; #FUNCTION# ====================================================================================================================
; Name...........: _Singleton
; Description ...: Check if no other occurence is running
; Syntax.........: _Singleton($sOccurenceName[, $iFlag = 0])
; Parameters ....: $sOccurenceName - String to identify the occurrence of the script.  This string may not contain any \ characters unless you are placing the object in a namespace.
;                  $iFlag          - Action if @error
;                  |0 - Exit if occurrence already exists
;                  |1 - Return if occurrence already exists
;                  |2 - Allow the object to be accessed by anybody in the system.  This is useful if specifying a "Global\" object in a multi-user environment.
; Return values .: Success      - 1
;                  Failure      - 0
; Author ........: Valik
; Modified.......:
; Remarks .......: You can place the object in a namespace by prefixing your object name with either "Global\" or "Local\".  "Global\" objects combined with the flag 2 are useful in multi-user environments.
; Related .......:
; Link ..........;
; Example .......; Yes
; ===============================================================================================================================
Func _Singleton($sOccurenceName, $iFlag = 0)
	Local Const $ERROR_ALREADY_EXISTS = 183
	Local Const $SECURITY_DESCRIPTOR_REVISION = 1
	Local $handle, $lastError, $pSecurityAttributes = 0

	If BitAND($iFlag, 2) Then
		; The size of SECURITY_DESCRIPTOR is 20 bytes.  We just
		; need a block of memory the right size, we aren't going to
		; access any members directly so it's not important what
		; the members are, just that the total size is correct.
		Local $structSecurityDescriptor = DllStructCreate("dword[5]")
		Local $pSecurityDescriptor = DllStructGetPtr($structSecurityDescriptor)
		; Initialize the security descriptor.
		Local $aRet = DllCall("advapi32.dll", "int", "InitializeSecurityDescriptor", _
				"ptr", $pSecurityDescriptor, "dword", $SECURITY_DESCRIPTOR_REVISION)
		If Not @error And $aRet[0] Then
			; Add the NULL DACL specifying access to everybody.
			$aRet = DllCall("advapi32.dll", "int", "SetSecurityDescriptorDacl", _
					"ptr", $pSecurityDescriptor, "int", 1, "ptr", 0, "int", 0)
			If Not @error And $aRet[0] Then
				; Create a SECURITY_ATTRIBUTES structure.
				Local $structSecurityAttributes = DllStructCreate("dword;ptr;int")
				; Assign the members.
				DllStructSetData($structSecurityAttributes, 1, DllStructGetSize($structSecurityAttributes))
				DllStructSetData($structSecurityAttributes, 2, $pSecurityDescriptor)
				DllStructSetData($structSecurityAttributes, 3, 0)
				; Everything went okay so update our pointer to point to our structure.
				$pSecurityAttributes = DllStructGetPtr($structSecurityAttributes)
			EndIf
		EndIf
	EndIf

	$handle = DllCall("kernel32.dll", "int", "CreateMutex", "ptr", $pSecurityAttributes, "long", 1, "str", $sOccurenceName)
	$lastError = DllCall("kernel32.dll", "int", "GetLastError")
	If $lastError[0] = $ERROR_ALREADY_EXISTS Then
		If BitAND($iFlag, 1) Then
			Return SetError($lastError[0], $lastError[0], 0)
		Else
			Exit -1
		EndIf
	EndIf
	Return $handle[0]
EndFunc   ;==>_Singleton

; #FUNCTION# ====================================================================================================================
; Name...........: _IsPressed
; Description ...: Check if key has been pressed
; Syntax.........: _IsPressed($sHexKey[, $vDLL = 'user32.dll'])
; Parameters ....: $sHexKey     - Key to check for
;                  $vDLL        - Handle to dll or default to user32.dll
; Return values .: True         - 1
;                  False        - 0
; Author ........: ezzetabi and Jon
; Modified.......:
; Remarks .......: If calling this function repeatidly, should open 'user32.dll' and pass in handle.
;                  Make sure to close at end of script
;                  01 Left mouse button
;                  02 Right mouse button
;                  04 Middle mouse button (three-button mouse)
;                  05 Windows 2000/XP: X1 mouse button
;                  06 Windows 2000/XP: X2 mouse button
;                  08 BACKSPACE key
;                  09 TAB key
;                  0C CLEAR key
;                  0D ENTER key
;                  10 SHIFT key
;                  11 CTRL key
;                  12 ALT key
;                  13 PAUSE key
;                  14 CAPS LOCK key
;                  1B ESC key
;                  20 SPACEBAR
;                  21 PAGE UP key
;                  22 PAGE DOWN key
;                  23 END key
;                  24 HOME key
;                  25 LEFT ARROW key
;                  26 UP ARROW key
;                  27 RIGHT ARROW key
;                  28 DOWN ARROW key
;                  29 SELECT key
;                  2A PRINT key
;                  2B EXECUTE key
;                  2C PRINT SCREEN key
;                  2D INS key
;                  2E DEL key
;                  30 0 key
;                  31 1 key
;                  32 2 key
;                  33 3 key
;                  34 4 key
;                  35 5 key
;                  36 6 key
;                  37 7 key
;                  38 8 key
;                  39 9 key
;                  41 A key
;                  42 B key
;                  43 C key
;                  44 D key
;                  45 E key
;                  46 F key
;                  47 G key
;                  48 H key
;                  49 I key
;                  4A J key
;                  4B K key
;                  4C L key
;                  4D M key
;                  4E N key
;                  4F O key
;                  50 P key
;                  51 Q key
;                  52 R key
;                  53 S key
;                  54 T key
;                  55 U key
;                  56 V key
;                  57 W key
;                  58 X key
;                  59 Y key
;                  5A Z key
;                  5B Left Windows key
;                  5C Right Windows key
;                  60 Numeric keypad 0 key
;                  61 Numeric keypad 1 key
;                  62 Numeric keypad 2 key
;                  63 Numeric keypad 3 key
;                  64 Numeric keypad 4 key
;                  65 Numeric keypad 5 key
;                  66 Numeric keypad 6 key
;                  67 Numeric keypad 7 key
;                  68 Numeric keypad 8 key
;                  69 Numeric keypad 9 key
;                  6A Multiply key
;                  6B Add key
;                  6C Separator key
;                  6D Subtract key
;                  6E Decimal key
;                  6F Divide key
;                  70 F1 key
;                  71 F2 key
;                  72 F3 key
;                  73 F4 key
;                  74 F5 key
;                  75 F6 key
;                  76 F7 key
;                  77 F8 key
;                  78 F9 key
;                  79 F10 key
;                  7A F11 key
;                  7B F12 key
;                  7C-7F F13 key - F16 key
;                  80H-87H F17 key - F24 key
;                  90 NUM LOCK key
;                  91 SCROLL LOCK key
;                  A0 Left SHIFT key
;                  A1 Right SHIFT key
;                  A2 Left CONTROL key
;                  A3 Right CONTROL key
;                  A4 Left MENU key
;                  A5 Right MENU key
;                  BA ;
;                  BB =
;                  BC ,
;                  BD -
;                  BE .
;                  BF /
;                  C0 `
;                  DB [
;                  DC \
;                  DD ]
; Related .......:
; Link ..........;
; Example .......; Yes
; ===============================================================================================================================
Func _IsPressed($sHexKey, $vDLL = 'user32.dll')
	; $hexKey must be the value of one of the keys.
	; _Is_Key_Pressed will return 0 if the key is not pressed, 1 if it is.
	Local $a_R = DllCall($vDLL, "int", "GetAsyncKeyState", "int", '0x' & $sHexKey)
	If Not @error And BitAND($a_R[0], 0x8000) = 0x8000 Then Return 1
	Return 0
EndFunc   ;==>_IsPressed