#include-once

; #INDEX# =======================================================================================================================
; Title .........: Code39 barcode generator
; AutoIt Version : 3.3.12.0
; Description ...: Create a Code39 barcode from supplied data
; Author(s) .....: David Williams (willichan)
; Dll ...........:
; ===============================================================================================================================

; #CURRENT# =====================================================================================================================
;_Bcode39_GenCode
; ===============================================================================================================================

; #INTERNAL_USE_ONLY# ===========================================================================================================
;__Bcode39_BitmapSetResolution
;__Bcode39_CreateImage
;__Bcode39_DataInit
;__Bcode39_EncodeData
; ===============================================================================================================================

; #VARIABLES# ===================================================================================================================
; ===============================================================================================================================

; #CONSTANTS# ===================================================================================================================
; ===============================================================================================================================

#include <GDIPlus.au3>
#include "StringSize.au3"; by Melba23 - http://www.autoitscript.com/forum/topic/114034-stringsize-m23-new-version-16-aug-11/
#include <File.au3>

; #FUNCTION# ====================================================================================================================
; Name ..........: _Bcode39_GenCode
; Description ...: Creates a Code39 barcode from supplied data
; Syntax ........: _Bcode39_GenCode($sData[, $iCheckChar = Default[, $sOutFile = Default[, $iDPI = Default[, $fMinBar = Default[,
;                  $fBarHeight = Default[, $iPrintText = Default]]]]]])
; Parameters ....: $sData               - A string value to encode into a Code39 barcode
;                  $iCheckChar          - [optional] Whether to include a check character.  Default is False.
;                  $sOutFile            - [optional] Where to write out the BMP file. Default is to create a random temp file
;                                         0 = Copy to the clipboard (planned, but not yet implemented)
;                                         1 = Create a randomly named temp file (Default)
;                                         String = Write to specified path\filename (assumes valid path and filename)
;                  $iDPI                - [optional] The dots-per-inch setting for the BMP file.  Default is 96
;                  $fMinBar             - [optional] Width of the narrowest bar, in inches (0.01 is the recommended standard for
;                                         most handheld scanners). Default is 0.01 inches
;                  $fBarHeight          - [optional] The height of the barcode, in inches (15% of the width or 0.25 inches
;                                         ,whichever is larger, is the standard). Default is to automatically calculate according
;                                         to this standard.
;                  $iPrintText          - [optional] Whether to print the data string under the barcode. Default is True.
; Return values .: String containing the location and name of the BMP file created, or "" if copied to the clipboard.
; Author ........: David E Williams (willichan)
; Modified ......:
; Remarks .......: Invalid characters in $sData will be skipped rather than generating an error
;                  Copying to the clipboard is not yet implemented.  Random temp file will be made instead.
; Required ......: http://www.autoitscript.com/forum/topic/114034-stringsize-m23-new-version-16-aug-11/
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _Bcode39_GenCode($sData, $iCheckChar = Default, $iExtended = Default, $sOutFile = Default, $iDPI = Default, $fMinBar = Default, $fBarHeight = Default, $iPrintText = Default)
	If (not IsString($sData)) Or (StringLen($sData) < 1) Then Return ""
	Local $vBarcodeObj[8], $t
	If $iCheckChar = Default Then $iCheckChar = False;
	If IsNumber($sOutFile) Then $sOutFile = 1 ;;; *** treat any number like a temp file, in case the user messed up ***
	If ($sOutFile = Default) Or ($sOutFile = 1) Then $sOutFile = _TempFile(@TempDir, "~", ".bmp", 7)
	If $iPrintText = Default Then $iPrintText = True ;print text at bottom of barcode
	If $iDPI = Default Then $iDPI = 96 ;Dots Per Inch used for the image file
	If $fMinBar = Default Then $fMinBar = 0.01 ;Minimum bar width in Inches
	Local $iModWidth = Ceiling($iDPI * $fMinBar) ;Number of Dots per Minimum Bar
	$vBarcodeObj[1] = $iModWidth
	$vBarcodeObj[2] = $sOutFile
	$vBarcodeObj[3] = $iDPI
	$vBarcodeObj[6] = $iCheckChar
	$vBarcodeObj[7] = $iExtended
	$t = __Bcode39_EncodeData($sData, $vBarcodeObj)
	If $iPrintText Then
		$vBarcodeObj[0] = $t[0] ;encoded text
	Else
		$vBarcodeObj[0] = ""
	EndIf
	$vBarcodeObj[5] = $t[1] ;bar encoding
	If $fBarHeight = Default Then
		$fBarHeight = ((StringLen($vBarcodeObj[5]) * $iModWidth) / $iDPI) * 0.15
		If $fBarHeight < 0.25 Then $fBarHeight = 0.25
		$fBarHeight = Ceiling($iDPI * $fBarHeight)
	EndIf
	$vBarcodeObj[4] = $fBarHeight
	__Bcode39_CreateImage($vBarcodeObj)
EndFunc   ;==>_Bcode39_GenCode

Func __Bcode39_BitmapSetResolution($hBitmap, $nDpiX, $nDpiY)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipBitmapSetResolution", "handle", $hBitmap, "float", $nDpiX, "float", $nDpiY)
	If @error Then Return SetError(@error, @extended, False)
	Return $aResult[0] = 0
EndFunc   ;==>__Bcode39_BitmapSetResolution

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __Bcode39_CreateImage
; Description ...: converts bar/space data into a BMP
; Syntax ........: __Bcode39_CreateImage(Byref $vBarcodeObj)
; Parameters ....: $vBarcodeObj         - Array containing barcode data
; Return values .: None
; Author ........: willichan
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func __Bcode39_CreateImage(ByRef $vBarcodeObj)
	Local $i, $cx
	Local Const $iFontMargin = 14
	;;create image
	_GDIPlus_Startup()
	Local $hBitmap = _GDIPlus_BitmapCreateFromScan0((StringLen($vBarcodeObj[5]) + 14) * $vBarcodeObj[1], $vBarcodeObj[4] + $iFontMargin, $GDIP_PXF01INDEXED)
	__Bcode39_BitmapSetResolution($hBitmap, $vBarcodeObj[3], $vBarcodeObj[3])
	Local $sCLSID = _GDIPlus_EncodersGetCLSID("bmp")
	Local $hBmpCtxt = _GDIPlus_ImageGetGraphicsContext($hBitmap)
	_GDIPlus_GraphicsSetSmoothingMode($hBmpCtxt, 0)
	_GDIPlus_GraphicsClear($hBmpCtxt, 0xFFFFFFFF)
	;;draw lines
	For $i = 1 To StringLen($vBarcodeObj[5])
		If StringMid($vBarcodeObj[5], $i, 1) = 1 Then
			$cx = (($i - 1) + 7) * $vBarcodeObj[1]
			_GDIPlus_GraphicsFillRect($hBmpCtxt, $cx, 0, $vBarcodeObj[1], $vBarcodeObj[4], 0)
			If @error Then ConsoleWrite("unable to draw bar " & $i & @CRLF)
		EndIf
	Next

	If $vBarcodeObj[0] <> "" Then
		$i = _StringSize($vBarcodeObj[0], 8, Default, Default, "Arial")
		$cx = ((StringLen($vBarcodeObj[5]) * $vBarcodeObj[1]) / 2) - ($i[2] / 2)
		_GDIPlus_GraphicsDrawString($hBmpCtxt, $vBarcodeObj[0], $cx, $vBarcodeObj[4] + 1, "Arial", 8)
		If @error Then ConsoleWrite("unable to draw text - " & @extended & @CRLF)
	EndIf

	;;save and dispose of image
	_GDIPlus_ImageSaveToFile($hBitmap, $vBarcodeObj[2])
	If @error Then ConsoleWrite("unable to write to file" & $vBarcodeObj[2] & @CRLF)
	_GDIPlus_GraphicsDispose($hBmpCtxt)
	_GDIPlus_BitmapDispose($hBitmap)
	_GDIPlus_Shutdown()
EndFunc   ;==>__Bcode39_CreateImage

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __Bcode39_DataInit
; Description ...: loads array with coding data
; Syntax ........: __Bcode39_DataInit(Byref $aEncoding, Byref $vBarcodeObj)
; Parameters ....: $aEncoding           - variable to fill with coding data array
;                  $vBarcodeObj         - array containing barcode object information
; Return values .: None
; Author ........: willichan
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func __Bcode39_DataInit(ByRef $aEncoding, ByRef $vBarcodeObj)
	Switch $vBarcodeObj[7]
		Case False ;Standard Set
				;;;Char, Bar Pattern, Value
			Local $aEnc[128][3] = [ _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				[' ', '100011101011101', 38], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['$', '100010001000101', 39], _
				['%', '101000100010001', 42], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['+', '100010100010001', 41], _
				['', '', 0], _
				['-', '100010101110111', 36], _
				['.', '111000101011101', 37], _
				['/', '100010001010001', 40], _
				['0', '101000111011101', 0], _
				['1', '111010001010111', 1], _
				['2', '101110001010111', 2], _
				['3', '111011100010101', 3], _
				['4', '101000111010111', 4], _
				['5', '111010001110101', 5], _
				['6', '101110001110101', 6], _
				['7', '101000101110111', 7], _
				['8', '111010001011101', 8], _
				['9', '101110001011101', 9], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['A', '111010100010111', 10], _
				['B', '101110100010111', 11], _
				['C', '111011101000101', 12], _
				['D', '101011100010111', 13], _
				['E', '111010111000101', 14], _
				['F', '101110111000101', 15], _
				['G', '101010001110111', 16], _
				['H', '111010100011101', 17], _
				['I', '101110100011101', 18], _
				['J', '101011100011101', 19], _
				['K', '111010101000111', 20], _
				['L', '101110101000111', 21], _
				['M', '111011101010001', 22], _
				['N', '101011101000111', 23], _
				['O', '111010111010001', 24], _
				['P', '101110111010001', 25], _
				['Q', '101010111000111', 26], _
				['R', '111010101110001', 27], _
				['S', '101110101110001', 28], _
				['T', '101011101110001', 29], _
				['U', '111000101010111', 30], _
				['V', '100011101010111', 31], _
				['W', '111000111010101', 32], _
				['X', '100010111010111', 33], _
				['Y', '111000101110101', 34], _
				['Z', '100011101110101', 35], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0], _
				['', '', 0]  _
			]
		Case True  ;Extended Set
				;;;Char, Bar Pattern, Value
			Local $aEnc[128][3] = [ _
				['', '1010001000100010111000101010111', 72], _
				['', '1000100010001010111010100010111', 49], _
				['', '1000100010001010101110100010111', 50], _
				['', '1000100010001010111011101000101', 51], _
				['', '1000100010001010101011100010111', 52], _
				['', '1000100010001010111010111000101', 53], _
				['', '1000100010001010101110111000101', 54], _
				['', '1000100010001010101010001110111', 55], _
				['', '1000100010001010111010100011101', 56], _
				['', '1000100010001010101110100011101', 57], _
				['', '1000100010001010101011100011101', 58], _
				['', '1000100010001010111010101000111', 59], _
				['', '1000100010001010101110101000111', 60], _
				['', '1000100010001010111011101010001', 61], _
				['', '1000100010001010101011101000111', 62], _
				['', '1000100010001010111010111010001', 63], _
				['', '1000100010001010101110111010001', 64], _
				['', '1000100010001010101010111000111', 65], _
				['', '1000100010001010111010101110001', 66], _
				['', '1000100010001010101110101110001', 67], _
				['', '1000100010001010101011101110001', 68], _
				['', '1000100010001010111000101010111', 69], _
				['', '1000100010001010100011101010111', 70], _
				['', '1000100010001010111000111010101', 71], _
				['', '1000100010001010100010111010111', 72], _
				['', '1000100010001010111000101110101', 73], _
				['', '1000100010001010100011101110101', 74], _
				['', '1010001000100010111010100010111', 52], _
				['', '1010001000100010101110100010111', 53], _
				['', '1010001000100010111011101000101', 54], _
				['', '1010001000100010101011100010111', 55], _
				['', '1010001000100010111010111000101', 56], _
				[' ', '100011101011101', 38], _
				['!', '1000100010100010111010100010111', 50], _
				[Chr(34), '1000100010100010101110100010111', 51], _
				['#', '1000100010100010111011101000101', 52], _
				['$', '1000100010100010101011100010111', 53], _
				['%', '1000100010100010111010111000101', 54], _
				['&', '1000100010100010101110111000101', 55], _
				[Chr(39), '1000100010100010101010001110111', 56], _
				['(', '1000100010100010111010100011101', 57], _
				[')', '1000100010100010101110100011101', 58], _
				['*', '1000100010100010101011100011101', 59], _
				['+', '1000100010100010111010101000111', 60], _
				[',', '1000100010100010101110101000111', 61], _
				['-', '100010101110111', 36], _
				['.', '111000101011101', 37], _
				['/', '1000100010100010111010111010001', 64], _
				['0', '101000111011101', 0], _
				['1', '111010001010111', 1], _
				['2', '101110001010111', 2], _
				['3', '111011100010101', 3], _
				['4', '101000111010111', 4], _
				['5', '111010001110101', 5], _
				['6', '101110001110101', 6], _
				['7', '101000101110111', 7], _
				['8', '111010001011101', 8], _
				['9', '101110001011101', 9], _
				[':', '1000100010100010100011101110101', 75], _
				[';', '1010001000100010101110111000101', 57], _
				['<', '1010001000100010101010001110111', 58], _
				['=', '1010001000100010111010100011101', 59], _
				['>', '1010001000100010101110100011101', 60], _
				['?', '1010001000100010101011100011101', 61], _
				['@', '1010001000100010100011101010111', 73], _
				['A', '111010100010111', 10], _
				['B', '101110100010111', 11], _
				['C', '111011101000101', 12], _
				['D', '101011100010111', 13], _
				['E', '111010111000101', 14], _
				['F', '101110111000101', 15], _
				['G', '101010001110111', 16], _
				['H', '111010100011101', 17], _
				['I', '101110100011101', 18], _
				['J', '101011100011101', 19], _
				['K', '111010101000111', 20], _
				['L', '101110101000111', 21], _
				['M', '111011101010001', 22], _
				['N', '101011101000111', 23], _
				['O', '111010111010001', 24], _
				['P', '101110111010001', 25], _
				['Q', '101010111000111', 26], _
				['R', '111010101110001', 27], _
				['S', '101110101110001', 28], _
				['T', '101011101110001', 29], _
				['U', '111000101010111', 30], _
				['V', '100011101010111', 31], _
				['W', '111000111010101', 32], _
				['X', '100010111010111', 33], _
				['Y', '111000101110101', 34], _
				['Z', '100011101110101', 35], _
				['[', '1010001000100010111010101000111', 62], _
				['\', '1010001000100010101110101000111', 63], _
				[']', '1010001000100010111011101010001', 64], _
				['^', '1010001000100010101011101000111', 65], _
				['_', '1010001000100010111010111010001', 66], _
				['`', '1010001000100010111000111010101', 74], _
				['a', '1000101000100010111010100010111', 51], _
				['b', '1000101000100010101110100010111', 52], _
				['c', '1000101000100010111011101000101', 53], _
				['d', '1000101000100010101011100010111', 54], _
				['e', '1000101000100010111010111000101', 55], _
				['f', '1000101000100010101110111000101', 56], _
				['g', '1000101000100010101010001110111', 57], _
				['h', '1000101000100010111010100011101', 58], _
				['i', '1000101000100010101110100011101', 59], _
				['j', '1000101000100010101011100011101', 60], _
				['k', '1000101000100010111010101000111', 61], _
				['l', '1000101000100010101110101000111', 62], _
				['m', '1000101000100010111011101010001', 63], _
				['n', '1000101000100010101011101000111', 64], _
				['o', '1000101000100010111010111010001', 65], _
				['p', '1000101000100010101110111010001', 66], _
				['q', '1000101000100010101010111000111', 67], _
				['r', '1000101000100010111010101110001', 68], _
				['s', '1000101000100010101110101110001', 69], _
				['t', '1000101000100010101011101110001', 70], _
				['u', '1000101000100010111000101010111', 71], _
				['v', '1000101000100010100011101010111', 72], _
				['w', '1000101000100010111000111010101', 73], _
				['x', '1000101000100010100010111010111', 74], _
				['y', '1000101000100010111000101110101', 75], _
				['z', '1000101000100010100011101110101', 76], _
				['{', '1010001000100010101110111010001', 67], _
				['|', '1010001000100010101010111000111', 68], _
				['}', '1010001000100010111010101110001', 69], _
				['~', '1010001000100010101110101110001', 70], _
				['', '1010001000100010101011101110001', 71]   _
			]
	EndSwitch
	$aEncoding = $aEnc
	$aEnc = 0
EndFunc


; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __Bcode39_EncodeData
; Description ...: encodes data into bar/space code for generating barcode
; Syntax ........: __Bcode39_EncodeData($sData, Byref $vBarcodeObj)
; Parameters ....: $sData               - data to be converted to a barcode
;                  $vBarcodeObj         - array containing barcode object information
; Return values .: array contining encoded data
;                      [0] - text for placing at the bottom of the barcode
;                      [1] - bar/space coding to generate barcode
; Author ........: willichan
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func __Bcode39_EncodeData($sData, ByRef $vBarcodeObj)
	Local Const $sStartStop = "100010111011101"
	Local $aReturn[2] = ["", $sStartStop & "0"]
	Local $iCheckVal = 0
	Local $aEncoding, $iTmp

	; make text uppercase if extended is not used to allow more valid characters
	If Not $vBarcodeObj[7] Then $sData = StringUpper($sData)

	__Bcode39_DataInit($aEncoding, $vBarcodeObj)
	Local $iLoop
	For $iLoop = 1 to StringLen($sData)
		$iTmp = Asc(StringMid($sData, $iLoop, 1))
		If $iTmp > 127 Then ContinueLoop
		$aReturn[0] &= $aEncoding[$iTmp][0]
		$aReturn[1] &= $aEncoding[$iTmp][1] & "0"
		$iCheckVal += $aEncoding[$iTmp][2]
	Next
	If $vBarcodeObj[6] Then ;Add Checkcharacter if selected.
		$iCheckVal = Mod($iCheckVal, 43)
		$aReturn[1] &= $aEncoding[$iCheckVal][1]
	EndIf
	$aReturn[1] &= $sStartStop
	$aEncoding = 0
	Return $aReturn
EndFunc   ;==>__Bcode39_EncodeData

