#include-once

#Region Additional Drawing & Measuring

Func _GDIPlus_GraphicsDrawStringRect($hGraphics, $sString, $nX, $nY, $nWidth, $nHeight, $sFont = "Arial", $nSize = 10, $iStyle = 0, $iAlign = 0, $iVertAlign = 0, $iFormat = 0, $iColor = 0xFF000000)
	Local $hBrush = _GDIPlus_BrushCreateSolid($iColor)
	Local $hFormat = _GDIPlus_StringFormatCreate($iFormat)
	_GDIPlus_StringFormatSetAlign($hFormat, $iAlign)
	_GDIPlus_StringFormatSetLineAlign($hFormat, $iVertAlign)
	If $sFont = $v4Setting_UI_FontFamily Or $sFont = "vPaint UI Glyphs" Then
		Local $hCollection = _GDIPlus_FontPrivateCreateCollection()
		_GDIPlus_FontPrivateAddFont($hCollection, @ScriptDir&"\Data\Fonts\"&$sFont&".ttf")
		Local $hFamily = _GDIPlus_FontFamilyCreateFromCollection($sFont, $hCollection)
	Else
		Local $hFamily = _GDIPlus_FontFamilyCreate($sFont)
	EndIf
	Local $hFont = _GDIPlus_FontCreate($hFamily, $nSize, $iStyle)
	Local $tLayout = _GDIPlus_RectFCreate($nX, $nY, $nWidth, $nHeight)
	Local $aInfo = _GDIPlus_GraphicsMeasureString($hGraphics, $sString, $hFont, $tLayout, $hFormat)
	If @error Then Return SetError(@error, @extended, 0)
	Local $aResult = _GDIPlus_GraphicsDrawStringEx($hGraphics, $sString, $hFont, $aInfo[0], $hFormat, $hBrush)
	Local $iError = @error, $iExtended = @extended
	_GDIPlus_FontDispose($hFont)
	_GDIPlus_FontFamilyDispose($hFamily)
	If IsDeclared("hCollection") Then _GDIPlus_FontPrivateCollectionDispose($hCollection)
	_GDIPlus_StringFormatDispose($hFormat)
	_GDIPlus_BrushDispose($hBrush)
	Return SetError($iError, $iExtended, $aResult)
EndFunc

Func _GDIPlus_GraphicsGetStringBounds($sString, $sFont = "Arial", $fSize = 10, $iStyle = 0, $iFormat = 0)
	Local $tGraphics = _GDIPlus_GraphicsCreateFromHWND(_WinAPI_GetDesktopWindow())
	Local $hBrush = _GDIPlus_BrushCreateSolid()
	Local $hFormat = _GDIPlus_StringFormatCreate($iFormat)
	If $sFont = $v4Setting_UI_FontFamily Or $sFont = "vPaint UI Glyphs" Then
		Local $hCollection = _GDIPlus_FontPrivateCreateCollection()
		_GDIPlus_FontPrivateAddFont($hCollection, @ScriptDir&"\Data\Fonts\"&$sFont&".ttf")
		Local $hFamily = _GDIPlus_FontFamilyCreateFromCollection($sFont, $hCollection)
	Else
		Local $hFamily = _GDIPlus_FontFamilyCreate($sFont)
	EndIf
	Local $hFont = _GDIPlus_FontCreate($hFamily, $fSize, $iStyle)
	Local $tLayout = _GDIPlus_RectFCreate(0, 0, 0.0, 0.0)
	Local $aInfo = _GDIPlus_GraphicsMeasureString($tGraphics, $sString, $hFont, $tLayout, $hFormat)
	If @error Then Return SetError(@error, @extended, 0)

	Local $aResult = [$aInfo[0].Width, $aInfo[0].Height]

	_GDIPlus_FontDispose($hFont)
	_GDIPlus_FontFamilyDispose($hFamily)
	If IsDeclared("hCollection") Then _GDIPlus_FontPrivateCollectionDispose($hCollection)
	_GDIPlus_StringFormatDispose($hFormat)
	_GDIPlus_BrushDispose($hBrush)
	_GDIPlus_GraphicsDispose($tGraphics)
	Return SetError(0, 0, $aResult)
EndFunc

#EndRegion

#region GDIP.au3
; some magic from GDIP.au3
Func _GDIPlus_ImageClone($hImage)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipCloneImage", "hwnd", $hImage, "int*", 0)

	If @error Then Return SetError(@error, @extended, 0)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[2]
EndFunc   ;==>_GDIPlus_ImageClone

Func _GDIPlus_BitmapSetResolution($hBitmap, $nDpiX, $nDpiY)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipBitmapSetResolution", "hwnd", $hBitmap, "float", $nDpiX, "float", $nDpiY)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_BitmapSetResolution

Func _GDIPlus_FontCollectionCreate()
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipNewInstalledFontCollection", "int*", 0)

	If @error Then Return SetError(@error, @extended, 0)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[1]
EndFunc   ;==>_GDIPlus_FontCollectionCreate

Func _GDIPlus_FontCollectionGetFamilyCount($hFontCollection)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetFontCollectionFamilyCount", "hwnd", $hFontCollection, "int*", 0)

	If @error Then Return SetError(@error, @extended, 0)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[2]
EndFunc   ;==>_GDIPlus_FontCollectionGetFamilyCount

Func _GDIPlus_FontCollectionGetFamilyList($hFontCollection)
	Local $iI, $iCount, $tFontFamilies, $pFontFamilies, $aFontFamilies[1], $aResult

	$iCount = _GDIPlus_FontCollectionGetFamilyCount($hFontCollection)
	If @error Then Return SetError(@error, @extended, -1)

	If $GDIP_STATUS Then
		$GDIP_ERROR = 1
		Return -1
	ElseIf $iCount = 0 Then
		$GDIP_ERROR = 2
		Return -1
	EndIf

	$tFontFamilies = DllStructCreate("hwnd[" & $iCount & "]")
	$pFontFamilies = DllStructGetPtr($tFontFamilies)
	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetFontCollectionFamilyList", "hwnd", $hFontCollection, "int", $iCount, "ptr", $pFontFamilies, "int*", 0)
	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then
		$GDIP_ERROR = 3
		Return -1
	EndIf

	ReDim $aFontFamilies[$iCount + 1]
	$aFontFamilies[0] = $iCount
	For $iI = 1 To $iCount
		$aFontFamilies[$iI] = DllStructGetData($tFontFamilies, 1, $iI)
	Next

	Return $aFontFamilies
EndFunc   ;==>_GDIPlus_FontCollectionGetFamilyList

Func _GDIPlus_FontFamilyGetFamilyName($hFontFamily, $iLANGID = 0)
	Local $tName, $pName, $sName, $aResult

	$tName = DllStructCreate("wchar[" & 32 & "]")
	$pName = DllStructGetPtr($tName)
	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetFamilyName", "hwnd", $hFontFamily, "ptr", $pName, "ushort", $iLANGID)

	If @error Then Return SetError(@error, @extended, 0)
	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then Return 0

	$sName = DllStructGetData($tName, 1)
	Return $sName
EndFunc   ;==>_GDIPlus_FontFamilyGetFamilyName

Func _GDIPlus_ImageForceValidation($hImage)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipImageForceValidation", "hwnd", $hImage)

	If @error Then Return SetError(@error, @extended, -1)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageForceValidation

Func _GDIPlus_ImageGetAllPropertyItems($hImage)
	Local $iI, $iCount, $tBuffer, $pBuffer, $iBuffer, $tPropertyItem, $aSize, $aPropertyItems[1][1], $aResult

	$aSize = _GDIPlus_ImageGetPropertySize($hImage)
	If @error Then Return SetError(@error, @extended, -1)

	If $GDIP_STATUS Then
		$GDIP_ERROR = 1
		Return -1
	ElseIf $aSize[1] = 0 Then
		$GDIP_ERROR = 2
		Return -1
	EndIf

	$iBuffer = $aSize[0]
	$tBuffer = DllStructCreate("byte[" & $iBuffer & "]")
	$pBuffer = DllStructGetPtr($tBuffer)
	$iCount = $aSize[1]

	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetAllPropertyItems", "hwnd", $hImage, "uint", $iBuffer, "uint", $iCount, "ptr", $pBuffer)
	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then
		$GDIP_ERROR = 3
		Return -1
	EndIf

	ReDim $aPropertyItems[$iCount + 1][4]
	$aPropertyItems[0][0] = $iCount

	For $iI = 1 To $iCount
		$tPropertyItem = DllStructCreate($tagGDIPPROPERTYITEM, $pBuffer)
		$aPropertyItems[$iI][0] = DllStructGetData($tPropertyItem, "id")
		$aPropertyItems[$iI][1] = DllStructGetData($tPropertyItem, "length")
		$aPropertyItems[$iI][2] = DllStructGetData($tPropertyItem, "type")
		$aPropertyItems[$iI][3] = DllStructGetData($tPropertyItem, "value")
		$pBuffer += DllStructGetSize($tPropertyItem)
	Next

	Return $aPropertyItems
EndFunc   ;==>_GDIPlus_ImageGetAllPropertyItems

Func _GDIPlus_ImageGetBounds($hImage)
	Local $tRectF, $pRectF, $iI, $aRectF[4], $aResult

	$tRectF = DllStructCreate($tagGDIPRECTF)
	$pRectF = DllStructGetPtr($tRectF)
	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetImageBounds", "hwnd", $hImage, "ptr", $pRectF, "int*", 0)
	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then Return -1

	For $iI = 1 To 4
		$aRectF[$iI - 1] = DllStructGetData($tRectF, $iI)
	Next
	Return $aRectF
EndFunc   ;==>_GDIPlus_ImageGetBounds

Func _GDIPlus_GraphicsDrawImageRectRectIA($hGraphics, $hImage, $nSrcX, $nSrcY, $nSrcWidth, $nSrcHeight, $nDstX, $nDstY, $nDstWidth, $nDstHeight, $hImageAttributes = 0, $iUnit = 2)
	Local $aResult = DllCall($__g_hGDIPDll, "int", "GdipDrawImageRectRect", "hwnd", $hGraphics, "hwnd", $hImage, "float", $nDstX, "float", _
			$nDstY, "float", $nDstWidth, "float", $nDstHeight, "float", $nSrcX, "float", $nSrcY, "float", $nSrcWidth, "float", _
			$nSrcHeight, "int", $iUnit, "hwnd", $hImageAttributes, "int", 0, "int", 0)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_GraphicsDrawImageRectRectIA

Func _GDIPlus_ImageGetFrameCount($hImage, $sDimensionID)
	Local $tGUID, $pGUID, $aResult

	$tGUID = _WinAPI_GUIDFromString($sDimensionID)
	$pGUID = DllStructGetPtr($tGUID)
	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipImageGetFrameCount", "hwnd", $hImage, "ptr", $pGUID, "uint*", 0)

	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then Return -1
	Return $aResult[3]
EndFunc   ;==>_GDIPlus_ImageGetFrameCount

Func _GDIPlus_ImageGetFrameDimensionsCount($hImage)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipImageGetFrameDimensionsCount", "hwnd", $hImage, "uint*", 0)

	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then Return -1
	Return $aResult[2]
EndFunc   ;==>_GDIPlus_ImageGetFrameDimensionsCount

Func _GDIPlus_ImageGetFrameDimensionsList($hImage)
	Local $iI, $iCount, $tBuffer, $pBuffer, $aPropertyIDs[1], $aResult

	$iCount = _GDIPlus_ImageGetFrameDimensionsCount($hImage)
	If @error Then Return SetError(@error, @extended, -1)

	If $GDIP_STATUS Then
		$GDIP_ERROR = 1
		Return -1
	ElseIf $iCount = 0 Then
		$GDIP_ERROR = 2
		Return -1
	EndIf

	$tBuffer = DllStructCreate("byte[" & $iCount * 16 & "]")
	$pBuffer = DllStructGetPtr($tBuffer)
	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipImageGetFrameDimensionsList", "hwnd", $hImage, "ptr", $pBuffer, "int", $iCount)
	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then
		$GDIP_ERROR = 3
		Return -1
	EndIf

	ReDim $aPropertyIDs[$iCount + 1]
	$aPropertyIDs[0] = $iCount

	For $iI = 1 To $iCount
		$aPropertyIDs[$iI] = _WinAPI_StringFromGUID($pBuffer)
		$pBuffer += 16
	Next
	Return $aPropertyIDs
EndFunc   ;==>_GDIPlus_ImageGetFrameDimensionsList

Func _GDIPlus_ImageGetPalette($hImage)
	Local $iCount, $iColorPalette, $tColorPalette, $pColorPalette, $aResult

	$iColorPalette = _GDIPlus_ImageGetPaletteSize($hImage)
	If @error Then Return SetError(@error, @extended, -1)

	If $GDIP_STATUS Then
		$GDIP_ERROR = 1
		Return -1
	ElseIf $iColorPalette = 0 Then
		$GDIP_ERROR = 2
		Return -1
	EndIf

	$iCount = ($iColorPalette - 8) / 4
	$tColorPalette = DllStructCreate("uint Flags;uint Count;uint Entries[" & $iCount & "];")
	$pColorPalette = DllStructGetPtr($tColorPalette)
	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetImagePalette", "hwnd", $hImage, "ptr", $pColorPalette, "int", $iColorPalette)
	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then Return -1
	Return $tColorPalette
EndFunc   ;==>_GDIPlus_ImageGetPalette

Func _GDIPlus_ImageGetPaletteSize($hImage)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetImagePaletteSize", "hwnd", $hImage, "int*", 0)

	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then Return -1
	Return $aResult[2]
EndFunc   ;==>_GDIPlus_ImageGetPaletteSize

Func _GDIPlus_ImageGetPropertyCount($hImage)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetPropertyCount", "hwnd", $hImage, "uint*", 0)

	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then Return -1
	Return $aResult[2]
EndFunc   ;==>_GDIPlus_ImageGetPropertyCount

Func _GDIPlus_ImageGetPropertyIdList($hImage)
	Local $iI, $iCount, $tProperties, $pProperties, $aProperties[1], $aResult

	$iCount = _GDIPlus_ImageGetPropertyCount($hImage)
	If @error Then Return SetError(@error, @extended, -1)

	If $GDIP_STATUS Then
		$GDIP_ERROR = 1
		Return -1
	ElseIf $iCount = 0 Then
		$GDIP_ERROR = 2
		Return -1
	EndIf

	$tProperties = DllStructCreate("uint[" & $iCount & "]")
	$pProperties = DllStructGetPtr($tProperties)
	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetPropertyIdList", "hwnd", $hImage, "int", $iCount, "ptr", $pProperties)
	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then
		$GDIP_ERROR = 3
		Return -1
	EndIf

	ReDim $aProperties[$iCount + 1]
	$aProperties[0] = $iCount

	For $iI = 1 To $iCount
		$aProperties[$iI] = DllStructGetData($tProperties, 1, $iI)
	Next
	Return $aProperties
EndFunc   ;==>_GDIPlus_ImageGetPropertyIdList

Func _GDIPlus_ImageGetPropertyItem($hImage, $iPropID)
	Local $iBuffer, $tBuffer, $pBuffer, $tPropertyItem, $aResult

	$iBuffer = _GDIPlus_ImageGetPropertyItemSize($hImage, $iPropID)
	If @error Then Return SetError(@error, @extended, -1)

	If $GDIP_STATUS Then
		$GDIP_ERROR = 1
		Return -1
	ElseIf $iBuffer = 0 Then
		$GDIP_ERROR = 2
		Return -1
	EndIf

	$tBuffer = DllStructCreate("byte[" & $iBuffer & "]")
	$pBuffer = DllStructGetPtr($tBuffer)
	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetPropertyItem", "hwnd", $hImage, "uint", $iPropID, "uint", $iBuffer, "ptr", $pBuffer)
	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then
		$GDIP_ERROR = 3
		Return -1
	EndIf

	$tPropertyItem = DllStructCreate($tagGDIPPROPERTYITEM, $pBuffer)
	Return $tPropertyItem
EndFunc   ;==>_GDIPlus_ImageGetPropertyItem

Func _GDIPlus_ImageGetPropertyItemSize($hImage, $iPropID)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetPropertyItemSize", "hwnd", $hImage, "uint", $iPropID, "uint*", 0)

	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then Return -1
	Return $aResult[3]
EndFunc   ;==>_GDIPlus_ImageGetPropertyItemSize

Func _GDIPlus_ImageGetPropertySize($hImage)
	Local $aSize[2], $aResult

	$aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetPropertySize", "hwnd", $hImage, "uint*", 0, "uint*", 0)
	If @error Then Return SetError(@error, @extended, -1)

	$GDIP_STATUS = $aResult[0]
	If $GDIP_STATUS Then Return -1

	$aSize[0] = $aResult[2]
	$aSize[1] = $aResult[3]
	Return $aSize
EndFunc   ;==>_GDIPlus_ImageGetPropertySize

#endregion

#Region ColorMatrix
Func _GDIPlus_ColorMatrixMultiply(ByRef $tCM1, $tCM2, $iOrder = 0)
	Local $iX, $iY, $iI, $iOffset, $nT, $tA, $tB, $tTmpCM

	If $iOrder Then
		$tA = $tCM2
		$tB = $tCM1
	Else
		$tA = $tCM1
		$tB = $tCM2
	EndIf

	$tTmpCM = DllStructCreate($tagGDIPCOLORMATRIX)

	For $iY = 0 To 4
		For $iX = 0 To 3
			$nT = 0

			For $iI = 0 To 4
				$nT += DllStructGetData($tB, "m", $iY * 5 + $iI + 1) * DllStructGetData($tA, "m", $iI * 5 + $iX + 1)
			Next
			DllStructSetData($tTmpCM, "m", $nT, $iY * 5 + $iX + 1)
		Next
	Next

	For $iY = 0 To 4
		For $iX = 0 To 3
			$iOffset = $iY * 5 + $iX + 1
			DllStructSetData($tCM1, "m", DllStructGetData($tTmpCM, "m", $iOffset), $iOffset)
		Next
	Next
EndFunc

Func _GDIPlus_ColorMatrixRotateBlue(ByRef $tCM, $nPhi, $iOrder = 0)
	_GDIPlus_ColorMatrixRotateColor($tCM, $nPhi, 1, 0, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixRotateColor(ByRef $tCM, $nPhi, $iX, $iY, $iOrder = 0)
	Local $nVal, $tRotateCM

	$nPhi *= $GDIP_RAD
	$tRotateCM = _GDIPlus_ColorMatrixCreate()

	$nVal = Cos($nPhi)
	DllStructSetData($tRotateCM, "m", $nVal, $iX * 5 + $iX + 1)
	DllStructSetData($tRotateCM, "m", $nVal, $iY * 5 + $iY + 1)

	$nVal = Sin($nPhi)
	DllStructSetData($tRotateCM, "m", $nVal, $iY * 5 + $iX + 1)
	DllStructSetData($tRotateCM, "m", -$nVal, $iX * 5 + $iY + 1)
	_GDIPlus_ColorMatrixMultiply($tCM, $tRotateCM, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixRotateGreen(ByRef $tCM, $nPhi, $iOrder = 0)
	_GDIPlus_ColorMatrixRotateColor($tCM, $nPhi, 0, 2, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixRotateHue(ByRef $tCM, $tPreHueCM, $tPostHueCM, $nPhi)
	_GDIPlus_ColorMatrixMultiply($tCM, $tPreHueCM, 1)
	_GDIPlus_ColorMatrixRotateBlue($tCM, $nPhi, 1)
	_GDIPlus_ColorMatrixMultiply($tCM, $tPostHueCM, 1)
EndFunc

Func _GDIPlus_ColorMatrixRotateRed(ByRef $tCM, $nPhi, $iOrder = 0)
	_GDIPlus_ColorMatrixRotateColor($tCM, $nPhi, 2, 1, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixScale(ByRef $tCM, $nScaleRed, $nScaleGreen, $nScaleBlue, $nScaleAlpha = 1, $iOrder = 0)
	Local $tScaleCM

	$tScaleCM = _GDIPlus_ColorMatrixCreateScale($nScaleRed, $nScaleGreen, $nScaleBlue, $nScaleAlpha)
	_GDIPlus_ColorMatrixMultiply($tCM, $tScaleCM, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixSetSaturation(ByRef $tCM, $nSat, $iOrder = 0)
	Local $tSaturateCM

	$tSaturateCM = _GDIPlus_ColorMatrixCreateSaturation($nSat)
	_GDIPlus_ColorMatrixMultiply($tCM, $tSaturateCM, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixShearBlue(ByRef $tCM, $nRed, $nGreen, $iOrder = 0)
	_GDIPlus_ColorMatrixShearColor($tCM, 2, 0, $nRed, 1, $nGreen, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixShearColor(ByRef $tCM, $iX, $iY1, $nD1, $iY2, $nD2, $iOrder = 0)
	Local $tShearCM

	$tShearCM = _GDIPlus_ColorMatrixCreate()

	DllStructSetData($tShearCM, "m", $nD1, $iY1 * 5 + $iX + 1)
	DllStructSetData($tShearCM, "m", $nD2, $iY2 * 5 + $iX + 1)

	_GDIPlus_ColorMatrixMultiply($tCM, $tShearCM, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixShearGreen(ByRef $tCM, $nRed, $nBlue, $iOrder = 0)
	_GDIPlus_ColorMatrixShearColor($tCM, 1, 0, $nRed, 2, $nBlue, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixShearRed(ByRef $tCM, $nGreen, $nBlue, $iOrder = 0)
	_GDIPlus_ColorMatrixShearColor($tCM, 0, 1, $nGreen, 2, $nBlue, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixTranslate(ByRef $tCM, $nOffsetRed, $nOffsetGreen, $nOffsetBlue, $nOffsetAlpha = 0, $iOrder = 0)
	Local $tTranslateCM

	$tTranslateCM = _GDIPlus_ColorMatrixCreateTranslate($nOffsetRed, $nOffsetGreen, $nOffsetBlue, $nOffsetAlpha)
	_GDIPlus_ColorMatrixMultiply($tCM, $tTranslateCM, $iOrder)
EndFunc

Func _GDIPlus_ColorMatrixTransformLuminance(ByRef $tCM, $nRed = $GDIP_RLUM, $nGreen = $GDIP_GLUM, $nBlue = $GDIP_BLUM, $nAlpha = 1)
	Local $iX, $iY, $aRet[4], $aLums[4] = [$nRed, $nGreen, $nBlue, $nAlpha]

	For $iX = 0 To 3
		$aRet[$iX] = DllStructGetData($tCM, "m", 21 + $iX)

		For $iY = 0 To 3
			$aRet[$iX] += $aLums[$iY] * DllStructGetData($tCM, "m", $iY * 5 + $iX + 1)
		Next
	Next

	Return $aRet
EndFunc
#EndRegion

#region IA funcs
; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesGetAdjustedPalette
; Description ...: Adjusts the colors in a palette according to the adjustment settings of a specified category
; Syntax.........: _GDIPlus_ImageAttributesGetAdjustedPalette($hImageAttributes, $pColorPalette[, $iColorAdjustType = 0])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $pColorPalette	 - Pointer to a $tagGDIPCOLORPALETTE structure that on input, contains the palette to be
;                  +adjusted and, on output, receives the adjusted palette
;                  $iColorAdjustType - The category whose adjustment settings will be applied to the palette:
;                  |0 - Color or grayscale adjustment applies to all categories that do not have adjustment settings of their own
;                  |1 - Color or grayscale adjustment applies to bitmapped images
;                  |2 - Color or grayscale adjustment applies to brush operations in metafiles
;                  |3 - Color or grayscale adjustment applies to pen operations in metafiles
;                  |4 - Color or grayscale adjustment applies to text drawn in metafiles
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: None
; Related .......: $tagGDIPCOLORPALETTE
; Link ..........; @@MsdnLink@@ GdipGetImageAttributesAdjustedPalette
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesGetAdjustedPalette($hImageAttributes, $pColorPalette, $iColorAdjustType = 0)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipGetImageAttributesAdjustedPalette", "hwnd", $hImageAttributes, "ptr", $pColorPalette, "int", $iColorAdjustType)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesGetAdjustedPalette

; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesReset
; Description ...: Clears all color- and grayscale-adjustment settings for a specified category
; Syntax.........: _GDIPlus_ImageAttributesReset($hImageAttributes[, $iColorAdjustType = 0])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $iColorAdjustType - The category for which color adjustment is reset:
;                  |0 - Color or grayscale adjustment applies to all categories that do not have adjustment settings of their own
;                  |1 - Color or grayscale adjustment applies to bitmapped images
;                  |2 - Color or grayscale adjustment applies to brush operations in metafiles
;                  |3 - Color or grayscale adjustment applies to pen operations in metafiles
;                  |4 - Color or grayscale adjustment applies to text drawn in metafiles
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: None
; Related .......: None
; Link ..........; @@MsdnLink@@ GdipResetImageAttributes
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesReset($hImageAttributes, $iColorAdjustType = 0)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipResetImageAttributes", "hwnd", $hImageAttributes, "int", $iColorAdjustType)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesReset

; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesSetGamma
; Description ...: Sets or clears the gamma value for a specified category
; Syntax.........: _GDIPlus_ImageAttributesSetGamma($hImageAttributes[, $iColorAdjustType = 0[, $fEnable = False[, $nGamma = 0]]])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $iColorAdjustType - The category for which the gamma value is set or cleared:
;                  |0 - Gamma adjustment applies to all categories that do not have adjustment settings of their own
;                  |1 - Gamma adjustment applies to bitmapped images
;                  |2 - Gamma adjustment applies to brush operations in metafiles
;                  |3 - Gamma adjustment applies to pen operations in metafiles
;                  |4 - Gamma adjustment applies to text drawn in metafiles
;                  $fEnable		     - If True, gamma correction for the specified category is applied, otherwise cleared
;                  $nGamma		     - Real number that specifies the gamma value
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: Gamma correction controls the overall brightness of an image, typical range is from 1.0  to  2.2;  however,
;                  +values from 0.1 to 5.0 could prove useful under some circumstances
; Related .......: None
; Link ..........; @@MsdnLink@@ GdipSetImageAttributesGamma
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesSetGamma($hImageAttributes, $iColorAdjustType = 0, $fEnable = False, $nGamma = 0)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipSetImageAttributesGamma", "hwnd", $hImageAttributes, "int", $iColorAdjustType, "int", $fEnable, "float", $nGamma)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesSetGamma

; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesSetNoOp
; Description ...: Turns on or off color adjustment for a specified category
; Syntax.........: _GDIPlus_ImageAttributesSetNoOp($hImageAttributes[, $iColorAdjustType = 0[, $fEnable = True]])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $iColorAdjustType - The category for which color correction is turned on or off:
;                  |0 - Color adjustment applies to all categories that do not have adjustment settings of their own
;                  |1 - Color adjustment applies to bitmapped images
;                  |2 - Color adjustment applies to brush operations in metafiles
;                  |3 - Color adjustment applies to pen operations in metafiles
;                  |4 - Color adjustment applies to text drawn in metafiles
;                  $fEnable			 - If True, color adjustment for the specified category is turned off; otherwise, color
;                  +adjustment for the specified category is turned on
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: None
; Related .......: None
; Link ..........; @@MsdnLink@@ GdipSetImageAttributesNoOp
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesSetNoOp($hImageAttributes, $iColorAdjustType = 0, $fEnable = True)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipSetImageAttributesNoOp", "hwnd", $hImageAttributes, "int", $iColorAdjustType, "int", $fEnable)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesSetNoOp

; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesSetOutputChannel
; Description ...: Sets or clears the CMYK (Cyan, Magenta, Yellow and Black) output channel for a specified category
; Syntax.........: _GDIPlus_ImageAttributesSetOutputChannel($hImageAttributes[, $iColorAdjustType = 0[, $fEnable = False[, $iColorChannelFlags = 4]]])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $iColorAdjustType - The category for which the output channel is set or cleared:
;                  |0 - Color adjustment applies to all categories that do not have adjustment settings of their own
;                  |1 - Color adjustment applies to bitmapped images
;                  |2 - Color adjustment applies to brush operations in metafiles
;                  |3 - Color adjustment applies to pen operations in metafiles
;                  |4 - Color adjustment applies to text drawn in metafiles
;                  $fEnable			 - If True, output channel for the specified category is set; otherwise, output channel for
;                  +the specified category is cleared
;                  $iColorChannelFlags - The output channel, can be any combination:
;                  |0 - Cyan color channel
;                  |1 - Magenta color channel
;                  |2 - Yellow color channel
;                  |3 - Black color channel
;                  |4 - The previous selected channel
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: None
; Related .......: None
; Link ..........; @@MsdnLink@@ GdipSetImageAttributesOutputChannel
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesSetOutputChannel($hImageAttributes, $iColorAdjustType = 0, $fEnable = False, $iColorChannelFlags = 4)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipSetImageAttributesOutputChannel", "hwnd", $hImageAttributes, "int", $iColorAdjustType, "int", $fEnable, "int", $iColorChannelFlags)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesSetOutputChannel

; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesSetOutputChannelColorProfile
; Description ...: Sets or clears the output channel color-profile file for a specified category
; Syntax.........: _GDIPlus_ImageAttributesSetOutputChannelColorProfile($hImageAttributes[, $iColorAdjustType = 0[, $fEnable = False[, $sFileName = 0]]])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $iColorAdjustType - The category for which the output channel color-profile file is set or cleared:
;                  |0 - Color or grayscale adjustment applies to all categories that do not have adjustment settings of their own
;                  |1 - Color or grayscale adjustment applies to bitmapped images
;                  |2 - Color or grayscale adjustment applies to brush operations in metafiles
;                  |3 - Color or grayscale adjustment applies to pen operations in metafiles
;                  |4 - Color or grayscale adjustment applies to text drawn in metafiles
;                  $fEnable			 - If True, output channel for the specified category is set; otherwise, output channel for
;                  +the specified category is cleared
;                  $sFileName		 - Fully qualified color-profile file name (see remarks)
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: If the color-profile file is in the %SystemRoot%\System32\Spool\Drivers\Color directory, then file name
;                  +parameter can be the file name. Otherwise, it must contain the fully-qualified path name
; Related .......: None
; Link ..........; @@MsdnLink@@ GdipSetImageAttributesOutputChannelColorProfile
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesSetOutputChannelColorProfile($hImageAttributes, $iColorAdjustType = 0, $fEnable = False, $sFileName = 0)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipSetImageAttributesOutputChannelColorProfile", "hwnd", $hImageAttributes, "int", $iColorAdjustType, "int", $fEnable, "wstr", $sFileName)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesSetOutputChannelColorProfile

; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesSetRemapTable
; Description ...: Sets or clears the color-remap table for a specified category
; Syntax.........: _GDIPlus_ImageAttributesSetRemapTable($hImageAttributes[, $iColorAdjustType = 0[, $fEnable = False[, $aColorMap = 0]]])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $iColorAdjustType - The category for which the output channel color-profile file is set or cleared:
;                  |0 - Color adjustment applies to all categories that do not have adjustment settings of their own
;                  |1 - Color adjustment applies to bitmapped images
;                  |2 - Color adjustment applies to brush operations in metafiles
;                  |3 - Color adjustment applies to pen operations in metafiles
;                  |4 - Color adjustment applies to text drawn in metafiles
;                  $fEnable			 - If True, color-remap table for the specified category is set; otherwise, color-remap table
;                  +for the specified category is cleared
;                  $aColorMap		 - Array of old and new colors:
;                  |[0][0] - Number of old and new colors
;                  |[1][0] - Old color 1
;                  |[1][1] - New color 1
;                  |[2][0] - Old color 2
;                  |[2][1] - New color 2
;                  |[n][0] - Old color n
;                  |[n][2] - New color n
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: During rendering, any color that matches ane of the old colors in the remap table is changed to the
;                  +corresponding new color
; Related .......: None
; Link ..........; @@MsdnLink@@ GdipSetImageAttributesRemapTable
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesSetRemapTable($hImageAttributes, $iColorAdjustType = 0, $fEnable = False, $aColorMap = 0)
	Local $iI, $iCount, $tColorMap, $pColorMap, $aResult

	If IsArray($aColorMap) Then
		$iCount = $aColorMap[0][0]
		$tColorMap = DllStructCreate("uint[" & $iCount * 2 & "]")
		$pColorMap = DllStructGetPtr($tColorMap)

		For $iI = 1 To $iCount
			DllStructSetData($tColorMap, 1, $aColorMap[$iI][0], ($iI - 1) * 2 + 1)
			DllStructSetData($tColorMap, 1, $aColorMap[$iI][1], ($iI - 1) * 2 + 2)
		Next

		$aResult = DllCall($__g_hGDIPDll, "uint", "GdipSetImageAttributesRemapTable", "hwnd", $hImageAttributes, "int", $iColorAdjustType, "int", $fEnable, "int", $iCount, "ptr", $pColorMap)
	Else
		$aResult = DllCall($__g_hGDIPDll, "uint", "GdipSetImageAttributesRemapTable", "hwnd", $hImageAttributes, "int", $iColorAdjustType, "int", $fEnable, "int", 0, "ptr", 0)
	EndIf

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesSetRemapTable

; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesSetThreshold
; Description ...: Sets or clears the threshold (transparency range) for a specified category
; Syntax.........: _GDIPlus_ImageAttributesSetThreshold($hImageAttributes[, $iColorAdjustType = 0[, $fEnable = False[, $nThershold = 0]]])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $iColorAdjustType - The category for which the color threshold is set or cleared:
;                  |0 - Color adjustment applies to all categories that do not have adjustment settings of their own
;                  |1 - Color adjustment applies to bitmapped images
;                  |2 - Color adjustment applies to brush operations in metafiles
;                  |3 - Color adjustment applies to pen operations in metafiles
;                  |4 - Color adjustment applies to text drawn in metafiles
;                  $fEnable			 - If True, color threshold for the specified category is set; otherwise, color threshold
;                  +for the specified category is cleared
;                  $nThershold		 - Number that specifies the threshold value (see remarks)
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: The threshold is a value from 0 through 1 that specifies a cutoff point for each color component. For example,
;                  +suppose the threshold is set to 0.7, and suppose you are rendering a color whose red, green, and blue
;                  +components are 230, 50, and 220. The red component, 230, is greater than 0.7255, so the red component will
;                  be changed to 255 (full intensity). The green component, 50, is less than 0.7255, so the green component will
;                  +be changed to 0. The blue component, 220, is greater than 0.7255, so the blue component will be changed to
;                  +255
; Related .......: None
; Link ..........; @@MsdnLink@@ GdipSetImageAttributesThreshold
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesSetThreshold($hImageAttributes, $iColorAdjustType = 0, $fEnable = False, $nThershold = 0)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipSetImageAttributesThreshold", "hwnd", $hImageAttributes, "int", $iColorAdjustType, "int", $fEnable, "float", $nThershold)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesSetThreshold

; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesSetToIdentity
; Description ...: Sets the color-adjustment matrix of a specified category to identity matrix
; Syntax.........: _GDIPlus_ImageAttributesSetToIdentity($hImageAttributes[, $iColorAdjustType = 0])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $iColorAdjustType - The category for which the color-adjustment matrix is set to identity:
;                  |0 - Color or grayscale adjustment applies to all categories that do not have adjustment settings of their own
;                  |1 - Color or grayscale adjustment applies to bitmapped images
;                  |2 - Color or grayscale adjustment applies to brush operations in metafiles
;                  |3 - Color or grayscale adjustment applies to pen operations in metafiles
;                  |4 - Color or grayscale adjustment applies to text drawn in metafiles
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: None
; Related .......: None
; Link ..........; @@MsdnLink@@ GdipSetImageAttributesToIdentity
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesSetToIdentity($hImageAttributes, $iColorAdjustType = 0)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipSetImageAttributesToIdentity", "hwnd", $hImageAttributes, "int", $iColorAdjustType)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesSetToIdentity

; #FUNCTION# ====================================================================================================================
; Name...........: _GDIPlus_ImageAttributesSetWrapMode
; Description ...: Sets the wrap mode of an ImageAttributes object
; Syntax.........: _GDIPlus_ImageAttributesSetWrapMode($hImageAttributes, $iWrapMode[, $iARGB = 0xFF000000])
; Parameters ....: $hImageAttributes - Pointer to an ImageAttribute object
;                  $iWrapMode 		 - Specifies how repeated copies of an image are used to tile an area:
;                  |0 - Tiling without flipping
;                  |1 - Tiles are flipped horizontally as you move from one tile to the next in a row
;                  |2 - Tiles are flipped vertically as you move from one tile to the next in a column
;                  |3 - Tiles are flipped horizontally as you move along a row and flipped vertically as you move along a column
;                  |4 - No tiling takes place
;                  $iARGB			 - Alpha, Red, Green and Blue components of the color of pixels outside of a rendered image.
;                  +This color is visible if the wrap mode is set to 4 and the source rectangle of the image is greater than the
;                  +image itself
; Return values .: Success      - True
;                  Failure      - False and either:
;                  |@error and @extended are set if DllCall failed
;                  |$GDIP_STATUS contains a non zero value specifying the error code
; Remarks .......: None
; Related .......: None
; Link ..........; @@MsdnLink@@ GdipSetImageAttributesWrapMode
; Example .......; No
; ===============================================================================================================================
Func _GDIPlus_ImageAttributesSetWrapMode($hImageAttributes, $iWrapMode, $iARGB = 0xFF000000)
	Local $aResult = DllCall($__g_hGDIPDll, "uint", "GdipSetImageAttributesWrapMode", "hwnd", $hImageAttributes, "int", $iWrapMode, "uint", $iARGB, "int", 0)

	If @error Then Return SetError(@error, @extended, False)
	$GDIP_STATUS = $aResult[0]
	Return $aResult[0] = 0
EndFunc   ;==>_GDIPlus_ImageAttributesSetWrapMode
#endregion
