#CS ===================================================================================================================================================================
	Name:				AXml
	Version:			1.0
	Pugixml:			1.2
	AutoIt version:		3.3.8.1
	License:			Freeware
	Author:				Antonio Do Rosario (Aipion)
	Website:			www.aipion.co.cc
	Descrption: 		XML parser component that is free for both commercial and non-commercial use.
							> Load/save XML files to an in-memory document object model (DOM).
							> Add name/value attribtes to nodes.
							> Numerous methods for iterating over the nodes in a document.
							> Built in XPath.
							> Full Unicode support.
							> MsXml Independent.
	Works on:			Microsoft Windows 2000, XP, Server 2003, Vista, 7, Server 2008. 32/64 bit
	Required: 			Dll file: AXml_x32.dll or AXml_x64.dll
	Script Function:	Are listed below
	Remarks:			Pugixml - The XML parser used in this component is by: Arseny Kapoulkine. The pugixml library is distributed under the MIT license.

	History:
						v1.0
						* First public vesrion.
#CE ===================================================================================================================================================================

#CS ====== Important Information =====================================================================================================
	The following table shows what each type of node contains.
	-----------------------------------------------
	Node Type					Has
	-----------------------------------------------
	Document 					No Name or Value
	Element						Name
	Declaration					Name
	PCDATA						Value
	CDATA						Value
	Comment						Value
	Doctype						Value
	PI							Name and Value

#CE ==================================================================================================================================

; ====== Set up ======================================================================================================================
Global $hDLL_AXml = DllOpen(@ScriptDir & "\AXml_x32.dll")
OnAutoItExitRegister("AXml_OnAutoitExit")
; ====================================================================================================================================



#cs ====== Function ==============================================================
Name...........: _AXml_LoadFile
Description ...: Loads XML document.
Syntax.........: _AXml_LoadFile($FilePath, [$Encoding], [$CondenseWhiteSpace], [$DocID])
Parameters ....: $FilePath        	 - Load this file.
				 $Encoding           - The encoding to use to open the file which are listed below.
										1 - The type of encoding to use is automaticily detcted. [Default]
										2 - UTF-8 encoding as defined in the Unicode standard; UTF-8 sequences with length equal to 5 or 6 are not standard and are rejected.
										3 - little-endian UTF-16 encoding as defined in the Unicode standard; surrogate pairs are supported.
										4 - Big-endian UTF-16 encoding as defined in the Unicode standard; surrogate pairs are supported.
										5 - UTF-16 encoding as defined in the Unicode standard; the endianness is assumed to be that of the target platform.
										6 - Little-endian UTF-32 encoding as defined in the Unicode standard.
										7 - Big-endian UTF-32 encoding as defined in the Unicode standard.
										8 - UTF-32 encoding as defined in the Unicode standard; the endianness is assumed to be that of the target platform.
										9 - The encoding of wchar_t type; it has the same meaning as either encoding_utf16 or encoding_utf32, depending on wchar_t size.
										10 - ISO-8859-1 encoding (also known as Latin-1).
				 $DocID				 - XML Document ID perveusly given by _AXml_LoadFile Function, otherwise -1 will create a new XML Document ID.
Return values .: Success        - Returns the XML Document ID which will be used by other functions.
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No error was encountered during parsing; the source stream represents the valid XML document which was fully parsed and converted to a tree.
						   1 - File could not be opened.
						   2 - It means that some I/O error has occurred during reading the file/stream.
						   3 - There was not enough memory during some allocation; any allocation failure during parsing results in this error.
						   4 - Something went horribly wrong; currently this error does not occur.
						   5 - Parsing stopped due to a tag with either an empty name or a name which starts with incorrect character, such as #.
						   6 - Parsing stopped due to incorrect document Declaration/Processing Instruction.
						   7 - Parsing stopped due to the invalid construction of Comment.
						   8 - Parsing stopped due to the invalid construction of CDATA.
						   9 - Parsing stopped due to the invalid construction of Doctype.
						   10 - Parsing stopped due to the invalid construction of PCDATA.
						   11 - Parsing stopped because starting tag either had no closing > symbol or contained some incorrect symbol
						   12 - Parsing stopped because there was an incorrect attribute, such as an attribute without value or with value that is not quoted (note that <node attr=1> is incorrect in XML)
						   13 - Parsing stopped because ending tag had incorrect syntax (i.e. extra non-whitespace symbols between tag name and >)
						   14 - Parsing stopped because the closing tag did not match the opening one (i.e. <node></nedo>) or because some tag was not closed at all
#ce===============================================================================
Func _AXml_LoadFile($FilePath, $Encoding = 1, $DocID = -1)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlLoadFile", "wstr", $FilePath, "int", $Encoding, "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=-1 Then Return SetError(_AXml_ParseResult_ID($DocID),-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_ParseResult_Description
Description ...: Returns the parsing result of loading the document as string.
Syntax.........: _AXml_XMLParseResult($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function. Use -1 for a last document created.
Return values .: Success        - Returns result as string
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_ParseResult_Description($DocID = -1)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "str:cdecl", "XmlParseResult_Description", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_ParseResult_Offset
Description ...: Returns the offset of last successfully parsed character if parsing failed because of an error in source data; otherwise offset is 0.
Syntax.........: _AXml_XMLParseResult($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function. Use -1 for a last document created.
Return values .: Success        - Returns the offset.
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_ParseResult_Offset($DocID = -1)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlParseResult_Offset", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=-1 Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SaveFile
Description ...: Saves XML document.
Syntax.........: _AXml_SaveFile($DocID, [$FilePath])
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $FilePath      - The filename to save as, if empty save current loaded file
				 $Indention		- The indention to use. [Default: @TAB]
				 $Encoding		- The encoding to use to save the file which are listed below.
									1 - The type of encoding to use is automaticily detcted. [Default]
									2 - UTF-8 encoding as defined in the Unicode standard; UTF-8 sequences with length equal to 5 or 6 are not standard and are rejected.
									3 - little-endian UTF-16 encoding as defined in the Unicode standard; surrogate pairs are supported.
									4 - Big-endian UTF-16 encoding as defined in the Unicode standard; surrogate pairs are supported.
									5 - UTF-16 encoding as defined in the Unicode standard; the endianness is assumed to be that of the target platform.
									6 - Little-endian UTF-32 encoding as defined in the Unicode standard.
									7 - Big-endian UTF-32 encoding as defined in the Unicode standard.
									8 - UTF-32 encoding as defined in the Unicode standard; the endianness is assumed to be that of the target platform.
									9 - The encoding of wchar_t type; it has the same meaning as either encoding_utf16 or encoding_utf32, depending on wchar_t size.
									10 - ISO-8859-1 encoding (also known as Latin-1).
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_SaveFile($DocID, $FilePath = "", $Indention = @TAB, $Encoding = 1)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlSaveFile", "int", $DocID, "wstr", $FilePath, "wstr", $Indention, "int", $Encoding)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_Encoding
Description ...: Returns the type of encoding used by the document.
Syntax.........: _AXml_Encoding($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - The encoding used to open the document which are listed below.
									1 - UTF-8 encoding as defined in the Unicode standard; UTF-8 sequences with length equal to 5 or 6 are not standard and are rejected.
									2 - little-endian UTF-16 encoding as defined in the Unicode standard; surrogate pairs are supported.
									3 - Big-endian UTF-16 encoding as defined in the Unicode standard; surrogate pairs are supported.
									4 - UTF-16 encoding as defined in the Unicode standard; the endianness is assumed to be that of the target platform.
									5 - Little-endian UTF-32 encoding as defined in the Unicode standard.
									6 - Big-endian UTF-32 encoding as defined in the Unicode standard.
									7 - UTF-32 encoding as defined in the Unicode standard; the endianness is assumed to be that of the target platform.
									8 - The encoding of wchar_t type; it has the same meaning as either encoding_utf16 or encoding_utf32, depending on wchar_t size.
									9 - ISO-8859-1 encoding (also known as Latin-1).
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_Encoding($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlEncoding", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_GetDeclaration
Description ...: Returns values of the attributes in the declaration node and goes to the node.
Syntax.........: _AXml_GetDeclaration($DocID, [$Attribute])
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Attribute		- 1 - Version
								  2 - Encoding
								  3 - Standalone
Return values .: Success        - Returns the string with results.	E.G. "1.0" for Version
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_GetDeclaration($DocID, $Attribute = 1)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlGetDeclaration", "int", $DocID, "int", $Attribute)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SetDeclaration
Description ...: Sets the values of the attributes in the declaration node.
Syntax.........: _AXml_SetDeclaration($DocID, [$Version], [$Encoding], [$Standalone])
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Version		- Set the Version attribute, if empty will not set it.
				 $Encoding		- Set the Encoding attribute, if empty will not set it.
				 $Standalone	- Set the Standalone attribute, if empty will not set it.
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
 Remarks ......: If all the Parameters of this function except $DocID are empty then this function will fail.
#ce===============================================================================
Func _AXml_SetDeclaration($DocID, $Version = "", $Encoding = "", $Standalone = "")
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlSetDeclaration", "int", $DocID, "wstr", $Version, "wstr", $Encoding, "wstr", $Standalone)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_GetDoctype
Description ...: Returns value of the Document type declaration (Doctype) node and goes to it.
Syntax.........: _AXml_GetDoctype($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the string with results.	E.G. "greeting [ <!ELEMENT greeting (#PCDATA)> ]"
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_GetDoctype($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlGetDoctype", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SetDoctype
Description ...: Sets value of the Document type declaration (Doctype) node.
Syntax.........: _AXml_SetDoctype($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Value			- Text to be set
Return values .: Success        - Returns the string with results.	E.G. "greeting [ <!ELEMENT greeting (#PCDATA)> ]"
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_SetDoctype($DocID, $Value)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlSetDoctype", "int", $DocID, "wstr", $Value)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_GetText
Description ...: Convenience function for easy access to the text inside current element.
Syntax.........: _AXml_GetText($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the string with results
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_GetText($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlGetText", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SetText
Description ...: Convenience function for easy set the text of the current element.
Syntax.........: _AXml_SetText($DocID, $Value)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Value         - Text to be set
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_SetText($DocID, $Value)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlSetText", "int", $DocID, "wstr", $Value)
	If @ERROR Or $aRetDCall_Main[0]=False Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SetCDATA
Description ...: Turns on or off a CDATA representation of the current element text.
Syntax.........: _AXml_SetCDATA($DocID, [$SetCDATA])
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $SetCDATA      - True (default) or False
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_SetCDATA($DocID, $SetCDATA = True)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlSetCDATA", "int", $DocID, "bool", $SetCDATA)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_IsCDATA
Description ...: Queries whether this represents text using a CDATA section.
Syntax.........: _AXml_IsCDATA($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns 1 or 0
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_IsCDATA($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlIsCDATA", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=-1 Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_GetNodeName
Description ...: Gets the name of the current node.
Syntax.........: _AXml_GetNodeName($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the string with results
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_GetNodeName($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlGetNodeName", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SetNodeName
Description ...: Sets the name of the current node.
Syntax.........: _AXml_SetNodeName($DocID, $Value)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Value         - Value to be set of the node
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_SetNodeName($DocID, $Name)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlSetNodeName", "int", $DocID, "wstr", $Name)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_GetNodeValue
Description ...: Gets the value of the current node.
Syntax.........: _AXml_GetNodeValue($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the string with results
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_GetNodeValue($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlGetNodeValue", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SetNodeValue
Description ...: Sets the value of the current node.
Syntax.........: _AXml_SetNodeValue($DocID, $Value)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Value         - Value to be set of the node
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_SetNodeValue($DocID, $Value)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlSetNodeValue", "int", $DocID, "wstr", $Value)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_RootElement
Description ...: Returns the root element name and goes to it. (The only top level element)
Syntax.........: _AXml_RootElement($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the Root element name (the only top level element)
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_RootElement($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlRootElement", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_FirstChild
Description ...: Returns the first child node name and goes to it.
Syntax.........: _AXml_FirstChild($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the string with results
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_FirstChild($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlFirstChild", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_LastChild
Description ...: Returns the last child node name and goes to it.
Syntax.........: _AXml_LastChild($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the string with results
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_LastChild($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlLastChild", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_Parent
Description ...: Returns the parent element name and goes to it.
Syntax.........: _AXml_Parent($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the string with results
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_Parent($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlParent", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_NextSibling
Description ...: Returns the next sibling node name and goes to it.
Syntax.........: _AXml_NextSibling($DocID, [$Name])
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Name          - Find next sibling element specified by name, if empty returns first founded
Return values .: Success        - Returns the string with results
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_NextSibling($DocID, $Name = "")
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlNextSibling", "int", $DocID, "wstr", $Name)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_PreviousSibling
Description ...: Returns the previous sibling node name and goes to it.
Syntax.........: _AXml_PreviousSibling($DocID, [$Name])
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Name          - Find next sibling element specified by name, if empty returns first founded
Return values .: Success        - Returns the string with results
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_PreviousSibling($DocID, $Name = "")
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlPreviousSibling", "int", $DocID, "wstr", $Name)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_RemoveNode
Description ...: Deletes the current node and goes to the parent element.
Syntax.........: _AXml_RemoveNode($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_RemoveNode($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlRemoveNode", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_RemoveAllChildren
Description ...: Deletes all children of the current node (it will be empty).
Syntax.........: _AXml_RemoveAllChildren($DocIDX)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_RemoveAllChildren($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlRemoveAllChildren", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_CreateNode
Description ...: Creates the node and returns its handle.
Syntax.........: _AXml_CreateNode($DocID, $NodeType, $Position, [$Name], [$Value], [$Handle])
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $NodeType      - Create this node. (Element, Comment, PI)
				 $Position		- 1 - Append
								  2 - Prepend
								  3 - Insert after
								  4 - Insert before
				 $Name			- Set the Name of the Node depanding on the type. if it is not requried by the type of Node, leave it blank.
				 $Value			- Set the Value of the Node depanding on the type. if it is not requried by the type of Node, leave it blank.
				 $Handle		- The Node handle of the destnation. Use 0 to use the last node.
Return values .: Success        - Returns the Handle
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_CreateNode($DocID, $NodeType, $Position, $Name = "", $Value = "", $Handle = 0)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlCreateNode", "int", $DocID, "int", $NodeType, "int", $Position, "wstr", $Name, "wstr", $Value, "int", $Handle)
	If @ERROR Or $aRetDCall_Main[0]=-1 Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_CreateNodeCopy
Description ...: Creates the node and returns its handle.
Syntax.........: _AXml_CreateNodeCopy($DocID, $NodeType, $Position, [$HandleCopy], [$HandlePaste])
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Position		- 1 - Append
								  2 - Prepend
								  3 - Insert after
								  4 - Insert before
				 $HandleCopy	- The Node to copy via handle. Use 0 to use the last node.
				 $HandlePaste	- The Node handle of the destnation. Use 0 to use the last node.
Return values .: Success        - Returns the Handle
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_CreateNodeCopy($DocID, $Position, $HandleCopy = 0, $HandlePaste = 0)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlCreateNodeCopy", "int", $DocID, "int", $Position, "int", $HandleCopy, "int", $HandlePaste)
	If @ERROR Or $aRetDCall_Main[0]=-1 Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_NodeHandle
Description ...: Returns the current node handle.
Syntax.........: _AXml_NodeHandle($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the Handle
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_NodeHandle($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlNodeHandle", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=-1 Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_GotoHandle
Description ...: Goes to the specified node or attribute and tells the type of handle.
Syntax.........: _AXml_GotoHandle($DocID, $Handle)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $Handle        - Handle of the node to go to
Return values .: Success        - Returns type of the handle and thay listed below.
									1 - Node.
									2 - Attribute.
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_GotoHandle($DocID, $Handle)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlGotoHandle", "int", $DocID, "int", $Handle)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_XPathNode
Description ...: Compute a node XPath expression and goes to it.
Syntax.........: _AXml_XPathNode($DocID, $XPath)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $XPath			- XPath expression.	E.G. "/a/b"
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_XPathNode($DocID, $XPath)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlXPathNode", "int", $DocID, "wstr", $XPath)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_XPathAttribute
Description ...: Compute an attribute XPath expression and goes to it.
Syntax.........: _AXml_XPathAttribute($DocID, $XPath)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $XPath			- XPath expression.	E.G. "/a/b/@attr"
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_XPathAttribute($DocID, $XPath)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlXPathAttribute", "int", $DocID, "wstr", $XPath)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_NodeType
Description ...: Returns the type of the current node.
Syntax.........: _AXml_NodeType($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns
									1 - Null			- Empty (null) node handle
									2 - Document		- A document tree's absolute root
									3 - Element			- Element tag, i.e. '<node/>'
									4 - PCDATA			- Parsed character data, i.e. 'text'
									5 - CDATA			- Character data, i.e. '<![CDATA[text]]>'
									6 - Comment			- Comment tag, i.e. '<!-- text -->'
									7 - PI				- Processing instruction, i.e. '<?name?>'
									8 - Declaration		- Document declaration, i.e. '<?xml version="1.0"?>'
									9 - Doctype			- Document type declaration, i.e. '<!DOCTYPE doc>'
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_NodeType($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlNodeType", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_CreateAttribute
Description ...: Creates an attribute on the current node and goes to it.
Syntax.........: _AXml_CreateAttribute($DocID, $AttributeName, $AttributeValue)
Parameters ....: $DocID          - XML Document ID given by _AXml_LoadFile Function.
				 $AttributeName  - Attribute name
				 $AttributeValue - Attribute value
Return values .: Success         - Returns 1
				 Failure         - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_CreateAttribute($DocID, $AttributeName, $AttributeValue)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlCreateAttribute", "int", $DocID, "wstr", $AttributeName, "wstr", $AttributeValue)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_GetAttribute
Description ...: Returns value of the attribute of the current element and goes to it.
Syntax.........: _AXml_GetAttribute($DocID, $AttributeName)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $AttributeName - Find attribute specified by name.
Return values .: Success        - Returns the string with results
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_GetAttribute($DocID, $AttributeName)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "wstr:cdecl", "XmlGetAttribute", "int", $DocID, "wstr", $AttributeName)
	If @ERROR Or $aRetDCall_Main[0]="-1" Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SetAttribute
Description ...: Sets attribute of the current element.
Syntax.........: _AXml_SetAttribute($DocID, $AttributeName, $AttributeValue)
Parameters ....: $DocID          - XML Document ID given by _AXml_LoadFile Function.
				 $AttributeName  - Attribute name
				 $AttributeValue - Attribute value
Return values .: Success         - Returns 1
				 Failure         - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_SetAttribute($DocID, $AttributeName, $AttributeValue)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlSetAttribute", "int", $DocID, "wstr", $AttributeName, "wstr", $AttributeValue)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SetAttributeName
Description ...: Sets name of the current attribute.
Syntax.........: _AXml_SetAttributeName($DocID, $AttributeName)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $AttributeName - Attribute name
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_SetAttributeName($DocID, $AttributeName)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlSetAttribute", "int", $DocID, "wstr", $AttributeName)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SetAttributeValue
Description ...: Sets value of the current attribute.
Syntax.........: _AXml_SetAttributeValue($DocID, $AttributeValue)
Parameters ....: $DocID          - XML Document ID given by _AXml_LoadFile Function.
				 $AttributeValue - Attribute name
Return values .: Success         - Returns 1
				 Failure         - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_SetAttributeValue($DocID, $AttributeValue)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlSetAttribute", "int", $DocID, "wstr", $AttributeValue)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_RemoveAttribute
Description ...: Deletes attribute with the specified name of the current element.
Syntax.........: _AXml_RemoveAttribute($DocID, $AttributeName)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $AttributeName - Attribute name
Return values .: Success        - Returns 1
				 Failure        - Returns 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_RemoveAttribute($DocID, $AttributeName)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlRemoveAttribute", "int", $DocID, "wstr", $AttributeName)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,0)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_FirstAttribute
Description ...: Returns name and value of the first attribute of the current element and goes to it.
Syntax.........: _AXml_FirstAttribute($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns an Array with the following results.
								  $hArray[0] - Returns 1
								  $hArray[1] - Attribute Name
								  $hArray[2] - Attribute Value
				 Failure        - Returns an Array with $hArray[0] - 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error calling the main function
						   2 - Error calling the GetDataSizes function
						   3 - Error creating the Buffer #1
						   4 - Error creating the Buffer #2
						   5 - Error calling the GetData function
#ce===============================================================================
Func _AXml_FirstAttribute($DocID)
	Local $aRetDCall_Main,$aRetDCall_GetDataSizes,$aRetStruct_GetDataSizes,$aRetStruct_CharPtr,$aRetStruct_Buffer1,$aRetStruct_Buffer2,$aRetDCall_GetData
	Local $hArray[1] = [0]

	$aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlFirstAttribute", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,$hArray)														;< Error calling the main function

	$aRetDCall_GetDataSizes = DllCall($hDLL_AXml, "int:cdecl", "GetDataSizes")
	If @ERROR Or Not $aRetDCall_GetDataSizes[0] Then Return SetError(2,-1,$hArray)												;< Error calling the GetDataSizes function
	$aRetStruct_GetDataSizes = DllStructCreate("int Data[2];",$aRetDCall_GetDataSizes[0])

	$aRetStruct_CharPtr = DllStructCreate("ptr x[2];")
	$aRetStruct_Buffer1 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,1)& "];")
	If @ERROR Then Return SetError(3,-1,$hArray)																				;< Error creating the Buffer #1
	$aRetStruct_Buffer2 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,2)& "];")
	If @ERROR Then Return SetError(4,-1,$hArray)																				;< Error creating the Buffer #2
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer1),1)
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer2),2)

	$aRetDCall_GetData = DllCall($hDLL_AXml, "int:cdecl", "GetData", "ptr" ,DllStructGetPtr($aRetStruct_CharPtr))
	If @ERROR Or Not $aRetDCall_GetData[0] Then Return SetError(5,-1,$hArray)													;< Error calling the GetData function

	Local $hArray[3] = [1,DllStructGetData($aRetStruct_Buffer1,1),DllStructGetData($aRetStruct_Buffer2,1)]

	Return $hArray
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_LastAttribute
Description ...: Returns name and value of the last attribute of the current element and goes to it.
Syntax.........: _AXml_LastAttribute($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns an Array with the following results.
								  $hArray[0] - Returns 1
								  $hArray[1] - Attribute Name
								  $hArray[2] - Attribute Value
				 Failure        - Returns an Array with $hArray[0] - 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error calling the main function
						   2 - Error calling the GetDataSizes function
						   3 - Error creating the Buffer #1
						   4 - Error creating the Buffer #2
						   5 - Error calling the GetData function
#ce===============================================================================
Func _AXml_LastAttribute($DocID)
	Local $aRetDCall_Main,$aRetDCall_GetDataSizes,$aRetStruct_GetDataSizes,$aRetStruct_CharPtr,$aRetStruct_Buffer1,$aRetStruct_Buffer2,$aRetDCall_GetData
	Local $hArray[1] = [0]

	$aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlLastAttribute", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,$hArray)														;< Error calling the main function

	$aRetDCall_GetDataSizes = DllCall($hDLL_AXml, "int:cdecl", "GetDataSizes")
	If @ERROR Or Not $aRetDCall_GetDataSizes[0] Then Return SetError(2,-1,$hArray)												;< Error calling the GetDataSizes function
	$aRetStruct_GetDataSizes = DllStructCreate("int Data[2];",$aRetDCall_GetDataSizes[0])

	$aRetStruct_CharPtr = DllStructCreate("ptr x[2];")
	$aRetStruct_Buffer1 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,1)& "];")
	If @ERROR Then Return SetError(3,-1,$hArray)																				;< Error creating the Buffer #1
	$aRetStruct_Buffer2 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,2)& "];")
	If @ERROR Then Return SetError(4,-1,$hArray)																				;< Error creating the Buffer #2
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer1),1)
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer2),2)

	$aRetDCall_GetData = DllCall($hDLL_AXml, "int:cdecl", "GetData", "ptr" ,DllStructGetPtr($aRetStruct_CharPtr))
	If @ERROR Or Not $aRetDCall_GetData[0] Then Return SetError(5,-1,$hArray)													;< Error calling the GetData function

	Local $hArray[3] = [1,DllStructGetData($aRetStruct_Buffer1,1),DllStructGetData($aRetStruct_Buffer2,1)]

	Return $hArray
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_NextAttribute
Description ...: Returns name and value of the next attribute and goes to it.
Syntax.........: _AXml_NextAttribute($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns an Array with the following results.
								  $hArray[0] - Returns 1
								  $hArray[1] - Attribute Name
								  $hArray[2] - Attribute Value
				 Failure        - Returns an Array with $hArray[0] - 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error calling the main function
						   2 - Error calling the GetDataSizes function
						   3 - Error creating the Buffer #1
						   4 - Error creating the Buffer #2
						   5 - Error calling the GetData function
#ce===============================================================================
Func _AXml_NextAttribute($DocID)
	Local $aRetDCall_Main,$aRetDCall_GetDataSizes,$aRetStruct_GetDataSizes,$aRetStruct_CharPtr,$aRetStruct_Buffer1,$aRetStruct_Buffer2,$aRetDCall_GetData
	Local $hArray[1] = [0]

	$aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlNextAttribute", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,$hArray)														;< Error calling the main function

	$aRetDCall_GetDataSizes = DllCall($hDLL_AXml, "int:cdecl", "GetDataSizes")
	If @ERROR Or Not $aRetDCall_GetDataSizes[0] Then Return SetError(2,-1,$hArray)												;< Error calling the GetDataSizes function
	$aRetStruct_GetDataSizes = DllStructCreate("int Data[2];",$aRetDCall_GetDataSizes[0])

	$aRetStruct_CharPtr = DllStructCreate("ptr x[2];")
	$aRetStruct_Buffer1 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,1)& "];")
	If @ERROR Then Return SetError(3,-1,$hArray)																				;< Error creating the Buffer #1
	$aRetStruct_Buffer2 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,2)& "];")
	If @ERROR Then Return SetError(4,-1,$hArray)																				;< Error creating the Buffer #2
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer1),1)
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer2),2)

	$aRetDCall_GetData = DllCall($hDLL_AXml, "int:cdecl", "GetData", "ptr" ,DllStructGetPtr($aRetStruct_CharPtr))
	If @ERROR Or Not $aRetDCall_GetData[0] Then Return SetError(5,-1,$hArray)													;< Error calling the GetData function

	Local $hArray[3] = [1,DllStructGetData($aRetStruct_Buffer1,1),DllStructGetData($aRetStruct_Buffer2,1)]

	Return $hArray
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_PreviousAttribute
Description ...: Returns name and value of the previous attribute and goes to it.
Syntax.........: _AXml_PreviousAttribute($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns an Array with the following results.
								  $hArray[0] - Returns 1
								  $hArray[1] - Attribute Name
								  $hArray[2] - Attribute Value
				 Failure        - Returns an Array with $hArray[0] - 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error calling the main function
						   2 - Error calling the GetDataSizes function
						   3 - Error creating the Buffer #1
						   4 - Error creating the Buffer #2
						   5 - Error calling the GetData function
#ce===============================================================================
Func _AXml_PreviousAttribute($DocID)
	Local $aRetDCall_Main,$aRetDCall_GetDataSizes,$aRetStruct_GetDataSizes,$aRetStruct_CharPtr,$aRetStruct_Buffer1,$aRetStruct_Buffer2,$aRetDCall_GetData
	Local $hArray[1] = [0]

	$aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlPreviousAttribute", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,$hArray)														;< Error calling the main function

	$aRetDCall_GetDataSizes = DllCall($hDLL_AXml, "int:cdecl", "GetDataSizes")
	If @ERROR Or Not $aRetDCall_GetDataSizes[0] Then Return SetError(2,-1,$hArray)												;< Error calling the GetDataSizes function
	$aRetStruct_GetDataSizes = DllStructCreate("int Data[2];",$aRetDCall_GetDataSizes[0])

	$aRetStruct_CharPtr = DllStructCreate("ptr x[2];")
	$aRetStruct_Buffer1 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,1)& "];")
	If @ERROR Then Return SetError(3,-1,$hArray)																				;< Error creating the Buffer #1
	$aRetStruct_Buffer2 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,2)& "];")
	If @ERROR Then Return SetError(4,-1,$hArray)																				;< Error creating the Buffer #2
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer1),1)
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer2),2)

	$aRetDCall_GetData = DllCall($hDLL_AXml, "int:cdecl", "GetData", "ptr" ,DllStructGetPtr($aRetStruct_CharPtr))
	If @ERROR Or Not $aRetDCall_GetData[0] Then Return SetError(5,-1,$hArray)													;< Error calling the GetData function

	Local $hArray[3] = [1,DllStructGetData($aRetStruct_Buffer1,1),DllStructGetData($aRetStruct_Buffer2,1)]

	Return $hArray
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_CurrentAttribute
Description ...: Returns name and value of the current attribute.
Syntax.........: _AXml_CurrentAttribute($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns an Array with the following results.
								  $hArray[0] - Returns 1
								  $hArray[1] - Attribute Name
								  $hArray[2] - Attribute Value
				 Failure        - Returns an Array with $hArray[0] - 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error calling the main function
						   2 - Error calling the GetDataSizes function
						   3 - Error creating the Buffer #1
						   4 - Error creating the Buffer #2
						   5 - Error calling the GetData function
#ce===============================================================================
Func _AXml_CurrentAttribute($DocID)
	Local $aRetDCall_Main,$aRetDCall_GetDataSizes,$aRetStruct_GetDataSizes,$aRetStruct_CharPtr,$aRetStruct_Buffer1,$aRetStruct_Buffer2,$aRetDCall_GetData
	Local $hArray[1] = [0]

	$aRetDCall_Main = DllCall($hDLL_AXml, "bool:cdecl", "XmlCurrentAttribute", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=0 Then Return SetError(1,-1,$hArray)														;< Error calling the main function

	$aRetDCall_GetDataSizes = DllCall($hDLL_AXml, "int:cdecl", "GetDataSizes")
	If @ERROR Or Not $aRetDCall_GetDataSizes[0] Then Return SetError(2,-1,$hArray)												;< Error calling the GetDataSizes function
	$aRetStruct_GetDataSizes = DllStructCreate("int Data[2];",$aRetDCall_GetDataSizes[0])

	$aRetStruct_CharPtr = DllStructCreate("ptr x[2];")
	$aRetStruct_Buffer1 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,1)& "];")
	If @ERROR Then Return SetError(3,-1,$hArray)																				;< Error creating the Buffer #1
	$aRetStruct_Buffer2 = DllStructCreate("WCHAR x[" &DllStructGetData($aRetStruct_GetDataSizes, 1,2)& "];")
	If @ERROR Then Return SetError(4,-1,$hArray)																				;< Error creating the Buffer #2
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer1),1)
	DllStructSetData($aRetStruct_CharPtr,"x",DllStructGetPtr($aRetStruct_Buffer2),2)

	$aRetDCall_GetData = DllCall($hDLL_AXml, "int:cdecl", "GetData", "ptr" ,DllStructGetPtr($aRetStruct_CharPtr))
	If @ERROR Or Not $aRetDCall_GetData[0] Then Return SetError(5,-1,$hArray)													;< Error calling the GetData function

	Local $hArray[3] = [1,DllStructGetData($aRetStruct_Buffer1,1),DllStructGetData($aRetStruct_Buffer2,1)]

	Return $hArray
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_NumChildren
Description ...: Returns a count of all the children in the current node.
Syntax.........: _AXml_NumChildren($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the count
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_NumChildren($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlNumChildren", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=-1 Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_NumAttributes
Description ...: Returns a count of all the attributes in the current node.
Syntax.........: _AXml_NumAttributes($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
Return values .: Success        - Returns the count
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_NumAttributes($DocID)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlNumAttributes", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=-1 Then Return SetError(1,-1,-1)
	Return $aRetDCall_Main[0]
EndFunc

#cs ====== Function ==============================================================
Name...........: _AXml_SelectNA
Description ...: Selects XML Node(s) and/or attribute(s) based on XPath expression.
Syntax.........: _AXml_SelectNodes($DocID, $XPathExp)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.
				 $XPathExp      - XPath expression.
Return values .: Success        - Returns an Array with the following results.
								  $hArray[0] - Returns 1
								  $hArray[1] - Contains the count of items selected.
								  $hArray[n] - Handle of the item selected.
				 Failure        - Returns an Array with $hArray[0] - 0 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error calling the main function
						   2 - Error getting the count of selected nodes
						   3 - Error getting the handles of the nodes
#ce===============================================================================
Func _AXml_SelectNA($DocID, $XPathExp)
	Local $hArray[1] = [0]
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlSelectNodes", "int", $DocID, "wstr", $XPathExp)
	If @ERROR Or $aRetDCall_Main[0] = -1 Then Return SetError(1,-1,$hArray)

	$aRetArray_XmlSelectNodes1 = DllStructCreate("int Data[1];",$aRetDCall_Main[0])
	If @ERROR Then Return SetError(2,-1,$hArray)

	$aRetArray_XmlSelectNodes2 = DllStructCreate("int Data[" &DllStructGetData($aRetArray_XmlSelectNodes1, 1,1)+1& "];",$aRetDCall_Main[0])
	If @ERROR Then Return SetError(3,-1,$hArray)

	$aRetArray_XmlSelectNodes1 = 0
	ReDim $hArray[DllStructGetData($aRetArray_XmlSelectNodes2, 1,1)+2]
	For $I = 1 To DllStructGetData($aRetArray_XmlSelectNodes2, 1,1)+1
		$hArray[$I] = DllStructGetData($aRetArray_XmlSelectNodes2, 1,$I)
	Next
	$aRetArray_XmlSelectNodes2 = 0
	$hArray[0] = 1
	Return $hArray
EndFunc


; ====== Functions For Internal Use ===============================================

#cs ====== Function ==============================================================
Name...........: _AXml_ParseResult_ID
Description ...: Returns the parsing result of loading the document. Works for last loaded docuent.
Syntax.........: _AXml_ParseResult_ID($DocID)
Parameters ....: $DocID         - XML Document ID given by _AXml_LoadFile Function.  Use -1 for a last document created.
Return values .: Success        - Returns the ID of error which are listed below.
									0 - No error was encountered during parsing; the source stream represents the valid XML document which was fully parsed and converted to a tree.
									1 - File could not be opened.
									2 - It means that some I/O error has occurred during reading the file/stream.
									3 - There was not enough memory during some allocation; any allocation failure during parsing results in this error.
									4 - Something went horribly wrong; currently this error does not occur.
									5 - Parsing stopped due to a tag with either an empty name or a name which starts with incorrect character, such as #.
									6 - Parsing stopped due to incorrect document Declaration/Processing Instruction.
									7 - Parsing stopped due to the invalid construction of Comment.
									8 - Parsing stopped due to the invalid construction of CDATA.
									9 - Parsing stopped due to the invalid construction of Doctype.
									10 - Parsing stopped due to the invalid construction of PCDATA.
									11 - Parsing stopped because starting tag either had no closing > symbol or contained some incorrect symbol.
									12 - Parsing stopped because there was an incorrect attribute, such as an attribute without value or with value that is not quoted (note that <node attr=1> is incorrect in XML).
									13 - Parsing stopped because ending tag had incorrect syntax (i.e. extra non-whitespace symbols between tag name and >).
									14 - Parsing stopped because the closing tag did not match the opening one (i.e. <node></nedo>) or because some tag was not closed at all.
				 Failure        - Returns -1 and sets the @error flag
				 @Error to:
						   0 - No Error
						   1 - Error
#ce===============================================================================
Func _AXml_ParseResult_ID($DocID = -1)
	Local $aRetDCall_Main = DllCall($hDLL_AXml, "int:cdecl", "XmlParseResult_ID", "int", $DocID)
	If @ERROR Or $aRetDCall_Main[0]=-1 Then Return -1
	Return $aRetDCall_Main[0]
EndFunc

Func AXml_OnAutoitExit()
	If IsDeclared("hDLL_AXml") Then
		DllClose($hDLL_AXml)
	EndIf
EndFunc