#include-once

#include <Color.au3>
#include <Math.au3>

; #INDEX# =======================================================================================================================
; Title .........: ColorEx
; AutoIt Version : 3.3.14.2
; Language ..... : English
; Description ...: Functions for working with colors.
; Author(s) .....: scintilla4evr
; ===============================================================================================================================

; #CONSTANTS# ===================================================================================================================
Global Const $__COLORCONSTANTS_CMYKMAX = 100
Global Const $__COLORCONSTANTS_XYZMAX = 100

Global Enum $COLOREX_DELTAC = 0, _
			$COLOREX_DELTAH, _
			$COLOREX_DELTAE
; ===============================================================================================================================

; #CURRENT# =====================================================================================================================
; _ColorConvertRGBtoCMYK
; _ColorConvertCMYKToRGB
; _ColorConvertRGBToXYZ
; _ColorConvertXYZToRGB
; _ColorConvertXYZToYxy
; _ColorConvertYxyToXYZ
; _ColorConvertXYZToCIELab
; _ColorConvertCIELabToXYZ
; _ColorConvertCIELabToCIELCH
; _ColorConvertCIELCHToCIELab
; _ColorConvertRGBToCIELCH
; _ColorConvertCIELCHToRGB
; _ColorCalculateDelta
; _ColorHarmonyTriad
; _ColorHarmonySplitComplements
; _ColorHarmonyAnalogous
; _ColorHarmonyComplement
; ===============================================================================================================================

#Region Color Conversions

Func _ColorConvertRGBtoCMYK($aArray)
	Local $aRet[4]
	Local $fRed, $fGreen, $fBlue
	Local $fC, $fM, $fY, $fK
	Local $minCMY

	$fRed = $aArray[0]/$__COLORCONSTANTS_RGBMAX
	$fGreen = $aArray[1]/$__COLORCONSTANTS_RGBMAX
	$fBlue = $aArray[2]/$__COLORCONSTANTS_RGBMAX

	If $fRed = 0 And $fGreen = 0 And $fBlue = 0 Then Return $aRet
	$fC = 1-$fRed
	$fM = 1-$fGreen
	$fY = 1-$fBlue
	$minCMY = _Min(_Min($fC, $fM), $fY)
	$fC = ($fC - $minCMY) / (1 - $minCMY)
	$fM = ($fM - $minCMY) / (1 - $minCMY)
	$fY = ($fY - $minCMY) / (1 - $minCMY)
	$fK = $minCMY

	$aRet[0] = $fC*$__COLORCONSTANTS_CMYKMAX
	$aRet[1] = $fM*$__COLORCONSTANTS_CMYKMAX
	$aRet[2] = $fY*$__COLORCONSTANTS_CMYKMAX
	$aRet[3] = $fK*$__COLORCONSTANTS_CMYKMAX

	Return $aRet
EndFunc

Func _ColorConvertCMYKToRGB($aCMYK)
	Local $aRet[3]
	Local $fC, $fM, $fY, $fK
	Local $iR, $iG, $iB

	$fC = $aCMYK[0]/$__COLORCONSTANTS_CMYKMAX
	$fM = $aCMYK[1]/$__COLORCONSTANTS_CMYKMAX
	$fY = $aCMYK[2]/$__COLORCONSTANTS_CMYKMAX
	$fK = $aCMYK[3]/$__COLORCONSTANTS_CMYKMAX
	$iR = Round($__COLORCONSTANTS_RGBMAX * (1 - $fC) * (1 - $fK))
	$iG = Round($__COLORCONSTANTS_RGBMAX * (1 - $fM) * (1 - $fK))
	$iB = Round($__COLORCONSTANTS_RGBMAX * (1 - $fY) * (1 - $fK))

	$aRet[0] = $iR
	$aRet[1] = $iG
	$aRet[2] = $iB

	Return $aRet
EndFunc

Func _ColorConvertRGBToXYZ($aRGB)
	Local $aRet[3]
	Local $fR, $fG, $fB
	Local $fX, $fY, $fZ

	$fR = $aRGB[0]/$__COLORCONSTANTS_RGBMAX
	$fG = $aRGB[1]/$__COLORCONSTANTS_RGBMAX
	$fB = $aRGB[2]/$__COLORCONSTANTS_RGBMAX

	If $fR > 0.04045 Then
		$fR = (($fR + 0.055)/1.055)^2.4
	Else
		$fR = $fR / 12.92
	EndIf

	If $fG > 0.04045 Then
		$fG = (($fG + 0.055)/1.055)^2.4
	Else
		$fG = $fG / 12.92
	EndIf

	If $fB > 0.04045 Then
		$fB = (($fB + 0.055)/1.055)^2.4
	Else
		$fB = $fB / 12.92
	EndIf

	$fR *= $__COLORCONSTANTS_XYZMAX
	$fG *= $__COLORCONSTANTS_XYZMAX
	$fB *= $__COLORCONSTANTS_XYZMAX

	$fX = $fR * 0.4124 + $fG * 0.3576 + $fB * 0.1805
	$fY = $fR * 0.2126 + $fG * 0.7512 + $fB * 0.0722
	$fZ = $fR * 0.0193 + $fG * 0.1192 + $fB * 0.9505

	$aRet[0] = $fX
	$aRet[1] = $fY
	$aRet[2] = $fZ

	Return $aRet
EndFunc

Func _ColorConvertXYZToRGB($aXYZ)
	Local $aRet[3]
	Local $fX, $fY, $fZ
	Local $fR, $fG, $fB

	$fX = $aXYZ[0]/$__COLORCONSTANTS_XYZMAX
	$fY = $aXYZ[1]/$__COLORCONSTANTS_XYZMAX
	$fZ = $aXYZ[2]/$__COLORCONSTANTS_XYZMAX

	$fR = $fX * 3.2406 + $fY * -1.5372 + $fZ * -0.4986
	$fG = $fX * -0.9689 + $fY * 1.8758 + $fZ * 0.0415
	$fB = $fX * 0.0557 + $fY * -0.2040 + $fZ * 1.0570

	If ($fR > 0.0031308) Then
		$fR = 1.055 * ($fR ^ (1/2.4)) - 0.055
	Else
		$fR = 12.92 * $fR
	EndIf

	If ($fG > 0.0031308) Then
		$fG = 1.055 * ($fG ^ (1/2.4)) - 0.055
	Else
		$fG = 12.92 * $fG
	EndIf

	If ($fB > 0.0031308) Then
		$fB = 1.055 * ($fB ^ (1/2.4)) - 0.055
	Else
		$fB = 12.92 * $fB
	EndIf

	$fR *= $__COLORCONSTANTS_RGBMAX
	$aRet[0] = ($fR > 255) ? 255 : (($fR < 0) ? 0 : Int($fR))

	$fG *= $__COLORCONSTANTS_RGBMAX
	$aRet[1] = ($fG > 255) ? 255 : (($fG < 0) ? 0 : Int($fG))

	$fB *= $__COLORCONSTANTS_RGBMAX
	$aRet[2] = ($fB > 255) ? 255 : (($fB < 0) ? 0 : Int($fB))

	Return $aRet
EndFunc

Func _ColorConvertXYZToYxy($aXYZ)
	Local $aRet[3]

	$aRet[0] = $aXYZ[1]
	$aRet[1] = $aXYZ[0] / ($aXYZ[0]+$aXYZ[1]+$aXYZ[2])
	$aRet[2] = $aXYZ[1] / ($aXYZ[0]+$aXYZ[1]+$aXYZ[2])

	Return $aRet
EndFunc

Func _ColorConvertYxyToXYZ($aYxy)
	Local $aRet[3]

	$aRet[0] = $aYxy[1] * ($aYxy[0]/$aYxy[2])
	$aRet[1] = $aYxy[0]
	$aRet[2] = (1-$aYxy[1]-$aYxy[2]) * ($aYxy[0]/$aYxy[2])

	Return $aRet
EndFunc

Func _ColorConvertXYZToCIELab($aXYZ)
	Local $aRet[3]
	Local $fX, $fY, $fZ

	$fX = $aXYZ[0] / 95.047
	$fY = $aXYZ[1] / 100
	$fZ = $aXYZ[2] / 108.883

	If $fX > 0.008856 Then
		$fX = $fX ^ (1/3)
	Else
		$fX = (7.787 * $fX) + (16/116)
	EndIf

	If $fY > 0.008856 Then
		$fY = $fY ^ (1/3)
	Else
		$fY = (7.787 * $fY) + (16/116)
	EndIf

	If $fZ > 0.008856 Then
		$fZ = $fZ ^ (1/3)
	Else
		$fZ = (7.787 * $fZ) + (16/116)
	EndIf

	$aRet[0] = (116*$fY) - 16 ; L
	$aRet[1] = 500 * ($fX - $fY) ; a
	$aRet[2] = 200 * ($fY - $fZ) ; b

	Return $aRet
EndFunc

Func _ColorConvertCIELabToXYZ($aLab)
	Local $aRet[3]
	Local $fX, $fY, $fZ

	$fY = ($aLab[0] + 16) / 116
	$fX = $aLab[1] / 500 + $fY
	$fZ = $fY - $aLab[2] / 200

	If $fX^3 > 0.008856 Then
		$fX = $fX^3
	Else
		$fX = ($fX - 16/116) / 7.787
	EndIf

	If $fY^3 > 0.008856 Then
		$fY = $fY^3
	Else
		$fY = ($fY - 16/116) / 7.787
	EndIf

	If $fZ^3 > 0.008856 Then
		$fZ = $fZ^3
	Else
		$fZ = ($fZ - 16/116) / 7.787
	EndIf

	$aRet[0] = $fX * 95.047
	$aRet[1] = $fY * 100
	$aRet[2] = $fZ * 108.883

	Return $aRet
EndFunc

Func _ColorConvertCIELabToCIELCH($aLab)
	Local $aRet[3]
	Local $fH

	$fH = __Atan2($aLab[2], $aLab[1])
	If $fH > 0 Then
		$fH = ($fH / ACos(-1))*180
	Else
		$fH = 360 - (Abs($fH)/ACos(-1))*180
	EndIf

	$aRet[0] = $aLab[0] ; L
	$aRet[1] = Sqrt($aLab[1]^2 + $aLab[2]^2) ; C
	$aRet[2] = $fH ; H

	Return $aRet
EndFunc

Func _ColorConvertCIELCHToCIELab($aLCH)
	Local $aRet[3]

	$aRet[0] = $aLCH[0]
	$aRet[1] = Cos(_Radian($aLCH[2]))*$aLCH[1]
	$aRet[2] = Sin(_Radian($aLCH[2]))*$aLCH[1]

	Return $aRet
EndFunc

Func _ColorConvertRGBToCIELCH($aRGB)
	Return _ColorConvertCIELabToCIELCH(_ColorConvertXYZToCIELab(_ColorConvertRGBToXYZ($aRGB)))
EndFunc

Func _ColorConvertCIELCHToRGB($aRGB)
	Return _ColorConvertXYZToRGB(_ColorConvertCIELabToXYZ(_ColorConvertCIELCHToCIELab($aRGB)))
EndFunc

#EndRegion

#Region Delta

Func _ColorCalculateDelta($aLab1, $aLab2, $iMode = $COLOREX_DELTAC)
	Switch $iMode
		Case $COLOREX_DELTAC
			Return Sqrt($aLab2[1]^2 + $aLab2[2]^2) - Sqrt($aLab1[1]^2 + $aLab1[2]^2)
		Case $COLOREX_DELTAH
			Local $fxDE = Sqrt($aLab2[1]^2 + $aLab2[2]^2) - Sqrt($aLab1[1]^2 + $aLab1[2]^2)

			Return Sqrt(($aLab2[1] - $aLab1[1])^2 + ($aLab2[2] - $aLab1[2])^2 - $fxDE^2)
		Case $COLOREX_DELTAE
			Return Sqrt(($aLab1[0]-$aLab2[0])^2 + ($aLab1[1]-$aLab2[1])^2 + ($aLab1[2]-$aLab2[2])^2)
	EndSwitch
EndFunc

#EndRegion

#Region Harmonies

Func _ColorHarmonyTriad($aLCH)
	Local $aLCH1 = $aLCH
	Local $aLCH2 = $aLCH
	Local $aRet[2]

	$aLCH1[2] = $aLCH1[2]-120
	If $aLCH1[2] < 0 Then $aLCH1[2] = 360+$aLCH1[2]

	$aLCH2[2] = Mod($aLCH2[2]+120, 360)

	$aRet[0] = $aLCH1
	$aRet[1] = $aLCH2

	Return $aRet
EndFunc

Func _ColorHarmonySplitComplements($aLCH)
	Local $aLCH1 = $aLCH
	Local $aLCH2 = $aLCH
	Local $aRet[2]

	$aLCH1[2] = $aLCH1[2]-150
	If $aLCH1[2] < 0 Then $aLCH1[2] = 360+$aLCH1[2]

	$aLCH2[2] = Mod($aLCH2[2]+150, 360)

	$aRet[0] = $aLCH1
	$aRet[1] = $aLCH2

	Return $aRet
EndFunc

Func _ColorHarmonyAnalogous($aLCH)
	Local $aLCH1 = $aLCH
	Local $aLCH2 = $aLCH
	Local $aRet[2]

	$aLCH1[2] = $aLCH1[2]-30
	If $aLCH1[2] < 0 Then $aLCH1[2] = 360+$aLCH1[2]

	$aLCH2[2] = Mod($aLCH2[2]+30, 360)

	$aRet[0] = $aLCH1
	$aRet[1] = $aLCH2

	Return $aRet
EndFunc

Func _ColorHarmonyComplement($aLCH)
	Local $aRet = $aLCH

	$aRet[2] = Mod($aRet[2]+180, 360)

	Return $aRet
EndFunc

#EndRegion

Func __Atan2($y, $x)
	If $x > 0 Then
		Return ATan($y/$x)
	ElseIf $x < 0 And $y >= 0 Then
		Return ATan($y/$x) + ACos(-1)
	ElseIf $x < 0 And $y < 0 Then
		Return ATan($y/$x) - ACos(-1)
	ElseIf $x = 0 And $y > 0 Then
		Return ACos(-1)/2
	ElseIf $x = 0 And $y < 0 Then
		Return -ACos(-1)/2
	Else
		Return 0
	EndIf
EndFunc
