#include-once
#include <StringConstants.au3>
#include <PowerPointConstants.au3>
#include <MsgBoxConstants.au3>

#Region #ToDo#
; #ToDo# ========================================================================================================================
; In Development:
;  _PPT_HeaderFooterList:
;      Replace with function _PPT_HeaderFooterGet. Use _PPT_HeaderFooterSet as a Template.
; _PPT_PresentationPrint:
;      Use PrintRange and PrintOptions to exclude "hidden" Slides from printing etc.
; ===============================================================================================================================
#EndRegion #ToDo#

; #INDEX# =======================================================================================================================
; Title .........: Microsoft PowerPoint Function Library
; AutoIt Version : 3.3.12.0
; UDF Version ...: See variable $__g_sVersionPowerPoint
; Language ......: English
; Description ...: A collection of functions for accessing and manipulating Microsoft PowerPoint files
; Author(s) .....: water
; Modified.......: See variable $__g_sVersionPowerPoint
; Remarks .......: Based on the UDF written by toady (see Links)
; Link ..........: https://www.autoitscript.com/forum/topic/50254-powerpoint-wrapper
; Contributors ..:
; ===============================================================================================================================
Global $__g_sVersionPowerPoint = "PowerPoint: 1.5.0.0 2021-08-31"

#Region #VARIABLES#
; #VARIABLES# ===================================================================================================================
Global $__iPPT_Debug = 0 ; Debug level. 0 = no debug information, 1 = Debug info to console, 2 = Debug info to MsgBox, 3 = Debug Info to File
Global $__sPPT_DebugFile = @ScriptDir & "\PowerPoint_Debug.txt" ; Debug file if $__iPPT_Debug is set to 3
Global $__oPPT_Error ; COM Error handler
; ===============================================================================================================================
#EndRegion #VARIABLES#

#Region #CURRENT#
; #CURRENT# =====================================================================================================================
;_PPT_Open
;_PPT_Close
;_PPT_ErrorNotify
;_PPT_CommentAdd
;_PPT_CommentDelete
;_PPT_CommentList
;_PPT_ConvertUnits
;_PPT_ExcelImport
;_PPT_HeaderFooterGet
;_PPT_HeaderFooterSet
;_PPT_PresentationAttach
;_PPT_PresentationClose
;_PPT_PresentationExport
;_PPT_PresentationExportGraphic
;_PPT_PresentationList
;_PPT_PresentationNew
;_PPT_PresentationOpen
;_PPT_PresentationPrint
;_PPT_PresentationSave
;_PPT_PresentationSaveAs
;_PPT_ShapeAdd
;_PPT_ShapeCopyMove
;_PPT_ShapeDelete
;_PPT_ShapeList
;_PPT_ShapeRangeSet
;_PPT_SlideAdd
;_PPT_SlideCopyMove
;_PPT_SlideDelete
;_PPT_SlideList
;_PPT_SlideRangeSet
;_PPT_SlideShow
;_PPT_TableGet
;_PPT_TableSet
;_PPT_TextFindReplace
;_PPT_TextSet
;_PPT_VersionInfo
; ===============================================================================================================================
#EndRegion #CURRENT#

#Region #INTERNAL_USE_ONLY#
; #INTERNAL_USE_ONLY#============================================================================================================
;__PPT_ErrorHandler
; ===============================================================================================================================
#EndRegion #INTERNAL_USE_ONLY#

#Region #FUNCTION#
; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_Open
; Description ...: Connects to an existing PowerPoint instance or creates a new one.
; Syntax.........: _PPT_Open([$bVisible = True])
; Parameters ....: $bVisible - [optional] True specifies that the application will be visible (default = True).
; Return values .: Success - the PowerPoint application object.
;                  Failure - 0 and sets @error.
;                  |1 - Unable to create PowerPoint object. @extended is set to the COM error code
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......: _PPT_Close
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_Open($bVisible = Default)
	Local $oPPT
	If $bVisible = Default Then $bVisible = True
	$oPPT = ObjGet("", "PowerPoint.Application")
	If @error Then
		$oPPT = ObjCreate("PowerPoint.Application")
		If @error Or Not IsObj($oPPT) Then Return SetError(1, @error, 0)
	EndIf
	$oPPT.Visible = $bVisible
	Return $oPPT
EndFunc   ;==>_PPT_Open

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_Close
; Description ...: Closes all presentations and the instance of the PowerPoint application.
; Syntax.........: _PPT_Close($oPPT[, $iSaveChanges = True])
; Parameters ....: $oPPT         - PowerPoint application object as returned by _PPT_Open.
;                  $bSaveChanges - [optional] Specifies whether changed presentations should be saved before closing (default = True).
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPPT is not an object or not an application object
;                  |2 - Error returned by method Application.Quit. @extended is set to the COM error code
;                  |3 - Error returned by method Application.Save. @extended is set to the COM error code
; Author ........: water
; Modified ......:
; Remarks .......: _PPT_Close closes all presentations (even those opened manually by the user for this instance after _PPT_Open)
;                  and the specified PowerPoint instance.
; Related .......: _PPT_Open
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_Close(ByRef $oPPT, $bSaveChanges = Default)
	If $bSaveChanges = Default Then $bSaveChanges = True
	If Not IsObj($oPPT) Or ObjName($oPPT, 1) <> "_Application" Then Return SetError(1, 0, 0)
	If $bSaveChanges Then
		For $oPresentation In $oPPT.Presentations
			If Not $oPresentation.Saved Then
				$oPresentation.Save()
				If @error Then Return SetError(3, @error, 0)
			EndIf
		Next
	EndIf
	$oPPT.Quit()
	If @error Then Return SetError(2, @error, 0)
	$oPPT = 0
	Return 1
EndFunc   ;==>_PPT_Close

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_ErrorNotify
; Description ...: Sets or queries the debug level.
; Syntax.........: _PPT_ErrorNotify($iDebug[, $sDebugFile = @ScriptDir & "\PowerPoint_Debug.txt"])
; Parameters ....: $iDebug     - Debug level. Possible values are:
;                  |-1 - Return the current settings
;                  |0  - Disable debugging
;                  |1  - Enable debugging. Output the debug info to the console
;                  |2  - Enable Debugging. Output the debug info to a MsgBox
;                  |3  - Enable Debugging. Output the debug info to a file defined by $sDebugFile
;                  |4  - Enable Debugging. The COM errors will be handled (the script no longer crashes) without any output
;                  $sDebugFile - [optional] File to write the debugging info to if $iDebug = 3 (default = @ScriptDir & "\PowerPoint_Debug.txt")
; Return values .: Success (for $iDebug => 0) - 1, sets @extended to:
;                  |0 - The COM error handler for this UDF was already active
;                  |1 - A COM error handler has been initialized for this UDF
;                  Success (for $iDebug = -1) - one based one-dimensional array with the following elements:
;                  |1 - Debug level. Value from 0 to 3. Check parameter $iDebug for details
;                  |2 - Debug file. File to write the debugging info to as defined by parameter $sDebugFile
;                  |3 - True if the COM error handler has been defined for this UDF. False if debugging is set off or a COM error handler was already defined
;                  Failure - 0, sets @error to:
;                  |1 - $iDebug is not an integer or < -1 or > 4
;                  |2 - Installation of the custom error handler failed. @extended is set to the error code returned by ObjEvent
;                  |3 - COM error handler already set to another function
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_ErrorNotify($iDebug, $sDebugFile = Default)
	If $sDebugFile = Default Then $sDebugFile = ""
	If Not IsInt($iDebug) Or $iDebug < -1 Or $iDebug > 4 Then Return SetError(1, 0, 0)
	If $sDebugFile = "" Then $sDebugFile = @ScriptDir & "\PowerPoint_Debug.txt"
	Switch $iDebug
		Case -1
			Local $avDebug[4] = [3]
			$avDebug[1] = $__iPPT_Debug
			$avDebug[2] = $__sPPT_DebugFile
			$avDebug[3] = IsObj($__oPPT_Error)
			Return $avDebug
		Case 0
			$__iPPT_Debug = 0
			$__sPPT_DebugFile = ""
			$__oPPT_Error = 0
		Case Else
			$__iPPT_Debug = $iDebug
			$__sPPT_DebugFile = $sDebugFile
			; A COM error handler will be initialized only if one does not exist
			If ObjEvent("AutoIt.Error") = "" Then
				$__oPPT_Error = ObjEvent("AutoIt.Error", "__PPT_ErrorHandler") ; Creates a custom error handler
				If @error <> 0 Then Return SetError(2, @error, 0)
				Return SetError(0, 1, 1)
			ElseIf ObjEvent("AutoIt.Error") = "__PPT_ErrorHandler" Then
				Return SetError(0, 0, 1) ; COM error handler already set by a call to this function
			Else
				Return SetError(3, 0, 0) ; COM error handler already set to another function
			EndIf
	EndSwitch
	Return 1
EndFunc   ;==>_PPT_ErrorNotify

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_CommentAdd
; Description ...: Add a comment anywhere on a Slide or connected to a Shape.
; Syntax.........: _PPT_CommentAdd($oPresentation, $vSlide, $vShape, $iLeft, $iTop, $sComment[, $sAuthor = @UserName[, $sAuthorInitials = @UserName[, $sProviderID = "AD"[, $sUserID = @UserName]]]])
; Parameters ....: $oPresentation   - Presentation object.
;                  $vSlide          - A SlideRange object as returned by _PPT_SlideRangeSet for a single Slide. See _PPT_SlideRangeSet for details.
;                  $vShape          - A ShapeRange object as returned by _PPT_ShapeRangeSet for a single Shape. See _PPT_ShapeRangeSet for details.
;                                     If you set this parameter to Default $iLeft/$iTop will be used to place the comment anywhere on the Slide.
;                                     If you provide a Shape the comment will be placed near the Shape. For details please see the $iLeft and $iTop parameter.
;                  $iLeft           - The position, measured in points, of the left edge of the comment, relative to the left edge of the presentation.
;                                     When $vShape is set the following values will be accepted to place the comment:
;                                     |1 - The comment will be placed at the left edge of the Shape
;                                     |2 - The comment will be placed at the right edge of the Shape
;                                     |3 - The comment will be placed at the top edge of the Shape
;                                     |4 - The comment will be placed at the bottom edge of the Shape
;                  $iTop            - The position, measured in points, of the top edge of the comment, relative to the top edge of the presentation.
;                                     When $vShape is set the following values will be accepted to place the comment:
;                                     |1 - The comment will be placed at the top (for $iLeft = 1 or 2) or left (for $iLeft = 3 or 4) side of the Shape
;                                     |2 - The comment will be placed at the middle of the specified edge of the Shape
;                                     |3 - The comment will be placed at the bottom (for $iLeft = 1 or 2) or right (for $iLeft = 3 or 4) side of the Shape
;                  $sComment        - The comment's text.
;                  $sAuthor         - [optional] The author of the comment (default = @UserName). Please see Remarks
;                  $sAuthorInitials - [optional] The author's initials (default = @UserName)
;                  $sProviderID     - [optional] The service that provides contact information (default = "AD"). Please see Remarks
;                  $sUserID         - [optional] The ID of the user providing the comment (default = @UserName)
; Return values .: Success - the Comment object.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a Presentation object
;                  |2 - Error occurred creating the SlideRange from $vSlide. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |3 - Error occurred creating the ShapeRange from $vShape. @extended is set to the error code returned by _PPT_ShapeRangeSet
;                  |4 - Error setting the Comment. @extended is set to the COM error code
;                  |5 - $iTop is invalid. Has to be a value between 1 and 3
;                  |6 - $iLeft is invalid. Has to be a value between 1 and 4
; Author ........: water
; Modified.......:
; Remarks .......: $sProviderID isn't well documented by MS. But "AD" (Active Directory) and "Windows Live" seem to be valid ProviderIDs.
;                  For $sAuthor I couldn't find a quick and easy way to get the full name of the logged on user. So the initials will be used here as well.
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_CommentAdd($oPresentation, $vSlide, $vShape, $iLeft, $iTop, $sComment, $sAuthor = Default, $sAuthorInitials = Default, $sProviderID = Default, $sUserID = Default)
	Local $oSlide, $oShape, $iShapeLeft, $iShapeTop, $iShapeHeight, $iShapeWidth, $oComment, $iCommentLeft, $iCommentTop
	Local $iCommentOffset = 22 ; Width/Height odf the Comment Icon
	Local $iCommentOffset2 = 5 ; Space between Comment Icon and Shape
	If $sAuthor = Default Then $sAuthor = @UserName
	If $sAuthorInitials = Default Then $sAuthorInitials = @UserName
	If $sProviderID = Default Then $sProviderID = Default
	If $sUserID = Default Then $sUserID = @UserName
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	$oSlide = _PPT_SlideRangeSet($oPresentation, $vSlide)
	If @error Then Return SetError(2, @error, 0)
	If $vShape = Default Then
		$oComment = $oSlide.Comments.Add2($iLeft, $iTop, $sAuthor, $sAuthorInitials, $sComment, $sProviderID, $sUserID)
		If @error Then Return SetError(4, @error, 0)
	Else
		$oShape = _PPT_ShapeRangeSet($oPresentation, $oSlide, $vShape)
		If @error Then Return SetError(3, @error, 0)
		$iShapeLeft = $oShape.Left
		$iShapeTop = $oShape.Top
		$iShapeWidth = $oShape.Width
		$iShapeHeight = $oShape.Height
		Switch $iLeft
			Case 1, 2 ; Left, Right
				$iCommentLeft = $iLeft = 1 ? ($iShapeLeft - $iCommentOffset) : ($iShapeLeft + $iShapeWidth + $iCommentOffset2)
				Switch $iTop
					Case 1 ; Top
						$iCommentTop = $iShapeTop
					Case 2 ; Middle
						$iCommentTop = $iShapeTop + ($iShapeHeight / 2) - ($iCommentOffset / 2)
					Case 3 ; Bottom
						$iCommentTop = $iShapeTop + $iShapeHeight - $iCommentOffset
					Case Else
						SetError(5, 0, 0)
				EndSwitch
			Case 3, 4 ; Top, Bottom
				$iCommentTop = $iLeft = 3 ? ($iShapeTop - $iCommentOffset) : ($iShapeTop + $iShapeHeight + $iCommentOffset2)
				Switch $iTop
					Case 1 ; Left
						$iCommentLeft = $iShapeLeft
					Case 2 ; Middle
						$iCommentLeft = $iShapeLeft + ($iShapeWidth / 2) - ($iCommentOffset / 2)
					Case 3 ; Right
						$iCommentLeft = $iShapeLeft + $iShapeWidth - $iCommentOffset
					Case Else
						SetError(5, 0, 0)
				EndSwitch
			Case Else
				SetError(6, 0, 0)
		EndSwitch
		$oComment = $oSlide.Comments.Add2($iCommentLeft, $iCommentTop, $sAuthor, $sAuthorInitials, $sComment, $sProviderID, $sUserID)
		If @error Then Return SetError(4, @error, 0)
	EndIf
	Return $oComment
EndFunc   ;==>_PPT_CommentAdd

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_CommentDelete
; Description ...: Delete a single Comment from a Slide or all Comments from a SlideRange.
; Syntax.........: _PPT_CommentDelete($oObject[, $vSlide = 0])
; Parameters ....: $oObject - Presentation object (if you want to delete > 1 Comment) or Comment object (if you want to delete a single Comment)
;                  $vSlides - [optional] SlideRange object as returned by _PPT_SlideRangeSet for the Slide(s) to process. See _PPT_SlideRangeSet for details (default = 0 = all Slides)
; Return values .: Success - 1, @extended is set to the number of Comments deleted.
;                  Failure - 0 and sets @error.
;                  |1 - $oObject is not an object or not a Presentation object
;                  |2 - Error occurred creating the SlideRange from $vSlide. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |3 - Error deleting a specified single Comment. @extended is set to the COM error code
;                  |4 - Error deleting all Comments from a Slide. @extended is set to the COM error code
; Author ........: water
; Modified.......:
; Remarks .......: If $vSlides defines a SlideRange all Comments from this SlideRange will be deleted.
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_CommentDelete($oObject, $vSlides = Default)
	Local $oSlideRange, $iDeleted = 0
	If Not IsObj($oObject) Or (ObjName($oObject, 1) <> "_Presentation" And ObjName($oObject, 1) <> "Comment") Then Return SetError(1, 0, 0)
	If $vSlides = Default Then $vSlides = 0
	If ObjName($oObject, 1) = "Comment" Then
		$oObject.Delete
		If @error Then Return SetError(3, @error, 0)
		$iDeleted = 1
	Else
		$oSlideRange = _PPT_SlideRangeSet($oObject, $vSlides)
		If @error Then Return SetError(2, @error, 0)
		For $oSlide In $oSlideRange
			For $i = $oSlide.Comments.Count To 1 Step -1
				$oSlide.Comments.Item($i).Delete
				If @error Then Return SetError(4, @error, 0)
				$iDeleted = $iDeleted + 1
			Next
		Next
	EndIf
	Return SetError(0, $iDeleted, 1)
EndFunc   ;==>_PPT_CommentDelete

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_CommentList
; Description ...: Get a list of all Comments in a presentation.
; Syntax.........: _PPT_CommentList($oPresentation)
; Parameters ....: $oPresentation   - Presentation object.
; Return values .: Success - two-dimensional zero based array holding the following information:
;                  |0 - Object of the Comment
;                  |1 - SlideIndex of the Slide where the Comment is located
;                  |2 - Comment Text
;                  |3 - Comment Author
;                  |4 - Comment AuthorInitials
;                  |5 - DateTime the Comment has been created
;                  |6 - The position, measured in points, of the left edge of the comment, relative to the left edge of the presentation.
;                  |7 - The position, measured in points, of the top edge of the comment, relative to the top edge of the presentation.
;                  |8 - Number of Replies to this Comment.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a Presentation object
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_CommentList($oPresentation)
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	Local $aComments[0][9]
	Local $iCommentsIndex = 0
	For $oSlide In $oPresentation.Slides
		With $oSlide
			If IsObj(.Comments) Then
				ReDim $aComments[UBound($aComments, 1) + .Comments.Count][9]
				For $oComment In .Comments
					$aComments[$iCommentsIndex][0] = $oComment
					$aComments[$iCommentsIndex][1] = $oSlide.SlideIndex
					$aComments[$iCommentsIndex][2] = $oComment.Text
					$aComments[$iCommentsIndex][3] = $oComment.Author
					$aComments[$iCommentsIndex][4] = $oComment.AuthorInitials
					$aComments[$iCommentsIndex][5] = $oComment.DateTime
					$aComments[$iCommentsIndex][6] = $oComment.Left
					$aComments[$iCommentsIndex][7] = $oComment.Top
					$aComments[$iCommentsIndex][8] = $oComment.Replies.Count
					$iCommentsIndex = $iCommentsIndex + 1
				Next
			EndIf
		EndWith
	Next
	Return $aComments
EndFunc   ;==>_PPT_CommentList

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_ConvertUnits
; Description ...: Convert units (like centimeters) to other units (like points).
; Syntax.........: _PPT_VersionInfo($iValue, $iConversion)
; Parameters ....: $iValue      - Int or Float to convert
;                  $iConversion - Conversion to execute. Can be one of the following:
;                  |1 - CentimetersToPoints
;                  |2 - InchesToPoints
;                  |3 - PicasToPoints
;                  |4 - PointsToCentimeters
;                  |5 - PointsToInches
;                  |6 - PointsToLines
;                  |7 - PointsToPicas
; Return values .: Success - Converted value.
;                  Failure - 0 and sets @error.
;                  |1 - $iConversion is invalid
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================Func CentimetersToPoints($iValue)
Func _PPT_ConvertUnits($iValue, $iConversion)
	Switch $iConversion
		Case 1     ; CentimetersToPoints
			Return ($iValue * 28.35)
		Case 2     ; InchesToPoints
			Return ($iValue * 72)
		Case 1     ; LinesToPoints
			Return ($iValue * 12)
		Case 3     ; PicasToPoints
			Return ($iValue * 12)
		Case 4     ; PointsToCentimeters
			Return ($iValue / 28.35)
		Case 5     ; PointsToInches
			Return ($iValue / 72)
		Case 6     ; PointsToLines
			Return ($iValue / 12)
		Case 7     ; PointsToPicas
			Return ($iValue / 12)
		Case Else
			Return SetError(1, 0, 0)
	EndSwitch
EndFunc   ;==>_PPT_ConvertUnits

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_ExcelImport
; Description ...: Import a Range of cells or a Chart from Excel and insert as Table or Picture into a Slide.
; Syntax.........: _PPT_ExcelImport($oPresentation, $vSlide, $oObject, $vShape, $iFlag)
; Parameters ....: $oPresentation - Object of the presentation to process.
;                  $vSlide        - Single Slide to process. Please see _PPT_SlideRangeSet for possible values.
;                  $vShape        - Table shape to copy the Excel cells to. If needed the table will be expanded to fit the Excel Cell Range.
;                                   If set to Default a new Table Shape will be created. This parameter is ignored for $iFlag = 2.
;                  $oObject       - Range object of the Excel cells to process or Excel Chart object.
;                  $iFlag         - Processing options. Can be a combination of the following values:
;                  |  1 - Insert the Range as a new Picture Shape
;                  |  2 - Insert the Range as a new Table Shape (editable)
;                  |  3 - Insert a Chart as a new Picture Shape
; Return values .: Success - the Shape object that has been created to hold the imported Excel Range.
;                  Failure - 0 and sets @error.
;                  | 1 - $vSlides is invalid. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  | 2 - Error when copying the Excel Range as picture to the clipboard. @extended is set to the COM error code
;                  | 3 - Error when pasting the Excel from the clipboard. @extended is set to the COM error code
;                  | 4 - Error reading the Excel Range. @extended is set to the COM error code
;                  | 5 - Error creating the Table Shape. @extended is set to the COM error code
;                  | 6 - Error writing Excel data to the Table Shape. @extended is set to the COM error code
;                  | 7 - $vShape is not a Table Shape
;                  | 8 - $vShape is not a Shape
;                  | 9 - $iFlag is invalid
;                  |10 - Error when copying the Excel Chart object. @extended is set to the COM error code
;                  |11 - Error when pasting the Excel Chart object. @extended is set to the COM error code
; Author ........: water
; Modified.......:
; Remarks .......: https://img.chandoo.org/vba/Automatically_Create_PowerPoint_From_Excel_VBA_Code.txt
;                  https://analystcave.com/vba-paste-from-excel-to-powerpoint/
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_ExcelImport($oPresentation, $vSlide, $vShape, $oObject, $iFlag)
	; XlPictureAppearance enumeration. Specifies how the picture should be copied.
	; See: https://docs.microsoft.com/en-us/office/vba/api/excel.xlpictureappearance
	Local Const $xlPrinter = 2     ; The picture is copied as it will look when it is printed
	Local Const $xlScreen = 1     ; The picture is copied to resemble its display on the screen as closely as possible

	; XlCopyPictureFormat enumeration. Specifies the format of the picture being copied.
	; See: https://docs.microsoft.com/en-us/office/vba/api/excel.xlcopypictureformat
	Local Const $xlBitmap = 2     ; Picture copied in bitmap (raster) format: bmp, jpg, gif, png
	Local Const $xlPicture = -4147     ; Picture copied in vector format: emf, wmf
	#forceref $xlPrinter, $xlBitmap

	Local $oShape
	Local $oSlide = _PPT_SlideRangeSet($oPresentation, $vSlide)
	If @error Then Return SetError(1, @error, 0)
	Switch $iFlag
		Case 1     ; Copy Range as Picture
			$oObject.CopyPicture($xlScreen, $xlPicture)
			If @error Then Return SetError(2, @error, 0)
			; Insert the picture into the Slide as a new Shape
			$oShape = $oSlide.Shapes.Paste
			If @error Then Return SetError(3, @error, 0)
		Case 2     ; Copy Range as Table
			Local $aData = $oObject.Application.Transpose($oObject.Value)
			If @error Then Return SetError(4, @error, 0)
			If $vShape = Default Then
				Local $iWidth = $oSlide.CustomLayout.Width * .60
				Local $iHeight = $oSlide.CustomLayout.Height * .60
				$oShape = _PPT_ShapeAdd($oPresentation, $vSlide, $MsoTable, Default, Default, $iWidth, $iHeight, UBound($aData, 1), UBound($aData, 2))[0]
				If @error Then Return SetError(5, @error, 0)
			Else
				$oShape = _PPT_ShapeRangeSet($oPresentation, $oSlide, $vShape)
				If @error Then Return SetError(8, @error, 0)
				If Not $oShape.HasTable Then Return SetError(7, @error, 0)
				Local $iShapeRows = $oShape.Table.Rows.Count
				Local $iShapeCols = $oShape.Table.Columns.Count
				If $iShapeRows < UBound($aData, 1) Then
					Local $iRows2Add = UBound($aData, 1) - $iShapeRows
					For $i = 1 To $iRows2Add
						$oShape.Table.Rows.Add(1)
					Next
				EndIf
				If $iShapeCols < UBound($aData, 2) Then
					Local $iCols2Add = UBound($aData, 2) - $iShapeCols
					For $i = 1 To $iCols2Add
						$oShape.Table.Columns.Add(1)
					Next
				EndIf
			EndIf
			For $iTableRow = 0 To UBound($aData, 1) - 1
				For $iTableCol = 0 To UBound($aData, 2) - 1
					$oShape.Table.Cell($iTableRow + 1, $iTableCol + 1).Shape.TextFrame.TextRange.Text = $aData[$iTableRow][$iTableCol]
					If @error Then Return SetError(6, @error, 0)
				Next
			Next
		Case 3     ; Copy Chart as Picture
			$oObject.Copy
			If @error Then Return SetError(10, @error, 0)
			$oSlide.Shapes.PasteSpecial($ppPasteMetafilePicture)
			If @error Then Return SetError(11, @error, 0)
		Case Else
			Return SetError(9, 0, 0)
	EndSwitch
	Return $oShape
EndFunc   ;==>_PPT_ExcelImport

; >>>> WORK IN PROGRESS <<<<
#cs
; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_HeaderFooterGet
; Description ...: Sets the Header/Footer for a Slide/Slides or the SlideMaster/NotesMaster/HandoutMaster/TitleMaster of a Presentation.
; Syntax.........: _PPT_HeaderFooterGet($oPresentation, $iFlag[, $sFooter = Default[, $sHeader = Default[, $sDateAndTime = Default[, $bSlideNumber = Default[, $vSlides = 0]]]]])
; Parameters ....: $oPresentation - Presentation object.
;                  $iFlag         - Processing options. Can be a combination of the following values:
;                  |  1 - Set the SlideMaster Footer
;                  |  2 - Set the NotesMaster Footer and/or Header
;                  |  4 - Set the HandoutMaster Footer and/or Header
;                  |  8 - Set the TitleMaster Footer
;                  | 16 - Set the Footer for the specified Slide(s)
;                  | 32 - Clear Header/Footer information that has been set for individual Slides to make sure all Slides display the Header and information you define for the SlideMaster
;                  | 64 - Specifies if the Footer/Header for the Slide(s)/SlideMaster/NotesMaster/HandoutMaster/TitleMaster will be visible
;                  |128 - Determines whether the footer, date and time, and slide number appear on the title slide. Applies to slide masters
;                  |256 - Updates all existing Slides with the specified Header/Footer settings
;                  $sFooter       - [optional] Text to use for the Footer. "" will reset the Footer text (default = keyword Default = do not set the Footer).
;                  $sHeader       - [optional] Text to use for the Header. "" will reset the Header text (default = keyword Default = do not set the Header).
;                  $sDateAndTime  - [optional] Format for the automatically updated date and time. Use the PpDateTimeFormat enumeration (default = keyword Default = do not set the DateAndTime).
;                  $bSlideNumber  - [optional] Displays the SlideNumber if set to True (default = keyword Default = do not set the SlideNumber).
;                  $vSlides       - [optional] Slide(s) to process. Please see _PPT_SlideRangeSet for possible values (default = 0 = all Slides).
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation has to be a presentation
;                  |2 - Error occurred when setting the Footer for the SlideMaster. @extended is set to the COM error code
;                  |3 - Error occurred when setting the Footer/Header for the NotesMaster. @extended is set to the COM error code
;                  |4 - Error occurred when setting the Footer/Header for the HandoutMaster. @extended is set to the COM error code
;                  |5 - Error occurred when setting the Footer for the TitleMaster. @extended is set to the COM error code
;                  |6 - Error occurred when setting the Footer for a SlideRange. @extended is set to the COM error code
;                  |7 - Error returned from _PPT_SetSlideRange. @extended is set to the error code returned by this function
;                  |8 - Error clearing individual Header/Footer information
; Author ........: water
; Modified.......:
; Remarks .......: CustomLayout and NotesPage can not be processed using this function.
;                  HeaderFooter objects aren't available for Slide objects that represent Notes pages.
;                  The HeaderFooter object that represents a header is available only for a NotesMaster or HandoutMaster.
;+
;                  Parameter $sDateAndTime: https://docs.microsoft.com/en-us/office/vba/api/powerpoint.ppdatetimeformat
;+
;                  See: https://stackoverflow.com/questions/35739457/vba-powerpoint-2010-replacing-text-in-headers-and-footers
; Related .......: _PPT_HeaderFooterList
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_HeaderFooterGet($oPresentation, $iFlag, $sFooter = Default, $sHeader = Default, $sDateAndTime = Default, $bSlideNumber = Default, $vSlides = Default)
		If IsObj($oPresentation) = 0 Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
		If $iFlag = Default Then $iFlag = 0
		If $vSlides = Default Then $vSlides = 0
		Local $oSlideRange
		; --------------------------------------
		; Set HeadersFooters for the SlideMaster
		; --------------------------------------
		If BitAND($iFlag, 1) = 1 Then
			Local $oMaster = $oPresentation.SlideMaster
			With $oMaster.HeadersFooters
				If $sFooter <> Default Then
					If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
					.Footer.Text = $sFooter
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $sDateAndTime <> Default Then
					If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
					.DateAndTime.UseFormat = True
					If @error Then Return SetError(2, @error, 0)
					.DateAndTime.Format = $sDateAndTime
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $bSlideNumber <> Default Then
					If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
					.SlideNumber.Visible = $bSlideNumber
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If BitAND($iFlag, 128) = 128 Then .DisplayOnTitleSlide = True
			EndWith
			; https://stackoverflow.com/questions/62076856/update-all-slide-masters-and-layouts-footers
			; Loop through the Designs
			For $oMaster In $oPresentation.Designs
				With $oMaster.SlideMaster.HeadersFooters
					If $sFooter <> Default Then
						If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
						.Footer.Text = $sFooter
						If @error Then Return SetError(2, @error, 0)
					EndIf
					If $sDateAndTime <> Default Then
						If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
						.DateAndTime.UseFormat = True
						If @error Then Return SetError(2, @error, 0)
						.DateAndTime.Format = $sDateAndTime
						If @error Then Return SetError(2, @error, 0)
					EndIf
					If $bSlideNumber <> Default Then
						If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
						.SlideNumber.Visible = $bSlideNumber
						If @error Then Return SetError(2, @error, 0)
					EndIf
					If BitAND($iFlag, 128) = 128 Then .DisplayOnTitleSlide = True
				EndWith
			Next
			; Loop through the CustomLayouts
			For $oLayout In $oMaster.SlideMaster.CustomLayouts
				With $oLayout.HeadersFooters
					If $sFooter <> Default Then
						If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
						.Footer.Text = $sFooter
						If @error Then Return SetError(2, @error, 0)
					EndIf
					If $sDateAndTime <> Default Then
						If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
						.DateAndTime.UseFormat = True
						If @error Then Return SetError(2, @error, 0)
						.DateAndTime.Format = $sDateAndTime
						If @error Then Return SetError(2, @error, 0)
					EndIf
					If $bSlideNumber <> Default Then
						If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
						.SlideNumber.Visible = $bSlideNumber
						If @error Then Return SetError(2, @error, 0)
					EndIf
					If BitAND($iFlag, 128) = 128 Then .DisplayOnTitleSlide = True
				EndWith
			Next
			; ==> (1) Gilt das Ändern aller Slides auch für Notes/Handouts?
			If BitAND($iFlag, 256) = 256 Then
				For $oSlide In $oPresentation.Slides
					With $oSlide.HeadersFooters
						If $sFooter <> Default Then
							If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
							.Footer.Text = $sFooter
							If @error Then Return SetError(2, @error, 0)
						EndIf
						If $sDateAndTime <> Default Then
							If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
							.DateAndTime.UseFormat = True
							If @error Then Return SetError(2, @error, 0)
							.DateAndTime.Format = $sDateAndTime
							If @error Then Return SetError(2, @error, 0)
						EndIf
						If $bSlideNumber <> Default Then
							If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
							.SlideNumber.Visible = $bSlideNumber
							If @error Then Return SetError(2, @error, 0)
						EndIf
						; ===>				If BitAnd($iFlag, 128) = 128 Then .DisplayOnTitleSlide = True
					EndWith
				Next
			EndIf
		EndIf
		; --------------------------------------
		; Set HeadersFooters for the NotesMaster
		; --------------------------------------
		; ==> Gilt das Ändern aller Slides auch für Notes? Siehe (1)
		If BitAND($iFlag, 2) = 2 Then
			$oMaster = $oPresentation.NotesMaster
			With $oMaster.HeadersFooters
				If $sFooter <> Default Then
					If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
					.Footer.Text = $sFooter
					If @error Then Return SetError(3, @error, 0)
				EndIf
				If $sHeader <> Default Then
					.Header.Text = $sHeader
					If BitAND($iFlag, 64) = 64 Then .Header.Visible = True
					If @error Then Return SetError(3, @error, 0)
				EndIf
				If $sDateAndTime <> Default Then
					If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
					.DateAndTime.UseFormat = True
					If @error Then Return SetError(2, @error, 0)
					.DateAndTime.Format = $sDateAndTime
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $bSlideNumber <> Default Then
					If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
					.SlideNumber.Visible = $bSlideNumber
					If @error Then Return SetError(2, @error, 0)
				EndIf
			EndWith
		EndIf
		; ----------------------------------------
		; Set HeadersFooters for the HandoutMaster
		; ----------------------------------------
		; ==> Gilt das Ändern aller Slides auch für Handouts? Siehe (1)
		If BitAND($iFlag, 4) = 4 Then
			$oMaster = $oPresentation.HandoutMaster
			With $oMaster.HeadersFooters
				If $sFooter <> Default Then
					If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
					.Footer.Text = $sFooter
					If @error Then Return SetError(4, @error, 0)
				EndIf
				If $sHeader <> Default Then
					If BitAND($iFlag, 64) = 64 Then .Header.Visible = True
					.Header.Text = $sHeader
					If @error Then Return SetError(4, @error, 0)
				EndIf
				If $sDateAndTime <> Default Then
					If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
					.DateAndTime.UseFormat = True
					If @error Then Return SetError(2, @error, 0)
					.DateAndTime.Format = $sDateAndTime
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $bSlideNumber <> Default Then
					If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
					.SlideNumber.Visible = $bSlideNumber
					If @error Then Return SetError(2, @error, 0)
				EndIf
			EndWith
		EndIf
		; --------------------------------------
		; Set HeadersFooters for the TitleMaster
		; --------------------------------------
		If BitAND($iFlag, 8) = 8 And $oPresentation.HasTitleMaster Then
			$oMaster = $oPresentation.TitleMaster
			With $oMaster.HeadersFooters
				If $sFooter <> Default Then
					If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
					.Footer.Text = $sFooter
					If @error Then Return SetError(5, @error, 0)
				EndIf
				If $sDateAndTime <> Default Then
					If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
					.DateAndTime.UseFormat = True
					If @error Then Return SetError(2, @error, 0)
					.DateAndTime.Format = $sDateAndTime
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $bSlideNumber <> Default Then
					If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
					.SlideNumber.Visible = $bSlideNumber
					If @error Then Return SetError(2, @error, 0)
				EndIf
			EndWith
		EndIf
		; -----------------------------------
		; Set HeadersFooters for a SlideRange
		; -----------------------------------
		If BitAND($iFlag, 16) = 16 Then
			$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)
			If @error Then Return SetError(7, @error, 0)
			For $oSlide In $oSlideRange
				With $oSlide.HeadersFooters
					If $sFooter <> Default Then
						If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
						.Footer.Text = $sFooter
						If @error Then Return SetError(6, @error, 0)
					EndIf
					If $sDateAndTime <> Default Then
						If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
						.DateAndTime.UseFormat = True
						If @error Then Return SetError(2, @error, 0)
						.DateAndTime.Format = $sDateAndTime
						If @error Then Return SetError(2, @error, 0)
					EndIf
					If $bSlideNumber <> Default Then
						If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
						.SlideNumber.Visible = $bSlideNumber
						If @error Then Return SetError(2, @error, 0)
					EndIf
				EndWith
			Next
		EndIf
		; -------------------------------------
		; Clear HeadersFooters for a SlideRange
		; -------------------------------------
		; See: https://docs.microsoft.com/en-us/office/vba/api/powerpoint.headerfooter
		If BitAND($iFlag, 32) = 32 Then
			$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)
			If @error Then Return SetError(7, @error, 0)
			For $oSlide In $oSlideRange
				$oSlide.DisplayMasterShapes = True
				If @error Then Return SetError(8, @error, 0)
				$oSlide.HeadersFooters.Clear
				If @error Then Return SetError(8, @error, 0)
			Next
		EndIf
		Return 1
EndFunc   ;==>_PPT_HeaderFooterGet
#ce

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_HeaderFooterSet
; Description ...: Sets the Header/Footer for a Slide/Slides or the SlideMaster/NotesMaster/HandoutMaster/TitleMaster of a Presentation.
; Syntax.........: _PPT_HeaderFooterSet($oPresentation, $iFlag[, $sFooter = Default[, $sHeader = Default[, $sDateAndTime = Default[, $bSlideNumber = Default[, $vSlides = 0]]]]])
; Parameters ....: $oPresentation - Presentation object.
;                  $iFlag         - Processing options. Can be a combination of the following values:
;                  |  1 - Set the SlideMaster Footer
;                  |  2 - Set the NotesMaster Footer and/or Header
;                  |  4 - Set the HandoutMaster Footer and/or Header
;                  |  8 - Set the TitleMaster Footer
;                  | 16 - Set the Footer for the specified Slide(s)
;                  | 32 - Clear Header/Footer information that has been set for individual Slides to make sure all Slides display the Header and information you define for the SlideMaster
;                  | 64 - Specifies if the Footer/Header for the Slide(s)/SlideMaster/NotesMaster/HandoutMaster/TitleMaster will be visible
;                  |128 - Determines whether the footer, date and time, and slide number appear on the title slide. Applies to slide masters
;                  |256 - Updates all existing Slides with the specified Header/Footer settings
;                  $sFooter       - [optional] Text to use for the Footer. "" will reset the Footer text (default = keyword Default = do not set the Footer).
;                  $sHeader       - [optional] Text to use for the Header. "" will reset the Header text (default = keyword Default = do not set the Header).
;                  $sDateAndTime  - [optional] Format for the automatically updated date and time. Use the PpDateTimeFormat enumeration (default = keyword Default = do not set the DateAndTime).
;                  $bSlideNumber  - [optional] Displays the SlideNumber if set to True (default = keyword Default = do not set the SlideNumber).
;                  $vSlides       - [optional] Slide(s) to process. Please see _PPT_SlideRangeSet for possible values (default = 0 = all Slides).
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation has to be a presentation
;                  |2 - Error occurred when setting the Footer for the SlideMaster. @extended is set to the COM error code
;                  |3 - Error occurred when setting the Footer/Header for the NotesMaster. @extended is set to the COM error code
;                  |4 - Error occurred when setting the Footer/Header for the HandoutMaster. @extended is set to the COM error code
;                  |5 - Error occurred when setting the Footer for the TitleMaster. @extended is set to the COM error code
;                  |6 - Error occurred when setting the Footer for a SlideRange. @extended is set to the COM error code
;                  |7 - Error returned from _PPT_SetSlideRange. @extended is set to the error code returned by this function
;                  |8 - Error clearing individual Header/Footer information
; Author ........: water
; Modified.......:
; Remarks .......: CustomLayout and NotesPage can not be processed using this function.
;                  HeaderFooter objects aren't available for Slide objects that represent Notes pages.
;                  The HeaderFooter object that represents a header is available only for a NotesMaster or HandoutMaster.
;+
;                  Parameter $sDateAndTime: https://docs.microsoft.com/en-us/office/vba/api/powerpoint.ppdatetimeformat
;+
;                  See: https://stackoverflow.com/questions/35739457/vba-powerpoint-2010-replacing-text-in-headers-and-footers
; Related .......: _PPT_HeaderFooterList
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_HeaderFooterSet($oPresentation, $iFlag, $sFooter = Default, $sHeader = Default, $sDateAndTime = Default, $bSlideNumber = Default, $vSlides = Default)
	If IsObj($oPresentation) = 0 Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If $iFlag = Default Then $iFlag = 0
	If $vSlides = Default Then $vSlides = 0
	Local $oSlideRange
	; --------------------------------------
	; Set HeadersFooters for the SlideMaster
	; --------------------------------------
	If BitAND($iFlag, 1) = 1 Then
		Local $oMaster = $oPresentation.SlideMaster
		With $oMaster.HeadersFooters
			If $sFooter <> Default Then
				If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
				.Footer.Text = $sFooter
				If @error Then Return SetError(2, @error, 0)
			EndIf
			If $sDateAndTime <> Default Then
				If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
				.DateAndTime.UseFormat = True
				If @error Then Return SetError(2, @error, 0)
				.DateAndTime.Format = $sDateAndTime
				If @error Then Return SetError(2, @error, 0)
			EndIf
			If $bSlideNumber <> Default Then
				If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
				.SlideNumber.Visible = $bSlideNumber
				If @error Then Return SetError(2, @error, 0)
			EndIf
			If BitAND($iFlag, 128) = 128 Then .DisplayOnTitleSlide = True
		EndWith
		; https://stackoverflow.com/questions/62076856/update-all-slide-masters-and-layouts-footers
		; Loop through the Designs
		For $oMaster In $oPresentation.Designs
			With $oMaster.SlideMaster.HeadersFooters
				If $sFooter <> Default Then
					If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
					.Footer.Text = $sFooter
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $sDateAndTime <> Default Then
					If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
					.DateAndTime.UseFormat = True
					If @error Then Return SetError(2, @error, 0)
					.DateAndTime.Format = $sDateAndTime
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $bSlideNumber <> Default Then
					If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
					.SlideNumber.Visible = $bSlideNumber
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If BitAND($iFlag, 128) = 128 Then .DisplayOnTitleSlide = True
			EndWith
		Next
		; Loop through the CustomLayouts
		For $oLayout In $oMaster.SlideMaster.CustomLayouts
			With $oLayout.HeadersFooters
				If $sFooter <> Default Then
					If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
					.Footer.Text = $sFooter
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $sDateAndTime <> Default Then
					If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
					.DateAndTime.UseFormat = True
					If @error Then Return SetError(2, @error, 0)
					.DateAndTime.Format = $sDateAndTime
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $bSlideNumber <> Default Then
					If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
					.SlideNumber.Visible = $bSlideNumber
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If BitAND($iFlag, 128) = 128 Then .DisplayOnTitleSlide = True
			EndWith
		Next
		; ==> (1) Gilt das Ändern aller Slides auch für Notes/Handouts?
		If BitAND($iFlag, 256) = 256 Then
			For $oSlide In $oPresentation.Slides
				With $oSlide.HeadersFooters
					If $sFooter <> Default Then
						If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
						.Footer.Text = $sFooter
						If @error Then Return SetError(2, @error, 0)
					EndIf
					If $sDateAndTime <> Default Then
						If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
						.DateAndTime.UseFormat = True
						If @error Then Return SetError(2, @error, 0)
						.DateAndTime.Format = $sDateAndTime
						If @error Then Return SetError(2, @error, 0)
					EndIf
					If $bSlideNumber <> Default Then
						If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
						.SlideNumber.Visible = $bSlideNumber
						If @error Then Return SetError(2, @error, 0)
					EndIf
					; ===>				If BitAnd($iFlag, 128) = 128 Then .DisplayOnTitleSlide = True
				EndWith
			Next
		EndIf
	EndIf
	; --------------------------------------
	; Set HeadersFooters for the NotesMaster
	; --------------------------------------
	; ==> Gilt das Ändern aller Slides auch für Notes? Siehe (1)
	If BitAND($iFlag, 2) = 2 Then
		$oMaster = $oPresentation.NotesMaster
		With $oMaster.HeadersFooters
			If $sFooter <> Default Then
				If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
				.Footer.Text = $sFooter
				If @error Then Return SetError(3, @error, 0)
			EndIf
			If $sHeader <> Default Then
				.Header.Text = $sHeader
				If BitAND($iFlag, 64) = 64 Then .Header.Visible = True
				If @error Then Return SetError(3, @error, 0)
			EndIf
			If $sDateAndTime <> Default Then
				If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
				.DateAndTime.UseFormat = True
				If @error Then Return SetError(2, @error, 0)
				.DateAndTime.Format = $sDateAndTime
				If @error Then Return SetError(2, @error, 0)
			EndIf
			If $bSlideNumber <> Default Then
				If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
				.SlideNumber.Visible = $bSlideNumber
				If @error Then Return SetError(2, @error, 0)
			EndIf
		EndWith
	EndIf
	; ----------------------------------------
	; Set HeadersFooters for the HandoutMaster
	; ----------------------------------------
	; ==> Gilt das Ändern aller Slides auch für Handouts? Siehe (1)
	If BitAND($iFlag, 4) = 4 Then
		$oMaster = $oPresentation.HandoutMaster
		With $oMaster.HeadersFooters
			If $sFooter <> Default Then
				If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
				.Footer.Text = $sFooter
				If @error Then Return SetError(4, @error, 0)
			EndIf
			If $sHeader <> Default Then
				If BitAND($iFlag, 64) = 64 Then .Header.Visible = True
				.Header.Text = $sHeader
				If @error Then Return SetError(4, @error, 0)
			EndIf
			If $sDateAndTime <> Default Then
				If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
				.DateAndTime.UseFormat = True
				If @error Then Return SetError(2, @error, 0)
				.DateAndTime.Format = $sDateAndTime
				If @error Then Return SetError(2, @error, 0)
			EndIf
			If $bSlideNumber <> Default Then
				If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
				.SlideNumber.Visible = $bSlideNumber
				If @error Then Return SetError(2, @error, 0)
			EndIf
		EndWith
	EndIf
	; --------------------------------------
	; Set HeadersFooters for the TitleMaster
	; --------------------------------------
	If BitAND($iFlag, 8) = 8 And $oPresentation.HasTitleMaster Then
		$oMaster = $oPresentation.TitleMaster
		With $oMaster.HeadersFooters
			If $sFooter <> Default Then
				If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
				.Footer.Text = $sFooter
				If @error Then Return SetError(5, @error, 0)
			EndIf
			If $sDateAndTime <> Default Then
				If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
				.DateAndTime.UseFormat = True
				If @error Then Return SetError(2, @error, 0)
				.DateAndTime.Format = $sDateAndTime
				If @error Then Return SetError(2, @error, 0)
			EndIf
			If $bSlideNumber <> Default Then
				If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
				.SlideNumber.Visible = $bSlideNumber
				If @error Then Return SetError(2, @error, 0)
			EndIf
		EndWith
	EndIf
	; -----------------------------------
	; Set HeadersFooters for a SlideRange
	; -----------------------------------
	If BitAND($iFlag, 16) = 16 Then
		$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)
		If @error Then Return SetError(7, @error, 0)
		For $oSlide In $oSlideRange
			With $oSlide.HeadersFooters
				If $sFooter <> Default Then
					If BitAND($iFlag, 64) = 64 Then .Footer.Visible = True
					.Footer.Text = $sFooter
					If @error Then Return SetError(6, @error, 0)
				EndIf
				If $sDateAndTime <> Default Then
					If BitAND($iFlag, 64) = 64 Then .DateAndTime.Visible = True
					.DateAndTime.UseFormat = True
					If @error Then Return SetError(2, @error, 0)
					.DateAndTime.Format = $sDateAndTime
					If @error Then Return SetError(2, @error, 0)
				EndIf
				If $bSlideNumber <> Default Then
					If BitAND($iFlag, 64) = 64 Then .SlideNumber.Visible = True
					.SlideNumber.Visible = $bSlideNumber
					If @error Then Return SetError(2, @error, 0)
				EndIf
			EndWith
		Next
	EndIf
	; -------------------------------------
	; Clear HeadersFooters for a SlideRange
	; -------------------------------------
	; See: https://docs.microsoft.com/en-us/office/vba/api/powerpoint.headerfooter
	If BitAND($iFlag, 32) = 32 Then
		$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)
		If @error Then Return SetError(7, @error, 0)
		For $oSlide In $oSlideRange
			$oSlide.DisplayMasterShapes = True
			If @error Then Return SetError(8, @error, 0)
			$oSlide.HeadersFooters.Clear
			If @error Then Return SetError(8, @error, 0)
		Next
	EndIf
	Return 1
EndFunc   ;==>_PPT_HeaderFooterSet

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationAttach
; Description ...: Attaches to the frist presentation where the search string matches based on the selected mode.
; Syntax.........: _PPT_PresentationAttach($sString[, $sMode = "FilePath"[, $bPartialMatch = False]])
; Parameters ....: $sString       - String to search for.
;                  $sMode         - [optional] specifies search mode:
;                  |FileName      - Name of the open presentation (e.g. "Presentation1.pptx")
;                  |FilePath      - Full path to the open presentation (default) (e.g. "C:\Temp\Presentation1.pptx")
;                  |Title         - Caption of the first window of the presentation (e.g. "Presentation1.pptx [Write protected]")
;                  $bPartialMatch - [optional] When $sMode = Title then $sString must fully match when False (default) or partial if True
; Return values .: Success - the PowerPoint presentation object.
;                  Failure - 0 and sets @error.
;                  |1 - An error occurred. @extended is set to the COM error code
;                  |2 - $sMode is invalid
;                  |3 - $sString can't be found in any of the open presentations
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......: _PPT_PresentationClose, _PPT_PresentationNew, _PPT_PresentationOpen
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationAttach($sString, $sMode = Default, $bPartialMatch = Default)
	Local $oPresentation, $iCount = 0, $sCLSID_Presentation = "{91493444-5A91-11CF-8700-00AA0060263B}"     ; Microsoft.Office.Interop.PowerPoint.PresentationClass
	If $sMode = Default Then $sMode = "FilePath"
	If $bPartialMatch = Default Then $bPartialMatch = False
	While True
		$oPresentation = ObjGet("", $sCLSID_Presentation, $iCount + 1)
		If @error Then Return SetError(1, @error, 0)
		Switch $sMode
			Case "filename"
				If $oPresentation.Name = $sString Then Return $oPresentation
			Case "filepath"
				If $oPresentation.FullName = $sString Then Return $oPresentation
			Case "title"
				If $bPartialMatch Then
					If StringInStr($oPresentation.Windows(1).Caption, $sString) > 0 Then Return $oPresentation
				Else
					If $oPresentation.Windows(1).Caption = $sString Then Return $oPresentation
				EndIf
			Case Else
				Return SetError(2, 0, 0)
		EndSwitch
		$iCount += 1
	WEnd
	Return SetError(3, 0, 0)
EndFunc   ;==>_PPT_PresentationAttach

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationClose
; Description ...: Closes the specified presentation.
; Syntax.........: _PPT_PresentationClose($oPresentation[, $bSave = True])
; Parameters ....: $oPresentation - Presentation object.
;                  $bSave         - [optional] If True the presentation will be saved before closing (default = True).
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - Error occurred when saving the presentation. @extended is set to the COM error code returned by the Save method
;                  |3 - Error occurred when closing the presentation. @extended is set to the COM error code returned by the Close method
; Author ........: water
; Modified.......:
; Remarks .......: None
; Related .......: _PPT_PresentationAttach, _PPT_PresentationNew, _PPT_PresentationOpen
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationClose(ByRef $oPresentation, $bSave = Default)
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If $bSave = Default Then $bSave = True
	If $bSave And Not $oPresentation.Saved Then
		$oPresentation.Save()
		If @error Then Return SetError(2, @error, 0)
	EndIf
	$oPresentation.Close()
	If @error Then Return SetError(3, @error, 0)
	$oPresentation = 0
	Return 1
EndFunc   ;==>_PPT_PresentationClose

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationExport
; Description ...: Export one/multiple/all slides of a presentation as PDF or XPS.
; Syntax.........: _PPT_PresentationExport($oPresentation, $sPath[, $sSlide = Default[, $iFixedFormatType = $ppFixedFormatTypePDF[, $iOutputType = $ppFixedFormatTypePDF[, $bUseISO19005 = True)]]]])
; Parameters ....: $oPresentation    - Presentation object.
;                  $sPath            - Path/name of the exported file.
;                  $sSlide           - [optional] A string with the index number of the starting (and ending) slide to be exported (separated by a hyphen) (default = Keyword Default = export all slides in the presentation).
;                  $iFixedFormatType - [optional] The format to which the slides should be exported. Can be any value of the PpFixedFormatType enumeration (default = $ppFixedFormatTypePDF).
;                  $iOutputType      - [optional] The type of output. Can be any value of the PpPrintOutputType enumeration (default = $ppPrintOutputSlides).
;                  $bUseISO19005     - [optional] Whether the resulting document is compliant with ISO 19005-1 (PDF/A) (default = True).
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $sSlide is an object but not a SlideRange object
;                  |3 - $sPath is empty
;                  |4 - Error occurred when exporting the presentation. @extended is set to the COM error code returned by the ExportAsFixedFormat method
;                  |5 - $sSlide is invalid. Has to be "StartingSlide-EndingSlide"
; Author ........: water
; Modified.......:
; Remarks .......: Method ExportAsFixedFormat only supports a single range of consecutive slides.
;                  To export a single slide simply pass the slide number as $sSlide.
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationExport(ByRef $oPresentation, $sPath, $sSlide = Default, $iFixedFormatType = Default, $iOutputType = Default, $bUseISO19005 = Default)
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	Local $oPrintRange
	If $sSlide = Default Then
		$oPrintRange = $oPresentation.PrintOptions.Ranges.Add(1, $oPresentation.Slides.Count)
	Else
		Local $aSlides = StringSplit($sSlide, "-", $STR_NOCOUNT)
		Local $iValues = UBound($aSlides)
		If $iValues < 1 Or $iValues > 2 Then
			Return SetError(5, 0, 0)
		ElseIf $iValues = 1 Then
			ReDim $aSlides[2]
			$aSlides[1] = $aSlides[0]
		EndIf
		If Number($aSlides[0]) = 0 Or Number($aSlides[1]) = 0 Then Return SetError(5, 0, 0)
		$oPrintRange = $oPresentation.PrintOptions.Ranges.Add($aSlides[0], $aSlides[1])
	EndIf
	If $sPath = "" Then Return SetError(3, 0, 0)
	If $iFixedFormatType = Default Then $iFixedFormatType = $ppFixedFormatTypePDF
	If $iOutputType = Default Then $iOutputType = $ppPrintOutputSlides
	If $bUseISO19005 = Default Then $bUseISO19005 = True
	$oPresentation.ExportAsFixedFormat($sPath, $iFixedFormatType, Default, Default, Default, $iOutputType, Default, $oPrintRange, $ppPrintSlideRange, Default, Default, Default, Default, Default, $bUseISO19005, Default)
	If @error Then Return SetError(4, @error, 0)
	Return 1
EndFunc   ;==>_PPT_PresentationExport

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationExportGraphic
; Description ...: Export one/multiple/all slides of a presentation in a graphic format.
; Syntax.........: _PPT_PresentationExportGraphic($oPresentation[, $sPath = @ScriptDir[, $vSlides = Default[, $sFilter = "JPG"[, $iScaleWidth = Default[, $iScaleHeight = Default[, $bKeepRatio = True]]]]]])
; Parameters ....: $oPresentation - Presentation object.
;                  $sPath         - [optional] Directory where the graphics should be stored. The graphics are named Slide<n>. n is the slide number (default = keyword Default = @ScriptDir).
;                                              Specify the path without trailing backslash. If the directory does not exist, it will be created.
;                                              Existing files with the same name will be overwritten.
;                  $vSlides       - [optional] Slide(s) to process. Please see _PPT_SlideRangeSet for possible values (default = 0 = all Slides).
;                  $sFilter       - [optional] The graphics format in which you want to export slides (default = JPG). See Remarks.
;                  $iScaleWidth   - [optional] The width in pixels of an exported slide (default = keyword Default = do not change the width).
;                  $iScaleHeight  - [optional] The height in pixels of an exported slide (default = keyword Default = do not change the heigth).
;                  $bKeepRatio    - [optional] If set To True the width:height ratio is preserved if width or height is set (default = True)
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $vSlides is invalid. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |3 - $sPath didn't exist and returned an error when the function tried to create it.
;                  |4 - Error occurred when exporting the presentation. @extended is set to the COM error code returned by the ExportAsFixedFormat method
; Author ........: water
; Modified.......:
; Remarks .......: The specified graphics format must have an export filter registered in the Windows registry.
;                  You need to specify the registered extension (JPG, GIF etc.).
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationExportGraphic($oPresentation, $sPath = Default, $vSlides = Default, $sFilter = Default, $iScaleWidth = Default, $iScaleHeight = Default, $bKeepRatio = Default)
	Local $oSlideRange
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)
	If @error Then Return SetError(2, @error, 0)
	If $sPath = Default Then $sPath = @ScriptDir
	If Not FileExists($sPath) Then
		If DirCreate($sPath) = 0 Then Return SetError(3, 0, 0)
	EndIf
	If $sFilter = Default Then $sFilter = "JPG"
	If $bKeepRatio = Default Then $bKeepRatio = True
	If $bKeepRatio = True And ($iScaleWidth <> Default Or $iScaleHeight <> Default) Then
		; Local $iRatio = $oPresentation.Pagesetup.SlideWidth / $oPresentation.Pagesetup.SlideHeight
		If $iScaleWidth = Default Then
			$iScaleWidth = $oPresentation.Pagesetup.SlideWidth * ($iScaleHeight / $oPresentation.Pagesetup.SlideHeight)
		Else
			$iScaleHeight = $oPresentation.Pagesetup.SlideHeight * ($iScaleWidth / $oPresentation.Pagesetup.SlideWidth)
		EndIf
	EndIf
	For $oSlide In $oSlideRange
		$oSlide.Export($sPath & "\Slide" & $oSlide.SlideIndex & "." & $sFilter, $sFilter, $iScaleWidth, $iScaleHeight)
		If @error Then Return SetError(4, @error, 0)
	Next
	Return 1
EndFunc   ;==>_PPT_PresentationExportGraphic

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationList
; Description ...: Returns a list of currently open presentations.
; Syntax.........: _PPT_PresentationList($oPPT)
; Parameters ....: $oPPT - PowerPoint application object to retrieve the list of presentations from
; Return values .: Success - a two-dimensional zero based array with the following information:
;                  |0 - Object of the presentation
;                  |1 - Name of the presentation/file
;                  |2 - Complete path to the presentation/file
;                  |3 - Title of the presentation derived from the first window.caption of the presentation
;                  Failure - 0 and sets @error.
;                  |1 - $oPPT is not an object or not an application object
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......: _PPT_SlideList, _PPT_ShapeList
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationList($oPPT)
	Local $aPresentations[1][4], $iIndex = 0
	If IsObj($oPPT) = 0 Or ObjName($oPPT, 1) <> "_Application" Then Return SetError(1, 0, 0)
	Local $iTemp = $oPPT.Presentations.Count
	ReDim $aPresentations[$iTemp][4]
	For $iIndex = 0 To $iTemp - 1
		$aPresentations[$iIndex][0] = $oPPT.Presentations($iIndex + 1)
		$aPresentations[$iIndex][1] = $oPPT.Presentations($iIndex + 1).Name
		$aPresentations[$iIndex][2] = $oPPT.Presentations($iIndex + 1).Path & "\" & $aPresentations[$iIndex][1]
		$aPresentations[$iIndex][3] = $oPPT.Presentations($iIndex + 1).Windows(1).Caption
	Next
	Return $aPresentations
EndFunc   ;==>_PPT_PresentationList

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationNew
; Description ...: Creates a new presentation.
; Syntax.........: _PPT_PresentationNew($oPPT[, $sTemplate = "", [, $bVisible = True]])
; Parameters ....: $oPPT      - PowerPoint application object where you want to create the new presentation.
;                  $sTemplate - [optional] path to a template file (extension: pot, potx) or an existing presentation to be applied to the presentation (default = "" = no template).
;                  $bVisible  - [optional] True specifies that the presentation window will be visible (default = True).
; Return values .: Success - the presentation object.
;                  Failure - 0 and sets @error.
;                  |1 - $oPPT is not an object or not an application object
;                  |2 - Error returned by method Presentations.Add. @extended is set to the COM error code
;                  |3 - Specified template file does not exist
;                  |4 - Error returned by method ApplyTemplate. @extended is set to the COM error code
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......: _PPT_PresentationAttach, _PPT_PresentationClose, _PPT_PresentationOpen
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationNew($oPPT, $sTemplate = Default, $bVisible = Default)
	If Not IsObj($oPPT) Or ObjName($oPPT, 1) <> "_Application" Then Return SetError(1, 0, 0)
	If $sTemplate = Default Then $sTemplate = ""
	If $bVisible = Default Then $bVisible = True
	If $sTemplate <> "" And FileExists($sTemplate) = 0 Then Return SetError(3, 0, 0)
	Local $oPresentation = $oPPT.Presentations.Add($bVisible)
	If @error Then Return SetError(2, @error, 0)
	If $sTemplate <> "" Then
		$oPresentation.ApplyTemplate($sTemplate)
		If @error Then Return SetError(4, @error, 0)
	EndIf
	Return $oPresentation
EndFunc   ;==>_PPT_PresentationNew

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationOpen
; Description ...: Opens an existing presentation.
; Syntax.........: _PPT_PresentationOpen($oPPT, $sFilePath[, $bReadOnly = False[, $bVisible = True]])
; Parameters ....: $oPPT      - PowerPoint application object where you want to open the presentation
;                  $sFilePath - Path and filename of the file to be opened.
;                  $bReadOnly - [optional] True opens the presentation as read-only (default = False).
;                  $bVisible  - [optional] True specifies that the presentation window will be visible (default = True).
; Return values .: Success - a presentation object. @extended is set to 1 if $bReadOnly = False but read-write access could not be granted. Please see the Remarks section for details.
;                  Failure - 0 and sets @error.
;                  |1 - $oPPT is not an object or not an application object
;                  |2 - Specified $sFilePath does not exist
;                  |3 - Unable to open $sFilePath. @extended is set to the COM error code returned by the Open method
; Author ........: water
; Modified.......:
; Remarks .......: When you set $bReadOnly = False but the presentation can't be opened read-write @extended is set to 1.
;                  The presentation was opened read-only because it has already been opened by another user/task or the file is set to read-only by the filesystem.
;                  If you modify the presentation you need to use _PPT_PresentationSaveAs() to save it to another location or with another name.
;                  +
;                  The PowerPoint object model does not allow to pass read/write passwords to open a protected presentation.
; Related .......: _PPT_PresentationAttach, _PPT_PresentationClose, _PPT_PresentationNew
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationOpen($oPPT, $sFilePath, $bReadOnly = Default, $bVisible = Default)
	If Not IsObj($oPPT) Or ObjName($oPPT, 1) <> "_Application" Then Return SetError(1, @error, 0)
	If Not FileExists($sFilePath) Then Return SetError(2, 0, 0)
	If $bReadOnly = Default Then $bReadOnly = False
	If $bVisible = Default Then $bVisible = True
	Local $oPresentation = $oPPT.Presentations.Open($sFilePath, $bReadOnly, Default, $bVisible)
	If @error Then Return SetError(3, @error, 0)
	; If a read-write presentation was opened read-only then return an error
	If $bReadOnly = False And $oPresentation.Readonly = True Then Return SetError(0, 1, $oPresentation)
	Return $oPresentation
EndFunc   ;==>_PPT_PresentationOpen

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationPrint
; Description ...: Print one/multiple/all slides of a presentation.
; Syntax.........: _PPT_PresentationPrint($oPresentation[, $iStartingSlide = 1[, $iEndingSlide = Default[, $iCopies = 1[, $sPrinter = Default]]]])
; Parameters ....: $oPresentation    - Presentation object.
;                  $iStartingSlide   - [optional] Number of the first slide to be printed (default = 1).
;                  $iEndingSlide     - [optional] Number of the last slide to be printed (default = keyword Default = last slide of the presentation).
;                  $iCopies          - [optional] Number of copies to be printer (default = 1).
;                  $sPrinter         - [optional] Sets the name of the printer (default = keyword Default = Active printer)
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $sSlide is an object but not a SlideRange object
;                  |3 - Error occurred when printing the presentation. @extended is set to the COM error code returned by the PrintOut method
;                  |4 - Error occurred when creating the WScript.Network object. @extended is set to the COM error code
;                  |5 - Error occurred when setting the default printer. @extended is set to the COM error code returned by the SetDefaultPrinter method
;                  |6 - Error occurred when re-setting the default printer. @extended is set to the COM error code returned by the SetDefaultPrinter method
;                  |7 - Error occurred when retrieving current default printer. @extended is set to the COM error code returned by the ActivePrinter property
;                  |8 - Error occurred when setting printer connection. @extended is set to the COM error code returned by the AddWindowsPrinterConnection method
; Author ........: water
; Modified.......:
; Remarks .......: Method PrintOut only supports a single range of consecutive slides.
;                  To print a single slide set parameters $iStartingSlide and $iEndingSlide to the same number.
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationPrint($oPresentation, $iStartingSlide = Default, $iEndingSlide = Default, $iCopies = Default, $sPrinter = Default)
	Local $bPrinterChanged
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If $iStartingSlide = Default Then $iStartingSlide = 1
	If $iEndingSlide = Default Then $iEndingSlide = $oPresentation.Slides.Count
	If $iCopies = Default Then $iCopies = 1
	If $sPrinter <> Default Then
		Local $sActivePrinter = $oPresentation.Parent.Parent.ActivePrinter
		If @error Then Return SetError(7, @error, 0)
		If $sActivePrinter <> $sPrinter Then
			Local $oWSHNetwork = ObjCreate("WScript.Network")
			If @error Then Return SetError(4, @error, 0)
			$oWSHNetwork.AddWindowsPrinterConnection($sPrinter)
			If @error Then Return SetError(8, @error, 0)
			$oWSHNetwork.SetDefaultPrinter($sPrinter)
			If @error Then Return SetError(5, @error, 0)
			$bPrinterChanged = True
		EndIf
	EndIf
	If Number($iStartingSlide) = 0 Or Number($iEndingSlide) = 0 Then Return SetError(2, 0, 0)
	$oPresentation.PrintOut($iStartingSlide, $iEndingSlide, Default, $iCopies)
	If @error Then Return SetError(3, @error, 0)
	If $bPrinterChanged Then
		$oWSHNetwork.SetDefaultPrinter($sActivePrinter)
		If @error Then Return SetError(6, @error, 0)
	EndIf
	Return 1
EndFunc   ;==>_PPT_PresentationPrint

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationSave
; Description ...: Saves a presentation.
; Syntax.........: _PPT_PresentationSave($oPresentation)
; Parameters ....: $oPresentation - Object of the presentation to save.
; Return values .: Success - 1 and sets @extended.
;                  |0 - File has not been saved because it has not been changed since the last save or file open
;                  |1 - File has been saved because it has been changed since the last save or file open
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - Error occurred when saving the presentation. @extended is set to the COM error code
; Author ........: water
; Modified.......:
; Remarks .......: A newly created presentation has to be saved using _PPT_PresentationSaveAs before
; Related .......: _PPT_PresentationSaveAs
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationSave($oPresentation)
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If Not $oPresentation.Saved Then
		$oPresentation.Save()
		If @error Then Return SetError(2, @error, 0)
		Return SetError(0, 1, 1)
	EndIf
	Return 1
EndFunc   ;==>_PPT_PresentationSave

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_PresentationSaveAs
; Description ...: Saves a presentation with a new filename and/or type.
; Syntax.........: _PPT_PresentationSaveAs($oPresentation, $sFilePath[, $iFormat = $ppSaveAsDefault[, $bOverWrite = False]])
; Parameters ....: $oPresentation - Presentation object to be saved.
;                  $sFilePath     - FilePath of the file to be written to (no extension).
;                  $iFormat       - [optional] PowerPoint writeable filetype. Can be any value of the PpSaveAsFileType enumeration.
;                                   (default = keyword Default = $ppSaveAsDefault (means: .ppt for PowerPoint < 2007, .pptx for PowerPoint 2007 and later).
;                  $bOverWrite    - [optional] True overwrites an already existing file (default = False).
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $iFormat is not a number
;                  |3 - File exists, overwrite flag not set
;                  |4 - File exists but could not be deleted
;                  |5 - Error occurred when saving the presentation. @extended is set to the COM error code returned by the SaveAs method
; Author ........: water
; Modified.......:
; Remarks .......: $sFilename is the filename without extension. "test" is correct, "test.pps" is wrong.
; Related .......: _PPT_PresentationSave
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_PresentationSaveAs($oPresentation, $sFilePath, $iFormat = Default, $bOverWrite = Default)
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If $iFormat = Default Then
		$iFormat = $ppSaveAsDefault
	Else
		If Not IsNumber($iFormat) Then Return SetError(2, 0, 0)
	EndIf
	If $bOverWrite = Default Then $bOverWrite = False
	If FileExists($sFilePath) Then
		If Not $bOverWrite Then Return SetError(3, 0, 0)
		Local $iResult = FileDelete($sFilePath)
		If $iResult = 0 Then Return SetError(4, 0, 0)
	EndIf
	$oPresentation.SaveAs($sFilePath, $iFormat)
	If @error Then Return SetError(5, @error, 0)
	Return 1
EndFunc   ;==>_PPT_PresentationSaveAs

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_ShapeAdd
; Description ...: Add a Shape to a single or multiple Slides.
; Syntax.........: _PPT_ShapeAdd($oPresentation, $vSlides, $iShapeType, $iX1, $iY1, $iX2, $iY2[, $vP1 = Default[, $vP2 = Default[, $vP3 = Default[, $vP4 = Default[, $vP5 = Default]]]]])
; Parameters ....: $oPresentation - Object of the presentation to process.
;                  $vSlides       - Slides to process. See Remarks.
;                  $iShapeType    - Constant from the MsoShapeType enumeration representing the type of Shape to add.
;                                   Currently the function supports the following ShapeTypes: MsoChart, MsoLine, MsoPicture, MsoTable, $MsoTextBox.
;                  $iX1           - The position (in Points) of the left edge of the Shape relative to the left edge of the Slide. See Remarks.
;                  $iY1           - The position (in Points) of the top edge of the Shape relative to the left edge of the Slide. See Remarks.
;                  $iX2           - The Width (in Points) of the Shape. See Remarks.
;                  $iY2           - The Height (in Points) of the Shape. See Remarks.
;                  $vP1           - [optional] Additional parameter for the Addxxxx method of the Shapes collection. See Remarks.
;                  $vP2           - [optional] Additional parameter for the Addxxxx method of the Shapes collection. See Remarks.
;                  $vP3           - [optional] Additional parameter for the Addxxxx method of the Shapes collection. See Remarks.
;                  $vP4           - [optional] Additional parameter for the Addxxxx method of the Shapes collection. See Remarks.
;                  $vP5           - [optional] Additional parameter for the Addxxxx method of the Shapes collection. See Remarks.
; Return values .: Success - zero-based one dimensional array with a Shape object for each Slide in the SlideRange
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - Error occurred creating the SlideRange from $vSlides. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |3 - Error creating the Shape. @extended is set to the COM error code
;                  |4 - $iShapeType is invalid
; Author ........: water
; Modified ......:
; Remarks .......: In the following table you find the parameters to use based on the ShapeType you want to create:
;                  SHAPETYPE      $iX1    $iY1    $iX2   $iY2    $vP1         $vP2        $vP3              $vP4
;                  $MsoChart      Left    Top     Width  Height  Style        Type        NewLayout
;                  $MsoLine       BeginX  BeginY  EndX   EndY
;                  $MsoPicture    Left    Top     Width  Height  FileName     LinkToFile  SaveWithDocument  Compress
;                  $MsoTable      Left    Top     Width  Height  NumRows      NumColumns
;                  $MsoTextBox    Left    Top     Width  Height  Orientation
;+
;                  When you pass Default for $iX1, $iY2, $iX2 or $iY2 then PowerPoint decides where to place the Shape.
;+
;                  For details please check the MS documentation:
;                  $MsoChart      https://docs.microsoft.com/en-us/office/vba/api/powerpoint.shapes.addchart2
;                  $MsoLine       https://docs.microsoft.com/en-us/office/vba/api/powerpoint.shapes.addline
;                  $MsoPicture    https://docs.microsoft.com/en-us/office/vba/api/powerpoint.shapes.addpicture2
;                  $MsoTable      https://docs.microsoft.com/en-us/office/vba/api/powerpoint.shapes.addtable
;                  $MsoTextBox    https://docs.microsoft.com/en-us/office/vba/api/powerpoint.shapes.addtextbox
; Related .......: _PPT_ShapeDelete
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_ShapeAdd($oPresentation, $vSlides, $iShapeType, $iX1, $iY1, $iX2, $iY2, $vP1 = Default, $vP2 = Default, $vP3 = Default, $vP4 = Default, $vP5 = Default)
	#forceref $vP1, $vP2, $vP3, $vP4, $vP5
	Local $oSlideRange, $oSlide, $iSlides
	If $vSlides = Default Then $vSlides = 0
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)
	If @error Then Return SetError(2, @error, 0)
	$iSlides = $oSlideRange.Count
	Local $aSlides[$iSlides]
	For $i = 1 To $iSlides
		$oSlide = $oSlideRange.Item($i)
		Switch $iShapeType
			Case $MsoChart
				$aSlides[$i - 1] = $oSlide.Shapes.AddChart2($vP1, $vP2, $iX1, $iY1, $iX2, $iY2, $vP3)     ; Style, Type, Left, Top, Width, Height, NewLayout
				If @error Then Return SetError(3, @error, 0)
			Case $MsoLine
				$aSlides[$i - 1] = $oSlide.Shapes.AddLine($iX1, $iY1, $iX2, $iY2)     ; BeginX, BeginY, EndX, EndY
				If @error Then Return SetError(3, @error, 0)
			Case $MsoPicture
				$aSlides[$i - 1] = $oSlide.Shapes.AddPicture2($vP1, $vP2, $vP3, $iX1, $iY1, $iX2, $iY2, $vP4)     ; FileName, LinkToFile, SaveWithDocument, Left, Top, Width, Height, compress
				If @error Then Return SetError(3, @error, 0)
			Case $MsoTable
				$aSlides[$i - 1] = $oSlide.Shapes.AddTable($vP1, $vP2, $iX1, $iY1, $iX2, $iY2)     ; NumRows, NumColumns, Left, Top, Width, Height
				If @error Then Return SetError(3, @error, 0)
			Case $MsoTextBox
				$aSlides[$i - 1] = $oSlide.Shapes.AddTextBox($vP1, $iX1, $iY1, $iX2, $iY2)     ; Orientation, Left, Top, Width, Height
				If @error Then Return SetError(3, @error, 0)
			Case Else
				Return SetError(4, 0, 0)
		EndSwitch
	Next
	Return $aSlides
EndFunc   ;==>_PPT_ShapeAdd

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_ShapeCopyMove
; Description ...: Copies or moves the specified Shape(s) of a single Slide to the specified Slide(s) in the same or a different Presentation.
; Syntax.........: _PPT_ShapeCopyMove($oSourcePresentation, $vSourceSlide, $vSourceShapes[, $oTargetPresentation = $oSourcePresentation[, $vTargetSlides = Default[, $iFunction = 1]]])
; Parameters ....: $oSourcePresentation - Presentation object of the source presentation.
;                  $vSourceSlide        - A SlideRange object as returned by _PPT_SlideRangeSet for a single source Slide. See _PPT_SlideRangeSet and Remarks for details.
;                  $vSourceShapes       - A ShapeRange object as returned by _PPT_ShapeRangeSet of the Shapes to process.  See _PPT_ShapeRangeSet and Remarks for details.
;                  $oTargetPresentation - [optional] Presentation object of the target Presentation where Shapes should be copied/moved to (default = keyword Default = $oSourcePresentation).
;                  $vTargetSlides       - [optional] A SlideRange object as returned by _PPT_SlideRangeSet for the target Slide(s) (default = Slide 1).
;                  $iFunction           - [optional] Specifies how to process the Shape(s):
;                  |1: Copy the specified Shape(s) to the target Presentation (default)
;                  |2: Move the specified Shape(s) to the target Presentation
;                  |3: Duplicate the specified Shape(s) on the source Slide. $oTargetPresentation and $vTargetSlides are ignored
;                  |4: Copy the specified Shape(s) to the clipboard. $oTargetPresentation and $vTargetSlides are ignored
;                  |5: Cut the specified Shape(s) from the source Presentation to the clipboard. $oTargetPresentation and $vTargetSlides are ignored
; Return values .: Success - 1
;                  Failure - 0 and sets @error.
;                  |1 - $oSourcePresentation is not an object or not a presentation object
;                  |2 - $oTargetPresentation is not an object or not a presentation object
;                  |3 - Error occurred creating the SlideRange from $vSourceSlides. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |4 - Error occurred creating the ShapeRange from $vSourceShapes. @extended is set to the error code returned by _PPT_ShapeRangeSet
;                  |5 - Error occurred creating the SlideRange from $vTargetSlides. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |6 - $iFunction < 1 or $iFunction > 5
;                  |7 - Error occurred when copying the Shape(s). @extended is set to the COM error code returned by the Copy method
;                  |8 - Error occurred when cutting the Shape(s). @extended is set to the COM error code returned by the Cut method
;                  |9 - Error occurred when inserting Shape(s). @extended is set to the COM error code returned by the Paste method
;                  |10 - Error occurred when duplicating Shape(s). @extended is set to the COM error code returned by the Duplicate method
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_ShapeCopyMove($oSourcePresentation, $vSourceSlides, $vSourceShapes, $oTargetPresentation = Default, $vTargetSlides = Default, $iFunction = Default)
	Local $oSourceSlides, $oSourceShapes, $oTargetSlides
	If Not IsObj($oSourcePresentation) Or ObjName($oSourcePresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If $oTargetPresentation = Default Then $oTargetPresentation = $oSourcePresentation
	If Not IsObj($oTargetPresentation) Or ObjName($oTargetPresentation, 1) <> "_Presentation" Then Return SetError(2, 0, 0)
	;	Local $iSlideCount = $oTargetPresentation.Slides.Count
	If $vTargetSlides = Default Then $vTargetSlides = 1
	If $iFunction = Default Then $iFunction = 1
	If $iFunction < 1 Or $iFunction > 5 Then Return SetError(6, 0, 0)
	$oSourceSlides = _PPT_SlideRangeSet($oSourcePresentation, $vSourceSlides)
	If @error Then Return SetError(3, @error, 0)
	$oSourceShapes = _PPT_ShapeRangeSet($oSourcePresentation, $oSourceSlides, $vSourceShapes)
	If @error Then Return SetError(4, @error, 0)
	$oTargetSlides = _PPT_SlideRangeSet($oTargetPresentation, $vTargetSlides)
	If @error Then Return SetError(5, @error, 0)
	Switch $iFunction
		Case 1     ; Copy
			$oSourceShapes.Copy
			If @error Then Return SetError(7, @error, 0)
			For $oSlide In $oTargetSlides
				$oSlide.Shapes.Paste
				If @error Then Return SetError(9, @error, 0)
			Next
		Case 2     ; Move
			$oSourceShapes.Cut
			If @error Then Return SetError(8, @error, 0)
			For $oSlide In $oTargetSlides
				$oSlide.Shapes.Paste
			Next
			If @error Then Return SetError(9, @error, 0)
		Case 3     ; Duplicate
			$oSourceShapes.Duplicate
			If @error Then Return SetError(10, @error, 0)
		Case 4     ; Copy to the clipboard
			$oSourceShapes.Copy()
			If @error Then Return SetError(7, @error, 0)
		Case 5     ; Move to the clipboard
			$oSourceShapes.Cut()     ; Cut/paste to another presentation
			If @error Then Return SetError(8, @error, 0)
	EndSwitch
	Return 1
EndFunc   ;==>_PPT_ShapeCopyMove

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_ShapeDelete
; Description ...: Delete a Shape or Shapes from a single or multiple Slides.
; Syntax.........: _PPT_ShapeDelete($oPresentation[, $vSlides = 0[, $vShapes = 0]])
; Parameters ....: $oPresentation - Object of the presentation to process.
;                  $vSlides       - [optional] Slides to process. See Remarks (default = 0 = all Slides).
;                  $vShapes       - [optional] Shapes to process on each selected Slide. See Remarks (default = 0 = all Shapes).
; Return values .: Success - 1, @extended holds the number of deleted Shapes.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - Number of Slides is > 1 but $vShapes is not one of the allowed values: 0, >0 or a string
;                  |3 - Error occurred creating the SlideRange from $vSlides. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |4 - Error occurred creating the ShapeRange from $vShapes. @extended is set to the error code returned by _PPT_ShapeRangeSet
; Author ........: water
; Modified ......:
; Remarks .......: With this function you can delete a single Shape from multiple Slides or multiple Shapes from a single Slide.
;+
;                  Parameter $vSlides: See _PPT_SlideRangeSet for the format of this parameter
;                  Parameter $vShapes:
;                    If Number of Slides is = 1: All values for _PPT_ShapeRangeSet are valid. See _PPT_ShapeRangeSet for the format of this parameter
;                    If Number of Slides is > 1: only 0 (all Shapes), integer (ID of the Shape) or a string (Name of the Shape) are valid
;+
;                  While looping through the specified Slides all specified Shapes are deleted with a single command.
;                  Errors are ignored and processing continues with the next Slide.
; Related .......: _PPT_ShapeAdd
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_ShapeDelete($oPresentation, $vSlides = Default, $vShapes = Default)
	Local $oSlide, $oSlideRange, $oShape, $oShapeRange, $iDeleteCount = 0
	If $vSlides = Default Then $vSlides = 0
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)
	If @error Then Return SetError(3, @error, 0)
	If $vShapes = Default Then $vShapes = 0
	If $oSlideRange.Count > 1 And _
			Not ($vShapes = 0 Or $vShapes > 0 Or IsString($vShapes)) Then     ; Check allowed values: 0 (all Slides), integer (single Slide), String (SlideName)
		Return SetError(2, 0, 0)
	EndIf
	; Loop through the specified Slides
	For $oSlide In $oSlideRange
		; Get the ShapeRange for every specified Slide
		$oShapeRange = _PPT_ShapeRangeSet($oPresentation, $oSlide, $vShapes)
		If @error Then Return SetError(4, @error, 0)
		For $oShape In $oShapeRange
			$oShape.Delete
			If @error = 0 Then $iDeleteCount += 1
		Next
	Next
	Return SetError(0, $iDeleteCount, 1)
EndFunc   ;==>_PPT_ShapeDelete

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_ShapeList
; Description ...: Returns a list of shapes in a slide.
; Syntax.........: _PPT_ShapeList($oPresentation, $vSlide)
; Parameters ....: $oPresentation - Presentation object to be saved.
;                  $vSlide        - Slide for which to list the shapes. The SlideRange object will be created from this Slide.
;                                   Please see Remarks in function _PPT_SlideRangeSet.
; Return values .: Success - a two-dimensional zero based array with the following information:
;                  |0 - Object of the Shape
;                  |1 - Property ID: Number that identifies the shape
;                  |2 - Property Name: Name created by PowerPoint or the user
;                  |3 - Property Type: Type of the Shape
;                  |4 - Property Visible: visibility of the Shape
;                  |5 - Property ZOrderPosition: position of the shape in the z-order
;                  |6 - Property HasTextFrame: Returns whether the shape has a text frame and can therefore contain text
;                  |7 - Property HasTable: Returns whether the shape is a table
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - Error occurred creating the SlideRange from $vSlides. @extended is set to the error code returned by _PPT_SlideRangeSet
; Author ........: water
; Modified.......:
; Remarks .......: The function only processes the first Slide if parameter $vSlide returns more than one Slide
; Related .......: _PPT_PresentationList, _PPT_SlideList
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_ShapeList($oPresentation, $vSlide)
	Local $oShape, $oSlide, $oSlideRange, $iShapes
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlide)
	If @error Then Return SetError(2, @error, 0)
	$oSlide = $oSlideRange(1)
	$iShapes = $oSlide.Shapes.Count
	Local $aShapes[$iShapes][8]
	For $i = 1 To $iShapes
		$oShape = $oSlide.Shapes($i)
		$aShapes[$i - 1][0] = $oShape
		$aShapes[$i - 1][1] = $oShape.ID
		$aShapes[$i - 1][2] = $oShape.Name
		$aShapes[$i - 1][3] = $oShape.Type
		$aShapes[$i - 1][4] = $oShape.Visible
		$aShapes[$i - 1][5] = $oShape.ZOrderPosition
		$aShapes[$i - 1][6] = $oShape.HasTextFrame
		$aShapes[$i - 1][7] = $oShape.HasTable
	Next
	Return $aShapes
EndFunc   ;==>_PPT_ShapeList

; #FUNCTION#============================================================================================================
; Name...........: _PPT_ShapeRangeSet
; Description ...: Creates a ShapeRange object with one or multiple Shapes from one or multiple Slides.
; Syntax.........: _PPT_ShapeRangeSet($oPresentation[, $vSlides = 0[, $vShapes = 0]])
; Parameters ....: $oPresentation - Presentation object.
;                  $vSlides       - [optional] Slide or Slides to create the SlideRange object from (default = 0 = all Slides).
;                                   Please see Remarks in function _PPT_SlideRangeSet.
;                  $vShapes       - [optional] Shape or Shapes to create the ShapeRange object from (default = 0 = all Shapes). Please see Remarks.
; Return values .: Success - ShapeRange object.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $vSlides is an object but not a Slide or SlideRange object
;                  |3 - Error occurred creating the SlideRange from $vSlides. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |4 - Error occurred creating the SlideRange from $vSlides. @extended is set to the COM error code returned by the Range method
;                  |5 - $vShapes is an object but not a ShapeRange object
;                  |6 - Error occurred creating the ShapeRange from $vShapes. @extended is set to the COM error code returned by the Range method
;                  |7 - $vShapes is an object but not a Shape or ShapeRange object
;                  |8 - $vShapes has invalid format
; Author ........: water
; Modified ......:
; Remarks .......: The ShapeRange parameter $vShapes can be defined using one of the following formats:
;                  * collection of individual Shapes:
;                    - ShapeRange object
;                    - Index number of the first and last Shape separated by a hyphen e.g. 1-3 or 1-* (* represents the last Shape)
;                    - Index number: -1 (ShapeRange consists of all selected Shapes)
;                    - Zero based array holding the index numbers or names of the Shapes
;                  * collection of all Shapes:
;                    - Shapes object
;                    - Index number: 0 (ShapeRange consists of all Shapes)
;                  * single Shape:
;                    - Shape object
;                    - Index number
;                    - Shape name
;+
;                  Just as you can work with several slides at the same time in the user interface by selecting them and applying a command,
;                  you can work with several slides at the same time programmatically by constructing a SlideRange collection and applying properties
;                  or methods to it. And just as some commands in the user interface that work on single slides aren't valid when multiple slides are
;                  selected, some properties and methods that work on a Slide object or on a SlideRange collection that contains only one slide will
;                  fail if they're applied to a SlideRange collection that contains more than one slide.
;                  In general, if you can't do something manually when more than one slide is selected (such as return the individual shapes on one
;                  of the slides), you can't do it programmatically by using a SlideRange collection that contains more than one slide.
;                  See: https://docs.microsoft.com/en-us/office/vba/api/powerpoint.SlideRange
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_ShapeRangeSet($oPresentation, $vSlides = Default, $vShapes = Default)
	Local $oSlideRange, $oShapeRange
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	; SlideRange
	If $vSlides = Default Then $vSlides = 0
	$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)            ; Create a SlideRange object
	If @error Then Return SetError(3, @error, 0)
	; ShapeRange
	If $vShapes = Default Then $vShapes = 0
	Select
		Case IsInt($vShapes)
			If $vShapes = 0 Then                                           ; Return all shapes of the SlideRange
				$oShapeRange = $oSlideRange.Shapes.Range
				If @error Then Return SetError(6, @error, 0)
			ElseIf $vShapes = -1 Then                                      ; Return all selected shapes of the SlideRange
				$oShapeRange = $oPresentation.Application.ActiveWindow.Selection.ShapeRange
				If @error Then Return SetError(6, @error, 0)
			Else
				$oShapeRange = $oSlideRange.Shapes.Range($vShapes)         ; Shape spezified by Index
				If @error Then Return SetError(6, @error, 0)
			EndIf
		Case IsObj($vShapes)
			If ObjName($vShapes, 1) = "Shape" Then                         ; Create a ShapeRange object ob the passed Shape object
				$oShapeRange = $oSlideRange.Shapes.Range($vShapes.ZOrderPosition)
				If @error Then Return SetError(6, @error, 0)
			ElseIf ObjName($vShapes, 1) = "ShapeRange" Then                ; ShapeRange object has been passed
				$oShapeRange = $vShapes
			Else                                                           ; Invalid object
				Return SetError(7, 0, 0)
			EndIf
		Case Else                                                          ; Create ShapeRange object from array, index range or name
			Select
				Case IsString($vShapes) And StringInStr($vShapes, "-") > 0
					Local $aShapesTemp = StringSplit($vShapes, "-", $STR_NOCOUNT)
					If @error Then Return SetError(8, 0, 0)
					If UBound($aShapesTemp) < 1 Or UBound($aShapesTemp) > 2 Then Return SetError(8, 0, 0)
					If $aShapesTemp[0] = "" Then $aShapesTemp[0] = 1
					If $aShapesTemp[1] = "" Then $aShapesTemp[1] = "*"
					If $aShapesTemp[1] = "*" Then $aShapesTemp[1] = $oSlideRange.Shapes.Count
					If Not ($aShapesTemp[0] <= $aShapesTemp[1]) Then Return SetError(8, 0, 0)
					Local $aShapes[$aShapesTemp[1] - $aShapesTemp[0] + 1]
					For $i = 0 To UBound($aShapes) - 1
						$aShapes[$i] = $aShapesTemp[0] + $i
					Next
					$oShapeRange = $oSlideRange.Shapes.Range($aShapes)
					If @error Then Return SetError(6, @error, 0)
				Case IsString($vShapes) Or IsArray($vShapes)
					$oShapeRange = $oSlideRange.Shapes.Range($vShapes)
					If @error Then Return SetError(6, @error, 0)
				Case Else
					Return SetError(8, 0, 0)
			EndSelect
	EndSelect
	Return $oShapeRange
EndFunc   ;==>_PPT_ShapeRangeSet

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_SlideAdd
; Description ...: Add one or multiple slides to a presentation.
; Syntax.........: _PPT_SlideAdd($oPresentation[, $vIndex = Default[, $iSlides = 1[, $sLayout = Default]]])
; Parameters ....: $oPresentation - Presentation object.
;                  $vIndex        - [optional] The name or index where the slide is to be added (default = keyword Default = add the slide at the end of the presentation).
;                  $iSlides       - [optional] Number of slides to be added (default = 1).
;                  $vLayout       - [optional] The layout of the slide (default = keyword Default = Layout of the preceding/next slide).
;                                   Can be any of the PpSlideLayout enumeration or a layout object.
; Return values .: Success - the slide object of the first slide added.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $vIndex is a number and < 1 or > current number of slides + 1
;                  |3 - Error occurred when retrieving layout of a slide. @extended is set to the COM error code
;                  |4 - Error occurred when adding a slide. @extended is set to the COM error code returned by the Add/AddSlide method
;                  |5 - $iSlides is not a number or < 1
; Author ........: water
; Modified.......:
; Remarks .......: If $vLayout ist set to Default then the layout of the slide with index $vIndex is used.
;                  If there is no such slide then the layout of the preceding slide is being used.
;                  If there is no such slide then layout $ppLayoutBlank is being used.
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_SlideAdd($oPresentation, $vIndex = Default, $iSlides = Default, $vLayout = Default)
	Local $oSlide, $oSlideReturn, $iSlideCount, $oSlideLayout, $iDiff = 1
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If $iSlides = Default Then $iSlides = 1
	If Not IsNumber($iSlides) Or $iSlides < 1 Then Return SetError(5, 0, 0)
	$iSlideCount = $oPresentation.Slides.Count()
	If $iSlideCount = 0 Then $iDiff = 2
	If $vIndex = Default Then
		$vIndex = $iSlideCount + 1
	Else
		If IsNumber($vIndex) And ($vIndex < 1 Or $vIndex > $iSlideCount + 1) Then Return SetError(2, 0, 0)
	EndIf
	$oSlideLayout = $vLayout
	For $i = 1 To $iSlides
		If $vLayout = Default Then
			$iSlideCount = $oPresentation.Slides.Count()
			Switch $iSlideCount
				Case 0
					$oSlideLayout = $oPresentation.SlideMaster.CustomLayouts.Item(1)     ; Titlelayout from SlideMaster
				Case 1
					$oSlideLayout = $oPresentation.SlideMaster.CustomLayouts.Item(2)     ; Layout of first content slide
				Case Else
					$oSlideLayout = $oPresentation.Slides($vIndex - $iDiff + $i).CustomLayout     ; Get the Layout of the slide where to insert the new slide
			EndSwitch
			If @error Then Return SetError(4, @error, 0)
		EndIf
		If IsObj($oSlideLayout) Then
			$oSlide = $oPresentation.Slides.AddSlide($vIndex - 1 + $i, $oSlideLayout)
		Else
			$oSlide = $oPresentation.Slides.Add($vIndex - 1 + $i, $oSlideLayout)
		EndIf
		If @error Then Return SetError(4, @error, 0)
		If $i = 1 Then $oSlideReturn = $oSlide
	Next
	Return $oSlideReturn
EndFunc   ;==>_PPT_SlideAdd

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_SlideCopyMove
; Description ...: Copies or moves the specified slide(s) before or after a specified slide in the same or a different presentation. Duplicates slides in the same presentation.
; Syntax.........: _PPT_SlideCopyMove($oSourcePresentation, $vSourceSlide[, $oTargetPresentation = $oSourcePresentation[, $vTargetSlide = Default[, $iFunction = 1]]])
; Parameters ....: $oSourcePresentation - Presentation object of the source presentation.
;                  $vSourceSlide        - Slide or Slides to process. Format of this parameter is described in _PPT_SlideRangeSet.
;                  $oTargetPresentation - [optional] Presentation object of the target presentation where slides should be copied/moved to (default = keyword Default = $oSourcePresentation).
;                  $vTargetSlide        - [optional] Index or name of the slide that the slides on the Clipboard are to be pasted before (default = keyword Default = copy after the last slide in the presentation).
;                  $iFunction           - [optional] Specifies how to process the slides:
;                  |1: Copy the specified slide(s) to the target presentation (default)
;                  |2: Move the specified slide(s) to the target presentation
;                  |3: Duplicate the specified slide(s) in the source presentation
;                  |4: Copy the specified slide(s) from the source presentation to the clipboard
;                  |5: Cut the specified slide(s) from the source presentation to the clipboard
; Return values .: Success - SlideRange object that represents the pasted objects for $iFunction = 1, 2 to another presentation and 3.
;                  |         1 for $iFunction = 2 when moving within the same presentation, $iFunction = 4 and 5.
;                  Failure - 0 and sets @error.
;                  |1 - $oSourcePresentation is not an object or not a presentation object
;                  |2 - $oTargetPresentation is not an object or not a presentation object
;                  |3 - Error occurred when copying the slide(s). @extended is set to the COM error code returned by the Copy method
;                  |4 - Error occurred when cutting the slide(s). @extended is set to the COM error code returned by the Cut method
;                  |5 - Error occurred when moving the slide(s). @extended is set to the COM error code returned by the MoveTo method
;                  |6 - Error occurred when inserting slide(s). @extended is set to the COM error code returned by the Paste method
;                  |7 - Error occurred when duplicating slide(s). @extended is set to the COM error code returned by the Duplicate method
;                  |8 - $iFunction < 1 or $iFunction > 5
;                  |9 - You can't duplicate slides to another presentation. $oTargetPresentation = $oSourcePresentation is needed
;                  |10 - $vTargetSlide is a number and < 1 or > current number of slides + 1
;                  |11 - $vSourceSlide is an object but not a SlideRange object
;                  |12 - Error occurred creating the SlideRange for the source presentation. @extended is set to the COM error code returned by the Duplicate method
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_SlideCopyMove($oSourcePresentation, $vSourceSlide, $oTargetPresentation = Default, $vTargetSlide = Default, $iFunction = Default)
	Local $oSourceSlideRange, $oTargetSlideRange
	If Not IsObj($oSourcePresentation) Or ObjName($oSourcePresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If $oTargetPresentation = Default Then $oTargetPresentation = $oSourcePresentation
	If Not IsObj($oTargetPresentation) Or ObjName($oTargetPresentation, 1) <> "_Presentation" Then Return SetError(2, 0, 0)
	Local $iSlideCount = $oTargetPresentation.Slides.Count
	If $vTargetSlide = Default Then $vTargetSlide = $iSlideCount + 1
	If IsNumber($vTargetSlide) And ($vTargetSlide < 1 Or $vTargetSlide > $iSlideCount + 1) Then Return SetError(10, 0, 0)
	If $iFunction = Default Then $iFunction = 1
	If $iFunction < 1 Or $iFunction > 5 Then Return SetError(8, 0, 0)
	If $iFunction = 3 And $oSourcePresentation <> $oTargetPresentation Then Return SetError(9, 0, 0)
	$oSourceSlideRange = _PPT_SlideRangeSet($oSourcePresentation, $vSourceSlide)
	If @error Then Return SetError(8, 0, 0)
	Switch $iFunction
		Case 1     ; Copy to target presentation
			$oSourceSlideRange.Copy()
			If @error Then Return SetError(3, @error, 0)
			$oTargetSlideRange = $oTargetPresentation.Slides.Paste($vTargetSlide)
			If @error Then Return SetError(6, @error, 0)
			Return $oTargetSlideRange
		Case 2     ; Move to target presentation
			If $oSourcePresentation = $oTargetPresentation Then     ; Move within the same presentation
				$oSourceSlideRange.MoveTo($vTargetSlide)
				If @error Then Return SetError(5, @error, 0)
				Return 1
			Else
				$oSourceSlideRange.Cut()     ; Cut/paste to another presentation
				If @error Then Return SetError(4, @error, 0)
				$oTargetSlideRange = $oTargetPresentation.Slides.Paste($vTargetSlide)
				If @error Then Return SetError(6, @error, 0)
				Return $oTargetSlideRange
			EndIf
		Case 3     ; Duplicate
			$oTargetSlideRange = $oSourceSlideRange.Duplicate()
			If @error Then Return SetError(7, @error, 0)
			Return $oTargetSlideRange
		Case 4     ; Copy to the clipboard
			$oSourceSlideRange.Copy()
			If @error Then Return SetError(3, @error, 0)
			Return 1
		Case 5     ; Move to the clipboard
			$oSourceSlideRange.Cut()     ; Cut/paste to another presentation
			If @error Then Return SetError(4, @error, 0)
			Return 1
	EndSwitch
EndFunc   ;==>_PPT_SlideCopyMove

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_SlideDelete
; Description ...: Delete one or multiple slides from a presentation.
; Syntax.........: _PPT_SlideDelete($oPresentation, $vSlide)
; Parameters ....: $oPresentation - Presentation object.
;                  $vSlide        - Slide or Slides to delete. Format of this parameter is described in _PPT_SlideRangeSet.
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - Error deleting slides. @extended is set to the COM error code returned by the Delete method
;                  |3 - Error occurred creating the SlideRange from $vSlide. @extended is set to the error code returned by _PPT_SlideRangeSet
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_SlideDelete($oPresentation, $vSlide)
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	Local $oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlide)
	If @error Then Return SetError(3, @error, 0)
	$oSlideRange.Delete
	If @error Then Return SetError(2, @error, 0)
	Return 1
EndFunc   ;==>_PPT_SlideDelete

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_SlideList
; Description ...: Returns a list of Slides in a Presentation.
; Syntax.........: _PPT_SlideList($oPresentation)
; Parameters ....: $oPresentation - Presentation object.
; Return values .: Success - a two-dimensional zero based array with the following information:
;                  |0 - Object of the Slide
;                  |1 - Property SlideID: a unique ID number
;                  |2 - Property SlideIndex: index number of the  slide within the Slides collection
;                  |3 - Property SlideNumber: slide number
;                  |4 - Property Name: name of the slide (format: Slide n when created by PowerPoint or user
;                  |5 - Property HasNotePages: Indicates whether the Slide has media that resides on a notes page
;                  |6 - Property Hidden: Determines whether the specified slide is hidden during a slide show
;                  Failure - 0 and sets @error.
;                  |1 - $oPPT is not an object or not an application object
; Author ........: water
; Modified.......:
; Remarks .......: For SlideID, SlideIndex and SlideNumber values please see:
;                  https://docs.microsoft.com/en-us/office/vba/api/powerpoint.slide.slideid
;                  https://docs.microsoft.com/en-us/office/vba/api/powerpoint.slide.slideindex
;                  https://docs.microsoft.com/en-us/office/vba/api/powerpoint.slide.slidenumber
; Related .......: _PPT_PresentationList, _PPT_ShapeList
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_SlideList($oPresentation)
	Local $oSlide, $iSlides
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	$iSlides = $oPresentation.Slides.Count
	Local $aSlides[$iSlides][7]
	For $i = 1 To $iSlides
		$oSlide = $oPresentation.Slides($i)
		$aSlides[$i - 1][0] = $oSlide
		$aSlides[$i - 1][1] = $oSlide.SlideID
		$aSlides[$i - 1][2] = $oSlide.SlideIndex
		$aSlides[$i - 1][3] = $oSlide.SlideNumber
		$aSlides[$i - 1][4] = $oSlide.Name
		$aSlides[$i - 1][5] = $oSlide.HasNotesPage
		$aSlides[$i - 1][6] = $oSlide.SlideShowTransition.Hidden
	Next
	Return $aSlides
EndFunc   ;==>_PPT_SlideList

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_SlideRangeSet
; Description ...: Creates a SlideRange object with one or multiple slides.
; Syntax.........: _PPT_SlideRangeSet($oPresentation[, $vSlides = 0])
; Parameters ....: $oPresentation - Presentation object.
;                  $vSlides       - [optional] Slide or Slides to create the SlideRange object from (default = 0 = all Slides). Please see Remarks.
; Return values .: Success - SlideRange object.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $vSlides is an object but not a Slide or SlideRange object
;                  |3 - Error occurred creating the SlideRange from $vSlides. @extended is set to the COM error code returned
;                  |4 - $vSlides has invalid format
; Author ........: water
; Modified ......:
; Remarks .......: The SlideRange parameter $vSlides can be defined using one of the following formats:
;                  * collection of individual slides:
;                    - SlideRange object
;                    - Index number of the first and last slide separated by a hyphen e.g. 1-3 or 1-* (* represents the last slide)
;                    - Index number: -1 (SlideRange consists of all selected slides)
;                    - Zero based array holding the index numbers or names of the slides
;                  * collection of all slides:
;                    - Slides object
;                    - Index number: 0 (SlideRange consists of all slides)
;                  * single slide:
;                    - Slide object
;                    - Index number
;                    - Slide name
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_SlideRangeSet($oPresentation, $vSlides = Default)
	Local $oSlideRange
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If $vSlides = Default Then $vSlides = 0
	; SlideRange
	Select
		Case IsInt($vSlides)                                                  ; Return all Slides of the Presentation
			If $vSlides = 0 Then
				$oSlideRange = $oPresentation.Slides.Range
				If @error Then Return SetError(3, @error, 0)
			ElseIf $vSlides = -1 Then                                       ; Return all selected Slides of the Presentation
				$oSlideRange = $oPresentation.Application.ActiveWindow.Selection.SlideRange
				If @error Then Return SetError(3, @error, 0)
			Else
				$oSlideRange = $oPresentation.Slides.Range($vSlides)        ; Slide specified by Index
				If @error Then Return SetError(3, @error, 0)
			EndIf
		Case IsObj($vSlides)
			If ObjName($vSlides, 1) = "_Slide" Then                           ; Create a SlideRange object ob the passed Slide object
				$oSlideRange = $oPresentation.Slides.Range($vSlides.SlideNumber)
				If @error Then Return SetError(3, @error, 0)
			ElseIf ObjName($vSlides, 1) = "SlideRange" Then                  ; SlideRange object has been passed
				$oSlideRange = $vSlides
			Else                                                             ; Invalid object
				Return SetError(2, 0, 0)
			EndIf
		Case Else                                                            ; Create SlideRange object from array, index range or name
			Select
				Case IsString($vSlides) And StringInStr($vSlides, "-") > 0
					Local $aSlidesTemp = StringSplit($vSlides, "-", $STR_NOCOUNT)
					If @error Then Return SetError(4, 0, 0)
					If UBound($aSlidesTemp) < 1 Or UBound($aSlidesTemp) > 2 Then Return SetError(4, 0, 0)
					If $aSlidesTemp[0] = "" Then $aSlidesTemp[0] = 1
					If $aSlidesTemp[1] = "" Then $aSlidesTemp[1] = "*"
					If $aSlidesTemp[1] = "*" Then $aSlidesTemp[1] = $oPresentation.Slides.Count
					If Not ($aSlidesTemp[0] <= $aSlidesTemp[1]) Then Return SetError(4, 0, 0)
					Local $aSlides[$aSlidesTemp[1] - $aSlidesTemp[0] + 1]
					For $i = 0 To UBound($aSlides) - 1
						$aSlides[$i] = $aSlidesTemp[0] + $i
					Next
					$oSlideRange = $oPresentation.Slides.Range($aSlides)
					If @error Then Return SetError(3, @error, 0)
				Case IsString($vSlides) Or IsArray($vSlides)
					$oSlideRange = $oPresentation.Slides.Range($vSlides)
					If @error Then Return SetError(3, @error, 0)
				Case Else
					Return SetError(4, 0, 0)
			EndSelect
	EndSelect
	Return $oSlideRange
EndFunc   ;==>_PPT_SlideRangeSet

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_SlideShow
; Description ...: Set properties for a slide show and run the show.
; Syntax.........: _PPT_SlideShow($oPresentation[, $bRun = True[, $vStartingSlide = 1[, $vEndingSlide = Default[, $bLoop = True[, $iShowType = $ppShowTypeKiosk]]]]])
; Parameters ....: $oPresentation  - Presentation object.
;                  $bRun           - [optional] If True then the function starts the slide show (default = True).
;                  $vStartingSlide - [optional] Name or index of the first slide to be shown (default = 1).
;                  $vEndingSlide   - [optional] Name or index of the last slide to be shown (default = keyword Default = the last slide in the presentation).
;                  $bLoop          - [optional] If True then the slide show starts again when having reached the end (default = True).
;                  $iShowType      - [optional] Type of slide show defined by the PpSlideShowType enumeration (default = $ppShowTypeKiosk).
; Return values .: Success - the slidewindow object when $bRun = True, else 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $vStartingSlide is a number and < 1 or > current number of slides
;                  |3 - $vEndingSlide is a number and < 1 or > current number of slides
;                  |4 - Error occurred when running the slide show. @extended is set to the COM error code returned by the Run method
; Author ........: water
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_SlideShow(ByRef $oPresentation, $bRun = Default, $vStartingSlide = Default, $vEndingSlide = Default, $bLoop = Default, $iShowType = Default)
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	Local $iSlideCount = $oPresentation.Slides.Count
	If $bRun = Default Then $bRun = True
	If $vStartingSlide = Default Then $vStartingSlide = 1
	If IsNumber($vStartingSlide) And ($vStartingSlide < 1 Or $vStartingSlide > $iSlideCount) Then Return SetError(2, 0, 0)
	If $vEndingSlide = Default Then $vEndingSlide = $iSlideCount
	If IsNumber($vEndingSlide) And ($vEndingSlide < 1 Or $vEndingSlide > $iSlideCount) Then Return SetError(3, 0, 0)
	If $bLoop = Default Then $bLoop = True
	If $iShowType = Default Then $iShowType = $ppShowTypeKiosk
	If $vStartingSlide <> Default Or $vEndingSlide <> Default Then     ; User specified start or end slide => Set RangeType
		$oPresentation.SlideshowSettings.StartingSlide = $vStartingSlide
		$oPresentation.SlideshowSettings.EndingSlide = $vEndingSlide
		$oPresentation.SlideshowSettings.RangeType = $ppShowSlideRange
	Else
		$oPresentation.SlideshowSettings.RangeType = $ppShowAll     ; Show all slides
	EndIf
	$oPresentation.SlideshowSettings.LoopUntilStopped = $bLoop
	$oPresentation.SlideshowSettings.ShowType = $iShowType
	If $bRun Then
		Local $oSlideShow = $oPresentation.SlideshowSettings.Run()
		If @error Then Return SetError(4, @error, 0)
		Return $oSlideShow
	EndIf
	Return 1
EndFunc   ;==>_PPT_SlideShow

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_TableGet
; Description ...: Read data from a PowerPoint Table Shape to an Array.
; Syntax.........: _PPT_TableGet($oPresentation, $vSlide[, $vShape = 0])
; Parameters ....: $oPresentation - Object of the presentation to process.
;                  $vSlide        - Slide to process.
;                  $vShape        - Shape to process on $vSlide. See Remarks.
;                                   When set to 0 the first Table Shape on the Slide will be processed (default).
; Return values .: Success - A two-dimensional zero based array holding the values of the Table Shape.
;                  Failure - "" and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - Specified Shape is not a Table Shape or no Table Shape could be found
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_TableGet($oPresentation, $vSlide, $vShape = Default)
	Local $oShape, $oSlide, $bFound = False
	If IsObj($oPresentation) = 0 Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, "")
	$oSlide = _PPT_SlideRangeSet($oPresentation, $vSlide)
	If $vShape = Default Then $vShape = 0
	If $vShape <> 0 Then
		$oShape = _PPT_ShapeRangeSet($oPresentation, $vSlide, $vShape)
	Else
		For $oShape In $oSlide.Shapes
			If $oShape.HasTable Then
				$bFound = True
				ExitLoop
			EndIf
		Next
		If Not $bFound Then Return SetError(2, 0, "")
	EndIf
	If Not $oShape.HasTable Then Return SetError(2, 0, "")
	Local $iTableRows = $oShape.Table.Rows.Count
	Local $iTableCols = $oShape.Table.Columns.Count
	Local $aData[$iTableRows][$iTableCols]
	For $iTableRow = 1 To $iTableRows
		For $iTableCol = 1 To $iTableCols
			$aData[$iTableRow - 1][$iTableCol - 1] = $oShape.Table.Cell($iTableRow, $iTableCol).Shape.TextFrame.TextRange.Text
		Next
	Next
	Return $aData
EndFunc   ;==>_PPT_TableGet

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_TableSet
; Description ...: Write data to a PowerPoint Table Shape.
; Syntax.........: _PPT_TableSet($oPresentation, $vSlide, $vShape, $aDataArray[, $iTableStartRow = Default[, $iTableStartCol = Default[, $iDataStartRow = 0[, $iDataStartCol = 0]]]])
; Parameters ....: $oPresentation  - Object of the presentation to process.
;                  $vSlide         - Slide to process.
;                  $vShape         - Shape to process on $vSlide. See Remarks.
;                                    When set to 0 the first Table Shape on the Slide will be processed.
;                  $aDataArray     - 1D or 2D array to write to $vShape.
;                  $iTableStartRow - [optional] First row (starting with 1) of the Table Shape to write $aDataArray to (default = 1).
;                  $iTableStartCol - [optional] First column (starting with 1) of the Table Shape to write $aDataArray to (default = 1).
;                  $iDataStartRow  - [optional] First row (starting with 0) of $aDataArray to write to the Table Shape (default = 0).
;                  $iDataStartCol  - [optional] First column (starting with 0) of $aDataArray to write to the Table Shape (default = 0).
; Return values .: Success - 1.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $aDataArray has to be a 1D or 2D array
;                  |3 - Specified Shape is not a Table Shape or no Table Shape could be found
; Author ........: water
; Modified ......:
; Remarks .......: If $aDataArray exceeds the size of the Table Shape $aDataArray will be truncated without raising an error.
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_TableSet($oPresentation, $vSlide, $vShape, $aDataArray, $iTableStartRow = Default, $iTableStartCol = Default, $iDataStartRow = Default, $iDataStartCol = Default)
	Local $aData[1], $iDataRow = 0, $iDataCol = 0, $oSlide, $oShape, $bFound = False
	If IsObj($oPresentation) = 0 Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If $iTableStartRow = Default Then $iTableStartRow = 1
	If $iTableStartCol = Default Then $iTableStartCol = 1
	If $iDataStartRow = Default Then $iDataStartRow = 0
	If $iDataStartCol = Default Then $iDataStartCol = 0
	If $iDataStartRow <> 0 Then $iDataRow = $iDataStartRow
	If $iDataStartCol <> 0 Then $iDataCol = $iDataStartCol
	If Not IsArray($aDataArray) Or UBound($aDataArray, 0) > 2 Then Return SetError(2, 0, 0)
	$oSlide = _PPT_SlideRangeSet($oPresentation, $vSlide)
	If $vShape <> 0 Then
		$oShape = _PPT_ShapeRangeSet($oPresentation, $vSlide, $vShape)
	Else
		For $oShape In $oSlide.Shapes
			If $oShape.HasTable Then
				$bFound = True
				ExitLoop
			EndIf
		Next
		If Not $bFound Then Return SetError(3, 0, 0)
	EndIf
	If Not $oShape.HasTable Then Return SetError(3, 0, 0)
	Local $iTableRows = $oShape.Table.Rows.Count
	Local $iTableCols = $oShape.Table.Columns.Count
	If UBound($aDataArray, 0) = 1 Then     ; Copy the 1D array to a 2D array
		ReDim $aData[UBound($aDataArray, 1)][1]
		For $i = 0 To UBound($aDataArray, 1) - 1
			$aData[$i][0] = $aDataArray[$i]
		Next
	Else
		$aData = $aDataArray
	EndIf
	Local $iDataRows = UBound($aData, 1)
	Local $iDataCols = UBound($aData, 2)
	For $iTableRow = $iTableStartRow To $iTableRows
		If $iDataRow <= $iDataRows - 1 Then
			For $iTableCol = $iTableStartCol To $iTableCols
				If $iDataCol <= $iDataCols - 1 Then
					$oShape.Table.Cell($iTableRow, $iTableCol).Shape.TextFrame.TextRange.Text = $aData[$iDataRow][$iDataCol]
					If @error Then Return SetError(999, @error, 0)
					$iDataCol += 1
				Else
					ExitLoop
				EndIf
			Next
			$iDataRow += 1
			$iDataCol = $iDataStartCol
		Else
			ExitLoop
		EndIf
	Next
	Return 1
EndFunc   ;==>_PPT_TableSet

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_TextFindReplace
; Description ...: Find & replace text on a single or selected slides.
; Syntax.........: _PPT_TextFindReplace($oPresentation, $sFindText, $sReplaceText[, $vSlides = 0[, $vShapes = 0[, $iOccurrence = 0[, $bMatchCase = False[, $bWholeWords = False]]]]])
; Parameters ....: $oPresentation   - Object of the presentation to process.
;                  $sFindText       - The text to be searched for.
;                  $sReplaceText    - The replacement text. To delete found text use an empty string ("").
;                  $vSlides         - [optional] Slides to process. See Remarks (default = 0 = all Slides).
;                  $vShapes         - [optional] Shapes to process on each selected Slide. See Remarks (default = 0 = all Shapes).
;                  $iOccurrence     - [optional] Number of times to replace $sFindText in a TextFrame (default = 0 = all occurrences).
;                  $bMatchCase      - [optional] Determines whether a distinction is made on the basis of case (default = False).
;                  $bWholeWords     - [optional] Determines whether only whole words are found (default = False).
; Return values .: Success - 1, @extended holds the number of successfully performed replacements.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - $sFindText is empty
;                  |3 - Error occurred creating the SlideRange from $vSlides. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |4 - Error occurred creating the ShapeRange from $vShapes. @extended is set to the error code returned by _PPT_ShapeRangeSet
;                  |5 - Number of Slides is > 1 but $vShapes is not one of the allowed values: 0, >0 or a string
; Author ........: water
; Modified ......:
; Remarks .......: With this function you can delete a single Shape from multiple Slides or multiple Shapes from a single Slide.
;+
;                  Parameter $vSlides: See _PPT_SlideRangeSet for the format of this parameter
;                  Parameter $vShapes:
;                    If Number of Slides is = 1: All values for _PPT_ShapeRangeSet are valid. See _PPT_ShapeRangeSet for the format of this parameter
;                    If Number of Slides is > 1: only 0 (all Shapes), integer (ID of the Shape) or a string (Name of the Shape) are valid
;+
;                  While looping through the Shapes of a Slide errors are ignored and processing continues with the next Shape.
; Related .......:
; Link ..........: https://www.thespreadsheetguru.com/the-code-vault/find-and-replace-all-powerpoint-vba-macro
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_TextFindReplace($oPresentation, $sFindText, $sReplaceText, $vSlides = Default, $vShapes = Default, $iOccurrence = Default, $bMatchCase = Default, $bWholeWords = Default)
	Local $oSlide, $oSlideRange, $oShape, $oShapeRange, $oShapeTextFrame, $oTextRange, $sShapeText, $oTempText, $iReplaceCountTotal = 0, $iReplaceCount = 0
	If $bMatchCase = Default Then $bMatchCase = False
	If $bWholeWords = Default Then $bWholeWords = False
	If $iOccurrence = Default Then $iOccurrence = 0
	If $vSlides = Default Then $vSlides = 0
	If $vShapes = Default Then $vShapes = 0
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	If StringStripWS($sFindText, $STR_STRIPALL) = "" Then Return SetError(2, 0, 0)
	$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)
	If @error Then Return SetError(3, @error, 0)
	If $oSlideRange.Count > 1 And _
			Not ($vShapes = 0 Or $vShapes > 0 Or IsString($vShapes)) Then     ; Check allowed values: 0 (all Slides), integer (single Slide), String (SlideName)
		Return SetError(5, 0, 0)
	EndIf
	; Loop through the specified Slides
	For $oSlide In $oSlideRange
		; Get the ShapeRange for every specified Slide
		$oShapeRange = _PPT_ShapeRangeSet($oPresentation, $oSlide, $vShapes)
		If @error Then Return SetError(4, @error, 0)
		For $oShape In $oShapeRange
			; Get the TextFrame. If the Shape contains no TextFrame continue with the next Shape
			$oShapeTextFrame = $oShape.TextFrame
			If @error Then ContinueLoop
			; Get TextRange of the TextFrame. If there is an error continue with the next Shape
			$oTextRange = $oShapeTextFrame.TextRange
			If @error Then ContinueLoop
			; Get the Text of the TextRange
			$sShapeText = $oTextRange.Text
			If @error Then ContinueLoop
			; Ensure there is text to search through
			If $sShapeText <> "" Then
				; Find first instance of "FindText"
				$oTempText = $oTextRange.Replace($sFindText, $sReplaceText, Default, $bMatchCase, $bWholeWords)
				If IsObj($oTempText) Then
					$iReplaceCountTotal += 1
					$iReplaceCount = 1
				EndIf
				; Find any additional instances of "Findtext" (if exists)
				While IsObj($oTempText)
					If $iReplaceCount >= $iOccurrence Then ExitLoop
					$oTextRange = $oTextRange.Characters($oTempText.Start + $oTempText.Length, $oTextRange.Length)
					$oTempText = $oTextRange.Replace($sFindText, $sReplaceText, Default, $bMatchCase, $bWholeWords)
					If IsObj($oTempText) Then
						$iReplaceCountTotal += 1
						$iReplaceCount += 1
					EndIf
				WEnd
			EndIf
		Next
	Next
	Return SetError(0, $iReplaceCountTotal, 1)
EndFunc   ;==>_PPT_TextFindReplace

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_TextSet
; Description ...: Set the Text of a Shape or Shapes of a single or multiple Slides.
; Syntax.........: _PPT_TextSet($oPresentation, $sText[, $vSlides = 0[, $vShapes = 0]])
; Parameters ....: $oPresentation - Object of the presentation to process.
;                  $sText         - Text to insert into the Shapes.
;                  $vSlides       - [optional] Slides to process. See Remarks (default = 0 = all Slides).
;                  $vShapes       - [optional] Shapes to process on each selected Slide. See Remarks (default = 0 = all Shapes).
;                  $iFlag         - [optional] This flag specifies how to process the Text:
;                  |0 - Replace the existing Text in the Shape (default).
;                  |1 - Prepend the Text to the existing Text in the Shape.
;                  |2 - Append the Text to the existing Text in the Shape.
; Return values .: Success - 1, @extended holds the number of processed Shapes.
;                  Failure - 0 and sets @error.
;                  |1 - $oPresentation is not an object or not a presentation object
;                  |2 - Number of Slides is > 1 but $vShapes is not one of the allowed values: 0, >0 or a string
;                  |3 - Error occurred creating the SlideRange from $vSlides. @extended is set to the error code returned by _PPT_SlideRangeSet
;                  |4 - Error occurred creating the ShapeRange from $vShapes. @extended is set to the error code returned by _PPT_ShapeRangeSet
;                  |5 - $iFlag is invalid. Has to be >= 0 and <= 2
; Author ........: water
; Modified ......:
; Remarks .......: With this function you can set the text of a single Shape of multiple Slides or multiple Shapes of a single Slide.
;+
;                  Parameter $vSlides: See _PPT_SlideRangeSet for the format of this parameter
;                  Parameter $vShapes:
;                    If Number of Slides is = 1: All values for _PPT_ShapeRangeSet are valid. See _PPT_ShapeRangeSet for the format of this parameter
;                    If Number of Slides is > 1: only 0 (all Shapes), integer (ID of the Shape) or a string (Name of the Shape) are valid
;+
;                  While looping through the specified Slides all specified Shapes are modified. Errors are ignored and processing continues with the next Shape or Slide.
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_TextSet($oPresentation, $sText, $vSlides = Default, $vShapes = Default, $iFlag = Default)
	Local $oSlide, $oSlideRange, $oShape, $oShapeRange, $iTextCount = 0
	If $vSlides = Default Then $vSlides = 0
	If $vShapes = Default Then $vShapes = 0
	If $iFlag = Default Then $iFlag = 0
	If $iFlag < 0 Or $iFlag > 2 Then Return SetError(5, 0, 0)
	If Not IsObj($oPresentation) Or ObjName($oPresentation, 1) <> "_Presentation" Then Return SetError(1, 0, 0)
	$oSlideRange = _PPT_SlideRangeSet($oPresentation, $vSlides)
	If @error Then Return SetError(3, @error, 0)
	If $vShapes = Default Then $vShapes = 0
	If $oSlideRange.Count > 1 And _
			Not ($vShapes = 0 Or $vShapes > 0 Or IsString($vShapes)) Then     ; Check allowed values: 0 (all Slides), integer (single Slide), String (SlideName)
		Return SetError(2, 0, 0)
	EndIf
	; Loop through the specified Slides
	For $oSlide In $oSlideRange
		; Get the ShapeRange for every specified Slide
		$oShapeRange = _PPT_ShapeRangeSet($oPresentation, $oSlide, $vShapes)
		If @error Then Return SetError(4, @error, 0)
		For $oShape In $oShapeRange
			Switch $iFlag
				Case 0
					$oShape.TextFrame.TextRange.Text = $sText
				Case 1
					$oShape.TextFrame.TextRange.Text = $sText & $oShape.TextFrame.TextRange.Text
				Case 2
					$oShape.TextFrame.TextRange.Text = $oShape.TextFrame.TextRange.Text & $sText
			EndSwitch
			If @error = 0 Then $iTextCount += 1
		Next
	Next
	Return SetError(0, $iTextCount, 1)
EndFunc   ;==>_PPT_TextSet

; #FUNCTION# ====================================================================================================================
; Name...........: _PPT_VersionInfo
; Description ...: Returns a string holding version/date of the PowerPoint UDF.
; Syntax.........: _PPT_VersionInfo()
; Parameters ....: None
; Return values .: Success - string holding version/date of PowerPoint.au3.
; Author ........: water
; Modified.......:
; Remarks .......: Example of the returned string: "PowerPoint: 1.0.0.0 2020-09-01"
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _PPT_VersionInfo()
	Return "  " & $__g_sVersionPowerPoint
EndFunc   ;==>_PPT_VersionInfo
#EndRegion #FUNCTION#

#Region #INTERNAL_USE_ONLY#
; #INTERNAL_USE_ONLY#============================================================================================================
; Name ..........: __PPT_ErrorHandler
; Description ...: Called if an ObjEvent error occurs.
; Syntax.........: __PPT_ErrorHandler()
; Parameters ....: None
; Return values .: @error is set to the COM error by AutoIt
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func __PPT_ErrorHandler()
	Local $sError = "COM Error Encountered in " & @ScriptName & @CRLF & _
			"UDF version = " & @CRLF & _PPT_VersionInfo() & @CRLF & _
			"@AutoItVersion = " & @AutoItVersion & @CRLF & _
			"@AutoItX64 = " & @AutoItX64 & @CRLF & _
			"@Compiled = " & @Compiled & @CRLF & _
			"@OSArch = " & @OSArch & @CRLF & _
			"@OSVersion = " & @OSVersion & @CRLF & _
			"Scriptline = " & $__oPPT_Error.Scriptline & @CRLF & _
			"NumberHex = 0x" & Hex($__oPPT_Error.Number, 8) & @CRLF & _
			"Number = " & $__oPPT_Error.Number & @CRLF & _
			"WinDescription = " & StringStripWS($__oPPT_Error.WinDescription, $STR_STRIPTRAILING) & @CRLF & _
			"Description = " & StringStripWS($__oPPT_Error.Description, $STR_STRIPTRAILING) & @CRLF & _
			"Source = " & $__oPPT_Error.Source & @CRLF & _
			"HelpFile = " & $__oPPT_Error.HelpFile & @CRLF & _
			"HelpContext = " & $__oPPT_Error.HelpContext & @CRLF & _
			"LastDllError = " & $__oPPT_Error.LastDllError
	If $__iPPT_Debug > 0 Then
		If $__iPPT_Debug = 1 Then ConsoleWrite($sError & @CRLF & "========================================================" & @CRLF)
		If $__iPPT_Debug = 2 Then MsgBox($MB_ICONERROR, "PowerPoint UDF - Debug Info", $sError)
		If $__iPPT_Debug = 3 Then FileWrite($__sPPT_DebugFile, @YEAR & "." & @MON & "." & @MDAY & " " & @HOUR & ":" & @MIN & ":" & @SEC & " " & @CRLF & _
				"-------------------" & @CRLF & $sError & @CRLF & "========================================================" & @CRLF)
	EndIf
EndFunc   ;==>__PPT_ErrorHandler
#EndRegion #INTERNAL_USE_ONLY#

