#include-once
#include <AutoItObject.au3>
#include <LinkedList.au3>

#include "BlenderConstants.au3"

#include "Blender\Object.au3"
#include "Blender\Scene.au3"
#include "Blender\Action.au3"

#include "Blender\Mesh.au3"

; #INDEX# =======================================================================================================================
; Title .........: Blender UDF
; UDF Version ...: 1.3 (Blender 2.76)
; AutoIt Version : 3.3.14.2
; Language ......: English
; Description ...: Functions for receiving and editing data from Blender 3D software.
; Author(s) .....: scintilla4evr
; Files .........: autoit-blender.py
; ===============================================================================================================================

Global Const $__Blender_sRegKey = "HKCU\Software\Blender UDF for AutoIt"
Global Const $__Blender_sRegKey_ObjectData = $__Blender_sRegKey & "\PythonObjectData"

Func _Blender_Startup($bStartUDFs = True)
	If $bStartUDFs Then _AutoItObject_Startup()

	RegWrite($__Blender_sRegKey, "", "REG_SZ", 0) ; Ensure that both keys exist
	RegWrite($__Blender_sRegKey_ObjectData, "", "REG_SZ", 0)
EndFunc

Func _Blender_Shutdown($bStopUDFs = True)
	If $bStopUDFs Then _AutoItObject_Shutdown()
EndFunc

#Region Objects

Func _Blender_GetScene($sName = "Scene")
	Return __bpy_CreateObject("bpy.data.scenes['"&$sName&"']")
EndFunc

Func _Blender_GetCamera($sName = "Camera")
	Return __bpy_CreateObject("bpy.data.cameras['"&$sName&"']")
EndFunc

Func _Blender_GetObject($sName = "Object")
	Return __bpy_CreateObject("bpy.data.objects['"&$sName&"']")
EndFunc

Func _Blender_GetMaterial($sName = "Material")
	Return __bpy_CreateObject("bpy.data.materials['"&$sName&"']")
EndFunc

Func _Blender_GetMesh($sName = "Mesh")
	Return __bpy_CreateObject("bpy.data.meshes['"&$sName&"']")
EndFunc

Func _Blender_GetLamp($sName = "Lamp")
	Return __bpy_CreateObject("bpy.data.lamps['"&$sName&"']")
EndFunc

Func _Blender_GetTexture($sName = "Texture")
	Return __bpy_CreateObject("bpy.data.textures['"&$sName&"']")
EndFunc

Func _Blender_GetBrush($sName = "Brush")
	Return __bpy_CreateObject("bpy.data.brushes['"&$sName&"']")
EndFunc

Func _Blender_GetWorld($sName = "World")
	Return __bpy_CreateObject("bpy.data.worlds['"&$sName&"']")
EndFunc

Func _Blender_GetText($sName = "Text")
	Return __bpy_CreateObject("bpy.data.texts['"&$sName&"']")
EndFunc

Func _Blender_GetCurve($sName = "Curve")
	Return __bpy_CreateObject("bpy.data.curves['"&$sName&"']")
EndFunc

Func _Blender_GetLattice($sName = "Lattice")
	Return __bpy_CreateObject("bpy.data.lattices['"&$sName&"']")
EndFunc

Func _Blender_GetMetaball($sName = "Mball")
	Return __bpy_CreateObject("bpy.data.metaballs['"&$sName&"']")
EndFunc

Func _Blender_GetImage($sName = "Image")
	Return __bpy_CreateObject("bpy.data.images['"&$sName&"']")
EndFunc

Func _Blender_GetAction($sName = "Action")
	Return __bpy_CreateObject("bpy.data.actions['"&$sName&"']")
EndFunc

#EndRegion

#Region AutoIt equivalents of Blender objects

Func _Blender_CreateVector($x = 0, $y = 0, $z = 0)
	Local $obj = _AutoItObject_Create()

	_AutoItObject_AddProperty($obj, "x", $ELSCOPE_PUBLIC, $x)
	_AutoItObject_AddProperty($obj, "y", $ELSCOPE_PUBLIC, $y)
	_AutoItObject_AddProperty($obj, "z", $ELSCOPE_PUBLIC, $z)

	_AutoItObject_AddMethod($obj, "py", "__bpy_Vector2Py")

	Return $obj
EndFunc

Func _Blender_CreateVector2D($x = 0, $y = 0)
	Local $obj = _AutoItObject_Create()

	_AutoItObject_AddProperty($obj, "x", $ELSCOPE_PUBLIC, $x)
	_AutoItObject_AddProperty($obj, "y", $ELSCOPE_PUBLIC, $y)

	_AutoItObject_AddMethod($obj, "py", "__bpy_Vector2D2Py")

	Return $obj
EndFunc

Func _Blender_CreateQuaternion($w = 0, $x = 0, $y = 0, $z = 0)
	Local $obj = _AutoItObject_Create()

	_AutoItObject_AddProperty($obj, "w", $ELSCOPE_PUBLIC, $w)
	_AutoItObject_AddProperty($obj, "x", $ELSCOPE_PUBLIC, $x)
	_AutoItObject_AddProperty($obj, "y", $ELSCOPE_PUBLIC, $y)
	_AutoItObject_AddProperty($obj, "z", $ELSCOPE_PUBLIC, $z)

	_AutoItObject_AddMethod($obj, "py", "__bpy_Quaternion2Py")

	Return $obj
EndFunc

Func _Blender_CreateColor($r = 0, $g = 0, $b = 0)
	Local $obj = _AutoItObject_Create()

	_AutoItObject_AddProperty($obj, "r", $ELSCOPE_PUBLIC, $r)
	_AutoItObject_AddProperty($obj, "g", $ELSCOPE_PUBLIC, $g)
	_AutoItObject_AddProperty($obj, "b", $ELSCOPE_PUBLIC, $b)

	_AutoItObject_AddMethod($obj, "py", "__bpy_Color2Py")

	Return $obj
EndFunc

#EndRegion

#Region INTERNAL: Parse Python strings and return Blender objects

Func __bpy_ParseMaterial($sPyStr)
	$aRegexp = StringRegExp($sPyStr, '(?i)<bpy_struct, Material\("(.+)"\)>', 1)
	If Not IsArray($aRegexp) Then Return SetError($BUDF_ERR_REGEXP, 0, 0)

	Return _Blender_GetMaterial($aRegexp[0])
EndFunc

Func __bpy_ParseMesh($sPyStr)
	$aRegexp = StringRegExp($sPyStr, '(?i)<bpy_struct, Mesh\("(.+)"\)>', 1)
	If Not IsArray($aRegexp) Then Return SetError($BUDF_ERR_REGEXP, 0, 0)

	Return _Blender_GetMesh($aRegexp[0])
EndFunc

Func __bpy_ParseCamera($sPyStr)
	$aRegexp = StringRegExp($sPyStr, '(?i)<bpy_struct, Camera\("(.+)"\)>', 1)
	If Not IsArray($aRegexp) Then Return SetError($BUDF_ERR_REGEXP, 0, 0)

	Return _Blender_GetCamera($aRegexp[0])
EndFunc

Func __bpy_ParseObject($sPyStr)
	$aRegexp = StringRegExp($sPyStr, '(?i)<bpy_struct, Object\("(.+)"\)>', 1)
	If Not IsArray($aRegexp) Then Return SetError($BUDF_ERR_REGEXP, 0, 0)

	Return _Blender_GetObject($aRegexp[0])
EndFunc

Func __bpy_ParseWorld($sPyStr)
	$aRegexp = StringRegExp($sPyStr, '(?i)<bpy_struct, World\("(.+)"\)>', 1)
	If Not IsArray($aRegexp) Then Return SetError($BUDF_ERR_REGEXP, 0, 0)

	Return _Blender_GetWorld($aRegexp[0])
EndFunc

#EndRegion

#Region INTERNAL: AutoIt to Python methods and vice-versa

Func __bpy_Vector2Py($oSelf)
	Return "("&$oSelf.x&", "&$oSelf.y&", "&$oSelf.z&")"
EndFunc

Func __bpy_Vector2D2Py($oSelf)
	Return "("&$oSelf.x&", "&$oSelf.y&")"
EndFunc

Func __bpy_Quaternion2Py($oSelf)
	Return "Quaternion(("&$oSelf.w&", "&$oSelf.x&", "&$oSelf.y&", "&$oSelf.z&"))"
EndFunc

Func __bpy_Color2Py($oSelf)
	Return "Color(("&$oSelf.r&", "&$oSelf.g&", "&$oSelf.b&"))"
EndFunc

Func __bpy_ParseValue($sPyStr)
	If StringRegExp($sPyStr, "(?i)<Vector \((.+), (.+), (.+)\)>") Then
		$aRegexp = StringRegExp($sPyStr, "(?i)<Vector \((.+), (.+), (.+)\)>", 1)

		Return _Blender_CreateVector($aRegexp[0], $aRegexp[1], $aRegexp[2])
	ElseIf StringRegExp($sPyStr, "(?i)<Vector \((.+), (.+)\)>") Then
		$aRegexp = StringRegExp($sPyStr, "(?i)<Vector \((.+), (.+)\)>", 1)

		Return _Blender_CreateVector2D($aRegexp[0], $aRegexp[1])
	ElseIf StringRegExp($sPyStr, "(?i)<Quaternion \(w=(.+), x=(.+), y=(.+), z=(.+)\)>") Then
		$aRegexp = StringRegExp($sPyStr, "(?i)<Quaternion \(w=(.+), x=(.+), y=(.+), z=(.+)\)>", 1)

		Return _Blender_CreateQuaternion($aRegexp[0], $aRegexp[1], $aRegexp[2], $aRegexp[3])
	ElseIf StringRegExp($sPyStr, "(?i)<Color \(r=(.+), g=(.+), b=(.+)\)>") Then
		$aRegexp = StringRegExp($sPyStr, "(?i)<Color \(r=(.+), g=(.+), b=(.+)\)>", 1)

		Return _Blender_CreateColor($aRegexp[0], $aRegexp[1], $aRegexp[2])
	EndIf

	Return $sPyStr
EndFunc

#EndRegion

#Region INTERNAL: bpy complex objects (Scene, World, etc.)

Func __bpy_CreateObject($sPythonData, $oParent = 0, $iTimeout = 2000)
	RegWrite($__Blender_sRegKey, "", "REG_SZ", 4) ; Can I haz object pleez?
	RegWrite($__Blender_sRegKey, "PythonInput", "REG_SZ", $sPythonData) ; Put the Python code for Blender to evaluate

	Local $hTimer = TimerInit()

	Do
		If TimerDiff($hTimer) > $iTimeout Then
			RegWrite($__Blender_sRegKey, "", "REG_SZ", 0) ; Give up

			Return SetError($BUDF_ERR_TIMEOUT, 0, 0)
		EndIf
	Until RegRead($__Blender_sRegKey, "") = 3

	Local $iCount = RegRead($__Blender_sRegKey_ObjectData, "")

	$oReturn = _AutoItObject_Create()

	_AutoItObject_AddProperty($oReturn, "pyexpr", $ELSCOPE_PUBLIC, $sPythonData)
	_AutoItObject_AddProperty($oReturn, "parentObj", $ELSCOPE_PUBLIC, $oParent)

	_AutoItObject_AddMethod($oReturn, "get", "__bpy_Obj_GetProperty")
	_AutoItObject_AddMethod($oReturn, "set", "__bpy_Obj_SetProperty")
	_AutoItObject_AddMethod($oReturn, "setString", "__bpy_Obj_SetStringProperty")
	_AutoItObject_AddMethod($oReturn, "query", "__bpy_Obj_QueryProperty")

	$propNameList = LinkedList()

	For $i = 2 To $iCount+1
		$sName = RegEnumVal($__Blender_sRegKey_ObjectData, $i)
		$sValue = RegRead($__Blender_sRegKey_ObjectData, $sName)
		$sValue = __bpy_ParseValue($sValue)

		$ListRegKey = RegRead($__Blender_sRegKey_ObjectData&"\"&$sName, "")
		If Not @error Then
			$key = $__Blender_sRegKey_ObjectData&"\"&$sName
			$list = LinkedList()

			For $j = 0 To (RegRead($key, "")-1)
				$value = RegRead($key, $j)
				$list.add($value)
			Next

			_AutoItObject_AddProperty($oReturn, $sName, $ELSCOPE_PUBLIC, $list)
		Else
			_AutoItObject_AddProperty($oReturn, $sName, $ELSCOPE_PUBLIC, $sValue)
		EndIf

		$propNameList.add($sName)
	Next

	_AutoItObject_AddProperty($oReturn, "propertyNames", $ELSCOPE_PUBLIC, $propNameList)

	RegWrite($__Blender_sRegKey, "", "REG_SZ", 0) ; Clear up the mess
	RegWrite($__Blender_sRegKey, "PythonInput", "REG_SZ", "")
	RegWrite($__Blender_sRegKey, "PythonOutput", "REG_SZ", "")
	RegDelete($__Blender_sRegKey_ObjectData)
	RegWrite($__Blender_sRegKey_ObjectData, "", "REG_SZ", 0)

	Return $oReturn
EndFunc

Func __bpy_Obj_GetProperty($oSelf, $sPropName)
	Return __bpy_ParseValue(__bpy_RequestData($oSelf.pyexpr&"."&$sPropName))
EndFunc

Func __bpy_Obj_SetProperty($oSelf, $sPropName, $vValue)
	Local $sData = $vValue
	If IsObj($vValue) Then $sData = $vValue.py()

	Return __bpy_SendData($oSelf.pyexpr&"."&$sPropName&" = "&$sData)
EndFunc

Func __bpy_Obj_SetStringProperty($oSelf, $sPropName, $vValue)
	Local $sData = StringReplace($vValue, "\", "\\")
	If IsObj($vValue) Then $sData = $vValue.py()

	Return __bpy_SendData($oSelf.pyexpr&"."&$sPropName&" = """&$sData&"""")
EndFunc

Func __bpy_Obj_QueryProperty($oSelf, $sPropName)
	Return Execute("$oSelf."&$sPropName)
EndFunc

#EndRegion

#Region INTERNAL: Sending and receiving data

Func __bpy_RequestData($sPythonData, $iTimeout = 1000)
	RegWrite($__Blender_sRegKey, "", "REG_SZ", 1) ; Inform Blender, that we're going to request data
	RegWrite($__Blender_sRegKey, "PythonInput", "REG_SZ", $sPythonData) ; Put the Python code for Blender to evaluate

	Local $hTimer = TimerInit()

	Do
		If TimerDiff($hTimer) > $iTimeout Then
			RegWrite($__Blender_sRegKey, "", "REG_SZ", 0) ; Give up

			Return SetError($BUDF_ERR_TIMEOUT, 0, 0)
		EndIf
	Until RegRead($__Blender_sRegKey, "") = 3

	Local $sReturn = RegRead($__Blender_sRegKey, "PythonOutput")

	RegWrite($__Blender_sRegKey, "", "REG_SZ", 0) ; Clear up the mess
	RegWrite($__Blender_sRegKey, "PythonInput", "REG_SZ", "")
	RegWrite($__Blender_sRegKey, "PythonOutput", "REG_SZ", "")

	Return $sReturn
EndFunc

Func __bpy_SendData($sPythonData, $iTimeout = 1000)
	RegWrite($__Blender_sRegKey, "", "REG_SZ", 2) ; Inform Blender, that we're going to mess with data
	RegWrite($__Blender_sRegKey, "PythonInput", "REG_SZ", $sPythonData) ; Put the Python code for Blender to evaluate

	Local $hTimer = TimerInit()

	Do
		If TimerDiff($hTimer) > $iTimeout Then
			RegWrite($__Blender_sRegKey, "", "REG_SZ", 0) ; Give up

			Return SetError($BUDF_ERR_TIMEOUT, 0, 0)
		EndIf
	Until RegRead($__Blender_sRegKey, "") = 3

	RegWrite($__Blender_sRegKey, "", "REG_SZ", 0) ; Clear up the mess
	RegWrite($__Blender_sRegKey, "PythonInput", "REG_SZ", "")
	RegWrite($__Blender_sRegKey, "PythonOutput", "REG_SZ", "")

	Return True
EndFunc

#EndRegion
