#include-once

#include "PhotoshopConstants.au3"

; #INDEX# =======================================================================================================================
; Title .........: Adobe Photoshop UDF
; AutoIt Version : 3.3.14.2
; Language ......: English
; Description ...: A collection of functions for automating Adobe Photoshop software
; Author(s) .....: scintilla4evr
; Resources .....: Adobe Photoshop Scripting (http://www.adobe.com/devnet/photoshop/scripting.html)
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_Create
; Description ...: Create a Photoshop application object.
; Syntax ........: _Photoshop_Create([$bForceNew = False])
; Parameters ....: $bForceNew           - [optional] Force creating a new object
; Return values .: Photoshop application object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_Create($bVisible = Default, $bForceNew = False)
	Local $oObj
	If $bVisible = Default Then $bVisible = True
	If $bForceNew Then
		$oObj = ObjCreate("Photoshop.Application")
	Else
		$oObj = ObjGet("", "Photoshop.Application")
		If Not IsObj($oObj) Then $oObj = _Photoshop_Create($bVisible, True)
	EndIf
	If Not IsObj($oObj) Then Return SetError(1, 0, 0)
	$oObj.Visible = $bVisible
	Return $oObj
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_Quit
; Description ...: Close Photoshop
; Syntax ........: _Photoshop_Quit($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_Quit($oAppl)
	$oAppl.Quit()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_GetVersion
; Description ...: Get the app version
; Syntax ........: _Photoshop_GetVersion($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: Photoshop version string
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_GetVersion($oAppl)
	Return $oAppl.Version
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_GetPath
; Description ...: Get the location of Photoshop's executable
; Syntax ........: _Photoshop_GetPath($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: Photoshop installation path
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_GetPath($oAppl)
	Return $oAppl.Path
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_OpenDialog
; Description ...: Open the Open dialog to select files
; Syntax ........: _Photoshop_OpenDialog($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: Array of selected file paths
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_OpenDialog($oAppl)
	Return $oAppl.OpenDialog()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_GetForegroundColor
; Description ...: Get the foreground color
; Syntax ........: _Photoshop_GetForegroundColor($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: Color object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_GetForegroundColor($oAppl)
	Return $oAppl.ForegroundColor
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_GetBackgroundColor
; Description ...: Get the background color
; Syntax ........: _Photoshop_GetBackgroundColor($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: Color object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_GetBackgroundColor($oAppl)
	Return $oAppl.BackgroundColor
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_SetForegroundColor
; Description ...: Set the foreground color
; Syntax ........: _Photoshop_SetForegroundColor($oAppl, $oColor)
; Parameters ....: $oAppl               - Photoshop application object
;                  $oColor              - SolidColor object
; Return values .: Previous color as a SolidColor object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_SetForegroundColor($oAppl, $oColor)
	Local $oPrev = $oAppl.ForegroundColor
	$oAppl.ForegroundColor = $oColor
	Return $oAppl
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_SetBackgroundColor
; Description ...: Set the background color
; Syntax ........: _Photoshop_SetBackgroundColor($oAppl, $oColor)
; Parameters ....: $oAppl               - Photoshop application object
;                  $oColor              - SolidColor object
; Return values .: Previous color as a SolidColor object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_SetBackgroundColor($oAppl, $oColor)
	Local $oPrev = $oAppl.BackgroundColor
	$oAppl.BackgroundColor = $oColor
	Return $oAppl
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_GetActiveDoc
; Description ...: Get the active document
; Syntax ........: _Photoshop_GetActiveDoc($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: Document object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_GetActiveDoc($oAppl)
	Return $oAppl.ActiveDocument
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_SetActiveDoc
; Description ...: Set the document as active document
; Syntax ........: _Photoshop_DocSetActive($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Previous active document object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_SetActiveDoc($oDoc)
	Local $oPrev = $oDoc.Application.ActiveDocument
	$oDoc.Application.ActiveDocument = $oDoc
	Return $oPrev
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_GetDocs
; Description ...: Get a document collection for using in For .. In .. Next loops
; Syntax ........: _Photoshop_GetDocs($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: Documents collection object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_GetDocs($oAppl)
	Return $oAppl.Documents
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_GetRulerUnit
; Description ...: Get unit used in scripting system.
; Syntax ........: _Photoshop_GetRulerUnit($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: PsUnits value
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_GetRulerUnit($oAppl)
	Return $oAppl.Preferences.RulerUnits
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_SetRulerUnit
; Description ...: Set unit used in scripting system.
; Syntax ........: _Photoshop_SetRulerUnit($oAppl, $iUnit)
; Parameters ....: $oAppl               - Photoshop application object
;                  $iUnit               - PsUnits value
; Return values .: Previous PsUnits value
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_SetRulerUnit($oAppl, $iUnit)
	Local $iPrev = $oAppl.Preferences.RulerUnits
	$oAppl.Preferences.RulerUnits = $iUnit
	Return $iPrev
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ObjectGetType
; Description ...: Get the type of an object
; Syntax ........: _Photoshop_ObjectGetType($oPsObj)
; Parameters ....: $oPsObj              - Any object returned or created using functions in this UDF
; Return values .: Object type string
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ObjectGetType($oPsObj)
	Return $oPsObj.typename
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocOpen
; Description ...: Opens a file
; Syntax ........: _Photoshop_DocOpen($oAppl, $sPath)
; Parameters ....: $oAppl               - Photoshop application object
;                  $sPath               - Image path
; Return values .: Document object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocOpen($oAppl, $sPath)
	Return $oAppl.Open($sPath)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocAdd
; Description ...: Add a document
; Syntax ........: _Photoshop_DocAdd($oAppl, $fWidth, $fHeight[, $fResolution = 72[, $sName = ""[, $iMode = $PsNewRGB[, $iInitialFill = $PsTransparent[, $fPixelAspectRatio = 1[, $iBitsPerChannel = $PsDocument8Bits[, $sColorProfile = ""]]]]]]])
; Parameters ....: $oAppl               - Photoshop application object
;                  $fWidth              - Width
;                  $fHeight             - Height
;                  $fResolution         - [optional] Pixel resolution (in pixels per inch)
;                  $sName               - [optional] Document name
;                  $iMode               - [optional] PsNewDocumentMode value
;                  $iInitialFill        - [optional] PsDocumentFill value
;                  $fPixelAspectRatio   - [optional] Pixel aspect ratio (range from 0.1 to 10)
;                  $iBitsPerChannel     - [optional] PsBitsPerChannelType value
;                  $sColorProfile       - [optional] Color profile name
; Return values .: Document object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocAdd($oAppl, $fWidth, $fHeight, $fResolution = Default, $sName = Default, $iMode = Default, $iInitialFill = Default, $fPixelAspectRatio = Default, $iBitsPerChannel = Default, $sColorProfile = Default)
	Return $oAppl.Documents.Add($fWidth, $fHeight, $fResolution, $sName, $iMode, $iInitialFill, $fPixelAspectRatio, $iBitsPerChannel, $sColorProfile)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocCount
; Description ...: Gets the count of opened documents
; Syntax ........: _Photoshop_DocCount($oAppl)
; Parameters ....: $oAppl               - Photoshop application object
; Return values .: Document count
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocCount($oAppl)
	Return $oAppl.Documents.Count
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetIndex
; Description ...: Get the index of the document
; Syntax ........: _Photoshop_DocGetIndex($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Index of the document
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetIndex($oDoc)
	Return $oDoc.Application.Documents.Index($oDoc)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetApplication
; Description ...: Get the Photoshop object that the document belongs to.
; Syntax ........: _Photoshop_DocGetApplication($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Photoshop application object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetApplication($oDoc)
	Return $oDoc.Application
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGet
; Description ...: Get a document object
; Syntax ........: _Photoshop_DocGet($oAppl[, $iIndex = 1])
; Parameters ....: $oAppl               - Photoshop application object
;                  $iIndex              - [optional] Document index
; Return values .: Document object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGet($oAppl, $iIndex = 1)
	Return $oAppl.Documents.Item($iIndex)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetSize
; Description ...: Get the size of a document
; Syntax ........: _Photoshop_DocGetSize($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: 2-item array: [0] - width, [1] - height
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetSize($oDoc)
	Local $aSize = [$oDoc.Width, $oDoc.Height]
	Return $aSize
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetName
; Description ...: Get document name
; Syntax ........: _Photoshop_DocGetName($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Document name
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetName($oDoc)
	Return $oDoc.Name
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetFullName
; Description ...: Get document path
; Syntax ........: _Photoshop_DocGetName($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Document path
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetPath($oDoc)
	Return $oDoc.Path
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocIsSaved
; Description ...: Checks whether the document has been saved since the last change.
; Syntax ........: _Photoshop_DocIsSaved($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Boolean value
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocIsSaved($oDoc)
	Return $oDoc.saved
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetResolution
; Description ...: Get document resolution
; Syntax ........: _Photoshop_DocGetResolution($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Resolution in pixels per inch
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetResolution($oDoc)
	Return $oDoc.Resolution
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetResolution
; Description ...: Get document pixel aspect ratio
; Syntax ........: _Photoshop_DocGetResolution($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Document pixel aspect ratio
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetPixelAspectRatio($oDoc)
	Return $oDoc.PixelAspectRatio
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocClose
; Description ...: Close the document
; Syntax ........: _Photoshop_DocClose($oDoc[, $iSaveOptions = Default])
; Parameters ....: $oDoc                - Document object
;                  $iSaveOptions        - [optional] PsSaveOptions value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocClose($oDoc, $iSaveOptions = Default)
	$oDoc.Close($iSaveOptions)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocSave
; Description ...: Save the document
; Syntax ........: _Photoshop_Save($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocSave($oDoc)
	$oDoc.Save()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocSaveAs
; Description ...: Save the document
; Syntax ........: _Photoshop_DocSaveAs($oDoc, $sPath[, $oOptions = Default[, $bAsCopy = Default[, $iExtensionType = Default]]])
; Parameters ....: $oDoc                - Document object
;                  $sPath               - Output file path
;                  $oOptions            - [optional] SaveOptions object (create using _Photoshop_Create*SaveOptions function)
;                  $bAsCopy             - [optional] Save as a copy
;                  $iExtensionType      - [optional] PsExtensionType value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocSaveAs($oDoc, $sPath, $oOptions = Default, $bAsCopy = Default, $iExtensionType = Default)
	$oDoc.SaveAs($sPath, $oOptions, $bAsCopy, $iExtensionType)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocResize
; Description ...: Resize the document
; Syntax ........: _Photoshop_DocResize($oDoc[, $fWidth = Default[, $fHeight = Default[, $fResolution = Default[, $iResampleMethod = Default[, $fAmount = Default]]]]])
; Parameters ....: $oDoc                - Document object
;                  $fWidth              - [optional] Output width
;                  $fHeight             - [optional] Output height
;                  $fResolution         - [optional] Output resolution
;                  $iResampleMethod     - [optional] PsResampleMethod value
;                  $fAmount             - [optional] Resizing amount
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocResize($oDoc, $fWidth = Default, $fHeight = Default, $fResolution = Default, $iResampleMethod = Default, $fAmount = Default)
	$oDoc.ResizeImage($fWidth, $fHeight, $fResolution, $iResampleMethod, $fAmount)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocResizeCanvas
; Description ...: Resize the document canvas
; Syntax ........: _Photoshop_DocResizeCanvas($oDoc[, $fWidth = Default[, $fHeight = Default[, $iAnchorPos = Default]]])
; Parameters ....: $oDoc                - Document object
;                  $fWidth              - [optional] Output width
;                  $fHeight             - [optional] Output height
;                  $iAnchorPos          - [optional] PsAnchorPosition value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocResizeCanvas($oDoc, $fWidth = Default, $fHeight = Default, $iAnchorPos = Default)
	$oDoc.ResizeCanvas($fWidth, $fHeight, $iAnchorPos)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocFlatten
; Description ...: Flatten all layers
; Syntax ........: _Photoshop_DocFlatten($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocFlatten($oDoc)
	$oDoc.Flatten()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocMergeVisibleLayers
; Description ...: Flatten all visible layers
; Syntax ........: _Photoshop_DocFlatten($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocMergeVisibleLayers($oDoc)
	$oDoc.MergeVisibleLayers()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocPaste
; Description ...: Paste the contents of the clipboard into the document
; Syntax ........: _Photoshop_DocPaste($oDoc[, $bIntoSelection = False])
; Parameters ....: $oDoc                - Document object
;                  $bIntoSelection      - [optional] Paste data into the selection
; Return values .: ArtLayer object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocPaste($oDoc, $bIntoSelection = False)
	Return $oDoc.Paste($bIntoSelection)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocRotateCanvas
; Description ...: Rotates the document by a defined angle in clockwise direction
; Syntax ........: _Photoshop_DocRotateCanvas($oDoc[, $fAngle = 0])
; Parameters ....: $oDoc                - Document object
;                  $fAngle              - [optional] Angle of rotation
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocRotateCanvas($oDoc, $fAngle = 0)
	$oDoc.RotateCanvas($fAngle)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocRevealAll
; Description ...: Expand the document to show clipped sections.
; Syntax ........: _Photoshop_DocRevealAll($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocRevealAll($oDoc)
	$oDoc.RevealAll()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_CreateBMPSaveOptions
; Description ...: Create BMP save options for use in _Photoshop_DocSaveAs
; Syntax ........: _Photoshop_CreateBMPSaveOptions([$bAlphaChannels = Default[, $iDepth = Default[, $bFlipRowOrder = Default[, $iOS = Default[, $bRLECompress = Default]]]]])
; Parameters ....: $bAlphaChannels      - [optional] Save alpha channels
;                  $iDepth              - [optional] PsBMPDepthType value
;                  $bFlipRowOrder       - [optional] Flips the row order. Available only when $iOS = $PsWindows
;                  $iOS                 - [optional] PsOperatingSystem value
;                  $bRLECompress        - [optional] Use RLE compression
; Return values .: BMPSaveOptions object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_CreateBMPSaveOptions($bAlphaChannels = Default, $iDepth = Default, $bFlipRowOrder = Default, $iOS = Default, $bRLECompress = Default)
	Local $oObj = ObjCreate("Photoshop.BMPSaveOptions")

	If $bAlphaChannels <> Default Then $oObj.AlphaChannels = $bAlphaChannels
	If $iDepth <> Default Then $oObj.Depth = $iDepth
	If $bFlipRowOrder <> Default Then $oObj.FlipRowOrder = $bFlipRowOrder
	If $iOS <> Default Then $oObj.OSType = $iOS
	If $bRLECompress <> Default Then $oObj.RLECompression = $bRLECompress

	Return $oObj
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Phoroshop_CreateGIFSaveOptions
; Description ...: Create GIF save options for use in _Photoshop_DocSaveAs
; Syntax ........: _Phoroshop_CreateGIFSaveOptions([$iColors = Default[, $iDither = Default[, $iDitherAmount = 75[, $iForced = Default[, $bInterlaced = False[, $iMatte = $PsWhiteMatte[, $iPalette = $PsLocalSelective[, $bPreserveExactColors = Default[, $bTransparency = Default]]]]]]]]])
; Parameters ....: $iColors             - [optional] Number of palette colors. Invalid, when $iPalette = $PsExact
;                  $iDither             - [optional] PsDitherType value
;                  $iDitherAmount       - [optional] Amount of dither in range 1-100
;                  $iForced             - [optional] PsForcedColors value
;                  $bInterlaced         - [optional] Interlace rows
;                  $iMatte              - [optional] PsMatteType value
;                  $iPalette            - [optional] PsPaletteType value
;                  $bPreserveExactColors- [optional] Protect colors on the image when using dithering (only with $iDither = $PsDiffusion)
;                  $bTransparency       - [optional] Preserve transparent areas
; Return values .: GIFSaveOptions object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Phoroshop_CreateGIFSaveOptions($iColors = Default, $iDither = Default, $iDitherAmount = 75, $iForced = Default, $bInterlaced = False, $iMatte = $PsWhiteMatte, $iPalette = $PsLocalSelective, $bPreserveExactColors = Default, $bTransparency = Default)
	Local $oObj = ObjCreate("Photoshop.GIFSaveOptions")

	$oObj.DitherAmount = $iDitherAmount
	$oObj.Interlaced = $bInterlaced
	$oObj.Matte = $iMatte
	$oObj.Palette = $iPalette

	If $iColors <> Default Then $oObj.Colors = $iColors
	If $iDither <> Default Then $oObj.Dither = $iDither
	If $iForced <> Default Then $oObj.Forced = $iForced
	If $bPreserveExactColors <> Default Then $oObj.PreserveExactColors = $bPreserveExactColors
	If $bTransparency <> Default Then $oObj.Transparency = $bTransparency

	Return $oObj
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_CreateJPEGSaveOptions
; Description ...: Create JPG/JPEG save options for use in _Photoshop_DocSaveAs
; Syntax ........: _Photoshop_CreateJPEGSaveOptions([$bEmbedColorProfile = Default[, $iFormatOptions = $PsStandardBaseline[, $iMatte = $PsWhiteMatte[, $iQuality = 3[, $iScans = 3]]]]])
; Parameters ....: $bEmbedColorProfile  - [optional] Embed the color profile in the document
;                  $iFormatOptions      - [optional] PsFormatOptionsType value
;                  $iMatte              - [optional] PsMatteType value
;                  $iQuality            - [optional] Image quality in range 0-12
;                  $iScans              - [optional] Number of scans to make to incrementally display the image on the page
;                                         in range 3-5. Used only when $iFormatOptions = $PsProgressive
; Return values .: JPEGSaveOptions object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_CreateJPEGSaveOptions($bEmbedColorProfile = Default, $iFormatOptions = $PsStandardBaseline, $iMatte = $PsWhiteMatte, $iQuality = 3, $iScans = 3)
	Local $oObj = ObjCreate("Photoshop.JPEGSaveOptions")

	$oObj.FormatOptions = $iFormatOptions
	$oObj.Matte = $iMatte
	$oObj.Quality = $iQuality
	If $iFormatOptions = $PsProgressive Then $oObj.Scans = $iScans

	If $bEmbedColorProfile <> Default Then $oObj.EmbedColorProfile = $bEmbedColorProfile

	Return $oObj
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_CreatePNGSaveOptions
; Description ...: Create PNG save options for use in _Photoshop_DocSaveAs
; Syntax ........: _Photoshop_CreatePNGSaveOptions([$iCompression = 0[, $bInterlaced = False]])
; Parameters ....: $iCompression        - [optional] Compression strength in range 0-9
;                  $bInterlaced         - [optional] Interlace rows
; Return values .: PNGSaveOptions object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_CreatePNGSaveOptions($iCompression = 0, $bInterlaced = False)
	Local $oObj = ObjCreate("Photoshop.PNGSaveOptions")

	$oObj.Interlaced = $bInterlaced
	$oObj.Compression = $iCompression

	Return $oObj
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocAddLayer
; Description ...: Create a layer in the document
; Syntax ........: _Photoshop_DocAddLayer($oDoc[, $sName = Default[, $iBlendMode = $PsNormalBlend[, $fOpacity = 100[, $fFillOpacity = 100[, $bVisible = True]]]]])
; Parameters ....: $oDoc                - Document object
;                  $sName               - [optional] Layer name
;                  $iBlendMode          - [optional] PsBlendMode value
;                  $fOpacity            - [optional] Layer opacity
;                  $fFillOpacity        - [optional] Layer interior opacity
;                  $bVisible            - [optional] Layer visibility
; Return values .: ArtLayer object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocAddLayer($oDoc, $sName = Default, $iBlendMode = $PsNormalBlend, $fOpacity = 100, $fFillOpacity = 100, $bVisible = True)
	Local $oObj = $oDoc.ArtLayers.Add()

	If $sName <> Default Then $oObj.Name = $sName
	$oObj.BlendMode = $iBlendMode
	$oObj.Opacity = $fOpacity
	$oObj.FillOpacity = $fFillOpacity
	$oObj.Visible = $bVisible

	Return $oObj
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetLayer
; Description ...: Get a layer from the document
; Syntax ........: _Photoshop_DocGetLayer($oDoc, $iIndex)
; Parameters ....: $oDoc                - Document object
;                  $iIndex              - Layer index
; Return values .: ArtLayer object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetLayer($oDoc, $iIndex)
	Return $oDoc.Layers.Item($iIndex)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetLayerCount
; Description ...: Get layer count in a document
; Syntax ........: _Photoshop_DocGetLayerCount($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Layer or LayerSet count
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetLayerCount($oDoc)
	Return $oDoc.Layers.Count
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetLayers
; Description ...: Get a layer collection for using in For .. In .. Next loops
; Syntax ........: _Photoshop_DocGetLayers($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: Layers collection object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetLayers($oDoc)
	Return $oDoc.Layers
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerSetGetLayer
; Description ...: Get a layer from the layer set
; Syntax ........: _Photoshop_LayerSetGetLayer($oDoc, $iIndex)
; Parameters ....: $oDoc                - LayerSet object
;                  $iIndex              - Layer index
; Return values .: ArtLayer or LayerSet object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerSetGetLayer($oLayerSet, $iIndex)
	Return $oLayerSet.Layers.Item($iIndex)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerSetGetLayerCount
; Description ...: Get layer count in a layer set
; Syntax ........: _Photoshop_LayerSetGetLayerCount($oDoc)
; Parameters ....: $oDoc                - LayerSet object
; Return values .: Layer count
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerSetGetLayerCount($oLayerSet)
	Return $oLayerSet.Layers.Count
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerSetGetLayers
; Description ...: Get a layer collection for using in For .. In .. Next loops
; Syntax ........: _Photoshop_LayerSetGetLayers($oDoc)
; Parameters ....: $oDoc                - LayerSet object
; Return values .: Layers collection object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerSetGetLayers($oLayerSet)
	Return $oLayerSet.Layers
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocRemoveAllLayers
; Description ...: Remove all layers in a document
; Syntax ........: _Photoshop_DocRemoveAllLayers($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocRemoveAllLayers($oDoc)
	$oDoc.Layers.RemoveAll()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetActiveLayer
; Description ...: Get the active layer
; Syntax ........: _Photoshop_DocGetActiveLayer($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: ArtLayer or LayerSet object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetActiveLayer($oDoc)
	Return $oDoc.ActiveLayer
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocSetActiveLayer
; Description ...: Set the active layer
; Syntax ........: _Photoshop_DocSetActiveLayer($oDoc, $oLayer)
; Parameters ....: $oDoc                - Document object
;                  $oLayer              - ArtLayer or LayerSet object
; Return values .: Previously active layer
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocSetActiveLayer($oDoc, $oLayer)
	Local $oPrev = $oDoc.ActiveLayer
	$oDoc.ActiveLayer = $oLayer
	Return $oPrev
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerGetName
; Description ...: Get layer name
; Syntax ........: _Photoshop_LayerGetName($oLayer)
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
; Return values .: Layer name
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerGetName($oLayer)
	Return $oLayer.Name
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerSetName
; Description ...: Set layer name
; Syntax ........: _Photoshop_LayerSetName($oLayer, $sName)
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
;                  $sName               - New layer name
; Return values .: Previous layer name
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerSetName($oLayer, $sName)
	Local $sPrev = $oLayer.Name
	$oLayer.Name = $sName
	Return $sPrev
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerGetBlendMode
; Description ...: Get layer blending mode
; Syntax ........: _Photoshop_LayerGetBlendMode($oLayer)
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
; Return values .: PsBlendMode value
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerGetBlendMode($oLayer)
	Return $oLayer.BlendMode
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerSetBlendMode
; Description ...: Set layer blending mode
; Syntax ........: _Photoshop_LayerSetBlendMode($oLayer[, $iMode = $PsNormalBlend])
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
;                  $iMode               - [optional] PsBlendMode value
; Return values .: Previous blending mode
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerSetBlendMode($oLayer, $iMode = $PsNormalBlend)
	Local $iPrev = $oLayer.BlendMode
	$oLayer.BlendMode = $iMode
	Return $iPrev
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerGetOpacity
; Description ...: Get layer opacity
; Syntax ........: _Photoshop_LayerGetOpacity($oLayer)
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
; Return values .: Layer opacity
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerGetOpacity($oLayer)
	Return $oLayer.Opacity
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerSetOpacity
; Description ...: Set layer opacity
; Syntax ........: _Photoshop_LayerSetOpacity($oLayer[, $fOpacity = 100])
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
;                  $fOpacity            - [optional] New layer opacity
; Return values .: Previous layer opacity
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerSetOpacity($oLayer, $fOpacity = 100)
	Local $fPrev = $oLayer.Opacity
	$oLayer.Opacity = $fOpacity
	Return $fPrev
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerGetFillOpacity
; Description ...: Get layer interior opacity
; Syntax ........: _Photoshop_ArtLayerGetFillOpacity($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: Layer opacity
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerGetFillOpacity($oLayer)
	Return $oLayer.FillOpacity
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerSetFillOpacity
; Description ...: Set layer interior opacity
; Syntax ........: _Photoshop_ArtLayerSetFillOpacity($oLayer[, $fOpacity = 100])
; Parameters ....: $oLayer              - ArtLayer object
;                  $fOpacity            - [optional] New layer interior opacity
; Return values .: Previous layer interior opacity
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerSetFillOpacity($oLayer, $fFillOpacity = 100)
	Local $fPrev = $oLayer.FillOpacity
	$oLayer.FillOpacity = $fFillOpacity
	Return $fPrev
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerGetBounds
; Description ...: Get layer bounding rectangle
; Syntax ........: _Photoshop_LayerGetBounds($oLayer)
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
; Return values .: Layer bounding rectangle array
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerGetBounds($oLayer)
	Return $oLayer.Bounds
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerIsVisible
; Description ...: Check whether a layer is visible
; Syntax ........: _Photoshop_LayerIsVisible($oLayer)
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
; Return values .: True - the layer is visible, False otherwise
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerIsVisible($oLayer)
	Return $oLayer.Visible
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerSetVisible
; Description ...: Set layer visibility
; Syntax ........: _Photoshop_LayerSetVisible($oLayer[, $bVisible = True])
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
;                  $bVisible            - [optional] Set visibility to 'visible'
; Return values .: Previous visibility
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerSetVisible($oLayer, $bVisible = True)
	Local $bPrev = $oLayer.Visible
	$oLayer.Visible = $bVisible
	Return $bPrev
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerGetBoundsNoEffects
; Description ...: Get layer bounding rectangle not including effects
; Syntax ........: _Photoshop_ArtLayerGetBoundsNoEffects($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: Layer bounding rectangle array
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerGetBoundsNoEffects($oLayer)
	Return $oLayer.BoundsNoEffects
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerIsBackgroundLayer
; Description ...: Check whether the layer is a background layer
; Syntax ........: _Photoshop_ArtLayerIsBackgroundLayer($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: Boolean value: True - the layer is a background layer, False otherwise
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerIsBackgroundLayer($oLayer)
	Return $oLayer.IsBackgroundLayer
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerAdjustBrightnessContrast
; Description ...: Apply the Brightness & Contrast adjustment
; Syntax ........: _Photoshop_ArtLayerAdjustBrightnessContrast($oLayer[, $iBrightness = 0[, $iContrast = 0]])
; Parameters ....: $oLayer              - ArtLayer object
;                  $iBrightness         - [optional] Brightness in range -100 to 100
;                  $iContrast           - [optional] Contrast in range -100 to 100
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerAdjustBrightnessContrast($oLayer, $iBrightness = 0, $iContrast = 0)
	$oLayer.AdjustBrightnessContrast($iBrightness, $iContrast)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerAdjustColorBalance
; Description ...: Apply the Color Balance adjustment
; Syntax ........: _Photoshop_ArtLayerAdjustColorBalance($oLayer[, $iShadowsCyanRed = 0[, $iShadowsMagentaGreen = 0[, $iShadowsYellowBlue = 0[, $iMidtonesCyanRed = 0[, $iMidtonesMagentaGreen = 0[, $iMidtonesYellowBlue = 0[, $iHighlightsCyanRed = 0[, $iHighlightsMagentaGreen = 0[, $iHighlightsYellowBlue = 0[, $bPreserveLuminosity = True]]]]]]]]]])
; Parameters ....: $oLayer              - ArtLayer object
;                  $iShadowsCyanRed     - [optional] Amount of red shadows in range -100 to 100
;                  $iShadowsMagentaGreen- [optional] Amount of green shadows in range -100 to 100
;                  $iShadowsYellowBlue  - [optional] Amount of blue shadows in range -100 to 100
;                  $iMidtonesCyanRed    - [optional] Amount of red midtones in range -100 to 100
;                  $iMidtonesMagentaGreen- [optional] Amount of green midtones in range -100 to 100
;                  $iMidtonesYellowBlue - [optional] Amount of blue midtones in range -100 to 100
;                  $iHighlightsCyanRed  - [optional] Amount of red highlights in range -100 to 100
;                  $iHighlightsMagentaGreen- [optional] Amount of green highlights in range -100 to 100
;                  $iHighlightsYellowBlue- [optional] Amount of blue highlights in range -100 to 100
;                  $bPreserveLuminosity - [optional] Preserve the lightness of the layer
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerAdjustColorBalance($oLayer, $iShadowsCyanRed = 0, $iShadowsMagentaGreen = 0, $iShadowsYellowBlue = 0, $iMidtonesCyanRed = 0, $iMidtonesMagentaGreen = 0, $iMidtonesYellowBlue = 0, $iHighlightsCyanRed = 0, $iHighlightsMagentaGreen = 0, $iHighlightsYellowBlue = 0, $bPreserveLuminosity = True)
	Local $aShadows = [$iShadowsCyanRed, $iShadowsMagentaGreen, $iShadowsYellowBlue]
	Local $aMidtones = [$iMidtonesCyanRed, $iMidtonesMagentaGreen, $iMidtonesYellowBlue]
	Local $aHighlights = [$iHighlightsCyanRed, $iHighlightsMagentaGreen, $iHighlightsYellowBlue]

	$oLayer.AdjustColorBalance($aShadows, $aMidtones, $aHighlights, $bPreserveLuminosity)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerAdjustLevels
; Description ...: Apply the Levels adjustment
; Syntax ........: _Photoshop_ArtLayerAdjustLevels($oLayer[, $iInputRangeStart = 0[, $iInputRangeEnd = 255[, $fInputRangeGamma = 1[, $iOutputRangeStart = 0[, $iOutputRangeEnd = 255]]]]])
; Parameters ....: $oLayer              - ArtLayer object
;                  $iInputRangeStart    - [optional] Input color range start in range 0 to 253
;                  $iInputRangeEnd      - [optional] Input color range end in range ($iInputRangeStart+2) to 255
;                  $fInputRangeGamma    - [optional] Input color gamma in range 0.1 to 9.99
;                  $iOutputRangeStart   - [optional] Output color range start in range 0 to 253
;                  $iOutputRangeEnd     - [optional] Output color range end in range ($iOutputRangeStart+2) to 255
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerAdjustLevels($oLayer, $iInputRangeStart = 0, $iInputRangeEnd = 255, $fInputRangeGamma = 1, $iOutputRangeStart = 0, $iOutputRangeEnd = 255)
	$oLayer.AdjustLevels($iInputRangeStart, $iInputRangeEnd, $fInputRangeGamma, $iOutputRangeStart, $iOutputRangeEnd)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyAddNoise
; Description ...: Apply the Add Noise filter
; Syntax ........: _Photoshop_ArtLayerApplyAddNoise($oLayer, $fAmount[, $iDistribution = $PsUniformNoise[, $bMonochromatic = False]])
; Parameters ....: $oLayer              - ArtLayer object
;                  $fAmount             - Amount of noise in range 0.1 to 400
;                  $iDistribution       - [optional] PsNoiseDistribution value
;                  $bMonochromatic      - [optional] Create monochromatic noise
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyAddNoise($oLayer, $fAmount, $iDistribution = $PsUniformNoise, $bMonochromatic = False)
	$oLayer.ApplyAddNoise($fAmount, $iDistribution, $bMonochromatic)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyAverage
; Description ...: Apply the Average filter
; Syntax ........: _Photoshop_ArtLayerApplyAverage($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyAverage($oLayer)
	$oLayer.ApplyAverage()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyBlur
; Description ...: Apply the Blur filter
; Syntax ........: _Photoshop_ArtLayerApplyBlur($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyBlur($oLayer)
	$oLayer.ApplyBlur()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyBlurMore
; Description ...: Apply the Blur More filter
; Syntax ........: _Photoshop_ArtLayerApplyBlurMore($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyBlurMore($oLayer)
	$oLayer.ApplyBlurMore()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyClouds
; Description ...: Apply the Clouds filter
; Syntax ........: _Photoshop_ArtLayerApplyClouds($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyClouds($oLayer)
	$oLayer.ApplyClouds()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyDespeckle
; Description ...: Apply the Despeckle filter
; Syntax ........: _Photoshop_ArtLayerApplyDespeckle($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyDespeckle($oLayer)
	$oLayer.ApplyDespeckle()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyDifferenceClouds
; Description ...: Apply the Difference Clouds filter
; Syntax ........: _Photoshop_ArtLayerApplyDifferenceClouds($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyDifferenceClouds($oLayer)
	$oLayer.ApplyDifferenceClouds()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplySharpen
; Description ...: Apply the Sharpen filter
; Syntax ........: _Photoshop_ArtLayerApplySharpen($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplySharpen($oLayer)
	$oLayer.ApplySharpen()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplySharpenEdges
; Description ...: Apply the Sharpen Edges filter
; Syntax ........: _Photoshop_ArtLayerApplySharpenEdges($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplySharpenEdges($oLayer)
	$oLayer.ApplySharpenEdges()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplySharpenMore
; Description ...: Apply the Sharpen More filter
; Syntax ........: _Photoshop_ArtLayerApplySharpenMore($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplySharpenMore($oLayer)
	$oLayer.ApplySharpenMore()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyDeInterlace
; Description ...: Apply the De-Interlace filter
; Syntax ........: _Photoshop_ArtLayerApplyDeInterlace($oLayer, $iEliminateFields, $iCreateFields)
; Parameters ....: $oLayer              - ArtLayer object
;                  $iEliminateFields    - PsEliminateFields value
;                  $iCreateFields       - PsCreateFields value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyDeInterlace($oLayer, $iEliminateFields, $iCreateFields)
	$oLayer.ApplyDeInterlace($iEliminateFields, $iCreateFields)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyDiffuseGlow
; Description ...: Apply the Diffuse Glow filter
; Syntax ........: _Photoshop_ArtLayerApplyDiffuseGlow($oLayer[, $iGraininess = 0[, $iGlowAmount = 0[, $iClearAmount = 0]]])
; Parameters ....: $oLayer              - ArtLayer object
;                  $iGraininess         - [optional] Graininess amount in range 0-10
;                  $iGlowAmount         - [optional] Glow amount in range 0-20
;                  $iClearAmount        - [optional] Clear amount in range 0-10
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyDiffuseGlow($oLayer, $iGraininess = 0, $iGlowAmount = 0, $iClearAmount = 0)
	$oLayer.ApplyDiffuseGlow($iGraininess, $iGlowAmount, $iClearAmount)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyGaussianBlur
; Description ...: Apply the Gaussian Blur filter
; Syntax ........: _Photoshop_ArtLayerApplyGaussianBlur($oLayer, $fRadius)
; Parameters ....: $oLayer              - ArtLayer object
;                  $fRadius             - Blur radius in range 0.1-250
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyGaussianBlur($oLayer, $fRadius)
	$oLayer.ApplyGaussianBlur($fRadius)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyHighPass
; Description ...: Apply the High Pass filter
; Syntax ........: _Photoshop_ArtLayerApplyHighPass($oLayer, $fRadius)
; Parameters ....: $oLayer              - ArtLayer object
;                  $fRadius             - Filter radius in range 0.1-250
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyHighPass($oLayer, $fRadius)
	$oLayer.ApplyHighPass($fRadius)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyMaximum
; Description ...: Apply the Maximum filter
; Syntax ........: _Photoshop_ArtLayerApplyMaximum($oLayer, $fRadius)
; Parameters ....: $oLayer              - ArtLayer object
;                  $fRadius             - Filter radius in range 1-100
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyMaximum($oLayer, $fRadius)
	$oLayer.ApplyMaximum($fRadius)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyMedian
; Description ...: Apply the Median filter
; Syntax ........: _Photoshop_ArtLayerApplyMedian($oLayer, $fRadius)
; Parameters ....: $oLayer              - ArtLayer object
;                  $fRadius             - Filter radius in range 1-100
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyMedian($oLayer, $fRadius)
	$oLayer.ApplyMedianNoise($fRadius)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyMinimum
; Description ...: Apply the Minimum filter
; Syntax ........: _Photoshop_ArtLayerApplyMinimum($oLayer, $fRadius)
; Parameters ....: $oLayer              - ArtLayer object
;                  $fRadius             - Filter radius in range 1-100
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyMinimum($oLayer, $fRadius)
	$oLayer.ApplyMinimum($fRadius)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyOceanRipple
; Description ...: Apply the Ocean Ripple filter
; Syntax ........: _Photoshop_ArtLayerApplyOceanRipple($oLayer, $iSize, $iMagnitude)
; Parameters ....: $oLayer              - ArtLayer object
;                  $iSize               - Ripple size in range 1-15
;                  $iMagnitude          - Ripple magnitude in range 0-20
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyOceanRipple($oLayer, $iSize, $iMagnitude)
	$oLayer.ApplyOceanRipple($iSize, $iMagnitude)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyPinch
; Description ...: Apply the Pinch filter
; Syntax ........: _Photoshop_ArtLayerApplyPinch($oLayer[, $iAmount = 0])
; Parameters ....: $oLayer              - ArtLayer object
;                  $iAmount             - [optional] Pinch amount in range -100 to 100
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyPinch($oLayer, $iAmount = 0)
	$oLayer.ApplyPinch($iAmount)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyPolarCoordinates
; Description ...: Apply the Polar Coordinates filter
; Syntax ........: _Photoshop_ArtLayerApplyPolarCoordinates($oLayer, $iConversion)
; Parameters ....: $oLayer              - ArtLayer object
;                  $iConversion         - PsPolarConversionType value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyPolarCoordinates($oLayer, $iConversion)
	$oLayer.ApplyPolarCoordinates($iConversion)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyRipple
; Description ...: Apply the Ripple filter
; Syntax ........: _Photoshop_ArtLayerApplyRipple($oLayer, $iAmount, $iSize)
; Parameters ....: $oLayer              - ArtLayer object
;                  $iAmount             - Rippling amount in range -999 to 999
;                  $iSize               - PsRippleSize value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyRipple($oLayer, $iAmount, $iSize)
	$oLayer.ApplyRipple($iAmount, $iSize)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplySpherize
; Description ...: Apply the Spherize filter
; Syntax ........: _Photoshop_ArtLayerApplySpherize($oLayer, $iAmount[, $iMode = $PsNormalSpherize])
; Parameters ....: $oLayer              - ArtLayer object
;                  $iAmount             - Spherization amount in range -100 to 100
;                  $iMode               - [optional] PsSpherizeMode value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplySpherize($oLayer, $iAmount, $iMode = $PsNormalSpherize)
	$oLayer.ApplySpherize($iAmount, $iMode)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyTwirl
; Description ...: Apply the Twirl filter
; Syntax ........: _Photoshop_ArtLayerApplyTwirl($oLayer[, $iAngle = 0])
; Parameters ....: $oLayer              - ArtLayer object
;                  $iAngle              - [optional] Twirl angle in range -999 to 999
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyTwirl($oLayer, $iAngle = 0)
	$oLayer.ApplyTwirl($iAngle)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerApplyZigZag
; Description ...: Apply the Zigzag filter
; Syntax ........: _Photoshop_ArtLayerApplyZigZag($oLayer, $iAmount, $iRidges, $iStyle)
; Parameters ....: $oLayer              - ArtLayer object
;                  $iAmount             - Distortion amount in range -100 to 100
;                  $iRidges             - Number of ridges in range 0-20
;                  $iStyle              - PsZigZagType value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerApplyZigZag($oLayer, $iAmount, $iRidges, $iStyle)
	$oLayer.ApplyZigZag($iAmount, $iRidges, $iStyle)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerAutoContrast
; Description ...: Automatically adjust the contrast of the layer
; Syntax ........: _Photoshop_ArtLayerAutoContrast($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerAutoContrast($oLayer)
	$oLayer.AutoContrast()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerAutoLevels
; Description ...: Automatically adjust the levels of the layer
; Syntax ........: _Photoshop_ArtLayerAutoLevels($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerAutoLevels($oLayer)
	$oLayer.AutoLevels()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerDesaturate
; Description ...: Desaturate the layer
; Syntax ........: _Photoshop_ArtLayerDesaturate($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerDesaturate($oLayer)
	$oLayer.Desaturate()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerEqualize
; Description ...: Equalize the brightness of the layer
; Syntax ........: _Photoshop_ArtLayerEqualize($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerEqualize($oLayer)
	$oLayer.Equalize()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerInvert
; Description ...: Invert the colors of the layer
; Syntax ........: _Photoshop_ArtLayerInvert($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerInvert($oLayer)
	$oLayer.Invert()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerMergeDown
; Description ...: Merge the layer with a layer below
; Syntax ........: _Photoshop_LayerMergeDown($oLayer)
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
; Return values .: Merged layer (ArtLayer object)
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerMergeDown($oLayer)
	Return $oLayer.Merge()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerPosterize
; Description ...: Posterize the layer
; Syntax ........: _Photoshop_ArtLayerPosterize($oLayer, $iLevels)
; Parameters ....: $oLayer              - ArtLayer object
;                  $iLevels             - Number of tonal levels
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerPosterize($oLayer, $iLevels)
	$oLayer.Posterize($iLevels)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerTranslate
; Description ...: Move the layer relative to its current position
; Syntax ........: _Photoshop_LayerTranslate($oLayer, $fX, $fY)
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
;                  $fX                  - X coordinate
;                  $fY                  - Y coordinate
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerTranslate($oLayer, $fX, $fY)
	$oLayer.Translate($fX, $fY)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerRotate
; Description ...: Rotate the layer
; Syntax ........: _Photoshop_LayerRotate($oLayer, $fAngle[, $iAnchor = $PsMiddleCenter])
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
;                  $fAngle              - Angle of rotation in degrees
;                  $iAnchor             - [optional] PsAnchorPosition value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerRotate($oLayer, $fAngle = 0, $iAnchor = $PsMiddleCenter)
	$oLayer.Rotate($fAngle, $iAnchor)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerResize
; Description ...: Resize the layer
; Syntax ........: _Photoshop_LayerResize($oLayer[, $fHorz = 100[, $fVert = 100[, $iAnchor = $PsMiddleCenter]]])
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
;                  $fHorz               - [optional] Horizontal resize in percent
;                  $fVert               - [optional] Vertical resize in percent
;                  $iAnchor             - [optional] PsAnchorPosition value
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerResize($oLayer, $fHorz = 100, $fVert = 100, $iAnchor = $PsMiddleCenter)
	$oLayer.Resize($fHorz, $fVert, $iAnchor)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_LayerDelete
; Description ...: Delete the layer
; Syntax ........: _Photoshop_LayerDelete($oLayer)
; Parameters ....: $oLayer              - ArtLayer or LayerSet object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_LayerDelete($oLayer)
	$oLayer.Delete()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ArtLayerCopy
; Description ...: Copy the layer into the clipboard
; Syntax ........: _Photoshop_ArtLayerCopy($oLayer)
; Parameters ....: $oLayer              - ArtLayer object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ArtLayerCopy($oLayer)
	$oLayer.Copy()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetSelection
; Description ...: Get the selection of the document
; Syntax ........: _Photoshop_DocGetSelection($oDoc)
; Parameters ....: $oDoc                - Document
; Return values .: Selection object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetSelection($oDoc)
	Return $oDoc.Selection
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_SelectionSelectAll
; Description ...: Select the whole document
; Syntax ........: _Photoshop_SelectionSelectAll($oSel)
; Parameters ....: $oSel                - Selection object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_SelectionSelectAll($oSel)
	$oSel.SelectAll()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_SelectionDeselect
; Description ...: Deselect the selection
; Syntax ........: _Photoshop_SelectionDeselect($oSel)
; Parameters ....: $oSel                - Selection object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_SelectionDeselect($oSel)
	$oSel.Deselect()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_SelectionInvert
; Description ...: Invert the selection
; Syntax ........: _Photoshop_SelectionInvert($oSel)
; Parameters ....: $oSel                - Selection object
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_SelectionInvert($oSel)
	$oSel.Invert()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_SelectionFill
; Description ...: Fill the selection
; Syntax ........: _Photoshop_SelectionFill($oSel, $oFill[, $iMode = $PsNormalBlendColor[, $iOpacity = 100[, $bPreserveTransparency = False]]])
; Parameters ....: $oSel                - Selection object
;                  $oFill               - SolidColor or HistoryState object
;                  $iMode               - [optional] PsColorBlendMode value
;                  $iOpacity            - [optional] Fill opacity
;                  $bPreserveTransparency- [optional] Preserve source transparency
; Return values .: None
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_SelectionFill($oSel, $oFill, $iMode = $PsNormalBlendColor, $iOpacity = 100, $bPreserveTransparency = False)
	$oSel.Fill($oFill, $iMode, $iOpacity, $bPreserveTransparency)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ColorCreate
; Description ...: Create a RGB color object
; Syntax ........: _Photoshop_ColorCreate([$iRed = 0[, $iGreen = 0[, $iBlue = 0]]])
; Parameters ....: $iRed                - [optional] Red
;                  $iGreen              - [optional] Green
;                  $iBlue               - [optional] Blue
; Return values .: SolidColor object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ColorCreate($iRed = 0, $iGreen = 0, $iBlue = 0)
	Local $oObj = ObjCreate("Photoshop.SolidColor")

	$oObj.RGB.Red = $iRed
	$oObj.RGB.Green = $iGreen
	$oObj.RGB.Blue = $iBlue

	Return $oObj
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ColorGetRGB
; Description ...: Get RGB values of a color
; Syntax ........: _Photoshop_ColorGetRGB($oColor)
; Parameters ....: $oColor              - SolidColor object
; Return values .: 3-item RGB array
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ColorGetRGB($oColor)
	Local $aRet = [$oColor.RGB.Red, $oColor.RGB.Green, $oColor.RGB.Blue]
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_ColorSetRGB
; Description ...: Set RGB values of a color
; Syntax ........: _Photoshop_ColorSetRGB($oColor[, $iRed = 0[, $iGreen = 0[, $iBlue = 0]]])
; Parameters ....: $oColor              - SolidColor object
;                  $iRed                - [optional] Red
;                  $iGreen              - [optional] Green
;                  $iBlue               - [optional] Blue
; Return values .: 3-item RGB array (previous color)
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_ColorSetRGB($oColor, $iRed = 0, $iGreen = 0, $iBlue = 0)
	Local $aPrev = _Photoshop_ColorGetRGB($oColor)

	$oColor.RGB.Red = $iRed
	$oColor.RGB.Green = $iGreen
	$oColor.RGB.Blue = $iBlue

	Return $aPrev
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetHistoryState
; Description ...: Get a HistoryState object from a document
; Syntax ........: _Photoshop_DocGetHistoryState($oDoc, $iIndex)
; Parameters ....: $oDoc                - Document object
;                  $iIndex              - History state index
; Return values .: HistoryState object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetHistoryState($oDoc, $iIndex)
	Return $oDoc.HistoryStates.Item($iIndex)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetHistoryStateCount
; Description ...: Get history state count
; Syntax ........: _Photoshop_DocGetHistoryStateCount($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: History state count
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetHistoryStateCount($oDoc)
	Return $oDoc.HistoryStates.Count
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_DocGetHistoryStates
; Description ...: Get a history state collection for using in For .. In .. Next loops
; Syntax ........: _Photoshop_DocGetHistoryStates($oDoc)
; Parameters ....: $oDoc                - Document object
; Return values .: HistoryStates collection object
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_DocGetHistoryStates($oDoc)
	Return $oDoc.HistoryStates
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_HistoryStateGetName
; Description ...: Get HistoryState name
; Syntax ........: _Photoshop_HistoryStateGetName($oHState)
; Parameters ....: $oHState             - HistoryState object
; Return values .: HistoryState name
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_HistoryStateGetName($oHState)
	Return $oHState.Name
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Photoshop_HistoryStateIsSnapshot
; Description ...: Check whether the HistoryState is a snapshot
; Syntax ........: _Photoshop_HistoryStateIsSnapshot($oHState)
; Parameters ....: $oHState             - HistoryState object
; Return values .: True - HistoryState is a snapshot, False otherwise
; Author ........: scintilla4evr
; ===============================================================================================================================
Func _Photoshop_HistoryStateIsSnapshot($oHState)
	Return $oHState.Snapshot
EndFunc
