#Region - TeamViewer.au3 - Header
; #INDEX# ========================================================================
; Title .........: TeamViewer.au3
; AutoIt Version : 3.3.10.2++
; Language ......: English
; Description ...: A collection of function for use with TeamViewer API
; Author ........: mLipok
; Modified ......:
; URL ...........:
; URL ...........: https://www.teamviewer.com/
; URL ...........: https://www.teamviewer.com/en/integrations/
; URL ...........: https://integrate.teamviewer.com/en/develop/api/get-started/
; URL ...........: https://downloadeu1.teamviewer.com/integrate/TeamViewer_API_Documentation.pdf
; Remarks .......: This UDF was created based on TeamViewer_API_Documentation.pdf v 1.4.1
; Remarks .......: This UDF is using Free Chilkat component look here https://www.autoitscript.com/forum/files/file/433-chilkat-udf/
; Remarks .......: Documentation is "work in progress"
; Date ..........: 2017/02/08
; Version .......: 0.1.1 BETA - Work in progress
; ================================================================================

#cs
	v0.1.1 BETA
	2017/02/08
	.	First official version - mLipok


	@LAST
#ce
#EndRegion - TeamViewer.au3 - Header

#Region - TeamViewer.au3 - Options
#AutoIt3Wrapper_Au3Check_Parameters=-d -w 1 -w 2 -w 3 -w- 4 -w 5 -w 6 -w 7
;~ #AutoIt3Wrapper_Run_Debug_Mode=Y
#Tidy_Parameters=/sort_funcs /reel
#EndRegion - TeamViewer.au3 - Options

#Region - TeamViewer.au3 - Include
#include-once
#include <Array.au3>
#include <FileConstants.au3>
#include "Chilkat.au3"
#EndRegion - TeamViewer.au3 - Include

Global $oErrorHandler = ObjEvent("AutoIt.Error", "_TVAPI_ErrFunc")
Global $__g_sTVAPI_Version = "v1" ; Put the current API version in here. REMARK: only MAJOR NOT MINIOR VERSION This is not API DOC v1.4.1
Global $__g_sTVAPI_BaseUrl = "https://webapi.teamviewer.com" ; URL of the TeamViewer  API

Global Enum _
		$TVAPI_ERR_SUCCESS, _
		$TVAPI_ERR_GENERAL, _
		$TVAPI_ERR_COMERROR, _
		$TVAPI_ERR_STATUS, _
		$TVAPI_ERR_TVAPIARR, _
		$TVAPI_ERR_COUNTER

Global Enum _
		$TVAPI_EXT_DEFAULT, _
		$TVAPI_EXT_PARAM1, _
		$TVAPI_EXT_PARAM2, _
		$TVAPI_EXT_PARAM3, _
		$TVAPI_EXT_COUNTER

Global Enum _
		$TVAPI_RET_SUCCESS, _
		$TVAPI_RET_FAILURE, _
		$TVAPI_RET_COUNTER

Global Enum _
		$TVAPI_ARR_RESET, _
		$TVAPI_ARR_STATUS, _
		$TVAPI_ARR_STATUSTEXT, _
		$TVAPI_ARR_RESPONSETEXT, _
		$TVAPI_ARR_COUNTER

Global Enum _
		$TVAPI_PAR_STRING, _
		$TVAPI_PAR_BOOLEAN, _
		$TVAPI_PAR_COUNTER

#Region - TeamViewer.au3 - Chillkat ActiveX object initialization
__TeamViewer_ChilkatInit()

Func __TeamViewer_ChilkatInit()
	_ErrorLog_ChilkatWrapper(ConsoleWrite)
	_Chilkat_StartUp(@ScriptDir & '\ChilkatAx-9.5.0-win32.dll')
	_Chilkat_DllVersion($CHILKATOBJ_VERSION_950)

;~ 	Local $oGLOBAL = _Chilkat_GLOBAL_ObjCreate()
;~ 	If @error Then MsgBox($MB_ICONERROR, 'GLOBAL Declartion', '@error = ' & @error & @CRLF & '@extended = ' & @extended)

;~ 	_Chilkat_GLOBAL_UnlockBundle($oGLOBAL, "Anything for 30-day trial")
;~ 	If @error Then MsgBox($MB_ICONERROR, 'UnlockBundle', '@error = ' & @error & @CRLF & '@extended = ' & @extended)
EndFunc   ;==>__TeamViewer_ChilkatInit
#EndRegion - TeamViewer.au3 - Chillkat ActiveX object initialization

#Region - TeamViewer.au3 - API functions - Reports - WIP

; #FUNCTION# ====================================================================================================================
; Name ..........: __TVAPI_Reports_GetMoreConnections
; Description ...: Get all connections if there are more dann 1000
; Syntax ........: __TVAPI_Reports_GetMoreConnections(Byref $oJSON)
; Parameters ....: $oJSON               - [in/out] an object.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func __TVAPI_Reports_GetMoreConnections(ByRef $oJSON)
	Local $s_MoreConnUrl = '' ; offset url to get more connections
	Local $oJSON_MoreConn = $oJSON
	Local $a_ConnectArr[0]
	Local $oTemp = Null, $aTemp = 0

	While $oJSON_MoreConn.hasMember("next_offset")
		__TVAPI_DebugOut("More connections found ... " & $oJSON_MoreConn.StringOf("records_remaining"))
		_TVAPI_JSON_AddRecordsTo1DArray($oJSON_MoreConn, $a_ConnectArr)
		$s_MoreConnUrl = "?offset_id=" & $oJSON_MoreConn.StringOf("next_offset")

		__TVAPI_DebugOut(@CRLF)
		__TVAPI_DebugOut("Get more connections...")

		Local $oHTTP = __TVAPI_HTTP_Open("GET", "reports/connections", $s_MoreConnUrl)
		If @error Then
			__TVAPI_DebugError("__TVAPI_Reports_GetMoreConnections request failed")
			Return False
		EndIf

		$oJSON = __TVAPI_HTTP_Send($oHTTP)
		If @error Then Return SetError(@error, @extended, False)

		ReDim $a_ConnectArr[UBound($a_ConnectArr) + 1]
	WEnd

	; writes the last connections in the dictionary
	_TVAPI_JSON_AddRecordsTo1DArray($oJSON_MoreConn, $a_ConnectArr)

	Return $a_ConnectArr
EndFunc   ;==>__TVAPI_Reports_GetMoreConnections

Func _TVAPI_ConvertRecordsTo2DArray(ByRef $aRecords_1D)
	; Is it used ? Needed ?

	Local $oJSON = _Chilkat_JSON_ObjFromString($aRecords_1D[0])
	If @error Then Return SetError(@error, @extended, 0)

	Local $iMembers_cnt = $oJSON.Size
	Local $aRecords_2D[UBound($aRecords_1D)][$iMembers_cnt]

	; Make Header For ListView
	Local $sHeader = ''
	Local $sSeparator = Opt("GUIDataSeparatorChar")
	For $iMember_idx = 0 To $iMembers_cnt - 1
		$sHeader &= $oJSON.NameAt($iMember_idx) & $sSeparator
	Next
	$sHeader = StringTrimRight($sHeader, StringLen($sSeparator))

	For $iRecord_idx = 0 To UBound($aRecords_1D) - 1
		$sHeader &= $oJSON.NameAt($iRecord_idx) & $sSeparator
		For $iMember_idx = 0 To $iMembers_cnt - 1
			$aRecords_2D[$iRecord_idx][$iMember_idx] = $oJSON.StringAt($iMember_idx)
		Next
	Next

	_ArrayDisplay($aRecords_2D, '$aRecords_2D')
EndFunc   ;==>_TVAPI_ConvertRecordsTo2DArray
; Check if API is available and verify token is valid
Func _TVAPI_Ping()
	__TVAPI_DebugOut(@CRLF)
	__TVAPI_DebugOut("_TVAPI_Ping: Pinging API...")
	__TVAPI_DebugOut("Request [GET] /api/" & $__g_sTVAPI_Version & "/ping")

	Local $oHTTP = __TVAPI_HTTP_Open("GET", "ping")
	If @error Then
		__TVAPI_DebugError("_TVAPI_Ping: request failed")
		Return False
	EndIf

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP)
	If @error Then Return SetError(@error, @extended, False)

	If $oJSON.BoolOf('token_valid') = False Then
		__TVAPI_DebugOut("_TVAPI_Ping: Token is invalid")
		Return SetError(5, 0, False)
	EndIf

	__TVAPI_DebugOut("_TVAPI_Ping: Token is valid")
	Return SetError(0, 0, True)
EndFunc   ;==>_TVAPI_Ping

; get all connections of a company with all available fields

; #FUNCTION# ====================================================================================================================
; Name ..........: _TVAPI_Reports_GetConnections
; Description ...: List connection reports
; Syntax ........: _TVAPI_Reports_GetConnections([$sFromDate = Default[, $sToDate = Default[, $sUserName = Default[,
;                  $sUserID = Default[, $sGroupID = Default[, $sDeviceName = Default[, $iDevicelID = Default[,
;                  $bHas_Code = Default[, $sSession_Code = Default ]]]]]]]]])
; Parameters ....: $sFromDate           - [optional] a string value. Default is Default.
;                  $sToDate             - [optional] a string value. Default is Default.
;                  $sUserName           - [optional] a string value. Default is Default.
;                  $sUserID             - [optional] a string value. Default is Default.
;                  $sGroupID            - [optional] a string value. Default is Default.
;                  $sDeviceName         - [optional] a string value. Default is Default.
;                  $iDevicelID          - [optional] an integer value. Default is Default.
;                  $bHas_Code           - [optional] a boolean value. Default is Default.
;                  $sSession_Code       - [optional] a string value. Default is Default .
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _TVAPI_Reports_GetConnections($sFromDate = Default, $sToDate = Default, $sUserName = Default, $sUserID = Default, $sGroupID = Default, $sDeviceName = Default, $iDevicelID = Default, $bHas_Code = Default, $sSession_Code = Default)
	; Example: $sFromDate = '2016-12-01T00:00:01Z'

	Local $sURLParameters = ''
	If IsString($sToDate) Then $sURLParameters &= _
			'from_date=' & $sFromDate
	If IsString($sToDate) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'to_date=' & $sToDate
	If IsString($sUserName) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'username=' & $sUserName
	If IsString($sUserID) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'userid=u' & $sUserID
	If IsString($sGroupID) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'groupid=g' & $sGroupID
	If IsString($sDeviceName) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'devicename=' & $sDeviceName
	If IsInt($iDevicelID) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'deviceid=' & String($iDevicelID)
	If IsBool($bHas_Code) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'has_code=' & $sDeviceName
	If IsString($sSession_Code) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'session_code=' & $sSession_Code
	If $sURLParameters <> '' Then $sURLParameters = '?' & $sURLParameters

	__TVAPI_DebugOut(@CRLF)
	__TVAPI_DebugOut("Get all connections...")
	__TVAPI_DebugOut("Request [GET] /api/" & $__g_sTVAPI_Version & "/reports/connections")

	Local $oHTTP = __TVAPI_HTTP_Open("GET", "reports/connections", $sURLParameters)
	If @error Then Return SetError(@error, @extended, False)

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP)
	If @error Then Return SetError(@error, @extended, False)

	__TVAPI_DebugOut("Request ok!")
	If $oJSON.hasMember("next_offset") Then
		__TVAPI_DebugError("Next_offset")
		; our received, more than 1000, connections as array of dictionaries
		__TVAPI_DebugOut("Found: next_offset")
		Local $aMoreConnections = __TVAPI_Reports_GetMoreConnections($oJSON)
		Return $aMoreConnections
	ElseIf $oJSON.hasMember("records") Then
		__TVAPI_DebugOut("Found: records")
		; our received, less than 1000, connections as array of dictionaries
		Local $a_ConnectArr[0]
		Local $oJSON_MoreConn = $oJSON.ArrayOf("records")
		If @error Then Return SetError(4, 0, False)
		_TVAPI_JSON_AddRecordsTo1DArray($oJSON, $a_ConnectArr)
		Return $a_ConnectArr
	EndIf
	__TVAPI_DebugOut("no more connections received")

EndFunc   ;==>_TVAPI_Reports_GetConnections
#EndRegion - TeamViewer.au3 - API functions - Reports - WIP

#Region - TeamViewer.au3 - API functions - Groups - WIP

Func _TVAPI_Groups_ChangeGroupDetails()
;~ 	PUT /api/v1/groups/<gID> (change group details)
EndFunc   ;==>_TVAPI_Groups_ChangeGroupDetails

Func _TVAPI_Groups_CreateNewGroup()
;~ 	 POST /api/v1/groups (create a new group)
EndFunc   ;==>_TVAPI_Groups_CreateNewGroup

Func _TVAPI_Groups_DeleteGroup()
;~ 	DELETE /api/v1/groups/<gID> (delete a group)
EndFunc   ;==>_TVAPI_Groups_DeleteGroup

Func _TVAPI_Groups_GetGroupDetails()
;~ 	GET /api/v1/groups/<gID> (get group details)
EndFunc   ;==>_TVAPI_Groups_GetGroupDetails

Func _TVAPI_Groups_ListGroups($sName = Default, $bShared = Default)
;~ 	GET /api/v1/groups (retrieve list of groups)
	Local $sURLParameters = ''
	If IsString($sName) Then $sURLParameters &= _
			'name=' & String($sName)
	If IsBool($bShared) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'shared=' & (($bShared) ? "true" : "false")
	If $sURLParameters <> '' Then $sURLParameters = '?' & $sURLParameters

	Local $oHTTP = __TVAPI_HTTP_Open("GET", "groups", $sURLParameters)

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP)
	If @error Then Return SetError(@error, @extended, $oJSON)

	__TVAPI_DebugOut("Request ok!")
	Return SetError($TVAPI_ERR_SUCCESS, $oJSON.Size, $oJSON)

EndFunc   ;==>_TVAPI_Groups_ListGroups

Func _TVAPI_Groups_ShareGroupWithUser()
;~ 	POST /api/v1/groups/<gID>/share_group (share group with other user(s))
EndFunc   ;==>_TVAPI_Groups_ShareGroupWithUser

Func _TVAPI_Groups_UnShareGroupWithUser()
;~ 	POST /api/v1/groups/<gID>/unshare_group (unshare a group from certain users)
EndFunc   ;==>_TVAPI_Groups_UnShareGroupWithUser
#EndRegion - TeamViewer.au3 - API functions - Groups - WIP

#Region - TeamViewer.au3 - API functions - Sessions - WIP

Func _TVAPI_Sessions_Create()
;~ 	POST /api/v1/sessions (create new session code)
EndFunc   ;==>_TVAPI_Sessions_Create

Func _TVAPI_Sessions_GetInfo()
;~ 	 GET /api/v1/sessions/<code> (get info about a certain session code)
EndFunc   ;==>_TVAPI_Sessions_GetInfo
Func _TVAPI_Sessions_List()
;~ 	GET /api/v1/sessions (list session codes)
EndFunc   ;==>_TVAPI_Sessions_List

Func _TVAPI_Sessions_ModifyInfo()
;~ 	 PUT /api/v1/sessions/<code> (modify info for a certain session code)
EndFunc   ;==>_TVAPI_Sessions_ModifyInfo
#EndRegion - TeamViewer.au3 - API functions - Sessions - WIP

#Region - TeamViewer.au3 - API functions - Devices - WIP

Func __TVAPI_Devices_GetWrapper($iRemoteControlID = Default, $iGroupID = Default, $bOnlineState = Default, $iDeviceID = Default)
;~ 	GET /api/v1/devices (list all devices from the computers & contacts list)

	Local $sURLParameters = ''
	If IsInt($iRemoteControlID) Then $sURLParameters &= _
			'remotecontrol_id=r' & String($iRemoteControlID)
	If IsInt($iGroupID) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'groupid=g' & String($iGroupID)
	If IsBool($bOnlineState) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'online_state=' & (($bOnlineState) ? "online" : "offline")
	If $sURLParameters <> '' Then $sURLParameters = '?' & $sURLParameters

	If $iRemoteControlID = Default And $iGroupID And Default And $bOnlineState = Default Then
		If IsInt($iDeviceID) Then
			$sURLParameters = '/d' & String($iDeviceID)
		Else
			; @TODO error
		EndIf
	ElseIf $iDeviceID <> Default Then
		; @TODO error
	EndIf

	Local $oHTTP = __TVAPI_HTTP_Open("GET", "devices", $sURLParameters)
	If @error Then Return SetError(@error, @extended, False)

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP)
	If @error Then Return SetError(@error, @extended, $oJSON)

	__TVAPI_DebugOut("Request ok!")
	Return SetError($TVAPI_ERR_SUCCESS, $oJSON.Size, $oJSON)
EndFunc   ;==>__TVAPI_Devices_GetWrapper

Func _TVAPI_Devices_AddNew($iRemoteControlID, $iGroupID, $sAlias, $sDescription = Default, $sPassword = Default)
;~ 	 POST /api/v1/devices (add a new device)

	Local $oJSON = _Chilkat_JSON_ObjCreate()
	If @error Then Return SetError(@error, @extended, 0)

	If IsInt($iRemoteControlID) And StringLen(String($iRemoteControlID)) = 9 Then _
			$oJSON.AddStringAt(-1, "remotecontrol_id", 'r' & String($iRemoteControlID))
	If IsString($sAlias) And StringLen($sAlias) > 0 Then _
			$oJSON.AddStringAt(-1, "alias", $sAlias)
	If IsString($sDescription) And StringLen($sDescription) > 0 Then _
			$oJSON.AddStringAt(-1, "description", $sDescription)
	If IsString($sPassword) And StringLen($sPassword) > 0 Then _
			$oJSON.AddStringAt(-1, "password", $sPassword)
	If IsInt($iGroupID) And StringLen(String($iGroupID)) = 8 Then _
			$oJSON.AddStringAt(-1, "groupid", "g" & String($iGroupID))

	If $oJSON.size = 0 Then
		; for testing: @TODO error handling
		MsgBox(0, '_TVAPI_Devices_Add_New ', '$oJSON.size=' & $oJSON.size)
	EndIf
	Local $sJSON_String = _Chilkat_JSON_ObjToString($oJSON, 1)

	__TVAPI_DebugOut(@CRLF)
	__TVAPI_DebugOut("Adding new device ( RemoteID= " & $iRemoteControlID & ").")
	__TVAPI_DebugOut("$sJSON_String= " & $sJSON_String)
	Local $oHTTP = __TVAPI_HTTP_Open("POST", "devices")
	$oHTTP.setRequestHeader("Content-Type", "application/json; charset=utf-8")

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP, $sJSON_String)
	If @error Then Return SetError(@error, @extended, $oJSON)

	__TVAPI_DebugOut("! " & _TVAPI_GET_Status() & @CRLF)
	__TVAPI_DebugOut("! " & _TVAPI_GET_StatusText() & @CRLF)
	__TVAPI_DebugOut("! " & _TVAPI_GET_ResponseText() & @CRLF)

	Return SetError($TVAPI_ERR_SUCCESS, Int(StringTrimLeft($oJSON.StringOf("device_id"), 1)), $oJSON)
EndFunc   ;==>_TVAPI_Devices_AddNew

; #FUNCTION# ====================================================================================================================
; Name ..........: _TVAPI_Devices_ChangeDetails
; Description ...: Change single device details
; Syntax ........: _TVAPI_Devices_ChangeDetails($iDeviceID[, $sAlias = Default[, $sDescription = Default[, $sPassword = Default[,
;                  $sPolicyID = Default[, $iGroupID = Default]]]]])
; Parameters ....: $iDeviceID           - an integer value.
;                  $sAlias              - [optional] a string value. Default is Default.
;                  $sDescription        - [optional] a string value. Default is Default.
;                  $sPassword           - [optional] a string value. Default is Default.
;                  $sPolicyID           - [optional] a string value. Default is Default.
;                  $iGroupID            - [optional] an integer value. Default is Default.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _TVAPI_Devices_ChangeDetails($iDeviceID, $sAlias = Default, $sDescription = Default, $sPassword = Default, $sPolicyID = Default, $iGroupID = Default)
;~ 	PUT /api/v1/devices/<dID> (change device details)

	If $sAlias = Default And $sDescription = Default And $sPassword = Default And $sPolicyID = Default And $iGroupID = Default Then
		Return SetError(1) ; @TODO ErrorHandling >> Nofhing to change
	EndIf

	Local $oJSON = _Chilkat_JSON_ObjCreate()
	If @error Then Return SetError(@error, @extended, 0)

	If IsString($sAlias) Then $oJSON.AddStringAt(-1, "alias", $sAlias)
	If IsString($sDescription) Then $oJSON.AddStringAt(-1, "description", $sDescription)
	If IsString($sPassword) Then $oJSON.AddStringAt(-1, "password", $sPassword)
	If IsString($sPolicyID) Then $oJSON.AddStringAt(-1, "policy_id", $sPolicyID)
	If IsInt($iGroupID) Then $oJSON.AddStringAt(-1, "groupid", "g" & String($iGroupID))

	Local $sJSON_String = _Chilkat_JSON_ObjToString($oJSON, 1)

	__TVAPI_DebugOut(@CRLF)
	__TVAPI_DebugOut("Changing device (" & $iDeviceID & ") details")
	Local $oHTTP = __TVAPI_HTTP_Open("PUT", "devices/d" & $iDeviceID)
	$oHTTP.setRequestHeader("Content-Type", "application/json; charset=utf-8")

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP, $sJSON_String)
	If @error Then Return SetError(@error, @extended, $oJSON)

	__TVAPI_DebugOut("! " & _TVAPI_GET_Status() & @CRLF)
	__TVAPI_DebugOut("! " & _TVAPI_GET_StatusText() & @CRLF)
	__TVAPI_DebugOut("! " & _TVAPI_GET_ResponseText() & @CRLF)
EndFunc   ;==>_TVAPI_Devices_ChangeDetails

Func _TVAPI_Devices_Delete($iDeviceID)
;~ 	 POST /api/v1/devices (add a new device)

	__TVAPI_DebugOut(@CRLF)
	__TVAPI_DebugOut("Delete device (" & $iDeviceID & ")")
	Local $oHTTP = __TVAPI_HTTP_Open("DELETE", "devices", "/d" & String($iDeviceID))

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP)
	If @error Then Return SetError(@error, @extended, $oJSON)

	__TVAPI_DebugOut("! " & _TVAPI_GET_Status() & @CRLF)
	__TVAPI_DebugOut("! " & _TVAPI_GET_StatusText() & @CRLF)
	__TVAPI_DebugOut("! " & _TVAPI_GET_ResponseText() & @CRLF)
EndFunc   ;==>_TVAPI_Devices_Delete

; #FUNCTION# ====================================================================================================================
; Name ..........: _TVAPI_Devices_GetDevice
; Description ...: Get Details about Single Device by DeviceID
; Syntax ........: _TVAPI_Devices_GetDevice([$iDeviceID = Default])
; Parameters ....: $iDeviceID           - [optional] an integer value. Default is Default.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _TVAPI_Devices_GetDevice($iDeviceID = Default)
	Local $vResult = __TVAPI_Devices_GetWrapper(Default, Default, Default, $iDeviceID)
	SetError(@error, @extended, $vResult)
EndFunc   ;==>_TVAPI_Devices_GetDevice

; #FUNCTION# ====================================================================================================================
; Name ..........: _TVAPI_Devices_GetDevicesList
; Description ...: List Devices Details
; Syntax ........: _TVAPI_Devices_GetDevicesList([$iRemoteControlID = Default[, $iGroupID = Default[, $bOnlineState = Default]]])
; Parameters ....: $iRemoteControlID    - [optional] an integer value. Default is Default.
;                  $iGroupID            - [optional] an integer value. Default is Default.
;                  $bOnlineState        - [optional] a boolean value. Default is Default.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _TVAPI_Devices_GetDevicesList($iRemoteControlID = Default, $iGroupID = Default, $bOnlineState = Default)
	Local $vResult = __TVAPI_Devices_GetWrapper($iRemoteControlID, $iGroupID, $bOnlineState)
	__TVAPI_DebugError(@ScriptLineNumber & " :: " & VarGetType($vResult))
	Return SetError(@error, @extended, $vResult)
EndFunc   ;==>_TVAPI_Devices_GetDevicesList

; #FUNCTION# ====================================================================================================================
; Name ..........: _TVAPI_Devices_SaveToFile
; Description ...: Save all devices to JSON file
; Syntax ........: _TVAPI_Devices_SaveToFile($sJson_FileFullPath)
; Parameters ....: $sJson_FileFullPath  - a string value.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _TVAPI_Devices_SaveToFile($sJson_FileFullPath)
	__TVAPI_DebugOut("! $sJson_FileFullPath=" & $sJson_FileFullPath & @CRLF)
	Local $oJSON = _TVAPI_Devices_GetDevicesList()
	If @error Then MsgBox($MB_ICONERROR, '_TVAPI_Devices_GetDevicesList()', '@error = ' & @error & @CRLF & '@extended = ' & @extended & @CRLF & @CRLF & _TVAPI_GET_ResponseTextPretty())

	Local $sJSON_String = _Chilkat_JSON_ObjToString($oJSON)
	If @error Then MsgBox($MB_ICONERROR, '_Chilkat_JSON_ObjToString', '@error = ' & @error & @CRLF & '@extended = ' & @extended)

	Local $hFile = FileOpen($sJson_FileFullPath, $FO_OVERWRITE + $FO_UTF8_NOBOM)
	FileWrite($hFile, $sJSON_String)
	FileClose($hFile)

EndFunc   ;==>_TVAPI_Devices_SaveToFile
#EndRegion - TeamViewer.au3 - API functions - Devices - WIP

#Region - TeamViewer.au3 - API functions - Users - WIP
Func _TVAPI_Users_CreateNewUser()
;~ 	 POST /api/v1/users (create new company member)
EndFunc   ;==>_TVAPI_Users_CreateNewUser

; Creates a single company user:
; Field values in $v_DictUser will be used to create the given user.
; Defaults for some missing fields (permissions, password, language) must be provided.
Func _TVAPI_Users_CreateUser($v_DictUser, $sDefaultUserPermissions, $sDefaultUserLanguage, $sDefaultUserPassword)
	__TVAPI_DebugOut(@CRLF)
	__TVAPI_DebugOut("Creating user [" & $v_DictUser.Item("email") & "]...")
	__TVAPI_DebugOut("Request [POST] /api/" & $__g_sTVAPI_Version & "/users")

	Local $oHTTP = __TVAPI_HTTP_Open("POST", "users")
	$oHTTP.setRequestHeader("Content-Type", "application/json")

	; define fields
	Local $o_Create_PayLoad = ObjCreate("Scripting.Dictionary")

	; name parameter, required
	If $v_DictUser.Exists("name") And StringLen($v_DictUser.Item("name")) > 0 Then
		$o_Create_PayLoad.Add("name", $v_DictUser.Item("name"))
	Else
		__TVAPI_DebugOut("Field [name] is missing. Can; t create user.")
		; _TVAPI_CreateUser = false
		Return False
	EndIf

	; email parameter
	If $v_DictUser.Exists("email") And StringLen($v_DictUser.Item("email")) > 5 Then
		$o_Create_PayLoad.Add("email", $v_DictUser.Item("email"))
	Else
		__TVAPI_DebugOut("Field [email] is missing. Can; t create user.")
		; _TVAPI_CreateUser = false
		Return False
	EndIf

	; password parameter,
	If $v_DictUser.Exists("password") And StringLen($v_DictUser.Item("password")) > 5 Then
		$o_Create_PayLoad.Add("password", $v_DictUser.Item("password"))
	Else ; use defaultUserPassword parameter
		$o_Create_PayLoad.Add("password", $sDefaultUserPassword)
	EndIf

	; permission parameter
	If $v_DictUser.Exists("permissions") And StringLen($v_DictUser.Item("permissions")) > 0 Then
		$o_Create_PayLoad.Add("permissions", $v_DictUser.Item("permissions"))
	Else ; use defaultUserPermissions parameter
		$o_Create_PayLoad.Add("permissions", $sDefaultUserPermissions)
	EndIf

	; language parameter
	If $v_DictUser.Exists("language") And StringLen($v_DictUser.Item("language")) > 0 Then
		$o_Create_PayLoad.Add("language", $v_DictUser.Item("language"))
	Else ; use defaultUserLanguage parameter
		$o_Create_PayLoad.Add("language", $sDefaultUserLanguage)
	EndIf

	; @TODO
	Local $v_JSON_PayLoad ; = _JSONEncode($o_Create_PayLoad)

	If @error Then
		__TVAPI_DebugError("JSON Encode failed")
		Return False
	EndIf
	__TVAPI_DebugOut("$sPayLoad: " & $v_JSON_PayLoad)

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP, $v_JSON_PayLoad)
	If @error Then Return SetError(@error, @extended, $oJSON)

	__TVAPI_DebugOut("User created.")

EndFunc   ;==>_TVAPI_Users_CreateUser

; Deactivates a single company user:
Func _TVAPI_Users_DeactivateUser($s_UserId)
	__TVAPI_DebugOut(@CRLF)
	__TVAPI_DebugOut("Deactivating user [" & $s_UserId & "]...")
	__TVAPI_DebugOut("Request [PUT] /api/" & $__g_sTVAPI_Version & "/users/" & $s_UserId)

	Local $oHTTP = __TVAPI_HTTP_Open("PUT", "users/" & $s_UserId)
	$oHTTP.setRequestHeader("Content-Type", "application/json")

	; define update fields
	Local $o_Update_PayLoad = ObjCreate("Scripting.Dictionary")

	; active flag
	$o_Update_PayLoad.Add("active", False)

	; @TODO
	Local $v_JSON_PayLoad ; = _JSONEncode($o_Update_PayLoad)
	If @error Then
		__TVAPI_DebugError("JSON Encode failed")
		; _TVAPI_Users_DeactivateUser = false
		Return False
	EndIf
	__TVAPI_DebugOut("$sPayLoad: " & $v_JSON_PayLoad)

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP, $v_JSON_PayLoad)
	If @error Then Return SetError(@error, @extended, $oJSON)

	__TVAPI_DebugOut("User deactivated.")

EndFunc   ;==>_TVAPI_Users_DeactivateUser

Func _TVAPI_Users_GetUserInfo($iUserID)
;~ 	GET /api/v1/users/<uID> (get information for one user)

	Local $oHTTP = __TVAPI_HTTP_Open("GET", "users", '/u' & String($iUserID))

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP)
	If @error Then Return SetError(@error, @extended, False)

	__TVAPI_DebugOut("Request ok!")
	Return SetError($TVAPI_ERR_SUCCESS, $oJSON.Size, $oJSON)
EndFunc   ;==>_TVAPI_Users_GetUserInfo

; #FUNCTION# ====================================================================================================================
; Name ..........: _TVAPI_Users_ListUsers
; Description ...: retrieve list of users
; Syntax ........: _TVAPI_Users_ListUsers([$sEmail = Default[, $sName = Default[, $sPermissions = Default[,
;                  $bFull_list = False]]]])
; Parameters ....: $sEmail              - [optional] a string value. Default is Default.
;                  $sName               - [optional] a string value. Default is Default.
;                  $sPermissions        - [optional] a string value. Default is Default.
;                  $bFull_list          - [optional] a boolean value. Default is False.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _TVAPI_Users_ListUsers($sEmail = Default, $sName = Default, $sPermissions = Default, $bFull_list = False)
;~ 	GET /api/v1/users (retrieve list of users)
	Local $sURLParameters = ''
	If IsString($sEmail) Then $sURLParameters &= _
			'email=' & String($sEmail)
	If IsString($sName) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'name=' & String($sName)
	If IsString($sPermissions) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'permissions=' & String($sPermissions)
	If IsBool($bFull_list) Then $sURLParameters &= _
			(($sURLParameters <> '') ? '&' : '') & _
			'full_list=' & (($bFull_list) ? "true" : "false")
	If $sURLParameters <> '' Then $sURLParameters = '?' & $sURLParameters

	Local $oHTTP = __TVAPI_HTTP_Open("GET", "users", $sURLParameters)

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP)
	If @error Then Return SetError(@error, @extended, False)

	__TVAPI_DebugOut("Request ok!")
	Return SetError($TVAPI_ERR_SUCCESS, $oJSON.SizeOfArray("users"), $oJSON)
EndFunc   ;==>_TVAPI_Users_ListUsers

Func _TVAPI_Users_ModifyUserInfo()
;~ 	PUT /api/v1/users/<uID> (modify user information)
EndFunc   ;==>_TVAPI_Users_ModifyUserInfo

; Updates a single company user:
; Field values in $v_DictUser will be used to update the given user id ($s_Update_UserId)
; if email should be updated, the dict must declare a column "newEmail" with the new email value
Func _TVAPI_Users_UpdateUser($s_Update_UserId, $v_DictUser)
	__TVAPI_DebugOut(@CRLF)
	__TVAPI_DebugOut("Updating user [" & $v_DictUser.Item("email") & "]...")
	__TVAPI_DebugOut("Request [PUT] /api/" & $__g_sTVAPI_Version & "/users/" & $s_Update_UserId)

	Local $oHTTP = __TVAPI_HTTP_Open("PUT", "/users/" & $s_Update_UserId)
	$oHTTP.setRequestHeader("Content-Type", "application/json; charset=utf-8")

	; define update fields
	Local $o_Update_PayLoad = ObjCreate("Scripting.Dictionary")

	; name parameter
	If $v_DictUser.Exists("name") And StringLen($v_DictUser.Item("name")) > 0 Then
		$o_Update_PayLoad.Add("name", $v_DictUser.Item("name"))
	EndIf

	; password parameter
	If $v_DictUser.Exists("password") And StringLen($v_DictUser.Item("password")) > 5 Then
		$o_Update_PayLoad.Add("password", $v_DictUser.Item("password"))
	EndIf

	; permission parameter
	If $v_DictUser.Exists("permissions") And StringLen($v_DictUser.Item("permissions")) > 0 Then
		$o_Update_PayLoad.Add("permissions", $v_DictUser.Item("permissions"))
	EndIf

	; email parameter (column newMail must exist in csv)
	If $v_DictUser.Exists("newEmail") And StringLen($v_DictUser.Item("newEmail")) > 5 Then
		$o_Update_PayLoad.Add("email", $v_DictUser.Item("newEmail"))
	EndIf

	; active parameter (assume every user to be updated is also active per default)
	If $v_DictUser.Exists("active") And StringLen($v_DictUser.Item("active")) > 0 Then
		$o_Update_PayLoad.Add("active", $v_DictUser.Item("active"))
	Else
		$o_Update_PayLoad.Add("active", True)
	EndIf

	; @TODO
	Local $v_JSON_PayLoad ; = _JSONEncode($o_Update_PayLoad)
	If @error Then
		__TVAPI_DebugError("JSON Encode")
		; _TVAPI_Users_UpdateUser_TODO = false
		Return False
	EndIf

	__TVAPI_DebugOut("$sPayLoad: " & $v_JSON_PayLoad)
	Local $oJSON = __TVAPI_HTTP_Send($oHTTP, $v_JSON_PayLoad)
	If @error Then Return SetError(@error, @extended, $oJSON)

	If _TVAPI_GET_Status() = 204 Then ; ??? @TODO
		__TVAPI_DebugOut("User updated.")
		Return True
	EndIf

EndFunc   ;==>_TVAPI_Users_UpdateUser
#EndRegion - TeamViewer.au3 - API functions - Users - WIP

#Region - TeamViewer.au3 - API Helper functions

Func __TVAPI($iAPI_idx = Default, $vAPI_Content = Default)
	Local Static $aTVAPI_static[$TVAPI_ARR_COUNTER] ; Preset

	; Reset
	If $iAPI_idx = $TVAPI_ARR_RESET Then
		ReDim $aTVAPI_static[0]
		ReDim $aTVAPI_static[$TVAPI_ARR_COUNTER]
		Return 1
	EndIf

	; Invalid ParamType
	If Not IsInt($iAPI_idx) Then Return SetError($TVAPI_ERR_TVAPIARR, $TVAPI_EXT_PARAM1)

	; Invalid ParamValue
	If $iAPI_idx <= $TVAPI_ARR_RESET Or $iAPI_idx >= $TVAPI_ARR_COUNTER Then Return SetError($TVAPI_ERR_TVAPIARR, $TVAPI_EXT_PARAM1)

	; get current content
	If $vAPI_Content = Default Then Return $aTVAPI_static[$iAPI_idx]

	; setting new content
;~ 	ConsoleWrite("! Setting: $iAPI_idx=" & $iAPI_idx & "  $vAPI_Content=" & $vAPI_Content & @CRLF)
	$aTVAPI_static[$iAPI_idx] = $vAPI_Content
EndFunc   ;==>__TVAPI

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __TVAPI_DebugError
; Description ...:
; Syntax ........: __TVAPI_DebugError($sReason[, $iError = @error[, $iExtended = @extended]])
; Parameters ....: $sReason             - a string value.
;                  $iError              - [optional] an integer value. Default is @error.
;                  $iExtended           - [optional] an integer value. Default is @extended.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func __TVAPI_DebugError($sReason, $iError = @error, $iExtended = @extended)
	If $iError Or $iExtended Then __TVAPI_DebugOut($sReason & " : $iError = " & $iError & " $iExtended = " & $iExtended)
	Return SetError($iError, $iExtended, 1)
EndFunc   ;==>__TVAPI_DebugError

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __TVAPI_DebugOut
; Description ...:
; Syntax ........: __TVAPI_DebugOut($sText[, $iError = @error[, $iExtended = @extended]])
; Parameters ....: $sText               - a string value.
;                  $iError              - [optional] an integer value. Default is @error.
;                  $iExtended           - [optional] an integer value. Default is @extended.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func __TVAPI_DebugOut($sText, $iError = @error, $iExtended = @extended)
	If Not @Compiled Then ConsoleWrite($sText & @CRLF)
	Return SetError($iError, $iExtended, 1)
EndFunc   ;==>__TVAPI_DebugOut

; Fix Status Code Bug of MSXML $oHTTP Implementation (204 Status is 1223)
Func __TVAPI_FixHttpStatusBug(ByRef $oHTTP, $iError = @error, $iExtended = @extended)
;~ 	ConsoleWrite("! $oHTTP.status=" & $oHTTP.status & @CRLF)
	If $oHTTP.status = 1223 Then
		__TVAPI($TVAPI_ARR_STATUS, 204)
		__TVAPI($TVAPI_ARR_STATUSTEXT, "No Content")
	Else
		__TVAPI($TVAPI_ARR_STATUS, $oHTTP.status)
		__TVAPI($TVAPI_ARR_STATUSTEXT, $oHTTP.StatusText)
	EndIf
	__TVAPI($TVAPI_ARR_RESPONSETEXT, $oHTTP.ResponseText)
	__TVAPI_DebugOut("Status: " & _TVAPI_GET_Status() & " StatusText: " & _TVAPI_GET_StatusText())
	Return SetError($iError, $iExtended, 1)
EndFunc   ;==>__TVAPI_FixHttpStatusBug

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __TVAPI_HTTP_Open
; Description ...:
; Syntax ........: __TVAPI_HTTP_Open($sMethod, $sCommand[, $sURLParameters = ''])
; Parameters ....: $sMethod             - a string value.
;                  $sCommand            - a string value.
;                  $sURLParameters      - [optional] a string value. Default is ''.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func __TVAPI_HTTP_Open($sMethod, $sCommand, $sURLParameters = '')
	__TVAPI($TVAPI_ARR_RESET)

;~ 	Local $oHTTP = ObjCreate("Msxml2.XMLHTTP")
	Local $oHTTP = ObjCreate("WinHttp.WinHttpRequest.5.1")
	Local $sURL = $__g_sTVAPI_BaseUrl & "/api/" & $__g_sTVAPI_Version & "/" & $sCommand & $sURLParameters
	__TVAPI_DebugOut("> $sURL=" & $sURL & @CRLF)
	$oHTTP.Open($sMethod, $sURL, False)
	If @error Then Return SetError(@error, @extended, Null)
	$oHTTP.setRequestHeader("Authorization", "Bearer " & _TVAPI_AccessToken())

	Return $oHTTP
EndFunc   ;==>__TVAPI_HTTP_Open

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __TVAPI_HTTP_Send
; Description ...:
; Syntax ........: __TVAPI_HTTP_Send(Byref $oHTTP[, $sSendParameter = Default])
; Parameters ....: $oHTTP               - [in/out] an object.
;                  $sSendParameter      - [optional] a string value. Default is Default.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func __TVAPI_HTTP_Send(ByRef $oHTTP, $sSendParameter = Default)
	If $sSendParameter = Default Then
		$oHTTP.Send()
	Else
		$oHTTP.Send($sSendParameter)
	EndIf
	__TVAPI_FixHttpStatusBug($oHTTP)

	Local $sStatusResult = _TVAPI_GET_Status()
	If @error Then Return SetError(@error, @extended, $sStatusResult)

	Local $sResponse = _TVAPI_GET_ResponseText()
	Local $oJSON = _Chilkat_JSON_ObjFromString($sResponse)
	If @error Then Return SetError(@error, @extended, $TVAPI_RET_FAILURE)

	Return SetError($TVAPI_ERR_SUCCESS, $oJSON.Size, $oJSON)
EndFunc   ;==>__TVAPI_HTTP_Send

; #FUNCTION# ====================================================================================================================
; Name ..........: _TVAPI_AccessToken
; Description ...: Set/Get AccessToken
; Syntax ........: _TVAPI_AccessToken([$sParam = Default])
; Parameters ....: $sParam              - [optional] a string value. Default is Default.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _TVAPI_AccessToken($sParam = Default)
	Local Static $sAccessToken = '' ; $__g_sTV_Access-Token
	If $sParam <> Default Then $sAccessToken = $sParam
	Return $sAccessToken
EndFunc   ;==>_TVAPI_AccessToken

; #FUNCTION# ====================================================================================================================
; Name ..........: _TVAPI_ErrFunc
; Description ...: TVAPI User's COM error function. Will be called if COM error occurs
; Syntax ........: _TVAPI_ErrFunc($oError)
; Parameters ....: $oError              - an object.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _TVAPI_ErrFunc($oError)
	; Do anything here.
	ConsoleWrite(@ScriptName & " (" & $oError.scriptline & ") : ==> COM Error intercepted !" & @CRLF & _
			@TAB & "err.number is: " & @TAB & @TAB & "0x" & Hex($oError.number) & @CRLF & _
			@TAB & "err.windescription:" & @TAB & $oError.windescription & @CRLF & _
			@TAB & "err.description is: " & @TAB & $oError.description & @CRLF & _
			@TAB & "err.source is: " & @TAB & @TAB & $oError.source & @CRLF & _
			@TAB & "err.helpfile is: " & @TAB & $oError.helpfile & @CRLF & _
			@TAB & "err.helpcontext is: " & @TAB & $oError.helpcontext & @CRLF & _
			@TAB & "err.lastdllerror is: " & @TAB & $oError.lastdllerror & @CRLF & _
			@TAB & "err.scriptline is: " & @TAB & $oError.scriptline & @CRLF & _
			@TAB & "err.retcode is: " & @TAB & "0x" & Hex($oError.retcode) & @CRLF & @CRLF)
EndFunc   ;==>_TVAPI_ErrFunc

Func _TVAPI_GET_ResponseText()
	Return __TVAPI($TVAPI_ARR_RESPONSETEXT)
EndFunc   ;==>_TVAPI_GET_ResponseText

Func _TVAPI_GET_ResponseTextPretty()
	Local $sResponeText = _TVAPI_GET_ResponseText()
	Local $oJSON = _Chilkat_JSON_ObjFromString($sResponeText)
	Return _Chilkat_JSON_ObjToString($oJSON)
EndFunc   ;==>_TVAPI_GET_ResponseTextPretty

Func _TVAPI_GET_Status()
	Local $sStatus = __TVAPI($TVAPI_ARR_STATUS)
	If $sStatus <> 200 And $sStatus <> 204 Then
		__TVAPI_DebugOut("Unexpected ResponseText code. Received content was:")
		__TVAPI_DebugOut(_TVAPI_GET_ResponseText())
		Return SetError($TVAPI_ERR_STATUS, Int($sStatus), _TVAPI_GET_StatusText())
	EndIf

	Return SetError($TVAPI_ERR_SUCCESS, $TVAPI_EXT_DEFAULT, $sStatus)
EndFunc   ;==>_TVAPI_GET_Status

Func _TVAPI_GET_StatusText()
	Return __TVAPI($TVAPI_ARR_STATUSTEXT)
EndFunc   ;==>_TVAPI_GET_StatusText
; #FUNCTION# ====================================================================================================================
; Name ..........: _TVAPI_RequestOAuth2_AccessToken
; Description ...: OAuth2: get an access token by clientId and authorizationCode
; Syntax ........: _TVAPI_RequestOAuth2_AccessToken($sClientId, $sClientSecret, $sAuthorizationCode)
; Parameters ....: $sClientId           - a string value.
;                  $sClientSecret       - a string value.
;                  $sAuthorizationCode  - a string value.
; Return values .: None
; Author ........: mLipok
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _TVAPI_RequestOAuth2_AccessToken($sClientId, $sClientSecret, $sAuthorizationCode)
	__TVAPI_DebugOut(@CRLF)
	__TVAPI_DebugOut("Get token...")
	__TVAPI_DebugOut("Request [POST] /api/" & $__g_sTVAPI_Version & "/oauth2/token")

	Local $oHTTP = __TVAPI_HTTP_Open("POST", "oauth2/token")
	$oHTTP.setRequestHeader("Content-Type", "application/x-www-form-urlencoded")

	Local $sPayLoad = "grant_type=authorization_code&code=" & $sAuthorizationCode & "&client_id=" & $sClientId & "&client_secret=" & $sClientSecret
	__TVAPI_DebugOut("$sPayLoad: " & $sPayLoad)

	Local $oJSON = __TVAPI_HTTP_Send($oHTTP, $sPayLoad)
	If @error Then Return SetError(@error, @extended, '')

	__TVAPI_DebugOut("Token received.")
	Return SetError($TVAPI_ERR_SUCCESS, $TVAPI_EXT_DEFAULT, $oJSON.StringOf("access_token"))

EndFunc   ;==>_TVAPI_RequestOAuth2_AccessToken
#EndRegion - TeamViewer.au3 - API Helper functions

#Region - TeamViewer.au3 - JSON To AU3ARRAY - Helper Functinos
Func __TVAPI_JSON_ChilkatArrayToAutoIt(ByRef $oChilkatJsonArray)
	Local $iArraySize = $oChilkatJsonArray.size
	Local $aResult[$iArraySize]
	For $iArray_idx = 0 To $iArraySize - 1
		$aResult[$iArray_idx] = $oChilkatJsonArray.StringAt($iArray_idx)
	Next
	Return $aResult
EndFunc   ;==>__TVAPI_JSON_ChilkatArrayToAutoIt

Func _TVAPI_JSON_AddRecordsTo1DArray(ByRef $oJSON_MoreConn, ByRef $a_ConnectArr)
	If $oJSON_MoreConn.hasMember("records") Then
		$oTemp = $oJSON_MoreConn.ArrayOf("records")
		$aTemp = __TVAPI_JSON_ChilkatArrayToAutoIt($oTemp)
		_ArrayConcatenate($a_ConnectArr, $aTemp)
	EndIf
EndFunc   ;==>_TVAPI_JSON_AddRecordsTo1DArray
#EndRegion - TeamViewer.au3 - JSON To AU3ARRAY - Helper Functinos
