﻿#include-once
; #INDEX# =======================================================================================================================
; Title .............: ImageSearch UDF
; Version ...........: 2025.05.25.5 (Refactored)
; AutoIt Version ....: 3.3.14+
; Language ..........: English
; Description .......: Searches for an image on the screen using a fast DLL.
;                      This UDF library for interacting with the high-performance ImageSearch.dll !
; Hybrid DLL Loading : It prioritizes a local DLL in @ScriptDir and falls back to a truly embedded hex string version.
; Author(s) .........: Dao Van Trong - TRONG.PRO
;
; Note ..............: The embedded DLL version in UDF supports Windows XP, but does not support AVX2!
; (Because the DLL file with AVX2 support is large in size, it is not suitable for embedding in a script)
; (Please use the DLL in the same UDF folder - this DLL does not support Windows XP, but it supports AVX2, which speeds up searching a lot)
; -------------------------------------------------------------------------------------------------------------------------------
; #SECTION# QUICK START GUIDE FOR BEGINNERS
; -------------------------------------------------------------------------------------------------------------------------------
; Follow these 4 simple steps to use this UDF in your script:
;
; 1. INCLUDE THE UDF:
;    #include "ImageSearch_UDF.au3"
;
; 2. INITIALIZE THE LIBRARY:
;    ; Call this once at the start of your script.
;    _ImageSearch_Startup()
;    If @error Then
;        MsgBox(16, "Error", "ImageSearch DLL could not be initialized. Exiting.")
;        Exit
;    EndIf
;
; 3. PERFORM A SEARCH:
;    ; Search for "my_image.bmp" on the entire screen.
;    Local $aResult = _ImageSearch("my_image.bmp")
;
; 4. PROCESS THE RESULTS:
;    ; The result is ALWAYS a 2D array. Check the count in the first element.
;    If $aResult[0][0] > 0 Then
;        ConsoleWrite("Found " & $aResult[0][0] & " match(es)!" & @CRLF)
;        ; Loop through each match
;        For $i = 1 To $aResult[0][0]
;            Local $iX = $aResult[$i][1] ; X coordinate
;            Local $iY = $aResult[$i][2] ; Y coordinate
;            ConsoleWrite("Match #" & $i & " found at: " & $iX & ", " & $iY & @CRLF)
;            MouseMove($iX, $iY)
;            Sleep(1000)
;        Next
;    Else
;        ConsoleWrite("Image not found." & @CRLF)
;    EndIf
;
; NOTE: You do NOT need to call _ImageSearch_Shutdown(). It is automatically handled when your script exits.
;
; -------------------------------------------------------------------------------------------------------------------------------
; #SECTION# IMPORTANT CHANGES COMPARED TO THE OLD VERSION
; -------------------------------------------------------------------------------------------------------------------------------
; Please be aware of these key changes if you are migrating from an older version:
;
; 1. EXPLICIT INITIALIZATION:
;    - You MUST call _ImageSearch_Startup() at the beginning of your script. This prevents the DLL from being loaded
;      and unloaded unnecessarily and gives you better control over error handling.
;
; 2. CONSISTENT 2D ARRAY RETURN:
;    - ALL search functions now ALWAYS return a 2D array, even if 0 or 1 result is found.
;    - The number of found matches is always in $aResult[0][0].
;    - This eliminates the need to check if the result is a 1D or 2D array, which was a common source of bugs.
;
; 3. AUTOMATIC PARAMETER CORRECTION:
;    - It will automatically correct them to the nearest valid values (e.g., a tolerance of 300 becomes 255)
;      and continue the search. A warning will be printed to the console if debug mode ($g_bImageSearch_Debug) is enabled.
;
; -------------------------------------------------------------------------------------------------------------------------------
; #SECTION# USAGE GUIDE FOR SPECIAL PARAMETERS
; -------------------------------------------------------------------------------------------------------------------------------
;
; === $iTransparent ===
;   - PURPOSE: To ignore a specific color during the search, making it "transparent". This is useful for finding
;     images with irregular shapes or on varied backgrounds.
;   - FORMAT: The color value must be provided in 0xRRGGBB (Red-Green-Blue) hexadecimal format. AutoIt uses BGR format
;     for many of its functions, but this DLL requires RGB, so 0xFF0000 is pure Red.
;   - HOW TO GET THE COLOR: Use the AutoIt Window Info tool. Hover your mouse over the color you want to ignore,
;     and look at the "Color" value in the tool.
;   - EXAMPLE: You have an icon of a red circle on a solid green background (color 0x00FF00). To find this circle
;     anywhere on the screen regardless of its background, you would call:
;     _ImageSearch("red_circle.bmp", 10, 1, 0x00FF00)
;
; === $iMultiResults vs. $iFindAllOccurrences ===
;   These two parameters are powerful but often confused. They serve very different purposes.
;
;   $iMultiResults (Find *Any* N Matches)
;   ------------------------------------------------
;   - PURPOSE: To limit the TOTAL number of results when searching for MULTIPLE DIFFERENT images at once.
;   - BEHAVIOR: It's a "stop condition". The search stops as soon as this number of matches is found.
;   - BEST USED WITH: A list of images, like "ok_button.bmp|accept_button.bmp|go_button.bmp".
;   - EXAMPLE: You need to click either an "OK" or "Accept" button, whichever appears first.
;       _ImageSearch_Area("ok.bmp|accept.bmp", ..., ..., ..., $iMultiResults = 1)
;       The search will stop and return the location of the very first button it finds. It is highly efficient.
;
;   $iFindAllOccurrences (Find *Every* Match)
;   ------------------------------------------------
;   - PURPOSE: To find EVERY SINGLE instance of the specified image(s) within the search area.
;   - BEHAVIOR: It forces the DLL to scan the entire area and not stop after the first match.
;   - BEST USED WITH: Single or multiple images when you need a complete count or list of all locations.
;   - EXAMPLE: You are writing a bot for a game and need to find the coordinates of ALL enemy units on the screen.
;       _ImageSearch("enemy.bmp", ..., ..., ..., $bReturnAll = True) ; or _ImageSearch_Area(..., $iFindAllOccurrences = 1)
;       The result will be a 2D array containing the locations of every single enemy found.
;
;   SUMMARY:
;   - Use $iMultiResults when you need to find *one or a few* items from a *list of possibilities*.
;   - Use $iFindAllOccurrences when you need to find *every single copy* of an item.
;
; -------------------------------------------------------------------------------------------------------------------------------
; #SECTION# BEST PRACTICES
; -------------------------------------------------------------------------------------------------------------------------------
;
; - ENABLE DEBUG: Keep $g_bImageSearch_Debug = True during development. The console warnings are invaluable for
;   troubleshooting parameter issues and understanding the DLL's raw output.
;
; - CAPTURE SMART: When creating your search images, capture the smallest, most unique part of the object you need to find.
;   Avoid including large areas of plain background.
;
; - USE TOLERANCE WISELY: Start with a low tolerance (e.g., 10-20). A tolerance that is too high can lead to false
;   positives (finding the wrong image) and can slow down the search.
;
; - DEFINE AN AREA: Whenever possible, use _ImageSearch_Area() to limit the search to a smaller portion of the screen.
;   Searching a 200x200 pixel box is exponentially faster than searching the entire 1920x1080 screen.
;
; - CHECK @ERROR: Although the UDF now self-corrects many parameter errors, it's still good practice to check @error
;   after a search call to catch critical issues like a missing image file or a failed DllCall.
;
; Author(s) .....: Dao Van Trong - TRONG.PRO
; ===============================================================================================================================

; #CONSTANTS# ===================================================================================================================
Global Const $__IMAGESEARCH_UDF_VERSION = '2025.05.25.5'
Global Const $__ImgSearch_MaximumResults = 99

; #GLOBALS# =====================================================================================================================
Global $g_bImageSearch_Debug = True
Global $g_iImgSearchWait_SleepTime = 100
Global $g_sImageSearchDLL_Dir = @ScriptDir
Global $g_sImgSearchDllName_x64 = 'ImageSearch_x64.dll'
Global $g_sImgSearchDllName_x86 = 'ImageSearch_x86.dll'
Global $g_bImageSearch_IsEmbeddedUsed = False
Global $g_sLastDllReturn = ""
Global $g_sImageSearchDLL_Path = ""
Global $g_hImageSearchDLL = -1
Global $g_bImageSearch_Initialized = False

; #FUNCTION# ====================================================================================================================
; Name...........: _ImageSearch_Startup
; Description....: Initializes the ImageSearch.dll library. It finds a local DLL, or deploys the embedded version
;                  to the temp directory, then opens a handle to it and registers the shutdown function for
;                  automatic cleanup on script exit.
;                  This function SHOULD be called once at the beginning of your script.
; Syntax.........: _ImageSearch_Startup()
; Parameters.....: None.
; Return values..: Success - Returns 1.
;                  Failure - Returns 0 and sets @error.
; @error.........: 1 - DLL is already initialized.
;                  2 - Failed to find or deploy the DLL file.
;                  3 - DllOpen failed (the DLL might be corrupt or blocked).
; Author.........: Dao Van Trong - TRONG.PRO
; ===============================================================================================================================
Func _ImageSearch_Startup()
	If $g_bImageSearch_Initialized Then
		SetError(1)
		Return 1 ; Already initialized, not a critical error.
	EndIf

	$g_sImageSearchDLL_Path = __ImgSearch_InitializeDllPath()
	If (StringStripWS($g_sImageSearchDLL_Path, 8) == "") Then
		If $g_bImageSearch_Debug Then ConsoleWrite('!> FATAL ERROR: Could not determine DLL path.' & @CRLF)
		SetError(2)
		Return 0
	EndIf

	$g_hImageSearchDLL = DllOpen($g_sImageSearchDLL_Path)
	If ($g_hImageSearchDLL = -1) Then
		If $g_bImageSearch_Debug Then ConsoleWrite('!> FATAL ERROR: DllOpen failed for path: ' & $g_sImageSearchDLL_Path & @CRLF)
		SetError(3)
		Return 0
	EndIf

	If $g_bImageSearch_Debug Then ConsoleWrite('>> DLL loaded successfully.' & @CRLF & @CRLF)
	OnAutoItExitRegister("_ImageSearch_Shutdown")
	$g_bImageSearch_Initialized = True
	SetError(0)
	Return 1
EndFunc   ;==>_ImageSearch_Startup

; #FUNCTION# ====================================================================================================================
; Name...........: _ImageSearch_Shutdown
; Description....: Closes the DLL handle and cleans up the temporary DLL file if it was deployed from the embedded hex string.
;                  This is automatically called on script exit if _ImageSearch_Startup() was successful.
; Syntax.........: _ImageSearch_Shutdown()
; ===============================================================================================================================
Func _ImageSearch_Shutdown()
	If ($g_hImageSearchDLL <> -1) Then
		DllClose($g_hImageSearchDLL)
		$g_hImageSearchDLL = -1
		If $g_bImageSearch_Debug Then ConsoleWrite(">> DLL handle closed." & @CRLF)
	EndIf
	If $g_bImageSearch_IsEmbeddedUsed And FileExists($g_sImageSearchDLL_Path) Then
		FileDelete($g_sImageSearchDLL_Path)
		If $g_bImageSearch_Debug Then ConsoleWrite(">> Temporary DLL deleted: " & $g_sImageSearchDLL_Path & @CRLF)
	EndIf
	$g_bImageSearch_Initialized = False
EndFunc   ;==>_ImageSearch_Shutdown

; #FUNCTION# ====================================================================================================================
; Name...........: _ImageSearch
; Description....: A simplified function to search for an image on the entire screen.
; Syntax.........: _ImageSearch($sImagePath[, $iTolerance = 10[, $iCenterPos = 1[, $iTransparent = -1[, $bReturnAll = False]]]])
; Parameters.....: $sImagePath   - The full path to the image file to search for.
;                  $iTolerance   - [optional] The allowed tolerance for color variation (0-255). 0 is an exact match. Default is 10.
;                  $iCenterPos   - [optional] If 1, returns the center coordinates of the found image. If 0, returns top-left. Default is 1.
;                  $iTransparent - [optional] The color to treat as transparent (e.g., 0xFF00FF). Default is -1 (none).
;                  $bReturnAll   - [optional] If True, finds all occurrences. If False, stops after the first match. Default is False.
; Return values..: Always returns a 2D array.
;                  Success      - A 2D array. $aResult[0][0] contains the number of matches. Subsequent rows are [index, x, y, w, h].
;                  Failure      - A 2D array with $aResult[0][0] = 0 and @error set.
; Author.........: Dao Van Trong - TRONG.PRO
; ===============================================================================================================================
Func _ImageSearch($sImagePath, $iTolerance = 10, $iCenterPos = 1, $iTransparent = -1, $bReturnAll = False)
	Local $iFindAll = ($bReturnAll ? 1 : 0)
	Local $iMultiResults = ($bReturnAll ? $__ImgSearch_MaximumResults : 1)
	Return _ImageSearch_Area($sImagePath, 0, 0, @DesktopWidth, @DesktopHeight, $iTolerance, $iTransparent, $iMultiResults, $iCenterPos, $g_bImageSearch_Debug, 1.0, 1.0, 0.1, $iFindAll)
EndFunc   ;==>_ImageSearch

; #FUNCTION# ====================================================================================================================
; Name...........: _ImageSearch_Area
; Description....: Searches for one or more images within a specified rectangular area of the screen, with advanced options.
;                  This function will automatically validate and correct invalid parameters to ensure operation.
; Syntax.........: _ImageSearch_Area($sImageFile[, $iLeft = 0[, ...[, $iFindAllOccurrences = 0]]])
; Parameters.....: $sImageFile          - The full path to the image file. Can be multiple paths, separated by "|".
;                  $iLeft, $iTop, $iRight, $iBottom - [optional] The coordinates of the search area. Default is the entire screen.
;                  $iTolerance          - [optional] Color tolerance (0-255). Default is 10.
;                  $iTransparent        - [optional] Transparent color. Default is -1.
;                  $iMultiResults       - [optional] The maximum number of results to find. Default is 1.
;                  $iCenterPos          - [optional] Return center (1) or top-left (0) coordinates. Default is 1.
;                  $iReturnDebug        - [optional] Enable/disable DLL debug mode. Default is 1 (enabled).
;                  $fMinScale, $fMaxScale, $fScaleStep - [optional] Parameters for multi-scale searching.
;                  $iFindAllOccurrences - [optional] If 1, finds all occurrences. If 0, stops after the first match. Default is 0.
; Return values..: Always returns a 2D array.
;                  Success        - A 2D array with $aResult[0][0] > 0.
;                  Failure        - A 2D array with $aResult[0][0] = 0 and @error set.
; @error.........:   0 - Success
;                  -1 - DllCall Failed: Could not call the function in the DLL.
;                  -2 - DLL Return Error: Invalid format (could not find '{...}' and '[...]' parts).
;                  -3 - DLL Return Error: Invalid content within the returned brackets.
;                  -4 to -10 - Various DLL internal errors (see C++ source for details).
;                  -11 - UDF Error: Image file not found on disk.
;                  -12 - UDF Error: DLL not initialized. Call _ImageSearch_Startup() first.
; Author.........: Dao Van Trong - TRONG.PRO
; ===============================================================================================================================
Func _ImageSearch_Area($sImageFile, $iLeft = 0, $iTop = 0, $iRight = @DesktopWidth, $iBottom = @DesktopHeight, $iTolerance = 10, $iTransparent = -1, $iMultiResults = 1, $iCenterPos = 1, $iReturnDebug = 1, $fMinScale = 1.0, $fMaxScale = 1.0, $fScaleStep = 0.1, $iFindAllOccurrences = 0)
	; --- Auto-initialize if necessary ---
	If Not $g_bImageSearch_Initialized Then
		If Not _ImageSearch_Startup() Then
			SetError(-12)
			Return __ImgSearch_Make2DResultArray(0)
		EndIf
	EndIf

	; --- Parameter Validation and Correction ---
	If ($iTolerance < 0) Or ($iTolerance > 255) Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> UDF WARNING: Invalid tolerance value (" & $iTolerance & ") specified. Clamping to range 0-255." & @CRLF)
		If ($iTolerance < 1) Then $iTolerance = 0
		If ($iTolerance > 254) Then $iTolerance = 255
	EndIf

	If ($iLeft >= $iRight) Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> UDF WARNING: Invalid coordinates (Left >= Right). Swapping values." & @CRLF)
		Local $iTempLeft = $iLeft
		$iLeft = $iRight
		$iRight = $iTempLeft
	EndIf

	If ($iTop >= $iBottom) Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> UDF WARNING: Invalid coordinates (Top >= Bottom). Swapping values." & @CRLF)
		Local $iTempTop = $iTop
		$iTop = $iBottom
		$iBottom = $iTempTop
	EndIf

	If ($fMinScale < 0.1) Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> UDF WARNING: Invalid minimum scale. Resetting to 0.1." & @CRLF)
		$fMinScale = 0.1
	EndIf
	If ($fMaxScale < $fMinScale) Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> UDF WARNING: Maximum scale is less than minimum. Resetting to minimum scale." & @CRLF)
		$fMaxScale = $fMinScale
	EndIf
	If ($fScaleStep < 0.1) Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> UDF WARNING: Invalid scale step. Resetting to 0.1." & @CRLF)
		$fScaleStep = 0.1
	EndIf

	; --- File Existence Check ---
	If Not StringInStr($sImageFile, "|") And Not FileExists($sImageFile) Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> UDF ERROR: Image file not found - " & $sImageFile & @CRLF)
		SetError(-11)
		Return __ImgSearch_Make2DResultArray(0)
	EndIf

	; --- DLL Call ---
	Local $aDLLResult = DllCall($g_hImageSearchDLL, "wstr", "ImageSearch", "wstr", $sImageFile, "int", $iLeft, "int", $iTop, "int", $iRight, "int", $iBottom, "int", $iTolerance, "int", $iTransparent, "int", $iMultiResults, "int", $iCenterPos, "int", $iReturnDebug, "float", $fMinScale, "float", $fMaxScale, "float", $fScaleStep, "int", $iFindAllOccurrences)
	If @error Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> DllCall ERROR: @error = " & @error & @CRLF)
		SetError(-1)
		Return __ImgSearch_Make2DResultArray(0)
	EndIf

	Local $sDLLResult = $aDLLResult[0]
	$g_sLastDllReturn = $sDLLResult
	If $g_bImageSearch_Debug Then ConsoleWrite(">> DLL Return: " & $sDLLResult & @CRLF)

	; --- Robust Result Parsing using a RegExp-based helper function ---
	Local $iStatus = Number(__ImgSearch_StringBetween($sDLLResult, "{", "}"))
	Local $sContent = __ImgSearch_StringBetween($sDLLResult, "[", "]")

	If @error Or (($iStatus < 1) And (StringLen($sContent) < 1)) Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> UDF ERROR: Invalid format from DLL. Could not find '{...}' and '[...]' parts." & @CRLF)
		SetError(-2)
		Return __ImgSearch_Make2DResultArray(0)
	EndIf

	If ($iStatus < 1) Then
		SetError($iStatus)
		Return __ImgSearch_Make2DResultArray(0)
	EndIf

	SetError(0)
	Return __ImgSearch_ProcessResult($sContent, $iStatus)
EndFunc   ;==>_ImageSearch_Area

; #FUNCTION# ====================================================================================================================
; Name...........: _ImageSearch_Wait
; Description....: Waits for a specified amount of time for an image to appear on the screen.
; Syntax.........: _ImageSearch_Wait($iTimeOut, $sImagePath[, $iTolerance = 10[, ...[, $bReturnAll = False]]])
; Parameters.....: $iTimeOut     - The maximum time to wait, in milliseconds.
;                  $sImagePath   - The full path to the image file to search for.
;                  (Other parameters see _ImageSearch)
; Return values..: See _ImageSearch_Area. Returns the result of the first successful find, or the last result if timeout occurs.
; Author.........: Dao Van Trong - TRONG.PRO
; ===============================================================================================================================
Func _ImageSearch_Wait($iTimeOut, $sImagePath, $iTolerance = 10, $iCenterPos = 1, $iTransparent = -1, $bReturnAll = False)
	Local $iFindAll = ($bReturnAll ? 1 : 0)
	Return _ImageSearch_WaitArea($iTimeOut, $sImagePath, 0, 0, @DesktopWidth, @DesktopHeight, $iTolerance, $iTransparent, ($bReturnAll ? $__ImgSearch_MaximumResults : 1), $iCenterPos, $g_bImageSearch_Debug, 1.0, 1.0, 0.1, $iFindAll)
EndFunc   ;==>_ImageSearch_Wait

; #FUNCTION# ====================================================================================================================
; Name...........: _ImageSearch_WaitArea
; Description....: Waits for a specified amount of time for an image to appear within a specific area of the screen.
; Syntax.........: _ImageSearch_WaitArea($iTimeOut, $sImageFile[, $iLeft = 0[, ...[, $iFindAllOccurrences = 0]]])
; Parameters.....: $iTimeOut            - The maximum time to wait, in milliseconds.
;                  (Other parameters see _ImageSearch_Area)
; Return values..: See _ImageSearch_Area. Returns the result of the first successful find, or the last result if timeout occurs.
; Author.........: Dao Van Trong - TRONG.PRO
; ===============================================================================================================================
Func _ImageSearch_WaitArea($iTimeOut, $sImageFile, $iLeft = 0, $iTop = 0, $iRight = @DesktopWidth, $iBottom = @DesktopHeight, $iTolerance = 10, $iTransparent = -1, $iMultiResults = 1, $iCenterPos = 1, $iReturnDebug = 1, $fMinScale = 1.0, $fMaxScale = 1.0, $fScaleStep = 0.1, $iFindAllOccurrences = 0)
	Local $aReturn = __ImgSearch_Make2DResultArray(0)
	Local $hTimer = TimerInit()
	While (TimerDiff($hTimer) < $iTimeOut)
		Sleep($g_iImgSearchWait_SleepTime)
		$aReturn = _ImageSearch_Area($sImageFile, $iLeft, $iTop, $iRight, $iBottom, $iTolerance, $iTransparent, $iMultiResults, $iCenterPos, $iReturnDebug, $fMinScale, $fMaxScale, $fScaleStep, $iFindAllOccurrences)
		If $aReturn[0][0] > 0 Then ExitLoop
	WEnd
	Return $aReturn
EndFunc   ;==>_ImageSearch_WaitArea

; #INTERNAL_USE_ONLY#============================================================================================================
#Region Internal Functions

Func __ImgSearch_InitializeDllPath()
	; --- Step 1: Prioritize local DLL in @ScriptDir ---
	Local $sLocalDllPath
	If @AutoItX64 Then
		$sLocalDllPath = $g_sImageSearchDLL_Dir & "\" & $g_sImgSearchDllName_x64
	Else
		$sLocalDllPath = $g_sImageSearchDLL_Dir & "\" & $g_sImgSearchDllName_x86
	EndIf
	If FileExists($sLocalDllPath) And Not (StringInStr(@OSVersion, '2003') Or StringInStr(@OSVersion, 'XP')) Then
		If $g_bImageSearch_Debug Then ConsoleWrite(">> Local DLL found. Using: " & $sLocalDllPath & @CRLF)
		Return $sLocalDllPath
	EndIf
	; --- Step 2: Fallback to deploying the embedded DLL ---
	If $g_bImageSearch_Debug Then ConsoleWrite(">> No local DLL found. Deploying embedded version..." & @CRLF)
	$g_bImageSearch_IsEmbeddedUsed = True ; Set the flag indicating the embedded DLL is used
	Local $sDllHex, $sDllName
	If @AutoItX64 Then
		$sDllHex = __ImgSearch_DllBin_x64()
		$sDllName = $g_sImgSearchDllName_x64
	Else
		$sDllHex = __ImgSearch_DllBin_x86()
		$sDllName = $g_sImgSearchDllName_x86
	EndIf
	If Not FileExists(@TempDir) Then DirCreate(@TempDir)
	Local $sTempDllPath = @TempDir & "\" & $sDllName
	If (StringStripWS($sDllHex, 8) == "") Then
		ConsoleWrite("!> WARNING: DLL hex string is empty. Deployment skipped." & @CRLF)
		Return ""
	EndIf
	Local $hFile = FileOpen($sTempDllPath, 18) ; 16 (write) + 2 (create path)
	If ($hFile < 0) Then
		ConsoleWrite("!> UDF ERROR: Failed to open file for writing at: " & $sTempDllPath & @CRLF)
		Return ""
	EndIf
	FileWrite($hFile, BinaryToString($sDllHex))
	FileClose($hFile)
	If Not FileExists($sTempDllPath) Then
		ConsoleWrite("!> UDF ERROR: Failed to write embedded DLL to: " & $sTempDllPath & @CRLF)
		Return ""
	EndIf
	Return $sTempDllPath
EndFunc   ;==>__ImgSearch_InitializeDllPath

Func __ImgSearch_Make2DResultArray($iRows)
	Dim $aResult[$iRows + 1][5]
	$aResult[0][0] = $iRows
	Return $aResult
EndFunc   ;==>__ImgSearch_Make2DResultArray

Func __ImgSearch_ProcessResult($sContent, $iExpectedCount)
	If (StringStripWS($sContent, 8) == "") Then
		; This can happen if DLL returns {1}[] for a valid find but no coordinates.
		If ($iExpectedCount > 0) Then SetError(-3)
		Return __ImgSearch_Make2DResultArray(0)
	EndIf
	Local $aRecords
	If StringInStr($sContent, ",") Then
		$aRecords = StringSplit($sContent, ",")
		If @error Then
			SetError(-3)
			Return __ImgSearch_Make2DResultArray(0)
		EndIf
	Else
		; Explicitly handle the single result case to avoid StringSplit ambiguity.
		Dim $aRecords[2]
		$aRecords[0] = 1
		$aRecords[1] = $sContent
	EndIf
	If ($aRecords[0] <> $iExpectedCount) And $g_bImageSearch_Debug Then
		ConsoleWrite("!> UDF WARNING: Mismatch between expected count (" & $iExpectedCount & ") and parsed records (" & $aRecords[0] & ")." & @CRLF)
	EndIf
	Local $iRecordCount = $aRecords[0]
	Dim $aFinalResult[$iRecordCount + 1][5]
	Local $iValidCount = 0
	For $i = 1 To $iRecordCount
		Local $aParts = StringSplit($aRecords[$i], "|")
		If @error Or ($aParts[0] < 4) Then ContinueLoop
		$iValidCount += 1
		$aFinalResult[$iValidCount][0] = $iValidCount      ; Index
		$aFinalResult[$iValidCount][1] = Number($aParts[1]) ; X
		$aFinalResult[$iValidCount][2] = Number($aParts[2]) ; Y
		$aFinalResult[$iValidCount][3] = Number($aParts[3]) ; Width
		$aFinalResult[$iValidCount][4] = Number($aParts[4]) ; Height
	Next
	$aFinalResult[0][0] = $iValidCount
	ReDim $aFinalResult[$iValidCount + 1][5]
	Return $aFinalResult
EndFunc   ;==>__ImgSearch_ProcessResult

; #FUNCTION# ====================================================================================================================
; Name...........: __ImgSearch_StringBetween (User Provided)
; Description....: A robust, RegExp-based function to extract a substring between two delimiters.
; ===============================================================================================================================
Func __ImgSearch_StringBetween($sString, $sStart, $sEnd, $bCase = False)
	Local $iCase = (($bCase = False) ? 1 : 0)
	If StringInStr($sString, $sStart, $iCase) = 0 Then Return SetError(1, 0, "")
	If StringInStr($sString, $sEnd, $iCase) = 0 Then Return SetError(2, 0, "")
	Local $sPattern = "(?s)(?<=\Q" & $sStart & "\E).*?(?=\Q" & $sEnd & "\E)"
	If ($bCase = False) Then $sPattern = "(?si)(?<=\Q" & $sStart & "\E).*?(?=\Q" & $sEnd & "\E)"
	Local $aTmp = StringRegExp($sString, $sPattern, 1)
	If IsArray($aTmp) And UBound($aTmp) > 0 Then Return $aTmp[0]
	Return ""
EndFunc   ;==>__ImgSearch_StringBetween

Func __ImgSearch_DllBin_x64()
	; This function holds the hex data for ImageSearch_x64.dll
	; TODO: Add the hex string for the x64 DLL here.
	; File size: 78 KB
	; Architecture: x64
	; Generated by AutoIt Embedded File Generator
	Local $sB = '0x307834443541393030303033303030303030303430303030303046464646303030304238303030303030303030303030303034303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303045383030303030303045314642413045303042343039434432314238303134434344323135343638363937333230373037323646363737323631364432303633363136453645364637343230363236353230373237353645323036393645323034343446353332303644364636343635324530443044304132343030303030303030303030303030343130363044354230353637363330383035363736333038303536373633303831454641464430383045363736333038314546414338303833313637363330383145464143393038373236373633303830433146463030383043363736333038303536373632303836373637363330383145464143433038303436373633303831454641463830383034363736333038314546414639303830343637363330383145464146453038303436373633303835323639363336383035363736333038303030303030303030303030303030303530343530303030363438363033303041454444383636383030303030303030303030303030303046303030323232303042303230413030303033303031303030303130303030303030303030413030373033393042303030303130304130303030303030303830303130303030303030303130303030303030303230303030303530303032303030303030303030303035303030323030303030303030303030303530304230303030303430303030303030303030303030323030343030313030303031303030303030303030303030303130303030303030303030303030303030303130303030303030303030303030313030303030303030303030303030303030303030303130303030303030343034353042303035383030303030303134343430423030324330313030303030303430304230303134303430303030303045303041303031383142303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303634363137343631303030303030303030303030304130303030313030'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '36304635414345463230463131344343423837304343393035313843304230383230313231323343303445314335303241313634453738433439333543354334354332433339463031393146303742333735423433463634343036373531363937384436384239383341304531363037353843363646393431343134343243393031373031414343393645363233453931463342363846344534313234344146303537324531394634433830373642353734343041463035443346323035364245373032424546353936383741373338424631383036383644323139414631323039343434334238323043333243434334353337443643343745323139463939303437303833334544364645464345313435384239463745443435303037374344323942383131453238433239434632323934323343323932323034383341453134443338433034303146343134303237303837374346374637423435313434333538383333383031374531394646303834303244383343313032373434464538363846343038364130313437344431453343374632413846303037453135344231313438384434323031383630314230443830324542423032324634333443353436433433384239373133424338443637434331433834454241394235443833333930303345383743423532453138343130364330333735313730353833364143314534363033303946313030383737373242313036373136303146363842384439383238464431324442463337313635353630423039323330303338314342383943383835313731333735323230434345374438444630433746323133463041313536363445313444323132463734443636414531373644363438433734314245423945313430424138424435383734373346424342433743374531454646314438343046313033424638433131323443304634434546344238443543324432423042413030334234433335354433464241413734353135383145463334343239313634433743363343353141333346314237324146453033444332424644343830313031454231444445414144353741423841363938363636433734313233433633333341433838303630433338344546464346303632423632463137343746384541433130313743304530393939443842314333323138353838423630343441333339323932313346393345363732393942463535463645414539413930433230434433303532363443444443434438333444413545303343344434384244443530394331323944383343453731433234424234353630423837333239373034444542323034444236433446343543383644313441464331463444363844454643313435433041424343374330313830464241433531433732373246'
	$sB &= '32433639303634433638454431434531343939354432313546444438393536394638423035393942343246303245323839343230383143313844383933303432454332334643313631373436323330304234413144444641433032364543393038344541374336353135313243464331303145303335463442433045453234354634433532414233463133363636303531303846453132324243313439383138363836464643314638303234433342433337363437343933433939354544414330323733314638334133314538314541313731384243454436414131343436304133433835423246323536454446323345433734433543314638313330344338393545464330443834453533383830374643423439424133363034344530444237334634394541383843443430323337304436323630444232364239344441334543384545303135363130344330333330324244314138413844344242304630454332373632344635443145383443324242304432393135304541314231464338423338334638393134453445343930463432443041323538364341363334303538423934334143313036354330424236413030303230304239344530303235453232323343443339343334313342454532333130324336384541303442413839343635333730373031383334464639423832433630363037383144323939393339303042333938433546343146433533394632464530334538423831373030443042343345463834304135373336313237463235384435393830393042304545463634333738453044453046453230313734304342333038454431374546344238303337443643423044353831394530374642333931323834383044363030413839344237303833363337384645334531354238334639443243454339303243454534383231454531353443304535363139463037303330444635313630464341463735323846363431373830384630453241303638433332354341324639433531343845433032364433304236363732433233313634393730343837333141363238373742423341323431363030383444383930414535343231343438353736314136444635334431464138393130343142414631364335413435334244343945353732364330373531323045393239443430323838373446304436303839353333343041344441324330324242364444333634433744363330303832304334314143433937333236343136443931393233434242343836413232343138324334433846423631323033413032373431313243433246303546323331424434333735304230373530343842464445423137423545314333353334413433333838423033433032423338383444333531353039324639343846463930433731'
	$s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
	$s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
	$sB &= '33454445323230353133313939374230423045463130323546384531303239443132453631384342313130334338443845353133423037433638324244313834443235444131303643364435383836333332453731323031374239443743454146423437343946364536334137413537464636343137383031334334373433334545333031354134393039333433413541383435333632333645383336343235424535303130433332313530434446363839384239334344383035393744453544453239383731313132303139393842454538373930374530353242303438313846414143433632364237413545383439413430383237413237384442373938464134353145383145383233343432303434463036304338344643304544324531343031373830343633413843393630374441444432454237443732373842303146443133303441414337343130383039334144354234333341453730444146314344343538433643444230414543374335453438334134363530434538414230323234443142343637383745393634304231444542323143353631303337313445423433353238303231423534304245373434303633424441313030423744394531424430303846303236303833373934433232323445313037333631303442303333423034423030314542314141303437423339453445453230323037333243303341374338393531413342373338303036443431303646363630323533303030433335303938433333463438303342343932343336303146454435313846303836383230414635343732394630353937353435343934384144393439463435463336443233424630373139414446414343433439324246303443334246433246323638453230463139433735314334413844313430363439383343384646303834394545343434443444384243343044444334343734413244363533344333383336314545343044364133303946454236443833383935354332324333363932363732374332374441333532433538353344363635413531413037343435303337324339324330433238353034314534313832383330363234383138463841383135433539314333393838393837433139454332424535324331443038323641363237463136373738343033433930374130343031373642444533363841443044333330313238303034443330424134393542374444344141333335383244454641424531353342323930323030343134433632303032354136323132463438453143333438464336394646344335313038343942423244324244343636373434333038433338343438463732383034354341413736433243433334433043433536453135323636344346384238363534423036334445343937333638'
	$s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
	$sB &= '46353438303744374630303734324236433243324343304438353431414332414637343641343836443939413234364337323941463744314133303332463633393733313837363043344445333844304330333545344535423942324443453832303044344630393635433233343843333845413137373432313442463435384141344642333930414137363332334542374246393043393639313937413044324546443745444531303131413936374531383735424646323745384333423143344236324533464537363035414145434643433343324438364345433138323532434634333034353733373134303734373643363037343434353631354445313145314537314534343443453844454144363235443046333630343031413842303439454335363632414335454233463130443534304244383342344442313830354336414330313643323536443730344541443133304632384631313130374530453031383142303645423932303738313730364545313438313431323745343838334544343136373037343539433034323431313435463039353043373231463136303245423131323633383841344330394430393243303131343139454534464630383133343938334334303234384646434537354133433734363238303841433836313531323039393137343330364444353046314630453142334646313245363934383130463042324635333533413541323533453937464130333339443236393034463935444646433843383436323638363532464642303342343434343832343141303931443433454431303541383745444134393130464146423036354646384632353032354141353142463430413832303734304132343644444337354535353132423532413830383834433630313233383932413742444235453842443030453643383145323030454536454641333445463445324341433245364645423135313630383734313242414541444636344542303943304530303346364430463430433538343438413837463841363430463838383431303143363431363536303843353145453433374343454241343041394130433431443738314136374138363043424141363637304333443731384541374133303241393432344630394641353737423543353636323232303735363839303633413830313336433745324130354330343239333942304646334134343230303739303735393430453436373233394636343145314339343145303838464634344636433139463045413641353031314335323033303043303842333431334337464644313636433730303439333643363430303233343046423131374638333733373833423130424330453144313830453132303830433935383443433343'
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '43414546384239353243334333453642363443323237354143423945414244463545383831463734343335333046463444413041383831463730383742333844384135453139363933323931414433374431383243323034334133433344444530453038363036443734443944443746374534343830453830453538303830344546383731303345324145303344444545373145304143324539304535423931324146303537384335363246304537363641313336454231313532463131364443383145363030384139374132423037324544314536304636303742353930393130313431453045313041383438363634424639463344304330313939383231313043303945353630323831414138423430313443463833373933373433313333443231434639343539304636343031383034363531333533304430384643344130433035463945424436443237353046343543313330303038453530303632363030303141453244464141463036384230413937373134413146314643323845433635303442333933303235353235434142453434343339333032333437363331313241373736433545324535433841383736303843343531464539313634343835394434383645344338393634343938393636304635414143354139433045304538373035363134333535394338393139424430374538393830423245323131343546313134384331343733384542363637383238374445353733323831463138354530304637343241303032393846423339333837343232463530454233354539413233363141464134343039384530313042383434364535344338374238423234373845313937383643363435433830304246383634443236363945354630313839343536423244313341443542453338303133394534324131364246463942373945453636363143343831433446463738373330364435384430444331383535414634383946393637353646354337443737343038413531364335413746453841373533384331313945384634314243333046344137313641453930434234363430444238373839423132443739353346313042383343353138343330394434454345324346464636464332323034313430334332423734323033433244373431433343333037353135313739344644374631453437303133433738373430343343353837353038464346324542353038383942383530333832344436334543324143464644324642344546353030363535453745384641383643313830333837463734373734364131363345433331323138343636413036464246443637433231344345314432464541363034323433433530344538344330374545323537354538393441333843383438384243374338373333423438324246'
	$s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
	$sB &= '41314142304241383842413444344645323234313035384133394632463633433739363441374131304431334344334643393039354638374241464638344333424346463936434632323343443146303737374243344338424335313832313739304333463932373130353833373241463639394338363842323732353842373435423631453837383534323431383735333634424535333330383437384537423038454232453033323635383439354230324337313432453738324132363330453834354437383031354137313146463532414631303344444635303735373946434135313044373930333842443632343841324436313043383438343835353830443630343145303642323230353735303330303236314138334435384144413434344532313035353339383443353430303534373444373532373530443131363141324143463332423931333939353036363642334342323241433643423142333042433044303032363033443935444238423841424141303136324638393246363641314545413539313234394236433138393734383530333338423233433031373643344336384444383230304534463031333234313842463436303732323038464630353731303842303543433238303533383846443032413630364236413934304634394237303239343435443843433731343942384637303138353043454532354542314443364537354530463642323530314146363144333338423635454234384433323143414543333738383641393133443338323244344141314333333831324333454433433031334338433333463230343031353445323631353036344445383837413030324232383234364231363034433736393843414239463533414438443245413037323046304546363034453443413031333841314639373045414636303145303739304241393746453833383135373044383238464442433633303339443031433037333734364345384237424546303838363038343738464333303636383031383046463138443631313035333930383841393933304530454642423938303638444442424239373141423838354336383838383432463137373039313330343230333937344237414139443338314338343246364342363841413532434231344133343143314434303230384631363634353134413230363031303241323642424441324646434232303733323041353030303331423332303835423736373731463634413337433741324534363038304541383942383538333831433537363837373031413438343537463535394338353237363041414243373734353437334133443035453042363630413245313934443733423038444246313636393034444331303738313935313644'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '36364438413031303430383431383830303039343939353432414230314638344433424338373245394244353635433044373831464238304137353045384537393041343230373738393538314542303537443138363235383536343644444630333734383445383133433638333944464632384434363245313945343138333742323045343836383538303031423339414333433438393236313138363233463438343144353730343838353343444633363346373130304342363131374446304338463042313534313446334433354335353645444334443830393943454238334343464638334338434336394635363141333132393232344130373841304631413731394130353430413434384439444538303046453337424336363342444641304542343130463437454334313342454337354338423939304330413033354337333035443042323933364646453846323434393830314135463845394441303032463638373536463242343030353241333946303935384441443034433445433736323244344631464630303945383233423732303039463144374232453038464246433435423034383645464337354330303739333832343933334636464130304437314338433831393344433246394330413044463245344432384343453642394546354333413145344536413146333134463446343735424532463739313531423446363141373233413344354235333132303842383345334137383536314332334643364345434535373436323338423746333538393433453633414432464439423839313341453146433442363031343931454338313530354337413246303233453130334536303535364146423434333038373431423938353037453046333338383444304441303746464642414545394533304237303833354636383330323735313438333437303243363433384435314531464631383031454230373046313030324633304637463031334632313630343941333845353536453344413746313741343037413036373531344232394534354332374435304238303241324335303942444132374542303842354436303043463134353643334346313536333035354630444237453241343435303833313743323942463443413430333833383930433034354343344441324442384146303336434146464336343132424332463044413844383146384233463535383031374330324638363537383842463743344442313343304232443631413630303446373944304146413430354544424245433631424432414430373345393246433932383345323038363746463736313536324338383043323930464434373246323232313830314645353238343942384630434445323246414334364243354530'
	$sB &= '32343245303439463746354344314246453241413630323732344634423844344332444230383146393030303437374638353230334232324133343445303630333738333637413030303945303136343546384533313441433941324245304334374334414143433730373031444146343235464333303133383741323030303237384137343232324444304339344130444237464337313034373838334232423143453130303043343044364135423641463630313633384633464331334535313534413735303034393643453131304531344543453435304546424236373639414337313044373443364346443941363346303533343142383930464431423043334230414638373433373842344437303835433932424138383445423030414338463046304136383539453343363436413237463936303938463143423841373845323445393437463036373438463746363142304333343246353834383336453443383737383233313143383233354539304134383531463731303943353532363543444245424337303346463444314341343732443830454333313031313038424136304646353836453142413036463142334231373745464341304530344431434343314536413730444438373136433432324638433531353330383739413541343831453444454230444333433143303042384242433944393446424237363042363745414146423238344246303831333945373834423432464142373536423230344635424438463138353645344432383333434443364346373339343233333838423842384436353130303034334233324541313544313737303638393630354431353544313837324134313146383544383132453345314530353842384542384435303737413939433042334434343744323138364338413533333843434538434541413043414138434233313842353945303433313832324344413645323834383044413541363830343531304344463833413138323941344637354431464436453730313033463434393436394632313830463430383333443733424638384633463435453335363344393735313041434642393337343435383542413538334330344542353638443431423230333833383634333336453845433042453144414238333630313745313639453141383230443631303145334646343631433032414338364243444544384542374538304344313031433538304639363833343239343034313033383732383341303743343930353037313145304333313730393846314530323837303134373234433342393546454246364444383831443334463630353435394538413032373431344342424131354146463438333646303634303431384434383032304345394138423930'
	$sB &= '33344646454643424635444331353843433538314534454630433232334341343146374430343432334330343430424331423646314532333636363035303932384437303838353135383336463046384430443435433442344137464130334239413439443244333043304142384644303139433635363743363332433843354234303932324345383644303444343134364335303431394641324633343436333433343846443133343042433844463734453337463042453846313231433945333835423541303841323031464143453046324345453937464545423232353930313839303542434537413941304344363431443538323434394534423937373037314141323043314131333939354539303836324145313637313443373035454134323039433839313046454342344330313245343032374430393944423031423941424336383136303537304332463938443937453639333638353442453031343936383537463945313831304341324639303343333039384431383936463634303135453032453636373530413446333041453933453139453643424144353739353636373430384234373142384234453034433236393745303731303138384346303844393432333137313730313535463431344445423542343534393846364331383237353444423538333645304430393634324132303044333142394644354245393838333634303734304445464532323234323137443433334646363631394630334642303135363030453935453930323432383345313333303244363831464541363839373433343336353134354235413230434546383430323932413931364631463045444631373532464443353345303830333039344541304346423732304531423630414341464631333046363946353338303343324235333035374630464439324346373730454142443635384232423932353341393944313145363130324236424232433843323234454236383036353634363433303836363139384143353744333046383041453935324436454235373833433932303032343142373044374137304538344335383734323330384645373431453230333841344431414243364436444442413138324630363530344331464441303534383642433942464639463034453330304343324542303743343030413046363431303832303734313730344244423631383934434534344338303241303441384541433044384331373031453834384532384143323434453133463841383830314542313642444231414534324438454335344638463542433835353334363941323144433046423634434431343531423031453436373946394543313438313734313437433744463535354532454346383637454438313046'
	$s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
	$s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
	$s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
	$sB &= '44393241454232413244313234453930464434433545413437324133414242323331434532334531433230383746353934383045463334364431314646464236344336324241344432343734344433393236303731463631353942454139414143413637424443303541464638343845424633363444334231343633314533393142414539393033463945334432343145444143463437394545393246454231414330394439443730414133304145303344464239304538343946303044363539463333463037443734414438383035344233304434353044463638304130303039353230313943413031433843363839433638423241413242304438383538464135303531364542444444323338364345393032443541353742353733373845344434373746324230333030393738344431383243343441413846314646314238413031343238413134303933414332373535364532323435323146363034353038463732434543453639304630463645443131314434413844343436363831453246463046303846414638304637374137323430363141363430313630393044383137353233373643424634394241374434433033443245343632464344453730343239333333433234393835433337343941343330323546354245343833443846464132333438354434314238413738323732333442333344323843314231373046304233363239454339413034384236323739333634394232363731373436363146374338333842323345313133384332463245373046444146464430434343303438453031304239323036303942334342323437303830363533313038414330343036383846344437424542413439413041303633313536343931323338344432424637313835464539434534324531323545394138363636434641323734393137383032324542363337313438363643384634353330354537354244303243433633423337304638334436303330383735433842384345354343373043394534433242393546363041393238323236304431364643383338333742333545413544303846313132334232343842433132423631303546334541443534393033433735353446363046434237304638383336323037453734383137443030373138333344414142453738373438314546464131453145413145463842373535434531413637313030344542444243323335354443313538413243444330344139393745323831364132433742454338303430313344464430344545363443363343443030394441383944433230393033443743444344304543304246353341463343453741393434314346464336453830343243343230383933453446393539423144303044364345394641443741343230313835353939363833'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '34303139304438443936383544414141323633344235413742393636373639334142373837423741383730344335334636443631323944373030324235433944343346413834433242433634314636303341303830303930454633374230374634304145334542304531433032373431303736413545463045364631453038453338383831353945423037433630463030463139443438414439363731434237354338454233463334303830463445444338353045394639453230454545443845343046303038384331433230454230433431314242433938394142413045384145303837464643323432314141383342443144333732433746363730324130394344363538303138323037303430334138363934323841353833333434464136443746383545353431333039333746343134343843434238364333463245383633423033394638393134314443423630303934324133363432364442344342323138464145343042373531363034383835433035303643453233413545453630433031364342333338463141393439453638414234314441313246394333344639394139373434324335383733454344393431393743353338333235393538334545314236443143313036433735323543373035383631363946323946334541314138413130423132333533444338453543313039303330454234353532464437353132444430453145443235434445423043384434324346433735313446413933453634434544333634303842343030344542424233363533413939394144414238333135303231433044464433333034384643393237354144393532394234343831423441314544333132433134463831333345383246304138434232444634443239313634314536303334304544363944354643454645393044324330454236433831333933383730413446453944384633333342343130334343343930334543463236463830324632364441373245393831464645384141333534344238383642343333313645393143313043343644444139323246423046463230463537314338443737353842323843414145323741433143343636314534313745383031303138393241413936453035463135453043343433393143304638363432463835353435364633464245323633383543303637343244333835383743323833453338463443314637364630343438303133424639373731353242434641363337314432303830304130343741363545303242333434393242434337354635454133383138374436354530413633453334343631454239443838303038303830423346463841353336433438423445303438314330333637443734323731344245303832414638314145393044373430464646384237383933363436'
	$s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
	$s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
	$s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
	$s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
	$sB &= '46383133433041374630443432444234433842443234433839314343384346463333343543354335434339454232313432383538384133383738393546383845383032363230323745454231453431453736333242353343323446453130373438443742303239454344334532383434384332344343313435324334353445383530413131384644413836383133313643323831304132414345394335444637414230304530413645323938433934413631344641304338363542353241443435374243313137464644304537343533393532433730333343433237424132373033323542333935423637453944424146373130313045463744394131363634333132323031304446433134393639463138314543443830344343384136463539433031303338453536414136333430333042394433364334383537323930303943304446333630383145313237384135313831303439433743313632454230383536454641354444343739303539424637393638393032453634303034364230313234423338304145383132303142464145363135444434314631363732454546334337433031314338343538443034333844384238344233374631373443364132383930393633354437344146463931464441303442383137344131363434463839384646383430424546443245303737304130443633363630313445343836324435384137384530414639433139463038344646343532443833334435333730384531393831444439443630393843343342423243373033413745423430343834354241414131393036383944323138394141443446444133324635343939413642304145444338374336353339333435384245363345423543323445414433313430454641343344423635413441313039354244374232324131303645304138433542324143423731394230303444443246434444313736463339303542423646353032424138314143324244383446424530373642443232454639303846353143343533373439423030343930313939463645424435333135423334304437353343303842424646313034303134433135383343384434443545384444344546373543313537324334443046343244333635374331433235313032393342443337333136303942423932433436363431383046303444384442454530334344423544424134314336303332423735463034343736384231424330303343363430423846453041313045374443313045423633463934343842453232343834444541344536323033443434464233314333384435463031393446424345373731344541383532443235323138423838344339444330324537344245373933433436444632433844383731323130464135354530433146453038303435'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '45344330384530383345313046433632323838324336414643363343313542303735323835423643304244433834383035314431324445333433413032443034314331453930344341374135373537324136383741303844443233424632424144344543394634433936353344464630463636343836353931393339333933464242304133363042353336373844453634344135453835374230314239363446323636463744444636433130463846323035463445334436303737383346383431313135463137363138333334424234333233354541353435343733353736373336393937353333303232353846333541373434463345353842383334393736313232363342314241383635453746303834344246324430443339354536434631363436304442393930413945463635433344363244444432424145343830323734394146413238333134414646433543343845383439324231353641363239383541323143413045313643363038313830433239373934313337433837374146303130364441303633304133373930423733313633384333344443303046424644433646443439393432443444313645443038393130304545423931343432433338464436353938353635304541323835344641314442463236313542313141383042363739433539434638343739364238354438463431463743343330344344363636433238423634343530424533333934364536314542383345413630344436454139304637343538344533363037313330304541443941464532463539444634363832314437313942463338354138423845423338384137433136343231303543304543454336323930444643393043413943423437463235393532433446343433424646374344364539344645424331363230414342343141463144373637303238423636303933463337343334433231334134344542323634463634343741393942384433303436353443363334373443333341383242394330383432363435464332454536374330313844423739304237373643454231354143313234364138413235444430354346443145393646454445453339344337343538384244343530334542463142393445433831393533383634303439344342433339373043363845383936313941343142463639433337334631394134343032343836313730463836354637384532373338304536323639344537454337423936393131344346423142384535353645313133414239364632423235323737394135444437303734363133343733354533434239373537424239393046343346423237383139304336324130384434314146454236303736343837303145463341423233323533354630303337304639314130434646434634323937453032'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '38353043453837423634313536423039453043343844324538344346353233344134453836353933383542303943454332314437463538303039353739304239303030353630333234413741323230413130444639363434313638303033304532364543433143443937383745413236443639333132334241383543424242353335413432344343453832324136344534334245444138353034343346333331343845304337323339313232343133443441313730393043413431464133433034463336373734313644453031433444383330463139363943364339314537443335433844344544433638434237373644334445463438433445423538424231313036364533423446304542323830314442333043463941413843353741363632344346413544363541304241333445443543384230443939443446303930453030303734373738373738334344313337464644363044324133353937413030394139304334443338313135443737413344334635384135384434383043303532453044304636383244433943463646373345453431354542343041363244444243463734333441413341444534353146323943354430353931463430313430353331363934453133313931383931454446304244343733434535334230394646363633333144343332414134333843444431393232324538454234353937394134383336443236303630353037374539384430354444373735313039394645374442433636304631303645424344453632303645314241424138433434353630383041444345444643383531343633333230324631464539364533383233303933393134313945424538373531313338364643463132453341364239453035433143323830323636343331334537324630323937433232464541453945423238423430414141303133323141383230333336433333444437343545304633453430344142424338424330333646353741413230424634313646343232373035433531464546373930304242304233303330394135463637414343413643423541373045304236464638384634343835383439433037413435343841383137413244424144354136323230384330323745304430383033373531363034413246423639434643343239323130454243383930444236314231333042304641343041334432443534333535303045343046343443433144304441364245414643424633383034343144303230323430414430303937444242363638443431323330334332343534314236303245384633343439344431343446433930304130393534384143334237424430383235303045463646464132454134353534304235443232383243414331353041303330364142393338354331384231333642303237'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '46303336414243433835314342374141354146393138384337313833424130454531444135324330454441363534373545304531323342424538393937383431343336393837433136304443434137454232324343433143423535463041424342303046464330323839443834433939354134444539373037303042353838333233423845383941383845373244313739413736303935414638363035373843453642363545394445364143373043414430303031374331463035343435303745303230464143394339433943394341303932313432323943394339433943333636363736383637423942394442363031373238343146383030463545383131464534453437363332363435343046343433344632453445344534323231323030463238333339333931393339353637463643383239343339333933393339413442344243443839433843334333394630303838303230333039433943394339433343353236303732394339433943394337453930413442453943394339433943434344384538464334453445343639453130383132433441373234453445344534453836394341414238344534453445344543364434444545453237323334463445464130433832323032453237323732373237343035303738383632373237323732373938414543384532393339313237323746433043383332323933393339333933334334453541373039333933393339333743383639324134443339363933393342414343453230314530333632313637423737443146374533464634374444383246454445343336463230383746433031463236333830463437354237324431434335464543304630314339433943394339443832343434363430434230454445344338313844374538463432464131434539454343354542374343364633363533454332463941463834453143333943314646383935383534434336314345354531453030303232413830384644383046414633383746373237323432384643344130363836373635364536353931313333324543373236393633374646383846423430424345343243383930303036463341373033463445433846463931423039454634363936463733373437323635363136443242373845443130373243314246353437333739384536353441383638324231364437463346424331373643343331303046323030444438373236463732453037364236334446463031414633333238433134303337433938463442443230313645313034443033314534314430344439453138323033463241414632353446453545433243383037463934413830323434333236433231434630343534434638344441373232313137384334333546433534323245363442344243314644433832314333'
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '43433832303230363639414330463035303334353030433143423032444232384238324543324342334245324433453645363936313746304646393146323036453741303636353737324437414136333033384343353141363645323336384634304332424238414644424646314636433337363746304545363241454341324542463535373437433245363536453637363337383738383538353339383741393433363446313831453933453335373537333037303033363733373736393242333038333542364631373241314143323636353642314430374433433846463836453146373632453638304233333431374138363236344632463834313941353934313639383230314642354231323131334346323032463243373531363742343538333146373236313731394637423244414445303043343436313146303836333533454641444436444134333644364636344132364531463036423446363631414335323645314635453634373537303733424633303041394532453932373236313846374435343938343342324345324135343634363546313038374241463246363337353237314635383333303930414545353044324334363544364530313442333837323033313846454343343336343236424442333642343246364337323632363932423246373639423642353734383635344636323639373634433146453933303144313033413634353143303841304535374536373445323836363535343241344344374143424337413738353636453246364537323637434330314146453131383632373931363733463932463136444134313142423236424545303032463146353333384241433733413935324431374343394444453739333846423942304531453637423234304144383741443344314636433735373841413644353833423141323232463246303036383734373545443631383037414638453632463631434443434534414146303732454234303341343439463837364442343244314639463246453030393134443731343043364636323545373941453944443130353433324439463030314630303531323536343432364244464131364136444339373933463846443143313536363132463336344636413541433130363138424234343146334333464231303743314133374136393632354143463246413637414230333636453146413136393634314641433141454339303530334636463141384134323238363542383734304438374331323344354346323841453445363044414335303131454435324637333631423137423143443036313738433432464437444130373736453636393636463132464446333839374438393744443546304630363231443834364531323232313031313334'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '36373933453646304530443043353246323834354337303139323641363134303131463530303233304345433531463832443831393330304234453444314543324646334237303130463030454530304344303041433030383730303736303432394536373930304445463030303230443042304136394144323636393532303336374437303642424146303142434530334638463233443835333546373846323334364631463641443741433746313932323846453146463246364532323532314546303143453031414430313843303136373031353630313434374345444246334241443336433536363544433941304639363145384331383445443144424241373730393044433630444537314345454234364646433336304644443345453330463837433138343343434630433042304130373541464645364330363030373432464230363133313346323043463030414530303844304433314543314437303643303437303730414446303633353042324144464444464241363045314642324441303744313044393030463146304334324336354542433931344445303230453232343045303935463037434436314136373930363332344637324532453244424536453232363045464530413534313230363334313043444342464631443045393231334530313144303046433030443730304336303334313542324232444235334441313546303242354532354645323533343144303634433842464637464336314146303138453031364430313443303132373031313630313037363546423044433343434630453044304337323134453045354439324337413737303938453634303835343037303646463737374235314330463630443435314637343845314236343131464631333433304539323041463030384631313237364643453030364430303443303032313743383030344646334343463731313130443230344338334346464530303634323836304330323030304330313131303833363435344643313336384530383037394533323139324542443639423344443145314436344334303633344333303142453236433030433730304239374244303242434137453030353546314230414645314537384431364333363036363430463334304539323646353032453330333230353445303444363236313141463344414141443937333230323837434230333041453130363841433233344446413644424537353730363332383645393045383639463932343639433036454642314430423434444637313644383036454636363133313334393336384637443536313333323046363733383036363530364134433139324344373730364145303630463532433141344439394341343037454330374135394633'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '38323630373835303243333136303130413837433642333636464234413636313537343135303633343134463246453746433145363044413537384146383631393146374537303643384334393131453642463142344433333141443445323142353631423445314231423837354338334630333231343530303933373332383634304636313938354236353630364243453730463238423035383738324531374136313742324546303535313739313032373530324141453735303830454439314530363545444641434534333937363938343732313138334338333943354333453030303230393038383432324344463330373036313232364145344241424437413731384532444239313243313633304136323832373838373730324538374230333641444632453430303643303738363436383233344139374346384332343246393032424630313037383545433939353443393845383838443937323135363438303837354630323338423531334143373843373838393442464334454133394235413944314137414234463930313131354243393232393438373842303146383644423035333239383030413745384546334445303632323142463238373631424137333139324139344641304245434536353443343931433739383630363739313138433038314632463730373939324142363030324338454641344543333638314243423837393243353745304246323937393235313746304130393243423838313343383138374141303130383733454544443430454231303139343834333742363745323732393736344134453441374630383745414638303830363230414146323538313041353830313446334535303534303145363938303032453346303443303138433234313536363132443234423035443930373436353637304537393430443034303430383333383231383935463546343730433244443437453539354637313646343931363641413934313039373136464446413833353837353337393635364236454534323844433445344536453046313838383730334643383430413045303835423830313546413030383832434139363443343032444435463034313346324634453637353836443244304332393539344642314435373634304636373436383531344642334446303536313134354636463636354637323631354346423246423534363631433734463041413133303330323134333443464637373139353833343637363834453633323932303331333933393332324433323330333042423744393738343334324532303530324534413245303836433631373539413430443032303631373232434238363330433837383434453443353233333434413630384233413637383735364437'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '46304638313930354541463333374534463343414330453035393130314446303830454232394245334433374331363643353942343244393236373330304537424441454645303743393231463046363839333245413437343045443336433345454430333935313645343130393537423937463844393745304644463743304541433937383730464641393831363134354137433941363742333143393935373939373635383741393946423536374431333146374330454636393941463045354539413545424541373744344632343630304533373942313634383338304546343942463033444544463734373046423739433245373042383045454639434546363942363339443739444136323739453130323830453242333933454433383039463445373530454142394633374139333434374245414331363439413036344130413634434544374244414137433046314132354538304634304545424133313639434242443336433345454341334343413742343046363541393743444146373032423736383045333941413136453433434141303937364137443937304142464330463035414346373045464330454245303830453243414344463046453541433846444231423137354630453338414535453736443631363546414644464534413933343343414642303045324342313446463337443037444630464632313635343736463441394137443937433941423235434143423242384235373439314344313937363046444642354237453042353044423639414545423942443130304532313137323434313246354534343237344646333739433038383736433044464530363739464432334541424243424542303045373743323137373843323842454636393336443943414430444330453836443531454137463944453541373630454230313631383737463046333639334543443137443832303634443838413334353943344542323634463841413830464545303630343733464244454137393630464439324630463146313732343343323743464339334633463541354637414339373346323943374639413941423039433343323743464241453345334642464238394435333746394546444130453841444246363733464244454345423630464135444232463046433031374330464239364346433945344534303244433046314431373237434643393733314433423342353436303733463239433343383238323944394433453237434643394238423845304538303344444443424145443542304631443137304633363246393037433945453646334137423435434230454132343544354637374537443346303130444531363046323730363730354633364442443937373545304542434445373733'
	$s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
	$s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
	$s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
	$s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
	$sB &= '38384234433234303845393430374446354646303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030'
	$s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
	$s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
	$s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
	$sB &= '303030'
	Return $sB
EndFunc    ;==>__ImgSearch_DllBin_x64

Func __ImgSearch_DllBin_x86()
	; This function holds the hex data for ImageSearch_x86.dll
	; TODO: Add the hex string for the x86 DLL here.
	; File size: 69 KB
	; Architecture: x86 (For Windows XP - Not support AVX2)
	; Generated by AutoIt Embedded File Generator
	Local $sB = '0x307834443541393030303033303030303030303430303030303046464646303030304238303030303030303030303030303034303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303046303030303030303045314642413045303042343039434432314238303134434344323135343638363937333230373037323646363737323631364432303633363136453645364637343230363236353230373237353645323036393645323034343446353332303644364636343635324530443044304132343030303030303030303030303030303736323635463734333033304241343433303330424134343330333042413435383945393541343535303330424134353839454130413437303033304241343538394541314134334430333042413434413742393841343441303330424134343330333041413432323033304241343538394541344134343230333042413435383945393041343432303330424134353839453931413434323033304241343538394539364134343230333042413435323639363336383433303330424134303030303030303030303030303030303030303030303030303030303030303035303435303030303443303130333030394444443836363830303030303030303030303030303030453030303032323130423031304130303030313030313030303031303030303030304330303930304430443630413030303044303039303030304530304130303030303030303130303031303030303030303032303030303035303030313030303030303030303030353030303130303030303030303030303046303041303030303034303030303030303030303030303230303430303130303030313030303030313030303030303030303130303030303130303030303030303030303030313030303030303031384535304130303534303030303030313445343041303030343031303030303030453030413030313430343030303030303030303030303030303030303030303030303030303030303030303030303643453530413030313030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030443044383041303034383030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030303030363436313734363130303030303030303030433030393030303031303030303030303030303030303030303430'
	$s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
	$s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
	$sB &= '35463932424332384437383031383937444630334246453736303531463546454338324134323442373435304339304638363638444646303543444637353546433533353242343834443531343541374530373033443834363242463837354446304646454246454434363333433944304241303235344637453230463930433146374439304243383531303238324343464242423841324244383842464237324634384638354636373432413844363432343946454444423432423438423535333634364141463435313532353035374135354444433745374230393031324138334337303234453735444141363636383930464238433336304234303831413046433337464236333637303041443538344245354542314544463330343734313433354644384138363438384232303445384535303838353637434346394144373438313631303043423231433134334337424336333444444138353543303138313431434345413446463033343430463739414636423243353343373744464433373938353630304630333446423734354333393146373535383641323030364246444146444136384633363842463043313735463032343342463337343239424237333643463836463534343831383342434237353033384434383143363643433532373435303032374336374641303046434242303133423034324339324542303236304133463646413435413338393037463643333441434543434531433134314534373030313943423839334333354235463545454446453432433838334331303835313546383641383732363638353432324145453141364342383539304639354330453030383041373331423545354631303535303836354145393838304144453930434536364643433130353630433030443845333846344535463735304343323342373531303734314633313046423736444246414445384430313734323432303831384346464644303834433037353736433630323641344237454643334137354536354235464137304337463438433132324239373439453038444535424431333938324437363933464136304335393537463446304336303843303143304337394246353735394430364437373144343830364234363433343735453942373542304145343130354246464246383333444545413735354632383333443946303838414335444130413935443331383531353134324541303838383145353234354341323231313945343634354638433730304233344433314238453231433738303443343338423832424630344137333146303134463843364431464142364539363744313038394632384439424546313038413036444132424242424242303535313035324136423630383143304335'
	$s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
	$s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
	$s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
	$s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
	$sB &= '33303238414338333237363338363033424641304638444344334236423337314642393339333233343133384234343741344335303735333643333730313045414434304431434632464442313544433644453046383531383135433842384630413835313041454636304434393930464246463041433531364146423531354530383031314642344346384634413643364436394643434335303532323231433530303635333632364232333131454137434533333932454343323830323943443846373336414342453731394245383146413833393730313437323032384237434132303030344143453244333945364638343734384544333339363233433046383233343738424242333632323035323744353245393138344130384338433845443432344532333044414342303530463044394338423842434346304230433533383843303246363438333144424634463830363232423030424431364242323636393036393942304638383435303537383030333438373235433843373539463338393841453039383441444335383538323037303546323530453538423730363443353639313035443934443238424337443031343630463836383335374144453730303637343738354130463037354238304246334146313039384435303032363641323833433030413334333746453045304337354635324243324431463835304338443236343931314242434546443034433243313636303631393432344330464641464433433131313643313043343242394332353242434242383933323434393932463730444445324646384539303344314331464132384332433145383146303343324438354332443139444237454135424141423132373838303833374330423130384133433042324230433242463345333739384130413743323931413135314446444239313937324633323433304642444444414532443344334441443044453039303130453345354442433642464637363336453833334339444145393842463338384130324233353434374237393844313430433337353645314633444243363243334231443735453838384438423844433442423844444146443745444336314332344530364346433039344334303338464431413646393135414630334634303435443146343045414637433845453633354236304642383533323836384442373931433042444336434336343446383031304538393436343644453639304139303043393439384130463833383742343641343844343538424445424445424336433538374439313930463434353434453935314344324635464436394236433841343638353138424430313243314541353045324641424531464245433145313130304244313632343832'
	$s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
	$s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
	$s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
	$s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
	$sB &= '36323237373631383744413634323837344141463341423843384243383242433034454531454430454331463932324439384431343938433235463542303930463444394342464539414130413334433841393746423336323038383342463431353842353241344230343538354138423033334243363737464539463546454235343242463044314545303344364632433145453146303346323342344230384637443633353035433133434236374232413042384430344635373731463941343036443242433631303831463430344543463234444643463843333544303744363445303835314337343734443839344631304438444436323646343431373530454233333744354136363630334639303546304546463638313933334330303735363833433830393036374441334646393742433833343330343143303336343930353731323346423331463038313031433834303144394437313336383144323438304231453044303538433030423244443041434636384230463038303431373230333238354230464530343034413835463231383335453443373030343033463731374536374338443437453331344141463831303536383934363338453633434135363036433445353046434132333233384638343634303735314432434337323031434544303346423136383533304343323730434634373246433433303234303838393946444342384437373841443344433343374236303035363336314538433739433833423631354539384537424432384234333130393431424344304332313643424544323430303241353036353335303943333038344132353444443546314645333030333945344145444230434631323645324143333532304636433030314639373430453443454145313932343938363733344531303333423630323246383036414430353632303839334133343839333832303032444335344241313431443234333438333636343042433938383337384332374533433342463232384133353036313343393446303245344146453346343630353731444637343342434545434633343330343342333931394639433943373736343330343842374245434642413139323438303131303439323245313241384334454144394536463730453030413033354641414530433130363346393630383537373432333434393636323141413230323938424536303343334243313733313338433541334633454530463542374136373846343839434334453334443146383839303136363630333039434332353336374430373530414630374635354438453534393735334135334336323635303842313938424437323736434646353535323541334244333733313846463039384237363332304535'
	$s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
	$s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
	$sB &= '30364330323738423838424346393436433745353537313437463038423438333946303543353032303832344239463731423130303339324632434245413045363438423538443339323037433635314645383734343738413445313837303032343846383230433433314432343231423141453845383537303830434434314142463130373939383341433242463537303735363842353335443844423830433033434635453244383033433346303841464435394646313733313735434346383337394535384433354643304330303735313242303031353534423736384135333844464630323346333243303635373736393330383435463438383333393030334141313035373844413530373030323939393039393538433943394430334632343334334633363443334142463541454237313130334238393946344245324633373330414641433437443242463341383430464639423838303238424630334246393735314538443043314544413842323837384544324633453444334338423845433931313145433730314541434344424135304331444243374631413033414635363446394142384542343130393339343731344546323631363141313734343844314333363533433338393242364439323531353139423434464536364530343335363839373731303032303733334432363638393134303341384130433444324133314646464336463834313330384343323430383146393538383544393131443837413844303446423846333630303532443438364336323845383036463041443630364433433336313141343743333939433139333241363335394231443939463435454134363339453136364535373242433242463941334643463434383638463941463933424638304430394141303939314645303343313842344530383242434141413738323144343834453137373533324246413245373930434136394631383541344530433032433336343738363830384232313942303034393442444636394638413631353738314645343839323234303937364437384237393746463230374132443246423431454630334437343630334332334246304346463345453739463444303439363246344638433043324441363938333137343543364646344634434638463736383133323644363946333330343430353745414432303231344145334230413243393035463835354244464245383343363143353637354435384231333532314630354344413032323332303333413034303842423045423930453633354642303430423541364142323631393537344437433835384131413045463131453430373037444234463643313032373036424638364146383530323433393341373434343132324133'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '38303830434237303536453033313044414330313034383735444335453930314333383843433346464232303138314130383036393432344531453038373536303531334245343734374142343645383945363335443539453436333735453036384333373432314136383542343738363439464532424137343731303844353430303032353230364332363443303630364338323839314635324139353334304638444335373134303242434446343839434139463535464436383343373143454238313236323937353138414442464145304636454144374531303242353631454638363034333044313538393538314144393545344142323831303837463230374530413442344444433435324243323536333230384330374132383830434146454534353436423831423534303842433141323842463745303136373545373646323133463939463746453842433638424632363838303137303439453630433132433742333841434533444242313337304543314537303443314530333643323736383043323541393744384634363638423045423030443044353632383036443032343533354142323332414139303744313138323741424133303139394141364130393143413539353235413539354141444330394536393539343035443041383437393243343134424142394234333030343646333242463131413646334130413130414141383345433637443035303331374430363443303444463831303643434533323738414243363333313634384634464341354241464334393833453842413844374231423334463041433835303231433731333735374646383945313237383832354138374543453936443038423535343442353944323830464141384238453035463237383737433034353738323037353138353630434239464637444331373733463034464338413037353638383031324134304542343338333635313441454333433045463031383345373539384434443134304335303737383833304131463138453539353046463441363435383346304339383934323439423538463734443237314535343833374443413045393237354337303032413943353830333831303243383132304435364637303845313934393844303843324135324532463234343433303037343037353943423045374441443039374435314636334235303034323234323941333546383039334230383735303533333032303834314431323336443834323132363038353135314646384536384246383043313031344345304646353031364338394442394230344638454645433934314135354534364432423732363130434338454137334334313034303435373141394437343038383630323230423034393837354233'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '37324631454139383232383046384538423045373230324646313045383942383544313441334237353043373245433139373735364334384642344633453031393045384342454236413043313534354341333041314632343539323633363045383435373832464233414331433430394232443631333531354630393143354230453436303044454442303145304642314144334238333638304245303841343539444436364232464532453733304638423037423943374530464537324631393131383641323437463930373630413643383443363543363831334433383236413032443237313338323742313241383446453230463539303634413439303038433633393336354146433634313342374437434433314243333744453738343545414132304542384341323443434137323345384437414334314442373837413532333832343536443043414432304245303038354642464434393546304637444639363744443838334546303431363342464237323442303342323746454537374544313833454646333736443235383930374646443339463737374631424536323538353339374137353035333934354438373430453845424341344544333330383034414238423134454241423539424146424331373830364338383137443044443437333131334345444236453132413030313131343034454245363430453044383637393243414433453044434530453041463045334134443542443746394246323946464545303838374431433039353233364242364138373146344232333730383446443935413432324146424438344230314631323044344436434532344141303138383333323442324230314634393231344230303538343833434536303835393235394132433035423533333234423442343439414336383846373143313736343032453545353639314143383343413035364143313643334438433735303546444441384538443930423241333436333834393638303634303041363836454243423742464342354138333438464246463636433734304143304138393438303331303146414335424643444443363430323131383333383834383246343235454645374234314446413238443530464238314336333544363732434435333537363633393444453639423535304232444331443936454538434131354330424432373844313839424643303343334245353342374433433032373334444537434136333931444444433330323833373737443642424630433834343333463945303544423737463735313833303533373230384438383633303739343733333132373438373042424542383336304234383036303146463836303333303033463230323646303343333832303041324635'
	$s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
	$s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
	$sB &= '37334130364241303242373732344237413631384437423130394644323337453333343044354330334346333330433338423133384436334138313445443230383146434131453046353130393934363636414530444246343135344243443034333038393138354235413033463132433532383931304642383035464644434432314130443630434643344338363134384434343836313037453141363132384330463634413134384244373236304543343235454534453130304643303037324536303430374634374138443837304345413030303138313041413034303643304339363538323035353633414634453730363934383130354339454243393938433333303038314132463239453239344230394633324334433839343143304532413045323135383546444230383535363842374138303636303532413232453234303831323141334344423231314637453742383641413037433235383043383836383239443338424338324430393330303841373937414332343236323035303341324246304146354237434630353043313635424133373339353330434433424137384446364445303338414443424139453932323134314438304441364535344333344445323432413235434142343235353037383043313538313042324330353734323046413345324239454545434546383143313730454138453730313230383034333439384530313635413142394235304634353630323346393545314532343138334641303545303630303834383743423732374443323130314142334535313433313936463935354536303839344539324235304446463344354138353241333236413234353938423745333036343339363238333736353345433336424538314639424637434544344532383634334438453134463637333231383543303543343636343833374533443930314632423833354331393831364533443931383435453344304337323635393039333835344533443844383243383935343141453345334438463836324533443932414539434243333238413145334442353032384430453344423441434434453035343037313043433634364132433530393930394143333033304542303746313535453031414331353131393945354235374331303138424239314242384635444530353741324230334339443745304642353935394233313544343441323131393735383331394241464635333537424634454536343042424242424341434438464334383544343238353743463733323233443845423635413439363936344537414638463646424430394643394536444131373138333346304639304638453436394133394331313434454630313032323333444639443136434331383232384530'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '45384134313034353143363835314639373236414244344333323244354343304130324345413143353244343644303739344341304235304331443946314546364331373831383231313434303342333439353046363834373437313338453239314434414642393036313439353937393231344635454630464446304644453846443143393338373143453846444630464432423542453631443441353943434644373438303037353330323238303033303339393034334632334135334438464439384644423238373945314341384644343546333444463046444644344146333739443846444530464436334345383033393330373435384131464633423830444538463042433630313330343045423430343944383430423745343031304130323034443837354234423646323242303333304542323846353442394435413332453844303439304632464632354546343035343035364635443530324134303736353237323830314235384238443443463141313435333531343541303935443244454244424342333232414441314232424542304334383035423232464430313131424345434334464546374238353842324242313242324246393430383130323738413230434245333631444541374545324542424242303230344643424244314246463144373441323045304244464545364143433038303642434437304239443539433943364233304645354238323342444331373032453135303432423436423637424238424443464542313942353832333042393533323830353533453344364334343137343941343144354131373530363731423631353930453931364345423630374237303644413130373630363031383745354441453438383139343641344330444534374233373533363339324246343532383834413542433138353636344444413930413431334530303534433735413645423233343632454234463635363646423236314245454635423933313733433346373844434644333241393734323931414330353230443642313831303431333943383146373741304644413046443934333134433542424438413033453032313745453841393534444136313333463638413732343735363338433645434432373842433044313130374337343331373438434443353735443933434442414237353439424630363437424345453438344134443643394545373936413245383441314143334238413443324646463435343635464130304342393342323138353534344343303637363830383335324630433742393146333141353644303836343839323532333936353946414644353433303842353830383541324333333139413446453646424639353332373433423836333438334641313030'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '33303643363433303232444542303430413230384243323935324330463544464133373433453646383141443139444343394130463734324641323431463833393444413036363138393541383035383346444636304430343230383836303636433734333033303133303732384130393934353546434335413533373443443035463832343442373143444532363545423835333530303646313031464130313344463743313845373531334143343137383831364546354134454235333537333544334638333037303038413041324639303738304432433234344234304332443241444437333734344639433232383032323146314439344137384334353234353636413136363534384241313735434238303630333035454231423339384343334331303045373942344333463534343937413035323938303639433931303444443937324143333843454232313843314343383746463035373043303241344438443834303830434544424345434331463839413646433033463145434135454530304642464438303746374442343534443942303238363236353545413730453645304532464246433046363833383938323041304237383544423739304642383833453834373044413043444543393832303331383331343534374332453038314237364642303338363233414436423531394432303839393839453634423433354433424342393431303839433430303130313141463443344641363444364143313033323032304542434632334542383034374645383439393846303832303038424431333345354131313636414543424544394532304642434436463544364232384232303233443632334345363831313430464533353843423439353032463044364131353331304345424545303931434630373033464541313533464537373130413838303846323131333735453436383339463943383036303841304146343746373435453845343444373136464534453045413442414336343436343039364441433239373038303933393730333039373032464335353034423432313735414338443735463445434330344441303746394139424143433030334339423035353844344330444530423141394435383534343444354337364138393935363031353030383041393434453536393338434136344138434338453841344641384335343045373030413342433245434331413441343739394545364432343630363930393442304344303136454137423037434243383343364632414343303035384644433334433038443831433746463745334238304638413132363143314544333632314542383841353441333943443145304430303343303542433030453041343430424331314136303036334343'
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '38323930363538363637344346463333434346383038453943413642414341394143464633334344434541464330433241333437393945453737393432353436413834394546333739394545374238433844454638304132424346463333433446313632433338343234453630454443454633334337363838394530313938323546333145363644463745303641433235314639303736433030363738394233433046434644454630413638344134303737393945453746393031423439424231433738333846423744333945453735463642373735314131313630364342463230303644313544344536303130344544333242364241453231453834433630313831333641373846363832303333363733313845364339423644353646363538374245303036393646413743323332334444464637393538333736373635364536353732363936333437413037434536364438314244373730363835343733364646443041453930453444414339464638383242363936463733373437323635363136444543363741383445423636423037373337393445363534414145463733383742383236443538343839373230304431303032444236344636373236463732303030314646374546423738313842394644333633363430324144464637383831363830333034334434373243394136324332363831434531364638323037303242444646363445464243303532443032463136383931423535364545343944374646423642364536463737364544453738363336353730373436393646364533454434364446463746444539454639384539363632363136343230363136433643364636333631343941444531353044303030463030314235454334394539343545364543314644394446444533453536363937333735363632303433324232423036353235343341323034453646373443454246433035304231444137353637363436443145424337393230373436463230363330373737363646383646364437303643363533453130424332313030443946444136303746393245323634333546353434393444343530463445353535374645364642303532343944383136344434463445343535343431353235393433353435393630393746464444353034354434313734463443344334313534343534313134374136334341364438464333453033373634313644344639373235423430323431374338303039324332453743383733453442434639314142304636324346464646464639314138463534363031303230333034303530363037303830393041304230433044304530463130313131323133313431353136464641414646464631373138313931413142314331443145314632303231323232333234323532363237324332'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '42394538424646463846393334383044313742323131334638323232303645333438464142384442443133463934343036344333463846313133433646364236373846323342343743323736304237453533394639384339383437303139373231434630393939374333303630433836303646433737334438453435373437393732313942304242394338383046413134303945344441334246413234303633303937383034314445304436393646413733433236363042444537373942333034343737303945383444344630433135434338303442423537303341373630363039354234433146314234354639454334354535433039453436303032303733304438373937303237463636453034394646423134303645364642354333323244433132454438394637373032344332303246333635303646463830313430383345333339313130463331313838434146323644383235363938453943334538463843304236394230313833463237443435424544324438364534333742463937364643333232413545303630323741454643323032463643444534373043304138373330303635434644383446363945453744394645383036343438324332363638314642333031424230443837303944374146353844393131394336394437393039373035343743313645413342443943303642344346354630344146304343464231324438374146453845343146303837423045344546434634353630304338343737423643393636303146364631383739333233383645304336464344433433304537314534303137374242344545313433323738364631304346464442344237364444393830413030383037444633374632314534434546383735464137464443344445443841343732423531304137423431303836313345423930433730384137464531384546363330334331303933333638314630344137434644343145444546463830333638343846333734383330303233374242393030323530434646384646344642393230303530393237344646303442433538394539423034374645433038463345383238343443423038464230313639454637413037414546453746463038303631384234363438323044374236343446303239464638444533413045324538344337433739363538303836444333343236463937433946333536373036453946363639303938364136423538364446464437333732314346383936303438333733343943453438384230414537374634313033343146363742343937323630433846443736363243363242383630393739434646453433373836453736324546433436364234323745303738464641433230343642374237383832453736314242304230323346454430373743373946303631'
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$s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
	$sB &= '30303033323030333030303332303033353030324530303335303032453030333230303335303032453030333530303030303035383030314130303031303034433030363530303637303036313030364330303433303036463030373030303739303037323030363930303637303036383030373430303030303034343030363130303646303032303030353630303631303036453030323030303534303037323030364630303645303036373030323030303244303032303030353430303532303034463030344530303437303032453030353030303532303034463030303030303434303030303030303130303536303036313030373230303436303036393030364330303635303034393030364530303636303036463030303030303030303032343030303430303030303035343030373230303631303036453030373330303643303036313030373430303639303036463030364530303030303030303030303930344230303442303732304130303343363137333733363536443632364337393230373836443643364537333344323237353732364533413733363336383635364436313733324436443639363337323646373336463636373432443633364636443341363137333644324537363331323232303644363136453639363636353733373435363635373237333639364636453344323233313245333032323345304430413230323033433734373237353733373434393645363636463230373836443643364537333344323237353732364533413733363336383635364436313733324436443639363337323646373336463636373432443633364636443341363137333644324537363333323233453044304132303230323032303343373336353633373537323639373437393345304430413230323032303230323032303343373236353731373536353733373436353634353037323639373636393643363536373635373333453044304132303230323032303230323032303230334337323635373137353635373337343635363434353738363536333735373436393646364534433635373636353643323036433635373636353643334432323631373334393645373636463642363537323232323037353639343136333633363537333733334432323636363136433733363532323345334332463732363537313735363537333734363536343435373836353633373537343639364636453443363537363635364333453044304132303230323032303230323033433246373236353731373536353733373436353634353037323639373636393643363536373635373333453044304132303230323032303343324637333635363337353732363937343739334530443041323032303343324637343732373537'
	$s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
	$sB &= '303030'
	Return $sB
EndFunc

#EndRegion Internal Functions
