#include-once
#include <OutlookEX.au3>
#include <Excel.au3>

; #INDEX# =======================================================================================================================
; Title .........: Function library to import/export ical (RFC 5545) and vcf (RFC 6350) files
; AutoIt Version : 3.3.14.4
; UDF Version ...: 0.6.0.0
; Language ......: English
; Description ...: A collection of functions to import/export ical (RFC 5545) and vcf (RFC 6350) files
; Author(s) .....: water
; Modified.......: 20190722 (YYYYMMDD)
; Contributors ..: mikell
; Resources .....: https://tools.ietf.org/html/rfc5545, https://tools.ietf.org/html/rfc6350
;                  https://www.autoitscript.com/forum/topic/114406-csv-file-to-multidimensional-array/?do=findComment&comment=799820 (_WriteCSV)
; ===============================================================================================================================

; #CURRENT# =====================================================================================================================
;_OLT_iCal_VEvent_Import
;_OLT_vCard_Import
;_OLT_CSV_Import
;_OLT_Export
; ===============================================================================================================================

; #INTERNAL_USE_ONLY#============================================================================================================
;__OLT_LineSplit
;__OLT_Debug
;__OLT_iCal_DateTimeTranslate
;__OLT_iCal_DurationTranslate
;__OLT_iCal_VAlarmProcess
; ===============================================================================================================================

; #CONSTANTS# ===================================================================================================================
; Debugging
Global Const $iOLT_DebugOff = 0 ; No debugging (default)
Global Const $iOLT_DebugConsole = 1 ; Writes debugging messages to the console
Global Const $iOLT_DebugFile = 2 ; Appends debugging messages to a file specified by parameter $sDebugFile
Global Const $iOLT_DebugProperties = 4 ; Writes the Outlook properties used to create the event to the debugging destination
Global Const $iOLT_DebugCreateOff = 8 ; Do not create the event item. Allows to test the import without writing events to Outlook
; Processing
Global Const $iOLT_FlagProcessRecord = 1 ; Result of the callback function: Process the current record, then continue with the next record
Global Const $iOLT_FlagCancelRecord = 2 ; Result of the callback function: Do not process the current record, then continue with the next record
Global Const $iOLT_FlagCancelAll = 4 ; Result of the callback function: Do not process the current record, cancel processing all remaining records and exit the _OLT_* function
Global Const $iOLT_FlagReturnData = 8 ; The data passed to the callback function has been modified. Pass the data back to the calling _OLT_* function
; Date/Time
Global Const $iOLT_FlagDTAlLDay = 1 ; Date is an allday event
Global Const $iOLT_FlagDTUTC = 2 ; Date/Time is in UTC format
; Export
Global Const $iOLT_ExportTypeCSV = 1 ; Export Outlook items to a CSV file
Global Const $iOLT_ExportTypeExcel = 2 ; Export Outlook items to an Excel Workbook
Global Const $iOLT_ExportTypeVCF = $olVCard ; Export Outlook contact items to VCard format (.vcf)
Global Const $iOLT_ExportTypeICS = $olVCal ; Export Outlook appointment items to iCal format (.ics)
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name...........: _OLT_iCal_VEvent_Import
; Description ...: Import iCal events from an ICS file to an Outlook calendar.
; Syntax.........: _OLT_iCal_VEvent_Import($oOL, $vFolder, $siCalPath[, $sCallback = ""[, $iDebug = $iOLT_DebugOff[, $sDebugFile = @ScriptDir & "\OLT_Debug.txt"]]])
; Parameters ....: $oOL        - Outlook object returned by a preceding call to _OL_Open()
;                  $vFolder    - Calendar folder object as returned by _OL_FolderAccess or full name of folder where the event items will be created
;                  $siCalPath  - Path and file name of the ICS file to import
;                  $sCallback  - Optional: Name of a function to be called before a new event is created. See Remarks for details
;                  $iDebug     - Optional: Sets the flags for debugging. Possible values (can be a combination of one or more of the following values):
;                  |$iOLT_DebugOff (0)        - No debugging (default)
;                  |$iOLT_DebugConsole (1)    - Writes debugging messages to the console
;                  |$iOLT_DebugFile (2)       - Appends debugging messages to a file specified by parameter $sDebugFile
;                  |$iOLT_DebugProperties (4) - Writes the Outlook properties used to create the event to the debugging destination
;                  |$iOLT_DebugCreateOff (8)  - Do not create the event item. Allows to test the import without writing events to Outlook
;                  $sDebugFile - Optional: Path of the file to write debugging messages to (default = @ScriptDir & "\OLT_Debug.txt")
; Return values .: Success - two-dimensional zero based array with the following information:
;                  |0 - Object of the created Outlook event item. When $iDebug = 8 then this element is undefined
;                  |1 - Start date-time of the event item
;                  |2 - End date-time of the event item
;                  |3 - Subject of the event item
;                  Failure - Returns "" and sets @error:
;                  |1 - File $siCalPath does not exist
;                  |2 - The ICS file does not start with a BEGIN:VCALENDAR line
;                  |3 - Unable to create the Scripting.Dictionary object. @extended is set to the COM error code
;                  |4 - FileReadToArray returned an error. @extended is set to the FileReadToArray @error
;                  |5 - Error returned by built-in function Call. @extended is set to @error as returned by Call. For details please check the AutoIt help file
;                  |6 - Error when creating the Outlook item. @extended is set to @error as returned by _OL_ItemCreate
;                  |7 - The callback function did not set @extended so the function does not know how to process the record
; Author ........: water, mikell (RegExp for "DURATION")
; Modified ......:
; Remarks .......: This function processes the following properties according to RFC 5545 (properties not listed will be ignored):
;
;                  COMPONENT:PROPERTY	SECTION IN RFC 5545  OUTLOOK PROPERTY  			 COMMENT
;                  ------------------	-------------------  -------------------------   ---------------------------------------------
;                  VEVENT:CATEGORIES	Section 3.8.1.2.	 Categories
;                  VEVENT:CLASS			Section 3.8.1.3. 	 Sensitivity				 olNormal, olPrivate or olConfidentional. olPersonal does not get set
;                  VEVENT:DESCRIPTION	Section 3.8.1.5.	 Body
;                  VEVENT:LOCATION		Section 3.8.1.7.	 Location
;                  VEVENT:PRIORITY		Section 3.8.1.9.	 Importance
;                  VEVENT:SUMMARY		Section 3.8.1.12.	 Subject
;                  VEVENT:DTEND			Section 3.8.2.2.	 End						 specifies the non-inclusive end of the event
;                  VEVENT:DTSTART		Section 3.8.2.4.	 Start
;                  VEVENT:DURATION		Section 3.8.2.5.	 End 	 					 used to calculate the end property
;                  VEVENT:TRANSP		Section 3.8.2.7.	 BusyStatus
;                  VALARM:TRIGGER		Section 3.8.6.3.	 ReminderMinutesBeforeStart	 Only the minutes section of the duration is processed
;
;                  Callback function:
;                  This function gets called before a new event is created in Outlook.
;                  An array holding the properties to create the Outlook event item is passed to the function (read only).
;                  The information is stored as Outlook-property-name=value (e.g. AllDayEvent=True).
;                  You can then decide how to continue processing by setting the return value:
;                    0 - Create the Outlook event item
;                    1 - Do not create the Outlook event item - continue processing the next event
;                    2 - Cancel processing the iCal file, exit _OLT_iCal_VEvent_Import and return to the calling function
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _OLT_iCal_VEvent_Import($oOL, $vFolder, $siCalPath, $sCallBack = "", $iDebug = $iOLT_DebugOff, $sDebugFile = @ScriptDir & "\OLT_Debug.txt")

	If Not FileExists($siCalPath) Then Return SetError(1, 0, "")
	Local $aClass[][] = [["Public", $olNormal], ["Private", $olPrivate], ["Confidential", $olConfidential]]
	Local $aICS = FileReadToArray($siCalPath)
	If @error Then Return SetError(4, @error, "")
	Local $iIndex = 0, $iMaxIndex = UBound($aICS) - 1, $aEventsCreated[1000][4], $iEvent = 0, $oItemCreated, $bFlag, $sComponent, $sTemp, $sTempDate, $aOutlook, $vCBResult, $hDebugFile, $iCBFlag = 0
	Local $iError, $aLine, $iTokenValues, $iTokenParams, $aParams, $aValues, $sParamsRaw, $sValuesRaw
	Local $oDictEvent = ObjCreate("Scripting.Dictionary")
	If @error Then Return SetError(3, @error, "")
	Local $oDictParam = ObjCreate("Scripting.Dictionary")
	If @error Then Return SetError(3, @error, "")
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then $hDebugFile = FileOpen($sDebugFile, $FO_APPEND)
	If $aICS[$iIndex] <> "BEGIN:VCALENDAR" Then Return SetError(2, 0, "")
	If $iDebug Then __OLT_Debug("", $aICS[$iIndex], $iDebug, $hDebugFile)
	While 1
		If $aICS[$iIndex] = "BEGIN:VEVENT" Then
			If $iDebug Then __OLT_Debug("", $aICS[$iIndex], $iDebug, $hDebugFile)
			$oDictEvent.RemoveAll ; Reset all event related variables for the next event.
			$oDictParam.RemoveAll
			While 1
				$iIndex += 1
				If $iIndex > $iMaxIndex Then ExitLoop
				__OLT_LineSplit($aICS[$iIndex], $aParams, $aValues, $sParamsRaw, $sValuesRaw)
				If @error Then $aParams[0] = "Error"
				$iTokenValues = UBound($aValues)
				Switch $aParams[0]
					Case "Error" ; Ignore lines that could not be split into params and values
						If $iDebug Then __OLT_Debug("Ignored with Errors: ", "  " & $aICS[$iIndex], $iDebug, $hDebugFile)
					Case "DTSTART"
						$sTemp = __OLT_iCal_DateTimeTranslate($aParams, $aValues)
						$bFlag = @extended
						$sTempDate = (BitAND($bFlag, 2) = 2) ? "StartUTC=" : "Start=" ; Start date/time expressed as UTC or local date/time
						$oDictParam.Item("UTC") = (BitAND($bFlag, 2) = 2) ? True : False ; UTC
						$oDictEvent.Item("DTSTART") = $sTempDate & $sTemp
						$oDictEvent.Item("ALLDAY") = (BitAND($bFlag, 1) = 1) ? "AllDayEvent=True" : "AllDayEvent=False" ; AllDayEvent
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("DTSTART"), $iDebug, $hDebugFile)
					Case "DTEND"
						$sTemp = __OLT_iCal_DateTimeTranslate($aParams, $aValues)
						$bFlag = @extended
						$sTempDate = (BitAND($bFlag, 2) = 2) ? "EndUTC=" : "End=" ; End date/time expressed as UTC or local date/time
						$oDictParam.Item("UTC") = (BitAND($bFlag, 2) = 2) ? True : False ; UTC
						$oDictEvent.Item("DTEND") = $sTempDate & $sTemp
						$oDictEvent.Item("ALLDAY") = (BitAND($bFlag, 1) = 1) ? "AllDayEvent=True" : "AllDayEvent=False" ; AllDayEvent
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("DTEND"), $iDebug, $hDebugFile)
					Case "DURATION"
						$sTemp = __OLT_iCal_DurationTranslate(StringSplit($oDictEvent.Item("DTSTART"), "=")[2], $aValues[0])
						$sTempDate = ($oDictParam.Item("UTC") = True) ? "EndUTC=" : "End=" ; End date/time expressed as UTC or local date/time
						$oDictEvent.Item("DTEND") = $sTempDate & $sTemp
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("DTEND"), $iDebug, $hDebugFile)
					Case "SUMMARY"
						$oDictEvent.Item("SUMMARY") = "Subject=" & $aValues[0]
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("SUMMARY"), $iDebug, $hDebugFile)
					Case "DESCRIPTION"
						$oDictEvent.Item("DESCRIPTION") = "Body=" & $aValues[0]
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("DESCRIPTION"), $iDebug, $hDebugFile)
					Case "TRANSP"
						$sTemp = ($aValues[0] = "OPAQUE") ? ($olBusy) : ($olFree)
						$oDictEvent.Item("TRANSP") = "BusyStatus=" & $sTemp
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("TRANSP"), $iDebug, $hDebugFile)
					Case "LOCATION"
						$oDictEvent.Item("LOCATION") = "Location=" & $aValues[0]
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("LOCATION"), $iDebug, $hDebugFile)
					Case "CLASS"
						For $i = 0 To UBound($aClass) - 1
							If $aClass[$i][0] = $aValues[0] Then
								$oDictEvent.Item("CLASS") = "Sensitivity=" & $aClass[$i][1]
								ExitLoop
							EndIf
						Next
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("CLASS"), $iDebug, $hDebugFile)
					Case "CATEGORIES"
						$oDictEvent.Item("CATEGORIES") = "Categories=" & $aValues[0]
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("CATEGORIES"), $iDebug, $hDebugFile)
					Case "PRIORITY"
						If $aValues[0] = "High" Or (Number($aValues[0]) <= 4 And Number($aValues[0]) >= 1) Then
							$sTemp = $olImportanceHigh
						ElseIf $aValues[0] = "Low" Or (Number($aValues[0]) <= 9 And Number($aValues[0]) >= 6) Then
							$sTemp = $olImportanceLow
						Else
							$sTemp = $olImportanceNormal
						EndIf
						$oDictEvent.Item("PRIORITY") = "Importance=" & $sTemp
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $oDictEvent.Item("PRIORITY"), $iDebug, $hDebugFile)
						; Process nested components. Process all properties of the component or ignore all properties up to the END: statement
					Case "BEGIN"
						Switch $aValues[0]
							Case "VALARM"
								$oDictEvent.Item("TRIGGERREMINDER") = "ReminderSet=True"
								$sTemp = __OLT_iCal_VAlarmProcess($aICS, $iIndex, $iMaxIndex, $iDebug, $hDebugFile)
								$oDictEvent.Item("TRIGGERMINUTES") = "ReminderMinutesBeforeStart=" & $sTemp
							Case Else
								Do
									$iIndex += 1
									If $iDebug Then __OLT_Debug("Ignored: ", "  " & $aICS[$iIndex], $iDebug, $hDebugFile)
								Until StringLeft($aICS[$iIndex], 4 + StringLen($sComponent)) = "END:" & $sComponent
						EndSwitch
					Case $aICS[$iIndex] = "END:VEVENT"
						If $iDebug Then __OLT_Debug("", $aICS[$iIndex], $iDebug, $hDebugFile)
						ExitLoop
					Case Else
						If $iDebug Then __OLT_Debug("Ignored: ", "  " & $aICS[$iIndex], $iDebug, $hDebugFile)
				EndSwitch
			WEnd
			$aOutlook = $oDictEvent.Items()
			If BitAND($iDebug, $iOLT_DebugProperties) = $iOLT_DebugProperties Then __OLT_Debug("Outlook properties used: -----------------------------------", _ArrayToString($aOutlook, @CRLF) & @CRLF & "------------------------------------------------------------", $iDebug, $hDebugFile)
			$iCBFlag = $iOLT_FlagProcessRecord
			If $sCallBack <> "" Then
				If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' started.", $iDebug, $hDebugFile)
				$vCBResult = Call($sCallBack, $aOutlook)
				$iError = @error
				$iCBFlag = @extended
				If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' ended. Flag = " & $iCBFlag & ", @error = " & $iError & " (0x" & Hex($iError) & ").", $iDebug, $hDebugFile)
				If $iError Then Return SetError(5, $iError, "")
				If $iCBFlag <= 0 Then Return SetError(7, 0, "")
				If BitAND($iCBFlag, $iOLT_FlagReturnData) = $iOLT_FlagReturnData Then $aOutlook = $vCBResult ; Callback returned modified data
				If BitAND($iCBFlag, $iOLT_FlagCancelAll) = $iOLT_FlagCancelAll Then ExitLoop
			EndIf
			If BitAND($iDebug, $iOLT_DebugCreateOff) <> $iOLT_DebugCreateOff And BitAND($iCBFlag, $iOLT_FlagProcessRecord) = $iOLT_FlagProcessRecord Then ; Create the event item
				If $iDebug Then __OLT_Debug("", "Creating event", $iDebug, $hDebugFile)
				$oItemCreated = _OL_ItemCreate($oOL, $olAppointmentItem, $vFolder, "", $aOutlook)
				If @error Then Return SetError(6, @error, @extended)
				; add the created item to the result array
				$aEventsCreated[$iEvent][0] = $oItemCreated
				$aEventsCreated[$iEvent][1] = StringSplit($oDictEvent.Item("DTSTART"), "=")[2]
				$aEventsCreated[$iEvent][2] = StringSplit($oDictEvent.Item("DTEND"), "=")[2]
				$aEventsCreated[$iEvent][3] = StringSplit($oDictEvent.Item("SUMMARY"), "=")[2]
				$iEvent += 1
				If Mod($iEvent, 999) = 0 Then ReDim $aEventsCreated[$iEvent + 1000][4]
			EndIf
		EndIf
		$iIndex += 1
		If $iIndex > $iMaxIndex Then ExitLoop
		If $iDebug Then
			If StringLeft($aICS[$iIndex], 4) = "END:" Or StringLeft($aICS[$iIndex], 6) = "BEGIN:" Then
				__OLT_Debug("", $aICS[$iIndex], $iDebug, $hDebugFile)
			Else
				__OLT_Debug("Ignored: ", "  " & $aICS[$iIndex], $iDebug, $hDebugFile)
			EndIf
		EndIf
	WEnd
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then FileClose($hDebugFile)
	ReDim $aEventsCreated[$iEvent][4]
	Return $aEventsCreated

EndFunc   ;==>_OLT_iCal_VEvent_Import

; #FUNCTION# ====================================================================================================================
; Name...........: _OLT_vCard_Import
; Description ...: Import vCard contacts to an Outlook contacts folder.
; Syntax.........: _OLT_vCard_Import($oOL, $vFolder, $svCardPath[, $sCallBack = ""[, $iDebug = $iOLT_DebugOff[, $sDebugFile = @ScriptDir & "\vCard_Debug.txt"]]])
; Parameters ....: $oOL        - Outlook object returned by a preceding call to _OL_Open()
;                  $vFolder    - Contacts folder object as returned by _OL_FolderAccess or full name of folder where the contact items will be created
;                  $svCardPath - Path and file name of the VCF (Virtual Contact File) file to import
;                  $sCallback  - Optional: Name of a function to be called before a new contact is created. See Remarks for details.
;                  $iDebug     - Optional: Sets the flags for debugging. Possible values (can be a combination of one or more of the following values):
;                  |$iOLT_DebugOff (0)        - No debugging (default)
;                  |$iOLT_DebugConsole (1)    - Writes debugging messages to the console
;                  |$iOLT_DebugFile (2)       - Appends debugging messages to a file specified by parameter $sDebugFile
;                  |$iOLT_DebugProperties (4) - Writes the Outlook properties used to create the contact item to the debugging destination
;                  |$iOLT_DebugCreateOff (8)  - Do not create the contact item. Allows to test the import without writing contacts to Outlook
;                  $sDebugFile - Optional: Path of the file to write debugging messages to (default = @ScriptDir & "\vCard_Debug.txt")
; Return values .: Success - two-dimensional zero based array with the following information:
;                  |0 - Object of the created Outlook contact item
;                  |1 - FullName property of the created Outlook contact item
;                  |2 - CompanyName property of the created Outlook contact item
;                  Failure - Returns "" and sets @error:
;                  |1 - File $svCardPath does not exist
;                  |2 - The VCF file does not start with a BEGIN:VCARD line
;                  |3 - Unable to create the Scripting.Dictionary object. @extended is set to the COM error code
;                  |4 - FileReadToArray returned an error. @extended is set to the FileReadToArray @error.
;                  |5 - Error returned by built-in function Call. @extended is set to @error as returned by Call. For details please check the AutoIt help file.
;                  |6 - Error when creating the Outlook item. @extended is set to @error as returned by _OL_ItemCreate.
;                  |7 - The callback function did not set @extended so the function does not know how to process the record
; Author ........: water
; Modified ......:
; Remarks .......: vCard 2.1 1996-09-18 http://www.imc.org/pdi/vcard-21.txt
;                  vCard 3.0 1998-09 RFC 2426
;                  vCard 4.0 2011-08 RFC 6350
;
;                  This function processes the following properties according to RFC 6350 (properties not listed will be ignored):
;
;                  COMPONENT:PROPERTY	SECTION IN RFC 6350  OUTLOOK PROPERTY  				COMMENT
;                  ------------------	-------------------  -------------------------		---------------------------------------------
;                  VCARD:FN				Section 6.2.1		 fullname
;                  VCARD:N				Section 6.2.2		 lastname, firstname,			Family Names, Given Names, Additional Names, Honorific Prefixes, and Honorific Suffixes.
;                  											 middlename, title
;                  VCARD:ADR			Section 6.3.1
;                    TYPE=WORK                               BusinessAddressPostOfficeBox,
;                                                            BusinessAddress,
;                                                            BusinessAddressStreet,
;                                                            BusinessAddressCity,
;                                                            BusinessAddressState,
;                                                            BusinessAddressPostalCode,
;                                                            BusinessAddressCountry
;                    TYPE=HOME                               HomeAddressPostOfficeBox,
;                                                            HomeAddress,
;                                                            HomeAddressStreet,
;                                                            HomeAddressCity,
;                                                            HomeAddressState,
;                                                            HomeAddressPostalCode,
;                                                            HomeAddressCountry
;                  VCARD:TEL            Section 6.4.1
;                    TYPE=HOME								 PersonalHomePage
;                    TYPE=WORK								 BusinessHomePage
;                  VCARD:EMAIL          Section 6.4.2		 Email1Address
;                  VCARD:ORG			Section 6.6.4		 CompanyName
;                  VCARD:TITLE			Section 6.6.1		 JobTitle
;                  VCARD:CATEGORIES		Section 6.7.1		 Categories
;                  VCARD:URL			Section 6.7.8
;                    TYPE=HOME								 PersonalHomePage
;                    TYPE=WORK								 BusinessHomePage
;
;                  Callback function:
;                  This function gets called before a new contact is created in Outlook.
;                  An array holding the properties to create the Outlook contact item is passed to the function (read only).
;                  The information is stored as Outlook-property-name=value (e.g. FullName=John Doe).
;                  You can then decide how to continue processing by setting the return value:
;                    0 - Create the Outlook contact item
;                    1 - Do not create the Outlook contact item - continue processing the next vCard
;                    2 - Cancel processing the vCard file, exit _OLT_vCard_Import and return to the calling function
; Related .......:
; Link ..........: https://tools.ietf.org/html/rfc6350
; Example .......:
; ===============================================================================================================================
Func _OLT_vCard_Import($oOL, $vFolder, $svCardPath, $sCallBack = "", $iDebug = $iOLT_DebugOff, $sDebugFile = @ScriptDir & "\vCard_Debug.txt")

	If Not FileExists($svCardPath) Then Return SetError(1, 0, "")
	Local $aVCF = FileReadToArray($svCardPath)
	If @error Then Return SetError(4, @error, "")
	Local $iIndex = 0, $iMaxIndex = UBound($aVCF) - 1, $aContactsCreated[1000][3], $iContact = 0, $oItemCreated, $bFlag, $sTemp, $sTempDate, $aOutlook, $vCBResult, $hDebugFile, $iCBFlag = 0
	Local $iError, $aLine, $iTokenValues, $iTokenParams, $aParams, $aValues, $sParamsRaw, $sValuesRaw
	Local $oDictContact = ObjCreate("Scripting.Dictionary")
	If @error Then Return SetError(3, @error, "")
	Local $oDictParam = ObjCreate("Scripting.Dictionary")
	If @error Then Return SetError(3, @error, "")
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then $hDebugFile = FileOpen($sDebugFile, $FO_APPEND)
	While 1
		If $aVCF[$iIndex] = "BEGIN:VCARD" Then
			If $iDebug Then __OLT_Debug("", $aVCF[0], $iDebug, $hDebugFile)
			$oDictContact.RemoveAll ; Reset all contact related variables for the next contact.
			$oDictParam.RemoveAll
			While 1
				$iIndex += 1
				If $iIndex > $iMaxIndex Then ExitLoop
				__OLT_LineSplit($aVCF[$iIndex], $aParams, $aValues, $sParamsRaw, $sValuesRaw)
				If @error Then $aParams[0] = "Error"
				$iTokenValues = UBound($aValues)
				Switch $aParams[0]
					Case "Error" ; Ignore lines that could not be split into params and values
						If $iDebug Then __OLT_Debug("Ignored with Errors: ", "  " & $aVCF[$iIndex], $iDebug, $hDebugFile)
					Case "N"
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: ", $iDebug, $hDebugFile)
						If $iTokenValues > 0 Then
							$oDictContact.Add("LastName", "LastName=" & $aValues[0])
							If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("LastName"), $iDebug, $hDebugFile)
						EndIf
						If $iTokenValues > 1 Then
							$oDictContact.Add("FirstName", "FirstName=" & $aValues[1])
							If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("FirstName"), $iDebug, $hDebugFile)
						EndIf
						If $iTokenValues > 2 Then
							$oDictContact.Add("MiddleName", "MiddleName=" & $aValues[2])
							If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("MiddleName"), $iDebug, $hDebugFile)
						EndIf
						If $iTokenValues > 3 Then
							$oDictContact.Add("Title", "Title=" & $aValues[3])
							If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("Title"), $iDebug, $hDebugFile)
						EndIf
					Case "FN"
						$oDictContact.Add("FullName", "FullName=" & $aValues[0])
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("FullName"), $iDebug, $hDebugFile)
					Case "EMAIL"
						$oDictContact.Add("Email1Address", "Email1Address=" & $aValues[0])
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("Email1Address"), $iDebug, $hDebugFile)
					Case "ADR"
						Select
							Case $aParams[1] = "TYPE=WORK" Or $aParams[1] = "WORK"
								;	If $csv_BusiPOBox <> "" Or $csv_BusiAddress  <> "" Or $csv_BusiStreet <> "" Or $csv_BusiCity <> "" Or $csv_BusiState <> "" Or $csv_BusiZip <> "" Or $csv_BusiCountry <> "" Then
								;		MsgBox(16, "Error", "Work address already recorded for this contact, sorry I only support 1. script will continue but address will not be moved")
								;	Else
								If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: ", $iDebug, $hDebugFile)
								If $iTokenValues > 0 Then
									$oDictContact.Add("BusinessAddressPostOfficeBox", "BusinessAddressPostOfficeBox=" & $aValues[0])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("BusinessAddressPostOfficeBox"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 1 Then
									$oDictContact.Add("BusinessAddress", "BusinessAddress=" & $aValues[1])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("BusinessAddress"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 2 Then
									$oDictContact.Add("BusinessAddressStreet", "BusinessAddressStreet=" & $aValues[2])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("BusinessAddressStreet"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 3 Then
									$oDictContact.Add("BusinessAddressCity", "BusinessAddressCity=" & $aValues[3])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("BusinessAddressCity"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 4 Then
									$oDictContact.Add("BusinessAddressState", "BusinessAddressState=" & $aValues[4])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("BusinessAddressState"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 5 Then
									$oDictContact.Add("BusinessAddressPostalCode", "BusinessAddressPostalCode=" & $aValues[5])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("BusinessAddressPostalCode"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 6 Then
									$oDictContact.Add("BusinessAddressCountry", "BusinessAddressCountry=" & $aValues[6])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("BusinessAddressCountry"), $iDebug, $hDebugFile)
								EndIf
								;	EndIf
							Case $aParams[1] = "TYPE=HOME" Or $aParams[1] = "HOME"
								;	If $csv_hPOBox <> "" Or $csv_hAddress <> "" Or $csv_hStreet <> "" Or $csv_hCity <> "" Or $csv_hState <> "" Or $csv_hZip <> "" Or $csv_hCountry <> "" Then
								;		MsgBox(16, "Error", "Home address already recorded for this contact, sorry I only support 1. script will continue but address will not be moved")
								;	Else
								If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: ", $iDebug, $hDebugFile)
								If $iTokenValues > 0 Then
									$oDictContact.Add("HomeAddressPostOfficeBox", "HomeAddressPostOfficeBox=" & $aValues[0])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("HomeAddressPostOfficeBox"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 1 Then
									$oDictContact.Add("HomeAddress", "HomeAddress=" & $aValues[1])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("HomeAddress"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 2 Then
									$oDictContact.Add("HomeAddressStreet", "HomeAddressStreet=" & $aValues[2])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("HomeAddressStreet"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 3 Then
									$oDictContact.Add("HomeAddressCity", "HomeAddressCity=" & $aValues[3])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("HomeAddressCity"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 4 Then
									$oDictContact.Add("HomeAddressState", "HomeAddressState=" & $aValues[4])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("HomeAddressState"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 5 Then
									$oDictContact.Add("HomeAddressPostalCode", "HomeAddressPostalCode=" & $aValues[5])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("HomeAddressPostalCode"), $iDebug, $hDebugFile)
								EndIf
								If $iTokenValues > 6 Then
									$oDictContact.Add("HomeAddressCountry", "HomeAddressCountry=" & $aValues[6])
									If $iDebug Then __OLT_Debug("", "    " & $oDictContact.Item("HomeAddressCountry"), $iDebug, $hDebugFile)
								EndIf
								;	EndIf
						EndSelect
					Case "ORG"
						$oDictContact.Add("CompanyName", "CompanyName=" & $sValuesRaw)
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("CompanyName"), $iDebug, $hDebugFile)
					Case "TITLE"
						$oDictContact.Add("JobTitle", "JobTitle=" & $sValuesRaw)
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("JobTitle"), $iDebug, $hDebugFile)
					Case "CATEGORIES"
						$oDictContact.Add("Categories", "Categories=" & $sValuesRaw)
						If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("Categories"), $iDebug, $hDebugFile)
					Case "TEL"
						Switch $aParams[1]
							Case "TYPE=CELL", "CELL"
								$oDictContact.Add("MobileTelephoneNumber", "MobileTelephoneNumber=" & $sValuesRaw)
								If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("MobileTelephoneNumber"), $iDebug, $hDebugFile)
							Case "TYPE=WORK", "WORK"
								If $aParams[2] = "FAX" Then
									$oDictContact.Add("BusinessFaxNumber", "BusinessFaxNumber=" & $sValuesRaw)
									If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("BusinessFaxNumber"), $iDebug, $hDebugFile)
								Else
									$oDictContact.Add("BusinessTelephoneNumber", "BusinessTelephoneNumber=" & $sValuesRaw)
									If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("BusinessTelephoneNumber"), $iDebug, $hDebugFile)
								EndIf
							Case "TYPE=HOME", "HOME"
								If $aParams[2] = "FAX" Then
									$oDictContact.Add("HomeFaxNumber", "HomeFaxNumber=" & $sValuesRaw)
									If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("HomeFaxNumber"), $iDebug, $hDebugFile)
								Else
									$oDictContact.Add("HomeTelephoneNumber", "HomeTelephoneNumber=" & $sValuesRaw)
									If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("HomeTelephoneNumber"), $iDebug, $hDebugFile)
								EndIf
						EndSwitch
					Case "URL"
						Switch $aParams[1]
							Case "TYPE=WORK", "WORK"
								$oDictContact.Add("BusinessHomePage", "BusinessHomePage=" & $sValuesRaw)
								If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("BusinessHomePage"), $iDebug, $hDebugFile)
							Case "TYPE=HOME", "HOME"
								$oDictContact.Add("PersonalHomePage", "PersonalHomePage=" & $sValuesRaw)
								If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aVCF[$iIndex] & @CRLF & "  OUT: " & $oDictContact.item("PersonalHomePage"), $iDebug, $hDebugFile)
						EndSwitch
					Case "END"
						If $iDebug Then __OLT_Debug("", $aVCF[$iIndex], $iDebug, $hDebugFile)
						ExitLoop
					Case Else
						If $iDebug Then __OLT_Debug("Ignored: ", $aVCF[$iIndex], $iDebug, $hDebugFile)
				EndSwitch
			WEnd
			$aOutlook = $oDictContact.Items()
			If BitAND($iDebug, $iOLT_DebugProperties) = $iOLT_DebugProperties Then __OLT_Debug("Outlook properties used: -----------------------------------", _ArrayToString($aOutlook, @CRLF) & @CRLF & "------------------------------------------------------------", $iDebug, $hDebugFile)
			$iCBFlag = $iOLT_FlagProcessRecord
			If $sCallBack <> "" Then
				If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' started.", $iDebug, $hDebugFile)
				$vCBResult = Call($sCallBack, $aOutlook)
				$iError = @error
				$iCBFlag = @extended
				If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' ended. Flag = " & $iCBFlag & ", @error = " & $iError & " (0x" & Hex($iError) & ").", $iDebug, $hDebugFile)
				If $iError Then Return SetError(5, $iError, "")
				If $iCBFlag <= 0 Then Return SetError(7, 0, "")
				If BitAND($iCBFlag, $iOLT_FlagReturnData) = $iOLT_FlagReturnData Then $aOutlook = $vCBResult ; Callback returned modified data
				If BitAND($iCBFlag, $iOLT_FlagCancelAll) = $iOLT_FlagCancelAll Then ExitLoop
			EndIf
			If BitAND($iDebug, $iOLT_DebugCreateOff) <> $iOLT_DebugCreateOff And BitAND($iCBFlag, $iOLT_FlagProcessRecord) = $iOLT_FlagProcessRecord Then ; Create the contact item
				If $iDebug Then __OLT_Debug("", "Creating contact", $iDebug, $hDebugFile)
				$oItemCreated = _OL_ItemCreate($oOL, $olContactItem, $vFolder, "", $aOutlook)
				If @error Then Return SetError(6, @error, @extended)
				; Add the created item to the result array
				$aContactsCreated[$iContact][0] = $oItemCreated
				$aContactsCreated[$iContact][1] = StringSplit($oDictContact.Item("FullName"), "=")[2]
				$aContactsCreated[$iContact][2] = StringSplit($oDictContact.Item("CompanyName"), "=")[2]
				$iContact += 1
				If Mod($iContact, 999) = 0 Then ReDim $aContactsCreated[$iContact + 1000][3]
			EndIf
		EndIf
		$iIndex += 1
		If $iIndex > $iMaxIndex Then ExitLoop
		If $iDebug Then
			If StringLeft($aVCF[$iIndex], 4) = "END:" Or StringLeft($aVCF[$iIndex], 6) = "BEGIN:" Then
				__OLT_Debug("", $aVCF[$iIndex], $iDebug, $hDebugFile)
			Else
				__OLT_Debug("Ignored: ", "  " & $aVCF[$iIndex], $iDebug, $hDebugFile)
			EndIf
		EndIf
	WEnd
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then FileClose($hDebugFile)
	ReDim $aContactsCreated[$iContact][4]
	Return $aContactsCreated

EndFunc   ;==>_OLT_vCard_Import

; #INTERNAL_USE_ONLY#============================================================================================================
; Name...........: _OLT_CSV_Import
; Description ...: Imports data from a CSV file and creates Outlook items in a specified folder.
; Syntax.........: _OLT_CSV_Import($oOL, $oFolder, $iOlItemType, $aCSVData, $aProperties, $sCallBack = "", $iDebug = $iOLT_DebugOff, $sDebugFile = @ScriptDir & "\_OLT_CSV_Import.txt")
; Parameters ....: $oFolder     - Folder object as returned by _OL_FolderAccess where the items will be stored
;                  $iOlItemType - Indicates the Outlook item type contained in the CSV (can be $olContactItem of the OlItemType enumeration).
;                  $aCSVData    - 2D zero based array containing the data to be imported. Each row holds an item, each column a property for an item
;                  $aProperties - 1D zero based array holding the names of the properties to be created with data from $aCSVData.
;                    Means: When $aProperties[0] contains "FirstName" (PropertyName) then the value in $aCSVData[*][0] is interpreted as.
;                  $sCallback   - Optional: Name of a function to be called before a new item is being created. See Remarks for details.
;                  $iDebug      - Optional: Sets the flags for debugging. Possible values (can be a combination of one or more of the following values):
;                  |$iOLT_DebugOff (0)        - No debugging (default)
;                  |$iOLT_DebugConsole (1)    - Writes debugging messages to the console
;                  |$iOLT_DebugFile (2)       - Appends debugging messages to a file specified by parameter $sDebugFile
;                  |$iOLT_DebugProperties (4) - Writes the Outlook properties used to create the contact item to the debugging destination
;                  |$iOLT_DebugCreateOff (8)  - Do not create the contact item. Allows to test the import without writing contacts to Outlook
;                  $sDebugFile  - Optional: Path of the file to write debugging messages to (default = @ScriptDir & "\vCard_Debug.txt")
; Return values .: Success - Returns the number of created items.
;                  Failure - Returns 0 and sets @error:
;                  |1    - Invalid properties passed in parameter $aCSVStruct. @extended is set to the index (0-based) of first invalid property
;                  |2    - Error returned by built-in function Call. @extended is set to @error as returned by Call. For details please check the AutoIt help file.
;                  |3    - Error creating item. @extended is set to the COM error code
;                  |4    - Error when creating the Outlook item. @extended is set to @error as returned by _OL_ItemCreate.
;                  |5    - Error returned by built-in function Call. @extended is set to @error as returned by Call. For details please check the AutoIt help file
;                  |6    - The callback function did not set @extended so the function does not know how to process the record
;                  |1nmm - Error returned by __OL_CheckProperties when checking the properties specified in parameter $aProperties
;                  +       n is either 0 (property does not exist) or 1 (Property has invalid case)
;                  +       mm is the index of the property in error (zero based)
; Author ........: water
; Modified ......:
; Remarks .......: Callback function:
;                  This function gets called before a new item is created in Outlook.
;                  A single parameter is passed to the callback function: The index of the record from $aCSVData currently being processed.
;                  You can decide how to continue processing by setting the return value:
;                    0 - Create the Outlook item
;                    1 - Do not create the Outlook item - continue processing the next record
;                    2 - Cancel processing the input file, exit _OLT_CSV_Import and return to the calling function
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _OLT_CSV_Import($oOL, $oFolder, $iOlItemType, ByRef $aCSVData, $aProperties, $sCallBack = "", $iDebug = $iOLT_DebugOff, $sDebugFile = @ScriptDir & "\_OLT_CSV_Import.txt")

	Local $oItem, $iItemCount = 0, $hDebugFile, $iCBFlag = 0, $aItemData[UBound($aProperties)], $iError ; , $vCBResult
	$oItem = $oFolder.Items.Add($iOlItemType)
	If @error Then Return SetError(3, @error, 0)
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then $hDebugFile = FileOpen($sDebugFile, $FO_APPEND)
	; Check properties
	If $iDebug Then __OLT_Debug("", "Checking properties", $iDebug, $hDebugFile)
	If Not __OL_CheckProperties($oItem, $aProperties) Then Return SetError(@error, @extended, 0)
	If BitAND($iDebug, $iOLT_DebugProperties) = $iOLT_DebugProperties Then __OLT_Debug("Outlook properties used: -----------------------------------", _ArrayToString($aProperties, @CRLF) & @CRLF & "------------------------------------------------------------", $iDebug, $hDebugFile)
	For $i = 0 To UBound($aCSVData) - 1
		$iCBFlag = $iOLT_FlagProcessRecord
		If $sCallBack <> "" Then
			If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' started. Processing record #" & $i & " (zero based)", $iDebug, $hDebugFile)
			; $vCBResult = Call($sCallBack, $i) <== Not return needed as the data can be modified in the Global data array
			Call($sCallBack, $i)
			$iError = @error
			$iCBFlag = @extended
			If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' ended. Flag = " & $iCBFlag & ", @error = " & $iError & " (0x" & Hex($iError) & ").", $iDebug, $hDebugFile)
			If $iError Then Return SetError(5, $iError, 0)
			If $iCBFlag <= 0 Then Return SetError(6, 0, "")
			; If BitAND($iCBFlag, $iOLT_FlagReturnData) = $iOLT_FlagReturnData Then $aOutlook = $vCBResult ; Callback returned modified data <== Not needed as the data can be modified in the Global data array
			If BitAND($iCBFlag, $iOLT_FlagCancelAll) = $iOLT_FlagCancelAll Then ExitLoop
		EndIf
		If BitAND($iDebug, $iOLT_DebugCreateOff) <> $iOLT_DebugCreateOff And BitAND($iCBFlag, $iOLT_FlagProcessRecord) = $iOLT_FlagProcessRecord Then ; Create the item
			For $j = 0 To UBound($aProperties) - 1
				$aItemData[$j] = $aProperties[$j] & "=" & $aCSVData[$i][$j]
			Next
			If $iDebug Then __OLT_Debug("", "Creating item", $iDebug, $hDebugFile)
			$oItem = _OL_ItemCreate($oOL, $iOlItemType, $oFolder, "", $aItemData)
			If @error Then Return SetError(4, @error, @extended)
			$iItemCount += 1
		EndIf
	Next
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then FileClose($sDebugFile)
	Return $iItemCount

EndFunc   ;==>_OLT_CSV_Import

#cs
; #FUNCTION# ====================================================================================================================
; Name...........: _OLT_Contact_Export_VCF
; Description ...: Export Outlook contacts as VCF files.
; Syntax.........: _OLT_Contact_Export_VCF($oOL, $aContacts, $sExportDirectory[, $sCallback = ""[, $iDebug = $iOLT_DebugOff[, $sDebugFile = @ScriptDir & "\OLT_Debug.txt"]]])
; Parameters ....: $oOL                       - Outlook object returned by a preceding call to _OL_Open()
;                  $aContacts                 - 2D array as returned by _OL_Itemfind holding the EntryIDs of the contacts to be exported in column 0
;                  $sExportDirectory          - Directory where the contacts should be exported to. Will be created if it doesn't exist
;                  $sCallback                 - Optional: Name of a function to be called before a new event is created. See Remarks for details
;                  $iDebug                    - Optional: Sets the flags for debugging. Possible values (can be a combination of one or more of the following values):
;                  |$iOLT_DebugOff (0)        - No debugging (default)
;                  |$iOLT_DebugConsole (1)    - Writes debugging messages to the console
;                  |$iOLT_DebugFile (2)       - Appends debugging messages to a file specified by parameter $sDebugFile
;                  |$iOLT_DebugProperties (4) - Writes the Outlook properties used to create the event to the debugging destination
;                  |$iOLT_DebugCreateOff (8)  - Do not create the event item. Allows to test the import without writing events to Outlook
;                  $sDebugFile - Optional: Path of the file to write debugging messages to (default = @ScriptDir & "\OLT_Debug.txt")
; Return values .: Success - number of exported contacts
;                  Failure - Returns 0 and sets @error:
;                  |1- Could not create output directory as specified by $sExportDirectory
;                  |2 - _OL_ItemGet returned an error. See @extended for the error code
;                  |3 - _OL_ItemAttachmentSave returned an error. See @extended for the error code
;                  |4 - Error returned by built-in function Call. @extended is set to @error as returned by Call. For details please check the AutoIt help file
;                  |5 - The callback function did not set @extended so the function does not know how to process the record
; Author ........: water
; Modified ......:
; Remarks .......: VCF: Files are named LastName_FirstName_Companies. If a file already exists "_n" is appended. Where n > 0 until the filename is unique.
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _OLT_Contact_Export_VCF($oOL, ByRef $aContacts, $sExportDirectory, $sCallBack = "", $iDebug = $iOLT_DebugOff, $sDebugFile = @ScriptDir & "\OLT_Debug.txt")

	Local $hDebugFile, $iItemCount = 0, $sExportPath, $iIndex = 0, $sTemp, $iCBFlag = 0, $iError ; , $vCBResult
	If Not FileExists($sExportDirectory) Then
		If DirCreate($sExportDirectory) = 0 Then Return SetError(1, 0, 0)
	EndIf
	If StringRight($sExportDirectory, 1) <> "\" Then $sExportDirectory &= "\" ; Make sure there is a backslash at the end of the path
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then $hDebugFile = FileOpen($sDebugFile, $FO_APPEND)
	Local $iItemCount = 0
	For $i = 1 To $aContacts[0][0]
		$iIndex = 0
		$iCBFlag = $iOLT_FlagProcessRecord
		; $oContact = _OL_ItemGet($oOL, $aContacts[$i][0], Default, -1)
		$oContact = $oOL.Session.GetItemFromID($aContacts[$i][0])
		If @error Then Return SetError(2, @error, 0)
		; Replace invalid characters from filename with underscore
		$sTemp = StringRegExpReplace($oContact.LastName & "_" & $oContact.FirstName & "_" & $oContact.Companies, '[ \/:*?"<>|]', '_')
		$sExportPath = $sExportDirectory & $sTemp
		If FileExists($sExportPath & ".VCF") Then
			While 1
				$iIndex += 1
				If Not FileExists($sExportPath & "_" & $iIndex & ".VCF") Then
					$sExportPath = $sExportPath & "_" & $iIndex & ".VCF"
					ExitLoop
				EndIf
			WEnd
		Else
			$sExportPath = $sExportPath & ".VCF"
		EndIf
		If $sCallBack <> "" Then
			If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' started. Processing record #" & $i & " (zero based)", $iDebug, $hDebugFile)
			; $vCBResult = Call($sCallBack, $i) <== Not return needed as the data can be modified in the Global data array
			Call($sCallBack, $i)
			$iError = @error
			$iCBFlag = @extended
			If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' ended. Flag = " & $iCBFlag & ", @error = " & $iError & " (0x" & Hex($iError) & ").", $iDebug, $hDebugFile)
			If $iError Then Return SetError(4, $iError, 0)
			If $iCBFlag <= 0 Then Return SetError(5, 0, "")
			; If BitAND($iCBFlag, $iOLT_FlagReturnData) = $iOLT_FlagReturnData Then $aOutlook = $vCBResult ; Callback returned modified data <== Not needed as the data can be modified in the Global data array
			If BitAND($iCBFlag, $iOLT_FlagCancelAll) = $iOLT_FlagCancelAll Then ExitLoop
		EndIf
		If BitAND($iDebug, $iOLT_DebugCreateOff) <> $iOLT_DebugCreateOff And BitAND($iCBFlag, $iOLT_FlagProcessRecord) = $iOLT_FlagProcessRecord Then ; Create the VCF file
			If $iDebug Then __OLT_Debug("", "Saving contact as VCF-file: " & $sTemp, $iDebug, $hDebugFile)
			$oContact.SaveAs($sExportPath, $olVCard)
			If @error Then Return SetError(3, @error, 0)
			$iItemCount += 1
		EndIf
	Next
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then FileClose($sDebugFile)
	Return $iItemCount

EndFunc   ;==>_OLT_Contact_Export_VCF
#ce

; #FUNCTION# ====================================================================================================================
; Name...........: _OLT_Export
; Description ...: Export Outlook items as VCF, ICS, CSV or Excel Workbook.
; Syntax.........: _OLT_Export($oOL, $aItems, $iExportType, $sExportPath[, $sProperties = ""[, $bHeader = True[, $sCallback = ""[, $iDebug = $iOLT_DebugOff[, $sDebugFile = @ScriptDir & "\OLT_Debug.txt"]]]]])
; Parameters ....: $oOL                       - Outlook object returned by a preceding call to _OL_Open()
;                  $aItems                    - 2D array as returned by _OL_Itemfind holding the EntryIDs of the items to be exported in column 0
;                  $iExporttype               - Format of the export file to be created. Can be one of the following values:
;                  |$iOLT_ExportTypeCSV (1)   - Export Outlook items to a CSV file
;                  |$iOLT_ExportTypeExcel (2) - Export Outlook items to an Excel Workbook
;                  |$iOLT_ExportTypeVCF (4)   - Export Outlook contact items to a VCF file (vCard)
;                  |$iOLT_ExportTypeICS (8)   - Export Outlook appointment items to an ICS file (iCalendar)
;                  $sExportPath               - VCF and ICS:
;                                               Directory where the items should be exported to. Will be created if it doesn't exist
;                                             - For CSV and Excel:
;                                               Directory and filename/extension of the CSV/Excel file to be created
;                  $sProperties               - Optional but mandatory for CSV and Excel: Comma separated list of properties to export
;                  $bHeader                   - Optional: Write header line to Excel and CSV exports. Default = True
;                  $sCallback                 - Optional: Name of a function to be called before a new event is created. See Remarks for details
;                  $iDebug                    - Optional: Sets the flags for debugging. Possible values (can be a combination of one or more of the following values):
;                  |$iOLT_DebugOff (0)        - No debugging (default)
;                  |$iOLT_DebugConsole (1)    - Writes debugging messages to the console
;                  |$iOLT_DebugFile (2)       - Appends debugging messages to a file specified by parameter $sDebugFile
;                  |$iOLT_DebugProperties (4) - Writes the Outlook properties used to create the event to the debugging destination
;                  |$iOLT_DebugCreateOff (8)  - Do not create the event item. Allows to test the import without writing events to Outlook
;                  $sDebugFile - Optional: Path of the file to write debugging messages to (default = @ScriptDir & "\OLT_Debug.txt")
; Return values .: Success - number of exported items
;                  Failure - Returns 0 and sets @error:
;                  |1  - Could not create output directory as specified by $sExportPath
;                  |2  - _OL_ItemGet returned an error. See @extended for the error code
;                  |3  - SaveAs returned an error. See @extended for the COM error code
;                  |4  - $iExportType is invalid
;                  |5  - Error returned by built-in function Call. @extended is set to @error as returned by Call. For details please check the AutoIt help file
;                  |6  - The callback function did not set @extended so the function does not know how to process the record
;                  |7  - The output file $sExportPath for $iOLT_ExportTypeCSV nd $iOLT_ExportTypeExcel already exists
;                  |8  - Function _ArrayAdd returned an error. See @extended for the error code
;                  |9  - Function _Excel_Open returned an error. See @extended for the COM error code
;                  |10 - Function _Excel_BookNew returned an error. See @extended for the COM error code
;                  |11 - Function _Excel_RangeWrite returned an error. See @extended for the COM error code
;                  |12 - Function _Excel_BookSaveAs returned an error. See @extended for the COM error code
;                  |13 - Function _Excel_Close returned an error. See @extended for the COM error code
;                  |14 - Function _WriteCSV returned an error. See @extended for the error code:
;                  |     1 - No valid 2D-Array
;                  |     2 - Could not open file
;                  |15 - Parameter $sProperties is empty. Is mandatory for CSV and Excel export.
; Author ........: water
; Modified ......:
; Remarks .......: VCF: Files are named LastName_FirstName_Companies. If a file already exists "_n" is appended. Where n > 0 until the filename is unique.
;                  ICS: Files are named Subject_StartdateTime. If a file already exists "_n" is appended. Where n > 0 until the filename is unique.
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _OLT_Export($oOL, ByRef $aItems, $iExportType, $sExportPath, $sProperties = "", $bHeader = True, $sCallBack = "", $iDebug = $iOLT_DebugOff, $sDebugFile = @ScriptDir & "\OLT_Debug.txt")

	Local $hDebugFile, $oItem, $iItemCount = 0, $iIndex = 0, $sTempName, $sTempPath, $iCBFlag = 0, $iError, $sExt, $oExcel, $oWorkbook, $aExport[0][0], $aProperties ; , $vCBResult
	If $iExportType = $iOLT_ExportTypeVCF Or $iExportType = $iOLT_ExportTypeICS Then
		If Not FileExists($sExportPath) Then
			If DirCreate($sExportPath) = 0 Then Return SetError(1, 0, 0)
		EndIf
		If StringRight($sExportPath, 1) <> "\" Then $sExportPath &= "\" ; Make sure there is a backslash at the end of the path
	EndIf
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then $hDebugFile = FileOpen($sDebugFile, $FO_APPEND)
	Switch $iExportType
		Case $iOLT_ExportTypeVCF, $iOLT_ExportTypeICS
			For $i = 1 To $aItems[0][0]
				$iIndex = 0
				$iCBFlag = $iOLT_FlagProcessRecord
				$oItem = _OL_ItemGet($oOL, $aItems[$i][0], Default, -1)
				If @error Then Return SetError(2, @error, 0)
				; Build filename and replace invalid characters with underscore
				If $iExportType = $iOLT_ExportTypeVCF Then
					$sTempName = StringRegExpReplace($oItem.LastName & "_" & $oItem.FirstName & "_" & $oItem.Companies, '[ \/:*?"<>|]', '_') ; replace invalid characters with underscore
					$sExt = ".VCF"
				EndIf
				If $iExportType = $iOLT_ExportTypeICS Then
					$sTempName = StringRegExpReplace($oItem.Subject & "_" & $oItem.Start, '[ \/:*?"<>|]', '_')
					$sExt = ".ICS"
				EndIf
				$sTempPath = $sExportPath & $sTempName
				If FileExists($sTempPath & $sExt) Then
					While 1
						$iIndex += 1
						If Not FileExists($sTempPath & "_" & $iIndex & $sExt) Then
							$sTempPath = $sTempPath & "_" & $iIndex & $sExt
							ExitLoop
						EndIf
					WEnd
				Else
					$sTempPath = $sTempPath & $sExt
				EndIf
				If $sCallBack <> "" Then
					If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' started. Processing record #" & $i & " (zero based)", $iDebug, $hDebugFile)
					; $vCBResult = Call($sCallBack, $i) <== Not return needed as the data can be modified in the Global data array
					Call($sCallBack, $i)
					$iError = @error
					$iCBFlag = @extended
					If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' ended. Flag = " & $iCBFlag & ", @error = " & $iError & " (0x" & Hex($iError) & ").", $iDebug, $hDebugFile)
					If $iError Then Return SetError(5, $iError, 0)
					If $iCBFlag <= 0 Then Return SetError(6, 0, "")
					; If BitAND($iCBFlag, $iOLT_FlagReturnData) = $iOLT_FlagReturnData Then $aOutlook = $vCBResult ; Callback returned modified data <== Not needed as the data can be modified in the Global data array
					If BitAND($iCBFlag, $iOLT_FlagCancelAll) = $iOLT_FlagCancelAll Then ExitLoop
				EndIf
				If BitAND($iDebug, $iOLT_DebugCreateOff) <> $iOLT_DebugCreateOff And BitAND($iCBFlag, $iOLT_FlagProcessRecord) = $iOLT_FlagProcessRecord Then ; Create the item
					If $iDebug Then __OLT_Debug("", "Saving item as " & StringMid($sExt, 2) & "-file: " & $sTempName, $iDebug, $hDebugFile)
					$oItem.SaveAs($sTempPath, $iExportType)
					If @error Then Return SetError(3, @error, 0)
					$iItemCount += 1
				EndIf
			Next
		Case $iOLT_ExportTypeExcel, $iOLT_ExportTypeCSV
			If FileExists($sExportPath) Then Return SetError(7, 0, 0)
			If StringStripWS($sProperties, $STR_STRIPALL) = "" Then Return SetError(15, 0, 0)
			$aProperties = StringSplit($sProperties, ",")
			Local $aItem[$aProperties[0]]
			ReDim $aExport[0][$aProperties[0]]
			If $bHeader Then
				ReDim $aExport[1][$aProperties[0]]
				For $i = 1 To $aProperties[0]
					$aExport[0][$i - 1] = $aProperties[$i]
				Next
			EndIf
			Local $aTemp[1][$aProperties[0]]
			For $i = 1 To $aItems[0][0]
				$iIndex = 0
				$iCBFlag = $iOLT_FlagProcessRecord
				$aItem = _OL_ItemGet($oOL, $aItems[$i][0], Default, $sProperties)
				If @error Then Return SetError(2, @error, 0)
				If $sCallBack <> "" Then
					If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' started. Processing record #" & $i & " (zero based)", $iDebug, $hDebugFile)
					; $vCBResult = Call($sCallBack, $i) <== Not return needed as the data can be modified in the Global data array
					Call($sCallBack, $i)
					$iError = @error
					$iCBFlag = @extended
					If $iDebug Then __OLT_Debug("", "Callback function '" & $sCallBack & "' ended. Flag = " & $iCBFlag & ", @error = " & $iError & " (0x" & Hex($iError) & ").", $iDebug, $hDebugFile)
					If $iError Then Return SetError(5, $iError, 0)
					If $iCBFlag <= 0 Then Return SetError(6, 0, "")
					; If BitAND($iCBFlag, $iOLT_FlagReturnData) = $iOLT_FlagReturnData Then $aOutlook = $vCBResult ; Callback returned modified data <== Not needed as the data can be modified in the Global data array
					If BitAND($iCBFlag, $iOLT_FlagCancelAll) = $iOLT_FlagCancelAll Then ExitLoop
				EndIf
				If BitAND($iDebug, $iOLT_DebugCreateOff) <> $iOLT_DebugCreateOff And BitAND($iCBFlag, $iOLT_FlagProcessRecord) = $iOLT_FlagProcessRecord Then ; Create the item
					For $ii = 1 To $aItem[0][0]
						$aTemp[0][$ii - 1] = $aItem[$ii][1]
					Next
					_ArrayAdd($aExport, $aTemp)
					If @error Then Return SetError(8, @error, 0)
					$iItemCount += 1
				EndIf
			Next
			If $iExportType = $iOLT_ExportTypeExcel Then
				If $iDebug Then __OLT_Debug("", "Saving item as Excel: " & $sTempName, $iDebug, $hDebugFile)
				$oExcel = _Excel_Open(Default, Default, Default, Default, True)
				If @error Then Return SetError(9, @error, 0)
				$oWorkbook = _Excel_BookNew($oExcel)
				If @error Then Return SetError(10, @error, 0)
				_Excel_RangeWrite($oWorkbook, Default, $aExport, "A1")
				If @error Then Return SetError(11, @error, 0)
				_Excel_BookSaveAs($oWorkbook, $sExportPath)
				If @error Then Return SetError(12, @error, 0)
				_Excel_Close($oExcel, False)
				If @error Then Return SetError(13, @error, 0)
			Else
				If $iDebug Then __OLT_Debug("", "Saving item as CSV: " & $sTempName, $iDebug, $hDebugFile)
				_WriteCSV($sExportPath, $aExport)
				If @error Then Return SetError(14, @error, 0)
			EndIf
		Case Else
			Return SetError(4, 0, 0)
	EndSwitch
	If BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile Then FileClose($sDebugFile)
	Return $iItemCount

EndFunc   ;==>_OLT_Export

; #INTERNAL_USE_ONLY#============================================================================================================
; Name...........: __OLT_LineSplit
; Description ...: Splits a line into the parameter(s) and value(s) part.
; Syntax.........: __OLT_LineSplit($sLine, byRef $aParams, $aValues)
; Parameters ....: $sLine   - Line to split into parameter(s) and value(s).
;                  $aParams - 0-based array with the parameters part of a line.
;                  $aValues - 0-based array with the values part of a line.
; Return values .: Success - Sets arrays $aParams and $aValues with the extracted parameters and values.
;                  Failure - Returns "" and sets @error:
;                  |1 - StringInStr didn't find a match
; Author ........: water
; Modified ......:
; Remarks .......: A iCal or vCard line looks like:
;                    Parameter-1;Parameter-2;...;Parameter-n:Value-1;Value-2;...;Value-n
;                  The function does not correctly handle separator characters (:;) embedded in a string!
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func __OLT_LineSplit($sLine, ByRef $aParams, ByRef $aValues, ByRef $sParamsRaw, ByRef $sValuesRaw)

	Local $iPos = StringInStr($sLine, ":")
	If $iPos = 0 Or @error Then Return SetError(1, 0, "")
	$sParamsRaw = StringLeft($sLine, $iPos - 1)
	$sValuesRaw = StringMid($sLine, $iPos + 1)
	$aParams = StringSplit($sParamsRaw, ";", $STR_NOCOUNT)
	$aValues = StringSplit($sValuesRaw, ";", $STR_NOCOUNT)
	#cs
		Local $aSplit = StringSplit($sLine, ":", $STR_NOCOUNT)
		If @error Then Return SetError(@error, 0, "")
		$sParamsRaw = $aSplit[0]
		$sValuesRaw = $aSplit[1]
		$aParams = StringSplit($aSplit[0], ";", $STR_NOCOUNT)
		If UBound($aSplit) > 1 Then $aValues = StringSplit($aSplit[1], ";", $STR_NOCOUNT)
	#ce

EndFunc   ;==>__OLT_LineSplit

; #INTERNAL_USE_ONLY#============================================================================================================
; Name...........: __OLT_Debug
; Description ...: Writes debugging messages to a destination.
; Syntax.........: __OLT_Debug($sHeader, $sMessage, $iDebug, $hDebugFile)
; Parameters ....: $sHeader    - Header text which is prepended to the message. Useful when the message consists of multiple lines.
;                  $sMessage   - Message to write to the debugging destination
;                  $iDebug     - Sets the destination for the debugging output. Possible values:
;                  |0 - No debugging
;                  |1 - Writes debugging messages to the console
;                  |2 - Writes debugging messages to a file specified by parameter $hDebugFile
;                  $hDebugFile - handle to the debug file
; Return values .:
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func __OLT_Debug($sHeader, $sMessage, $iDebug, $hDebugFile)

	If StringStripWS($sMessage, $STR_STRIPALL) = "" Then Return ; Ignore empty lines
	If StringLeft(StringReplace($sMessage, " ", ""), 1) = @TAB Then Return ; Ignore lines starting with a tab
	If $sHeader <> "" Then $sMessage = $sHeader & @CRLF & $sMessage
	If StringLeft($sMessage, 6) = "BEGIN:" Or StringLeft($sMessage, 4) = "END:" Then $sMessage = StringLeft($sMessage & "------------------------------------------------------------", 60)
	Local $sTimeStamp = @YEAR & "/" & @MON & "/" & @MDAY & " " & @HOUR & ":" & @MIN & ":" & @SEC & "." & @MSEC
	Local $aLines = StringSplit($sMessage, @CRLF, $STR_ENTIRESPLIT)
	Select
		Case BitAND($iDebug, $iOLT_DebugConsole) = $iOLT_DebugConsole
			For $i = 1 To $aLines[0]
				ConsoleWrite($sTimeStamp & " " & $aLines[$i] & @CRLF)
			Next
		Case BitAND($iDebug, $iOLT_DebugFile) = $iOLT_DebugFile
			For $i = 1 To $aLines[0]
				FileWriteLine($hDebugFile, $sTimeStamp & " " & $aLines[$i])
			Next
	EndSelect

EndFunc   ;==>__OLT_Debug

; #INTERNAL_USE_ONLY#============================================================================================================
; Name...........: __OLT_iCal_DateTimeTranslate
; Description ...: Translates DATE or DATE-TIME to YYYY/MM/DD HH:MM:SS.
; Syntax.........: __OLT_iCal_DateTimeTranslate($sDateTime)
; Parameters ....: $sDateTime - DATE or DATE-TIME to be translated
; Return values .: Success - formatted DATE-TIME string. Flag values get returned by setting @extended. See Remarks.
;                  Failure - Returns "" and sets @error:
;                  2 = Pattern invalid. @extended = offset of error in pattern
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func __OLT_iCal_DateTimeTranslate(ByRef $aParams, ByRef $aValues)

	Local $sTemp, $sDTValue, $bFlag = 0
	If $aParams[1] = "VALUE=DATE" Or $aParams[1] = "DATE" Then
		$sDTValue = StringRegExpReplace($aValues[0], "(\d{4})(\d{2})(\d{2})", "$1/$2/$3") ; translate last value after the split (YYYYMMDD) to YYYY/MM/DD
		If @error Then Return SetError(@error, @extended, "")
		$bFlag = $iOLT_FlagDTAlLDay ; AllDayEvent = True
	ElseIf $aParams[1] = "VALUE=DATE-TIME" Or $aParams[1] = "DATE-TIME" Or StringLeft($aParams[1], 5) = "TZID=" Then
		$sTemp = StringReplace($aValues[0], "Z", "") ; Remove "Z" at the end of the string for UTC
		If @extended > 0 Then $bFlag = $iOLT_FlagDTUTC ; UTC = True
		$sDTValue = StringRegExpReplace($sTemp, "(\d{4})(\d{2})(\d{2})T(\d{2})(\d{2})(\d{2})", "$1/$2/$3 $4:$5:$6") ; translate last value after the split (YYYYMMDDTHHMMSS) to YYYY/MM/DD HH:MM:SS
		If @error Then Return SetError(@error, @extended, "")
	EndIf
	Return SetError(0, $bFlag, $sDTValue)

EndFunc   ;==>__OLT_iCal_DateTimeTranslate

; #INTERNAL_USE_ONLY#============================================================================================================
; Name...........: iCalDurationTranslate
; Description ...: Translates the DURATION property and adds the values to the start DATE-TIME.
; Syntax.........: iCalDurationTranslate($sDateTime, $sDuration)
; Parameters ....: $sDateTime - Start DATE-TIME (YYYY/MM/DD HH:MM:SS) the duration should be added to.
;                    If this parameter is set to "" then the minute part of the duration will be returned.
;                  $sDuration - Duration to be added to the start DATE-TIME.
; Return values .: Success - Calculated DATE-TIME. Format: YYYY/MM/DD HH:MM:SS.
;                  If $sDateTime is set to "" then the minute part of the duration will be returned (used by VALARM component)
;                  Failure - Returns 0 and sets @error:
;                  |1  - StringRegExp: No matches found
;                  |2  - StringRegExp: Bad pattern. @extended = offset of error in pattern.
; Author ........: water, mikell (RegExp for "DURATION")
; Modified ......:
; Remarks .......: RegExp taken from https://www.autoitscript.com/forum/topic/197397-translate-rfc-2445-duration-format/?do=reportComment&comment=1415924
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func __OLT_iCal_DurationTranslate($sDateTime, $sDuration)

	Local $aResult = StringRegExp($sDuration, '^([-+]?)P(?:(\d*)W)?(?:(\d*)D)?T?(?:(\d*)H)?(?:(\d*)M)?(?:(\d*)S?)', 3)
	If @error Then Return SetError(@error, @extended, 0)
	; If result does not have to be added to a DATE-TIME then just return the minute part of $sDuration (used for alerts)
	If $sDateTime = "" Then Return $aResult[4]
	ReDim $aResult[6]
	; Translate string to number
	$aResult[1] = Number($aResult[1])
	$aResult[2] = Number($aResult[2])
	$aResult[3] = Number($aResult[3])
	$aResult[4] = Number($aResult[4])
	$aResult[5] = Number($aResult[5])
	; Set to negative if +/- sign has been set
	If $aResult[0] = "-" Then
		$aResult[1] *= -1
		$aResult[2] *= -1
		$aResult[3] *= -1
		$aResult[4] *= -1
		$aResult[5] *= -1
	EndIf
	If $aResult[1] <> 0 Then $sDateTime = _DateAdd("w", $aResult[1], $sDateTime)
	If $aResult[2] <> 0 Then $sDateTime = _DateAdd("d", $aResult[2], $sDateTime)
	If $aResult[3] <> 0 Then $sDateTime = _DateAdd("h", $aResult[3], $sDateTime)
	If $aResult[4] <> 0 Then $sDateTime = _DateAdd("n", $aResult[4], $sDateTime)
	If $aResult[5] <> 0 Then $sDateTime = _DateAdd("s", $aResult[5], $sDateTime)
	Return $sDateTime

EndFunc   ;==>__OLT_iCal_DurationTranslate

; #INTERNAL_USE_ONLY#============================================================================================================
; Name...........: __OLT_iCal_VAlarmProcess
; Description ...: Processes the VAlarm component of an iCal file (from VALARM:BEGIN to VALARM:END).
; Syntax.........: __OLT_iCal_VAlarmProcess(ByRef $aICS, ByRef $aEvent, ByRef $iIndex, ByRef $iMaxIndex[, $iDebug = False])
; Parameters ....: $aICS       - Variable with the same name passed from the main function
;                  $iIndex     - Variable with the same name passed from the main function
;                  $iMaxIndex  - Variable with the same name passed from the main function
;                  $iDebug     - Variable with the same name passed from the main function
;                  $hDebugFile - Variable with the same name passed from the main function
; Return values .: Success - Time in minutes the alarms goes off prior to the event
;                  Failure - Returns 0 and sets @error:
;                  |1  - StringRegExp: No matches found
;                  |2  - StringRegExp: Bad pattern. @extended = offset of error in pattern.
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func __OLT_iCal_VAlarmProcess(ByRef $aICS, ByRef $iIndex, ByRef $iMaxIndex, $iDebug, $hDebugFile)

	Local $iAlarm
	If $iDebug Then __OLT_Debug("BEGIN:VALARM", "", $iDebug, $hDebugFile)
	Do
		$iIndex += 1
		If $iIndex > $iMaxIndex Then ExitLoop
		Select
			Case StringLeft($aICS[$iIndex], 8) = "TRIGGER:"
				$iAlarm = __OLT_iCal_DurationTranslate("", StringMid($aICS[$iIndex], 9))
				If @error Then Return SetError(@error, @extended, 0)
				If $iDebug Then __OLT_Debug("Processed: ", "  IN.: " & $aICS[$iIndex] & @CRLF & "  OUT: " & $iAlarm, $iDebug, $hDebugFile)
			Case $aICS[$iIndex] = "END:VALARM"
				If $iDebug Then __OLT_Debug("END:VALARM", "", $iDebug, $hDebugFile)
			Case Else
				If $iDebug Then __OLT_Debug("Ignored: ", "  " & $aICS[$iIndex], $iDebug, $hDebugFile)
		EndSelect
	Until $aICS[$iIndex] = "END:VALARM"
	Return $iAlarm

EndFunc   ;==>__OLT_iCal_VAlarmProcess

; #FUNCTION# ====================================================================================================================
; Name...........: _WriteCSV
; Description ...: Writes a CSV-file
; Syntax.........: _WriteCSV($sFile, Const ByRef $aData, $sDelimiter, $sQuote, $iFormat=0)
; Parameters ....: $sFile      - Destination file
;                  $aData      - [Const ByRef] 0-based 2D-Array with data
;                  $sDelimiter - [optional] Fieldseparator (default: ,)
;                  $sQuote     - [optional] Quote character (default: ")
;                  $iFormat    - [optional] character encoding of file (default: 0)
;                  |0 or 1 - ASCII writing
;                  |2      - Unicode UTF16 Little Endian writing (with BOM)
;                  |3      - Unicode UTF16 Big Endian writing (with BOM)
;                  |4      - Unicode UTF8 writing (with BOM)
;                  |5      - Unicode UTF8 writing (without BOM)
; Return values .: Success - True
;                  Failure - 0, sets @error to:
;                  |1 - No valid 2D-Array
;                  |2 - Could not open file
; Author ........: ProgAndy
; Modified.......:
; Remarks .......:
; Related .......: _ParseCSV
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _WriteCSV($sFile, Const ByRef $aData, $sDelimiter = ',', $sQuote = '"', $iFormat = 0)
	Local Static $aEncoding[6] = [2, 2, 34, 66, 130, 258]
	If $sDelimiter = "" Or IsKeyword($sDelimiter) Then $sDelimiter = ','
	If $sQuote = "" Or IsKeyword($sQuote) Then $sQuote = '"'
	Local $iBound = UBound($aData, 1), $iSubBound = UBound($aData, 2)
	If Not $iSubBound Then Return SetError(2, 0, 0)
	Local $hFile = FileOpen($sFile, $aEncoding[$iFormat])
	If @error Then Return SetError(2, @error, 0)
	For $i = 0 To $iBound - 1
		For $j = 0 To $iSubBound - 1
			FileWrite($hFile, $sQuote & StringReplace($aData[$i][$j], $sQuote, $sQuote & $sQuote, 0, 1) & $sQuote)
			If $j < $iSubBound - 1 Then FileWrite($hFile, $sDelimiter)
		Next
		FileWrite($hFile, @CRLF)
	Next
	FileClose($hFile)
	Return True
EndFunc   ;==>_WriteCSV
