#AutoIt3Wrapper_AU3Check_Parameters=-w 3 -w 4 -w 5 -w 6 -d


#include <Constants.au3>
#include "..\CryptoNG.au3"
#include <GUIConstantsEx.au3>
#include <WindowsConstants.au3>
#include <EditConstants.au3>
#include <FontConstants.au3>

;=========================================================
; GUI Log
;=========================================================
Const $CONSOLE_BACK_COLOR = 0xF5F5F5, _  ;WhiteSmoke
	  $CONSOLE_FORE_COLOR = 0x000000, _  ;Black
	  $CONSOLE_WIDTH      = 800,      _
	  $CONSOLE_HEIGHT     = 600

Global $frmConsole = GUICreate("Console Log (Press ESC to Close)", $CONSOLE_WIDTH, $CONSOLE_HEIGHT, 302, 218, BitOR($GUI_SS_DEFAULT_GUI,$WS_MAXIMIZEBOX,$WS_SIZEBOX,$WS_THICKFRAME,$WS_TABSTOP))
Global $ctlEditBox = GUICtrlCreateEdit("", 0, 0, $CONSOLE_WIDTH, $CONSOLE_HEIGHT, BitOR($ES_AUTOVSCROLL,$ES_AUTOHSCROLL,$ES_WANTRETURN,$ES_READONLY,$WS_VSCROLL,$WS_HSCROLL))
GUICtrlSetResizing($ctlEditBox, $GUI_DOCKLEFT+$GUI_DOCKRIGHT+$GUI_DOCKTOP+$GUI_DOCKBOTTOM)
GUICtrlSetFont($ctlEditBox, 11, $FW_MEDIUM, 0, "Consolas")
GUICtrlSetColor($ctlEditBox, $CONSOLE_FORE_COLOR)
GUISetBkColor($CONSOLE_BACK_COLOR, $frmConsole)
GUICtrlSetBkColor($ctlEditBox, $CONSOLE_BACK_COLOR)
GUISetState(@SW_SHOW, $frmConsole)


;============================================================================
; Examples - Uncomment the example(s) that you would like to execute
;============================================================================

show_udf_version_example()

;~ eunmerate_registered_providers_example()
;~ eunmerate_algorithms_example()
;~ eunmerate_key_storage_providers_example()

;~ aes_cbc_encrypt_decrypt_example()
;~ aes_cbc_encrypt_decrypt_multibyte_example()
;~ aes_cbc_encrypt_decrypt_with_iv_example()

;~ aes_cbc_encrypt_decrypt_file_example()
;~ aes_cbc_encrypt_decrypt_file_with_iv_example()

;~ aes_encrypt_decrypt_file_example() ;OLD AES EXAMPLE WITH DEFAULT IV

;~ aes_ecb_encrypt_decrypt_example()

;~ aes_gcm_encrypt_decrypt_example()

;~ tripledes_cbc_encrypt_example()
;~ tripledes_cbc_decrypt_example()
;~ tripledes_cbc_encrypt_decrypt_file_example()

;~ des_encrypt_decrypt_example()
;~ rc2_encrypt_decrypt_example()
;~ rc4_encrypt_decrypt_example()
;~ rc4_encrypt_decrypt_no_hash_example()

;~ hash_data_example()
;~ hash_file_example()

;~ pbkdf2_example()

;~ generate_random_example()

;~ create_rsa_key_pair_example()
;~ rsa_public_private_key_encrypt_decrypt_data_example()
;~ rsa_sign_and_verify_example()

;~ create_ecdsa_key_pair_example()
;~ ecdsa_sign_and_verify_example()

;~ convert_binary_to_string_example()
;~ convert_string_to_binary_example()

write_to_log(@CRLF & "Done")

;Wait for GUI close request
While 1
	If GUIGetMsg() = $GUI_EVENT_CLOSE Then ExitLoop
WEnd


;==========================================================================

Func show_udf_version_example()
	write_to_log(@CRLF)
	write_to_log("CryptoNG UDF v" & _CryptoNG_Version() & @CRLF)
EndFunc

;==========================================================================

Func eunmerate_registered_providers_example()

	Local $aResult[0]

	;Show available Cipher algorithms
	$aResult = _CryptoNG_EnumRegisteredProviders()
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	write_to_log(@CRLF & "Registered Providers" & @CRLF)
	For $row in $aResult
		write_to_log("- " & $row & @CRLF)
	Next

EndFunc

;==========================================================================

Func eunmerate_algorithms_example()

	Local $aResult[0]

	;Show available Cipher algorithms
	$aResult = _CryptoNG_EnumAlgorithms($CNG_BCRYPT_CIPHER_OPERATION)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	write_to_log(@CRLF & "Cipher Algorithms" & @CRLF)
	For $row in $aResult
		write_to_log("- " & $row & @CRLF)
	Next

	;Show available HASH algorithms
	$aResult = _CryptoNG_EnumAlgorithms($CNG_BCRYPT_HASH_OPERATION)
	If @error Then
		write_to_log(StringFormat("ERROR: @error = %s  Status = %X", @error, @extended) & @CRLF)
		Return False
	EndIf

	write_to_log(@CRLF & "Hash  Algorithms" & @CRLF)
	For $row in $aResult
		write_to_log("- " & $row & @CRLF)
	Next

	;Show available Asymmetric Encryption algorithms
	$aResult = _CryptoNG_EnumAlgorithms($CNG_BCRYPT_ASYMMETRIC_ENCRYPTION_OPERATION)
	If @error Then
		write_to_log(StringFormat("ERROR: @error = %s  Status = %X", @error, @extended) & @CRLF)
		Return False
	EndIf

	write_to_log(@CRLF & "Asymmetric Encryption Algorithms" & @CRLF)
	For $row in $aResult
		write_to_log("- " & $row & @CRLF)
	Next

	;Show available Secret Agreement algorithms
	$aResult = _CryptoNG_EnumAlgorithms($CNG_BCRYPT_SECRET_AGREEMENT_OPERATION)
	If @error Then
		write_to_log(StringFormat("ERROR: @error = %s  Status = %X", @error, @extended) & @CRLF)
		Return False
	EndIf

	write_to_log(@CRLF & "Secret Agreement Algorithms" & @CRLF)
	For $row in $aResult
		write_to_log("- " & $row & @CRLF)
	Next

	;Show available Signature algorithms
	$aResult = _CryptoNG_EnumAlgorithms($CNG_BCRYPT_SIGNATURE_OPERATION)
	If @error Then
		write_to_log(StringFormat("ERROR: @error = %s  Status = %X", @error, @extended) & @CRLF)
		Return False
	EndIf

	write_to_log(@CRLF & "Signature Algorithms" & @CRLF)
	For $row in $aResult
		write_to_log("- " & $row & @CRLF)
	Next

	;Show available Signature algorithms
	$aResult = _CryptoNG_EnumAlgorithms($CNG_BCRYPT_RNG_OPERATION)
	If @error Then
		write_to_log(StringFormat("ERROR: @error = %s  Status = %X", @error, @extended) & @CRLF)
		Return False
	EndIf

	write_to_log(@CRLF & "Random Number Generator (RNG) Algorithms" & @CRLF)
	For $row in $aResult
		write_to_log("- " & $row & @CRLF)
	Next

	;Show multiple available algorithms
	$aResult = _CryptoNG_EnumAlgorithms($CNG_BCRYPT_CIPHER_OPERATION + $CNG_BCRYPT_HASH_OPERATION + $CNG_BCRYPT_ASYMMETRIC_ENCRYPTION_OPERATION)
	If @error Then
		write_to_log(StringFormat("ERROR: @error = %s  Status = %X", @error, @extended) & @CRLF)
		Return False
	EndIf

	write_to_log(@CRLF & "CIPHER/Hash/Asymmetric Encryption Algorithms" & @CRLF)
	For $row in $aResult
		write_to_log("- " & $row & @CRLF)
	Next

EndFunc

;==========================================================================

Func eunmerate_key_storage_providers_example()

	Local $aResult[0]

	;Show available Cipher algorithms
	$aResult = _CryptoNG_EnumKeyStorageProviders()
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	write_to_log(@CRLF & "Registered Key Store Providers" & @CRLF)
	For $row in $aResult
		write_to_log("- " & $row & @CRLF)
	Next

EndFunc

;==========================================================================

Func tripledes_cbc_encrypt_example()

	Const $ENCRYPTION_KEY = Binary("0x3ABEE19C277607BDF290DB8757DE477F46E3526A384A7D60"), _
	      $MESSAGE        = "This is a super-secret message.", _
		  $ALGORITHM      = "3DES-CBC"

	Local $xEncryptedMessage = ""

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_3DES_CBC_EncryptData($MESSAGE, $ENCRYPTION_KEY)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG 3DES CBC Encryption Example" & @CRLF)
	write_to_log(StringFormat("%s Plain Text Msg    = %s", $ALGORITHM, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encryption Key    = %s", $ALGORITHM, $ENCRYPTION_KEY) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message = %s", $ALGORITHM, $xEncryptedMessage) & @CRLF)

EndFunc

Func tripledes_cbc_decrypt_example()

	Const $ENCRYPTED_MESSAGE = Binary("0xEA8D26200F44D9BF187AA28A6EA87BD9DED32F12FE5E7C030D60B3A1E705DDE5"), _
	      $ENCRYPTION_KEY    = Binary("0x3ABEE19C277607BDF290DB8757DE477F46E3526A384A7D60"), _
		  $ALGORITHM         = "3DES-CBC"

	Local $sDecryptedMessage = ""

	;Decrypt message
	$sDecryptedMessage = _CryptoNG_3DES_CBC_DecryptData($ENCRYPTED_MESSAGE, $ENCRYPTION_KEY)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG 3DES CBC Decryption Example" & @CRLF)
	write_to_log(StringFormat("%s Encryption Key    = %s", $ALGORITHM, $ENCRYPTION_KEY) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message = %s", $ALGORITHM, $ENCRYPTED_MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message = %s", $ALGORITHM, $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func tripledes_cbc_encrypt_decrypt_file_example()

	Const $ALG_ID         = $CNG_BCRYPT_3DES_ALGORITHM, _
	      $INPUT_FILE     = "ExampleLoremIpsumDataFile.docx", _
	      $ENCRYPTED_FILE = "ExampleLoremIpsumDataFile - Encrypted.docx", _
	      $DECRYPTED_FILE = "ExampleLoremIpsumDataFile - Decrypted.docx", _
	      $SECRET         = "Secret Password!", _
		  $SECRET_SALT    = "Secret Salt"

	Local $vEncryptKey = ""


	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_3DES, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt file
	_CryptoNG_3DES_CBC_EncryptFile($INPUT_FILE, $ENCRYPTED_FILE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	_CryptoNG_3DES_CBC_DecryptFile($ENCRYPTED_FILE, $DECRYPTED_FILE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG 3DES File Encryption/Decryption Example" & @CRLF)
	write_to_log(StringFormat('%s Input file     = %s', $ALG_ID, $INPUT_FILE) & @CRLF)
	write_to_log(StringFormat('%s Encrypted file = %s', $ALG_ID, $ENCRYPTED_FILE) & @CRLF)
	write_to_log(StringFormat('%s Decrypted file = %s', $ALG_ID, $DECRYPTED_FILE) & @CRLF)

EndFunc

;==========================================================================

Func des_encrypt_decrypt_example()

	Const $MESSAGE      = "This is a super-secret message.", _
	      $SECRET       = "Secret Password!", _
		  $SECRET_SALT  = "Secret Salt"

	Local $sAlgId = "", $sDecryptedMessage = ""
	Local $vEncryptKey = ""
	Local $xEncryptedMessage = ""


	;DES Example
	$sAlgId = $CNG_BCRYPT_DES_ALGORITHM

	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_DES, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_EncryptData($sAlgId, $MESSAGE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_DecryptData($sAlgId, $xEncryptedMessage, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG DES Encryption/Decryption Example" & @CRLF)
	write_to_log(StringFormat("%s Plain text message   = %s", $sAlgId, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key          = %s", $sAlgId, $SECRET) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key Salt     = %s", $sAlgId, $SECRET_SALT) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key (Hashed) = %s", $sAlgId, $vEncryptKey) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message    = %s", $sAlgId, $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message    = %s", $sAlgId, $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func rc2_encrypt_decrypt_example()

	Const $MESSAGE      = "This is a super-secret message.", _
	      $SECRET       = "Secret Password!", _
		  $SECRET_SALT  = "Secret Salt"

	Local $sAlgId = "", $sDecryptedMessage = ""
	Local $vEncryptKey = ""
	Local $xEncryptedMessage = ""


	;RC2 Example
	$sAlgId = $CNG_BCRYPT_RC2_ALGORITHM

	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_RC2_128, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_EncryptData($sAlgId, $MESSAGE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_DecryptData($sAlgId, $xEncryptedMessage, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG RC2 Encryption/Decryption Example" & @CRLF)
	write_to_log(StringFormat("%s Plain text message   = %s", $sAlgId, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key          = %s", $sAlgId, $SECRET) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key Salt     = %s", $sAlgId, $SECRET_SALT) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key (Hashed) = %s", $sAlgId, $vEncryptKey) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message    = %s", $sAlgId, $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message    = %s", $sAlgId, $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func rc4_encrypt_decrypt_example()

	Const $MESSAGE      = "This is a super-secret message.", _
	      $SECRET       = "Secret Password!", _
		  $SECRET_SALT  = "Secret Salt"

	Local $sAlgId = "", $sDecryptedMessage = ""
	Local $vEncryptKey = ""
	Local $xEncryptedMessage = ""


	;RC4 Example
	$sAlgId = $CNG_BCRYPT_RC4_ALGORITHM

	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_RC4_128, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_EncryptData($sAlgId, $MESSAGE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_DecryptData($sAlgId, $xEncryptedMessage, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG RC4 Encryption/Decryption Example" & @CRLF)
	write_to_log(StringFormat("%s Plain text message   = %s", $sAlgId, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key          = %s", $sAlgId, $SECRET) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key Salt     = %s", $sAlgId, $SECRET_SALT) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key (Hashed) = %s", $sAlgId, $vEncryptKey) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message    = %s", $sAlgId, $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message    = %s", $sAlgId, $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func rc4_encrypt_decrypt_no_hash_example()
	#cs
		This is an example of using a non-hashed, secret password.  The encryption key for RC4 can be
		any length between 40 bits (5 bytes) and 512 bits (64 bytes).  It can be plain text or a
		binary value.  All that matters is that the it's length is within the specified range.
	#ce

	Const $MESSAGE = "This is a super-secret message.", _
	      $SECRET  = "Secret Password!"

	Local $sAlgId            = "", _
	      $sDecryptedMessage = ""

	Local $xEncryptedMessage = ""


	;RC4 Example
	$sAlgId = $CNG_BCRYPT_RC4_ALGORITHM

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_EncryptData($sAlgId, $MESSAGE, $SECRET)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_DecryptData($sAlgId, $xEncryptedMessage, $SECRET)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG RC4 Encryption/Decryption without Hash Example" & @CRLF)
	write_to_log(StringFormat("%s Plain text message   = %s", $sAlgId, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key          = %s", $sAlgId, $SECRET) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message    = %s", $sAlgId, $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message    = %s", $sAlgId, $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func aes_cbc_encrypt_decrypt_example()

	Const $ALG_ID       = $CNG_BCRYPT_AES_ALGORITHM, _
	      $MESSAGE      = "This is a super-secret message.", _
	      $SECRET       = "Secret Password!", _
		  $SECRET_SALT  = "Secret Salt"

	Local $sDecryptedMessage = ""
	Local $vEncryptKey = ""
	Local $xEncryptedMessage = ""


	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_AES_128, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_AES_CBC_EncryptData($MESSAGE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_AES_CBC_DecryptData($xEncryptedMessage, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG AES Data Encryption/Decryption Example" & @CRLF)
	write_to_log(StringFormat("%s Plain text message   = %s", $ALG_ID, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key          = %s", $ALG_ID, $SECRET) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key Salt     = %s", $ALG_ID, $SECRET_SALT) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key (Hashed) = %s", $ALG_ID, $vEncryptKey) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message    = %s", $ALG_ID, $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message    = %s", $ALG_ID, $sDecryptedMessage) & @CRLF)

EndFunc

Func aes_cbc_encrypt_decrypt_multibyte_example()

	Const $ALG_ID       = $CNG_BCRYPT_AES_ALGORITHM, _
	      $MESSAGE      = "Hello / 你好 / こんにちは / مرحبا", _
	      $SECRET       = "Secret Password!", _
		  $SECRET_SALT  = "Secret Salt"

	Local $sDecryptedMessage = ""
	Local $vEncryptKey = ""
	Local $xEncryptedMessage = ""


	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_AES_128, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_AES_CBC_EncryptData($MESSAGE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_AES_CBC_DecryptData($xEncryptedMessage, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG AES Data Encryption/Decryption with Multibyte Characters Example" & @CRLF)
	write_to_log(StringFormat("%s Plain text message   = %s", $ALG_ID, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key          = %s", $ALG_ID, $SECRET) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key Salt     = %s", $ALG_ID, $SECRET_SALT) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key (Hashed) = %s", $ALG_ID, $vEncryptKey) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message    = %s", $ALG_ID, $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message    = %s", $ALG_ID, $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func aes_cbc_encrypt_decrypt_with_iv_example()

	Const $ALG_ID       = $CNG_BCRYPT_AES_ALGORITHM, _
	      $MESSAGE      = "This is a super-secret message.", _
	      $SECRET       = "Secret Password!", _
		  $SECRET_SALT  = "Secret Salt"

	Local $sDecryptedMessage = ""

	Local $vEncryptKey = ""

	Local $xEncryptedMessage = Binary(""), _
	      $xIV               = Binary("")


	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_AES_256, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Generate a random, 16 byte, IV
	$xIV = _CryptoNG_GenerateRandom($CNG_BCRYPT_RNG_ALGORITHM, 16)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_AES_CBC_EncryptData($MESSAGE, $vEncryptKey, $xIV)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_AES_CBC_DecryptData($xEncryptedMessage, $vEncryptKey, $xIV)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG AES Data Encryption/Decryption with IV Example" & @CRLF)
	write_to_log(StringFormat("%s Plain text message    = %s", $ALG_ID, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key           = %s", $ALG_ID, $SECRET) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key Salt      = %s", $ALG_ID, $SECRET_SALT) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key (Hashed)  = %s", $ALG_ID, $vEncryptKey) & @CRLF)
	write_to_log(StringFormat("%s Initialization Vector = %s", $ALG_ID, $xIV) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message     = %s", $ALG_ID, $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message     = %s", $ALG_ID, $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func aes_cbc_encrypt_decrypt_file_example()

	Const $ALG_ID         = $CNG_BCRYPT_AES_ALGORITHM, _
	      $INPUT_FILE     = "ExampleLoremIpsumDataFile.docx", _
	      $ENCRYPTED_FILE = "ExampleLoremIpsumDataFile - Encrypted.docx", _
	      $DECRYPTED_FILE = "ExampleLoremIpsumDataFile - Decrypted.docx", _
	      $SECRET         = "Secret Password!", _
		  $SECRET_SALT    = "Secret Salt"

	Local $vEncryptKey = ""


	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_AES_192, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt file
	_CryptoNG_AES_CBC_EncryptFile($INPUT_FILE, $ENCRYPTED_FILE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	_CryptoNG_AES_CBC_DecryptFile($ENCRYPTED_FILE, $DECRYPTED_FILE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG AES File Encryption/Decryption Example" & @CRLF)
	write_to_log(StringFormat('%s Input file     = %s', $ALG_ID, $INPUT_FILE) & @CRLF)
	write_to_log(StringFormat('%s Encrypted file = %s', $ALG_ID, $ENCRYPTED_FILE) & @CRLF)
	write_to_log(StringFormat('%s Decrypted file = %s', $ALG_ID, $DECRYPTED_FILE) & @CRLF)

EndFunc

;==========================================================================

Func aes_cbc_encrypt_decrypt_file_with_iv_example()

	Const $ALG_ID         = $CNG_BCRYPT_AES_ALGORITHM, _
	      $INPUT_FILE     = "ExampleLoremIpsumDataFile.docx", _
	      $ENCRYPTED_FILE = "ExampleLoremIpsumDataFile - Encrypted.docx", _
	      $DECRYPTED_FILE = "ExampleLoremIpsumDataFile - Decrypted.docx", _
	      $SECRET         = "Secret Password!", _
		  $SECRET_SALT    = "Secret Salt"

	Local $vEncryptKey = ""

	Local $xIV = Binary("")


	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_AES_192, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Generate a random, 16 byte, IV
	$xIV = _CryptoNG_GenerateRandom($CNG_BCRYPT_RNG_ALGORITHM, 16)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt file
	_CryptoNG_AES_CBC_EncryptFile($INPUT_FILE, $ENCRYPTED_FILE, $vEncryptKey, $xIV)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	_CryptoNG_AES_CBC_DecryptFile($ENCRYPTED_FILE, $DECRYPTED_FILE, $vEncryptKey, $xIV)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG AES File Encryption/Decryption with IV Example" & @CRLF)
	write_to_log(StringFormat('%s Input file            = %s', $ALG_ID, $INPUT_FILE) & @CRLF)
	write_to_log(StringFormat('%s Initialization Vector = %s', $ALG_ID, $xIV) & @CRLF)
	write_to_log(StringFormat('%s Encrypted file        = %s', $ALG_ID, $ENCRYPTED_FILE) & @CRLF)
	write_to_log(StringFormat('%s Decrypted file        = %s', $ALG_ID, $DECRYPTED_FILE) & @CRLF)

EndFunc

;==========================================================================

Func aes_ecb_encrypt_decrypt_example()

	Const $ALG_ID       = $CNG_BCRYPT_AES_ALGORITHM, _
	      $MESSAGE      = "This is a super-secret message.", _
	      $SECRET       = "Secret Password!", _
		  $SECRET_SALT  = "Secret Salt"

	Local $sDecryptedMessage = ""
	Local $vEncryptKey = ""
	Local $xEncryptedMessage = ""


	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_AES_128, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_AES_ECB_EncryptData($MESSAGE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_AES_ECB_DecryptData($xEncryptedMessage, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG AES ECB Data Encryption/Decryption Example" & @CRLF)
	write_to_log(StringFormat("%s Plain text message   = %s", $ALG_ID, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key          = %s", $ALG_ID, $SECRET) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key Salt     = %s", $ALG_ID, $SECRET_SALT) & @CRLF)
	write_to_log(StringFormat("%s Encrypt Key (Hashed) = %s", $ALG_ID, $vEncryptKey) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message    = %s", $ALG_ID, $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message    = %s", $ALG_ID, $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func aes_gcm_encrypt_decrypt_example()

	Const $MESSAGE   = "-->This is a test<--", _
	      $KEY       = "-SuperSecretKey-",     _                               ; 128-bit (16 bytes)
	      $NONCE     = _CryptoNG_GenerateRandom($CNG_BCRYPT_RNG_ALGORITHM, 12) ; 96-bit  (12 bytes)

	Local $aCipherData[0]
	Local $sDecryptedMessage = ""
	Local $xEncryptedMessage = Binary(""), _
	      $xAuthTag          = Binary("")

	;Encrypt plain text message
	$aCipherData = _CryptoNG_AES_GCM_EncryptData($MESSAGE, $KEY, $NONCE)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	$xEncryptedMessage = $aCipherData[0]
	$xAuthTag          = $aCipherData[1]

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_AES_GCM_DecryptData($xEncryptedMessage, $KEY, $NONCE, $xAuthTag)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG AES GCM Data Encryption/Decryption Example" & @CRLF)
	write_to_log(StringFormat("Plain text message        = %s", $MESSAGE) & @CRLF)
	write_to_log(StringFormat("Encrypt Key               = %s", $KEY) & @CRLF)
	write_to_log(StringFormat("Encrypted Message (Data)  = %s", $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("Encrypted Message (Nonce) = %s", $NONCE) & @CRLF)
	write_to_log(StringFormat("Encrypted Message (Tag)   = %s", $xAuthTag) & @CRLF)
	write_to_log(StringFormat("Decrypted Message         = %s", $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func aes_encrypt_decrypt_file_example()

	Const $INPUT_FILE     = "ExampleLoremIpsumDataFile.docx", _
	      $ENCRYPTED_FILE = "ExampleLoremIpsumDataFile - Encrypted.docx", _
	      $DECRYPTED_FILE = "ExampleLoremIpsumDataFile - Decrypted.docx", _
	      $SECRET         = "Secret Password!", _
		  $SECRET_SALT    = "Secret Salt"

	Local $sAlgId = ""
	Local $vEncryptKey = ""


	;Use AES encryption
	$sAlgId = $CNG_BCRYPT_AES_ALGORITHM

	;Create a hashed encryption key from text-based password/salt
	$vEncryptKey = _CryptoNG_PBKDF2($SECRET, $SECRET_SALT, 100, $CNG_KEY_BIT_LENGTH_AES_192, $CNG_BCRYPT_SHA1_ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Encrypt file
	_CryptoNG_EncryptFile($sAlgId, $INPUT_FILE, $ENCRYPTED_FILE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	_CryptoNG_DecryptFile($sAlgId, $ENCRYPTED_FILE, $DECRYPTED_FILE, $vEncryptKey)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG AES Encryption/Decryption Example" & @CRLF)
	write_to_log(StringFormat('%s Input file     = %s', $sAlgId, $INPUT_FILE) & @CRLF)
	write_to_log(StringFormat('%s Encrypted file = %s', $sAlgId, $ENCRYPTED_FILE) & @CRLF)
	write_to_log(StringFormat('%s Decrypted file = %s', $sAlgId, $DECRYPTED_FILE) & @CRLF)

EndFunc

;==========================================================================

Func hash_data_example()

	Const $MESSAGE     = "This is a super-secret message."
	Const $HMAC_SECRET = "Secret Password"

	Local $xBinaryHash = Binary("")


	;SHA256 Example
	$xBinaryHash = _CryptoNG_HashData($CNG_BCRYPT_SHA256_ALGORITHM, $MESSAGE)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log(StringFormat("%s Hash Data     = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Hash          = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $xBinaryHash) & @CRLF)


	;HMACSHA256 Example
	$xBinaryHash = _CryptoNG_HashData($CNG_BCRYPT_SHA256_ALGORITHM, $MESSAGE, True, $HMAC_SECRET)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log(StringFormat("HMAC%s Hash Data = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("HMAC%s Secret    = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $HMAC_SECRET) & @CRLF)
	write_to_log(StringFormat("HMAC%s Hash      = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $xBinaryHash) & @CRLF)


	;MD5 Example
	$xBinaryHash = _CryptoNG_HashData($CNG_BCRYPT_MD5_ALGORITHM, $MESSAGE)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG Hash Data Example" & @CRLF)
	write_to_log(StringFormat("%s Hash Data        = %s", $CNG_BCRYPT_MD5_ALGORITHM, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Hash             = %s", $CNG_BCRYPT_MD5_ALGORITHM, $xBinaryHash) & @CRLF)

EndFunc

;==========================================================================

Func hash_file_example()

	Const $FILE_PATH   = @ScriptFullPath
	Const $HMAC_SECRET = "Secret Password"

	Local $xBinaryHash = Binary("")


	;SHA256 Example
	$xBinaryHash = _CryptoNG_HashFile($CNG_BCRYPT_SHA256_ALGORITHM, $FILE_PATH)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log(StringFormat("CryptoNG UDF Version %s", _CryptoNG_Version()) & @CRLF)
	write_to_log(StringFormat("%s Hash File = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $FILE_PATH) & @CRLF)
	write_to_log(StringFormat("%s Hash      = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $xBinaryHash) & @CRLF)


	;HMACSHA256 Example
	$xBinaryHash = _CryptoNG_HashFile($CNG_BCRYPT_SHA256_ALGORITHM, $FILE_PATH, True, $HMAC_SECRET)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG Hash File Example" & @CRLF)
	write_to_log(StringFormat("HMAC%s Hash File = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $FILE_PATH) & @CRLF)
	write_to_log(StringFormat("HMAC%s Secret    = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $HMAC_SECRET) & @CRLF)
	write_to_log(StringFormat("HMAC%s Hash      = %s", $CNG_BCRYPT_SHA256_ALGORITHM, $xBinaryHash) & @CRLF)

EndFunc

;==========================================================================

Func pbkdf2_example()

	Const $PASSWORD       = "This is a super-secret password!"
	Const $SALT           = "Secret Salt"
	Const $ITERATIONS     = 1500
	Const $KEY_BIT_LENGTH = 192
	Const $ALGORITHM      = $CNG_BCRYPT_SHA1_ALGORITHM

	Local $xPasswordHash = Binary("")


	;PBKDF2 Example
	$xPasswordHash = _CryptoNG_PBKDF2($PASSWORD, $SALT, $ITERATIONS, $KEY_BIT_LENGTH, $ALGORITHM)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG Password-Based Key Derivation Function 2 (PBKDF2) Example" & @CRLF)
	write_to_log(StringFormat("PBKDF2_%s Password      = %s", $ALGORITHM, $PASSWORD) & @CRLF)
	write_to_log(StringFormat("PBKDF2_%s Salt          = %s", $ALGORITHM, $SALT) & @CRLF)
	write_to_log(StringFormat("PBKDF2_%s Iterations    = %s", $ALGORITHM, $ITERATIONS) & @CRLF)
	write_to_log(StringFormat("PBKDF2_%s Key Length    = %i bits / %i bytes", $ALGORITHM, $KEY_BIT_LENGTH, $KEY_BIT_LENGTH / 8) & @CRLF)
	write_to_log(StringFormat("PBKDF2_%s Password Hash = %s", $ALGORITHM, $xPasswordHash) & @CRLF)

EndFunc

;==========================================================================

Func generate_random_example()

	Local $xRandomBytes = Binary("")

	;Generate random bytes example
	$xRandomBytes = _CryptoNG_GenerateRandom($CNG_BCRYPT_RNG_ALGORITHM, 16)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG Random Byte Generator Example" & @CRLF)
	write_to_log(StringFormat("%i Random Bytes = %s", BinaryLen($xRandomBytes), $xRandomBytes) & @CRLF)

EndFunc

;==========================================================================

Func create_ecdsa_key_pair_example()
	Const $PUBLIC_KEY_FILE  = "example_ecdsa_publickey.blob", _
		  $PRIVATE_KEY_FILE = "example_ecdsa_privatekey.blob"

	;Create ECDSA Public/Private Key Pair
	_CryptoNG_ECDSA_CreateKeyPair($CNG_KEY_BIT_LENGTH_ECDSA_256, $PUBLIC_KEY_FILE, $PRIVATE_KEY_FILE)
	If @error Then
		write_to_log(@CRLF)
		write_to_log("ERROR: _CryptoNG_ECDSA_CreateKeyPair" & @CRLF)
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG ECDSA Public/Private Key Creation Example" & @CRLF)
	write_to_log("Public/Private ECDSA key pair successfully created." & @CRLF)
EndFunc

;==========================================================================

Func ecdsa_sign_and_verify_example()

	Const $MESSAGE          = "This is a test message."
	Const $HASH_ALGORITHM   = $CNG_BCRYPT_SHA1_ALGORITHM
	Const $PUBLIC_KEY_FILE  = "example_ecdsa_publickey.blob"
	Const $PRIVATE_KEY_FILE = "example_ecdsa_privatekey.blob"

	Local $xSignature  = Binary(""), _
	      $xHash       = Binary("")

	Local $bSignatureIsValid = False


;~ 	;Create ECDSA Public/Private Key Pair
;~ 	_CryptoNG_ECDSA_CreateKeyPair($CNG_KEY_BIT_LENGTH_ECDSA_256, $PUBLIC_KEY_FILE, $PRIVATE_KEY_FILE)
;~ 	If @error Then
;~ 		write_to_log(@CRLF)
;~ 		write_to_log("ERROR: _CryptoNG_ECDSA_CreateKeyPair" & @CRLF)
;~ 		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
;~ 		Return False
;~ 	EndIf

	;Generate the hash of the message
	$xHash = _CryptoNG_HashData($HASH_ALGORITHM, $MESSAGE)
	If @error Then
		write_to_log("ERROR: " & "_CryptoNG_HashData"         & @CRLF)
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return
	EndIf

	;Generate a signature from the hash (a.k.a. Signing the hash)
	$xSignature = _CryptoNG_ECDSA_SignHash($HASH_ALGORITHM, $xHash, $PRIVATE_KEY_FILE)
	If @error Then
		write_to_log("ERROR: " & "_CryptoNG_ECDSA_SignHash"     & @CRLF)
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return
	EndIf

	;Verify the signature
	$bSignatureIsValid = _CryptoNG_ECDSA_VerifySignature($HASH_ALGORITHM, $xHash, $xSignature, $PUBLIC_KEY_FILE)
	If @error Then
		write_to_log("ERROR: " & "_CryptoNG_RSA_VerifySignature" & @CRLF)
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage()     & @CRLF)
		Return
	EndIf

	;Display results
	write_to_log(@CRLF & "Example: ECDSA Sign & Verify" & @CRLF)
	write_to_log(StringFormat("Message                         = %s", $MESSAGE)           & @CRLF)
	write_to_log(StringFormat("Hash Algorithm                  = %s", $HASH_ALGORITHM)    & @CRLF)
	write_to_log(StringFormat("Hash (Binary)                   = %s", $xHash)             & @CRLF)
	write_to_log(StringFormat("ECDSA Public Key Blob  (Base64) = %s", _CryptoNG_CryptBinaryToString(FileRead($PUBLIC_KEY_FILE) , $CNG_CRYPT_STRING_BASE64)) & @CRLF)
	write_to_log(StringFormat("ECDSA Private Key Blob (Base64) = %s", _CryptoNG_CryptBinaryToString(FileRead($PRIVATE_KEY_FILE), $CNG_CRYPT_STRING_BASE64)) & @CRLF)
	write_to_log(StringFormat("Signature (Binary)              = %s", $xSignature)        & @CRLF)
	write_to_log(StringFormat("Signature (Base64)              = %s", _CryptoNG_CryptBinaryToString($xSignature, $CNG_CRYPT_STRING_BASE64)) & @CRLF)
	write_to_log(StringFormat("Signature Valid?                = %s", $bSignatureIsValid) & @CRLF)

EndFunc

;==========================================================================

Func create_rsa_key_pair_example()
#cs ====================================================================================================

	The keys that are created are "legacy" Microsoft CryptoAPI-compatible key blob files.  The key blob
	files can be converted to PEM or DER encoded key files using tools like OpenSSL.  The commands below
	are examples of how to convert the key files and how to view the key file information.

	See the _CryptoNG_CreateRSAKeyPair function header for more information.


	Commands to convert the key blobs to PEM or DER format using OpenSSL v1.1+:
	=========
	openssl.exe rsa -pubin -inform "MS PUBLICKEYBLOB" -in publickey.blob -outform PEM -out publickey.pem
	openssl.exe rsa -inform "MS PRIVATEKEYBLOB" -in privatekey.blob -outform PEM -out privatekey.pem

	openssl.exe rsa -pubin -inform "MS PUBLICKEYBLOB" -in publickey.blob -outform DER -out publickey.der
	openssl.exe rsa -inform "MS PRIVATEKEYBLOB" -in privatekey.blob -outform DER -out privatekey.der


	Commands to show key blob info using OpenSSL v1.1+:
	=========
	openssl.exe rsa -inform "MS PUBLICKEYBLOB" -pubin -in publickey.blob -noout -text
	openssl.exe rsa -inform "MS PRIVATEKEYBLOB" -in privatekey.blob -noout -text


	If you have OpenSSL installed and you want to convert the key blobs to PEM-encoded files, then
	uncomment (CTRL+Q) that section of the example.

#ce ====================================================================================================

	;Create RSA Public/Private Key Pair
	_CryptoNG_RSA_CreateKeyPairEx(512, "example_rsa_publickey.blob", "example_rsa_privatekey.blob", "example_legacy_publickey.blob", "example_legacy_privatekey.blob")
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG Public/Private Key Creation Example" & @CRLF)
	write_to_log("Public/Private RSA key pair successfully created." & @CRLF)

EndFunc

;==========================================================================

Func rsa_public_private_key_encrypt_decrypt_data_example()
	Const $ALG_ID           = $CNG_BCRYPT_RSA_ALGORITHM, _
	      $MESSAGE          = "This is a super-secret message.", _
		  $PUBLIC_KEY_FILE  = "example_rsa_publickey.blob", _
		  $PRIVATE_KEY_FILE = "example_rsa_privatekey.blob"

	Local $sDecryptedMessage = ""

	Local $xEncryptedMessage = Binary("")


;~ 	;Create RSA Public/Private Key Pair (Uncomment section below to create a new key pair)
;~ 	_CryptoNG_RSA_CreateKeyPair(2048, $PUBLIC_KEY_FILE, $PRIVATE_KEY_FILE, $CNG_BCRYPT_RSA_KEY_EXPORT_RSA)
;~ 	If @error Then
;~ 		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
;~ 		Return False
;~ 	EndIf

	;Encrypt plain text message
	$xEncryptedMessage = _CryptoNG_RSA_EncryptData($MESSAGE, $PUBLIC_KEY_FILE)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Decrypt encrypted message
	$sDecryptedMessage = _CryptoNG_RSA_DecryptData($xEncryptedMessage, $PRIVATE_KEY_FILE)
	If @error Then
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return False
	EndIf

	;Display results
	write_to_log(@CRLF)
	write_to_log("CryptoNG Asymmetric Public/Private Key Encrypt/Decrypt Example" & @CRLF)
	write_to_log(StringFormat("%s Public key file       = %s", $ALG_ID, $PUBLIC_KEY_FILE) & @CRLF)
	write_to_log(StringFormat("%s Private key file      = %s", $ALG_ID, $PRIVATE_KEY_FILE) & @CRLF)
	write_to_log(StringFormat("%s Plain text message    = %s", $ALG_ID, $MESSAGE) & @CRLF)
	write_to_log(StringFormat("%s Encrypted Message     = %s", $ALG_ID, $xEncryptedMessage) & @CRLF)
	write_to_log(StringFormat("%s Decrypted Message     = %s", $ALG_ID, $sDecryptedMessage) & @CRLF)

EndFunc

;==========================================================================

Func rsa_sign_and_verify_example()

	Const $MESSAGE          = "This is a test message."
	Const $HASH_ALGORITHM   = $CNG_BCRYPT_SHA1_ALGORITHM
	Const $PUBLIC_KEY_FILE  = "example_rsa_publickey.blob"
	Const $PRIVATE_KEY_FILE = "example_rsa_privatekey.blob"

	Local $xSignature  = Binary(""), _
	      $xHash       = Binary("")

	Local $bSignatureIsValid = False


;~ 	;Create and export a RSA Public/Private Key Pair
;~ 	_CryptoNG_RSA_CreateKeyPair(512, $PUBLIC_KEY_FILE, $PRIVATE_KEY_FILE)
;~ 	If @error Then
;~ 		write_to_log("ERROR: " & "_CryptoNG_RSA_CreateKeyPair" & @CRLF)
;~ 		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage()  & @CRLF)
;~ 		Return
;~ 	EndIf

	;Generate the hash of the message
	$xHash = _CryptoNG_HashData($HASH_ALGORITHM, $MESSAGE)
	If @error Then
		write_to_log("ERROR: " & "_CryptoNG_HashData"         & @CRLF)
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return
	EndIf

	;Generate a signature from the hash (a.k.a. Signing the hash)
	$xSignature = _CryptoNG_RSA_SignHash($HASH_ALGORITHM, $xHash, $PRIVATE_KEY_FILE)
	If @error Then
		write_to_log("ERROR: " & "_CryptoNG_RSA_SignHash"     & @CRLF)
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
		Return
	EndIf

	;Verify the signature
	$bSignatureIsValid = _CryptoNG_RSA_VerifySignature($HASH_ALGORITHM, $xHash, $xSignature, $PUBLIC_KEY_FILE)
	If @error Then
		write_to_log("ERROR: " & "_CryptoNG_RSA_VerifySignature" & @CRLF)
		write_to_log("ERROR: " & _CryptoNG_LastErrorMessage()    & @CRLF)
		Return
	EndIf

	;Display results
	write_to_log(@CRLF & "Example: RSA Sign & Verify" & @CRLF)
	write_to_log(StringFormat("Message                       = %s", $MESSAGE)           & @CRLF)
	write_to_log(StringFormat("Hash Algorithm                = %s", $HASH_ALGORITHM)    & @CRLF)
	write_to_log(StringFormat("Hash (Binary)                 = %s", $xHash)             & @CRLF)
	write_to_log(StringFormat("RSA Public Key Blob  (Base64) = %s", _CryptoNG_CryptBinaryToString(FileRead($PUBLIC_KEY_FILE) , $CNG_CRYPT_STRING_BASE64)) & @CRLF)
	write_to_log(StringFormat("RSA Private Key Blob (Base64) = %s", _CryptoNG_CryptBinaryToString(FileRead($PRIVATE_KEY_FILE), $CNG_CRYPT_STRING_BASE64)) & @CRLF)
	write_to_log(StringFormat("Signature (Binary)            = %s", $xSignature)        & @CRLF)
	write_to_log(StringFormat("Signature (Base64)            = %s", _CryptoNG_CryptBinaryToString($xSignature, $CNG_CRYPT_STRING_BASE64)) & @CRLF)
	write_to_log(StringFormat("Signature Valid?              = %s", $bSignatureIsValid) & @CRLF)

EndFunc

;==========================================================================

Func convert_binary_to_string_example()

	Const $PUBLIC_KEY_FILE = "example_rsa_publickey.blob"

	Local $xBinaryData = Binary("")
	Local $hFile = -1
	Local $sFormattedString = ""

	write_to_log(@CRLF)
	write_to_log("Convert Binary To String Examples" & @CRLF)


	;Make sure file exists
	If Not FileExists($PUBLIC_KEY_FILE) Then Return write_to_log("ERROR: " & $PUBLIC_KEY_FILE & " does not exist" & @CRLF)

	;Open, read, & close example public key blob
	$hFile = FileOpen($PUBLIC_KEY_FILE, $FO_BINARY)
	If $hFile = -1 Then Return write_to_log("ERROR: " & $PUBLIC_KEY_FILE & " does not exist" & @CRLF)

	$xBinaryData = FileRead($hFile)
	If @error Then Return write_to_log("ERROR: Unable to read " & $PUBLIC_KEY_FILE & @CRLF)

	FileClose($hFile)


	;Display string to binary example - HEXRAW
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_HEXRAW)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
	write_to_log(@CRLF & "$CNG_CRYPT_STRING_HEXRAW" & @CRLF)
	write_to_log($sFormattedString & @CRLF)


	;Display string to binary example - HEX
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_HEX)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
	write_to_log(@CRLF & @CRLF & "$CNG_CRYPT_STRING_HEX" & @CRLF)
	write_to_log($sFormattedString & @CRLF)


	;Display string to binary example - HEXASCII
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_HEXASCII)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
	write_to_log(@CRLF & @CRLF & "$CNG_CRYPT_STRING_HEXASCII" & @CRLF)
	write_to_log($sFormattedString & @CRLF)


	;Display string to binary example - HEXASCIIADDR
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_HEXASCIIADDR)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
	write_to_log(@CRLF & @CRLF & "$CNG_CRYPT_STRING_HEXASCIIADDR" & @CRLF)
	write_to_log($sFormattedString & @CRLF)


	;Display string to binary example - BASE64
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_BASE64)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
	write_to_log(@CRLF & @CRLF & "$CNG_CRYPT_STRING_BASE64" & @CRLF)
	write_to_log($sFormattedString & @CRLF)


	;Display string to binary example - BASE64 + NOCRLF
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_BASE64 + $CNG_CRYPT_STRING_NOCRLF)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
	write_to_log(@CRLF & @CRLF & "$CNG_CRYPT_STRING_BASE64 + $CNG_CRYPT_STRING_NOCRLF" & @CRLF)
	write_to_log($sFormattedString & @CRLF)


	;Display string to binary example - BASE64HEADER (BEGIN/END CERTIFICATE Header/Footer)
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_BASE64HEADER)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
	$sFormattedString = StringReplace($sFormattedString, "CERTIFICATE", "PUBLIC KEY")
	write_to_log(@CRLF & @CRLF & "$CNG_CRYPT_STRING_BASE64HEADER" & @CRLF)
	write_to_log($sFormattedString & @CRLF)

EndFunc

;==========================================================================

Func convert_string_to_binary_example()

	Const $PUBLIC_KEY_FILE = "example_rsa_publickey.blob"

	Local $xBinaryData = Binary(""), $xConvertedData = Binary("")
	Local $hFile = -1
	Local $sFormattedString = ""

	write_to_log(@CRLF)
	write_to_log("Convert String To Binary Examples" & @CRLF)

	;Make sure file exists
	If Not FileExists($PUBLIC_KEY_FILE) Then Return write_to_log("ERROR: " & $PUBLIC_KEY_FILE & " does not exist" & @CRLF)

	;Open, read, & close example public key blob
	$hFile = FileOpen($PUBLIC_KEY_FILE, $FO_BINARY)
	If $hFile = -1 Then Return write_to_log("ERROR: " & $PUBLIC_KEY_FILE & " does not exist" & @CRLF)

	$xBinaryData = FileRead($hFile)
	If @error Then Return write_to_log("ERROR: Unable to read " & $PUBLIC_KEY_FILE & @CRLF)

	FileClose($hFile)


	;Convert binary to formatted string (raw hex) and formatted string back to binary
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_HEXRAW)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)

	$xConvertedData = _CryptoNG_CryptStringToBinary($sFormattedString, $CNG_CRYPT_STRING_HEX_ANY)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)

	write_to_log(@CRLF & "$CNG_CRYPT_STRING_HEXRAW to Binary" & @CRLF)
	write_to_log("String:   " & StringUpper($sFormattedString) & @CRLF)
	write_to_log("Binary: "   & $xConvertedData & @CRLF)


	;Convert binary to formatted string (base64) and formatted string back to binary
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_BASE64)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)

	$xConvertedData = _CryptoNG_CryptStringToBinary($sFormattedString, $CNG_CRYPT_STRING_BASE64_ANY)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)

	write_to_log(@CRLF & @CRLF & "$CNG_CRYPT_STRING_BASE64 to Binary" & @CRLF)
	write_to_log($sFormattedString & @CRLF & @CRLF)
	write_to_log("Binary: " & $xConvertedData & @CRLF)


	;Convert binary to formatted string (base64 with header) and formatted string back to binary
	$sFormattedString = _CryptoNG_CryptBinaryToString($xBinaryData, $CNG_CRYPT_STRING_BASE64HEADER)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)
	$sFormattedString = StringReplace($sFormattedString, "CERTIFICATE", "PUBLIC KEY")

	$xConvertedData = _CryptoNG_CryptStringToBinary($sFormattedString, $CNG_CRYPT_STRING_BASE64_ANY)
	If @error Then Return write_to_log("ERROR: " & _CryptoNG_LastErrorMessage() & @CRLF)

	write_to_log(@CRLF & @CRLF & "$CNG_CRYPT_STRING_BASE64HEADER to Binary" & @CRLF)
	write_to_log($sFormattedString & @CRLF & @CRLF)
	write_to_log("Binary: " & $xConvertedData & @CRLF)

EndFunc

;==========================================================================
;
;==========================================================================
Func write_to_log($sMsg = "")
	GUICtrlSetData($ctlEditBox, $sMsg, 1)
EndFunc
