#cs
	This set of examples uses a json file that contains all of the
	NFL game score information for the first week of the 2018 regular
	season.  As you can see, the json file has a lot of information.
	These examples gather information from the json.

	I created this set of examples to highlight the difference between
	json parsers and json processors.  Basically, the differece between
	a json parser and a json processor is the difference between getting
	data from a json and getting information.  A processor can parse but
	a parse cannot process.  Parsers merely retrieve individual values,
	whereas processors can retrieve information.

	So for those that don't understand the value of a json processor
	over a json parser and think that it's too slow, as you can see by
	running the examples, a single invocation with a well-written filter
	can replace numerous lines of code and, depending on the efficiency
	of the script lines that it replaces, gets the results quicker.

	Keep in mind that what ordinarily requires a parser to loop through
	json to retrieve values and then the script code to process those
	values in order to get a result, only requires a single invocation
	of a processor.  So, again, a processor can be faster, more efficient,
	and easier to use than a parser, when you are looking for json
	information as opposed to just json values.

#ce

#include "jq.au3"
#include <GUIConstantsEx.au3>
#include <WindowsConstants.au3>
#include <EditConstants.au3>
#include <ColorConstants.au3>
#include <FontConstants.au3>

;=========================================================
; Global GUI
;=========================================================
Const $CONSOLE_BACK_COLOR = $COLOR_CREAM
Const $CONSOLE_FORE_COLOR = $COLOR_BLACK
Const $CONSOLE_WIDTH      = 800
Const $CONSOLE_HEIGHT     = 600

Global $frmConsole = GUICreate("Console Log (Press ESC to Close)", $CONSOLE_WIDTH, $CONSOLE_HEIGHT, 302, 218, BitOR($GUI_SS_DEFAULT_GUI,$WS_MAXIMIZEBOX,$WS_SIZEBOX,$WS_THICKFRAME,$WS_TABSTOP))
GUISetBkColor($CONSOLE_BACK_COLOR, $frmConsole)
Global $ctlEditBox = GUICtrlCreateEdit("", 0, 0, $CONSOLE_WIDTH, $CONSOLE_HEIGHT, BitOR($ES_AUTOVSCROLL,$ES_AUTOHSCROLL,$ES_WANTRETURN,$ES_READONLY,$WS_VSCROLL,$WS_HSCROLL))
GUICtrlSetResizing($ctlEditBox, $GUI_DOCKLEFT+$GUI_DOCKRIGHT+$GUI_DOCKTOP+$GUI_DOCKBOTTOM)
GUICtrlSetFont($ctlEditBox, 11, $FW_MEDIUM, 0, "Consolas")
GUICtrlSetColor($ctlEditBox, $CONSOLE_FORE_COLOR)
GUICtrlSetBkColor($ctlEditBox, $CONSOLE_BACK_COLOR)
GUISetState(@SW_SHOW, $frmConsole)

;=========================================================
; Global Constants
;=========================================================
Const $JSON_FILE = "nfl_scores_2018_week_01.json"

;=========================================================
; Global Strings
;=========================================================
Global $sExePath = ""

;=========================================================
; Main Processing Block
;=========================================================

$sExePath = _jqInit()
If @error Then Exit MsgBox($MB_ICONERROR + $MB_TOPMOST, "ERROR", StringFormat("ERROR: Unable to initialize jq - @error = %s", @error))

show_path_and_version()
pretty_print_json_example()
json_file_example("Count of how many games being played this week", $JSON_FILE, '. | length')
json_file_example("List matchups for the week"                    , $JSON_FILE, '.[] | .home.abbr + " vs " + .away.abbr')
json_file_example("List of games not started"                     , $JSON_FILE, '.[] | select(.qtr == null) | .home.abbr + " vs " + .away.abbr')
json_file_example("TSV list of all games (with date and score)"   , $JSON_FILE, 'path(.[]) as $path | getpath($path) | [$path[0][0:4] + "-" + $path[0][4:6] + "-" + $path[0][6:8], .home.abbr, .home.score.T, .away.abbr, .away.score.T ] | @tsv')
json_file_example("TSV list of games that are final (with score)" , $JSON_FILE, '.[] | select(.qtr | ascii_downcase? | startswith("final")) | [.home.abbr, .home.score.T, .away.abbr, .away.score.T] | @tsv')
json_file_example("List of winning teams (ties are excluded)"     , $JSON_FILE, '.[] | select(.qtr | ascii_downcase? | startswith("final")) | if .home.score.T > .away.score.T then .home.abbr elif .home.score.T < .away.score.T then .away.abbr else empty end')
json_file_example("List of tie games"                             , $JSON_FILE, '.[] | select(.qtr | ascii_downcase? | startswith("final")) | if .home.score.T == .away.score.T then .home.abbr + " vs " + .away.abbr else empty end')
json_file_example("MIN final score this week"                     , $JSON_FILE, '[ .[] | select(.qtr | ascii_downcase? | startswith("final")) | .home.score.T, .away.score.T ] | "MIN Score: " + (min|tostring)')
json_file_example("MAX final score this week"                     , $JSON_FILE, '[ .[] | select(.qtr | ascii_downcase? | startswith("final")) | .home.score.T, .away.score.T ] | "MAX Score: " + (max|tostring)')
json_file_example("AVG final score this week"                     , $JSON_FILE, '[ .[] | select(.qtr | ascii_downcase? | startswith("final")) | .home.score.T, .away.score.T ] | "AVG: " + ((add/length)|tostring)')

;Wait for GUI close request
While 1
	If GUIGetMsg() = $GUI_EVENT_CLOSE Then ExitLoop
WEnd

;=========================================================
; Functions
;=========================================================

Func show_path_and_version()
	write_log_line(@CRLF & "================================" & @CRLF)
	write_log_line(StringFormat("jq Path        = %s", $sExePath))
	write_log_line(StringFormat("jq UDF Version = %s", _jqUdfVersion()))
	write_log_line(StringFormat("jq Version     = %s", _jqVersion())   )
EndFunc

Func pretty_print_json_example()
	Local $sCmdOutput = _jqPrettyPrintJsonFile($JSON_FILE)  ;From string example

	write_log_line(@CRLF & "================================" & @CRLF)
	write_log_line("Desc   : Pretty-print JSON")
	write_log_line("Input  : " & $JSON_FILE)
	write_log_line(@CRLF & $sCmdOutput)
EndFunc

Func compact_print_json_example()
	Local $sCmdOutput = _jqCompactPrintJsonFile($JSON_FILE)

	write_log_line(@CRLF & "================================" & @CRLF)
	write_log_line("Desc   : Compact-print JSON")
	write_log_line("Input  : " & $JSON_FILE)
	write_log_line(@CRLF & $sCmdOutput)
EndFunc

Func json_file_example($sDesc, $sJsonFile, $sFilter, $sOptions = Default)
	Local $hTimer     = TimerInit()
	Local $sCmdOutput = _jqExecFile($JSON_FILE, $sFilter, $sOptions)
	Local $iTimerDiff = TimerDiff($hTimer)

	write_log_line(@CRLF & "================================" & @CRLF)
	write_log_line("Desc    : " & $sDesc)
	write_log_line("Input   : " & $sJsonFile)
	write_log_line("Filter  : " & $sFilter)
	write_log_line("Duration: " & StringFormat("%.3f seconds", $iTimerDiff / 1000))
	write_log_line(@CRLF & $sCmdOutput)
EndFunc

Func write_log_line($sMsg = "")
	GUICtrlSetData($ctlEditBox, $sMsg & @CRLF, 1)
EndFunc
