;============================================================================================
; Title ..........: BitMaskSudokuSolver
; AutoIt Version..: 3.3.14+
; Description ....: showcase for E4A bitmask handling
; Author..........: A.R.T. Jonkers (RTFC)
; Release.........: 1.1
; Latest revision.: 11 June 2020
; License.........: free for personal use; free distribution allowed provided
;							the author is credited; all other rights reserved.
; Tested on.......: W10Pro/64
; Dependencies....: Eigen4AutoIt (E4A) v4.7+
;============================================================================================
; Summary (Wotsit?)
;
; Eigen4AutoIt (E4A) provides extensive tools for producing and manipulating
; bitmasks, including conditional expressions and logical operands.
; This example script applies these to solve sudoku puzzles, which are
; mathematically simple, but can be computationally hard.
;
; It uses simple solving strategies and/or optimised brute-force,
; based upon a few tables and basic logic, applied to large bitmask
; matrices that track the remaining valid permutations for each 3x3 block
; (the basis of all approaches shown here).
;
; No guarantees, no warranties, no liability for damages, no refunds.
;	Use at your own risk.
;
;============================================================================================
;	Remarks
;
; * Find call _LoadSudoku(X) and parse 1-10 for testing one of the provided
;	sudoku puzzles; parse anything else to load an external .sdk file
;	(a sample file is provided: 9 lines of 9 digits (1-9) and dots for blanks).
;
; * Set flag $callSolver1st to False to force Brute Force-only (faster than _Solver).
;
; * Set flag $exitUpon1stHit to False to force Brute Force to analyse the
;	complete permutation space and collect all solutions if there are
;	more than one. If $exitUpon1stHit=True (the default), the programme
;	ends as soon as the first valid solution is found.
;
; * call _Eigen_Show_MatrixDisplay to show each solution in a separate window;
;	call _Eigen_Hide_MatrixDisplay to just write solution info to console.
;
; * Three tables (extension *.mat) are provided, but will be regenerated
;	if not found locally (slow, but done only once).
;
; * The number of updates sent to console can be controlled with flags
;	$verbose and $testing; writing less info = faster processing.
;
; * This script shows how E4A can be used to solve computationally hard
;	problems without requiring any advanced maths. Each permutations-related
;	operation evaluates over 362 thousand cases per single E4A call.
;	(NB: AutoIt E4A function calls parse matrix IDs only (integers), not data.)
;
; * How it works:
;	1. create work environment (note how multiple matrices (refs) are stored/accessed in arrays)
;	2. read in sudoku data and store the clues.
;	3. punch-in all clues, reducing the number of valid permutations per 3x3 block and all neighbouring blocks
;	4. [optional] call _Solver, which applies a few simple solving strategies; this will fail on extreme sudoku's.
;	5. call _OptimiseMax: Find the block with most remaining permutations and swap it to the Top-Left corner block (TL).
;		(NB: block rows and block cols can be swapped at will without affecting the solution.)
;	6. Find the next block with most remaining permutations and swap it to the Bottom-Right corner block (BR).
;	7. Brute-Force: cycle through all valid permutations in nested loops for
;		the six off-diagonal blocks, with early-outs for any block that has zero valid permuts left.
;		In the innermost loop, permutations of the three diagonal blocks are
;		thereby reduced to either 1 or 0. Evaluate and report if sudoku is valid.
;
; * Note in particular how the following functions are used:
;	+ _Eigen_ConditMask, to produce bitmasks;
;	+ _Eigen_CwiseLogicalOp with operator "xcl", as positive exclusion filter (1 = out, 0 = keep);
;	+ _Eigen_CwiseLogicalOp with operator "and", as negative exclusion filter (1 = keep, 0 = out);
;	+ _Eigen_CwiseLogicalOp with operator "or", to combine partially overlapping subsets;
;	+ _Eigen_ReDim_ExistingMatrix, to switch data representation between vector- and block-shaped;
;	+ _FillCandidValues, to multiply a LinSpaced_Colwise matrix ($candidvalues) with a bitmask ($candidates) to list remaining candidate values per cell (0=invalid);
;	+ _Eigen_Swap_Arow_Arow/_Eigen_Swap_Acol_Acol, to quickly re-arrange block references (the blocks themselves are untouched);
;	+ _FillMasksOfValidNeighbours, to create two subsets of 12,096 valid permuts, one horizontally and one vertically;
;	+ _MaskValidPermutations, to apply up to four valid-neighbour bitmasks to reduce a block permutation subset;
;	+ _Eigen_IsZero, as early-out for invalid block combinations;
;	+ _Eigen_GetSum, to determine how many valid permutations are left for a block;
;	+ _Eigen_GetCol_MaxVal, to retrieve the last-remaining valid permutation ID.
;
;============================================================================================
#NoTrayIcon
#include <WinAPIError.au3>
#include ".\Eigen4AutoIt.au3"	; EDIT THIS PATH, IF NEED BE	<<<<

; number of permutations in a 3x3 block containing digits 1-9 once each
Global Const $permuts=362880	; 9! (9 factorial = 9 x 8 x 7 x 6 x 5 x 4 x 3 x 2 x 1)

; global matrices, multi-matrix arrays, and related vars
Global $sudoku[81]	; 1D array
Global $matSudoku,$solution,$solutionRow,$solutionCol,$solutionBlock
Global $candidates,$reduced,$blockneighbours,$blockfreqs,$blockfreqsum
Global $retainRowVec,$cellRowVec,$valueRowVec,$digitColvec,$digitRowvec,$valuecolvec,$scratchRowVec,$blockRowVec
Global $blockPermuts,$digitInRows,$digitInCols,$matDigitBuffer
Global $blockIDadjRow1,$blockIDadjRow2,$blockIDadjCol1,$blockIDadjCol2,$matBlockCells
Global $listof7vals,$clueslist,$totalclues,$solutionsFound=0,$tstart=False,$tsolver=False

; block vars
Global $blockTL,$blockTC,$blockTR
Global $blockCL,$blockCC,$blockCR
Global $blockBL,$blockBC,$blockBR

; operational flags
Global $callSolver1st=True	; T: only call Brute-Force if/after Solver has failed; F: do Brute-Force only
Global $exitUpon1stHit=True	; T: early-out; F: exhaustive search for all solutions
Global $verbose=True		; T: write status updates to console; F: don't
Global $testing=False		; T: report punch-in/out and loop tracking (slow); F: don't

;_____________________________________________________________________________
; CREATE WORK ENVIRONMENT

; check E4A
If _Eigen_GetVersion<4.7 Then _FatalError("E4A version 4.7 or higher is required")
ConsoleWrite("Initialising BitMask Sudoku Solver..." & @CRLF & @CRLF)

; Beginning of Story...
_Eigen_StartUp("int")		; integers are most suitable for bitmask handling
_Eigen_SetLogicalBit0only()	; all CwiseLogicalOps are to act upon bit-0 only (for binary bitmask handling)

;_Eigen_Show_MatrixDisplay()	; ensure _MatrixDisplay calls are not ignored
_Eigen_Hide_MatrixDisplay()		; ignore all _MatrixDisplay calls (console output only)


; load 9x9 sudoku as square matrix (blank cells = 0)
; NB to load your own 2D 9x9 sudoku array, call _CreateSudokuMatrix(<your array>) instead
_LoadSudoku(6)	; test examples 1-10 included; parse anything else to load an .sdk-formatted text file instead (see sample.sdk)


; GENERATE ALL LOOKUP TABLES
; create block ID lookup table
$matBlockIDs=_Eigen_CreateMatrix_LinSpaced_Colwise(3,3,0,8)
$matBlocks=_Eigen_CreateMatrix_Zero(9,9)
$matBlockCells=_Eigen_CreateMatrix_Zero(9,9)

For $cc=0 To 2
	$startcol=$cc*3

	For $rc=0 To 2
		$startrow=$rc*3

		_Eigen_SetConstant_Block($matBlocks,$startrow,$startcol,3,3,$startcol+$rc)
		_Eigen_Copy_Ablock_ToBblock($matBlockIDs,$matBlockCells,0,0,3,3,$startrow,$startcol)
	Next
Next
_Eigen_ReDim_ExistingMatrix($matBlockCells,1,81)	; needed as rowvec only

; create block neighbour lookup table
Global $blockneighbours[9][4]

For $bc=0 To 8	; identify and store block's neighbours
	$blockneighbours[$bc][0]=Mod($bc+3,9)					; near row neighbour
	$blockneighbours[$bc][1]=Mod($bc+6,9)					; far row neighbour
	$blockneighbours[$bc][2]=(Int($bc/3)*3)+Mod($bc+1,3)	; near col neighbour
	$blockneighbours[$bc][3]=(Int($bc/3)*3)+Mod($bc+2,3)	; far col neighbour
Next

; create full permutation rowvector buffers (needed for creating/loading the file tables)
$matMrowvec=_Eigen_CreateMatrix(1,$permuts)
$matMcolvec=_Eigen_CreateMatrix(1,$permuts)	; "colvec" because the stored data relate to column indices (shape is rowvec)

; load the big tables from file (regenerated if not found in local dir)
_LoadBlockPermutations(); 9 rows (storing digits 1-9, base-1) x 362880 cols (block permutations)
_LoadDigitTables()		; 3x9 rows (row c.q. col 0/1/2 for digits 1-9, for each block permutation)

; create a mirror for the digits buffer
$matDigitBuffer = _Eigen_CloneMatrix($digitInRows,False)

; create block permutation mask buffers
; these arrays merely contain matrix IDs (integers, indices to $MatrixList)
; illustrating how to select from multiple matrices depending on input
Global $punchout[9]
Global $matpermut[9]
Global $validRowNeighbours[9]
Global $validColNeighbours[9]

For $dc=0 to 8	; only matrix IDs are stored here, not matrix contents
	$punchOut[$dc]=_Eigen_CreateMatrix_Ones(1,$permuts)
	$matpermut[$dc]=_Eigen_CreateMatrix_Ones(1,$permuts)

	$validRowNeighbours[$dc]=_Eigen_CreateMatrix_Zero(1,$permuts)
	$validColNeighbours[$dc]=_Eigen_CreateMatrix_Zero(1,$permuts)
Next

; We allocate all matrices of known dimensions BEFORE we do any work,
; so we won't waste time and effort if memory turns out to be insufficient.
; If more memory is needed, increase the EIGEN_ALLOCATE_RAM_GB value in
; the local Eigen4AutoIt.ini and rerun in x64-mode.

; create punchout mirror for block freqs
$matPunchOut=_Eigen_CreateMatrix(9,$permuts)	; content is based upon clues
$savePunchOut=_Eigen_CreateMatrix(9,$permuts)	; for restoring after trial & error

; create all RowVec buffers
$cellRowVec=_Eigen_CreateMatrix (1,81)
$cellRowVec2=_Eigen_CreateMatrix (1,81)
$cellRowVec3=_Eigen_CreateMatrix (1,81)

$valueRowVec=_Eigen_CreateMatrix (1,81)
$valueRowVec2=_Eigen_CreateMatrix (1,81)

$retainRowVec=_Eigen_CreateMatrix (1,81)
$scratchRowVec=_Eigen_CreateMatrix (1,81)

; create index lookup tables
$blockindexRowVec=_Eigen_CreateMatrix (1,81)
$rowindexRowVec=_Eigen_CreateMatrix (1,81)
$colindexRowVec=_Eigen_CreateMatrix (1,81)

$digitRowvec=_Eigen_CreateMatrix (3*9,1)	; row data stored as colvec
$digitColvec=_Eigen_CreateMatrix (3*9,1)

; create solution buffers
$solution=_Eigen_CreateMatrix_Zero (9,9)
$solutionRow=_Eigen_CreateMatrix_Zero (1,9)
$solutionCol=_Eigen_CreateMatrix_Zero (9,1)
$solutionBlock=_Eigen_CreateMatrix_Zero (3,3)

; create solver-related buffers
$candidates=_Eigen_CreateMatrix (9,81)	; digits x cells
$savecandidates=_Eigen_CreateMatrix (9,81)	; for restoring after trial & error
$candidValues=_Eigen_CreateMatrix (9,81)

$valuecolvec=_Eigen_CreateMatrix (9,1)
$valuecolvec2=_Eigen_CreateMatrix (9,1)
$valuecolvec3=_Eigen_CreateMatrix (9,1)
$scratchcolvec=_Eigen_CreateMatrix (9,1)

$blockfreqs=_Eigen_CreateMatrix (9,1)

$listof7vals=_Eigen_CreateVector(7)	; for Sort_Unique

; create candidate value mirrors
Global $candidate[9]
For $dc=0 To 8
	$candidate[$dc]=_Eigen_CreateMatrix_Ones (1,81)
Next

; create all buddy tables
$buddies=_Eigen_CreateMatrix_Ones (3,81)

Global $rowbuddymask[9],$colbuddymask[9],$blockbuddymask[9],$allbuddies[81]
For $dc=0 To 8
	$rowbuddymask[$dc]=_Eigen_CreateMatrix (1,81)
	$colbuddymask[$dc]=_Eigen_CreateMatrix (1,81)
	$blockbuddymask[$dc]=_Eigen_CreateMatrix (1,81)
Next

; fill index and buddy tables
For $cc=0 To 8
	For $rc=0 To 8
		$cll=$rc+($cc*9)
		$blockID=_Eigen_ReadMatrixValue($matBlocks,$rc,$cc)

		_Eigen_WriteMatrixValue($rowindexRowVec,0,$cll,$rc)
		_Eigen_WriteMatrixValue($colindexRowVec,0,$cll,$cc)
		_Eigen_WriteMatrixValue($blockindexRowVec,0,$cll,$blockID)

		_Eigen_WriteMatrixValue($buddies,0,$cll,$rc)
		_Eigen_WriteMatrixValue($buddies,1,$cll,$cc)
		_Eigen_WriteMatrixValue($buddies,2,$cll,$blockID)
	Next
Next

For $dc=0 To 8
	_Eigen_ConditMask_Row($buddies,0,"==",$dc,$rowbuddymask[$dc])
	_Eigen_ConditMask_Row($buddies,1,"==",$dc,$colbuddymask[$dc])
	_Eigen_ConditMask_Row($buddies,2,"==",$dc,$blockbuddymask[$dc])
Next

For $cll=0 To 80
	_Eigen_ConditMask_Row($buddies,0,"==",_Eigen_ReadMatrixValue($buddies,0,$cll),$retainRowVec)
	_Eigen_ConditMask_Row($buddies,1,"==",_Eigen_ReadMatrixValue($buddies,1,$cll),$scratchRowVec)

	_Eigen_CwiseLogicalOp_InPlace($retainRowVec,"or",$scratchRowVec)
	_Eigen_ConditMask_Row($buddies,2,"==",_Eigen_ReadMatrixValue($buddies,2,$cll),$scratchRowVec)

	_Eigen_CwiseLogicalOp_InPlace($retainRowVec,"or",$scratchRowVec)

	$allbuddies[$cll]=_Eigen_CloneMatrix($retainRowVec)
Next
_Eigen_ReleaseMatrix($buddies)
; Ready to roll!

;_____________________________________________________________________________
; PROCESS SUDOKU CLUES

; create list of all clue locations
$matClueMask=_Eigen_ConditMask($matSudoku,"!=",0)
$matClueList=_Eigen_FindAll($matClueMask,1)	; get offset,row,col
$totalclues=@extended
ConsoleWrite("Processing "&$totalclues&" Clues..." & @CRLF & @CRLF)
If $verbose And $totalclues<17 And $exitUpon1stHit=True Then ConsoleWrite("WARNING: at least 17 clues are required for a unique solution" & @CRLF)
If $verbose And $totalclues=17 Then ConsoleWrite("WARNING: 17 clues is barely enough; this may take an extremely long time." & @CRLF)

$matClueValues=_Eigen_CreateVector_FromAcell_Mask($matSudoku,$matClueMask,False)
$cluesList=_Eigen_CreateMatrix_FromABcols($matClueList,$matClueValues)
_Eigen_ReleaseMatrix($matClueList)
_Eigen_ReleaseMatrix($matClueValues)

; punch in/out each clue in the permutation mask of each relevant block
For $kc=0 To $totalclues-1
	$rw=_Eigen_ReadMatrixValue($cluesList,$kc,1)	; get clue location and value
	$cl=_Eigen_ReadMatrixValue($cluesList,$kc,2)
	$cll=$rw+($cl*9)
	$valueBase1=_Eigen_ReadMatrixValue($cluesList,$kc,3)

	_PunchIn($cll,$rw,$cl,$valuebase1-1)	; stored base-1, so convert
	_Eigen_WriteMatrixValue($solution,$rw,$cl,$valueBase1)
Next

; check that all clues are consistent with each other
For $dc=1 To 9
	_Eigen_ConditCount_Rowwise($solution,"==",$dc,$solutionCol)
	If _Eigen_ConditAny($solutionCol,">",1) Then _FatalError("invalid clues = no solution")

	_Eigen_ConditCount_Colwise($solution,"==",$dc,$solutionRow)
	If _Eigen_ConditAny($solutionRow,">",1) Then _FatalError("invalid clues = no solution")

	For $cc=0 To 2
		$startcol=$cc*3

		For $rc=0 To 2
			$startrow=$rc*3
			If _Eigen_ConditCount_Block($solution,$startrow,$startcol,3,3,"==",$dc)>1 Then _FatalError("invalid clues = no solution")
		Next
	Next
Next

; this was deferred until all clues were processed
If _UpdateCandidatesPermuts()=False Then _FatalError("invalid clues = no solution")
_CheckAllBlocks()	; fill blockfreqs & check whether solution is already found

;_____________________________________________________________________________
; SOLVE SUDOKU

; whittle down the number of permutations per block
If $callSolver1st Then
	_SaveCandidatesPermuts()	; prep restore
	If _Solver()=True And Not @error Then _CloseDown()	; no need for brute-force
	If @error Then _RestoreCandidatesPermuts()	; ensure valid state
EndIf

_OptimiseMax()
_BruteForce()
_CloseDown()

;=============================================================================
; Example Sudokus 1-10, from easy to extreme
; timings depend on hardware; provided for relative comparison only

Func _LoadSudoku($selection=0)

	Switch $selection

	Case 1	; easy (pure BF = 1 sec), found @ 50% done; solver takes: 12 sec.
		Global	$sudoku = [	0,3,0, 0,0,1, 9,6,0, _
							0,0,1, 0,4,6, 5,0,0, _
							0,5,7, 2,0,0, 3,0,0, _
							1,0,6, 0,0,2, 0,0,0, _
							0,0,0, 0,0,0, 8,9,0, _
							0,4,5, 7,3,0, 0,1,0, _
							0,0,0, 8,0,5, 4,0,0, _
							0,2,0, 1,9,0, 0,0,7, _
							8,6,0, 0,0,0, 0,5,3 ]

	Case 2	; easy (pure BF = 1 sec), found @ 83% done; solver takes: 15 sec.
		Global	$sudoku = [	4,0,0, 0,0,0, 5,9,0, _
							0,3,2, 0,0,0, 0,6,0, _
							0,7,0, 6,0,9, 2,0,3, _
							1,0,8, 0,4,0, 0,0,0, _
							7,0,0, 8,5,0, 0,0,0, _
							0,9,0, 0,0,0, 1,3,8, _
							6,0,0, 1,0,0, 0,0,2, _
							0,0,0, 2,0,6, 9,0,0, _
							0,8,9, 4,0,0, 0,0,5 ]

	Case 3	; moderate (pure BF = 2 sec), found @ 0% done; solver takes: 37 sec.
		Global	$sudoku = [	0,0,6, 0,0,9, 0,0,0, _
							0,7,0, 0,0,8, 2,0,5, _
							5,0,3, 0,7,0, 0,0,0, _
							8,1,2, 0,0,0, 0,0,0, _
							0,0,0, 9,3,0, 0,0,0, _
							0,0,0, 0,0,0, 0,4,6, _
							0,5,0, 0,0,0, 0,0,9, _
							0,0,0, 1,0,2, 0,0,0, _
							0,0,0, 0,0,0, 5,6,4 ]

	Case 4	; moderate (pure BF = 3 sec), found @ 0% done; solver takes: 38 sec.
			Global	$sudoku = [	1,0,0, 8,0,4, 0,0,0, _
							0,7,0, 0,6,3, 0,0,0, _
							0,0,0, 0,0,5, 0,6,4, _
							0,0,0, 0,0,0, 0,4,9, _
							5,3,0, 0,0,0, 0,0,0, _
							0,0,0, 1,3,7, 0,0,0, _
							2,0,0, 0,5,0, 0,0,0, _
							0,4,6, 0,9,0, 8,0,0, _
							0,0,8, 0,0,0, 2,0,7 ]

	Case 5	; moderate (pure BF = 4 sec), found @ 50% done; solver takes: 16 sec.
		Global	$sudoku = [ 5,3,0, 0,7,0, 0,0,0, _
							6,0,0, 1,9,5, 0,0,0, _
							0,9,8, 0,0,0, 0,6,0, _
							8,0,0, 0,6,0, 0,0,3, _
							4,0,0, 8,0,3, 0,0,1, _
							7,0,0, 0,2,0, 0,0,6, _
							0,6,0, 0,0,0, 2,8,0, _
							0,0,0, 4,1,9, 0,0,5, _
							0,0,0, 0,8,0, 0,7,9 ]

	Case 6	; hard (pure BF = 32 sec), found @ 8% done; solver + BF take: 31 + 34 = 65 sec.
		Global	$sudoku = [ 1,0,0, 0,0,7, 0,9,0, _
							0,3,0, 0,2,0, 0,0,8, _
							0,0,9, 6,0,0, 5,0,0, _
							0,0,5, 3,0,0, 9,0,0, _
							0,1,0, 0,8,0, 0,0,2, _
							6,0,0, 0,0,4, 0,0,0, _
							3,0,0, 0,0,0, 0,1,0, _
							0,4,0, 0,0,0, 0,0,7, _
							0,0,7, 0,0,0, 3,0,0 ]

	Case 7	; hard (pure BF = 42 sec), found @ 83% done; solver takes: 64 sec.
		Global	$sudoku = [ 4,0,2, 0,0,0, 0,3,0, _
							1,0,0, 6,0,5, 0,2,9, _
							9,0,0, 0,0,0, 1,0,0, _
							0,0,0, 0,4,2, 0,0,0, _
							0,8,0, 9,0,1, 0,5,0, _
							0,0,0, 8,5,0, 0,0,0, _
							0,0,3, 0,0,0, 0,0,8, _
							6,1,0, 5,0,4, 0,0,2, _
							0,4,0, 0,0,0, 5,0,7 ]

	Case 8	; hard (pure BF = 85 sec), found @ 35% done; solver takes: 38 sec.
		Global	$sudoku = [	5,0,0, 0,0,0, 0,2,4, _
							0,0,0, 1,9,0, 0,0,0, _
							7,0,0, 0,0,0, 0,5,0, _
							0,0,0, 0,0,0, 0,9,2, _
							9,3,0, 0,7,0, 0,0,0, _
							2,0,0, 3,6,0, 0,0,0, _
							0,0,8, 0,0,0, 9,0,0, _
							0,0,2, 6,0,4, 8,0,0, _
							0,0,4, 0,0,8, 0,0,1 ]

	Case 9; the Easter Monster (pure BF = 1,232 sec (20.5 min)), found @ 65% done; solver FAILS!
		Global	$sudoku = [ 1,0,0, 0,0,0, 0,0,2, _
							0,9,0, 4,0,0, 0,5,0, _
							0,0,6, 0,0,0, 7,0,0, _
							0,5,0, 9,0,3, 0,0,0, _
							0,0,0, 0,7,0, 0,0,0, _
							0,0,0, 8,5,0, 0,4,0, _
							7,0,0, 0,0,0, 6,0,0, _
							0,3,0, 0,0,9, 0,8,0, _
							0,0,2, 0,0,0, 0,0,1]

	Case 10	; extreme (pure BF = 2,495 sec (41.6 min)), found @ 81% done; solver FAILS!
		Global	$sudoku = [ 8,0,0, 0,0,0, 0,0,0, _
							0,0,3, 6,0,0, 0,0,0, _
							0,7,0, 0,9,0, 2,0,0, _
							0,5,0, 0,0,7, 0,0,0, _
							0,0,0, 0,4,5, 7,0,0, _
							0,0,0, 1,0,0, 0,3,0, _
							0,0,1, 0,0,0, 0,6,8, _
							0,0,8, 5,0,0, 0,1,0, _
							0,9,0, 0,0,0, 4,0,0 ]
	Case Else

		Local $fname=FileOpenDialog("Please select an sdk file",@ScriptDir,"(*.sdk; *.txt)",1)
		If @error Then _FatalError("file selection error: " & @error)

		If _FileCountLines($fname)<>9 Or @error Then _FatalError("file access error or unrecognised format; text file with 9 lines expected, containing 9 single digits (1-9) or dots ('.') for empty cells")

		Local $line,$fh=FileOpen($fname)
		If @error then _FatalError("unable to access file: " & $fname)

		For $rc=0 To 8
			$line=StringSplit(StringReplace(StringStripWS(FileReadLine($fh),8),".","0"),"",2)
			If @error or not IsArray($line) Or UBound($line)<>9 Then
				FileClose($fh)
				_FatalError("unrecognised format; text file with 9 lines expected, containing 9 single digits (1-9) or dots ('.') for empty cells; spaces are ignored.")
			EndIf

			For $cc=0 To 8
				$sudoku[$cc+($rc*9)]=$line[$cc]
			Next
		Next
		FileClose($fh)

	EndSwitch

	_CreateSudokuMatrix($sudoku,True)	; T: stored as 1D rowwise array

EndFunc


Func _CreateSudokuMatrix(ByRef $array,$stored1Drowwise=False)

	If UBound($array,0)>2 Then _FatalError("1D or 2D array expected")

	$matSudoku=_Eigen_CreateMatrix_fromArray($array)
	If @error Then _FatalError("unable to create input matrix")
	If _Eigen_GetMatrixSize($matSudoku)<>81 Then _FatalError("parsed array should have exactly 81 integer values (0 for blanks, 1-9 for clues)")

	If Not _Eigen_IsVector($matSudoku) Then	; 2D-specific checks
		If _Eigen_GetRows($matSudoku)<>9 Or Not _Eigen_IsSquare($matSudoku) Then _FatalError("parsed 2D array should have 9 x 9 integer values (0 for blanks, 1-9 for clues)")
	EndIf

	; check valid content
	If _Eigen_IsZero($matSudoku) Then _FatalError("parsed sudoku array is completely empty (all zeroes)")
	If _Eigen_GetMinVal($matSudoku)<0 Then _FatalError("negative values in sudoku array cannot be processed")
	If _Eigen_GetMinVal($matSudoku)>0 Then _FatalError("no blank values found to resolve in parsed sudoku array")
	If _Eigen_GetMaxVal($matSudoku)>9 Then _FatalError("largest value in array exceeds maximum of 9")

	; reformat valid vector input to 9x9 square matrix
	_Eigen_ReDim_ExistingMatrix($matSudoku,9,9)	; no change if already 2D
	If $stored1Drowwise Then _Eigen_Transpose_InPlace($matSudoku)

	_MatrixDisplay($matSudoku,"input sudoku (press <Esc> to proceed)")

EndFunc

;_____________________________________________________________________________
; Bitmasks and lists

Func _StoreMaskOfCellsWithNvalues($tempRowVec,$N,$mincount=1)
; sum the candidates bitmask colwise (=per cell)
; stores a rowvector bitmask of 81 cells in whatever rowvec var is parsed

		If $mincount<1 Then Return False

		_Eigen_MatrixSpecs_Colwise_Single($candidates,2,$tempRowVec)	; 2=sum
		_Eigen_ConditMask_InPlace($tempRowVec,"==",$N)

		Return (_Eigen_GetSum($tempRowVec)>=$mincount)
EndFunc


Func _CreateListOfCellsWithNvalues($tempRowVec,$N,$minsize=1)
; expects a parsed 1x81 rowvector to be filled
; stored values are size-ordered; offset/row/col/value are base-0

	If $N<1 Or $N>9 Then Return SetError(1,0,False)

	; create a colmask for $candidates for cells with exactly N values
	If _StoreMaskOfCellsWithNvalues($tempRowVec,$N,$minsize)=False Then Return SetError(1,0,False)

	; list cells with N values
	_Eigen_Redim_ExistingMatrix($tempRowVec,9,9)
	Local $list=_Eigen_FindAll($tempRowVec,1)
	Local $listsize=@extended
	_Eigen_Redim_ExistingMatrix($tempRowVec,1,81)	; needed again

	; fill candidates mirror with the actual values
	_FillCandidValues(); NB stored values are base-1 (0 = empty = invalid candidate value for cell)

	; remove all cols (cells) with fewer/more than N values
	Local $tempcandid=_Eigen_CreateMatrix_FromAcols_Mask ($candidValues,$tempRowVec)

	; store all non-zero values per listed cell
	Local $tempvalues=_Eigen_CreateVector_FromAcell_Mask($tempcandid,$tempcandid)
	_Eigen_CwiseScalarOp_InPlace($tempvalues,"-",1)	; make values base-0
	_Eigen_Redim_ExistingMatrix($tempvalues,$N,$listsize)

	; append values to list
	_Eigen_AppendCols_InPlace ($list,$N)	; example of an alias-wrapper call
	For $rc=0 To $N-1
		_Eigen_Copy_Arow_ToBcol($tempvalues,$list,$rc,$rc+3)
	Next

	_Eigen_ReleaseMatrix($tempcandid)
	_Eigen_ReleaseMatrix($tempvalues)

	Return SetError(0,$listsize,$list)
EndFunc


Func _FillMaskOfOverlaps($inputmask,$outputmask,$rowmask=True)
; expects a parsed digitprofile inputmask (a colvector of size 27)
; produces a permuts rowvec mask of overlaps

	Switch $rowmask		; select block digit profile
		Case True
			_Eigen_CwiseBinaryOp_Colwise($digitInRows,"+",$inputmask,$matDigitBuffer)
		Case Else
			_Eigen_CwiseBinaryOp_Colwise($digitInCols,"+",$inputmask,$matDigitBuffer)
	EndSwitch

	_Eigen_ConditAny_Colwise($matDigitBuffer,">",1,$outputmask)	; permut mask (1) = overlap

EndFunc


Func _FillMasksOfValidNeighbours($permutID,$blockID)
; create row/col mask of those permutations that do not produce a row/col duplication clash
; 1. add the content of our target column (= permutation) to ALL cols
; 2. flag as valid only those cols with zero overlap in rows/cols (2=overlap; 0/1=no overlap)
; 3. store it in the target block's bitmask array for valid (row c.q. col) neighours

	_Eigen_CwiseBinaryOp_ColwiseCol($digitInRows,$permutID,"+",$digitInRows,$matDigitBuffer)
	_Eigen_ConditAll_Colwise($matDigitBuffer,"<",2,$validRowNeighbours[$blockID])	; mask (1) = valid

	_Eigen_CwiseBinaryOp_ColwiseCol($digitInCols,$permutID,"+",$digitInCols,$matDigitBuffer)
	_Eigen_ConditAll_Colwise($matDigitBuffer,"<",2,$validColNeighbours[$blockID])	; mask (1) = valid

EndFunc


Func _MaskValidPermutations($blockID, $validmask1=0, $validmask2=0, $validmask3=0, $validmask4=0)
; reduce valid permuts by applying 1-4 "and" bitmask filters (negative exclusion: 0 = out; 1 = keep)

	_Eigen_Copy_A_ToB($punchout[$blockID],$matpermut[$blockID])	; keep original in $punchout array

	If $validmask1>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask1)
	If $validmask2>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask2)
	If $validmask3>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask3)
	If $validmask4>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask4)

EndFunc

;_________________________________________________________________________________
; Load / Create the Lookup Tables

Func _LoadBlockPermutations()
; 9 rows (digits 1-9, base-1) x 362,880 cols (block permutations)

	Local $filename="blockpermut.mat"

	If FileExists($filename) Then
		$blockPermuts = _Eigen_LoadMatrix($filename)
		Return
	EndIf

	If $verbose Then ConsoleWrite("Generating lookup table file " & $filename & "; please wait..." & @CRLF & "NB This only needs to be done ONCE." & @CRLF)

	$blockPermuts = _Eigen_CreateMatrix (9,$permuts)
	Local $matC = _Eigen_CreateMatrix (9,1)

	Local $rc=-1
	For $c0=1 To 9
		_Eigen_WriteMatrixValue($matC,0,0,$c0)

	For $c1=1 To 9
		If $c1=$c0 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,1,0,$c1)

	For $c2=1 To 9
		If $c2=$c0 Then ContinueLoop
		If $c2=$c1 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,2,0,$c2)

	For $c3=1 To 9
		If $c3=$c0 Then ContinueLoop
		If $c3=$c1 Then ContinueLoop
		If $c3=$c2 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,3,0,$c3)

	For $c4=1 To 9
		If $c4=$c0 Then ContinueLoop
		If $c4=$c1 Then ContinueLoop
		If $c4=$c2 Then ContinueLoop
		If $c4=$c3 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,4,0,$c4)

	For $c5=1 To 9
		If $c5=$c0 Then ContinueLoop
		If $c5=$c1 Then ContinueLoop
		If $c5=$c2 Then ContinueLoop
		If $c5=$c3 Then ContinueLoop
		If $c5=$c4 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,5,0,$c5)

	For $c6=1 To 9
		If $c6=$c0 Then ContinueLoop
		If $c6=$c1 Then ContinueLoop
		If $c6=$c2 Then ContinueLoop
		If $c6=$c3 Then ContinueLoop
		If $c6=$c4 Then ContinueLoop
		If $c6=$c5 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,6,0,$c6)

	For $c7=1 To 9
		If $c7=$c0 Then ContinueLoop
		If $c7=$c1 Then ContinueLoop
		If $c7=$c2 Then ContinueLoop
		If $c7=$c3 Then ContinueLoop
		If $c7=$c4 Then ContinueLoop
		If $c7=$c5 Then ContinueLoop
		If $c7=$c6 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,7,0,$c7)

	For $c8=1 To 9
		If $c8=$c0 Then ContinueLoop
		If $c8=$c1 Then ContinueLoop
		If $c8=$c2 Then ContinueLoop
		If $c8=$c3 Then ContinueLoop
		If $c8=$c4 Then ContinueLoop
		If $c8=$c5 Then ContinueLoop
		If $c8=$c6 Then ContinueLoop
		If $c8=$c7 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,8,0,$c8)

		$rc+=1
		_Eigen_Copy_Acol_ToBcol($matC,$blockPermuts,0,$rc)
		If $verbose And Mod($rc,1000)=0 Then ConsoleWrite("Processed " & $rc & " of " & $permuts & @CRLF)
	Next
	Next
	Next
	Next
	Next
	Next
	Next
	Next
	Next

	If $verbose Then ConsoleWrite("Saving file..." & @CRLF & @CRLF)
	_Eigen_SaveMatrix($blockPermuts,$filename,True)
	_Eigen_ReleaseMatrix($matC)

EndFunc


Func _LoadDigitTables()
; 3x9 rows (row c.q. col 0/1/2 for digits 1-9, for each block permutation)

	Local $filename1="digitInRows.mat"
	Local $filename2="digitInCols.mat"

	If (FileExists($filename1) And FileExists($filename2)) Then
		$digitInRows=_Eigen_LoadMatrix($filename1)	; permut x colvectors (of size: 27 rows: 0/1/2 for digit 1, 0/1/2 for digit 2, etc)
		$digitInCols=_Eigen_LoadMatrix($filename2)	; permut x rowvectors^T (of size: 27 rows: 0/1/2 for digit 1, 0/1/2 for digit 2, etc)
		Return
	EndIf

	If $verbose Then ConsoleWrite("Generating lookup table files " & $filename1 & " and " & $filename2 & "; please be patient..." & @CRLF & "NB This only needs to be done ONCE." & @CRLF)

	Local $matB = _Eigen_CreateMatrix (9,1)
	Local $matDrows = _Eigen_CreateMatrix (3,1)	; colvector
	Local $matDcols = _Eigen_CreateMatrix (1,3)	; rowvector
	Local $matDcolsT = _Eigen_CreateMatrix (3,1)	; rowvector

	$digitInRows = _Eigen_CreateMatrix_Zero (3*9,$permuts)	; colvectors
	$digitInCols = _Eigen_CreateMatrix_Zero (3*9,$permuts)	; rowvectors

	For $pc=0 To $permuts-1
		_Eigen_Copy_Acol_ToBcol($blockPermuts,$matB,$pc,0)

		_Eigen_Redim_ExistingMatrix ($matB,3,3)
		For $dc=1 To 9
			$offset=($dc-1)*3
			_Eigen_ConditAny_Rowwise($matB,"=",$dc,$matDrows)
			_Eigen_ConditAny_Colwise($matB,"=",$dc,$matDcols)
			_Eigen_Transpose($matDcols,$matDcolsT)

			_Eigen_Copy_Ablock_ToBblock($matDrows,$digitInRows,0,0,3,1,$offset,$pc)
			_Eigen_Copy_Ablock_ToBblock($matDcolsT,$digitInCols,0,0,3,1,$offset,$pc)
		Next
		_Eigen_Redim_ExistingMatrix ($matB,9,1)

		If $verbose And Mod($pc,100)=0 Then ConsoleWrite("Processed " & $pc & " of " & $permuts & @CRLF)
	Next

	If $verbose Then ConsoleWrite("Saving files..." & @CRLF & @CRLF)
	_Eigen_SaveMatrix($digitInRows,$filename1,True)
	_Eigen_SaveMatrix($digitInCols,$filename2,True)

	_Eigen_ReleaseMatrix($matB)
	_Eigen_ReleaseMatrix($matDrows)
	_Eigen_ReleaseMatrix($matDcols)
	_Eigen_ReleaseMatrix($matDcolsT)

EndFunc


Func _FillCandidValues()
; uses $candidates bitmask multiplication of colwise values 1-9 as a filter

	_Eigen_SetLinSpaced($valuecolvec,1,9)	; 0 = empty
	_Eigen_CwiseBinaryOp_ColwiseCol($candidates,0,"*",$valuecolvec,$candidValues)

EndFunc


Func _UpdateCandidatesPermuts()
; store the remaining permuts per block from $punchout[block] to $matPunchout (all blocks)

	For $dc=0 To 8
		_Eigen_Copy_Arow_ToBrow($punchOut[$dc],$matPunchOut,0,$dc)
		_Eigen_Copy_Arow_ToBrow($candidate[$dc],$candidates,0,$dc)
	Next

	; sum remaining active permuts per block
	_Eigen_MatrixSpecs_Rowwise_Single($matpunchOut,2,$scratchcolvec)

	Return Not (_Eigen_ConditAny($scratchcolvec,"==",0))	; F: invalid configuration
EndFunc


Func _SaveCandidatesPermuts()	; prep restore

	_Eigen_Copy_A_ToB($candidates,$savecandidates)
	_Eigen_Copy_A_ToB($matPunchOut,$savePunchOut)

EndFunc


Func _RestoreCandidatesPermuts()
; restore state for next attempt

	_Eigen_Copy_A_ToB($savecandidates,$candidates)
	_Eigen_Copy_A_ToB($savePunchOut,$matPunchOut)

	For $dc=0 To 8
		_Eigen_Copy_Arow_ToBrow($matPunchOut,$punchOut[$dc],$dc,0)
		_Eigen_Copy_Arow_ToBrow($candidates,$candidate[$dc],$dc,0)
	Next

EndFunc


Func _OptimiseMax()
; Block IDs (row=offset) are LinSpaced Colwise in a 3x3 grid, from 0 to 8.
; Identify the two blocks with the largest number of active permutations left;
; swap these onto the block diagonal, and identify the other block positions

	_UpdateCandidatesPermuts()	; ensure permuts are up-to-date
	_CheckAllBlocks()			; ensure freqs are up-to-date
	_Eigen_Redim_ExistingMatrix($blockfreqs,3,3)	; reshape as block
	If $verbose Then _Eigen_ShowMatrix($blockfreqs,"Remaining block permutations")

	Local $blockMF=_Eigen_GetOffset_MaxVal($blockfreqs) ; find block ID with highest frequency
		If $verbose Then ConsoleWrite(@CRLF & "Swapped Block ID " & $blockMF & " (in major square, now at TL) has the highest freq of " & _Eigen_GetMaxVal($blockfreqs) & @CRLF)

	Local $maxValBlockrow=_Eigen_GetRow_MaxVal($blockfreqs)
	Local $maxValBlockcol=_Eigen_GetCol_MaxVal($blockfreqs)

	If $blockMF>0 Then	; update the grids
		If $maxValBlockrow>0 Then
			_Eigen_Swap_Arow_Arow($matBlockIDs,$maxValBlockrow,0)
			_Eigen_Swap_Arow_Arow($blockfreqs,$maxValBlockrow,0)
		EndIf

		If $maxValBlockcol>0 Then
			_Eigen_Swap_Acol_Acol($matBlockIDs,$maxValBlockcol,0)
			_Eigen_Swap_Acol_Acol($blockfreqs,$maxValBlockcol,0)
		EndIf
	EndIf

	; punchout MF target all directly neighbouring blocks
	_Eigen_SetZero_Row($blockfreqs,0)
	_Eigen_SetZero_Col($blockfreqs,0)


	Local $blockMF2=_Eigen_GetOffset_MaxVal($blockfreqs) ; find block ID with next-highest frequency in minor square
	If $verbose Then ConsoleWrite(@CRLF & "Swapped Block ID " & $blockMF2 & " (in minor square, now at BR) has the next-highest freq of " & _Eigen_GetMaxVal($blockfreqs) & @CRLF)

	Local $maxValBlockrow=_Eigen_GetRow_MaxVal($blockfreqs)
	Local $maxValBlockcol=_Eigen_GetCol_MaxVal($blockfreqs)

	; if current BR block is suboptimal, swap with optimal (most permuts)
	If $maxValBlockrow<2 Then
		_Eigen_Swap_Arow_Arow($matBlockIDs,1,2)		; 2D grid row swap
		_Eigen_Swap_Arow_Arow($blockfreqs,1,2)		; update freq table
	EndIf

	If $maxValBlockcol<2 Then
		_Eigen_Swap_Acol_Acol($matBlockIDs,1,2)	; 2D grid col swap
		_Eigen_Swap_Acol_Acol($blockfreqs,1,2)	; 2D grid col swap
	EndIf

	; diagonal blocks (globals)
	$blockTL=_Eigen_ReadMatrixValue($matBlockIDs,0,0)	; Top Left block
	$blockCC=_Eigen_ReadMatrixValue($matBlockIDs,1,1)	; Centre block
	$blockBR=_Eigen_ReadMatrixValue($matBlockIDs,2,2)	; Bottom Right block

	; upper off-diagonal blocks (globals)
	$blockTC=_Eigen_ReadMatrixValue($matBlockIDs,0,1)	; Top Centre block
	$blockTR=_Eigen_ReadMatrixValue($matBlockIDs,0,2)	; Top Right block
	$blockCR=_Eigen_ReadMatrixValue($matBlockIDs,1,2)	; Centre Right block

	; lower off-diagonal blocks (globals)
	$blockCL=_Eigen_ReadMatrixValue($matBlockIDs,1,0)	; Centre Left block
	$blockBL=_Eigen_ReadMatrixValue($matBlockIDs,2,0)	; Bottom Left block
	$blockBC=_Eigen_ReadMatrixValue($matBlockIDs,2,1)	; Bottom Centre block

	If $verbose Then ConsoleWrite(@CRLF & "Remaining diagonal block ID " & $blockMF2 & " (now at CC) has a freq of " & _Eigen_ReadMatrixValue($blockfreqs,1,1) & @CRLF)

EndFunc

;_________________________________________________________________________________
; punch candidate values in/out

Func _PunchIn($cll,$row,$col,$valueb0)
; final cell-value assignment

	If $testing Then ConsoleWrite("PunchIn cell " & $cll & ", row " & $row & ", col " & $col & ", value = " & $valueb0+1 & @CRLF)

	Local $blockID=_Eigen_ReadMatrixValue($matBlocks,$row,$col)

	$blockIDadjRow1=$blockneighbours[$blockID][0]
	$blockIDadjRow2=$blockneighbours[$blockID][1]

	$blockIDadjCol1=$blockneighbours[$blockID][2]
	$blockIDadjCol2=$blockneighbours[$blockID][3]

	; create mask for rows
	_Eigen_SetZero($digitRowvec)
	_Eigen_WriteMatrixValue($digitRowvec,($valueb0*3)+Mod($row,3),0,1)
	_FillMaskOfOverlaps($digitRowvec,$matMrowvec,True)	; T: rowmask, F: colmask

	; create mask for cols
	_Eigen_SetZero($digitColvec)
	_Eigen_WriteMatrixValue($digitColvec,($valueb0*3)+Mod($col,3),0,1)
	_FillMaskOfOverlaps($digitColvec,$matMcolvec,False)	; T: rowmask, F: colmask

	; include ONLY clue's own block's permuts that match the clue
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockID],"and",$matMrowvec)
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockID],"and",$matMcolvec)

	; exclude in the two row-adjacent blocks all permuts with clue value in same ROW
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockIDadjRow1],"xcl",$matMrowvec)
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockIDadjRow2],"xcl",$matMrowvec)

	; exclude in the two col-adjacent blocks all permuts with clue value in same COL
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockIDadjCol1],"xcl",$matMcolvec)
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockIDadjCol2],"xcl",$matMcolvec)

	; and for solver
	_Eigen_Copy_A_ToB($allbuddies[$cll],$retainRowVec)
	_Eigen_CwiseLogicalOp_InPlace($candidate[$valueb0],"xcl",$retainRowVec)

	; zero out all values for the target cell in $candidates
	For $dc=0 To 8
		_Eigen_WriteMatrixValue($candidate[$dc],0,$cll,0)
	Next

	$reduced+=1

EndFunc


Func _PunchOutInBlockRow($block,$row,$valueb0)
; candidate removal in up to 3 cells in a block (row)

	If $testing Then ConsoleWrite("PunchOutInBlockRow block " & $block & ", row " & $row & ", value = " & $valueb0+1 & @CRLF)

	; create mask for rows
	_Eigen_SetZero($digitRowvec)
	_Eigen_WriteMatrixValue($digitRowvec,($valueb0*3)+Mod($row,3),0,1)
	_FillMaskOfOverlaps($digitRowvec,$matMrowvec,True)	; T: rowmask, F: colmask; T2: flip the bits (reverse condition)

	; exclude in the block all permuts with clue value in specified row
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$block],"xcl",$matMrowvec)	; permuts mask

	; and for solver
	Local $before=_Eigen_GetSum($candidate[$valueb0])
	_Eigen_CwiseLogicalOp ($blockbuddymask[$block],"and",$rowbuddymask[$row],$scratchRowVec)	; cells mask
	_Eigen_CwiseLogicalOp_InPlace ($candidate[$valueb0],"xcl",$scratchRowVec)
	$reduced+=($before-_Eigen_GetSum($candidate[$valueb0]))

EndFunc


Func _PunchOutInBlockCol($block,$col,$valueb0)
; candidate removal in up to 3 cells in a block (column)

	If $testing Then ConsoleWrite("PunchOutInBlockCol block " & $block & ", col " & $col & ", value = " & $valueb0+1 & @CRLF)

	; create mask for cols
	_Eigen_SetZero($digitColvec)
	_Eigen_WriteMatrixValue($digitColvec,($valueb0*3)+Mod($col,3),0,1)
	_FillMaskOfOverlaps($digitColvec,$matMcolvec,False)	; T: rowmask, F: colmask

	; exclude in the block all permuts with clue value in specified col
	_Eigen_CwiseLogicalOp_InPlace ($PunchOut[$block],"xcl",$matMcolvec)	; permuts mask

	; and for solver
	Local $before=_Eigen_GetSum($candidate[$valueb0])
	_Eigen_CwiseLogicalOp ($blockbuddymask[$block],"and",$colbuddymask[$col],$scratchRowVec)	; cells mask
	_Eigen_CwiseLogicalOp_InPlace ($candidate[$valueb0],"xcl",$scratchRowVec)

	$reduced+=($before-_Eigen_GetSum($candidate[$valueb0]))

EndFunc


Func _PunchOutCellCandidate($cell,$valueb0)
; eliminate one candidate value from a single cell

	If $testing Then ConsoleWrite("PunchOutCellCandidate cell " & $cell & ", value = " & $valueb0+1 & @CRLF)

	Local $row=_Eigen_ReadMatrixValue($rowindexRowVec,0,$cell)
	Local $col=_Eigen_ReadMatrixValue($colindexRowVec,0,$cell)
	Local $block=_Eigen_ReadMatrixValue($blockindexRowVec,0,$cell)

	; create mask for rows
	_Eigen_SetZero($digitRowvec)
	_Eigen_WriteMatrixValue($digitRowvec,($valueb0*3)+Mod($row,3),0,1)
	_FillMaskOfOverlaps($digitRowvec,$matMrowvec,True)	; T: rowmask, F: colmask

	; create mask for cols
	_Eigen_SetZero($digitColvec)
	_Eigen_WriteMatrixValue($digitColvec,($valueb0*3)+Mod($col,3),0,1)
	_FillMaskOfOverlaps($digitColvec,$matMcolvec,False)	; T: rowmask, F: colmask

	; exclude in the block all permuts with clue value in row+col cell
	_Eigen_CwiseLogicalOp_InPlace ($matMrowvec,"and",$matMcolvec)	; permuts mask
	_Eigen_CwiseLogicalOp_InPlace ($PunchOut[$block],"xcl",$matMrowvec)

	; and for solver
	_Eigen_WriteMatrixValue($candidate[$valueb0],0,$cell,0)
	$reduced+=1

EndFunc


Func _PunchOutValueForMaskedCells($valueb0,$tempRowVec)
; eliminate one candidate value for all masked cells in a parsed rowvec bitmask (81 cells)

	If _Eigen_IsZero($tempRowVec) Then Return

	Local $list=_Eigen_FindAll($tempRowVec,1)
	Local $lastlist=@extended-1
	If $lastlist=-1 Then Return

	For	$lc=0 To $lastlist
		_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list,$lc,0),$valueb0)
	Next
	_Eigen_ReleaseMatrix($list)

EndFunc


;_____________________________________________________________________________
; solution-related

Func _CheckAllBlocks()
; 1. Are all block permutation counts reduced to unity?
; if so, construct the full solution and check its validity in terms of
; having 9 unique values for each row and each column
; NB we don't ever need to check the blocks themselves,
;	because their permutations are by definition valid

	_Eigen_ReDim_ExistingMatrix($blockfreqs,9,1)	; reshape as colvector
	_Eigen_MatrixSpecs_Single_Rowwise($matPunchOut,2,$blockfreqs)	; 2=sum remaining permuts per block in colvector
	_Eigen_ReDim_ExistingMatrix($blockfreqs,3,3)	; restore block shape

	If _Eigen_IsOnes($blockfreqs) Then
		If $verbose Then ConsoleWrite(@CRLF)

		For $bc=0 To 8
			_StoreBlock($bc,_Eigen_GetCol_MaxVal($punchout[$bc]))
		Next
		_CheckSolution()			; does not return if single solution is sought and valid solution is found
		Return SetError(3,0,True)	; invalid solution

	Else
		If _Eigen_IsZero($blockfreqs) Then Return SetError(3,0,True)	; invalid configuration
		If _Eigen_ConditAny($blockfreqs,"==",0) Then Return SetError(1,0,True)		; invalid guess; configuration produces empty permut subset in at least one other block
		If $testing Then _Eigen_Show_Matrix($blockfreqs,"Remaining block permutations")

		If _Eigen_GetSum($blockfreqs)=$blockfreqsum Then Return SetError(2,0,True)	; no improvement in entire solver pass
		$blockfreqsum=_Eigen_GetSum($blockfreqs)	; for next pass
	EndIf

	Return False	; not done yet
EndFunc


Func _StoreBlock($blockID,$permutID)
; construct a single block in the solution

	If $verbose Then ConsoleWrite("Block " & $blockID & " contains permutation: " & $permutID & @CRLF)

	_Eigen_ReDim_ExistingMatrix($solutionBlock,9,1)
	_Eigen_Copy_Acol_ToBcol($blockPermuts,$solutionBlock,$permutID,0)

	_Eigen_ReDim_ExistingMatrix($solutionBlock,3,3)
	_Eigen_Copy_Ablock_ToBblock($solutionBlock,$solution,0,0,3,3,(Mod($blockID,3)*3),(Int($blockID/3)*3))

EndFunc


Func _CheckSolution()
; does each value 1-9 occur exactly once in each row and each column?
; NB no need to check block integrity (valid by definition)

	For $dc=1 To 9
		_Eigen_ConditCount_Rowwise($solution,"==",$dc,$solutionCol)
		If _Eigen_IsOnes($solutionCol)=False Then Return

		_Eigen_ConditCount_Colwise($solution,"==",$dc,$solutionRow)
		If _Eigen_IsOnes($solutionRow)=False Then Return
	Next

	$solutionsFound+=1
	If $verbose Then _WinAPI_Beep()
	If ($solutionsFound=1 And $exitUpon1stHit) Then _CloseDown()

EndFunc


Func _FatalError($errmsg="an unknown error occurred")
; Oops!

	ConsoleWrite("Fatal Error: " & $errmsg & @CRLF)
	Exit(1)

EndFunc


Func _CloseDown()
; All done, folks.

	If $tsolver Then ConsoleWrite(@CRLF & "Solver took: " & Round((TimerDiff($tsolver)/1000)-(($tstart)?(TimerDiff($tstart)/1000):(0)),2) & " seconds" & @CRLF)
	If $tstart Then ConsoleWrite(@CRLF & "Brute-Force took: " & Round(TimerDiff($tstart)/1000,2)-(($verbose=True)?(1):(0)) & " seconds" & @CRLF)	; subtract 1 sec for beep
	ConsoleWrite(@CRLF & "Solution(s) found: " & $solutionsFound & @CRLF & @CRLF)

	If $solutionsFound>0 Then
		_Eigen_ShowMatrix($solution,"Sudoku solution found:")

		; display solution IF _Eigen_Show_MatrixDisplay() was called before (or _Eigen_Hide_MatrixDisplay() was NOT called before and Eigen4AutoIt.ini setting allows it)
		_MatrixDisplay($solution,"Solution found (press <Esc> to Quit)")
	EndIf

	ConsoleWrite("Please wait; closing down..." & @CRLF)
	_Eigen_CleanUp()	; ...End of Story
	Exit(0)

EndFunc

;_________________________________________________________________________________
; solver methods

Func _NakedSingle()
; FIND all instances of cells with a single candidate value left
; REMOVE all permutations with other candidates at that location ("punch-in")

	If _StoreMaskOfCellsWithNvalues($cellRowVec,1,1)=False Then Return True

	Local $prevred=$reduced
	Local $list=_Eigen_FindAll($cellRowVec,1)
	Local $cll,$lastlist=@extended-1

	; collect single value for each cell
	_Eigen_MatrixSpecs_Colwise_Single($candidates,14,$valueRowVec)	; 14=maxvalrow

	For $sc=0 To $lastlist
		$cll=_Eigen_ReadMatrixValue($list,$sc,0)
		If $testing Then ConsoleWrite("RESOLVED cell: " & $cll & @CRLF)

		_PunchIn($cll,Mod($cll,9),Int($cll/9),_Eigen_ReadMatrixValue($valueRowVec,0,$cll))
	Next
	_Eigen_ReleaseMatrix($list)

	If $reduced>$prevred Then
		If _UpdateCandidatesPermuts()=False Then Return False
		If $verbose Then ConsoleWrite(@TAB & "NakedSingle reductions: " & $reduced-$prevred & @CRLF)
	EndIf

	Return True
EndFunc


Func _HiddenSingle()
; FIND in each unit any cell whose number of candidates can be deduced to be one
; REMOVE all permutations with other candidates at that location ("punch-in")

	Local $prevred=$reduced

	For $dc=0 To 8	; value base-0
		If _HiddenSingleUnit($dc,$rowbuddymask)=False Then Return False
		If _HiddenSingleUnit($dc,$colbuddymask)=False Then Return False
		If _HiddenSingleUnit($dc,$blockbuddymask)=False Then Return False
	Next

	If $verbose And $reduced>$prevred Then ConsoleWrite(@TAB & "HiddenSingle reductions: " & $reduced-$prevred & @CRLF)

	Return True
EndFunc


Func _HiddenSingleUnit($valueb0, ByRef $buddymask)

	Local $cll,$rw,$cl,$prevred=$reduced

	For $bc=0 To 8	; buddy loop
		_Eigen_CwiseLogicalOp($candidate[$valueb0],"and",$buddymask[$bc],$scratchRowVec)

		If _Eigen_GetSum($scratchRowVec)=1 Then

			$cll=_Eigen_GetCol_MaxVal($scratchRowVec)
			$rw=_Eigen_ReadMatrixValue($rowindexRowVec,0,$cll)
			$cl=_Eigen_ReadMatrixValue($colindexRowVec,0,$cll)

			_PunchIn($cll,$rw,$cl,$valueb0)
		EndIf
	Next

	If $reduced>$prevred Then Return _UpdateCandidatesPermuts()

	Return True
EndFunc


Func _NakedPairs()
; FIND two cells in a single unit that both have the same two possible values
; REMOVE the same two candidate values from all remaining cells in the unit

	If _StoreMaskOfCellsWithNvalues($cellRowVec,2,2)=False Then Return True

	If _NakedPairUnit($rowbuddymask,"rows")=False Then Return False
	If _NakedPairUnit($colbuddymask,"cols")=False Then Return False
	If _NakedPairUnit($blockbuddymask,"blocks")=False Then Return False

	Return True
EndFunc


Func _NakedPairUnit(ByRef $buddymask,$label)
; expects global $cellRowVec pre-filled

	Local $prevred=$reduced
	Local $list,$list2,$value1b0,$value2b0,$cell1,$cell2,$block,$lastlist,$lastlist2

	For $rc=0 To 8
		_Eigen_CwiseLogicalOp($cellRowVec,"and",$buddymask[$rc],$retainRowVec)
		If _Eigen_GetSum($retainRowVec)<2 Then ContinueLoop

		$list=_Eigen_FindAll($retainRowVec,1)
		$lastlist=@extended-1
		For $pc1=0 To $lastlist-1
			$cell1=_Eigen_ReadMatrixValue($list,$pc1,0)

			For $pc2=$pc1+1 To $lastlist
				$cell2=_Eigen_ReadMatrixValue($list,$pc2,0)

				_Eigen_CwiseBinaryOp_ColCol($candidates,$cell1,$cell2,"-",$candidates,$valuecolvec)
				If Not _Eigen_IsZero($valuecolvec) Then ContinueLoop

				; get the values
				_Eigen_Copy_Acol_ToBcol($candidates,$valuecolvec,$cell1,0)
				$value1b0=_Eigen_GetRow_MaxVal($valuecolvec)
				_Eigen_WriteMatrixValue($valuecolvec,$value1b0,0,0)
				$value2b0=_Eigen_GetRow_MaxVal($valuecolvec)

				; find all OTHER cells in current unit (so excluding cell1/2)
				_Eigen_CwiseLogicalOp($candidate[$value1b0],"and",$buddymask[$rc],$scratchRowVec)
				_Eigen_WriteMatrixValue($scratchRowVec,0,$cell1,0)	; punch out our pair
				_Eigen_WriteMatrixValue($scratchRowVec,0,$cell2,0)

				If Not _Eigen_IsZero($scratchRowVec) Then
					$list2=_Eigen_FindAll($scratchRowVec,1)
					$lastlist2=@extended-1

					For $lc1=0 To $lastlist2-1
						_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list2,$lc1,0),$value1b0)
					Next
					_Eigen_ReleaseMatrix($list2)
				EndIf

				; repeat for second value
				_Eigen_CwiseLogicalOp($candidate[$value2b0],"and",$buddymask[$rc],$scratchRowVec)
				_Eigen_WriteMatrixValue($scratchRowVec,0,$cell1,0)	; punch out our pair
				_Eigen_WriteMatrixValue($scratchRowVec,0,$cell2,0)

				If Not _Eigen_IsZero($scratchRowVec) Then
					$list2=_Eigen_FindAll($scratchRowVec,1)
					$lastlist2=@extended-1

					For $lc1=0 To $lastlist2-1
						_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list2,$lc1,0),$value2b0)
					Next
					_Eigen_ReleaseMatrix($list2)
				EndIf

			Next
		Next
		_Eigen_ReleaseMatrix($list)
	Next

	If $reduced>$prevred Then
		If _UpdateCandidatesPermuts()=False Then Return False
		If $verbose Then ConsoleWrite(@TAB & "NakedPair reductions in "&$label&": " & $reduced-$prevred & @CRLF)
	EndIf

	Return True
EndFunc


Func _NakedTriplets()
; FIND three cells in a single unit that all have the same three possible values
; REMOVE the same three candidate values from all remaining cells in the unit

	If _StoreMaskOfCellsWithNvalues($cellRowVec,3,3)=False Then Return True

	If _NakedTripletUnit($rowbuddymask,"rows")=False Then Return False
	If _NakedTripletUnit($colbuddymask,"cols")=False Then Return False
	If _NakedTripletUnit($blockbuddymask,"blocks")=False Then Return False

	Return True
EndFunc


Func _NakedTripletUnit(ByRef $buddymask,$label)
; expects global $cellRowVec pre-filled

	Local $prevred=$reduced
	Local $list,$list2,$value1b0,$value2b0,$value3b0,$cell1,$cell2,$cell3,$block,$lastlist,$lastlist2

	For $rc=0 To 8
		_Eigen_CwiseLogicalOp($cellRowVec,"and",$buddymask[$rc],$retainRowVec)
		If _Eigen_GetSum($retainRowVec)<3 Then ContinueLoop

		$list=_Eigen_FindAll($retainRowVec,1)
		$lastlist=@extended-1

		For $pc1=0 To $lastlist-2
			$cell1=_Eigen_ReadMatrixValue($list,$pc1,0)

			For $pc2=$pc1+1 To $lastlist-1
				$cell2=_Eigen_ReadMatrixValue($list,$pc2,0)
				_Eigen_CwiseBinaryOp_ColCol($candidates,$cell1,$cell2,"-",$candidates,$valuecolvec)
				If Not _Eigen_IsZero($valuecolvec) Then ContinueLoop

				For $pc3=$pc2+1 To $lastlist
					$cell3=_Eigen_ReadMatrixValue($list,$pc3,0)
					_Eigen_CwiseBinaryOp_ColCol($candidates,$cell1,$cell3,"-",$candidates,$valuecolvec)
					If Not _Eigen_IsZero($valuecolvec) Then ContinueLoop

					; get the values
					_Eigen_Copy_Acol_ToBcol($candidates,$valuecolvec,$cell1,0)
					$value1b0=_Eigen_GetRow_MaxVal($valuecolvec)
					_Eigen_WriteMatrixValue($valuecolvec,$value1b0,0,0)	; punchout for next maxval
					$value2b0=_Eigen_GetRow_MaxVal($valuecolvec)
					_Eigen_WriteMatrixValue($valuecolvec,$value2b0,0,0)	; punchout for next maxval
					$value3b0=_Eigen_GetRow_MaxVal($valuecolvec)

					; find all OTHER cells in current unit (so excluding cell 1/2/3)
					_Eigen_CwiseLogicalOp($candidate[$value1b0],"and",$buddymask[$rc],$scratchRowVec)

					_Eigen_WriteMatrixValue($scratchRowVec,0,$cell1,0)	; punch out our triplet
					_Eigen_WriteMatrixValue($scratchRowVec,0,$cell2,0)
					_Eigen_WriteMatrixValue($scratchRowVec,0,$cell3,0)

					If Not _Eigen_IsZero($scratchRowVec) Then
						$list2=_Eigen_FindAll($scratchRowVec,1)
						$lastlist2=@extended-1

						For $lc1=0 To $lastlist2-1
							_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list2,$lc1,0),$value1b0)
						Next
						_Eigen_ReleaseMatrix($list2)
					EndIf

					; repeat for second value
					_Eigen_CwiseLogicalOp($candidate[$value2b0],"and",$buddymask[$rc],$scratchRowVec)

					_Eigen_WriteMatrixValue($scratchRowVec,0,$cell1,0)	; punch out our triplet
					_Eigen_WriteMatrixValue($scratchRowVec,0,$cell2,0)
					_Eigen_WriteMatrixValue($scratchRowVec,0,$cell3,0)

					If Not _Eigen_IsZero($scratchRowVec) Then
						$list2=_Eigen_FindAll($scratchRowVec,1)
						$lastlist2=@extended-1

						For $lc1=0 To $lastlist2-1
							_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list2,$lc1,0),$value2b0)
						Next
						_Eigen_ReleaseMatrix($list2)
					EndIf

					; repeat for third value
					_Eigen_CwiseLogicalOp($candidate[$value3b0],"and",$buddymask[$rc],$scratchRowVec)

					_Eigen_WriteMatrixValue($scratchRowVec,0,$cell1,0)	; punch out our triplet
					_Eigen_WriteMatrixValue($scratchRowVec,0,$cell2,0)
					_Eigen_WriteMatrixValue($scratchRowVec,0,$cell3,0)

					If Not _Eigen_IsZero($scratchRowVec) Then
						$list2=_Eigen_FindAll($scratchRowVec,1)
						$lastlist2=@extended-1

						For $lc1=0 To $lastlist2-1
							_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list2,$lc1,0),$value2b0)
						Next
						_Eigen_ReleaseMatrix($list2)
					EndIf
				Next
			Next
		Next
		_Eigen_ReleaseMatrix($list)
	Next

	If $reduced>$prevred Then
		If _UpdateCandidatesPermuts()=False Then Return False
		If $verbose Then ConsoleWrite(@TAB & "NakedTriplets reductions in "&$label&": " & $reduced-$prevred & @CRLF)
	EndIf

	Return True
EndFunc


Func _NakedQuads()
; FIND four cells in a single unit that all have the same four possible values
; REMOVE the same four candidate values from all remaining cells in the unit

	If _StoreMaskOfCellsWithNvalues($cellRowVec,4,4)=False Then Return True

	If _NakedQuadUnit($rowbuddymask,"rows")=False Then Return False
	If _NakedQuadUnit($colbuddymask,"cols")=False Then Return False
	If _NakedQuadUnit($blockbuddymask,"blocks")=False Then Return False

	Return True
EndFunc


Func _NakedQuadUnit(ByRef $buddymask,$label)
; expects global $cellRowVec pre-filled

	Local $prevred=$reduced
	Local $list,$list2,$value1b0,$value2b0,$value3b0,$value4b0,$cell1,$cell2,$cell3,$cell4,$block,$lastlist,$lastlist2

	For $rc=0 To 8
		_Eigen_CwiseLogicalOp($cellRowVec,"and",$buddymask[$rc],$retainRowVec)
		If _Eigen_GetSum($retainRowVec)<4 Then ContinueLoop

		$list=_Eigen_FindAll($retainRowVec,1)
		$lastlist=@extended-1
		For $pc1=0 To $lastlist-3
			$cell1=_Eigen_ReadMatrixValue($list,$pc1,0)

			For $pc2=$pc1+1 To $lastlist-2
				$cell2=_Eigen_ReadMatrixValue($list,$pc2,0)
				_Eigen_CwiseBinaryOp_ColCol($candidates,$cell1,$cell2,"-",$candidates,$valuecolvec)
				If Not _Eigen_IsZero($valuecolvec) Then ContinueLoop

				For $pc3=$pc2+1 To $lastlist-1
					$cell3=_Eigen_ReadMatrixValue($list,$pc3,0)
					_Eigen_CwiseBinaryOp_ColCol($candidates,$cell1,$cell3,"-",$candidates,$valuecolvec)
					If Not _Eigen_IsZero($valuecolvec) Then ContinueLoop

					For $pc4=$pc3+1 To $lastlist
						$cell4=_Eigen_ReadMatrixValue($list,$pc4,0)
						_Eigen_CwiseBinaryOp_ColCol($candidates,$cell1,$cell4,"-",$candidates,$valuecolvec)
						If Not _Eigen_IsZero($valuecolvec) Then ContinueLoop

						; get the values
						_Eigen_Copy_Acol_ToBcol($candidates,$valuecolvec,$cell1,0)
						$value1b0=_Eigen_GetRow_MaxVal($valuecolvec)
						_Eigen_WriteMatrixValue($valuecolvec,$value1b0,0,0)
						$value2b0=_Eigen_GetRow_MaxVal($valuecolvec)
						_Eigen_WriteMatrixValue($valuecolvec,$value2b0,0,0)
						$value3b0=_Eigen_GetRow_MaxVal($valuecolvec)
						_Eigen_WriteMatrixValue($valuecolvec,$value3b0,0,0)
						$value4b0=_Eigen_GetRow_MaxVal($valuecolvec)

						; find all OTHER cells in current unit (so excluding cell 1/2/3/4)
						_Eigen_CwiseLogicalOp($candidate[$value1b0],"and",$buddymask[$rc],$scratchRowVec)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell1,0)	; punch out our quad
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell2,0)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell3,0)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell4,0)

						If Not _Eigen_IsZero($scratchRowVec) Then
							$list2=_Eigen_FindAll($scratchRowVec,1)
							$lastlist2=@extended-1

							For $lc1=0 To $lastlist2-1
								_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list2,$lc1,0),$value1b0)
							Next
							_Eigen_ReleaseMatrix($list2)
						EndIf

						; repeat for second value
						_Eigen_CwiseLogicalOp($candidate[$value2b0],"and",$buddymask[$rc],$scratchRowVec)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell1,0)	; punch out our quad
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell2,0)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell3,0)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell4,0)

						If Not _Eigen_IsZero($scratchRowVec) Then
							$list2=_Eigen_FindAll($scratchRowVec,1)
							$lastlist2=@extended-1

							For $lc1=0 To $lastlist2-1
								_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list2,$lc1,0),$value2b0)
							Next
							_Eigen_ReleaseMatrix($list2)
						EndIf

						; repeat for third value
						_Eigen_CwiseLogicalOp($candidate[$value3b0],"and",$buddymask[$rc],$scratchRowVec)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell1,0)	; punch out our quad
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell2,0)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell3,0)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell4,0)

						If Not _Eigen_IsZero($scratchRowVec) Then
							$list2=_Eigen_FindAll($scratchRowVec,1)
							$lastlist2=@extended-1

							For $lc1=0 To $lastlist2-1
								_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list2,$lc1,0),$value2b0)
							Next
							_Eigen_ReleaseMatrix($list2)
						EndIf

						; repeat for fourth value
						_Eigen_CwiseLogicalOp($candidate[$value4b0],"and",$buddymask[$rc],$scratchRowVec)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell1,0)	; punch out our quad
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell2,0)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell3,0)
						_Eigen_WriteMatrixValue($scratchRowVec,0,$cell4,0)

						If Not _Eigen_IsZero($scratchRowVec) Then
							$list2=_Eigen_FindAll($scratchRowVec,1)
							$lastlist2=@extended-1

							For $lc1=0 To $lastlist2-1
								_PunchOutCellCandidate(_Eigen_ReadMatrixValue($list2,$lc1,0),$value2b0)
							Next
							_Eigen_ReleaseMatrix($list2)
						EndIf

					Next
				Next
			Next
		Next
		_Eigen_ReleaseMatrix($list)
	Next

	If $reduced>$prevred Then
		If _UpdateCandidatesPermuts()=False Then Return False
		If $verbose Then ConsoleWrite(@TAB & "NakedQuads reductions in "&$label&": " & $reduced-$prevred & @CRLF)
	EndIf

	Return True
EndFunc


Func _HiddenPairs()
; FIND two cells in a single unit that are the only ones that have
; some or all of the same two possible values (but they can have other values in addition)
; REMOVE all other values from these cells

	Local $prevred=$reduced

	For $dc=0 To 7
		For $dc2=$dc+1 To 8

			; create cell-candidate mask
			_Eigen_SetZero($valuecolvec)
			_Eigen_WriteMatrixValue($valuecolvec,$dc,0,1)
			_Eigen_WriteMatrixValue($valuecolvec,$dc2,0,1)

			_Eigen_CwiseLogicalOp($candidate[$dc],"or",$candidate[$dc2],$cellRowVec)

			If _HiddenPairUnit($rowbuddymask,$dc,$dc2)=False Then Return False
			If _HiddenPairUnit($colbuddymask,$dc,$dc2)=False Then Return False
			If _HiddenPairUnit($blockbuddymask,$dc,$dc2)=False Then Return False
		Next
	Next

	If $reduced>$prevred Then
		If _UpdateCandidatesPermuts()=False Then Return False
		If $verbose Then ConsoleWrite(@TAB & "HiddenPairs reductions: " & $reduced-$prevred & @CRLF)
	EndIf

	Return True
EndFunc


Func _HiddenPairUnit(ByRef $buddymask,$value1b0,$value2b0)
; expects global $cellRowVec to mask the two cells with the two values

	Local $prevred=$reduced
	Local $cell1,$cell2

	For $rc=0 To 8
		_Eigen_CwiseLogicalOp($cellRowVec,"and",$buddymask[$rc],$retainRowVec)
		If _Eigen_GetSum($retainRowVec)<>2 Then ContinueLoop

		$cell1=_Eigen_GetCol_MaxVal($retainRowVec)
		_Eigen_WriteMatrixValue($retainRowVec,0,$cell1,0)
		$cell2=_Eigen_GetCol_MaxVal($retainRowVec)

		; ensure all parsed values are candidates here
		_Eigen_CwiseLogicalOp_ColCol($candidates,$cell1,$cell2,"or",$candidates,$valuecolvec2)
		_Eigen_CwiseLogicalOp_InPlace($valuecolvec2,"and",$valuecolvec)
		If _Eigen_GetSum($valuecolvec2)<2 Then ContinueLoop

		For $vc=0 To 8
			If $vc=$value1b0 Or $vc=$value2b0 Then ContinueLoop

			If _Eigen_ReadMatrixValue($candidate[$vc],0,$cell1)=1 Then _PunchOutCellCandidate($cell1,$vc)
			If _Eigen_ReadMatrixValue($candidate[$vc],0,$cell2)=1 Then _PunchOutCellCandidate($cell2,$vc)
		Next
	Next

	If $reduced>$prevred Then Return _UpdateCandidatesPermuts()

	Return True
EndFunc


Func _HiddenTriplets()
; FIND three cells in a single unit that are the only ones that have
; the same three possible values (but they can have other values in addition)
; REMOVE any additional values from these three cells

	Local $prevred=$reduced

	For $dc=0 To 6
		For $dc2=$dc+1 To 7
			_Eigen_CwiseLogicalOp($candidate[$dc],"or",$candidate[$dc2],$cellRowVec2)

			For $dc3=$dc2+1 To 8
				_Eigen_CwiseLogicalOp($cellRowVec2,"or",$candidate[$dc3],$cellrowvec)

				; create cell-candidate mask
				_Eigen_SetZero($valuecolvec)
				_Eigen_WriteMatrixValue($valuecolvec,$dc,0,1)
				_Eigen_WriteMatrixValue($valuecolvec,$dc2,0,1)
				_Eigen_WriteMatrixValue($valuecolvec,$dc3,0,1)

				If _HiddenTripletUnit($rowbuddymask,$dc,$dc2,$dc3)=False Then Return False
				If _HiddenTripletUnit($colbuddymask,$dc,$dc2,$dc3)=False Then Return False
				If _HiddenTripletUnit($blockbuddymask,$dc,$dc2,$dc3)=False Then Return False
			Next
		Next
	Next

	If $verbose And $reduced>$prevred Then ConsoleWrite(@TAB & "HiddenTriplets reductions: " & $reduced-$prevred & @CRLF)

	Return True
EndFunc


Func _HiddenTripletUnit(ByRef $buddymask,$value1b0,$value2b0,$value3b0)
; expects global $cellRowVec to mask the three cells with any of the three values

	Local $prevred=$reduced
	Local $cell1,$cell2,$cell3

	For $rc=0 To 8
		_Eigen_CwiseLogicalOp($cellRowVec,"and",$buddymask[$rc],$retainRowVec)
		If _Eigen_GetSum($retainRowVec)<>3 Then ContinueLoop

		$cell1=_Eigen_GetCol_MaxVal($retainRowVec)
		_Eigen_WriteMatrixValue($retainRowVec,0,$cell1,0)
		$cell2=_Eigen_GetCol_MaxVal($retainRowVec)
		_Eigen_WriteMatrixValue($retainRowVec,0,$cell2,0)
		$cell3=_Eigen_GetCol_MaxVal($retainRowVec)

		; ensure all parsed values are candidates here
		_Eigen_CwiseLogicalOp_ColCol($candidates,$cell1,$cell2,"or",$candidates,$valuecolvec2)
		_Eigen_Copy_ACol_ToBCol($candidates,$valuecolvec3,$cell3,0)
		_Eigen_CwiseLogicalOp_InPlace($valuecolvec2,"or",$valuecolvec3)
		_Eigen_CwiseLogicalOp_InPlace($valuecolvec2,"and",$valuecolvec)
		If _Eigen_GetSum($valuecolvec2)<3 Then ContinueLoop

		For $vc=0 To 8
			If $vc=$value1b0 Or $vc=$value2b0 Or $vc=$value3b0 Then ContinueLoop

			If _Eigen_ReadMatrixValue($candidate[$vc],0,$cell1)=1 Then _PunchOutCellCandidate($cell1,$vc)
			If _Eigen_ReadMatrixValue($candidate[$vc],0,$cell2)=1 Then _PunchOutCellCandidate($cell2,$vc)
			If _Eigen_ReadMatrixValue($candidate[$vc],0,$cell3)=1 Then _PunchOutCellCandidate($cell3,$vc)
		Next
	Next

	If $reduced>$prevred Then Return _UpdateCandidatesPermuts()

	Return True
EndFunc


Func _HiddenQuads()
; FIND four cells in a single unit that are the only ones that have
; some or all of the same four possible values (but they can have other values in addition)
; REMOVE any additional values from these four cells

	Local $prevred=$reduced

	For $dc=0 To 5
		For $dc2=$dc+1 To 6
			_Eigen_CwiseLogicalOp($candidate[$dc],"or",$candidate[$dc2],$cellRowVec3)

			For $dc3=$dc2+1 To 7
				_Eigen_CwiseLogicalOp($cellRowVec3,"or",$candidate[$dc3],$cellrowvec2)

				For $dc4=$dc3+1 To 8
					_Eigen_CwiseLogicalOp($cellRowVec2,"or",$candidate[$dc4],$cellrowvec)

					; create cell-candidate mask
					_Eigen_SetZero($valuecolvec)
					_Eigen_WriteMatrixValue($valuecolvec,$dc,0,1)
					_Eigen_WriteMatrixValue($valuecolvec,$dc2,0,1)
					_Eigen_WriteMatrixValue($valuecolvec,$dc3,0,1)
					_Eigen_WriteMatrixValue($valuecolvec,$dc4,0,1)

					If _HiddenQuadUnit($rowbuddymask,$dc,$dc2,$dc3,$dc4)=False Then Return False
					If _HiddenQuadUnit($colbuddymask,$dc,$dc2,$dc3,$dc4)=False Then Return False
					If _HiddenQuadUnit($blockbuddymask,$dc,$dc2,$dc3,$dc4)=False Then Return False
				Next
			Next
		Next
	Next

	If $verbose And $reduced>$prevred Then ConsoleWrite(@TAB & "HiddenQuads reductions: " & $reduced-$prevred & @CRLF)

	Return True
EndFunc


Func _HiddenQuadUnit(ByRef $buddymask,$value1b0,$value2b0,$value3b0,$value4b0)
; expects global $cellRowVec to mask the four cells with any of the four values

	Local $prevred=$reduced
	Local $cell1,$cell2,$cell3,$cell4

	For $rc=0 To 8
		_Eigen_CwiseLogicalOp($cellRowVec,"and",$buddymask[$rc],$retainRowVec)
		If _Eigen_GetSum($retainRowVec)<>4 Then ContinueLoop

		$cell1=_Eigen_GetCol_MaxVal($retainRowVec)
		_Eigen_WriteMatrixValue($retainRowVec,0,$cell1,0)
		$cell2=_Eigen_GetCol_MaxVal($retainRowVec)
		_Eigen_WriteMatrixValue($retainRowVec,0,$cell2,0)
		$cell3=_Eigen_GetCol_MaxVal($retainRowVec)
		_Eigen_WriteMatrixValue($retainRowVec,0,$cell3,0)
		$cell4=_Eigen_GetCol_MaxVal($retainRowVec)

		; ensure all parsed values are candidates here
		_Eigen_CwiseLogicalOp_ColCol($candidates,$cell1,$cell2,"or",$candidates,$valuecolvec2)
		_Eigen_CwiseLogicalOp_ColCol($candidates,$cell3,$cell4,"or",$candidates,$valuecolvec3)
		_Eigen_CwiseLogicalOp_InPlace($valuecolvec2,"or",$valuecolvec3)
		_Eigen_CwiseLogicalOp_InPlace($valuecolvec2,"and",$valuecolvec)
		If _Eigen_GetSum($valuecolvec2)<4 Then ContinueLoop

		For $vc=0 To 8
			If $vc=$value1b0 Or $vc=$value2b0 Or $vc=$value3b0 Or $vc=$value4b0 Then ContinueLoop

			If _Eigen_ReadMatrixValue($candidate[$vc],0,$cell1)=1 Then _PunchOutCellCandidate($cell1,$vc)
			If _Eigen_ReadMatrixValue($candidate[$vc],0,$cell2)=1 Then _PunchOutCellCandidate($cell2,$vc)
			If _Eigen_ReadMatrixValue($candidate[$vc],0,$cell3)=1 Then _PunchOutCellCandidate($cell3,$vc)
			If _Eigen_ReadMatrixValue($candidate[$vc],0,$cell4)=1 Then _PunchOutCellCandidate($cell4,$vc)
		Next
	Next

	If $reduced>$prevred Then Return _UpdateCandidatesPermuts()

	Return True
EndFunc


Func _InteractionBlockBlock()
; If any number of candidates for a single value in two blocks AB in same blockrow/blockcol
; occur ONLY in the same two cell rows/cols
; REMOVE this candidate from block C in the same two cell rows/cols

	Local $prevred=$reduced
	Local $before,$blockA,$blockB,$list,$listunique,$listunique2

	For $dc=0 To 8	; single value loop
		_Eigen_Copy_A_ToB($candidate[$dc],$valueRowVec)	; take a candidate value

		For $bc=0 To 8	; target cells in two blocks in the same block row/col ($bc=blockC)
			$blockA=$blockneighbours[$bc][0]	; block row neighbours
			$blockB=$blockneighbours[$bc][1]

			_Eigen_CwiseLogicalOp($blockbuddymask[$blockA],"or",$blockbuddymask[$blockB],$cellRowVec)
			_Eigen_CwiseLogicalOp_InPlace($cellRowVec,"and",$valueRowVec)	; filter for our value

			If _Eigen_GetSum($cellRowVec)>1 Then
				$list=_Eigen_CreateVector_FromAcell_Mask($blockindexRowVec,$cellRowVec)
				$listunique2=_Eigen_Sort_Unique($list)	; returns a colvector

				If @extended=2 Then	; found in both blocks
					_Eigen_ReleaseMatrix($list)
					$list=_Eigen_CreateVector_FromAcell_Mask($rowindexRowVec,$cellRowVec)
					$listunique=_Eigen_Sort_Unique($list)	; returns a colvector of unique rows

					If @extended=2 Then	; two rows found
						_PunchOutInBlockRow($bc,_Eigen_ReadMatrixValue($listunique,0,0),$dc)
						_PunchOutInBlockRow($bc,_Eigen_ReadMatrixValue($listunique,1,0),$dc)
					EndIf

					_Eigen_ReleaseMatrix($listunique2)
				EndIf

				_Eigen_ReleaseMatrix($list)
				_Eigen_ReleaseMatrix($listunique)
			EndIf

			$blockA=$blockneighbours[$bc][2]	; block col neighbours
			$blockB=$blockneighbours[$bc][3]

			_Eigen_CwiseLogicalOp($blockbuddymask[$blockA],"or",$blockbuddymask[$blockB],$cellRowVec)
			_Eigen_CwiseLogicalOp_InPlace($cellRowVec,"and",$valueRowVec)

			If _Eigen_GetSum($cellRowVec)>1 Then
				$list=_Eigen_CreateVector_FromAcell_Mask ($blockindexRowVec,$cellRowVec)
				$listunique2=_Eigen_Sort_Unique($list)	; returns a colvector

				If @extended=2 Then		; found in both blocks
					_Eigen_ReleaseMatrix($list)
					$list=_Eigen_CreateVector_FromAcell_Mask ($colindexRowVec,$cellRowVec)
					$listunique=_Eigen_Sort_Unique($list)	; returns a colvector

					If @extended=2 Then	; two cols found
						_PunchOutInBlockCol($bc,_Eigen_ReadMatrixValue($listunique,0,0),$dc)
						_PunchOutInBlockCol($bc,_Eigen_ReadMatrixValue($listunique,1,0),$dc)
					EndIf
					_Eigen_ReleaseMatrix($listunique2)
				EndIf

				_Eigen_ReleaseMatrix($list)
				_Eigen_ReleaseMatrix($listunique)
			EndIf
		Next
	Next

	If $reduced>$prevred Then
		If _UpdateCandidatesPermuts()=False Then Return False
		If $verbose Then ConsoleWrite(@TAB & "InteractionBlockBlock reductions: " & $reduced-$prevred & @CRLF)
	EndIf

	Return True
EndFunc


Func _IntersectionRemoval()
;FIND any one value occuring 2-3x in the overlap of two units, and nowhere else in one unit of the two.
;REMOVE that number from elsewhere in the other unit of the two.

	Local $prevred=$reduced
	Local $blocktotal,$rowtotal,$coltotal

	For $dc=0 To 8
		_Eigen_Copy_A_ToB($candidate[$dc],$valueRowVec)	; take a candidate value mask

		Local $sum
		For $bc=0 To 8
			_Eigen_CwiseLogicalOp($blockbuddymask[$bc],"and",$valueRowVec,$cellRowVec)
			$blocktotal=_Eigen_GetSum($cellRowVec)	; mask for value in block
			if $blocktotal<2 Then ContinueLoop

			For $rc=0 To 8	; count remaining overlap of block, row, and value masks
				_Eigen_CwiseLogicalOp($blockbuddymask[$bc],"and",$rowbuddymask[$rc],$cellRowVec2)		; mask for overlap block/row
				if _Eigen_IsZero($cellRowVec2) Then ContinueLoop	; no block/row overlap

				_Eigen_CwiseLogicalOp_InPlace($cellRowVec2,"and",$valueRowVec)		; mask for value in overlap block/row
				$overlaptotal=_Eigen_GetSum($cellRowVec2)
				If Abs($overlaptotal-2.5)<>.5 Then ContinueLoop	; 2/3 hits in overlap?

				_Eigen_CwiseLogicalOp($rowbuddymask[$rc],"and",$valueRowVec,$cellRowVec3) 	; mask for value in row
				$rowtotal=_Eigen_GetSum($cellRowVec3)

				If ($blocktotal>$overlaptotal And $rowtotal=$overlaptotal) Then
					_Eigen_CwiseLogicalOp($cellRowVec,"xcl",$cellRowVec2,$scratchRowVec)
					_PunchOutValueForMaskedCells($dc,$scratchRowVec)

				ElseIf ($blocktotal=$overlaptotal And $rowtotal>$overlaptotal) Then
					_Eigen_CwiseLogicalOp_InPlace($cellRowVec3,"xcl",$cellRowVec2)
					_PunchOutValueForMaskedCells($dc,$cellRowVec3)
				EndIf
			Next

			For $cc=0 To 8
				_Eigen_CwiseLogicalOp($blockbuddymask[$bc],"and",$colbuddymask[$cc],$cellRowVec2)
				if _Eigen_IsZero($cellRowVec2) Then ContinueLoop	; no block/col overlap

				_Eigen_CwiseLogicalOp_InPlace($cellRowVec2,"and",$valueRowVec)
				$overlaptotal=_Eigen_GetSum($cellRowVec2)
				If Abs($overlaptotal-2.5)<>.5 Then ContinueLoop	; 2/3 hits in overlap?

				_Eigen_CwiseLogicalOp($colbuddymask[$cc],"and",$valueRowVec,$cellRowVec3)
				$coltotal=_Eigen_GetSum($cellRowVec3)

				If ($blocktotal>$overlaptotal And $coltotal=$overlaptotal) Then
					_Eigen_CwiseLogicalOp($cellRowVec,"xcl",$cellRowVec2,$scratchRowVec)
					_PunchOutValueForMaskedCells($dc,$scratchRowVec)

				ElseIf ($blocktotal=$overlaptotal And $coltotal>$overlaptotal) Then
					_Eigen_CwiseLogicalOp_InPlace($cellRowVec3,"xcl",$cellRowVec2)
					_PunchOutValueForMaskedCells($dc,$cellRowVec3)
				EndIf
			Next
		Next
	Next

	If $reduced>$prevred Then
		If _UpdateCandidatesPermuts()=False Then Return False
		If $verbose Then ConsoleWrite(@TAB & "IntersectionRemoval reductions: " & $reduced-$prevred & @CRLF)
	EndIf

	Return True
EndFunc


Func _Xwing()
; FIND in two rows exactly two cells each containing the same digit candidate
; (that is, no other cells in either row should have this candidate)
; If these two cells occur in the same two cols, so the 4 cells are crosswise buddies (same 2 rows, same 2 cols)
; REMOVE all OTHER occurrences of this value in the same COLS
; repeat for cols

	Local $prevred=$reduced
	Local $row1,$row2,$col1,$col2,$list,$list2

	For $dc=0 To 8
		_Eigen_Copy_A_ToB($candidate[$dc],$valueRowVec)	; take a candidate value mask

		; reshape temporarily to sum candidates rowwise & colwise
		_Eigen_Redim_ExistingMatrix($valueRowVec,9,9)
		_Eigen_MatrixSpecs_Rowwise_Single($valueRowVec,2,$valuecolvec)		; 2=sum
		_Eigen_MatrixSpecs_Colwise_Single($valueRowVec,2,$valuecolvec2)	; 2=sum
		_Eigen_Redim_ExistingMatrix($valueRowVec,1,81)

		; do exactly two rows (or two cols) each have two cells with target value? if not, do next value
		If _Eigen_ConditCount($valuecolvec,"==",2)<>2 And _
			_Eigen_ConditCount($valuecolvec2,"==",2)<>2 Then ContinueLoop

		; filter for targets present
		If _Eigen_ConditCount($valuecolvec,"==",2)=2 Then	; corner rows detected

			; get the two row indices
			_Eigen_ConditMask_InPlace($valuecolvec,"==",2)
			$row1=_Eigen_GetRow_MaxVal($valuecolvec)

			_Eigen_WriteMatrixValue($valuecolvec,$row1,0,0)
			$row2=_Eigen_GetRow_MaxVal($valuecolvec)

			; mask all cells in row1/row2 that have candidate value $dc, store in $scratchRowVec
			_Eigen_CwiseLogicalOp($rowbuddymask[$row1],"or",$rowbuddymask[$row2],$scratchRowVec)
			_Eigen_CwiseLogicalOp_InPlace($scratchRowVec,"and",$valueRowVec)	; filter for target value
			If _Eigen_GetSum($scratchRowVec)<>4 Then ContinueLoop

			_Eigen_Redim_ExistingMatrix($scratchRowVec,9,9)
			$list=_Eigen_FindAll($scratchRowVec,1)	; 4 entries
			_Eigen_Redim_ExistingMatrix($scratchRowVec,1,81)

			$list2=_Eigen_Sort_Unique_Col($list,2)	; col coordinate
			if @extended<>2 Then ContinueLoop

			$col1=_Eigen_ReadMatrixValue($list2,0,0)
			$col2=_Eigen_ReadMatrixValue($list2,1,0)
			_Eigen_CwiseLogicalOp($colbuddymask[$col1],"or",$colbuddymask[$col2],$cellRowVec)

			_Eigen_ReleaseMatrix($list)
			_Eigen_ReleaseMatrix($list2)

		Else	; corner cols detected
			_Eigen_ConditMask_InPlace($valuecolvec2,"==",2)
			$col1=_Eigen_GetRow_MaxVal($valuecolvec)
			_Eigen_WriteMatrixValue($valuecolvec,$col1,0,0)
			$col2=_Eigen_GetRow_MaxVal($valuecolvec)

			; mask all cells in col1/2 that have candidate value $dc in $scratchRowVec
			_Eigen_CwiseLogicalOp($colbuddymask[$col1],"or",$colbuddymask[$col2],$scratchRowVec)
			_Eigen_CwiseLogicalOp_InPlace($scratchRowVec,"and",$valueRowVec)
			If _Eigen_GetSum($scratchRowVec)<4 Then ContinueLoop

			_Eigen_Redim_ExistingMatrix($scratchRowVec,9,9)
			$list=_Eigen_FindAll($scratchRowVec,1)	; 4 cells
			_Eigen_Redim_ExistingMatrix($scratchRowVec,1,81)

			$list2=_Eigen_Sort_Unique_Col($list,1)	; row coordinates
			if @extended<>2 Then ContinueLoop

			$row1=_Eigen_ReadMatrixValue($list2,0,0)
			$row2=_Eigen_ReadMatrixValue($list2,1,0)
			_Eigen_CwiseLogicalOp($rowbuddymask[$row1],"or",$rowbuddymask[$row2],$cellRowVec)

			_Eigen_ReleaseMatrix($list)
			_Eigen_ReleaseMatrix($list2)
		EndIf

		; create final cell target mask
		_Eigen_CwiseLogicalOp_InPlace($cellRowVec,"and",$valueRowVec)	; filter for target value
		_Eigen_CwiseLogicalOp_InPlace($cellRowVec,"xcl",$scratchRowVec)	; exclude the four corner cells

		_PunchOutValueForMaskedCells($dc,$cellRowVec)
	Next

	If $reduced>$prevred Then
		If _UpdateCandidatesPermuts()=False Then Return False
		If $verbose Then ConsoleWrite(@TAB & "Xwing reductions: " & $reduced-$prevred & @CRLF)
	EndIf

	Return True
EndFunc


Func _XYZwing()
; FIND three cells that are 1. pairwise buddies, and
; 2. have 3,2,2 candidates out of 3: ABC, BC, AC.
; REMOVE C from all cells that share any unit with all three cells

	Local $prevred=$reduced
	Local $list,$list2,$list3,$lastlist2,$lastlist3
	Local $value1b0,$value2b0,$value3b0,$val,$ext

	$list3=_CreateListOfCellsWithNvalues($valueRowVec,3,1)
	if @error Or $list3=False then Return True
	$lastlist3=@extended-1

	$list2=_CreateListOfCellsWithNvalues($valueRowVec,2,2)
	if @error Or $list2=False then
		_Eigen_ReleaseMatrix($list3)
		Return True
	EndIf
	$lastlist2=@extended-1

	For $c1=0 To $lastlist3
		$c1cell=_Eigen_ReadMatrixValue($list3,$c1,0)
		$c1row=_Eigen_ReadMatrixValue($list3,$c1,1)
		$c1col=_Eigen_ReadMatrixValue($list3,$c1,2)
		$c1bl=_Eigen_ReadMatrixValue($matBlocks,$c1row,$c1col)

		_Eigen_WriteMatrixValue($listof7vals,0,0,_Eigen_ReadMatrixValue($list3,$c1,3))
		_Eigen_WriteMatrixValue($listof7vals,1,0,_Eigen_ReadMatrixValue($list3,$c1,4))
		_Eigen_WriteMatrixValue($listof7vals,6,0,_Eigen_ReadMatrixValue($list3,$c1,5))

		For $c2=0 To $lastlist2-1
			$c2cell=_Eigen_ReadMatrixValue($list2,$c2,0)
			$c2row=_Eigen_ReadMatrixValue($list2,$c2,1)
			$c2col=_Eigen_ReadMatrixValue($list2,$c2,2)
			$c2bl=_Eigen_ReadMatrixValue($matBlocks,$c2row,$c2col)

			; check: are c1/c2 pairwise buddies?
			$link12=(($c2row=$c1row) Or ($c2col=$c1col) Or ($c2bl=$c1bl))
			If Not $link12 Then ContinueLoop

			$c2val1=_Eigen_ReadMatrixValue($list2,$c2,3)
			$c2val2=_Eigen_ReadMatrixValue($list2,$c2,4)

			_Eigen_WriteMatrixValue($listof7vals,2,0,$c2val1)
			_Eigen_WriteMatrixValue($listof7vals,3,0,$c2val2)

			For $c3=$c2+1 To $lastlist2
				$c3cell=_Eigen_ReadMatrixValue($list2,$c3,0)
				$c3row=_Eigen_ReadMatrixValue($list2,$c3,1)
				$c3col=_Eigen_ReadMatrixValue($list2,$c3,2)
				$c3bl=_Eigen_ReadMatrixValue($matBlocks,$c3row,$c3col)

				; check: are c1/c3 pairwise buddies?
				$link13=(($c3row=$c1row) Or ($c3col=$c1col) Or ($c3bl=$c1bl))
				$link23=(($c3row=$c2row) Or ($c3col=$c2col) Or ($c3bl=$c2bl))
				If Not ($link13 And $link23) Then ContinueLoop

				$c3val1=_Eigen_ReadMatrixValue($list2,$c3,3)
				$c3val2=_Eigen_ReadMatrixValue($list2,$c3,4)
				If ($c3val1=$c2val1 And $c3val2=$c2val2) Or _
					($c3val1=$c2val2 And $c3val2=$c2val1) then ContinueLoop

				_Eigen_WriteMatrixValue($listof7vals,4,0,$c3val1)
				_Eigen_WriteMatrixValue($listof7vals,5,0,$c3val2)

				$list=_Eigen_Sort_Unique($listof7vals)
				$ext=@extended
				_Eigen_ReleaseMatrix($list)
				If $ext<>3 Then ContinueLoop	; early-out

				; identify value C (the single overlapping one in the two pairs)
				$val=(($c2val1=$c3val1 Or $c2val1=$c3val2)?($c2val1):($c2val2))

				; identify all cells that share any unit with all three cells
				_Eigen_CwiseLogicalOp($rowbuddymask[$c1row],"or",$colbuddymask[$c1col],$cellRowVec)
				_Eigen_CwiseLogicalOp_InPlace($cellRowVec,"or",$blockbuddymask[$c1bl])

				_Eigen_CwiseLogicalOp($rowbuddymask[$c2row],"or",$colbuddymask[$c2col],$cellRowVec2)
				_Eigen_CwiseLogicalOp_InPlace($cellRowVec2,"or",$blockbuddymask[$c2bl])
				_Eigen_CwiseLogicalOp_InPlace($cellRowVec,"and",$cellRowVec2)

				_Eigen_CwiseLogicalOp($rowbuddymask[$c3row],"or",$colbuddymask[$c3col],$cellRowVec2)
				_Eigen_CwiseLogicalOp_InPlace($cellRowVec2,"or",$blockbuddymask[$c3bl])
				_Eigen_CwiseLogicalOp_InPlace($cellRowVec,"and",$cellRowVec2)

				; exclude our three cells
				_Eigen_WriteMatrixValue($cellRowVec,0,$c1cell,0)
				_Eigen_WriteMatrixValue($cellRowVec,0,$c2cell,0)
				_Eigen_WriteMatrixValue($cellRowVec,0,$c3cell,0)

				If Not _Eigen_IsZero($cellRowVec) Then _PunchOutValueForMaskedCells($val,$cellRowVec)
			Next
		Next
	Next
	_Eigen_ReleaseMatrix($list2)
	_Eigen_ReleaseMatrix($list3)

	If $reduced>$prevred Then
		If _UpdateCandidatesPermuts()=False Then Return False
		If $verbose Then ConsoleWrite(@TAB & "XYZwing reductions: " & $reduced-$prevred & @CRLF)
	EndIf

	Return True
EndFunc


Func _Solver()
; returns True when exhausted due to 1. solution found, OR 2. permutation set emptied => abandon invalid configuration

	Local $solverpass=0
	$tsolver=TimerInit()	; global
	$blockfreqsum=-1		; global

	Do
		$reduced=0	; global
		$solverpass+=1
		If $verbose Then ConsoleWrite(@CRLF & "Starting Solver pass: " & $solverpass & @CRLF)

		If $testing Then ConsoleWrite("Solver analysing Single Values..." & @CRLF)
		If _NakedSingle()=False Then Return SetError(1,0,True)
		If _HiddenSingle()=False Then Return SetError(1,0,True)

		If $testing Then ConsoleWrite("Solver analysing Exposed Units..." & @CRLF)
		If _NakedPairs()=False Then Return SetError(1,0,True)
		If _NakedTriplets()=False Then Return SetError(1,0,True)
		If _NakedQuads()=False Then Return SetError(1,0,True)

		If $testing Then ConsoleWrite("Solver analysing Hidden Units..." & @CRLF)
		If _HiddenPairs()=False Then Return SetError(1,0,True)
		If _HiddenTriplets()=False Then Return SetError(1,0,True)
		If _HiddenQuads()=False Then Return SetError(1,0,True)

		If $testing Then ConsoleWrite("Solver analysing Intersections..." & @CRLF)
		If _InteractionBlockBlock()=False Then Return SetError(1,0,True)
		If _IntersectionRemoval()=False Then Return SetError(1,0,True)

		If $testing Then ConsoleWrite("Solver analysing Wings..." & @CRLF)
		If _Xwing()=False Then Return SetError(1,0,True)
		If _XYZwing()=False Then Return SetError(1,0,True)

		_CheckAllBlocks()	; does not return if single solution sought and found
		If @error Then Return SetError(1,0,True)	; pass error up the calling chain immediately
	Until $reduced=0	; no improvement = stop looping

	If $verbose Then ConsoleWrite(@CRLF)
	If $solutionsFound>0 Then _CloseDown()	; fail-safe exit

	Return False	; F = it did get through to the end, but unsuccessfully so far
EndFunc

;_____________________________________________________________________________
; BRUTE FORCE TRIAL AND ERROR
; guesses the contents of the six off-diagonal blocks in optimised order,
; thereby reducing permuts to unity or zero in the three diagonal blocks,
; without having to evaluate these at all.

Func _BruteForce()

	; create lists of all remaining candidates for all non-maxfreq blocks
	; lists for upper diagonal blocks
	Local $matTClist=_Eigen_FindAll($punchOut[$blockTC],1)
	Local $totalTC=_Eigen_GetMatrixRows($matTClist)

	Local $matTRlist=_Eigen_FindAll($punchOut[$blockTR],1)
	Local $totalTR=_Eigen_GetMatrixRows($matTRlist)

	Local $matCRlist=_Eigen_FindAll($punchOut[$blockCR],1)
	Local $totalCR=_Eigen_GetMatrixRows($matCRlist)


	; lists for lower diagonal blocks
	Local $matCLlist=_Eigen_FindAll($punchOut[$blockCL],1)
	Local $totalCL=_Eigen_GetMatrixRows($matCLlist)

	Local $matBLlist=_Eigen_FindAll($punchOut[$blockBL],1)
	Local $totalBL=_Eigen_GetMatrixRows($matBLlist)

	Local $matBClist=_Eigen_FindAll($punchOut[$blockBC],1)
	Local $totalBC=_Eigen_GetMatrixRows($matBClist)

	ConsoleWrite(@CRLF & @CRLF & "Starting Brute-Force..." & @CRLF & @CRLF)
	$tstart=TimerInit()


For $tcc=0 to $totalTC-1
	$TCpermutID=_Eigen_ReadMatrixValue($matTClist,$tcc,0)	; offset = permut ID

	_FillMasksOfValidNeighbours($TCpermutID,$blockTC)

	If $verbose Then
		ConsoleWrite(@CRLF & "Processing TC (block " & $blockTC & ")  permut " & $tcc+1 & " of " & $totalTC & @CRLF)
		ConsoleWrite(Round(100*$tcc/$totalTC,2)& "% done in " & round(TimerDiff($tstart)/60000,1)& " min." & @CRLF)
		If $exitUpon1stHit=False And $solutionsFound>0 Then ConsoleWrite("Solutions found so far: " & $solutionsFound & @CRLF)
		ConsoleWrite(@CRLF)
	EndIf


For $trc=0 to $totalTR-1
	$TRpermutID=_Eigen_ReadMatrixValue($matTRlist,$trc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validRowNeighbours[$blockTC],0,$TRpermutID)=0 Then ContinueLoop

	_FillMasksOfValidNeighbours($TRpermutID,$blockTR)

	_MaskValidPermutations($blockTL,$validRowNeighbours[$blockTC],$validRowNeighbours[$blockTR])
	If _Eigen_IsZero($matpermut[$blockTL]) Then ContinueLoop

	If $verbose Then
		ConsoleWrite(@CRLF & "Processing TC (block " & $blockTC & ")  permut " & $tcc+1 & " of " & $totalTC & @CRLF)
		ConsoleWrite("Processing TR (block " & $blockTR & ")  permut " & $trc+1 & " of " & $totalTR & @CRLF)
		ConsoleWrite(Round(((100*($tcc+($trc/$totalTR)))/$totalTC),2) & "% done in " & round(TimerDiff($tstart)/60000,1)& " min." & @CRLF)
	EndIf


For $clc=0 to $totalCL-1
	$CLpermutID=_Eigen_ReadMatrixValue($matCLlist,$clc,0)	; offset = permut ID
	If $testing Then ConsoleWrite(@TAB & "Processing CL (block " & $blockCL & ")  permut " & $clc+1 & " of " & $totalCL & @CRLF)

	_FillMasksOfValidNeighbours($CLpermutID,$blockCL)

	_MaskValidPermutations($blockTL,$validRowNeighbours[$blockTC],$validRowNeighbours[$blockTR],$validColNeighbours[$blockCL])
	If _Eigen_IsZero($matpermut[$blockTL]) Then ContinueLoop

	_MaskValidPermutations($blockCC,$validColNeighbours[$blockTC],$validRowNeighbours[$blockCL])
	If _Eigen_IsZero($matpermut[$blockCC]) Then ContinueLoop

	_MaskValidPermutations($blockCR,$validColNeighbours[$blockTR],$validRowNeighbours[$blockCL])
	If _Eigen_IsZero($matpermut[$blockCR]) Then ContinueLoop


For $blc=0 to $totalBL-1
	$BLpermutID=_Eigen_ReadMatrixValue($matBLlist,$blc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validColNeighbours[$blockCL],0,$BLpermutID)=0 Then ContinueLoop
	If $testing Then ConsoleWrite(@TAB & @TAB & "Processing BL (block " & $blockBL & ")  permut " & $blc+1 & " of " & $totalBL & @CRLF & @CRLF)

	_FillMasksOfValidNeighbours($BLpermutID,$blockBL)

	_MaskValidPermutations($blockTL, $validRowNeighbours[$blockTC],$validRowNeighbours[$blockTR],$validColNeighbours[$blockCL],$validColNeighbours[$blockBL])
	If _Eigen_GetSum($matpermut[$blockTL])<>1 Then ContinueLoop

	_MaskValidPermutations($blockBC,$validRowNeighbours[$blockBL],$validColNeighbours[$blockTC])
	If _Eigen_IsZero($matpermut[$blockBC]) Then ContinueLoop

	_MaskValidPermutations($blockBR,$validRowNeighbours[$blockBL],$validColNeighbours[$blockTR])
	If _Eigen_IsZero($matpermut[$blockBR]) Then ContinueLoop


For $crc=0 to $totalCR-1
	$CRpermutID=_Eigen_ReadMatrixValue($matCRlist,$crc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validRowNeighbours[$blockCL],0,$CRpermutID)=0 Then ContinueLoop
	If _Eigen_ReadMatrixValue($validColNeighbours[$blockTR],0,$CRpermutID)=0 Then ContinueLoop
	If $testing Then ConsoleWrite(@CRLF & "Processing CR (block " & $blockCR & ")  permut " & $crc+1 & " of " & $totalCR & @CRLF & @CRLF)

	_FillMasksOfValidNeighbours($CRpermutID,$blockCR)

	_MaskValidPermutations($blockCC,$validColNeighbours[$blockTC],$validRowNeighbours[$blockCL],$validRowNeighbours[$blockCR])
	If _Eigen_IsZero($matpermut[$blockCC]) Then ContinueLoop


For $bcc=0 to $totalBC-1
	$BCpermutID=_Eigen_ReadMatrixValue($matBClist,$bcc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validRowNeighbours[$blockBL],0,$BCpermutID)=0 Then ContinueLoop
	If _Eigen_ReadMatrixValue($validColNeighbours[$blockTC],0,$BCpermutID)=0 Then ContinueLoop
	If $testing Then ConsoleWrite(@CRLF & "Processing BC (block " & $blockBC & ")  permut " & $bcc+1 & " of " & $totalBC & @CRLF & @CRLF)

	_FillMasksOfValidNeighbours($BCpermutID,$blockBC)

	_MaskValidPermutations($blockBR,$validRowNeighbours[$blockBC],$validRowNeighbours[$blockBL],$validColNeighbours[$blockTR],$validColNeighbours[$blockCR])
	If _Eigen_GetSum($matpermut[$blockBR])<>1 Then ContinueLoop

	_MaskValidPermutations($blockCC,$validColNeighbours[$blockBC],$validColNeighbours[$blockTC],$validRowNeighbours[$blockCL],$validRowNeighbours[$blockCR])
	If _Eigen_GetSum($matpermut[$blockCC])<>1 Then ContinueLoop


	; update solution matrix
	If $verbose Then ConsoleWrite(@CRLF & "Checking this solution:" & @CRLF)

	_StoreBlock($blockTL,_Eigen_GetCol_MaxVal($matpermut[$blockTL]))
	_StoreBlock($blockTC,$TCpermutID)
	_StoreBlock($blockTR,$TRpermutID)

	_StoreBlock($blockCL,$CLpermutID)
	_StoreBlock($blockCC,_Eigen_GetCol_MaxVal($matpermut[$blockCC]))
	_StoreBlock($blockCR,$CRpermutID)

	_StoreBlock($blockBL,$BLpermutID)
	_StoreBlock($blockBC,$BCpermutID)
	_StoreBlock($blockBR,_Eigen_GetCol_MaxVal($matpermut[$blockBR]))

	_CheckSolution()
	If $verbose Then ConsoleWrite("Resuming scan..." & @CRLF)

Next
Next
Next

Next
Next
Next

EndFunc
