;============================================================================================
; Title ..........: BruteForceSudokuSolver
; AutoIt Version..: 3.3.14+
; Description ....: showcase for E4A bitmask handling
; Author..........: A.R.T. Jonkers (RTFC)
; Release.........: 1.1
; Latest revision.: 2 July 2020
; License.........: free for personal use; free distribution allowed provided
;							the author is credited; all other rights reserved.
; Tested on.......: W10Pro/64
; Dependencies....: Eigen4AutoIt (E4A) v4.8+
;============================================================================================
; Summary (Wotsit?)
;
; Eigen4AutoIt (E4A) provides extensive tools for producing and manipulating
; bitmasks, including conditional expressions and logical operands.
; This example script applies these to solve sudoku puzzles, which are
; mathematically simple, but can be computationally hard.
;
; It uses optimised brute-force, based upon a few tables and basic logic,
; applied to large bitmask matrices that track the remaining valid
; permutations for each 3x3 block.
;
; No guarantees, no warranties, no liability for damages, no refunds.
;	Use at your own risk.
;
;============================================================================================
;	Remarks
;
; * set flag $showGUI=True for a simple dynamic display of BF-cycling.
;	This SLOWS EVERYTHING DOWN to a crawl, so don't use it for timings!
;	Also, the e[X]it button is purely ornamental.
;
; * set flag $showGUI=False to write output only to console (much faster).
;	The solver will then use whichever selection 1-9 was last used (as stored in .ini file)
;
; * The number of updates sent to console can be controlled with flag
;	$verbose; writing less info = (marginally) faster processing.
;
; * Set flag $exitUpon1stHit to False to analyse the complete permutation
;	space and collect all solutions if there are several.
;	If $exitUpon1stHit=True (the default), the programme ends as soon as
;	the first valid solution is found.
;
; * Three tables (extension *.mat) are provided, but will be regenerated
;	if not found locally (slow, but done only once).
;
; * This script shows how E4A can be used to solve computationally hard
;	problems without requiring any advanced maths. Each permutations-related
;	operation evaluates over 362 thousand cases per single E4A call.
;	(NB: AutoIt E4A function calls parse matrix IDs only (integers), not data.)
;
; * block IDs are numbered Colwise base-0, so TL=0, BL=2, CC=4, TR=6, and BR=8 (unswapped).
;	T=Top, C=Centre, B=bottom, L=left, R=Right.
;
; * How it works:
;	1. create work environment (note how multiple matrices (that is, their integer indices) are stored/accessed in arrays)
;	2. read in the sudoku data and store the clues.
;	3. punch-in each clue, reducing the number of valid permutations per 3x3 block and all neighbouring blocks (in both directions)
;	4. call _OptimiseMax: Find the block with most remaining permutations and swap it to the Top-Left corner block (TL).
;		(Block rows and block cols can be swapped at will without affecting the solution.)
;	5. find the next block with most remaining permutations and swap it to the Bottom-Right corner block (BR).
;	6. the resulting matrix of block IDs has four possible orientations:
;		a. regular, b. regular, but transposed, c. with rows and cols reversed, and d. the latter, but transposed
;		Identify which of these four yields the smallest permutation space for
;		the two blocks in the outermost Brute-Force loops (TC & TR)
;	7. within this chosen TC x TR grid, sampling suggests that the solution most likely resides
;		on or near the reverse diagonal (that is, from bottom-left to top-right),
;		and on that axis most likely near or above the mid-point.
;		Re-order a joint listing of the remaining TC & TR permuts based on these likelihoods.
;	8. Brute-Force: cycle through all valid permutations in nested loops for
;		the six off-diagonal blocks, with early-outs for any block that has zero valid permuts left.
;		In the innermost loop, permutations of the three diagonal blocks are
;		thereby reduced to either 1 or 0. Evaluate and report if sudoku is valid.
;
; * the BF algorithm internally solves the sudoku variant with swapped blocks,
;	but the GUI displays the blocks in their original location (matching the clues).
;
; * Note in particular how the following functions are used:
;	+ _Eigen_ConditMask, to produce bitmasks;
;	+ _Eigen_CwiseLogicalOp with operator "xcl", as positive exclusion filter (1 = out, 0 = keep);
;	+ _Eigen_CwiseLogicalOp with operator "and", as negative exclusion filter (1 = keep, 0 = out);
;	+ _Eigen_CwiseLogicalOp with operator "or", to combine partially overlapping subsets;
;	+ _Eigen_ReDim_ExistingMatrix, to switch data representation between vector- and block-shaped;
;	+ _FillCandidValues, to multiply a LinSpaced_Colwise matrix ($candidvalues) with a bitmask ($candidates) to list remaining candidate values per cell (0=invalid);
;	+ _Eigen_Swap_Arow_Arow/_Eigen_Swap_Acol_Acol, to quickly re-arrange block references (NB the blocks themselves are untouched);
;	+ _FillMasksOfValidNeighbours, to create two subsets of 12,096 valid permuts, one horizontally and one vertically;
;	+ _MaskValidPermutations, to apply up to four valid-neighbour bitmasks to reduce a block permutation subset;
;	+ _Eigen_IsZero, as early-out for invalid block combinations;
;	+ _Eigen_GetSum, to determine how many valid permutations are left for a block;
;	+ _Eigen_GetCol_MaxVal, to retrieve the last-remaining valid permutation ID.
;
;============================================================================================
#NoTrayIcon
#include <GUIConstantsEx.au3>
#include <WinAPIError.au3>
#include ".\Eigen4AutoIt.au3"	; EDIT THIS PATH, IF NEED BE	<<<<

; number of permutations in a 3x3 block containing digits 1-9 once each
Global Const $permuts=362880	; 9! (9 factorial = 9 x 8 x 7 x 6 x 5 x 4 x 3 x 2 x 1)

; global matrices, multi-matrix arrays, and related vars
Global $sudoku[81]	; 1D array
Global $matSudoku,$solution,$solutionRow,$solutionCol,$solutionBlock
Global $candidates,$reduced,$blockneighbours,$blockfreqs,$blockfreqsum,$transp
Global $retainRowVec,$cellRowVec,$valueRowVec,$digitColvec,$digitRowvec,$valuecolvec,$scratchRowVec,$blockRowVec
Global $blockPermuts,$digitInRows,$digitInCols,$matDigitBuffer
Global $blockIDadjRow1,$blockIDadjRow2,$blockIDadjCol1,$blockIDadjCol2,$matBlockCells
Global $clueslist,$totalclues,$solutionsFound=0,$tstart=False,$selection=0

; block vars
Global $blockTL,$blockTC,$blockTR
Global $blockCL,$blockCC,$blockCR
Global $blockBL,$blockBC,$blockBR

; operational flags
Global $showGUI=True	;False		; T: (SLOW!) display partial solution in progress; F: don't
Global $exitUpon1stHit=True	; T: early-out; F: exhaustive search for all solutions
Global $verbose=True		; T: write status updates to console; F: don't

;_____________________________________________________________________________
; CREATE WORK ENVIRONMENT

SRandom(@AutoItPID+@MSEC)

; check E4A
If _Eigen_GetVersion<4.8 Then _FatalError("E4A version 4.8 or higher is required")
ConsoleWrite("Initialising BitMask Sudoku Solver..." & @CRLF & @CRLF)

; Beginning of Story...
_Eigen_StartUp("int")		; integers are most suitable for bitmask handling
_Eigen_SetLogicalBit0only()	; all CwiseLogicalOps are to act upon bit-0 only (for binary bitmask handling)
_Eigen_Hide_MatrixDisplay()	; ignore all _MatrixDisplay calls (console output only)
_Eigen_reSetVerbose()		; ignore all _MatrixDisplay calls (console output only)


; create blockID lookup table
$matBlockIDs=_Eigen_CreateMatrix_LinSpaced_Colwise(3,3,0,8)
$matBlocks=_Eigen_CreateMatrix_Zero(9,9)
$matBlockCells=_Eigen_CreateMatrix_Zero(9,9)

For $cc=0 To 2
	$startcol=$cc*3

	For $rc=0 To 2
		$startrow=$rc*3

		_Eigen_SetConstant_Block($matBlocks,$startrow,$startcol,3,3,$startcol+$rc)
		_Eigen_Copy_Ablock_ToBblock($matBlockIDs,$matBlockCells,0,0,3,3,$startrow,$startcol)
	Next
Next
_Eigen_ReDim_ExistingMatrix($matBlockCells,1,81)	; needed as rowvec only

; create block neighbour lookup table
Global $blockneighbours[9][4]

For $bc=0 To 8	; identify and store block's neighbours
	$blockneighbours[$bc][0]=Mod($bc+3,9)					; near row neighbour
	$blockneighbours[$bc][1]=Mod($bc+6,9)					; far row neighbour
	$blockneighbours[$bc][2]=(Int($bc/3)*3)+Mod($bc+1,3)	; near col neighbour
	$blockneighbours[$bc][3]=(Int($bc/3)*3)+Mod($bc+2,3)	; far col neighbour
Next

; create full permutation rowvector buffers (needed for creating/loading the file tables)
$matMrowvec=_Eigen_CreateMatrix(1,$permuts)
$matMcolvec=_Eigen_CreateMatrix(1,$permuts)	; "colvec" because the stored data relate to column indices (shape is rowvec)

; load the big lookup tables from file (regenerated if not found in local dir)
_LoadBlockPermutations(); 9 rows (storing digits 1-9, base-1) x 362880 cols (block permutations)
_LoadDigitTables()		; 3x9 rows (row c.q. col 0/1/2 for digits 1-9, for each block permutation)

; create a mirror for the digits buffer
$matDigitBuffer = _Eigen_CloneMatrix($digitInRows,False)

; create block permutation mask buffers
; these arrays merely contain matrix IDs (integers, indices to $MatrixList)
; illustrating how to select from multiple matrices depending on input
Global $punchout[9]
Global $matpermut[9]
Global $validRowNeighbours[9]
Global $validColNeighbours[9]

For $dc=0 to 8	; only matrix IDs are stored here, not matrix contents
	$punchOut[$dc]=_Eigen_CreateMatrix_Ones(1,$permuts)
	$matpermut[$dc]=_Eigen_CreateMatrix_Ones(1,$permuts)

	$validRowNeighbours[$dc]=_Eigen_CreateMatrix_Zero(1,$permuts)
	$validColNeighbours[$dc]=_Eigen_CreateMatrix_Zero(1,$permuts)
Next

; We allocate all matrices of known dimensions BEFORE we do any work,
; so we won't waste time and effort if memory turns out to be insufficient.
; If more memory is needed, increase the EIGEN_ALLOCATE_RAM_GB value in
; the local Eigen4AutoIt.ini and rerun in x64-mode.

; create punchout mirror for block freqs
$matPunchOut=_Eigen_CreateMatrix(9,$permuts)	; content is based upon clues
$savePunchOut=_Eigen_CreateMatrix(9,$permuts)	; for restoring after trial & error

; create all RowVec buffers
$cellRowVec=_Eigen_CreateMatrix (1,81)
$cellRowVec2=_Eigen_CreateMatrix (1,81)
$cellRowVec3=_Eigen_CreateMatrix (1,81)

$valueRowVec=_Eigen_CreateMatrix (1,81)
$valueRowVec2=_Eigen_CreateMatrix (1,81)

$retainRowVec=_Eigen_CreateMatrix (1,81)
$scratchRowVec=_Eigen_CreateMatrix (1,81)

; create index lookup tables
$blockindexRowVec=_Eigen_CreateMatrix (1,81)
$rowindexRowVec=_Eigen_CreateMatrix (1,81)
$colindexRowVec=_Eigen_CreateMatrix (1,81)

$digitRowvec=_Eigen_CreateMatrix (3*9,1)	; row data stored as colvec
$digitColvec=_Eigen_CreateMatrix (3*9,1)

; create solution buffers
$solution=_Eigen_CreateMatrix_Zero (9,9)
$solutionRow=_Eigen_CreateMatrix_Zero (1,9)
$solutionCol=_Eigen_CreateMatrix_Zero (9,1)
$solutionBlock=_Eigen_CreateMatrix_Zero (3,3)

; create solver-related buffers
$candidates=_Eigen_CreateMatrix (9,81)	; digits x cells
$savecandidates=_Eigen_CreateMatrix (9,81)	; for restoring after trial & error
$candidValues=_Eigen_CreateMatrix (9,81)

$valuecolvec=_Eigen_CreateMatrix (9,1)
$valuecolvec2=_Eigen_CreateMatrix (9,1)
$valuecolvec3=_Eigen_CreateMatrix (9,1)
$scratchcolvec=_Eigen_CreateMatrix (9,1)

$blockfreqs=_Eigen_CreateMatrix (9,1)

; create candidate value mirrors
Global $candidate[9]
For $dc=0 To 8
	$candidate[$dc]=_Eigen_CreateMatrix_Ones (1,81)
Next

; create all buddy tables
$buddies=_Eigen_CreateMatrix_Ones (3,81)

Global $rowbuddymask[9],$colbuddymask[9],$blockbuddymask[9],$allbuddies[81]
For $dc=0 To 8
	$rowbuddymask[$dc]=_Eigen_CreateMatrix (1,81)
	$colbuddymask[$dc]=_Eigen_CreateMatrix (1,81)
	$blockbuddymask[$dc]=_Eigen_CreateMatrix (1,81)
Next

; fill index and buddy tables
For $cc=0 To 8
	For $rc=0 To 8
		$cll=$rc+($cc*9)
		$blockID=_Eigen_ReadMatrixValue($matBlocks,$rc,$cc)

		_Eigen_WriteMatrixValue($rowindexRowVec,0,$cll,$rc)
		_Eigen_WriteMatrixValue($colindexRowVec,0,$cll,$cc)
		_Eigen_WriteMatrixValue($blockindexRowVec,0,$cll,$blockID)

		_Eigen_WriteMatrixValue($buddies,0,$cll,$rc)
		_Eigen_WriteMatrixValue($buddies,1,$cll,$cc)
		_Eigen_WriteMatrixValue($buddies,2,$cll,$blockID)
	Next
Next

For $dc=0 To 8
	_Eigen_ConditMask_Row($buddies,0,"==",$dc,$rowbuddymask[$dc])
	_Eigen_ConditMask_Row($buddies,1,"==",$dc,$colbuddymask[$dc])
	_Eigen_ConditMask_Row($buddies,2,"==",$dc,$blockbuddymask[$dc])
Next

For $cll=0 To 80
	_Eigen_ConditMask_Row($buddies,0,"==",_Eigen_ReadMatrixValue($buddies,0,$cll),$retainRowVec)
	_Eigen_ConditMask_Row($buddies,1,"==",_Eigen_ReadMatrixValue($buddies,1,$cll),$scratchRowVec)

	_Eigen_CwiseLogicalOp_InPlace($retainRowVec,"or",$scratchRowVec)
	_Eigen_ConditMask_Row($buddies,2,"==",_Eigen_ReadMatrixValue($buddies,2,$cll),$scratchRowVec)

	_Eigen_CwiseLogicalOp_InPlace($retainRowVec,"or",$scratchRowVec)

	$allbuddies[$cll]=_Eigen_CloneMatrix($retainRowVec)
Next
_Eigen_ReleaseMatrix($buddies)


; store lists of masked cells per block for display
$cellIDsBlock0=_Eigen_FindAll($blockbuddymask[0],1)
$cellIDsBlock1=_Eigen_FindAll($blockbuddymask[1],1)
$cellIDsBlock2=_Eigen_FindAll($blockbuddymask[2],1)
$cellIDsBlock3=_Eigen_FindAll($blockbuddymask[3],1)
$cellIDsBlock4=_Eigen_FindAll($blockbuddymask[4],1)
$cellIDsBlock5=_Eigen_FindAll($blockbuddymask[5],1)
$cellIDsBlock6=_Eigen_FindAll($blockbuddymask[6],1)
$cellIDsBlock7=_Eigen_FindAll($blockbuddymask[7],1)
$cellIDsBlock8=_Eigen_FindAll($blockbuddymask[8],1)

;_____________________________________________________________________________
; PREP GUI

If $showGUI Then
	$cellsize=40
	$cellshiftH=$cellsize*.65
	$cellshiftV=$cellsize*.82
	$digitscale=.6
	$linewidth=4
	$linecolor=0xa0a0a0
	$darklinecolor=0x404040
	$cluescolor=0x0000a0
	$guesscolor=0x800000
	$size=9*$cellsize

	Global $GUI = GUICreate("Brute-Force Sudoku Solver",$size+$linewidth,$size+$linewidth)

	Global $digits[0]
	For $cc = $cellsize to $size Step $cellsize
		GUICtrlCreateLabel("",0,$cc,$size,$linewidth)
		GUICtrlSetBkColor(-1,((mod($cc,$cellsize*3)=0)?($darklinecolor):($linecolor)))

		GUICtrlCreateLabel("",$cc,0,$linewidth,$size)
		GUICtrlSetBkColor(-1,(mod($cc,$cellsize*3)=0)?($darklinecolor):($linecolor))


		For $rc = $cellsize to $size Step $cellsize
			_ArrayAdd($digits,GUICtrlCreateLabel("",$cc-$cellshiftH,$rc-$cellshiftV,$cellsize*$digitscale,$cellsize*$digitscale))
			GUICtrlSetFont(-1,18,700,Default,"Arial")
		Next
	Next

	Global $dummy[10],$aAccelKeys[10][2]
	For $dc=0 To 9
		$dummy[$dc]=GUICtrlCreateDummy()
		$aAccelKeys[$dc][0]=$dc
		$aAccelKeys[$dc][1]=$dummy[$dc]
	Next
	GUISetAccelerators($aAccelKeys)

	$msg=" Press   1 to  9 to select a sudoku   or     Press 0  to load  an sdk    file.  "
	For $rc=0 to 8
		For $cc=0 to 8
			GUICtrlSetData($digits[$rc+$cc*9],StringMid($msg,$cc+$rc*9,1))
		Next
	Next

	GUISetState(@SW_SHOW, $GUI)

	While 1
		Switch GUIGetMsg()
        Case $GUI_EVENT_CLOSE
            Exit
		Case $dummy[0]
			$selection=0
			ExitLoop
		Case $dummy[1]
			$selection=1
			ExitLoop
		Case $dummy[2]
			$selection=2
			ExitLoop
		Case $dummy[3]
			$selection=3
			ExitLoop
		Case $dummy[4]
			$selection=4
			ExitLoop
		Case $dummy[5]
			$selection=5
			ExitLoop
		Case $dummy[6]
			$selection=6
			ExitLoop
		Case $dummy[7]
			$selection=7
			ExitLoop
		Case $dummy[8]
			$selection=8
			ExitLoop
		Case $dummy[9]
			$selection=9
			ExitLoop
		EndSwitch

		Sleep(250)
	WEnd
	IniWrite("BFSS.ini","selection","lastused",$selection)

	Opt("GUIOnEventMode", 1)
	GUISetOnEvent($GUI_EVENT_CLOSE, "_CloseDown")

	; blank the grid
	For $cc=0 To 80
		GUICtrlSetData($digits[$cc],"")
		GUICtrlSetColor($digits[$cc],$guessColor)
	Next

Else
	$selection=Number(IniRead("BFSS.ini","selection","lastused",Random(1,9,1)))
EndIf

;_____________________________________________________________________________
; PROCESS SUDOKU CLUES

; load 9x9 sudoku as square matrix (blank cells = 0)
; NB to load your own 2D 9x9 sudoku array, call _CreateSudokuMatrix(<your array>) instead
_LoadSudoku()

; create list of all clue locations
$matClueMask=_Eigen_ConditMask($matSudoku,"!=",0)
$matClueList=_Eigen_FindAll($matClueMask,1)	; get offset,row,col
$totalclues=@extended

; write update to console
ConsoleWrite("Processing "&$totalclues&" Clues..." & @CRLF & @CRLF)
If $verbose And $totalclues<17 And $exitUpon1stHit=True Then ConsoleWrite("WARNING: at least 17 clues are required for a unique solution" & @CRLF)
If $verbose And $totalclues=17 Then ConsoleWrite("WARNING: 17 clues is barely enough; this may take an extremely long time." & @CRLF)

; collect the clues and store them in our list (in appended columns)
$matClueValues=_Eigen_CreateVector_FromAcell_Mask($matSudoku,$matClueMask,False)
$cluesList=_Eigen_CreateMatrix_FromABcols($matClueList,$matClueValues)
_Eigen_ReleaseMatrix($matClueList)
_Eigen_ReleaseMatrix($matClueValues)

; punch in/out each clue in the permutation mask of each relevant block
For $kc=0 To $totalclues-1
	$offset=_Eigen_ReadMatrixValue($cluesList,$kc,0)
	$rw=_Eigen_ReadMatrixValue($cluesList,$kc,1)	; get clue location and value
	$cl=_Eigen_ReadMatrixValue($cluesList,$kc,2)
	$cll=$rw+($cl*9)
	$valueBase1=_Eigen_ReadMatrixValue($cluesList,$kc,3)

	_PunchIn($cll,$rw,$cl,$valuebase1-1)	; stored base-1, so convert to base-0
	_Eigen_WriteMatrixValue($solution,$rw,$cl,$valueBase1)

	If $showGUI Then
		$cellIDsBlock=Eval("cellIDsBlock" & _Eigen_ReadMatrixValue($matBlocks,$rw,$cl))	;blockID
		_Eigen_WriteMatrixValue($cellIDsBlock,_Eigen_Find_Col($cellIDsBlock,0,$offset),1,$valueBase1)

		GUICtrlSetData($digits[$offset],$valuebase1)
		GUICtrlSetColor($digits[$offset],$cluesColor)
	EndIf
Next

; check that all clues are consistent with each other
For $dc=1 To 9
	_Eigen_ConditCount_Rowwise($solution,"==",$dc,$solutionCol)
	If _Eigen_ConditAny($solutionCol,">",1) Then _FatalError("invalid clues = no solution")

	_Eigen_ConditCount_Colwise($solution,"==",$dc,$solutionRow)
	If _Eigen_ConditAny($solutionRow,">",1) Then _FatalError("invalid clues = no solution")

	For $cc=0 To 2
		$startcol=$cc*3

		For $rc=0 To 2
			$startrow=$rc*3
			If _Eigen_ConditCount_Block($solution,$startrow,$startcol,3,3,"==",$dc)>1 Then _FatalError("invalid clues = no solution")
		Next
	Next
Next

; this was deferred until all clues were processed
If _UpdateCandidatesPermuts()=False Then _FatalError("invalid clues = no solution")
_CheckAllBlocks()	; fill blockfreqs & check whether solution is already found

_OptimiseMax()	; rearrange block refs
_BruteForce()	; cycle through remaining off-diagonal block permuts
_CloseDown()	; report result

;=============================================================================
; Example Sudokus 1-10, from easy to extreme
; timings depend on hardware, cpu load, disk activity and other factors; provided for relative comparison only

Func _LoadSudoku()

	Local $fname=""

	Switch $selection
	Case 1	; easy
		Global	$sudoku = [	0,3,0, 0,0,1, 9,6,0, _
							0,0,1, 0,4,6, 5,0,0, _
							0,5,7, 2,0,0, 3,0,0, _
							1,0,6, 0,0,2, 0,0,0, _
							0,0,0, 0,0,0, 8,9,0, _
							0,4,5, 7,3,0, 0,1,0, _
							0,0,0, 8,0,5, 4,0,0, _
							0,2,0, 1,9,0, 0,0,7, _
							8,6,0, 0,0,0, 0,5,3 ]
	Case 2	; easy
		Global	$sudoku = [	4,0,0, 0,0,0, 5,9,0, _
							0,3,2, 0,0,0, 0,6,0, _
							0,7,0, 6,0,9, 2,0,3, _
							1,0,8, 0,4,0, 0,0,0, _
							7,0,0, 8,5,0, 0,0,0, _
							0,9,0, 0,0,0, 1,3,8, _
							6,0,0, 1,0,0, 0,0,2, _
							0,0,0, 2,0,6, 9,0,0, _
							0,8,9, 4,0,0, 0,0,5 ]
	Case 3	; moderate
		Global	$sudoku = [	0,0,6, 0,0,9, 0,0,0, _
							0,7,0, 0,0,8, 2,0,5, _
							5,0,3, 0,7,0, 0,0,0, _
							8,1,2, 0,0,0, 0,0,0, _
							0,0,0, 9,3,0, 0,0,0, _
							0,0,0, 0,0,0, 0,4,6, _
							0,5,0, 0,0,0, 0,0,9, _
							0,0,0, 1,0,2, 0,0,0, _
							0,0,0, 0,0,0, 5,6,4 ]
	Case 4	; moderate
		Global	$sudoku = [	1,0,0, 8,0,4, 0,0,0, _
							0,7,0, 0,6,3, 0,0,0, _
							0,0,0, 0,0,5, 0,6,4, _
							0,0,0, 0,0,0, 0,4,9, _
							5,3,0, 0,0,0, 0,0,0, _
							0,0,0, 1,3,7, 0,0,0, _
							2,0,0, 0,5,0, 0,0,0, _
							0,4,6, 0,9,0, 8,0,0, _
							0,0,8, 0,0,0, 2,0,7 ]
	Case 5	; moderate
		Global	$sudoku = [ 5,3,0, 0,7,0, 0,0,0, _
							6,0,0, 1,9,5, 0,0,0, _
							0,9,8, 0,0,0, 0,6,0, _
							8,0,0, 0,6,0, 0,0,3, _
							4,0,0, 8,0,3, 0,0,1, _
							7,0,0, 0,2,0, 0,0,6, _
							0,6,0, 0,0,0, 2,8,0, _
							0,0,0, 4,1,9, 0,0,5, _
							0,0,0, 0,8,0, 0,7,9 ]
	Case 6	; hard
		Global	$sudoku = [ 1,0,0, 0,0,7, 0,9,0, _
							0,3,0, 0,2,0, 0,0,8, _
							0,0,9, 6,0,0, 5,0,0, _
							0,0,5, 3,0,0, 9,0,0, _
							0,1,0, 0,8,0, 0,0,2, _
							6,0,0, 0,0,4, 0,0,0, _
							3,0,0, 0,0,0, 0,1,0, _
							0,4,0, 0,0,0, 0,0,7, _
							0,0,7, 0,0,0, 3,0,0 ]
	Case 7	; hard
		Global	$sudoku = [ 4,0,2, 0,0,0, 0,3,0, _
							1,0,0, 6,0,5, 0,2,9, _
							9,0,0, 0,0,0, 1,0,0, _
							0,0,0, 0,4,2, 0,0,0, _
							0,8,0, 9,0,1, 0,5,0, _
							0,0,0, 8,5,0, 0,0,0, _
							0,0,3, 0,0,0, 0,0,8, _
							6,1,0, 5,0,4, 0,0,2, _
							0,4,0, 0,0,0, 5,0,7 ]
	Case 8	; hard
		Global	$sudoku = [	5,0,0, 0,0,0, 0,2,4, _
							0,0,0, 1,9,0, 0,0,0, _
							7,0,0, 0,0,0, 0,5,0, _
							0,0,0, 0,0,0, 0,9,2, _
							9,3,0, 0,7,0, 0,0,0, _
							2,0,0, 3,6,0, 0,0,0, _
							0,0,8, 0,0,0, 9,0,0, _
							0,0,2, 6,0,4, 8,0,0, _
							0,0,4, 0,0,8, 0,0,1 ]
	Case 9; the Easter Monster (extreme)
		Global	$sudoku = [ 1,0,0, 0,0,0, 0,0,2, _
							0,9,0, 4,0,0, 0,5,0, _
							0,0,6, 0,0,0, 7,0,0, _
							0,5,0, 9,0,3, 0,0,0, _
							0,0,0, 0,7,0, 0,0,0, _
							0,0,0, 8,5,0, 0,4,0, _
							7,0,0, 0,0,0, 6,0,0, _
							0,3,0, 0,0,9, 0,8,0, _
							0,0,2, 0,0,0, 0,0,1]
	Case 10	; extreme
		Global	$sudoku = [ 8,0,0, 0,0,0, 0,0,0, _
							0,0,3, 6,0,0, 0,0,0, _
							0,7,0, 0,9,0, 2,0,0, _
							0,5,0, 0,0,7, 0,0,0, _
							0,0,0, 0,4,5, 7,0,0, _
							0,0,0, 1,0,0, 0,3,0, _
							0,0,1, 0,0,0, 0,6,8, _
							0,0,8, 5,0,0, 0,1,0, _
							0,9,0, 0,0,0, 4,0,0 ]

	Case Else
		$fname=FileOpenDialog("Please select an sdk file",@ScriptDir,"(*.sdk; *.txt)",1)
		If @error Then _FatalError("file selection error code: " & @error)

		If _FileCountLines($fname)<>9 Or @error Then _FatalError("file access error or unrecognised format; text file with 9 lines expected, containing 9 single digits (1-9) or dots ('.') for empty cells")

		Local $line,$fh=FileOpen($fname)
		If @error then _FatalError("unable to access file: " & $fname)

		For $rc=0 To 8
			$line=StringSplit(StringReplace(StringStripWS(FileReadLine($fh),8),".","0"),"",2)
			If @error or not IsArray($line) Or UBound($line)<>9 Then
				FileClose($fh)
				_FatalError("unrecognised format; text file with 9 lines expected, containing 9 single digits (1-9) or dots ('.') for empty cells; spaces are ignored.")
			EndIf

			For $cc=0 To 8
				$sudoku[$cc+($rc*9)]=$line[$cc]
			Next
		Next
		FileClose($fh)

	EndSwitch

	Switch $selection
		Case 1 To 10
			ConsoleWrite("Solving internal example sudoku #" & $selection & @CRLF & @CRLF)
		Case Else
			ConsoleWrite("Solving external sudoku in sdk file " & $fname & @CRLF & @CRLF)
	EndSwitch

	_CreateSudokuMatrix($sudoku,True)	; T: stored as 1D rowwise array

EndFunc


Func _CreateSudokuMatrix(ByRef $array,$stored1Drowwise=False)

	If UBound($array,0)>2 Then _FatalError("1D or 2D array expected")

	$matSudoku=_Eigen_CreateMatrix_fromArray($array)
	If @error Then _FatalError("unable to create input matrix")
	If _Eigen_GetMatrixSize($matSudoku)<>81 Then _FatalError("parsed array should have exactly 81 integer values (0 for blanks, 1-9 for clues)")

	If Not _Eigen_IsVector($matSudoku) Then	; 2D-specific checks
		If _Eigen_GetRows($matSudoku)<>9 Or Not _Eigen_IsSquare($matSudoku) Then _FatalError("parsed 2D array should have 9 x 9 integer values (0 for blanks, 1-9 for clues)")
	EndIf

	; check valid content
	If _Eigen_IsZero($matSudoku) Then _FatalError("parsed sudoku array is completely empty (all zeroes)")
	If _Eigen_GetMinVal($matSudoku)<0 Then _FatalError("negative values in sudoku array cannot be processed")
	If _Eigen_GetMinVal($matSudoku)>0 Then _FatalError("no blank values found to resolve in parsed sudoku array")
	If _Eigen_GetMaxVal($matSudoku)>9 Then _FatalError("largest value in array exceeds maximum of 9")

	; reformat valid vector input to 9x9 square matrix
	_Eigen_ReDim_ExistingMatrix($matSudoku,9,9)	; no change if already 2D
	If $stored1Drowwise Then _Eigen_Transpose_InPlace($matSudoku)

EndFunc


Func _ShowBlock($blockID,$permutID)

	Switch $blockID
		Case 0
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock0,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 1
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock1,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 2
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock2,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 3
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock3,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 4
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock4,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 5
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock5,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 6
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock6,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 7
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock7,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 8
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock8,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next
	EndSwitch

EndFunc


Func _ClearBlock($blockID)

	Switch $blockID
		Case 0
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock0,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock0,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock0,$rc,1))))
			Next

		Case 1
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock1,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock1,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock1,$rc,1))))
			Next

		Case 2
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock2,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock2,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock2,$rc,1))))
			Next

		Case 3
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock3,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock3,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock3,$rc,1))))
			Next

		Case 4
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock4,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock4,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock4,$rc,1))))
			Next

		Case 5
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock5,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock5,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock5,$rc,1))))
			Next

		Case 6
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock6,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock6,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock6,$rc,1))))
			Next

		Case 7
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock7,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock7,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock7,$rc,1))))
			Next

		Case 8
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock8,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock8,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock8,$rc,1))))
			Next
	EndSwitch

EndFunc

;_____________________________________________________________________________
; Bitmasks and lists

Func _StoreMaskOfCellsWithNvalues($tempRowVec,$N,$mincount=1)
; sum the candidates bitmask colwise (=per cell)
; stores a rowvector bitmask of 81 cells in whatever rowvec var is parsed

		If $mincount<1 Then Return False

		_Eigen_MatrixSpecs_Colwise_Single($candidates,2,$tempRowVec)	; 2=sum
		_Eigen_ConditMask_InPlace($tempRowVec,"==",$N)

		Return (_Eigen_GetSum($tempRowVec)>=$mincount)
EndFunc


Func _CreateListOfCellsWithNvalues($tempRowVec,$N,$minsize=1)
; expects a parsed 1x81 rowvector buffer to be filled
; stored values are size-ordered; offset/row/col/value are base-0

	If $N<1 Or $N>9 Then Return SetError(1,0,False)

	; create a colmask for $candidates for cells with exactly N values
	If _StoreMaskOfCellsWithNvalues($tempRowVec,$N,$minsize)=False Then Return SetError(1,0,False)

	; list cells with N values
	_Eigen_Redim_ExistingMatrix($tempRowVec,9,9)
	Local $list=_Eigen_FindAll($tempRowVec,1)
	Local $listsize=@extended
	_Eigen_Redim_ExistingMatrix($tempRowVec,1,81)	; needed again

	; fill candidates mirror with the actual values
	_FillCandidValues(); NB stored values are base-1 (0 = empty = invalid candidate value for cell)

	; remove all cols (cells) with fewer/more than N values
	Local $tempcandid=_Eigen_CreateMatrix_FromAcols_Mask ($candidValues,$tempRowVec)

	; store all non-zero values per listed cell
	Local $tempvalues=_Eigen_CreateVector_FromAcell_Mask($tempcandid,$tempcandid)
	_Eigen_CwiseScalarOp_InPlace($tempvalues,"-",1)	; make all values base-0
	_Eigen_Redim_ExistingMatrix($tempvalues,$N,$listsize)	; reshape for appending

	; append values to list
	_Eigen_AppendCols_InPlace ($list,$N)	; example of an alias-wrapper call
	For $rc=0 To $N-1
		_Eigen_Copy_Arow_ToBcol($tempvalues,$list,$rc,$rc+3)
	Next

	_Eigen_ReleaseMatrix($tempcandid)
	_Eigen_ReleaseMatrix($tempvalues)

	Return SetError(0,$listsize,$list)
EndFunc


Func _FillMaskOfOverlaps($inputmask,$outputmask,$rowmask=True)
; expects a parsed digitprofile inputmask (a colvector of size 27)
; produces a permuts rowvec mask of overlaps

	Switch $rowmask		; select block digit profile
		Case True
			_Eigen_CwiseBinaryOp_Colwise($digitInRows,"+",$inputmask,$matDigitBuffer)
		Case Else
			_Eigen_CwiseBinaryOp_Colwise($digitInCols,"+",$inputmask,$matDigitBuffer)
	EndSwitch

	_Eigen_ConditAny_Colwise($matDigitBuffer,">",1,$outputmask)	; permut mask (1) = overlap

EndFunc


Func _FillMasksOfValidNeighbours($permutID,$blockID,$transpose=False)
; create row/col mask of those permutations that do not produce a row/col duplication clash
; 1. add the content of our target column (= permutation) to ALL cols
; 2. flag as valid only those cols with zero overlap in rows/cols (2=overlap; 0/1=no overlap)
; 3. store it in the target block's bitmask array for valid (row c.q. col) neighours

	Switch $transp
		Case False
			_Eigen_CwiseBinaryOp_ColwiseCol($digitInRows,$permutID,"+",$digitInRows,$matDigitBuffer)
			_Eigen_ConditAll_Colwise($matDigitBuffer,"<",2,$validRowNeighbours[$blockID])	; mask (1) = valid

			_Eigen_CwiseBinaryOp_ColwiseCol($digitInCols,$permutID,"+",$digitInCols,$matDigitBuffer)
			_Eigen_ConditAll_Colwise($matDigitBuffer,"<",2,$validColNeighbours[$blockID])	; mask (1) = valid

		Case Else
			_Eigen_CwiseBinaryOp_ColwiseCol($digitInRows,$permutID,"+",$digitInRows,$matDigitBuffer)
			_Eigen_ConditAll_Colwise($matDigitBuffer,"<",2,$validColNeighbours[$blockID])	; mask (1) = valid

			_Eigen_CwiseBinaryOp_ColwiseCol($digitInCols,$permutID,"+",$digitInCols,$matDigitBuffer)
			_Eigen_ConditAll_Colwise($matDigitBuffer,"<",2,$validRowNeighbours[$blockID])	; mask (1) = valid
	EndSwitch

EndFunc


Func _MaskValidPermutations($blockID, $validmask1=0, $validmask2=0, $validmask3=0, $validmask4=0)
; reduce valid permuts by applying 1-4 "and" bitmask filters (negative exclusion: 0 = out; 1 = keep)

	_Eigen_Copy_A_ToB($punchout[$blockID],$matpermut[$blockID])	; keep original in $punchout array

	If $validmask1>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask1)
	If $validmask2>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask2)
	If $validmask3>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask3)
	If $validmask4>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask4)

EndFunc

;_________________________________________________________________________________
; Load / Create the Lookup Tables

Func _LoadBlockPermutations()
; 9 rows (digits 1-9, base-1) x 362,880 cols (block permutations)

	Local $filename="blockpermut.mat"

	If FileExists($filename) Then
		$blockPermuts = _Eigen_LoadMatrix($filename)
		Return
	EndIf

	If $verbose Then ConsoleWrite("Generating lookup table file " & $filename & "; please wait..." & @CRLF & "NB This only needs to be done ONCE." & @CRLF)

	$blockPermuts = _Eigen_CreateMatrix (9,$permuts)
	Local $matC = _Eigen_CreateMatrix (9,1)

	Local $rc=-1
	For $c0=1 To 9
		_Eigen_WriteMatrixValue($matC,0,0,$c0)

	For $c1=1 To 9
		If $c1=$c0 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,1,0,$c1)

	For $c2=1 To 9
		If $c2=$c0 Then ContinueLoop
		If $c2=$c1 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,2,0,$c2)

	For $c3=1 To 9
		If $c3=$c0 Then ContinueLoop
		If $c3=$c1 Then ContinueLoop
		If $c3=$c2 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,3,0,$c3)

	For $c4=1 To 9
		If $c4=$c0 Then ContinueLoop
		If $c4=$c1 Then ContinueLoop
		If $c4=$c2 Then ContinueLoop
		If $c4=$c3 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,4,0,$c4)

	For $c5=1 To 9
		If $c5=$c0 Then ContinueLoop
		If $c5=$c1 Then ContinueLoop
		If $c5=$c2 Then ContinueLoop
		If $c5=$c3 Then ContinueLoop
		If $c5=$c4 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,5,0,$c5)

	For $c6=1 To 9
		If $c6=$c0 Then ContinueLoop
		If $c6=$c1 Then ContinueLoop
		If $c6=$c2 Then ContinueLoop
		If $c6=$c3 Then ContinueLoop
		If $c6=$c4 Then ContinueLoop
		If $c6=$c5 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,6,0,$c6)

	For $c7=1 To 9
		If $c7=$c0 Then ContinueLoop
		If $c7=$c1 Then ContinueLoop
		If $c7=$c2 Then ContinueLoop
		If $c7=$c3 Then ContinueLoop
		If $c7=$c4 Then ContinueLoop
		If $c7=$c5 Then ContinueLoop
		If $c7=$c6 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,7,0,$c7)

	For $c8=1 To 9
		If $c8=$c0 Then ContinueLoop
		If $c8=$c1 Then ContinueLoop
		If $c8=$c2 Then ContinueLoop
		If $c8=$c3 Then ContinueLoop
		If $c8=$c4 Then ContinueLoop
		If $c8=$c5 Then ContinueLoop
		If $c8=$c6 Then ContinueLoop
		If $c8=$c7 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,8,0,$c8)

		$rc+=1
		_Eigen_Copy_Acol_ToBcol($matC,$blockPermuts,0,$rc)
		If $verbose And Mod($rc,1000)=0 Then ConsoleWrite("Processed " & $rc & " of " & $permuts & @CRLF)
	Next
	Next
	Next
	Next
	Next
	Next
	Next
	Next
	Next

	If $verbose Then ConsoleWrite("Saving file..." & @CRLF & @CRLF)
	_Eigen_SaveMatrix($blockPermuts,$filename,True)
	_Eigen_ReleaseMatrix($matC)

EndFunc


Func _LoadDigitTables()
; 3x9 rows (row c.q. col 0/1/2 for digits 1-9, for each block permutation)

	Local $filename1="digitInRows.mat"
	Local $filename2="digitInCols.mat"

	If (FileExists($filename1) And FileExists($filename2)) Then
		$digitInRows=_Eigen_LoadMatrix($filename1)	; permut x colvectors (of size: 27 rows: 0/1/2 for digit 1, 0/1/2 for digit 2, etc)
		$digitInCols=_Eigen_LoadMatrix($filename2)	; permut x rowvectors^T (of size: 27 rows: 0/1/2 for digit 1, 0/1/2 for digit 2, etc)
		Return
	EndIf

	If $verbose Then ConsoleWrite("Generating lookup table files " & $filename1 & " and " & $filename2 & "; please be patient..." & @CRLF & "NB This only needs to be done ONCE." & @CRLF)

	Local $matB = _Eigen_CreateMatrix (9,1)
	Local $matDrows = _Eigen_CreateMatrix (3,1)	; colvector
	Local $matDcols = _Eigen_CreateMatrix (1,3)	; rowvector
	Local $matDcolsT = _Eigen_CreateMatrix (3,1)	; rowvector

	$digitInRows = _Eigen_CreateMatrix_Zero (3*9,$permuts)	; colvectors
	$digitInCols = _Eigen_CreateMatrix_Zero (3*9,$permuts)	; rowvectors

	For $pc=0 To $permuts-1
		_Eigen_Copy_Acol_ToBcol($blockPermuts,$matB,$pc,0)

		_Eigen_Redim_ExistingMatrix ($matB,3,3)
		For $dc=1 To 9
			$offset=($dc-1)*3
			_Eigen_ConditAny_Rowwise($matB,"=",$dc,$matDrows)
			_Eigen_ConditAny_Colwise($matB,"=",$dc,$matDcols)
			_Eigen_Transpose($matDcols,$matDcolsT)

			_Eigen_Copy_Ablock_ToBblock($matDrows,$digitInRows,0,0,3,1,$offset,$pc)
			_Eigen_Copy_Ablock_ToBblock($matDcolsT,$digitInCols,0,0,3,1,$offset,$pc)
		Next
		_Eigen_Redim_ExistingMatrix ($matB,9,1)

		If $verbose And Mod($pc,100)=0 Then ConsoleWrite("Processed " & $pc & " of " & $permuts & @CRLF)
	Next

	If $verbose Then ConsoleWrite("Saving files..." & @CRLF & @CRLF)
	_Eigen_SaveMatrix($digitInRows,$filename1,True)
	_Eigen_SaveMatrix($digitInCols,$filename2,True)

	_Eigen_ReleaseMatrix($matB)
	_Eigen_ReleaseMatrix($matDrows)
	_Eigen_ReleaseMatrix($matDcols)
	_Eigen_ReleaseMatrix($matDcolsT)

EndFunc


Func _FillCandidValues()
; uses $candidates bitmask multiplication of colwise values 1-9 as a filter

	_Eigen_SetLinSpaced($valuecolvec,1,9)	; 0 = empty
	_Eigen_CwiseBinaryOp_ColwiseCol($candidates,0,"*",$valuecolvec,$candidValues)

EndFunc


Func _UpdateCandidatesPermuts()
; store the remaining permuts per block from $punchout[block] to $matPunchout (all blocks)

	For $dc=0 To 8
		_Eigen_Copy_Arow_ToBrow($punchOut[$dc],$matPunchOut,0,$dc)
		_Eigen_Copy_Arow_ToBrow($candidate[$dc],$candidates,0,$dc)
	Next

	; sum remaining active permuts per block
	_Eigen_MatrixSpecs_Rowwise_Single($matpunchOut,2,$scratchcolvec)

	Return Not (_Eigen_ConditAny($scratchcolvec,"==",0))	; F: invalid configuration
EndFunc

;_________________________________________________________________________________
; punch candidate values in/out

Func _PunchIn($cll,$row,$col,$valueb0)
; final cell-value assignment

	Local $blockID=_Eigen_ReadMatrixValue($matBlocks,$row,$col)

	$blockIDadjRow1=$blockneighbours[$blockID][0]
	$blockIDadjRow2=$blockneighbours[$blockID][1]

	$blockIDadjCol1=$blockneighbours[$blockID][2]
	$blockIDadjCol2=$blockneighbours[$blockID][3]

	; create mask for rows
	_Eigen_SetZero($digitRowvec)
	_Eigen_WriteMatrixValue($digitRowvec,($valueb0*3)+Mod($row,3),0,1)
	_FillMaskOfOverlaps($digitRowvec,$matMrowvec,True)	; T: rowmask, F: colmask

	; create mask for cols
	_Eigen_SetZero($digitColvec)
	_Eigen_WriteMatrixValue($digitColvec,($valueb0*3)+Mod($col,3),0,1)
	_FillMaskOfOverlaps($digitColvec,$matMcolvec,False)	; T: rowmask, F: colmask

	; include ONLY clue's own block's permuts that match the clue
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockID],"and",$matMrowvec)
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockID],"and",$matMcolvec)

	; exclude in the two row-adjacent blocks all permuts with clue value in same ROW
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockIDadjRow1],"xcl",$matMrowvec)
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockIDadjRow2],"xcl",$matMrowvec)

	; exclude in the two col-adjacent blocks all permuts with clue value in same COL
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockIDadjCol1],"xcl",$matMcolvec)
	_Eigen_CwiseLogicalOp_InPlace($PunchOut[$blockIDadjCol2],"xcl",$matMcolvec)

	; and for solver
	_Eigen_Copy_A_ToB($allbuddies[$cll],$retainRowVec)
	_Eigen_CwiseLogicalOp_InPlace($candidate[$valueb0],"xcl",$retainRowVec)

	; zero out all values for the target cell in $candidates
	For $dc=0 To 8
		_Eigen_WriteMatrixValue($candidate[$dc],0,$cll,0)
	Next

	$reduced+=1

EndFunc

;_____________________________________________________________________________
; solution-related

Func _CheckAllBlocks()
; 1. Are all block permutation counts reduced to unity?
; if so, construct the full solution and check its validity in terms of
; having 9 unique values for each row and each column
; NB we don't ever need to check the blocks themselves,
;	because their permutations are by definition valid

	_Eigen_ReDim_ExistingMatrix($blockfreqs,9,1)	; reshape as colvector
	_Eigen_MatrixSpecs_Single_Rowwise($matPunchOut,2,$blockfreqs)	; 2=sum remaining permuts per block in colvector
	_Eigen_ReDim_ExistingMatrix($blockfreqs,3,3)	; restore block shape

	If _Eigen_IsOnes($blockfreqs) Then
		If $verbose Then ConsoleWrite(@CRLF)

		For $bc=0 To 8
			_StoreBlock($bc,_Eigen_GetCol_MaxVal($punchout[$bc]))
		Next
		_CheckSolution()			; does not return if single solution is sought and valid solution is found
		Return SetError(3,0,True)	; invalid solution

	Else
		If _Eigen_IsZero($blockfreqs) Then Return SetError(3,0,True)	; invalid configuration
		If _Eigen_ConditAny($blockfreqs,"==",0) Then Return SetError(1,0,True)		; invalid guess; configuration produces empty permut subset in at least one other block

		If _Eigen_GetSum($blockfreqs)=$blockfreqsum Then Return SetError(2,0,True)	; no improvement in entire solver pass
		$blockfreqsum=_Eigen_GetSum($blockfreqs)	; for next pass
	EndIf

	Return False	; not done yet
EndFunc


Func _StoreBlock($blockID,$permutID)
; construct a single block in the solution

	If $verbose Then ConsoleWrite("Block " & $blockID & " contains permutation: " & $permutID & @CRLF)

	_Eigen_ReDim_ExistingMatrix($solutionBlock,9,1)
	_Eigen_Copy_Acol_ToBcol($blockPermuts,$solutionBlock,$permutID,0)

	_Eigen_ReDim_ExistingMatrix($solutionBlock,3,3)
	_Eigen_Copy_Ablock_ToBblock($solutionBlock,$solution,0,0,3,3,(Mod($blockID,3)*3),(Int($blockID/3)*3))

	If $showGUI Then _ShowBlock($blockID,$permutID)

EndFunc


Func _CheckSolution()
; does each value 1-9 occur exactly once in each row and each column?
; NB no need to check block integrity (valid by definition)

	For $dc=1 To 9
		_Eigen_ConditCount_Rowwise($solution,"==",$dc,$solutionCol)
		If _Eigen_IsOnes($solutionCol)=False Then Return

		_Eigen_ConditCount_Colwise($solution,"==",$dc,$solutionRow)
		If _Eigen_IsOnes($solutionRow)=False Then Return
	Next

	$solutionsFound+=1
	If $verbose Then _WinAPI_Beep()
	If ($solutionsFound=1 And $exitUpon1stHit) Then _CloseDown()

EndFunc


Func _FatalError($errmsg="an unknown error occurred")
; Oops!

	ConsoleWrite("Fatal Error: " & $errmsg & @CRLF)
	Exit(1)

EndFunc


Func _CloseDown()
; All done, folks.

	If $tstart Then ConsoleWrite(@CRLF & "Brute-Force took: " & Round(TimerDiff($tstart)/1000,2)-(($verbose=True)?(1):(0)) & " seconds" & @CRLF)	; subtract 1 sec for beep
	If $showGUI Then ConsoleWrite("Note: Try it WITHOUT the GUI (set $showGUI=False) for a FASTER result!" & @CRLF)
	ConsoleWrite(@CRLF & "Solution(s) found: " & $solutionsFound & @CRLF & @CRLF)

	If $solutionsFound>0 Then _Eigen_ShowMatrix($solution,"Sudoku solution found:")

	If $showGUI Then
		Sleep(2000)
		GUIDelete($GUI)
	EndIf

	ConsoleWrite("Please wait; closing down..." & @CRLF)
	_Eigen_CleanUp()	; ...End of Story
	Exit(0)

EndFunc

;_____________________________________________________________________________
; BRUTE FORCE TRIAL AND ERROR
; guesses the contents of the six off-diagonal blocks in optimised order,
; thereby reducing permuts to unity or zero in the three diagonal blocks,
; without having to evaluate these at all.

Func _OptimiseMax()
; Block IDs (row=offset) are LinSpaced Colwise in a 3x3 grid, from 0 to 8.
; Identify the two blocks with the largest number of active permutations left;
; swap these onto the block diagonal, and identify the other block positions
; Then choose the smallest permutation space for all four TC x TR configurations

	_UpdateCandidatesPermuts()	; ensure permuts are up-to-date
	_CheckAllBlocks()			; ensure freqs are up-to-date
	_Eigen_Redim_ExistingMatrix($blockfreqs,3,3)	; reshape as block
	Local $bfreqs=_Eigen_CloneMatrix($blockfreqs)
	If $verbose Then _Eigen_ShowMatrix($blockfreqs,"Remaining block permutations")

	Local $blockMF=_Eigen_GetOffset_MaxVal($blockfreqs) ; find block ID with highest frequency
	If $verbose Then ConsoleWrite(@CRLF & "Swapped Block ID " & $blockMF & " (in major square, now at TL) has the highest freq of " & _Eigen_GetMaxVal($blockfreqs) & @CRLF)

	Local $maxValBlockrow=_Eigen_GetRow_MaxVal($blockfreqs)
	Local $maxValBlockcol=_Eigen_GetCol_MaxVal($blockfreqs)

	If $blockMF>0 Then	; update the grids
		If $maxValBlockrow>0 Then
			_Eigen_Swap_Arow_Arow($matBlockIDs,$maxValBlockrow,0)
			_Eigen_Swap_Arow_Arow($blockfreqs,$maxValBlockrow,0)
		EndIf

		If $maxValBlockcol>0 Then
			_Eigen_Swap_Acol_Acol($matBlockIDs,$maxValBlockcol,0)
			_Eigen_Swap_Acol_Acol($blockfreqs,$maxValBlockcol,0)
		EndIf
	EndIf

	; punchout MF target all directly neighbouring blocks
	_Eigen_SetZero_Row($blockfreqs,0)
	_Eigen_SetZero_Col($blockfreqs,0)


	Local $blockMF2=_Eigen_GetOffset_MaxVal($blockfreqs) ; find block ID with next-highest frequency in minor square
	If $verbose Then ConsoleWrite(@CRLF & "Swapped Block ID " & $blockMF2 & " (in minor square, now at BR) has the next-highest freq of " & _Eigen_GetMaxVal($blockfreqs) & @CRLF)

	Local $maxValBlockrow=_Eigen_GetRow_MaxVal($blockfreqs)
	Local $maxValBlockcol=_Eigen_GetCol_MaxVal($blockfreqs)

	; if current BR block is suboptimal, swap with optimal (most permuts)
	If $maxValBlockrow<2 Then
		_Eigen_Swap_Arow_Arow($matBlockIDs,1,2)		; 2D grid row swap
		_Eigen_Swap_Arow_Arow($blockfreqs,1,2)		; update freq table
	EndIf

	If $maxValBlockcol<2 Then
		_Eigen_Swap_Acol_Acol($matBlockIDs,1,2)	; 2D grid col swap
		_Eigen_Swap_Acol_Acol($blockfreqs,1,2)	; 2D grid col swap
	EndIf

	If $verbose Then ConsoleWrite(@CRLF & "Remaining diagonal block ID " & $blockMF2 & " (now at CC) has a freq of " & _Eigen_ReadMatrixValue($blockfreqs,1,1) & @CRLF)

	; minimise TCTR parameter space
	Local $regular=_Eigen_ReadMatrixValue($bfreqs,0,1)*_Eigen_ReadMatrixValue($blockfreqs,0,2)
	Local $flipped=_Eigen_ReadMatrixValue($bfreqs,2,0)*_Eigen_ReadMatrixValue($blockfreqs,2,1)
	Local $regularT=_Eigen_ReadMatrixValue($bfreqs,1,0)*_Eigen_ReadMatrixValue($blockfreqs,2,0)
	Local $flippedT=_Eigen_ReadMatrixValue($blockfreqs,0,2)*_Eigen_ReadMatrixValue($blockfreqs,1,2)

	Switch _Min(_Min(_Min($regular,$flipped),$regularT),$flippedT)
		Case $regular
			$transp=False
		Case $regularT
			_Eigen_Transpose_InPlace($matBlockIDs)	; blockfreqs is no longer needed, so act only on matBlockIDs
			$transp=True
		Case $flipped
			_Eigen_Reverse_Rows_InPlace($matBlockIDs)
			_Eigen_Reverse_Cols_InPlace($matBlockIDs)
			$transp=False
		Case $flippedT
			_Eigen_Reverse_Rows_InPlace($matBlockIDs)
			_Eigen_Reverse_Cols_InPlace($matBlockIDs)
			_Eigen_Transpose_InPlace($matBlockIDs)
			$transp=True
	EndSwitch

	; assign the diagonal blocks (globals)
	$blockTL=_Eigen_ReadMatrixValue($matBlockIDs,0,0)	; Top Left block
	$blockCC=_Eigen_ReadMatrixValue($matBlockIDs,1,1)	; Centre block
	$blockBR=_Eigen_ReadMatrixValue($matBlockIDs,2,2)	; Bottom Right block

	; assign the upper off-diagonal blocks (globals)
	$blockTC=_Eigen_ReadMatrixValue($matBlockIDs,0,1)	; Top Centre block
	$blockTR=_Eigen_ReadMatrixValue($matBlockIDs,0,2)	; Top Right block
	$blockCR=_Eigen_ReadMatrixValue($matBlockIDs,1,2)	; Centre Right block

	; assign the lower off-diagonal blocks (globals)
	$blockCL=_Eigen_ReadMatrixValue($matBlockIDs,1,0)	; Centre Left block
	$blockBL=_Eigen_ReadMatrixValue($matBlockIDs,2,0)	; Bottom Left block
	$blockBC=_Eigen_ReadMatrixValue($matBlockIDs,2,1)	; Bottom Centre block

EndFunc


Func _BruteForce()

	; create lists of all remaining candidates for all non-maxfreq blocks
	; lists for upper diagonal blocks
	Local $matTClist=_Eigen_FindAll($punchOut[$blockTC],1)
	Local $totalTC=_Eigen_GetMatrixRows($matTClist)

	Local $matTRlist=_Eigen_FindAll($punchOut[$blockTR],1)
	Local $totalTR=_Eigen_GetMatrixRows($matTRlist)

	Local $matCRlist=_Eigen_FindAll($punchOut[$blockCR],1)
	Local $totalCR=_Eigen_GetMatrixRows($matCRlist)


	; lists for lower diagonal blocks
	Local $matCLlist=_Eigen_FindAll($punchOut[$blockCL],1)
	Local $totalCL=_Eigen_GetMatrixRows($matCLlist)

	Local $matBLlist=_Eigen_FindAll($punchOut[$blockBL],1)
	Local $totalBL=_Eigen_GetMatrixRows($matBLlist)

	Local $matBClist=_Eigen_FindAll($punchOut[$blockBC],1)
	Local $totalBC=_Eigen_GetMatrixRows($matBClist)

	; the final optimisation rearranges the processing order in TC and TR (the two outermost loops)
	; the solution most likely resides close to the reverse diagonal in TC x Tr
	; create rowID and colID lookup tables for TC x TR matrix
	Local $xmax=$totalTC-1
	Local $ymax=$totalTR-1

	Local $vecrows=_Eigen_CreateMatrix_LinSpaced_Rowwise(1,$totalTR,0,$ymax)
	Local $veccols=_Eigen_CreateMatrix_LinSpaced_Colwise($totalTC,1,0,$xmax)

	Local $rowIDs=_Eigen_CreateMatrix_FromA($veccols,$totalTR,1)
	Local $colIDs=_Eigen_CreateMatrix_FromA($vecrows,1,$totalTC)
	_Eigen_ReleaseMatrix($vecrows)
	_Eigen_ReleaseMatrix($veccols)

	; for each cell, calculate the distance from the reverse diagonal (bottom-left to top-right)
	_Eigen_CwiseScalarOp_InPlace($rowIDs,"*",$ymax)
	_Eigen_CwiseScalarOp_InPlace($colIDs,"*",$xmax)
	_Eigen_CwiseBinaryOp_InPlace($rowIDs,"-",$colIDs)
	_Eigen_ReleaseMatrix($colIDs)

	_Eigen_CwiseUnaryOp_InPlace($rowIDs,"abs")	; distance is never negative
	_Eigen_CwiseScalarOp_InPlace($rowIDs,"/", _Max(1,Sqrt(($xmax^2)+($ymax^2))))
	_Eigen_ReverseRows_InPlace($rowIDs)	; reverse the diagonal


	; create a list of TC, TR refs ordered by distance to the reverse diagonal (nearest-first)
	Local $TCTRlist=_Eigen_FindAll($rowIDs,0)
	Local $list,$list2,$newlist
	Local $targetrow=Round($totalTC/2,0)	; use this for sudoku's of mixed difficulty (easy to hard)
;	Local $targetrow=$totalTC-1				; use this instead for the most extreme sudoku's only

	; order rows per distance class by abs row distance from $targetrow,
	; as sudoku population sampling suggests that duration outliers are
	; on average smallest when starting there
	_Eigen_AppendCols_InPlace($TCTRlist,1)
	_Eigen_Copy_Acol_ToAcol($TCTRlist,1,3)
	_Eigen_CwiseScalarOp_Col_InPlace($TCTRlist,3,"-",$targetrow)
	_Eigen_CwiseUnaryOp_InPlace($TCTRlist,"abs")
	_Eigen_SortRows_ByCol_InPlace($TCTRlist,3)

	Local $maxVal=_Eigen_GetMaxVal($rowIDs)
	For $dc=1 To $maxVal
		$list2=_Eigen_FindAll($rowIDs,$dc)
		If @extended>0 Then
			_Eigen_AppendCols_InPlace($list2,1)
			_Eigen_Copy_Acol_ToAcol($list2,1,3)
			_Eigen_CwiseScalarOp_Col_InPlace($list2,3,"-",$targetrow)
			_Eigen_CwiseUnaryOp_InPlace($list2,"abs")
			_Eigen_SortRows_ByCol_InPlace($list2,3)

			$newlist=_Eigen_CreateMatrix_FromABrows($TCTRlist,$list2)
			_Eigen_ReleaseMatrix($list2)
			_Eigen_ReleaseMatrix($TCTRlist)
			$TCTRlist=$newlist
		EndIf
	Next
	_Eigen_ReleaseMatrix($rowIDs)

	Local $totalTCTR=_Eigen_GetMatrixRows($TCTRlist)
	Local $tccount=-1,$trcount=-1

	; ready ,set, GO!
	ConsoleWrite(@CRLF & @CRLF & "Starting Brute-Force..." & @CRLF & @CRLF)
	$tstart=TimerInit()

For $tctrc=0 to $totalTCTR-1	; TC & TR loops are replaced here by single loop that assigns both
	$tcc=_Eigen_ReadMatrixValue($TCTRlist,$tctrc,1)
	$tccount+=1

	$TCpermutID=_Eigen_ReadMatrixValue($matTClist,$tcc,0)	; offset = permut ID
	_FillMasksOfValidNeighbours($TCpermutID,$blockTC)


	$trc=_Eigen_ReadMatrixValue($TCTRlist,$tctrc,2)
	$trcount+=1

	$TRpermutID=_Eigen_ReadMatrixValue($matTRlist,$trc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validRowNeighbours[$blockTC],0,$TRpermutID)=0 Then ContinueLoop

	_FillMasksOfValidNeighbours($TRpermutID,$blockTR)

	_MaskValidPermutations($blockTL,$validRowNeighbours[$blockTC],$validRowNeighbours[$blockTR])
	If _Eigen_IsZero($matpermut[$blockTL]) Then ContinueLoop

	If $verbose Then
		ConsoleWrite(@CRLF & "Processing TCTR step " & $tctrc+1 & " of " & $totalTCTR)
		ConsoleWrite("; " & Round(100*$tctrc/$totalTCTR,2)& "% done in " & round(TimerDiff($tstart)/60000,1)& " min." & @CRLF)
		ConsoleWrite("Processing TC (block " & $blockTC & ")  permut " & $tcc+1 & " of " & $totalTC & @CRLF)
		ConsoleWrite("Processing TR (block " & $blockTR & ")  permut " & $trc+1 & " of " & $totalTR & @CRLF)
		If $exitUpon1stHit=False And $solutionsFound>0 Then ConsoleWrite("Solutions found so far: " & $solutionsFound & @CRLF)
		ConsoleWrite(@CRLF)
	EndIf

	If $showGUI Then
		_ShowBlock($blockTC,$TCpermutID)
		_ShowBlock($blockTR,$TRpermutID)
		_ClearBlock($blockCL)
	EndIf


For $clc=0 to $totalCL-1
	$CLpermutID=_Eigen_ReadMatrixValue($matCLlist,$clc,0)	; offset = permut ID

	If $showGUI Then
		_ShowBlock($blockCL,$CLpermutID)
		_ClearBlock($blockBL)
	EndIf

	_FillMasksOfValidNeighbours($CLpermutID,$blockCL)

	_MaskValidPermutations($blockTL,$validRowNeighbours[$blockTC],$validRowNeighbours[$blockTR],$validColNeighbours[$blockCL])
	If _Eigen_IsZero($matpermut[$blockTL]) Then ContinueLoop

	_MaskValidPermutations($blockCC,$validColNeighbours[$blockTC],$validRowNeighbours[$blockCL])
	If _Eigen_IsZero($matpermut[$blockCC]) Then ContinueLoop

	_MaskValidPermutations($blockCR,$validColNeighbours[$blockTR],$validRowNeighbours[$blockCL])
	If _Eigen_IsZero($matpermut[$blockCR]) Then ContinueLoop


For $blc=0 to $totalBL-1
	$BLpermutID=_Eigen_ReadMatrixValue($matBLlist,$blc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validColNeighbours[$blockCL],0,$BLpermutID)=0 Then ContinueLoop

	If $showGUI Then
		_ShowBlock($blockBL,$BLpermutID)
		_ClearBlock($blockCR)
	EndIf

	_FillMasksOfValidNeighbours($BLpermutID,$blockBL)

	_MaskValidPermutations($blockTL, $validRowNeighbours[$blockTC],$validRowNeighbours[$blockTR],$validColNeighbours[$blockCL],$validColNeighbours[$blockBL])
	If _Eigen_GetSum($matpermut[$blockTL])<>1 Then ContinueLoop

	_MaskValidPermutations($blockBC,$validRowNeighbours[$blockBL],$validColNeighbours[$blockTC])
	If _Eigen_IsZero($matpermut[$blockBC]) Then ContinueLoop

	_MaskValidPermutations($blockBR,$validRowNeighbours[$blockBL],$validColNeighbours[$blockTR])
	If _Eigen_IsZero($matpermut[$blockBR]) Then ContinueLoop


For $crc=0 to $totalCR-1
	$CRpermutID=_Eigen_ReadMatrixValue($matCRlist,$crc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validRowNeighbours[$blockCL],0,$CRpermutID)=0 Then ContinueLoop
	If _Eigen_ReadMatrixValue($validColNeighbours[$blockTR],0,$CRpermutID)=0 Then ContinueLoop

	If $showGUI Then
		_ShowBlock($blockCR,$CRpermutID)
		_ClearBlock($blockBC)
	EndIf

	_FillMasksOfValidNeighbours($CRpermutID,$blockCR)

	_MaskValidPermutations($blockCC,$validColNeighbours[$blockTC],$validRowNeighbours[$blockCL],$validRowNeighbours[$blockCR])
	If _Eigen_IsZero($matpermut[$blockCC]) Then ContinueLoop


For $bcc=0 to $totalBC-1
	$BCpermutID=_Eigen_ReadMatrixValue($matBClist,$bcc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validRowNeighbours[$blockBL],0,$BCpermutID)=0 Then ContinueLoop
	If _Eigen_ReadMatrixValue($validColNeighbours[$blockTC],0,$BCpermutID)=0 Then ContinueLoop

	_FillMasksOfValidNeighbours($BCpermutID,$blockBC)

	_MaskValidPermutations($blockBR,$validRowNeighbours[$blockBC],$validRowNeighbours[$blockBL],$validColNeighbours[$blockTR],$validColNeighbours[$blockCR])
	If _Eigen_GetSum($matpermut[$blockBR])<>1 Then ContinueLoop

	_MaskValidPermutations($blockCC,$validColNeighbours[$blockBC],$validColNeighbours[$blockTC],$validRowNeighbours[$blockCL],$validRowNeighbours[$blockCR])
	If _Eigen_GetSum($matpermut[$blockCC])<>1 Then ContinueLoop


	; if we get this far, update the solution matrix
	If $verbose Then ConsoleWrite(@CRLF & "Checking this solution:" & @CRLF)

	_StoreBlock($blockTL,_Eigen_GetCol_MaxVal($matpermut[$blockTL]))
	_StoreBlock($blockTC,$TCpermutID)
	_StoreBlock($blockTR,$TRpermutID)

	_StoreBlock($blockCL,$CLpermutID)
	_StoreBlock($blockCC,_Eigen_GetCol_MaxVal($matpermut[$blockCC]))
	_StoreBlock($blockCR,$CRpermutID)

	_StoreBlock($blockBL,$BLpermutID)
	_StoreBlock($blockBC,$BCpermutID)
	_StoreBlock($blockBR,_Eigen_GetCol_MaxVal($matpermut[$blockBR]))

	_CheckSolution()
	If $verbose Then ConsoleWrite("Resuming scan..." & @CRLF)

Next
Next
Next

Next
Next

EndFunc
