;============================================================================================
; Title ..........: CrossFilterSudokuSolver
; AutoIt Version..: 3.3.14+
; Description ....: showcase for E4A bitmask handling
; Author..........: A.R.T. Jonkers (RTFC)
; Release.........: 1.2
; Latest revision.: 23 July 2020
; License.........: free for personal use; free distribution allowed provided
;							the author is credited; all other rights reserved.
; Tested on.......: W10Pro/64
; Dependencies....: Eigen4AutoIt (E4A) v5.1+
;============================================================================================
; Summary (Wotsit?)
;
; Eigen4AutoIt (E4A) provides extensive tools for producing and manipulating
; bitmasks, including conditional expressions and logical operands.
; This example script applies these to solve sudoku puzzles, which are
; mathematically simple, but can be computationally hard.
;
; It uses optimised brute-force, based upon a few tables and basic logic,
; applied to large bitmask matrices that track the remaining valid permutations for each 3x3 block
; (the basis of all approaches shown here).
;
; No guarantees, no warranties, no liability for damages, no refunds.
;	Use at your own risk.
;
;============================================================================================
;	Remarks
;
; *	set $XfilterLastPass=0 for multi-pass Xfiltering until no more improvement (fastest for extreme sudokus)
;	set $XfilterLastPass=1 to Xfilter once, followed by Brute-Force (fastest for easy sudokus)
;	set $XfilterLastPass=N (N>=2) to Xfilter up to N times before Brute-Force
;	set $XfilterLastPass=-1 to skip Xfiltering altogether (just using Brute Force)
;
; * Three tables (extension *.mat) are provided, but will be regenerated
;	if not found locally (slow, but done only once).
;
; * How it works:
;	1. create work environment (note how multiple matrices (refs) are stored/accessed in arrays)
;	2. read in sudoku data and store the clues.
;	3. punch-in all clues, reducing the number of valid permutations per 3x3 block and all neighbouring blocks
;	4. Cross-Filtering (multi-pass): For each block: given the block's
;		current-remaining valid permutations, apply all combinations of its
;		four possible blockrow and blockcol neighbours as compound filters,
;		invalidating those permutations that lack simultaneous support from all four neighbours.
;		Speed is gained by processing block frequencies in descending order.
;	5. call _OptimiseMax: Find the block with most remaining permutations and swap it to the Top-Left corner block (TL).
;		(NB: block rows and block cols can be swapped at will without affecting the solution.)
;		Find the next block with most remaining permutations and swap it to the Bottom-Right corner block (BR).
;	6. Brute-Force: cycle through all valid permutations in nested loops for
;		the six off-diagonal blocks, with early-outs for any block that has zero valid permuts left.
;		In the innermost loop, permutations of the three diagonal blocks are
;		thereby reduced to either 1 or 0. Evaluate and report if sudoku is valid.
;
; * Note in particular how the following functions are used:
;	+ _Eigen_ConditMask, to produce bitmasks;
;	+ _Eigen_CwiseLogicalOp with operator "xcl", as positive exclusion filter (1 = out, 0 = keep);
;	+ _Eigen_CwiseLogicalOp with operator "and", as negative exclusion filter (1 = keep, 0 = out);
;	+ _Eigen_CwiseLogicalOp with operator "or", to combine partially overlapping subsets;
;	+ _Eigen_ReDim_ExistingMatrix, to switch data representation between vector- and block-shaped;
;	+ _FillCandidValues, to multiply a LinSpaced_Colwise matrix ($candidvalues) with a bitmask ($candidates) to list remaining candidate values per cell (0=invalid);
;	+ _Eigen_Swap_Arow_Arow/_Eigen_Swap_Acol_Acol, to quickly re-arrange block references (the blocks themselves are untouched);
;	+ _FillMasksOfValidNeighbours, to create two subsets of 12,096 valid permuts, one horizontally and one vertically;
;	+ _MaskValidPermutations, to apply up to four valid-neighbour bitmasks to reduce a block permutation subset;
;	+ _Eigen_IsZero, as early-out for invalid block combinations;
;	+ _Eigen_GetSum, to determine how many valid permutations are left for a block;
;	+ _Eigen_GetCol_MaxVal, to retrieve the first-found/last-remaining valid permutation ID.
;
;============================================================================================
; To increase available memory on x64 hardware+OS, enable this line and adjust EIGEN_ALLOCATE_RAM_GB in Eigen4AutoIt.ini before running this script
;#AutoIt3Wrapper_UseX64=Y

#NoTrayIcon
#include <ColorConstants.au3>
#include <File.au3>
#include <GuiConstantsEx.au3>
#include <GuiStatusBar.au3>
#include <Misc.au3>
#include <ProgressConstants.au3>
#include <Timers.au3>
#include <WindowsConstants.au3>

#include ".\Eigen4AutoIt.au3"	; << PLEASE EDIT THIS PATH FIRST, IF NEED BE

;_____________________________________________________________________________
; GLOBALS

; Number of permutations in a 3x3 block containing digits 1-9 once each
Global Const $permuts=362880	; 9! (9 factorial = 9 x 8 x 7 x 6 x 5 x 4 x 3 x 2 x 1)

; global matrices, multi-matrix arrays, and related vars
Global $sdks,$sdkDone,$sdklevel,$sdkSelected,$aSudoku[81]	; 1D array
Global $mSudoku,$solution,$solutionRow,$solutionCol,$solutionBlock
Global $candidates,$digitInRows,$digitInCols,$matDigitBuffer
Global $retainRowVec,$cellRowVec,$valueRowVec,$digitColvec,$digitRowvec,$valuecolvec,$scratchRowVec,$blockRowVec
Global $clueslist,$totalclues,$solutionsFound=0,$tstartbrute=False,$selectedLevel=0,$tstartcross=False

; blocks-related
Global $blockTL,$blockTC,$blockTR
Global $blockCL,$blockCC,$blockCR
Global $blockBL,$blockBC,$blockBR
Global $blockneighbours,$blockfreqs,$blockfreqsum,$blockPermuts,$orderBF
Global $blockIDadjRow1,$blockIDadjRow2,$blockIDadjCol1,$blockIDadjCol2
Global $blockSolved[9],$blockTag[9]=["TL","CL","BL", "TC","CC","BC","TR","CR","BR"]

; operational settings
Global $XfilterLastPass=1	; 0: keep filtering until no more improvement; 1-N: exit after at most N passes; -1: disable
Global $defaultLevel=Number(IniRead("XFBFSS.ini","selection","lastused",1))
Global $delayInSec=2		; results display duration before proceeding with next sudoku
Global $msgDuration=4		; how long any message is displayed

; flags
Global $showGUI=True			; T: (SLOW!) display partial solution in progress; F: don't
Global $pickRandomSudoku=True	; T: pick from sdk list at random; F: process list sequentially
Global $verbose=True			; T: write status updates to console; F: don't
Global $exitUpon1stHit=True		; T: early-out; F: exhaustive BF search for all solutions

;_____________________________________________________________________________
; CREATE WORK ENVIRONMENT

SRandom(@AutoItPID+@MSEC)

; check E4A
If _Eigen_GetVersion<4.9 Then _FatalError("E4A version 4.9 or higher is required")
ConsoleWrite("Initialising BitMask Sudoku Solver..." & @CRLF & @CRLF)

; Beginning of Story...
_Eigen_StartUp("int")		; integers are most suitable for bitmask handling
_Eigen_SetLogicalBit0only()	; all CwiseLogicalOps are to act upon bit-0 only (for binary bitmask handling)
_Eigen_Show_MatrixDisplay()

; GENERATE ALL LOOKUP TABLES
; create blockID lookup table
$matBlockIDs=_Eigen_CreateMatrix_LinSpaced_Colwise(3,3,0,8)
$matBlocks=_Eigen_CreateMatrix_Zero(9,9)

For $cc=0 To 2
	$startcol=$cc*3

	For $rc=0 To 2
		$startrow=$rc*3

		_Eigen_SetConstant_Block($matBlocks,$startrow,$startcol,3,3,$startcol+$rc)
	Next
Next

; create block neighbour lookup table
Global $blockneighbours[9][4]

For $bc=0 To 8	; identify and store block's neighbours
	$blockneighbours[$bc][0]=Mod($bc+3,9)					; near row neighbour
	$blockneighbours[$bc][1]=Mod($bc+6,9)					; far row neighbour
	$blockneighbours[$bc][2]=(Int($bc/3)*3)+Mod($bc+1,3)	; near col neighbour
	$blockneighbours[$bc][3]=(Int($bc/3)*3)+Mod($bc+2,3)	; far col neighbour
Next

; create full permutation rowvector buffers (needed for creating/loading the file tables)
$matMrowvec=_Eigen_CreateMatrix(1,$permuts)
$matMcolvec=_Eigen_CreateMatrix(1,$permuts)	; "colvec" because the stored data relate to column indices (shape is rowvec)

; load the big tables from file (regenerated if not found in local dir)
_LoadBlockPermutations(); 9 rows (storing digits 1-9, base-1) x 362880 cols (block permutations)
_LoadDigitTables()		; 3x9 rows (row c.q. col 0/1/2 for digits 1-9, for each block permutation)

; create a mirror for the digits buffer
$matDigitBuffer = _Eigen_CloneMatrix($digitInRows,False)

; create block permutation mask buffers
; these arrays merely contain matrix IDs (integers; indices to $MatrixList),
; illustrating how to select from multiple matrices depending on input
Global $punchout[9]
Global $matpermut[9]
Global $validRowNeighbours[9]
Global $validColNeighbours[9]

For $dc=0 to 8	; only matrix IDs are stored here, not matrix contents
	$punchOut[$dc]=_Eigen_CreateMatrix_Ones(1,$permuts)
	$matpermut[$dc]=_Eigen_CreateMatrix_Ones(1,$permuts)

	$validRowNeighbours[$dc]=_Eigen_CreateMatrix_Zero(1,$permuts)
	$validColNeighbours[$dc]=_Eigen_CreateMatrix_Zero(1,$permuts)
Next

; Note: we allocate all matrices of known dimensions BEFORE we do any work,
; so we won't waste time and effort if memory turns out to be insufficient.
; If more memory is needed, increase the EIGEN_ALLOCATE_RAM_GB value in
; the local Eigen4AutoIt.ini and rerun in x64-mode.

; create punchout mirror for block freqs
$matPunchOut=_Eigen_CreateMatrix(9,$permuts)	; content is based upon clues
$savePunchOut=_Eigen_CreateMatrix(9,$permuts)	; for restoring after trial & error

; create all RowVec buffers
$cellRowVec=_Eigen_CreateMatrix (1,81)
$cellRowVec2=_Eigen_CreateMatrix (1,81)
$cellRowVec3=_Eigen_CreateMatrix (1,81)

$valueRowVec=_Eigen_CreateMatrix (1,81)
$valueRowVec2=_Eigen_CreateMatrix (1,81)

$retainRowVec=_Eigen_CreateMatrix (1,81)
$scratchRowVec=_Eigen_CreateMatrix (1,81)

; create index lookup tables
$blockindexRowVec=_Eigen_CreateMatrix (1,81)
$rowindexRowVec=_Eigen_CreateMatrix (1,81)
$colindexRowVec=_Eigen_CreateMatrix (1,81)

$digitRowvec=_Eigen_CreateMatrix (3*9,1)	; row data stored as colvec
$digitColvec=_Eigen_CreateMatrix (3*9,1)

; create solution buffers
$solution=_Eigen_CreateMatrix_Zero (9,9)
$solutionRow=_Eigen_CreateMatrix(1,9)
$solutionCol=_Eigen_CreateMatrix(9,1)
$solutionBlock=_Eigen_CreateMatrix(3,3)

; create solver-related buffers
$candidates=_Eigen_CreateMatrix (9,81)	; digits x cells
$savecandidates=_Eigen_CreateMatrix (9,81)	; for restoring after trial & error
$candidValues=_Eigen_CreateMatrix (9,81)

$valuecolvec=_Eigen_CreateMatrix (9,1)
$valuecolvec2=_Eigen_CreateMatrix (9,1)
$valuecolvec3=_Eigen_CreateMatrix (9,1)
$scratchcolvec=_Eigen_CreateMatrix (9,1)
$blockfreqs=_Eigen_CreateMatrix (9,1)

Global $maxOrderBF=100
$orderBF=_Eigen_CreateMatrix_Zero ($maxOrderBF,9)

; create candidate value mirrors
Global $candidate[9]
For $dc=0 To 8
	$candidate[$dc]=_Eigen_CreateMatrix_Ones (1,81)
Next

; create all buddy tables
$buddies=_Eigen_CreateMatrix(3,81)

Global $rowbuddymask[9],$colbuddymask[9],$blockbuddymask[9],$allbuddies[81]
For $dc=0 To 8
	$rowbuddymask[$dc]=_Eigen_CreateMatrix (1,81)
	$colbuddymask[$dc]=_Eigen_CreateMatrix (1,81)
	$blockbuddymask[$dc]=_Eigen_CreateMatrix (1,81)
Next

; fill index and buddy tables
For $cc=0 To 8
	For $rc=0 To 8
		$cll=$rc+($cc*9)
		$blockID=_Eigen_ReadMatrixValue($matBlocks,$rc,$cc)

		_Eigen_WriteMatrixValue($rowindexRowVec,0,$cll,$rc)
		_Eigen_WriteMatrixValue($colindexRowVec,0,$cll,$cc)
		_Eigen_WriteMatrixValue($blockindexRowVec,0,$cll,$blockID)

		_Eigen_WriteMatrixValue($buddies,0,$cll,$rc)
		_Eigen_WriteMatrixValue($buddies,1,$cll,$cc)
		_Eigen_WriteMatrixValue($buddies,2,$cll,$blockID)
	Next
Next

For $dc=0 To 8
	_Eigen_ConditMask_Row($buddies,0,"==",$dc,$rowbuddymask[$dc])
	_Eigen_ConditMask_Row($buddies,1,"==",$dc,$colbuddymask[$dc])
	_Eigen_ConditMask_Row($buddies,2,"==",$dc,$blockbuddymask[$dc])
Next

For $cll=0 To 80
	_Eigen_ConditMask_Row($buddies,0,"==",_Eigen_ReadMatrixValue($buddies,0,$cll),$retainRowVec)
	_Eigen_ConditMask_Row($buddies,1,"==",_Eigen_ReadMatrixValue($buddies,1,$cll),$scratchRowVec)

	_Eigen_CwiseLogicalOp_InPlace($retainRowVec,"or",$scratchRowVec)
	_Eigen_ConditMask_Row($buddies,2,"==",_Eigen_ReadMatrixValue($buddies,2,$cll),$scratchRowVec)

	_Eigen_CwiseLogicalOp_InPlace($retainRowVec,"or",$scratchRowVec)

	$allbuddies[$cll]=_Eigen_CloneMatrix($retainRowVec)
Next
_Eigen_ReleaseMatrix($buddies)


; store lists of masked cells per block for display
Global $cellIDsBlock0=_Eigen_FindAll($blockbuddymask[0],1)
Global $cellIDsBlock1=_Eigen_FindAll($blockbuddymask[1],1)
Global $cellIDsBlock2=_Eigen_FindAll($blockbuddymask[2],1)
Global $cellIDsBlock3=_Eigen_FindAll($blockbuddymask[3],1)
Global $cellIDsBlock4=_Eigen_FindAll($blockbuddymask[4],1)
Global $cellIDsBlock5=_Eigen_FindAll($blockbuddymask[5],1)
Global $cellIDsBlock6=_Eigen_FindAll($blockbuddymask[6],1)
Global $cellIDsBlock7=_Eigen_FindAll($blockbuddymask[7],1)
Global $cellIDsBlock8=_Eigen_FindAll($blockbuddymask[8],1)

;_____________________________________________________________________________
; PREP GUI

If $showGUI Then

	Global $progress[9],$labelProgress[9]
	Global $barblockorder[9]=[0,3,6,1,4,7,2,5,8]
	Global $keypressed=False

	Global $cellsize=40
	Global $cellshiftH=$cellsize*.65
	Global $cellshiftV=$cellsize*.82
	Global $digitscale=.6
	Global $linewidth=4
	Global $size=9*$cellsize
	Global $guiblocksize=$size+$linewidth

	; define colours
	Global $litelinecolor=0xa0a0a0	; light grey
	Global $darklinecolor=0x404040	; dark grey
	Global $cluescolor=0x0000a0		; dark blue
	Global $guesscolor=0x800000		; dark red
	Global $acceptcolor=$COLOR_GREEN

	; Create main GUI
	$wintitle="Cross-Filter Sudoku Solver"
	$wintitleSufffix="     [ press <Esc> to Quit ]"
	$GUI = GUICreate($wintitle & $wintitleSufffix, $guiblocksize*2, $guiblocksize)
	GUISetState()

	; create sudoku grid in left panel
	Global $GUI1 = GUICreate("Sudoku",$size+$linewidth,$size+$linewidth, 0, 0, $WS_POPUP, $WS_EX_MDICHILD, $GUI)
	Global $digits[0]
	For $cc = $cellsize to $size Step $cellsize
		GUICtrlCreateLabel("",0,$cc,$size,$linewidth)
		GUICtrlSetBkColor(-1,((mod($cc,$cellsize*3)=0)?($darklinecolor):($litelinecolor)))

		GUICtrlCreateLabel("",$cc,0,$linewidth,$size)
		GUICtrlSetBkColor(-1,(mod($cc,$cellsize*3)=0)?($darklinecolor):($litelinecolor))

		For $rc = $cellsize to $size Step $cellsize
			_ArrayAdd($digits,GUICtrlCreateLabel("",$cc-$cellshiftH,$rc-$cellshiftV,$cellsize*$digitscale,$cellsize*$digitscale*1.2))
			GUICtrlSetFont(-1,18,700,Default,"Arial")
		Next
	Next

	; display instructions to user
	$msg=	"Select: " & _
			" 1= Easy " & _
			" 2= Hard " & _
			" 3=Xtreme" & _
			"Press anyother key to load  an sdk    file.  "
	For $rc=0 To 8
		For $cc=0 To 8
			GUICtrlSetData($digits[$rc+$cc*9],StringMid($msg,$cc+$rc*9,1))
		Next
	Next
	GUISetState()

	; create 9 block bars in right panel
	Global $GUI2 = GUICreate("Bars",$size+$linewidth,$size+$linewidth, $guiblocksize, 0, $WS_POPUP, $WS_EX_MDICHILD, $GUI)
	For $bc=0 To 8
		_CreateBar($GUI2,$bc,$barblockorder[$bc])
	Next
	GUISetState()

	; activate and wait for user input on difficulty level or sdk-file to load
	WinActivate($GUI)
	GUIRegisterMsg($WM_KEYDOWN,"IsPressed")

	While Not $keypressed
		Switch GUIGetMsg()
			Case $GUI_EVENT_CLOSE
				GUIDelete($GUI)
				Exit(0)
		EndSwitch
		Sleep(250)
	WEnd
	GUIRegisterMsg($WM_KEYDOWN,"")
	IniWrite("XFBFSS.ini","selection","lastused",$selectedLevel)

Else
	$selectedLevel=$defaultLevel	; read from XFBFSS.ini
EndIf

_LoadSudokus($selectedLevel)
HotKeySet("{Esc}","IsPressed")

;_____________________________________________________________________________
; MAIN LOOP

For $sdkc=1 To $sdks[0]

	; initialise work environment
	_PrepAllBuffers()
	If _LoadNextSudoku($sdkc)=False Then ContinueLoop	; skip invalid input

	; punch-in all known values
	If _ProcessAllClues($sdkc)=False Then ContinueLoop

	; evaluate remaining candidates
	_UpdateCandidatesPermuts()
	If _CheckAllBlocks()=True And $exitUpon1stHit Then ContinueLoop	 ; already solved

	; reduce each block's permutation space in 1-N passes (controlled by $XfilterLastPass)
	If $XfilterLastPass>=0 Then	; set negative to skip Xfiltering altogether
		If _CrossFilter()=False Then ContinueLoop

		; evaluate remaining candidates
		_UpdateCandidatesPermuts()
		If _CheckAllBlocks()=True And $exitUpon1stHit Then ContinueLoop	; already solved
	EndIf

	; apply brute force to find solution(s)
	_OptimiseMax()	; rearrange block refs to move highest two freqs to TL & BR
	_BruteForce()	; cycle through remaining off-diagonal block permuts
	_Report()		; show final result

Next

_CloseDown()

;=============================================================================
; GUI-related

Func _ShowBlock($blockID,$permutID)

	Switch $blockID
		Case 0
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock0,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 1
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock1,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 2
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock2,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 3
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock3,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 4
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock4,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 5
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock5,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 6
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock6,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 7
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock7,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next

		Case 8
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock8,$rc,0)],_Eigen_ReadMatrixValue($blockPermuts,$rc,$permutID))
			Next
	EndSwitch

EndFunc


Func _ClearBlock($blockID)

	If $blockSolved[$blockID] Then Return	; reduce flicker

	Switch $blockID
		Case 0
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock0,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock0,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock0,$rc,1))))
			Next

		Case 1
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock1,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock1,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock1,$rc,1))))
			Next

		Case 2
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock2,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock2,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock2,$rc,1))))
			Next

		Case 3
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock3,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock3,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock3,$rc,1))))
			Next

		Case 4
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock4,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock4,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock4,$rc,1))))
			Next

		Case 5
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock5,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock5,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock5,$rc,1))))
			Next

		Case 6
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock6,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock6,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock6,$rc,1))))
			Next

		Case 7
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock7,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock7,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock7,$rc,1))))
			Next

		Case 8
			For $rc=0 To 8
				GUICtrlSetData($digits[_Eigen_ReadMatrixValue($cellIDsBlock8,$rc,0)],((_Eigen_ReadMatrixValue($cellIDsBlock8,$rc,1)=0)?(""):(_Eigen_ReadMatrixValue($cellIDsBlock8,$rc,1))))
			Next
	EndSwitch

EndFunc


Func _SetColorBlock($blockID,$color)

	If $color=$acceptcolor And Not $exitUpon1stHit Then Return

	Switch $blockID
		Case 0
			For $rc=0 To 8
				GUICtrlSetColor($digits[_Eigen_ReadMatrixValue($cellIDsBlock0,$rc,0)],$color)
			Next

		Case 1
			For $rc=0 To 8
				GUICtrlSetColor($digits[_Eigen_ReadMatrixValue($cellIDsBlock1,$rc,0)],$color)
			Next

		Case 2
			For $rc=0 To 8
				GUICtrlSetColor($digits[_Eigen_ReadMatrixValue($cellIDsBlock2,$rc,0)],$color)
			Next

		Case 3
			For $rc=0 To 8
				GUICtrlSetColor($digits[_Eigen_ReadMatrixValue($cellIDsBlock3,$rc,0)],$color)
			Next

		Case 4
			For $rc=0 To 8
				GUICtrlSetColor($digits[_Eigen_ReadMatrixValue($cellIDsBlock4,$rc,0)],$color)
			Next

		Case 5
			For $rc=0 To 8
				GUICtrlSetColor($digits[_Eigen_ReadMatrixValue($cellIDsBlock5,$rc,0)],$color)
			Next

		Case 6
			For $rc=0 To 8
				GUICtrlSetColor($digits[_Eigen_ReadMatrixValue($cellIDsBlock6,$rc,0)],$color)
			Next

		Case 7
			For $rc=0 To 8
				GUICtrlSetColor($digits[_Eigen_ReadMatrixValue($cellIDsBlock7,$rc,0)],$color)
			Next

		Case 8
			For $rc=0 To 8
				GUICtrlSetColor($digits[_Eigen_ReadMatrixValue($cellIDsBlock8,$rc,0)],$color)
			Next
	EndSwitch

EndFunc


Func _CreateBar($hGUI,$index,$blockID)
; for displaying remaining valid permutations per block on a log10-scale

	Local $height=$index*$guiblocksize/9
	Local $barheight=($guiblocksize/9)*.5
	Local $labelwidth=100
	Local $margin=10
	$labelProgress[$blockID]=GUICtrlCreateLabel("Block " & $blockTag[$blockID] & ": ######",$margin,$height+$barheight*.6,$size,$labelwidth)
    $progress[$blockID]=GUICtrlCreateProgress($labelwidth+$margin*2, $margin+$height, $guiblocksize-$labelwidth-$margin*3, $barheight)

	; enable multi-coloured bars
	DllCall("UxTheme.dll","int","SetWindowTheme","hwnd",GUICtrlGetHandle(-1),"wstr",0,"wstr",0)
    GUICtrlSetStyle(-1, BitOr($GUI_SS_DEFAULT_PROGRESS, $PBS_SMOOTH))
	GUICtrlSetBkColor(-1, $COLOR_CREAM)

EndFunc


Func _HighLightBar($blockID,$color=$cluescolor,$highlight=True)

	Switch $highlight
		Case True
			GUICtrlSetColor($labelProgress[$blockID],$color)
			GUICtrlSetFont($labelProgress[$blockID],9, 800) ; bold

		Case Else
			GUICtrlSetColor($labelProgress[$blockID],$COLOR_BLACK)
			GUICtrlSetFont($labelProgress[$blockID],9, 400) ; regular
	EndSwitch

EndFunc


Func _UpdateBar($blockID,$value)

	Local $color,$logvalue=_Min(4,(_Max(0,Log($value)/Log(10))))

	Switch Ceiling($logvalue)
		Case 0
			$color=$acceptcolor
		Case 1
			$color=$COLOR_YELLOW
		Case 2
			$color=0xFF8C00	; $COLOR_ORANGE
		Case 3
			$color=$COLOR_RED
		Case Else
			$color=$COLOR_PURPLE
	EndSwitch

	GUICtrlSetColor($progress[$blockID], $color)
	GUICtrlSetData($labelprogress[$blockID],"Block " & $blockTag[$blockID] & ": " & $value)
	GUICtrlSetData($progress[$blockID],100*((1+$logvalue)/5))

EndFunc


Func _BlankGridBars()

	; blank the grid
	For $cc=0 To 80
		GUICtrlSetData($digits[$cc],"")
	Next

	For $bc=0 To 8
		_SetColorBlock($bc,$guesscolor)
		GUICtrlSetData($labelprogress[$bc],"Block " & $blockTag[$bc] & ": ######")
		GUICtrlSetData($progress[$bc],0)
	Next

EndFunc


Func _FillAllBars()

	For $bc=0 To 8
		GUICtrlSetData($labelprogress[$bc],"Block " & $blockTag[$bc] & ": 362,880")
		GUICtrlSetColor($progress[$bc], $COLOR_PURPLE)
		GUICtrlSetData($progress[$bc],100)
	Next

EndFunc


Func _UpdateAllBars()

	_Eigen_ReDim_ExistingMatrix($blockfreqs,9,1)

	For $bc=0 To 8
		_UpdateBar($bc,_Eigen_ReadMatrixValue($blockfreqs,$bc,0))
	Next

EndFunc


Func IsPressed()

	$keypressed=True
	If _IsPressed("1B") Then _CloseDown()	; <Esc>

	For $cc=0 to 3
		If _IsPressed("3"&$cc) Then
			$selectedLevel=$cc	; number keys 0-3
			Return
		EndIf
	Next
	$selectedLevel=0	; any other key

EndFunc


Func _FatalError($errmsg="an unknown fatal error occurred")

	ConsoleWrite("FATAL ERROR: " & $errmsg & @CRLF)
	If $showGUI Then MsgBox(16,"Fatal Error", $errmsg,$msgDuration)
	_CloseDown(-1)

EndFunc


Func _Error($errmsg="an unknown error occurred")

	ConsoleWrite("ERROR: " & $errmsg & @CRLF)
	If $showGUI Then MsgBox(48,"Error", $errmsg,$msgDuration)

EndFunc


Func _Report()

	If $solutionsFound>0 Then
		_Eigen_ShowMatrix($solution,$sdklevel & " sudoku"&(($selectedLevel=0)?(""):(" #" &$sdkSelected))&" solution found:")
		If $showGUI Then
			_Eigen_SetOnes($blockfreqs)	; just for display
			_UpdateAllBars()
		EndIf
	EndIf

	; report durations
	If $tstartcross Then ConsoleWrite(@CRLF & "Cross-Solving took: " & Round(TimerDiff($tstartcross)/1000,2) & " seconds" & @CRLF)
	If $tstartbrute Then ConsoleWrite(@CRLF & "Brute-Force took: " & Round(TimerDiff($tstartbrute)/1000,2) & " seconds" & @CRLF)
	ConsoleWrite(@CRLF & "Solution(s) found: " & $solutionsFound & @CRLF & @CRLF)

	Sleep($delayInSec*1000)

EndFunc


Func _CloseDown($exitcode=0)

	HotKeySet("{Esc}")
	GUIRegisterMsg($WM_KEYDOWN,"")

	If $verbose Then ConsoleWrite(@CRLF & "Please wait; closing down..." & @CRLF)
	If $showGUI Then GUIDelete($GUI)

	_Eigen_CleanUp()	; ...End of Story
	Exit($exitcode)

EndFunc

;_____________________________________________________________________________
; File-related (pregenerated lookup tables and example sudokus)

Func _LoadBlockPermutations()
; 9 rows (digits 1-9, base-1) x 362,880 cols (block permutations)

	Local $filename="blockpermut.mat"

	If FileExists($filename) Then
		$blockPermuts = _Eigen_LoadMatrix($filename)
		Return
	EndIf

	If $verbose Then ConsoleWrite("Generating lookup table file " & $filename & "; please wait..." & @CRLF & "NB This only needs to be done ONCE." & @CRLF)

	$blockPermuts = _Eigen_CreateMatrix (9,$permuts)
	Local $matC = _Eigen_CreateMatrix (9,1)

	Local $rc=-1
	For $c0=1 To 9
		_Eigen_WriteMatrixValue($matC,0,0,$c0)

	For $c1=1 To 9
		If $c1=$c0 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,1,0,$c1)

	For $c2=1 To 9
		If $c2=$c0 Then ContinueLoop
		If $c2=$c1 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,2,0,$c2)

	For $c3=1 To 9
		If $c3=$c0 Then ContinueLoop
		If $c3=$c1 Then ContinueLoop
		If $c3=$c2 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,3,0,$c3)

	For $c4=1 To 9
		If $c4=$c0 Then ContinueLoop
		If $c4=$c1 Then ContinueLoop
		If $c4=$c2 Then ContinueLoop
		If $c4=$c3 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,4,0,$c4)

	For $c5=1 To 9
		If $c5=$c0 Then ContinueLoop
		If $c5=$c1 Then ContinueLoop
		If $c5=$c2 Then ContinueLoop
		If $c5=$c3 Then ContinueLoop
		If $c5=$c4 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,5,0,$c5)

	For $c6=1 To 9
		If $c6=$c0 Then ContinueLoop
		If $c6=$c1 Then ContinueLoop
		If $c6=$c2 Then ContinueLoop
		If $c6=$c3 Then ContinueLoop
		If $c6=$c4 Then ContinueLoop
		If $c6=$c5 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,6,0,$c6)

	For $c7=1 To 9
		If $c7=$c0 Then ContinueLoop
		If $c7=$c1 Then ContinueLoop
		If $c7=$c2 Then ContinueLoop
		If $c7=$c3 Then ContinueLoop
		If $c7=$c4 Then ContinueLoop
		If $c7=$c5 Then ContinueLoop
		If $c7=$c6 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,7,0,$c7)

	For $c8=1 To 9
		If $c8=$c0 Then ContinueLoop
		If $c8=$c1 Then ContinueLoop
		If $c8=$c2 Then ContinueLoop
		If $c8=$c3 Then ContinueLoop
		If $c8=$c4 Then ContinueLoop
		If $c8=$c5 Then ContinueLoop
		If $c8=$c6 Then ContinueLoop
		If $c8=$c7 Then ContinueLoop
		_Eigen_WriteMatrixValue($matC,8,0,$c8)

		$rc+=1
		_Eigen_Copy_Acol_ToBcol($matC,$blockPermuts,0,$rc)
		If $verbose And Mod($rc,1000)=0 Then ConsoleWrite("Processed " & $rc & " of " & $permuts & @CRLF)
	Next
	Next
	Next
	Next
	Next
	Next
	Next
	Next
	Next

	If $verbose Then ConsoleWrite("Saving file..." & @CRLF & @CRLF)
	_Eigen_SaveMatrix($blockPermuts,$filename,True)
	_Eigen_ReleaseMatrix($matC)

EndFunc


Func _LoadDigitTables()
; 3x9 rows (row c.q. col 0/1/2 for digits 1-9, for each block permutation)

	Local $filename1="digitInRows.mat"
	Local $filename2="digitInCols.mat"

	If (FileExists($filename1) And FileExists($filename2)) Then
		$digitInRows=_Eigen_LoadMatrix($filename1)	; permut x colvectors (of size: 27 rows: 0/1/2 for digit 1, 0/1/2 for digit 2, etc)
		$digitInCols=_Eigen_LoadMatrix($filename2)	; permut x rowvectors^T (of size: 27 rows: 0/1/2 for digit 1, 0/1/2 for digit 2, etc)
		Return
	EndIf

	If $verbose Then ConsoleWrite("Generating lookup table files " & $filename1 & " and " & $filename2 & "; please be patient..." & @CRLF & "NB This only needs to be done ONCE." & @CRLF)

	Local $matB = _Eigen_CreateMatrix (9,1)
	Local $matDrows = _Eigen_CreateMatrix (3,1)	; colvector
	Local $matDcols = _Eigen_CreateMatrix (1,3)	; rowvector
	Local $matDcolsT = _Eigen_CreateMatrix (3,1)	; rowvector

	$digitInRows = _Eigen_CreateMatrix_Zero (3*9,$permuts)	; colvectors
	$digitInCols = _Eigen_CreateMatrix_Zero (3*9,$permuts)	; rowvectors

	For $pc=0 To $permuts-1
		_Eigen_Copy_Acol_ToBcol($blockPermuts,$matB,$pc,0)

		_Eigen_Redim_ExistingMatrix ($matB,3,3)
		For $dc=1 To 9
			$offset=($dc-1)*3
			_Eigen_ConditAny_Rowwise($matB,"=",$dc,$matDrows)
			_Eigen_ConditAny_Colwise($matB,"=",$dc,$matDcols)
			_Eigen_Transpose($matDcols,$matDcolsT)

			_Eigen_Copy_Ablock_ToBblock($matDrows,$digitInRows,0,0,3,1,$offset,$pc)
			_Eigen_Copy_Ablock_ToBblock($matDcolsT,$digitInCols,0,0,3,1,$offset,$pc)
		Next
		_Eigen_Redim_ExistingMatrix ($matB,9,1)

		If $verbose And Mod($pc,100)=0 Then ConsoleWrite("Processed " & $pc & " of " & $permuts & @CRLF)
	Next

	If $verbose Then ConsoleWrite("Saving files..." & @CRLF & @CRLF)
	_Eigen_SaveMatrix($digitInRows,$filename1,True)
	_Eigen_SaveMatrix($digitInCols,$filename2,True)

	_Eigen_ReleaseMatrix($matB)
	_Eigen_ReleaseMatrix($matDrows)
	_Eigen_ReleaseMatrix($matDcols)
	_Eigen_ReleaseMatrix($matDcolsT)

EndFunc


Func _LoadSudokus($level=$defaultLevel)

	Global $sdks[1]

	Switch $level
		Case 1
			_LoadEasySudokus()
			$defaultLevel=1
		Case 2
			_LoadHardSudokus()
			$defaultLevel=2
		Case 3
			_LoadExtremeSudokus()
			$defaultLevel=3

		Case Else
			$defaultLevel=0
			$fname=FileOpenDialog("Please select an sdk file",@ScriptDir,"(*.sdk; *.txt)",1)
			If @error Then _FatalError("file selection error code: " & @error)
			If _FileCountLines($fname)<>9 Then _FatalError("file access error or unrecognised format; text file with 9 lines expected, containing 9 single digits (1-9) or dots ('.') for empty cells")

			Local $fh=FileOpen($fname)
			If @error then _FatalError("unable to access file: " & $fname)

			Local $line=""
			For $rc=0 To 8
				$line&=StringReplace(StringStripWS(FileReadLine($fh),8),"0",".")
				If @error Then
					FileClose($fh)
					_FatalError("unrecognised format; text file with 9 lines expected, containing 9 single digits (1-9) or dots ('.') for empty cells; spaces are ignored.")
				EndIf
			Next

			FileClose($fh)
			_ArrayAdd($sdks,$line)
	EndSwitch
	$sdks[0]=UBound($sdks)-1

	Global $sdkDone=$sdks
	For $sc=1 To $sdks[0]
		$sdkDone[$sc]=False
	Next

EndFunc


Func _LoadNextSudoku($selected)

	If $pickRandomSudoku Then
		Local $todo=_ArrayFindAll($sdkdone,False)
		If @error Or Not IsArray($todo) Then _CloseDown(1)
		If UBound($todo)<1 Then _CloseDown(2)

		$sdkSelected=$todo[Random(0,UBound($todo)-1,1)]
	Else
		If $selected<1 Or $selected>$sdks[0] Then Return False
		$sdkSelected=$selected
	EndIf
	$sdkDone[$sdkselected]=True

	$aSudoku=StringSplit(StringReplace(StringStripWS($sdks[$sdkSelected],8),".","0"),"",2)
	If @error Or Not IsArray($aSudoku) Then Return False

	_CreateSudokuMatrix($aSudoku,True)	; T: stored as 1D rowwise array

	Return True
EndFunc


Func _CreateSudokuMatrix(ByRef $array,$stored1Drowwise=False)

	If UBound($array,0)>2 Then _FatalError("1D or 2D array expected")
	If UBound($array,0)=1 and UBound($array,1)<>81 Then _FatalError("1D sudoku array has unexpected shape")
	If UBound($array,0)=2 And (UBound($array,1)<>9 Or UBound($array,2)<>9) Then _FatalError("2D sudoku array has unexpected shape")

	$mSudoku=_Eigen_CreateMatrix_fromArray($array)	; global
	If @error Then _FatalError("unable to create input matrix")
	If _Eigen_GetMatrixSize($mSudoku)<>81 Then _FatalError("parsed array should have exactly 81 integer values (0 for blanks, 1-9 for clues)")

	If Not _Eigen_IsVector($mSudoku) Then	; 2D-specific checks
		If _Eigen_GetRows($mSudoku)<>9 Or Not _Eigen_IsSquare($mSudoku) Then _FatalError("parsed 2D array should have 9 x 9 integer values (0 or dots for blanks, 1-9 for clues)")
	EndIf

	; check valid content
	If _Eigen_IsZero($mSudoku) Then _FatalError("parsed sudoku array is completely empty (all zeroes)")
	If _Eigen_GetMinVal($mSudoku)<0 Then _FatalError("negative values in sudoku array cannot be processed")
	If _Eigen_GetMinVal($mSudoku)>0 Then _FatalError("no blank values found to resolve in parsed sudoku array")
	If _Eigen_GetMaxVal($mSudoku)>9 Then _FatalError("largest value in array exceeds maximum of 9")

	; reformat valid vector input to 9x9 square matrix
	_Eigen_ReDim_ExistingMatrix($mSudoku,9,9)	; no change if already 2D
	If $stored1Drowwise Then _Eigen_Transpose_InPlace($mSudoku)

EndFunc


Func _LoadEasySudokus()

	_ArrayAdd($sdks,"..3.2.6..9..3.5..1..18.64....81.29..7.......8..67.82....26.95..8..2.3..9..5.1.3..")
	_ArrayAdd($sdks,"2...8.3...6..7..84.3.5..2.9...1.54.8.........4.27.6...3.1..7.4.72..4..6...4.1...3")
	_ArrayAdd($sdks,"......9.7...42.18....7.5.261..9.4....5.....4....5.7..992.1.8....34.59...5.7......")
	_ArrayAdd($sdks,".3..5..4...8.1.5..46.....12.7.5.2.8....6.3....4.1.9.3.25.....98..1.2.6...8..6..2.")
	_ArrayAdd($sdks,".2.81.74.7....31...9...28.5..9.4..874..2.8..316..3.2..3.27...6...56....8.76.51.9.")
	_ArrayAdd($sdks,"1..92....524.1...........7..5...81.2.........4.27...9..6...........3.945....71..6")
	_ArrayAdd($sdks,".43.8.25.6.............1.949....4.7....6.8....1.2....382.5.............5.34.9.71.")
	_ArrayAdd($sdks,"48...69.2..2..8..19..37..6.84..1.2....37.41....1.6..49.2..85..77..9..6..6.92...18")
	_ArrayAdd($sdks,"...9....2.5.1234...3....16.9.8.......7.....9.......2.5.91....5...7439.2.4....7...")
	_ArrayAdd($sdks,"..19....39..7..16..3...5..7.5......9..43.26..2......7.6..1...3..42..7..65....68..")

	_ArrayAdd($sdks,"...1254....84.....42.8......3.....95.6.9.2.1.51.....6......3.49.....72....1298...")
	_ArrayAdd($sdks,".6234.75.1....56..57.....4.....948..4.......6..583.....3.....91..64....7.59.8326.")
	_ArrayAdd($sdks,"3..........5..9...2..5.4....2....7..16.....587.431.6.....89.1......67.8......5437")
	_ArrayAdd($sdks,"63..........5....8..5674.......2......34.1.2.......345.....7..4.8.3..9.29471...8.")
	_ArrayAdd($sdks,"....2..4...8.35.......7.6.2.31.4697.2...........5.12.3.49...73........1.8....4...")
	_ArrayAdd($sdks,"361.259...8.96..1.4......57..8...471...6.3...259...8..74......5.2..18.6...547.329")
	_ArrayAdd($sdks,".5.8.7.2.6...1..9.7.254...6.7..2.3.15.4...9.81.3.8..7.9...762.5.6..9...3.8.1.3.4.")
	_ArrayAdd($sdks,".8...5........3457....7.8.9.6.4..9.3..7.1.5..4.8..7.2.9.1.2....8423........1...8.")
	_ArrayAdd($sdks,"..35.29......4....1.6...3.59..251..8.7.4.8.3.8..763..13.8...1.4....2......51.48..")
	_ArrayAdd($sdks,"...........98.51...519.742.29.4.1.65.........14.5.8.93.267.958...51.36...........")

	_ArrayAdd($sdks,".2..3..9....9.7...9..2.8..5..48.65..6.7...2.8..31.29..8..6.5..7...3.9....3..2..5.")
	_ArrayAdd($sdks,"..5.....6.7...9.2....5..1.78.415.......8.3.......928.59.7..6....3.4...1.2.....6..")
	_ArrayAdd($sdks,".4.....5...19436....9...3..6...5...21.3...5.68...2...7..5...2....24367...3.....4.")
	_ArrayAdd($sdks,"..4..........3...239.7...8.4....9..12.98.13.76..2....8.1...8.539...4..........8..")
	_ArrayAdd($sdks,"36..2..89...361............8.3...6.24..6.3..76.7...1.8............418...97..3..14")
	_ArrayAdd($sdks,"5..4...6...9...8..64..2.........1..82.8...5.17..5.........9..84..3...6...6...3..2")
	_ArrayAdd($sdks,"..72564..4.......5.1..3..6....5.8.....8.6.2.....1.7....3..7..9.2.......4..63127..")
	_ArrayAdd($sdks,"..........79.5.18.8.......7..73.68..45.7.8.96..35.27..7.......5.16.3.42..........")
	_ArrayAdd($sdks,".3.....8...9...5....75.92..7..1.5..8.2..9..3.9..4.2..1..42.71....2...8...7.....9.")
	_ArrayAdd($sdks,"2..17.6.3.5....1.......6.79....4.7.....8.1.....9.5....31.4.......5....6.9.6.37..2")

	_ArrayAdd($sdks,".......8.8..7.1.4..4..2..3.374...9......3......5...321.1..6..5..5.8.2..6.8.......")
	_ArrayAdd($sdks,".......85...21...996..8.1..5..8...16.........89...6..7..9.7..523...54...48.......")
	_ArrayAdd($sdks,"6.8.7.5.2.5.6.8.7...2...3..5...9...6.4.3.2.5.8...5...3..5...2...1.7.4.9.4.9.6.7.1")
	_ArrayAdd($sdks,".5..1..4.1.7...6.2...9.5...2.8.3.5.1.4..7..2.9.1.8.4.6...4.1...3.4...7.9.2..6..1.")
	_ArrayAdd($sdks,".53...79...97534..1.......2.9..8..1....9.7....8..3..7.5.......3..76412...61...94.")
	_ArrayAdd($sdks,"..6.8.3...49.7.25....4.5...6..317..4..7...8..1..826..9...7.2....75.4.19...3.9.6..")
	_ArrayAdd($sdks,"..5.8.7..7..2.4..532.....84.6.1.5.4...8...5...7.8.3.1.45.....916..5.8..7..3.1.6..")
	_ArrayAdd($sdks,"...9..8..128..64...7.8...6.8..43...75.......96...79..8.9...4.1...36..284..1..7...")
	_ArrayAdd($sdks,"....8....27.....54.95...81...98.64...2.4.3.6...69.51...17...62.46.....38....9....")
	_ArrayAdd($sdks,"...6.2...4...5...1.85.1.62..382.671...........194.735..26.4.53.9...2...7...8.9...")

	_ArrayAdd($sdks,"...9....2.5.1234...3....16.9.8.......7.....9.......2.5.91....5...7439.2.4....7...")
	_ArrayAdd($sdks,"38..........4..785..9.2.3...6..9....8..3.2..9....4..7...1.7.5..495..6..........92")
	_ArrayAdd($sdks,"...158.....2.6.8...3.....4..27.3.51...........46.8.79..5.....8...4.7.1.....325...")
	_ArrayAdd($sdks,".1.5..2..9....1.....2..8.3.5...3...7..8...5..6...8...4.4.1..7.....7....6..3..4.5.")
	_ArrayAdd($sdks,".8.....4....469...4.......7..59.46...7.6.8.3...85.21..9.......5...781....6.....1.")
	_ArrayAdd($sdks,"9.42....7.1..........7.65.....8...9..2.9.4.6..4...2.....16.7..........3.3....57.2")
	_ArrayAdd($sdks,"...7..8....6....31.4...2....24.7.....1..3..8.....6.29....8...7.86....5....2..6...")
	_ArrayAdd($sdks,"..1..7.9.59..8...1.3.....8......58...5..6..2...41......8.....3.1...2..79.2.7..4..")
	_ArrayAdd($sdks,".....3.17.15..9..8.6.......1....7.....9...2.....5....4.......2.5..6..34.34.2.....")
	_ArrayAdd($sdks,"3..2........1.7...7.6.3.5...7...9.8.9...2...4.1.8...5...9.4.3.1...7.2........8..6")

EndFunc


Func _LoadHardSudokus()
; source: http://magictour.free.fr/top95

	$sdklevel="hard"
	_ArrayAdd($sdks,"1....7.9..3..2...8..96..5....53..9...1..8...26....4...3......1..4......7..7...3..")
	_ArrayAdd($sdks,"4.....8.5.3..........7......2.....6.....8.4......1.......6.3.7.5..2.....1.4......")
	_ArrayAdd($sdks,"52...6.........7.13...........4..8..6......5...........418.........3..2...87.....")
	_ArrayAdd($sdks,"6.....8.3.4.7.................5.4.7.3..2.....1.6.......2.....5.....8.6......1....")
	_ArrayAdd($sdks,"48.3............71.2.......7.5....6....2..8.............1.76...3.....4......5....")
	_ArrayAdd($sdks,"....14....3....2...7..........9...3.6.1.............8.2.....1.4....5.6.....7.8...")
	_ArrayAdd($sdks,"......52..8.4......3...9...5.1...6..2..7........3.....6...1..........7.4.......3.")
	_ArrayAdd($sdks,".524.........7.1..............8.2...3.....6...9.5.....1.6.3...........897........")
	_ArrayAdd($sdks,".923.........8.1...........1.7.4...........658.........6.5.2...4.....7.....9.....")
	_ArrayAdd($sdks,"6..3.2....5.....1..........7.26............543.........8.15........4.2........7..")

	_ArrayAdd($sdks,"..5...987.4..5...1..7......2...48....9.1.....6..2.....3..6..2.......9.7.......5..")
	_ArrayAdd($sdks,"3.6.7...........518.........1.4.5...7.....6.....2......2.....4.....8.3.....5.....")
	_ArrayAdd($sdks,"1.....3.8.7.4..............2.3.1...........958.........5.6...7.....8.2...4.......")
	_ArrayAdd($sdks,"6..3.2....4.....1..........7.26............543.........8.15........4.2........7..")
	_ArrayAdd($sdks,"....3..9....2....1.5.9..............1.2.8.4.6.8.5...2..75......4.1..6..3.....4.6.")
	_ArrayAdd($sdks,"45.....3....8.1....9...........5..9.2..7.....8.........1..4..........7.2...6..8..")
	_ArrayAdd($sdks,".237....68...6.59.9.....7......4.97.3.7.96..2.........5..47.........2....8.......")
	_ArrayAdd($sdks,"..84...3....3.....9....157479...8........7..514.....2...9.6...2.5....4......9..56")
	_ArrayAdd($sdks,".98.1....2......6.............3.2.5..84.........6.........4.8.93..5...........1..")
	_ArrayAdd($sdks,"..247..58..............1.4.....2...9528.9.4....9...1.........3.3....75..685..2...")

	_ArrayAdd($sdks,"4.....8.5.3..........7......2.....6.....5.4......1.......6.3.7.5..2.....1.9......")
	_ArrayAdd($sdks,"1.....7.9.4...72..8.........7..1..6.3.......5.6..4..2.........8..53...7.7.2....46")
	_ArrayAdd($sdks,"4.....3.....8.2......7........1...8734.......6........5...6........1.4...82......")
	_ArrayAdd($sdks,".......71.2.8........4.3...7...6..5....2..3..9........6...7.....8....4......5....")
	_ArrayAdd($sdks,"6..3.2....4.....8..........7.26............543.........8.15........8.2........7..")
	_ArrayAdd($sdks,".47.8...1............6..7..6....357......5....1..6....28..4.....9.1...4.....2.69.")
	_ArrayAdd($sdks,"......8.17..2........5.6......7...5..1....3...8.......5......2..4..8....6...3....")
	_ArrayAdd($sdks,"38.6.......9.......2..3.51......5....3..1..6....4......17.5..8.......9.......7.32")
	_ArrayAdd($sdks,"...5...........5.697.....2...48.2...25.1...3..8..3.........4.7..13.5..9..2...31..")
	_ArrayAdd($sdks,".2.......3.5.62..9.68...3...5..........64.8.2..47..9....3.....1.....6...17.43....")

	_ArrayAdd($sdks,".8..4....3......1........2...5...4.69..1..8..2...........3.9....6....5.....2.....")
	_ArrayAdd($sdks,"..8.9.1...6.5...2......6....3.1.7.5.........9..4...3...5....2...7...3.8.2..7....4")
	_ArrayAdd($sdks,"4.....5.8.3..........7......2.....6.....5.8......1.......6.3.7.5..2.....1.8......")
	_ArrayAdd($sdks,"1....6.8..64..........4...7....9.6...7.4..5..5...7.1...5....32.3....8...4........")
	_ArrayAdd($sdks,"249.6...3.3....2..8.......5.....6......2......1..4.82..9.5..7....4.....1.7...3...")
	_ArrayAdd($sdks,"...8....9.873...4.6..7.......85..97...........43..75.......3....3...145.4....2..1")
	_ArrayAdd($sdks,"...5.1....9....8...6.......4.1..........7..9........3.8.....1.5...2..4.....36....")
	_ArrayAdd($sdks,"......8.16..2........7.5......6...2..1....3...8.......2......7..3..8....5...4....")
	_ArrayAdd($sdks,".476...5.8.3.....2.....9......8.5..6...1.....6.24......78...51...6....4..9...4..7")
	_ArrayAdd($sdks,".....7.95.....1...86..2.....2..73..85......6...3..49..3.5...41724................")

	_ArrayAdd($sdks,".4.5.....8...9..3..76.2.....146..........9..7.....36....1..4.5..6......3..71..2..")
	_ArrayAdd($sdks,".834.........7..5...........4.1.8..........27...3.....2.6.5....5.....8........1..")
	_ArrayAdd($sdks,"..9.....3.....9...7.....5.6..65..4.....3......28......3..75.6..6...........12.3.8")
	_ArrayAdd($sdks,".26.39......6....19.....7.......4..9.5....2....85.....3..2..9..4....762.........4")
	_ArrayAdd($sdks,"2.3.8....8..7...........1...6.5.7...4......3....1............82.5....6...1.......")
	_ArrayAdd($sdks,"6..3.2....1.....5..........7.26............843.........8.15........8.2........7..")
	_ArrayAdd($sdks,"1.....9...64..1.7..7..4.......3.....3.89..5....7....2.....6.7.9.....4.1....129.3.")
	_ArrayAdd($sdks,".........9......84.623...5....6...453...1...6...9...7....1.....4.5..2....3.8....9")
	_ArrayAdd($sdks,".2....5938..5..46.94..6...8..2.3.....6..8.73.7..2.........4.38..7....6..........5")
	_ArrayAdd($sdks,"9.4..5...25.6..1..31......8.7...9...4..26......147....7.......2...3..8.6.4.....9.")

	_ArrayAdd($sdks,"...52.....9...3..4......7...1.....4..8..453..6...1...87.2........8....32.4..8..1.")
	_ArrayAdd($sdks,"53..2.9...24.3..5...9..........1.827...7.........981.............64....91.2.5.43.")
	_ArrayAdd($sdks,"1....786...7..8.1.8..2....9........24...1......9..5...6.8..........5.9.......93.4")
	_ArrayAdd($sdks,"....5...11......7..6.....8......4.....9.1.3.....596.2..8..62..7..7......3.5.7.2..")
	_ArrayAdd($sdks,".47.2....8....1....3....9.2.....5...6..81..5.....4.....7....3.4...9...1.4..27.8..")
	_ArrayAdd($sdks,"......94.....9...53....5.7..8.4..1..463...........7.8.8..7.....7......28.5.26....")
	_ArrayAdd($sdks,".2......6....41.....78....1......7....37.....6..412....1..74..5..8.5..7......39..")
	_ArrayAdd($sdks,"1.....3.8.6.4..............2.3.1...........758.........7.5...6.....8.2...4.......")
	_ArrayAdd($sdks,"2....1.9..1..3.7..9..8...2.......85..6.4.........7...3.2.3...6....5.....1.9...2.5")
	_ArrayAdd($sdks,"..7..8.....6.2.3...3......9.1..5..6.....1.....7.9....2........4.83..4...26....51.")

	_ArrayAdd($sdks,"...36....85.......9.4..8........68.........17..9..45...1.5...6.4....9..2.....3...")
	_ArrayAdd($sdks,"34.6.......7.......2..8.57......5....7..1..2....4......36.2..1.......9.......7.82")
	_ArrayAdd($sdks,"......4.18..2........6.7......8...6..4....3...1.......6......2..5..1....7...3....")
	_ArrayAdd($sdks,".......4...2..4..1.7..5..9...3..7....4..6....6..1..8...2....1..85.9...6.....8...3")
	_ArrayAdd($sdks,"8..7....4.5....6............3.97...8....43..5....2.9....6......2...6...7.71..83.2")
	_ArrayAdd($sdks,".8...4.5....7..3............1..85...6.....2......4....3.26............417........")
	_ArrayAdd($sdks,"....7..8...6...5...2...3.61.1...7..2..8..534.2..9.......2......58...6.3.4...1....")
	_ArrayAdd($sdks,"......8.16..2........7.5......6...2..1....3...8.......2......7..4..8....5...3....")
	_ArrayAdd($sdks,".2..........6....3.74.8.........3..2.8..4..1.6..5.........1.78.5....9..........4.")
	_ArrayAdd($sdks,".52..68.......7.2.......6....48..9..2..41......1.....8..61..38.....9...63..6..1.9")

	_ArrayAdd($sdks,"....1.78.5....9..........4..2..........6....3.74.8.........3..2.8..4..1.6..5.....")
	_ArrayAdd($sdks,"1.......3.6.3..7...7...5..121.7...9...7........8.1..2....8.64....9.2..6....4.....")
	_ArrayAdd($sdks,"4...7.1....19.46.5.....1......7....2..2.3....847..6....14...8.6.2....3..6...9....")
	_ArrayAdd($sdks,"......8.17..2........5.6......7...5..1....3...8.......5......2..3..8....6...4....")
	_ArrayAdd($sdks,"963......1....8......2.5....4.8......1....7......3..257......3...9.2.4.7......9..")
	_ArrayAdd($sdks,"15.3......7..4.2....4.72.....8.........9..1.8.1..8.79......38...........6....7423")
	_ArrayAdd($sdks,"..........5724...98....947...9..3...5..9..12...3.1.9...6....25....56.....7......6")
	_ArrayAdd($sdks,"....75....1..2.....4...3...5.....3.2...8...1.......6.....1..48.2........7........")
	_ArrayAdd($sdks,"6.....7.3.4.8.................5.4.8.7..2.....1.3.......2.....5.....7.9......1....")
	_ArrayAdd($sdks,"....6...4..6.3....1..4..5.77.....8.5...8.....6.8....9...2.9....4....32....97..1..")

	_ArrayAdd($sdks,".32.....58..3.....9.428...1...4...39...6...5.....1.....2...67.8.....4....95....6.")
	_ArrayAdd($sdks,"...5.3.......6.7..5.8....1636..2.......4.1.......3...567....2.8..4.7.......2..5..")
	_ArrayAdd($sdks,".5.3.7.4.1.........3.......5.8.3.61....8..5.9.6..1........4...6...6927....2...9..")
	_ArrayAdd($sdks,"..5..8..18......9.......78....4.....64....9......53..2.6.........138..5....9.714.")
	_ArrayAdd($sdks,"..........72.6.1....51...82.8...13..4.........37.9..1.....238..5.4..9.........79.")
	_ArrayAdd($sdks,"...658.....4......12............96.7...3..5....2.8...3..19..8..3.6.....4....473..")
	_ArrayAdd($sdks,".2.3.......6..8.9.83.5........2...8.7.9..5........6..4.......1...1...4.22..7..8.9")
	_ArrayAdd($sdks,".5..9....1.....6.....3.8.....8.4...9514.......3....2..........4.8...6..77..15..6.")
	_ArrayAdd($sdks,".....2.......7...17..3...9.8..7......2.89.6...13..6....9..5.824.....891..........")
	_ArrayAdd($sdks,"3...8.......7....51..............36...2..4....7...........6.13..452...........8..")

	_ArrayAdd($sdks,".2..3..9....9.7...9..2.8.....48.65..6.7...2....31.29..8..6.5......3.9....3..2..5.")
	_ArrayAdd($sdks,"..........79.5.18.8..........73.68..45.7.8.9...35.27..7.........16.3.42..........")
	_ArrayAdd($sdks,".....3.1..15..9....6.......1....7.....9...2.....5............2.5..6..34.34.2.....")
	_ArrayAdd($sdks,".1.5..2..9....1.....2..8.3.5...3......8...5..6...8.....4.1..7.....7.......3..4.5.")
	_ArrayAdd($sdks,".......8....21....96..8.1..5..8...1..........89...6.....9.7..5.3...54...48.......")
	_ArrayAdd($sdks,"5..4...6...9...8..64..2.........1...2.8...5..7..5.........9..8...3...6...6...3...")
	_ArrayAdd($sdks,"5....7.1..4.......19.......9......45...3.9..64.3..6...73.5..........2.79.1..6....")
	_ArrayAdd($sdks,"..14...7.6..7....8.....19...1......34..28......36.9.........5.4.8...23...74......")
	_ArrayAdd($sdks,"..5.......7...9.2....5..1..8.415.......8.3.......928..9.7..6....3.4...1.2.....6..")
	_ArrayAdd($sdks,"1.....3.8.6.4..............2.3.1...........958.........5.6...7.....8.2...4.......")

EndFunc


Func _LoadExtremeSudokus()
; sources: https://norvig.com/sudoku.html; https://www.extremesudoku.info/

	$sdklevel="extreme"
	_ArrayAdd($sdks,"..4..8..7.2..3..5.8..9..1..1..5..3...9..7..8...8..1..6..2..3..5.4..2..7.6..8..2..")
	_ArrayAdd($sdks,"...57..3.1......2.7...234......8...4..7..4...49....6.5.42...3.....7..9....18.....")
	_ArrayAdd($sdks,"85...24..72......9..4.........1.7..23.5...9...4...........8..7..17..........36.4.")	;
	_ArrayAdd($sdks,"..53.....8......2..7..1.5..4....53...1..7...6..32...8..6.5....9..4....3......97..")
	_ArrayAdd($sdks,"12..4......5.69.1...9...5.........7.7...52.9..3......2.9.6...5.4..9..8.1..3...9.4")
	_ArrayAdd($sdks,"7..1523........92....3.....1....47.8.......6............9...5.6.4.9.7...8....6.1.")
	_ArrayAdd($sdks,"1....7.9..3..2...8..96..5....53..9...1..8...26....4...3......1..4......7..7...3..")
	_ArrayAdd($sdks,"1...34.8....8..5....4.6..21.18......3..1.2..6......81.52..7.9....6..9....9.64...2")	;
	_ArrayAdd($sdks,"...92......68.3...19..7...623..4.1....1...7....8.3..297...8..91...5.72......64...")
	_ArrayAdd($sdks,".6.5.4.3.1...9...8.........9...5...6.4.6.2.7.7...4...5.........4...8...1.5.2.3.4.")

	_ArrayAdd($sdks,"7.....4...2..7..8...3..8.799..5..3...6..2..9...1.97..6...3..9...3..4..6...9..1.35")
	_ArrayAdd($sdks,"....7..2.8.......6.1.2.5...9.54....8.........3....85.1...3.2.8.4.......9.7..6....")
	_ArrayAdd($sdks,".....6....59.....82....8....45........3........6..3.54...325..6..................")	;;
	_ArrayAdd($sdks,"1.......2.9.4...5...6...7...5.9.3.......7.......85..4.7.....6...3...9.8...2.....1")
	_ArrayAdd($sdks,"8..........36......7..9.2...5...7.......457.....1...3...1....68..85...1..9....4..")
	_ArrayAdd($sdks,"6.2.5.........3.4..........43...8....1....2........7..5..27...........81...6.....")
	_ArrayAdd($sdks,"6.2.5.........4.3..........43...8....1....2........7..5..27...........81...6.....")
	_ArrayAdd($sdks,".6.5.1.9.1...9..539....7....4.8...7.......5.8.817.5.3.....5.2............76..8...")
	_ArrayAdd($sdks,".2.3......63.....58.......15....9.3....7........1....8.879..26......6.7...6..7..4")
	_ArrayAdd($sdks,".4..5..67...1...4....2.....1..8..3........2...6...........4..5.3.....8..2........")

EndFunc

;_________________________________________________________________________________
; candidates-related

Func _PunchIn($cll,$row,$col,$valueb0)
; punch candidate values in/out

	Local $blockID=_Eigen_ReadMatrixValue($matBlocks,$row,$col)

	$blockIDadjRow1=$blockneighbours[$blockID][0]
	$blockIDadjRow2=$blockneighbours[$blockID][1]

	$blockIDadjCol1=$blockneighbours[$blockID][2]
	$blockIDadjCol2=$blockneighbours[$blockID][3]

	; create mask for rows
	_Eigen_SetZero($digitRowvec)
	_Eigen_WriteMatrixValue($digitRowvec,($valueb0*3)+Mod($row,3),0,1)
	_FillMaskOfOverlaps($digitRowvec,$matMrowvec,$digitInRows)

	; create mask for cols
	_Eigen_SetZero($digitColvec)
	_Eigen_WriteMatrixValue($digitColvec,($valueb0*3)+Mod($col,3),0,1)
	_FillMaskOfOverlaps($digitColvec,$matMcolvec,$digitInCols)

	; include ONLY clue's own block's permuts that match the clue
	_Eigen_CwiseLogicalOp_InPlace($punchOut[$blockID],"and",$matMrowvec)
	_Eigen_CwiseLogicalOp_InPlace($punchOut[$blockID],"and",$matMcolvec)

	; exclude in the two row-adjacent blocks all permuts with clue value in same ROW
	_Eigen_CwiseLogicalOp_InPlace($punchOut[$blockIDadjRow1],"xcl",$matMrowvec)
	_Eigen_CwiseLogicalOp_InPlace($punchOut[$blockIDadjRow2],"xcl",$matMrowvec)

	; exclude in the two col-adjacent blocks all permuts with clue value in same COL
	_Eigen_CwiseLogicalOp_InPlace($punchOut[$blockIDadjCol1],"xcl",$matMcolvec)
	_Eigen_CwiseLogicalOp_InPlace($punchOut[$blockIDadjCol2],"xcl",$matMcolvec)

	; and for solver
	_Eigen_Copy_A_ToB($allbuddies[$cll],$retainRowVec)
	_Eigen_CwiseLogicalOp_InPlace($candidate[$valueb0],"xcl",$retainRowVec)

	; zero out all values for the target cell in $candidates
	For $dc=0 To 8
		_Eigen_WriteMatrixValue($candidate[$dc],0,$cll,0)
	Next

	If $showGUI Then
		_UpdateBar($blockID,_Eigen_GetSum($punchOut[$blockID]))
		_UpdateBar($blockIDadjRow1,_Eigen_GetSum($punchOut[$blockIDadjRow1]))
		_UpdateBar($blockIDadjRow2,_Eigen_GetSum($punchOut[$blockIDadjRow2]))
		_UpdateBar($blockIDadjCol1,_Eigen_GetSum($punchOut[$blockIDadjCol1]))
		_UpdateBar($blockIDadjCol2,_Eigen_GetSum($punchOut[$blockIDadjCol2]))
	EndIf

EndFunc


Func _FillCandidValues()
; uses $candidates bitmask multiplication of colwise values 1-9 as a filter

	_Eigen_SetLinSpaced($valuecolvec,1,9)	; 0 = empty
	_Eigen_CwiseBinaryOp_ColwiseCol($candidates,0,"*",$valuecolvec,$candidValues)

EndFunc


Func _UpdateCandidatesPermuts()
; store the remaining candidates per block from $punchout[block-specific] to $matPunchout (all blocks)

	For $dc=0 To 8
		_Eigen_Copy_Arow_ToBrow($punchOut[$dc],$matPunchOut,0,$dc)
		_Eigen_Copy_Arow_ToBrow($candidate[$dc],$candidates,0,$dc)
	Next

	; sum remaining active permuts per block
	_Eigen_MatrixSpecs_Rowwise_Single($matpunchOut,2,$scratchcolvec)
	If _Eigen_ConditAny($scratchcolvec,"==",0) Then _FatalError("invalid clues = no solution")

EndFunc


Func _PunchOutInvalids($bl,$list,$offset,$CCparsed=False)
; expects rowvec/colvec bitmask with invalid=1, valid=0

	Local $index,$permutID,$invalids=_Eigen_FindAll($list,1)
	_Eigen_ReleaseMatrix($list)

	For $rc=0 To _Eigen_GetLastMatrixRow($invalids)	; returns -1 if matrix does not exist
		$index=_Eigen_ReadMatrixValue($invalids,$rc,0)
		$permutID=_Eigen_ReadMatrixValue($offset,$index,0)
		_Eigen_WriteMatrixValue($punchout[$bl],0,$permutID,0)	; punch-out invalid permutation

		If $index<$maxOrderBF Then
			_Eigen_DeleteMatrixValue($orderBF,$index,$bl)	; no longer relevant, and orderBF has to match list of valid permuts
			If $bl<8 Then _Eigen_InsertMatrixValue($orderBF,$maxOrderBF-1,$bl,0)	; restore last in col, otherwise everything shifts up
		EndIf
	Next
	_Eigen_ReleaseMatrix($invalids)

	Local $remaining=_Eigen_GetSum($punchout[$bl])
	_Eigen_WriteMatrixValue($blockfreqs,$bl,0,$remaining)

	If $remaining=1 Then
		$blockSolved[$bl]=True
		_StoreBlock($bl,_Eigen_GetCol_MaxVal($punchout[$bl]))
		If $showGUI Then _SetColorBlock($bl,$acceptcolor)
	EndIf

	If $showGUI Then _UpdateBar($bl,$remaining)

EndFunc

;_____________________________________________________________________________
; solution-related

Func _CheckAllBlocks()
; 1. Are all block permutation counts reduced to unity?
; if so, construct the full solution and check its validity in terms of
; having 9 unique values for each row and each column
; NB we don't ever need to check the blocks themselves,
;	because their permutations are by definition valid
; returns done-status: T: solved or error; F: continue

	_Eigen_ReDim_ExistingMatrix($blockfreqs,9,1)	; reshape as colvector
	_Eigen_MatrixSpecs_Single_Rowwise($matPunchOut,2,$blockfreqs)	; 2=sum remaining permuts per block in colvector
	If $showGUI Then _UpdateAllBars()	; accesses blockfreqs as colvector

	_Eigen_ReDim_ExistingMatrix($blockfreqs,3,3)	; restore block shape
	If _Eigen_IsOnes($blockfreqs) Then
		If $verbose Then ConsoleWrite(@CRLF)

		If _CheckSolution()=True Then
			_Report()
			Return True
		EndIf

		_Error("Invalid solution")
		Return SetError(3,0,True)	; invalid solution

	Else
		If _Eigen_ConditAny($blockfreqs,"==",0) Then Return SetError(1,0,True)		; invalid guess; configuration produces empty permut subset in at least one other block
		If _Eigen_GetSum($blockfreqs)<$blockfreqsum Then $blockfreqsum=_Eigen_GetSum($blockfreqs)	; for next pass
	EndIf

	Return False	; not done yet
EndFunc


Func _StoreBlock($blockID,$permutID)
; construct a single block in the solution

	If $verbose Then ConsoleWrite("Block " & $blockID & " (" & $blockTag[$blockID] & ") contains permutation: " & $permutID & @CRLF)

	_Eigen_ReDim_ExistingMatrix($solutionBlock,9,1)
	_Eigen_Copy_Acol_ToBcol($blockPermuts,$solutionBlock,$permutID,0)

	_Eigen_ReDim_ExistingMatrix($solutionBlock,3,3)
	_Eigen_Copy_Ablock_ToBblock($solutionBlock,$solution,0,0,3,3,(Mod($blockID,3)*3),(Int($blockID/3)*3))

	If $showGUI Then _ShowBlock($blockID,$permutID)

EndFunc


Func _CheckSolution()
; does each value 1-9 occur exactly once in each row and each column?
; NB no need to check block integrity (valid by definition)

	For $dc=1 To 9
		_Eigen_ConditCount_Rowwise($solution,"==",$dc,$solutionCol)
		If _Eigen_IsOnes($solutionCol)=False Then Return False

		_Eigen_ConditCount_Colwise($solution,"==",$dc,$solutionRow)
		If _Eigen_IsOnes($solutionRow)=False Then Return False
	Next
	$solutionsFound+=1

	Return True
EndFunc

;_____________________________________________________________________________
; Bitmask- and list-related

Func _PrepAllBuffers()

	If $showGUI Then _BlankGridBars()

	; timers
	$tstartcross=False
	$tstartbrute=False

	For $bc=0 To 8
		$blockSolved[$bc]=False

		; enable all permutations in the bitmasks
		_Eigen_SetOnes($punchOut[$bc])
		_Eigen_SetOnes($matpermut[$bc])
		_Eigen_SetOnes($candidate[$bc])

		; clear all valid-neighbour bitmasks
		_Eigen_SetZero($validRowNeighbours[$bc])
		_Eigen_SetZero($validColNeighbours[$bc])

		$cellIDsBlock=Eval("cellIDsBlock" & $bc)
		_Eigen_SetZero_Col($cellIDsBlock,1)
	Next
	_UpdateCandidatesPermuts()

	; clear solution
	$solutionsFound=0
	_Eigen_SetZero($solution)
	_Eigen_SetZero($orderBF)

	; reset block ID refs (_OptimiseMax() may have swapped these)
	_Eigen_SetLinSpaced_Colwise($matBlockIDs,0,8)

	; set all block permutations to maximum (for display)
	_Eigen_SetConstant($blockfreqs,$permuts)
	If $showGUI Then _FillAllBars()

EndFunc


Func _StoreMaskOfCellsWithNvalues($tempRowVec,$N,$mincount=1)
; sum the candidates bitmask colwise (=per cell)
; stores a rowvector bitmask of 81 cells in whatever rowvec var is parsed

		If $mincount<1 Then Return False

		_Eigen_MatrixSpecs_Colwise_Single($candidates,2,$tempRowVec)	; 2=sum
		_Eigen_ConditMask_InPlace($tempRowVec,"==",$N)

		Return (_Eigen_GetSum($tempRowVec)>=$mincount)
EndFunc


Func _CreateListOfCellsWithNvalues($tempRowVec,$N,$minsize=1)
; expects a parsed 1x81 rowvector to be filled
; stored values are size-ordered; offset/row/col/value are base-0

	If $N<1 Or $N>9 Then Return SetError(1,0,False)

	; create a colmask for $candidates for cells with exactly N values
	If _StoreMaskOfCellsWithNvalues($tempRowVec,$N,$minsize)=False Then Return SetError(1,0,False)

	; list cells with N values
	_Eigen_Redim_ExistingMatrix($tempRowVec,9,9)
	Local $list=_Eigen_FindAll($tempRowVec,1)
	Local $listsize=@extended
	_Eigen_Redim_ExistingMatrix($tempRowVec,1,81)	; needed again

	; fill candidates mirror with the actual values
	_FillCandidValues(); NB stored values are base-1 (0 = empty = invalid candidate value for cell)

	; remove all cols (cells) with fewer/more than N values
	Local $tempcandid=_Eigen_CreateMatrix_FromAcols_Mask ($candidValues,$tempRowVec)

	; store all non-zero values per listed cell
	Local $tempvalues=_Eigen_CreateVector_FromAcell_Mask($tempcandid,$tempcandid)
	_Eigen_CwiseScalarOp_InPlace($tempvalues,"-",1)	; make values base-0
	_Eigen_Redim_ExistingMatrix($tempvalues,$N,$listsize)

	; append values to list
	_Eigen_AppendCols_InPlace ($list,$N)	; example of an alias-wrapper call
	For $rc=0 To $N-1
		_Eigen_Copy_Arow_ToBcol($tempvalues,$list,$rc,$rc+3)
	Next

	_Eigen_ReleaseMatrix($tempcandid)
	_Eigen_ReleaseMatrix($tempvalues)

	Return SetError(0,$listsize,$list)
EndFunc


Func _FillMaskOfOverlaps($inputmask,$outputmask,$digitInRowsCols)
; expects a parsed digitprofile inputmask (a colvector of size 27)
; produces a permuts rowvec mask of overlaps

	_Eigen_CwiseBinaryOp_Colwise($digitInRowsCols,"+",$inputmask,$matDigitBuffer)
	_Eigen_ConditAny_Colwise($matDigitBuffer,">",1,$outputmask)	; permut mask (1) = overlap

EndFunc


Func _FillMasksOfValidNeighbours($permutID,$blockID)
; create row/col mask of those permutations that do not produce a row/col duplication clash
; 1. add the content of our target column (= permutation) to ALL cols
; 2. flag as valid only those cols with zero overlap in rows/cols (2=overlap; 0/1=no overlap)
; 3. store it in the target block's bitmask array for valid (row c.q. col) neighours

	_Eigen_CwiseBinaryOp_ColwiseCol($digitInRows,$permutID,"+",$digitInRows,$matDigitBuffer)
	_Eigen_ConditAll_Colwise($matDigitBuffer,"<",2,$validRowNeighbours[$blockID])	; mask (1) = valid

	_Eigen_CwiseBinaryOp_ColwiseCol($digitInCols,$permutID,"+",$digitInCols,$matDigitBuffer)
	_Eigen_ConditAll_Colwise($matDigitBuffer,"<",2,$validColNeighbours[$blockID])	; mask (1) = valid

EndFunc


Func _MaskValidPermutations($blockID, $validmask1=0, $validmask2=0, $validmask3=0, $validmask4=0)
; reduce valid permuts by applying 1-4 "and" bitmask filters (negative exclusion: 0 = out; 1 = keep)

	_Eigen_Copy_A_ToB($punchout[$blockID],$matpermut[$blockID])	; keep original in $punchout array

	If $validmask1>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask1)
	If $validmask2>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask2)
	If $validmask3>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask3)
	If $validmask4>0 Then _Eigen_CwiseLogicalOp_InPlace ($matpermut[$blockID], "and", $validmask4)

EndFunc


Func _ProcessAllClues($sdc)

	; create list of all clue locations
	$matClueMask=_Eigen_ConditMask($mSudoku,"!=",0)
	$matClueList=_Eigen_FindAll($matClueMask,1)	; get offset,row,col
	$totalclues=@extended

	Local $newtitle="Solving "& $sdklevel & " sudoku #" & $sdkSelected & "; run " & $sdc & " of " & $sdks[0] & " (" & $totalclues & " clues)"
	If $showGUI Then WinSetTitle($GUI,"",$wintitle & ": " & $newtitle & $wintitleSufffix)

	If $verbose Then
		ConsoleWrite(@CRLF & $newtitle & @CRLF)
		If $totalclues<17 And $exitUpon1stHit=True Then ConsoleWrite("WARNING: at least 17 clues are required for a unique solution" & @CRLF)
		If $totalclues=17 Then ConsoleWrite("WARNING: 17 clues is barely enough for a unique solution;"&@CRLF&"this may take a long time." & @CRLF)
	EndIf

	; add the actual values of the clues to our list
	$matClueValues=_Eigen_CreateVector_FromAcell_Mask($mSudoku,$matClueMask,False)
	$cluesList=_Eigen_CreateMatrix_FromABcols($matClueList,$matClueValues)
	_Eigen_ReleaseMatrix($matClueList)
	_Eigen_ReleaseMatrix($matClueValues)

	; punch in/out each clue in the permutation mask of each relevant block
	For $kc=0 To $totalclues-1
		$offset=_Eigen_ReadMatrixValue($cluesList,$kc,0)
		$rw=_Eigen_ReadMatrixValue($cluesList,$kc,1)	; get clue location and value
		$cl=_Eigen_ReadMatrixValue($cluesList,$kc,2)
		$cll=$rw+($cl*9)
		$valueBase1=_Eigen_ReadMatrixValue($cluesList,$kc,3)

		_PunchIn($cll,$rw,$cl,$valuebase1-1)	; stored base-1, so convert to base-0
		_Eigen_WriteMatrixValue($solution,$rw,$cl,$valueBase1)

		If $showGUI Then
			$cellIDsBlock=Eval("cellIDsBlock" & _Eigen_ReadMatrixValue($matBlocks,$rw,$cl))	; blockID
			_Eigen_WriteMatrixValue($cellIDsBlock,_Eigen_Find_Col($cellIDsBlock,0,$offset),1,$valueBase1)	; use rowindex to store clue values

			GUICtrlSetData($digits[$offset],$valuebase1)
			GUICtrlSetColor($digits[$offset],$cluesColor)
		EndIf
	Next


	; check that all clues are consistent with each other
	For $dc=1 To 9
		_Eigen_ConditCount_Rowwise($solution,"==",$dc,$solutionCol)
		_Eigen_ConditCount_Colwise($solution,"==",$dc,$solutionRow)

		If _Eigen_ConditAny($solutionCol,">",1) Or _Eigen_ConditAny($solutionRow,">",1) Then
			_Error("invalid clues = no solution")
			Return SetError(1,0,False)
		EndIf

		For $cc=0 To 2
			$startcol=$cc*3

			For $rc=0 To 2
				$startrow=$rc*3

				If _Eigen_ConditCount_Block($solution,$startrow,$startcol,3,3,"==",$dc)>1 Then
					_Error("invalid clues = no solution")
					Return SetError(2,0,False)
				EndIf
			Next
		Next
	Next

	Return True
EndFunc

;_____________________________________________________________________________
; BRUTE FORCE TRIAL AND ERROR
; guesses the contents of the six off-diagonal blocks in optimised order,
; thereby reducing permuts to unity or zero in the three diagonal blocks,
; without having to evaluate these at all.

Func _OptimiseMax()
; Block IDs (row=offset) are LinSpaced Colwise in a 3x3 grid, from 0 to 8.
; Identify the two blocks with the largest number of active permutations left;
; swap these onto the block diagonal, and identify the other block positions

	_Eigen_Redim_ExistingMatrix($blockfreqs,3,3)	; reshape as block
	If $verbose Then _Eigen_ShowMatrix($blockfreqs,"Remaining block permutations")

	Local $blockMF=_Eigen_GetOffset_MaxVal($blockfreqs) ; find block ID with highest frequency
	If $verbose Then ConsoleWrite(@CRLF & "Swapped Block ID " & $blockMF & " (in major square, now at TL) has the highest freq of " & _Eigen_GetMaxVal($blockfreqs) & @CRLF)

	Local $maxValBlockrow=_Eigen_GetRow_MaxVal($blockfreqs)
	Local $maxValBlockcol=_Eigen_GetCol_MaxVal($blockfreqs)

	If $blockMF>0 Then	; update the grids
		If $maxValBlockrow>0 Then
			_Eigen_Swap_Arow_Arow($matBlockIDs,$maxValBlockrow,0)
			_Eigen_Swap_Arow_Arow($blockfreqs,$maxValBlockrow,0)
		EndIf

		If $maxValBlockcol>0 Then
			_Eigen_Swap_Acol_Acol($matBlockIDs,$maxValBlockcol,0)
			_Eigen_Swap_Acol_Acol($blockfreqs,$maxValBlockcol,0)
		EndIf
	EndIf

	; punchout MF target all directly neighbouring blocks
	_Eigen_SetZero_Row($blockfreqs,0)
	_Eigen_SetZero_Col($blockfreqs,0)


	Local $blockMF2=_Eigen_GetOffset_MaxVal($blockfreqs) ; find block ID with next-highest frequency in minor square
	If $verbose Then ConsoleWrite(@CRLF & "Swapped Block ID " & $blockMF2 & " (in minor square, now at BR) has the next-highest freq of " & _Eigen_GetMaxVal($blockfreqs) & @CRLF)

	Local $maxValBlockrow=_Eigen_GetRow_MaxVal($blockfreqs)
	Local $maxValBlockcol=_Eigen_GetCol_MaxVal($blockfreqs)

	; if current BR block is suboptimal, swap with optimal (most permuts)
	If $maxValBlockrow<2 Then
		_Eigen_Swap_Arow_Arow($matBlockIDs,1,2)		; 2D grid row swap
		_Eigen_Swap_Arow_Arow($blockfreqs,1,2)		; update freq table
	EndIf

	If $maxValBlockcol<2 Then
		_Eigen_Swap_Acol_Acol($matBlockIDs,1,2)	; 2D grid col swap
		_Eigen_Swap_Acol_Acol($blockfreqs,1,2)	; 2D grid col swap
	EndIf

	; diagonal blocks (globals)
	$blockTL=_Eigen_ReadMatrixValue($matBlockIDs,0,0)	; Top Left block
	$blockCC=_Eigen_ReadMatrixValue($matBlockIDs,1,1)	; Centre block
	$blockBR=_Eigen_ReadMatrixValue($matBlockIDs,2,2)	; Bottom Right block

	; upper off-diagonal blocks (globals)
	$blockTC=_Eigen_ReadMatrixValue($matBlockIDs,0,1)	; Top Centre block
	$blockTR=_Eigen_ReadMatrixValue($matBlockIDs,0,2)	; Top Right block
	$blockCR=_Eigen_ReadMatrixValue($matBlockIDs,1,2)	; Centre Right block

	; lower off-diagonal blocks (globals)
	$blockCL=_Eigen_ReadMatrixValue($matBlockIDs,1,0)	; Centre Left block
	$blockBL=_Eigen_ReadMatrixValue($matBlockIDs,2,0)	; Bottom Left block
	$blockBC=_Eigen_ReadMatrixValue($matBlockIDs,2,1)	; Bottom Centre block

	If $verbose Then ConsoleWrite(@CRLF & "Remaining diagonal block ID " & $blockMF2 & " (now at CC) has a freq of " & _Eigen_ReadMatrixValue($blockfreqs,1,1) & @CRLF)

EndFunc


Func _GetBFlist($blockID)

	Local $list=_Eigen_FindAll($punchOut[$blockID],1)
	Local $listsize=_Eigen_GetMatrixRows($list)

	; if Xfiltered, re-use FindAll result's row indices (col 1, all zero) to store/sort the Xfilter sums;
	; these sums indicate the likelihood of individual permuts fitting other blocks' constraints,
	; so reorder the list descending, using that key
	If $XfilterLastPass>=0 And $listsize>1 Then	; otherwise $orderBF is not filled, so sorting is pointless
		_Eigen_Copy_Ablock_ToBblock($orderBF,$list,0,$blockID,_Min($maxOrderBF,$listsize),1,0,1)	; saved for BF
		_Eigen_SortRows_ByCol_InPlace($list,1,False)	; re-order by descending XF sums
	EndIf

	Return SetError(0,$listsize,$list)
EndFunc


Func _BruteForce()

	; create lists of all remaining candidates for all non-maxfreq blocks
	Local $matTClist=_GetBFlist($blockTC)
	Local $totalTC=@extended

	Local $matTRlist=_GetBFlist($blockTR)
	Local $totalTR=@extended

	Local $matCRlist=_GetBFlist($blockCR)
	Local $totalCR=@extended

	Local $matCLlist=_GetBFlist($blockCL)
	Local $totalCL=@extended

	Local $matBLlist=_GetBFlist($blockBL)
	Local $totalBL=@extended

	Local $matBClist=_GetBFlist($blockBC)
	Local $totalBC=@extended


	ConsoleWrite(@CRLF & @CRLF & "Starting Brute-Force..." & @CRLF & @CRLF)
	$tstartbrute=TimerInit()

For $tcc=0 to $totalTC-1
	$TCpermutID=_Eigen_ReadMatrixValue($matTClist,$tcc,0)	; offset = permut ID

	_FillMasksOfValidNeighbours($TCpermutID,$blockTC)

	If $verbose Then
		ConsoleWrite(@CRLF & "Processing TC (block " & $blockTC & ")  permut " & $tcc+1 & " of " & $totalTC & @CRLF)
		ConsoleWrite(Round(100*$tcc/$totalTC,2)& "% done in " & round(TimerDiff($tstartbrute)/60000,1)& " min." & @CRLF)
		If $exitUpon1stHit=False And $solutionsFound>0 Then ConsoleWrite("Solutions found so far: " & $solutionsFound & @CRLF)
		ConsoleWrite(@CRLF)
	EndIf

	If $showGUI Then
		_ShowBlock($blockTC,$TCpermutID)
		_ClearBlock($blockTR)
	EndIf


For $trc=0 to $totalTR-1
	$TRpermutID=_Eigen_ReadMatrixValue($matTRlist,$trc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validRowNeighbours[$blockTC],0,$TRpermutID)=0 Then ContinueLoop

	If $showGUI Then
		_ShowBlock($blockTR,$TRpermutID)
		_ClearBlock($blockCL)
	EndIf

	_FillMasksOfValidNeighbours($TRpermutID,$blockTR)

	_MaskValidPermutations($blockTL,$validRowNeighbours[$blockTC],$validRowNeighbours[$blockTR])
	If _Eigen_IsZero($matpermut[$blockTL]) Then ContinueLoop

	If $verbose Then
		ConsoleWrite(@CRLF & "Processing TC (block " & $blockTC & ")  permut " & $tcc+1 & " of " & $totalTC & @CRLF)
		ConsoleWrite("Processing TR (block " & $blockTR & ")  permut " & $trc+1 & " of " & $totalTR & @CRLF)
		If $exitUpon1stHit=False And $solutionsFound>0 Then ConsoleWrite("Solutions found so far: " & $solutionsFound & @CRLF)
		ConsoleWrite(Round(((100*($tcc+($trc/$totalTR)))/$totalTC),2) & "% done in " & round(TimerDiff($tstartbrute)/60000,1)& " min." & @CRLF)
		ConsoleWrite(@CRLF)
	EndIf


For $clc=0 to $totalCL-1
	$CLpermutID=_Eigen_ReadMatrixValue($matCLlist,$clc,0)	; offset = permut ID

	If $showGUI Then
		_ShowBlock($blockCL,$CLpermutID)
		_ClearBlock($blockBL)
	EndIf

	_FillMasksOfValidNeighbours($CLpermutID,$blockCL)

	_MaskValidPermutations($blockTL,$validRowNeighbours[$blockTC],$validRowNeighbours[$blockTR],$validColNeighbours[$blockCL])
	If _Eigen_IsZero($matpermut[$blockTL]) Then ContinueLoop

	_MaskValidPermutations($blockCC,$validColNeighbours[$blockTC],$validRowNeighbours[$blockCL])
	If _Eigen_IsZero($matpermut[$blockCC]) Then ContinueLoop

	_MaskValidPermutations($blockCR,$validColNeighbours[$blockTR],$validRowNeighbours[$blockCL])
	If _Eigen_IsZero($matpermut[$blockCR]) Then ContinueLoop


For $blc=0 to $totalBL-1
	$BLpermutID=_Eigen_ReadMatrixValue($matBLlist,$blc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validColNeighbours[$blockCL],0,$BLpermutID)=0 Then ContinueLoop

	If $showGUI Then
		_ShowBlock($blockBL,$BLpermutID)
		_ClearBlock($blockCR)
	EndIf

	_FillMasksOfValidNeighbours($BLpermutID,$blockBL)

	_MaskValidPermutations($blockTL, $validRowNeighbours[$blockTC],$validRowNeighbours[$blockTR],$validColNeighbours[$blockCL],$validColNeighbours[$blockBL])
	If _Eigen_GetSum($matpermut[$blockTL])<>1 Then ContinueLoop

	_MaskValidPermutations($blockBC,$validRowNeighbours[$blockBL],$validColNeighbours[$blockTC])
	If _Eigen_IsZero($matpermut[$blockBC]) Then ContinueLoop

	_MaskValidPermutations($blockBR,$validRowNeighbours[$blockBL],$validColNeighbours[$blockTR])
	If _Eigen_IsZero($matpermut[$blockBR]) Then ContinueLoop


For $crc=0 to $totalCR-1
	$CRpermutID=_Eigen_ReadMatrixValue($matCRlist,$crc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validRowNeighbours[$blockCL],0,$CRpermutID)=0 Then ContinueLoop
	If _Eigen_ReadMatrixValue($validColNeighbours[$blockTR],0,$CRpermutID)=0 Then ContinueLoop

	If $showGUI Then
		_ShowBlock($blockCR,$CRpermutID)
		_ClearBlock($blockBC)
	EndIf

	_FillMasksOfValidNeighbours($CRpermutID,$blockCR)

	_MaskValidPermutations($blockCC,$validColNeighbours[$blockTC],$validRowNeighbours[$blockCL],$validRowNeighbours[$blockCR])
	If _Eigen_IsZero($matpermut[$blockCC]) Then ContinueLoop


For $bcc=0 to $totalBC-1
	$BCpermutID=_Eigen_ReadMatrixValue($matBClist,$bcc,0)	; offset = permut ID
	If _Eigen_ReadMatrixValue($validRowNeighbours[$blockBL],0,$BCpermutID)=0 Then ContinueLoop
	If _Eigen_ReadMatrixValue($validColNeighbours[$blockTC],0,$BCpermutID)=0 Then ContinueLoop

	_FillMasksOfValidNeighbours($BCpermutID,$blockBC)

	_MaskValidPermutations($blockBR,$validRowNeighbours[$blockBC],$validRowNeighbours[$blockBL],$validColNeighbours[$blockTR],$validColNeighbours[$blockCR])
	If _Eigen_GetSum($matpermut[$blockBR])<>1 Then ContinueLoop

	_MaskValidPermutations($blockCC,$validColNeighbours[$blockBC],$validColNeighbours[$blockTC],$validRowNeighbours[$blockCL],$validRowNeighbours[$blockCR])
	If _Eigen_GetSum($matpermut[$blockCC])<>1 Then ContinueLoop


	; update solution matrix
	If $verbose Then ConsoleWrite(@CRLF & "Checking this solution:" & @CRLF)

	_StoreBlock($blockTL,_Eigen_GetCol_MaxVal($matpermut[$blockTL]))
	_StoreBlock($blockTC,$TCpermutID)
	_StoreBlock($blockTR,$TRpermutID)

	_StoreBlock($blockCL,$CLpermutID)
	_StoreBlock($blockCC,_Eigen_GetCol_MaxVal($matpermut[$blockCC]))
	_StoreBlock($blockCR,$CRpermutID)

	_StoreBlock($blockBL,$BLpermutID)
	_StoreBlock($blockBC,$BCpermutID)
	_StoreBlock($blockBR,_Eigen_GetCol_MaxVal($matpermut[$blockBR]))

	If _CheckSolution()=True And $exitUpon1stHit=True Then
		If $showGUI Then
			For $bc=0 To 8
				_SetColorBlock($bc,$acceptcolor)
			Next
		EndIf
		Return
	EndIf

	If $verbose Then ConsoleWrite("Resuming scan..." & @CRLF)

Next
Next
Next

Next
Next
Next

EndFunc

;_____________________________________________________________________________
; CROSS-FILTERING SOLVER
; Each block has two row neighbours and two col neighbours.
; Given a centred target block (XCC), these five form a cross (centre + 2 blockrow- and 2 blockcol neighbours).
; Create four bitmasks for the valid neighbour permuts for XCC only
; and evaluate their co-dependence, thereby invalidating permuts in
; both the target block and (possibly) one or more of its neighbours.
; Additionally, accumulate stats per XCC permut to re-order BF processing (most likely cases first)

Func _CrossFilter()

	; reshape as colvector for easy access
	_Eigen_ReDim_ExistingMatrix($blockfreqs,9,1)

	; if the clues already identify any unique blockID, update the solution
	For $bc=0 To 8
		If _Eigen_ReadMatrixValue($blockfreqs,$bc,0)=1 Then
			$blockSolved[$bc]=True
			_StoreBlock($bc,_Eigen_GetCol_MaxVal($punchout[$bc]))
			If $showGUI Then _SetColorBlock($bc,$acceptcolor)
		EndIf
	Next

	; create buffers for sorting blockfreqs (col 0 = blockIDs)
	Local $unsortedfreqs=_Eigen_CreateMatrix_LinSpaced_Colwise(9,2,0,17)
	Local $sortedfreqs=_Eigen_CloneMatrix($unsortedfreqs,False)	; F: no data copied

	; keep looping over all blocks until no more improvement, or up to N passes
	Local $prev,$blID,$lastbc,$oldfreq,$crosspass=0
	$tstartcross=TimerInit()	; global
	Do
		$prev=_Eigen_GetSum($blockfreqs)
		$crosspass+=1
		ConsoleWrite(@CRLF & "Applying cross-neighbour filters; pass " & $crosspass & (($XfilterLastPass>1)?(" of " & $XfilterLastPass):("")) & @CRLF)

		_Eigen_Copy_Acol_ToBcol($blockfreqs,$unsortedfreqs,0,1)
		_Eigen_SortRows_ByCol($unsortedfreqs,1,False,$sortedfreqs)	; F: descending
		$lastbc=((_Eigen_GetMinVal($blockfreqs)>1)?(8):( _Eigen_Find_Col($sortedfreqs,1,1)))

		; evaluate from largest to smallest blockfreqs (faster than sequential),
		; so the largest freq is reduced before it shapes the permutation space of its neighbours
		For $bc=0 To $lastbc	; impose cross filters on each block in turn
			_Eigen_ReDim_ExistingMatrix($blockfreqs,9,1)
			$blID=_Eigen_ReadMatrixValue($sortedfreqs,$bc,0)
			$oldfreq=_Eigen_ReadMatrixValue($blockfreqs,$blID,0)	; updated after sorting

			If $oldfreq>1 Then
				_CrossSolver($blID,$oldfreq)	; the main X-Filter call
				If @error Then
					_Eigen_ReleaseMatrix($unsortedfreqs)
					_Eigen_ReleaseMatrix($sortedfreqs)
					Return SetError(1,0,False)
				EndIf
			EndIf
		Next	; keep going until max # passes reached OR no more improvement
	Until $crosspass=$XfilterLastPass Or $prev=_Eigen_GetSum($blockfreqs)

	_Eigen_ReleaseMatrix($unsortedfreqs)
	_Eigen_ReleaseMatrix($sortedfreqs)

	Return True
EndFunc


Func _CrossSolver($blXCC,$blfr)
; expects central block ID (0-8) of a virtual cross-shaped set of blocks
; updates punchout[$blCC] after applying stack-tiled 216-filters
; Note that all references to CC/CL/CR/TC/BC here refer to a virtual cross of blocks (with CC=midpoint)
; and NOT to the physical block locations

	; identify target block's near and far neighbours
	Local $blXCL=$blockneighbours[$blXCC][0]
	Local $blXCR=$blockneighbours[$blXCC][1]
	Local $blXTC=$blockneighbours[$blXCC][2]
	Local $blXBC=$blockneighbours[$blXCC][3]

	If $showGUI Then	; activate text markers
		_HighLightBar($blXCC,$COLOR_RED)
		_HighLightBar($blXCL)
		_HighLightBar($blXCR)
		_HighLightBar($blXTC)
		_HighLightBar($blXBC)
	EndIf

	; create remaining valid permuts bitmask from CC
	Local $oldXCClist=_Eigen_FindAll($punchOut[$blXCC],1)	; needed later too
	Local $width=@extended
	Global $newPermuts=_Eigen_CreateMatrix(1,$width)	; GLOBAL using intermediate rowvector to reduce memory footprint; used in _CreateNeighbourFilter()

	; create a temporary virtual list of CC offsets (permutIDs) only
	Local $offsetsXCC=_Eigen_ReMap_MatrixCols($oldXCClist,0,1)
	Global $maskpermuts=_Eigen_CreateMask_FromOffsets($offsetsXCC,1,$permuts)	; GLOBAL used in _CreateNeighbourFilter()

	; create bitmasks of all remaining CC candidates for each of CC's block neighbours
	Local $permutsXCL=_CreateNeighbourFilter($blXCL,$width,$digitInRows)
	Local $offsetsXCL=@extended

	Local $permutsXCR=_CreateNeighbourFilter($blXCR,$width,$digitInRows)
	Local $offsetsXCR=@extended

	Local $permutsXTC=_CreateNeighbourFilter($blXTC,$width,$digitInCols)
	Local $offsetsXTC=@extended

	Local $permutsXBC=_CreateNeighbourFilter($blXBC,$width,$digitInCols)
	Local $offsetsXBC=@extended

	_Eigen_ReleaseMatrix($newpermuts)
	_Eigen_ReleaseMatrix($maskpermuts)


	; apply tiled filters from CC's row neighbours, col neighbours, and jointly
	; this is the crux of this technique
	Local $permutsXTCBC=_StackTiles($permutsXTC,$permutsXBC,$width,$blXTC,$blXBC)
	Local $invalidXTC=@error
	Local $invalidXBC=@extended
	_Eigen_ReleaseMatrix($permutsXTC)
	_Eigen_ReleaseMatrix($permutsXBC)

	Local $permutsXCLCR=_StackTiles($permutsXCL,$permutsXCR,$width,$blXCL,$blXCR)
	Local $invalidXCL=@error
	Local $invalidXCR=@extended
	_Eigen_ReleaseMatrix($permutsXCL)
	_Eigen_ReleaseMatrix($permutsXCR)

	Local $permutsXCC=_StackTiles($permutsXTCBC,$permutsXCLCR,$width)
	_Eigen_ReleaseMatrix($permutsXTCBC)
	_Eigen_ReleaseMatrix($permutsXCLCR)


	; collect the summed occurrences; also needed for BF sort order (sum = likelihood)
	Local $sums=_Eigen_GetSum_Colwise($permutsXCC)	; row vector
	_Eigen_ReleaseMatrix($permutsXCC)

	Local $maskedSums=_Eigen_ConditMask($sums,">",0)
	Local $remaining=_Eigen_GetSum($maskedSums)

	If $remaining=0 Then	; something went terribly wrong
		_Eigen_ReleaseMatrix($oldXCClist)
		_Eigen_ReleaseMatrix($sums)
		_Eigen_ReleaseMatrix($maskedsums)

		_Eigen_ReleaseMatrix($offsetsXCC)
		_Eigen_ReleaseMatrix($offsetsXCL)
		_Eigen_ReleaseMatrix($offsetsXCR)
		_Eigen_ReleaseMatrix($offsetsXTC)
		_Eigen_ReleaseMatrix($offsetsXBC)

		If $invalidXTC>0 Then _Eigen_ReleaseMatrix($invalidXTC)
		If $invalidXBC>0 Then _Eigen_ReleaseMatrix($invalidXBC)	; fails on easy 6
		If $invalidXCL>0 Then _Eigen_ReleaseMatrix($invalidXCL)
		If $invalidXCR>0 Then _Eigen_ReleaseMatrix($invalidXCR)	; fails on easy 6

		_Error("Invalid Sudoku?")
		Return SetError(1,0,$blfr)
	EndIf

	Local $nonzeroSums=_Eigen_CreateVector_FromAcell_Mask($sums,$maskedSums) ;produces colvector
	_Eigen_Copy_Ablock_ToBblock($nonzeroSums,$orderBF,0,0,_Min($maxOrderBF,$remaining),1,0,$blXCC)	; save for BF
	_Eigen_ReleaseMatrix($nonzeroSums)

	; flip valid CC bitmask to mark invalid permuts
	_Eigen_CwiseLogicalOp_InPlace($maskedSums,"not")
	If $verbose Then ConsoleWrite("Block ID " & $blXCC & " (" & $blockTag[$blXCC] & ") yields " & $remaining & " out of " & _Eigen_GetSize($sums) & " valid permutations" & @CRLF)

	If $remaining<$blfr Then	; are permuts reduced?
		_PunchOutInvalids($blXCC,$maskedSums,$offsetsXCC, True)	; releases list of invalids too	; T: centre of cross
	Else
		_Eigen_ReleaseMatrix($maskedSums)
	EndIf

	_Eigen_ReleaseMatrix($offsetsXCC)
	_Eigen_ReleaseMatrix($sums)
	_Eigen_ReleaseMatrix($oldXCClist)

	; invalidate redundant permuts in neighbouring blocks as well
	If $invalidXTC>0 Then _PunchOutInvalids($blXTC,$invalidXTC,$offsetsXTC)
	If $invalidXBC>0 Then _PunchOutInvalids($blXBC,$invalidXBC,$offsetsXBC)
	If $invalidXCL>0 Then _PunchOutInvalids($blXCL,$invalidXCL,$offsetsXCL)
	If $invalidXCR>0 Then _PunchOutInvalids($blXCR,$invalidXCR,$offsetsXCR)

	_Eigen_ReleaseMatrix($offsetsXCL)
	_Eigen_ReleaseMatrix($offsetsXCR)
	_Eigen_ReleaseMatrix($offsetsXTC)
	_Eigen_ReleaseMatrix($offsetsXBC)

	If $showGUI Then	; deactivate text markers
		_HighLightBar($blXCC,Default,False)
		_HighLightBar($blXCL,Default,False)
		_HighLightBar($blXCR,Default,False)
		_HighLightBar($blXTC,Default,False)
		_HighLightBar($blXBC,Default,False)
	EndIf

EndFunc


Func _CreateNeighbourFilter($bl, $newcols, $digitInRowsCols)
; for the parsed block, store valid CC neighbours only

	; collect remaining valid permuts for the current neighoubring block
	Local $list=_Eigen_FindAll($punchOut[$bl],1)
	Local $totalrows=@extended
	Local $offsets=_Eigen_CreateMatrix_FromAcols($list,0,1)
	_Eigen_ReleaseMatrix($list)

	; Note that the reference var ($permutsTile) is local,
	; but the created E4A matrix itself is by definition global
	Local $permutsTile=_Eigen_CreateMatrix($totalrows,$newcols)

	; create a new bitmask containing only permuts valid for CC
	; this becomes our duplication "tile" for this block
	For $rc=0 to $totalrows-1
		$permutID=_Eigen_ReadMatrixValue($offsets,$rc,0)	; offset = permut ID
		_Eigen_CwiseBinaryOp_ColwiseCol($digitInRowsCols,$permutID,"+",$digitInRowsCols,$matDigitBuffer)
		_Eigen_ConditAll_Colwise($matDigitBuffer,"<",2,$matMrowvec)

		_Eigen_CreateMatrix_FromAcols_Mask($matMrowvec,$maskpermuts,False,$newpermuts)	; F: don't invert mask
		_Eigen_Copy_Arow_ToBrow($newPermuts,$permutsTile,0,$rc)
	Next

	Return SetError(0,$offsets,$permutsTile)
EndFunc


Func _StackTiles($matA, $matB, $newcols, $bl1=0, $bl2=0)
; expects A and B to have same number of cols (but can have different number of rows)
; each B row is duplicated Arow times in an AND buffer with which to filter matA
; completely zero results indicate invalid A and/or B permuts (stored separately)

	; create bitmask buffers
	Local $rowdupes=_Eigen_CloneMatrix($matA,False)	; tile buffer
	Local $dupes=_Eigen_GetMatrixRows($rowdupes)

	Local $stacked=_Eigen_CloneMatrix($matB,False)	; we don't accumulate in a vector because the summed results inform permuts' relative likelihoods (used in BF)
	Local $rowvec=_Eigen_CreateRowVector($newcols)

	Local $invalidA=0,$invalidB=0,$doInvalids=($bl1<>$bl2)

	If $doInvalids Then
		$invalidA=_Eigen_CreateColVector_Ones(_Eigen_GetMatrixRows($matA))
		$invalidB=_Eigen_CreateColVector_Zero(_Eigen_GetMatrixRows($matB))
		$invalidA4Brow=_Eigen_CloneMatrix($invalidA,False)	; copy shape only
	EndIf

	; duplicate each matB row as an A-sized tile for filtering matA
	For $rc=0 To _Eigen_GetLastMatrixRow($matB)
		_Eigen_CreateMatrix_FromArow_Tiled($matB,$rc,$dupes,$rowdupes)
		_Eigen_CwiseLogicalOp_InPlace($rowdupes,"and",$matA)
		_Eigen_GetSum_Colwise($rowdupes,$rowvec)
		_Eigen_Copy_Arow_ToBrow($rowvec,$stacked,0,$rc)

		If $doInvalids Then	; check for/store all-zero result (invalid input permuts)
			_Eigen_IsZero_Rowwise($rowdupes,$invalidA4Brow)
			_Eigen_CwiseLogicalOp_InPlace($invalidA,"and",$invalidA4Brow)
			_Eigen_WriteMatrixValue($invalidB,$rc,0,_Eigen_IsZero($rowvec))
		EndIf
	Next

	If $doInvalids Then _Eigen_ConditMask_InPlace($stacked,">",0)	; turn sums into bitmask
	_Eigen_ReleaseMatrix($rowdupes)
	_Eigen_ReleaseMatrix($rowvec)

	If $doInvalids Then
		_Eigen_ReleaseMatrix($invalidA4Brow)
		If _Eigen_IsZero($invalidA) Then
			_Eigen_ReleaseMatrix($invalidA)
			$invalidA=0
		EndIf

		If _Eigen_IsZero($invalidB) Then
			_Eigen_ReleaseMatrix($invalidB)
			$invalidB=0
		EndIf
	EndIf

	; return up to 3 matrixIDs
	Return SetError($invalidA,$invalidB,$stacked)
EndFunc
