#Region Header

#CS
    Title:          Syntax Highlighter for AutoIt v3 Code
    Filename:       Au3SyntaxHighlight.au3
    Description:    Allows to highlight AutoIt v3 syntax to html/bbcode format.
    Author:         G.Sandler (a.k.a (Mr)CreatoR), www.creator-lab.ucoz.ru, www.autoit-script.ru.
    Version:        0.1
    Requirements:   AutoIt v3.3 +, Developed/Tested on WindowsXP Pro Service Pack 3
    Uses:           
    Notes:          *Idea* based on similar old project: http://www.autoitscript.com/forum/index.php?showtopic=34236

#CS Example script
#include <GUIConstantsEx.au3>
#include <WindowsConstants.au3>
#include <IE.au3>

$sFile = FileOpenDialog("Open Au3 Script File", "", "AutoIt Script (*.au3)")
If @error Then Exit

$sAu3Code = FileRead($sFile)
$sAu3Syntax_HighlightedCode = _Au3_SyntaxHighlight_Proc($sAu3Code, 0)

$sTmp_File = @TempDir & "\Au3_HighlightSyntax.htm"

$hFile = FileOpen($sTmp_File, 2)
FileWrite($hFile, $sAu3Syntax_HighlightedCode)
FileClose($hFile)

_IEErrorHandlerRegister()

$oIE = _IECreateEmbedded()
GUICreate("AutoIt Syntax Highlighter", @DesktopWidth-10, @DesktopHeight-70, 0, 0, $WS_OVERLAPPEDWINDOW + $WS_VISIBLE + $WS_CLIPSIBLINGS + $WS_CLIPCHILDREN)

$GUIActiveX = GUICtrlCreateObj($oIE, 10, 20, @DesktopWidth-5-20, @DesktopHeight-200-40)

$GUI_Button_Back = GUICtrlCreateButton("Back", 10, @DesktopHeight-200, 100, 30)
$GUI_Button_Forward = GUICtrlCreateButton("Forward", 120, @DesktopHeight-200, 100, 30)
$GUI_Button_Home = GUICtrlCreateButton("Home", 230, @DesktopHeight-200, 100, 30)
$GUI_Button_Stop = GUICtrlCreateButton("Stop", 340, @DesktopHeight-200, 100, 30)

GUISetState()

_IENavigate($oIE, $sTmp_File)

While 1
    Switch GUIGetMsg()
		Case $GUI_EVENT_CLOSE
			_IEQuit($oIE)
			GUIDelete()
            Exit
		Case $GUI_Button_Home
            _IENavigate($oIE, $sTmp_File)
        Case $GUI_Button_Back
            _IEAction($oIE, "back")
        Case $GUI_Button_Forward
            _IEAction($oIE, "forward")
        Case $GUI_Button_Stop
            _IEAction($oIE, "stop")
    EndSwitch
WEnd
#CE Example script

#CE

#include-once
Opt("MustDeclareVars", 1)

#EndRegion Header

#Region Global Variables
Global $sAu3_Keywords_File 		= @ScriptDir & "\Resources\au3.keywords.properties"
Global $sAu3_Styles_File 		= @ScriptDir & "\Resources\au3.styles.properties"

Global $sAu3SH_Funcs_URL 		=  __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.functions.url", "http://www.autoitscript.com/autoit3/docs/functions/%s.htm", 0)
Global $sAu3SH_UDFs_URL 		= __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.udfs.url", "http://dundats.mvps.org/help/html/libfunctions/%s.htm", 0)
Global $sAu3SH_Keywords_URL 	= __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.keywotds.url", "http://www.autoitscript.com/autoit3/docs/keywords.htm#%s", 0)
Global $sAu3SH_Macros_URL 		= __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.macros.url", "http://www.autoitscript.com/autoit3/docs/macros.htm#%s", 0)
Global $sAu3SH_PreProcessor_URL = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.preprocessor.url", "http://www.autoitscript.com/autoit3/docs/keywords.htm#%s", 0)

Global Enum _
	$iAu3SH_Cmnt_Style = 1, $iAu3SH_CmntBlck_Style, $iAu3SH_Nmbr_Style, $iAu3SH_Fnc_Style, $iAu3SH_Kwd_Style, $iAu3SH_Mcro_Style, $iAu3SH_Str_Style, _
	$iAu3SH_Oprtr_Style, $iAu3SH_Var_Style, $iAu3SH_SntKys_Style, $iAu3SH_PrPrc_Style, $iAu3SH_Spec_Style, $iAu3SH_Abrv_Style, $iAu3SH_COM_Style, $iAu3SH_Udf_Style

Global $aAu3SH_Styles[$iAu3SH_Udf_Style+1]

For $i = $iAu3SH_Cmnt_Style To $iAu3SH_Udf_Style
	$aAu3SH_Styles[$i] 			= '<span class="S' & $i & '">'
Next

Global $sAu3SH_CloseTag 		= '</span>'

Global $iAu3SH_AddURLs 			= 1
#EndRegion Global Variables

#Region Public Functions
; #FUNCTION# ====================================================================================================
; Name...........:	_Au3_SyntaxHighlight_Proc
; Description....:	Allows to highlight AutoIt v3 syntax to html/bbcode format.
; Syntax.........:	_Au3_SyntaxHighlight_Proc($sAu3Code, $iOutput = -1)
; Parameters.....:	$sAu3Code - AutoIt v3 plain code.
;					$iOutput  - [Optional] Sets the output format:
;					                                      -1 - (Default) Return CSS classes header and the AutoIt Syntax Highlighted code as string (wrapped with code tags)
;					                                       1 - Return CSS classes header and AutoIt Syntax Highlighted code as array, where...
;					                                            [0] = CSS styles.
;					                                            [1] = AutoIt Syntax Highlighted code.
;					                                       2 - Return the result as Html formatted string.
;					                                       3 - Return the result as BBCode formatted string (html tags replaced with bbcode tags).
;					
; Return values..:	Success - Returns AutoIt Syntax Highlighted code (see $iOutput parameter above for more details).
;					Failure - Sets @error to 1 if the $sAu3Code is empty string.
;
; Author.........:	G.Sandler (a.k.a (Mr)CreatoR).
; Modified.......:	
; Remarks........:	*Idea* based on similar old project: http://www.autoitscript.com/forum/index.php?showtopic=34236
; Related........:	
; Link...........:	
; Example........:	Yes (see #Region Header).
; ===============================================================================================================
Func _Au3_SyntaxHighlight_Proc($sAu3Code, $iOutput = -1)
	If StringStripWS($sAu3Code, 8) = '' Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $sPattern1, $sPattern2, $sPattern3
	Local $sReplace1, $sReplace2, $sReplace3
	
	Local $sUnique_Str_DSQ = '%@~@%'
	Local $sUnique_Str_IN = '!~@%@~!'
	
	While StringInStr($sAu3Code, $sUnique_Str_DSQ)
		$sUnique_Str_DSQ &= Random(10000, 99999)
	WEnd
	
	While StringInStr($sAu3Code, $sUnique_Str_IN)
		$sUnique_Str_IN &= Random(10000, 99999)
	WEnd
	
	; Get all strings to array (all between "", '' and <> chars), so we can replace later with unique marks
	$sPattern1 = '(?m)(("|'')[^\2]*?\2)' ;'(?s).*?(("|'')[^\2]*\2).*?'
	$sPattern2 = "(?si)#include\s+?(<[^\>]*>).*?"
	
	Local $aDoubleQoute_Strings = StringRegExp($sAu3Code, $sPattern1, 3)
	Local $aInclude_Strings = StringRegExp($sAu3Code, $sPattern2, 3)
	
	; Replace all the strings with unique marks
	$sPattern1 = '(?s)("|'')([^\1])*?(\1)'
	$sPattern2 = "(?si)(#include\s+?)<[^\>]*>(.*?)"
	
	$sReplace1 = $sUnique_Str_DSQ
	$sReplace2 = '\1' & $sUnique_Str_IN & '\2'
	
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
	
	$sPattern1 = '([\(\)\[\]\<\>\.\*\+\-\=\&\^\,\/])'
	$sPattern2 = '(?i)(#include)' & $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '-' & $sAu3SH_CloseTag & '(once)'
	
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\1' & $sAu3SH_CloseTag
	$sReplace2 = '\1-\2'
	
	; Highlight the operators, brakets, commas (must be done before all other parsers)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	; Fix for the #include-once pre-processor
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
	
	$sPattern1 = '(\W+)(_)(\W+)'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\2' & $sAu3SH_CloseTag & '\3'
	
	; Highlight the line braking character, wich is the underscore (must be done before all other parsers)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	$sPattern1 = '(\D(?:\s+)?' & $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\.' & $sAu3SH_CloseTag & '|\D(?:\s+)?\.)(\w+)'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_COM_Style] & '\2' & $sAu3SH_CloseTag
	
	; Highlight the COM Objects (must be done before all other parsers)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	; Highlight the function
	$sAu3Code = __Au3_SyntaxHighlight_ParseFunctions($sAu3Code)
	
	; Highlight the UDFs
	$sAu3Code = __Au3_SyntaxHighlight_ParseUDFs($sAu3Code)
	
	; Highlight the keyword
	$sAu3Code = __Au3_SyntaxHighlight_ParseKeywords($sAu3Code)
	
	; Highlight the macros
	$sAu3Code = __Au3_SyntaxHighlight_ParseMacros($sAu3Code)
	
	; Highlight the PreProcessor
	$sAu3Code = __Au3_SyntaxHighlight_ParsePreProcessor($sAu3Code)
	
	; Highlight special keywords
	$sAu3Code = __Au3_SyntaxHighlight_ParseSpecial($sAu3Code)
	
	$sPattern1 = '([^\w#@])(\d+)([^\w])'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Nmbr_Style] & '\2' & $sAu3SH_CloseTag & '\3'
	
	; Highlight the number
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	$sPattern1 = '([^\w#@])((?i)0x[abcdef\d]+)([^abcdef\d])'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Nmbr_Style] & '\2' & $sAu3SH_CloseTag & '\3'
	
	; Highlight the hexadecimal number
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	$sPattern1 = '\$(\w+)?'
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_Var_Style] & '$\1' & $sAu3SH_CloseTag
	
	; Highlight variables (also can be just the dollar sign)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	; Highlight send keys (must be done before the strings been restored, because send keys should be highlighted inside the strings)
	$aDoubleQoute_Strings = __Au3_SyntaxHighlight_ParseSendKeys($aDoubleQoute_Strings)
	
	; Replace back the unique marks with the original one and wrap them with "string" tags
	For $i = 0 To UBound($aDoubleQoute_Strings)-1 Step 2
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_DSQ, $aAu3SH_Styles[$iAu3SH_Str_Style] & $aDoubleQoute_Strings[$i] & $sAu3SH_CloseTag, 1)
	Next
	
	For $i = 0 To UBound($aInclude_Strings)-1
		$aInclude_Strings[$i] = StringReplace($aInclude_Strings[$i], '<', '&lt;')
		$aInclude_Strings[$i] = StringReplace($aInclude_Strings[$i], '>', '&gt;')
		
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_IN, $aAu3SH_Styles[$iAu3SH_Str_Style] & $aInclude_Strings[$i] & $sAu3SH_CloseTag, 1)
	Next
	
	; Highlight commented lines / comment block (plus extra parsers due to need of the loop, see the function's body)
	$sAu3Code = __Au3_SyntaxHighlight_ParseComments($sAu3Code)
	
	; Replace tabs with 4 spaces, in IE the tabs not looking good :(.
	$sAu3Code = StringReplace($sAu3Code, @TAB, '    ') ;'&nbsp;&nbsp;&nbsp;&nbsp;'
	
	; Get the CSS styles header (to use on return)
	Local $sStyles_Header = __Au3_SyntaxHighlight_GetKeywordStylesHeader()
	
	; Check the $iOutput flag
	Switch $iOutput
		; Return CSS classes header and AutoIt Syntax Highlighted code as array ([0] = CSS styles, [1] = Au3 Code)
		Case 1
			Local $aRet[2] = [$sStyles_Header, $sAu3Code]
			Return $aRet
		; Return AutoIt Syntax Highlighted code as string
		Case 2
			Return $sAu3Code
		; Return the result as BBCode formatted string
		Case 3
			Return __Au3_SyntaxHighlight_ConvertHtmlToBBCodeTags($sAu3Code)
		; Return CSS classes header and the AutoIt Syntax Highlighted code as Html formatted string
		Case Else
			Return $sStyles_Header & @CRLF & '<pre class="codebox"><span>' & $sAu3Code & '</span></pre>'
	EndSwitch
EndFunc
#EndRegion Public Functions

#Region Internal Functions
Func __Au3_SyntaxHighlight_ParseFunctions($sAu3Code)
	Local $aFuncs = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.functions")
	Local $sPattern1, $sReplace1
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aFuncs[0]
			$sPattern1 = '([^\w\$]+|\A)((?i)' & $aFuncs[$i] & ')(\W+|$)'
			$sReplace1 = '\1<a href="' & StringFormat($sAu3SH_Funcs_URL, $aFuncs[$i]) & '">' & _
				$aAu3SH_Styles[$iAu3SH_Fnc_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	Else
		For $i = 1 To $aFuncs[0]
			$sPattern1 = '([^\w\$]+|\A)((?i)' & $aFuncs[$i] & ')(\W+|$)'
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Fnc_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	EndIf
	
	Return $sAu3Code
EndFunc

Func __Au3_SyntaxHighlight_ParseUDFs($sAu3Code)
	Local $aUDFs = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.udfs")
	Local $sPattern1, $sReplace1
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aUDFs[0]
			$sPattern1 = '(\W+|\A)((?i)' & $aUDFs[$i] & ')(\W+|$)'
			$sReplace1 = '\1<a href="' & StringFormat($sAu3SH_UDFs_URL, $aUDFs[$i]) & '">' & $aAu3SH_Styles[$iAu3SH_Udf_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	Else
		For $i = 1 To $aUDFs[0]
			$sPattern1 = '(\W+|\A)((?i)' & $aUDFs[$i] & ')(\W+|$)'
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Udf_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	EndIf
	
	Return $sAu3Code
EndFunc

Func __Au3_SyntaxHighlight_ParseKeywords($sAu3Code)
	Local $aKeywords = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.keywords")
	Local $sPattern1, $sReplace1
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aKeywords[0]
			$sPattern1 = '(\W|\A)((?i)' & $aKeywords[$i] & ')(\W|$)'
			$sReplace1 = '\1<a href="' & StringFormat($sAu3SH_Keywords_URL, $aKeywords[$i]) & '">' & $aAu3SH_Styles[$iAu3SH_Kwd_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	Else
		For $i = 1 To $aKeywords[0]
			$sPattern1 = '(\W)((?i)' & $aKeywords[$i] & ')(\W)'
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Kwd_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	EndIf
	
	Return $sAu3Code
EndFunc

Func __Au3_SyntaxHighlight_ParseMacros($sAu3Code)
	Local $aMacros = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.macros")
	Local $sPattern1, $sReplace1
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aMacros[0]
			$sPattern1 = '(\W+|\A)((?i)' & $aMacros[$i] & ')(\W+|$)'
			$sReplace1 = '\1<a href="' & StringFormat($sAu3SH_Macros_URL, $aMacros[$i]) & '">' & $aAu3SH_Styles[$iAu3SH_Mcro_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	Else
		For $i = 1 To $aMacros[0]
			$sPattern1 = '(\W+|\A)((?i)' & $aMacros[$i] & ')(\W+|$)'
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Mcro_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	EndIf
	
	Return $sAu3Code
EndFunc

Func __Au3_SyntaxHighlight_ParsePreProcessor($sAu3Code)
	Local $aPreProcessor = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.preprocessor")
	Local $sPattern1, $sReplace1
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aPreProcessor[0]
			$sPattern1 = '(\W+|\A)((?i)' & $aPreProcessor[$i] & ')([^\w!-]+|\Z)'
			$sReplace1 = '\1<a href="' & StringFormat($sAu3SH_PreProcessor_URL, $aPreProcessor[$i]) & '">' & _
				$aAu3SH_Styles[$iAu3SH_PrPrc_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	Else
		For $i = 1 To $aPreProcessor[0]
			$sPattern1 = '(\W+|\A)((?i)' & $aPreProcessor[$i] & ')([^\w!-]+|\Z)'
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_PrPrc_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Next
	EndIf
	
	Return $sAu3Code
EndFunc

Func __Au3_SyntaxHighlight_ParseSpecial($sAu3Code)
	Local $aSpecial = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.special")
	Local $sPattern1, $sPattern2, $sReplace1, $sReplace2
	
	For $i = 1 To $aSpecial[0]
		$sPattern1 = '(\W+|\A)((?i)' & $aSpecial[$i] & ')(.*)<span.*?>(.*?)</span>(.*?)'
		$sPattern2 = '(\W+|\A)((?i)' & $aSpecial[$i] & ')(.*)'
		$sReplace1 = '\1\2\3\4\5'
		$sReplace2 = '\1' & $aAu3SH_Styles[$iAu3SH_Spec_Style] & '\2\3' & $sAu3SH_CloseTag
		
		$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
	Next
	
	Return $sAu3Code
EndFunc

Func __Au3_SyntaxHighlight_ParseSendKeys($aStrings)
	Local $aSendKeys = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.sendkeys")
	Local $sPattern1, $sPattern2, $sReplace1, $sReplace2
	Local $sStrings
	
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_SntKys_Style] & '\1' & $sAu3SH_CloseTag
	$sPattern2 = '(?si)([\^+!#]*?{\S(?:[\h]+?[\h\d]*?)?})'
	$sReplace2 = $sReplace1
	
	For $i = 0 To UBound($aStrings)-1 Step 2
		$sStrings &= $aStrings[$i] & @CRLF
	Next
	
	For $i = 1 To $aSendKeys[0]
		$sPattern1 = '(?si)([\^+!#]*?{' & $aSendKeys[$i] & '(?:[\h]+?[\h\d]*?)?})'
		$sStrings = StringRegExpReplace($sStrings, $sPattern1, $sReplace1)
	Next
	
	$sStrings = StringRegExpReplace($sStrings, $sPattern2, $sReplace2)
	Return StringSplit(StringStripWS($sStrings, 3), @CRLF, 2)
EndFunc

Func __Au3_SyntaxHighlight_ParseComments($sAu3Code)
	Local $aSplit_Code = StringSplit(StringStripCR($sAu3Code), @LF)
	Local $sSpaces, $aSplit_Comment, $iSubCommentStarted_Count = 0
	Local $sPattern1, $sPattern2, $sPattern3, $sPattern4, $sPattern5, $sReplace2, $sReplace3, $sReplace5
	
	$sPattern1 = '(\s+)?[^(&lt|&gt)];' ; Commented line pattern
	$sPattern2 = '<\w+\h?[^>]*?>|</\w+>' ; Remove all tags pattern
	$sPattern3 = '(?i)(\s+)?#(cs|comments.*?start)(.*)' ; #comment-start search pattern. ".*?" in "comments.*?start" is the "-" seperator, can be wrapped with tags
	$sPattern4 = '(?i)(\s+)?#(ce|comments.*?end)(.*)' ; #comment-end search pattern
	$sPattern5 = '(?i)(?<=(<span class="S))(?>((\d++)">.*?)</span>)( *+)\1\3">' ; Clean double tags
	
	$sReplace3 = '\1' & $aAu3SH_Styles[$iAu3SH_CmntBlck_Style] & '#\2\3'
	$sReplace5 = '\2\4'
	
	$sAu3Code = ''
	
	; Go thru the code and check each line...
	For $i = 1 To $aSplit_Code[0]
		; Commented line
		If StringRegExp($aSplit_Code[$i], $sPattern1) Then
			; Remove all tags
			$aSplit_Comment = StringSplit($aSplit_Code[$i], ';')
			$aSplit_Code[$i] = $aSplit_Comment[1] & $aAu3SH_Styles[$iAu3SH_Cmnt_Style] & ';' & _
				StringRegExpReplace($aSplit_Comment[2], $sPattern2, '') & $sAu3SH_CloseTag
			$sAu3Code &= $aSplit_Code[$i] & @CRLF
		; Comment block
		ElseIf StringRegExp($aSplit_Code[$i], $sPattern3) Then
			; Remove all tags
			$aSplit_Code[$i] = StringRegExpReplace($aSplit_Code[$i], $sPattern2, '')
			
			; Add the comment *open* tag
			$sAu3Code &= StringRegExpReplace($aSplit_Code[$i], $sPattern3, $sReplace3) & @CRLF
			
			$iSubCommentStarted_Count += 1
			
			; Now check each line for ending of the comment block
			For $j = $i+1 To $aSplit_Code[0]
				$i = $j
				
				; Remove all tags
				$aSplit_Code[$j] = StringRegExpReplace($aSplit_Code[$j], $sPattern2, '')
				$sAu3Code &= $aSplit_Code[$j] & @CRLF
				
				; Check if current line of code is the (sub)start of comment block. If so, make a "note" for it (inrease the comments-start counter by one)
				If StringRegExp($aSplit_Code[$j], $sPattern3) Then
					$iSubCommentStarted_Count += 1
				EndIf
				
				; Check if current line of code is the end of sub comment block. If so, decrease the comments-start counter by one (to allow the ending of all comments)
				If $iSubCommentStarted_Count > 0 And StringRegExp($aSplit_Code[$j], $sPattern4) Then
					$iSubCommentStarted_Count -= 1
				EndIf
				
				; Check if current line of code is the end of (all) comment block(s). If so, exit this current loop
				If $iSubCommentStarted_Count = 0 And StringRegExp($aSplit_Code[$j], $sPattern4) Then
					$sAu3Code &= $sAu3SH_CloseTag
					ExitLoop
				EndIf
			Next
		Else
			; Clean double tags (in sequence of keywords - operators for example: == )
			$aSplit_Code[$i] = StringRegExpReplace($aSplit_Code[$i], $sPattern5, $sReplace5)
			
			$sAu3Code &= $aSplit_Code[$i] & @CRLF
		EndIf
	Next
	
	Return $sAu3Code
EndFunc

Func __Au3_SyntaxHighlight_GetKeywordStylesHeader()
	Local $sStyle
	
	Local $sRet = _
		'<head>' & @CRLF & _
		'<style type="text/css">' & @CRLF & @CRLF & _
		'.codebox' & @CRLF & _
		'{' & @CRLF & _
		'	BORDER-BOTTOM: #AAAAAA 1px solid;' & @CRLF & _
		'	BORDER-LEFT: #AAAAAA 1px solid;' & @CRLF & _
		'	BORDER-RIGHT: #AAAAAA 1px solid;' & @CRLF & _
		'	BORDER-TOP: #AAAAAA 1px solid;' & @CRLF & _
		'	PADDING-RIGHT: 8px;' & @CRLF & _
		'	PADDING-LEFT: 8px;' & @CRLF & _
		'	PADDING-BOTTOM: 8px;' & @CRLF & _
		'	PADDING-TOP: 8px;' & @CRLF & _
		'	FONT-SIZE: 12px;' & @CRLF & _
		'	FONT-FAMILY: Courier New, Verdana, Courier, Arial;' & @CRLF & _
		'	COLOR: #465584;' & @CRLF & _
		'	WHITE-SPACE: pre;' & @CRLF & _
		'	BACKGROUND-COLOR: #f0f5fa' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'a' & @CRLF & _
		'{' & @CRLF & _
		'	text-decoration:none;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'a:hover' & @CRLF & _
		'{' & @CRLF & _
		'	text-decoration:underline;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'pre' & @CRLF & _
		'{' & @CRLF & _
		'	font-family: Verdana, Arial, Helvetica, sans-serif, "MS sans serif";' & @CRLF & _
		'	line-height: normal;' & @CRLF & _
		'	margin-top: 0.5em;' & @CRLF & _
		'	margin-bottom: 0.5em;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'span' & @CRLF & _
		'{' & @CRLF & _
		'	font-family: Courier New;' & @CRLF & _
		'	color: #000000;' & @CRLF & _
		'}' & @CRLF & @CRLF
	
	For $i = 1 To 15
		$sStyle = __Au3_SyntaxHighlight_PropertyRead($sAu3_Styles_File, "style.au3." & $i, "", 0)
		$sRet &= ".S" & $i & @CRLF & "{" & @CRLF
		
		If StringInStr($sStyle, "bold") Then
			$sRet &= "	font-weight: bold;" & @CRLF
		EndIf
		
		If StringInStr($sStyle, "italics") Then
			$sRet &= "	font-style: italic;" & @CRLF
		EndIf
		
		If StringInStr($sStyle, "#") Then
			$sRet &= "	color: " & StringRegExpReplace($sStyle, ".*((?i)#[a-z0-9]+).*", "\1") & ";" & @CRLF
		EndIf
		
		$sRet &= "}" & @CRLF & @CRLF
	Next
	
	Return $sRet & '</style></head>' & @CRLF
EndFunc

Func __Au3_SyntaxHighlight_ConvertHtmlToBBCodeTags($sAu3HtmlCode)
	Local $sStyle, $sTags_Start, $sTags_End, $sRet
	
	For $i = 1 To 15
		$sStyle = __Au3_SyntaxHighlight_PropertyRead($sAu3_Styles_File, "style.au3." & $i, "", 0)
		
		$sTags_Start = ''
		$sTags_End = ''
		
		If StringInStr($sStyle, "bold") Then
			$sTags_Start = '[b]'
			$sTags_End = '[/b]'
		EndIf
		
		If StringInStr($sStyle, "italics") Then
			$sTags_Start = '[i]' & $sTags_Start
			$sTags_End &= '[/i]'
		EndIf
		
		If StringInStr($sStyle, "#") Then
			$sTags_Start = '[color=' & StringRegExpReplace($sStyle, ".*((?i)#[a-z0-9]+).*", "\1") & ']' & $sTags_Start
			$sTags_End &= '[/color]'
		EndIf
		
		; Replace <a href=> tag with bbcode [url=] tag...
		$sAu3HtmlCode = StringRegExpReplace($sAu3HtmlCode, '(?si)(.*?)<a href="([^"]*)" class="S' & $i & '">(.*?)</a>(.*?)', '\1[url=\2]\3[/url]\4')
		
		; Replace all the styles with bbcode
		$sAu3HtmlCode = StringRegExpReplace($sAu3HtmlCode, _
			'(?si)(.*?)' & $aAu3SH_Styles[$i] & '(.*?)' & $sAu3SH_CloseTag & '(.*?)', '\1' & $sTags_Start & '\2' & $sTags_End & '\3')
	Next
	
	; Replace all the html entities and <br>s
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '<br>', @CRLF)
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '&lt;', '<')
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '&gt;', '>')
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '&nbsp;', ' ')
	
	Return $sAu3HtmlCode
EndFunc

Func __Au3_SyntaxHighlight_PropertyRead($sFile, $sProperty, $sDefault = "", $iRetArr = 1)
	Local $aFileRead = StringSplit(StringStripCR(FileRead($sFile)), @LF)
	Local $sRet
	
	For $i = 1 To $aFileRead[0]
		If StringInStr($aFileRead[$i], $sProperty & "=") Then
			$aFileRead[$i] = StringTrimLeft($aFileRead[$i], StringLen($sProperty & "="))
			If StringRight($aFileRead[$i], 1) = "\" Then $aFileRead[$i] = StringTrimRight($aFileRead[$i], 1)
			$sRet &= StringStripWS($aFileRead[$i], 3)
			
			For $j = $i+1 To $aFileRead[0]
				If StringInStr($aFileRead[$j], "=") Then
					ExitLoop 2
				ElseIf StringLeft($aFileRead[$j], 1) <> "#" Then
					If StringRight($aFileRead[$j], 1) = "\" Then $aFileRead[$j] = StringTrimRight($aFileRead[$j], 1)
					$sRet &= " " & StringStripWS($aFileRead[$j], 3)
				EndIf
			Next
			
			ExitLoop
		EndIf
	Next
	
	If $sRet = "" Then
		$sRet = $sDefault
	EndIf
	
	$sRet = StringStripWS($sRet, 3)
	
	If $iRetArr Then
		Return StringSplit($sRet, " 	")
	EndIf
	
	Return $sRet
EndFunc

Opt("MustDeclareVars", 0)
#EndRegion Internal Functions
