#include-once
#include <File.au3>
#include <_zip.au3>
#include <EditConstants.au3>
#include <WindowsConstants.au3>

; #INDEX# =======================================================================================================================
; Title .........: GOLLOG
; AutoIt Version : 3.3.16.0++
; Language ......: English
; Description ...: Smart logging system
; Author(s) .....: NSC
; Last revision..: 20220913
; ===============================================================================================================================

; ------------------------------------------------------------------------------
; This software is provided 'as-is', without any express or
; implied warranty.  In no event will the authors be held liable for any
; damages arising from the use of this software.

; #VARIABLES# ===================================================================================================================
; internal use
Global $gollog_count = 0 ;total number of chars in editbox
Global $gollog_lastlog = "sicrlf" ;carriage return default yes "sicrlf" or no onetime "nocrlf", see func GOLLOG
Global $gollog_cachelog = "" ;to count number of chars on single line
Global $Gollog_edit = "" ;name of edit control
; to initialize with func _GOLLOG_INI
Global $gollog_guititle = "" ;name of gui for checkin' if activate on screen loggin'
Global $geleft = 0 ;left coord edit box
Global $getop = 0 ;top coord edit box
Global $gewidth = 0 ;width of edit box
Global $geheight = 0 ;height of edit box
; optionally to initialize with func _GOLLOG_INI, otherwise default values used
Global $GEfontcolor = "0x00FF00"
Global $GEbkcolor = "0x000000"
Global $GEfont = "consolas"
Global $GEfontsize = 8
Global $gefontweight = 800
Global $gollog_logpath = @ScriptDir & "\logs"
Global $gollog_logfile = $gollog_logpath & "\" & StringTrimRight(@ScriptName, 4) & "_LOG_"
; ===============================================================================================================================


; #CURRENT# =====================================================================================================================
; GOLLOG_INI
; GOLLOG
; GOLzipLOG
; ===============================================================================================================================

; #INTERNAL_USE_ONLY#============================================================================================================
; cleanedit
; ===============================================================================================================================

#comments-start
	Changelog:
	; V.2.31 2014-2015-2016-2018 only functions up to this version
	; V.3.00 2018/07/30 first attempt to convert to UDF
	; V.3.02 2018/08/03 bugfix in log folder creation
	; V.3.03 2021/04/08 removed a undeclared variabile, caused a warning... $test = guicreate...
	; V.3.10 2022/04/22 previously when the edit ctrl was full it was destroyed and recreated, now is only cleaned
	; V.3.15 2022/05/03 rename and rationalization of global variables, minor bugfixes
    ; V.3.16 2022/09/13 GOLLOG_INI creates read only edits, to prevent accidental user inputs, $gollog_logpath bugfix

	ideas to do

	don't repeat timestamps, at least on GUI , or highlight only time changes
	in line option to display also as balloon text one or more lines
	in line option to write only on screen and not on disk
	in line option to change font size / color only on one line (looking  $GEfontsize, $gefontweight)
	search a way to determine when the guictrledit is "full"

#comments-end

; #FUNCTION# ====================================================================================================================
; Name...........: GOLLOG_INI
; Description ...: Initialize variables for GOLLOG edit, and creates the GOLLOG EDIT BOX
;				   if the Gui window is not existant ($gollog_guititle) also creates the windows gui;  in this case, the parameters of
;				   coords and dimensions are used for the new gui window, but are coords and dimensions are inverted
;				   due to different parameters order in GUICreate and GUICtrlCreateEdit
;				   if not called gollog only write to disk with default setting (because $gollog_guititle = "")
;				   if called with &guititle = "", the edit box related variables are ignored but you can still define values for the log file
; Syntax.........: GOLLOG_INI ($gollog_guititle, $geleft, $getop, $gewidth, $geheight[, $gefontcolor [, $gebkcolor  [, $gefont [, $gefontsize [, $gefontweight [, $gollog_logfile ]]]]] )
; Parameters ....: $gollog_guititle  - name of gui
;				   $geleft - left coord of the edit (for windows gui: width)
;				   $getop - top coord of the edit (for windows gui: height)
;				   $gewidth - width of the edit (for windows gui: left coord)
;				   $geheight - height of the edit (for windows gui: top coord)
;				   $gefontcolor - optional color of font in edit control
;				   $gebkcolor - optional color of background of edit control
;				   $gefont - optional font type
;				   $gefontsize - optional size of font
;				   $gefontweight - optional weight of font
;                  $gollog_logfile - optional complete path\name of the logfile (datetime will be still added to the name)
;				   otherwise default path\name will be used
; Return values .: Success - Returns "OK"
;                  Failure - error message, sets @error = 1
;					set @extended
;                  |1 - left cooord missing
;                  |2 - top coord missing
;				   |3 - widht missing
;				   |4 - height missing
; Author ........: NSC
; Modified.......:
; Remarks .......:
; Related .......: GOLLOG
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func GOLLOG_INI($Fguititle, $Fgeleft, $Fgetop, $Fgewidth, $Fgeheight, $FGEfontcolor = "0x00FF00", $FGEbkcolor = "0x000000", $FGEfont = "consolas", $FGEfontsize = 8, $Fgefontweight = 800, $Flogfile = $gollog_logpath & "\" & StringTrimRight(@ScriptName, 4) & "_LOG_")
	$gollog_guititle = $Fguititle
	$geleft = $Fgeleft
	$getop = $Fgetop
	$gewidth = $Fgewidth
	$geheight = $Fgeheight
	; optionally to initialize with func _GOLLOG_INI, otherwise default values used
	$GEfontcolor = $FGEfontcolor
	$GEbkcolor = $FGEbkcolor
	$GEfont = $FGEfont
	$GEfontsize = $FGEfontsize
	$gefontweight = $Fgefontweight
	$gollog_logfile = $Flogfile

	If $gollog_guititle = "" Then
		Return "paramter GUI title missing"
	EndIf
	If $geleft = "" Then Return SetError(1, 1, "edit: left cooord missing")
	If $getop = "" Then Return SetError(1, 2, "edit: top coord missing")
	If $gewidth = "" Then Return SetError(1, 3, "edit: widht missing")
	If $geheight = "" Then Return SetError(1, 4, "edit: height missing")

	If Not WinExists($gollog_guititle) Then
		If $geleft = "" Then Return SetError(1, 1, "GUI: width missing")
		If $getop = "" Then Return SetError(1, 2, "GUI: heigh missing")
		If $gewidth = "" Then Return SetError(1, 3, "GUI: left cooord missing")
		If $geheight = "" Then Return SetError(1, 4, "GUI: top coord missing")
		GUICreate($gollog_guititle, $geleft, $getop, $gewidth, $geheight) ; removed $test =
		; modify parameters to fit edit in "auto" generated GUI
		$gewidth = $geleft - 10
		$geheight = $getop - 10
		$getop = 5
		$geleft = 5
		cleanedit()
		GUISetState(@SW_SHOW)
	Else
		If $geleft = "" Then Return SetError(1, 1, "left cooord missing")
		If $getop = "" Then Return SetError(1, 2, "top coord missing")
		If $gewidth = "" Then Return SetError(1, 3, "widht missing")
		If $geheight = "" Then Return SetError(1, 4, "height missing")
		cleanedit()
	EndIf
	Return "OK"
EndFunc   ;==>GOLLOG_INI

; #FUNCTION# ====================================================================================================================
; Name...........: GOLLOG
; Description ...: writes one line of log
;				   if there isn't edit control, the line is only written on the log file
;				   if there is also an associated edit control, it writes both on screen and on file
; Syntax.........: GOLLOG(ByRef $logtext)
; Parameters ....: $logtext  - a string of text to be written down on disk and on screen
;				   if $logtext containg at the end a substring like "|nocrlf50" that line does not CR ("noCRLF")
;                  except if the max specified number of chars on the line ("50") is excedeed
;				   next call of Gollog() without substring "|nocrlf50" restores auto Carriage Return behaviour
; Return values .: n/a
; Author ........: NSC
; Modified.......:
; Remarks .......:
; Related .......: GOLLOG_INI,cleanedit
; Link ..........:
; Example .......: no
; ===============================================================================================================================
Func GOLLOG($logtext)
	If Not FileExists($gollog_logpath) Then
		DirCreate($gollog_logpath) ; pre-UDF this was in the scripts
	EndIf
	$gollog_count += StringLen($logtext)
	Local $gollog_logfiletimerange = @YEAR & @MON
	Local $linelimit = StringRight($logtext, 2)
	Local $acapo
	If StringRight($logtext, 9) = "|nocrlf" & $linelimit Then
		$logtext = StringTrimRight($logtext, 9)
		$acapo = "no"
	Else
		$acapo = "si"
		$gollog_count += 4
		If $gollog_count > 10000 And $gollog_guititle <> "" Then
			;Sleep(3000)
			cleanedit()
			;	MsgBox(64, "debug", $conta)
			$gollog_count = 0
		EndIf
	EndIf

	If $acapo = "no" And (StringLen($gollog_cachelog) <= $linelimit) Then ;pearl perla no CRLF if > linelimt
		If $gollog_lastlog = "nocrlf" Then
			If WinExists($gollog_guititle) Then ; no GUI so no on screen log
				GUICtrlSetData($Gollog_edit, $logtext, 1)
			EndIf
		Else
			If WinExists($gollog_guititle) Then ; no GUI no on screen log
				GUICtrlSetData($Gollog_edit, @MDAY & "/" & @MON & "_" & @HOUR & ":" & @MIN & " " & $logtext, 1)
			EndIf
		EndIf
		$gollog_cachelog = $gollog_cachelog & $logtext
		$gollog_lastlog = "nocrlf"
	Else
		If $gollog_lastlog = "nocrlf" Then
			If WinExists($gollog_guititle) Then ; no GUI no on screen log
				GUICtrlSetData($Gollog_edit, $logtext & @CRLF, 1)
			EndIf
			$gollog_cachelog = $gollog_cachelog & $logtext
			_FileWriteLog($gollog_logfile & $gollog_logfiletimerange & ".txt", $gollog_cachelog)
			$gollog_cachelog = ""
		Else
			If WinExists($gollog_guititle) Then ; no GUI no on screen log
				GUICtrlSetData($Gollog_edit, @MDAY & "/" & @MON & "_" & @HOUR & ":" & @MIN & " " & $logtext & @CRLF, 1)
			EndIf
			_FileWriteLog($gollog_logfile & $gollog_logfiletimerange & ".txt", $logtext)
		EndIf
		$gollog_lastlog = "sicrlf"
	EndIf
EndFunc   ;==>GOLLOG

; #FUNCTION# ====================================================================================================================
; Name...........: GOLzipLOG
; Description ...: takes all log files older than specified number of months, and zip these in a unique zip file, in the same folder
; Syntax.........: GOLLOG(ByRef $months2notZIP)
; Parameters ....: $months2notZIP  - number of months (age of log files) to not zip
; Return values .: n/a
; Author ........: NSC
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: no
; ===============================================================================================================================
Func GOLzipLOG($months2NOTzip) ; zipping old log leaving unzipped only n months
	GOLLOG("ZIPping log files older than " & $months2NOTzip & " months")

	Local $hSearch = FileFindFirstFile($gollog_logfile & "*.txt") ; searching for logs
	Local $logconta = 0
	Local $gollog_logfiletimerange = @YEAR & @MON
	While 1 ; single file processing cycle
		Local $sFileName = FileFindNextFile($hSearch)
		; If there is no more file matching the search.
		If @error Then ExitLoop

		Local $stringtime = StringTrimRight(StringRight($sFileName, 10), 4) ;obtaining year-month like 201609

		If $gollog_logfiletimerange - $stringtime > $months2NOTzip Then ;zipping

			If Not FileExists($gollog_logfile & ".zip") Then
				If Not _Zip_Create($gollog_logfile & ".zip", 1) Then
					GOLLOG("ERROR " & @error & " creating " & $gollog_logfile & ".zip")
				Else
					GOLLOG("Created new log archive: " & $gollog_logfile & ".zip")
				EndIf
			Else
				GOLLOG("adding to archive: " & $gollog_logfile & ".zip")
			EndIf
			If Not _zip_additem($gollog_logfile & ".zip", $gollog_logpath & $sFileName) Then
				GOLLOG("ERROR " & @error & " zipping:  " & $gollog_logpath & $sFileName)
			Else
				GOLLOG("Added: " & $gollog_logpath & $sFileName)
				$logconta += 1
				If Not FileDelete($gollog_logpath & $sFileName) Then
					GOLLOG("ERROR - Unable to DELETE log file " & $gollog_logpath & "\" & $sFileName)
				EndIf
			EndIf
		EndIf
	WEnd
	GOLLOG("Finished = " & $logconta & " log files zipped")
EndFunc   ;==>GOLzipLOG

; #INTERNAL_USE_ONLY#============================================================================================================
; Name...........: cleanedit
; Description ...: create the edit box delete edit box and recreate it with same parameters
;				   when the edit box is "full" of chars delete edit box and recreate it with same parameters
;				   parameters are global variables specified in GOLLOG_INI
; Syntax.........: cleanedit()
; Parameters ....: n/a
; Return values .: n/a
; Author ........: NSC
; Modified.......:
; Remarks .......:
; Related .......: GOLLOG_INI
; Link ..........:
; Example .......: no
; ===============================================================================================================================
Func cleanedit() ; cleaning of edit every n lines (in program put if $nlines > xlines then this function)
	If GUICtrlGetHandle($Gollog_edit) = 0 Then
		$Gollog_edit = GUICtrlCreateEdit("", $geleft, $getop, $gewidth, $geheight,  BitOR($ES_AUTOVSCROLL, $ES_AUTOHSCROLL, $ES_WANTRETURN, $WS_BORDER,$ES_READONLY))
		GUICtrlSetData(-1, "" & @CRLF)
		GUICtrlSetFont(-1, $GEfontsize, $gefontweight, 0, $GEfont)
		GUICtrlSetColor(-1, $GEfontcolor)
		GUICtrlSetBkColor(-1, $GEbkcolor)
		GUICtrlSetCursor(-1, 3)
	Else
		GUICtrlSetData($Gollog_edit, "")
	EndIf
EndFunc   ;==>cleanedit


