; #INDEX# =======================================================================================================================
; Title ............: ImageSearch_UDF
; AutoIt Version ...: 3.3.16.1+
; Language .........: English
; Description ......: Advanced image search library
; Author(s) ........: Dao Van Trong - TRONG.PRO
; Version ..........: 3.2 (2025.10.15)
; DLL Version ......: ImageSearchDLL v3.2+
; ===============================================================================================================================
#include-once
#include <Array.au3>
#include <File.au3>
#include <WinAPI.au3>

;~ #RequireAdmin
;~ #AutoIt3Wrapper_UseX64=n
;~ #AutoIt3Wrapper_UseX64=y
;~ #pragma compile(x64, false)
;~ #pragma compile(x64, true)

; #CONSTANTS# ===================================================================================================================
Global Const $__IMGSEARCH_UDF_VERSION = '3.2'
Global Const $__IMGSEARCH_MAX_RESULTS = 64
Global Const $__IMGSEARCH_DEFAULT_TOLERANCE = 10
Global Const $__IMGSEARCH_DEFAULT_SCALE_STEP = 0.1

; DLL Error Codes (matching C++ ErrorCode enum)
Global Const $IMGSEARCH_ERROR_INVALID_PATH = -1
Global Const $IMGSEARCH_ERROR_FAILED_TO_LOAD_IMAGE = -2
Global Const $IMGSEARCH_ERROR_FAILED_TO_GET_SCREEN_DC = -3
Global Const $IMGSEARCH_ERROR_INVALID_SEARCH_REGION = -4
Global Const $IMGSEARCH_ERROR_INVALID_PARAMETERS = -5
Global Const $IMGSEARCH_ERROR_INVALID_SOURCE_BITMAP = -6
Global Const $IMGSEARCH_ERROR_INVALID_TARGET_BITMAP = -7
Global Const $IMGSEARCH_ERROR_RESULT_TOO_LARGE = -9
Global Const $IMGSEARCH_ERROR_INVALID_MONITOR = -10

; ===============================================================================================================================
; #GLOBAL VARIABLES# ============================================================================================================

Global $g_bImageSearch_Debug = True
Global $g_sImageSearchDLL_Path = ""
Global $g_hImageSearchDLL = -1
Global $g_bImageSearch_Initialized = False
Global $g_sLastDllReturn = ""
Global $g_sImageSearchDLL_Dir = @ScriptDir
Global $g_sImageSearchDLL_CustomPath = ""
Global $g_bImageSearch_UseEmbeddedDLL = False
Global $g_sImageSearch_TempDLLPath = ""

; ===============================================================================================================================
; #CURRENT# =====================================================================================================================
; Startup/Shutdown Functions:
;   _ImageSearch_Startup
;   _ImageSearch_Shutdown
;   _ImageSearch_SetDllPath
;
; Core Search Functions:
;   _ImageSearch
;   _ImageSearch_Area
;   _ImageSearch_InImage
;   _ImageSearch_hBitmap

; Utility Functions:
;   _ImageSearch_CaptureScreen
;   _ImageSearch_hBitmapLoad
;   _ImageSearch_MouseClick
;   _ImageSearch_MouseClickWin
;   _ImageSearch_ClearCache
;   _ImageSearch_GetVersion
;   _ImageSearch_GetSysInfo
;   _ImageSearch_GetLastResult
;   _ImageSearch_WarmUpCache
;
; Wait & Click Functions:
;   _ImageSearch_Wait
;   _ImageSearch_WaitClick
; ===============================================================================================================================

; ===============================================================================================================================
; STARTUP & INITIALIZATION

; ===============================================================================================================================
; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_Startup
; Description ...: Initializes the ImageSearch library by loading the DLL
; Syntax ........: _ImageSearch_Startup()
; Return values .: Success - 1
;                  Failure - 0 and sets @error:
;                  |1 - No DLL found
;                  |2 - DllOpen failed
;                  |3 - Architecture mismatch
; Author ........: Dao Van Trong
; Remarks .......: Must be called before using any search functions
;                  Automatically called on script start
; ===============================================================================================================================

Func _ImageSearch_Startup()
	If $g_bImageSearch_Initialized Then Return 1
	; Priority 1: Custom path
	If $g_sImageSearchDLL_CustomPath <> "" And FileExists($g_sImageSearchDLL_CustomPath) Then
		$g_sImageSearchDLL_Path = $g_sImageSearchDLL_CustomPath
		If $g_bImageSearch_Debug Then ConsoleWrite(">> Using custom DLL: " & $g_sImageSearchDLL_Path & @CRLF)
	EndIf
	; Priority 2: Auto-search
	If $g_sImageSearchDLL_Path = "" Then
		Local $aDllNames[3]
		If @AutoItX64 Then
			$aDllNames[0] = "ImageSearchDLL_XP_x64.dll"
			$aDllNames[1] = "ImageSearchDLL_x64.dll"
			$aDllNames[2] = "ImageSearchDLL.dll"
		Else
			$aDllNames[0] = "ImageSearchDLL_XP_x86.dll"
			$aDllNames[1] = "ImageSearchDLL_x86.dll"
			$aDllNames[2] = "ImageSearchDLL.dll"
		EndIf
		For $i = 0 To 2
			Local $sPath = $g_sImageSearchDLL_Dir & "\" & $aDllNames[$i]
			If FileExists($sPath) Then
				$g_sImageSearchDLL_Path = $sPath
				ExitLoop
			EndIf
		Next
	EndIf
	Local $sArch = _GetFileArch($g_sImageSearchDLL_Path, True)
	If ((($sArch = "x86") And @AutoItX64) Or (($sArch = "x64") And Not @AutoItX64)) Then
		If $g_bImageSearch_Debug Then ConsoleWrite('! Wrong Dll Arch [' & $sArch & ']: ' & $g_sImageSearchDLL_Path & @CRLF)
	Else
		If $g_bImageSearch_Debug Then ConsoleWrite('>> Dll Arch:  ' & $sArch & @CRLF)
	EndIf
	If ((($sArch = "x86") And @AutoItX64) Or (($sArch = "x64") And Not @AutoItX64)) Or ($g_sImageSearchDLL_Path = "") Then
		; Priority 3: Embedded DLL (fallback)
		If $g_bImageSearch_Debug And ($g_sImageSearchDLL_Path = "") Then ConsoleWrite(">> No external DLL found, trying embedded DLL..." & @CRLF)
		$g_sImageSearchDLL_Path = __ImgSearch_ExtractEmbeddedDLL()
		If $g_sImageSearchDLL_Path <> "" Then
			$g_bImageSearch_UseEmbeddedDLL = True
			$g_sImageSearch_TempDLLPath = $g_sImageSearchDLL_Path
			If $g_bImageSearch_Debug Then ConsoleWrite(">> Using embedded DLL" & @CRLF)
			$sArch = _GetFileArch($g_sImageSearchDLL_Path, True)
			If $g_bImageSearch_Debug Then ConsoleWrite('>> Dll Arch:  ' & $sArch & @CRLF)
		EndIf
		If $g_sImageSearchDLL_Path = "" Then
			If $g_bImageSearch_Debug Then ConsoleWrite("!> ERROR: No DLL found (external or embedded)" & @CRLF)
			Return SetError(1, 0, 0)
		EndIf
	EndIf
	$g_hImageSearchDLL = DllOpen($g_sImageSearchDLL_Path)
	If $g_hImageSearchDLL = -1 Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> ERROR: DllOpen failed: " & $g_sImageSearchDLL_Path & @CRLF)
		Return SetError(2, 0, 0)
	EndIf
	$g_bImageSearch_Initialized = True
	OnAutoItExitRegister("_ImageSearch_Shutdown")
	If $g_bImageSearch_Debug Then
		Local $sVersion = _ImageSearch_GetVersion()
		Local $sSysInfo = _ImageSearch_GetSysInfo()
		Local $sDllType = ($g_bImageSearch_UseEmbeddedDLL ? "[Embedded]" : "[External]")
		ConsoleWrite(">> ImageSearch DLL loaded successfully " & $sDllType & @CRLF)
		ConsoleWrite(">> Version: " & $sVersion & @CRLF)
		ConsoleWrite(">> System: " & $sSysInfo & @CRLF)
	EndIf
	Return 1
EndFunc   ;==>_ImageSearch_Startup

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_Shutdown
; Description ...: Closes the DLL and cleans up resources
; Syntax ........: _ImageSearch_Shutdown()
; Return values .: None
; Author ........: Dao Van Trong
; ===============================================================================================================================

Func _ImageSearch_Shutdown()
	If Not $g_bImageSearch_Initialized Then Return
	If $g_hImageSearchDLL <> -1 Then
		DllClose($g_hImageSearchDLL)
		$g_hImageSearchDLL = -1
	EndIf
	; Clean up embedded DLL if used
	If $g_bImageSearch_UseEmbeddedDLL And $g_sImageSearch_TempDLLPath <> "" Then
		If FileExists($g_sImageSearch_TempDLLPath) Then
			; Retry mechanism vì file có thể bị lock
			Local $iRetries = 3
			While $iRetries > 0
				FileDelete($g_sImageSearch_TempDLLPath)
				If Not FileExists($g_sImageSearch_TempDLLPath) Then
					If $g_bImageSearch_Debug Then ConsoleWrite(">> Cleaned up embedded DLL: " & $g_sImageSearch_TempDLLPath & @CRLF)
					ExitLoop
				EndIf
				Sleep(100)
				$iRetries -= 1
			WEnd
			If FileExists($g_sImageSearch_TempDLLPath) Then
				If $g_bImageSearch_Debug Then ConsoleWrite("!> Warning: Could not delete temp DLL: " & $g_sImageSearch_TempDLLPath & @CRLF)
			EndIf
		EndIf
	EndIf
	$g_bImageSearch_Initialized = False
	If $g_bImageSearch_Debug Then ConsoleWrite(">> ImageSearch DLL closed" & @CRLF)
EndFunc   ;==>_ImageSearch_Shutdown

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_SetDllPath
; Description ...: Sets a custom DLL path (must be called before _ImageSearch_Startup)
; Syntax ........: _ImageSearch_SetDllPath($sPath)
; Parameters ....: $sPath - Full path to the DLL file
; Return values .: Success - 1
;                  Failure - 0 (file not found)
; ===============================================================================================================================

Func _ImageSearch_SetDllPath($sPath)
	If Not FileExists($sPath) Then Return 0
	$g_sImageSearchDLL_CustomPath = $sPath
	Return 1
EndFunc   ;==>_ImageSearch_SetDllPath

; ===============================================================================================================================
; CORE SEARCH FUNCTIONS (NEW SIMPLIFIED INTERFACE)
; ===============================================================================================================================
; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch
; Description ...: Searches for an image on the entire screen
; Syntax ........: _ImageSearch($sImagePath[, $iTolerance = 10[, $iResults = 1[, $iCenterPos = 1[, $fMinScale = 1.0[, $fMaxScale = 1.0[, $fScaleStep = 0.1[, $iReturnDebug = 1]]]]]]])
; Parameters ....: $sImagePath     - Image file path(s), multiple separated by "|"
;                  $iTolerance     - [optional] Color tolerance 0-255 (default: 10)
;                  $iResults       - [optional] Max results to return 1-1024 (default: 1)
;                  $iCenterPos     - [optional] Return center (1) or top-left (0) coordinates (default: 1)
;                  $fMinScale      - [optional] Minimum scale factor 0.1-5.0 (default: 1.0)
;                  $fMaxScale      - [optional] Maximum scale factor (default: 1.0, must be >= $fMinScale)
;                  $fScaleStep     - [optional] Scale increment step (default: 0.1)
;                  $iReturnDebug   - [optional] Include debug info (1) or not (0) (default: 0)
; Return values .: Success - 2D Array
;                  |[0][0] - Number of matches found
;                  |[i][0] - Match index
;                  |[i][1] - X coordinate
;                  |[i][2] - Y coordinate
;                  |[i][3] - Width of matched region
;                  |[i][4] - Height of matched region
;                  Failure - Array[1][5] with [0][0] = 0 and sets @error
; Author ........: Dao Van Trong
; Remarks .......: - Transparency detection is AUTOMATIC (based on image alpha channel)
;                  - GPU/SIMD optimization is automatic
;                  - Cache is automatic
;                  - Scale search: Set $fMinScale < $fMaxScale to enable
; Example .......:
;   ; Simple search
;   $aResult = _ImageSearch("button.png")
;   If $aResult[0][0] > 0 Then
;       MouseClick("left", $aResult[1][1], $aResult[1][2])
;   EndIf
;
;   ; Multi-scale search
;   $aResult = _ImageSearch("icon.png", 10, 5, 1, 0.8, 1.2, 0.1)
;
;   ; Multiple images
;   $aResult = _ImageSearch("btn1.png|btn2.png|btn3.png", 20, 10)
; ===============================================================================================================================

Func _ImageSearch($sImagePath, $iTolerance = 10, $iResults = 1, $iCenterPos = 1, $fMinScale = 1.0, $fMaxScale = 1.0, $fScaleStep = 0.1, $iReturnDebug = 0)
	Return _ImageSearch_Area($sImagePath, 0, 0, @DesktopWidth, @DesktopHeight, 0, $iTolerance, $iResults, $iCenterPos, $fMinScale, $fMaxScale, $fScaleStep, $iReturnDebug)
EndFunc   ;==>_ImageSearch

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_Area
; Description ...: Searches for an image within a specified screen area
; Syntax ........: _ImageSearch_Area($sImagePath, $iLeft, $iTop, $iRight, $iBottom[, $iScreen = 0[, $iTolerance = 10[, $iResults = 1[, $iCenterPos = 1[, $fMinScale = 1.0[, $fMaxScale = 1.0[, $fScaleStep = 0.1[, $iReturnDebug = 1]]]]]]]])
; Parameters ....: $sImagePath     - Image file path(s), multiple separated by "|"
;                  $iLeft          - Left boundary (0 = entire screen)
;                  $iTop           - Top boundary (0 = entire screen)
;                  $iRight         - Right boundary (0 = entire screen)
;                  $iBottom        - Bottom boundary (0 = entire screen)
;                  $iScreen        - [optional] Monitor index (0 = primary, negative for entire desktop)
;                  $iTolerance     - [optional] Color tolerance 0-255 (default: 10)
;                  $iResults       - [optional] Max results 1-1024 (default: 1)
;                  $iCenterPos     - [optional] Return center (1) or top-left (0) (default: 1)
;                  $fMinScale      - [optional] Min scale 0.1-5.0 (default: 1.0)
;                  $fMaxScale      - [optional] Max scale (default: 1.0)
;                  $fScaleStep     - [optional] Scale step (default: 0.1)
;                  $iReturnDebug   - [optional] Debug mode (default: 0)
; Return values .: Same as _ImageSearch
; Author ........: Dao Van Trong
; Remarks .......: Set bounds to 0 to search entire screen
; ===============================================================================================================================

Func _ImageSearch_Area($sImagePath, $iLeft = 0, $iTop = 0, $iRight = @DesktopWidth, $iBottom = @DesktopHeight, $iScreen = 0, $iTolerance = 10, $iResults = 1, $iCenterPos = 1, $fMinScale = 1.0, $fMaxScale = 1.0, $fScaleStep = 0.1, $iReturnDebug = 0)
	If Not $g_bImageSearch_Initialized Then _ImageSearch_Startup()
	If Not $g_bImageSearch_Initialized Then Return __ImgSearch_MakeEmptyResult()
	If $g_bImageSearch_Debug Then $iReturnDebug = 1
	; Validate and normalize parameters
	$sImagePath = __ImgSearch_NormalizePaths($sImagePath)
	If $sImagePath = "" Then Return __ImgSearch_MakeEmptyResult()
	$iTolerance = __ImgSearch_Clamp($iTolerance, 0, 255)
	$iResults = __ImgSearch_Clamp($iResults, 1, $__IMGSEARCH_MAX_RESULTS)
	$iCenterPos = ($iCenterPos = 0 ? 0 : 1)
	$fMinScale = __ImgSearch_Clamp($fMinScale, 0.1, 5.0)
	$fMaxScale = __ImgSearch_Clamp($fMaxScale, $fMinScale, 5.0)
	$fScaleStep = __ImgSearch_Clamp($fScaleStep, 0.01, 1.0)
	$iReturnDebug = ($iReturnDebug = 0 ? 0 : 1)
	; Call DLL - NEW SIMPLIFIED SIGNATURE
	Local $aDLL = DllCall($g_hImageSearchDLL, "wstr", "ImageSearch", _
			"wstr", $sImagePath, _
			"int", $iLeft, _
			"int", $iTop, _
			"int", $iRight, _
			"int", $iBottom, _
			"int", $iScreen, _
			"int", $iTolerance, _
			"int", $iResults, _
			"int", $iCenterPos, _
			"float", $fMinScale, _
			"float", $fMaxScale, _
			"float", $fScaleStep, _
			"int", $iReturnDebug)
	If @error Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> DllCall error: " & @error & @CRLF)
		Return SetError(1, @error, __ImgSearch_MakeEmptyResult())
	EndIf
	; Parse result
	Local $sResult = $aDLL[0]
	$g_sLastDllReturn = $sResult
	If $g_bImageSearch_Debug Then ConsoleWrite(">> DLL returned: " & $sResult & @CRLF)
	Local $aResult = __ImgSearch_ParseResult($sResult)
	; Check if parser set error (from DLL error code)
	If @error Then
		Return SetError(@error, 0, $aResult)
	EndIf
	Return $aResult
EndFunc   ;==>_ImageSearch_Area

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_InImage
; Description ...: Searches for a target image within a source image (file-to-file search)
; Syntax ........: _ImageSearch_InImage($sSourceImage, $sTargetImage[, $iTolerance = 10[, $iResults = 1[, $iCenterPos = 1[, $fMinScale = 1.0[, $fMaxScale = 1.0[, $fScaleStep = 0.1[, $iReturnDebug = 1]]]]]]])
; Parameters ....: $sSourceImage   - Path to source (Source) image file
;                  $sTargetImage   - Path to target (Target) image file(s), multiple separated by "|"
;                  $iTolerance     - [optional] Color tolerance (default: 10)
;                  $iResults       - [optional] Max results (default: 1)
;                  $iCenterPos     - [optional] Return center (1) or top-left (0) (default: 1)
;                  $fMinScale      - [optional] Min scale (default: 1.0)
;                  $fMaxScale      - [optional] Max scale (default: 1.0)
;                  $fScaleStep     - [optional] Scale step (default: 0.1)
;                  $iReturnDebug   - [optional] Debug mode (default: 0)
; Return values .: Same as _ImageSearch
; Author ........: Dao Van Trong
; Remarks .......: Useful for pre-processing images or testing without screen capture
; Example .......: $aResult = _ImageSearch_InImage("screenshot.png", "button.png", 20)
; ===============================================================================================================================

Func _ImageSearch_InImage($sSourceImage, $sTargetImage, $iTolerance = 10, $iResults = 1, $iCenterPos = 1, $fMinScale = 1.0, $fMaxScale = 1.0, $fScaleStep = 0.1, $iReturnDebug = 0)
	If Not $g_bImageSearch_Initialized Then _ImageSearch_Startup()
	If Not $g_bImageSearch_Initialized Then Return __ImgSearch_MakeEmptyResult()
	If $g_bImageSearch_Debug Then $iReturnDebug = 1
	; Validate
	If Not FileExists($sSourceImage) Then Return __ImgSearch_MakeEmptyResult()
	$sTargetImage = __ImgSearch_NormalizePaths($sTargetImage)
	If $sTargetImage = "" Then Return __ImgSearch_MakeEmptyResult()
	$iTolerance = __ImgSearch_Clamp($iTolerance, 0, 255)
	$iResults = __ImgSearch_Clamp($iResults, 1, $__IMGSEARCH_MAX_RESULTS)
	$iCenterPos = ($iCenterPos = 0 ? 0 : 1)
	$fMinScale = __ImgSearch_Clamp($fMinScale, 0.1, 5.0)
	$fMaxScale = __ImgSearch_Clamp($fMaxScale, $fMinScale, 5.0)
	$fScaleStep = __ImgSearch_Clamp($fScaleStep, 0.01, 1.0)
	$iReturnDebug = ($iReturnDebug = 0 ? 0 : 1)
	; Call DLL
	Local $aDLL = DllCall($g_hImageSearchDLL, "wstr", "ImageSearch_InImage", _
			"wstr", $sSourceImage, _
			"wstr", $sTargetImage, _
			"int", $iTolerance, _
			"int", $iResults, _
			"int", $iCenterPos, _
			"float", $fMinScale, _
			"float", $fMaxScale, _
			"float", $fScaleStep, _
			"int", $iReturnDebug)
	If @error Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> DllCall error: " & @error & @CRLF)
		Return SetError(1, @error, __ImgSearch_MakeEmptyResult())
	EndIf
	Local $sResult = $aDLL[0]
	$g_sLastDllReturn = $sResult
	If $g_bImageSearch_Debug Then ConsoleWrite(">> DLL returned: " & $sResult & @CRLF)
	Local $aResult = __ImgSearch_ParseResult($sResult)
	If @error Then
		Return SetError(@error, 0, $aResult)
	EndIf
	Return $aResult
EndFunc   ;==>_ImageSearch_InImage

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_hBitmap
; Description ...: Searches for a target bitmap within a Source bitmap (memory-to-memory search)
; Syntax ........: _ImageSearch_hBitmap($hBitmapSource, $hBitmapTarget[, $iTolerance = 10[, $iLeft = 0[, $iTop = 0[, $iRight = 0[, $iBottom = 0[, $iResults = 1[, $iCenter = 1[, $fMinScale = 1.0[, $fMaxScale = 1.0[, $fScaleStep = 0.1[, $iReturnDebug = 1]]]]]]]]]]])
; Parameters ....: $hBitmapSource - Handle to Source bitmap (HBITMAP)
;                  $hBitmapTarget   - Handle to Target bitmap (HBITMAP)
;                  $iTolerance      - [optional] Color tolerance (default: 10)
;                  $iLeft/$iTop/$iRight/$iBottom - [optional] Search region in Source (0 = entire bitmap)
;                  $iResults        - [optional] Max results (default: 1)
;                  $iCenter         - [optional] Return center (1) or top-left (0) (default: 1)
;                  $fMinScale       - [optional] Min scale (default: 1.0)
;                  $fMaxScale       - [optional] Max scale (default: 1.0)
;                  $fScaleStep      - [optional] Scale step (default: 0.1)
;                  $iReturnDebug    - [optional] Debug mode (default: 0)
; Return values .: Same as _ImageSearch
; Author ........: Dao Van Trong
; Remarks .......: Fastest method for repeated searches (no disk I/O)
;                  Bitmaps must be created with GDI/GDI+ functions
; Example .......:
;   $hScreen = _ScreenCapture_Capture("", 0, 0, 800, 600)
;   $hIcon = _GDIPlus_BitmapCreateHBITMAPFromBitmap($pBitmap)
;   $aResult = _ImageSearch_hBitmap($hScreen, $hIcon, 10)
;   _WinAPI_DeleteObject($hScreen)
;   _WinAPI_DeleteObject($hIcon)
; ===============================================================================================================================

Func _ImageSearch_hBitmap($hBitmapSource, $hBitmapTarget, $iTolerance = 10, $iLeft = 0, $iTop = 0, $iRight = 0, $iBottom = 0, $iResults = 1, $iCenter = 1, $fMinScale = 1.0, $fMaxScale = 1.0, $fScaleStep = 0.1, $iReturnDebug = 0)
	If Not $g_bImageSearch_Initialized Then _ImageSearch_Startup()
	If Not $g_bImageSearch_Initialized Then Return __ImgSearch_MakeEmptyResult()
	If $g_bImageSearch_Debug Then $iReturnDebug = 1
	; Validate
	$iTolerance = __ImgSearch_Clamp($iTolerance, 0, 255)
	$iResults = __ImgSearch_Clamp($iResults, 1, $__IMGSEARCH_MAX_RESULTS)
	$iCenter = ($iCenter = 0 ? 0 : 1)
	$fMinScale = __ImgSearch_Clamp($fMinScale, 0.1, 5.0)
	$fMaxScale = __ImgSearch_Clamp($fMaxScale, $fMinScale, 5.0)
	$fScaleStep = __ImgSearch_Clamp($fScaleStep, 0.01, 1.0)
	$iReturnDebug = ($iReturnDebug = 0 ? 0 : 1)
	; Call DLL
	Local $aDLL = DllCall($g_hImageSearchDLL, "wstr", "ImageSearch_hBitmap", _
			"handle", $hBitmapSource, _
			"handle", $hBitmapTarget, _
			"int", $iTolerance, _
			"int", $iLeft, _
			"int", $iTop, _
			"int", $iRight, _
			"int", $iBottom, _
			"int", $iResults, _
			"int", $iCenter, _
			"float", $fMinScale, _
			"float", $fMaxScale, _
			"float", $fScaleStep, _
			"int", $iReturnDebug)
	If @error Then
		If $g_bImageSearch_Debug Then ConsoleWrite("!> DllCall error: " & @error & @CRLF)
		Return SetError(1, @error, __ImgSearch_MakeEmptyResult())
	EndIf
	Local $sResult = $aDLL[0]
	$g_sLastDllReturn = $sResult
	If $g_bImageSearch_Debug Then ConsoleWrite(">> DLL returned: " & $sResult & @CRLF)
	Local $aResult = __ImgSearch_ParseResult($sResult)
	If @error Then
		Return SetError(@error, 0, $aResult)
	EndIf
	Return $aResult
EndFunc   ;==>_ImageSearch_hBitmap

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_hBitmapLoad
; Description ...: Load image file and convert to HBITMAP handle
; Syntax ........: _ImageSearch_hBitmapLoad($sImageFile[, $iAlpha = 0[, $iRed = 0[, $iGreen = 0[, $iBlue = 0]]]])
; Parameters ....: $sImageFile - Path to image file
;                  $iAlpha     - Alpha channel (0-255, default=0 transparent)
;                  $iRed       - Red component (0-255, default=0)
;                  $iGreen     - Green component (0-255, default=0)
;                  $iBlue      - Blue component (0-255, default=0)
; Return values .: Success - HBITMAP handle (must DeleteObject when done)
;                  Failure - 0 and sets @error:
;                  |1 - DLL not initialized
;                  |2 - DLL call failed
;                  |3 - Invalid file path
; Author ........: Dao Van Trong
; Remarks .......: Remember to DeleteObject() the returned HBITMAP when finished
; Example .......: $hBitmap = _ImageSearch_hBitmapLoad("image.png", 255, 255, 255, 255) ; White background
;                  ; ... use $hBitmap ...
;                  _WinAPI_DeleteObject($hBitmap)
; ===============================================================================================================================

Func _ImageSearch_hBitmapLoad($sImageFile, $iAlpha = 0, $iRed = 0, $iGreen = 0, $iBlue = 0)
	If Not $g_bImageSearch_Initialized Then
		If Not _ImageSearch_Startup() Then Return SetError(1, 0, 0)
	EndIf
	If Not FileExists($sImageFile) Then Return SetError(3, 0, 0)
	Local $aResult = DllCall($g_hImageSearchDLL, "handle", "ImageSearch_hBitmapLoad", _
			"wstr", $sImageFile, _
			"int", $iAlpha, _
			"int", $iRed, _
			"int", $iGreen, _
			"int", $iBlue)
	If @error Or Not IsArray($aResult) Or $aResult[0] = 0 Then
		Return SetError(2, @error, 0)
	EndIf
	Return $aResult[0]
EndFunc   ;==>_ImageSearch_hBitmapLoad

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_CaptureScreen
; Description ...: Capture screen region and return as HBITMAP handle
; Syntax ........: _ImageSearch_CaptureScreen([$iLeft = 0[, $iTop = 0[, $iRight = 0[, $iBottom = 0[, $iScreen = 0]]]]])
; Parameters ....: $iLeft   - Left coordinate (default=0)
;                  $iTop    - Top coordinate (default=0)
;                  $iRight  - Right coordinate (default=0, full width)
;                  $iBottom - Bottom coordinate (default=0, full height)
;                  $iScreen - Monitor index (0=all, 1=primary, 2=secondary...)
; Return values .: Success - HBITMAP handle (must DeleteObject when done)
;                  Failure - 0 and sets @error:
;                  |1 - DLL not initialized
;                  |2 - DLL call failed
;                  |3 - Invalid screen region
; Author ........: Dao Van Trong
; Remarks .......: Remember to DeleteObject() the returned HBITMAP when finished
; Example .......: $hBitmap = _ImageSearch_CaptureScreen(0, 0, 800, 600)
;                  ; ... use $hBitmap ...
;                  _WinAPI_DeleteObject($hBitmap)
; ===============================================================================================================================

Func _ImageSearch_CaptureScreen($iLeft = 0, $iTop = 0, $iRight = 0, $iBottom = 0, $iScreen = 0)
	If Not $g_bImageSearch_Initialized Then
		If Not _ImageSearch_Startup() Then Return SetError(1, 0, 0)
	EndIf
	Local $aResult = DllCall($g_hImageSearchDLL, "handle", "ImageSearch_CaptureScreen", _
			"int", $iLeft, _
			"int", $iTop, _
			"int", $iRight, _
			"int", $iBottom, _
			"int", $iScreen)
	If @error Or Not IsArray($aResult) Or $aResult[0] = 0 Then
		Return SetError(2, @error, 0)
	EndIf
	Return $aResult[0]
EndFunc   ;==>_ImageSearch_CaptureScreen

; ===============================================================================================================================
; UTILITY FUNCTIONS
; ===============================================================================================================================
; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_ClearCache
; Description ...: Clears the internal bitmap and location cache
; Syntax ........: _ImageSearch_ClearCache()
; Return values .: None
; Author ........: Dao Van Trong
; Remarks .......: Useful for freeing memory or forcing re-scan after image updates
; ===============================================================================================================================

Func _ImageSearch_ClearCache()
	If Not $g_bImageSearch_Initialized Then Return
	DllCall($g_hImageSearchDLL, "none", "ImageSearch_ClearCache")
	If $g_bImageSearch_Debug Then ConsoleWrite(">> Cache cleared" & @CRLF)
EndFunc   ;==>_ImageSearch_ClearCache

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_GetVersion
; Description ...: Gets the DLL version string
; Syntax ........: _ImageSearch_GetVersion()
; Return values .: Version string (e.g., "ImageSearchDLL v3.1 [x64] 2025.10.15 :: Dao Van Trong - TRONG.PRO")
; ===============================================================================================================================

Func _ImageSearch_GetVersion()
	If Not $g_bImageSearch_Initialized Then _ImageSearch_Startup()
	If Not $g_bImageSearch_Initialized Then Return ""
	Local $aDLL = DllCall($g_hImageSearchDLL, "wstr", "ImageSearch_GetVersion")
	Return (@error ? "" : $aDLL[0])
EndFunc   ;==>_ImageSearch_GetVersion

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_GetSysInfo
; Description ...: Gets system capabilities and cache statistics
; Syntax ........: _ImageSearch_GetSysInfo()
; Return values .: System info string (includes CPU features, GPU support, cache size, screen resolution)
; Example .......: ConsoleWrite(_ImageSearch_GetSysInfo() & @CRLF)
;                  Output: "CPU: AVX2=Yes AVX512=No | GPU: DX12=Yes | Screen: 1920x1080 | LocationCache: 5/100 | BitmapCache: 12/100 | PoolSize: 75"
; ===============================================================================================================================

Func _ImageSearch_GetSysInfo()
	If Not $g_bImageSearch_Initialized Then _ImageSearch_Startup()
	If Not $g_bImageSearch_Initialized Then Return ""
	Local $aDLL = DllCall($g_hImageSearchDLL, "wstr", "ImageSearch_GetSysInfo")
	Return (@error ? "" : $aDLL[0])
EndFunc   ;==>_ImageSearch_GetSysInfo

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_GetLastResult
; Description ...: Gets the raw DLL return string from the last search
; Syntax ........: _ImageSearch_GetLastResult()
; Return values .: Raw result string (e.g., "{2}[100|200|32|32,150|250|32|32]<debug info>")
; Remarks .......: Useful for debugging or custom parsing
; ===============================================================================================================================

Func _ImageSearch_GetLastResult()
	Return $g_sLastDllReturn
EndFunc   ;==>_ImageSearch_GetLastResult

; ===============================================================================================================================
; WAIT & CLICK FUNCTIONS
; ===============================================================================================================================
; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_Wait
; Description ...: Waits for an image to appear on screen with timeout
; Syntax ........: _ImageSearch_Wait($iTimeout, $sImagePath[, $iTolerance = 10[, $iCenterPos = 1[, $iSleepTime = 100[, $fMinScale = 1.0[, $fMaxScale = 1.0]]]]])
; Parameters ....: $iTimeout       - Timeout in milliseconds (0 = wait forever)
;                  $sImagePath     - Image file path(s)
;                  $iTolerance     - [optional] Color tolerance (default: 10)
;                  $iCenterPos     - [optional] Return center (1) or top-left (0) (default: 1)
;                  $iSleepTime     - [optional] Sleep between checks in ms (default: 100)
;                  $fMinScale      - [optional] Min scale (default: 1.0)
;                  $fMaxScale      - [optional] Max scale (default: 1.0)
; Return values .: Success - 2D Array (same as _ImageSearch)
;                  Timeout - Empty array with [0][0] = 0
; Author ........: Dao Van Trong
; Example .......:
;   ; Wait 5 seconds for button
;   $aResult = _ImageSearch_Wait(5000, "button.png")
;   If $aResult[0][0] > 0 Then
;       MouseClick("left", $aResult[1][1], $aResult[1][2])
;   Else
;       MsgBox(0, "Timeout", "Button not found")
;   EndIf
; ===============================================================================================================================

Func _ImageSearch_Wait($iTimeout, $sImagePath, $iTolerance = 10, $iCenterPos = 1, $iSleepTime = 100, $fMinScale = 1.0, $fMaxScale = 1.0)
	Local $hTimer = TimerInit()
	While True
		Local $aResult = _ImageSearch($sImagePath, $iTolerance, 1, $iCenterPos, $fMinScale, $fMaxScale)
		If $aResult[0][0] > 0 Then Return $aResult
		If $iTimeout > 0 And TimerDiff($hTimer) > $iTimeout Then
			Return __ImgSearch_MakeEmptyResult()
		EndIf
		Sleep($iSleepTime)
	WEnd
EndFunc   ;==>_ImageSearch_Wait

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_WaitClick
; Description ...: Waits for an image and clicks it when found
; Syntax ........: _ImageSearch_WaitClick($iTimeout, $sImagePath[, $sButton = "left"[, $iClicks = 1[, $iTolerance = 10[, $iSleepTime = 100]]]])
; Parameters ....: $iTimeout       - Timeout in milliseconds (0 = wait forever)
;                  $sImagePath     - Image file path(s)
;                  $sButton        - [optional] Mouse button: "left", "right", "middle" (default: "left")
;                  $iClicks        - [optional] Number of clicks (default: 1)
;                  $iTolerance     - [optional] Color tolerance (default: 10)
;                  $iSleepTime     - [optional] Sleep between checks in ms (default: 100)
; Return values .: Success - 1 (image found and clicked)
;                  Timeout - 0 (image not found)
; Author ........: Dao Van Trong
; Example .......:
;   ; Wait and click button
;   If _ImageSearch_WaitClick(5000, "button.png") Then
;       MsgBox(0, "Success", "Button clicked!")
;   Else
;       MsgBox(0, "Failed", "Button not found")
;   EndIf
; ===============================================================================================================================

Func _ImageSearch_WaitClick($iTimeout, $sImagePath, $sButton = "left", $iClicks = 1, $iTolerance = 10, $iSleepTime = 100)
	Local $aResult = _ImageSearch_Wait($iTimeout, $sImagePath, $iTolerance, 1, $iSleepTime)
	If $aResult[0][0] > 0 Then
		If _ImageSearch_MouseClick($sButton, $aResult[1][1], $aResult[1][2], $iClicks, 10) Then Return 1
		Return MouseClick($sButton, $aResult[1][1], $aResult[1][2], $iClicks)
	EndIf
	Return 0
EndFunc   ;==>_ImageSearch_WaitClick

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_MouseClick
; Description ...: Clicks mouse at coordinates (screen or current position)
; Syntax ........: _ImageSearch_MouseClick($sButton[, $iX=-1[, $iY=-1[, $iClicks=1[, $iSpeed=0[, $iScreen=0]]]]])
; Parameters ....: $sButton - Button: "left", "right", "middle" (default: "left")
;                  $iX, $iY - Coordinates (-1 = current position)
;                  $iClicks - Number of clicks (default: 1)
;                  $iSpeed - Speed 0-100 (0=instant, default: 0)
;                  $iScreen - Monitor index (default: 0)
; Return values .: 1 on success, 0 on failure
; ===============================================================================================================================

Func _ImageSearch_MouseClick($sButton, $iX = -1, $iY = -1, $iClicks = 1, $iSpeed = 0, $iScreen = 0)
	If Not _ImageSearch_Startup() Then Return SetError(-1, 0, 0)
	Local $aRet = DllCall($g_hImageSearchDLL, "int:cdecl", "ImageSearch_MouseClick", "wstr", $sButton, "int", $iX, "int", $iY, "int", $iClicks, "int", $iSpeed, "int", $iScreen)
	If @error Then Return SetError(1, 0, 0)
	Return $aRet[0]
EndFunc   ;==>_ImageSearch_MouseClick

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_MouseClickWin
; Description ...: Clicks mouse in a window
; Syntax ........: _ImageSearch_MouseClickWin($sTitle, $sText, $iX, $iY[, $sButton="left"[, $iClicks=1[, $iSpeed=0]]])
; Parameters ....: $sTitle - Window title/class/handle
;                  $sText - Window text
;                  $iX, $iY - Relative coordinates in window
;                  $sButton - Button (default: "left")
;                  $iClicks - Number of clicks (default: 1)
;                  $iSpeed - Speed 0-100 (default: 0)
; Return values .: 1 on success, 0 on failure
; ===============================================================================================================================

Func _ImageSearch_MouseClickWin($sTitle, $sText, $iX, $iY, $sButton = "left", $iClicks = 1, $iSpeed = 0)
	If Not _ImageSearch_Startup() Then Return SetError(-1, 0, 0)
	Local $aRet = DllCall($g_hImageSearchDLL, "int:cdecl", "ImageSearch_MouseClickWin", "wstr", $sTitle, "wstr", $sText, "int", $iX, "int", $iY, "wstr", $sButton, "int", $iClicks, "int", $iSpeed)
	If @error Then Return SetError(1, 0, 0)
	Return $aRet[0]
EndFunc   ;==>_ImageSearch_MouseClickWin

; ===============================================================================================================================
; INTERNAL HELPER FUNCTIONS
; ===============================================================================================================================
; Parse DLL result string
; Format: {count}[x|y|w|h,x|y|w|h,...](debug_info)          - Success
;         {error_code}[]<Error Name>(debug_info)            - Error

Func __ImgSearch_ParseResult($sResult)
	; Extract count or error code between { }
	Local $sCountStr = __ImgSearch_ExtractBetween($sResult, "{", "}")
	Local $iCount = Number($sCountStr)
	; Check if error (negative count or has error message)
	Local $sErrorMsg = __ImgSearch_ExtractBetween($sResult, "<", ">")
	If $iCount < 0 Or $sErrorMsg <> "" Then
		; Error occurred
		If $g_bImageSearch_Debug Then
			ConsoleWrite("!> DLL Error [" & $iCount & "]: " & $sErrorMsg & @CRLF)
			; Extract debug info if exists
			Local $sDebugInfo = __ImgSearch_ExtractBetween($sResult, "(", ")")
			If $sDebugInfo <> "" Then ConsoleWrite("   Debug: " & $sDebugInfo & @CRLF)
		EndIf
		Return SetError($iCount, 0, __ImgSearch_MakeEmptyResult())
	EndIf
	; Extract matches between [ ]
	Local $sMatches = __ImgSearch_ExtractBetween($sResult, "[", "]")
	; Create result array
	Local $aResult[$iCount + 1][5]
	$aResult[0][0] = $iCount
	If $iCount = 0 Or $sMatches = "" Then Return $aResult
	; Parse matches: x|y|w|h,x|y|w|h,...
	Local $aRecords = StringSplit($sMatches, ",", 3)
	Local $iValid = 0
	For $i = 0 To UBound($aRecords) - 1
		Local $aParts = StringSplit($aRecords[$i], "|", 3)
		If UBound($aParts) >= 4 Then
			$iValid += 1
			$aResult[$iValid][0] = $iValid
			$aResult[$iValid][1] = Number($aParts[0])  ; X
			$aResult[$iValid][2] = Number($aParts[1])  ; Y
			$aResult[$iValid][3] = Number($aParts[2])  ; W
			$aResult[$iValid][4] = Number($aParts[3])  ; H
			If $iValid >= $iCount Then ExitLoop
		EndIf
	Next
	; Resize array to actual valid count
	$aResult[0][0] = $iValid
	ReDim $aResult[$iValid + 1][5]
	; Debug output if enabled
	If $g_bImageSearch_Debug Then
		Local $sDebugInfo = __ImgSearch_ExtractBetween($sResult, "(", ")")
		If $sDebugInfo <> "" Then ConsoleWrite(">> Debug: " & $sDebugInfo & @CRLF)
	EndIf
	Return $aResult
EndFunc   ;==>__ImgSearch_ParseResult

; Extract string between delimiters
Func __ImgSearch_ExtractBetween($sString, $sStart, $sEnd)
	If $sString = "" Or $sStart = "" Or $sEnd = "" Then Return ""
	Local $aRet = StringRegExp($sString, "(?si)\Q" & $sStart & "\E(.*?)" & "(?=\Q" & $sEnd & "\E)", 3)
	If @error Then
		Local $iStart = StringInStr($sString, $sStart)
		If $iStart = 0 Then Return ""
		Local $iEnd = StringInStr($sString, $sEnd, 0, 1, $iStart + StringLen($sStart))
		If $iEnd = 0 Or $iEnd <= $iStart Then Return ""
		Return StringMid($sString, $iStart + StringLen($sStart), $iEnd - $iStart - StringLen($sStart))
	EndIf
	If IsArray($aRet) And UBound($aRet) > 0 Then Return $aRet[0]
	Return ""
EndFunc   ;==>__ImgSearch_ExtractBetween

; Normalize file paths
Func __ImgSearch_NormalizePaths($sInput)
	If $sInput = "" Then Return ""
	; Remove duplicate delimiters
	While StringInStr($sInput, "||")
		$sInput = StringReplace($sInput, "||", "|")
	WEnd
	; Trim delimiters
	$sInput = StringStripWS($sInput, 3)
	$sInput = StringRegExpReplace($sInput, "^\|+|\|+$", "")
	; Validate files exist
	If Not StringInStr($sInput, "|") Then
		Return (FileExists($sInput) ? $sInput : "")
	EndIf
	Local $aPaths = StringSplit($sInput, "|", 3)
	Local $sValid = ""
	For $i = 0 To UBound($aPaths) - 1
		If FileExists($aPaths[$i]) Then
			$sValid &= ($sValid = "" ? "" : "|") & $aPaths[$i]
		EndIf
	Next
	Return $sValid
EndFunc   ;==>__ImgSearch_NormalizePaths

; Clamp value between min and max
Func __ImgSearch_Clamp($vValue, $vMin, $vMax)
	If $vValue < $vMin Then Return $vMin
	If $vValue > $vMax Then Return $vMax
	Return $vValue
EndFunc   ;==>__ImgSearch_Clamp
; Create empty result array

Func __ImgSearch_MakeEmptyResult()
	Local $aResult[1][5] = [[0, 0, 0, 0, 0]]
	Return $aResult
EndFunc   ;==>__ImgSearch_MakeEmptyResult

; #FUNCTION# ====================================================================================================================
; Name ..........: _ImageSearch_WarmUpCache
; Description ...: Pre-loads images into cache for faster subsequent searches
; Syntax ........: _ImageSearch_WarmupCache($sImagePaths)
; Parameters ....: $sImagePaths - Pipe-separated list of images to preload
; Return values .: Success - Number of images cached
;                  Failure - 0
; Author ........: Dao Van Trong
; Remarks .......: Call this during app initialization for better performance
; Example .......:
;   _ImageSearch_WarmUpCache("btn1.png|btn2.png|icon.png")
; ===============================================================================================================================

Func _ImageSearch_WarmUpCache($sImagePaths)
	If Not $g_bImageSearch_Initialized Then _ImageSearch_Startup()
	Local $aFiles = StringSplit($sImagePaths, "|", 2)
	Local $iCached = 0
	For $sFile In $aFiles
		If FileExists($sFile) Then
			; Dummy search to trigger cache
			_ImageSearch($sFile, 0, 1, 1, 1.0, 1.0, 0.1, 0)
			$iCached += 1
		EndIf
	Next
	Return $iCached
EndFunc   ;==>_ImageSearch_WarmUpCache


; #FUNCTION# ====================================================================================================================
; Name............: _GetFileArch
; Description....: Determine the architecture (x86, x64, ARM, etc.) of the executable file or library (EXE, DLL, SYS, OCX...).
; Syntax.........: _GetFileArch($sFilePath [, $bAsText = True])
; Parameters.....: $sFilePath - File path to check
;                  $bAsText - True => Returns the description string (default)
;                             False => Returns the architecture code (e.g., 32, 64, 65,...)
; Return values..: Success: + When $bAsText = True → Returns the description string (e.g., "SCS_64BIT_BINARY", "ARM64", ...)
;                           + When $bAsText = False → Returns the code (eg: 64, 65, ...)
;                  Failed → SetError(code, ext, message)
; Author.........: Dao Van Trong - TRONG.PRO
; =========================================================================================================================

Func _GetFileArch($sFilePath, $bAsText = True)
	If Not FileExists($sFilePath) Then Return SetError(-1, 0, '')
	Local $tType = DllStructCreate("dword lpBinaryType")
	Local $aRetAPI = DllCall("kernel32.dll", "bool", "GetBinaryTypeW", "wstr", $sFilePath, "ptr", DllStructGetPtr($tType))
	If @error = 0 And $aRetAPI[0] Then
		Local $BinaryType = DllStructGetData($tType, "lpBinaryType")
		Switch $BinaryType
			Case 0
				Return SetError(0, 0, ($bAsText ? "x86" : 32)) ; (I386) SCS_32BIT_BINARY
			Case 6
				Return SetError(0, 6, ($bAsText ? "x64" : 64)) ; (AMD64)SCS_64BIT_BINARY
			Case 1
				Return SetError(0, 1, ($bAsText ? "DOS" : 1)) ;SCS_DOS_BINARY
			Case 2
				Return SetError(0, 2, ($bAsText ? "WOW" : 2)) ;SCS_WOW_BINARY
			Case 3
				Return SetError(0, 3, ($bAsText ? "PIF" : 3)) ;SCS_PIF_BINARY
			Case 4
				Return SetError(0, 4, ($bAsText ? "POSIX" : 4)) ;SCS_POSIX_BINARY
			Case 5
				Return SetError(0, 5, ($bAsText ? "OS216" : 5)) ;SCS_OS216_BINARY
				;Case Else
				;    Return SetError(1, $BinaryType, "Unknown (API Code: " & $BinaryType & ")")
		EndSwitch
	EndIf
	Local $hFile = _WinAPI_CreateFile($sFilePath, 2, 7)
	If $hFile = 0 Then Return SetError(2, 0, "Error: Cannot open file")
	Local $tDosHeader = DllStructCreate("char Magic[2];byte[58];dword Lfanew")
	Local $aRead = _WinAPI_ReadFile($hFile, DllStructGetPtr($tDosHeader), 64, 0)
	If Not $aRead Or DllStructGetData($tDosHeader, "Lfanew") < 64 Then
		_WinAPI_CloseHandle($hFile)
		Return SetError(3, 0, "Error: Cannot read DOS header")
	EndIf
	If DllStructGetData($tDosHeader, "Magic") <> "MZ" Then
		_WinAPI_CloseHandle($hFile)
		Return SetError(4, 0, "Error: Not a valid PE file")
	EndIf
	_WinAPI_SetFilePointer($hFile, DllStructGetData($tDosHeader, "Lfanew"))
	Local $tNtHeaders = DllStructCreate("dword Signature;word Machine;word NumberOfSections;byte[18]")
	$aRead = _WinAPI_ReadFile($hFile, DllStructGetPtr($tNtHeaders), 24, 0)
	If Not $aRead Then
		_WinAPI_CloseHandle($hFile)
		Return SetError(5, 0, "Error: Cannot read NT headers")
	EndIf
	If DllStructGetData($tNtHeaders, "Signature") <> 0x4550 Then
		_WinAPI_CloseHandle($hFile)
		Return SetError(6, 0, "Error: Invalid PE signature")
	EndIf
	Local $Machine = DllStructGetData($tNtHeaders, "Machine")
	_WinAPI_CloseHandle($hFile)
	Switch $Machine
		Case 0x014C
			Return SetError(0, 32, ($bAsText ? "x86" : 32)) ; (I386) SCS_32BIT_BINARY
		Case 0x8664
			Return SetError(0, 64, ($bAsText ? "x64" : 64)) ; (AMD64) SCS_64BIT_BINARY
		Case 0xAA64
			Return SetError(0, 65, ($bAsText ? "ARM64" : 65))
		Case 0x01C0
			Return SetError(0, 66, ($bAsText ? "ARM" : 66))
		Case 0x01C2
			Return SetError(0, 67, ($bAsText ? "THUMB" : 67))
		Case 0x01C4
			Return SetError(0, 68, ($bAsText ? "ARMNT" : 68))
		Case 0x0200
			Return SetError(0, 69, ($bAsText ? "IA64" : 69))
		Case 0x01C5
			Return SetError(0, 70, ($bAsText ? "ARMV7" : 70))
		Case 0x0162
			Return SetError(0, 71, ($bAsText ? "R3000" : 71))
		Case 0x0166
			Return SetError(0, 72, ($bAsText ? "R4000" : 72))
		Case 0x01F0
			Return SetError(0, 73, ($bAsText ? "POWERPC" : 73))
		Case 0x01F1
			Return SetError(0, 74, ($bAsText ? "POWERPCFP" : 74))
		Case 0x0EBC
			Return SetError(0, 75, ($bAsText ? "EBC" : 75))
		Case 0xC0EE
			Return SetError(0, 76, ($bAsText ? "CEE" : 76))
		Case Else
			Return SetError(7, $Machine, "Unknown (Machine: 0x" & Hex($Machine, 4) & ")")
	EndSwitch
EndFunc   ;==>_GetFileArch

; ===============================================================================================================================
; EMBEDDED DLL SUPPORT
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: __ImgSearch_ExtractEmbeddedDLL
; Description ...: Extracts embedded DLL to temp directory
; Syntax ........: __ImgSearch_ExtractEmbeddedDLL()
; Return values .: Success - Path to extracted DLL
;                  Failure - Empty string
; Author ........: Dao Van Trong
; ===============================================================================================================================

Func __ImgSearch_ExtractEmbeddedDLL()
	Local $sHexData, $sDllName
	If StringInStr(@OSVersion, 'XP') Or StringInStr(@OSVersion, '2003') Then
		$sHexData = (@AutoItX64 ? __ImgSearch_GetEmbeddedDLL_XP_x64() : __ImgSearch_GetEmbeddedDLL_XP_x86())
		$sDllName = (@AutoItX64 ? "ImageSearchDLL_XP_x64.dll" : "ImageSearchDLL_XP_x86.dll")
	Else
		$sHexData = (@AutoItX64 ? __ImgSearch_GetEmbeddedDLL_x64() : __ImgSearch_GetEmbeddedDLL_x86())
		$sDllName = (@AutoItX64 ? "ImageSearchDLL_x64.dll" : "ImageSearchDLL_x86.dll")
	EndIf
	If ($sHexData == "") Then Return "" ; No embedded DLL
	Local $sTempPath = @TempDir & "\" & $sDllName
	; Write binary to file
	Local $hFile = FileOpen($sTempPath, 2 + 8 + 16) ; Binary + overwrite
	If $hFile = -1 Then Return ""
	FileWrite($hFile, Binary($sHexData))
	FileClose($hFile)
	If Not FileExists($sTempPath) Then Return ""
	If $g_bImageSearch_Debug Then ConsoleWrite(">> Extracted embedded DLL to: " & $sTempPath & @CRLF)
	Return $sTempPath
EndFunc   ;==>__ImgSearch_ExtractEmbeddedDLL

; #FUNCTION# ====================================================================================================================
; Name ..........: __ImgSearch_GetEmbeddedDLL_x64()
; Description ...: Returns hex string of embedded x64 DLL
; Syntax ........: __ImgSearch_GetEmbeddedDLL_x64()
; Return values .: Hex string of DLL binary (empty = not embedded)
; Author ........: Dao Van Trong
; Remarks .......: PLACEHOLDER - Add hex string here to enable embedded DLL
;                  To generate hex: FileRead(DLL_PATH, 16) then convert to string
; ===============================================================================================================================

Func __ImgSearch_GetEmbeddedDLL_x64()
	; This function holds the hex data for ImageSearchDLL_MD_x64.dll
	; File size: 93 KB
	; Architecture: x64
	Local $sHexData = '0x4D5A90000300000004000000FFFF0000B800000000000000400000000000000000000000000000000000000000000000000000000000000000000000080100000E1FBA0E00B409CD21B8014CCD21546869732070726F6772616D2063616E6E6F742062652072756E20696E20444F53206D6F64652E0D0D0A24000000000000006F657E712B0410222B0410222B041022227C83223B041022608E142323041022608E152337041022608E13232E041022608E11232D04102252851123220410222B041122320510225D8F19232A0410225D8F10232A0410225D8FEF222A0410222B0487222A0410225D8F12232A041022526963682B041022000000000000000000000000000000005045000064860300C875F4680000000000000000F00022200B020E32007001000010000000A002005004040000B00200000000800100000000100000000200000600000000000000060000000000000000300400000400000000000002006001000010000000000000100000000000000000100000000000001000000000000000000000100000000029040080010000942404006C040000002004009404000000900300FC2100000000000000000000802A04002C000000000000000000000000000000000000000000000000000000400704002800000020110400400100000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000555058300000000000A00200001000000000000000040000000000002C461D7400000000800000E055505831000000000070010000B00200006401000004000000000000ECE61F7400000000400000E02E727372630000000010000000200400000C00000068010000000000EC56213800000000400000C0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000332E303200555058210D2408'
	$sHexData &= '0AB965919D691801E862EE03004654010000980300490900F2F9FF6FFE4883EC28B91000E809021534488D0D9B7C030048C74008007BF3DBFD4889058C1D890830D2430283C428E95077C8177618C8CC017F3889587F77BF67766EEC8D05F10C010F57C0660F7F053BEF3CB3EE2BF30714F0D7CFE05F9A54AE803F4C8BC0BA21E4B72590F77624AE01BE8C90FE821E0563A73089299FF37B840CF240E47B62F2405E1EDC7BE17BDE7BC9411EF2E37BE07B9E7B1BB14B1EA77B12FF38A6826F70055D404D442430B9301748221FBCE038014D402D52401CE4210F164F404C400A4080BCE4911340A6427998BC21BFB942D69DB7839F2D3F895C2408576A208C0547916FBFDD3E8BD926018BFA0E4908FF1585542240F6C70159B7FDDA74271ECB100475142840544DC304C47EF0AF7430EA205FC3BA18CB0110BC2F7667F6BFBE8B0948FF25F65362DF4053D63AD67578F08B05833DF233C44928E7C70E342BB8872005EE145418CE284A733BFE85C074034010B9C1F206B0C3057F243833C94C5C9890028F0F111BBE5BFBB04810704C688B531016084CD689507DF0617B23B633CCFC142C20305BC3A72586AB25CC5FAC415B0A0E0E041174833F4CF0624F8BC2C35F497A9227C30776704C9E180814535556AFDD3ABCBD38498BF0486C247848A7D9F000037C57D02A9ABA1E2896CE04C7EFD3'
	$sHexData &= 'F8AD2C680408E83B52B9FF000F48C14F87053F44385F5E5D4B7F55509F7C3674A12090E5E03A3C452833FCBFD9B7C0A2D1390580792210C9744D4983F81072470FB7CAC7F6FF77D46EC1C4E27D79C0C5FE7F4500C5F877201F84E316FCEDFFC4C17E6F01C5FD754D2AFDD7C985C90F854004F27DDB47DBB4C01078C126480E3BC86076D67BDDDBE716040F874005390FBFA7C926080EDBECDF6D61C90670C900387723F3410F9E1C75C1069F7BDE76D7C8669E7576960810574476DDF846DE60915E7E44604584FE5B412B6A6722B60CBCB6D1E94803C8498D044AD31E280AC84D67505DC33CD1571B3C4DEAD0523A43B7FF82166A8722C10AC073136641391442337C156F440C42741CDCC02472ED8E5DB928B501299F1059D11DF2B33A087FDA0EF87770B96251D9141AD274352FD9E1CAEFE109C4C175750A83F9FFE3067A83833404EAC276D4B2B0C2F10D6B7737BA1E5A6690DCECA1EBBF09090268C80684663BCAC77061B8BCD184D9D0B748F52507014D181A124C60DED0EE9F84FE4933D141AC004854D241B901866DD38D0BC1EA04B204026C0C0284E469F0FF044B6643390443450F42CA41FC7608B159B6B910F7D16D7E93AB0D426C2A7D988348B1CD164E85F7836B16C9B3390601140153D89AA32B8F19A44441620C43535BC5662406751D9B6BB860B52D73E7E36442C24301AB'
	$sHexData &= 'D5155F2070F0A102EDEB7E77E51253900B1EB0AACBCBFA02683E4D235FDB42BAA0202A510846B5EB6A68807771D2D0457D3DCFA101FF3F77F9DA077CF3AB24C11ADD4CF6C301740DC2AF073C7836CF4018E8C3787911E001C97FF976B484AC1C83C19B4CCA41F95E36C1A241104108640677F9DEDA61949B01823949071C322C1FA54CD84A16B87DFAD84A5F46B25DAEB0CB70EF489D76C00362487C3AD9DD36E0DC1F48488D70082C12151303F2A8383BCE2102223FFD752E0DE7019E4B321F0313C8D0D6E57FBD5EAF309223B775DF0D57C161BB05B24CE74752FF70DBE0905F3E933660C12844E0FF77DC414C8E5AF630794741F7D80F1F400049831755B5F3E902B8CD1EE016039AC266C1025C75D3E0029C106603D2442BC15A6A3BC355CA306645947C6C75D02040FBEDE0FE26C7012D00EB347F15165DA6044428D11E7EC5254F674DDC166070A9867C26589F40C33542158A8A148DC8C68E01F37098EC867B748E6623971FBA8B5F53209B104A0926977C9386E3304079809C2A3F1D74BE49D1D50019850445038FE885AD09502D46A50558C3DE7D517CDC05E45F448902D3894A08C30E0D03215F0146D86456F6F621335018204B084C8BFABDFEC2EDDB493906750E8B0B390A08B001A932C0992E2D17B950424C8A61AA9EFF3C4439027503B001C31F01AFD574F592891113FF3EFD'
	$sHexData &= '15DE85209241564157EAC94881ECA01293555870F041342028F8A138F2E74D17D0FF52F09A172BEA4DF79C374CF83E175C71FF5302767810741541B8027A4E1563739BA6E0ABC84A706CB6060655D7B8C02D6812CEBA109015807DEF0F5BDA66056047127A45E754CF34055B7B738B55EFFA0F7636124548FFC220FA3687B5CD0010721E1E48D8C238E808F0BEBFDD06F81F77A083C227EB0AB905E0CD29D7FE7E4B0C766145B74C897DC706CF0F10077323DAE61D0C4F100E4D1C47180F67C1DA76C607003A7F101D2445B75A34E394CF12FE72D90B4637F83C15A702D1C6451F08585E76180C24AB47A483A5420E396ACFB73E5A41E492725206D64300C3E23E48171C278B9C24E862029578A53AC4415F415E6CC34E5EC889BF5F71FD46287EE225409E18B360BBCA21BFAA3D325040BD8B72DAD2BCE8088B2A64CE8B5A04C51A06543EBC2DAE687C24380F76D6700FACCE95D25774483C97CA953BB60B5C40406150B93583466B8838700555820707349CF995375897F6DA3EB89B85896F1892CA895F1C107720873330F409509F900200D3A0876091BF7870B0E9C05A22300BBCBB3DB04B716130322944510D4D38959BECDAF90240DF2C8BA05806F2CC6F1D390E03827491804B07072F9B086A457B81E3684107170B3BC8F43B75ABAD4264583E49BF4D468E906C0A1B74D05AD898B7AB'
	$sHexData &= '7EDA69432A5E05A4B1DFD8DC309C3041BCC4AE7A410AA7660F451C8E07800FCA13E0CF7A14181C237F78C3A1BD90E1DE014F2C064C462A4030FFC8434587F6C24A0AF979B0AAB045DF3F1906F9032B678F2F38D3CC4D015EDA5424280C7C4CF7C7336E16414304F84130702133E22266EAE205A123057862EB0D7A0D857D9782B74DF503567B10061896CB9039E0D042C5E867B39C83079138CD50C6406C072D555FF8F82645D6751CC5B34A0E0886692F426E1A0C29D2C2A564C4973C95EF774E1B893B4CD23043C300B2BD96E434272F8A0287865F9130FF5E2107D1DF11EAF6A112DF4ABF6DDF4E2342C85E7CDF121C5438857CF8028D6C1F189295032E42F77B3393D6A845F35FD25C2805F885C9741D4A01DC122B8CD6D800BA448C56C08D6FEE49FF200D76816118FFF3CE2F3153B47281499C029D3F9003E490EF20F9F5D08EE408832A042D6966C1559FD9B0E53F10F8F8FF41B3543E4C3681FF809698007515496BC164FFBF542CD5D33A00366E01756549BAF38C90FB7F37FC9407FCF4B2496C49F7E90A4D8D041149C1F8184988C7DBDDFFC1E93F4C03C14969C85B4C2BC912C900CA9AB8DD612B52F77AD13AFA18D9C1E83FED6C6922E034C012D589137B3BB481DB9960FF5F69C245898951606F2BBA39C803E4449FF32BAB6AB272481E736B4040597FF11737108946F220FF53'
	$sHexData &= '189098090FBD1BD1A4900FCB8E4678FD61E9D31F085D7597BBA8742410B95D4973E0C0A867C9D82E683CEFBAC456E2CE40060F114606F0DE231D40804089467C76225E5C97553BFC7E2098923E103BFA74B9DA8BB258457A06157C3090CF611B0A6CEDF340400A569F7E116C8AB56049FA38852AE9C3588745989633D2099705F6A0E46730268E3B07DD9ABF9D73837C3A0074748B0D6266657C0425587DFF4B932C3E1404C88B0C02390DB3707E2D6FF70EE782AAF1B8833D9E9975182885C0EDB7CE2D020789414CEE33E3064E1636F489052F0E5E2E07026204741956719B7131EB206F3879A9529D566F0B594A901F030AFF8EC3770A11E84093BFF10BDA473ADA04F13A874210F0FF424A20C07B6C8AB8688B923070585C1CE21734798838FE6616889064993A33B88B474C3DC7BFF62FDA7F750AFFC88916E926D940B5018B144A07F007AC394032EDC73880667462950818CF9AFDA87A837E68A6065668EBEC6870FAF9B6B7A6064E70CF614084ED4064B68F8618181C623C00C11BB5BF4AF00FB17B10757344139B5210B903595185FFAFC72C4B100F45F883FF027556F260A7615959BAF039740CD479FDCE04F7CC8B83AC1867141DB906CEECB9F24158C683B00183DEDB504FF31218DCB88ACEC14368A5A0EB0883F8D809020350080F491AD1A3608230BF6678E35BFE20A6CBFAC3'
	$sHexData &= '57DD769A59183728154E3995F7FEC10F19A0817B4C2B0CC7434CFE10E928FCC21B6F708B4E68BD93FE644532F63B608F7282FF30C03939851F1CDA740D10D1B3DCECEB241B4EB8418C9D04A2BE7908E0A7E3331456700F8BC58747104FD35C8695470E07CFEB2B377CF00341B60135E338FA4584F6198DCEFEC15373F54F108BD8278BD02A9AEF1FB6792BD1747506713B6D64BEA605FF7F3BD87463876F1083FD03745BDA4F60CAFDC0E1CB68386081BFD9C916C787E5F386D317FEF9DB3980BFCDFB000264528D57607A184A87337B3B2F74E9A147E9F10DE586F26C24680CC3876BC25DCF91927F0977A9B80C4115E34106720142971750C71F7969EA52037F5B90FE0334AD36606F0EFFA69009798376081016156811DF46485C4137C6F0F340CA3881BB94652B6DC31F51C783946F42A883BB7BD1859FB0027D16364873198E7284A3B72B7CEA7A48183872723B9BCA250BFF08F924970339C1364ABF1BB94C328728BF0A32EDB4073D11BD9AABF84F62B8412E308E28BFC55E17625922DB7436BB30BF457472939593C31B10355C42FEE67B0C83FF609F219BD7889F143C73F0E500A15E3D63BFDE38C255A764F777C1082201264CB0FDB860BF38CA750784CB35B7D23E62C4C1902C4B74141EF6D874FC483BCB0F95C2E6204338E9BA40C3B0151503BF193D81A245AC2F919F445A0A69'
	$sHexData &= '9449713CD142F05E985F902116EDCAC67F6A940E34839590F241690CC43820E80ACB8B77E287D66A3481A7F7A51F4433EC7A0A89EE96858E8CCC5FED6C0326B891720E5F566DC0080D740285014538EF0C0365B8061081BE689562520E2B8AC1FA03882E78578E08D1001FBAD07745997D3D2C6FB960DCE17C5A5278C4CDA6625B8B7E2BDA4C84CF3D7663F341B0C1FB920CDFAE0C73BD6F6E4E44C6B15E201BA666FC76823CCC9F0944B335C9D0A9209F10CF975387725E389E4C0706DFE3897910F40C18BE4FB44D74780FEC8BF0441C32EA660E0CCAFF7BCF5B1A413806485852280630897168C7412012DB7B4EC1486C227006B980D95702C12FDC7204F0FF10EF6C9FBD18780833D20A2A481040BB880C90A091913F0F118398A8B831329B6D5393C81BE80DF808803F236F467CD8E089B318016CD6AE6A01D0CD442C1C49200D1E9A0BCE280EFF744B47E883E7FBEDE02D8A31898B30016938C1468603F53B1050B61DF21C1C0C271C8D896464F6ADE5480D5058A519CE6F40539B609E1C68C5E9B0E26B780CFD024745FC6EB8B2835027E9D917E974D29A83120C77FE220952314E9D3BAF58E533E0199A5F1018A9F1014E1608C2AF7478B7C2004DC102F912C2CE64F0220D8D8E91858B9E4C03AE0ED3EC29171DB108319B270C03C934E099CA62287B1CE71A830C0A9008385A754AA0'
	$sHexData &= '4049B108C730E0B12D7A80060074BA81C179FD5597B9AA1145483E1A4C90EB05D576764395807B0D4EAA2FFD2A05F0A610390075136C582A05F005DA040F94C0FF45249F4ED5DFDCC31D3C3F18E8545589E2F21AF14B8BFBBA78785E6824918D3650630D354EBC6C90B948937340774E1C092796500CEEC84F9D0C9269725F10CB7407F0D248447C8C89394C963807F8C9D6AE31B0B7D00BD2BE549B28667CE3D67A087C0E1C742005D020DC41B00160D070080F19DAD50EF081E42D954BF3F06BD7EE53FA1312BD3818209E58CF40E2B83CAB88C6FAB070AF529517C33F404CF96982BDE0923050C9F30F7F8E7A5A79C2134FDC2AD5B1E789EB10C03AEBE4C728B95D17C3B9D06F383C41B829F0F2E16F5122C6668D6BC747044788F597982EE8CDB2D0DB244F180620CD5C28D7F6FC7B06380E484889475878068788B31FE03DE58754680670AF89B77889BDBEA8956057898FDC88876F39453BDB18B853376A20C0B44703642F12F31224AF076FA08C9FA54989BF8067BA2F3601764F786416CC9055166D0D99AE8721D1484ED372D550D7109C71D75832D261E780407A54A5CC54CE956B6EAD30555FDABF41BAFF75237CD7081DFB201FCA90FB37E0E1D34E25D3EBE28960F50F1841B08D15B8F5E47408F1EE0607F2380B3FEC9056AEEBAF59EBF8EBF67F1820577A43E902F2805E836E0D'
	$sHexData &= '70F68FB28015212F00BC1AA7605F24527F0425D483FBB7CCE1028608BB10BF2E222A8D9E4C07337D48401F5D1C5778B9400528AB33F653EF286C0C70383C464D906006C31A6AC1C85C21BEA7B05DD8C4759D20BFA5172E39EB2C4F97100109F148CF89773825BBD24D3FCBEB7EB528CEAC68B8D602674808121208343C7D70C16E5808F172C748530AF99EF6E7D3B51104D778789B21F04841ABE14056CAEB0E4472FCA12EB4D70723EC4C8D3929D56C1B025B20067328E38102B745052EDC025B10067318577E8FBC7434F0261A8424E023F1C7084C40BC64F1415C0CBC517ED8FCA88D53D8A75B2C3C610B4D017AA35A07B18E2D6838B7474EC14C58102D0FD3142C2FE03BFB751D56C7167668906A8C24A560482766B082EBD0221E7778335E96AB6DE9BB0C21C07CE596D6ADF85340868423E9E9326096C6A8C08473ACF291357038701320AD169C346290308C84242ECF0122ED013094B238BC847A20B38C443B1AE679713AC24174381B701D1C7252123890CBC18DBB5A41D80F271BC5465D0C8C01738F159667A4F018205F98ECD41450545A182B8002753FF134709A7000B570A70415E128E3669F0A76600B3C7B086390037FB0C5530ECBC98B465E047561807BAFB54915D4750456104F83C0FF6BF03A007425E88D41B101450FB6C1C7C435E8A4D17E8B35D8E1036358116CBEF50E'
	$sHexData &= '3644D588FE410E4D0EF808F002B361245C10EA8D54A55A4F5DD8C270206F2D79DDCE824DDF6C3D280C9F40F07C210C8B4868700E2586B50A817F030F3E3578F904201C0858253FC17001AD9621B817162B41E148006438F4FF6008C33F102ED54C357F186B21C02147AA84906316880D2ADB6EEBD94C4B800666F2F6EA5280AA4DE2D0CE4A0E44EB009B9598E2AD4C3438CEA7A72B2BF039D8BF62B62B7C00146E082E385075385D431C010B11C3D587455436C06D2B8E3F134C4437C0E9A9C84125C511464F73934C2DA077C3EDB319B9908890306B3898BAE4A831DF40B2EA118609062030A728AB18F7F04B18C3DC494E9CA8434618F35B08D70616472A0367883C395E10DF147DA0530BE109302D8157FAAE01A52E3954691B7830012B2A29F8093CD97F536509151A4720C931D040E5183230F9CF127258098B511883EA01742A46A0B9521D1083FA038C2727DD79F0770F4F270FD41A0E291F58125BB239A1032B4C09DD01C8AB40587FAC52059295174A2873640E202742BF5D52FE2765C14856DF619A9D5360BAAA80FF300E4AD7F081ED83E92849093B0822BBE62D83F91F32B2CABF203A331BD83C7F7753799FED18EB33B21D37EB19100E101C0B9EE7E55207CC515483F0F612B37210061875837080151C487F26B710527F01FCB41BF425D07ACAC0D1F949E60F8CFD5BD115242E'
	$sHexData &= '55448C41A683E0DF2D41003A0095B801176C1A7305CA04C366BA5CDEFFFDDD74070D2F757041C1E81066047C6A440FB7510642069C5D33F240FAFA59397413EFAE79028030F9470D2F74402441CA956FB002994B043F741AE0EDA6B32E70333F664439F4480F44C8DDC17E6708A4037CF675E8982E83769D7221DE0ABAD8110038B314C0C274FF3CFAF9BB42A2137AF7BD26023A6C46A90743C31FFE6CFF16061B379736C496DA15C2BF11FE0776E41AEB0308DFBABCD0C104FDAC07C264767D85016FAC304F8D3C3690CF4880347C1A94487C3386031929D8E023487911415E4323ED6FDA761D2C7B18B63B1BEB394B8D141F84CB051DC04F302DAC4C3EE354DF7A74F009A030C101103F4D3F84255C500EB86D8CD9B7ADB9E93774DB1E43100CE74D8D2C4353B4205B6B182790D59E2C270ECCB75BB343412C38F825F714F0492BF676B782F366FEDCE074553004C64D2BFC32F07DBB8ED424FF742CFE4D0F42C75C057C7005FF2700493BF7720276272F7FA482834D7A664710393370606A4A5484F50AEC0602EF5F79EB06752B483973100F8246E2572590A1A810076C0B8EE1FFCE7E66890471EB6A0E901473EB5F4D3BFD7508E5B2C3A678FE54EB7D45FE478DF8A4C0411A84531838730AF6B48B26F6019E43FA9C69DA95BE4613C7044A5C341A16D34B149CC07D0E41B90F516C0CFCEE'
	$sHexData &= 'C2BECF17DAD648D1FD488BC52653BC0E416D61DA9E1EC3B80EAC1CDA4A891147C43CFF0DC41F1F5441556F8BEC2280E9F3272F2F0FF04C8BFA4C55A83EC0375E7449104A1E7D4210C1990DAFA37A0AC74270ECC2FCF1433A7A35D74933034D268462B0AE344B489582C7DB1E106BD06C7DE006E858D6BADFE17517FABEE041B8894CE6E890A2A978F17DD0EB1DA3C39CF8D399D9830214D076502C283C5E10DD84995820880E07613BBF1D88D0180450488D55D00841BD5453A1FA4B3BD014643A04A4AF59E8442A7E022475ED2AACF1FDB74DB8F8C8B942B54CEA7A6FD8A8B8B8C0A8C26DEA9A708B924E563136F9D5C2B68C2BD188E2F04518488B41B570DEBCC84883E9F91F776C33E02AF00E895DC25D3C83C72320DAFE8EC84532E40C75A00EE330F8C4B133BFAF7341B401ADBCED3638BDD64CD0732110420174E0C2A5C316CF2A12510FDDF6F1049D458BCD15F4C602B0484CA0CFBFB9BBDCBF0461ED07CC7102EBDB48CE4138FA352BC3437974737E88FE741E945FBEE1FF073B0F094883C714AA0BB8FE442B942A9A001B0E32A8A45DC8F012330E3EB81439E083C310F16E0577A8425CEB5B4C94A80645A048D3F73E2F83B889FC4D3D3C987B8743064C045CF800742B97C0409AE6153B7B61B03CE69E7D885CDE6C676F86998532EEEB04AB7DB8DA2DD23630484D50A6D134A9F4D5'
	$sHexData &= '120C5C9F8070D23075AFF48D0F83FB01752585A158D2DC7029E44F6A8F78F36D80F3FB95EB7654027555284D3E7A6D764344D474287E82C37510C1B055E23A8EE03D66C2A9FFED5AE093743FEB304584E475A02D1E842A4E283202870A41DA29781F1E895A10830D9ABF11198D4610C3B733DC94263CA871A084AD3E4C8569C6CFEB0AC314FB3608820B0E48C1FFFC2A722D3C166C1A927875262C41E866C0451E14701C2F49C3B9642BDE74DAEB457440ABECF0F8BA5241BEADF647084D4A91340AD12E191117C2D2F1A6A23451EB12FCCE48075F17B417FF792455B3A09CB4127DE016A4036E4DFC07153AEEB6C5B33549B50F106476070E4DE07E59C3C14F1C755CC7D8788B660320C3D416D0F13D4B7B1F2920CB45B8A2C8DFA48B687883BC0AC7C4F0A73E72B9D080415D415C4E00E9D0A6AF7F62230D3068490C5325279C20B7C4706EC22EB42050A07818D04D8B30B4871078F65083376F0EDD1A551C2CE6C56EE0FE48667B204DF87C14418B06B99248222FDB01C765122FF084113EA4104E81AFABC2AF3875568B07374CAB88F60E35C409CAF44803F104F2E04E352E9794100E10822358333A035FD7071E9CD10AD693A4E002D761EB23D9B49112442FFE237C561E2FA5749FEBA1D415EC028F4CD8AF12E8E323F054C58CB5BAEF5DA34AE62FC44866C7015CBCC1C784A5D9024D3E'
	$sHexData &= '38D64D059B0967303AE8571D84169C06BFFE355C90330327672A602596BC902568C585081445DFE9B3B607E9843FF1013D71E9AD069F471FC44B27601357F1418904C615FB4C279FB2421F18DC90C6D355AF46840D1F8028544E389E4640983CEF7DD34E282A48265860E742A3411646481D1047D0A1F39AC0D84A68048E88AF189FB7CBF78D4E683946839BD7C6DB48FA500B0704297F5032813C585E85011098600C2FCC28D79F4865418EEC42465846602A8D00FB0DDB660F6F0D4E5C72074030E405672C24AF3846DF72304291ADE3AEB53206980EC3E0EACE090D398C240D8C0871E9C9E30410C8E8C9308A0FD94C2126E4C25E18903B59D570F291062F02F11ED84C83AE4FC0764DE788C7ABB855440E19570E209CE8C0F9FA3E6588564F35FCD44490310322BC315EC3A2D7220EDF6CB49FAFAAD758D6880F121C90567E0B19A98C06C646BD840591EF2677083141DD32E9C1BE046745E85E44F80FE26B460304551FEC3957187313740699059A5F100B6F1CEF30DAD3D25FC67E174F41B88CCDFC8003850346295853693AAC0F270F1F33C876C6DB2AFF00742CDA64D44590E7F21CEF07EFFFA5520CD7CC125724BFEA18CF470FB007A604012200EB1728663F6914C70C41B12277B2BC850CD2611E07EF5B8EAC184645FF450788AB90435E00A6270F3C52D202DF2F818EB5A206BFDF'
	$sHexData &= '5A70B583718B915F0D34158531363D4BFA811BF9CC4378839DC343680096C4DB7B1F3E2811058ECE845E25971A68FF1531183111CDFA57D68809D8A5312AD31F435030384D6053E4E05DC29041118117F004542F623CB499B0000A10B4319B89028CD13A30A89D431801FB200136FC904C10D4156045595FC965B369AC6037DC92CC60DAA854611FA95161D49F675441AF47292BAFA9F5160653A28CECD451988CFC4F53CF7BC003624F28291A4848E605CE216834374C058755ACC7C1FEE200207880250FC9EF350ADEC61CE7BC024F385EF5ED389C7CB72DF99BCBBFAEBFE00B64648BA4324830D6C9209530504850FC045C2A41BE2D4301C7037197DAC0671EECFF8F5004528B491C8BC183E0033C03756A24BF12B622ABB96D448B4210416F3FC0DFB9FF1241F6C001B4C9894B040CBAE00A73C8006FB72534143D0C30A0750AB8FDA320B37728403A3D03210AAC0FFBA47D72107424033DB80213F6C10224DC9B8D74944A10EB06C8C0075EECB37A38684A203C09B8710F6C29CB01FF38FA7C20C26268458B5439A801AD070916352D2B1CDA4718DC4E3481F92FBC3F52160382FF824163C01083C820C1E804EB35EF2B307DD35A407711742270101D086E6EB72A16182032EB0D1C7B740E2674092815F06F6A740433C0EB05B8618AA1D7C4066140B58C306EE05BDF994590278289CEFF'
	$sHexData &= '02B00858ED44F9A68BEA88E12504821C4EE109FC060E3CBEC96C4341A4483B4310C2BD706F201270CCEBC20AA99FF834300902E030A338E9F93F3F8C7878D272905DBF1869AA6D31A8E27B1BE0782756722DFE8CDE7F0755E3A604367F66837F2C2E75299A1305FA0FB7472E66C40D3C1C1A9B0197F02A301AC380FF5F696A4C74D2EB4A418BDEEB4583FB05750DA003DD5094C5863ABB12EA33B24A6AF09E2E4D20B8E00078302AE3281938462C9B61C127005B84B245CB282FE83C180652BEDC02070B8426B3A5D62E085102FF2EC18B02F156E416A434240A14C7411C3B1ED911D6F7963E7520CE1C48981848DBC1E10C4220C84E6FB0D144820ED6C7401C2EF50813E2C18D783A5A2CE9C9DDB90616DDDA38114048C34A28AFE828C2C5DAB420DF7718B91F6B6DD4B6DE9C145B25E2E76ABDA040DEDF06BD992EC87509411FB28D42FE418905E56E290ECCDB85152E5046E44980D1FA5233DE0A61DA72925710D9DAF3342A4F0FCF512D1A5162998AF92821E4337A6A41907F20C4C64A2C1F0B48584DF209FD02F144826A8EFB6F45FE14A5120E1F078ADE9824802702027054F0461E4999456933EDB076F747559B7E0E481C046E721F49C70611E5E0110F9EE3860E3AF47E52816EEB68BA39268BFB201ACA3817C10C3D6B389530690E464C601FC48F8C8D3C4866137E5B0DFCC27230CA'
	$sHexData &= '66F3ABBFA4891CC0F7418980A714967DA02BD4C0B0DCFF983EA70538A404D61618CD453DE1F4F800F7FF443D663F28FA6EEB20A603995385ED0FBD8A1FB695C3BD4984163BD7FCAC18051D96169F26EB5226B2D61480CACFFC4E423E12A13038B7C758C8772F77C2A02730F8630688AD0E3F38D4E089829F7F0450EB2E70E0810EE8287F41102C2800C08306440C530E29B8E0FCB42BE902FE3EB0C01FDABC3E06FA457ABB446AD909934A5E883BD72197361920DB5A1C7B9506A0BA480F42DF913623D421156AFBC67D19169875F0145E0F76883EE98C3D2D3C132E415FE2590A1AF50639EE4C464EC70789DB3608727166089E42700E2A82256E3E0ADCADA1D7753047FEBAB722C2BE33D478D03D0E9AC104B59E570A34FAFEFFFFFF51A770FE8AFE897E10EB152A0EB64C8D053E823C92CE7552BCBD38C253EA148F3B0B0C79627F03FA027847388209884971231CC100B55F55FC061F0CAA7547FF0FBAE10A7332EEFC842266345919B91EE5026801424304E8C2FB027F8F3F61EB5F7707B90AADDE83E11083C920FD5511FEC1E904EB2E83FA49C9EB05B901C0C79383B5678913B02316AEE094C819ED229C8A60448932C0B3353519231F548C4976D84B3FE3F84C1E891BCEB74110F625E2FA56621D823F24B27514C702034342046111FE87DA28494633FF893ABD9EE4000F3FDFD78518'
	$sHexData &= '7714948CA8E81BEA084D16D09D9A5333BFB4D08AEB08E680D5E6A5F1483441A99EDEA18553003BC6741E9FBB75A202D10949C64C3BA957ABF5F675E74C2A3548202BC6ED7B204F8D067C2573157A46047D08A8D0730404084CDB5973C0812520478BEFAC0128FA804C492698FE3B0FC0B6CDEBCC48082F55008A9A0739469DA00A4E4D1AB0C8CB187A69B11333D0A9345B482BFD5004EE30170F3220D42B601374241F3771B8D8650F20680CAF6B572A07028BE9C8F7D87555F15F6F4870740D8C44E98BCDEB3ADB05CDA2CB44F40F570C47CA8960C4E3C26708990FB60724B80DC5CE184B9D4E535C40489290D29AE5A5C371B52222CDA8C6B0A450CD38F9F1387AA5F6459AD854088C70685A22E1C78985ABDA9841C7433596C8A82B85AEE4706378C179AD5C86358A4B7588E526491F861F84AE431C1EE435A3CE4B8F499A3F1118F7005800DC104C220424E6D96BA634886010837E74E92A95A7C350DC80E257524DE94A874343793E39890B9D8158E7D46CF825922DC2C4DED4B1E90711BE42959E30BB48E04930EC39002A6D7DE062E5FC4264FB128B45F03C144771183E020B2E69CBC0C0F25C93A8D0D70ED8A11AC2E1C26847D75820C7EE44C02F4DCF4DB45DC0D841C91FD53372F21CA0B0B8D03DF2344DFAC2D0A041B5013C07CF9B5F00061020C8A0B88327E4B9A0B0F875400960'
	$sHexData &= 'E8426C3B90FF451D19A64492EE0F43745B30426A387940481F1B522038254A24451212AF83C1895867924FE599795D11381040185BFE782A4820F005114934AC6FA18FF3288F282A5D08D217290940500093431A124D4D7CAEB5C65CAC2C38B748E26F998D48B3381058483975E24C153BF7506F5A80295508FC005DE2C7E045CB505C296F945D3C883BA0DA90DEBB6CDD149D018645D04E78C03AD8A64109531287ECEE2C8412855E1C36D0A3158E7C4F7D7C20D5CE5D89889523DB12DB90875C899D697D904B7C29EC3E734478A8805C89B34C7012A11223DCAAC4809FE5239214B49B3F44C0880E54C01242E01480A7FD2457E287A51E4F4F4585F684414BFC114F41BEA9B8C10A0A846F5A4C3BC1771D5688B70235AEBA429C41EB5538AF7910E98E33634CA74CF20EA472490E008E439C1283A00D99BD4DDC9A8B8331F14166377C4C764BE2ACD236E60612A6590D21F8968859092BDDD74FBF411FD29E3106251E2CC3B2DB3CE20EEECC267C8866CAF8898250D0C7DBAEF88F961153FB41C566E9D8881024924E813E82C19EF55AE434268BC320C4E5791F1AC39F38E8BE415751184532EDC55CD52C783830C5EF03F12023051FE808727CE0083D1D9E30787563B1902F518482D0897199CD0EF2214E9B7D88EB17F030C8E10967759FDB5E7DDE0FCD342C10002D3C5385EF457A6D4CCC'
	$sHexData &= '1D383A08132EC2AB082945C027D02A601558AF79C78E79D948819BC013E03C030B262F1CC496275CD230A8E82885816C062C5001FC44237CA2688919A9417E27F1040B88F041B7D1B0192D027DBE9FC2E157A11F8DAC24A0D88EACBCEC246002500292EE8540865A010F6DF1F74C230538BFFEE444896456F8407C9F1719FF653B4540148A3FC138A1B9508D2A3106C486EEDA06DCC684B3F5A155587729C799D2321FBC6C60877D8CAB9C2DAF42646C0D4027448DAA79B7C7816820EC440D3C6458A4AB8DC873EEC835B6A5C0DA33615EFB0C633EC0F4EB34CF5CB8502AE4BA02EE976E9B3524F7FE018503B954F06EE04803CBB60841FF10C8CEED07AE15F327647507B06783B6340AF047BC277F7B96FB0D3E8B8DD17E050834D4026A20747C85983100E06BF061001B703E103E1339137C182BD60F2BC2EB2E8947E0DC5C047546CB54DE792D79375C7088A86398A4A3D00342C8161843440CB21819A6109EAFAE0E30B8597098A4EB08B92BC8EA6BF0CE29079DF72F6339683B2CA4C7C32FB122A1E08ADA48FFC838F3DC9687F00718F0498D5601DB0A1D5817C41DA14C60F7F0C8F32D02134C64E96F41BF07850207FA7EDC7B3FDB1D0AD30DC7560C15C20FB701A382FA1DFE3BC773350FB636042F0406825C36CC75E64A8D0D527BD80C763B12807C3400F83BFE1BEFD0CE74B8E902EB'
	$sHexData &= 'E5602D4E8D1C76CEABC6B397BA13B26DFF36F8B2ECEC765C4C3BDE740F4983EB82771534DAD2DB3ADE48D1FB9681BD854D56395F41550F8642382C9634830F4D820B2F4C2459D4D01847A03B09942917FB9ECFB47FFFC966833C4A2E751A893505EB3F4E96F268D7E8DC51044E3EB61FF6154019F68215A7A3BB58064A11608E0775847B44BD62FD18CC8DCB915775BDEDB00250020BBA4AC860023B7B1A178BBF3C99156350388397A304CE9D841A13389C96827BC0183237326441978217354B881A6CCB08A4CB7789C5FC018092E69B1A329E99073223FB3B745049DF589A95EA1CA8A421DFDC262BD591FC9EFB013FA4BA9C7589C92801231CED7D79478274A91301023D1E9169B4343BA69849BE56313511F558E1D1F00B0E3E181E0F01D048C1E24C0B3B397B01C21424EB068C0F566C67070CC618BF30E66EDBFB66E2A28BF8CE7D19C60502FA000CFC250FE19138291D77693B3246155032D2701B730CA8FAFF4712563C24063C060F94C232C032C98412246FBFBC5DA87C1F52078BC381E30E23C0C1E8FDFE9AFB05240181FB1556C1A788059CBE880D978D7112D90ABF1F0555570EFCB0F933F6160A2E09BECA7AF5776748980CD898C88B8C80FC689A95C4AFBA03C8FFE1D05C2A4F3E95A7F2109346D23D492B2A4FE57934802BAF229EE75379E6191D2C1044C0B0F354076B0CBA'
	$sHexData &= 'E3FAE0EFC6099C900D7C346686134498BB8168B7126E8059789021B3F0DCB09AB22066BA104269C0704840F7833C11DE2AC1A28E534836C8C92058485881801DC0169F09100CEC28D5489128214732C8C93828380144929123952B28EC3313CFC3BC8907850ADB62AE3483696D984CBBBD4815688F2E8FC6EB1B066CB01067925E691866907D67F4069EE7799E866B625950475058E7793E35BF55D435348B36C1CC7EE95B19FF218B511C85D27E1AE52B08410FAF2C01D0F508F9020EC29E7AD25C2B4D7FF3C458482DD01BCAE28911662ED87634602611412AB0067EC910F06A0C0C87B74A300526A14A9EA814A3A92BFBB2CAB8107FEC800216D23BB96CE6027023915BE20B893888393D58E802055A9A95F4C65E0F60B9D21EC0D3CC6DF6010AC83D023887D47DBD8BD51BD753380C7E5E34C276DB9072FFE7833D0FB4E37EE2265215EF1E0FE1B6AE03A75009587A05E63011B8271D4BE52C81EF29F900DFE702473C393C7567F838DD08F1D793F71538506D330EB19F21342F70988F805F804149FD57EBC6B98007936FC094382916EBB32B8B70F0864001C65D263E8C4504AD00B97F970668893C44D0031B0FCBC9E502C8D1090714380842040FC20D9C9458AC900E528A2287425903E5763CDE15AD280EC605AC82C2AC5CBD750BE961A8334C87C4355EB845C95AB05E963F01B82D4C'
	$sHexData &= '96E602459B2A0D9928480DD66CCA17F774590E8356EF5CE6ADC9B0A6683BA46D761EA035998E98143B128CAC241E32669C6B405211DBD234430233EBB6B7828DBC219D31411649495B33AA0B481E7F0BC4038C564EF90160CA7F02D8307D85BA023A75E72242B227EA31C210820E945FFA814E1090C83805BE20B125C8CF41C6EB05219DC6A11F56C899F54C35D0D73038F97C2A2D1C30DF482885240B6B7C814055531823573637FD80800112E30270127E339441FF440FB6B51709AB31520F175BABD1ADC61C90252570272A9B31C5D12F69705EF5338051E386373E4E8FF4012371A6B4778875348CE26A1EDBE3AF273D1B37020D0C70298D91D3DE8E17F6FE0C6C9A8864112AA98DBCC4A28858651A3E00DED065F8355E6807E13D4267164C60648009A6C0FFB2543C4C3963107447C5308DD2BD9D4D40150047156C8530570965A4483040C883255F0A88F770C725770A8E8008C8323710A622DE793AA9339437F0053964138D1665B4437AE34FB23DECF3017B01560306C8B9F2E88D28F91B8EA784E4F3018D8FD07F4DF88BC124223C02A045C8A9B1D2CC7515A845F054F8C6E9C60868284AC08D77DEDC632B4AA00CD8AC6272C05D098BBC6228537D4CAF146C875270D28890E82F430BCB07F38DF2BF5093224D5EE57086807005B818945B5DC37F55D0CA0F0D80BA20FFE00117394D'
	$sHexData &= 'F6AB3CE069A568538003DC33F6627D2F48BE4083FFBB25232284E49CF2CBBA2A742749B9B3312EE82C0D067F46007F5B8C040A9ED8490FAFD948FFB0E5C914E9C872ED2FB0CD2C36CFE553B004F230C0488CF14B69EE48C871328ADE953F793F01351A34740DB02722CFF38AFF0DACC8F111B4DAC8BF6A75305E1C8ECBA2244823C0B928B29D41DE9D8F621E387FC9413E3A122F9BF1F92270002CC490F74613F1F0116F1C0ED109667E2892CD20D3B1384C084D73486564E83030462023CFC9003AE01836C7464C181DB4F66F545F587D006385E8E628C70AB6063F700855802FA98C3AF195B059BC900B910F2446F5CE705B253EC68DB055360C0DB920D246DB13B21576E0336E9595C9CFFD61A990DB8D4DDEB02E56A1AD3AB019DFB19BCB66982FD017F04875EF890BBA42D9D06F09E0C8C9E001735DC395502AC4BFB60C10561706F2E80900131FAC7C61A5652601EFDCAD827A83C02312F904E24F050FD449C150ED846CA8399E77FCD2691002673D1635FE1803F6BD0DAEC81F1958FA08418BF5BF16A88C3BB96BD76400C014ED1B2F3A6CA97F0EE0FFC6D1C7F715F83FBCE432C0EB02B001887A84C07505E98799986E3456D7895F2DB00AEF1C941B9360B80ED09623DD19480897C2131E10E58E384A7810238D05A456805B006C17E2142EA914924376D752B8A28B59CC3F95815218'
	$sHexData &= '9F100A9642BA27EB2E5F29BDB9209442FF8368A790025A784CC129163CC2E94C3C8568919B743EE2466264DD1587114E6C41F449676A01948911233889381D5684D0BBB0CA46C30591BCB92328F7945D448A2D5B6BCEC9022C88BEACEC3FD2B678900745B800B5036CA7D348671877B4585B566E9C351B2FB40C352EB72B05EA58940CAC85B814B66F7970084D2A29C60C34010EE6C1CE0F2A0159FFED7FE9EE7BA2388DB01703692C1C2D20121E1576D75F3E7CC15B22387A26451703C8A642F260ADBA90B6EB30388C0A83940D3886859A987E230DBBB0493F9D157A1D13628BD405A9907614C6F29154DCA53B42ACC7C0899051835A27DE23ADE7AB572B873AAC34607409294C5CCFEB77E9EB7E01E7D0D8A9F38C4048B94AC993C70768E9195C1E729DDFA81D2D5830232383085854D588032EA18B5E2EEA1BA11B38E5521C5F1AE4A5408C983CD888A10FF085F94839714569E6C0CE4D58A9DC6B1403166409E0C1F2D0063CEB546BAD1794D273833C71E149B94F1F49BA901993FF0B3E4C33C84D0FAFCACA220233715028AE146177964C48998B15A61EB85D4D93F2C9D13415981A95B7D83A2F845230200F8307B8DF230DE213560315F30D82716D584AFD85EB5CE88DBE24704268915871845A841B0DCFC8FB9B7FBCFE2E28DE0A1DCA76D9BEAF967646B6159ED312058C43F0A7FDFF'
	$sHexData &= 'C84823D04C18774D8B04D07DEFB04AB09713669C362C583074C3F7032E3B1D692A72BA6D444217D87554D7C33F90B8C9D7227E801C1850E7B9B748489328145648A7E7EF6AA1F00202DAE8202C333B3149A35129A91E21A52306C4411A79B4B7A18BD02A4B172C8AF0B119DF3658E27026CEA027802F68A1E1905AA948E070C8C3B172C70F578F50254212658F41FC38D24ECC45974C4610DFC689C0AD9934CB33C949BF4CC2231C4366F4923906279F5FCF3E90CF73C648207E123D7612CD919B41F93D68126D077F79B179BE3020DF11826BB9D64212107D1D3537C43476483E39F954AED80DB111CF9F11C8AD38ECF28A11C8F70212D9E2BB6841D779D8C7C47C11E70AE2C0D9DF3316C78C0682D6C8B5133FDF905730059FCBA5AB110CE4809CA31195115E9087145804F21018DD86D8F1578B0CC80F1071201DA5F79DA507524DC72A866CF17E2F0332C8C410B29D1CC88C4B1C420557BB6D2F3A9A20C6EFF2104B3008114DFF508068796715848C65B0749F001EE04FE6593530137F875EA7295F18C5D04D9FE2AF738D45D8DD548D9FB807F555C72ABC965783BC2EAA10DB042C105EF2E2D2C21BCB2A0910F40F550372426C10590501D302E5E60F9316AFB80FAA6C368A45C7F2125241C6461802EDA2EE01EB050D006B440EC6B5701BF292E000562873F0976B269E188CDF0EA0974E'
	$sHexData &= '3C65B98FCD4C16C4E92FCF3D3D27CDCA2A0F7DCFCFD743383DC7014B13D318620C780D8174217B0F030C38307FC37EB79207B2017C6F0F670FC046EBF2590F6FEC7230F2E3835170044F1006C00E5A79C9A7CEAE0E990E14C886E1A240E420DC4A96801CF23E537021668C7A199081E3650E6FDFFFE3BB7C0E0F4C0E2D8DD6E4B9E3E442BFB76BC71FCF8799E717273668172F0139C820DF750EF2C353CFF70D4AFF72E10D86C94B0ECF0DBA0D77E139C071330E71AD0DBB0806930A7ADFFAB7172C2F3807B54F1029292B07DB01F469046683F9A7E0B2E06476510A1551CA24D1D8AB6F903CD04C270D6BD16DBBEFC3652FE91C0D241474088777BCC1B2C2C0EB0BE4070DF8063C07E4780DFA0C4D7F4C9E2CC278E10CCF0C21835C1AFF0CD00D46CA20CD105A0C649E9A69028528300F62249C9A06E9571FCFA5631A5615DAB34033ED2404B90B96B7D002F35500580EE1013FAAE201F94C006E3DDB46A505CFD00285D5F42E8842907507D6F4C4F3CD7425779F74AE11D37F7BEE0BDC0CF2C1811DE60BD91DD80BE487438F737948A0B2AF0BBDE5351D9D0B71880B144275B0466C7FBE9E059C312B28DD8749CE04CDC4917CFAD6D4FEB71638570B808152BC380B71754BA0036DA179D92A800B9DA8CCA11D535EC8AD5335F0E272F0B278DC3A0FC0E8F50AFD79DF0F083BC0770C34BFCC20'
	$sHexData &= 'E4498D50FFD87D7C3A4F0A23CAD8D7298BD8CEAD1D3CEDFFBCC32CC2A634C5004203BFF250A8833C0E0075233D77DE95962E9089040EAC8D22001C6DBE7EBB4023C3AEC18C38B409CB2904691E660771501A3AD22291F2B0D2C1BA510230549538771F891D479BFF05484849E4B968900ADDDE2D320A5279B47210C61903C3DE070A60D36C7E4C23C048F2094EC0495487C26052DFFE09E9690A9E03E009B64E2F6F59934AE54CC94137F9325EB44F089A584717B59775DB7435438BC6815A59930C24735D12D3F6D500C108EB07C50B39A40CF54BDE83D4B08422002BDFEA02703B4AF95BD74A8442ADBFDD4A6BC5D00B5AB9120A98481AD0F1CC021CC8A6EA2F88CC5CDDE5921708109A26051ACD020EE55577050B1C87C4DE42DB7D30CE6C0836354A122015630808053C4BB61DD8A9C88650C825753EF07468241D4303A7489C4CC74B72A4CBAC17810E42C814094CDB6D381849A590174ACF7A4710751CE825571C7D8BDB7DD4F807E30DE90CF0E80A48AD2A5C2D2E92EB2D4CFFF38E50D05CF7F44314071E17A6080B85B45D1F0CF7000A9E9A0FC7BF2446ED8D70454B0676270ECF4A207D12E105BC9083BC1086076C075B07A772908758074E07F80E72699AF9CFC71C1F07CFE5905C0B07070B04C265909707F506C78B58833760BE82447BDE5C92E67DC71ED7100ECFF33CCFDF18E7'
	$sHexData &= '20EF283301D60E0E1EC716737D3A70AF7E583DAB0637AC0DD50038913524DF5DCF65130651DB506506026BA3E20F864E77FED98515AF0849C3DA03D9DA34065D94B219D81F06864BE1506C4D17E7F1055E1CE430CBE60586EBEC3DCD9BA9D4E820BE6EDA125E08275650597D82617122C990796F8226335F084DA7C870A48325D2F26858202B979D77F3C928E5D9200F59396EDB17CA0BCA1404863A4CDBAD4E3367C27ECC2500FC328FB0083EDC19A1930DE0C49C634C2C003A64F247644718B63200F879035F6CFFC878FFC050F8498968F80E48F0A9AD67F3F040E8E8570F5FE0804C54CA8B4F20C01802CD5448521FA0E8E1B7D0EBF840FD48522BCA7D9574276F3BD973663C9AEB4276A09FA6DD36BC30C2F8D876144FED77AAFBADCC73EB1B3ED9189DBFD6FD02EDC333D2962D03FB4C77D977787FAC0948DF12089900988D161668D6166C0A409F26EB4A262E0468651D59AA6E2F3A28DF3003DB3FFA775982351D30DCE8E03EE9F3055A577C1F7C46997ED8C0255FC6C6246070D3012DA9AC20EE822604AAA1DF3CC0E3B6C53902017A6DCA3DF2C7383C901DB7D4F200E4D76DD93C9093BC16C80AC84BD9C11DE0AE6305838CF8940C18172D6E43C9940D82120E74BAABAA05F823660148483BC87343500D5B4F8FBE743A742EB34B6853344F493D024C74060803DB80B655EA010449'
	$sHexData &= '080651EC053E583A8607F518EB0C7F98A91B3A0D84CE5967D0BE22ACA99F807921C0118F5BBD14FD010670E2B51885808CFF4E0E8CC4081F63A818B6CA420DD622B001B80678A8629C2C06B461C580072AE175E3F198A006A84340788D6F3CC6559CA180929766C1D7EA0752091F818F402C45602B4DBA8102013E7882407067914E38EA3C028A702048B3BA83484F04BB4A7B006ADE264D9F017F11037C5FEB28458BDFC2FAAF01B038064066C7454800666BDBBE9A50188955402E481CB84D449E8008B0FB50204488559A58F0692AC2CDF08B104DF64C3E494D247CDEEA28694C10484498E89AB58B628B78D45DA33870E26CDEE40E3045288E6FE6B745404E08145E1089561844C4885620C6D86E85D246210449064628019D782E756ADA90CD50597D3CBB7B37805EB3C3B10F85290EB04080CBF85F7303186BFB2BC1E92184197A1BD10917C2290B80DA60E0B93A1C19EF76349C2078033331FA05D5161C74543D458B87B878B8FC64E8418CB84778A42EB84910741BBAF00AFEE74680FEB0A0E475810BCCEF11B902B505D4A06628EB335C3C819C6045C610DF7D609AC1B77C45E2D301CFCC8B23B880878DBE10D9878217C1742DD1751CC9750A2E140A0191CA60D2092C1423FA809F972E13B05077087BE03017B10307D944C1CA8053D0F074E47EA83878819F05DD84F027CF1B01A8'
	$sHexData &= '4420205A7E742320688E96F28B2A22DEC2444E15E27DBC96E47B0E8F17FFCBD7D8FE0EE98045F05012720C3D0A20265D8134B63EB2050D017A53ED0FC050895D580A5CEC209CC9C81B3A49CBC8C9BB85A050660A3C4717B9E9DBB9F8AA08185A6935CD921014CCAFC3FB6461E6FC2E10BF2E777C16A3086A07045449287F8E1A0822AAAB54700869437C07338D0C8DE381F94AF3E64ABD42D04883EAFA8CD4EB92C1C27A3591973848113BF8C037A016308F084D893E4DD396A3E7260610F8B8FC450D0547B05601688384808F254F9E30D0C4D79DF5D04F2F0703807D19498080B07D2E4E7C3B451B0ECCF8ACC838C232D253BBD30D72405668C259C29A4F5656C04160D2C1E3C1D1C1A1D95C2042C3C12DF0D4C1E09C08F39AF807053E0CA013F85F8EC10A2C96C108615C225F66E589C1B3D124F84728362CB7D875BB0A1F9CA54DC943F84D301AFF7604B9EB2276243842BD96C4C924CE3308676A0C1CE4F2D43A0E1A85445C4198891244148A30262AEE34E207C8AE6C3DB614F4FF8BC17BD8FF26AEC04C63E84C03EA458BF91C7C6A6D2D1C40E03F7AC72C74856F63ABFD0EF6C1EE1814C6CE18FEC1EF08F5F688F88D46FF3CFDA481DA5C6EC60F5B3BEF55DB0E28F7BA5E0EB6C022C01C9FB6236259C6D4B1102CD8D85C40BCAF2B6936C7F890686EC61A28C495D0C6747744C1B2A460'
	$sHexData &= 'FF924C96094E5598AA445C2C49CA0A6C4E754034451C4EC154954AA7BC00646C5F646468D3823CC885FF6C81FF5C38E44A37C87870BD707485D27C874B1E7881FA30D58BD6C1E208BC67701B6D0BD010B71122C0EFD151D1431E3CAEC82589148841E9A8C3B2FFC07A8C56BC5E96C54F64FFC1193C7DBF382F3B8B453B4D20499CC99B563819BE658021B3D465760871803CFA654CED341F3BC4C24C504D4958C802029D4CBE4CF4B0AD051470C8A010F7F12A7C6C2C0E0947D03455E342157ECF5828FE4132F01D94943672144803CAF095DA31F4DD6CE6EB09C64540D35D056C075E105A38010EAC4317B420C809468B330A785BD3BB9E7B22660826223B047B080610660176011B664320ADD3AA0AE8B8B1069900860FB97DC05C8E46006B86A60A141C8EF288502C853B617B4E74174C702BBC0A0382338BCB3109F03A61C643BC2E17B3F1B780808F5D808E52F581878C80895C0C4885A710E2A0733B72A92C045610464646200C0D5670CD1921CB886907085C4438971E2BC264082DC39B392087007984DE84DE9707B2F27BBD75CDFECEAB91873C90BD9CBDDDCE2039C8CB15BD26BD14BD0039057206BDD1BC1502B272CCBCF6CCE30E718804BD5414048C7526DE0BA1D3715511C11641A3608A71635F284420312A1B116BE5BB81181B237F5553796C9ACACA78FD4803A898A4BAD6AA'
	$sHexData &= '019E1002CBF1ED3C0EA35D0AABF80852E204E9853FD21498C018348B4A1885C955CB14FF0895FA8B421C4C15500A009A0C9B665208D1FA02C88298BEB5251AC2D050364549F485F6DAB60BCF0F4181F8631881FE175ADDFE5BF2DF48B815AE47E17A140849F7E2181A1008163CDC6E1F1B0F7FB40641BBB19C47D84D171B7F590F86D94CBEB979027805F0379E49BD3349BEF0452FB023BF437014814933D5B10D5F3320D6164481F833D01900980CC8020390D9B4FB1EC1E006F012CBE9020A767DABE4C83AD94D03C34D8872ABDE6DC74118AA4C745CF38DF7BC83B810E2F95006852541B15C3E1A720A201CC823622BF906CB0B34C2A3CD3495A68FEFCA7C44645EE7851D8B0D4444DAFBB40D253329F2CE65F6940D0C7F2876F4077FCDFB2E2674BD900C44985115FBFC781171BA015F204F808B93879C882FE7CA03CB830C7B1AA65515BC20BF223D41FA1859BECA21972EC884391CA1A4132BC86F7C3387245E0E394DD66DCA46C6836FAD85C8008BD8496E069D8185703DBD33592F9D0E990A7C0B1001308CDA27C261B0A88B95D992376F90155033646801241C4879F9023A6B90019001BC7F2F49B9CAEB41F6C204754E6E60A0E5B9B908C9742540B8B5242F7C78010AC171089524301550E010C438320687CC44BD69C740BCEB063D70389523875C488A70708A0B5D0B97BA85FC4D'
	$sHexData &= '90803C30C3BDB00FB88985DC14F2CDA39525A9EF7D21F78DBC15AF95C0BD83E42EE442BA368D33C1FB7E2BC15B448B818BA31C133D025B9DE5D5841B18EFDE32A8AD1ABBBA1A06D7E65E4C14418955B008FFE66E161C4588552041C6453E85C187D2446755161CE70B0DAB18347831A9229EBE84F77150DAF2B49C46152522D6740C6B154B547D69BACF62302CC8013DACB402E687C75E95FE7DCAA7742C02ABAA91085A2B38601C543212E8645C4D627BCD341DAD9EE92C4C981A6FCA6899CC260FA1616CDD5D5A1E42E04CA178E768F84B608615200D3ACD1C042058D548AE32029563675499399C632C458BEAA72B76E35A00C3A1C3E852760B3A0F09F7034DF8305E8A8F9268064B34A36220EE609B16338E049AC9D1FAECE10706DF3F8E4539C8188DC7E133540FDE38FE48982A7AEB0502507EF180E3034FB218714CBB4AE119F4585FAF60DB4E58011C63C8D8CC04885A4B9AB746F031F841C1EFF851E304C88D615DC82F0348350E7D802CC7557369A8F18CC745CC294DA007A861ABB145C8233821CFF394365A8454D0D080811AB61A1932178DE73969529088888CC7400D5B0F320D38D219DC67FC308BCEC1E1E9C8109505AAFAF211414963C6184C82FF009EF187C6B3443BF0C08CB6786161FA5148FFC241413B438CF08D66331CBE78D301C008C67F44943FD650407EF728C986'
	$sHexData &= '6995D1F845DE75E83DF034FAF279007205F033DBCAEB358DD241EE418BD6AC769D5E5EFFC835F86EE275DB296D1DD5600875F82F7D024E3BD42ECE545CE9331036524CF97508BA07D7D065F319CE6FC845E920C8947C2544515A1338183CA83CE8F79091B06B4B90484F5F6FD3CB397423AD22D71E9064D6D3070D15C8C1E27A199778DE8B689A850652A8B3AB96F00B9DF4983823F47353954F8998C79BB70853A39F5D789C880001A52DD0804A999E9C906FE82DCE948A9688DD0A45452A96FEC7F8D87DA7A15E2520BB48883BC680A7507B1B9003C2785FA68050CE02ADC9783B83243604F2ABEB0300CE867A1648DF78486407436D0BE49D5029D47B0BEE6608061059F420E752A8173FFA78C363C70FCD9145383FC8C6BF79280F14187D63FA34078179F0F5BD990BD29131ED9FDDC89EEF218C34B12EC101593920B7C264B1597939C855B196C2CEB0DFB06673813CCDB04EBFB0D6438282390C95DF31B70CDCA5491A0D458672F2E04678A57EB07843A97C4E79B0A3C0784765EDF3D5588EBF12B5A12E9A7F659078D899EB1E7623482BD89D8F5A11403D2D07F45C9A48402030B0BF51A9A59F1FE04C03E2FC2FF937448BEF440FAFE966418B3484448BF6410674D4D2441D41ED9980011F92D46641EF988001EF08A434C6EFD23487FCB0C7459C949498CDC8850C9CA8A0A032218334'
	$sHexData &= 'A4A8B4C8204D33ACACB0B44843156AFA41F5400F582E472F789A44A6D23CD43BFFFC91CC3838422FCE8B200B8B153BA901F79E0A4BF8553E2448641249D927B4C3502A29B50855744D1BF6C3E409421DA4F9015A50E6504DF673227321D4F3366673080610181B506A2E9B20341620284B4B13EA6F20109498873A8020152D78427DD80F5A741A0AC33AA10EB09278AF414338D4205A43F884C61A12F009E490A65DFFA40CB5EDC4490EB57578DB6E814DB00655104B41224D1218796E0B4F148845209921C843792D11A9883417905380540795B50359F9BC95B5B0ACAABC431EC8CB33BEE0ACD1AC1CE4E5C812BE4AAC5BAC49ACAA6139903BAC61007222C007FAABB08D64E5F5AB1FBC28D2C0629B7C9A084810C11604C14F688C74E31DA6327ED6686EC7458876EB040BBAC5163E4A6986A7363003252B7F9446BC24200348036A60CCC65B851F815403AF688818404CE6A4F0D4A902D918500B63407420C65F48987D38C809448879280B2FA5ACB49CACA8BA20D0ED66BC4170B9A79CA4637DFC753B04CE36EB1E4C6375782FAAEA2C1A6430232D54B710F7174195287262C0190BFABF9EA41858C80C364EFC0194050E4E98069859CB1103054EBF11984C316095087C8412983770020F3B8FF9852311B8843D92E80420F46700AEA498A0E89319BF31A0522A1C385337151E163BE1C0AC'
	$sHexData &= '85D3CCAED03845E0B8D8C2073F8A1AF6FAF748E0E1F505470A818F8C98F56CA00DBFADCACA25433801E15A59D36CF398D8C0C81B2B07E4B3F281A97BB904BBB1A92F073920A2A9E3BA1BA9AF90272B2CA91AA90CA94DD33423D0D0C8C8C0E4E48134C0D7A8C05652F99CD2A8FCB8C02A61301E83E4448018A81444EA8DC34C0172D201C0D0C0801B42139F03D58B43081DC04893E8C8D0FF87C1A56066804CF4615D380503F20AC898F1981A77CD730AB075EB3E76F4740AAC974A4AC0F0762CD62AA9F60BE7ADCCEB1D34F172B5D019C20D714B8D1C79C96D2183BEE19EF1C94446586D0CB755B00CBD963E548434D8981235175223E006FFDE8118814DE1E766FF53C810C89B56404C4CD24C2158443C4372C8F33C4044501CF23C2748484C503CCFC9855C545458462021875CF7C810E200C00160D33495E69A3C3030989860F535CF3434644B1CAD25AF3A26E270B731BB855C1A81F4C8C084CC085C1AEF6FAE1CAD9B32C9885D418491ACAC39310A680A8156880B04534243D84360D31149DC884B3AA2FA72FDE18843214488CAE80131010C041EEB21B1EC2DB68B6D8092741439047C7B289A844F166F18C7620A8BE4403FB69D842D3CC8B9CD2396060F041448C50184B3610240F907DF76BDD8493BCE74151C02894210B811CF4615F083410814A648615B78EF4FA101A7EC372D500B'
	$sHexData &= '9F579F85A347B4394E095739D0B5B136F34D7023BF05A0A302AE435AE6A2EE2581DF9605E70CA0B8EDEE288905DE10E20DCF0C6E66135DB136367ED05E76B681BBDD79DA17FCF0FF2504E44DC404A51F637EFA3990950736E9A2ABB334E4902119150412C1978373154A444BD075262EBD1B39C8379EFEDE5646272F60D5F95515180A7E29608087610CEDD93BD87F69812478E80C9B1C5E8AECE772F02A3EEB0232DB4C42344204161C7AFD3500B8945F99BB7000CCE92CE660A1EAF80027076E894DF7707F30B19AC47E9CBB280A10DC8D5DEA45E794EF85C936BF09371D16D79DDC84F28B5DD78B7DDBF606345A74DF40E3442BFB049F0F5FF4C7895DAFCEB3EB3CB94C5F4CB94D5F2AED761486AF8BD81E3C4EB38BF8A7F3AA342C4F5D1B1E1780676F10C08D4E010AA6D3F412B9AE7A770B428D988975E74EB29AED0AED1C143A3E0183F82E0D4703AA59411594EF54EE32182C5E19DFB00CCA46403B453BE2895245F726189E0E66C784AF7C3A4F60EF056EE7413BD42CB780A424C1EE83EF6E4ABF440341B344395DBBC8CF8B183EDC132CE837C88AD33BFC1ABF538B8926E1469828CA3BF05780E66B4CC32CC749CA0ECE967CC3BC9101CB48EF98CB15B8B6350CE30C5DCF3EF23720C6A6C55FF50CD845547E668B1B412BDDC6E27C2C7CC481FCD27F540D7F4CC0232A7B1DB7B3F0CA'
	$sHexData &= '3C0645DA5E6D96AF204CB6BEA72CA1C3954E2DF8D33E2197C8BC33207610891CA0775F22455067AF078104D220C6D02DFED10180D36A30E8205E74909A00DECCEC2BA72A2A52F894D38EAF564642099C836E719736E8CC27D71E95F1AE01773AF6A70FF8AEFB2D5BDB750EB10BA7692ECFF086A0A91C883A811A0FAB1F53552571E4C80488DD9E9300B15B0A6AF94143B883FA08DA57440FE9DC12E30E9424D842344713192E1CC74252CBF81FCE42349EBC160640886E28EB7C68AD8F910B00963CD764DA4561371637403862745B6B2424CE4AEFFE10261312F9441E8208E9857486853C04116C89E6F3E63D048A0E612140886CBEF13E0819DBCDAC5332249FD9C10A0518C901FF22D64D606F37D0602BBAE104E84CEBD9DC82236DC18874A3DE10C98B4904E34DA7E17A18890E802004073BC169135A5B4830442C1C480F3B6DB82013A02136B424905BC34A0A0F7088750464AEA4AC70EB3BBDEF12318BD50DE18B0ECA0812D2C70424002B7C3D4CE104A404C0244C04F60ADD3BD52508448B4BF238F122EB377DA3941333F64C6525A22D3A5A10074D15101BDF5EC6DA0738F80A1C823C26604103C51E4931E2F66880987DCA1F8B4EB0FD438B04834EDA8500437409DCC1E9BF9188FF183BCD7C50468B0C0B35B6D02BD18BCA8810ADFAF7D90F48CA72CA7F7C38D85B138646080AD0C1'
	$sHexData &= 'EA083544F9B9F68C965C413A107336C2F7D81E57FDA6C2C27F2841FFC010C77C8E8800A7C1878B738B0DF817768E3BF744A31CB00198C0A048D303344B5D0A0CB8D338628715BB442C774378E9E23C7920A653A6531CA9B8A218527A4D0C437202A6535AD8A4D2620C2071F050CFC51D95BC24F86E4005355814A0D6EBE2BE73867E70C1004B5AE2F8D7C745F2A2E2F58BD86E556C4D00FF565C819ED6226DE630400FBEC77FF3B7FFC5F96EE8C4C1796ED610EFC0C4E25100E870D200C4E35518B750C4DBED010A6D18D2015A6D400832CE4AC55520D0E4A64C176610F0B238BED04D668A86C01841C3C7C543E0042E972FC0815289542C80347E7CBF8357225E68C5D9EFE40A0C8B94DD28E2FF7414C5FD72D118C5ED66E04017256CBBDD5F2C247241C5F5DB1D2212F66F048B40BF6D7F77DB151519EDD8CBC5E5D8C21EEBC940D8D5C5DDDFDA0F89447467DBB248109CE00C525C540868281E5789148D480706BE53DCF75ED80F8D3A334C6690FE22A6701824504C2303AA51E978FBC74ECE7C5145C8194138EEF8C0CB3BCF7F6FC941CB608F8FB73454BDD26F848EE71837C8FE7C488B05E66197136A8B53F6B8182D180C7C453BFCE3A4FDE9EB63604CE93D7D3C2374697A20E9C561603CD5D3FD226067E4080140FAE096EFF8BCE1F7AA2845103F16D6B0205A180852068CDA033BE9CC'
	$sHexData &= 'F8CE411831423206BB22CD73D230081C7EBC24578745CA70011C780175FF7E1A891EF9F6EB3CBEF67E320A086B9420EA7CD60B08E3885ED2F43A1930EDBF1B0F4CE6ED62F27D487CE062D20A7A3E0D7EFFEE62F17E487F65010D6D02758E2141D6783AEC5674DCC3C5F7B6D0C18202803C88462918286C99810F2603C373322488D30CEC76F4FB0F75A885BF497F0F6EDB63C15862D1B46F1487827422F3CE62F1AD17DB3772D21862F30C1FC90493C9DB6F2BA9C685A8695B0484446D484D13B9DDDBDC51DBD417180A65C2FFED087423EBC27875483ECD06C4E1FB93213409BBD9B52E6638295F98312AD4D1BFECB290448580BFC38ED0E0BA0F3BD3E24C0BC2D1C07F870017463A75E649F7D04C854807C3A2C397DC0B6FD0610BCC6FF956CA8DA9D4A340D434C35BA7082B56C26E369C247C3C0500BD1D2DF5D4020084BFA91A8B1252079D02C11DF71CFFFF48551A9E01C5C4A17E10392FFFE4941E1A05CE4CDCCA8BC6C1E0DB9FAFAF1840C024C07D18C0012E66E1EE7CCF5745C9EC4F5C0D7F83ADAEDED5042160DA70D13F4E7BBA7A5D46D9AF41D0C6D070CA55DB756900D0A944CA74C158703082C0DDB408611E34553DB2A8AC101A26CB79E5C1600D245F7EB4C1814A0A74307F728ADFD48144C2FA183BC67C5447F22DF0EDA1043E04CA2BC88BC1D66CAD12995A307F743AB05355'
	$sHexData &= 'A4088ECA3D7F67AA57ABB27BAA10757F4B38D15ED483C1107C878016C5476188442D02E65D11A2143DBD2AA424500EC8563B533B28A9A76C080169D35A4A01FA044C1594D07AAEAC44041512F845C639010F9CC0C33FD281EA2B13EF0001A330C8C623AA95F0A91D4202A2F0DAC680392FDC64CD8C0F0F11B87A2C89849F115068531F9C4E010F75AD9EACF1404E2970A20CB7046908064CB10C82CE816309306430E4697F39847DB0E8181C1C89AF2077BF0651283D7CD9D1C20CE1285D8EA440612D2C74E2326029481308604F503E7AF21E58E800BB23D119799FA7DA983FEB17260E0C70CDDCE73BBF04A5DED1449348380D6D683633779D16BCAF244181661CF4638E1DFF15292A0C360A2841BDA7E0C5E20BDEDA5E47E8FA8A2B461CFFC00620D00602F7F504552C68E1A495AFD283FD2086509794CA13E0EBC25F683D178F33DB458D6500C10FDD74B4DD66900CDC78FB7632CFB0DC750B8B46A9EB07623CACA11368A0EA4467E98B5A9E847408B23B0A0A01581948014BE5A43260506870AD910B0A188B78F5E0102925A1FC174C6583A385AE2AC0433B9450EFCEED741C220049C77F02C64208EA2DE1351C8445767B1B380AB7A2BE9707F26C215234257C51B068B481533E0F45205EA9D1FFC3EB0C1FF72915B81BF4DD64ACEAD22B44869C538EC0F7D1C1FDD76949BEB76DDBB605'
	$sHexData &= '6A3307096363C4E5787745C0AEDC0BFC17B118807BC6145601DA0D8075B1847B51DA24095ED454C249848C3FB2E9A26B941F1723D5ECE3E0DC904C9CA845514DD424900C40499298C10368951056084556EC0E89918708A07706C32A103C00E4AD5053C640024B9581B6C6CE81B5F77C985FE729A7CEA70E4C8C85D1EC41AD52C6DC18FCD55C9C63D87172B086A2E7FE37610C1F4889AF01491352F1B00A744B030032236978C9250DEB0AFD8E347C06158475BDF90911C2DB4076DEBC34B03BE14BD94883EB2A802369FBCB152A0A8F541BD735701AB841EA6E5EEEE161E28568E9196C3A53D398FDA6FDAD3B46B44342E938887F2EDDCEC2E5A91818D56AF4F52BA07F052ACD0288CFCE755889CED4CA238B087D5F9F143F5F4C08881472E820361398F93E8D17A66CEB4488E4CC3BA7DEA2598C897972F713BE3F7CA3FCEB050CA24CB1B0C238549ED5562703CB791850CF469240F0448E8442A91CC0F30E1C4C0E4B52C4B580F351DF6345D284613B3B561074652101D543BE1088354FF34C440C48084C0C875CD6DD86748092E43CDDD3BD341811602023682811F7FCA970304C896CC71AB9547F73776C6483460838B0373006E9173DA49C90F7362F561B2E74264C092B3BD83B0CB8032F5CFFC70D3BF825F2926AD260C65AF00001692E07003840748338C5B904052C1C7F124A89316C'
	$sHexData &= '7FC948C7069C8D28702DE2393A41D96E40305F84AFDDAB2A046A646A10298B313BAF08FCC2710434B31360D9E30538C661285FCC761127E828C864395183D28D3FA7B308FF00C648D90170D6CAB25B0BA6074251107C86303A11BE7402884480CE682CDC79E7B87A09853F4468A06100B8BDCF3830411024104944709C7D1B8F88547745B07C7BCDB6820EB2F0F2DE432003F03CEF84C7DE1628C63C33E144988E888232CDF120CE44A679489567E4C76E775049495C1919B93C383C404464904306485058B05206196068B29CE44795186664504938491C1CCB805034BFA563911CCF822B4864F905F52A7EB1ACFFC63B732E8C148CFD2AA6C1C2676B9C28C045252DBE1DCC19F861A4716410C3A4004B7E7F18F60911C4908B0C42A14184EC13BCB60E10DFE8F4850BF96CC720C54408409848880EE37CF79078D80E9868980C2BE9C9578C700A8956C2373F79D0C833DB8BF3D04A173534BA5078302C840F4C195C03F97CC8698540080DD9014008F359F95C2FDD015008D808C39D0CC623812A1083903897D9583D40BCED290FF28A400849E0012533CBEC0A898B3C7F58CEDFF29D34F2C5DCB5899DC005A988D30AC841A8887F451CE04D8D7EA2174C85D2490F49723BA888CF458B6085FC05B4C18FADFC08A040E605210E1CF8EB5E5B04503D497854240F7ABFDBBAB6862FC160467DFC'
	$sHexData &= '140D0C481CC1F344B829EF6B3C1808115D8404E06658F3D93EB40BA8042470446CD8496E929BA7CC0405D5DF0009CACAA6DD15B722F890803FA0DF22CFB3F9880AD7233C465C0888776FF322BDB42FCA8582624D6906CE3D861CC7A2285F1807E3C8889DF2297807C816C60F318807299007702A9BB322310A000604F691A030B255B9B71470174DC04E20243964B832C5C37D132B298B9D1CA4C036DFEDBA5DBC1185080A52A5C00D2BE38B8DC48833017F102BC8EB4245299EE75B20BC24DCB94D1F190AE87C6769B94E21089608789FDFB94F5DFA9DC88B410F57C0BFDE23448D1408458D414A08A610684AF30F9AC041391C244D363BDD301170C20C82945D6A83C1F1B0418D424870684E1447861B98720E7891560CAC8182C28B108CAA4028B05F52A0413BCB7E0508BC06BA05C49DFC1F8B5E1CCC3BDA2BDA02E1FF55DA1C443BD97D6D3BD37D69FC4013DADF2BC3AC8BC32BC20C9034203BD00770D06C44CCC2224A626443D34E8D50A16684D38EBCD01CD8E9AC35852C07D31C1ECC95E110A8E71EC0930006CE954C80A5D3C026F768A12B44C019661C27745341DD9DC8981918BAFC70CA7EEF1C20D85A56FC90E3F4F131957DD74C9033C8837E6071EDBCBD47E701D01C4F6C9B3C79977DBC9AF2CD1D906F10E4C36E538239E432ED9835A8676D24875C26A04C3A10E490CB983107'
	$sHexData &= '873CE4DB9A56203715D3CC177792417E35FA9998564C37B95C72C8DE5C50C23201410E28B84F2C1FAEBB3898517C56574BCE0E9055E5CBA84E437281BC585810985C2378813CD85BCB411176C883163F223C353F0EE9529913103864E6FC344DD346E804A8A09888583A85D4902DBC1960A580A75028716BE0018B47D264E6FE0A45AE7921F50C95197F83ED0A397AAC90770BD5CB578580281E780733FC41E40AA4918F32FBB2E87C2A8B3DFC979C4DCB811CE404ED9723CBAF0944D84C1D8028802C6BF76029354051E41C13C763CADFD7D72CCA2593BC2D2FF4C9502B8E8BE3C4C91754CBBC424EC87FC90C2432C963425E81F5C8D7B118809CC3DAC8CA10608F052AA7BE25438A032457FA7AC9A50339C8091A9650C9EA5EB6B0E11D78359D1D560C845C6C6F10B03800B4D696555ED888613E40DEC745077971BD65C8DB8C952AC8240739247095EEC7472007395495BAC777C1F50A8EC70F2AC7DB0EF2425EEDC6B494CEC6E4740D404B963CF6155E2B11AE3C240075190CC4D33CCD7E1412180B1CB60DF11EC3C01F80BD0A19919CB76F9C592CCB26448B8DF00C55C1009EA4B885A575DE504C8D45B83A58E16C6DA240384695F4EC0A07085FA9463D0700EFFB1DB159B4DB17A95645B2453BC77E0508C17E031AC1D6EC0A5F4C1C3BD7E4D21E18D22E2BC87EA1F81EC0173BFA0FA5B8'
	$sHexData &= '1FA2FD0C90452BC744C668BAD789956C0E046EB399C6F80685700AA223DB17F04C63C2649530777CE583F0EEE5062C2C16358ED7C6B8EF8B8581AF9EBE4F2BB351E8EE8C83BD37007E59DFB050C37D53A9D28BE1028D043B723778B017F003C78785D8D714E5B33684425127507B430AED0C886AC119FFC33B9DB3FA866CB07CB14839E9015F9CE0F4840A400DBD9C71C674177C72008A55E07C46AEDD83F1B094BCE48626CCF823B5226460EB0B33DB23EF7D887E04183056A0B3578CF7AD3CFD1F36B00C119677730154027539E43B5B8108DF56830121FC250877AC474AC1800110D591B6A5F121F2B837BCED9E1F009E1D00391CBDA635EB318A7675C4C735380A03CF9CB722373402C0A4FBFF490F44DF8F0766D7DCF7722BCF65952973DC5E6C7A64DA0E47741339A01A0C4937B46DCD2C09207B0110FF725A0053DA439D09D0A83638454CF8E0C39095C36F270F900DE009C4A990DF0F8E1D90C3E8C41D95532B72C8F783956FC1FA0571781E3CE47B0AD0483744E2C225C72E068F2837A698F090CB500C72C2A3A7A0C4292DC26FE0C1211F8201938D15A3C162913C396A8F84C1C787493377D59B01438C903143BF494C22D0E711A109F9B42E5ECF52080708A1B993798B185541833E37B802BE1A9611C49E0268E3111D03A9111A6C7CAAA5B71D9986DCC49230152D2039C374F83A'
	$sHexData &= '11AE9D6A7660DF47CD19B0D4529A19A1354D23391B3CB5E97E2A8044257C20AAEB3A0A463786C0467E45C8C3A03D1C2CA31060869F8C5C6070458B664C18590ABBF5FC020F9D70C7456F91069874A0083D0766BDD8A49C7828CBC45CCAF0ADC20EE21570D0B654CA281D254AADF2DE152FD9761540C0C230C200FE51B8747604B384DB885C247131BA23E41CB896FE09B88175173F0448BFD32ADEB5ADD9CF4DC4CD32E11A0F326003CAE1AC20DCA43903163278066DCEB75085D201B727AE702CA1FCE42807CB8B8D66D3BE525AC14C3BE87319BCD5E2055EFDC58ED1BD5083CE02EB13AA53BEA81DF050FC682804FFC80C2B68BD4034CC9040F6C604F31BD1FB741183E6FB39572C027410210FCB49FD10FE201D703272203084E7C907785884A1CDD3FE5006C0A32E483A5438473D000EAB418DE6743A601EB744B095C1C809DEE75D0C5B30062C3F130EC31BA2407DC859DDB486232FE6806A1CBC983F11BE06EE8BBD2E78442B85BEB2B9BC342A889D84CB8D04540A4E48C7EA3B154C6927B6288F481B376955823E392CF45708042303758CA18701584CAD63C7FE1B1EC48248C9EE1380737803EA07E9A693A35D2A8BD744F85C179DEC74E42024BCA11008FF00D35217B889BD1069141BE76F17EC858C0F11851CA4D39632B831240601189F4E853B95653B6D3C5B8C10ECE9176690EB'
	$sHexData &= '5BFF53ADAEEDFDB41688FFC089BDA782BCED1FC0037C5C67ECEB0CDBCD8055932ADA100D884CC3952B274849C395C13C7688AA38500AD23030D5020429332CE7750006E1959C9E0AA7CFAA6C82197A38CE6D194387C9837B7892D07474747A19A12038C1E37562FCF01560B934ACD27A84887B838015F37F8D90F1F680C53080A7731D325810F6632CFF1B77C599A8020428DFB8C95B0D5C4E18EC06618DC3821762ABCE1B538F207279C4648090B7DA7F18668A2FC67234669014C61927A0C908D6C7FD03162C2080DFC365C3CAF64CF36658F173CE5EBD8801108E870533F681FBDC0143020F86274DC90413D7171017204CC407C249EDCAFB7F9B8AC982061003E940169F9367E4F80538FC3004068F313C2328007378DEB46015EB9B67E96CF9C086064F4D5896FC1CDCA91C10062232605C054F83221A043D855CDD607138B20CEF0658801578E578FFCB9B48BBA51E8EAFA9BF85D736859B5C085FF32ACF3B08BFC6C5D238523BC1741F5074E053314C0FAFCB5628DB2014AB78562CC1DA162CF81E76EE75962315BE53400B3169CDD444744D1A840E001F0F9731D86621D1BDB530FC9206616892D723488B4F78696CDEF94883EFFF1F7ECEEA470EC2419A831CCD82156983CDF219E142695C3C081FDE334569384046A187CFC142BA2F6CBE8EA5F82D8847802CC8BA1B3BBDA3B4B715'
	$sHexData &= '0F16443B8DF4192882A5E201D7FFA613D28E6CB02008CD9933B16054806F273013400E90141074F80747460664303038742214B84E5C0D553007573070ECF37C283590C72C28C57E457BCC4CEB194FAAF3AAB5B1FA42F4908D82B7E0D04AE7BC48C84D691BD5F1B449E82471036310E26079B6200765D3C83349A3CB28CF6807872BC86093CEF8CB2326AF00B9400721AB834B7B744010711C6458D2476AC5CB4839740CDBABC0C0D888631C94F8B8F4D2ED49FFC53789AA23025C86B62D97F0487A58C5521F0D1990A9822F32505128915C3B85586B0DE7CA8676A5682BA56052FCD101E12BE74963464C78075023A5128054E0B966E05C65D2CFD1095C562E0FA4BD09C6C1097672F2B9095A8230098D939787D45453824009DE6B49E02DE0C4DD0D253397596206D17F58CB5EE022DD8101F35C26DAC00995EA495B12C833FFFFB9ECD001AE13B8ECB609F1C3213C6BDF583B44AA48373146065419041C50A0AEF802E88B441908995CD1F845A5FE9367B4180C3C02EB06458BFA5CD07970A4A11FFBA49941BE136DB3D734E036A5053952E495DA1A08654A0EF945BD2F0CD742D3CA9AFFC7D6FCEED01103D9A44C52FBB2A1877C46DC9B8705B60E7806C685F3C4131CD7105289C6ACDAD99CA5191895282A660E9C71673F83CE40D4C12375102C20AA2D7972405E98097809C009CB7B2E79'
	$sHexData &= 'C00947616E904F76A49A18DB2D7009A80970353220C30EDC38091F303F3530D01A208211B197EB9605CA068FD3207BE6BF882090C19812E749562739F51C8C17F46F263D5CE9543CE6B45139681D4EDBA3D3FF7117B580252F6020739AFE1D148001156F7F9C2251EC097EB46125980797066144BE1CA8B0F8C0BB213541395E60FCE1DF3CAA18DE4CDE4CB8DB34B6D7F97F92F182DE1B43988BDA48C1FB12488BC353803170D85C0B0905AC818B9C7EB8C123904FE5DA2B07887802CD249FF645302498B80558A04D69973C989870A825C0254F73A0B058A8A316202EC80DC1A8F9C1B2544CE16C0C3DBE9884E0790E4A04103AA28491D78943E0F85EC97C38B9E479CF517CB026C8D8B882E392E7CB1890021930C97714F8F80488E6A0445A02FD2BE48981394B4CED27783492A150946A09B868917A69682D49B249BDA10EC40E35821F22694C631E421028A6F149F7E0481D3DA671A4E41F859F436EC8BD1E6F4DDC81CE0499A0D30060A9F445982423CD6DD228F0782A3661C1E043827941EADBEDA7E40B0348F7E32BD96683C3FE41A49986C78919379ED20489D899327B7B2906B01C102006487867C5AE59B02D9D8CA3770F45789DCECB6300E70C9051783C6200B38D3158578918D36C28697F039B783E26B3D9E453BC8FA89DB0C80072CB2FA3A80AE49A7BC0314B5BA9B826396464'
	$sHexData &= 'C8D0B8C86E75444A9E4983BDF0800FF775AB1F6AE8F2726219CFCB20073007E4056FF238072F753007DB2D47AE095EB15AA918C9437E79400A500AAFB55905C2CC39B384BD8748C22C90B725AD012DAEE421AF9BF8000801CF86994F77468B03A108BCF696295DA433A71801EF342587284C02A818C8C8259701B80D2F25934CC8B83848236F6162B2AA3898C8E5C8250FB9AF984C20934C586868462639F25858040C4399E4DA5B33AB78880D9377004D88A878ABC981CC4ED6AEA9039823CF0132A8909838C8C8C825017D382025934CB8C8799E5B184602B4B8B8E4D80719180FBAEE1EBB1B9249266418D8E839F2BC97801DB6B0D838045B468A261C6A0BA7395EDE58D3B79DBC6E7BEFCC0D98E72FE7364AF8D0D5E71B5199B64B7367D7FFB310490E1919202810F2280E90201A83DCAC8C8C4CEC10B130404807C82487304019A82D78115C6514D66464BA36CBD14050DA253964584036B397E018B04F0965632027B90A83F8080272F27C0C173AA8F8049091874C1659A80418402699000015B9E4F2F9F2A0D800F814BB90570FFDA7AFF802B7E719401E680213981C32C9CB5802D80212A132C9B5B739B738488E3CCF011158387803BDE590491045390E9049AEB718280F493EDC7930502702C2E4E490073D0298A9BD7CFF584D68AB089267E60D7FB04AAAB83DE510E1091DCFA939'
	$sHexData &= '00998E7583C5C892F794EE0AB70657BC98DF2358E30B5871D8A6C598CF65D324303EB5B878BD63490D90C8F2815A98386C633E34D2C60BBFD11455AAD373B68893DE7BD6A89B10E4352EE42CD0060743BA611F5CB865580BC86919B80733B50139E62C61F818191B90DB8E253338D8670232200358F8C1C0AE9D5EB5044D014B6EBB965C40A119102BB064408009726D406ECF010333B803322003389858C006322078CF6C300CC858983B330CC8800CB818D88E5832303BF8140772C92B81400A200041185860FD7B78E4DA1AA39B665B04D86048BBD8A38B196B032D19F8C44DC00664601938584DC8800CC87898B803C1E01879DB502BDD4193061630BDDEE4520CF098B01FE12C619199BED182DA12121EAEAE20F02806498E49C746435A5D7A37901E83CE10A31140975414A8084F0EC881D798087808C00836CF252FC0084750900832C82F4DA9080843182103C1900E5E3A7AA127988F5A031BD81960398B9DE04BC15AA9BBE35B21BF8D238407BB4AC11B811CF8BD4203751B8790B7850A205109471AF7288591A077BF74186B836D9A8443EF5104214132C8B006E052002466A1B0789741464E0007F9D8E08CC92427D8D80A819CA8599CB0BFE533946108636E19BE6D71283C42AAC6EA3AD81938326FCB45D063139758D910BE9BCC150BD5FD79C45B100BDB393D404E5DC318A1B8'
	$sHexData &= '6D906FF26A8DEEA01DBF877CD5506424557D93E490CB693756146167971C72B91845331C32C9432E29FF9F2044483CE40DCB9F339317F2E4F26C909FD66C214F5EC8549FBA6C209F6A16166B3DA5B10146F2DA9EA05F48C5901C8A5C6D4CC8AB59B14A9E114AC3E411514FA07E0D6A62D17924349987023B228FFB9D19A88505AB01D9231A8C342D6506D81EAD674B4D00551C715E317536A21BE00B3E9FED42450843D04BC053B022444C7D5BA043E4C85561FFE05802A4BF09C2A155BFC745B7DB636908CF02B2893A047A0800BCC07B0610355671084E5E032FDE46181F01F0E7F21AF82B1C1B07F607206CC9E9441AB83BB5FD1DA4704018443B481C135D033B4870454DDF0913FF19B8D0450745E1E9188B55F75645FE75708E1545FB85C0B44DE703185966609C721F3276524898228533793585C79F0FFC0DC9AC0140F834F04D1FC40FA1810D961E0A5728118270489916920F08BD217B7F206B487C68AD74C40740C6980FB601C71A680D67025E966E03AD01594506063E8D6B6EDF8E8CC707DE23B23F01253C27D09E81F285780F99FCB70F56C40FA07423807D8E7939CB80AD884D7355DF411C184CDC003BC3F90601A9BC2001E02C2D6020328C32F2549EF6B71030203828D13C95A740304838C1E016F053A0414910C3FF142C89667A0E836940204453272483359021D3BEB28A'
	$sHexData &= 'A0B1653F10C400E4F810C4CAA97FB0FB5005454604D96FDA55E4731F0113E18179AE1CC2F23B051E05DCBAF7F6C1A285D4C28B31CFF76C17781785837B915A8B13BA3ED901A6F0038BF0F3B64BF2CC4E56D0034C2758B09FE4A94CF85490E08B5350BCA605799A7022E84CBE50C8D33C86C32648300CC8D30C40201038A7B9E4D31CF002301890A71990D02814B0C490A7192010902CE1D09CF01C3A6213B009F8C449BA371F21F20AE4C23A04C5CB0FE097A50DB69C98CC49E1302F52BC1B5003B78C9C6050206DA1591872044822DD2E53990D4C935061DA0DB9D48C041479B3DE33317C8D4B90AD7D3964700205309901333534E44299F0726C4512E933DB3251AAC97BCE100D4FAE501372397207C89850D07335AB2465AB283272C9C8D030085AC824133230B0C0948C3C1330B010934CC8E5EC971090A039F24C203890F001B54546260B7E334C20B70B6D70A580408C5CF2CC449ED00D0EF68E848CD06F4D6099B9E479AE4850B00C9309B91CA196B03072F0BE90404D8BCD9430CE90644266533E959010E4798EE420CC107064422EA7DB9570F0790E8EE400CFC0F0503221B329709550D0F23C4772E0CED030848C8C5C061530B95CC8C9B0C04CB0B91CB9E41001C69410BF90930990A066490F7E09B947097490118624EEAB5873D90866482432878517D82247D183531F1CF24D5B'
	$sHexData &= 'B9832DB0300372C880D050F0870CC821701090801C322030B0500372C850D0702AC94B8699F090000119D8B603A9333090011990D050F020C90119701004DF9A2C3C837BFB90061B6C20E1C706E4DAC1AD0319305040066440709064400664B0D0F0102B981100AA182B9CAFEDA00500FA68A38CA9C540EB76314E4001BC4DA9E0062F10C76878989789002FAC22F0F852F24856245E4771D90CD8BA8468842033BE6D7A0231B68B408BDFF2167008C84EFD2D48A4515F9D8DA24A0B810C16F176F67DEE5BD82DDA0C397845138A0199432C225E639584472040043EFECE29FDA1FA3BD872549048C459270CD8D155C04AF0C3C040678B511103199538CABC6E11E7EA5CC01888B56DEE6DC608F4A77B9F7B5C815CEC93696307C85720833E05751F362838C919746235617C854CF2F8943109752F8DF0938CD7751041C70686E48AE0B30C0710EB4FAF062A23171BAE8120400E17B933F21E02047D054D784C9188E04B60E969A23376DD5F57838C2D7C718BFA7744B7303C097D0609B43AF489E6F2E6DF017534240EB160C4CF149CF7EA8D77FFF144C0DBBDBA70026B21319FA41D1BA92329B68BCD7423DD8C4C985F4314605B01FB4F6B893FAACE40023BDE7D0BB93249045740E73B1EDF7CB5E48953C3E3B6CBC6ECB14AEB93E15F3B1782B8A6BF23499FBC74E88D6BB11AD8834AF74D89'
	$sHexData &= '6DDADCD66BC8B80C7BC066E844DAA6038012408B974C388FD201C0D61CE2E8665E78ACA806CE1FFB34DC3F1843684666FD8B5D7717AC9059B410F6A27E4032F6A001F569626C41BE5BD8533FFCF78D5B208BC8C745BB010E4169C469CD667F6099F7F95CDEC048A724C7BD4A7A4D12B71ABF68B341B828AD03EDDE7C1AC78054FB5AA72CAF5605E16CB6D05DD8102E8C75A5B5D404CF20D70FB6F69F7A21A8D91C185A4045F7092A7E085CF861C49B196033AD6DD8459774DEA687404FF382FF662D2714C4458BEF2BC1452BE84293D55E6FC2830FAFC8862AC5D2B6184EE203C8EC9EE6C90E67BDFEF62EC177060A51C1EB0D0F28C18FB79E7E37971619DE0D34FA342FC8760AFE19820E2942EB5F8D041B7ADE77333568FE608E450F4C128AC0ADEBF67C4790831C7587905D595DCEE33DC406B57B2897F1188BD19684FE5C86F6C5333F994559D7F4DE7ECA7AA25A40828B9F368FB59F098330C40DF86B466482C04B602AC83C87C3B0DF8945AB7D0EB3ED79DE3B980CBF06D3DBE714A77B36CDF3448965BBC7CF7DE3EFFC7DD1854E679C7D41BF58019336BAB7A198CF90C24A4130A1BCA2204B9558F8A29330A5E68C109CEB435E12701148174CBB071F24DDB3237A70E38FCFC9295C2BFBB68D57FFF07BB7FDB4F4DCEB0F1C8BD7201B84807D7F48BD2375ECBCF80EAC3A176C6155D5F9'
	$sHexData &= '094B199CA6062B209BD2BE975EE30C59601852F47CD85B1F8CBE29A320EEB769C3E216B51EF34C12BF3D14E8C96DEF26F96C444269C63B24B3001F3CA406FF4C5BE9740C92789EA31A9BBACE53B0668A005C3CF7C1D829D2461097DF41561E595A3008FC4620B19AC22008E940159BB835FC008461A0DB6514CDF6DA293A15C4C510174DF608D0239C4F7808C6F1A611864EB98CA7CECA2B12D55008300819395893FF22606ADAA65D2650F1042A7A08EA37078B3F7063FB077C3D0889E00BE547E0B525ADC2CE5A2120D5516D63ECA572A3D8483322748A58F5110BC6FD24E9C3895E0A8B4C8431056E15360FC3F59DEEBE6D842095E13DC3EE958E00000EEB13A536E194702A56BD28B1A1600AB55F300416530E5F108A2C6474B469034F317520E39CCB2119EF94B04328EE0830EB1178838D07F9FF97FC3986148235E9506A9D910B906059B43EA1B5DF28394128B59A282CC57F272B2B1F2290F87008C844603A87158E076545FFB71484AD186B1A5D8D9B653CA48D018BCF884EB0119C30AB17098FA293BFB753FC8400F6187DCD02CD2E597500B2CDF139AF89F1D24C3BBE21CE20747CA395C235766F671F7221034762496060BED35EF7FC57C3D74192867075290F8F9622D3881ADF9F803A4A1E6A61BB0E9C403C80D9C1C8C28722FFFFB99703B2C0F360DD57FD31D8E5F2729057FD'
	$sHexData &= '316F577832950C48BEB14BE0797007D51B18857C376490A3690F06519789E086BB3059EA2BC6DACB24EAC25621C85270979DD46921C9809C2A7192561C4DC8930756FB4D56C0A01414563B81A0F0DD3727681D38A22DEDEE01A77CE0594040C40A021F504849BC5D01E385527C772B344D9775D13004C4EFD96F90D14993605C33F673E70D8D674C8D0C56497C033144BF2B923C764D03FFF0B6903731FF9D204A8B1CFB076473D139CB7555FF552A8FE421FF9339FACB22F8966C305550FFC65F69A83430268280FEB188CB8152DBB2C6EF6AC0EB1890D793302475BD7660BDEE7761818D140BB91AB826C1E204A336D550988D1EC532EB8117BC83261193FE3CFBEB8A198ADBAAD34702985EEB0232A1FDC485575C14DDCC4065454E17CD404001598068E9F88FEB02CB9C1BB28400D28B4CD0C7620D5D6207000624257DACFBC3FF453BCF450F4DF933DBCC49D8B864242654519BF24FD83E53B2016D1180FEAC83FF08041270460203AED49A09687E099AECB52607D7D667248416278B5B497D7452DC16F9184233DED908033603BA2C83E8A6D78C44F444809180B1022E2CA94787A2CFF2040203530166F5288C0B199777CAB3505848935048496C0472DFB9C145909FB13C3F68B4E18122CBDAC0F904D4EAD12EC9432F0101F31A10F660D16383C49787EBD52A892308620618FB66AFB4'
	$sHexData &= '854C491F1B5340464063219B279E882C829D5224746FDE26FC3BC87C6C3B667F66223414073BD0E2ADC5ED7C5A3B5C3C7F540C740F33FC491D44396341E5162D668A44041D451144120AA585292E040C2645B2CC241ADE41468F8731BCAA28445B517704163589BB36246E2447FF5F8FC84B37EEC00B3E39081C37A19B481DF7C3BF3815017A027CF1B994CCA4E86461BB327CD04219A876F6CB5EF083C8018912BCF8423AEC28A3A00589F3F21D4AE001E3546A8A1E462A00A77AE20B9D2339CF80BB9EF7D4C1451686145C180C4279466020A8378477871A28BF7F8A8CB453DEC438388CE43BED9EDAD21E943F48CEED6C6490501BA0004D9CEED115BC89AA509054441021E0F682A59CB9575E6542E0EA03D9356520238DBF03A92D00EA3CE52095201618B040C30A73C003FFE5C881E48B3C8178A00906E404C5480197412087AFC34ADC2D89A534DCC150A756E348461E8E0FA680100E0B99E128E79099A487851308E8A92E1A2EFED37089AC73B0B44950C80C1BFBC57083F041607FD94E5E9A34F9BCC8243BEF6FADBC18C51E77F9B9DE242F185E4CC5544024E8DBF0858700C3188D8F17EA7ED83DDD45D34197C769BFD0B587049F0CCCAF8B455F0A93A6699AA667976F9B77BB17049ACB7FA2859F64105D3A745FDF844DD78FDBAB027D199AED2A0475BF440E8F44C78BA4410B9230'
	$sHexData &= '20EF1BF4E5903B5121A063B1A4404DDAD924795A7B878BE96F255D40E0C48ED68BCD8001818B05E99C38004895FF7C0D043692D763F1D00639AB3978A0F1C2444E01879C417217090F2D12614B1A494726BA232A5009584746976E12F12B60C853B981F2663B026860A2B9707F0FB6F844394410A02F5E02466001463270F0131B0911AD8DE002E777BE4FF0F80565B581B7BE60080610A0138C88772136E33F2561B7E06D413870FC6710729D163672D7E44F480843080D285F47FE90A4D9DA604485683C610380B3A4ED7A08B792C2BB9217EF1727D2475A3D152C24FED1419E41DE20182C28344083419E303C387970031F017CFD8D4B45281030062B00FFA5FB35188D41C1E00857289066904BD0F62CFE9DCAB05F54440BC05B30D80838C0A65934578B94904AE53D9F6438E0D2910E09793C81FA5D703CDC01E9EA4D0105CF6C0A2F7410ED1207021F4461F7C8DA704C164118C2EB112C51B9A24F783C003778EB889622BC34118960406E9E4060147801405BE0412C97400369D6361D212126AA58C4792B7F421A7302063C1626BE65A540FE7E730246F672C40201474490AF3485E08A6F782D4D2317244E010D77CCB8D042A9D220B82DAA68BC2BF308404D90B80242DF684D4CF4351B01136C3B0E004E5B1AD38DB4F6860D6A644DEE0BF07C4708498D065C5EA0505C089A8B745814'
	$sHexData &= 'BB990D25604D03F040EE257C37E72841BDCA2D15104CAD2E3A80A1A432ECFC20498B07492B0C0A558002C2A49F088883363BC028E75C021DBE2123FDEC4B032743EF1C74F6E025FC0F8D48FFF7C1A88B14984DB300BB02110C5FB67E38064E47CD44C420EB880A1246101C89489CAE4160DB9FA0EE01E4C37438C375E8391B85345AA352FE400A238623126D7EDADEE54B6F9109C776670E90237F231229DB48D1FB4C8BC37879AE4C2C1C78987B65E22D3B07A887DBE48AA68D98B5B023E107CF05B0110F8250864741B9112027EB2BA9817D09EDD640F19396676D08C7756892848AB46C2423D68C13D576080672400AC1EA80C36D620C4875077811A3E1A82D22892BF8904EF7DF25A3465B6564D2D787624078BBC099576C127446593866037A843018182E99C61534BCB71A0E140F125DE5D0CA0B3B101D00A8455AB98DE341063AC2F8120AACC2280B291D1BAEDC4CEA4841AD00F1B836984B6E1720DED45953310C173D4748D0A73EE47432F02C2826961F3334B020260B2F6434A91C0C0B4776900369AD2C1EF045245AD151C06F0CF4A9193803ABA5AAA82D7C16E2A0331244A39087AC1C9B45CC33C033EFE5971B794370096F781C9455908F302FF7430C300388554118409969B932B5F928C67201FC06F2562E46202095472A7381A4CF821C0E3D3C1F4A218C17297B83EE04700D'
	$sHexData &= 'DF78AD9D4313C524F4032C6D6E8B85B8AD203DFA6EB94FECDDE71B852B3DE66E022D570C54280C0EC06C927CFF020ACE9BC9C0D245D38B18C59FDD0702B4C66D020F15BD8B15602895B87B0A6A0C306016E11EBA5864E7994767331CD1A80A7B2E48218423111909B7B908938AB0DA03D918B8972A8830F889710DE57A0A9DED9850C91205EBDA160E44B4CA36D29DC49FD027881CBA0083038CA5D5A00FB5789D85205F059FC33002878FBE235D01106983FA2110A2CC0D40B1CC6B7B3E40DBD743E7E07440666EAB852FA072893828125627CDB11FA8D0EB173FCB650201AC9066301FA08C7B80745A400F245D5D0D2EF7C47C1EE48001AF080FAB2842C1DFAB0958A1A84FC8EB0517894AFBEBF50D4F3102C128CA33C933D2AC86AEBCD6FA8905376C02ED18384BD233C9FEC51969691AE7EB3933DB4D4C14DEF64A2BE20D0563F6171BF8D24C221DF36BE4391D14D2A9BC976C1D0B10EB02717550BABDAFC258FAA0823DCC8711186B6D30C168011D9215368E5F4890402FE4D42BE0E96B282E4A41F78A3E903E90F20736B36F701D5C4B833B00752D4D3009420663F1DD250F56D4415C045B68777B60033FF1243233E27E841B918D3CF57B7118735C354AF00CC41CB822689D56F02A16DB807924E814629F88439AA644700D4045DA4C40775E20133B2B20542CADECA17524744B3B8D00'
	$sHexData &= '7F6A903B05F09DDD05E7E135B6C7B75C6E80100A25AFBD9274247255664C906A4D20519F401EA512AE87482A30F1C0088F0693E0DE05E848F9DB880121858502CD8836C04F8841132B4981F87D0F8700A524F41847FFA3AD04A9DC93FCFCC4636B8F038113521C569E63CB10405A24647717670DFC11D07E4706C604C57289552E4B544E6046A3891184188C24640E617B3BF0772857BA554A104E3DEC025449161C423B41783E92DEA60160A44DCEB3FCB51A384F367A0E0ADC327A2545283B95BB93C69175BBB869C860C8486BD37E896FF511DCCCB9163F0EECCC2CEBAA26B648CB01BF95955DC03240FEC002AE1A1A0B932C6C752E7C7C719B39C7C697220C65D711DA58B6FE163D5021DA405C8BD01384A7AB9CAC76CE0498DB76DF750D32C9A81274068995290FB75C2166B83786365DA917B62388C18885AB36F40E7A0A84B914C2B958E40964221F20897E0147425BBB284603044310F423884318D9469B52C0494139857409DC913AA3437D300D127A2B8C286CA79049891E0F22DADB660C7ED121994611951B1A3C0611D18B9DBC118C2FFC4150C26C03C25B301AD50796ACBC6A330003C002F26244329F10252B1741E87A0424803B504CCA41642B4BED3900490CC88F0C2481D06D9B18797F7A416B63E5C04212E95E069403029F20E0667ACD55304CB1DB44B1CD0C7442CA33E0'
	$sHexData &= '04064D05D106A3B42E7C31C2880B45084A4AF87E05350B2F03CA743EFEB0CBD0ED9B3A4C3148B5FA04C3BCBB6A1DCE0033D0ACE8F73C84865179619F0AEB3933625021C04EFB94021C4A016B1813FACEC1630769C0D1C63388840A6DAD6F9BF6AC16EC541642048A301881E1C03B0C3C74DEBE2E16742A163BB6064D2297E85431C00493B00D3CA4B63BCAB0F1DDAEF44F18CFEBD9854A9FD9CAE860C402459B493B4ADA0EEAC4752D9F922BE16019E46C052D320B84976F552A312817B296B0D847E30A29700110C11AF0B443CAD39013909E3CCE7689131A3CCF26EF3A4B1A3A2550534AB039190385A0109D78310E022F8861D0E4E2241F56E84C62B5249A00E227C249E2B3AF41A9BC6F26B5BF10402022CF30A95510FADFB938452F7900C92AB4380155AFC0BFF15291C88C058D7501687341FAF39D3704776243CF1CEAC8781F3DB88102033A1E4120386529636492701ABE66CF2261B2F0F72669EF2AFF06E7C645DF19EF182A9A4571F2D38610A33351C8BC180AA4D5C61565827A7D17BA6AC625FF1E297F644A3E42231D09AC8E141EC683DAB0FE2570A9490CD0484CA6D24AE8CF08240AB2F361E83A5E0BFC9721015474B66DC414040263D00B18899047DC4F236A49FF6CC1A6D64711C02B19B6B25C4C7D08B2B9C54C55849E24EAFB9CD06AA01253C976AF7DC0433A64F235A5C7'
	$sHexData &= '105E4CCC486BC3383BB5F05188035A063BA4655C81A17F40CFC820AB82D32638301D659CD0D202639BD937381ABAA88930B0040E26461EAC9FB7F08419566C90B7659EC68B0A2F74139D405B5D11B55A57812A00433A281412715AC5D89389113F4A042F24E147E64110743158802C4DA0490626E20D30A0C03E12F78324F36343D24820478938B231C3FD40B1A1576D922C3FE2852384155C18B968504032C299A36658D2E940E5EC48E27C04478752040DA6747213A4EF9E58064038F42C7A0100C891E33E8DF6364DF3611D964FF9358FF829E85C4E1CD7B5C0A95445293FC6C524687F702997B5431D63A11AC808C510D0C814BC5C8248B950C9D857875A7250ABB5F88E8DDB94DC4930B000084000D68454C9473E5E3A603694E91FC6475D3BC1F2E4E93366F819488DBC5825A3C78D52705F81D58EE983413D03A25833017FAB480EC92E716C783EAFA0025FD0ACDD6F11C82340B13549A722E4D13C18567CA9C54E6724BB1F3D78867E009F8B79089E1285E7814A0D483BDF0E799B4807C3FF85FC4AD8612BC124480FAFC2438B91D01EC36AA17431C9C17138D34182BC9146AD7C080F695720BF3E25D803F75132720BC00AFB80FA02367E04C30E98066C109B5CBF11BC811E2A005651187E142E2F2D41043DC125CD1DCB4AD1C02119873403CAD00309C25982ACF50D9C08D6E51F17'
	$sHexData &= '49B052309A7136109CFC7972C8C0DF57F920C50DD24FA57471DE7443E9B9F8F145DE75BD95572C7438A58354D0E481799A16890F07474799913A265F7FDF2C0D05A45433FA0D3601BF75F639D789301661A0318BF336B0002FEB21E326B3E811D2A3BB55187EB20132B677E11906408877283C70280926BDCE5882FAB00264F53C701051266BBB355236174F68401814180C74375DCD1C1C7CB68847200F21A0C0EEDF2141887021C64728E974448B092C99E4351B520127E03EE18E405481C55641939F0AA78E3F3745D5024C894A166002BC100618F502F32E100EC826B27AE991D36816490B7F20509DA25A9F6C5F746E575553917411F706405656213F2613E0171025A823FFA28E4026922C23E011518840F55AF2E43924A9D024A92B5F54F0E8CC40D54808488B00159D1F3432194F084C8B0D581032E0DF0D1423735C84B4B6E6390D3CD604A20C0847E6699E080C0C109242B92BBCA318BA28063096182173F4791AC728284818C828C8EC31C8303849A49C8F0DC4E4D6779F581D56569A28F23BCA261B9D508AF5000EB2AED2F603088A1BA925C2A2BA283C056069DDB21F374C16960B1B38285141F808700F48BA519547388BB36671036B168C22C2D0386914374CD4793817C606B206C2C369B410026782F0346D7140704A3676761F481A0CC476C7285F594A13A45EFF53440854'
	$sHexData &= '65BF0394168134C50D34184435387284708857C78B39A633C3110BB01DBF1FE46004F54A4F272CBA306723840760FBC4DF35BB83D2360E3993E947023C22841F27D0C7E429C4011F6C1C18EF9686972B0304404014374C012125403751D9C317A3183E17FFCA210E62F9866294596A03DA4823D8A3C3D5689136BF4056D55A00AB1581FAE11B656F20019375086C2301766F37D10B8669DF0873C3152758507F49BA387F7C182003381F302DAC203D17A979298026BDA4A9962A03647E2A0C3F8877C176184243372022C03D26297835FC7C7C4646360F110730382F6BD214AC2DDE3E1E4EDE212707CEF348DE0822E53E261D890017566304B5B0D51D77234A87571E46469650ABD1D0660C43D0B84BA99FD0FA52026D62A579F312E0FFC04CC2F3DECDC1EC26CC493E85DDDD3D772305C6438905BF290F443A37F31A7A7030C6368B0D9AD36D6229203CC80F5FF33CC9387A52020F77E50DDBC481A468E7335CC0F3CFC6333B8928A0926E10031A08004837A53AF694B6434AF91BC603494120BDDFED3A9FF24A12FF052ACF351BC7AB869446B3FFF9543E07F428FF51FAE5510786BF29D13B1DC25103C4516423242FBD510D49F3409CF5491E7F327B08C35BD17CA96D68E383E740D7F51CC341E5295867B047434FA7079C5FA54B56EB3547BC47804B19079F20692C1726D62AB6EFB4F090'
	$sHexData &= 'CE4905D17313CE0490E048C305671A48764C3AA7057B504815D0763C4A1B02383F6AADCCC2074A07D247CE14BC8C5D5C95004B5F23062124076DD8686262B1CCEF098D05EBDF3F6FF0B5FA08E52CCA76572C2C4E8FD8C97488BC04C901BE6BAA850BB723A4C3730806E39E1F5874068B47080E9D1136E98171A9F88405B90C5779E2352A571F01C058511042114634DD0B1629DF101E100F3960162C6F920E4790AE4872F69940E907A47E81362C068EE1A3D22A5E75691A25B828E46C1B16BEA8DC3F48023D1446471ACF3A17D2C0285C105A8021FF5E008CDBED81C00296C275E74AF1743C422561819E743758ED1D367EF0D1F8E3EB626810C2A2B934FAFA0E42D28207C7A9A0EB450B4BDEAD089C6E92D97422C403EFB66E0110C3CAC302443C75AA2075B8E166134747C34978616690B2FDD0DBC158E366909918EFA58299938CE7794141670DCF56DA50FB4D8BC3A1158B1AB95FBD9064BB4BFE1BF1A50955089CF0D1D8B69D1297379A270E67DF229EE01017260E317050EAC0B9C83BC0A19D0BF304E06001427BC930D5988CBD728E01E2D954CAC04DE8D06F1267F8E5D7C1EF6C2C616C0710A81E5DE04184300EF707629752EF8D559ACE1D8C7B40B4370CA2FF02742A7E0EE262183BD1210CC31DFC493F45C8860CC7421823EAA6225289E451D42094A85258CC405B917849ACBF83'
	$sHexData &= '45882D41D89406C38BBCE8C2D2A97769208D810BC4675E33218F664650CFBC2245DB118537B16016AFA13490632541F0F54ACC2252B1FA8204747BF6F37892F0B5D435667B1844AC4D5E4B188F2A0D5D584643C004EEB6A161B1EB5A40037415E43EB0FBEBDB0D0F0275D84E0881A220082BA660BD12AB2D7D1100BF02742F0543BF8845F8A4CAC1E305738BA89844830EFF7C7F6283EF017438CCB8EF00B7E132ABB2BBB0F5901A8B80F804948BDB13EE427B6CD5A61F60696B565A1ACC706F34909CDC4EA762015C20578626A6DD408DD0F32A4A850D4CD0B064B074F84439194226BB20202871C8907CC0E9FE30282C85643A831C0C21B90AB111322082141A1B9F601731BA4002F11038D0B9D28B91C8201CCCD0D18674FABF25283180C4FABC9111784A087468F46A59533A014F826EE3C179145F33D29D2C511AC424D50F0A8B5041BE7DF518CB832F9BC2EB8C246A68023F1C3AC24A6EAFF91F010319913F9A3F0E1E014B0571D6BF0A2BF81D1101412566611DC6219C185F8BDAEF038844944BF3E40A0B1CE944FE1801C01B0F7472BDD00160F1223A601B56869D4817CC88071AD580E20BB5693AD1092E4F5C4B4F427D60A4470FD3E1F35858A446274C954F9CA976F0024A15412828189642612186E59A421F9108309C9748F9483FC44DE919071F580A176E9E3530961C9998A78C'
	$sHexData &= 'A1A056D88412061849FC5DE8121474644D04BE1818FA4C88017F0CE844A8595C3669770A03A3D3FC795C275CA136A75E0692DEE2F641905005C80772C676341788700E3293675E65786E479227A4C10F867EEB0E208E4479D4105A482B1902071B64DCA44E030E0F5F1A2AD042F1A6014A524803A9ADF4D7C97518C2EB135806E008BD156A6CBCEB059BCE82099B2788BE2CE206229CBE70F124497E2567F43DA6565BD07CCC0A08DC09E911BA7FA031437A4CB935A6C16506495743094351AC41D4031BDF55322047BD4F39E449330F76BFFFF719A4B2593856FE0F710FAB89670AC60FBA7621285E5B156027E180B9E73248333816F8A1964917A4F98CD84F86A3529A05274D19764686E201C5EBB8C34601BB7341AD40467B5748893BBF516438C2720C4F500605BCC0F9D1E436EE000D41B87C358D81027D0A3952577B0C4B8D97BF47CA135C483B387486BD2786733C49CE7919365981AD9BDF7E8BD21AF496A698A6094C49A8C16A1A044CC8904A57055C497C4D894A62126B49D4B18DB85C4885EA805F11F7EF121DAB259F6661C263C2E10110EC4F5A97190396A3D70E74133868B7A21E1E0A5C9468E1CD0E1CAB681A5CFF83B9E35BC4195BA69601AA995841D508D7487DD8619A663B173044A1E49E028B51BE1C216C229646A099C1158812988A55C265580D7E2A3F4D41F949FC90'
	$sHexData &= '7E600D4DBB04060B473987EA8983850249F5514DF89E1B7DE00B0D4D44C7409F934EC1DF41C6477C01884772C8B246DED340526BF242893B219DFA372D894F741A68B5D262BABC839E4690A782C3A10502C350BBB1B40E4903B41BF0612B4CF9357472CA863C603D87F0B2693ADD74D604BF01275553DEDD0AF120FAAA74344770496FE32A41942454E575D256C697438908403D61023E0ED144B0106A2D055157608B435259D60D0B04422D1D827315EB1572D6035DB08EB12505415615641A129FD628C90BC5BE044AA54E101568714CE923E10F03D74C3930751141EEC14FB1A80BE66883EE02FF35045B62EF3C45F77E1D3B0C5398F6C674170B0DAA85A978500C3896376C971A5D2D39F219230F6E93204318CF208B9397434E2A48434B53973CCD200F574707913C27234747C7598841896F78F11515B464361FD93F97E601C010C188886BC0619203F96F0284BEA212B62DC849504D631121E15185CC52541D42DDB3F53A41E0574410429854A448E20A82940A494BC33F468CBF3AE3DB7A4E9327C363C787D243013551DCE1E1A342093D2F2618E92E4E3220082C23468B2AD79CEFC418BB9D26C33F0E00BC6D3F43DCD8E90FC32D239592F21F273051E7980A7BB69EB18D9D9AAE773EA11E2CE929E94D1F301C00D8724A207E06F54B432736520F448804418B6424D894EB0D24772A'
	$sHexData &= 'A1F8C8D1B28C11523573B3594D8E4C7C18CFCD4470294C758EFF2A4FA51DFB1009300728A99D4E79F7E6054F6840B75023BE14730983E87757F0DB6F3EDB598B5D1C398CB02B0D4465D8F3E745F78A49762268C03BD9D8CB1F492B284E404E0B8A92182E87A109A58E0DE9E11B72AF17A43D9CA11986D82C3D809A5C98982545772B0F17AA5C5F6376C15D33057F042B0FBE18EBB34948B619AFA41003D8E27E1FA633470AB6CB400C0B1497DC1631F062C5AE064FEBE1985D079A3EA60A0096B8DD2EFD0388030D493F07AE0662BF383FC23FD5823EC051504C630284BD362416EA73099A01F40A2459D5FF50247A129A63F8B8AFF85EC07AA3DAF4D7B520101607B05DF9631F5A28C3F6D146FABE87893E4AB52BACE76A0873359AD6CCAA5FD9FB3951FE752A265044E18300FE9C18E1756C34FE66B8F8BD28067D8D01F742809BC19EA66F794A7474087B0E549B6EAB621AEB7A28C675C7944B3864E9FE34531174524CE0A2C45ACA4E509A15F8AE86FA97E44863B001BC407B1BDC288B13100527FD1D090D50C7003209D644D85B1F2EDB974E68077F98805EC3E121C8FF0101101C87534283EAAD424742F3359380400D317C58DEAE67C54B0E504C217C4AFFF44136D265ADF3534A82496622026911B23879138078BA4E81B023423C884141C032D71DCD91898962034168ABC4D14B68AB'
	$sHexData &= '953E581582EB546F66CF8055270237E42478D55D185741EC521D58579D87714D011D48594F4397554824112890624858A323FB5EA6ACE7625991653C4414B2FE703DB3A7EB8E6CD89F2BE87444DF3F8C1D45B88BC5BA01690713D9A8386EE876DD433240BAC643726628C185CB394603F94C992E466B80030160285D489D58A6D879DAA5123F7D9E855A951FDC704D0189864528097EF1DD4C050EEA7B43807B7CF23423AD6C7980BB4643432A858C0C938B308C70C204FFEF94923898616040AA48178D96F8ABA31010B6255315A981A43A0774CA682268AA013BBCF036B52E585F2D6D3226B9C0FFB3A0493EE4420247A5808160CA3AAF9B707274269A581A312A6004DC8B81F5140AC0F99507847BC241504C8B084D42065979413041684D5728EB64CBC3EF406E6628A45648B804D983E68C0D9DA42EF6FA0474A181F8BFF20232C94183E10208F6B0D46EA99E02113D6C48C8492B8584C9BE83840BC045580F4952FB77A7A0490318654D8B7B3806072EC0D6429116304340790AFA0E372838727304497E2A184969C9A0A9326830558AAAA6713F5E7A3BBC6C787B68116C45773BA07B6A05756298628D3C5E3684F0158814F3893FE9148B67029050A69E5532F4B64D572D24246310F10C201A84968ABA517434C781CEE5E029310A4B4471C4DFAD86AF891AEB0749C7020832098B87D6'
	$sHexData &= '7A87AF39A20DE62E422B4842A6C2491A03A55E3F404330D40094D6CA35CCD01CF3954197B2F9F00389D24A83E002C10216F40004ABCA1D1EB0A54AA53884D2F231DCC346FCB50E97498B2F04432100931D1CA08A30F7C81B616563340516E90F08A105E91F09F44E83E3038055AC43D756842145985CED27A4618BDC85FF47C5C91CECB2F6EB243AC04DDE43C15984809A1635BA039460ECCE77721914FB29EED39F0574637671594A8D1C4F2F4B9A0BA514681FD3A9580C92C04E68C30E8A4F0E184D890A5F388BFAC081D87CC90F69517CC56900386A8B42E170514A0A78BFD83271C259504963134DE44D580814999C05F601C3D31623C53C402EA17528CB82E80992E164FA29B21ECD891E7602418903820022A4694EECEE0B1DB2DDBF3E41B933C60DCF55CD48187800763F9CDFDF8D1CD1740D0A50FE7406D402752C9A7D4A2BA741410AC308D60127E88411DE2314D0DC5141E2C1C2C35AC1C37F6427CABA91443416EF866E005A7F4A1CE46644934D78DDA4096CC4904A24BA68C614DF4924491611A1C64D68DA3498C6732814007B0AAF124B40381846A056629EFA40024C024D5E031E688EC51EA4468DE88D73180B1C202E3DD60FCC49314881A23F732C9F472DD62C364170E84A3576B125D148030818FCEB6E526E9DE889FD2F5AEA273ECD0F86AC06CF56EB1C837F0F8387BD26'
	$sHexData &= '604CEC3FB9259041378B5A107D6478274954A8A8E283E7E0F647461E38182DEAE0C374F42EBDB1518FFA0DCD25A203F63B2DF5B6E7C635A7A0A012923E78EACC4AAA50834FB603EFF054FD23F34145580589086857AB8689BA43C4220630387A37CC1EBC0D502620EB2F1F635076299268069C541581666C4D6C4F35AA1B3D224D9C944E480B098332C08F1EA525CF4A4CC66C0F9E556ED65310834B76FF08D8C22251392139EB042A7490F033877E6C1252B1FF510154A86515283347526D0BEC7157F70D161CC6724AEBBC452BA3122F4E152651086F4263238BB703D238E152715982D0443E87554D2D1920D884BF3CE80808836370FEFD3896B549FB4E0768E03025729F200A9B800F97E9110B975AA4BF6948C1018C95EA2D81B89B3191333080692A173E98182C37C378374151833601E40E52A61678F7144D0998B4A653CB16148C45235491904F964EB80116B1969947401F538FC097A689112024B94E730C252B4DD90C5942C4C7D0009E3A856DBBFF8DB1587185F0E1AA7F066863400437FB6C809C5F48840835062B08DD7D77458D881B45898C08542B8D4E089C763E087BFB0501109A496B5117170805CEB5A85563FF7F7C92E556A7419892AB60959F68C9818C9C9F476868A60AD2C86418FB6E4A1E720899CD04F800197940309F4A3C7813EBD42ADF43C2CAD28C595225246E'
	$sHexData &= '43A336B4D4AF73D0751B8AD29472EBC1A161491C42A012D4BD37D1FB643D404858EA4FABCC805E72FB07772B7C9E03DB72C008814C431348D25A52E53B92C713F2A5104AC3C576315A87C0E4244F244432E5177220E8E9488D4D582A124F55AFB07A530474A2707EE646F8EB08A8C21190995F98D62C03A730567A5D83838B97900F5AC95211B08205600C335B8FCD2D64589FDFBE0AAD0085FB45EA969F1D1E403BC62550190F7720B5F0411E180F42C60433E1C011B0210FCF05A4E45685C6761E123007527E37F00AB81DB125A6A4FFEB2A9EA7E0A17D4AEBBCFC78E747970EA849F808890068DAC340D5740CCF7649AAE00A81363B472A38485741F59F2A1D5F1831195839580415FEC551472C901FD6C32F2E3376C04C28B8343343F3B8A4C3255F683F44F292018E455B514D2D432003335B607808290C464997CBD331465ED73E6E6064B03CCF6DA0B01A16C148775B600DED06FFB90F1853357CBB7F92F93D84134DA4C710B652AFC81AEF6845498D2C179FFA697A3F458B71EACD56C9CB773F6222E9336B6ED1EADF2E072D840E121607C8BAD80BA52005D7CE4B1717C3D36952B51D0B408F2831BDE42552B5746771D350030F6F104EC35F69E27965A949BE4948D399D87A17F1753FCF23B175900B0BEB19A9052CCD12428AD7B4E1A8E16F0F60D18316C333D45FCF7A32645086C1'
	$sHexData &= '7E98D00470DBAA12F1F850092C6C72D44C08B230CE0F5391279C861650C8BB58DC7C83C06A408BF9DCB456EF435037B8F73E4C15ADD3DFDA1D4A104674E07714493BF477C50E9CFEF2D03534FCFAFF51FE1045003A00C526DA4EED3A2F3E878D1C46D44C9F95E800E08A374BC762442D6863412A5633C5197841CAFFEF5806F437B471955CC3AC2927A6BFD69C1462D4DA430E8B9C3F1FCC3AC5C49ED9E40E894F96A3C1F0450D2CB0221F04340193A5A1655782BA26782EACB63BFB1F4FB8A3CA054EC0772D4E414AA55D87CC9B091CEAE008495D321149D98D3C319F1853242D4622A578CB23A90BC04C8F0943C890FC493BF8760549092F0FC853C9FE0039FD00F9FD009E421E2155C3756690F294EC90FE746BECFC008E2483B7754993BF5FFB2057F25E24BC4283CB045B240D4592513CBBCF57F2249088FC0074A3D450908BA4CAC197AD78F9909F0F3CFC0055FD0018FC0063213900700108AC1F8C4D6559F6CF653DB031FDF95B06961E934D0B412CDD80084ED84D91558B32B960393EC8CE2CC1BD00562549922405D69002C9AE2C04796DEFCBBC86C2F79E3424C7693414D25A0027D1D90E195C49B97B9204DF0F4B5B0DF8CB5C8440C0C83419A24C62422FCD67563FC2C08A4911570662D62D3202517F08670A1043E84C1AC37AFB904D3BE9476BCDEB12AD110C91C834492EC26F'
	$sHexData &= '1243EB4D0934DB0A69C95C8A7D405027A4D61B40927B762D804156CA0D472A721DD282D7B6505DEBD0C96922FCDF68431B170C12086BC038CD1BBA0607986B70980A95A111D3670A199FD9CD4D6BE8384AB099D2008ACB0EA0DC676909387BDFCDDB1659EA89450A47040C04083ACDD33C080C0C102435AB5A7E57444D1867BB8E1760C538818D7FF4D145BC30EE0175BEDE01F96B1340BF1058CA961129F49218E4A43BF6F2871A45FC884D8D4618620694799AB7688B46E641040C08080C011985A70C10D4D043D2DCE162624D89601A6A231CD93066EBB2B41E4938C638BE5238416B02DF4038D8759CE3E906E989886559D11823E353706882779A4F5618F00447245F06C80A46700777F00040230E250EB8CD0477270E2901611006A006895B7F4C5C303829B042220C2549A501F51D2DEC7599DA1FD05351ECC782D0D7666A487C63135D8D5B188B039A36808AE0433A0A0808D099A6690C0C10F141F0DCC8F31C4D4D031F0E4B1BE499413021634323E3D00FDF86744D215B388943411FB49EFBD61F498C3E4BBDBC90584DA59BB1EDEAA7C88A73F2546BEF38A41C44D6FF2B26194889A668F3B2835B715AF6F53EAF54884EC6D67C4C8D464E181C2189A7147F01E472B91CB8414141415C83B39800A706B9011FC8F3540EB8494861419EE7793E214883C2C14049CE9482DA52C25C1E'
	$sHexData &= '9F862DBD3C90F2EE76C8CF22232397064646468B53C8C304092B342C470DE09B0436F34E8D342B2A16D0768E12330C92494D8CB03EA8EE6D5CE9C2FA6C2B40CA3B2773FAB4888903848943830A81A02773320A7F0A7D98F6CE29A9355C77ACC7177E020A6CE65C7F629D8C38402B0B6C7F84E1E1E97828253C414B9D4E3B0DAC9F5E8B427B604EE6640A790A770AC93340D29E41C1C20E090673D6946EF20CC3B0F3DF4911494C0C26C483DFE64C251CC93A4559E41540858C6779BF8F7C5CC2E10F80E3010F4D8B700589A9E1300BA0D0213C94123DF438B0F50B8CAE5DC8DB49BD56BA2442B7FF020337E9B3B2F7E92E49C1FEBF071FA19CC60E4C03F0AC0F8E7AF649D1887674F5FE496BDE8CDFE2C8CCCC062C713811CBAC3D034FB00EACDDD230E16F8FCDACDCB0B7D1DA5C72FF90F9875DCFDCD67C125503C52275C7D120EE3C35FC2249AAF133D2263C0AC77D37BB5C9C63794F2846EE1ECB82836C41B86DB9AD8775E19D0F8D3CFC684879901CC19814091E0F0B066D55195AFE3ECB624368C15EA6C4CF381F59C8ED528BFA2A8B043BD08C3B04A2799AA70CDB08DF0CE37C01784D2C3B10E4E7CF05C07420C93E3218A7F2EC160A110FE8F7F0694F3C95FFF807BE14E0CA60749D31C728277B32CC70116A46B00388619DC91AA89E061302330B418F0C49E14537D6029DD64A180CD7'
	$sHexData &= '3F488DE0C1AA9FF4475F83E8C8898046CCC10936D87F46D07D46D47B46D879121CC1D5E4E081017D12981298BE7BB67908AC10987477C7D9C8466607D8BCDB46CCBA4C595946D046D4B8C8570CA9784C5AE8C74DABE010D933D2D1DAC8708B82059584709D5EBCE909E8AA65D77FFA04F44CDCED50EA40E73611E2F7792C2C5ECB1010E40A471C9BC6DE342D47204487649E260C21DE21AD081C1CFB12AD07D01A1A1B5C80367A434641B3132D63C43EE34B109919EBD659AC4721379A64AC0E1F2F504F0479C5DF016FE8475E95823F7730066188A41A4E01EF2A76833460237F3FD4E4A89A139C7EC485030A78569E4885ED61B74B53684673D02CE744FE1D2A51ADF0D0E8CF7BD40C7D89AA548C7CCB13AB93DE4D7BD60CFA8E0EBA2F53DA03BC392B9B6A8A38A70491745293273F23CFC604CF4C8BC5CE6ED7693E605CF76F1E37C11DD434DED66CD58D2892B17BABBCFC4565964C6A55C06AD903E069E1E680E6FC33ED847A8358F0CA6266C40C64EC0398576B3F9E79D06432B14BD6043EB754CE16A721443EAC130EE05009A552A9E858F09938D1939E0EA40818A31D9DA226FB7228425E9D016469567DD4EAB3850971D205481C64A5B97F0FE461DC6196A15AC384C3797FD08165C163266043D070D8654643536901874183AB83AB5361555F303849D529526943F855659539DB11A0'
	$sHexData &= '345608C2C304AE1682320FAC4630A498C69F420A167C159783C3D05B4A600180C12D63627CE99D284D1A30DF0F0A6234322A34BA3E5F0849813868F333F848AA11664983EEF9C9B0FB9420012A170225975C720304053F6072C90607AC23563093763A5A50B703C0C0B0429D50C0F666282C241D346D3332744528D214B60729F56A49DF0FB92D5850FF8CA91C75F149890EC4A68B5A494E3B30151CCC9651E9AC5500680598010518466A0E406D7D1682892713ED420D2D611CF70AD48A0C8D9455DB521B8F07833A6B1806A02AF5CE20244E34C11801FF8F6E7807F3480F2AC1EB151E80DC8E05C683E1080BC1252AB04228DC58647E38A81E699E99D450D2F657D7EB18CFA7F276AAE99083E05CC8D1D037A185D2CC0C5ECA0F2FCBE28FD0BD7465AE180223AB33C9B7FDDC250E0D4130C17216185CC110A300565B12B800278087707B8C2CE203C1B92C4B5BB72D0F47C80CF9731952E8949F531C5C8D3CFD2B0382E25D8619C9DC4C1501E90C450D28975EB8611D100F149F2D155790A241191229132BA9691855FF5C26E2918686DA1A76B7F36687085EAAB2304923C7C604C14D761226133E6C891C56002F1A721740EBC73A750665A3AFF81F0C4654C1087505580C83F16815859664265EDB051C8C560E2BD10E7842E844C95A07D08978D2280CCA40353038B8FE0272392677D02E7A'
	$sHexData &= '1200418D5E76276145DA0A013C24DEFE14E1966A87B214D5B768C1E903F1D11CBA68AB4C033C3E4DB21D6CDC60823F72C119642CA87A38BEC045D38EB208C39F4090DCD61DDFE9C22B4D5E1C0748BF7944C91B52CC9FCD6748475190FFF8F9B08BD421CF8D8D292029DEFD4CE0D5B2BE053C0F458867BBE4F80930692314E995952345C645A5242FEA8D0230F2D26D084BD9C6A2960E87B6F0C9F14B11458C485B4D896681147D80385D00CC4D2A238F0E5D38C5DBD3CB0B79905CC368CB90CA55C85E2DD9FBC1A032721CDDD9D9B1CF5374FA6FDE5CEAC44D45C10EE0552E5C5558DAAE631061C503C96424C83F052E4A06AACF5F5BA3B142462044281A064908EB302855A00424107EDE5C00E150DC14210B8B87925576DC084EEBBBF92DDE944E0A248A5CA468643BFF42B2164D89CF53A3CC060672424D45E6A6CFC930480C4D00224C89344F6F2C00A815384CE95500CA394408C176378B5A00A6384752554A98343098774D001662DF55710C0D8D7E71720B070DBD49CED1D66FB186C4DA62376A3248BB9EDB2E43CF24C30F87A6558130A0A322E8D00C6A219C780B022A24B1F6A6DD8619D3AAC4DD0B8684C983F4B8AC8D4381B4AAE38BCFD03B1B222C064906104865A4D0AE4949AD878606B85A5CB0C08493963B5F49764EE80D16863EDAD081D1BADF504A3EEBD91E2414F8FE8834'
	$sHexData &= '9DC298D61BC67D262A5BBB8F1DEB56C5B9BA25ED0BB8EB104A2C85E73B60F8504843BFC9214F0E036B921F90E702E46C0603CD4132F208E17CED1C24073203DDED21043281C5388103A1B854C1FC89F8410C63450FB7E10197A0C6C84C625419518F6C6D80A1426EC0CC68604F608F792A934CC46DD7F81720B99C486D86FF400D5C726D8672970060B5704D38896E290731AC614EF106D6D223585650878D5224696438243E8A5C4D1803B7BA443E021772550C8326153B6F1969A4D6C9944CEC8A3AAC856E40E81A83D0974CCB845EB4418318A15FC2B38C47C56918034668ABE09680773B2AFDDD98540C9C315E1C8A008E13504EE7C0422C009C642C3C33F62A722314BD61991BED2056FECB6F023A5EFE211F8A5540488BD5483BE8772F4896F222692A6F08803CA868CCF7576F081B4354C2743718C125CD8192AA3677BAA5BCAAC6C1F7FA07366D02715119972CFD1418FA06464997A52D2EE876D645330A614004577F909C40A6FF7058845CC8330F4332D238188D1E361A66A41A29444731824809455879C8705EC8B90EE418764395620BE157081BCB614D8D470150493E0A590F76451054086FEA5161CCB2B601F58006C2C07F466D470754F3858BC148414B08781B0D343182D5AC1C5997DC07C74801380A479D9DCBD59D6996752B326081590E4F5174FF0B66598719AF4CBD0F'
	$sHexData &= 'E90900FF0C6010E4300E90C35956C9D3BC26D54303D34F30945898101CFFC4375A9FC681F1F24C8D89ECE1B58C475FC1A1C2510DF145AE18DBC190F0C17395C6400800252BE02038CF3CC401011A3D1FBEA8A4F532C0DF6ACE01015C4A411156A05C5A6F163315A40F009D237AAF52BAA5DF001B2EE1DC8105BCA5E43C1913C7474C69746EBACD1237730197C083BFE80D80BBEC723CF6720BE3B4AFFE1C4098489300DD40F57309C612EF3883F29CB494F6C2151CF2A328F50E0F2083BB263A269F3BCD61173D8D8B4DA2264BD843387B74E5C9E3DE073238694A2D3112FE544E0EA092D6D37C10CD0A00E844D44027B07058A7B902A94343824B5ABFD490693864FFCC30CF201EC8916720283DDEC6DF46829B42AA375332351C0CC4687B20F2900E131B6520F3DD33184058BB4B20A28BD66934E86D5D798DC6BF601880B9F169389AFD9B1383B9E30AC850E831354DD7622757B03C3823A0F836AE0A3150BD1C37A57CA6525310B29A5693A1253ED9BC48464ED4880BDD5E34DD801D1A1F5ACBC80BB9423895DC859C421E1EDE02DEE7D3158DA91E760E91E6209B2838DC7619D042069093DADAD41C5279398816DCD42804C82BD9D3DBFEDB032147C92828EFF2428690C9382DDB24A79007B6DC9ADCC80032C8DADAD49A26674865E3DA1F690A819CD8DA04D9A7C90243BC94171F8A5370'
	$sHexData &= '48D438502127B48E466E47B1E0D54E1402F8639F7809739D56482DDFA52E6BC90E81AACBDA692D7187F4D0B51F837920F994691FA10FDD27DC227B7920A6B15EA281CC52742D3C7B10B6277FE3B124135E740CF8F71C2D4FE1ADF34C75B088C67FE1D9994E6B50A1FD3BE2901AAC7E3DA8807F5E84504D8BFA498B7034F11B5A09DA0DD7902BEE882F442D7471751D95F64BBFE8890A5FC2A0725041C31652ED1E61E36CF725487A1DB10AFF40E90E31E64C2BE3895A0C5C3298AF6E5022C0A16396C0EB08DE06B220AD070D7D770724B6829F2C0BBE2BC227A15A3CE0D07AD0483AE21C237A4CA41D4CC1C3F280CE68488E92510F28F0CE4C88E6F7D418407A224923CF4390A78D2C6F76E82ED4146834244DD6847D085B8715840DDCC98DD446833320D3D448BD2D3949BAC780998470C89F2F05430B5BDD798106E7869D7C1DB00246B5D90E9486F6E8902C0D91D605A57B05820C4F692B249DB2C811760F459B832D47A2242FDE4B5556DEDA1D31C1281D27D6A5D5761383ED36813E3CC4B2839802817BAC06C31AE00A9F6D7070280028C8C1BBE019887CB5CA0E087038CD006349288AF08082F3B689758D57501C18EC8A788DFFC69AE183C113D6579E7AEC18C986F3706801AC5901D2910D2B7FB0310103D9E4E18681111624CAD8B981D2A81CE200029AC0E13BD714750CDE20C383B3'
	$sHexData &= '1EC3003CBA00D98904F7C36CC50591204AF399AC81685007B2231C2D5CD20E589917200526879D9FC0A555991F3F7AE294EA285FE8A93BD017A8DDDA8B7E8B3428070E5898D40D4E511852F6111A580E427018063A00D0FC78204C8960E06889809B3453C42832091D28A396CAA521827402BA05BB08A81AA7D944223D3502831F7604DEC1AC053F20D7D308C9725D44B8D534466A3EE0044E00859930CE0692DC0476825703C275C3D2B1897902004D9BDFFB04F31B4B2A041403E8AC3B7105E05A5417D8D1CFDDB152FCEF7E5DB81C24E9112275288F8108408CE7A7A8C5A1C96CB561041FAE0166275D5F81830AA7387BD431D5C1D20D46C12648BD4B33108C09BF48D5B0E07D4307C448D36027B0033F7CC14C24C1AB82351D40C17FBF068363135444BB7A2C4FE1E0895D007F870C78693C1E09B46758A005A53D18759414DE09B38BC587A769A5610A545500F254E002EF60A2557DBCEBACB574C05505FF49ED414B4E38105F56185B0C00A727E160E83459FA8023D0CD4E203288B096DD0E754096F11152C271AF7FA2D897B452C65C5E103F96875C9A229ECBD838DB1149D33C464E5C812D0FA97A46089C8B1152313574D7BF3FF933F24154584C8BE24D05CFFC01CC849ECB004920307442E800023FB1C2A71E614CC1584936EC6C23C1761DAB1C04468AC5A173EEAEF0C6B0EE9C50'
	$sHexData &= 'DB15C1461466E246F57D821A1050BC2C0D7AA82B78E97122FC0BD607E20FE248878123D6B84AC0846D56FF44088E110CBABEFC0A9857E0C12ECE488B344A051438180660591476800831E76900A08AAA474EF3A0A3844FF7B26EF3FBB3F5420FDD30A5902205D9B2B450B60C0DCAFE51EF8C3815B92DCB2A01E5917BA0E4DBA5B6FB01ACFB01F882F1CDBE81EC4C3BFA70EE56706070BC213729051F1F06ECD7357D0282C9C11C52CB4A1C006E93D48C75017E13770EAE1AD6FC75B03447FA652258BC380D567F19A28395778D51FF500603D38D85F60594B6300569FAFA14E50A340072DB66E7FA0132DDB7E0DEFA9BDA05B918F77A005EAB11D12231A11CA9410BC2A2A1A98F92C4AE1DA4925A47559F2AF4D90C0E8D0927CF1B237411C01F58414BE0F2F9364A15052365DAE87509E019825647EB22E4E9D944303080B241405315E141646D15BF8891D550E321985AF8AB50E540A0CE0334D06C0DF4D280F913476C0CB0872DA536184B4F2081927AB23C114B5AEDBEDD92CE5508038D85184C8F3C98AC513F8100272A0C60313DFFD3DA85F1A9CF57180B906C8F0C878BF8136BC911DAA455185D8B5559A1C15F071E2C33028C6DACB296002221D85AB166A39F99302371351C6ADD81B56164D90C9E3D4D5B80F10E60100CF879500100FCF52249181DC44FCC9706BE00105BC05669465E'
	$sHexData &= '13E5034B704D44122E302149A934AC2CAB0431C10038A47D6021D3DAAB2E7410608614CCCFC4E1DCB61F09284D03E457D1496BD7489C713B318E33954400BC3526B9BDAD042762AF8C8D344189C51BAF839269804C4C82302CC845901A9EBF09018B02E8EB2B290EA30A3F948632486040C9C0959CE42F86324C4398344D8A4949E1E069C8DD784848C67905469C8903E42E000B5EAE8FEBD605D730914BFBBA14C082C80FA7C82AFD58FF60F6F3B99F703A34585C68808711C80456F6807972001373780107EA0B350D054B0CA88326DC793769B181C1B446CC004BCF1050AB27593C19A73DBEB09A8B3D39513A2D035A3BA3017725977F54231F0828B3384C8BC72DC1CFB04EE69409807B4880D4AEFD94C0EBAD3206A9B0A7017962301FDB0C0184BCB692485FBB050E0158C938FF062097AF01740C005A7985DA7F67E0068BFA2C4F5E856B6BCB91C9A6869C4413FBE8EC1430481F38282817FFE97FC6111B9856BB31B5A895A5B0F7B9FFA5B144270AD66CFE0A688F2678D11189E472F66F43CF1049CEFFC1F9046C148213A8C1076749BAE9A99261893A5DB8B1C46A01E260FDDEAF88AD6D9E43F892F935DA85FFE6D2D40AB93DA6FF292F4C779E4FCF44A1F54AD308B4A3345890E3432B050B88CF491ECF83E1B5ACC34DB48D34191EBD7C852931494D4CF28B164D15F1407D9005C019'
	$sHexData &= '928782FB3A0E8D37824D083814C22BC7E38B060409921F143A488B5642584297105A1F5DA0906848DDE5AEEB34B4F4405D8C7805618DEF89E8515EC34AE988C1BF56DB49D37CB1484D8BE869B871096D3EEE4D8BE1FD04CB7232B25254E88B89C070062E5682043F74E15FE4854B8B8F48FFC5C011B1BD5858434E49F833C8013954498B8F8B484A248A571424514A50DCA210860E5F89560849A5C57205703E5471C01AABB849492EAF841F57794D2B2B43F32484C793FC09298D8D227C4A3804C824688D28538409B91F442C498D33742A94285D8CDEBFF4B1A8049E4E8D3C95F03127976E8CD4401D14BC8D7C18007E38824AE809B5544A13BA99301D35CDB40B198ACB2413A1382A21C8D643685225AC408F425FA12BD642013F55142D24ADC338C6E93B1083541E8EDF7030E411507851458EE5901772008EE58E8D682F2984B94FF23F6461B1B043344335029924B82C42A49446E1C53C474B5D9721A3702171CDB3008A9146A875075D00D286642D1CB2D68B4E29DEC9A20F6F1528799267904A3070385D08892111A60A54823A410ED27FE9019D88F62541C6191919414141DC4E02CFA0402C502806D9783F096A411823004F02E7D8286E9D04DA2BB6C03840517641A0594828B0384C9865744E304E31F92CB0AA8045A9DB8A08C29AD17FA55E62E3D8E9AE39BE9C1A1F053657187A'
	$sHexData &= 'A41969416D1F2FD2D344CBD5B238CA1AAB0A9FEFBF1E0A11A1DF7302B001D4BA8646F8D16645C4DC021B92385926A8237BB2DA19507BFF22681114F845C0FA6998A6690AC408C8969E46D3A8CC43A7D03CC15D059A07B20C7A0EE06349CFF310E818791B169247C0A17ED24718E88F202DF741FFD4847DE9A0F9AF915D73A6FEE63FE8109A665FEB96C8A9549E2A0807474747BCA062DCA404186EEDE32F3D4FAECA8B4544060AC43ECDD33646040CC808CC0CF30794DDA28ED0F382EB605B3E6E55D8F2DEE90AEC08BA24385AFE1CD73D0DD008F57448FF6E06A552A93C46464646580771BBA69FF7F2388F2438181C75C22E2D2BA4C1081229272548379A48B1E53B49D0CF3A40FCA5906C8BB1B434629AE16A244A01E5F90007D4C41F50252A555A6EBCDA29C328F1B2B12BD52AD18995536207BD94CC2BCD2B690953AC05A24BFCCA4F099F94D24C8D62498BEC493BC42FFCB7CD403BE9358BFD49C1E7054D5BD9D21C79F6EB40492C49494AC12C8406473D422268F2498BCFEA481F421FA0C1FE05A8C1E10599960A327A08F0040CD5E72815313580CE082B09489700BC9731EDC00618001C0B48B50A106AAC37A54B47160F6AD685D121904676389D48DF55796A646A181A1A3150DD23034A8910DF8D37D483525220386E753BF06303B2E605809434207441BB556C1C5F9D809D1CC204'
	$sHexData &= '200CCA206BE29A18CE1C3C0F92C08203A7A3D39CBCA1456B9E432B49C25B30CFC3DA8C1919D810018B6F974A031B0D5FB962A1810FDD75EFCD0A86423A85E03770E4070A583B5203E6687604AEE8BD8037920E36455048AD114B850D40AC081FD5890A8A27424BA076AC9F2A02EC05EEF8EA06C0E31FE3A0BEDA93896F38B92810B7854263560AFC4262518B4FD9E057C3E5C9F75F388C5FFA9E402B924178D6D608AD7906904D51BF4C454038124777B79101D889A5CCBD90974267AD0389AC7747780A2A084D3A70DCD3B5A94B97493AA3F508208FCF7C28D6321504E9CBB92016704889132716B5E4126141C628318C85860F98D4A910B14D3F0406F9A53AD5A950D23C0EB4C3E25B97D2301C0EA03997D208969F0E239670030776A5079E072F0F30032A7DC05E33CA50ACC2E035F2102F4D6A4140281C334405DE61CBA569FFD5029411E9406138305F331DB04F3F30D0EF07EC94A6031208088F4728C48BAB0F1EDAF3C34E1C38B71BD341CA243B2F702114DEEB66EFD0EBD6C0291D309E7F06AE059E4D9E574EA869BE68EF1DD8305F75D2CF5FCFF086105C63D4562827E8F32B9F30493BDC969C1E55E048011CA11E5A15189F03C050580304442D25A3A59AD645B8369E2A4961D049B953750AACE1B79B4FCE495481F7D0CB0E42C8BF0F6909E09B5566E495FED002C69C3BC24C668D'
	$sHexData &= '04604B4D3C027F5917D8415185EDC915B02457912C4CC9608CC913488BCD6CEF050DC124752A4DE249C1FCED030534CC71E1E81C5120859A64E33A5C4314E96CA92CA1741630C7110B8345B57AD332B5C4EA2538207BE54126941B3A0C5BAA560806B6CE1AFA75CD62B1A5770A7465695650D85801AC089F3CA13990DF93C0A90008050443B8D92D51D3C440284148C68D4930AB802D9CBEE60EB630C8138A09E075B08A642AFB62EC4004549A9AA84C3BC7745AF3E6505A4BDF9E51C42E1992515212DD3989B02FB275B1DDAC602B4EB02003C6E4D0767457DA48331A0D9AA8BFDE00E269A340FCDC404050F08D75F2C7DDAC75B27A1E5918190D4B72C8DAA886F02E1390D1360E6B08D6A091B34358A7A4AB9E0A5654C78836F94EC6C7D38204E036C66000E408F71C2E0E043719F52562400DB70CCC72534C013F5947C0DC4D8BE036E5CA919C079CED027424689F113FC94C9CF34C8D7A67F7493BC7BA8CAC35F00CF1355AB51BF5DA00D3F6C5977C9A93FE4ECE9CF39CEA34B990ABCEAECA6FD40B1B02BC8B418B0C24414639F510EF1730B5097B0E794C8BC547041E23A18DD7C4D525875497810214474702122A605EF02D496DD41F4CCB310C5133EB707E58214F5A1427E3CF0C3B270A98C09F54AB467E50BCACDA3D0F482BD50408983C69F19F5403A514C095DECDD3F83056D8263D'
	$sHexData &= '057A988F004F9F5A4C68642B40FB48FD8EC5E7348EE124E09FBAAE051C3154248CB129E455C9924C699F0D16EF8AB1ABF89249304263DC31B8890CC104C237B335F6A620D7C644D708637CD651681B0E4E3A0D139E2D22DF01FABE019CB848097413F0C642F8013AC6CD3122E504EB5A1C25649762077F5F485EED2CF7304067B0FACF799AF613E41C905E03490BEA549AA7C34141DA2454E9E0D9BDE86E7A8E4A3AD0DDDD0634D411D58949FD3E5A001172BD30E8EF4CEBC300D478E37A1819971138FF1AAAD6B55EEA925B95C1711CD03F55DF9CE4EB7207E81C05A2124933C871DF740564E9D8A003A2188FDEFC4CF2BCA4C4A21805F1D0013981A218A1CBF89010628B670C1AE8F1DCD4F09D27D5646B3631043EF7516E31100950450025104D10D254036624004D454AA30C7A4D007F294AA04124E13C66901F90401B68A4E742E0E9AE9B4231D849CF09E839728381C6A176426437A0855F9F829E8235E81C7BA6833F699E77EE0169069E49034106799EE74B494B430B7532728383C65C169408068D1E7FA441471F63377B914519F425BF937B012BE4C9745D5F1E85BC8A5FA484A48A0E801C42A48AA48A0E14C94BA43DA44C090A994AE04F3E760381E03148B80EEB3018112BD615BDA82534F20AE8712C081EFDA7D32990931E2DCE5336038C464EC0FF26045E5A64050B0F4E5A08'
	$sHexData &= '39B115A7D934483A88E4B04246D21CA0123605FDEB65A7D92E988095DBCBBD0469B22558ACD4486A0252B40739DAFD3FD05326FBCB5FD9496BCB3889043941F33CEF6F6E4439040E08080C83F63ACF0C10FA0818A4E03D02DC39280966471827A6496656024F28304C2846464F8130036F184D07CD904AFAA1DA90A69DEE618DA9575657183A91C093EA7F017864029E94A67E640167436F981A410E49E6693B7141D00C0CD45232A01BB037D8BC14DCF776486AD806E0F640C8210A4AE2B7DE324AD84D743B018F54DEE068C8495F6BF387E05981387AA79F720366904076AB62FFE2FE0DA7EAC0B3064818380ED911B03CCD747751439F522930181F1C4843CE0E60C11BAC41D0000D9D0E1643BB3815B1D008E2D97558CD03105058E749983E84C0E9DF90E7784C8B6CA014D3415E5E9D025D254C1F552539236BC30DE3BCC12BCEB91713420C740C3020833C023512D43FA9C8CF25715708F10244176F8AFCA928A374242099A9CE060DD84ADF8D7232C68CA8991D9C98B6C93548892F3994144E85A9CEA9804D5E0821D7A4CDA735C83667660154D111D663E903293F958A81D7FBCBF4928C838AC9682D01C036913C2A9074994F147D4922C94DE10480A0FC8802A94B8D0CB688D840EF908A8BA38957090974D9DF284970B128AC025DC05DB561E26D0A3F48920DA297010DA3D4FDD715'
	$sHexData &= 'A0129934D12D2187DB49BA170A432AB631884BC03B2BACDB6205D848CA0D761458019F45BCE8D7385DD1ACB099996AC2F528DA0A8F2247ACE19690DFA4F938C02EAC0AC4CC7850CD6CCA41484942BE157249E3BB458BF5EB3D205EF24AACE1AC83B3B40A6B0FBBF33C1AA1D4F1EEF78F4AC924FAC82E0E1708C39202B8E3898A20824E3C2C1E341A4758F466487248C4CD2DE9C426356A96E319B896495B0C664A1428754A4DAC19BFB1F17584402218A984770BE2030E069E2B675382B7143E4097D667535850F852D1EC8543BDE9800BB2449E7F2E0FCC45DF2ED295C3785061E00CBA183674479AE7765DDF579F4F076FB990E6796F2FC7FE3AC679591D7A357B3E744140E1D876189FA3F2D2CCDE5051055AB9A0F069859B8708692C39A2C193E0C90B895D0864958E0518D5DACE9574B2639C22971AFEAD35267E0D35CB949A630D631067BBCF160C15C0A7C105A5A6253F1930B185D1F204444F7D494A4923D15B03162A618F9A5A30D81E025618D7A721054E8B9BBF1802101A54C62A1F200106D4EA97879800C6DD2F4C3B48B726F14320740797EB134D2B61085ADDC9FE99405DA2EB95C1849618920612DB5368B5B088F7BE410BE858035B5E66EFC60229260E15FB0CB570C4BE567B3FD54A530BC1F9CCD0AF358B9B9100388BC0BFEFB7B4C00312A8B503B5E0C43C81AFC16A4181'
	$sHexData &= '3B3E7AF5738CD496B5AA24099AF02D6C7708AF02B7EAC55475AEF586010F214D2BF7DED44E00AE060233D239E00253735AA8C2B37F170A4E2D90D91E93F071F4118B68ABC5A5534A18FC449604AB01CF48304738D99FC2550B5F551F3C64429A61FFC4B0CEC997749E14486828202EE9223373252719F9FFEB4835B0D469C5D86213B07D1F0236BF2E7BF848896E3C3C2D0CB1499F064AA19774459158E9A4076C91638BD5ED0B0D22BEADD02F762103503D2C1BA82B85B792D6A1DD5A183990747C5D6AE110A957DE563F412A0F99BFC7494C4DC900C0B276B36BE14A49F08964344EFE5C9A61CBCFD4488BC7E0C925975714488BF9F84F9003C9E5B27CF6B27C62C02C9DB2234B8DB5D0203A182C4D458595F9423024CEFC801148F207766B4980B0E45F18FE488BCD0FB7C01802CEB5226580A242C07CE64E4BB8C92119332249CF54E0AE15B0C9C8C1D4CE562FA7745E2D36890C56493788B1C061AC94455A0DC45381A7E492A183FFFFB6590A2427B3E4955426B9EDD7F817069003C8FFB3EA6B3719249A71F959FFA2DF96344FFA57014588243EA2213980E9443E0147231CD45A90128FC7308A88147F1A1123E9DF387FC661C9C943B574161C20A782E5B57A7BD3E160902C3EE63288B8C1E1242EFDF5572D2EAD8139C5B4CD0FC607507C29AA3C052B969236241AA6D69F91402E404B'
	$sHexData &= '640CA315521E5DF17020210507AD124EA620B1313148884C14BB0730F7B255597308E951C912564C60BF6F3A65F5FF09914B239C12AEA00FA3D19385404B7F9D4A2964DA7F30680B79543FBA890489FC25C0516FD5B7BE23A08E09E4DC89E480BC907BE09A699C7C1EC9234D9CB701B7C4AE409EBC839AB7CF2122A086225E7F9C9243F2669A239A4E9A5C400D91908A1B9C906CF85880BFBFC23B7C5C7B4BBE8D84BBAB5B1F0A4F2A807A633725B7AED8A4990E1F6C430BC6431C74D8AC08540CDC1243E20A011C761B480DDF80B96719A27834B4C3FF878D64922B28B907E4452D69E098699ABC401E214D9AB9C4AC409EBC8398B9CFE46B6951FF5F66983280542CFE9898720132012828A842C85028BF036F2011974B30BE183426017F8D590820028B3F514B38761BFA21C4FFD9BA48042C06A55191DC244EA31F08C39FC02B581E998B7F4940A8C953806B3597001483C011CF7F82C0801A2A2E223134DF2096E5D100A125045A9EF43B29D552DE386645127627099694305E104BC0FE6B84D21E0C693616107E6993D274524B3E0825C048956FF629D802152AD23E495D9D1E562DE0405A07F56444AC789950EA8B90A1A2E748EDA48A8748F926ED877412E4021B48C3645E5AF240195E7E599078C818DF5841C75E69A2B03C9223ECFF4F3B9007A4E6D54C0610184FC1B1701EB9D418'
	$sHexData &= '182FCC4E3B34411CC178A961E62E21A0657D18D4D1524DF397C31866900EBF6C29403D58E33E3FAD772CD55378477D991670E7C3D560F5464B38405C6800EF6C0624B0189F00568A748FEE7478B53813430555A17160DE7FDE2ABD75376B571050B6655CC88DC0B5260E044600E26D894D6963C5276B635CD60ABF67A2A38062B4F11DEAF2414D0D0377BA4822547211487618CF05CD2E496BD003C53196F6A0CE14C64885E500BA1E4420173651C8BF77BEB769AD2DB5D80A01C949DE0E698C838B1D1A9A88348AC0BF9625A13E94056FC7BFBE22929C7C48D5FFEB39255CC870C24CAF54CE9411009BEE07DBEAF58201031137402866E82DB5B64C06565CEE2E2D77EC31FEF87410CCC0AE43E883C70806F49F410FCBCBF39307780AC02300804DD09140D70BCFFDE1C3C3A4306604A44F0C6D506A7FD6050B225E03BB70BB01E7481B4653A357BC873F0F7A69C0560172FB087314B003AFF15A6E3BC3CAC1DEF05AC322ECEBDAB9E14A835DE9807435BCBCA6D0D8538D91CDC1D80CDDC996ED27E45424ADC1D8C1A883439AE6488D68E545E8684B8D24B856BBF2CEE3A5C24A0594F5D1E90AB44279D1202EAED94AC0FBF004460E76180D19684A48A4821C6DB0D802497259B6962BFA40BCFBE19C5A365115F21A4C03FF2861DF2AB570F36C8BFE4BFA4501054703309E3FD6E8D40E2033D2'
	$sHexData &= 'EC03A11800DC1B91CDEB34EBEDE3CEB69E05A2361E0666F23369AC4A0E06D9B2506F28DBB38C85F40D77003396865075504D8AA7E894CE644F011D80778B83230D39892D0F17E48E704C5930333E01DE3C062120A7C3BA8581E455C73FC3FE3C0079C874BAC3F8C3F8C823E485C3C80EBAF200E455D1B9AAB98F404E045AB92FB92F4A1E211CB9DBB811A3E424C7B8881497F4BF11EDE629A5C1C5FFBDC700C9656ACBC78E3FB5A7342546587E35C2F7ABB1A185B98025C555BA121433942E1288004EC1D220638F00EF3ED80DD228D201393989BB0ACB334ACFEBB8014E338D67D7F7F18C373585D39C9035F8B33CB9A6F0FC8D00340AEB0E63F9AE291D5EE18E78461206D09911FF5223494F13C90EA134F00D97910D69630AB41963401E0E691F4BCD00586E8B645866B9E800900E74424580BB64191AAFEDE9C84179B279524EBB8D4B798D9F8B80F12A1483776C3E2FE06A5F288DAA23C7169A6E540187653F2597B16C209F30A85E5083A885A2C59C1C49C510A4C5CC38855D2296FB1441033D403EFF101EF2212800018BF2609206CD7396414B6686714072680811F9FF0001B4C5A45421E4CC424D1E349F33C8E700C930BCC81DA74899049682549F0639C38ABBEF740C50861894489FFACCC5DC2405208AEE5B8A2A0FA2B1C9065E9E606E6350D828E9F73A0D11CA85AD2D5408CF40'
	$sHexData &= '08BC13F5B5D148185E1E28D2A0D5F3303886E0C61A3B8D05DF00109073CF6D779005745CF14883E8761ADEEF9509384102768045A2BD04A54D6FC6409AAA1B30083AF67042D5C83298F1F0E03101CC64F3CECE6AE01D646AAC264CA5D0780F1581C5C5E2FC904FCFCD2311194D1A8A004D974772C818BEC645185D6DAC061801ACDAB85835E3358662185862FF88A152C24ABF8844320468457478E391F2D1FE726E7807A3713B50813BD67D7BE2268C40BFAE8D57385F266E68EB8D4F708C03DF10FA00F8D7F6D85B1BC94C6BC58CAC2B0366023C882641404EBE8B791BCE0B548904388BB638041008069ECA53080C0C101343779651941E418BA30FDAA0EB707C8E7A4C4148C4F73641A06BD5F2CD943AC8E779E70D008B443ACC0ED008BB4D789ED40CD822E089AD03D79E5A18004084A9B403000ACA6DACE57DC0C623A2DF8D75C2C053C264CCD481C1E1C79E9C143D6F774FE89A10DED74304D5DAFBFD1414D3430CD14310C7C1705BDADA7CEEBEE67CDC2A88ADDB38567B5446B753464646460E83C7BB182528C9E9A3520C1D92DF57712C9E019CEE6946A1362A0D6D34F94F0295E803D68E283C86DB9AB2CD686FFFC0595A1BF8814D6BE0B960D6C215D010E8AA3471A1D18010662C2E047E93F4070B4701E53C48DBB2CF0B152D22172D433BBCBD9A5BF34D8A1C492BF4A02E72C9C9'
	$sHexData &= 'B59DCBD3D385C66807CB5BB44D9E5B4267049A622D724C3D37B8925390592BDBDEA48D87B4CF17876F67E11D2D5BD5E3D92DB9640249D54885C368A1655BC18748C1D9C0BEEACB28C82E13CFD617EB01EE5864BF8DE93F824C5489FF85309203A4045EB7B5B2346166297E45872070AA38962EDFE0A304387BC849CB7515D642F6615D19421200B78B7AE772DA64F573289FE07630F256CE163557096082FE6C53F572DD8C6E96F034816F22FD73378118D8C0A721DB1DBA317410C61FD793881AF8DF480A6C72C94D3BF4763D1D3B36819641CD7FD5991C6D306C217D3BDDEDF8F50927B857A472C67C2A90AF5B98F680CF5A2AE7A2E6911EEA1245F70DECF6EB457B5F11D4DC4650F06C7533B9597B274DA5C3D3C8EE2CCD7C40B889AC72E3644173127BE1CE4C3618E083D0621FC22ECD3D389B9A82D3183D98099C49527380BC88B07FAD70252FE404E64C5A8339D58555FF31C147813C00F9C1ED242F008A6E7FE88CA92100399A8120481C9693C18989177A008275E9433880153B9068034203739160CCAA16524F49C071F25816B042D0D47D1FB64E15F454465BCF85384D943029D5557797089C080C4C03F338032FA1661A6076D9CB6D0F304E251E042FD4042BD048CC0CBF2FC8F68C45FD5A482B28A67C9C3DDA0A1BBDBB3DC439486A407B47695B91BCB18A8449ECB3A6562FD092'
	$sHexData &= 'F80BBD9C3357972D682F6C00B0387D599FFF1985DA4C5E72D1FB945B7A842058FC4EC841B0A4D01B612FF0BCE7825B4BC7A4CECFBE2FB42EB8B07E0EF3BA2F5BD008DD3BE37C5E072D5D8327CF7357CB2B472813C2422E2CB278A48A98E8BF04C76124AF2EBFD5BF05072F78B7A6B640BE2C5DE83294402A05D8369517D20BCF24EACCCCA90BDB24CC2DCF7C2BF9CAEACC241E492BF69DD38C762097D3CB5B2EB8702ECE94D4CEC1853B0BACA0CF00720079EACCEACC5BC923E4EACCEACC654417465B5BCF07C901E4EACCEACCEA0B19A00BBF30A2640890D030205DB841E088BE301EC84032D030EACCB214F24AD7BAED64924A17BE30D03093973487EACC30EACCB0D832C8D6AC303DF868A4BABF0F10C142061C6CA9E4F392031E6B8B4927CC2E21853C61BC3FA99DC0BB793619A197B5AD0A81DF614AAEADE4383F3026AC41C6DFDAA89930C87F6768CF20EA12F5D2903875068E790047E901E5763416D9187E8DD0253C056850581660F412E6A6CF788B0BFC4BD88E237A0A817B8912C7430078779708346F0180BFF1F0AE00BE57D9F9806A007410D520F06706395F9CEB2D807FA4D25C6E2E27807F4747579A5786DE532C47380756023A15C485580353CB6ADE2890AA600CC72627F366074690D1280C49663DC3E30B38390E66F772856CB5AD3EAF2E6825173252BE031CF29961BF60'
	$sHexData &= '1F665090930CD2D920B5771EAE140E404B40BE489E07B279CF3A77C3787FBC1C904147A87848016319480097EFC1D25006B15D9393E6A44A0340CB4823C3F0B440D200900377502A788424F760FD2B844DC2DFBF990A61ABF1D5DBA7C77C4A0768F47920E9177FC60D60141F0570114AF759740B29885C1424971152058C4A01958860AA1229CC5F08234170747A576AC9C1D5E7564A46A322E101A2238D905C043DC8114ECB3EDF987B843389AF3C0FBDC8B7200C4DEC48D3E03CE7231B1F74C2AF78F47400031387D242FFD336470A0ADD0BC760280D081F53EDFFC166E78D143F7CDA660C61527E386247463006403132D10ED1C0045C192F5ADDF825849E4088E72D49E0440F072B0FB6413E08399610FEF8124D33C74CEE96934C32ED2012151314156132C92416170E0882EF9618152328E614B666E24E8B0CD03C0E8D1C04A208C502CB9A89149CBA436A54EB7549843B12DA4E0E2A40106E10D07A43C0D12E383BBF1DAA110C42084D891049601296BEEF51744A1441087938B77B515F880F527B74501C699EF6DA75F166513A120C4A0140C29DF7DE6C4806527C41648CC6A7824ED3DCEC813E8144C66CA5E50583568047C31A11BC2A8147EBB3BF15AE3A1921F9277315506143793A1D4C8BE93ABBD371A119B5AD7FE77220EBE303A523BBC0086878A20A6D08BB6108E428DC72CF'
	$sHexData &= 'D251D852782E99D3E3E5C3491A581F421B0E20F0F349895D43FF497D2B9CA88F7D423FBC75520DE1E3E09FAF0F700A7032CD29D006A7801FD25FA4B10037C87A8B2C381A505790A6760497855F1AA24F8D0C125C29E00AC4D0F41D38B934C37F4201484DDF81EA2804302EFA6F037D184939FDBC0BC22F75AE3F49897FE07F49D2184EA3C6086F4812B207767732DDEF340C5629506B164DD669B4CA85D20DB0C2718578503D9D0300A657261E401758329301E3A98D8F2F38D00C56B8606709DFF7EB7C121F7453B191DAD3CEFB722A05DD4F28207606FDEB088DC0ED9B28367573A475ADFD81235A6D0C19DB1269B8D61C4BFD3FE158294A85BE194CF5DF1C721083411F558B54808E6A487BDFDE301E28F05D6CA3108F0ADFEB907F03C2341B45F9C1D800E310662D844311584EC63C2AF08021E41C2BCA27583413BFC07643F321E33D1A34CB49D3C8095C3503AF60A570C4721C5F28B9648A1F1FD5C48B64ACFF7455801DF1C3A875444B746F259F03936DFD707B4CF094F6885736E1D92771768482D9B924396A8393084FFFB9BC1840FFC59C28790CC22ABF13AD8142C803C801DF91230358DAD9FCA0D148C1DA0F1782B595047F6DB7920960E48F0DDF661B1999DFE7EF467018136A7C95F6463075B7D221ED3A4F57284644D7E9994A8D32622E4A19896CD92A7356AF5304BEF30648'
	$sHexData &= '8BCEE64180680C8493EEC8009A0818380605A86D065EE0863270492B57404024B9835B46038928FC2BE52D39EC41E14158FA580C0505D1067FC2BEC8683B20757CFFD891015C48FE75BE5BEF032B693CC7EB201C5F781FEB553CF70607674DFFF20F9443D03B0CF742FFE87C0C003F56898840019A88480138D680FCC87414037107577403B802A283E7FD26FFEB0DE05847DF091EF07C7FCBC796C6400C000D0D0EB78D5C325FE886C467EAF16E8C3EB96A6E05088BB5A131AF004F45F765006807103456D23D83792A421686278EA696AF0E105A83AEB8C4138D279821604209879C90D1D894D41A4977BB4E11F84A9B3F33DEA6AE700A74E858C2E4E45A39800FE96636448B312E696B258C338CB68B79B73C816B4E717B1542E51094083830E35CBE58C140488B48094715580F5644A20A4518A136248C762C00B42F223B02418908423CCDD38C9B0C08080C0C186DBCCFDAC2181410F7437300603876774F0E25E87B6D8937346F6E77687F0C9A042CC34B3825703A5AC08D00D3AA6560DCCD9F02924350E87D0839813CA16B19A71558951CCBA71F08481E41EABD0AE401726169F1A4161801F213A51C912E39A1855C55391A1E6CE080B67B4A03E3B6AD4927273760F2F1303840028F6E0A0B4628A3029EF633029CC20625B6D4D6505442A82D35F13CE869C0DB325427420B024178A1'
	$sHexData &= '61158D0700937FC0920D56A38A54280552411F8C6CD458C7F359B2F34B0EC84B66C065496747B98358B52D37F7008E4626A33A30ECB40C6F383014094AAC9D7BF9629DEE7DAC4FEA5443A68729F31639407D3165A78DDC44535625DA121488241A0C218838955A110D5F5D272641583FAF9F04278513FC804B380BFA0547C64138D861B6ADA9DA4039047908D07366B779A7B44543680678838861F2B081EF7B580660C79773F76AB05532502B89BB9C3D0C64F6CE48A0A866B00DC07A40CF6FD03783E0886966764CE1D218F0F2C97C12B58991ECF8EF32943C2ABFAA5238853C9017E063696513F20A394D655065D92BE4C92128C665C9411EC951637663346374148A508EB579BD507C22D00AA0CB2018933CE4E20620CE288B64509AE7E279066070C84048800696914338848A15D2CBAD48C8412E2397C8D0D8DA84909091D4E0058C10605F2CC320C2A78B61FEC50B60B835ED104C2359007D18361E2C58C0A239080061321F906330F0D75B83500A649C8B504120C907DD14B9405AC94445DF28C9019900FF480952E1800F23FFFEB8493C5094A0E78CA1440394373A2842F25329F9C6107E795C904F1B7C0F848A127AACC887C096019FC99C0499E156DEC2323A2788D3560368F59C835F45067FF90FB6DA37A4087A817A84DBA228B2605484DF7E19590805CB018D7941726420FFBC'
	$sHexData &= 'C959D5B0CBD6521B01602AE483FF819DECC82B8A817F577FB12C09C28F47F30C5F020A5AA575513F49611F870D01B500D0FB5789E02B9FFA1C3FD8052234DE969BB3FA7C9F35A7215D9F2885A9082407363C9F6D9CAB99182A3F5C1FC0D566190BF7021CA1C719303D3C37D0BD230B718CC6B80BAD05330D1143CB45DB1CF44B383B6A73E232D1DA7B4718831AE813580DD947D085C344D100299CF86889823967330AC63816F8802325FADE5F0E8C2F81314A6DFE5DBE2660A7C116F045C3EC8AC058B341546C032712C782DC693971F053AF13827125E402D0A4E0414CE30F32C3EBD05F03CA072306C79A4DE01C5BF69B1B1845603C6AF9E23671E95AD70765471E10C55FF5FBF39E27E415202E5E4E5C1C2A7924135EC15D724A0E91FF986B5DA7203945279AF7ABF20A9023BB957E5C953C92139E5A635C2587480E115C179C849C2239BB5B779850402160D6D86CB45408DD0E4211C6F32CB11E30308A4460821FD9F5E10407301FF46C1B2B080DA16087B3833F96648C1293726A2BE420FFF61C92137244A7F61CE411C901F7CC8F994F58250FF7CC5F1EBB0CC8CD98FCCC3FFD9668437240FB0C257F2C04BE10E05C795E82C7155F792F77790E5CC0FE56EB87587F69B89101476B377401D92F6B7407C70E3F37009CF6D4C38CE7996E41DE12FFF84C1F7C21D92C4C5EA94B2BE431E2'
	$sHexData &= '01DA0CBF71919CE4946C7E2954D8ACB998F22D3991B9485E59957D2EE157486341FCA4B44D0C82F44CA7174C3CBF41B904891D0466EF033FA8A4F0B59E8D5111AE55E3D37A41ED4512208BC8FFE66601DD080474352D740583F925752B9CA54F48F76F2E9BA7E97938CB03D554085739216DE354AF7F0078234A6EF058FD918D4B58886DC24181E2800F5B07E6D5261D80F0CB6C3C60A1808D0B4989768D431DDBDA00104BE8368048E0B49F27C8458953E0C470070F278C0FCFF9FF7482E0442915975307F8BE18FA6977FB3E1E76B9E9FD659C261A8ECE80375A4F809D2994C1464D25E14346BF9068108AB49EDA9F7750FCB482A5003004244158E0FDCFC814D857434F6B2FA4C24462D080F2C853A7600BFF8BE918EBE11739C35C2BCD5322443DEC2DCC30F0F875487BC2CF0AC8097517306F67E245E9AC28A6451E69933490A20C1A8444EB0805E12EF46444E738A41140BA5BA8701844E0FA50415E777B71D9D4A1D98CEF520F9058018EFBB486D05CD33BBA26BFB7181878B580C6234183C9F0A7B02A7D75FF3830703F68A1F65C100F44C7811D1A0CE11460122F855436C3EFA6D189E226148037C02A0C39C8816F68520657245E05227F4A3AFF7B825A5FA9726C0A6CA0C2794831CF56BE86849D16EB76E7E8601FCEBE01EB89033D080E86598A062C0E0BA724376923970D73A738'
	$sHexData &= 'EEFA51200C494D2E975224241D504B7A93C3D750912FAF09A3B0DC4001128BF8C7A0016D418AD86E025D30DA6632FB92D0A163119410A94B060E36265702F0F83F73022283C8FF41BCA7108602D837154474021098A0CECC8258F2EF286987525A4A106656209B4AB77B442A1300A34D0D080F85D9F0B072235DAA6A1B51DD359C7650123918203ECEF5101D28ED4940E2502DD83E92C9750B1616510D5213EC0C37C050EB648CBA45D47EBE6C441096EE8872AF78EB1217E844431EBC9DA13C4A22C14523C4A805450A0410BCAFCC5ED593048CB4DA0128A3FE43D148B900F73D7B2AE1CE3E663C2E14066D02F5D05F0D13F646144F9D86A06CFBC54556CABB36814536F101231209FC4944C1C041C1E115E31C18B5316048E04F59F0BF0EE4C84F81608BC3BAD241B19F645F2601AB41E0AC0D1E6B4797789CF641C04D291212F2361F034116EECB794F6AFBC0415754C6C7A3838143C19C52DD0DD48AD22E1C0781180A47BE08D9446F68F41FC28D239743B622C2B04FB217A5FDF05036F553E20CEB07C71BB4838308671EA018F784F98C3423DD7453AB28E96C2AF83D18D25AC736B3BD4E1A28CD7D158521BA9771309AC15A6AE728870C18BE074D00E6401D0D6988C89E925E7440D1A001D730401EB0026148A0D76E904E26E1B832DBB40DA223102F2AC025E500E3D8BFA607A191A17C'
	$sHexData &= '9BE8079CAACAAB20034E8D47070F5875650E548BAE75582CAF834EC9F1510DAC6EDD579828E54E368F5770563DF0D5050C7430BDB8BE06FEB0E6200201388533F6C646DC11433B40B7405DB8A065C2DAC8711013FFE1273C1365DF4D317FC645B00164E9F3E08975B4E54CEB39992FDBC1DCC904BDB9CDB8FFDEAD2021408AFECD72A0408875B01C22330518B43E5B1A4D017311830486B24901ABB077606A40D877E97B55D80C0760D4666E4D63685B85DB160831741158740CB706B32FF03A034079FBDCB1033FA27F02C075DFE8DA8D7823E4F2C5810804CAC85B2A4D70DC69EE2F7D1A14888D16DDE4E50392764C20485051B5736F5745E8A8FC7383F00193A06AE80A3E183697824DAE09EC0DDE70750965EBA6132E834DE86D1B0CCE2D67840313097EC08612F4B40527041B330CB895DF7A4C1D03D3EBDF0B9A9BDB03741E080174193274140BA1377F2182E83B740A83E826F86A750288ED43A5AE4D257A8298500007158E4DF72034E5D47CCA4040C5FC3492F45098246E85DB029A9B6922B909042E28A0E8122826224BED21A1E16BAE6F0DD60F182918BEDE083724600A1313C0EB081560102CC2E5E030C553DA2879E0E13EA479BA3582D46E4B9BC3B8037C8EFD4B1B269EE6F2160FB64C02FF01B08BB0C205790442803CDE441D3D7026C875E7ED1A6314F48137E42AE2477775'
	$sHexData &= '4B0E25774DEB15961618422F9E1B532BD636C7502A2670064E30E09B3DF1CF5105C97501BF895A15BB6BE926741AA220D89D0D4259E7514AA80F9CF442548028787EF080805376A76603D172C23A823A193F04FE5212BC3817C8752B63A5FD0FB3CA75EC83E01E46D01490793FD02464A938750C3C0269C59050EDDF2F0158295F51279B304958E0D670390085264C331EAC28F16FBFC727F9A804A707D9A820742564FA30761F84E266131E0877104EFCDE92C4EEC997C103C9EB24DE9B3D63D9101B3040FE0876130A50D4897896C8FB0E0E6DE00C224E2A2325EB9524550206F8321A76DDB6F1240C0CC003B9FDDA08404E01756B745CE969EBD3B9382804106E09C67750A0728A6482FE07F0D94732DEC64405482075EF4A42BB16C5923621A4F0103F445FFE02D34FA4A088C62E3D807C40631F89A9104AE2EB6AFF409903F6098CEB67D114DF61487CC4FA741AD2DE92735B6FEA3069E8C1E4B750993D5A1FDF7313D70CE6B3A10530A1C3799C929ED14D58010E7DD72599D282940C5FB4BE5B046E9208740B45F2EBBEB362F1E93FB2EBDEBE6AF637AC380604CB76D1495B66B6787E69093C0F8ABFC05F25008E4CA228D8C5E161DB78532500C0DB6518DB01B324B83E5FC5E5C6308DA7C757C1D657C149EB501B7401101C7426746E757C95007C04C1F610C4E26D8C09C0012570CB05'
	$sHexData &= 'E0E755F4D262C2254903E4341A0DC81316DF9D0431209E0EA21B011717F190428A716B0DD0BCC036031849B4F074CE896934D8B274CA314D9C80123A2DDCC847400987A790F39DC0EEAE3C07260B10E7EB9986830C51494F993F58F96CE12DA0B6933626FEB21069124C8C4E1307E2F54A09B44244B7580FC0294970440374D5B40E5577090F1896D2020A73AD56826854C0613C43F5D878E2C223661F43D1710084F03642BA097CBF0D562A3A61F071721F4475E39D80F01135020BB8EEBC25E105D0FD10721972881B0A2C2A2788DC4863785D44B14CEB9EE942DBC4C633000CEB4C325133C3D144A841F4A567F5DD8D5890685D4427EE85846DC1E6F07AE70F88F7451FBE415B8147E4EF06FD242D072B0E48271F5E5BF312BC24E74BBED5C45FF760E3316A84FE742B7F9D39BC6F5F0955F15C57CBB60105EF310F5454CEBE45D21141F285EFB405F334F3467ED02AB88931B0D8342013C8877515881C613930175C10070B38A4E02362C32AB282BC759F4C0F53F90A566261FE75E20FD8C614BC875DD11708BA95EFB2036A1698EB071108406C23BD7411C00558401858163E6081BF57014F7264DDF203178163A5C504497745094F9FA00D4C8ED0617C1FF022202872581C2D6C37199C4F840E3544AFCC7D3364549F00FC5FF0C5F1EFC9AEC5DD62DA6E379BE876D972D405367536D222'
	$sHexData &= 'DFC60FE19BBF1647D0C5EDDBFC50C135FFEA4BC2FFBD7D8855AF73BC83E307747FB07A0CA12E1BF90000FFBA5CF7B1B76DDB6671EAFDDBCDC5D5FEB6DBEEDF05CA17E9EFB0F576E2A672D5CC6D365D207B185FDD0A6547C8D6D48BCBF9D3E0386FF87FD8D2F7D2FFC823D07421ECBDC2B91FDADD670178EB19C22214C91BDEDB02C5431FEB0A9FB1B0C170308A2D171AA142F85F5B5DBF0929422FCD554250854A68C9CE32B2516F01F8256B14774FDA14FD780024C80E847F71A08481045D2C9320958072AE83AE3B1883E23FE4187AF9CCADE0010F003212CACEF30EEDA613BBBFEB80DF75CDE98333D2F6001B584401EF055C133A7D66BE0391D4A20E9EC6FF117C35E07C86330E85E014D895F7B9D217A07DD7D0456010C020F31CDC7B3AC818D96B0930C99200BAE74D40D11100B080B9FF24C1C4C37922C2010BD41AB6FB0E0485E36922D1010BD2023204F0CE6C16D0033E0A6D484837143AC2004A31C67079E6DE20368C4E131FC2F800E06072426C31810D68BD0573550C2B56E9A32078107D6EC8BA7DC0202C10856C1D5E6F67112D1A5349CB4DE25A382A47A20340C75CE4ECA14102D4E7D2794027EB073B7C750FE2F14181E69A5481D1DD74745C2A52425529A3887FDFF3450FBCFE167A9318C03AAABABD77BD4DD472064606544D3BC57252D6EC216C2048C1BF8F7B7C0E4BCC'
	$sHexData &= '14E017C2191075C2EB2F6A9ACF634433F0759E2A1DB4D728AAFC080F23CC7580FEB5F76DA1CEFF90825C4C1CC7C508C2DEBE90D7C5794D821DBFE0F179554D2BFD1EF3445E16A9DD9AF6C6A2C7773F9A45582438D80017D632C484BF868B1BD8A5474180EF983D0B74060793824334CE32B41822F2A1F360C65E007F0BF969FDF84BC6EBD59E09E0760E66C4EBCDA210EDB0680D53CAFAE5044208952ED27470E1FC1F05A1B5600A89497F66D8B41042DFFA77390AEF0B24EDB03639C9751645E05CD0200D180A2089B441B0B40AAE2AAE6F796EF763C3B7AD354844FEEB934815A8B0976B10523F1AB20ABD3D6822012F2B0170414AF5610C398D1ACC40E2121FCA12E749FA0E82C2DBCA6C05088C27A44732EC41F40B2B2B18598B083409A5D1CCF68AAD38824CD0BCEE2EC2E1503152163E80D16D659CAC62930EAC8DD334EFBD8F9712FC3A106312FE87EE85A3F163A003D1EFBD6848E307942BEF35D25009234DBB36F057D884ABFF03014E0D7C845F59E6C310EB0FA68D1C4333D41A21E28E6C1380EDBBB674D876D3E1F1D09EF0AAF0187AC990D1EEE7DF999AD2EA9573768904A20B38E143EB7058ACFBA1CB6A9016D62B82F983A5C2519F0395FFC08297AC36F607036FDC3D4E772B510F57C6766B68D04C6E4C7E4480A170412EAF699EACD167AF5476AC0687110E4F32B7DEF33A71'
	$sHexData &= '48C2AB68416DC2C141787C2BD2C2025F6271034DDA74C25CEA03751DF17FC61B0E00044A6639044B750D52AAF02661BCEDED349E7D689EDBDA75D4B5133B269401ACAC1F08A5D99F908C112CFF25D53EB20AD75C2E97CBD9DBDDDFE1D9D673B9E3E5CC4B9F4EE03460210D936801F802352E0123BC434F11EE580201742C3C80B965DA0CC8E143BE049CCD0C3DDA7203E81A8085CE084A1EB6409E6A5C08E901B2AF91D2FF0D092CEA56E8374A12C1C11066F7C15201C3900E3DD816C910B020C4B717FAECB4289867323C1694EFDDCFB32B1624858B15DB31C089057F28896D0B9C820383B9146704D05E64590E1A3810461BA7BB5EFA3B00734359DD6562EFC724CCD72CC6309780AE2522253700CFA6D2EB450638150F41B74FE533A24EFC6696EBD652FF74DED2700D63FB9C444C8A1405D74C687E2A1E06104A3B937CFACD6669F7213D2071521BE4EB75F8AE1117DEC7E7615209F83E21062A20F0DFDFF7BE95161E21184821FF30E748784D288408EA4633C0017C571302B10D68E875EE3730F389AFB001EBF7770710F10B52F71E28EB191F10D1F001DBB5079462EB3522F66659244318170E4C171826E1B4AC959F9723286A5AD38575691219B123EC7F10C0F6DE230929483780351721F0890B40AB50E9C6D6A35E4D75166611E1001B1A49CDC7FECE4083F8387FA2542458F7F02A'
	$sHexData &= '0C1DE15D22EADE658B40D7096865283823D0E03D206012ACBA05291470C30E56CF494D282778787F0354C605217A0163781B6420E7140973EBEA612A1123C1F73D41C08A073089D2756768D2105C6C6A232808249D8F453D2AD264660222FC61E09DE41EDA742EC2336274268EA4253002EB44E946E405A1642FAA0C597046BA1DAAB3487D640C84C7C1017205F8D3B8E8DD245786B84D5A74DFF41550390545D8FDF07848630D78107B337E77C2350D03CA81395045785FB80B024D094161BC41187554D0DACE012ADE14C21814410665A05D51CE4C06CAF9EF842E4449F8188B4A0C4C3BC1720A86FB2DA3037272085A10753FC38BDF5E85D24B837A24007D130A67F8E81D0E060E022D18C3478AD98185640F2B60C01876EA0F42488715AA63718F3421606F9F22005486AA0CB2841D3FD98ACBA53D80FB629F833D7A582ECB874E13190ECDD3BA646311A632DC2302D75A266CFF1D9BD0795F188C48F7D81BC006FFC8FA013D529199D8F718E07220AFDA07EF651882078846C783FA0386A6021799857A0D116303BDC0007C1291349E085748040A54378B1F70067C418A44DF10E7D63D3D8DC20F3EEB0179EBF20811616C824F2F3CBA1C2E83D5F4AEDCCD1123BF3DB00A04571315690402E3896201D013A25458F0FA41BE1B526FFF7A64445810BBCD03DE49C436893BEB74375E802FFE'
	$sHexData &= '1E7528B943DE8E5EA4C00018D85A055AEBAD6D6033F6043619CBA11F360B6863C710C310FF262C8235FE43DC63456A42EB0FFB30842AD039DC2809E188439702FABCD9FC53D07723C81620A3F80FB6AB5C44021FF06981374FE81AA04D9D1D1BB8B0E041BB27B1E79FF5D533756916DCFB1B020CC2BB92A3708AD88832FF8CDBA188881D61E6521BAAC7050DB5AEB47118CFE00ED7A8110DF12471DC1341A13329073C6FC0E33EEF0CC906C9C87BBA295DAC203E86371FFD34DB1AA8B860024032FF99183C1CBA6F4B663FA11C9CC086521252FA13EED59D8F17A48A88C6BA61073AD4D0045A84C29C5209EA7E41FFD1FF05B9943DAE356C16B7B72218686090079097F016DC1F408AF98B05797E607F0D7996800D347F589486BB2A6030337C3D061E741D3669753675989BEC9822605AF04D6CEB5F9F932A7A0A66C4AE2B6048D8740585E859C8F0A3079069ACA44227A24018331A66688EE4F940D35BC2030C8850DD19D0FB0F3915DCB47FE7021D560D244021B8D26B81C363AC7C62A28C6A25D00DF6EBB0B43A6025E041E25E00C2E103EF1D4C60903CEFE3F532503444C0772820F2010F6CDA369EB511E68132DF37908EB48F0FA0E899E608740E44ABD3C247EC3D60E81E3C608BB8753CE374255ACE61C882B04B104F6AFE39E58A6820F6DB0F4B634EE980D1409D094385056FF8C246'
	$sHexData &= '3819A53083212CAB70A2447615F1632A9C09111C3C574DFA94EA6C5DCA00621DC4FFB776699F3E7CD104DA418B0883E1F841F600048BD210D27415F48CF6BFE86EF70863D04D03CA4C23CAEB0384CA179E00DDA84A8B143810E229BE4A380CAA01030F7412C850C485F00EB9F03123E99886B8CEC28E8586CE6866E964D71A184223611010FFFF2B8B81F16E74656C81F2696E65490BD18B3735F5FFD80E0081F347656E750BD39A428BF975CE0EA7F45E25F03F1ECC14225FE23C35FA1411523DC006AFDCDE6510583D6006020C213D701A05B0F95513A18FFC502016B975ACF85F9205A3C1731444148F5DBCAF684BA70144BE8414EB07267B62402BE2DC166F0101F6FD077C421222F6F2E83CEF6F36D30FBAE309730B62025229F034F5027C0D2540FAB35F099B44CAB8BE3BE87C0F701F92F061ABDBFCD18247786142F0752198E0FE62C2B22AC14F2111F44921A4BDB7BB785614C0E714731F48EF342ADE79DA993B201D06481B230D7C001220AC692ECF915AAB341C90431AF012C3C556B0D95C851DFAE320B2E0740890575C3BCDD02003B0CE12D2C220749CCBD3B4623020B705B56946C00DFAA272BDBD3B38F84423D12AFD16AF1B3BD17537EFFDBCFFC322C23AC275254898830D7D0A4050DB399CE069966B067D7C20B770B46FF0487320C5E617730C480CF00233ED4D29006116'
	$sHexData &= 'E113734E9F4383024E6E15483100804DF4D613CBA20681E1FF229A0D298E487DB4740DD9DAF702FE6C0B3A23D088151C80F90176DEF2ED6D30E2BF1E0CA9157315D8E0F422ECD7C0090C35F01F34E728264C5303C36C4B1038B28416C4B9025DC330F9422E5E615E003E5E00D0680D94A74FCC69C90590A28FE70048C2DDEEB150CC39058CE8C20AE83E7C8BC9B81FE005A11CB382FC1E4DFDA0FF98BB32A2DF2D992B3AC375773E03A70D2AFE45B8FB2EEC7A08761AF650152F4BCBAEE07A3117E92EACB732137A1FE1451820F06CDA15889AF2331606F0C148DA7B9924C192231833DB30B8684402C114760BE0BCC31B2AAE1CA7D002CC5AFC64678304D26E14686A2EC978C4791E6924D288426ADC0E16BF5CAF137EAE83081FF81002B3281C3ABF46B95C00EDF2405676A2EF9CE89CF834C8C1960C7277C360815CC8BCBC72B934B36E5A320A0C1E2897CACBE52A2CDE31D0C2B9545E2ED46632505297CBE572E4BED0BA7C72B95C2E96887A8C0981E0F09E32771097027A4407498B2ED20C2D050C58826090912033161516E8865C0459B004FF02FCF12466F6D8B5451B30F7D94403C844854B167584840402882302319E1B9CDFA520C5B9B0BBAD0BB7B95C2A2FB132A395872A9DA0D27FFFE0BFB4806C83783233F88AE556A44F8EE9705C1F8058D84EEA0136C88A40175F3BBB1DF98D'
	$sHexData &= '8A280FDC1F3047338C1FA60D767220BC5F1F300D7D2047262C001C6C64FC84F93F5401EA6AA7510871DB203B246AA5BB533D288B5DCED64E4CE9237E751F4A7D9045B234089C00214C904BC2A7862B5CD5056BB854D53435FF81103C72501BFC401307DF3831F34D40B91B3C33F10E902B22830506E814D11B4CDFB6B169872D294CE1294527C420181E5F370D2DC051EE847535DFFD6DCDB92089352D621373B7869DA92BC95520552F4F60604B4BBD28C89119060C595F40133232C91CBC4028203C7C6C615199296E28180BF2961CD09F30C950C8EA46F68A1F4BF6836530FEA51CECB130D849C3AA786CC97017C3DBF3173B9DC7400E7262331B2C592AA163959C312A0D37862507EC80572EC87BD3922867883C2E3846DE89281F37AC4E382C720DA1148F47707B781F1E70FEF860E867474B6021271302FD12DA5EB05F47080C17B92DD3DE2CFF1F485F6004E4422E6060C855F016CF177F201403647A20FED92EC790E40A70686B485C72100A877E30303B18C98530277F21B9605B9F5F20B93DB5455D7F002D9C3F20E4DAB50B789FE017C0D6222FB9A00001B7261802FB1798CF7F02699CED7918FD8C8D40868110CE5A6B6E68950EAC85373378DE6362DF762E23B4551C9D082A56F29C9D20F72869920B06964BCCFF88935C72ED681748280D46DE33E0695201AD01287C1E591803'
	$sHexData &= '06C503464C3F2382C0B24001CFB985B7E4218D4D68ABEAE1712D8A5B55409CA85A4A5B25C516B561597C4BE8D9BD8F2680DB9E66F04B14361465B0881F362A882A0993BD44FB013C5D1B8CB70F969D66887B0B8FC676025C30C710193F2C45383EEFF2401A118A7810859D642DDF128DB8FE1910DEECB94F85FD029C82979B595983BC5FAED1252C61BF3832750497154D700F79C9E391CC2570004C2611A8262F7000D71C2A4AC0A99FB000930C0497450DB0C5908BA2772E239F8D92A2413401607016D70BE490B0013117D16D1A9F58F97CF7389908024C0FDF204A5E2639DFA855246EDACE6A107F2E23654246C9990E3B45678CBAA81E037F286C3F20DA150462CFFF389FB06023C0F73F7EADE4C0A548001E194E27567242094EFE709A13B687F71773ACB35578BB2B2F1F478831E4926B77202085525A8ABFD7C50A2D698CFF303E788E10CEF911A1378D60212E772623774190EBB87F80AD6097A6016717282F7A7CF53CB456995F1083653CE7EDC43A84C84FB1921462B280F80040B8C8CA21D79F3434180195AB121B51313410435EF29202AC211002C94BC9CB2C221002EE08896A61DDF79CD037827C18C9C80129A076AD32F22ED803F017084721D49277029C5E784CC2C515AA57FF3899189E21B53868B9981C0C412EC09F3037DA3A87301F254EC7916985B8230C09B35C324C'
	$sHexData &= '6F171953F130E0AD324330756B8E648AA58BFFA8387084B5D17B7F81EB7E301A9C10261E81A51C38C91756D4EFFFB409BA1F2B07391D2340093A2040091507E521D8078AA8784BAE699208D80338F62AB6DA1001475F400B77DBD2718A500769175FDF484EF27803F80817A00297012C2C0791E7EFA208BF01200CB80FB6B6D8E9FCA72F1760082215B81D99A74FD28C5C9516B1601060B56B47D513408120B1F91CA3E1300719AA4610078FD2CE2680171F6C2FF3588F6D6D9201305F8F58814C456B2F3D20925780CBDF0A211B1D4A5E1EF2400A9B1D400A92CAC9430C40BF070572141722F7986BAF8E837D57B8067772ED926B1778584738E425975CD818F8056B03BBB617B82F00EF4082E3920505FF4BAE697B595FC7B8ED225AC92008A7F8A7B5252779D80418022F975C72C938985858AE05649205785F98E4DA25D72F38B85F18476B01B9D8BF6DD3924B6E1710302DB0646003F8BF05D60682C027AF034703C325D71738786F0302C5926B8F0347146492D72FB8089808E1C944D910EF486BC28A2E0BD91FD681659060609928F16A31D35DB893B976C9C17AF0D017B0E4924B2E90705069492EB93010BE03E3924B2E030303B5931C50BD5010054705B6E4B55FF001D7C9B5C0AE2F30EF2F10D7D65E7290F002D72FD05C72C92550B03072C92597901070722D105CF0DF2FD0C925'
	$sHexData &= '975C30B010BAB1AA71903F4F1BF5E0561F011E1F528955A5B8CF946825975C6C1758280B9029D078379802CC0A6C87E0221866A2112B3B5F802C6AA45805DFB42F87366FF51E2EACBB723906E04B0C175820E56D2F0728430C178840DACBD1A09F20C40C17A1A15670837CEF456084455F05E303CA4100BA045468006CF7F715626190CB29084F3801289009930701C59CAEDA130F3A1797A81730E938D0B78E190D2738AEB15F0B1C9F0B4A19625B5F5A6C2505365071101C0BD53F9F280F28B4E4206650E780855243DFDEE440A688B720309674EB8DA9BE58202307A31702FD23462E081F70BD118F95909917D863576D93751C1B5D8C5D380E7DF053069A607D9450496F49B836DE0352654F6BB64814D370D322CBCB384202E1B5028EECE0EE3CF31AC74D504C3A7094EECAD2F7A213428FACE115203C4CB56C8C3B4457C829E8DB902BBCCC4C23682C0381452B2948DB6C0E4D504D48C509680D89761ECF3FADD5A0340921AFA6C0D25F17BFD88C935CED24B62C3F6CB5B5B554A5ECFF5F1F93AFE860271F01DBBCC4089C42DD740C45ABEDB4B74FF02C47C71C170C01B9E49240282805AC6060DF0C1F60AA9ADCD7283F126080BC853F5F771CCFC064A466482A77865E48551F335837253008107F13D94786305815911246C033AD9C13CF105558DD01EB9CA4F2582A99BF0125401969'
	$sHexData &= '9E814040314056F264A46F40ED7C382807A75F20B0FED6820905B11BFC9216A25ADB4A604EA05BD19DDF389BAC1058C19FAF097CAC069F4F70E991CCDF501E0C95D4C01061D85CA34B17CCCD90D70FE9AE6A7D537F385808C0414E142D9A10D80E42FE92D7014040FF15B00EB8C364E43840E0BF3EA8353660C981885384C35D318D8ADCE9F8026B9F963C2D9F3C90460E5BC4ED5916C8300B0E06CEBF8B108392769B5AEEBACBD6DB33F0070245708B5568A55C389004978C05F55F650303E91DEBD853101855ED7FA5A753860A7A967E821852B89916846DAD02EE3E8916F99DF7F266A35F1D39264C1601BEF003903FBFE1FDC7798EDCAEC7BF6D781C6193BC402E517D38613BE4DB92383F55FD0039171D80ED73BCBDFC977E3A58A78DFFE695F00800EE531899179C4714C2A1C7021183AA41673026052B64A0383E502CB8B22071898B15DE364F59E92665811CC8DBF0030CD0613EBE70851CC8919E30D534708E3FEB3D536F530FFB8905FA3544E2A22DC074010E11A1C437E15DF410E865FB6692113C31CB10330FE6F8266E1DC46A0CB75100016C15D0010F4E2E1F004E28033E28469E9CDC1F1C0EFE2768B9CD4E5B01C4261FD60FEE9C9C8C3C8E42745E44793BB99DB44F28120FF641723B3919DCC2A6BF9E7272720B0F786858391979724856406A7639393939889CACC48C3C39'
	$sHexData &= '39E0FE0C411E9C9C9C9C2E4E6470B69C9C9C8294A8019EDC169CA02EBF0FF0242F4E4E4E4644688C98234F4E4EC6EA0A3016272727272C3A4C9AC9C89127562E1A318AE464E4C9CA1032528A7232F2E4CE183388C039197972FE403484CC9C9C8C3C0E355AA6F67972DB9E3836AF0FC422373C39391966AAEE32389E9C9C8C76BAF81839197972DB7F0FB40C3A62469E8C3C382E16F82DDE4E4E4E4EAA88684C234F4E4E321AF42CD027272727AE7A1EE6939393912BAC6620C9C9C9C8DC2A9C580CF2E464E4C6297E36FC287A727232C2A47C782E1B3921273FD830E227DA0F72727272C8B49E0A727272721C283844727272725462707EDCB6137B8A27FF263DAF0F124E469E9C3C8C3CA4BE4E4E4E4ED6E2ECFE2776624F083DCF763C1F423E2E76322102F26F826217BBDD2F8C0F7A6F5C1F177B5BF0AC425F0FB642CF3FB7DBE40C6F723E1F360F1EDD4E6E274E4FAAE60F5AB3173BB92FC498FFB63D2F1A4F4E6E27240F2E38823DDC6E2723E8502FAC1F1089BD9D9CC0CA0FE03DDF9393D993A0DC420FD2C86D7BB193BED46FB83A2F0F64E4C9C9E8FE1C3B32E4E4E4E43E587690F2E4E4E4ACC4F60C3C7272723218304258FB68E7769CEFD8010110C00FB1E4C8070223F02410173BB157C922BE81022CA340DD82DC6E0FC0D1D00F90F60055BA6F20F90213AC0FDED8116C197F0111'
	$sHexData &= '12015F106DC53F07210078F7FFEDFC002D007300790E640065006600610075006C00741EFF18DAF96C006F00630E6521020304050607489B6D6F0D07000F0D061E01DDBD2DC8010F0D047E0D1939993F0202026EBB05991F010F0407FF41464E667F0303039291939103030303909193593F020202735B90D91F010F85FF8C9C8C0C040404046EDB2283041F0F0D464E466EFF04040404919391410404049193D992047F0303E464649003030303E4649664033F020216643664021F010FC9C814DC42FF0505323204DB053BFF05058C1C32720505050523838C9C05050505C824232705050532C8C8C9050505051C32723205050505838C9C8C0505050520232723050505C8C8C92CFF04040432723232040404048C9C8C1C0404040423272383040404042327B3257F030303C8C9C820030303C8C92CC9033F02022DC86CC8021F010F8A4415AAE1FFFF191964E406060606460E1939060606069141464E06060664929193060606061964E464060606060E1939190606060641464E460606069291939106060606909193D9FF04040464E46464040404041939193904040404464E460604040404464E664B7F0303039193914103030391935992033F02025B90D990021F010FC805018A06FFFF05723232C8050505059C8C1C32050505052723838C05050505C9C824230505053232C8C805'
	$sHexData &= '0505058C1C32720505050523838C9C050505056D2523270505FF3232C8C8040404048C1C32720404040423838C9C040404042D25232704047F3232C8C803030303DC46327203033F026CB1908C021F0F39D1C335081218020781B9308302940F070276602E2C341F0F0702B0D0424D183F1F0F423581B907024A7F0B0B2DD43F1F0F07FD3778A202470F090A0B0C0D0E0FB0846C891B1E00011BB45B732E077F3F7E3F2C4B4B2C1F1B0BFFC4B2B5B57F1B3F1FEDD646BB1A7F3F7E3FB0F6120B1F1B0FB6F6D642FFFF1BFF1F5B5BEC0D1BFF3F7EB54B2CB43F1F1AFFB5B46D6D7F1B7F1F766BA9BDFF7F3F7E3F8215AC851F0BFFDBB6258708091BFF1FD6D2D2061B7F3F605B0BED7E3F1FF3A56D6BA9FF7F1B7F5B4BEDAD1FFF7F3F7EC14B2CB43F1F1BFFB6F6D656FF1BFF1F5B5BEC0D1BFF3F7EB54B2CB43F1F1AFFB5B46D6D7F1B3F1F766BA9BDFF7F3F7E3F506489851F87FF5B72400E0A0B0A0B1F5B4B4B5B1B7F3F7E0B4B2CB43F1F1BFF4BDBD6127F1B7F1FB796DA5BFF7F3F7E8A9758683F1F1BFF6CEDADADFF1BFF1F608B621B1BFFFFDA25165A3F1F1AFF5ADAB6B67F1B3F1FBBB5D4DEFF7F3F7E3FA0B8C4421F1AFF96F656B0FF1BFF1FD342C1B677FF3F0296251692031F1BFF96B6AD857F1B3F1F6E2DB5B7FF7F3F7E082EB1D03F1F1AFF6DEDAD0DFF1BFF1F6E2DB46C77FF3F'
	$sHexData &= '7ED52EB1D03F1F1AFFD612BBB57F1A7F1FDAADA5F6FF7F3F7E3F2C2A24161FC36B612B2AFFFF1F7260696B977F3F0203B1B0C4421F1BFFB5B46D2D7F1B7F1F766BA9BDFF7F3F7E3F556589851F0FFFC4D66E6DFF1AFF1FA585AAB01BFFFFAB5D62A13F1F1AFFADA56D6B7F1B3F1FB45B4BEDFF7F3F7E3F04854B2C1F0BFF2D210782FF08091F05A12A2C1BFFFFCA125B6C3F1F1BC40241AAFFFF7F84AA2CB11F1BFF41852C10FF3FFF4B1004AAFFFFFF36D8DA5B1BFF1F1BB4C04255FFFF3F6DB54B2C1F1AFF7FBDB5C46E1A7F1FFF85766BA97F3F7E3F0B4165891F06FFFFC0C2024B1F1B7FC442BBB53F7E3F1FB6B5D0B21BFF7F1BA5F6D6D23F1FFF7F2516DAAD3F7E3F1FDE5A80C00FFFFF1B2AB2C1D6FF1F1BFF62A10516FF3F1F766BAB5D1AFF7F1A7F4BEDAD251FFF7F3F4B2CB45B7E3F1F0AD85A020AFFFF1FB79658361B7F3F7E359758683F1F13FFB5C46E2D7F1A7F1F766BA9BDFF7F3F7E3F406089851F0EFFA9C0120BFF1F2DB0507077FF7F3F52CD120B1F12FFA8BDB5C43F1FFF7FF6A1255A3F1F100E0E0C029B0BDBDC0C0A07020A080F07020806DB5C58681F0F070206043F2E2CB4501F0F07A116AA6D0204027F3FD55C58681F0F070202B60A150AFFBF1FE885B8B47F3F6AD04B703F1FFFAC2D1890BF1FA208B5037F7FFFD760300CFEF646040755D50C3402041FB6B9ED'
	$sHexData &= '43201C1C18021814070214100FB0D036170702100C1F0F166A2FB807963F1FDA4284850F0F7F84166AA13F1F8B120B42FFCF251582119F76E916B6B5013F9FE7562D640FFF1F0B58D88A055F637FDD5887109F0F0E0884B43FBF200D9CC01A9E10022A0737FF6C28B9680100FF03403B471B7F1F0744C4A0B96307EF407E220E98300F18F49CDD5E1E029A4007C00FA02FB1E8C3FEFF9062616420616C6C6F636174696F6E7FB5B5FF1F9208556E6B6E6F776E20657863657055BDFBFFBF7F72726179206E6577206C656E6774682F7374726976C1A6BD1620746F6F286F003A436742CCB2ADB56532633E7942656D7FF616B3FF7D08754661696C6A207363686564759E16FF9978736865206F726521FF6869732066DC05CFE0756E6307726E2A7420624466FE5F1B88602026612064656661756C741A1FB467070C75543074C66B5F726F1311D6DA6B6548700C6D90D89402811D6C7475B6040812882D04E6660A606965761F5136F6C3B141734E73666941FC9908BB6624B6749297CF98B8976F023F0637077BBB3F740B000E2200222C08016469626E82050240BC795F2AF4793A3A19EC68EF9B7573002A3F69C8CC9D60A9D91244274F6F701CFC149C132B2B1F55004E00432FBBF73F114F47004C004F004200410E5212020B16285E543FDEDC7473B67265086B87E76D6F76630CC05FDD'
	$sHexData &= '2C6C1B691073772C6B6CF820C1985834E8691817885EC4E642490076B369EC5598BBEA200070D868126FA6EF7B7EED0A2A6D2267006566276D436D796F1C1F1A697A652E6FDED1DA790A6C2A647B722E6DDDA05DF61363008F6F74E2BC57700E846376651A6EE7F7BD4736765663AB63526E007420D812BC78576B73B3D76E76EF5336753A6F702A62653ADFBB1A0E400F7336615B68BCEFAE3BFB6732D637701E6305C3C0E093628F770F9E82DE61653B726A2A05ABBD0F0776476F4B786FDA1D286A1F2922624E7499DA817AABDCEF54B0B5B760FF7E741F7FBE4FE20D523273CFFB74632FBEF75A69A667976F0F4C871DEE616ADF6D27A75FF9CD968B1A6401553A6B006E8CD7E1E17777137257DDBEFF19A6067D005B005D003C577B677C7B3A385E1F64CF7E2A4106DA3D1DFC480045005F0A4D225316265276C3EEF62B265600320A0144362E1D1CC1BD4F0F2A1F5F53C4F6BEEF4F0B53AF4C4A441FFDFB06CFCFE24741AE58003500318C8D59DABA1F1729FE445B1A3C03E33A1E73DB5EBCD5539E73F770373F6CD3EEC3F778364F3A2F4E8A563B762FC26572551CB0F73A70DA739F72931EDDC1B4432CDF9A1B6EAC0DA6F79BAFAB9F63B0EE7C426F6B6D2F6BDDF6D42F671A747F4370CFBB7D6F45E7B63DE7665B00277C9F1EDD0023692B3D1F53706081150B5F8303D3880D3CAF23'
	$sHexData &= '5F6C456F5DC3CF377F83E9AD29235E6803C5965CB13ACF536FEAF12A30482649C64D32DED18B03A0790B00E3AF059B82C58F3F3AEF760306C1745F57735F9A69D2346E5D6730068A146C878FB33D868307D72F6377741BFFC8D1D21859D6D22FED7D6B182F6F70663D434DB4F6BD3B072DDF633D124B1D836FF33F3C3A1CE06BBE2EFB73E16A0707EB063F68D81AB6ECFB377F37AFDFC1D690E35F629F6B8A260D0F2FFE3D6B9BDE1B055E6A752266FFB185ADBD73E67AAF5F015D8E69F53F8E6F706D342C4DF7C7791E8B5F653DC2C376E70103A7640AEFC7B6674F6764DF92AC18F077782F29C231CD01494F53B3019D708044F24C4F33F73E7EAFE232125BAE3600349E201E30063BEFDDBD353231120A1620023A0FC0C507AE448E3E56BFE2FD80F95E72772D1F9ADEEB1852BA477650173E59F706764E4E01434255AF74C2C38ECBC22566D675D3562F277C067384D97BF86F253A780B3F4D1AD32E5633B7374CE3980A339F74A743D355D72E66937A222F5F42F76874436D5750A2EF53972B16EC5337C3757223C0080FDE32C6D74D8DFFFF53657450726F63657373447069417761720E16434E6C81816FDE78743F5049471C045ABD47EBE0A0E255620F5CDB5F2AF6A2AD831658DB0063E18944D0AFCD7FC1F6B88F756E1A64653A645F6D61702F73F1FFD2046A3501696E76616C6964'
	$sHexData &= '208BBDF1B9FF706F7369522F5F1B13E85F76695A396F68617361B237FE68206275636B657EDA8F19B4D86FE67175653C543EDD5F72E4C8DFB7700F018790733071C0391B6476B01F10606FA4E891938B1DFB400F3020DFED91D9BB50BFBA029B981FE0BF00D8C9B09312BA0FA0306F72F6B69DB1A8FFAF71008F876064E4F702B95FB9B8C69C404EF0B9F0AFB8707480C19391133F9C104B16807F7B323B3932D008FFF1F0E00F39417072B0A0EFBA58473E5C629FFE3980A130D9899023C6B09ED88FB7B7C9B3101F000FF0BF0B500FA010BCDD266F994116A0BF959A6ED2753F728A720F823D400EB278ADF82F2347CE7098E8DF87D07640744E042792D0A2988F2CE0294E14DC0FF08F9C60BFE2C8D993F9F4F02FAA48F8A04E468E9C6FF3409E88F2F1BD404E4EF1F19BE839C509B99FE0A5386FC7709B488E1CB7F0B670A88F5872227AD7E08FE2D0E4849C3CBEA03060B140DB6D72764CD00F4290EF402FDBDBF6A23F5F13811F3F3CA00F6C27B39390803F3B3A2F849EB42BCB5E39395039465E7E9FB6201A20AC2F216F13C89D389E2F0DA0EBC80939A9A82BE0AB2864B816BB7FB00FAFA026E4C809B670F190F18EC06EC32F600F206F299D9CE1532FF06F4DA075922367B80F7BAF481047B0DC6E27DD1F40870F8D3F936E83CC8E45804F44990F9FE4C8994EA57F9A18E290B9BD60'
	$sHexData &= '229F2FBC900F50422E3A393020AFA00B32F2649D4B30309FC96D6278F2508FF29FF2886E27E4F0509AC01F10AF1EC3809CC570A8DFB860ECF6A2D88F5FF0700F60CF49B7AE139D785F0F90760239D7A05F1140A3C8F172F2A6DB3F02CFA6F8E0C309763BF6A02F905FAEB89F884E2E76800F70603F6E2F9826AE3FB328FFA0B8866242ADEF504FBBE9849C873099782F800F2E900B161FFBB6B0B7C5DEBFD8100FCFD7F06FDE26A84EA2E89FB16FFBD49ECC4ECE1FA7B8C86FA5291083229CB86FAB78FFF0C3311C4FB8AF606F12833A4B10568238375B09C23A66AFF00E76A8CD8BA05620417F4385011B83D826070F64079A0B24170065070A69A805014041E090BE22034040CFAA43BA3D57380F48610B8242ACFE7C4F272727B7400F5058606860553459C99C7C39B2FEBF40C8E4037C54E8C01F20537D5080610300585CD275CBAA30067F6EB16B57703F805787C8D31D6D01994F40586929695E405B01F6BC01C15D022BC8C2026FD016F2F6706D03C6BD0200E84FC496CDE6607BB0C510A9CF1304277958C0B0724F6069368FD8BE02CFD081F8B7782127D82B30649F70B1C09E623838BF0F10C302EF5B49B379506308BAF04F48369B03E084D0C318AAAFA4D99C605F28BD40007F097925058E0290B9DA3B25DB70AB7E60AEA0A92FFF2B69360F908B78B5C0A02DDBC148C7CEB33648'
	$sHexData &= 'B28FA78A4FF3B076B808AB1FE084C47048203EFF48B0157C20AD02477FB37944306E1FF07C38C9824D4B49982EAC68C1B6398A52C858B0FF6F0713C9B358B1AB00D6889282AB7DBE10ACEF548E03B379C4BE20C8029F1067A0BE94A1584960D703971C300C968FBE87531417140FA8BF907482EBB4B837E06DEF60AD13C9A7451F3E88BF3013C986AB537EC9FE38AD98D26C26D8C6027F6BE0C27890CFC305032E0AF0B1AAE0242730C5C0C75F6325CD766872DE50B8E8621FABC1D7AFFF9F53F22C385EFE60B5AE98A7D809867BCF00C49F6085C12B693650B688585C3F9B53EC0490C19F586E08B915C19534D8180F241578819660BEFEAF435C431E60BCC7C0084E517D26A8A847A073AFD15C4427A8BC1FC4E0A6D5F174DE38CEB01FD73185EC38B39EC0BC09131427E96FB4887E4F88E61087029E8F0079819C92CDF0B030B0305350011632F877A16CC2E030C0D63690B598607A500F18B43FD8D125EFAA7F035EC830EF7729E405F8BE663F404E14E030B19F10CA41314598A84F70FF74A3097908C690961F0822C2109C044FC8AF9B1F5790FF882E20BAA0B4AD3A4E1C8FF0BFAFC0B1C28A01F67F045F583228E7CF9B535C17BA1FA863A0ACB5674A3698B230029F30420A7918C07682E0D58E4EE89FB0863FEB131372D8B010B7515D10ACC7AFB90725ED154CE8E7E0777E090F2438'
	$sHexData &= '70A0279E3E9907C8C8BBC87BCEC8B30E9047217F1054E8EAC4694278E6A79C0E05C77768B67E5D080D420E9FCF2F8203E4407ED83768B851259E5D4E68B9C3BF787DE185764EE85E3E98579C0F7069FE10B566D744333F630E70B3269013C938B580B09038C1844B605FB0C58124479CBFD03487A6D5C79F488BE04A1ECDF8E8B5CF50C3C46AFAC467CF248718F48E2F48BC29E404D5714FD8BB1336C00126FF90B6FFE05AC144B7FFBF1A2885BCA0AC36D9F56A070EF8CF68C94E30CA212F90296048C5B0075438F0BB0FC06E7FC567424EC0B0A0A8C69FAFC9D7049876044F331E7462B8B7FFF8B9AF4777B5042F05F7BDAFAD96404EB0AE2704C5258FAFB8C379FF0983EB857EEF3B41A489DF90FFD8A9EF53C7C314A00F9F909C48369B7C90BCE0B828C11D098A93D0B97F2711E2E10E98C71E5E10BFD5811528F73F04C123E4A0BE28C78F551C6184E6BEA86F5E5D7A69965FF8B7D329E4408858BD9FC82B694E207208F0AF055F2C52663F48BA3F849C404EA06158AB7955E18D060F50B6949247092E08B08D01EED067A0C24E56BC378A45DF7F2E883EB099422B468F5F70DA44D3A6F828BBDF683FCD2136C22E9F106F70C1F04A9ABD78308A0F8A1A0939C8B8D713EC14E180CFA0629FB85E704DB1B67F486FD0B5AFD5A449984F6F15151248BF57820943AC6FB65F32813C64C9B077'
	$sHexData &= 'B3E105370687D86A35C571821F10C70F4085D7BAD57F2F38E770B25AED41D21EBFD8C88F04F22117E6C9D8C0026F9DA12A0F304E031068ABD09E4FB6E49114E130BCFEBDD0B66CAE4A03FF9038AF6B9A48569E38BEC0F72156C510066F2EA91D1DB17F78C7FFD56A8F46A67FB8FFE797DCA940AFE06E588312042FBF043881FCEF83C8C58FC5B1404E687287B0E482E1A07D1FB7222528F5D73FC74C9A07A51604885E5497679E7810FF50C90A039B4268B7843AF1D6808F4E8FC18001F248C6C7B02A7A63A6C78603221CF6BFC2EFCE953427106E04107D93379ADF66043FC1442A39350F48B3ACD52EB1A67E407FC459ABE1466F6EA77630220F57CFF07B002B693638A9A81E4AB0831FB97F0601B8DA85DEE87F70031372822F40BD69BA0364828FAD053FC4B5CC01219F587E13C8A9E178C26FE06370034B9C90B3B8AC4EC001F2A09F20B7F83A20C2A9D7A0B82F016AD74AD4572E284FBCBAF4C4C60F90BCE4542885FFB87E9B13C10350BBDF60BF78BE9C404E24F0C450AFB8024C10BFA077F05D5417041FD83788D7AA10585E07FF865A2443008639210728EFE0C950A9A510410217E64127F0D78F5061D0A94917B608E738C22C39D1B01EEFE8BCA24BAC6E27E7AF215E4C09CF60AE0FA003E411E8C7C877600DD337208F1F50398534086E1870C0E0A3D100360F20C69040A62E6F7B'
	$sHexData &= '1A67886116CFA0B00E5CEDEE709FF89F149B40AE2E78C6BF035E6BAD966ED0C64F1E10E341D8BABF60C5125CCD4368AAAF371A07143B679F78C5A7538AEFC69F708F2A16445720FF6F03090A1CB0760FD38B1889AFF7B0C06F1D71E10410BF8FC078A9190B466F3F894013CC6F01973855E1406E60FFE05D069407545F60C2D08964A86F0878C5C4A366ACDF5F10C80189E4A8FF08497324AFCFC0BA04B042260831306FE045A41047C7C73B6BB3B697B78FB8C923F0A01BDF08BB3F9871C804C4288E507D0B2607016B0F6291424ED0C1B6359818F18F587F182F809ADD3DAF02942E20B40E978125973E4FB3F9065728852F94F646306D95F6699B2C12AE140B16F30956CD66B30EE8158D0DAC237417E6D36CB6241E500E2C33830F53FF1F5C83A14755A65E52534453528A078600639819FF95FF42A101DAC108AF4204443A5C446174615C5FEDD66454A26A7E5F12432B2B0A49C5B6FFFF6D616765536561726368444C4C5C52656C1673652DBAED82FF5F7836342E706462B72507AF248C87F31F564743544C0010042E93DB87BF76246469771226A021027FECE1C36D6E5F330F27243030002003C5209B34901F78E6A6B37DE0B177977964592638077367F7E069648C2435273867978624569E766366671E7008435237B1A70DFC434157274C6F422AB749804F55B8978319635AE79F'
	$sHexData &= '44C872205F62274FD0490EE44BECD8274FE04C2195895DE82743F0F925A43244F89F00685F6297015008274F104BEC72205418274F1BCAC3E720E03FC672EFEA20DB781E30002454D7264CDA46C69022AD7CCA4E78C8E7FD49766F6C746D642EF42C0417741D767A008467CFCF1F727B0F76F9746324494128275A5AD74FD96023035438274F7CBA3F9E4010A76C731F509409D849F38D243EE4D80424641DA9341FD7E8D8B078981875A3B026CE031F088F1CA2CD91B72080014665D01D2C01073A645B3932FC1E261400333030203757101F0334F58463AF67BE1C2736B1B637180986CFDD70521E100FF683384CB47E602A1FB73B5F3A73B6E06762731F026ECFE60090FC217089C0D1A4FA605E3F7376B8311B0CF90C1726F0036E01220279508000238450ED073F1BFD130500010A04000A340601FC0EBD06320670195E06520230A82D8DFF5F48864612050012620E700D600C500B5BA3F25B98191D3E9202502C16077995916F6F1208061E32B7D976A8117A04621EA03254D9A6660BFDBD5D086406020C6018E466561BB9CD764782000F4297D260B736838C0C7230521926FFB6430284341D060114000CF00AE008EC69ABF570076006803033C4C49214DDEDCFD3DECDDB040C9046BC01023A0D80D6B6EFFF0A74004C022C0436024504BE1B067E341206B2837298E05CCF8C1FE23E'
	$sHexData &= 'A6E99EE6085219170A72BC576976CF2CDA3A35103C060292CA37140E1FB65004D802361986BBAFC3075844605D4F66001147834F33DD2F70867985DA56F3E0EFE60A9043603EB2300424020E46010FAE097E760F640B06F60F720B0E77ACB7BB1B7E68B5DAAF0E6B00210508057406D4B8D74917C2063C34CC4391E2C11F191337D711ED224D870C0F0C52B708DBED4183EF36110818DBB7C02E4004D486B31B5A41A3122612540D0666792A95FF12520EE00C700B603C454C109C370C1ED10202806F07F27C3864DB1669DB467CF8F0F2747F858C30361602228660F9F04E9707BF5016822BA60E5F06BF1FF2540EB6A7D0D9E07628FFE8707660042E021C00EF10542A0D27F8FD0857878FE77B10DC2E15DC1E2F6E1FF937051E09095205E0036002300C00702D5B8B6C36BC2D20DC1FCF13AA7407609F650EC514B59B74D610340B06328B19DF4A679AE7080732679414461F959974AE021FFFA69FF7BE2438B908C106E6040812B919F8AB50D4C91E06131020BBD26CB7C6B0347E14485ACE0180EC0C6F8AEFF4045002A89F32E7FC6429F696F6DA06DD080E0670980A460B703F27BBFB0B420750065E4F24DD2E90045C691E184E9CE69716D29FE7D32557502859710A0A2DDEF75E77A0365C352836790A8A6F60333BAEFF0858001602AA085E0A7F509F10F5A94C4872A0A9B6DDE28FEE'
	$sHexData &= '6DB750397C320A3A9E04590302FBDE37F27772CC4438D908E7060DDE3095DFF0C708A940424B0204EFDD12EB60DA0E35CC3514CA61EA1458283EF8CE118008AEBE063EE4F6FA7972680728DE32DE027A048BE6D97C2F06641514F27754369582FEDEDE77DEC087EEEEFD80368CB02C15408A700BBA0B867EBB6350104482025AC0081C02CA0A1ED06C772CBC19250613642206342101F526804906CEC7A894E3779FF70EA6B510E106F2100A8941033E3E887ABDDCB8661788DA2EF034BC4FF0F7D10B605502F3060C0EE96602DAFC0B16C62637D495050E840034042E0251C9A1E02E2C380CCEB6DF18DF4FA36EE522DF7204471E570E0D8DAE0336920B2F443CD47DA16C36E64DDF54DFC506AE0D52804EEA8E3F357F8E0796AA397F1F1928091E07DE36DB54190634180112B6E00D700C20CF53793ABFA48AADD20C748002830CB2BE69627FAF5F1AD780503240987E1770089A01EB1AFC042008680C9E6697704737F286F0DFF910905EB6930F282FD70B2630D31C5C81A21F09E6085245DBB66A013AF63AD42635C4A630545E0C1709BCFE7208F006E00470378EA01A341A2245036E06F210F04AD0FA8FBE4AB3418F6CE072751082E01DDCE105F6B047D5E03C015EC92070C05FBA04510E02B2DF68EAF0FF4126C6193215F013E011D00FC01F0B42799A7D1FB07486B9C4E89ABEAD1638'
	$sHexData &= '0A3626080699160134DDCD76F32F1414640A0634095210706FE835FA7D2AF102E1580C10182F5C796EF7FA702E6C383700E0066F0E0D9EE673A49220E129E1400831782A5FC0368C9C2A867985AA76D6561A0266004F1ADE1587A4126CE18F751087DD2BB80D070EC0814AD0320B906C87E3860AF40258045622047912D3C4050D8F0F9A0F680F5369E030E86217ACB51D58C1E3BCE1CFC17234A79B068C5787D857E1D13D3F82F2DF60361C392C083A1071DE7D021C041C06C711050011DFED45E632A4F714E2CF1D100458439E6115DE024F0E1E07167C953C45084CE21FD6D3E5215E3E5D979240BF43817FB31A06D207E0057004600312C866A5D92780E26289E296E27860BA3B80320B4008A49E4A2D0355BAC434B6EFF7B8C19E3A3DE46ECD50362506E05968C0211A7C40DE1FA75CAC3CDDEFF83A01E3A6E3651DCF274EA03D898639DCE29BE7591E250A13129212F010E09D6CB58256E6DF34E33D9EED83E576445B0411E6E50532C474FB0B46246814001C343506012A0013799E6F8E177886320181B1A5FBF2E00EEFF081033A0D0158E02AB71A023AD0040CF20BBEE3374F610216EE02AD48D006D60829F4D20CA660AFFFF1030E6D030C11070AD7020C06388303E4FF276825001F3456001F014CFC8FC56F804F0205E420060A0E75707FBF9FEE2E0009317A2065EE053E4F1610'
	$sHexData &= 'F20234E588DFFE0466065408740AEE085906025008A7B37987E23EBF09041E22A632DD691D018F660656162980F9C00E5607C6123DE7E8F812011616708780E489A66583EF08A306080AF521022A7F07BE740B90500A5582029004920690089F6EBA6096BFC07FC9D4E4154F835E1FD07DE00474B0A05DD102671F280D34530601694E2080F0BAF7F8E473CC9E77F83805E5F6061F06F980FF7A60274D16E559FC9045040CD8005C6858F076024204CAA4043C017F54D36C81676421201A9ECDDB45FA0DE00BBF58E5CB0065F091E77973840890362C3BD51BB92902C804067B521E7142828F9A70B6005C1856DE025E066E898EF0CF010D420970086027261D3FFDB6DE1701100BF009E007C08B30029FC07B95E6C8E572D110F3060C569A177FB7364C3C987A3C82E032FFC57FEE2B6066A4149000660236A66608320A5E0865DDBC451D6A197EAA00CE2E1D6443DE8C71B3063442013C00DBCF30E633BC5769D239108F061ADEC38BF74AB3013D5AC1023E1F1662655FA698BDDBE741A390C106A2793273650D420FF535046FFA3F1D14B9CC0262044606820879BC3A0C9010EA79DEE0741ABC372C0B001A1A1A300CFF0E3E13F011E00FD00DC0EFD07EDAE7733EC338DD11E70622140A80BE0C1C0F32B030B570D09EC0683A760F582EF109033EBE2577C50B203487118E101219E73BFFBF'
	$sHexData &= 'E3EF1F30F90F1AA9267E06250308420AD6064A0C860EBA47C3E0A61092841244868E0AC6B109AE4706BE5E0F7CAFD2EC075CE78B69107E068FE707A8C0471E70386638CD3D23B791D0A0D806088664E960C057558C858852046C068A069BCDAD598C220D68061E64253424011CCEF48404EF265007C4E7FF2ECF7BCDC6DA0470B9F0067904E779D8FC02FD0304BD021A09761C1BF9FC968A16AEF07F04E80DE8D19F13B21AE8900CC204CC022CC0744061140299F77938C8E73CE845E8E03E90F0D099EDF61115D615541D06341CD6BDA7327B636770799206087884020766DE629017FCDDEB6A830A3417400E800052022970EA066DDB88A25DDE1E063C028F491ED818B1CB838F1F111E0B01D7791E4E1B1A00F85466437350F7ECF510E93038E95696D24E057ED3DDF72EEF6A9D400A8AB204A103063808CC857E09E68721FE15914748F0A6D29B7E40E9837FE8BD49105606040732203CAA6155F0C10C9886DE045D76ACB4E76976513F785C6E8188C4F03A9602A27A62E4C7E1173EDDA04FA9B9E966903F50D3A77BF209AC042A74066D5DF3E0D7D0E9D5A7362E390FEB108AB07678191E1A764006AA2B9E61013627D00CC0CA096008AFE3BD4AF308EA821110580616ECA5119E3ABA2A0F02059A0DC2CDAA874F90808AA1046E7167DF78E092C4C14C25E103761608EF8C2D49E1021AB6'
	$sHexData &= 'AC8C0837C70B2A023D40A5041201884C107D0AE344F0FE12490216FE12BA180A778CEA62D57F1337951002420219360E287832FF030EB03A68331669C7D009C0077006E7F934F8600530045077C41203CD9EEEE1CD2AEB1E0E40403C69099A7A074FBE013141083EB8402107876DC29197E081065013AE033AC9046FD30DC1A0124C04A30582AD0A3F8104229106D480FC088178BFE0FFAE351C380CF610D5161A99041CF1A675081A89021E6035BA8299ECC61B98AE330F43D380E77E12139E3228FF7B95E60188EB1A9110AA06325269F8080E5041B02A610158F41B3E0AD907086C023406D50D084819271DF8AECD4E0117F6F00BE009D02B784A7DAC67A800DF01110087D4788DE7F4EB0FCF030A1D5FFD180696DE02077B6FADB5540916429B6B2E21CE46B8405F188E11F032D00BC02BFF2F46FB30065001302E273620F01EE01CD01AC018701760A60B57FD1630155019310E1F6428FE27E114F8B72076F016E014D012C0104F78ECCF7B0F104C368510A806BD0E6C9A0D44B6E71033E160C37C37D87BF5700A30351CD4B0E2501E8002FB847E2B4654021312D4DD5E376C55FFF10D18090302794C49387804F608520AF85F24032E04F10702FA0E92004E08A6CD82575F5256111F1F68FE77C314E81A54164614B270106065F3D9008F04ED0DED19ED2418DC6010420EE100FF370D0B'
	$sHexData &= 'C2E804C65342B8B8003AA8B9003298BA9BCFF3F4002D88BBAEBCBD86017C013E9B0314AF60ED720B69ED6C42D36C57F0D8425E8C42EE65D38F01253A3095C6611FE1210CF5E477F91A0372042941043502778E3D7E1D2DB5021F411DFA1E0EC023DF97B5E10D4269231D15056436FB200A721E42697AB1B2AE066C01A010300220E0CE663DBEFA381E7ECFA26D3A2525025DD97C2F0D21C56130953E42447FDFEC7047219A0D3A4F20B1C11C3A80EF0E0E86227229411CE2702FDFCB160B2C1E3AC1411C610537C183B38133A3761B7D06053F74EB77DD06EFBE362B455407592946E406DFBD49186AB880C508DF6122426596E15EE582171B1A646464E4611A1919189191B36C1B0D1BE117178691D986163F29161577B2C8ECB53637060DE4F60E0BFD78295307CCB2D96C130D0829120D1B61769B458E6109D1E11B112923B36C640A100D1B616164382CB30B2914CE4DCCDE8A50153F4553618C8CCC8E610729130D0F91DB968D0C0DFB1B0FF123B391050E29220E7D1936481B556571762D833CEC723E04FF4DAFF8B4062D050AA80C200AF5B85A105A126C10B00A28FE5FBEAA14B88E246D036410BA0A200C500A9D0357C9F8DF43480A850416F9020A3A322228A6FFFFFF2059042420208903265A285A2A6C284E20C02C202E5A302832FC885FF4BF84125D0338B83A3C3C9244110446'
	$sHexData &= '2044FF7F2BB460D514D64A69034C204EC24C645045054A55FFFFFFFF02526054410342A456A90238EC583E5AC45C2D045EBD035C38602A5CDC38CE62FCFFFFFF2C642C662668266A266C266E7E70287228742C762C78287AEE7C307E2A80FFFFFF77F0822E849C86AA88CC8AAA8C6E8E2E907492349476963498789A349C7FC3FFFF789E34A076A234A476A634A882AA3AAC7EAE1AB4B2B2B4B4B6B2FFFFFFFFB87EBA3ABC7EBE3AC07EC23AC47EC63AC87ECA3ACC72CE3AD084D242D4AD0E62FE5FB0E12D03D632D891CE2018560A89041A601C601ECC662E2DD0060A449F24F89E1F921CBCF10200DAC510DEF160319E1F66604895CBF01B5A348C9E0D081602C102068808C86B82B19FDFFCF1341DFAE605F26A0219300B66B4AA5BA3398030F205FE10DE488FED49F3064E0A77C1410D323232D21A121211113232323210100F0FF08D2C320E0D6F22B3914F4841140D4F0796CD8ECC1B41C10C0D06291BB36C766441C10B0D05291B412C03B223C10A0D251B66D9ECC841C1090D03291B41CB0CA946C167AC291BD1DFF0D1DE549C008A28042806280824C2FFFF457F0C280E28102812D228162418B41A941C7A1E2E207E2232247AFFB77FC72632AE2A322C7A2E3230740C346E362C386E3A283C6E3E035D90FE28406E4228446E7A6E7F855F24346E4E524CA1078E04000DB206D8275D'
	$sHexData &= 'E1F004E0B7DE211026C40AB74880BA3413F14C360C36A006A0E57D3B63244F640B1FD126B402D6423B5E1F67DBF6CDBFBF7205F0037016060016E4060E64F00554B66CB68A0C10046E0E0500F41F5C58A40F192408F61B005EA6F26025273FB00B3679CFFE0BF4190004D41AC65EC486381FEEFCA9B43F0A010D0106017FFA9B71C62008E6563412010F010C01EE7D6F2007B466523E28BD10C406A8EF479974324006A92E02DCB82DC0D6FE0BC66417D6341606010E01549E6EDD077FF462FD4C9FCFFA04F58EA02E1D0C4AC1F3C3A57E1816D205E06097ED4343A548431E43160AF4B605B259B05806F00F9E28100CF5EBE6C1DA1F9E1C07A209F0D3C2A811D3473620A09521790E011800B02308B7129BC1565E738F37D3C69A8A3E056267562D38C4AF441E1E17F015E013C08A0F37A1BE96B7D0FE7B00AFDD18789087BAEE069CE5F58D0A06FD4B42ED8A4D0992A71301686738D273B7D006C07B4F1CF6F5BCF76C96330329106C067D18B6D0440F87E1ED506FD062B2A572F90BCA32B03A7076ACC5A7629A0D1104526F5E5E1C141674F6004BE51018C0BD61FBBFAC32089905102612AD0410A9E6CA089822162610E4C1E168DF309EE225795024461E8242DFC86BA6CCF6D510E2F6FACE80F916D31502AB086D020282636C4DA1AC9AEC119E1182C6690D8C51B7FFF79C18FCDE081706'
	$sHexData &= '020E4036ACB60D242E9E2CC63460060158A7714305074054BCFBC3F3346E496D103D60367C7A80BBCEFB62105B0114981A3C105356FBFF0E3EC1180A6E0046049A06E410C6B612AEAAFC706707001C9E1C640F8331A3D906340D526B3F2F7D9E4FA838B5C9DAF79612DF95C010D4B43A80930A3038FE7CF8C56CD20406D1F7B9F8C9050CCA2F997DA7D4C606E8044CFAB607C14A35CD201F1AB7FF59FF9E32F80836060E0816404CBD853BB8E0A0D5016A6915A5A1023E88830558E216F276F40896908A3417F0A60C56084707BE066CB148508F2F64F8E703F09D6D1074F85E201B15C001809CAFB7F7791A5C94179DA90C080110303B46321260FE5E792ADFD69F1106D20657C0C9E2BE833FEB0AC5303EDC4C62659AB09B66E00AE0524A0625F804CBF341B708606E835D7DBD01B906ECF89F1FCF30976B5F70645E2917B0967D061CF95F1F87DC14CD360D06340C7277642E9181F214FEF9F9236E25D9BC5DA876FF503265324B9E61831FE73333349D846B1F11670D0CBC3C95EFE7D8E1E860C3036C3AA89E9E07C00742BBD03A7000AE0DAEE406054F9E6D93D64035E65635F9540B2DB4ED7D1F6F06F8E71F67422D99614F64D624F6BEA7C0F4E30644FA5F1FB5249B17F7303749371FBD631B84379FE02EF4A4ED2EEC06987FDE1F2038C6A0EF8D16DA1F678E803EA364B6367946441F'
	$sHexData &= 'E752D97C9DC086F0560E390CFB4D2F244D2600A7AE346DB4CB7B1E474E28001F67DCADE6615605425B00DB8225377709AE903B7EED3B325B9B34801F5F3449B3D95C64303C4C3CB037C73AB01F01127E0A0900124368B7973211132613341506B2D67BCD83DF10FC1908260604D0F16B7547A02704B2045E022AE080914FF623E1792ADD6F4851700CC817C2477F3AFB684D83B1E1F760889293107CD40235D68455827F0424063A08880C1E9CAF0741E90E079F98A696578CCDA1FCC2026FB4FC0B3F70146E47063E6288DB86063209E00E370493CD8A2FD770487E8C4870D97F6E0D762E27AFA0490100B64978456293011FC7004AF69A9BEC963A4AF11F1917AE1C03295CCAA3E62800C6869C522F01AB70E85E1186CF66E07BFE4E8CFD02007A95109CFD9FA01D6CF8B004D5E65503C636BE364CB299C34E07AED60BF278164A8143460886B6D05030D8C1DB68465170FD471FDDA914D287FEFDCF4903B9850F8711D73EE16B5AFFFE1D0823FE16B04D3614986080063E9E3EF4E506BE8A76396634EA6A00F0F0877F96543CD8F22CDEEE54FE1F0151667BD056AE51F4764C7E47647E4254510A4647FAC0F7FFF05D71C806FEBF9D10045C4AF2B08729010B9632F3217863871B0F0400305F5A6293667F5FB41F46003990DB428FF0E45C549AED606E5660FC08B674B10BCD6DB82FBE1F67'
	$sHexData &= 'ACDD37044700562FBF5B69FEC85213060013E4040560619EEC62CB66896160FF1F069FC6AF85BC560007525BE0F9F527C0A7740C0B97C0621D6CD26CA6EA62B41F12348BFEBA6E12740967C063E6DC6398C884589B1F170EC2BDCB0334000341105206E0CD4FFEEE490003E5050EBACA06A60408341616426804CC86E11711468E7E14F012E010D0EB66C4079F0AF7849F910876687BDEB906D61032CD010630E81D797F0FD8680A0DA2D0CC18D01502EEBBBFEF1BEC08C85E0A0C0ECFFD0180044E020D7F91089F9A0C61BA6A9E04FD0C084D04C400DBC10E38060C220C32D678CD4FA7EAEFFC2906010E069053B39F5AFEA902F04A572800BC95930132029447590221E9FF11FF4075EFD54C9F65106C069E552A600387086994A45A701713F1008D0302678C4C496BDD11FF5F5A3E23BE142A4F370A4ABFBC72C54D537979CCD04785E932741E85CD0019111406110A4E17A89BD7F8A6020255C40DF6060EE80A37C1F75E30C60A063D2A40B9030485EF447C07C60804743DEEA9D0E676360618645E296966831FD7137279B93CCF64026D7A70F0C516F835F0088DD43A04B5068FC8295C112E329F6C3B8E30CEB7FE5079EE4C7A791B6CD2881F47B07AF95A4BB2AC7B1F760579F584A742C0DF07F4377888D366CE107C1406F3B4C4B6E0771F560000C2015EC800010F6E32E65435152350'
	$sHexData &= 'BF207FA36A2DC9497F1FBF438E0B6B37275260CB66C170804E938094031F79C7ABC0B7D4038FDD03CB5701F138026FFC030378D065C705047E0C04435298700B169D4E15672F792ACD6E67300432394020509D0F4006888807576277F25804614FA602DF706F830DF987010089871F472088031B15364A88171F01167EC786754E75D4CE67C412005534BF5607E9FE92500167B21F50217512DBA2171E17D40613C49674B3E153B35E0B640416808C86B48C3C6DB165C8041F6600002324CFF30000005E4BF217B78E0100228F1F01780D9EF71ACE788E6D8665564FA248B4122EC066675A64B3AD266092868D9290DB1FCF9307C397A405AD056890E69300BED70D32F30B1D94EFB22758B3A062069F0F40967E70966C684FB87C0727001F163D1D18B760211C9E1C74D976E36C1154BE60986E9B989EB6D8B210061F560000344D0B790056074212F7BCBAD00F5011AE69B66AC84F509AAE469A6416CB26361F04AF4F39D392ABAFF0F7740D6293667BA09BA6D09BB81F6F216F53F75600AFD9CC4F9E62D018080018B6C405CB66F3E7060E00409D6E9D1007F2BCEC621F08B75E00001D931A16160457259A88D5184E0E17B26781AFD33BA0565ADFA910B0075C3760DA87F20481BCB323F907E715001492D85C6EA04A2F4FE1E8C0EB920655C2017736B60231D647E0EFBD031408761D0848'
	$sHexData &= '060E0C10840C1D896B063CAD321CBE7477A3A8A420D7253EF04F33F0776297480F50FE94043C06BC0E76AF8205ADDF2337AF7808FCA6D60B6C8781180489020603056F1E9FA827BF96069FB1B8089F101EEC02B80AAC8C3D0CC77018F934FAB76FF4640DAB586CD9AC5E9FABC4081F8985BC14C75600150041EE3F6206C0E624EA6913F1561FE41ED406BF175B369B2EB63BAD1C091F08B690E765B75E0000BF6C9E2BF9D00470C46470AF96D8A4D9ABAF7C1FBF551092B75E005F822DC9E630B26BB21F5F324D6B233F37E05B9E66732828D41D12BEC596CD66A0B4EAB42C0A1FB4D892A736D45F9AA2432D9752BAD0932157A7090D861A9E12FEF0243669B6B7B63FB88C1F432D96269F57EFCF41C5488235C635C62A1FFB63B73E12C4C654084FE0B9BE616293662ABAE41F0C2EE47989376E0000017D83C8F8265454AFFF62D9CC6B1737172E178E12EE6406B165B379D0BDF0BD640B1F2CE4AD62275E00E6021C5627625F239AED24CF23F4E450BFBE7EBFC0796B894D1FBF5E009160150BE77FDB03440EC6C48EE0C0B60EC12E165B36200C1F26C48912A14D570F581B25DE90C2DEC20F1F15580B950F375F82810CC8E0F406486CC16B562E06C0471F118B4D2EFFC45FFC37C9AB7F3C0D45180C6800AA020150C0AD47702ED6C64D0763A2FF118F0A09CF6C6928FF0D0308921786F98C'
	$sHexData &= '0D2E40989770B156013C6E04FF604D2E3C9C4FA6EE56CBEF15EE35A6069864BF091FD868A6BCBF26A07A1F6762A40110FF540EDB3BD0F2867407F4E506EC670B65DD601F4F64FF306A6063D440CD071F010BCDA678A63606340A327D6F085A5B6047C046E706480ECE9FB496BF1F3F3205E093667BB3EF90CE5EC6CE8046485A621F4600B429B8785FD60876CF23D09CA38289AF1DB61DB60DC4B6F7150E0A2F9206C4AF1FEC42F2B48600004E1F816C5E1BD7F0D143D22DF6BE2E2FB206300FD71FC6DA85B6D7874E1F000B62CD97B73E73E48FFF0A000A520F0308D00F0BE3E7173EC02258009E0719433E016E4280ED225BFF0F035F48024710D9797214101D102C77F9D97C0E381041104E100C2087E680060F0286B2D7113772A870EF70107910E13E88BF8B06060CC03A9A006F0DC0BF051108EC043E0226065E47AECD8FC9F02F54B0DB7EF36C9ECDEBDB94101B1BE4D43BBE4E8D662FF706A41047B0D126763627001F9F31249D34015F0F011084A4A11F0F273CCDC9E610E44BE40F0F0A9682AF9B2F570638112727A1B5D126001F9F03E1104807DFFFBC770D3C466BC6C110DA06EB5E40B5F10CCE956EBA4EC6851CB00D9609C03B817B24C4E24020B850A31AFE3704105A0224049C067C08B46CFC9A23CD6C3006BF0FB2141253F23DEF55211030064108BE3501E6F7C078807078'
	$sHexData &= '3812AB1851824E60E451030C52CC2AFC7789C85C5F74502406B936D8AE98AF1F4FC02604B6E4D01F770732D71860160417DAF32CC050ECCE89EC12142E14DEF7559CF61E0627D206AC1297335384C51FD7548F2ABE0316E79202D0213AAE3A076840239E2E3E24F60EF6BDF6C25409FF60EE5E060C13E7B1F775B79627F4061413E71F9E3728710E47A600002EE30B7900111C0B76C646489A1B12FF8B89F7CC130308D6060A879FED4BD3B0298040967851082E9E0E1DE1578BF8020440068E680A8402C4003C45D0052B5EA61568066F00E51B08FE587642CB9F8E12000DB23F84E6EA11FF11FE10B2299B574DFF149745145814DCCDF100D470453417600AA24073DC4C8E8E0668987E0186572055BFC770FF6680C196CDCFFF64141F360FF08A1FA414C6AD14B414322E3F573000021C0030165D90A6B90E22211C6FDCA04FEB3E38E96715062B0A0850D9F01D7836102E5E80669A82CECA685380FD4027A85967352852880337E07845DF6716A104D2A6270CEC054097CA7EAC3208140216F75DE6829F524C82295608802EFBD65E159FB770CFB043E6BF62785782B0642660A402BA1707B9E1FB06A03C60A508024E870A36DB4361CF5F06345E0156AE452030D76FA0E886CCF6016F740F06640E9413BF297E19199215E01925FEFAF8AB29361717B617D210072F20F374BCD601EE1408'
	$sHexData &= '982A1C088F9E14E112782096192AF62BF44C8542477040B8F475833F175CC27228BC031ACE97211E4AC538585E6804160ADE0B175F31D53730DE71064C16F7E779DA621F5E0000002E48B19000CC4E0C1C38B568460B463E0A4C7C6D28964E166611A2001C82517F0F192EE40501F99E1F0087A40061B602BD2C07962E06342D858207B586A7CFF379C52A874F48174D17766A36CC981FAF270E64D3B0FC392802000F53CE020250F154EA190807A60670E9C0F34C6F090807F6063E9E53F8E01115462E154615B3F9747F153211E0BF02DC2A4B2BC2DAEB4E2701AE16B91F7C5EA7FC6F08670446F3152C4E9E77D23EE81E295732E85AB7F6163D3F1F3E3F09BB801D5836761A723A7013DEE61B8260D77546522D6B530F9E096E10A750D70706D7C0A77905120D0477AE22004F86014F18505AE7B4EF18033F01A4B61D1B2EE01B37ED56641D463F8DC02B2B25AFB643BF981AE433B1236F8F88A04D32C59629403F602182EB5623C02E976A38157CAF5B9FA03C1F99D80B643838547F28279053F2301FD018B04CF675AB6D0F3723102EA01977C4B0BCADAFE81A2FF7BC2979C819042EE0886E58B315627FF7724ADA623F4E1930536AB89253E01A5C5F3825A776503C9F585C70224C05DD22C707543F4090A9E295685E4F90088683BB057F2E374EED924037FF60174C24A7888F801908DA'
	$sHexData &= '0310ED01FC2603275DC9943CCA3FEC684A4EC9AB4F3440E642DAD5AE6412889FA827A7E494BC72286A3C8E3AC9293925F227703D938C954C5C1F10A8A064A200A00007210AE840716D05452401794E451580FF003B12B520F3FF05993F45CD5D20D266D46E2B1271EE02B31DFF4E9520DE0175986F028FC8C5F0C51F8E380347022AD83201009F0162ABE432EF1F804A098F029330077F900704997EF02E3F41563F247F94F1FF5F46616B655F6E6F5F636F70186C6C616276A58CB76C20616461702A40563C1ECA78FBDB6D62185F343E403F494E063F556E69FB54517898054040594197FB2318FF62617369635F5F405F575522D6DE5B3338405F1E61697473204034645876A79671507D6F722932406666B3CF2F41454255A15008E86000F38F08B65A357F72756E74696D169A010C87575F9FCF10AC9B3143435F4378C1B0FF616E63656C416E645294B68379CB287475AFFC36FEAD5AC654045FBE706C404540646574D6DA670A8C6C486375DC650A79D85370B0FD554541415F4E0230307A56DE73B606158CD07472345F45B0F6DD32F1486F6C7A72832018244869D25F1A1ECC30A298590F1A0E682C370D407045B0FF31FA8A10C4B1603F30071F0CD7DC3606796EC376658FB859C323DDDC4D1E6A526573525B33D08CBF35705F5CDB9B0B51884024240C48C755C1173257E01F9CAA'
	$sHexData &= 'BF5F4D98566F6964546F4124A2F14C7446080B2CA2B120614B6823D83E1A4136FE585A1339C4B182E7583517DF339C0F0332C642490D465860050D3E42886D485FAA3B24AAF7BD506978F6CC6002F742756666EA060006AA38AB92314D77345655B4E0B2363634730C18221AAB325F348A3A42B746E871C1EF74DD63B51712950034D51F00D94C0032464BD5856B8B8C67E21B4865A882DF31435CD308057F021BA32AEF66A10AAC9D5F475FD8E8C09B3752303F348C5306582312B541A30307C78840435FF865B69D00ABB94B585008555F7F51D78A86484264AA427333A9E18029C0A042B200A94C555F334D81120DA3386557E1014469215F5F494E98279E807075AD84C10CB404C3EF8F5319A9AD4E614F7761634EC75F2FD263D49162C39F58100CA2911F11620D6963DB7F30E38040465F847630106FFF9A406055175643ADA04535E3D921CDA58DD57FD65F5FD12B062C4F7479D08CB8F0220A6E666F40BF428744660C5126142188E33725FB584D175732DE562955542520FFCD818A41E53435586A9F62104C0C9962DA9F840306A47F46B6552CD2F1DCC6670E79326382AA41C37F27518F8FD89F526566E8601097BE0E5F6F626A32ED69F80D15FF55BE50504CF59AA043DD4045254996695A2D4C5574E958B3334133E0C37C5404386CF7B6E070F84E6F412585BAC42CC71BCC2425'
	$sHexData &= '8E358D879732378C0A71DB79BE4D41155675FF5348288E11C1565521409F0A5241978D511BAB0B29711DBAC2A98F6F73E1FF210613129F1F515BA0518F6F451851889F99582752143BD00A30A2FF0073A869C102C1F04A6459DF9B31767C476469D0757383177018064D6F396F66A22085FC3C65616DCF6A3A12745F47895171D5D4303F4974624048A815AC3A5F5F040B3463ACDB9F3803037C5F4465F4A848B110CD67BFA25001C4993F2B410CABD97050F2A21F696F55D5845591B77F5DBA03588D312DC00E60108DE068D0F0009F1D58559356131F94288040F30C40A81A851F55C87311334D485D07086BDBB840A2001F8781B6A00025BF55B088033DBF3A34AA6A714FA3AA4AA5B93AC4AA1BBFFF04A4B62AB77F988013F05A43582F4DC0822DEFCB186D152A8D5F01A90B1FAFB1D5EA18557F1D3FA4C58A4108636BC43ACE82587B3932C36C58412CB5DF552756103F75625527B6B77FE0992A228D3645851A4651DFC7C0AA4BABA3FF7981419D1FFF69AE88B008AF4585B8104A9F46AA6350B71F0104286299048558D5BF9FB55097127FF72A02C080DCE335A1315A9F216CCDD6860EB9F2201A62D368820AC82D720ACCA04067D05A651BBA148C97DA01ECD038065B6DA2430B8D594B750528C45F5F915621029D5F14F20C103143CB913A42BB87D02AC0BA088DBF10C16CACCDABBA'
	$sHexData &= '81ABA53F942082101F220606403F57055C4202EB800889212F9B2662D530BFBFC6DB04AF4469F680756DBD6AAB081581770710041220CF164E3310371E40235FF37916B7B0E4D02E0711106C9FE67B0E87162CE89013125E90E92FA78CEF0E9E124F164234DF7B9A13F4900EE81604D9F03B9B4FFB7E158E1480155A1716F6799EE66017933CC003188F3979B36F700E91182FA016CAD0D37C9EE7E044E000196827CFF3BA0F3D163C407D809A4DF73C92708F19661A783EA5D97C701AD81A880E1D1BF33CEFCD5F200E421670609F7C4AFBDC98470E9E1D8EA00EE31DCDF37C134FF016E71ED91E296FF6359B1F00DA300E881F479016DE7B9ACD26C046532008600E3DCDB7E08120C72E3721144037FB3CEF0EB31644E01922A7500EDF86DFC17022DF0FAD226F0EEF227C23DB056723D63A24400EA8244FF37913C7B016E260DAF07ECFD37C9A258C8025A69CB0D93C1B3E7F26E6E0DA26CB2638A5393826AE28F00EE77D7F48D220DB020FE31654D9F7349FF0182A6C200E4B2A77079FE7BD500ECD2E90D04A2B37DFB72218770F95169CDB3ECFF334959BB0A0C02BAF79FB0EDF0FFD5F102C6406C0F2269B87DB0E2CB32C16ECBEEEEE73F0002DD6460F361600DC609CE769406E18E779DD8A5F0FF1162CF1E7693ECD032E40102E2350A7F9349F301130682030DFBBD37C9E7CE050319C'
	$sHexData &= '0F6932F7B4EF6CBE34DD0EDD331684F00ECD66F3DD5934DE166034FB3538DE00DBF7349B36C33788D00E0439EE28DF6DE2E13B0E443957EBCEFBD64F0FB5465C0FBB1653693ECF70C0BD3A803A04F79E66463BDC500ED10648FBBC778EB63C061416509E3CBF9A6FC337A00EDD3C6F46933D1C9B4FB37DC03E16EC402CF0400C464FB3F934481046A84790E0D1F7349B479148D0A00E7649E61DDF7D6708E10EF24B5E16004C06B0F756FD964C0F3F4D4F0E99067B4FF3BD9846A016274EC4300E90069FE6F3DCFC8F16E324E2F059504FB3F934546050C55164D0DAF7349B510D53A0200E2B565ED8E01BD97C30563357400E7059BEA7FD1ECCE3020F7B61164C800E1965EFAAE7D32ED02065651F657666F3BC773634E40E64164470EFDDEFEA00677F0F2469362E600E9CEF69DFA2690F0E126B2EDC200E023DED3BBC6D8EE56E0EDF6F16A4E08D6CCFF60EF5728E0CE67376237330ADF89EF60E86772EA8900EEE772FBEFBCEF60E1C7AA640E70E3C7DDE16407D98DFD36C4480E4500EA88300B706130A3E270FF3BCEE6E0184EE2E0F0716B81067FB9EE67987C8800E8589EE20E93CCDA77D0E7C8A1664808AADC07BE7E1F3B00F95B6E90E6516F7349FE67CEA7039A698400E537D67C1E5AC6658EB0E4FAD7F50393ECF7B0ED41670E0B2AEF777DFD3BEC00E45B15EB8500E46B2BE176C'
	$sHexData &= '369B4DB2F5B308EC00B46DB62ACDE6D32070B642BA38BA69DFD36C28C150300E38C316E4F09D83EFA00E86F796ED0E6EF9EF69DF597E98F10E00FA16EC400E69B16611DB0201BEF2020FD37E6B16470F1F0316CC9ACFF3BA0F4616E046D804F0756B1671170F8AC4799EE7160C8A93284EDBAD58D70F8C0816543CCDA7ED0F2B09166C3009BC7C369FE6F3C0330B94500BE80FAD53C469CCC7E7799ED70FD61620D6EC3CB3F9349FF013124C20129513BD239B4F68A013F81400155616F33CCDA7168440166FA070BEFB9EE64B19B0500E451E6646601E6CDFD96C621F9CF60EBE20D678D99AF7DDF320225E2E20224623EC3B9BED530E08258E1CF7102516F3693ECF78704D2788502767FB9E66E428E8F00E5E29D640F8E89ACF7B0EDF1650102A6406ED7BDAD787700E392B2E7C400E282C16AC4D81F73D300EF92C2FCEA6946F5522962E7F0F775F77DA0B2F16F80F4C060CF9A60E6F5522DED82FBF7F0F3CCFF3BC96162896D43CE05589B8FBF2317E163FE775F7BA0FA006782E0FDE168C69B72A113F0EA633A7344FA59C3333B016358938CDB834C097AF5B9388CF0FC01610BAD37C9AFAC01F36240F298883CFF3163450B836979DF65B93C70F163716509A449CD70F2D1664C779DD79DD0F9E2E740FDF168844AC49C46FE779DD9AA70F5616B4569CE7799E62C86270D879DD9A449F'
	$sHexData &= '0FD616E812719EE7D6EDFC11E7756B1F0F341614795EB7266F0F76163C76E7799EE7AD50ADB36069DF1C9FC0B83A7FC00E8B3B2E70DBAD49C45F0F143C1622CEEB4E8C0F1F16A0F3BC6E4D470FCE16BCCE66F83CCFDCD0E03E3DB740A77D4AFB0E8F3EBEE00E364016F4B4CFDCBE400EC142662CFC020E8C431684359FA7F99043C0A40044309DE337C006EE0FC544371045CFDD37B22E5445600E884666770EF139B219B246EF104644475047AD48C4B266488F9ED79DF70FB92ED00FC916E8C96FE43B79DAE05E1649200E41442C9BB349BF5049A049BA7BDF8AEF0FCF06FC760FF4DF8A44FC0610FD02EF0F7E3DCFD37C1620FD7E863490349B4FF32E654B44704B794C587B4AB3F9A04C624F700ED016445CECCEA40FCC508FEEB4DF8A670F4B5216C40F2AEDF3BC6316D8B08C53D6E8531D226E0E1E5404FE02EF5E77DA6F0F5055163C0F791650B64FF379803257601058763F5A6F8EEF6C08EC0E0F5B7F100E2FE6F3AC70062F16DD7CFEE4E00C9FE03B5C57405C665C473667DF0E045D2F105D995DA05D4EFB9C34D4E02E5F8EA4DA6E4522070F1D6016C4E7795E770F2316D82329DD3A449CE89F0F8F9DE6D37C160CFF8F4B6120E7799ED70F5116385157486DB70E116F0FB362166CF33CAF3B0FB91688B9BF6E1D22CE985F0F715E77DA956316C00FB516DCD376EB10570F21642E'
	$sHexData &= 'EC693EFFBA0F3A160400030040246614369F66F33066946C64A06CA06F9E66F36C38016F57707860D36CBEFB0E4373A61650730F76D8AF5B3788CF0F6D16203ECFF33C6D9434A04279AD1BC469443FE7799ED70F86169886B0AC5EB76E103F0FE616BC204EF379E6107CD077DA6ED5D70F847D16F00FCFF3345F8A160C038A901C901AC4693E127F38EB4EDBAD970F5C802E4C0F6216419CF7CE680F1678DD79DDAA970FD416A00FF2F3DEA87B06B4A7817E06C416C02CF89EE62483E0300E0884661804D9E779EF0E781648D034864771F4CDBE400E188747200E6787B7E77D6B06FF0FD74664CD20EE760F18889616FFBCEEBC6E0F9D2E880FD6169C419CE6F3E07D8CACEEB4DD9A5F0F8F8E16D40F10BFEEBC9C16F80FB506080503EEDEB766770F6D062C2E0F7B9ACFBDB106404716399250400E9A411CE052928F67F3BA73DF0FD12F460FDF16847B6FAACFE0719307800EBE0694A7D3BA559F0E2894C70F962EB0AF5B3388170FD616D875A7F93CD61597EC0F2A79B34FF3160006305B978F60168B966FF63990C497EFD0163998DA6FCD202F0FE899161CE7795E770FEE1638EEF4DD8A419C48E70F8DF3BCEEB49B16780F93169093158338CF99A09779DD69BB0F269D16C80F2C16E406719EE72C32F4BAD3762B370FE39E16200FE9F93CCFF31640E9EF50709F013E4FB3A29F70B019'
	$sHexData &= 'A0D7ED7BDA37200E2DA22E7C300EDCA416B8D97CF73DE00E4FA6D61650A6FEA738CD66B3540800A85FAB8869BB3582E70FF1AC16D47EDD79DD0FF716F00FFD06000903DA6F8D20870F58AF162CE7795E770F5E164C5E64DD1A419C5C270F16F3BCEEB4B2168C0F1C16AC1C158238CF22BCE779DD69BB0F88B416DC0F8E16FC85207EDD0F94060C0A03875E77DA6F0FD7B7163C0FDD165C419CE779DDE36CEEB4DD0A370FCCB9169C0F38CFF3BCD216B8D2D8C869BB7581BF0FDFBC16F0CFD37CDD0FE516180BE5EB2810A7F93CF0CFBD503B6DB72E570F39BF16700FCEF33CAF3F16903F45A0DA6E5520F70FD2C016D05C775E770FD816F00FDE06B72A100608A70F3CAF3BED7AC216300F80164C805520CEF3865C775E77DA6E0FE8C32E780FEE1694409CE779EEF4A4EEB4DDAA170F78C516D00FEFBEEEBC7E16F00F8406000DBE0EE9DEAA6FAEC53FF00E29C6C70E699B4FFBDC1770EAC75E20F0C70FC8A75D7D23100E5DC84F46C8F64D9E93C9C9CA0E30CAA727CDF7DE0E88065446B0CAD1E01388D37C2E47CB64FFBA558158370FF5779ACFF316B8F504CCCC0FAB02715E1C16DC0FFCBAD3760F24CD16F80F2F06100E0379DFAA40070F852E2009C479DD0FBE1638EB4EDBADDF0F35CE165C0F8B166E4D20CE709F0F795E77DA91CF168C0FC216A0C29A409CE7C8B027DBAD09C48F0F7D'
	$sHexData &= 'D116799EF74EDC0F16F8DCE2E23C4FF3080FE2E8204D205604DF475E77DA6E0F9DD316480FFC1664EB4EF379FC02D4740F08161DF83CCF8C10349C0F46D4DFC98B34D41672804666D596BAF734AC700E9B06E7E07B537C0E14D6EF200EAF06D80FC737D91CBED616D70E63D737BEF7B46F0E27D846FC300EB81620107C4FF369C0E7D954F00EA2DB5807884FB7CF77BB1581070FC9DC7E16F3BAD3760F13DD16D40F5116E8EFF33CCF51A6F8B07FDFA611C137FB4EA60EA7E0D7B00E38E1079DCDD937400E67E22F70E20AE440AC03C45EAFEEB4DD3AE70F31E5164C0F3BCFF3BC7B16687BB97C0F27F894B60EE68E8C0EDFE70730E8A57D230F26E8B00E09EA469C0588D33E0E08EB16FCE7799AAF5B0F7516501275B1B70A10E768DF0F38AF3B6D17EC2E780F501690058855804FA7DB9DB65B0FFCED16C40F0DEE16E7E3EB4EDC0F3306EC121EEE620D2096EEDFA7FD56014F0FA3F01638799EE7750FC31650C3E960A7E6C2E7F078F197F186794E9AB31710F28590C4B629F439D0FBF28FF38E37F5BBBBEF698C400E50F646160F87F8DE37FB662F900EC7FA17D00E99FBC614FB9EF6BD860EB6FC1620C00E80FDBF2C9BB3DF0F96FE2FA0FE65FFBBDF1A40970F130026163ACFF3BA0F211684708B7B01EE75EF900E9E475E2E1E3AEDC9D10001061967B97DE77B0E551E90DB0E8E017701'
	$sHexData &= 'CFC9E6E40101D2010202794636672F30024B02505E7957E7397097072EB89E02D339C06C4E94F03C4D14F92E08037416373C5EA5E0F3406A0496BC045E049BB3EF6C88150E1A062F20069C069B239B23A006CA07D0074C087B938323800806080ED406E6C867A15F161E092009CF5B9E9354609E5E0909DC9B3C27090A0E38069E93E70CA7165860785B37794E80900FFF774EDE64C1000BDE100B1620E0C47B137C570C4F580EB5060CF6BDAF011EB80ED80C470FFF1777F034D30006DAD8C6DC16DBF03DCD4F0EA8500E700EFF16EF6CBBC9900F090FAE1CE00E3CF7B4239B0F18DC721276B4740E9EE6F3BCCA16A8CCF914F0FC82EF29F80EC9151E14E80EF31567B46F6473F4153116B00E8D184628FB9E66FB901916021CC4100E451D3FB3D9BE81360EAC1E2E1017B01E2A409C662521E8FFDAEEB4DD0FEB2216600F00231680F6795E770F441690AC12248E7C9EF79E34300E4E165050B924698E7C131FBC16372538257493B768370FAD259F16E4B77C4E9EE4F9FC24260FBA7B9EDB39173C9C7CD60FCC91CFADDB2F0FE017E01A27EDF33CCD1C27A73CA840287654DCBEB73B0F64282F0F8D1790B7E173F2CACCE328CF2EDF04DFC81F29200EAA2907AC0E522AEEF7BE776E16540EA42A5F0FBA2BE6E153DAF72EBC0E3F2CEEE40E682DBE23DCF71E18030FA506F6A80EC52DFBDE6D'
	$sHexData &= 'CE772C2EC7304E46F40E143116A2EF3DABC70EDF067C2E0432B67DCBE6C81F32283264320FA0F74616BC17303336AF33F00EF20679DED9BE887610342E163416B0CE6C76DBF6ECDC0F170F3B3510DDF779DED35C79167CCCEA354766F37BBF0F0836170F683614DE1BF9669F8CA9365FDC160237260EDEF66CDF820604DF1E0038382F164CD93C794EC0E6003A313A93E7E4194F75759BBDEF6C9FFC1A3B062CE50E523B2F239B4FFB0F343C2E944C3C7A3CA439F2397916CAD0013D1F3DDBB0E7A445C0F12F3EBE3D3EB23D7923641682F03F4619409C3CB737400E6917699AB833B279F2E15041764190D93ECFC9B9DDFBDCEC5E10429B276FB23942168C42446B4434473E23EF2B45494572F28C6C9E36485F486086F68D2C78004B661E4B400E694B2F9B7C276F9816B5F846334C0ECD93B7E25D4C7716F9684D85694F9E914DB0D6044E068253EF75EAE8EE9A0EEA0614737BB3B7A6B850F651A7180E601778A7BD7DEF9E51770FC417285206673EB76E803CD60FA617A60353377AEF69640F0E2706749E0FE4B97D6F3E53770F5717576B02347ADD0FA10660CE8D6C762B5F0FF853BCDAFE139E54E716D08066549F9C3C272F1650609090CFC973F2B0B0DCE06541CBBE9155D00E2756D70128A04056565491243619DF3234401E0147A0F507B491416464728B6CFD27981E6C6F62616C'
	$sHexData &= '4D7C7279B7B693AD53B745782A4638654C69CF28C0EC627261281849697A65782A5BB75326737448D08F0D163E401D542441F89C00C07346321331437EEF1900EBCFC6495A5175652C01B4D692500666CA6D38276F19082DD04C6F616457D66EDB6E1A57616B966C6C3A6E64E40A562CCBD6836DF0626CAB5468DFA36DDDD2A953306570575352904163940036F77175697217C0636B6880BA0D043069768E6086AB5DA931430E2C7A6CFB1407B3594D7500333080F839CE9B084015D053302C4F62487B0F50726C724274BCB6E1C1574B5422705024681A5319CB6A36444C6145E5E116DABA53440F3E83B5EC9966DAED150672588FEC2658AC571F4A234EC82600BFDB41D2134174500610B6B862FA6A2B09566CA060014E611A4279A3E142117960539605D68520CC6139353D3218F05EF19941B370D4414E5349A9EE3162578D478F6F5769640DAB0028B65E546F246C4370B866383474BCD224B31FDFC0A38A694D2A6167C49B8F1FF7DE3B9F80D74F41806367868C2A7A2342504A5137766E150EC1477F251055A45F77633CA0B0B83CED6936342862206CBE638EF3363516D81BDE59E5694487386D902C3C63835629E110646814664AD0B6E56988A658006C633C6F9ED66C50A8705E7E8E633AC768A0C31A5572886E0197A7FD9936AE0C7371727476E6887285612601615F446E2273'
	$sHexData &= '78026C7B5F344C2DDABBBB6F54EE2E0F1A5F00862D1C96190C40A06B74CCBDC189FF5F69F2986D5D34285E120A926722D4EC61BB5D287776726776453979694BCC46B84D7FD19D9AA3AA76D6646DBA3E1E57858143685373AA6C74C3449F4F08E855666359661481A30D6C747012E20E76F8F486C97B82B2246332736E0F04420032D6546D6D336BA26FB05F767377704EBA662C7D77D55B7C0EED5A656B6A24660C2D9F5987FD774AEE447075F329F8EEAF106763142858565878D98ADC776C0481F66609CF61AC5611E40476F97B0F1F73747269F6129D065C28123449060070187D1CDFC58042B500BB06772E31D31E6DD5B54B444342440B9B8DDE74137251572B6F9E02C30B9A427A585390DB5B12503099D1B8CCC94C14A6873390A63A39030F746825C404DB599F47725468699A4DB2C5B4B62B3539D37606C8AFA31261E17B0330DF75539D4F660F04A3163F4D9A27484249975866F0544D4150298DDF738BC258AA6D9BED4A80C22FCEB19F0304152F3253DA6F8A42BD0FC5342B4400246599D9C357B02384771D525E4927B8BA82AB41159955650CAF01299D506E896D1E0B365D48F86768358804C3261184987570E912ACC0C2534830357E3E151B0B6A6C78513F5F58420A1714856FBA7F1BB859284244405A3E3B081430D798BC083B337C1108AB0ACE65F6DD37C4405F4A4031'
	$sHexData &= '0859693C309180DFEE91637674265744555F4D6289C0871B80CD3256306C9E096E7315124041FDBCF3EBC2D6B492BFF948A1804728E05F1550746A6FD7D0DE42F06C435F4E46584577C127C7A8226F798741454673C082087478515B6D6D2C13D9256DB3474E9A1F526574687742436EC200C78064036114BCDD02B46D5FC93B77611507026C2F355F7008F36E1758705C2F695962EBDE14CCC07AA8FA500C04DE1632401D85850584506D8A6D7083CB58953F308FC77B0938496279443C100A347464B18F4603DB384FBB49040E6E2251B7CE8A8F4014D9CA4E2F2458C57E4F73667840E55B0202340B0B7D01A70D5FAE31A12C0593814B3B7D6958F020054889670E4C417172177366D8B5011D29804724854ED380646EEC4A45910ACC2C0C85EF2B90190261314721179D55F3315580F68124D84081A2B640E458C085B1012E5156E8D27963B0B74C45654D8B58F84DD05C92834781787393AB3806206D4DB9A0E8904178014D04ED5623086E93028120D5509931CB040D07E645481098A0BD695C65815885A21518E865DF802B8836B9E736E1365D50BB30D65036120702D32DBB319D03846A673231B7F41CD5866A8548834A5CD552158D854B8C061C954D2B51D817714D676F6F1ED10228C285794A75E120E044F65CEAD42E2AA1535E4B88A81BD1FD2C566661632E8B25348DDA02BC99'
	$sHexData &= '756E8C696699EE0B6CD0FD48BE33B8443126AF05EDE70F976F758F703B20DD659110AB33696EA95FB66041390E957043266CA7D51077617918B645DABE6E6F987640BBDD8666082E57021148406130330B58E6D5704851605F9E54470445201E5F49146E1C04DF0633FCFB45305A5C2212F179E3726C73D651C1672F69776004070C4341FE130B51452440EB184008B4046175104D122B32EC73A7486E55E2C64DD44370709F6F2E889C7030A3281DC3A5E4963331A31158A835716C7175351ED578CE8F6F70E58312DC18331704A1C6D84B7DA610831035815E4B83814BEBBB19202420D224EF978171E14165B69781EDF6496E434B5F11E363C165607786D504A10E58911B2A092C46EF6265823408C1A523588D624619068E71D85857AADC7082556E68F16445067018B8A1370430B7D6FBBD5F749160167D5B2537B303E0AEE9E8B05B31B1848F5D7D3667D3A1472C61BB45762843044F7632670B517B32E036798563812D173614587D65C63AA25D63648345F3364CD40E7E95573E6B49AABD2402301487AC585110132B63915C54BD489A4B0284D31818FA649085D838C7A1FB40A947135D40705C9C11620B66711BD826A80C13C642647706C2E84A1B74456BECEABA139469676EB7463E0EC228012823BF021F58C222DAD168D205B0739440F759086C06580079A01854DE5377604064'
	$sHexData &= '6F779A1DA0D4ECB53C497329563CD860A00307458843683056362BEE642F7323385D6B18DC2E794D6F1836739B5408CC09781C6D18526817026A2C05734C50B06ED500C279D7533105785C7D6D6F75A66553055B64766BE157FB41569B4620934D107205CE53A0B2B684980CBB31786078786FF1BE500360F64C730FD4CAD7D4E102315F995F64D53A05BED15F6C743B433E706D021C8A4C66CC5F7267D4C2468457673106BC9581A22A704A760EC050BABD54EB1C300730A8E87AB077000E1FEE0F66A281F229889500558B463FFFCE9A983048FD340001F23C5708101008373FF7233808F0F026211D1808200D08BC6A17620148011808236257841059BACD5704A9F0100188181DE8FDFFDB57A0A810F09024C030B02828F0E00190F07002F008013857FCFFA61EC0A0C0404828409030F0C0013020F010B05838F7DBB7F65060F0804060046801283040260270304838302AF86BA6B24012306ABC035030605070EFF69B6DF80240327C6060B05050484AF0D002BE5020BC99DDE0803E0AEE1A1801480001ACFF010C64860600C875F468DD10D8FC0F0022200B020E3200484E01FB9640467D58158023E636F5DC1724063F0695A8C9410300AAB2D4C6B6690D1F1591A945A63D2C532B42DD40C1C03AD33A360E2501D05D4D649F8CC420A7BA5601A8DADEAF247DE0A5028F016AC4369AE5'
	$sHexData &= '01307960693DCE274675BA5F60F78F8C461DF48B203D67EF329D06E3A700E44CD43D7F4D45EC961F9C40044FF3BDC90DE03D4E50033C3064EAE49103862000C093E73D58D5C7904E226C6E646AECC620BC9FDC45F2ED43860437E18E4FF6205EC561054C650E4616EFE4DBC9D0944F06216642378486F07D0093EA0F57E49224491200000054FF0000000000000000000048894C240848895424104C8944241880FA010F859B02000053565755488D358DABFEFF488DBE0060FDFF488D87547C0300FF30C7005F25008E505731DB31C94883CDFFE85000000001DB7402F3C38B1E4883EEFC11DB8A16F3C3488D042F83F9058A1076214883FDFC771B83E9048B104883C00483E9048917488D7F0473EF83C1048A10741048FFC0881783E9018A10488D7F0175F0F3C3FC415BEB0848FFC6881748FFC78A1601DB750A8B1E4883EEFC11DB8A1672E68D4101EB07FFC841FFD311C041FFD311C001DB750A8B1E4883EEFC11DB8A1673E483E8037219C1E0080FB6D209D048FFC683F0FF7458D1F84863E87238EB0E01DB75088B1E4883EEFC11DB7228FFC101DB75088B1E4883EEFC11DB721841FFD311C901DB75088B1E4883EEFC11DB73ED83C102EB0541FFD311C94881FD00FBFFFF83D102E81AFFFFFFE960FFFFFF5E5F5889074889F7564889F748C7C600480200B2095357488D4C37FD5E56'
	$sHexData &= '5BEB2F4839CE7332565EAC3C80720A3C8F7706807EFE0F74062CE83C0177E44839CE731656AD28D075DF5F0FC829F801D8AB4839CE7303ACEBDF5B5E4883EC28488DBE00D003008B0709C0744F8B5F04488D8C30941404004801F34883C708FF155C20000048958A0748FFC708C074D74889F94889FAFFC8F2AE4889E9FF15462000004809C074094889034883C308EBD64883C4285D5F5E5B31C0C34883C4284883C704488D5EFC31C08A0748FFC709C074233CEF77114801C3488B03480FC84801F0488903EBE0240FC1E010668B074883C702EBE1488B2DF41F0000488DBE00F0FFFFBB00100000504989E141B8040000004889DA4889F94883EC20FFD5488D873702000080207F8060287F4C8D4C24204D8B014889DA4889F9FFD54883C428C6053C000000FC488D8E00F0FFFF6A015A4D31C050E829000000585D5F5E5B488D4424806A004839C475F94883EC804C8B442418488B542410488B4C2408E95126FEFFC356488D35C860FEFF48AD4885C07414515241504883EC28FFD04883C42841585A59EBE55EC300000068070480010000000C11048001000000548C03800100000010110480010000000000000000005000000000000000000000000000000000000000000000000080000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007000000000000000000000000000000000000000000000000000000000000000700000000000000000000000000000000000000000000000000000000000000070000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000017070480010000000000000000000000400100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '000000000000000000000000004050038001000000000000000000000000000000000000003867028001000000486702800100000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007CCA0280010000000000000000000000000000000000000040670280010000005067028001000000586702800100000060670280010000006867028001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000200100000002000008018000000600000800000000000000000814152A90000010001000000380000800000000000000000814152A9000001000904000050000000A42004005803000000000000000000000000000000000000814152A90000010002000000780000800000000000000000814152A900000100090400009000000000240400910000000000000000000000A0C00300580334000000560053005F00560045005200530049004F004E005F0049004E0046004F0000000000BD04EFFE000001000A00E90702000F0002000300000000003F000000000000000400040002000000000000000000000000000000B8020000010053007400720069006E006700460069006C00650049006E0066006F0000009402000001003000340030003900300034004200300000005E001B000100460069006C006500440065007300630072'
	$sHexData &= '0069007000740069006F006E00000000005B007800360034005D00200049006D00610067006500200053006500610072006300680020004C0069006200720061007200790000000000460013000100500072006F0064007500630074004E0061006D0065000000000049006D0061006700650053006500610072006300680044004C004C002000780036003400000000004E001700010049006E007400650072006E0061006C004E0061006D006500000049006D0061006700650053006500610072006300680044004C004C005F007800360034002E0064006C006C00000000005600170001004F0072006900670069006E0061006C00460069006C0065006E0061006D006500000049006D0061006700650053006500610072006300680044004C004C005F007800360034002E0064006C006C000000000034000A00010043006F006D00700061006E0079004E0061006D00650000000000540052004F004E0047002E00500052004F000000340008000100500072006F006400750063007400560065007200730069006F006E00000033002E0032002E0030002E00300000003A000D000100460069006C006500560065007200730069006F006E000000000032003000320035002E00310030002E00310035002E003200000000008800320001004C006500670061006C0043006F00700079'
	$sHexData &= '00720069006700680074000000A9002000440061006F002000560061006E002000540072006F006E00670020002D002000540052004F004E0047002E00500052004F002E00200041006C006C0020007200690067006800740073002000720065007300650072007600650064002E000000440000000100560061007200460069006C00650049006E0066006F00000000002400040000005400720061006E0073006C006100740069006F006E00000000000904B004F8C303003C3F786D6C2076657273696F6E3D27312E302720656E636F64696E673D275554462D3827207374616E64616C6F6E653D27796573273F3E0D0A3C617373656D626C7920786D6C6E733D2775726E3A736368656D61732D6D6963726F736F66742D636F6D3A61736D2E763127206D616E696665737456657273696F6E3D27312E30273E0D0A3C2F617373656D626C793E0D0A000000000000000000000000000000D4260400D4250400000000000000000000000000F6260400E42504000000000000000000000000001B270400F42504000000000000000000000000003A270400042604000000000000000000000000005B270400142604000000000000000000000000007A270400242604000000000000000000000000009C27040034260400000000000000000000000000BC2704004426040000000000000000'
	$sHexData &= '0000000000DD27040054260400000000000000000000000000E727040064260400000000000000000000000000F32704007426040000000000000000000000000000280400942604000000000000000000000000000D280400A426040000000000000000000000000018280400B426040000000000000000000000000029280400C426040000000000000000000000000000000000000000003C28040000000000000000000000000048280400000000000000000000000000542804000000000000000000000000005A28040000000000000000000000000070280400000000000000000000000000762804000000000000000000000000007E280400000000000000000000000000862804000000000000000000000000008E28040000000000000000000000000096280400000000000000000000000000B028040000000000A028040000000000BE280400000000000000000000000000CE280400000000000000000000000000DA280400000000000000000000000000E2280400000000000000000000000000EA2804000000000000000000000000006170692D6D732D77696E2D6372742D636F6E766572742D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D66696C6573797374656D2D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D686561702D'
	$sHexData &= '6C312D312D302E646C6C006170692D6D732D77696E2D6372742D6C6F63616C652D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D6D6174682D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D72756E74696D652D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D737464696F2D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D737472696E672D6C312D312D302E646C6C0047444933322E646C6C00676469706C75732E646C6C004B45524E454C33322E444C4C004D535643503134302E646C6C005553455233322E646C6C00564352554E54494D453134302E646C6C00564352554E54494D453134305F312E646C6C0000005F776373746F7569363400005F6C6F636B5F66696C6500006672656500005F5F5F6C635F636F6465706167655F66756E6300000073717274000061626F727400000066676574630000007763736C656E0000426974426C7400004764697046726565000047657450726F634164647265737300004C6F61644C6962726172794100005669727475616C50726F7465637400005F4D74785F6C6F636B00000047657444430000006D656D73657400005F5F4378784672616D6548616E646C657234000000000000FFFFFFFF000000008C290400010000000A0000000A00000028290400502904'
	$sHexData &= '0078290400201201004016010050190100601E0100501E0100A0130100F00F0100F01001000015010070160100A3290400AF290400C9290400E0290400F72904000E2A0400222A0400392A0400532A0400672A04000000010002000300040005000600070008000900496D616765536561726368444C4C5F7836342E646C6C00496D61676553656172636800496D6167655365617263685F4361707475726553637265656E00496D6167655365617263685F436C656172436163686500496D6167655365617263685F476574537973496E666F00496D6167655365617263685F47657456657273696F6E00496D6167655365617263685F496E496D61676500496D6167655365617263685F4D6F757365436C69636B00496D6167655365617263685F4D6F757365436C69636B57696E00496D6167655365617263685F684269746D617000496D6167655365617263685F684269746D61704C6F61640000000004001000000040A748A750A758A7001004001C00000010A178A190A198A120A238A240A248A250A258A20000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	Return $sHexData
EndFunc   ;==>__ImgSearch_GetEmbeddedDLL_x64

; #FUNCTION# ====================================================================================================================
; Name ..........: __ImgSearch_GetEmbeddedDLL_x86()
; Description ...: Returns hex string of embedded x86 DLL
; Syntax ........: __ImgSearch_GetEmbeddedDLL_x86()
; Return values .: Hex string of DLL binary (empty = not embedded)
; Author ........: Dao Van Trong
; Remarks .......: PLACEHOLDER - Add hex string here to enable embedded DLL
; ===============================================================================================================================

Func __ImgSearch_GetEmbeddedDLL_x86()
	; This function holds the hex data for ImageSearchDLL_MD_x86.dll
	; File size: 81.5 KB
	; Architecture: x86
	Local $sHexData = '0x4D5A90000300000004000000FFFF0000B800000000000000400000000000000000000000000000000000000000000000000000000000000000000000000100000E1FBA0E00B409CD21B8014CCD21546869732070726F6772616D2063616E6E6F742062652072756E20696E20444F53206D6F64652E0D0D0A240000000000000080D51F0DC4B4715EC4B4715EC4B4715ECDCCE25ED4B4715E8F3E755FCEB4715E8F3E745FDEB4715E8F3E725FCCB4715E8F3E705FC0B4715EBD35705FCDB4715EC4B4705ED3B5715EB23F785FC5B4715EB23F715FC5B4715EB23F8E5EC5B4715EC4B4E65EC5B4715EB23F735FC5B4715E52696368C4B4715E0000000000000000504500004C010300B475F4680000000000000000E00002210B010E3200400100001000000090020080CA030000A0020000E003000000001000100000000200000600000000000000060000000000000000F0030000040000000000000200400100001000001000000000100000100000000000001000000040E803008001000094E40300AC03000000E003009404000000000000000000000000000000000000C0E9030028000000000000000000000000000000000000000000000000000000CCCC030018000000DCD50300C00000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000555058300000000000900200001000000000000000040000000000002C461D7400000000800000E055505831000000000040010000A00200003801000004000000000000ECD61F7400000000400000E02E727372630000000010000000E00300000A0000003C010000000000EC16210C00000000400000C0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000332E303200555058210D0908'
	$sHexData &= '0AF7BBBA4CCCBC9220EBAB0300742A0100006A030049090039F73FFFFF6A08E8090226446800025B60C7400400A300034788C7000BDB1BF63F36297B83C408C3CC01B92CB0640146E9A67D95EC9041593DCC2BC0CB25A72DBF5C209CC94126909C685C50B7DDDF1F558BEC6AFF3443F564A19950510C031040D67677FF33C5508D45F464A325C745FC0D6A1810F036FB33FFC1114C0183C404890089F81150C7050AE90EE9DE5857135C607FC61DE477B702B941506A103F64076808A83CD886134C0E803F4ED0CC80E0DD822F91048B4D08890D8B598BE55DE4C1A4C16E8BA0A25E577BFB1F2C9E568BF1FF7604C7063A9A8CFF150AC3FFFE7F63688B4508A801741DA804750E5623508BC65E5DC27ABD1DBE022056E631BC6423FF31C10D4FFB4144C39F83EC08278992AE356838B80CF8218D0C50CE7F3EF3FF9F3C85C074038946081D74270F57C0E1C162FB0F11008B56B24DF83A188948337870BB6650085E1EFC33CD6C257EA51B78B063ADC01E5E21CC01FF714CAC252992015CC241B8FE20515CBEF09FDA8D4D14516AE0B6680B967510040CE6726CFF702E28F8AE52308762FC83C9FFDE1C440F48C11F29D8C0DABFE4E0C458C4894424FBDF0BFE545666895424162A8BD18BF21218578B7D083905D0DBFFFFFFDC744683FF1072410FB74C241A660F6EC1C4E27D79C0C5FE7F6820'
	$sHexData &= 'C5EDEFBEFDF8770F1F4000186F06C5FD7521FDD7C885C90F85FA036DDAF7FFA983C01083C6208D48103BCF4E76DA12040F876E5BDBDA3440BF9726080C61C00670C800CC367FDB387726101F4400000F100675C10698DB356DDBEAC966A0752F9808105C4A76E0177CD87F937756F30F7E4FD084D28B681C7440CDF0BF7D67B6C0BCC9D1E903C88D044A5FF4D65436481BC333CCD5352A3BFE0FFE8F8BC13BC77313668B743E3934428D0C42741840ECAD66D82472F23257205C2C9A0E2BD6251FBFF810EED0DA490C5680895A04B8182AD7D01AC889C008C5353061183C0AF884C99F020C2B3BDC9B83F9FF9D0503C220A3C2FE303876DB7A1D7739C7A2343C6CADE6FF6C667F7E0E840233B4CAF3C8C93BF41089C2E1193986108FD87D0AFCB07DB79761D180FAFF740EB6CA0808F7D1601B363CAE0C672C044E946DC987DF701BC083E0FE401F0C5F86641816DD0C1A87DBF6E7716DB70C4606144266CA759BEBA8D61C5EEA5C14F365DF0BB6FAD3D68D460450E78AD85ED600EA91094E06C220E1607DB4BDA5BB5E8B4904B8268B083245C1C3BE66D9857F7877635CF2F6D64BB1CB82640B6A58C195852E6927B6018D417201598160B0A87F32B0221DDB3DBB4C04F01C5B8B004B51A64A30D96D60D01F68D89CE080C1F1562038320D037615EC4B61B80B5653A8C065DFBC56ADD861B6DF'
	$sHexData &= '2D542072367DDD100860ED9B02E35089DC7BBE5F5D331F0CBEF4AA060C90E4F014147FE072AE332CCC1FB0D570B8320AA57FDFBB92578AE468AE344D084D498E027F38D2F9FBBF154FF07D704DFA0A85F6792FF7DEB8CD54CDFCFFFF83E902F7E6C1EA038D049203C02BF083C6306689314275A1763BFCE083C1FEB82D202001EB2059811DF886511EFA50518BCFB06B3C03230A5CBA8BC75FBDFE0713F2BF0CF488648D4E04C645F801F0B0E8F09CF41C3D160135F386CB09B20F9978C3E4040D87209F99787EC20EAC8A71042960B0A1035B076684CA147ADFD8D244F88B550C8910BC7A083FB851039E018D550208AF52FF50FEEA6605E27504548B5604163B4A04750FDB37F83B103B067509B0015E7A8032C0119F094C617F708474603B426CBBBF0A828B063B450C75075C59E95D38991E8911DCD2A284AA05EE074B538BDCF404A47293016F966B04896CF005B8184D8E765342202E11FF4DD7E15040EC041B8B532085D2743EFEFF37FC8B73248D4B10522BC283F802721E83FE0F8D42028943200F47FFEFB8D82A66A17044120411C644110200EB116A0268F48F267023F0E2021EC16C8B4B683018DE7655D4FF73102A1408661B7E8D34011E837DE8084DE48414D9FCFFED8B4324C2202BC6894DEC51523BC87725837B24040E864E37FB8D4310420603F056DB1AEC1BBCF61D72C4'
	$sHexData &= '0CC60406CE0CC0512AC55EF0DB3F4DE883F90F76283ED44181F900107272128B50F06F073B58C12364C00EF81F77478BC25150F0374008AF43C70F1107C74710802DCD3E61C414142854C21EE616F0603613557CBFC55F3F8BE35BC3B9051CCD29DD4432D4809F147E37C9A71F24539B4DDC8975F857989AEEF8068B7D745D0853BA07BE07D36D387C6EF00FD43D008301CC705A4C062B880C39A653F48461F0800C213741DC2D7BD30AAB871E7E10C65F895E0C81183C36BD9B185E5BDDF90DED90AB10DF2420D0105D1516B4562218908BCCD634075070B2E61F1A55CCFF754C3E2E0EC1DB1B9A098C8BC8DA208379143666746130860985BA5A1C0B41FB12B79A50C60828A77A10390E82A929D8BE99091922AB320884057F5B6F4E0C8A24895610B3040339D8BFB40CBF35382818A8D174046F7C74F479E2100CBF1A0B3CE2E3631C10C2880E2958018FA13F75E9108D47008320072607E255D060B7212799844946384E47A5E0D6B0470AC7894F40075F6D587683D89F92A59F6066249693DF31817CCCF86A146EC3BF028B48D0010BD751567023C86E1ADD0D2FD08BCAFCC48806C746F55F695F8D790112148A014184C075F92BCF028CBE589A8BCEE86D8C5FF0BF791A390D33CA3808A70A7529FF505F36ED5C439A225A1032832A5C017B454AE83EDF8B86A6516EEC2A45360E9ABC13'
	$sHexData &= 'F4D0DB08604EA54085C974087A7DDF205A5251EB076A0DB0973421CEE060410A9791774AD7E3A1EEE83D721D050B2605A0F0AF5FAD064C4E3720D1217E2B58AA4E85B804893EEB1B5783C06CC1A1909E2E750B22FE70B0D95F9A8CEB0989061584FB5F880E15D899FF8CC35F4A0C42260F7FB12A7C2E5898688D9B3D87183F0FCC607EB3949C880A1F98682E4C48EC33C021868411AE8E85079B8E70108AFF0715B66A1812C33F481481E1FFF3FFFF816D086BB0C9AE68891C103FE508B90CEF20F941D65CC908830423FFE3C557E0106AD957895DF495B46D933EBDE0F88955FC61E8188BF062FCFDC52FC08BFA4880969800751CCA18506A6457232AFE1BF7359C89038BC3895304AA5B587FA2E7B74600366E01754D494245340C6ADFFDBEDD00682B8BDAAE53506B2BF029CA9A3B1BFAC7B7BD6D8F495250653B53C4A7B70F7BF8EB3A5393BC8BF35B3356517A76508167A275DE514C5177079FA08D40FA76450384F44A2458F013FA8978508930559FC921908B809AD0539099069F4D9DE0ABCC53750889C5FF7610C2DD781FE8FFD0561E013DD8620CAAA30D10913F0A4BD909105660EB7AEB6A0795080C8056C6FC4975E08F40051260897608AA89AB2E069F32D9F4AC7E074FE432FC80510C00134C0E6AC05450F9FE082C7078F0FD9842A660DF70F122C09534B9000C4F53A542C399'
	$sHexData &= 'BF6AAAD2C252AFE616E07F26183C65B4790F837DF8007466FEFD7505122CED0D442C8B0C887848003B810C0FD0C2FD267E2D681975DD80833D29D7EA701B70178A20599B1D8E3F8C2CCCD13EDE8CF70447E80648D1741320A178403AE52B57F67AAECC55227FF806E210F02B1D589F85448F0E0C38AB9F379D7555E020085347D96DC2770942F88D4F088701CCF0FF588C0274085F3E4382F301F2021F7F407E8D730C22EC7660D468773C945415B6F82C3DFF007FBC5BD1DC701AC84F6E662F1837871532887AAFC864EFFA1DB0E48939CA00837B3C3405894B3CCFE2B3DAEB065840704B40CB94FFA5E8FE7CC7EC807DF300AF16F1EF050577A6F08D5FC6CE4AF00FB10BC06371D79D134DB903E7C62B3BC60F1557F0DFD283FE0275478D7734565B741805CCFE107C8BA863750C486A0689182B9E5490FD56C6476401F3FC0C33E3D6F0279E83C8FF2604B4C10148C74084994ADF043B5BEBE490A161DFCD0C0D0F0E23A9C000B11872404E178B1742E20DFA00125133D2C03938740A06813E1436C110F1EB749D3308207A03894A2C6CB810FB10A4635340345072B80282058AC1E08703A29CC49E10A2388B27EB07CB4203FF19BC3E1777B9B9F38BF085F6747283FED8A7D8FC037707FF248517983D3BF0745CC76078517CF8037450037553F386EF0F9F807F6400E475168B1D56E456F6'
	$sHexData &= '46707783D35E223174F05619C1500418EDA9900C860811D87B073F510C4F07B8C18B0ECE103312E009CF4105DE04E943814103815FFD708B06207C8B06B4684A95E88292B0CEC192E74435DF996C042688FE0550578D7E2857F547E13F19280147837E64027D15536776B79CFD5756277CF35B8DECC70D6E0DE2C3BF7179255783721808F0618BC75C4604FC93BCCDFD138E187E084F75091B5CFF60676CFED4275F51A111FCE554390523965F9AD00E15FF2983C104E909013FAC1F382D74002F404E24E606FE9F233E138CCB2374158B113BCE0F9523F005AB18B6C00A5210C74624476155A4A128F76BC302F6F03F313C278940AA156803F857B72E9207D7BF387CDA50E603CCB0D0F9F83FD4BB8DCF12F0C747B93A33C98B5627145FA824EE0C8B028BCADCBA1ACEBB7F894F2483FBFF75524E620241934B06194FCF471629B19257DBCA0DF0852BB61345E83EAC8281D6E4C08AD0DE8B5670DBE719D03AE80CE20357680619A0BFB0E008F726EB08971203B98E87CCB421F11500A40ABC071B3D070BE4D53F38A0D386168A81CC317FC0008214215F10B4D4DFEFDC0D91D4CC0DF88D4E7A751EC0484ECDA3D0811029D1B51F22B7E18E84DCA20C500117E8F71278B0EC146CCCF4063C0944D520449CDF98D913DE70AF94895DF85606848903C743EDDB0F469A0C080C5141082B8CC1F878'
	$sHexData &= 'AFAF8602F6FC4C34373F773757D69AFF2D944B04CEF03C8B75FC8B7B460476524618AB43F85731EE2950A1FFC1FE028D0CB7894B083AC3151F5E357C62979CBF8B09C3F01E776325DF10523435D4A73057655C160CB56788A45F04660E08487BAC6384467C463038386D8491D9180D1C3C9A2814506623970240440D4C627C7334EE06407D1083CBFFF44F1969052C14425054C4587D47A079066886E4DA869420DF4FD93B869C4F7C002180877C67E786A0F3780235A4A89A0F966313AC04FF744147DD35E89902373CBE33898EB00A5B61F24BB4677534BB0A77D66510481D087525831C793403122C7598EEBCE4906387C013C4B800CE7EF889B6C8B786CC1D4648F00EC6600A43521EF561C42F721215185F084B350485E350F85B1A5060C4E4C5FF38E01BB52D369B4BD280C703DF83F9E40FC97C5558C6178D8BBCE1FA32F8858BB3C9141F0F60BA6691161D08117380145DA762F65BAF6490C7DF390D10896555ED0045A0F18080B600D65088576FC07531DB4DE43E189C73B71FCEB3EB10D22047C31D8B613CFBF7CD05164D837E0C7C0C7563EB065154F00D3F8B4E0433C020040F94C040552AD60EBA7EDD21FC08204AB0BD9C1E176C5026D72D70DF261620DC26CC2F2CB04160DF5F958BC81C1C6C17DF07C821535F304D94A3188AE02E848B30057FE6EEDE8D81B857D89B9C6A28'
	$sHexData &= 'FB9ECEFFA1A9D8895DDC8D737803BE470003020D04010D086A70B558F704F888D890CEC01CDB0041403B3828180C02C8A0039650D74BCDE6C0A9E46C045D33A842ADFCE040E4BE6A664E7107DEF004FF554275E88AD873CF8F8210F2164E7ECCC3280E51E38AC3C5837940438E8E22EAC33F95FCC22F9014837D0CFFF64DEC4E580C8E752E683B29E186058301648B89B00BB6D93436561810188B4F04D8098F8D985A37F678645F4E2268C30B7ADF7521221C9938710345E851790540365F3F39D51A8149063457BDB8B4B8C1D8B24026DF08032BF1CBF1837F04236B45150C44FA10FE3E16EA5A14DC8057F4E87419A852776ABF6018ACE919E92023F88B118D4187BC9314E6A43652EB301CB7A2690AE8108D8FC84BDC40A1F041A856461498F8C3351528600CFA10B80832BA0027DAC0FCC094E0C4C3EA4DAF348BD0391A7C02A6FC08B1DD1CCA403BC8B1EB09505601AD610DD8000CD0EC8B014D87028CCEC609011870E877084E3F318C42AD793D9F3D0E39E43A4648EC9211503071C5D97CE0010984A8A055760BF576043EF5D7013C46F070EA94DC5423D08B03105EB74A7AD870083C37B060F0EFC912003BFE7550F608FF335631E274C95ACB17572C88A8E7139489D53B95F855D48D7A08C607208B48F3A7A2E045B49CAC04ACB48955B0455BE05AB818D8896740D0BE32A3E20604'
	$sHexData &= '288BF440E0078211B00D4AE44DD092D07260B0825610892CCC880569F148EFC8354824DD79AC2932BD01E92BC8C0D8F0404E2E08D824E163E01B97DF401F2CA3183168700E68D004E38439F4023F0F750807ED12685AF23D5F648E44C8081F9D3241203C385379F0073B1AE901492C331A0B861D5E0822BC525224E71BA9AE2F30B11CF061F81FF475F08D5EAB28895DE4C743D39FDDCC80058285537E2C1C897702D2E11ADE11B9BC4DA579ECE62E3108C2D214135468E8560CA7109CDC8C7F3A51B96AFD8C0FF18D4E205D4E10F7703B2CB8CCDD994E0CFAE139B1B84D558B760456331251F3FD521EA800DF618D4630528AB4086FB470357005DF087390865A0237BC5ADE11A2003E82031194F9EA4608E09FC6E403771FE97E023BB5288C3EC0C30BD079AFBD72EC8C08C33328903E700676013882E77C825F1481E5CFA6F8EC27EC608D45EC8C928B989F90905C4108DF7490909D81FF8C145F3ADCD1520C288D41FF0F482A00BAC32B772BEB130DD0EB0CECEB05BA1506AC81819A413C35F0AA3B66903FEB2F51FB4E550DA10F1F40297806EE0D3CCF7F868D9F0DB2C3F00B14087F74F22BF1D1FE57BFAA6201100F8C0C2BE0018BF80FB7D81432FFFF83E0DF2D41003A00C1EF1083F81A730A8CB5DAED2591C3205C7409082FFA1F6DBBBD60C776047C646659064406EE832EF3FDB5FB'
	$sHexData &= '3C072AFB2F754E347412DAA76DFB746705750432EB3F227902F6376BDEB1DF2B0E46043F58082E740D4AA0E97FC6FF3F7510663979220A44DD6ABD4A2BB7C0967C410E83CF5B8E73341A2B7426045BB315F13BC27421FE30EAF92E15835C8CCD9BF91083C00275EC19C1F31EA566810A2A1C7840C9E7E6203D7F0575076E82C7BAB7563F72FE1CC00C83FE46E2CF6640C2FEEB8BC25E318242D7C0DF4EBA7DFC97EAF806723BC704240A1C1604722FD66FA07BE6BA196A0E040C61780CFCED19BE4008F7AADAE00EF106B0014F32C0EBF0A0FA459F0EDEC3FF1853F0BD05A12AB3F3837B1407FCDF74957C338B4B2C4DF88D14098D04328955F0458B052A5640D1F851F8E7B87BADE12DDC06E50F8325837DFC1E416968D016B646642D25E05D78D2102F7F147A897DF876D7B6DD15920FB4D18B47C85DF4C241BD3C6CCD5EC33003101C34FC55ECAFA7E8DD03828BD6E619B2FC1C62C06170F276F8BB591F0A28E8AE342B68D1FA3B4A86B36C51550042C270F235B76BEF11D52D7E3B753282110F870B3CD8163444E8D8F003EC3BD331F677BF15155C023720397710612EA5E9E1E034781AE1164E0161F760B70C3BF0672C550C3BFB152E9ABB5828B8611B9614DC2B8BF78E2CBC4C3B1F37524AA419A2DF4CD3F033C966890C308003510D84DF22FCCFBF07B6F6C04C2DB18B4310B757141B'
	$sHexData &= '7E58604EDA77327C83FA0783C8ED860D50008980565351C9F89360684350C589C2D46AE1EFC17F71076570EB4F1EA69F468639D4685842EB0E4202C195CA40FE37328B77029A7A56D7F43BCE73185601EA7F7B11FCC1283C485CEB106A5C9DFEEFC3E1858CC81E2BDA662BC1D1FB3BD877389C6151E1DDD2199A44803204BF39451C1B508D0C4E5281FC9FDF800C4652AE8BCF53A12C472C7C6B2F1ED64C5F1889C5C22114040756C1EF228D22627C240CD4318B4164500D5551CE9A465D144F642F160E2BC6340C00FF7408DF6B235BD48D54905E1614A418DA6ECC12031C0B5B1610AC225C89C31F6330C944D0FA1622700602942BA51FA994498E1410E3513950D9895CB4EA2039CF34FCC0D4A247DE242B2C2BF015B890E7CBFECB894474C345440CB61EF75F946E0C393AF507B3501518A7B64B681CA0192FF474ACB8227B874223F10180100E890AD5878460C74178958BC140C3E151D335B47E8F369AA430E3DA728955B4AEC472F786FBAA6D7E45C880751533D2BE832A6AF6058855D88975DC922CEB56C6331F2B54B8EA5056CE6F2CA59418598B38C4390B44A65B28BF5C85476A5E55C82A7E1311D698D0741066833A6003E0C1DBDE863A83C2021E75F0EB043146D28E21C6663C690CE48C44C0DF3360689839FC0A397C5F55E09358F02BCA5152561A2F5800071EE06A136AAD09'
	$sHexData &= '4C9334A0D59E4EB466BFC0E4A31B82B85E7BBD417B296AF932B180D86A42010CDAE05C4A197A56D872C1CA5019694D11B051D63B3CEEAA4FB4B2DC5EB1730F4D30220AB5A98D463F303E6F453F4375E090E468E0897DC03BC7FF08B7015933AE6ABF8BC683C608BB8DFAC56A01DC8C288B387A50576CD845ACACC033261084339656F016ECAF8C2C14B8573578C43B2EAD28986B2AB8B683C192FB4C8C0528388990E0DA3BF7D570BBDFECAB31E914A03BF225321F400ADB9FA96F5F5A8D700234741853E74AF79F0E2F0C0783C662FCAFE2A42BD4F9883B7DE8741289077C51C366BB4FC983C7B20AEB5866C424769F06C1B8724CE09E8C36461D02FC0F56EB3D397DE07406942874722927BA2F79188C86470BE56D9A8512759D5C77F6FDB086C9ABB4E908223B4CCF70C30685343B228D79F0297704FF201B26863761340A7DC47CF7D9E1740BC20C893CB7CF8BF78BD7F62DEE5710740E8BF88BC84450E8D02BB4169876EBC1F903D6722B44607525DEBD7618E972F404F0C1A4C7C5E20C0EA4EF6F78B2ACC741F848B834EE6690D6B4253DBEE0756E84DC3BCA7357B1FA0AD18B37E95ADD8A807D42C5EBC40A4583CAE8D188C214CA02932D85C4C513EB5A8BEA30C02BD3AB49AAC14290C41ABA76BA0E7C7735837DDC880AFD75061D70A06D18125072B48A89CD55F5CDC40345AC33FF26'
	$sHexData &= '3C20D8EBF08177A60EA4B6B45285CF042E43BA76C7850CC2BFDB65A6487539D64483F801721F33EAF5A5807001C9A18DCEE8CDF53230DC5E084102EB0FEA2788EEB5A2C8AD7DB01113E6D0E68C3E095E868B89011040BD02281F131145D8299B82479A079D7ECEE8C1F8426257800382C53381FAFBA67CE57F71FC83C2232BCE8D41798BCE5251D8C5A9EB8DDCE4B535BBDC09870D7607762E9E2E0C4DDF484B2618912E4CAC50899DD3AC9150E048A68972C8290ABF3B5720261EC004FA88A434B4E648EAC80ED0CCE48B8570A88CCCD85010B0B7C58710B82AD8B018D485D2338CC61E5337A9E6002A779D787C138B01E78235C02A47CE8901D937A937A94FD46EDA17C83F9009DC03C708E4D78C2AE1164807534AD0AF54F132C08845A657E603C133C904C7456C6203C271073B1638818A860E845B118A8B25BEFED01C84525FFCAE702CEB1E86B8EB03FCB8857F6E67AAA5B001EBA38883835F233EC275E4B2060A1E5D236D74DC575D146259DC6A4EBB9A6AEBA00BB991D4087786B5F65246D404D84BE090F6ED8960E0D50E33D24C1441EB2057241BD554130E53E04F002B9C83CE4A038AF0C6A0A2D0C6B50843D0A3853F146D2D3A100564A26B21DA0DC6076B4B6F065E9FA29DE85BBF182D4604C145468664221418530DB19A7CC3BFF08311479C88D0C4C0E3C25021C452FC0654A3'
	$sHexData &= 'C9856EF410A0B441A022FC5D871C3BC87541D239D852232E3FEF49AD7D60952F7F3FC43D8D5060401010C436309CB82E1866102C2BAEE0881C45A8C6631C1018AE0612F2D2577B2D32504704161A301935AD20BE1489771C5FAFBFAB53931CA308AB4D53194FB4E83404100C44427C1236FC7D3018E0545F97C847AADA3B1969B02428142A2C0C3CF10370D222402C888D4E44BF84B77166BFDD0A0F45F8C74169828B41E98E10145C5AF00A628955EC8A1084D2A6E8F052A50E57A5F1A0D03857930C009F95AF44723BEB74EC7B409A82D70C5078CC1410F09460415E1457CF2BE0CCC160EADEB2C73D9DF00056BBAD1ED465B603561E733AFC2C1A31E000BD86BB2AC10F1C20C1548BD48D48C6823F3A0484EC5F8A08D4DD441F96C93A8942A26A508BD744AAA1F1823BAC7050D5109089E445D424B09A2CC15B6C9DAC807E253E580F8D4144760747C92966DF3C3EF218A33D00CEB004B46F18886021638B0FF33618C6526AC14201010FF0A922D2ADC825612410C07AC14E872F54E1D15082BCB24C10E902C002C99F78F22AB1E80D347010B8514CD40CF05068E0D514365C2038C49E0F95C1708D0442D3FF7F03CE034DCC03C8394F1473168B77DECE6CA43FA4CCBD86EC7E5650DD2105C0C88B530C16D350E0DD5E3BF07731EC0C2818C9D73EA6B8E030177E2152B85603C152503BB8A3'
	$sHexData &= 'DDFF66315803C6C6040800EB0D5652AAA8566A55B64FF1D48BC1C630D893BFDF7231808D46039783F90F6A0374B62BB02F0034A11230039800CFDEB3133633A316A558F10173A2164C54707DE4494FDA83D53DFA96C677285CCA33A81ED2D7CC5F78501B124417E438462A95ED89040AA80A04043804CCC6D958DAE8C80767E382E10475E44AB8740410D6CE6B9F43A517CDFA0F66C758225A551B8FE26A220DC65C53C441694C6E65D0426EED2B5B3E78C6700083BC841484182C577108815401DF3EFC844032B5835C649037685F58292C8B46441602C14F25923FEE31B35278149704580F9289D04CC68A40EB2C28C0420E7F5999484E5A9228143558C00221CDBE08B8C04826DB4D08019487ADA51D64A86B9F7D81EC08A9020BD6450EB5463FD10F7C678418D002DBD56CA37069558C5848D08D5E5E792E618D70385239E3E08D85B305B3E11FDFF9C3135831029F3A4017C2581A47970756A2DA5825A7DA81A63A3C1475F642137AA0037250758D01668061AD012A2C1F444BE2084506D04F44F10F24430D57817FED693858C9889182E03C6807715769276090D8388076E61E1A54585FCC8868692025271B8A3A5C6882434E713F3D1D78192B2AF0013A8B720454D881EC4C5271E686D878FB1B8D940C78BFE04AA052925E2C0CB86D08F07F771A57BFFF0A8B5610F6C2010F444A51A1'
	$sHexData &= 'F403FEC2C1E80AC229C4076EC5C03DC0A0525FC7010AC8B01DA1393D03250AE3DDA83FC1EA0483E201E65F891121FCAF6106EEFA38407711742683E80237B75621C4E8A21C1232EB0D7B7412123E7D3626746A74348901CD7ADEF057F9155FB8E0870C34308B511C4283E003FF99D50A143524D7563C03754DF0E4216836107B338B411BB868023A61774C345A5E20159258B4E7B8071B9C9907B895EBDC8E0FB6B654EBC32F94C802EBCED1EA24740516EBAFF17F608C837934078D71208BCEE6506AF189AA8D03C818FE920F644E75923AD210FADE12410FE925D6424CEAD6AC90387AD885EC8C8008F8AEFE8109E0434756B91C4048413CC820874A1F3D5544C1915832F0C542002D5ACBD682C6E2E13F304010168D4A02668B02F4662729EEFEAAF52BD1D1FACB45873BA0A0DA74FA681056656A64D0001D2683CC5456CB4DB0669CEDA012C8185820CC015124129CDAE6B0DF7C7E00A70E6F1CE2328B45B0C9230A4C0F45808041776B195782457861BCCE0EFE9A7538448B18FCDF531F66837F2C2E75240FB7472E7ED7F56BD5AEF81A1628300075745363FFFF1646BDFFD7EB3F33F6EB3B83FE05750ED1EBF6C327A3C50D9E2FBE126528FA9E22058323ACA6C824CC4D01B1805E4E49B9E68408774D0F45F13817D8C003394810EB0CA1E061A423D70DD7802417F03F3D016498848D9F'
	$sHexData &= '280675C880D13EFE421A47DD05B0240ADA421C062D06770D76D305A08B4E1C2C0346203486D18642AB4D284A0C1470B73BE32DF04A04562EBD2C8D7A404DE48D0DFDE48F51020183C1CA8845EC3C80525880D1F96861CC425CA450AD29DF48C18F8013F2331F8BE2F8FF03F8CB8D1473D1C3D1F83BF07272837DECA45EC0133C310FDC715B70BF76CC50248855FC27E0814C24379DCC20D8900A88CFBD461AE90B813F34208C5C264230342087EB7E81F881EC5CAEA584245D8098B8581A862833B78F5B4173FF7638F5CA12744AD1451F037148152A7C243C565769D44C17443E59D4082DA7A1DC228E74C77106042CBDAD80AC0F8B8CAF121115572573040313CD44B1E00DF11652B72AE0BC048C64B5976B1540DFD0D23FFCBF2B841CF29FBD14AFEC9DB19A7A100749CF33FF165312C217BA657640689E52FC3BF0548F11EE8BCB85DB75755168261AEA448AD351D3013AE5BC46E82B5487E21BA316F6572AFE534A61BE720F3546606E5509ADC61188751F281BFBDB1799BC6EFFEBCA5153893B5DE4722D2A593810E0EBB7AAA85028B50E97386F8B5D8EE3A0E14C969F0DB20AE95CD16D61DB4E6F2BAA4349C0FCF11DC86BFD90FF3736002B7C03A206E0A22FCEF8AF61DF093BD80F42C3BB2B12468D045D02F857018791D3891FEB5B6A086FDE048F66A4436C804983FB02727E48D3FD'
	$sHexData &= '608D43FEB90691C1C8D98D560497B4C1D3964AFE2C96520F5D82DFCA6A0EB85199716CA08928069E020D7F455839E44C488BF2EE0816D29908FF0CCA73482328F62D3028D0D9964A101024088B9A46C4C5FB8C0ADE019459782407DA28BD240F8E481C4CAF6260CD2A239F0C56E0B1A051E40825B46CA750C02909422C75057942672CE94F0C00D8F59ADAD1799F510CB0BDD1285909122185D2832B262D9D7DE0DF800172C83DFD38E361D18992D086C742150F81D5278AC70243E0CE63E031C64DD17A3838430D9541534843003B08674B9E7C7C009854F4901618C48C9B1E6D7707265F25CE30C876B5433F6B0C0ED6108D3C46BE7DBC0BA8048EFF88CC208CC29786A8061B466251472FF7B68975A40130E9E8C83BF178C708AC814AF8E6067C23CF73354DB7C320F283FA73FA0604540817C26ADCC6D0F88291502446292C1619ED7701488D7DD4560F470851B669A730ADB5043E56BC8B66E09F02C7EB0E6685A12C33D2C645D3E12FA0A1EEC944B83F4D8C3A9820585E4FC8841C2E79BED47F3F0C748B18EC523546CEA82BCADE6209D809BE0AE25AF0C042F2773D822A03C1070C064D8E075245A97217F0AD470AC003C0505293CC388320B8C033F6682EA638D459206E6851527FA7CC2937688502840AC31C7D01805F8BCABC2A55CDED1F0A2383EA02741E08017419322C8BA36E6E'
	$sHexData &= 'B7D10C7B740D082674086A740378825F020778954C8C88FCA2135810E1FE13D20885C0750CCA10A089019CFAB88AE8C74063BB824977C92D8A88690F90C434BF4B1CA0951C20E8650E39D01F3E73D0DA00A78845BD50CA0A5CFB293975ACA28554166C3032A7392B97E5A7313D4979714ED7997E0A0B62914858610E414DACB1071210EB76E00F1088D298E295F247E41854878B533C833CDF964E493F3A8D1D24F8358B112208FB1702B3DE948B93477C8582EE07B34DB88873303C1518C30DCEF33E88B8274FF31E2EACA0022F3CB277BFD0002AB7C685670A593A85EE59A0B6230C0D38AC6B60C62D18B4073268B74747895567E440302E41336FF9669F9E854427340C6689E8899E66F0BAB547C73963124807FB489C76444C7439B5E5C16AEAD53652F53F30BB250E4E6C22B84B80BD6781D38A4D9F47D83B19497CAE9782BA0839C63FCCC2F3C2D3CCB8033AB85F70BFF00BF2D0898D60A91750609C89F83E2E115E2BB861059603E97E304EFB859B54E481FA38721DD5040152D1D349880848EBC87D2697083882906A1C466A13285E0474B25525B1D2B07B0E583C50E2ED35FB864CD1E9F3AB1380088B2F975D1E0538115168B5E105276A3BEDC4CC1C60FF08841B0C719A1E8875ABC023332BBA73FADE21F466AEA15503DF529A857117C7219EB55BC0C2AF78B54E1D087E33F93979'
	$sHexData &= '7397FD83BD10CBDBD75F0079BCF7FF7565103EA409CB66E61A8843B85A1946C77758BFE55D702BF8228D047D176D5D70EB815CE939B03058157905804D5728420B288FFF5013954A27C37426BC551F1A0F06A7F27420580E095B12A220D154A787CE72E323E1100BD9E4531E10BF060C0AAE554F8D473FA460B0EF7A6CC8F983C20481118638EF0473C45B8807EB10F7ADA39E6820F8843901529C634CD745DF14496C95851E50AB032ABEBB7480B8A0C6A10E9D6CED5548552760E0CD5580DF82F9E534B971B0B8C764A443B89D0CB8C86AF64F63C8CFCC3BD1EADE551BC15416DB10E5705AE56A00CFB4AF80288299A4EC67B879694A4B704D9C3F9ACC92C81EB0083988F0989B08175C9C2F88769B54008B90D4D9F1220C58F57489388B0014488B07CF30D08449318F3E70AACCC44DB4507C05DF8B4CC5808FA40F868053681D7AB7A5A1E9167AD1011E79315F3EFE81ECD40CBB23C1A0655C118D38BCCE46F384043468406A5B60A399B96A003230E934297015323C1F20351500EFC741A85800A86F0FE3A5185099B4D1146A7E58CC2DD53A9C0159898ECC45C6021D202D7676B5F09398C78405A79B3827A4E6BEEA4816419889840D6F8658C888B49BB9F8DE58AD809BA1A4905F940DD810B08276D8AC5050F503D3546AC8F0A20C1E1A9520026DC0845022EA0962D4A16C689F70F803'
	$sHexData &= 'C630D1221085A9FA7D6821CE3C04247F6C57B8407C69251C60BCFAE6106CBCE4129DC9EE888C1D50AE8527579404F8B1B0757042E04E019EF8107215C14FDD80658D105104246C16A8ECD05CE94D2401CF20FE5007E3BA61AFB951BA345E5CC27709FC663BC273347E3D3A43A320C63E59019EFF5D69BF461275E28D347340063B80BC350074313BF3A7F0FEFE742883EE02EBE53A90668B169F9ACA7B24B50EACA10E44052726FE83E166EB042BF3D1FE462B58426E21551279041A1C65C65714E276EC3332FE5D4241FE2E75158D48FFAF3307141D3507AB48978D48B2692E514A8C153D9E547C9C67ECAC60617092A43A74F5112439A9E1DD5D6CA28D7168DD443098D76737CDB4E44EB0CED442443294C34EB85024E567BD8BCE5EFF251FA5F5182DA2443540C9D36C3D0C246A48F6108A9B817F140FACD61EC1EA1EC8CA0FA428A037FDFFC1E00203F013D185D27707720A83EE7205BE646935611E6032A04FC6A9486F15033B09A0941A19FE7246BDC272793EFC890BC6B677BAAA0B6A416E33FFBE976BC664C4A78140B42B60223D2ADC683CAA741B3C47D1C6FADC8BB5FD7CDF32C05F88688BC321B0011FAF19E386D37F8079095E91611EB4019207D5800C15D45CC3FF1819DD01725C4047E83652D5DD3759ED50516446ECF00D06665E75F40031638099B0A85D008C3B0190ABB12A'
	$sHexData &= 'A9DFF01D3F3815283CC08D7D48C96E551F48F60FA28B7629DA5B89075F41BC899C464F0889570EF0017D10C6A1B27E5092600045DEDD59014754707A71C3BB89C06968C1E81A24800037C25CA26D69FF408ED17A3F557F68360157B44C0A0A947746DD57B734535D30BE5C14EB3D0D50B9F2E7CAEB3690EB2F8E10EB2850EB212B7FAE5C88EB1AC8EB138F08EB0C38EBFFFA1A950564BB078CEB8F9051C2102B818BAF671C2A1936E750FB5B95D8052F3D70101B8440109147F893B8E4C740B140E30D23BA0899845CA27905C073129836C0D01C02E7FABAA5B8458025036A23E0C2488A9AD07710A309875D2329B71772E1957D929BA0582A4721E8B007106984B5E96CAF43A110A70B5CDD8202387992D488CFC4690D90D073E7798B7066B45B5A0653619EE7799E4C453E373029EC0C2D7A221DBF7E1AB616C0096A1C7E13D7B7F0EF320FAF510C2B01C1F8B003105021290D663F60118CC051FCEC4518749D856E041C08E806104E432CD446D0344D33240A1008140C6D38029B1C4610F3166E164EA318000901F8D660A4717BD156FDFD5008816C33F639351898763866D403AC9000BFD1948FBDFB97909003CE4823C8138CFF34886DCC8970D0F4EF30463B6F72D35EC3DF904956698585E040025FB038A1E40DC8B5D4FDFFFFB7C7D4690ACC0A117C2EB5D0FD0378FF1A655FCD8D85D8'
	$sHexData &= '16507BDC1FC0332F68483D0216A5601BA82004DC8D8D3B4A3EA1FF520851DC11802BCAD1F9FB5B98D5023784A6AE8D3409BF0E22D659F76C0F47CD6CB3565067FBE0F6C11FEB18F1FF3151B96369DD9E840BC6853F00F2B4E154B9EFFF59BA39AD6E6CB85343D9223F274DFC102427C66DC08DC82425B4FDA527665C5CDF85C4FD012F855B618FB98F6AC7061367680D8BCE2DFB00327113E48B5DECB81055D2C373600DC38BEBBB75808ED6B2B76A0E06F5A1D1EA9194C56F2CEB9A8BA80B28FF3FB001D1D889E41D1F800075A004004FC90D78CF4602BEBD8C1AA889559C754A95100C0A1DE492625E67C38AAC0FFE6CFFC2A847A126802945AC030F84FBE595390BE5DD9157AC84C745B04E4C4CF8081D6A1B597D3E0962CA6E84B4CC22B9EB0CB34D98B784A432116073DF80944310E86498843F2654A5822530E060CCB65660380E9676547D147DB00D56AC34A2EB0FAD5620E0A24E068BFCFE74295BF80BC7BFB7AD632CC2E935B82683C702D0386412A0C0DF9EFE75E088F9481FF811D9A8CCC740D8CC914E289CDC36EB76C1D2D0F058A89C20B41A379D494FC8B43022B6C28D7D9A006883B43210CAA00AFA3D6A1F0C100C226690FF4DA846BF96C28C0B2CB7003739AC44D1CB62AA45A8A195C601703B3AFC68AC57281500431F40A11072906FF400049374730E048A4510D4EAA067'
	$sHexData &= '1FA25D0C238885087F0C58A8291409804C5DBBB1E8140610018DE87BD2423D108D787720FD2D335443283D8D1041A60A8DBD3D1052B9649010100C28BB9E9A4C28F4835BB1B7B66413E355C086D40768D128A54E20E2F6C02F940E740A724483AA74748BD63D157C48EDCEBAF89810BE6C2C1A5B4C37BCEEFDB4B6A086D872E8D88BC877B1642EA8747E9443D86B6FC9A165CBD3CB6CEF7DB053B0155C31CB28FFB54301B95917335843FFA1CDFB31E545E06A01993A74A2DF4514D9100C1AC430F99F33E862C830F5A54DE420B9103FC02A050C2AB411F2AA0A58B71F4110061F421E4001C18985186421EC78D40A14231C678085907D9E8B7A102DB0DB2B805FB2BEC59D1C8103720FB60C0292CE694D8D02DAF193AC0116EF31843A98CE202320619546D234881C30F0A6EDB885FF20FF2321875C2E0120202EB90CC8201C3838C7E49043707489E348B1A9A8DD7F9FC7205D90D019F056FD66C1A472609C82B4FEC6B58F05757BC2F422C6A4DDA1DB85E41A26F8F156E3A5E72BED61B0BA16AAF3A4CC2C05A500A70B6A5E154B66DBE5ACFC1021A9FCF0DC5B7B57A9F7ADF51D00679A3A0661AD00F0E4B8C7361C7A2AF80F712A39036FFCFE6A700942B008814D3FC8E6BD049F97C009E0FE05394172CCDCFE24CDC8CD97C1CC0AF872C804C9E4F4F84CD28C4CE40BC8B4319C4E858BB5DB'
	$sHexData &= '0A49DB09BD7B6A45D4E4C480434CDA50E6C80F2814C11ADB91A04C073D5D4E5732C84FED716A16572020490EA95C1C38402308849188C601DD029EA556361023059C096F0C08A310FC905315BE1B7F41B08D3828731C01C50558C1217C6C73AD0410619AA751D9DFF8386EC1DE47500019CEC02958BAA808747A011EE50F6C1C0B80BD4E4F3A3B7B558BD6EF43BC958CA8650B14CD3BE1CE6E30D8FE19EF6071D709CBBD2F83BDEC89AD86DED7EAE80C0F47271A1447C125DD0D3D4EADE4B94B3A36909F0E0949998477EB1E361A3F163C697EDF05752BD72D5460C10921D1FA3CF0FEA4953807A1EAD010D90B39D05B41913F6E044026A93A03F404F0302F21076EDD8A02490EE932F308000420157273ABC1D8DA185874C7C175D46007CC304AE20CC91C48207CCC24DFC701D29C0C240406BD2003C8302417532A7EE09E24D32C0A6C0C3A02784A624767F8C411A193EBFC118993660CB435897F2A5C19D9810A904C07AB38F0945E80D66C1F091E081E2477A002031C7110D1020B0863B02223B2AF3FF57427FFFD999F861BB1A464FD6419B518DF20745FDC1B520748F252507D5C0564AAD9392F1723D2004323EC01776324866EB4C30FC873143F309C5FDC8D133192CA4EA0501F0418E40541FD4C19D44FC457F9837F2570EE051E80C5DBD85816005EEEBA0C5402152BE801A586C015'
	$sHexData &= 'A1F7BD5F4B77C41D8333C9BAC7DB7418BC23FE77BFB6040E4133C269D0D5CB72EF52832CD564F0C45B881B81C1FBBE6C8B1516B4E01645D052EFF1BF3B1B744C8B4A203BE79873418B027E4EE19894025388038349FAD4EA188D46F8D2649D1840C4BF8911993211486F6CC1D752B7E845C0B9946A5DEC687349D3490C3905D7E0D663C0522FCCC666EB36E0179875C81D1A72D2710A394AEC0EBCA89318C8445E1D2BB98045285917C5C21370F2C39F42E029E4904588897594C4028A0FB490E1093D3416A26275ACC58518FA81ACA2EBF8935CFC23E41C8B7DAC8BF0EA0FCE2415DB358BC6C25256F69C202DF658F97396FEE4BBF8E3C40572738DEF56A6002E0F12B9D226F519E4F207174233C169C8D651D65BB7083EC497636FF5DB4AC923D122CE89850643044C8818DBBEF01C01F428881902EBB2B43B768B7C2213EC7311AE9F7593B08ED31CF18B349056459E46FC19AD70799846D86CDF2312141DF1DA03E2FBD151A345DDFB11FFEDB1D222139810B0479CA5CE49917A5D2E7DD78260042CC568652EE14718014F88EB0B3790A6B7093200B5042A1A3462C404AD5F408E90388D5F708200CEBAACF7370852791EFE837E463ED1853CEA50164139A03B84043E04758E8B7A06A0575C0812A5BDCF4AE3E283428AB04E4810C9C7C207D353116648283188E397D8770C634D4A2319E8'
	$sHexData &= 'B8199009F72417151A07A70F861B9B68B84B4E76F6E96D3F8183D29CB0040580043F8105630D986D511A393802A1DA9840E8FAA6B75A0C516CD3B38D4AFFF2236E68BC7F1F8D42FF89114811EA86BFFE8C4E23C68D3C85D93C0FDE1F6A30EB64BD0333399AED1A6599BA8B2339B48A6A8B02BB80C10C81AA9845BCC0D0280FC758803426C234EDD2C00A10C4AE0F1F74DD411861984128454089B9A1E004B8A31704F664FD0215F043075E5B4EFF03C810498B50795845236925173B36BFF612494BFD56FFF57414F7A6537CF07619D3EB86CDC7051D358D22386D99DF1AE45558FF42E53C728616DC99F96BBAAF5A62CCC810D80E8CB81360804880B781B89180433C04BB126A0643959D374A1AFE3B01DEF20CC8241EF80BA452F94CCF7DC8C468F19503B96C6C79C78B72202E76F87BF5A845A403CE91A04A360CC2094061ECC9E9489C99326E1EDE761B07804071170D707F8F3E85F69B9D033595C74687E0C29CBF8F5CC3A2EC2197ACA4D1CC21C395410699A8A8A01C6A0137270B206A206F2FCB7E3097890C038D5F828B8D0DC6D155D4C184C701A20D2E1D8941D2D94204F0D427939AD0890EF903F16104059850D3A9FFD70C3CEA93ACD7A8C1ACF24CCCC0763BBFA8A413BD020CA096888352A6CCF89A81C7AC72CF341C43E0A0E9F74DD88FEC161C0F8D082C707A0CEB0D4DF181E3'
	$sHexData &= '3868B1EC7536B644B7AF014C348D7C3F1C08FB4B6881809057ADC7965D9F7C6D2C4106E4BA571E1053A2D2551B5FDDBF6C9A4A1E4335F92B7484341DF7B6C8A1D7FC14E099730EA31D573F8C3233872FA85A1D1FD647465B0503C2136834DEF07BAE85DBFC8B03EB02C848045CC2871C4923CE5E486B3C4CAA1E50FB95AB5F01EC83792C41093B4710751AFA29B21CD619B259F494E9D97409CF46EB9D1D2DE432C8C1271918BC44009C569A0987883260F4B65EC848A80E0E57101548869A86270377FA33F27BE0F93BA04F23C78D340E81A972C820A00E750194E7033AD8C7A8D1C0E9C1CC690161467E35786003A945E01EBE20D3B4030A0A18E8EA0932890469766CD03A1A2F22BD7CA1053D2FFA4699A1C6E1AC3864763FA1624C9F762948037FA48F86A71A980C4FE95C1158D0D4027F6477CDEB0AD7A83D8A0BC0C575214739A2D5B4895F75C21D420E245D08F6C31BC0175E1C895DA60590230E9C181D481A110F19488D86EA4850082FFD2500782B1D06E8544C326CE0459C13E57252583232B044DB683907917EB7FADCCB56DFC253AE1BBA7FED398AA6F8E6890CF82740F4C72380C5C543F49F6C0802D0F0D3EC454F04BA0C1106A335A383AAF4E0FEDF68142455FAC1F9023BF173058D3CB2EB300F05D73322EC58D6F076FFB729C40F51AEE4BF0C5AEB132BF1C1E60256AC8257'
	$sHexData &= 'EA6A00575103FE22D88F7B811D2AC21F890B6211FD16BCE37EA32D897B845308EFEB3F29825604D5E2088096021E2781FEEF4D81BD145094CBEB3BD5F0606FE28D043E766AF105AC901943C3C955443A1A08BFAD95B51AA1F9F480BB43D507EE04D38009BED1B08AA293D808B15D1B0AABCF74A61D0A607FE8367EC32B4910AE69C0ABAA00FE85DD5DBC733C3B3518742DEC348B5708C74708F01F480719338903894B7C0CBDEB7D837D140CEB07FE01BC7C430D0A0923DF7C10B7805F807E152F6E7A4488063B92740656807F2CDD022E4C7E6B43E4149342DF448B4238853C000155E8427869E66E3C0A54F4FE55203CE4564059AC9A8D416C208C44A16A0760F6E4722ED40133C967020C528D47870A98A81F8050092549305472B880FFBB1CFF020B2EE48D782C4BCEECCE59A79DBF8D5598C4A38D58135D2602395877C01501D9C7C4D782DF6E00433BD296C0B06616C4585DB5A8331170EA75BCC210898D8F847F6FB94DC08A4814888D6B984DC42C27BD1BE83BC87429C5BAB9A00E3802B6B0843C8AD6709BC3E670888553D2B2DC3000370C034048BD6C4DB4C859D76C988EEE89BE8A51C4188EFF89770C884714C64710C8FE106617BF0DB0E3C5006978BC91F4C72E705C170928C2AE231B412F016F00AE2022370378332871084EC8CF2095FEC98D4F8B4598C3F09908668765CFE9'
	$sHexData &= '52CD0A7A1857CD723682174A305ADF8BF08B6195C2575C2BACC704238091F0B178F188EF1BC090805738AE68206C8E3EF4EB3CF689B54833010FD7B53D0752824F77D6686A02816C53251244B06C8F1E615F1901357E70EBC33D0A70DFDE6AC68BBC93081485842F426438258B40C49287B196DD5E902F17E1393F476C40F68E5E41468EDE6FF6EE1520532615FC7DDE8F15C8B13676F61794E4CF5C6387017BE4A93E160EC62B1B69FFC7003D0A20261A0701C5006A26AB1C901E0A949A0A839B53684738CA88889E614A5B0E8CBB4C6B51313C54CE95913484C8883FDFD458B0B28C65BA83E0AAC30A1E4040DC0FAF40AE4FE190D82D4805235E3C4B047D4C5E4233E04A1E602DABE440E642E714CC200885C2C7AD870C6F29AEF0AEB965422B60A542C1C3869700A2877469F30F70980C783317C017C209E650807DE132ABC4468F86F4E43BE81757656013A7F58CA132F08C32DC48F9845426C28C4F9B15C22B17C1E98995C3A64F3B153D8B17A5D0B9BD23CC42A5B09483A1D47FCC0E289C76D44E3C69C2977411C70AA852089C2930CD37EC04C5121B35B2E32C141618C9D4DE07BF0B0AC9E96D7DD06255FF8DF888CC3FC864048AEB1A761B2BC88D106B84C1348DA337B811B48566AF957C378BB5744FE00136CF954813445154B4064C4CFF8B90AF42FFE668DB0CC603C233D2C1E102'
	$sHexData &= '5FE9B43F778955E44C3484C18BF18E8616EDC1E8181E38FBEEF8E962EA30C003D544EC844266F80393899C9C0F683402CD260D54A06D28A050E651F2C03C0F5BC6B9ED68802E5EC81F1AAEDB3CC670950E59C116209F010FBED9041C248036517A58D95D20B26FB7E8302C7DE86189BD005DB31446331D65396B9790871C75B5180F1B586CD9D90B77B8CB2A3CE1AB8A85ECFCFE4D93E644B6E536141AFCA43D300AFB0F49C83808DF310AEF810B878DA50F13ECC0D94EC16A5E70D99D052C3BAD1995958A3EE846188CDD3B31DE8DC70E70658B858E34B78AD2640C55840C81FA5CA499E14976C0083E1F2A2E1EFB8B23D83413C866B6620C25A2080B1A8415003B2699E8CC549042FD38146F22E73B95198CC71F3F5EB8EE8B4D8B71418B739F6BE058973BF25BCC0119AE4CED21777FFD2DF0470880BD0174598B75D0B8E8282B75CCAC86A0C6C1FED01247C89C0DEC70EC44F7F1C5138E9ABDB01ECA5473DC8982B35128E6F41D3BF401FF5F807C8A03FF4A8C3A5E03CE431BF04D1F72E7C6206A241147C0C1F3B421751AE0C237BF15066D02BC25E9700CC740BF4834012300CABB01115008C7021A2EE19EC49748401ACBA7C029ADA4FA54330C0A032916D5C69FC293296D45D8B565240CE0D02E1B7859418A3288A0B0C4C0E57240428E6E028E106463F861B40D17821C5446608BC478'
	$sHexData &= 'E144382809F1F050160202838E52DA89300450C3915C9C1E3CC3EE0106B0A6C0CC8906E18AB105AF4ED03A5608F3CE036CCD7BF50CE115C018015830C090011FE223C2130F6FF14F7E9900F702E0751DF0A35430C0F2D9602908A8EA264F2AAA0B8CDB1768B8985C442B11480107F2A8079C8A99A2484844830A2CD9409CD04805D24C3A1E54C85505CD640B03A7F2078BCFBDEE818B4202E5AC29888029A798038B42E808038F4AEBC309986886FCDFC2022BAC6C8A088874AB710D0C9C108EECBF0E9EC3DFCAA1ED11807EE27421975AB514153901F29053147F45702002220CF810FA89BD6CC26D43F7FCEE94FE541B53B84B0C16E04C207281FD625C27CEAEEACE7A684B8D70A2DC5C2EEAA2F3DED0154F34A1DE68E0770C2B0F2E44CC6F5700E126853385D243627C73A6EDF6715FED81FA0349FB49DE16FEB81F85EB51EE8C997803E8F7E14DC1EA053BDD0B77A34EC22C857CFE7B778B3F1F83E149EBDFB604B7F620AC80352C611CBE5D93FB44B70133C80D04C922190303B5AFBC1BE683258BD41AD18BC8C1E837F8FDFF02C1E10605B979379E03CA03C131393BB59D729F8B90130B0F5BF9F0FE2ABF099721E558000564E88429085BDDC4C35EBFEB99E4C98FC4C7F0FEF0FE834CF29CECFE0808404A33C92044F8032606079000FFB5D6C62679AEB560F4904D22D74D28C631E4D3'
	$sHexData &= '2603D22C10E8A86880256FB561327074493C1CE425771C61C1A535D455BC146027AFD21DD70A0760FE92AF342395D9B4A4B8068B780C9F9567E5897DB0CCFEB48BFEB88DDB8E1CC94DA4A4B88B708885532B3A24AC8BB29F76941DF01CB8FAA48B5576A8682D9F080F386E68505D892A89E0C07E0C661A129E5F2E97ACA8A4B900C646A4F1002C4764FE1D06B9F29A1164FE79084FD01C4AE622FDBCA2E620C7C885CCBC8D9C8BC354D407C38D463AC645FDC31832C90CB510580F2DF5D30AA2F7D18D291F127DBDC70BFDEA274E3C168F003C50768E50E9191BEB0BF34F1B4EC37CCA98A9E9C70BAD8D89EC414FE266EFC145578CE8219C0A51AB026982C9D71AE0DFE0E616E43B740A3058ECED9128C51E5794608BD10A283E69106D680D088A5515D10D9C824FC162AC33D2D2A41DCCD26E9B5EA3B517303CBB4C0B957839A00C96D2E14082F0427AC1C6F7D029E7AE75E8841D3FB44885DBDA3ACA03C852B08BC19BBE400B8E8BD19CEA10C1EE08AD1FE87CDF8580A77CBF74A39686032D902D0E938FDF584E48C1CCFEBC904017E89DFE0B35C608A46CEE2C9D25402CBDA558B08F6C2073617B0764601E5183B5B504184C1825AE832A47166885981CFE8520F0E5BB2B8D3ED5123EF4FD022D7AA0B9180DB9D007DEEDF8398D8D93C0D9932602500085F68E691D0C748E5B958241807CA0'
	$sHexData &= '816C5CDFF4FD74CBECE500FEFCFD5B741AD140EF08E58DFBDEA2406FA00B5FC6AD5B4BDE8394400B1A74C7813C366C8B9519D31F727C58FD890C8240117B45F1E35337DE8CAAB18BAF8B5977E6F8C742BD3B3791F49A4F995A5F50E5DD78989E2ED7057D1E530F45C878605DB9796541012FC85D3008E3EF4B26AEFB9BC722771089004F97840C6B1BFBBB0FBC51E2509EE49475488BF06D0D74B70C80336A07500C95E0AAF54160F4B5356C4B3025DFF025546455214BCB41C7AEE8508B0FF01698AE02F5651B704965E42399088BBD8047DB001BD8441E838B3533FFD67BD20C9C76D6A06472270802BC51AD4F190F478EC78570E1BB95D27880757B7B98F4C085D087D021203821AFC0D4B1B5D89A53A150A06A957B5CB44A022182E9018B10F0216B51D68A5741D62890BD77EAF10FAF098E78D4F78882442D0A1300666ACFD4C24334E2092E303D95B900B8B5270A46D48B42DAD88B4272E452B9DCC702424244FE7300080EDC2B22CEC6B964A4EB9CD504B484103E639561BFF60F039B64F12CBB3587480C80C8049BCA7DE99BD43B0217D4C0AB00F5A162BBCAD4C06E4BA176913508543E92C6413B352F832D0998FCC2C0924BBC6149DCAF8906C903C853A4FFD45056D4A0AA5429D599532E3D8C7AC19407E087C2A18089F0CCA6FC10FEB2EB1B761C2BF02751D0DCE348703D08C6A3'
	$sHexData &= '95C4FE905EF800CFBC8D7888A223598C3358CF408F4F40401A9043C242AC7C0FC941500CB02F87549AE7F9EF584CBDBCFC0098CCD39F229F22096988853FC2407A61927C98C39864A4215CB14798985CB017B9408B83584A0EC885B5B5341EB20530053CFEECFD3C4F3E4F40FE30FEECFD388B3C909724FE2CFEA50B94178005383F4965A4102A987C341AE2E2E430FE8B53372CE202214628931A4BC3E5153724FE08184CDC1C6280D0561AE2C10BD00CA4BB31CB201B7CCE51757A098D28464589141FE20F01F0413BF7049D508BACE38F1C93B7418BB9E80E3570DD3BA9612F60805704616A2458AD0C1412453F93805F0C3CEB8D770C4612291F007248FE4DBC4B68006A07430001917400CE185B07070C06204941389813B84D5EBFE65652A9FC015C0D8B45E0E4E8A48000A6D4E66C000C06C0C25D003402D357E9320461B780C55DC68930570640647457F7562F338045A094CA7A1E0003525CE08D729901B80963E4E800035CE6E9349B000727B9EF049F07E1E8315D1E8BEFB801EF4B09168D4E082D264F3AB00B140EA1F890A05AE817688B86902A38399901182A05D06DBC1474E020721FDF64ECAD8DEB11AD40180090AA26DCEB040BBB801C28009F45DA405583086C88609DEF79C070FF680B5C85FA0EC86343750888561864A9979FF897CC45AC506A18579C6010A9085C4AA9'
	$sHexData &= 'C8FEF2BD7776926D31458468162C3C0FD2DA02E081FF5D163D156A972B14E36C2DBCA814DA519F0716F68830486528E784A86C0F2CB74675538A27AE6C88150A25E02218038606E73C0B339A4060CE90812C805E93F07DCC50E71C8B3AC2EEF4E6BCD840B805909F78CF7EA9CE244E80668EE832B6A0BD0230CE92AFC7A8E009F742B8D8E1F50516A5CEE0CE0D2F1475CBD901490E72E5D17403A5C1088145C9A5C1CBA9EE402E8484E487E010585581CBA9120442BF8791B23AAC905275B8259C666CAF8545884EC08CC75555887920FFAB88074CC6D4B738E3CF88088272EB387636AA7BE5998ABD0CD4EB1E2BF16422E077B44CB5C3518D340123956AC4F68975967598B655FE7C8079249CFF997845C2234E72B977EC64823368685A676690302457247B64A8CB47C2A34781A85422E101C8A8544D807BC18511F7A8D9041BA387114F2381A8A89350FF06646A2B4580656475E10EC843B54480D9042727B922694C54FF5092CF412A4054FF48FF08412664343C57437C59E165F748038F844FC12C80558068A88D8C1CC8C98544403C38C2A5F4B00F2792433E456F888D85346484B7E4645C648B0439E48875A8D4E58F8457E53BD76CA6AC8B8D7CC7F29D4B7874258983AF53F03D3B4D845B8CDA6C13E9C80CC87747DEF02103C6C216F41BB6880423756FD24EAE4870357B681B531B09'
	$sHexData &= 'CEFC17697272128E4870B061EB7A91050CFAC7F68888458F87789D44983CFE32168863C40A201F8FA7402146231FA334700A8479EB220A46FAEAC49A19193D800726BFC7EB067A2861D124F7465318F69F2C0C66511CE000D580005C0155651D3D454E2CEC18D864DFAA3B5108741516FC421000F624DED3834104141C5A9378EFA504ECBADC312943C806AC478010A810C218F292D410D4AEFB85F17ED03B3ED47405A30D68239FB15602CAA9F0634A624422018969BBEC037B1B7866FC29D2ED3A5B60156ECAF9D417609D06021FE739C3181BB24FE07E3076590D5CB4636A7F2168B6B3015041072BDA81EC0D947B1630142B8AA2C32D41692DF632DB2B9F708938C860F853062F5049A83A1004B085C97E2C8D0056E08D55A48CAABC5C20FF5D0EFB8B7DA48BD7381EAC2BCF8B75B02BF3897DBC303E8CDC74752EEB30BB2BF14B6A62406A4C5EF86A4D3610CF616CF7D86A4E42106A4F181E4DC04B40A7E007E8425E14DA925036E4A827085F12AE06D137CDB5235E0C231E0CF403F99A108AB7495EC003D13BFAB0D8D05D02862268745593BDB5055655E822D016C14A03F328C0F75EA6379E03D3395D9CB454F048C794C0605A509A5275D01C2ABBBC4E48D6F238409FA65995F639DC4CC8088D42A6758AB0011CD8C422C4EF6D74980C1A7D10648B19171D1AF82B923B750C7C417813'
	$sHexData &= '0420C28B382BB0B8FE226E384048FF7E53717F4B1E5CF123077F437514D475B4293C309E4A32C50849AC34AE72D48219578305844CD227DCD456C6B7DF23531B0C56752833C0C3F56D6192F850A50C8D10DDA8D36820E470E4CC1F7744C208FCB89C61C7247E49AF6002D4086BD966F8CAD357AD30B4B1AB7DAD93621C1CE02F14CB6AE3EC751A64203041C82B09AC9F4610A1C411322CC10A229A8D172ED0CBFCB0BB476A55081B4514504EAAAB7C160E9E28760C22095C529D21863170C8D4341C67D8A31C3F14804E21BA8426523A4384506FFD610C0E8903489D894301944B07D74343ADC903B4013615B34369010174C32F8E9184835F2C9A7E053F194410564ADB0F885EB0D3D8068219C8921516099C8604D62A801F3B0BC23BC310122842010491F6DC03C62010B2F8DA4A81E07514F83E08EB457E7F5F6C5CE919FF38BAFF0FB0212B1A200380D655FC9AA0B670D47C0585086662F434FE710046D6879A085AC2708D0CA332B0F4FCA4700028127C1C98D4014169CA26AF09AC7DDA0A1482D003C12AE01A4E83A35581405283FA986F2D156240E11C8ABECA3B0A6C26A2A07C82B404882EFBF69B500E04C32BC29933C2083BC67F5E8B263BD39EFDB7D0C1EA080A5A08334420D9836D0B2638EB10672A8BE662E134FEA0413BCF7C9408004F507C55BA41EC3BAC84AFDE81C28A6E10'
	$sHexData &= 'B001035F5E32C0305401583F382B94F0967C421C7E07D51504061814F57CD76161B29828008A128B722F57DC54084744242C0A0E3B47B29845A4F93B243803450C100F8BA780E67D0AE6112D2F5EC34E0E7F1B134068176FE51C282BD71220170E24A91890380224241DDC79F10624302821088A37E00D090D57ED0F283E9BB00F269650DEBFC7821C2E3BFFC51487F859B2908B4424188A185014583C766E8830C0DE2A7A21C66536341C6D6069EDAC8BEE304481E2CD60579A0381B1EB9BD2DD0F1024F76EEDAE8426C0FE1CCC1072D118CCD8E6EDBDC4CD66D9D7C33D44745DDE10560CBDB6B5F6880F54E64CD40ACE3660D512C10CC5DEFBCCB606F9D01C68E51468CD1CC224E17B36EF6CEED01CC465D7E00E5CC86DCDE7637E55DAB719788A265D18C15403E6C985572CBC28B14DFC8D327994F043883FF9048AF7C0C3B1D8E2DF5E7C6260A3D91F87149270C08B143812183D1C1EA484D2C77F664BEED346A7C34DF0C0023D47B4EF7D28F94313C03DC171FEE1288B82413BCE7C837EF1A03DC7D4FA103B0638492191C487C3FAF0FD00A38BC0B21F21D0CBC08AB6827D1705018454023B060F9C415E17B5525DC33F46A1121632C99000E87501AA9D828BD92C4424D8247B6CD87D28163AA40129B7F18BB903AF6B751B4602DD0DD0112E220CF365D14DA87C04A48B46B6420C109834'
	$sHexData &= '6DB768161010A77045A1E0CFC63706AD5350300EADB06ADA07675D4410C7008C6E90085184B80AB44404234AC466F0DFAFA9B834480CEB123DFF75A86A06D64B3700078D57102015E8841D7A45188D14B8A0220847002EB2A10B2E0B1E64361296DA6CD272252480C1BB0D326A08811857C70285EEDEAF00DF4708EB4737B904EB3424EF4DD26A060CED6A06A290A420241161277524B19E1CADB78B46F801BF36B4817E10F4010DB786FD0FEF8CAB0028BFB3C70F47F83A2B68187CF8411033D24020F7F7EE2A3216B6D2F705B78D83FF01A41026C16C2CA64FEAE982D26515599412EA0D44B033FF4A3511BAA03A84D73E7AACAC356B863FC140EB07C49EAFBEC480E109E7C2A0AA98328CD3344D73D20A901094204684CD56980068B8774A3DA4025AB384A4C34273EE3A9CE3A8253C6F1355681B0AE36CC05B8565B3F443825DA262E1B61104EA80D0C6A0A8FB516F0677B9131E6941E039F48E6CD3700D389EC438865716723836930B2F092B8501EE8E8077588D2B85DE56EB06E96CA8C2D8020E3B0C740E4085FD030C30C6400401831E9A16F32DE000F1B4DCACA02740B7561F74E54D3A443021B0913885F4EDD2748E02518BCAEBA4B144435B5C4A477EB6B88CB054B455D73AB4927DE022DC50AC855312109461B6CF024E609D110EA2A1807EEE00E005740A8079745E1E1B0C4720'
	$sHexData &= 'CFCC931E3C1C0EF026F124426C0BB9F5328A5E1A0D02FE19A5A38B2EBA9073735BADA0046F74B4510AEE84B043BCA7C6AE83EDCC821FDCB6B5C310B88B203578501D88046A66E96AACDD3E2A2E8B5382C28B0BABA200872DA5B24B781CBDE1C4043E6CDCDC604DF335E11A5E1E5136C3FCE40432554A383B164AF81858A9B90C1E10BD5143AB39A80147210E1EB6CB91DBE860058112F1BE02A20199650082E45A936D5450427578CB0ED8FF120C584DF513BCB8F590BF55B88C82DD9B59C0FF300E361FC286244ECE562CEB0570BF257C0CAEAC9984688FAD1FA8DAE0BC034D10B8208B43896A082650646C19251428A04C91A1356A218DB8F7DDBCA1860A41849727C2571EC60B4B043B4B08604F9C1030B99324A87B3574B486CAE7D19DD29F922188F310101A6A0D3D8CE794699AC9ECFC1814FCC40A249CB7806F86C88343042CB97583E31560C28DB9086180A7B4E2B8204D9C5E587200358B45071174040D6874A4001F0E8747BA928E3BF8F7E1F0F01DACBE46C0D23BF023017300EE9DE95D4BB93C2092FE4500289CCC1F46F6129C208603274E86162B5A871B8187B6B96612EC0606934A683502BBD920EAD06C136399083B961CDD14AE60BB32080CD63B6E26048B400C19BAFA070C8F775F4B108A15278A60AAD207EE7108A88BAE8E22A8432E26069EF02B2D9ADA21BADFA21CD6'
	$sHexData &= '209028EE68B186430CDDCC33434CEC381870BEA66A4314612B17D4F05214A4E26987E858CC25ED53D371095A4E2A4E5400542E24BB9284EAC0C42E875C7E19C8D0D4DC24239749E0E4469C7C98E0D0EBEEA612CEE432D17183339E134C0F0F8303EB034C7A47796EF8311D03B92CF042C58CC56DA0EB1E84C67B86B046293FC2422EA03F226E038B520D7F201550A4764022898432C8180833B489846C7B5F853CE4504C141008B4FB603CB5D0FCDF780E0ADC3A27E0878D8D80FC770A045E84D2251E3A4C849C9B304DE9B8FDC047C4E4B8FD27C8FD37C88B02B1D0FDBF0680BB331266675089BDD39B708902B3339F933CF9EBB8FDB8FDB4FDD0FDDAF2E421D0FD08FE120CD22B004CA9462C9EE1D1EDAC95D4DE704E38EA0C0501A820FEBA880F49D03CFC76377999A3F9D4FC330F4EC2E05D73016024EC2FC18B00DEB2970FADA03810BC5DCCA0406DC0ED5FA7523DB6463C0F46C87D2F092E6B471A981EC18B3A14ED3490F63E2FC882B06E436A72EDB40E7C8505E0B52BC1810A9471803F8DE082D6A740A640A9845D43C19DB2C204FD1D0AD723298CDE77DDB151A559719CB457D90BBC052F025DD99D6F2B27804B7BF45C5EC6A7914E44D86B9DD880D10A0037FB4EABB2C81EC8FEC8D15CBE4B47EEF8FC0B9C88D1E572B9908CCCA45F42081FA806CBC2966C59A7A253103998401B40'
	$sHexData &= 'E7B8592A481CD80B7D2EE2B38D442BD0287F1C35667DDE1BF1898D74EB426DA43DB79968C3431E4E12748A7897664FB88B6B4D0075CF47BF2B03709AA47A4CE17CD3D0D095A026CC6C758172BC963B07612EC0B5F8FB900F0F4D5AB19CE0D13B3118619A4E44744BD7E35CFC9D9C9185B35E3180C0443BE320756957AEF45588FCC29A5B3A855C5F88DFEA900E10DF7D3B95837E043BD07E38A519A90C81A244A4866764658D8C8D88C8C8BB364C20405A872BFBFFEFFB0EBFB9297339AD7D573BC17D53FF764F3C0788189E21D547DAE9C050BCAD1C4C01010CB9FDB27C35328DBC8BA9F8840D1452D5FC915077615DDBF2C8D258D81ED1159C1FE0D72BFB8B131B8584B8045C490C40420F8009578D4CD1897B82AFDD2FB7BA1254F87A5ACCE46578D91A1A2EED8AAC8F77AA9C45C9D0376CEF066A4425FE88BAF8405169E27C65FDD4FC5D2EA37BA5683D38331718B6E470197F9CFC322127E4900C8890FC21875CC890E88C19D86172D4C82DBC3964400E2CA03006E4900184346C6AD80D4C4A4D9CB6BCDC583EECFD70555CB41727E91E6959548D5C17D77A23B0B05FFDD05F38BD1C812515D8A6C9B5A1BF085F286B093F4270FF31511EFE5A42D15622A6BFC4C5CA1C2021D52FFE6F8BCA8D41020B23C177571B8D21D1F9744F5247682CB8E6F0539515124251A28F05808C50353355D1'
	$sHexData &= 'D3F16D3B8B461C1DA668D36A3E832509650E07408B3D0153E46C101D3D310EA5E88E22413BB3C81B603A9E35358EAD277DA963EA98D775D76D0AE4008ED7650AC9451C5F5F0B650C4B121F13D75DED99A86B0F61C69993FAE02A900B0ECC7FF155868BBDBE229CB969D6837F24B74DAE6DA444D157A8322800499A2ACCB43CF152B177640299E02F7777458B0F03F24F38CFF0BF2BBA954C0F4F3CC5D1AD8A6B5510B555402ACE705DBBF07111AB8F6A709DFF370579A31C3E8B58203475162D354DD30A0C10100A3CA15B48DAF3C8CD8051EC410FDB197D005D5C0124FCE97AA06D54AC48C785AC0A22574AC5C5DA56C223BC8D1192369C3F4F70D6B6C443360C6A19F8C894762428D8487298920240F60B6D8EE1D684FBA295D0F0A065B4BED05C5A01C8CACA5BEFEE158A8414643B959B7E068B0FB2050FBD395108623BC20F78DD478AC0E08D8C2B959B8BB79A93F07DAC122B5D8ACDAAF0B4C283EF75AFD18885F43144FFEC3CECA7D0523950930148CC3D77D2605CFC857137E2BBBBEE7C856150474C83BD31007E6133FF2FE15D01FF8DFFD1712094420370F0CE7E701A2AEC158D34883A91C431F4BF8B3ACA8D0C888D049551F1EBC6CEF71447643BAD7CB32ADE370EBD2367EB5D1852707B7C28C88FABBE2AC8B427802CB8129200BC8E133CFE67522C0DFC34D081B791248B0633D2'
	$sHexData &= '9205B60302C2FADAE8750312883AE9AD20EDBCB47F6F11B5799533E41783F80223D381E5753768FB8518E1C0A659FEB440ECC9630B8F9417BACA3CD908CF44B00EB1C49030C89481B5C44B22C22D8909AF529C114BE94AF322573CDC767733FF3CD380EC57514BA8566A835DEEA333126350030B7EB5E412617267FC26B6D28E39105DFC9654740F4F120F7841793AEC7142ABF8577C078D7E013BF8729B4F3EC0E01A19C43D36FACC469043A642C4FB1C902B991917CC279153C891C49240A1D8C206BEC63DDC5006C211241DCF601A11232662691BCF4DBA54321C3724C15BF49093B19515784A2F3D06DF8BBD3E388733F455572341414C66904F17589A65948CE69391918C90941D833EF8F1AB8A2E762823CD528D7B8951C029BA63193B3C7C75D3F7611EB4CD3E8BDC3A766E8B1DD3A353B97DB8DF56B31FE7B4433AF7CC20993C0C78AC7B479A8C2C21910C35020CD425E12BD983C6189CAD200BD157BBEB368DBF65CD8D3C985B50423D4C22B1A96105710816451A2407E21DB5074E019EB64B8B46306A6873E02EBDA40DA0E10F87546A849D1410985D8E2B5D5AC824047C5736CF5B691D7C0E0DD05CC32DDDFFF31615800F54C10F2FD0761B579401D9D9BB36C685AB12013577071D017145348E90348CA6F02761F403898D69FBD9A7FC9568375F95AC8B0CCA17771EA8C0A50BC7'
	$sHexData &= '04C2C5051C480F43DA2558F77AFF31FEDB80B9773C1823B058C47C7CFFD030854043E9B0003545A05670186B0B08CD4A600CC00447260366E0A2890A8955A0AE34CDDF3ABB100AA88A46148845AC15AD07DFEF6178940181CF5A088D19FCE0E25393171226807DB015304B7FDDD8F2EAFBE3E9C3BE1F8A8ABA8881631575F0C78BED221A513A3E0ABCEDBC50B7CC83CF2813BC4D1E06F961287CE3F69ED00493EEC8532C095FC00FD8FC280204E90BD9063EE0FB5F616B59FB979F15A80274143AFDC53B48E0921F85241C609BB4C0FD12204BA28E6C403BABFCD2E34F85E1384F512B737446518253030BAA4209460C7EC1FFB5F006A44A62D95C168E9C705D37A17262ECF8939C4529BF296C9C82D5D1B98D95C4D670A43A68711CDC80359F8815561AF46F8BAE875DE1D62B8D582B6089884300C57D097152B49E6D0903CA250F8F21A803C1171C0C695C3F6351C02A98AC522299825C7845AFD16173CFD06D23304675A804A457569858C0E90A5DE92CF81CEE12283C3E3F36BB2C303DB10842857427208F9AE110A701FFBD8B16E7C5B520C724252C65EA98E4D5343528734ABDD83B90B08171D2CFEB5893A76DDB9998D37B3E7017894BB040BDB07297E057C07C8E6F3CEB0B39CD2E025E82D4613B52F04890F774FD87C630987A62A19558C35B5E7CADBCD70417AFF3C5D96233FFB5FB'
	$sHexData &= 'B2CC9DA1D54324322DE293C02B7A3C43BBB9C4FD066AA84A2838D8834E79FA8870FA6A7C0AF727D20310489A7B08958027580C5D936F021BAC8D1C2BBAB22E3B04A95657259FACED9470B006620B78C1C160D046959981042F56958C39FC954F1EF19CCD7DD62AD7F1C7848100ACF3523304D09889F63F96EEBD312C07840A381737A401A701528DC22E390014C03AAAAC0EA90F8220D3C57F79006E004F5150C40F2C0352C4C4871CD800BC0B577E3A63539AD83477F97C0307E0EB59AC4D588719878427820173898B4DFCEB58C17FE5D5D58BC12BC6C03D4A20FA84861F3706564C6B288BE616678723ECFE5E0F2AD9EA19BEF00BF47B2F3BF187D4C9C6C16A9CBEDF1A06DC822FAC37EB771C56E4F2798D0CFCFCF8157237F21B0A144D8E6813B87C233C1610FA3EF685D97BC01CECF530AABC3B1785CA55382F8CF4A7F1130575D42C113458F020FB8B77A2BA70B6523CF8D559321B3F48384706E9084C2E12DEB90BCF2F1131B640FB0043AE48F7741E23C7201225E712BC91B0D9325B330D38453CE00807070B8A74951F372BF443011F7F37D791517AF8108AE0CFCDC3E0100E1E2EDFD915A7200FFC2D1A9AEC9A1CD22612F0F4C1C55D34E38B7BEE600EE82DBFA0B70C2A45A4517CACFC43744B106155A871593060EF45172CBD1F678F8D376C945D2C3137D7BE15F899640F3BBDAD'
	$sHexData &= '16855CB8B4C04350578B6C8EEDC1378DA20832807D84007955372EC05B856125E7946205064162E5852904D8AD8CFED79102581A301895D80AB008978CB8EDF1BBEC782F338B612690FE23BE4D0B192EF954F78F838FEC7C2BC7297503F47474A8517AB58F7CBCD17E82D5DDEB16554D870E888903776C6EC98F5F993C958D0C8541F9F22537CD56C1F28D46F2515606194E4C34D7383C5C161D4C75DA20FF2B5790CB854440B311060B5ED900EB6D6E022FD9B092B22B86E45472E0FCFC2D24E4DD048E88FC1FF4FC4DB4F7DF9F3845B07426463510AD94D77729368E3B459C74099F37D412C6739C628B25C470448BF78685740C2BC68E41F08BD9BE80E0C2CE6CCE2CE96CD9A1A2821FDC63E5EBCDA70C34021934232D2C1493B14A932B5B9C1B2FC1F9AB69C1CD8B4E068EF5A430EDDBA7EDD0687E0BD6C11DD72E640AE1DC9AC108083524CF4170FD18202258E3134D3627CC52992A8408BD3708CA25830C0808049D1C6492202058FD5C4A17D6B713E83833C010910C5CD0D21037F9FE5AFFC30F6BF72C8B0431078B4431049D19B81067DB7834EC8A6D09EA2A0899BAD1F801496457DD360C1D53FF89ADDAB78A20139BC83341334196FB2CB63BD10F30C80C8B4F473BFE6B032954B7DAD25990C3902EBD6EDD92781D2A479FE4750750A06039780A0508AE184800DD2A08470ACEAD77'
	$sHexData &= '7426698E26F038705C0651A44299E421A7846728FD70FD2B7831D1E1EB8780ED90497A089FA8AC3EF0CBE598983A394644F8F90A8740BB1A561871078CDB43991FB18B95A4B93B2F6E11708554A5A878F843F7C5E5D8CCFC65D08CF190F84D462E97DB3D35C890A83E19B954DE6B04FD88C03FC1C87739BC9C35D8400DE2D33872A4418AE260D1543E5CF8D41484C9BCE8107E0098BCC4B80F44C25C0896BA6DE5C9D82E5F4E55A808F94239B4D9C3C49943F4F94D7C73646533C1B4F14438F900FD864B333B7C3545509827DBC8779998A0DB68466B98C4FF91AC478D75EA8B4DB82B4DB404E823B42E69C91266907CF3207F1BEE02F7E1C883C10E8BCA85C908504AADEA50A9BDB034CF23FC488072924B1BB9C7B849349858C44B2760CC21B04A8BB0FEDB3F23C07F277C04C07321F7D983D000F7D80274B479B4EA6C6C70B91020B0705898207283AD8AEB0D3317C0EC5B0F521C206A0668D8512505AE25BCC04B85DB7C4E0561AFE0FCF4FC949CE481E0FCF0FC06B6D9BBB9388D95E0B74CCAB0F9ADF59DD62B36389374B94D1B502D6930C8F001299577B64E8DBD687C490B81D7C44721062BC1CA9B497CD3A447837E14C02F120F2DCAC6898D2788C657765AF04CE2EAE0EA48CC57482964B62FC5FFA2E01C9A63A9EB125233891CC057FF8BCD4AC39535F0D905AB55101608DB599687'
	$sHexData &= 'CD0D702B044641E822E161ADCBD21C7CA960A5D24FE01EC0795828504AFB12741015500B3C996C70429ACBE68FE4B0FB51F83925976652C8B8934C2BC966A4C853F0F8B583ED70BF54516722FB73816C46DA30E4E05518FAED4A2ECD5620AC107020DA8434235730586498695EF24DE33892F4386926A41959485A5048D346727C70E050EEB225CD5B604E095C23CDD4AE98FB71DC685DCDC8813478945E80CD487325D8805F90493372209060984833D25CC09861A892F6E91C8C0B6232C084347CDD6DA031632CD847C14BC36490A428FF498F03692E9B9828FF65F03281B491BC667010107A1CC83667E208FB684C5C5E2D8CD171E0FE695E9A03D900F86AB0C8FCAC183940B06B1018F84CCFC880D06CFE310969660780E36D30B892CA8E6E6638FC548C33216DF6D8B36F7048704B9D924B00DC9024D5EB409AA7A2716098331A8870A59AF8D7D00E480E60F8708A1441DE8C1DDF70F8E374F86682E4B460C100FC6F21438DCA5CF810A514172417B8AB83486E490EB9344C585CE682649248386D32C9219744484C38D95C904C306C94FE24931C52404430680E68A30AC16B2CDE76413E8390FEE0C1AB9A0B920B83186AC4C924875C282C184B734132B069142699E490C0C4B00072692E4868F410AF24931C1400F6EA6C0E68C1671CFBBCB92D71E7C1ABC108FB59D94C906624FF18FB'
	$sHexData &= '797BDB05E0C1ABC110FF04AE992065048A2F70DB627BC1AB83F0FAB9341324643C0072C8490E04F0FA284C65734163ECFA387973494E3CABC1D8FAE4A5992062D4E8FAC9ED6D17E0C1ABC1C0432ECD0561BCD020996492D4C0A839A4B2B960ACFBB8BCC50B4926A8C2C9486533405FA4FAA872814B72ACAB8390904B73415E94A0482699E4A490801C72692E5D8C909405C92493807892432ECD5C7C888CB920996478685B4C72C8A574787C6834172493605A6492490EB970746097E6826450595C604C32C921645048E4D25C90584C58924926395C4838875C9A0B5744484C4132C9243830E4904B73563440442E482699302055931C72692C303420CD05C9241854F46492432E282C18B2B92099E05304F9CD2539A4F0F4ABC1C26682E4F0F852DCFBB8208334A3F9F9AB3417245707C8510C92490EB9D8DCC84B334172F8F950C4879CE490080CF8F9B0B93417244FF4C07292490EC4B0E0F9486533414E0CFFF039864B72F4AA45FEE4B299204DDCF9FFE4CD0D39FFABC1C8F9909766824CC4D8F9E4922B5CE107C8F9219767824BFCF9542530C9F9F9A345099C364AC034AFCD05C92DC12049AC32C92195F93034206926484E98F94894B924C770A845ACAB0BB820B9C180472EC921ED83C09094AA2EA4E802B9A6A00AA799468768F9AF417AA13945F971F98264DA90F9C150C92197E6'
	$sHexData &= '444C60645C904C325038432639E4D234484C389A0B924920421CC924875C303420657341320841FCF7934C7248181C08E82B9B0992F740ECF8F8F76F6FBB20E0C1ABC1D8F8BA3413243FD4E89B1B324CD7ECABC15C9A0B92C03EBCD032C92487D4C0A8904B73413DA4B8482699E4BCA8901C72692E3C8CA0A405C924939078439A03C43B84BA882EE964928C7832D011680668B53A84A964D9305DB0A992750416A390704C6C2034179CC0BD678D9502E1ED00090F470FC5341817C22FD1075C2F61158D71052A7F4072FE238B8D7CFEFE64DA9041FEC168294C5A80AC780B431AF07DB2FE596A1E16DDBC178C8833AFBCF45B8B55903C8B45942A7018011A1BD57369DE4B9EB4CA60B88C1700964BBCB43969A4606FD640C1E203D54C5271C011033ADCC814465C90023C2F07A94B72DC9F38B3B351696F74314C7216611AED9CC2B2FB3B85501B0C8D874CD29B8D6F3FAC8392A984402BD8A52B637220FECCBB53514309635BDC904CC821FD130361951CDCB21A7A554355760C900169AB4F2C10E886006AE8144314864206421516220FE03473922050F76680C1B924182F10004E63857FC159400BE4048F5FB689B8674C494314584C50A12E19415E0D68518FDF4C6EB42690415CF10E6C41ED788B7D897DA40AD624A0990112A879A86D0D14E3273BACED2B894604516708400C076A2F00'
	$sHexData &= 'C55906CEF7CB48DCB8D904754CD95DAC435600164810520DD8B507284CB8142EA67EF60D7101010785C00F8B4E083B51F41D5AF7DA16B5105052242E2D119A914B6B2375AA4F9684545310B214D0E6B637A0CB91145015A8A055023445D44CDA703F212385272E2F80D1ADCC47D007BC5616160254B5DEC46A13B846182AEE122EB6E6B48A10EEFF771CD414D63496B2341C89D71A080E51FDD081C6BFEA0C525651FF3085C7E648BCB0D85008DF709F651889A8E47144C00F1C11FD4D01DC1882439F625573483848D5D7740368978326FC03E91811D6F2445CDDDED44D8E8EC51997F017A4D541119FA91C511300F00C10D334ED591489180A101C14ADC1344D2018243AC19E202F0587A1005FC2803D429B8D784E1A501336903A0C4B052094B4C5252AD061E844C92F488E0EFF904E52853C6A13DF20041711B678817FA642D4011511241D418DBA381AFB87F086F17726555432943AE4FB53D0BE1C58F201152F91C28B1614313AD2BCE8078C8E8839A8E159181B409C34A31F334590072C303CC949F39C940248383BA3F1C2980E1D5634B9C9A51D60BB28783039B934CFA005902CA43C2797E606A818A8079AE7E4D2C014AC08D84BF39C5C10B0096045ACA4930CB43A2A8893F6292808B8360B1E41417C4F101F40FE3D166E20010D76A85121899594253A8784373539880029435E40'
	$sHexData &= 'A429F578FF0957016154EF93C2979273D24B6B20FDD4483B85B775B8DA580E2B7C09C5A8BCC7987320ED66642F0F6438B4690EC848701090FC70690EA4191150B088469A031912E88813469A036968ACA0149A91E640D4A015809091E640A8B8169066A439BCB817980E64A439A4D018A00EA41969D019B0A09A031969E81A84E89B0369461BC89C00FF6EE6402E1C6C2F1D8C3407D264E098181E3423CD8154181FF88950DE8194582033D2DC36ADCB3830211969066410FE904822D28CB4032864482328818C3407BC6024522DDCCC242F25D4FCBE85F11ACAF61C98FC28E421BBD204FDF0FC45E2C4C032C6E6860C72580004A3B9627915F260FF3CFEE0B644286BB9A34EAE425E7328FE5C38E0860C810F3CA39C0C905C7348245832C924875C48101C72324044202400C9249310304C72C8C90C40443093AB9093F8FD2C879CE490080CF8FD1890910324F4FD285A2599E42C18B0434EA622CF14F092492E70DD73E000E4909301DC10140748269900C8FCFE2EC92123D8DCA3F22AE415A1E8FEC4FDE0921C32F8FCA39C0C905C73B0E4C032C92487C4B0D01C723240ACE0E400C92493D0984C72C8C9CCA8AC9827032493B894C84C32C921CCB880879C0C90B490944032C92480A0931C72327CB0B4A00009C92468490E39199C787C64806492688864492639E4989C8890930192508460'
	$sHexData &= '482699E464507092434E064C80841920996470386C92490E39484C38E4648064583468904926396C582039A4C1AB4D8C6D303485BCC1259D2778FF1BDF885C544B486E709B36AD971B4021244E061C50219498E4905440FD8B6490CB55FC181C5321873C08FDFCEC17649091FCFCFCA327032437B9D8D4E84C32C921ECD8C0879C0C90BCD0D44032C924C0A8931C7232A4B8BCA8C900C924908C934C72C8A0A49078212703241488904C32C98C78005491B10C5C01212703B4B94458904C32C95C483024879C0C2C4044394032C93018FCFB93490E19282C1826134B9931DAC78289C60A7F26A924B07E285620E8740061F0700E027F3E8D59028BF21D8BD17DD5540372E0C3CBD1F91DC05840924DE075DB7E37B06A33DB4433837DF4078D500F4706846536F1B704585F1443C6DA407837C8CEF7B7125843683BD872CD2D4D0650A3DE05C8751950BAA0C4D8D9A6913584C216EB5139194A0ED007DC4E727928057515EC744AC925B3D4ED49F825FC242B0F0975239404750EC7060887B6E904ED10EB3F650675E5322197201820401E6D5C5A0C1A0204D5263AE600B9E0371D44876342F826205BA41DABD1E51F9476EB03628BDAAB500B801D985DF0E055FC3FC5540A11BC1533F68512F86BEBDB7E3FE060588D43FF04CC847EB8623020459F049E929B80D3B5BF99F66A0AFFD721F83B75'
	$sHexData &= 'F47D046A322A4623151ABA12F07CD31F040EE9247F6C685411688F5A8D0E2C64F226411B08D2145058B10917A11C14BB0FCA6CB66224341AC74483076F22F0542428C18BCB55E035B40C3E893D5640D706DAC1485AC64013E3B72AE0E00118CF2ED7B8F00A8B7C2474AD36031C69C7176AA54CDF9593A70E544099F7F9C0500170ADF29E80C82858690E186DC0DE663B3C4840FE89448D5E50B0773F00F44720C01057FF744A1F5C61A0412C0301C4D3807C8B08351A141719B80CE9F016D06E0E1A658B8638C7BEE97E10CE3BC7A47657C90F4FF804E146048E26C26B84EEB61228C8CE28EF5B82D9C8DEC23A303BA5A781031C5CE6C07A09FC6F0E2EC87706F20F51C0EB050E45F229F8F699B390302FC8760D5615AAE0DDC3EB4C40B9DF03C08BF178FF45E0DF07EA4CC1B4143BF97C2F6476F1EE76CA2E0304501D3410BC6756CE859814943DD404AEA0F77ED14A2E2CA2E02C7E926E38BBD7E23E30A29ACC60A983E17930683CBC34A75CC399FE2CBF5C2B4E6E27CF6870780F486063504E4E647CCD803CD8C77552E4D3BEFC5CC187CBFA75168D4706F7558211FDB00F8CE69CEB31398619AFCCBE2C59304F182BFE8D57E049E863A0FE8BD782F745D341421714611AFED18FEE3E5A8B6C386E34807940923B2E3FB2D8133948C6A376EC581C938DA3C81D3B3E8FFD8FFBC1C3BB070EFF'
	$sHexData &= 'E607F930C020B6F174B001F926010F94BF81EC047B2D7C1ABAFCDE8D990828E09DC03AB42C8DD2383B332A048AEC6F0996C31CFF3729DB84FE146068E00818C647041BC4071CC6BEFCF2227081D31EA709B8555B0B2C90431E55DF4E9038A02A0C8C5C7B0D5E52557E3002B5C02C020FCE0D140A3718C12938F8AF1DFBAB6AE8B720DF2107E857611C3B2D05B33556C5AABC3D96AD240B8875C8F726ED66B081CC1C28DCC264E6024A21E08EA449732B1CE4F980DC8D04D5BD9721E31D44338B73850CF74CE3388DB7E28CB73B41563D8DBDC10F470DF49A4785E5C0F71955204187219A2B5149F4B7F748DF31DE08663B0E751E66D0B7F694661848F43B4E02750FFD1494DB3AC42875DE332875817A646CC8913738BCCE0C7DEB1833C0ACD986C3398D2D0C40339869FCEE8BBDCD8B77043B7708A448BD083C432AF745E2C0C84913550B6812F189461CD820780DC233304B5E563AB05E888199B2ECD3CCF77D2CB94831D3400EBA081F0866BBC152BF1C0A0C3B304E6602F05AC8414ECDBF4ED0340805EFF147FA8BD985DB4523028BF3EE02A1E50200FE0683C6C684B2B7E4359305B8F3E29428536036573A0C107C44B18DEBA8033930E8701D42251F9EAAD853396C5A1AFF8495D437B76DAAEC64F63B253C591D1886CE8020F00F97B6EBF0A41EB27FBD4372D31EB1746A33F0F056350B'
	$sHexData &= '55EB2557E9A6542C8B442C88B6DC5C477F4EF4011B205692194B4880BDF73CB0EF0D21C6E90C0453CC9B34D3969CC089205B1D180B0AA820F8C481044A8E82407FAA0B080C2BE03887E414AC78571579211F510499B61D18031464A1607F3987902DAD4413ECF7692980D5192D12E8542B91364BF76FECDF55AA2969D7B0899DC8D5EAEBCB3538EC0D392F86706B67012298C2D12BC68B41B6D7F7CEC1F8055E1E300B7C9C4788EEE16CB765C14433DB756A57F49720FFD3C1E205EAE0413867386D0432F4C0A722F65C423E0982C4F7210F58C919B522F0310760386427CCC122E49201B934D4D0D0299860E5D47568A943A51F8D8006A0868216CC82E19184335A4251167860D48679C2CC6805043216090F1F50E1E07B20130C6527EB20C102FC976131E78B3C32EB9FB52D2080F033C00F1BBA9A405F539C7D5B5AD81E168E78A0B10B43D297F78453B0290E3F394D1427504AD80F4D06BE42818969FC100F49F0B86427F0286986B1988B2410AA60ECAF5D10BAFBC29510540EE40232C05440A002460BAA0704BAB6364A20090D008694D406CB4D1C76E00202DE4DF30B2836F068E601525CFE2261C542744E48D3EB0AC5095431F02870AA7497680C775716CC20232E210812C7031CDC30586B9F1C07473487180C08EA18A916E054AB2523588D505FEA0903439E0EF523356447683934'
	$sHexData &= 'B81CBDCD10040CCA45C78410DC92C3F5FA1E4C2725231D0E41661F181450271022203DB00FA2148B55047C03D682FFFB0E943B207C640A1C7F5E3BD67C5A3BC6E6E7A4E1207F54DC201C1C33FF931B230DD4F8F80C1C133A54FC94C30FB6F36A51128B343C7C441C191185BB707849181C0073B92B9AE41A063F686407A4355ADF0F8CB741C00686E026A80A4470C18C309A9A89338ED2F7239A2A36283D94B1A2F4267C64DC18E8350B2C240840BCF3344FF30C2C14301834B137064F1C38D6354524BCCE0C03C80671302250354FE5D2285C3454389902C20AB1605FC21F2D83605DBABC8DB6982F38FCB45AD3529C27A0896E128F2F9609A0DFD192340E2085023C901C6F8175946C38C42E0C446334A323DFAA10DF0D1418232323231C202428239797232C0A3001343818690700120C3C40A44A5B12226044C31C058AD55F7FD709A9B0587E8D86E48765429A80E428E766788B0EAED1702410F914A5792A2F4160205818F2699ECA64245C282801E479713444C04849686C924126E4C82410A6E4422635287444C44174BD24E3908918025FE81CADA499F0E8E6143043449310CB3C9291496E97243028CF4865643438024C504C417224B0341D0C1C867E77FF42F326D475185E0A148B4D08102233C11A1BB854D4C8014E18BF4F0DA3A11432582046283F627DE8872C0481E22F84A532'
	$sHexData &= '3F55E88845E45802096A049626A671A72A721A5F4A78DA681B2A55B03EE39C1595F126E82C8D1455022B10D00DF2012C248A6C220A079725DCE3138384ADC0EB2C1486009EA5012A8262862612450457EFE85BE2A0D00A04C970368947118933760F7D452466612DADD903EFEC516E99AF7E8E0F69DD260A67173D15930983096232C800A1B5368AC8FF83D8B886D05E8549C212C4EB2EDB1BEC4A7D100F4D1885D250D400D41289B86CBCFFC7710C8A57103C7D1428CA20D204D61E8B100242B412DADB6DD87D183EB832CC15662644C8C981F80E4F903E6DDCE04E43982598BFE4F5B6088D54EDF68E21CD32C024000BC828BC534E4E2AFFB8B4BC388627372BDCE4516758C279784082322ED068C489EE03DD3446E90D2855D088058AEC94FF12A2D0C5203E8209EBC4B42CC82147D108C35F856C135F3052BFA9BB605C4DBFB911D9F837F10193DC553416378CC591018DD763BCF75B76901B90B2692F38DD1771D41AB8ACBC644D8B6E36C520BE6C63A0E3513ECEA72D89885440A034ACF483D239B13002619CA22D60041DDEA733C23870CF5DA471A8DAC64F134CF0FF0CC361BCE87DA08BF36833D129CFAA243C3FB4C03331380D95DC1585943634752EACCB1883A5E5CE8CC9142308751C89C45E9322136F6925A5533D38D92EEDA78C07CFB442FCE52EDA8D7A4F20E0A7890FC02FCC'
	$sHexData &= '503508400E7E24814DB008CC0A15A1948CE69DA34B425D0EE466820F822C01E7A8D8770D4E40176A1172340029BED26A1170308D34F037A0FF9DD18C3BC674192BC37515510D878C41331C08500BC0020637E892A915E012B4882D0B86E17779012EDDA8B2474C8BFC6161931CC89C2DBD62D564C974D5E7818585945EDB43182B4EE5038BCB7B2743105878CEE34191C90820A3C6018C939A6B5740E1C9F19315C64FAB2605DE2F8AC3A7D38800856B9CA55526DE8EACA3B9E8A74031C1568BD05188D27240EE5C656920AF9F945806EF36F0630DAA4C150CC0A38108893A5069DE26ED0EF898B30305219A7D2B0528848142D071B68F060E18F0BC8BF80A8A1DB89434B6D0E81502BFF05F8084DB648B0D9E6A6419B87517DE8B0C910FF404F08DB1BE81EBC1A84694E0882DE80F44C8519AD0483DBFF02556860266424E6830F95FFE3AA090A2E83D3176F4E1FBDC10010F85A9567095C0393CE1DBF526012F21D85623204DD25DF1EC066A12D0EB122BF8020D3E8D99261928FC8CEB60687102BE64469610060B08EB9EC027A20C4750F6315000CA1B1F3623595EC5131060405466607CCF165C23A157741150AD5419D1380D45502D4CF93A41620A256477830BA7621FC2005FE883C9D48711E38B01C3283940497F4FC6C8C0D152244BA059C01A40E8C835789001EFE7C46FE48E302E08'
	$sHexData &= 'EE3EBAECDB590518EC40170BA16298FF4A37DD47D77C062825B08B929649048D482DE1DD3C9D833B590C7340603C99302C7E5EFE94C69B98F6807914345BA03B5809209DDC3C0A8B403FDAD19201D20C22230AA7CF1345788448253542B84A36AC562B23D519B714FD7AD1819D1418BC32354708E8C6750C8A018342FCE4F8AA281F3B451406A2B478A2A39492122A64C3EC5E4D94AE83E594E779D02463FF500EC3B8400E18DC25E8037C46E0853399694405E9C63B7ED2CC41584FC2473469C1E803BE1F3234891DE8A3208358CC5A0DF0178AD8E8C2D2EF3493683BF277A97337A013A90F7428A12527453CCE6004711660FCE8CE5C137E283B46504551921698A8741021214160078D681090360BDA26EB0D4275DC9076D91F45C79C69635650247470C72F380565A8026ABC315B1600265FC30C0ACA0C0F8C4B0E72EABF509188027B01180C9DA5528D8B213811C57322BA51117F018DFD1DC8F0F1752933D250084653B05F0E43D0F1FFE09119E28204B101EB0A32C983FA12BD02F73BA455EC668B8575527AE98A85775C109AF112884DA0EBC506D5D4E284C92B363A508C04DC296DBD6C4C64C3D20332AE90D8811C902E6F2447049F315180179A2889B55A015A1176D438085001AD696BA4E0299F894638A47C07C3508247C7C56F439FB67560708930DA6B00218C78EDA464726AC8'
	$sHexData &= '81D0392240F7958B8D90817C450C692829F368506863778C19D8081FF55F20CA05F4203B320F4CCA0ACCC0ED39305E0AC88BC17D275357883F536D91D28584E8D1861612B8868D5A027FFA2228C595542BD38B5E101380171A24FA895DFC38645C844B34137C8BDE899454D09EABE4FCC0431F1A29FDEF08F543EB0F5231260047FEF80D34C02969B633060158FE161B461070486911EBD368E328A4E951872430E8CF421502413914497859C31F50D580C24A061C0A02D3D3D3E0E4A8409F98806B3A71BF8B032BCEBB81674268E8328B7CFFF7F1FF18248B74182085FF7C177F0E4A11120D7F043BF176072B4B13A3EBF183DF5E0ED5D84B68B76B0E750A4C1838B6D8906D7A1C4566D1A6956881DDC803CB0374D628F6FD6B7720BA583C52193BCB7415350C4F8450D30FEB02B0018CDC69ADC453B76A750ABA387C399B90FEEC00F48744181425C001DEF3C11E05743B4E377F04319316B68D86841940CA19387B9B6A76776118B8CD3BC1E045FF44AA99EB2E83C688D7FFEBC53C60188623382A0001D27F0ECE671C3B45E87553E6BBEBB81C2283CA04E7C797672BEDE90E24EB5936F2E3DD13C318209CF95066DC5E74B5EBC97CB834680E28AC02EFCA105C11A095B23992CEE468DF34DC16F9A66A005235100BA46EF5060EDEB6381D283EA8BC0856040691303809129DA05A40C31B4F'
	$sHexData &= '2AA13E7F0E4661127089A64D200748F04A6937F5454532C8C1F91AC9FF575BA08652C5749C066BCE2C030F5F89085E9AD42A0BF9295AC9A3DF510D1F929143416EEC5423DC1A9FEC00742D9E3AC9D017B4633AAAA3263C0A06B950495419D310A1A0D7AC5FA1D737A43F31460A01DD457C08742EDBFA3475DE0C1485B4F0DB406D6F2C4E8D48184FF586A4807751610E794DEFA8FC1BFF514D1886E41440F36258DD3D6468C01815362D892786F48BF8982DE23F1CB5E308D7FB4C8D8798B5024F68899828896040246610F730A768893B4E0F0A62411060C9FC0E0A492B8B8F9058152ADDAF8B115746F9A538765A933168C81B411F5197096BC128572B68A96B71C4107CAEC04EF8106F3C51D568C857016ACC1151ECF068A270306745C47ED17EA261AAFC4CC6EE8975E0C61C168C64C3DC0E3A3083C5C4D5121F8E574B225BEB9F89378D4E9D7C200D5E472A8B8E181621AC400D39B5C3501D5D16C4943563C12D214109488EA01FC4C9296243FF51DFCAD0882347E4843FEC7D103F68B817933DA4AF453E79901AF42332713C0ECC45B8902377A47D4BBCACAC1417193059E444D3BC8C3403198007EB088B4B4A51B169933413A956994EA0A0C50742510B4E084B0671902B6EFD22C7060C0A0A397BD57840A825F3FE51F821350512510F8F0260793580D604B96284F53DFAE0B21C7F03'
	$sHexData &= '5004C1782A492A3E741130DC3114578B7A10FCCB0F13D78D42238BD75F515215A00D07061FD1C872F3CE94A624721A5C6D00890EC679B0C006A44204EA3B56436294F4B38CA55F50979F0C57C33C8B460E0FE02055B35A7989E4178804305BCF5F5251A8C5420A1F502B6D28888E04D028A8AF2EFF042B01D9C0990651A8DF4237F212B7F902C7425542002640DA1648573433D2F2928B8B5B104418B55C9041AE60FF5C8C6A580735E709924751AC5FB72C51A34286FCAF13494FFF528B5D6FF444838DA239CAA2CED2A7326FFCC20E1B95FAECFD002FE1C5D06E807F748AB2461CE28FE2D0A2720C3ECB184888034200E2B55C0224745DC0786A7308F9328F1AF02E935717935B2EACEBB06E339F036B1A0E56FE5663666BB690A132948A14F4699AA6690A08080C0C10D2E178A31036CEE224E5873B322841142BE641249663044224ABE33EB88E470A365EE7B879721487037F59A61CE2AB092CA96BD02C4019848083DF2E1E160EB5082AC14F2E1FC4FD538B59CB4D8B062912C9CB5BB43850669A864798464726690C6A35E7A2CBD4096A458B0941A006ACE37F3440C335856B3F2D3437A1568413B7A35389360B8B3A481DE8378D4F94223E4A4E45BCACDB620D9319FEAEE55E22FF33C2278861255F333489480DDFEC241E14FE5CF4507F9C8B0030AD08ED8E23DEBA33F265EC7BD4D2'
	$sHexData &= '460340740ED3B260C2F683C60C1A75F2FE4BBF7E6D15F55FF702A1B0092EE723781B0D620B9303E1C0D0A232109F886409DC5F08FF31392E051F203DBF504C6A5E36D6356E522FDFB1840B1380265440AD82197FC1E8033BC1760FC4ACFF30EE7B1FF66C922C549457832E8900091A0E07A940661D7E5F215A509C7C4603C1E9023B375023A57E26C8DE4CA9BC1AECF3AB5F785F822C20216F32157F524BA470895A9CD58356A21088D9B92418AD972714A951C64EA5DFE72EDB53AC7E18071C25AB8AD06A60803F086A10A10258A742D09F01CA4B1BCD0DADB4A1965436D3886A60E3274D94D88E3676BC8B3D7FD8892199D2C67F0F44D94B23C38D34D33E62BC6D000E8B79201BAE15766E664A3C87309E439D666347187628286686D6346598D6CF5B273858079F5F468C5869E1439C5F928B17A63A778A710D6C6E3B45AC7E6376339C7B34DE7E8A1008DBBF0C3B2BF1A499919765531E77D6642A303F59AEA743013F74291383344757203F03C588950846BF3F8462E3774F1CD85714890E79FC580DBD7731F810CA008D1C460198251D6A5D5257F4AE786A4A388650893749D0095E8741D4BF8B4904418BD05BD03B3A04EDC255FC005DFC757D135918C3FE943955F47472E1C60B41826E8B7718584155EB05383BC63AD96E0B49AFB1198CFA565253106B60302B1E2FE00625D652CAB1'
	$sHexData &= 'DD000CECC3D3123A0E8D486CE860BB90C7890F3D69A8AB9054B21AA72D4422DC1760022F68E91115202C04B91D0BDA61630D42D21B8CC350801401FF83A1E9F3A4072BF08B5114D1FEE0BF1C149C288AFA89711003F6565034906788895056E44581D6DFDF52980C70923A3085E858221106DA7C771A4CBDF77B10140714046B9669340CEEC810F5A612BA1CCCC004D0681001E310E90433E045D95046C13CFEF7AF4810ACD0CB160C04010801FD93589152FEDED0342F3055D8044655446B2686E23AE255C4780AE0C1838AE0FDBCCAB5EA2675D0CD329B3A22B52E0679E698A6EC662219F40B3F42837DC402224D8129D02C428E90496605068F498655D8D442A18DEC2ECED68BC11B743696F8084C33F6B258513A5C5F75D03EC207B7C26455CC1690086A74D86103564CE534202CCE424C271882C0106F1A3E194441030D7501527E7B404A14DA31A1E706244074503C58808ACC46380D3CDFB496ED14FF1A34021323ED5BDE40643E68CE979C586C707F700E1BD90D8B5A506C897070283258066194E441ABE15EEDB649482614AC3E518D89D40C19C7BD4BAB284CB16F22154E67BF52D5E107C8202C8D7714567C50C112AE31014FBD7AA5DE921A39807D0C69745B7A241E93A6746DE44240C15100E75FA64AC40F96046EA385DB4B93B00B9B4A176B61126E1F084FA6089895B7694046'
	$sHexData &= 'E004BB01B4EB031BE67EDF7412C10C0A0275DA4B0A8C6A3C221932C0036DB81410DE904449DD294F8A9A02090F45D950056B9A5DE0EB30F8C1FD83EB01743208DFA78B8FBBFDC61B4A361B6AB00131DC7080D58D5F7C732856614E7EC01AEB467BE07CCF6E07C7430A885369F02C3514A49FCA440A3BD15F1F2DA5C896AC09ECBA479AD41580D9C82517929A73EC3FC8760909824B3B484E0CB09F530DAD7E8B98C10A58700FFF1E4AA92AEC1C4A40011F412CA48483C31AC36A1FF8248D4AEAC213C2487AE5B0044475ED14CABE55D075747E6C8B7670631DE809AB606817DA6BB38C0F23905204C95A5400C30659EFAA3F31C819A771A01123C1FF319B18EB209D9F092B54389C489F8BD149B620A5D01190498F9CF49E186D73084EB383E8046A0EC1ADA24642BA334370ABC62D3E9F95F69B80A7029801F8395110721A52101AA5186AC4C87128EB8076D1BD01FE1F914ADFC07A103BF77717897210088C8D94B54021A8DC22DA725E5B62DDB94E7C9ACF2BC75C29414E478D96A450D23C7AE40B6B957521049608CC48ABA124D35F2235895853FF48C1091C7CCA8227E010565F80802835A5BFE02A538BDFF953175E06F84CAD5B22DE4752FB52E10B4A521F771FDC8086129F652F6660C90544D7C2C70C58F0DB0F1007D64EA50CB2C6473E890E38340E495D9BC24E1DA41F6F38D5A915'
	$sHexData &= '19404173844ED9E401386F28DD2850E4C19F55A792D44FF0C257837802C53519CAC746A5C7A8044AB12868524FACD501297F5B732207E13AA59F9DA83A62D4F2109400FFE7127E3E81FBFE6954C083FB07771B895E105067830D519EC47CC0DD5587FD8BFB940781FF617630B88ABFB87B06BF1703C03D0B428D4823FA0EC60CE8766B51D76821C4922E206F0A2376804AC4F8188177414253FFDF7275C4EB148D412383E0E08948FC364C49B10ADBBEFDE048E16B5422FC2450897E1411281B654A3314238FA493A90A28077C8E06180C9FE2E60F303C8E1EA47AEB6EF609B0C042D5007735649401840E3A1FF7C958D002F0DBE65D949A9EE218154D088C7AE3DD03CFCFC6041900E95753404EA01AFA576FD36D10B5119F8D275BC90551411D1F0CAC1B2F059D0641839A1C741F01257F3C162516136D8C0664028AFF0FFF2183344F56B10F0FDB5AF10E0F815232FE23B9992806EED0BF79D4E528A30B4B061680240F281E994F143EEBC9F34B015D285615EBE0F5135D1B6CDF4439DC0069536AA8731D026A7C56A1F92256BE4F50E07C0A424C00935266995F360A648900AB7F0A3A14FF01A63BC2721B8B096C14512C2C2DB6FE4D100790D46BC0EEF90C3FA000D0055CFFD156890203D502520A669059C6062EBE65EC06C23709ADC7E2FF913176F8B0364F824A38880D5F2A50BC94EB'
	$sHexData &= '61BAF83897067C12473178EB2174FC837E4C00742CDA68A8FF8750643EB9B9C87418FF764C01235A13B90844932995EEF799C15E9EB80CA3020EC67F5E00EAA685662E0B4510B08120533E04AD774CF42A849B9C4903C9E2569B685D058AA410BE3632F05282DF4847480188473E2F759BC28A2882C7735E3E894F771F1EFC40895744C7478B5F695FF80F891421293F123C01D8C003555A1312AA921698331A10940E3A92977FD8731E2D3D1D629B6A7B614B47C8D0A9F8A41DAF42A3D877CE20894740D03C3B2ED00508443906B857548B082BD1D02A851792471CC2080C2C5223892B0477408B384C52D1FA4B73890B767A7B70C3898973EE7B14EB23CC7AE20AEA038A00C60A0CF60D4BBE43105C8CA4CE4A23208215C10A1DDFC70CFE7F102E8D4F3C3908751A837D140175D6FF28BD306E20750CE2C6FE83D0FF319EC3DDEB0614200B720C66826D83E25A420A1F1D0C330AEE4F380BC175068DA29286298661515647F4795E712C052307632049000C2010C93559082B0070101C3FED21F1C980BA1140BF18C9D1063F3CC8205B039F533D387CAB5B47C4E88BFA2097519B00EE8B02826405D59FF614FF0A1A1C4802890A7E96C24E3FCE5CD6C9750A6F37601A7E1F8B4E544698F6701AEC460C5C5A4E51A2DC147670095A647C104C7522A4B3E929E963C0B9DE338021C49B8CC14465'
	$sHexData &= 'E9F09B5EDFC5F820F1015B964E508B01A5463A631CEBC87602D0A6DF846A948D4EBA4DBC2A5297C6D86EC03C535212951D0F0E90361013AF031EC8C6E892E1159A8808EB0F6410CC3D449DC1D5C65C8E73B366F6168B7D594DF925CC734C80467CDEEE06ECD051E2C4F1878E036E2638E98483E853CF4C982A795CF18DCFF7B90F74CC8B1C5D0EB155D053609776912BF87A3BC7E82F24D1929285C709AECD353C6EB4043AA8486A05E8C544BEC910FC52FF704CD710A65BAE21AA1D1EF38204F0230EE3CCED2C19400771C95D5166D3C1EE31A56E07BEC3EBA2792BAD28C3794EBE208B8B7526AE3094E4B82BF10376DA41EC3A7E247F844AF6DE4DF1440EFF4EC904D63C0552EB6452851686B72EBD999FAC898F05D8E600C407A5D578E60014503CED8B804705CED20657FF501C6CF85442730B3FC778161CAB422EC6D86640C39CE1DE009FF1BB410FB755A38B812F0C70732868DF7406F6E8665304FE751D0833C94C217ED67E428300FE2EFBAE840F44C1558CBA839130045E5D5951397C2999A68363145B3605B06E4639421A561074851698DA125E0CC08B03E45BB7DE7B13894650AE8BEFE2481E54891F324EA313E7107947347B0755C90C16C3336B86B5AA3FA313534C80FCF7BFBBC35DD083431AF154751540C035C1AA885B9769209D19A4795B313025C868B732BD20205F8B32'
	$sHexData &= '2063891E5E7355B593D57D695FA37134A10A6D24385E2C7A536151703E251038CB1E1E8CD6A07ACFA7C1BF77D8CFD31E028406FCDCD0F974F33CDAD84640506F8086E740D4120102C6FD35FC0A4075D8EB9908D4662BD874145C600CC9E92256F0397E32DC3BD875BCB64DD839A4700B521FFEC6463E0EFEDEB02EEACCF1CB8B4173B0672DAF08818C59C31F908555A12D9F06B482F5A5B52C32DCE0C0877E6107285B896E7E4009422946807E4842AD210AA6680933575A08238896532F1A98C046B424A480ACAC33101BFF152C8023B185C4CC5644C244444615C728894308D5019FC00C6AC8538E1D6241276834D5907264C0CBE47368213B0DEC02138BED51EBFA425424223C56191038111010286E7F13E03B4A50B6068B302BCEEB23F642047552D61E9ED12A3496E8F2461624344280B6695E4403C942037691BE8C46A475C02C7224343269771B1F9E26BD13B7FDEB0C28E41552884BD4E1350F29141FE9D4F10E519594529A24A2BD622B13850AF61C047404B2A5F9FE2168D283E7022102B1287B0F63C9840962E984C90F7417E34A56035D14D0134D184F89EF0AA62F924E2C54090A18D3A1DA6570A2EB110D08395049574E767303890893FE38BCC40B18F890589939A2DD83000A87AD72088AD4E0566513500B44A433138793BA365E83426CBCFB464078529E708B642C29AA29'
	$sHexData &= '298D229F1CD0D7240C04E0A68854BE9ED79D8ED0A02C330C70312DD300AFC124C47C58D73120FBFE6DE10C482E388D3C5A1E2BCF57C018DD83EE384E12806B8076CA188984101258541A42644D405FFD9403F804C55E69403A3494B95BDF20EA6078C3D255188BC2E8D019E4C149F0D618B3A1133A5CDB8BFADAD428EE96958BE3DB64D90E375C211F0EC2BB1C18904B3A83C334EC29BDCC42956812019B7F373EA9389984C2EB6083E20380D40C37C0FA03E42C9D1B6EC1A79E5E90850797699619665B328E53EC3D6B629A00CA4111B9C61D948A7610D5037D0CE5E28DA13A7AE9D8105C88B26764D9F81312D810C5D79CFC1874161C7B0C2800DF7927783C8D147B5498784028ED38F71A216031729BB33B8BC50EE6F9285683286EA2583C24A0E51819DCC9125894941990A9902BDF3814159D9C52E0B95A4A3BDE0AAA1EDEB84A1409DA4F3BF973C14D4407F207734206C874ADAC58023C607F35B02072385642F03BF7762189163331A28A400B7C466176CF2E1E2C891C6CD8047A567880ADDF8FB9EE09D79F214A745568BE7D52455C0C3B30763E4D34F9B8DF5DD4740CC4FE7406D20275299EED9E2D55C64236BE1CB0CE330F15B455B6B7C122CFC644E1B03AE5815F3F6C03E11942FF09F647A82B6CB02DFF67D8765A750B0F1C2CF884FB47C45730640AFF69DC641E8D1C48CB66E0'
	$sHexData &= '3BC3A542BBD97326B85849890A3E200B4B32CB642CAEDE18975E8E4F385FBEBF01953066FC58DED1FB83FB2060DD4BDF73051AEB7AE93F7355E3436CB8226767E103C0BFD4095664750700476B575B085C722B0F863FC95C3A97361767CC8145F4866D69E600F210B835A4821DDCF72AE908D33033DB349430B96F4A693476B809F807172EE57E128B5DF450565389F39531D6C8148B55027090DE6E03D303CB72A31E150F6C47B247E74733085D874747D67C0417C308D7591D2D1C092C52335C0F39241E4062005E3897F110864AF2A8BA0CDA53C580A22740A6B2707AB3127DABB2F0772FABE4834FA24187AD5EFF104F20C91127014044A0C117F6024BACC1AC8BC14B010658859F8412C4A115DA01C620EBCC38D2F4001D02000F0ABC30ECA695B841EB0DF3DB0C0E83890FB09003C9EF160CC8040E0A15865D86EEB675CE2B9A112C610E9341108B270BB9F0913083663CFE13012C2EE00F3FA963A313791831CA5A05A1AF09049257BF53B040236928EB4C8544CF46C1CAD21E48E416162A6DB51C104C1550D643DA7D5F8E6D8DBEBA2C4388C8077442405EB57D0001179191E04DDF4E0942B5AC70AB08641800C0A3EB744A5B6F037F46980A0C5C33A4986D07432AD2397B44312E9CD2826F316D0F246500232016BCEB4DEA65892BBF7D8220904C5F581F9C9626E438B0CD27A72258'
	$sHexData &= '1F67EC44C01308253861C0EB0DE3750C7EACC4E343B7013BC67517D6437D09E3B5FF98145E05FC2BF0D1FE81FE0B6C5ED81A0AAF2077288977915763A1B5209F5614790A1060B702CB07E2AE0A18A525142945E027B05C74B7263A16B52A2562D83315B000FE8D432B49FC2F0071C333DB44582383E3E08943FCEB0B27581D06558BD84C39FF75B0E440FEF8891F538947141E311E42153B5F7863818A8CFC4DA80C57056C215A2DFC0E0983F6CC8CBF89F16D5C56A432BB2F4B696DF7E31926E3DF10F2BA82BEE012F7E2305342BF2EDEC126DA8BC8422D5577947E03CEBE0963F083D330F956A3CA4E20A6CED1B803D96D315A057C130CCBCE13D078F0F8FB37FD8BD72BD11B5D04ACDA01D1EB03D113151D69C20BF667005EC1EB1D894C4B1A8532507ECF688F049CA9FC9F3F61F03F92EB0B8BCA83EE0175DF88832F046A50807790728D7CC8C4BE155E7FE7F883C7967652F63BC662E05F7C248EDF0BC39F93D271680147084E38FFC577997A6E616A772256515704A263908F340FC6280057C072C7022D08526D2BE0719AD8B82C57C920D771C02350132A160538447883C324A65CC7774C57437CC37642EBC91B53E3C18E974DFC1122894FA602168C17201E005F082137004B1F0C40A744D008C5EE5B5A02AE940407666055589A2752C956B72803E365F58098007F8B32B150F8A03F'
	$sHexData &= '70518956F556BF6ECF5640121449B62659D7B914C5BB598852C53F143DCAA9AA381A42577F701A652579E52989705150C74003B96A0AE30681081826B8CA2FB9BB17C943556D2484C44870507F049ACBC924B9D9D95BB75766685E7B477275C433C8D3541D8E787AE74751211FA472F8703857895888955C353E7708C154C48A7FC5B955C06EE8830E57106C55FC63C71F21C8EA71628B5F6C04326E44FC2DD0F483CE8FF1763712A60328BA3D00C4080EF2B155715D3FB4693AFEC57137D7393ED1E867BCBBC0E4C87607BED9EBBA03C3D3DB09032462758220C0738064A80E4F0E1128411EFCB026AE8413366677147650C288BA0A0485CCD8859F97C7458D0C455B90824C9501193106874E8E9243E3223F5B097CA75157A0322418CC7B4E8F7FA10678A11C4D1053579F5A250804A5A04A5A70D8605095721B8E4C0F892D5D11771FF283FBE2320C4B0CB804426A378A68B3F848C67617167E071792185E9404EC0AC009E31D882BDAD1FB7CE1AB441DE2F8C703CE56DF02F0039F8D3C1B02566180A205E7FCF0687E6684B58C2BF34A12DA3603CF0BB80CA40D3CE1DD14B862245E5F83516E07A0296C2F279FBC2FDFBB7A05D57644CA57920C07560321B58D790BCF1827AC0402EEB72A405AAD2ABB01C6067931B4DF53E518F1001A42E405077F4900ABFA5DE886743920395C50A125F4'
	$sHexData &= '39247C11630B6CDB50BA93FA042BF004EC39389EA1186A5ADC8307CF4D2E95490738CF935C0CAD4F83E0BB432A9523744A0738622893883ABF846C25CDE52F8B0739399B4321F883A6EB244EEBCD21E042CB1F8108FF75E49E2D3B94BC6AC429A6287403928EA12B40D712044ECE38480C5D463820221C44AF625D744176C03CBF1812105D076A0053CFC901C881DCC7F20200185F542D0430850C3CD4A45E681F8C4EAC19AE1FC0926AE24546CA0673184A58A0F01ABB806F7ED8B8E9A28B2EDC2BCBF7E9B4C1FA92E946E9038BC2E4C1E81FB44AF5870C3AB3771E1C6F8636802E3B76C1DABA5D74D1A41542E3052BC6A3903895753D2B086777A035855B861C30E00EDC0FD0199A4BC883C174D03B13B804BECA47A534E0CF9C6BB00E43C814D8A3316BC12C0057E46074434EF63490837C80E4412777A5C200E89F76BA8D702383E6C65710AB5535ACA4CF2B4DCC0EAEA1D86AD1039CA2802EA8240CBCB46BC02C927A0287BAE47CB69E657DAEB8DA4D454EE05FDA506BC22CA2DF2B775BB248C201A47CE801DD5455C0750B3A07FCA856EB1446F799D80D8BD780A52CE1CF1001EF9106D31DA227771ACF420B16438B5343D818402A0B123143B08913339CCFC38306C36B962833604304A699AA4CB90FB50CCCD2B13F2BC76BD12CC5518A94E0E6025044F32BF23AE2C5E0C30940FED653'
	$sHexData &= '6726BA62A4CF1DBC38CC8783674AE83A552687C326420B162A0602EC86570B0CEFD8F06826907450D3B2192CB2D7C3BDD3B998C3110BA93562B4ACE09ECD90005AD49856C81C60344A8C51CD4378B0F77E16790C974E44F77299819F1E862F480EB0BDF7BEB3E6018F78CC96A36BC62C897D90FE0C5403B6516BD658CFF499A826575C03D7F8ADC06CBB76442658DCB152D0D50EAE53CEBF8CE4F8CBD73A0525A19F5465F14875C3DFF60D6B6018EEF62160B1C2704E3DAC62523C90F87479367F3AEA4AE5B81F0A18132FAC6AFC0D837B089EAC487963E16A545394D1246BF02C63137F217CC0D45003F7F95F94D654DCC8CEB0383612085027DE5CF065980878DE9EE45255D7FBB0D1FA400306C150B487DC3145ABB05A89530852CA3091F53BC67D12D0C2E096297E3F7D669C740E0DCEAA42B42375B089C66253746F087F8D2E7CFBD6E1ECC1A14E49578BD01A1AFF2304E17A474A6BCA2CC785D60FEA04395C443910349EE669DA8B0C04C408C80C004FAB69CC0F689E9643948A3124D42034AF5FB29A0F18D90C243934C0D3E428C8B8A0BBC1C41EFDD8185A6A7039778A540A89071F28D84542AC64DB3C5C96469949C0876222F514017B6C0DB6DAC541A88F4A6F78AC63B0681F531645D1BFD0A5366E2B7BDE42C6C334107588C096583C68F0202410E746E546041D0A1EE301DF410C'
	$sHexData &= 'DAA361F02804DB34E40A14A0526009D71AD574405761198365B56C4C630CBE72383C50B7E28D47142B26E395367E1C2BF750FC33D269C69656C3E9393C378E255705412764D10C0FF3BC83EE2C6E123A58E2221CEE617A50E937A929E4D5C03F549098B181F0F9807F189D7C2DEE21D8770A513BFE237636902ECD2DD9073B4757CF06C8A504B7CE7C8008EA6030C9477B86B3AD43F01C3044F3075332B6461488B6D4A9B3C1AC4715BB810A1383005B25DA27A40711AF8B622D21F4734259535DFFC5DF29E30A3938C40720C268F426DC58CF02DE543D58BE7D773DAF9F97D62BD7DD216AFA068DF5B819C046E0825103D718A5F9253436D0E40CC820BF1E2ECA68D0E5ACE6AEFBCBC48BFE2B7DC046E083E7E0B9EBCE5DC4BC2BC1D6B8C77D1006760DA0BC7F4816835766BFB44DC0EA23AE3AEF081ECA4DC86E3B8F3AFC8251F2AE51BDAE5C6E6C195088058855506F3A867EB57E9A480EC8A6C0C1E1055FBF7E29DB7EBFCCDBFAEBF97524F7DC149B3918D75DDC69B4E41E9F208370E5F46C5C3A1CD928E82A50626BD2642FD70705C670C04DE2CCA51E252703E4D07F8E4060C040188EF057D7C38A047D60FA797DA4B533C3F0413A521F7F76CD5CCBDF6AE08D7EE08D4F0461D0D03A97CB24DCE0E43682A1917147A1E0B5001AAEC47302264204DDC2B5C843BC30F5411C8902917EB058'
	$sHexData &= '2BF9E8C1FF0550ECDB20BF65AD0BD0D70ABA524220AB42018364F07EA3E9D55F204309EB5505DF242E96981A1ECE8D7A0406C929AD42A054682F434DE003F58B3FAD36DE65F943012433C848B7CBD010FB1703F1DD4E20EE69D9158B4A1823CB68A35858E6C804243A0594CCFA9A3CCF949648019F1AEF08740C2624741675F44225A4ABD0C65004B304DC09FDF8817A08E40A750B428C6521CA969446D25449C03847E083501BDC413EF0660150E020006E4F8762742E14F659410815816319BE1964017C174EC074F20F5804C5AF6D66D11CC06790325AD03B0F285B02BAB1CA4134FD0F2FB6D7FB8CCBC4818F18D3D22CC2F2BF94703D97DD04034BDD5DFEDCC2B4D02410666B2635DCB9086D835D803B5E0E7310D453DD546E027307C1FE1A02E3CF4EFC34EC50EEECBCD32351185C6978453A7904A6D01C8B79CA0C1BF0D78C3B50A351401B659D5084570411A9A23740893E89568C32F211371A6F235979E60CD83B693CFC773CE075288934D88974D856E4F72718AEE9FDEBB53BCF752056A3B63D684F39549075049D20397152DF554568012CD412550CB9D85AB6B5D41CC34A10120A2D99672802A6BEC405A44122EED0A51F5AFA1FB3C669F04B41A7316FCAC2744CE9EA56528D4ECB6FC727E019DA6A1C88C647F7E0B8E304817B7A711C07F38A18A4F98D4304E024D35A039501ED'
	$sHexData &= 'F9C3EA5A56A60853472000B90CC7D7431B384EC0F1431CD17671C95299A499E8E8E8E88D01E405431C500D87C36F85EE3C568D4708384441401F0BC0108DEB03238490E396429C2C1E76BF4C42172C3C4B0C34A2F6AFF855C2C68B34C13B73F805893CC1EB123BF21159E950690DD626C17CC1E4D60AF9D889381FD8F80850DC6353083C115A3CF5F2EC7D52CBF032A8468462FE69047CC8850E933F857A2BD1B8BD5817EC19C1FA90FAD1EFFED07607BF756821DE23EB0703FAB842FE9642F211496A8A5D168B41893587AEC8A46B4A80A07450246557481A00EB0DECA296453AAD1A549FC1B321C332F96D962883F4763C7DC42E10517A974CA22B75FCC04CD2E096BAE4A336095AC8180750DA61CD768A08EB8973410EA913B0B39F9E085A6E00B553F8860F80E4B686D7CE7314578D7A00561A1590F8183BF9B8740269B0F986D26FE501989C86A4B0C6ACD21879F17607BFC63B2EA9EC00B186D2F6F04D280145600CAB005DF68D3C12F47667BA8F801E246756D155008F043C66FE0C1200FC0174AA31B6443102572E06C0DED957FA8D47E0A88123FA5133C35E950E2059D3575322A051C8C33410671130D8770C37B637025B79A731A2045F44695C8E807CFF5F0E8B01C030303BC2C52802BA8888D0138256819BFF087EF883CE0F895DFC268BF1016B041C8F8133C9211F5D87E443FF'
	$sHexData &= '0062E0C9CB4E01C8A14C055C5820B10DE39A816F4F8807EBAC51C1F822A002306810406A22E08B81830E5076488B3765424540F34DFC415924A23710B389185E5FB912D85604493689180FD462A3410135A0823A50B7C31F4427A4278D710CD55AC5B6609A0CE94E1BFC5F77A2F518248D9183C6818D8042A279C2015C4E45972028922808DC51D61731C044A1A496C6D178C5562394D0870313A9C58D4BEFCB57A1AC33FEBF555B2164907D104435A25E8E7E2C3F570A42037835004F3486465492E89B617C33DD0D80BE4FFD7DBEED018A32B0245799CC84C0293CC6860440F1DE430138868E0C75FA2D9E91D8C4E851A21F464EF83B3426BE4E341E8D5E5C9FD998105AA2A960E40B1ED89B3B74EAB3754C3810740C49E31674509F548A9992661B3C12264D0E02F8CC57421B85016F1F87CCB09AFFBFBD0C18A2152111BE0630D17040631B1074689F5FDE5C124F0AD7A0725DEE987511480C435D5350485A3FBEC15B1346AD03971C7416257805EC0A42462C53C0700501BC2924BFC65585A403ACA9A74DA769BF5178B8417DB2104D21924C7FD43004C81C0A0874F443D2FC798D5F148D69C646747FD838DE72467E750F0918037E781D0C344D601A4C5E489A2C315F3258EDC8544C833F1C4D303C007E7E002F1E6CBAC6302D4C3F3981514C3737F133F4FEA8B932D87DD598807413C7'
	$sHexData &= 'EF260372CE1C14194E085EE923750F0B1A74BFC037011C4B5C2E3702821F7F43A6B24B0849EDB4D48C234A040340DBD2D244FC0A75DE232AA41309837BB0627A6DFD2E0D5E1A544E8B04A9D369FD700B6A20677FE64A00D81E604BEE1485358D5490291E414B80921A84281D7235BF3834824E90E9A0C4799C2089020F505FA6FAD8DFC4AFCD202BFE8A8BF82B205AEF9500F70C1E20EEB6744A603C1F08C614A096A81F00023181341DA5DCEE93A3C3EB0E719638CF7DE72434D7AC33061C04E175C552311D2B4C3980C1DBF00883C890FC5D28EFAC7710CE1EBC3E5E4AF002500613DDCA8BC1677F828A3C888B4A1E7849483CE96E38C32CF6DFD28B14880C026068054FD2B2821E7A049D23A29978B6A2250D6E4BB3A4F61B1F08FB6ECC92FB5AB82C8DC6B0D41328251A6186E760062E851C678BB6F418D42E43A4A15080B2FB14FD08D760741FAC4CA30F1FFF9DDDEAF6EB076746890D83EF0175C5AE9139383C8CFB5216F22C6A42B772D091BDB68E831F0F06FEDB4ABB422F00981D34F634F51824A5B9C1A75A2CE3068B4E066972C9C84672F65BDA9643FE1B1F6A7D44B02DE30C4742AE48B1417B0E3B0634F7911C957507506903014903CED330680007080A185C5002F7A607D66E48E0C2EE6D9C5F9098FA6122C203C68383E33BB488C13A385EC51278E0197F426EE2125AD04E7D'
	$sHexData &= '1EFFC00161B88D0E83781C07A21E264AFC064FC235711EFAECF2426E28A98910A849C956BE55FAC4A04D178BD890DE0AA820CC21CB0847A14A5066E46A82041A490D05F40699F4945842EAE6DC0AB627E2DF4EE83CFF4FEA3B2E741E00BA28B1A64A26DBEE4F6B5C3635F43945E475E5694DF814F0757CDDFDA927F001EB23221D052C8931C617F02C1A304EC3AF6770615125049EE3BFC8069F029C7E56087F75E4500E29C1BB7D66C1217F66D10A4133C369D8E081AC58A5A655C3508D0CD8C2047583D621B24D86D68B368B18A5F510A83A4F395DE46A41238D82E4DEBC3C83199882033B5D48E69345A20C2B882A0810AD823D042C09931C6F13A9CCC1549E4DAE182B61919A18DF50A585506A46B40848B009AAAA7540E209482ACF8491CAAD0EA57771A8026880065F94E0D1BA44337FD29BC1B73508890A7405E2EB9329DEEB03AFA50A7CF6837F0869607E230C05EE039FFC1130A059B2815C3C4F102AB256156397B05CE0261A47862A1280C8009F6AAB923085EA53389717A8F4433D8894055EA00396433A7DDC39B39D12AA55AE64EC75583241BF560131AC967D3BC30EE06DB6EB05420C9B68151AEB98F10E8E2B4558003EE026DCE89692890934AB725AF24A7A8D5E63C0A8CDAE8FF81E4EB101C255F04EDA56522DB38503F21F8C8BC27A16655E5B24A8D4D23D408EB06B793E'
	$sHexData &= '4029343F908C897A10C5D39787C90C5835BDCD98C246E63294945F95DCEB93EA84F25788A2DCC510148BC7CCD3AA5C436AAACCB2F87000DC6420626690EC4E68188C1CFDDAF6505719968669453416CE0AA380073BE2661E40503CD7D261184258B82828C690E23BAA9875B50735432F513BFFFF4E41037189FA03CE4AFB4E2BF0424D11E105780D679098943F96FC80D52DF56762DFD781FCB23E96848AF0BA2BA92BA3A08887D066EAC98B904598747442B256E247C25675103DA9EC3C6D854C032DD4879408D1346009A91184C9C301A84F453844FD44415528DC3096BF70B450C596A9309622DEEFD00AA905016DF02D84625FADD717A4B7A70E350E640149BE85AA0842625128DF9732908954EF883224B6854714A51A8F09A094368F11FCD856732E92DF5F1F03C9E614F00228535156521B08B2105141AD38252C8788AFA12C248010383F9A3A0C83B5C7BC5944506BB00CEED6B430A404A62BFC3BAC8F0556C13BD0767760389C02B63B16775BA0E214103805A5535622F289078D0506B06B6725D848288A037C3284503A9D2283B4062B1FEF66C00343290A8C208009F8B4936199C34316D4AC180D0C8084CC883D591B1DE253D79D3578EBB8F4B2F39B69647C815B86C8073D9B362053001CF8FEB947260E6EE99B56E7221D5C4B6B2770C1324003A4F941CF74BF092F0C06282218'
	$sHexData &= '36967B147014078945F49BB5A8A8F2C98C500A65B24B200C8DA4553A1961DFE5AC3D5153524566031814F9FF622105E4281C7248DB7FEA9C30807E3E133209A4A15ABD764E6837E40DBBE37F9EE76ECB7C1F1E4E88463E06024309A03346E00C0DE29601A97C43ECF8578B89A9EEF87517FB5F5C7109024B420A04ACE4D005B0011A8D43B5BF9B0480283C27B6425F9748248D2060919F01E802C9C88B5C945E39F94671E5BC43BC84211CC220F87508700013C3DFB7758514658817AD464F8585F49C1547C057618B1FED9E80E28D70012BCA774D328C6A66B8318BD1D1F17A05ABEA23209E7B1A01B7F73B54C60F43D881FB6D2BDD93995AE1EB073C552A44AED27041F9076E62C145851E281138ADC6AF06F566EB2B1EC17511E6C85150560080EA43C5EB268B0F7619FCE916B1728C408D14C6B181F046E02C522B53FFAC2B1C1840CFB1DC0289926AC0CE7F9EC61A360B5F44428EE1C1EA68135F43527990FC9FB18B4B9FACFE40C36DB9F881FF2BFD5FC8F00AE49FAC4D10653C61863541111AAE0A428EC10CCA6903013B2F79258B0B43041EA94CC31B625083C2086757CB6C085D36B4AB61C1080C365F04E602D0139865F47156804332C5B273F495D1617CB7A1641622009C00CCFC543CA83204062BA112C62EE384A0E4A392432A572AC7F9C7196BA81CA112FF5F60830E7EFF435E'
	$sHexData &= '1880731422E3834F8E50D28D3407C632B61CF85776608B335592044D548693BA551259CA33047218C338309602087FBBD6EC942B7356B0605BF290E8047FDE10E4C8850CF0FCA29EF22039E4F8F4A299C98B9017A267A29987D3F058B0E4F8DAE32C520318503989CE142AB55C518DD2C27D0693E1C104474A7668F4D83D781152E114CE78487F526BF5A21866890D3241C51C051053E2EC129BCB56BFBAC4BE75104BAE0D101F5C06045E562E1E1800037081514EE08B81131E458DB7881448DA60D938EC03082105A502BA354A5DAC08B095F5D9889B40BF4104548A382063A44065FFF6DE24894E249402899764C787B00001B702C0A412B4268BF4BAA8D4450151C582A954B40E908ADB413A23AC1AF6BDB457864C8BC42C204DDC94D9B8087F0D1C49B49722A05766182C3E0B0F0E0DB4DA1C8E348D580DD60F0F83CAFF024DB95F83A943B7398949898F17B73CD6634E8BC24D1C7AED22C57A23500A2575E01CC41AE4562118A54852183700B101842AA23A7ECD5452C087C5FF4C682839565DF9704118F5A35A70E80C2165B2C9F8DC00B032B20E8527E1243E74368B07C6077D13845016C62C8D7F2C13B103C07C3875CADE27A10DC6CE2519A0A71B53C35F3BCA745CDE8348BFF8D44141138072B941414729605101DD461430B020380E17C176AAE3C12CCA759F2B5E21607402BFAA'
	$sHexData &= '5802AC3C12623EC39DC42F41D4F683EF2C054A629584FD25930C5635F64F341273135E519B6C75CD7FC3E8D6D56C390DFFFA091D80CEFD541E2C920B88F92B58A56C1BE8A675AFF441185C9E2F9FDA8BF13D065A5862B0209188744072F03B919C83C72C9CCE0C72B8A95F568D4026803681C220B47849403BDE55E6555033E0E31526A8D48D715EC0F61B815121D43EF4FE95E6C1A91C04084645D0255D5A1883D4D82416521DE29C6416EAA976D41424324045ECC362431B41CB4B21D85848D2B2F44300AA6ED2CF4647622006F0D4E9ED60874656933C42CC1059F9F030425245C845CC62F0C65945D0680C9EC238B8E3E9707B12AC56DCC64C60B2AF43FF38410886DDEA7E82159CF6D08575C4130D12464DA988C0ED006E6C3C894E1E931880597D1E851980835DB470444D9ADAF1A0BCB2A73C97C804478AFB1FCD7804387030779C3F021697BA3770A4C82BCEB80E8D66019E2A6C0202C7784480F4C624BA152ACF663D9301EC710059AAEF7680AB8C343C08F0B8376D34A902A0E900F09EA1F2FF48AAEA8D04318BCA3BC2E800796EC301DC31499A4F4EC82399E8AAEAAAE32127B1403C99808695C4D2402CE656FBA80ACE628D5118AAD8D604010F029AD0D4C35CE03690A9F30D159418041D5A530BA9C557C86E0F5642C0F97E5C8BD028E01C2C662A1CC7C8210F5895931B092C4A'
	$sHexData &= '8B57470F10232A131581C7C850D7EC3A4CC10DC098EFEC4089378385705FE5D84FDEDC030E1D46201606FD395C50E8DD9F573648F1C22078F0C2CE8523D6088487FB4CE1333FA9EE00D481C79842C7470B0C4C7E851CAB967E3480E8C70453A0002374E79C036D2426211639A23F2940A86298E0C7CE3C83BC0B8183C430E8C2081A81DE902164E281E053E085E6620436E0D080E8420DE00F76AE47B083D22AF04A044F05FC9DBF81B3C449898BC40A3901818375E42C85E7C26FC3C6833B9458950403FF01249253578E95B0C4A79E51F9A48D92C0698996F1069196B8B81D022E10CB426C89A1F84214895224CF06BF86B040BEE4503D86B4FC897D708C0CB80F0E2A2E90439E97BF4F034719199962473D4747476EB691A6564701C201E1F88D8E43462C0C0948EA83703215044B8CAEE7A7D21207BEBF8E96F9203408F223ED1941F89210CBF2C4C0ADF19C7C521FB1821B32181E844951A14A2C8CEAB74A4E541A7F57BDA4341A30A6B885FF2B5C0C76BC3BF80AB0298D77208975C86109872D1A8A066CDE653CE6040D461046080870C2F94614E3F865AD0EC70C75C018A1B1CB15B1D4271458A6D1354EC69B194C4CE438DE75C4742265A88C5FF27FE078DB6598CB52E434598C6C697C8A624275DEA48C042A4CCE3D45D0D06013DAD0F842EB656E3C57F5160C261C3314ACCBCCB097'
	$sHexData &= 'E9479D47A2CE075B25F6EF205975D689380F20CD92C6C083C6208C7C983000AFE48975C8D0350772901BB80FC29A7C827EA8AEC0268047301F57FDB4A095283F2D3EF8801626A42E5C057DD7C6A980EC4AE81BE026638B23B007C3B20B50FF26765B7E2BD76207C1FA05E77F8626B2B2068D04178BD1FCF75E4BC681FA6D318BDAC1E3056A5217013604283D3F419958A1275CB20650A6517A20B1FF6753D9185C9387F0D8C215D0B09003CF9ACC2C7D1E4A0C7E428D412066A6CBF93436437F840115AD0B83B29343252184861802133EE1A9660E565A8B0E57C26C900FE76C8B56EC210E81380D7C573AE5410C6659431815FC8FA43B893E8D4001C1E06CC7CE6134C056E68CA508D44A31371F8CFCD2007BFC028945F8024B02E6F33FB37CD9B1965A0ADE280E006A9985C14A3A4536211F7FB3773F2B4AA28DE44B02F6EB3A3A504B267910A9B37053AD417B50A0C086385C490360086DFC2504460D968E4648EB8545B50C5068BEB1C4855D0348063762198B84871A0BAE980C11112A01F45F88C121368628308D0C335E5B9D4A35345FE1BF723206D60C10AB470CC092C807B518443B9854EB035778E9E2CA93E71F7DB54FCF3BC7C0ED3579A93EF97531CD363A310C0FFF9F40AD2527A65141B54806838641A93EC7D792351D4E03FF0AC60A014555BD216AD2138584177CD8CF752198'
	$sHexData &= 'F83BD674042601F00593E056BD1920D9C6414A5E2E75E8EB4C505921E43A182CEB3E7F1B9F4DFC53042FF25E2EB9E4F206C706C6F20BC99122E95C2C41C33C91AA92800968260A238AA93BC7E02DC10086AE8B2BA6E6003C140891800238149F085634BDD1E141EB19C8798B851CDCB037D081FCF22839C8F0B71BB716D3BBDD187B2B538D1C49C8BC90474803B716B70F66983084FCD6E8867052018662D43B7868730F100188512E562BC3D13ABC10BD4A635041870178078B936E7C700F661745AFBA7180CD57C0184C506F1B8391A3F82E52B55A4EAE6FF0FC3C542DBA713B573F20879203B8F4B8EF85BC485EB8EFB8EF0899AAE4B8E8FC889223B9FCFC86D21964FCF42A343964A83F03F8D6D26AB70F3E91BC55CDA684CABAD358413ED0CA2B95ABB8391E41D0FEDEF2CDBACE118BCB3BC3752B6BD92C79218FA839BACEBAC7585A7D026B55F82C3A11070B585E3E10120235760A18040B8A202C9C5156150170180014A03F4873A85A77A52CF8735576312CFE502977472595B9A51F470F5A723561A72D425075AF716BC8DB036F14800B217FAA992AB6BF8EC856B86766008B33B34355606105645A3D91663EE00C0CF2BC6F3C08790D7095F0B837F3BC6AFC55EF2D732BEA3C89C1E70285FFF97C6442BDFF47233BC7BC6A0232C909BC63578155D30ADE800A0C8E5C3253C0053640'
	$sHexData &= '456295F406D7FC0433C914D3140B9A5268E92AB7EC80A2186882EF2E359DCE83E6F4333FF8FA30209434F4379F9B72C2620227132895C1B5B214DEC784DFD2102ED1FA3DA19B90537A2814BB41A43B5BB45500154DBE5A411810C03961F8453BB61BD2ACBC51F8BE5539770EE809D92B42020AC821991B5F3ABE553A90404EBE4E0E5D8A66841E0E401CD88F3C4F3966016B043E0372A31D425B4EFC2E405D812A0AC3DFE80083B9810DF577D6C1D065DF22EC2DB7C3320E6E5605F4502B354DFBAED0C30768F6745C376A198BD8973D5D1007F375F2514A3518C30C39C67B4404076402C697E1E269C937F723549C1B37718F8CDECE3B61D58DEE15189DC200FF20FF18235510128483F89B2204D05644C25BFD59903BF175124E9A081F8CAD5CF1BF5A0C534B0257130B5A9B17040808AC762974134B8BFA11A56B96550417B50AD0AB09A752367604EBC6BD8F12FECA0E5B5F58085EA1D5D6DB9E201C30C5259FD6D60A48E68EE4C6A7879B3FC037A0445EA60EA19BFCEC14EE81DC0F5316C0D1EE74C75730AC9DC032920E3BDA1E812EA48ADAEB0CC9C0328911B6423AFFF81242C8C02B5EC87050A8BA6C5144863805123C1B1D34C920023D60F2187B1A64266427F89247AC92032D407447A61A18001F108FD94C3C46EBC437C158C80E2FCBB622B62B27781015C4CA0745A14830E1771E'
	$sHexData &= '3CB1E82434CDC153104177BE0E23AE8321231BC153B860C44974771308124C8249101C8B849720FCB2E38ED6DBC0F83398008346BD24F30357A145C2921965425CC28A4F817209E1127F79600D3D860356CDFF6855E26040578BC2BB8B7B483686260B7DF4C97109167A5D43076462280BC7FFEB2EE11C9223C2C071E866838D0A5E73143A03C6C217056FE60C0FA652765CC145205C5FDE63C4562B97F8C985EA125E8041C647B8845B9BB356AFA643405831B471404C01AB5F19306B914C2BB68463A9405FD9341783A5391F0EC190693D6D259808D23853C10ADD7F5222B980C0172A60214F417A18B1310819807F80C2B920B3306F8B32B40E052C285F57DA3E02C8019C9F34E9A4E5221EA5E01A714CE9F7A4A3FE978135AD9BBF04511801803F78994232A0C7F8C6485A482A0A28CC49C7DAD476F4CB0742D90939813492D9C643FE2003C8C602BBD80F42D88D4321C9C843C6430BB8F04067FC585F145E4035F120BA044430CC4DD2EFB45076748B1F53991080C2B02D02127DFC2A068582114059AC90A729F9AD3503C266890C4653EDA98C7CF2DA8D43DA5153C7305672B12157E31392F500D0C413534E86A511324CE1108452E5089FFF57A1A0F8014EC786CC568D720146E441010CCBC5C75A9442F2A2C78B000EC429900F3F8185D2A4A85685FCAB05C030F441880417C6441701'
	$sHexData &= '9C03A9029D1B5E5F8B18D88153A78A026C36BB62C3880CA35E58579521C934A47F0CF20E6C38AA8553C8E39C287910C8B4C8E824577281105BCC01B7858851207AE00681B04F2063B0B64A01C8DDE48222CEC3C72B515023327018BD070163180E87C303823A408105D5774330D9BAE84A8F35184A0825F40508D29BCF24F1A8354B08BF846F09F69C08C2E36905C64CF277184743788042DF8B8EC07428E03075CB322AEC4A1E11A305CB3769330472CB411B6690A390A942FF51011AA507F4192C0986C31BC38B898584E985080812C41FB2821D409D46A92250359F863335D9ACE9697C4D00C9F0CC393222F1BBE3C00C42F4E10B80B98EA194C62D0F19A0DFB0CD0F9213442565CD145233909CCD1E179023AA58555FB9080C9217BFB0C95D230D5F48443F303232241322404459401A71300B863ECB9C3F4E2CF8148B08B78026289E10AAB0C8253886C50504D0E4D068564303EB1260BE3BC0937C4D5F18208B492C50413A9B7D258F4A1838823F58356A1318249AE195C492307D8D5FAD4860413E8BEC18A4F921018B4E2C08C7176CAB5337FFBB08324158905137BF047E5093431F08FF0430A8206C244CBF4AA80E361FDC5E53A1D3504CBF92B0021BF9730CE42357202F182AD55F3B10533F581185A71F1788C3335344039FFFF0A21A93400897C3E0B17004FFF1589AB42C784150'
	$sHexData &= 'B8C563006708DB0CD050996ADA4A080A0C0C73A9AD66639F9F2B25D8629B74F9D1686376B0D6B599BFD0291420763E8215EC02B805EA4EF1AC35DA112CCB8CC35F1F3B124C6A8AD8105D6B5608A00244B82C5E90ED80152D990EFB0F09A1825F676A2550A908CE36CA8B697C863E063BC7DED19A1C05F0E9113FD1BA33367092838C23D1BA746E20DA04052B4BCA03B559474E4D3C27990108C854852F0136B2B03689330C788001D040F833F88E9A75A0B000F4965941084BA3CF84383DA6704CCED6CD074284FD2154FAF3AB4DA3E0511A8CFE25F0812653AE60F2BE2E574A048C002EE00108845F0C01721C853B083E0873139F1AEE09B1728FD392030EDFCB899E1411A88670603D0A57D39791C80BF93111DBEB3FD397121C044ED38B77246E920E35105DFCD18502C428A4BB608406FA86D266EE0A76FA7D2BF18D3C888162E1B6B838882BF8EE750580A5351B1A28EDBD53022E783BFE77425B56B8912924744B526DF84D6A07704A04E00206C74CEBF887BD85576D69792453EE0D855E66E3C11CB553F5AD026882EF5CB203C303FCBD081185DDBBC75305295D5C2686832C343040C09A6511D105F792A3C521013561051630338DA05757912A0EEE2CB9BF901CE4132055555505D5722279208794D577AE9017F2D577D56B903908B9428C8C902B992A8C8C28E492838C90C20AE464'
	$sHexData &= '908C9800B9C00535294F29105A456408304B5FA00E787104BC2F5D11F7853090BA65BF11689056A738921283A35EB061C9B196D9941FC80461C8703F6D9F30214425F20ED73CEC12FA4E6A406A125D26098060C7067F2E428B85F499016BFCF68A94038A584E4446703A0CD85089968960F42561CD10FD1244DB44070951AC1A70224DFF420603012269FD81D536A4C833C029D29C2092FF9A5050066AF5448B8EACC78D7886461C064B78A42622C55FAD00721419BF53BB119889052DE093F0C6746125D58481BD4D93CBFFA10D34689D65D79C2A319099B5B8A5909AACB1173F683230D224BC9871961C37B44FC0046C5516DF998CAA3BB462CE649C5FD7A417317F98E08DDAC108E4860487C45DC57010C1F5C8D1F0A89A587F58A8517284361F18B8260E85324340DC348C0212930C85DFADD148886AD58D4B2C1043245349A95A28284284FD19064CE0578D430C5A8083C48EB470DF8184747E28F19C6225CDCF26B94B1C02193830CC03A84BBA4069E01C10052D7364084C195EAE44A8084A779D7300AB1B57DFC2959076421DE44A12825A68AEF09037A420ECEEF80EE80C5AC053617D7A3F8D343F804F7B81528C2C03C1500A58FF55FFB104934094F7DF1BFF6B4DFC50D217408C0203FE6BD79EED86BF4E1A8904198B441AE84419040E130808A86D9EF71E0C0E0C100B108814DFC4'
	$sHexData &= 'AA4A8EC1BA03CB0B53C64060E2520ED8066AA1F6939E2E1275438DC22E10CE753E746BD44BEAB5D2F0CB74D4DC1AD8344FF384D20CDC08E00C838A3CF3E4A8E8A21E75B7D3A1F07D579E0CE04749DD62D18E03F356068307C676873A7172A9741B67B6AE744141C13BA5524141A4A343C08104FE397DF87CB08EBC83C198B83630027C42C9E572B94242424203B02A281F3F050903305DF0E9C0A753C0F75783F8284223F5491CDE157E400B13816692CA580E28E0C1670290C75008FFD380EF8B9A810ADCD7BAF3CC1A762E988E283C37255724389847E6393F2F01ABDD573A6D0A775736ECF8575633092C0930D6F661EAD8C6569B2150D6981735C196316108C492147C020A9E505746F4B521E8A23ECF0708A1AB396CC108072ADC5BC76E976C2CFC56FCCF565750810AD8042055E225EC5070B1EB2EB60C31565329B1DA82771E62CB835343F8A646435F42AA83F119C25C9800676385349A5E2C9E60E8F826B2C6732A8BD836D440DB6B6069D3D7ECF4561925AE0456003ADE72DB2045541585C6D954536DD9DD613717534C2A1B0CC32C780AE2B13E72E0A88C5CFB09C1C204D8CCD9AC8131507848BF5BE45B8CD5B0C126FE1722220C1A25A8FC3A0E866A30D012B1DEE8A2D8BA5BFC0BAA7D0C72C68CA49C8F4DE05AFCD71CC873586D281CEE44708D471B08233F7DF0FFD3206A1905'
	$sHexData &= '13741A7C50C05114293092D0C3F0CEB5E91BDEF804EC7274A4255D43CAB07DE8B6AC61F07D64C17523E674701EA6D36909BA812BA6BE48C044E0816E12EB3C184BC575DFB450752AC1C65628C888CE4CC483EB32724C050BD4D34322E912E3471B0C9ED818D0AA943945D08A2F81E0A88958047D908432C5DF4612EBA490BDA52C1F2C5CED9F41D028C00DDAE14F8AA1AA78189EBFC38C5E450726DFE0F515258748F4EC921350851FE2153454C921F71C14EBD1C28325A4C604A9F0AADAFE6DF82EB8694CC226C0BEF17D680A4CC0551F3C9C8A204EAEB034BA2240C014FE150A7A0428D81BC0C1E6059A8804D4A640E2F8C382E00148C1E760E34CB0B67C041E5A042445441FAB95F54BC689441E1C60823CE1CADC7CA5FDEC9633FA2E97DD3908AAA49690787772F704DF85E403C24D28B80A02DE19FC85D8FE7D47304F18011EDE7E30E1A521DA90D853E3FCA000F7268B03973904B7F17A1B148B431C9877B4BA7B82BEC0184DF377EDDB78AC03C7555F5019268C09979F1445DE840B0593E563300C3070CE1EE40035A1916A720648F8970E32247FF94C8B45F8F8F94C8209DF4A6DF09C31A193609A1DF445AE82036D4056403D1E26A4367BD69D56559BAC4CF27CF82B7DF0F8F8F94C5B09B8952CB25BE40013569BF94C4A4E218FF94CF94C0891DA418353DF03B99AB09F2088640290'
	$sHexData &= 'E020C9200F6420F94C20935C2447E020F995CAA9904CF94C20203FC92123E020F94C83EB20831CE421F94CE6332014134633399F4F699DD2547F4F0A63DC4204E9178CE7096E042BAC1414644F18A311A73EC2607507B483C1208A86B2953C69BAB1C2CCFF984C74A4B9DAF60B705C252C908B798EB6C228A3C011F14910EC7C7D01B24EE2D3C44A4E08E818A5B56060DEA340241CB0D0CF4C412D5C2DBF5EA6204124BFE0106F047E098A10C35F1DF0F589C02457C8474428B41EA10F135605395402775E5F60BF97AC0D2E122B58A609247C1B022B59C3FF04906B2BE999243F187927042C5F58E517B4420634AB36D0437A8FBCBBE02A286045776452D00A0FB122804F77A2814726CD5682DB06B609E80059848FA4975C013880BF8D6D35889BFBEA1ED800A6082187C9C0740D4CA6F510F017F3EB2CEB26C545682082C91B5E842E94D14329806A34E4446C09BD01B4149CE853E4F376831195B4C97423120479DE9250D539C03B45C4C0450C6339C07B1AB0102CC427EAB92D0723D25EFAA2A3C0290C84A88603E48754455FFF59251CE40A3924F18EACC2920BA975D0AE18CC4C2DE4FF476A85644047863DB2D6D07569A0094406C80D26A485CE22860DB0453B0DACE473FBFCEBC6E98426471F3CFFB5042314D04AF719B6AA55799F3F106B263125510317906C73C9E66220163F1013'
	$sHexData &= '853B40AD0FBDC896159EC94401A5C7D3E0982701A104F80B9C7062A4E841CCD8E0EF4E04805D8D4901D3E78D4B6C63A32E7E3FC7D0CC441C8D5F7C897B1C8943185E26F05A2B263BC64EEDF80E06562C3FE081F2AB3EF7FBE50969FA500933FA0A0A69FF1F170B26F0DD4169D70F8B7B1823FA1700C185728B1CFAE476316E10FE3BDE75078902D4EB6D8B9478C87A41784A7A08752F4A615D10C10E1C5826F7EE012AB20B98891740547E22B833B7ECF8895914027E73333BDACAB77DE71A3874743D1675F46A895D5D3B08784A13E8C45A58003CBCAF225165788C8B2CC142AD839306A22CF921BC62EF70F5E7EBC5E1420E115F97041870798681FFBEFDE88BC657EA7087D4FFF850BC39B9745800FFE63646FA538AA5734C36368BFE06A0F84AD1F01102608748B81F0A9E51100123BE68625502D100FA94C411E0EE3B710DE0D2033F6A741C80BFE20A12F91A0C104233CB69D93E19A73541D78DE8CBC2FDCD84C6490C23D38D0CD188E1DD0CA0440A787104E9D40716009AF786837BD4534A12E0DDF632EEF7B4E001BCB26E2025F03941FE754F11ABAF44691351127CA39B0A8935D10BE2CE4968215DA85E093B242C3EB030890AD631979A06B5B4FC7AD25572BAA8BFCBF7C61A391A744DBE5B0091E6080F42456E69F06DB90F0818065F8700C217D24D0068BA6B34153D99451821A0'
	$sHexData &= '1502FE56CE3AC148F8702E19F489300012E30676AAA83C1A51F73BF8016B26D931EEAC561B69002602E2EBCE5FF9301398D1ED70323E44224CC818FC47965330210608BD4091848A57CD36C02384F9124D4448BF7B0348ECFC26565D8907311B05AC198DDFB73BD4407337883E807F42754256C0030E5A357E2C8356A1B6170F6A9C2C4824CB8E29C6BE0335108F81ABB924BF85C042601FAB8440B775B061906F2210DFFC60100114170CAC3FC00239E4599738A5ADF4C2DA75CCA7F1E984C061B9E809223C6D5DB9794AFC83207401FE05521EC17E63EB03B07DC81CAEBA02096AB6E04514C2A01EE5D490E206FBD60901E30F1D29326120CEF50423340382428C04573ABD11B8B8144A9158D70D08FB45BCA1212BB5C12183C01CB4011108727AA01E7E138BF8FD5BC0005DB0CC960FEB379870D84A6DDB450DC1600B4AD0B85FB8594A4F038D7DD045740683C7168BD82D2A88B82788BC583BCF0278A174895052548BB22B71561C1175F685FB4DD470F30E9A6AD6A686950830BC2B063473583028940E0A092A1BD8646A81C15F104308373656B993C415BDE0E8029B244B933C2C744606ED6C2B5B543A475708B45D98B8AC85D6017E571C0CF7E1800B066908400608091F40584718F88955C85408A1E2194DD05E307ECA5607EADA4F04D07D84D061D4F3C009E04058CAC8BFB85ABFCC'
	$sHexData &= '319BF0D11417924083AFF6BBA9549AAC02DC4242421ED2BA52427214564A7DDC8084B2B26FD9C199362658E61078046189ECB5C88DE459761600DD273C59E351F807C904D41A83E41A8CC448080B0A985687B79F401E01F2F539F84B1E841CA09AF899C00206AC42FF9043809CF769FA7B0721AF22985C50621017242CDF8D1098B8D9E8768D06742A0C71C6E0B1BF255339ECF0408D728A213443143B5B64C04FBD431CCC02568B335A061456033DB64158600C780A6AAE39696241E6698930C50821E482FC9041DE045F59DD14A1C6652D7F37A6380C33E865461B42924B462C4D535AC5E014E0E4850AB721C7759900B53B756640484797721A77A80404E844405D431A870D7BD049FF6E1418B8D6FE7FD64E7142CA35190C0060935693757FC0274865500350F0D3EACAA1D88F46200D0C9480F9BCAEE578A5674638B6BC48CC8642104F300D34976D4C8C5460282C0258792108235CA360AC47A3196474352D4D538A2E8814668C78E264A4328E905EC6A14206AA9F1C56901CC57F7D99148007C22E7FE09340082910F801C855B14945A128232323CD3840202444232323231C484C50131A162754A9FC245CF438C57895467C877E888081940496BF60282CE0347005F9038B09334510C9402D68DAFCF0C069203B043A00840B78E41D5C782A2026ED65AEFC0D5B4F82B0331B05846AFD'
	$sHexData &= '1F536524901C3F24C90B60259E1CE054C6C020BF742C95B1000E5AFE56D7021E15C07F9910E19E277861DA0E14144F0755C691BFEA50DA02C897F635C8E8011CBFDE15689F800258E60812B88C857CE2808C1F34B504F298C4DF3C0481B4385F2B07C14532353818DD456022485673B05CC883BE19988D0D0E8D55BBFF635C7702790D3F18E0522884906B48C8A31C3F2890018E23AC3F28080E21871640B92C9FBBAA24855FC14FAD0B13991F7341F186ADE42AFADFC9EC71BABCAEBE970B0B3939B43F98C4EC2B644F18DF5A1D78C0C7C4B91C19C8FCE88B4EC306197F450B8C7903C8D4DF2856982808669F0590835685AC1FC83058385D9F4096E81017C13C584221EC20E1706673146C600537FF41B8F6DDDC0256751960880F661A042C22DFBDB28C82107D65882B581B0344074A70C493FAD46A1CE6C7C6425B58988B400CB4070A0040CA265CC4288C27EE6D533781C614110F3C58F875DC17EC9E006B55802475DF97206C038AD83F4BF002F4BD3C0074173E50108EF1006D883BC86DC76EE058FA3AE053E8AC1857D0890E32B44F04ABD25B4D16B6B890035424E40AA61837FF0F2FE8E508F72B968D9BC3EBB73C5010C480E27E38DC4AA69B05EBDF72A8A2B89DBF97F0009007040534217910F2057B05340239AA90DD807900C8030714075B11923C0807141FA87C0B58D01020E4'
	$sHexData &= '6120445FB7B09204BEB8FCF7A522328EE51F1AE3E004C2C1DF5C0270E20FCDE48B7E0803548630028E89894187310CBF420EAA605E3F101305C3A99AF87F9A819C2890F89B249180E4449B247AC2418D9E06E79CB747275E0C0CDF9B40372C0978A3181F2C00906B010EBD3FECB5C9B785FF4607AC7F7B482B79078C5F1A50C2032120A6EAECBE954CCDDF807E13B02057F248E96CF05385745E11E07F9883DBF7212B49FCAA6AAC0F1CEBA4327669516A5CF8C758B0C0A0178D15CC43A1BD4B196A3D1380C873735B0198740F0804742D2D740500860470257523BA4406A4AD52FF0168040827E11E60A8C9E653150925A1604E2B9C0C6181E19E3C0C08994A14E2BE0BD21E70E02775181E01999659121072B73087E0240482B319B65DDA90961C7D1E7413DF54FF0C76C23962C8BF13A8B9E9FD025A670314A04C1C13DFC1C34B949A40750F60E046785757DA0E82AE33C510C48774A481619DA08B0A87571117EBD06CD781FAEC30A0159BD8752A6B576386FD041FC714F1285333DB535970F4208417FED800872D2630375667BEB7E02D037F2ED67521C8849DE34158D10F6434BE1A84188A58E8C1E804A800BE9B0684D6885DF85E0D1B6C28C05BAD0EB380B24A4343E20006A9C8D17090465774527A564A238236D80765CD297650009C015D0DC841627C228CE11DECD8AD0C6D0D1D0E'
	$sHexData &= 'C9300B8833D208646F4347468FB80224FC80B0088AD3800536F07A80B4E3FEFD0464861484955F9A808FD60A2084D244C38B3E020AE602EE6A575852021117058581DD38C4530C5CBA2482A8B440194D604F0A045AA888FDDCEE6F4B270F8454DBF6C32A1110EE5189BB91004A0AF6C19044399C870728C120C1F3C166238466697594F4AED2203ED08DF150834C719841780002430D35A634C350938BB78B7259F9FD558BB57C16898D548D74502C85CFE57858EB2124A5FDBA2924854B45F010584C072FCDEC50D88AFFD2061E3C0C5E000566753F7697F0D95AE3D5A2770C0B73F01A656EB56D2C240B7130741E3045571363869609C20A9770FB21071B78188BC1FE9DF58686F0DEC022E015506896C9C0B90D3A309848F076FF492F05340F1F8A8BF8E998D12673484383E6286AB09C2622B6E50DFF497ADC82464B154455A6C42200060907130CA8D30722521E33F7C2C37478483BADBA08DC488E0974B84BC1826786ED66EB0771E824CD207C1874286A18325801E0C368DA8BCEECE7DE4700A06D08E94F0C10181ED2C04F6F036A325F5697C768A002BCF599C9A29B8C9E135633F677CA566A07815AB8655D61665444D24358642E999879468446A3A93E1E01860D343420028650B5CB44E69EDB502CCB2713A7CF85F06A05580E3439BB218D2D85021387550000890437DC70D7D868'
	$sHexData &= '2255C4D9B794C4E942738A7C14F26AEE46DB8004FC57C710139D8CE89AAE0FEA0D33C964CCD9597853E07615210E0831740916D376C16BF07157990262CE0C5E3BC629072DEC6384070E85D45653F856908A57876ECC9DD8EB6B04EC71719E5A584454DA1E6CB6225A5709D4B976BF6DF08219EB3CBE751E834DF4014E5EB49B776353A2FEAA5BEB1ABB880BECFBF057EB491AEBE8CE96884DCC1A3045359A5753C610382B082720F03A8DDCDCDCBC08193274140B740F3B740A3040DADC2674056A75528A3653D1BE714588D6C3C042622DA7D08BC6C4C6FCDC4C02FF8034DDA3596C64F87E44ED602961062B51849A8E847810D9288552C2CDDB2F6E07B8090458EB2683FE1F6A023462F10BDE68017C6C65304B0B531FAC6B2EC0811E49BBFCEE5369B104B45C2A687E1042F329A846B6AA572BC44636BF13D0D6C9F20267F7343825A40B619C88266BA20A56695E70618D389745638C7ED67C5827A41BCA128B0F22312F436BB71484EB0E0A14BE0E09EA46B020A81676E806BC143EE8C3337D95AF1F021480190A0475D3623F5B82E99CEB09FA835959FBEC4147B7A1283DA02609709F25C815A19B4274BFE6580889B7C9C3A574FDCB5CFB41BF505DC31BB860B18075BF1014486954602DC8BC1014C6F69F4EC002C60431012275F227006FA718570F28CA028A608D886F19233CD2E2BF'
	$sHexData &= '433B178C7B669012660F3817C8759078FFB52C84207CEF83E11E03D1361853C085EF1FB6F7C1963CC6E84908BF40EEB0011261C945057FE042094B16143801DF580712701088F905100670CC746A1097859D4C146D16D0E83A50487CC1E90580E1DAC126736CA9A4BD611DBC120DB4B01E0027BD15A61E3683136B98AAAF9AEF9E1A922E1ACC1066173F64E0406E5601754D3FB3599299E3E01B504F4C4B73DA6D1BA6A31C0C245BEDF362D5188CEB6FA950D6B735043BCFBA353476D28019AE9F32F04A069F5012AE7057996548CE75E913F186A16C4AFE7A77DB0721494C04484F1180BC493812F8689407DCE28545400E258368AE12DE3381DF8B957D7CB7144AD7E2B6369E881074E6526E10A194F1F4EBC35F408318C40214CECA0D25B082C2576628386C91420225011839FA2B05FC44BC17330108BFCFDFFECDF6C5F96EC10661C070C000C4E37D18D03C1F00C5ED753C3E00850093754A63A5EA282436EC9C1C5ABEDA12FE028A382BCA3E09C4E2758C8DED6DFB4875C206DBC1591D031AF877D202C10BE60C663938B843B325BC857D34F3304C2DC1765F03C12135B0F0240DBE20A23CB0C42450384912687C7B6410062BDF7F7526C17E8ABECB094610EAEB969B863A64E0CCCC9F68A5A4108859BCF1C8C6A30856664DBC2D007F10CEA483E00F59FB0441F2D003F146A8B4603C1E'
	$sHexData &= '0CB004E45907E30F4C02198D5052AC4D46357959D84DEAE62903F16A111C75B8741B2C0B0C380DA056BA08E17E4B03013A61C8717222341A19DA417C75A85877270A3888B1CDBCD1EEA224D0EEDB72206402EB312B751CC631E560037914FCDB45DF99C2CC28C2C4BD0F83A5C16820C06A75EC9402E106A91CE7E4B003FAB67CC3290879B08D65C22DA1C0CED15459D0EE0D984604C89265D0C1C1E0C14F503498B895DF7E6D87988BD01C1EE460C33172D0397DAC746746E3B51A2009398A3A3564E06E4DCB547CCF75E6C2BED336ACDA0B30E3427ED01DBEF0355A0FBDC0320A631FABE0830702EB97FD03C6853C03C30F6B745A631F562E186E65970223FC61E0103BB843E052D8565B179358B8208CE9C384DA8A00DF245880B0A64C5A4A5B121A84AA947B6FDB93D98F1C4C1EE0067F65DEFE08725354CF7B7A1A2D4EA00E3580FEF4B7C79B567D335C773E52F1EFC9C5D5DBC33EFD77DB6076D10672D3053436CCC5E5DFC69E47C08FC6379D9682FC50C035FF836CF12BEA021BDB04BF83E607311BF09C166260C0F4F05098BBCBDB1DD1B55DF0E405FDB4C4D69D61B0BD0CBE3D80C716BE903463CED398D6FF561ADED0F7D24823D0743042F7BD74BDCAB81F342BC1BE1DC6B68AAB71742BFA461F29741B811D76473FB8F096710C45EC1D82C201457ABF64017E03368242FE02CF730D'
	$sHexData &= '803C31FA3564D18802EB077FC2550756DF0CA351C0C2123846402180BF3A147752C5E1EFDB06331B137CC23E16C3CAAB5C5D4ACEB403C486E1181FDE0104DADACA06C02A32D0C3B7A100C8F34EF5EB6475CF7E520756C7C34507F069E31D60C61298C6FBF0BC2BD9FF84EE45B6D7C011C3CE20ABA008301D800EA09234D8BDA432527916F840F020F6CE4865D000C8FF8237F80793DC81E37922C1015E11EF53DE6EC2020BCE0330C7530113DBE8C173EB3B1D03D8187518741016D1C001E55F101574C32F6C74309430762B7817FE03FDF908770FB9B23BC81BC0F7D8031ADC5BB307A930400608480F46C02785CD44DA55B0161050B951D8D80FAE7605B908712C9F59E0401EB8005325D13A5238C452071BB839FB24142BF92718E12FF083FA2008206F5250BE0A1F523D8BEEF2D9F58B06858D664BE7A503F95C36B0B946C84DA0014E50F0C0C78FBFED74B7F8F575010181E75500B31F4AB88B29D38850FF85F94CA7E02258D703CA10502016E9DB02FDB27206780218FC5401EB32360E1743AA3E242550512480E3DA7BE17D9D8B676689ECEFF104BB3DF51706E3203B630FBBD72841EB9A628D5FA0EE5CE9EC65782B1C3B840B1EACCEED0D12230A43EF09E88EC03E99F5BB75A8139C8106C3F6C97468EDE0FD152D2B4B148911BAC79A50C176822BC1774B1703C8CFDBE802D260AFA0'
	$sHexData &= 'A9EF45000767D6AA8FEC207B70905087C7BB92A7852FC16ABA68A1CBD722DCC304D7E9F44D568B821974012F850F93103B84407C0203F7CD45A58540DBD1C0E455B58A251228B93EDEFA1D1C92ED2153FFBB03C140B97815BE87264E4000C14FF0172C6B8D72E08D5020BD83EE04721419D71416BE0A3B08751312E40426FFADFDFB73EF83FEFC2C746D8A0A3A30271AFD74628A4A013A48016E6E9ADB751A18FE74550202750DFF7448321CBC69030374408B3B09C322C7394B95540FF08074086E0BDF20B0308E0B8183C5F7CE0F862A9CE9E914AA0DE7B2C1CD033B9B85234224FF48A05B07813F4B24D404B3D63884483A46D327042CC223DFD401785F10F8DE07E7177C29A5925683322BCE9ED2BB77A4041EDC9588C1E80512D1A42ACE1BEE2A16982A2221ECC057B7B7A0D1111E126D4D02125FFB1B07239954107F995FEF886776FAAF22D98BC25F21F129E0D1E8529CD80C825518F14DDC57D1A551688483EAD6C0EB1A11263FD10D366CFF2250C610EB0C8D344E3B6E0F8E4A21C488683460B4F0E0BD2435F6D31EF751F8AA0040C8B4D1E91C7D8D5D1ED8B93428A5C0CE471422D8B0C10D5A080AEDA0E04E83D80F46A28163F04A3347CE9DDB47DCD090FE37D0FF0868B032D1598E9C0D7531BE1BCA7688520F5770C8B6AA4738A6027DA84E10CCA61C156ACE32EFA6A663E5E816'
	$sHexData &= '3B3AD8A0C29AE3A395EFFFFC3BC48A8A3237B48A76D21CFD411E42282620FE8334378302027515FF0318889035031732013514837F6422ECEB43786A141505520381D0E032B20FE1C19AF866390E7523319F200989444F645835D4EC41FEF162D0E2EE8EEAD04E93F075D1AEE2B211203FFFF460307428D8AE314F386CAD166468C3BB269CCB64181E3754A3C038B72F54B8410CC4526A0634E224339DF833C3650730C270E1C04ABF9013FB705DDC15DC1F177C2E5F2F9D218C412C0B3034E572B95C383C4044C82B81CB484CBD5AFA198C9252C0FE1348FA653D2C77048477D3C18880C3222AC680111800D85D5263C0BB09EB13A1EB43290C4609FFD74AC0A16B58F3488F0F608106C991A1676DD6CA083386CE8906C967F0B7A1E62582681D35C06DC40E24C41C7C3B0D215DEA1E14CAC3EC313079BEE2F4AD60F05ADD504BBD41A8B484BF89135689089DF71D4AF910008913C046F015A46B05DC44BCD9BFA357E0B521C134500FC7603F189F2A00EB264C2FEBEC2625662956C4F4B579B8DEA6DD41A9DAC4A0D14C2E7E05ECCDEB0D61334459FC0FAD00ABDC193EE66A06A781021AC656CF54088BDE48ED483C03C85A4114FCFFDF6EFA03D010066BF02803F2EB158B4A0C394D0C720AA6B6C8699CFA3945120CD01F36BA802838E79CC36D80621944F95931750EA688C6B8B2186B0109'
	$sHexData &= 'AB84FC10EB04046828604E6CCC6E8B516C0E4EF03264C3B0010661D260F73707102E16EB181F71504950BF81EDAC034E45335656C61227EA846D2E69F6C0590F952BFDF443B8F7EEF7170711EBED5813091822E0F136B0CF7519D0229CA50216F86A3C087401777F556A1C4347F0CE8A654A59595D310CB676BB03A6042D6859293400E0BE8377DC1C7421C059E90B1D8BC021A5F2051A8203D160569F3104D461C396F8190A6FEBE96B6D3DF12E5AFE552123B77C74041F4F5405746B08AED2620B260422748EC311135C34581D1009B5E9B0532BEB30F6B95CBE1B244A0B080C100FFBE0721418450101DA733680805D05C781CC13E2B00FF0DCEA198CF9B84D5A98390576BF054522F0AE5DA10C3C81B81950453E7E2ADF3C754CB90B018818753E72B96C3430E037B67D74596D8DFB1B2B27837824007C21A0B0E0EB3B042D851F3CA28138540112085210C14AB0DD0C0E8B65E84186A25504A47EF0757A474763401B098AB9AB4D6BAEA187750006380C120641D845C1FD0F08C5E1396EE58304FF2E4B6EEB0BC485C08E9D624A20AA44193F7B8D80C15B7D294DF7D8598F692DECE64828C738339042047870416D8AC81A40002C12E3910095828FE0033C835CD4774081BF0A1C888BB3107C25804FB33F42EB168B44BE7A4D9C0C285C8BC878B1852133FC2479E58B7EF7C15D824E11C0'
	$sHexData &= '105989BB6FEC16207B8BF7EBC913D7DEC5AC76AC4D57C7F08DB329087C2B04202FD883FF1E75303301F6E5D5232814B7E0DE1C375954FAFCEB1C8B837307E03D92923E8989FC53324B0744FF368E2C1CC9ED33D4338A00A57B200911064054A559891C5F0C0521673829547AF8EC812AF95A474845FA10DAD14FFD324B242C0359EE6A100F7E7AAAA5D0E3B15984E32BE9FA42493DAFB6FAB301885DE7DE4A5B2C49B146F8BB2BFBC7050A806FE4192C71139A4D755FA5FB1032AA096987B863C809C4DDEB561E4833807529AFE858203BB8CFCC542A947AA1023274FF1DBDDBE74D382BDF84DB7543A5A4778B15307076741F332871598C01DFC25F940C6A02203693920236496CE77E35DBA11C2D0F8A9038E3A429084955286A33E522076CB9CB8713F0A179FA7D201860E07048A395F646B8618353AA0B33E03A3C8380002102756F05819A4B50F9083275D695596774BAC8702C9FC4A3250984207A41F6F1ABCF2CAC949B14D850418B407EE018CB670ABAD1A46D0CE318A705320862860F39F9F1C941631C7F8A8C2DB808E61D4373FF580A83FF028791F036E8EB31085374A6EC6B0DCC2DC80ACD2DAF2D6F7900392AA60130DC9C75E038ECA327742353502F85DBBD2DA030D31CBDA8078247BB5E037548B5743509EB24184C3D5110014006E1101EA62AAAB1E3610001814DA4752135'
	$sHexData &= 'A4410E61574F836379F16ED0342B0A47D603F7CE8F393B1941F9580C395F321E84997160D72CBA9E45854A6801CB017AA191F05082516ABBB6811FE14033312C53DB8018B8D1A281DA065E59905B61606502A2C01740FF04F1696E65490A356E74656C47FB5F50DADC6A013547656E7516582CC2FF9D5A597575397825F03FFF0F3DC006ED2BB72F5C233D6006020C1C3D70153D500603280E6CC2D33C0307031A830DB96E10D424A4E033DBC6CD44C228CA78A25DBB4C08A5EC186A022F183A828E706C584D8D50A14CBB22038973DC4B7A5D8B292365027E8CECF7B7DB3389C3CC749D0288D8017C2B2DC1BE24754177708BE86ADB0EB6E0245839D67C1D501E553320815D83217DCC2C09012DE1B1B19033C3DB42D462109483E1FEC70187F02EDDA34189893B76D63EB80C3CE12964045376A916640276EF1C1863AE03427623D067E0855FAAA06A5202010AF00734943E76556226293382FAE00623CE5E4146DAC113440F2DBA0803D73CF841F6C32074795B942025534197DBBAD803D0A1DA5861969E6032056457FBD6C62B204EDABACCE423C2E26B04AA00F230B82C73CCB236C7B7407CDBB7890636BEEC84EB0B3963F7B3A315577680301025FAFE41A34CAD696631E80860DF4DBD0D46E0FA06C1EF6C008D37C0021AE706A32881CF29DAE7F85DC8F7D1238DF7D7BEEB3DF79DB2AC'
	$sHexData &= '7D3C01760F5ABF1931CAB3F10D10D6680259CD29C3CB684A2375C175F4EC47566752BBE40F36F702850C4D023136E3346F05564334163905A6A03293E05E6A59F6A6C2FB37CCDE445364FF35B5D0A73D118C6C068D2BE0AE02392E9331C4988918D8654F48EC161B0E241843DAFABF82C065B209DE26F0CA72B9106730B9FD56F658161E1D483111340EDF0AD21A19388829F09A096CC41E33D0C10210DF0D033F864EE640BBBED0B0121A6E760485AE264AA63665B5D21C37E2F7557A4F38EB0E2A0A0D1146C1E010F86B7855B28973C65F11805EC3EF907D8F0A38CB2C171D3259C3B8AC4CC1701622EFA0206E269C8308244C18EE7C0D31811B0244AABB2B643050062CB848BB093BF32D0E50E87319573372E950845052B957505072B97C8D0F580B646C702E97CBE574787C5054E572B95C9CA098E85C2E97CBE0DCD8D4C0C4E572B9E4D0CCC4D57F7CB57F3D05187C5C81F9B4C2F5A71901F20EB5754C0FAE80D0FF81A8FC83F03FA8817442A9A48254C2FC52AEBEBF227122A920742DA908042991468C7C4710081B9320A87A2F7EC6B88F1044C88BAEB8902F9E0A02056433835756537C950D253CE80BC07D1447C2AA8F452CF7D8F7C8D80041DE794B1D1C361CD2B26792181C182E3305ECAC5C4C0E7F33D292A6BB9F51D8850ED3EB4116335461BFFF7727D1EBD1D9D1EAD1D80BDB'
	$sHexData &= '75F43EF0F76482DFBFDFCE9232A4E603D1720E3B4E770872073B56760119081C884E8C86070BF59BD8E083DA005B5E5F835E558AE00B3633ED631547456539F8009928F08BC32818808205EEC60F28EB478579B7F7E50F094E2B301B241C33DB1210700D2CF8144D79A150D3C840308636C95D7EBFD3BD9E08EA9A0A0C75091E04F7E1CFBE07F632530A5708EE140313E103D3EFD5AD52B701F6C9902075163E85C0FFC4E27918C062F17D087A0C7E16C2FCFA4A29F456C099D1E03D6A9E7353D72A00802C1AC319BE5E02BEFD021C0FBAE818C1E904E00780E9BE0749FF5A780FA5C2D3E0C32AE35D28DC780856669C08BA5A802B97EFD18289B00BB4B86328A2C5650AFFB4C3CA2602BE062C909098D55520E1088D204AD4C8EC41AF8C41CEC15000025FB55A0BA9608D4B400155C9D05EB7EED40F75D0ED735B727CDFE80DBC6968139B0CE8F3B07D985C1B5865F451DC4E32B4279F01509C59F8C20032ED8A4D45349FEC255D88A0FC5F9F6025970CB15FE48CDA4072825F44DCB85F2F19481E001A1CE46A2863949049E8F9532CAD13447D4C8CF01D181D9303055DC3104592892D03DC3FF88826282EEC7FE81F07721A0EE01FC354C830200D3E198CCFDCD880143C071693C4E4E20DB976C0A3D45B4ED110C11E0FDC3BC420126EC1487FDA15ADF66062D1EBCF25E1123BC4A8FF7EEDB7'
	$sHexData &= '023EEAE0D13A1C8365E0FE2EDC4BF3693F1BBCC3C0D4262CAC07042715A2E0D861A8ADE99263E1F8E05F81D371BB7DBC9DEC41019D74407032768FB8D72690D39FC574F045C8F5867569DF14CFE0216F55F0BE490C5DC2FC8CCD93862EE43B34E4FE73F93EB7329FC331024D800C6DC8FDC3D0E0BE3839C6308B7F05EC0EB57C4F24BA83152C7C6F79C50BC664ED7FD0B58B2093D0798F83355C72D0A09BDF039EE649AC3C24ACB4E9C3AAC0870EBC0B917654E044C6E41FD07DA201AC018A98D539E93099C73FF0F05C1B4226F0F0854C42AE76B1FFDF6872C0A663885FD85F78E4862513B4D3B0C901A603ACFFC7E047810376CC6FC8DFC82BD99590144E2A0C8ED1756ACFC8486323A4AD8A21B87D7E74393969230E88B8A0693E878A8DB314384C30869514046AD0AA10E4809D20F1A0FB4D4F0405D33EBB83A51D65031D943CA8F65DC17A2FC3CF1970E0C865D89140584C6D01F4AEAA05A298234D8A01C3910402870C2918C8FCEF0F17050632A07F8820ACE421C94059B09A97437522ACFDB625D72EF08F847E6CFFB3D2544D74A4083FDF5D82A060A4F72484D5620D465582B15BDFC09A93542C0E0C40460CC951385D8D2F15300D610A721043E10179810CFCCA4C8D5C2C69EE1CFE1C88C3904B8EE94020AC2493B449FE14E4CC452906B8FF4938B01C0C2790FECA98FF0832C9DB8E'
	$sHexData &= '598C8FCAD8F0919521089120418406061990DF28503882F3D96DE9A76EC90BE4283FD4CB2027457F61BF463C4ECCA8EA801D20A493BCAD026CABFF9FDF8BBC405AA368CC2C7F60030D586F80892EC68A31F3FFB880DDE652C3180F7A82E0982AB1EE0BB4E576D2861CCC9C8E08C8AF13B0E5E4CCCC9FE0C106DB25D08FEF48BC1F420E2043CCCD245E55AA46BF6DCD586841034BEE97AE133E5C45BC4D04154C0C4E51E313CEEC8FCD840AA301013F4F551B8C85EC980FB12B80DAD4167B802C7A5179F963DD446383A540F4FE8DAF3136398C13D5CC69321405698E2CB5F020240F5645C097CD43162E5CFE44DE5CFE58435E322010F0FE171CF282BC9D60FEA433D0B343B697FD024C1408FEE42B0110849DB4BC50D4487711C887C8CA01456604E0FD13C8A120C9D0FDCE3949C39248B25845BD434E72A4585CFF68FF7A011396D4CDCEE803445744DFC435CF7C7A20B13465E9014654A6D974C25EA843BCDC44C0C2C041FCCBA71A832EA75D4BBB8CB47BB7087B142570247406E428988D60F4E4C1E40A27601FE86D90A719E4E8B8BCA92527695A00B4D00CC37436358B2B254BFEE8FEC3474381A521FFC920FEA43539C9D8FDB8FD741CB83C6BE9C82D706EB527D3A99C5CFC3B2A76044BC56751E02BA06D8D518ECB414CDFA519023E4D8CE9A64C6B6B6BCBE9512B1572EC155E41DF68'
	$sHexData &= 'FE57C0CA60F81AD768FDEDDA87AFBD48BC83F9FDBFC3D6E01164DA241CABF8D921DB52F0C03ABCFE34516E0350DA3FAC2AECDEE2D88B96FC2F44FC576C653AC2A6838C832D045CB23964495AFB70FD28B886249708419869873CE82D78F8901527996492A8C0D8E8F72639E42108F92038C07192495068F9F34D5B9B36802B989941E0729249A657B0C8F8FE6B494E4B41B0FB6D4E4BF290FBF0F82B18FA996492432030386492492648506092492699687880492699649098A8434E72C8C0FAD828FF879C6492F01008FBE0931CF290FE00F8B0184E32C924803038FC8A6A924348009980BC383073E0F78C546079D040FEB04C9E4E1AD0B0DEA4D46972716013CDBB988A458157D40003F04B8EE092E8FCE13024D30690FC781590937C8BEBA8FCD7AFD860933CE42408FD40FEFF9B649263412058382BC924934C70508824934C3268A080934C32C9B898D0B04E32C924E8C800FF99649243E018F872C849263010FE486145934C2860BF4086428467C3FB2E1D0C39D456F7495E204347F7B8D5A8907C994D7FE4013F1C25970CC8BCFCD4462D92A23C9C1F4CC8949CD600BCA0C9C9B14F950FA8B4C8D4D6A57FEF8481A82107609C00B9A7CE25110144DC398D881CD70C3F5004190AAE75E0BF11D823146A00D7850E3B9BA47E7C94152BDC2B0168C8100420BEBEA455A2B2FDF921A1E23D'
	$sHexData &= 'FB4E46EC1583C1386B81569213C86CFDD7701B0ACEBE5FD5FC0F988C8DD46D3C6B545CD87B9EE4B9920BD908CCECD8C460882BE0617D580E8E2E6B74DFC4C0085841275548AE0DB193C0FFA842AC76C6BC8F86384E310183D9109FED9203165BE044EB57C87068AB46F4D4C0926B7B709FF3B96DD20CC50FD8260C90570C4BCC01D9A401239D34BFF0C828AE0D706FC479E8FFBC28060EEC21DA1CB8200292FDBC162919E05FDA5496E6E7026D53848365ECE41B58060A01749B0E908B7FF0DC1F00D60E90DCD8AD698F466DC99FD07C0EBC530C0A524C6DDBB06D469210DB0F437900273DDBB40F5C05AB89BFCC59018880300F8F38F286C9DBE83FE94CC9CC2D9964A8EFCCDCE23A7B51DFE92C0FC94CDFF2A795BCDC50DF01510C9B9C3A567FDC847F2407C854E4B06C33F0728B8E6CEF1FA70181EFD79C0E81C1741BBEB0F1B23B0F2117AC152CAF3891DCDC3F1AF08AE8E91C9DDD301FD5B0AA142D1F517DAEE4DD5CC4EC5CEF3E054B5EDD88FEDC8B5CB8B397418DFE4C2FD8FC0E1AC226690EC0E80F0FC34B15AFDDB4E1CFBF921324EDECAEB0DE10EA781051EFDE547F5C3B92E7CCECC86B8090A310962FEC8801D706E86FACBF964B8676106C6FECE07C8013073F273C850019152A3590DF63074CDFC8EFECE44FBC2B55030C4FDF530132C4160D7FCC84163B84DDDFCC798CB858B9'
	$sHexData &= 'ADDF59745CBE4BC1A8B8C331028D292AB8F7FDF1316F4ACDC2A6E0739FE0FA0AE12FFB062CBFF074250FE07CF01FE014A6A8A3E03F286CB1E9B48F591D1C28B976418FB07FE4584E949CE17CE214D94AA65333F001C0C20F1A014CA9C15610E50180D81D919BA1136A080601674FF94788454D84D0AD5FBC3C44DE0DA096EC953519828788D4615703F81A529330C0116C83841C749D2914234A78CA117C13C12A9402DCA38DE5200DC64D9B7F9CC82403C99C746C5BBE950C6C5E1158136084C0A750454C87BA275898BF47565AFFD8881784224314560A1B47B7315E47216D819B058BD112D4B16201CAD8BF1FC05AB1A58E81C698440CA470126517914C6DDFC8B01EA000216CEC1541739F9214B141073CCFF3FCAEFD02009E887C6E5EE7353FF3C83F24FC36064EF39EE7F9941703007A644A14262EDFF334DF1206FC16E0C6062EFEFB799EE76906EED8C8B8A89EE779FE4015030054607286967C9EE779AECAE8F60816F33CCFD318384E5A6CDAEE3CCF7E92AC010C040E3AF33CCFF3066E8EB2D6E2CFF39EE60E0532520E5E7482CDE6F33C94E22006C603623C4FF33DD00610075496CEA7F9BCFB1208EE06C800093ECFD37C9E80C40A0A4C96E09FE7693E2E0B6EBCF8520C9E779F77CE06D6180D8E069CDE3ECFD37C1A0E3882CA1A0F3C4FF36968A403866648799EE6F32EFA02D8'
	$sHexData &= 'B89EE7799EE7846C482604E6F33CCDD001743C02BC00799E7F9E7632F2FF0200B06420F33C4FF3DAFE945A200279AFBBE6DCFDECD60142163A0628F33C4FF36AFC7C8898A4CFF33CCFB4C2D0DEEAFEBEC5703C145F121E067C11799EE76992ACC4D0DA7B309CE7ECF6B88612E79E93373B2C130FDC26EC7664FBEEEDF6066C37460FAA179E06CC171FDED96C380814A644145C13162EF73CEF79ECD006AE9420264F7906B7827FFA3FB622160E79DEF33C1896A006AAB4E0DD9EE7D2CA046FD6DE8A9E52F0E43EBEBE0FA6360EEE9EE7BDFB060410A606380E445E7C349FE77996B2CA4811FCE73DCDF7126E1E11A25E26DEEFF6657F010131C011196C11300650D89DE73D603690B0160121A8B62F0224CF0629CAFFDA8ABDB1011112013F07FF533200210078002D0073007939DE743B0E640065FA00610075360CEDFC33A21E6C006F00630E6521B6BDBD69E0040506070D07000F0DB620236D061E01010FE46476F70D047E3F020216643664021F010F3999B9ED0407FF7F03034E460619030303034E664946033F020241664346021F013232CC6D0F85FF04048B0C32720404041F19B9B96D0F0DFF04044606193904040404664B464E04047F039141464E030303599291930303033FD99091930202021F53705B90010F42FF05106C272305053BFFC8C8C9C80505050532723272050505'
	$sHexData &= '058C9C8C0C050505052327239305050505C8C9C820050505723272C8050505059C8C0C320505050527B3808C05FF0404C9C820230404043272C8C8040404048C0C327204040404CC968C9C04047F0323838C9C03030303B324232703033F02B321232702021F0155A8B6200FE1FF90912B12FF060664E46464060606061939193906060606464E46060606060691939149060606E46464900606060639193964060606064E460619060606064E664B4606FF04049391414604040464E4909104040404191964E404040404992D193904047F0346061939030303036649464E03033F026643464E02021F0104286E410F06FFC8202317FF05050572C8C8C9050505050C32723205050505938C9C8C0505050520232723050505C8C8C9C80505050532723272050505058C9C8C0C050505052023B795FF0404C8C8C9C80404040432723272040404048C9C8C0C040404042023B7947F0303C9C8C9C8030303034232721B3F02020FD7B0C51F0F081218C20CE644020702940FB0D004E60702761F0F0B3581B90702183F1F04E6C2420F07024AB4500BD57F3F1FE0892E2C0F070247B225F6DF0F090A0B0C0D0E0F1B1E0001CDB9C0121B077F3F2DB1D06E7E3F1F1BD6D6B22C0BFF7F1B1BED12CB3F1F1A7F3F4B2CB45B7E3F1F1B5B0BC1DA0FFFFF1BB137D8DAFF1F1BFFB1D06E6D3F7E3F1FB6B5'
	$sHexData &= 'D52E1AFF7F1BA5F6D6D27F1FFF7FB016DAAD3F7E3F1F961C0A560BFF08094B1B6CDB1BFF1F1B7F2DB45B4B3F7E3F1FADA5826DF3FF7FB5B796B61B7F1FFF7FB1D06E2D3F7E3F1F5B5B052F1BFFFF1BB137D8DAFF1F1BFFB1D06E6D3F7E3F1FB6B5D52E1AFF7F1BA5F6D6D23F1FFF7F2516DAAD3F7E3F1F0139409187FF0A0B2D6D6DC90A0B1F1B7FB1D06E2D3F7E3F1F5B4B2C2C1BFF7F1B6A6F2D6D7F1FFF7F62A1DD5A3F7E3F1FB7B62A5E1BFFFF1B8A6DB0B5FF1F1BFF5868812DFF3F1FDBDA6A971AFF7F1B527B6B693F1FFF7F120BEDD63F7E3F1F5BC182E21AFFFF1B05DB5ADAFF1F77FF58484E0B3F02031FB6165A961BFF7F1BD4DE5ADA3F1FFF7FC442BBB53F7E3F1FB73620B81AFFFF1BD0B2B5B5FF1F77FFC442BBB53F7E3F1FECD656BB1AFF7F1A7F96DA5B4B1FFF7F3F905868B77E3F1FADA8B0A8C3FFA5ADAD85FF1F977F120BC9813F02031FB6B5C4C21BFF7F1BA5F6D6D27F1FFF7F2516DAAD3F7E3F1FBBB555950FFFFF1AFFAAC2125B1F1BFF89859616FF3F1FB6ADAD761AFF7F1B3F2DB5B7961FFF7F3F2EB1D06E7E3F1F0B1C081214FFFF0809AAB0B4841F1BFF6CB11584FF3F1F04A92A4B1BFFFFB2C4120B7F1F1BB44010AAFFFF3F03824B2C1F1AFF065B7B6BFF1BFF1F1658A8CA1BFFFF3FAD7689851F1AFFB796D8AD7F1A7F1FFFD06E2DB57F3F7E3F21A82CB11F'
	$sHexData &= '06FF58586069FF1F1B7F5868B7163F7E3F1FB6165A961BFF7F1BD4DE5ADA3F1FFF7FC442BBB53F7E3F1F5B0B10B80FFFFF1B4536D8DAFF1F1BFF2CB4C042FF3F1F6E6DB54B1AFF7F1AA9BDB5C47F1FFF7F8985766B3F7E3F1F5B4B40610AFFFF1FD612CB061B7F3FE6120BED7E3F1F13FF96D8ADA57F1A7F1F6E2DB5B7FF7F3F7E082CB1D03F1F0EFF15586201FF1F05160A2E77FF7FAA5962A13F1F12FFB5B796583F1FFF7F3EB4440B3F1F100E0E0C73619BDB020C0A07020A080F07029B0B0B6D08061F0F070206048585166A3F1F0FD442B5CD070204027F3F9A0B0B2D1F0F07020256A142A1FFBFBD1097D61F7F3F0D7A090E3F1FFFB5050352BF1F14A176807F7FFF1A0C8641FEF64604AA9A81E60702041F36B77DA8201C1C180218140702141016DAE6C20F0702100C1F0F42ED051707963F5B88B0D01F0F0F7FD0422D543F1F51624188FFCFC14230629F8E4BB7B06EAF8F0F38B76A21EF1F0BC0C2562C4F637FADC63A848FFF704020A43FAF206BE004D68E10021A0727FF01D810C84500FF0360D12EB6011F07362206CD1107DF9D58B6882822E6A7EF0E97B3BDCD9E7005B006A016AC50620FB2FBFF616420616C6C6F636174696F6E4FAE30556E6BFF7FDBFF6E6F776E2065786365703D5F72726179206E6577206C654D7B7BF76E677468BD737472691620746F6F286F006C6B'
	$sHexData &= 'CDDB3A202A67656E656336793A656D77DF0D09B3CE167761696C5A2073FF3737FB63686564758E166865206F72652100546869732066E02E78EE756E63E7626E0A7420623C33F3FFDA50582026612064656661756C741AF4DB2BEDD975543074B66B62726F6BF42070B689086B0C6D9048947475AE3381C006CC0802700A6038B6397369657657493973BB66F67E0E73666939EC1CA6748287F7BA7B6F125C023F06012E070BC0BEFB4100DE2200222C0800646952BB9B60812094795F0AD4793A3A8B757376063BD8002A3769B08C123C6727586A274F6F70132B2B69073F050F55004E00431F27F1DABDFF47004C004F004200410E521202543F3C5FB0408EB474739672656D6F7663F60E5B1BE4B85FD526367724716838D86B6CE8E8B069E08AD87023D9491A00760A7317D1EB692220007090CFAF9DBD68126F9E0A2A6D226700656679EFE27D276D43D712697265D1DA796D266F0A6C2A64731BB1F7DE722E6D13637F5F9CF70AAE740E646366651A6EFCBE7748567656639B63526E0074045B82F7785753739BDACDEE1D5336753A6F702A62657B57C1E13A20F77336615BF7DD75E768FB6732D637701E6018189C63836277EEC153B0CE61653B726A60B5F7E02AF776476FEF4DBBA34B286A1F2922624E74CBD5560F9BD4EF5480ADBD05FF7E741F7FF75D126F523273EFFB745A1B7B'
	$sHexData &= 'F1BD69A667976F0F4C87DFC7710F536D27A6476EF0FC9E07B77855326B006E7737DBEBF06F0B724F003E007D005B7C1A59F9005D003C007B007CDF64EFDFD171AF7EF24106480045005FFA4DA3BD7647F2531626522B26567BEDBE0ADE5F57442E264F0F22FBDEC183075F333F0B539F704C6CEF4C42441FCFC2F8BE6FC24753BB32172968AFC1C146C3A33A271E73D88BB799337E73B77037D8AEBDC3AEEA5F473A2F9D14AD762E2F7A6572ADE268EF3A70D273977243DE8F3B983B2CA78AC36E8B2EB6C1F42F538FE26FA8C55877636D27C7675BD176EDB2749F7017779E1FF76EDF3DCF6653002700B32E6DC3A369BB3D6FF30168B5AF836E63B5A78EB7AE81AB5F3BC7BFA2B722B777A35668835B72C10E3ABF5367C76BC004487649B64D1A471F3E8CC879CB005B580A167BA78727E70B300886744FB7D1A468159FE65DDF07C55B3430FF97733D0C070F6E27635F740BCF88DEB13759064E3F70363D67F6B1F6DDD2072DA7DB772E6D75DC2673375AC0A3C101DFDE63731EAE36700BB63F68A3435A0AD237DF9F833DB81C5B87626BCF64EF4D718C57E32862751A6668DBDE4D4773CE7A9F4FBB347ABA2F61C7BF706DA7F75D8AEE791EDB4EF3754F1CFB0183BB7F64576D009FF05EC764B72F788E68B6800F01499753237C4D078344E24C1F8BDF3B4033BA32125BAE380036FE7B77EF'
	$sHexData &= 'BD201E3006353231120A1620027181EBCE3A0F448E0E562F602E70672E72472DF73A5E781F022A47765017BD9393A29E594E4332558DF0B0E39F5B8225277C7B78EB7A276F1E2502780B2E5681D93F4DCB47BD7219D3374C571E3DCCC6F14B434B937ADA2F86DAAAAE5F421757502C58D1E8A25F53EB371E78B85C7A75729E325EE3FFFFB8D753657450726F6365737344706941776172DB4460133E16436FFE78743F8829961350496F471C547C6A5B625F5C8B5F2A76C5156B5F8B0063332C31C0CE259FE02FD81E756E1A646522645F6D61702F731A7F419A5235FE6E76616C696420BED8199F07706F7369722F5F76697FDA31356239006F6861736820629B839DF175636B659ED217D67175F6BBB1F6653C543ECD36AFC000019CB006F86C9ECD8A7089100088F0A075BD5FC3DD5772827257AC2C2E30D7DA68EF0620668020103F683726D93EEDB15C76D8CD508ECBF08C3B27DB37FEEC060C7060463EDF6CEFCED006F0FEA56469E00000144FD0DA73B6D91EA234E90007E8D0C0575B0CB970B1E492BE3BA04E4A1B10DC94DE6FDEA636A73DB7B007A09087A1180E309ECDE6D9060DD0E008F0095010F63CCFF636480600705030206F35DF35989D4780063E9F7C084247B6EF0E012EA7001607D0ACA466C3698F1FA9F4179DCC369CE0139C90CD8D908C50A3801F299D7611BC500660'
	$sHexData &= '1F2CAE0AF31636DA8E86F4EC06306C3805CF100037A4EC16D9401527E4D3CBCA00A71466E8BB69CA91B0F170E6AC01FEE00667F63CCFC0A0907097ABFBB4EFBD6A3E5FB65F3E5E608E5D24A50EEFBDE7BD165A665A5A46591E586E5775E8E1BCF006B026AF4A18A60DD9F00E330C06A06016BCC373E4C80B00A7882860F4CE705EC1C1102FA6F006E06C18F94817B6582CCE09B0DEB3E0D3039E481E6B1099069FAF9BE0D3666565803F652FAB06B1DEF7143CB7E663616E24BD06C30964C57BC97E5C16F140E1DD09DEA204EE5017CE30067ECDDEFBB4CFC7AEA0A81E694DC086B82E7033C59319A7D7A0848F0EBD7792CF0E0F10A424C88006D034D94EB216A5B4A738060CE0677327E92FD0A74C4506CFB069387B9EA0904037AA24AF2FF9D8DEAD642E20072ECFA00493AB230F6E10A9E0769071921507C726A3B0B148793A38F2D650ABCC16FE8E70BCB3E7E2F646E0E06700C006B047565CF5907F78DE25E0A8987CE54D5427CBE6CBA06F12833A808085200E2FF0B63526340BF02343D87C0EA04020419CB61AB551571D0182027B27F041267F07ADF6ADE585871F456FB0A209FA98C03FC0993EFAA024DA5EF86D37638CEC01E2829102B3AC2F03D40C13639E0102C149DAA9B8F6C148AA06801732C3D02F401F90ECB5C1704E6C151E7C06843FEDC6A662BFFF83663617378207C864'
	$sHexData &= '01A8D4050909D94EECA4A4E7233826A970169D90CD3770AEB029E0B6B01C0F06C17B2C5E1CB507AB9E4E209B742070A2A8E69C4EEC9B3A0F1F2EA7B4274060FBEE846CAD441EC4E62E9EF8F63079A764A0E09FD83624549DDA6BE73C861CDF46A09EA508BB7AF8AFA40E547E48E7C16E90206F0E6637D6E0812E8F9F16BF1897723B819C34E89FE43629100203E4E601AE54161C03E27D13C1B02C8731E6A1A49E44F0C081A71EA6A55E1F3C64C129D936AAD4AD10263930A7F6BE09AA96A6549EE86FB08070B076621FAA1845CF223AC14EA72408B7A7A48FB107C86FA65623A000A074B7E8C41E30119E807FAF50E5BD878B6F0CF6D415460A137B27E4A31C2E90EED4271D0543C10386D78C36389005279068AD78C618C0B2E9C526ACDF28E096A72854D43D1341A68F881FA213B2E03E0816A4C0A900374EB2A30D0E94BE9DA030A89B8D9D40AB0CAD1F8CAF078F83139EAC06BCD6FEB0F96140B63C5EA2D4090FA47C8BEA04B23FC4A9AC4F0BA4E90B96903E1179BA2B157FA7A05EAC0F8900233756B0401605AAE47F7638E078B608B0674E2726906D22E7A07E2EC4AFD3AD059E2E2A0F083F04E7B338714CA8F7A140578F17058B047E96F8EEC46443607816D0BFDEAB4831843484BE1E60C5800552B61F473C406EDF1FDE9FF45E18A002AB640D5FAE84B68FF6641C172C444FCF2887'
	$sHexData &= '2690AB4EA8106630180998F1E2981847A83F388C261860F10670EF1FAC3CBA409AAE11ADFEA6386D04C30E67B7DE905970B50837C698E69EDC72C3A363CFA820763F88031B4EC8041C20282EA3E8E1E23821A248CF8486030A3867B7B0DE55272C90299EB73C703061DF102E3C0B26227F6CA6A034E40172C722D7662DE80248645AC8A80EE7E2B192170704AE878BE08257F4FF68D609A6B30EDFC8294430C76B820BCFAE44EC275827BD12CC8737AE2FA70EA3834F07DFA79C412E905DF63BB81C6B0C6FA605B7BC7EFCCFD0CCE380F7ADDF0601B3154184BE97272B23DE89BD789ED0273D10CE0C27C4676213E06F449DBC70B1E6600730AB6FAE59F02AD2B766D4AE4436B0B815AE5429FED7EF3071E000B058AF80127B80DC5E2E5008A1D8AFB8B73518AE872733F8CE16E19B7D16AF68A9AC46A278AFDEB5210B0EA30026E4363EA5D70BA23B3F25FFA1B48743AC7A1790268887D701252658026FA4BF1E4463EF2B28769C7F67A4B0A389E1C41FA7783FA51E142C12C82B78D6B6F0C82E4C706FA5762EE0F1408907A414470322CC36210F1857AF94CC1F29697127A3C406CF9C5E4C13C19E08AFB04FB140F156153E46B0F7CE941A1301469FE770DD7863CD6701BE978D218B1EADD856A5E438C0C54167984E5C6713EB9860A7347F2A4CC889F7AE185457039E53D78FB0A81CF7C1F0'
	$sHexData &= 'C404C006E72698293AA6D4A63E3784072E880C7F675C975C860013682F04AB5C7002F93D60A98C8E306C035B2552F2101FCF9A56E08336AF34CF9F09637ADDE77406E4070CF8198B373FC404B7E1DC9203A6B04F272F7002598EA3CC2AEC571D1ECD48C9D00297A6C0024713170FAC4614C2891289777711C0251846B2E7166F2870818B5F2F2660C7354E1438AD4718C03E728260857E6FA3E860C602171086B72CC00939A99ECC4EA4384D5578933F9707585C450F1C76B8AEC7243171E084A16C9F6020EFC1F05F6FAEA8D64001912E4EA2F0FE6427E0AD648EE700A68C098625566F6657DAB80E80B12EF857B76AC418DEDEBC0FDFAC60715717E03EFF8A70423EA7389F90CF051648C56E2BBE4D704304162708D8C1A1315F77F01F5CD5B29D9E40BE373877507452605367C47F765438119C4E57AF58CF2E3E924C801FF7784E108CF64F9E144423D0A7C4673E9F28DF4C3841A933C7AF8457E070714C089F64CF17A421C167F73F40AEA20D1F5F076FC40B2A1D11672FBFC354A91FCEAA8017A0A7BC15A15B1F14EF2F27D99422EF8EA0181CB0C170820067AC944775E04180F7AE708F92933D70016FB0140858E941B04FA1C4564B5E9A601FA6DF4B8B688207D4AF62D4C189DF20C0CE00C7B1466CC42F0729AE1AAB9F7703D7089EB8902C0C961F41704E24BEE68FBA400E30AF5622'
	$sHexData &= 'AC765816E3AF16EF5335693E4FDB927D4606B8ED20474DF3799EE7709DCDFD2D48E6F33C4F7CA0C0E00D4979F7799E5F95D5464A6606C0FB3CCDE7F5574BA3EB3E4CCE9EE6F3BC06C3ED1D4D558D3ECDE779BDFD734EFE554F9ACDE76990E96F501051B0F379F779FD45526606E53553B3F93C4F75ADF58B547055F3799A4FDA1056A1F6145C9ACFBB4F6E525EAE06D00D5F853E4FF379C60E6091D50D61349FE7694D97DF0D624B799ACFF398D50D633D8EF3349FE7B0E52D646590E7799ACFC50565457DBDEEF33CCD2E665D8DCD36677E799ACFF306BDFD35686DA8DDE7DDE7E52569FE06DD1D6A76B5789EE7069DDDFA187D80E6F39EB602C4C31634F80E40513E9EA7F9BC8C06F8343F5FE7EC3CCDE779EFF25D4065CD7C6AE0F3D56C116276061D9BCD66DB72070E7E241B10CF34C10C72DB6C36D54131018A43FE860000FF2B57EB2626576E0252534453A6585CD117FBE1BFD1FF0646B5C8DF6455AD346C6F443A5C446174615C5FDD2A840EBA6AAE5F12432B2BD8F6FFBF0A496D616765536561726368444C4C5C52656C1673652DB75DF4BF5F7838362E706462552507AFDE23FE04A84743544CC080202E14EC3E128E2464694F80116EF0A316C2EC276DDF70A6906D57D8B0244F8C206B1E4103B0D21E61799770268C6964EDFFBA7B642435258C7302A72E30306366671E94B1EE'
	$sHexData &= 'AF3C0443525424584341479827CA24AF324C9C4E1C550E569990B85ABC9F44817C895DC0274FC4902FB1CB49C8274FCC542676394CD02743D49790CA8444D89F12BB1CC8DC50E027799703F94FE454E8261879361F265AC7744039721078934D4640AD1FCA21DF6B9F5826A01472B6F790CDF8C1B40173787D7EACC3CDF727698C766F6C746D642E38C40C0F3AEC90047A008C674E44C80F76F92772746324494148275A5ADEC8C02A4C4F54502608DDC366434F5827746C731F604EEB463EE80824B7D13F245AE8C0E8CD675026201789134CB4C718F5F5028F65B1F08D70C8A2F616180132B7F8C846876C26140033671CF818B2D91D3427A8FB581C37B8F651A92048A09F6FA0D93E4CB6211EF00DC4902F1E5827BBD2EC0E7367E8561C62737387EEAA07E760FF733E24303127DCD6219FA0F00332075D11C80480070800368A2F01023ED4A72205931947C174BB95160CDF3F023670577B8B64A402A03F0E948BE4DAABBFD4E00ACC1521530507E0EC4472C22437400F3312AE1662A707C258AF578422B976D1BF9057B0E422B9E4C0DCF05239919CC308382004E7AD453234AD0E8D546EB75F1F67876F0378B3B501BBEF90870F4F390F6E8159E787A46903EAE4ECC81D9420411F173C6A8C347850C1208E8705F8D61E399139303938013782D370F22E47521FC4449761B71072C46467'
	$sHexData &= '801F7CDBB09102884F030786780938397B108722C01F1D94C682537422A722CC4FC4CCA7E5BD938BC0C8736E092C746BF6C508010F509F77B4BC6363470E089C7D506B7E5F3A21040F2905E846868D313F07394F0689E096002FDF6497A060A4C125B327069897911F48C10E70023A7802A7E1E482801F88903FC5ECD3C24255270F67C66BB78B82200FE0FFE867E96E2792543B200F396703FBD02219884F0E92916E648801937704C40E4E4E2EC0C8D34F39D14ED2FEDEC708873C1019E98E1C3C293C317701444572ED2270CF7072C9C9C9A0A8B3AC89E4DA45E057D868D14EC53D105FC8045716C925EF4D664838C9B7022B5FCF676C2E926B17B05798F089028C5407C4971CF996933E3E0E400248031D23478E50045B0566B7C9C9453BC920B7D7A00597BC8FD786CC643F272727133F3F3F3FE4079C8E486FC9A8C7C9C0E6D696833F3F97774EC189EBD4DF600447CA082EB848CEBFCA28C047CBEC9688F02F27873CCF288A544E623A05FF495B8EBC402040281E4B90916EE4570362970CC06F27D386009FA0C01FCC0223674F669FD7E2040FED05F8C9C9C997064103050403EC4B83C9020F0ACB48010F6EED14C1980F41402F8F6F27B7934B53AF5E69EF8906C3C04E662F95A07F08DEC90979C809CBAC6EE803C85F38CBE0CFCC080260276FC941E0DEE8F02F27E4216F03DE04CC'
	$sHexData &= '1C711599283A5D2FCC50D7424532D23D42386703842395DBC97078778004C0BC9D5C24C0C887D002A200938CD8CDFF2723DBF243100E1801202872918C748702486017C948776867017CA072918C74BF03A8D0E4345D70DA37E54FCDE88E6CCBA944460E20012B008123474E0333043E043B397B4BAF1F68072F7307081FBC3408327B0F0FCE70393939954549C0E3CE1E3939EFFA4502050F0D05397B76DBCF1F15FF4520094F2B0A05999D1C360A0B1F3E6BA2E13457CF0CC745D74ECEBE9745C20187CD01019701324B242358570C728AE1368867E82F4640B64E0C3B46592F02FF5F869DDCDE3E610F767E5F0870DD82CC1F899F919781F102871F0FFC096F919CA203B6AFCFD030C548773B46D0AFE9677368EFF276E454472047481F54023D19393B5F030F6A04758006479E8C9C2F880793A904B40AB7939123BC0BC79E0FD29391B327DD0F2FE510F0EFC893B3FB0F4BE6F114F9152FE464F6C84C044806170F0E19793272F6241A2F2C1B37421DCEDE761C4A1D671F55200F601D91234F466B7623812472E4C8938C97269F27C1268C1C397229CC2AD72BE2912F479EEA2AF52E49002FC191B7930B162F21322C244F6F234F6C0F4937365F1F6B3891B3272376813A0F8C3B3972E4C8973C9F3DAA3EB53F23478E1CC040CB41D642E1F2E5C89143EC44F7454A0246478E1C390D4718'
	$sHexData &= '4823492E4AE4C89123394B444C4F4D8E1C39725A4E654F70507B51C8912347865291539C541C3972E4A755B256BD57C8589123478ED359DE5AE95B39F2E5C8F45CFF5D4B0A5E155F23478E1C20602B6136624172E4C891634C6457656266478E1C396D67786883698E6AE4C89123996BA46CAF6D8C1C3972BA6EC56FD070DB903CE4273F3312D424F4960B704C374C0E610522DD8A378F27FF9E9CB45196A0ABB602912347CE0FC103CC04D705F972E4C8E206ED07F8084D030923478E1C0E0A190B240C2F72E4C8910D3A0E450F5010478E1C395B11661271137C14E4C89123871592169D178E1C3972A818B319BE1AC91BC8912347D41CDF1DEA1E1C39F2E5F51F4E00200B2116229123478E21232C2437250592C78142E7D5CCE322B9A45680F8C083E4E571F7D624D63C3CD847C84F00D65040792C4208193922FFD6D6327276DAC740010F4802501C393B795801051F60066807084E468E70087878570999E4190A0BFC279223C22FB2FFD74FB02423DD054FB967016498CE9058575F08DB906AAA984F9D0EEEC2C85B4B1F5602611F04BAD903B13F5073060F7ECF0504C10946FC3FD8242FABE0A8ED50C00F6E07C3A14255BE02F7D838772447A5133978FFD892D3014C51E7D898721AA60567406FD8D01123C054510782DBD9828F2951780F8A375D149CE48AD9348FC013C929B80FD2'
	$sHexData &= 'F7D96852D122F926579407918C54DE663B04C836F27672707B0F83028BEFBC60919CDA0CAF52C86B279293DA40530057924B2E926C3098EC768A2E708FDAC467EC5729886E17D00FD88F1FA74EC8DB007427DFDB3402203AC1660F6CDF541047912179DBB72E20055527D643AF01FF940487EC844CC10627A477AE1F89E4545C5F97DBD88BE4B4DB54500F5867DC0C8BE49293B0B84089E49293F0F8744846BADB55300F386701A872915CBB7057D4B0028BE4A506DD00F92E66CF370EF81F56970E080324A78D1C16041EFFDD544572491B565080CA4572C980ACC0172C929107D8AF56F90E6EB3F8B7572F57AF0E1004ED748D1C1E0526D7DE34B7ECC99113576057687602872E9291CA7E0178B0745C24D757A4F0B7CEA189E4DED058CF5828145CB4D3BFDF046760F7022717EDDF30BF909BC46D7635A7068774EFA4F2673C689F58D09F58D8023055184BA7CF0579400E05C7DFB801EAC41E242F0FDFECE00C5910893D425E5918E020CF8BA2A3C0EBE21FE0FFF61682DB5950CF589F1F60030F2E92916E799703A0C02884930B67C8D0FF4CC589E4E05A1027E13D484E04FFE1385A4064D8E44BE75A483F0410087761509705A71E9007E4E14C02033702037EC91574491F9D2FA0202717C9D080884EC8955C90E4051792137242055E057DE4223972E238E268C09C920BE4C8D0E27C21'
	$sHexData &= '27E48406F7073EDB6E9D3407A5FEAFD80F1F2D19D8FB0228DD06F03FD00268A29D7F2BE03F2C91F740869D2CA0CFD42D8C069E011984ABA3B7E34402D206B2CD0FAC36E4A01E70CD7021170C1C17E56B581A897F46C7B5C99DA05CA740901037817282B3A1074007E4281F844E4E72F03F00BC77C1A9AD8787195017388E6A537092D4E4842F18F75C0BA1F41020173714377B31BC0D2057C804A744CEC12939B11CB711A008905F1BD94672C4A67005909F04A7E404176429101F6A58045D808FA7A960DCD2AFAF766A2BA4704F3728C0C72068432C4E0147F2D13E5CFC0227B41403D6B5577225521E1577BB922939E017FC7227A37D255792D2FD0277644A4EC97011940F52501785AF5A120337004E6351D9222E8C083311808C025E452D540180FF10B4207900017DA0F628F9B119BF445541BB45570175987F403B536D01FF5F08C556323287C6437401F702980C1EC1DB0FF79A641F9C01098AC8BC18C2572E3F4156725C696D655F06FB3D21EF4073746440403FB6294514BE5F377FF6F6FF873C6C616D6264615F313E403F313F00243F305608B39FF9FF5F46616B655F6E6F5F636F70798061626C2096B7CFB176617074D2406B34494E063F556E85800785696875BCC06E4BD65941BE96027369635FCDECAF2107405F57552238405F854B706F1E6169747320315067DF20150D8C'
	$sHexData &= '2832402F4142B76666B3559F5008E65E005A334035B76D074C54B66B5F06796EC17465CDFE0DC18D7340554D1E6A52DA9A89C265734ABD3530FFDA969B0B514145402424146EAA6011320B9F02E4730532464B058E14266FF56440A12ACEFF55506978656C4275666661B5624105D361650A8A548F4733892041494D6724AE630AAFF5763E5314AF324245F75A60056D46A642586D48A9E01824AAB1A47B400DF6F63048005F4E314D2E5B861816D834AE3036343F74DB9AC88C9B5E98EE6538436F1012055A437554650A7947A24A0DD7A867FE8DAFA6B0FD193752303F340A5F536061D3E57DB84A6E75E067EA6489894095696D706C5F22B860D3BC9064651A696C730B6DA580536E9C555F7B83AE150C486664A6426F33BBE24AD1C551305871DFFBE85F9C5F4DD8566F6964546F3C744675CE2CAC4830F5A78A83570BCB7C41361A5A5D6D7A8E163758357B180E2CACFF2F79B09088B0FF3A4D5574E63F31EF435805234A6F2E1C1514A700DF58445DAB47A06155856FA144A31021FFC2A5052208CC2F219718D194A38FC583AFAA09EF5F496E32882D74C47275315F3B173B6040D10077531754A90A693F43FC46D800DB5F4381416E6452756E0F2CBA0E517391CB284413EC4045FD5526E451385C56E83030731A21F82150845C6C5F4571DC9160ED486F6C76727F9F5A4C4521148F77'
	$sHexData &= '5F31E13E2D2788BF82416948433F96E6DA429E5F5F13460C1BE72FF10E0D48417F7479C722B809E0BA6E666F8FF7A9384C1362E2FD94264C1C4F46B14488ADE32A66670E79323FD1465423798F8548ABAE946616AEB82848C471626A32512F920D4D67555A69BE616C2E0A1A7440584D33086D7461F054D8386C5B5018DC054E6F41E1F9B81814A05374F945F76875A46A3E7327EF2B066C89CF4377A0E0B06857C45E73AD800AAFDC694A64C9F6B5EB8BCF476469D2757365172F4E5DDB47376F6648656115A4A5906DC59F8F04586398B16E00A2E953A78F5C50A0E05856566F550D9806DC50504C3745655401355673BB045060C3822ADBA2B1856A4F7953055673BF8FA74744810B1149A93E2334600F216CD941C42569D220D02841831A2863381B89C0CCDA52986992B46581881BC143D81C19ABF68C5D6BA04082A55B3DAF0144008C1D012C45815D78EB607F2258DF496F7362404837331E4A30516483E84EB040AF4465F48100CD18BC07A21258552F830260402A0FC41811C515174D299013696F2F28587569E7B7006BC645953B68C401ABFFA65335AD6A404FD1C08A8A5591458A4B6B8DDF9709368A6065A744A0111C353F40D211186D31811D491E3435AD14408155CFA90C56540A0F6AC4AA4235ADB5429EA9E02F324BD66AC402DF451222E07522402356A140441810CF8F30'
	$sHexData &= '028E80FA43F845AD0511CF2F818012188D2A20C185589FEB065B0590454701ABC4A1C191DA2A0A7577A167018C543A5074349ACE4B584BDB322F5071282C551F0410CC0A79ABAAB0A6937F040A101DE7EF01A32B40C75C6BC4AA7FCF5F5250A050618BE0D63AA74FC60A429E314360A1466CCF47851256DD400FC9270D682F376901C183149745DF15224488DDC7D500245825635A35E1613FFF442B8CEAC33C8F200A89158D06421466360F2EAC2062F9EF81D10CAC3B75A5209E0749ADD6AA2E61DFB8B082682FC71A402084C536C14715E75F44694C58756DCA4227F19501F7228A809029AA226254DD230808B6097F12DAFCC302000020600147214164F49F93D06472281E6C6F62616C4DD6866E367C727953A724B3DFDA45782A4638654C6962726128186DE8F68979697A655326737448C00701CFA88F75549C12BAC85441734622137BCF12FA3143B3CFC6495A6FA8B5F651756592500666CA6D984EDE32521DD04C6F61645728C0DAF6D273674C46CA33D4CE6C581E1A9434570A92502774416C6C6E64819DA1D6D956626910E16DDD00D25468E0DF5330657036F7DFA3575352C641637175697217F6636B6F1725A4D2EE69768E43B62BC90C12D43B81344E60C30C7F4D750EC90CDC9C079C2EB0B199A1537A2C4F6278307B37EE6CE674435719549A0DAF6D26705024681A53484C'
	$sHexData &= '6145906E9E848153447B66B80D0B3E6ADAED1516EB602D06725893571FEF36BB094A234EC841D2136C71D5D041749A62FA6E2BC506622908014E611E149197604279A360539A08C21C2E092F3935159F695D3D321841B370D841237605EF4E5349A98D478FDD90E21E73576964BAA854B230B3866F246C3834ACCBBE8187406EA78A694DBD57202DC63C419B3B0C1F3FEE9F80DB4F458C7548C7CE2A7A2342EA6E50318A14193B95829FCDA2DAA40184625F77636E6379DA1DD669363428C68C765E00AEF66C5E271A17415A9E3644EB9807AD5BC0CE7A6814666DA5D91CA46D23420A02A8502090B6006C6346DA70B03CCF6C907EA63A2BB0F866105C40626D5FC69C087172745F706A26B0B8E669CD8AD00A722996A73DFC64665E4AC0AEA835C38540885F9812695E7476761A9CB55F69DA4E9418249E3358B36F92274B590E72A5B0C400277D803D6C8B3F5B5FB7902E1CDB3360B47783A7E668C076E24A6DE1B66D61AB15226967063D0A67760AB2C7C02F9E58F3A66853D0E11C77F7006CA26C330BD1CAF066772C793A4485A1D762656B010112745613282407775E3CC2DE6DF6ED6D6DC85F7673777058E6665F3646766B05E696EC10846714636A38538BC566AA0F63516670D6BA4D614EAA3F11660F389F82DB0DCA148C2863AD63D5F0061C6D9A127786773C28B05EED69E4006782'
	$sHexData &= '00AD7D3C9D06300005EBD9A57418A642FB2E475BF5289BDB4B4466A3F7B44342447413985157C0F0C2C22B6F9A424A37B094A728B738E2999B9399B6D13AA6AD336E5B498B6C58821FB06799EAF41BBAAE25BF4772DA12136C746869C04D2B29CB76190F39CFE6BBDACE531261277553A3161B4B434F664D4DAE66F00F0447484249544D4150498D58AA9758DF73B39B92C290C0A110A878699F6B9F3253E86F28A619208A882B44206D1A205866E348606768ECE12BB8AB44AC772952A649275C5DC1D5412FC155B919B81A16C5C95312E191557A3035B0E46341822D9C757055EB574695A748D0C03F8597411D2E02F5594175DF4C70C5AAAE84405F4A4031095A28DDCBEC58693C304563767429794ADF9044555F4D6249281DE043E73256306CFB010302D7C841621CF76183236B04BF504244489F0D7854D65F315074C242ACD6760D8E6C415F4E445843CA883273448296A5032C784E5643027478E3C63261B4598725A4F9D0D66BAF5265746877051060644243A00DBB5AE13561CA175F4DB2BDF0763B776115155F7008020BEE52C35C2BCA80D9ED2B5962BE7AD45B5ADDC8F8A050158050843554C13138259ABF58912AFB28B63F308D4941457791015A2B8B407F5838C40454BEB46D86CED6D47A208140880430B8686448C740F52D2562655450E268102571404E0952B19FAD4F73'
	$sHexData &= '667840E19FA855CC02317BA04E2B8A68311D9C9782D97A6BB7FF6E319828785B70C38123289173840CBB16A09F7E4722836E66E11A9034D90A29A248057DDBCE15C80C55315590A390995F31554CC5F4C0D65F5506B505869483B10202883C4A2C7963B0A74C45554D8956F81DD05C6881477F787397AB380680594D417241C72178E94D03DA9D16E66A8F21B582AB507327EEDEE580863548695A658D021FD07F584142B341B46911BF95C7E980024C3687309A0203B6B050360E073112B5F39A1E3231AF6A13B501BA81487F1A2E4A2E494A0101866AA1814DE7EC89385229676F6F1C50ECC228B39F30664A8C04333A3C4AFCA212BB534149507A456F44F756666163285D2234696FD416E0756E44696691F248CEB6C004AE2FAE4431220AEE15A0FD8D6F75852487807692770C9A33696E99E179D9411F0A399704D60B3253E4776179737E09866D879C7640ABA9AC1422B5582A64750304736023551F5732DC234233396D7852341938061A5FA89B0DC61C2E6F203D49B25BA18660636E4E220D43B3D298E0308AF1C1E19BB32E16CC673F691AE160607743C656C40F16A2C022409F7861758E0282405E837324B122C3A54846DC2A20DC944370705DE02072C24AFB692E3792DBA8DB31319F6BA1D60C9B507175C6A31A07A0B56F70D0801BE109538251633BF347A19C8441881AA554A5'
	$sHexData &= '03C08317FF83C272E819DEBBCF04C285411FA849CD04B6DB6E43495F11D5FAB13AAC373F588C0CABD9E30F2A716E5D8F524AA3DE1AA4C11F5870C6AE540BC56FD87012556E681070B1B1E964458F62A50D6067F777ACE649695F3F59F670B2566182EDE4F0B1EB66AC5931792267B800EC02897188E029F445768A32670769B90D67516F832EEC2276B97F587B633544BB5A5364814538A81D8CF97493575B49046C366B4920A48591A260510F22B9A8581FAF48B9E40A47C1F21859888DA3F664B59FF98C7A3401114070FC1821B6004C716D82CAC0C313AC2464778CAEB4810672454F8EAE0B01EE69676E97588B008F10C9A5B20BF801F900DDF41062A504A8C08264FC7CAED967BD1D6649732956CEC0EADA59A04D542410C365466F186AF65617B00AD53A501FACDBF48C858953316D6F753D01164FB2656B6D9B39AD979F9645841973F3D5A61048994D10726E18EB9D9A51287E794D6F66783395ECB22943688C9FA502E8D6C6E88050C0182C70437878B55B8712C204CD5FBB5F64670EE0521D360E3B2EC06C3D644690E761E3C0767233637763F0DE566A31162A701900270C7A4EC7C01CC0D854E73CE43A577050A216A863570E3BC33B58421138FE9B5200DF62EEFB341B7FF7FF26A30C0C06170A16170B1B0D1C12060A0AFFFFFFDB0D09060E1E0C1E061221131C0F163D06162E'
	$sHexData &= '1B20F0400123F042011114DFDAFFFF17073307210610262A071C0C1D202011092604181EA00F140B0CF6FFFFB72E1420AE22124412F0FA000735321D0C4127441C0D4646300EBADDEFFFFF250E09371A66150F4A570E21100B272C2A062C78C70CEB7125A8BFA0FD0D2F5E0F2423FE38001D080D0E080A11FD827F35480C3E2A1239641337060B0A3D12228DEDEDB7BF641A1D0F171F050400120E520D592E300D154F1F8A2DFFFF6FFF0D26AD12626E5E0C480D1A0712E00C142AF0C601101A96123A297B1264FFFF5FF01E9C125C8A28080F0F8B24920F4F157E4742088E1257770F267CA0E06BFF0F110E0C0F2C05060009AA18234835FFFF6FFB9C230507002E4422F088010DF0190390907F10F0DF0121F015BEFDDFFE016075BC14F016B8F05003122F3966123740C7164B011A4411FCEDFFDF2202063015192A250F200F06225D154932FCD5CC100F6ED47FDFF0FF37209501D90FF036014C43545E1B3E24109E3202811B32BF4020FE13447EF02F0356B2A412590D062686FFFFF61B1F0A13002D4729242C7D2A09391D09171A7521551828F96707FF231B09066637301233010F4D23F04F0104BF42D4079E0D1305782214281F281176FF7FC30807F88736223F143C55E8802F302112681084B8FB8B2DFD496A14051505F62A202A0C821376E5D89B443D191708A605F0CB02E6B36F'
	$sHexData &= 'ED17A03DF0E10114252C105406501432050E2AFDED5F7047281416230F2E074D0C0E0B5437D223132807581D6F0DB716124F06763E042D0E944A28073BFE86BB4B1F121A067A08AB06110B09090606400D7F45C507058052AA0F0E350F210A06360E2BB8DCDA0A3F4C0F463E093021B644826B0DED600D127E1294C77C0748053E0BEC61371A055B5E8944220B1EBC090F13D87F6F56580F1D0D8B17F0220106A680198B65FD9B2408132C11300D6613DE0126E81B20E2062976E25EF01C0134B5FF82AD381136193C2EF03701F00402B5F0446A0F41F19BEDFE062A1639EE0D9613F067A0F0871F86150E3E0B05124E15F084017D316ADF4C40065E1E8F0C1DFACE4B063BC1D79A362A132E3C9D244E0BBA76C1BF6B6F1110F82E94171FC4F0250258F0460145D87F33EC0631905B135246280661633A3365AD2978163318262BF704715DF7FB2E4D2C050A0F0A450509191614BC6922091E1D47F00B4414FE1B23E0FFB79B780D3C0FC0D102F0E80112A61F241D1C1A1C2C9F82DFBEA60236CBF085207DF01402136C98273569BE81E0FF432929102FBAF07C017D1AAAA00312B1F8864F2D30308201010B190568338379EFEC1E9B4C12604328B1F67ED12B2BAA3E0273F08101301A6FA7FD82DF58CD01080CF057720E0433F0C206E02524D8B1DD2363BC9C1914743C5FE067BFFFDF72FC00'
	$sHexData &= '0F0BF07E015D1DE639393A390137350FF09BD14BECB5174284DF03BD96AFBDE2760F970FEC020DC636C801543301F05BF8FF7FF30E5F3329252533B2094E252C0672107A114389182D3D0A55FE7FF5EF02360F5A132F3116255A12401B161A0F540E190D1D11F9D33AAAEA0E68160C565F0C249ABBDB5DFFC7559B3E21146E1AF094541EF000093C0115FC5FFD0FB922062B1EC04E130F1A140A0AF0BAC42AB8A91075060F0222052A142356ADFAA66E00061C0924073E060B0A070D8217DD3A580916091A0B6009510F1BF4EDD6F5A6194E232B09832A3DA4D3F00A7E21ED5BD76F0E901B0D523446D01E8C960F1C061804856DD7DBF0124E0C64BA2F010E773E2B04066EFFB705F8040EA00614120823DA2B08F0EC02127AF00D01687409BE007A1BA03C0E4756364F0505055730BE0D20BC1F68320631FCFF7FC14E1918333FB173159C20180F82428A62B7C269AA4A2E4110FDAD1956A0185E844222C02B07AA7A0F1EAF2FF8DF761E086B04F0910360F070036C12621E5332223E103DD82A37F0810358030E703DEA6A07ACDDEAAFD421DA1299EEFF6F30D412D9040D8B15330F2D0F16284713156811F79D00FF3524360D1B150D78D61B253A10155B11DCD1F07F2C272F120DD212076529A624103A0719A8FD071015960B642AF006031107476F4DF7F01598572C118396562C1820FB3B'
	$sHexData &= '7EC52217381407371A0D1498B4024A3F5C3060862A25A063F0BE0C2D90E9DB821D0B325E020A01028AEF01223118A83D3EFE50EB12260D1F50AC10FE3A33FF58D1B4A621BFFA1652083FD0BF0B0F40A6E036074C449B4D123237FFF6B601E6181D766C27692A6ECA013E5D1C2E43147C804FFB2E0C31083115A20C3F062E11261D1E0D823F6418185B53049C1B06FEFF9ADF08C9128DF0D105810E5906476E0F31260F1673651DB45E1D5F3640F06C069D300A7CACE2AB1752186C3D12061A2F3E3A0113F3FDEDDB5021DF0F335E201E171855188F151BE1DB377D422FBE8B371EA940375977340E443E1227B168FF17A0332428111549192E2708111F6F3FC4DED824063235093B1C9927190217370A2B6D01BEE80A50169012290C260C74470E1625FC6FF845E60C1B900A170BE6FE000A3E2D6B147717425F00D737AE2A158145069C301218120B23FF8A5D8E1F86DC44747493014C4F71F06802EDDFF41D7E1FC9F0D001FE3A034B27F07F0206014E7145C10AEE1EDC3F0511C216A2ED3728BE860D0728080D0D0E88A454FFB7B386334E174F15183E120A680D28124B292D36C17741F4EB2F0D21540D4280D8186912C099E3EDC23D1A120C994F5866005AB6457B3BE836100B080A0B1A203C0A08052470C09AB6342A524A6DA6AADF4C16141D3B0F09A63AF8220C23055801AC38133A86'
	$sHexData &= '060157FBB9568852026810343B3FF423C0FFF6FFFF304F2420202D52364A714D233F6242485D38432A3F2E3A4079270D571B08FCFF4A48593C0D3D2B0D69530D404A4BF6402E8DB6E8FF1038A3320DA66A339448393A0A0206CEDB7F470178E79A403A78122F4886374292482EFFDFF1FF3E4D3D383A4722354842213F344240672F3A366958393638383BF1BB6EF64740685024001D153E481D0B0D190A33DF1EF119C61518E03D0816707066F0BDAA53E134050C9D1C04FE260E08ECDCADF91D1404143514091601A32848052969CDEF86D03C58187814180410180C106CEC1D9C034824141736F7DCF61E6B450F591555F7DCB9B308722304878D47F6DCDDBB0C91506C4BB95BDBDEF65E172F5BBD01C8C6BEF7B64D4177FF0843D9DBDEE71599107DD35D9D11BCD97B89411B1443417B0B9EBD7D0439450973DE82776F3738C97D23046F4108F1BB4303166C0BB2DD999B236DDB86CB5336550DD7706F7BEFAFD75D0703CEDE7B0B42935575B7D7C1C3DACB1B3B71331DF6DFF06E16E11BF0640F200C2008043DDFA6B930090308000C0828140810ECEEDAE71A1020101D1C1004CEB6BDB4F75558200507110BDDDADBDEC7C36149B51CB7BD37BB1008A89B0C24434F2C42BBC165133CCA0CCDB1B5D0C2B57319E3BD861BB8517A14081427ECC128DA4C011510F15A8687056B4763C5B36F'
	$sHexData &= '430B8718100618191AD250B00C289C28213BF168F7DE894B0C0014CAAB62DB566943DF4F45E1566C9D610B67C476BC8B21334F17BDB3913D1C04204E4A0006014F79DA591D021C041421FB7B6F7B3335113B31F0242CDD2FC437A1208AF0B801B4248E28249880EDFFFF6F20B8B8F0600230B43C20A82C1C28881850304850F00C021C5C68856E48F62444F0D8582CD0D6BE7DDBD41C3C2C700268403448F01001543C6A301AFEF63B742EF03830C82420343C8802F0580150A825E0FFFF2848483C24344470A0F0C00248F0F40134A868FC1DED4C010500B475F4F720BE1052A8210B010E32A973427A2C007BF73DDB7B776F8A10700E1217140667268B6D640F58900368DBB2F73C4078106F0F133A1BDA6C2F1F2CAB9A46EBC5606E90735AADD9B6702E6C1F4ED0AC703D2D79771BCF103E4014622B20E7019C3CF0803FAE615E604357F9EDFF8C461DF4202AF08BBC6FCFA6A80F6402EEED2D7F07ECA61FB4042E4D04387174F26C77F00F0C034F56205CC0DE8373369F16D7604E0688C73E79449620949F726579DE0DBCDFF6EE4E204A9D194E3EA620984288272F9A244912FE0F18AA030000005492FF000000000000000000000000807C2408010F851B02000060BE00A002108DBE0070FDFF8D872C470300FF30C70072B12157505783CDFFEB0E909090908A064688074701DB75078B'
	$sHexData &= '1E83EEFC11DB72EDB80100000001DB75078B1E83EEFC11DB11C001DB730B75288B1E83EEFC11DB721F4801DB75078B1E83EEFC11DB11C0EBD401DB75078B1E83EEFC11DB11C9EB5231C983E8037211C1E0088A064683F0FF7475D1F889C5EB0B01DB75078B1E83EEFC11DB72CC4101DB75078B1E83EEFC11DB72BE01DB75078B1E83EEFC11DB11C901DB73EF75098B1E83EEFC11DB73E483C10281FD00FBFFFF83D1028D142F83FDFC760E8A02428807474975F7E942FFFFFF8B0283C204890783C70483E90477F101CFE92CFFFFFF5E5F58890789F7B900600200EB318A07473C80720A3C8F7706807FFE0F74062CE83C01772483F90472248B072C09751986C4C1C01086C429F801F083E904AB83E901720A8A0747EBD683E90173C08DBE008003008B0709C0743C8B5F048D843094D4030001F35083C708FF9610D60300958A074708C074DC89F95748F2AE55FF9614D6030009C07407890383C304EBE16131C0C20C0083C7048D5EFC31C08A074709C074223CEF771101C38B0386C4C1C01086C401F08903EBE2240FC1E010668B0783C702EBE28BAE18D603008DBE00F0FFFFBB0010000050546A045357FFD58D871F02000080207F8060287F585054505357FFD5588D9E00F0FFFF8DBBACCC03005731C0AA5949506A0153FFD1618D4424806A0039C475FA83EC80E94C71FEFFEB1A56BE'
	$sHexData &= 'D0730210FCAD85C0740D6A0359FF742410E2FAFFD0EBEE5EC20C000000E4CC0310D4D503102C570310D4D503100000000000004000000000000000000000000000000000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000700000000000000000000000000000000000000000000000700000000000000000000000000000000000000000000000700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000ABCC031000000000C0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000040200310F8C102106D0000008C730210000000000000000000000000000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '000000000000000000ACC3021000000000000000000000000000000000000000009073021000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000200100000002000008018000000600000800000000000000000814152A90000010001000000380000800000000000000000814152A9000001000904000050000000A4E0030058030000000000000000000000000000000000'
	$sHexData &= '00814152A90000010002000000780000800000000000000000814152A900000100090400009000000000E40300910000000000000000000000A0600300580334000000560053005F00560045005200530049004F004E005F0049004E0046004F0000000000BD04EFFE000001000A00E90702000F0002000300000000003F000000000000000400040002000000000000000000000000000000B8020000010053007400720069006E006700460069006C00650049006E0066006F0000009402000001003000340030003900300034004200300000005E001B000100460069006C0065004400650073006300720069007000740069006F006E00000000005B007800380036005D00200049006D00610067006500200053006500610072006300680020004C0069006200720061007200790000000000460013000100500072006F0064007500630074004E0061006D0065000000000049006D0061006700650053006500610072006300680044004C004C002000780038003600000000004E001700010049006E007400650072006E0061006C004E0061006D006500000049006D0061006700650053006500610072006300680044004C004C005F007800380036002E0064006C006C00000000005600170001004F0072006900670069006E0061006C00460069006C0065006E0061006D00650000'
	$sHexData &= '0049006D0061006700650053006500610072006300680044004C004C005F007800380036002E0064006C006C000000000034000A00010043006F006D00700061006E0079004E0061006D00650000000000540052004F004E0047002E00500052004F000000340008000100500072006F006400750063007400560065007200730069006F006E00000033002E0032002E0030002E00300000003A000D000100460069006C006500560065007200730069006F006E000000000032003000320035002E00310030002E00310035002E003200000000008800320001004C006500670061006C0043006F0070007900720069006700680074000000A9002000440061006F002000560061006E002000540072006F006E00670020002D002000540052004F004E0047002E00500052004F002E00200041006C006C0020007200690067006800740073002000720065007300650072007600650064002E000000440000000100560061007200460069006C00650049006E0066006F00000000002400040000005400720061006E0073006C006100740069006F006E00000000000904B004F86303003C3F786D6C2076657273696F6E3D27312E302720656E636F64696E673D275554462D3827207374616E64616C6F6E653D27796573273F3E0D0A3C617373656D626C7920786D6C6E733D2775726E3A73'
	$sHexData &= '6368656D61732D6D6963726F736F66742D636F6D3A61736D2E763127206D616E696665737456657273696F6E3D27312E30273E0D0A3C2F617373656D626C793E0D0A00000000000000000000000000000038E60300C0E503000000000000000000000000005AE60300C8E503000000000000000000000000007FE60300D0E503000000000000000000000000009EE60300D8E50300000000000000000000000000BFE60300E0E50300000000000000000000000000DEE60300E8E5030000000000000000000000000000E70300F0E5030000000000000000000000000020E70300F8E5030000000000000000000000000041E7030000E603000000000000000000000000004BE7030008E6030000000000000000000000000057E7030010E6030000000000000000000000000064E7030020E6030000000000000000000000000071E7030028E603000000000000000000000000007CE7030030E6030000000000000000000000000000000000000000008EE70300000000009AE7030000000000A6E7030000000000ACE7030000000000C2E7030000000000C8E7030000000000D0E7030000000000D8E7030000000000E2E7030000000000EAE703000000000004E80300F4E7030012E803000000000022E80300000000002EE803000000000036E80300000000006170692D6D732D77696E2D'
	$sHexData &= '6372742D636F6E766572742D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D66696C6573797374656D2D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D686561702D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D6C6F63616C652D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D6D6174682D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D72756E74696D652D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D737464696F2D6C312D312D302E646C6C006170692D6D732D77696E2D6372742D737472696E672D6C312D312D302E646C6C0047444933322E646C6C00676469706C75732E646C6C004B45524E454C33322E444C4C004D535643503134302E646C6C005553455233322E646C6C00564352554E54494D453134302E646C6C000000005F776373746F7569363400005F6C6F636B5F66696C6500006672656500005F5F5F6C635F636F6465706167655F66756E630000006365696C000061626F727400000066676574630000007763736E636D70000000426974426C7400004764697046726565000047657450726F634164647265737300004C6F61644C6962726172794100005669727475616C50726F7465637400005F4D74785F6C6F636B00000047657444430000'
	$sHexData &= '00776373737472000000000000FFFFFFFF00000000CCE80300010000000A0000000A00000068E8030090E80300B8E803004035010070390100503C0100304001002040010000370100303301002034010030380100A0390100E3E80300EFE8030009E9030020E9030037E903004EE9030062E9030079E9030093E90300A7E903000000010002000300040005000600070008000900496D616765536561726368444C4C5F7838362E646C6C00496D61676553656172636800496D6167655365617263685F4361707475726553637265656E00496D6167655365617263685F436C656172436163686500496D6167655365617263685F476574537973496E666F00496D6167655365617263685F47657456657273696F6E00496D6167655365617263685F496E496D61676500496D6167655365617263685F4D6F757365436C69636B00496D6167655365617263685F4D6F757365436C69636B57696E00496D6167655365617263685F684269746D617000496D6167655365617263685F684269746D61704C6F6164000000C00300140000008D3AAF3CCC3CD03CD43CD83C00D0030014000000D43518361C3624367C369436000000000000000000000000000000000000000000000000'
	Return $sHexData
EndFunc   ;==>__ImgSearch_GetEmbeddedDLL_x86

; #FUNCTION# ====================================================================================================================
; Name ..........: __ImgSearch_GetEmbeddedDLL_XP_x86()
; Description ...: Returns hex string of embedded XP-compatible x86 DLL
; Syntax ........: __ImgSearch_GetEmbeddedDLL_XP_x86()
; Return values .: Hex string of DLL binary (empty = not embedded)
; Author ........: Dao Van Trong
; Remarks .......: PLACEHOLDER - Add hex string here to enable embedded DLL
; ===============================================================================================================================

Func __ImgSearch_GetEmbeddedDLL_XP_x64()
	; This function holds the hex data for ImageSearchDLL_MD_XP_x64.dll
	; File size: 52.5 KB
	; Architecture: x64
	Local $sHexData = '0x4D5A90000300000004000000FFFF0000B800000000000000400000000000000000000000000000000000000000000000000000000000000000000000F80000000E1FBA0E00B409CD21B8014CCD21546869732070726F6772616D2063616E6E6F742062652072756E20696E20444F53206D6F64652E0D0D0A240000000000000097C4F37AD3A59D29D3A59D29D3A59D2940EB0529D2A59D29BCD30329D2A59D29BCD33729D8A59D29BCD30129D7A59D29BCD33629D6A59D29DADD0E29DAA59D29D3A59C290DA59D29C8383229D2A59D29C8380629D2A59D29C8380729D2A59D29C8380029D2A59D2952696368D3A59D29000000000000000050450000648603004376F4680000000000000000F00022200B020A0000D0000000100000009001002062020000A00100000000800100000000100000000200000500020000000000050002000000000000800200000400000000000002004001000010000000000000100000000000000000100000000000001000000000000000000000100000008C75020084010000E4730200A801000000700200E403000000000200E01300000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '00000000000000000000000000555058300000000000900100001000000000000000040000000000002C461D7400000000800000E0555058310000000000D0000000A0010000C600000004000000000000ECD61E7400000000400000E02E7273726300000000100000007002000008000000CA000000000000ECA61F9A00000000400000C000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000332E303200555058210D2408'
	$sHexData &= '0A95FA09AE84C323D3DC47020016C2000000F201004905002CADFDFFFF48895C2408574883EC20488D05D75F0100488BF98BDA2A010E49BAAD7DFB08FF15855522F6C30174091CCF1AC7553FC777B36FCD04663064C4205FC3CC1634FF251E30BEE67EBFCC014053978BD97FA45450C744EFFF69D70800007E6785C07403894310B9182305ADFDB91DA3241F8B5310624C504C0A5196EDFE2B8948088950104C8900CF5BC30BBFEDECDB26C2BC0D0D570BD18BC8D7D39E9D79D354CE282248100828B7B53489018605E15E99D81D1CE194936E3FD9866DC93636F7523F7F38C3E106AF5278825C8DE4806C76C9523B53EFF012DB01C200A39228A785C9741930FB910D3C76510E4C8B00BA1400FC5FF03EC641FF107E288F816118FFF3FFFF8149BA45D0359E023DC1BF3FEF1919424620F9085A606D008330042B16C4C3E0822D375556E681EC6024E55FFCB7B620FE82FF2C58100F2970D80F28F13C33FF89C96BCBC37818783C444016387DF3C5CD508C24D84B9D4F90C784249066ADC5EF9202224533C09758084C643C97EFCE946E9A7D63480484E55D0C226D69BA404158452E4D743FEFFE7089BC24C8900CD08D570253286EB05536FEB34EE81A15E95A29991D2A9B507E4FF0C50A48C116DBA6C87803FE125508675500967F70694DCECE4E26D32BE8050094DF15E7825C65032FCB2E'
	$sHexData &= 'C1DCFB8AB3EF4983C8B650011F922C527B180872051CC57F647403EB0303CC66837C50FE2E75885F186E6FFFCA5B5390291A93BCEE77300D43F36F729C2488D24AC3090A22B4700EFAC4F04D89E25F5E5D613F8D99981D1DD95B2B9B0EF784A47C3717C79CDF780515CCCA33C497570D3A8560CB206820406AEDEE16974C340D28BA64ADC1E81E480CBED5FFEE80B832193BCA0F42D13BD00F4FC2807A33CC28B5F7DDE6012A6CE078C3432D4522C78BDF8079F4C4EB815B74DA06E210C53F3CB6BBFAE60ED783F8027459E40A1646C44B37C7B76F448B1DF5D60C4585DB75336F203EC18581B912251C8133A4C7DD6CDD28093068FF50C705BD40D7AECB769E56BEDE504B48BFDB0ED9B02203CB109A33C9115AFBEAAD080804240FA2E404894C1A89E00C2513FC0B2A017D1AC605A4D501D1100CBE61AFD87304060CB81FDDDDDDDD7CC1EA1A8C8980E2018F8815667C8FFE4EAB7BFFC10A83F909775E6E51EA79F4F73F63C18B8C821016D803CAFFE1F24351912727AFC30E7BBB3352DB9393936B9BD30B534FEFB5576E0F5B6690FE15B4F60606C4EEF33CCF7B0EE606DED6CEC6244028CEBE7F305B407A5DE6894B081B40285B10BDEC056F0673208BFAAEF133DB411C1B2ED79FBC915A30162A40498D4B801254266464934B6048308D3C39D9644B30DB593017694BD341483B2E4370A9'
	$sHexData &= 'CEBBA68962B8900CC09EEA6C683E14059CD808159A9DB6018799F70CA88BD782616D1576DE143C6F33EC1548AF49BA3351D61B3003D26A32608BC8C83B08FE2E2DB7C64C8D184D498B72A0D90C1C067328E30F3A1F5E079F18569A5002085F9B2C0C7CC8C4402233F61B36BCFAF439356D5E8974D47427E24118BBEEE0A307CF7110404430ACC9FFC2F6C10F7C6689310ABFBCE908089AF7D1067743B90401197048EAC83D021869DD03EE777C48586868BBC04C7C3D2DF89F5766F2AF48F7D16041FF4EEEF73E0738189ED93DF916082660B6CFBFDA0E480F430DD233D2AA024918B27252184373B7CB335B0A06041E3314DBEBDC9A79B1215D085155DC18D3C103F18CF333A1D2700D10701E44C45EDA8B30115F4032415441554CEC6C47D8AE9060C245C8396AA0004D85273FA32AC745F852DEEDFEC37FFF44896DC0418D7507F44C396A10751D94184C896910667D985CD03229F8157150900A759BC167B33875E86DE06DD0B9424DD0013AF007AF904D8BE574E08AF09D737FD274400F1F8400013B280645E8F86F7B5820F80862CA265DD00EDA420FB70C6177EFDA6C70794A6642246349FFC47F4C3BE2BBE75DC772C832CD8F4841BA5C4C7A066FDA1D359266C245EA8E49C166833CDC83B5C1482F5C252A14485B98024DF7916232C1429E6EAEF214A6776F2F6128F874685008110A'
	$sHexData &= '207F3A3E0F0F6CB0530A9B497860F9083530D5DF731C9CE04F8D442A6E8FCF87CF3EA556073EEB0B0A948907208361DBBCD014E04710064F1849D25E7BDAEB049675FED40A6AD07E2FE06215EFC718F8AD41542CC21FF060415D415CB19E53652AB453A14156FCAC6600E4C579F2D4B028FEA656C5BB30ECF585A8758BE97CB8F11E4980FB2B42440FB6B5105E0816E2E7E2B4205A5008CC90389F55EC3AC8439B554DD89C0447A74218E0992BD846CB0339214F55AF46DC433BC8B7545DC804D080504583F6AA0D03F8FC8C5F10743F8ED726DA4E439078B64E4761707BEFEB55785F3A83BD901AFBC2A99A8D788EE8FB8090EFE3032CD1450150E086C34D1B6401BE418BD5BE8146A33E87B486B1353CD6366D87A519BE251CBCF5C03C56189A63C4F84F58DFC8E0EF5A3526085A441424F30F108D18491EF1C7163FF7455CD600C3030E903BB6047FF8F033C9174F837D68A58BE680CD5015484F048D9063A56254E7E2C4415E3E308598335BE740F2803BE0A84630D501793FC3831EDF207023F923281B0737FC38BA2233C96042104DAC281207506D104A18685CF9F002D5D35F02C240CE486BDB1FB218FF54D8D890493BC872DFA66853C571481E6453C944754EC80B399D4414537444C08120477C52B80F4C9E7B860114D0CEC53ADECA4001078CA2D0AF8F769507F2D34B14F48EBA1F'
	$sHexData &= 'DA08CB65247EC6364A828BB043887536BC02ACC84C8F01E50EE1BA714680182BA9D7834175BAEF31B1C78170E36682798C93800F40504CD5A8DBCB38AD7539E4C8F068C1A0DAC5F62F71D7FC837910000F84FE11449E61AB3061351816B6F61FE2CFF1375C284532ED44886C4E2A743933FF35FE27D1DE5C83D3E26BD26449D4FDBF1BE33342DD133D802C740CFFC783FF037CD9E4FE77C7702041B5017B4584ED7505E9A9C2395271C910500CBC576077FA1552ED1059E6FF760634C7E209C8EBDC885C84B8E01AB6508F4CA42468EEF6815F2283FC36C63E3C48660079E171077FB98926304ABDE8F29A2640B0C93E340766390269DBED0EB883C202ACC9EBE6B393ED2BB0AA2B5ED1DE7CFF6169D2CFF70AEE47199424784A6D92CC1964EE5C3C29DC907F234A8FED2BB029410133112BC06B9B97786844EB1B93C28C50F061FBAC1958EF023C0F9CC93CAE15B8F94541B01CCBA00D7ED3E0E1742ACFF047BF0E4EC8F4F527C8230D9E765FE2241636AC90E5E872609A1AC720AB5E50E4D7F45D4872A3ACEB8B05E6403958DA3B8C83A08BCC68E0F39D720F8A328143EB0BC743B0708EF6948C00D1FDD1E3740E870965603F9ABA11E05CC8A0F075A6495F7AC3405630695CDBFCE031C4B9A88B7C244087582EBF9595828E50964F843A258D8FC9AF2DB8600C994D8B051740ACD09962CC04'
	$sHexData &= 'A8E0305F062B2DD9B3407E03A16D72A5706C30067018613A25D90810BEA2991688F1A013AC1B10D44D3EBBE91E54CE08DFEC4F280DD1C9267D673BF43BC17426501839FE1C2EB9C90EF8133B937807A628EB0AF4DB766BF1907C533F5EF974620E5F40101DA94B75A07D4E7448184B9603114957CA05AC608941084F6C7BA07C28374B10F6F941C1B5D48F4E289089432066C1CB780478A534DEFF0D67C93320E80E58F92C6EA03C63C76D3DEF81BDAB1FE3385513BE1CEE5C9C515015168E3E23481F9F69E40C25852B991A0DAF814C8DB8707F2039C21323FE2200867CB809B020C4ADBCCA0707290437B4C0C69C404EC9F23C76C85EC89ACA25835C5C1BD9EDA2872D6915436A40AA388907083C43D334ED8947040A4008440C20853BE34E44101E96D060F37DA63943406F327B4077504F0091F9D1EDEC240838181BBE1A958D57108F344FF3B8EA8B47385F0C3C64407C9F4EF36844085B40C2C8C7C9A98A1B8BD8744DC71540CB91A722C26B6BFC3F0714DFBB8594A2760D181638151AC228A608BADC56B6F7CD9C4CB99124499204044E15562BCA53F970318E01730ED2964AC8FFADDBDACC98C2C01535403B40084C8904335CB618B41D1B1BD99F84373899E328250B7212947CABCCCA0BC5153BBEEB0F20FAC45857F7A90432C0557015B59095BAFD648000F2B189701E55E27E68A1'
	$sHexData &= '6CC017E845F545BF3177BA6537F1F20A1E68C8E1DC8BBD3A233F9EDC68498BCC45AF6642C61DB8FC912937672AC6063ADE87DDFAAFEBF24DA712A761ADA2E27217520724387C040A3C1F4DD33408400C444E023F8E3C90C61A3B45DF4DF60F8EB61AE4D76606C7494DC73DF6781C4745EF4545F3434581BC638FF74145FB3FA8C59167401EF53D7373E3C9CBCBDC3D65C556C545C7537992A13DB80D37C5C1CD65E548F87748A93AC10D7734ECB816402C5B088C433DFCFC1D1996C4AF3D7DDF080F82F2F00A3223E0E910352C170F6DF21C0FFF6ED42AFF56563E2CD44527452B452F39883C5346C8977CFF9513722078C4B847EACF13F2903957C4833D446407E8C2C10F864615C44427145D10BD29DBC292109F8415A3229B93E4B0CFDBE455C71AAFDD8C6A9437E28AAD457DB77EDF74569E8078490378AFC000759848108079121E36C4A79676B017327E1FF776799F74F4EB204E08137E4110750D3BE5C7BBDF492574ED193BC775AD426EC3B74FDA76180F85EA193B10F88B5808B8FB807B3DCEC93C54B9BEEF0D1D0F3CB8356044C84CD680309C5C362318B43B3028BEF7EEF2189B318BDF807FF4BAE4FBC2DEE5B9BC332A14F00028E634F85E452E35E314EB65ED607D38BA1406D3496410011A24F94C4FCC561733C1C3E51DE425460818FFEE637D764B5844D647ED45BCED082A3A'
	$sHexData &= '81BE75A0F085A9E4E9E0DD4C46749717C8CE3AB53A3E9C1FA75D1AEEDFC72E9D3C2592475F95C28718FCD610C23F4E33BB1767FF69EA02C098D04B2B0C3625374D344D78D6ED522830638014C2FE5D1B5D1E5534507E90BF3568F80C379966ABC14982493BDBC48FF80A331FE81051C06953F185FF74415A2912487CF0C011282C07D152243C16E700AA620F97C4396365BB57A837B208061042C2B03917BFEFE049C1BDBE75A9D24D891BC2342BC43D97C10831143C220C79704CB3D2FF3B9F643999DBF2024B39803DE1C03310DD3392B63010675079811C225F0DC004399013F202EB38D3BFC5430EF2C8BFBEBFBBBF5B16EEB7F22CBE5E34191FBE870348F24E9C8B63893454278A5FB40853678BD94083E1691EEE0F4101CCE997189C0C0B4F3642030757C209B5493963F2AA18BBC3CD5BC31F80B3B3112A0D6876526EBD90279BE7C833CFE40CD4BE31943C19CFBCBE1E742E390AE2539E109938EF61051A1358792C6608408136A00136181769C27970C0184604BFA00897268C96180891BC404E45AF2C371337AF646543D1B4BEA5BEE0CCC8427C2B54550105D085BE90837C1EAD41DF3364BEADCEA0BC4ABE57B1ACBA43CCB1F26E0DBC3F322F38F41097AA915840C877C38C8B86D8DDE3BBFF32D3698855D24068DF56E68A63C64540EB188CE690CF68208AB178EAC9BBF1FB4575'
	$sHexData &= 'DBCF31FBE40C46C803E4DBBCC3BCC2D7812A8AE40073704013E86B34546D05701C1C252E24A72A9ABD478B6635B2815AF34C8963440818234C35E08C03B2EDBCB27C9B3BDA3D6BED6867642460680A50BBDFE198404750C11DA3831B2107921E034883BC240F5939AB3FDD3162BCBF30393948BCCE423ABC055CB62322644405257024BC1058DE5618DFE1AC2734AC7089F09E38044906698554ED491A84C9408533E52536435458FFEFACFEBABAE4C841213C213C5D5448A38E585FFF218A0BE4BDBA3363209BE3C29A3301B982BA48FEF2E477BA6DBA4C89256ABA1990016448322441852306171EFFE82F8C635F015E10F308F6A7DE5C85DB743AD4E943C48B014F03CBFFCF32FB803BB8EE23BE0806FBBB268A0ED9859054BA5AC1150E21A785326B324D87BF6DB30CBAFFCB361D02B305F3B92CF642D1BD0732F210A8FB9D201FD7492B8F22B1B7E32DA2AEC253DAC96850CA1F1478D97C99EB7B808C306B3885DC72D47B3F046E03D284507008AC2488BAF00D7451E2DEFF42783FDF53F9E43EB8984DA2FF0381E300ED702EB0FC683E0552B8E806C8B1001FFD40387841751D1F4C391073065EBBC57B1810EB0AC86E3974E87F1EF83B807800740D4C3B5918729FB0E0C1C778EB0E84514861F4F003BD9FD81223FEA0280A05D81A5A201B3CE0F0AB5C1815589B6B386B2074DBE3150F'
	$sHexData &= '1E4DE06F1731BB7DE0060ED404CFE8F0CE82E02EB7B5C31A542C61D4899DBC443EF00346B7047E0806E41A3BF0745868240C82DF065E82D39DF146081138A04C1DF1080316702E0FA8DAB9BC8B305423BE1749A52BE0BEDCD533D022A20A910ECB8B8450C2922F841F3650AC7C1BE42BEB24FBB8BAB6142C0B7E7E7F5D95B60C2B1D8649C1FB206305841736DC8BAF2264736860BCE10AAB55115BB5ECEEC71BEE1449F22FCF4B84346A2ED8F4C0E0E8C3742C10838946AF951D7013CA67531450FA789EE789A61E10101010431C7C392427592B019318853CA810DF5F2FE3DB4288C7FA114C6CB9E8DF172CF8F64D2BC2E8F802DAC1490F42C80C760DD4D136C333D2CC70F7F146C8DE417F0874E162ECC91E1C4A8D148D04FE7FA700058D42038038FF720E4903C94803C20276EA3F06F032C0C3B001C3DF54C20567614157B26C24C0112D809D3A45B0F7705081C92E8A31AA76F2141E846DAA24B6B5E2F665E5B379B455AFB5692A9AB5C0FBEAC32F0E7801FF75800688C2CA4F87A5BFA6B4A82FC26F908FAC66B5678FF929B61403CE69BA081875097C9ED0E0FADCEC201C59D9FCCD4D081E2E45207518FF1378FB7508571C7CEC3A8B4820894E47EEDFB4200A24240FB64828884E289F4336234FF19C4CC64640A2F8E0CC2D8358A5A530182F377B041A9DD5B22E03E8A9A2FE84D7C237'
	$sHexData &= '3703028D05C438F005348C2232004C48EC35C097F1CB8D68412C74FFF5208D483E08EB084D8BEE4958A7A0F9174E6DB84D85ED75104102DA2301CFB6BB90DF24418B5E45892EC6200D58A5597D3806492EDE6C85705DB6FF105744DC34DC3B1F563449114D6E04E7273952E9373838D12D4C63DE155FB064646E4844E40F8E9415E1678A255E5C174181FC007D7469BB4ED68F1981FB17D5407F02968C02340FBAE012E83E11FF720C3D0A202600C68A75050D01101AD035D760D368CE6C9DC809E87490DB565141461A180BA547607D9820504618DF6681B0E8AF8C5E248BC3410FAF1263F8FBAC8EE2E6C022AC909F9BE32A7ABCC92ABB012C8F46062BB50B2701A87C36732A845CCE0A0C90E3D32C041EB1002EC38DDA70039BE219452E69458CFE459813C276588F2C7DA038961002E1FE40DB9C50BBFF88A403E6E7D42509AC0E1DFD3C15DB2BD53F3392FC040F8C842D3DF05D95B06AFB014D8D5F0822C4FCB0EC027EF7ABB27A468D34A56B7A3C0F41BF93FA3B52083F384BF88BD1C1EA1803DEB46FC2C941C1E9100CC1E80884D25EB0BD9DFD2A31413AD7118BC2646EC00F5B0428D4B6AD4258005ED04CC1231C59C2A779DEDF0658C3F3440F2CC82EC0C8C971AB559ECACBC15D5A54EDEEDF58C979054532C9EB0F9F453BCF0C4FBCD25C2ACF1430C0C0C0C75FE4BD77C060045E0D'
	$sHexData &= '5C415AC1E2D86DEEBE081E0BD00E5D11238D43BB6E8210266A0689144E8B43AB7C20B0FC0672D8458BC8410CBCE509360A0845D92B19451041823B5815DB7CD02AD7ACE4A1852AC934C0B804121474D2D772CA8255D02C78D745CF9DF96005052B2ED2C518A4C0ADE711CECB03429EBC22CD2C002C000C48C580A2010421830C14D5EBEBB21F72410283C30449062FD76FECA2FFCCE6296C323D6FE065AF80084099D57FFD2F4B493BFC0F8DA043478D5435BC2AFB23E40CBF2E0C80FAFF1155572E8ABD0E443BCBBAC3759A6E202FC0C341D53B57B90B57B5C10ECDF04963CA35419CFFC7EF017F78800A4B154C03F8833744DA814403BC1F48663E18E041BEE37F2C30F0904781586B1D27807C07FE023BDA74114B255C84C0A95B18DFB0750396C6884628B99E60D6DE4166442530681BB83004A55004DC7008065E409537F0F6AD660E280001EB03BADE8B4620349B77C1340A245046288843D6488778E1CB8BCB4329005831D8A90A9BE01E6C3326C727091CB8B0B96F252E57217CC94D30DD9040022B207E5F415E8BE22A7150DF5365A881B21203866898022020B606E32F8F8CA8067898F848982CB27CA02E4D8BE933362925EA14FAE0444E82E20DF18BF28953F98B4920983564863FB03CE78B47244815E817DDE2FFBF044F084D2BCA49C1F9020FAFC148984C3BC820362E1C3633'
	$sHexData &= '9647455981FA4DC289B4DB164181F81940EDFF66A114DE3064761F48B815AE47E17A140849281663C1F7E11AC14CA4D1E8FA6728C662B0E80682B863D64D112EBE0A743F4E8D1C8539FFFD2FFD030A81C1B979379E7CC1E0064803C8B8E35FF913E80233D94903D04D03D3B8D172D513CFA6812C40930C64585840D89820A36D3BE4F38441D1BB2187175C318F545E3B55D6D0908B57207C93873474F2283524DA801CD259DFD6323F5AC19243BDA4D43424ED0B2C568D95F96F945C90208D911962288977E2DC1837464D8A0C8D41884DC68F1E638EC83D41C64573E9593890BDE92E8119EE19DBBCFD2240852DD68CA1F047B5364CB5E3B69766618DEF1764BA2016E33B33033CE930049CC8DF2D414F1CE38AC9B2B11277D0C6E41CA1F8E038A8B6066642522391FD5BDF0CF80F895D082F954E883C1D041A54C83B633220027970232A78BE57C959193B90B351732AE423E01FE9BF945A893F75D833C98BD9423D5F7C6076FAF90E3553394F24173417A2055ED0D6F1886DD041BC560890A8756CBE63DEBEDE24D1ED5B4074B706AFC3417CEF078BF30ABC00B88BCAC1E978C241C1E8EF2CAC40CA860884E333CD413A190A88B2CC11EC79D7E72042C58EC2B5C6700EA40274C1CE8A10998AB9C690D251A2357FC2413BD4C412D061DB550977CC2F61C43133CCDDF7C0C1E108820BC80E65'
	$sHexData &= '11C01441810EE40B74AA04D0F0023F2C443BD07C32FD7EFF8122FCFE00F53B5F241FEC44C630E24442735A40C8E76E21E5C78ACD93BC45F82E7D1092952B2D3D3AE390335F589758D0705AC71C9734E07C2C3BAD09420ECF88016461A06E579874577598554021E6FBA8A3C7EA5E8BD84C3A404C767BEFEBF0BA07933021F9F6D8BA0225A2C929192EC32054500076A0F85C2808D8EF75342806C90C8742517446BF979274551965202B139B39813C8A529120791FECE84A03E085570CBA877AE44122329B8970F778ADA4410D0B7C60A83920074798184DD81F2F9043060E04EC1EAB554C254556200665E183B6E7CE4745068BC7EDD8B46B831EFB18EDFD4176231FED188AC22E057F90ABA82B60EFC373659C5579A03430575C6B2208FEC80E5813634C1F6F7AB14A263453CDB213E914657D80C33C1158632F432138BC0575B058A8E0EA8288FEB8A8DC65C4444B21FF42E0506FBE858AC8FF473ADD418D5B5E08498DF14D44C177FC8AEE54C6448D24B5DD28F423853CB5477EE4C8A10A3F37D137D109E40099D7C629480E90CEC650B0C410B690D6FCA9310A1A1241DB9F042A78CB38BB1041DFD7588414DCD1D111402E95C0C2C7291811AC9ED9D2CB27070A0E0BD139A4395231A090A4AA019303050704D93A91925C50C83A910295FA3C854D48FFCE370C928097858E81443B4B462D'
	$sHexData &= 'F42F74493BFF3E3BE7478D142362D987900B46BE3BAF3AD30F51685154C345AED086042F47FF09FFA1030BEF1B4C03F64403EA801D56D844893FFFCB6DB82758E1154536CC31FF7F844301CB76193C8B4DA00F45B1720AD9C8890A17B81733FFF376C28EA6178C3C78080610C00CACD02013940BDD420610140C162CF94207444518498BD536638463E31E573CF9818520B0C918FF18200F9087F318DB17477EA02014BDADBD31C14A35F142628D68474673E1119680800EBC5668A1DDA50A798B09F032D1BF2A4C2A9BD9E000DF44C0177A02F00C05649EF2D93C4FA2ED535D9E17133F01EDB2489E2D76A7A16DAF6713B0F24C896DB755A79E569DCC6F90CF5818169E178BA9120223ED2024E7BAE2E10FE35A815722D144BD5C91F13726EB8975778B5DEFE28E23F256E8CCD20F81FE9716FBF09C09482F9A17F04DC0C523E5A02B1AB6F8FD0E9E2106387FE27F33D2DD88178066E6091846243A466F3C899D17F54C322A533C3819A815ABE0460B9E3B7845896E1034851F8C8F3497894B2169C45D9B1ABF5F57D9BC2A8F498B4E3F1709C9E972B04134D1C840334E6A809FFB2AC517C4FB480FAFFE028AE1F505761B7060C8E5C8BFCE16049029C40FAF07C1CC9E95706407A915373540F00617D45D7745A6F1558C0E30C7BA7F9ACF41E6000A02872D30B903B0CBF226E60E224883FE71'
	$sHexData &= 'A1295F434E418D5E22C6DB800348C92C1FA0428E3C40414041A30E6043C40424211A4170CB412A10DD908211DC7FDAD04518108CA95BC1A2035821D14F5E111C03CF42124212A4624021A60141060A0E04D7FE30B9A090FE02BBD01CBAD93F7C87C49F4CF897EB0BE15E3F483BFEE8009A5FE044532E1F80428E1C441C441C187C61C32FD9EF3C008707085B131FF21CC01B3A440375771B3041BD8F025F783F3ECA387D0F114B7F81211D255C4188644CA59B9A1806BA01C31030D8B3D1CE14E80D2B909707408AF620E0003B8897B47EA8358B2A05C5203826EE519EB7796FC9492B11202040040E2408B7F0E779280C2C48B867BB00BCA03F48F7EA48C1FA221C24B15E523F4803D0FA4D0B23BC1648B72C8CDA254D1A81389DB00AA75B3F8BD992FD948E42953EE6FAAE0B1FB9F69FDD6D960D056E4D3BD875264C445A25A57641248EB833D23AF89FDF7715B10F474A551D3B85DB7E644D2BC3C027E4293F49F7E83BDA7F4215B79576F6E28B63D0A20492435C83D3398DDB89070A4483041E0F08AD3D8D4A080C27480A9D9702EA62437C4CA47778A7322E2FDA33E28818885F10A81874908E384E4AEC35420354CCF11EB0BFD940871DAD04E81EFAB67E266350F0243655C8457418E6D7876623AA584454785C2B46499BED7E2BCCEB35B94C6EE65A0EB94DBB52A55D72F0184D4EBF34'
	$sHexData &= 'D769954A3F4FF81832E4FD2AC17183FDFF7E1A038D2C3E83FB17DD61C416CE9E8D1C0CBAF68D145D5BDB9A3E4C0C42FF18F0124FF04CFCBD3DEFB74520FC4103CC8D4126F8F85646013BE89AA235AD2AE81047EA22EA1E1AD8AD2DE824444F822BEE04DFBA7E233EE28740F23C81FD867F3497C32D6DF17F2C47E50D911CBB861E413F081920F4749D42756ED647A933109F51D0E63C84C304CE4CDAE81E146690D58758EE5EE85B6EA7553420866B73348178243257B8602D189DA62A3840D9244E136212C66C2F6109187EA22000CC501D5370A22D2EDECDC24689A0C5D25A7A3EBC11DB5A485C2808018BD838495B1608F207F809F60C234ED3F6D9750E2ECD01B8F82D756D47A11AC50FA3DF56025A33B0E032D204D904226DFB9688324A1610C977008B2E2E8B722014040666043BC13B96DE13F22A5AFC602CC78591FD0B413B432429AA581EE81357C2784C68AC3008596B27E5704C9117C04AE2E66EEB3F9578B7BDEA85FF7EA6CFB881800069C921F7E903D19A4C4D8D98200C070A081F01F8BD99061AC42BC279690A41428DDAE3417A1CFF5D50BDE04B781FE749EBA2124D8B13DE04EC186A2CAFF104CB4503F0D4D26BAA72544C08105A0880E322BF98E446C70274C682FAC64DBA827F34367CD749B62BEB9F458B143510BCA02B3CB00B187D07B6F6EE7C5A1413AEB6CAE4442B'
	$sHexData &= 'C80FB7BD911CB82EEA106608D01A41C0FB2DB01D0045C141582BEC750EB7790341F7A90FD81C021A007EB78B5ACF7F6108C77F5C1058FFC30D1A41BF9C3BDE7C88458B638BDE24D83C8B4403BE00FFC703C6A95EA00D7C60F936492CBCEA93C3873E0503EE6848753084CC0312B3CCB559077EBA385E5BC334D706190F642A46D7DF584D0C2B0790274D0C4D0C1C2429550E14DD4D0C7878101DD4129811A4AF3E5A745AE950D782400F2922101C76080384C70DBA506F3D55C40FBFC71816B6C7C26CD4CC10EFED39C76DB3FB1061F60670F6003B4D08AC22BE15FAF1DD1C0F8F713761583FBBB7870CC90482C0D283FE038D4C16C7ED6B1F3F2ED2FEDDF30664950051BCDB5BB022662C6EDC26C40672D018F0DBE6EDDB6E3A66D806D7C33DFFFF627462F3419EB779DB0C9020DBE74CD468E516CFC1F76CEFD90660C5D526CD06F9D0E12ECFC9A77D26F9C21EEE65D6C4E085DCA73DE68663DFDAC143BC0115FF4CD44183C204ACC16042033BC6FA101D7B554B6C66D68E4CA849411E292ACE88BEE8448B1C13C8A393C359BDCBC303C99434EBCB448C5E782A3F3BB0BE1565D81E127C87EE248BF9B30EA6E8B7213BFB7D43F9E97A4CC3541A897AD1487047A2A7D29128284CF9E5E5CF733448EA7858B001EBCAC43AA5C958FF3F95CD4DC1F9D800970F29B4A7F464E524C000D082DF44F2'
	$sHexData &= '6B2E89A5082C04E9363CCEF66E8597443087148D9FF093D0D508928F5AF76DC609CDAC410F40DFA0A7EDF4851620413940205350051824BDC8F8912444383D398D380D458D47F715D80D42155C42EB0D41B806A810451650565D0C0A5B5D08AB52F99CAA3B4724374F86A217B74860B58F4C12757F5F265D17110A77D6CC88D0A477494EC38B4500CF44FBFB880F311C44883A803DC18CB456E14F32EEA44A2CEB058A483C0A7E126F708B4F24D203C3B89FA0DBDFDE786F4103C710A38975A7894DABE01175AF329A419A14CFC7B7201DD021D35587AEB727E19DD09FA9B80CDC0FE9F697CFB7FF150006F280BD873BE16720402FE461E4202BC6E0611654428ED84E0C1CD84105F3D92B9E554F774C9CD7FE0E4DFC49E0000608BF8017EA48543BE8742BBDD35C117F8DCC72384D895D00BE40DAE22AE2D11042A2FF31CC7A18B440205EFB1FEA8172C5502F483BF774319EFFC168445E503B59E8FF155805207C203C038D7BF8A09A355A867BE87E9C5378AFB16EDED775D399324AE9870A9496D9F3BC4C524D0699E4794D4D1B4D0C6BF1088F5D10B7F1021A144AF61673489249101C69B008FF1F0CC80460E60BCC562348894B06F655FE5C2143C675D249A14CEB2AC0259976DE7D000608FD1459172AA43BFC2EDF90CF3AAD8B007AD8000F0F33959F837930B4D98D4AC10E534918322E'
	$sHexData &= 'C055D6E688FF4343183F38B1D3585FA89013F26958285D7FEA41E56958D648CA62BB0C7E320C4752546074B87C547B8DA3E2D2669048C664C407DF51B1C0EC18A505FAA544480F44FB807636A923B71B4A4C2BCE4B3092CC238D8CC785B92C794830190238F3FF156203D50A79366E57188D770110F372982458C228322E60971C38859D1F1D58F7819BF8AC700985086CE5DBA653082AB8A84408409823DF7D3E488890783A0E9868A087939FA85823447EF0C4130EA8059EF24CF88DAE4EC698BA0C544966C08182607E3AC6AC6B3E58DB45A4685789CD3353164CFD7DB6385DFE153F1685D2750FDF1000FF820CC7052C89332D4B0F2ECF493B7A695D651FD6158C5C51399A606A0CD4EF1727BD68020D700245ED50F9DC953770548856A6CE14B8DCCB2415A7FD19F6E47D0367D85F1AE2145C2E61505B0CA646AB8201FE9041BD08DBB660C52DCE653658454013ACECEFA7948B6B8893840D404F6FF9CECB58E3FECCEB0C8705C069E789BDC80AD0D8484457D3499C309C58B90A0D0B310EEA4FC19E9CECEC55BA6C1046543C1025321C52110D25CDFFF6ED0E2FE07710410AC17606450F28E1EB0444E04F88CF7D5B58372BCC6242778EDCAE7437C84E5CF39005E11B1DD4EDA6BDEE103F0F4ACB6241C3EB0308C199DF77763905B36659C0643D4212BD6D9B2E20C7C0663A6ED0BA5BD2'
	$sHexData &= 'F34589EE122EC804680970E878976119E28866C7454401ED0DB2098E48044014660C309AD6EE3AEEE78BF74CB404D0A88586A669A0D8D40606052AB4FD9359CC144E20D4219EFF6F4321D3146C65508B75548B7D58412BFC8B219B426F262BC6EB33CB74DAA1CF5B68A6D01A668BF03F972B6FD01859F8B94F4C418D145FE80DF73C8D3C06C47A3BCC7C0CA08BE13BD9FF57A1C87CE028143BCE7C0A8D47FF8BF1247448150BC83018C8A814DED428367E1C3BC27E063721F0AEAC8955A8521C96587E043B2868A7B2CF32DE184E65A675096E059D5EC040C466D88BC144C8FE88A674D4E004576E3BF10FD795F2BAD8208F9A8BD64121EA3511F1463C2D3C63685B09FF80513D1C0FB6F87FCCFA6BA8F0057A4084FF741E4562DD94838722BB9DABF5CF05A23EDAB3FB002B7DA4BAFD3AAF976F82408006063CF91E02879332C10F9806FEE7E816FB4C4ADD41837E60007E2D14596FF9C4E70463C11B3901F170FC007DD604492EE655D426239360093895837CC2480430047B9F385CEB2F7A3340387D91C7D2E10934571CB639B26A1341C21F7AB7843E8B20B311FA509284158393D310EA0C6604EEDC2B164E7D1E835C2EADB85FB3B937480D50318D296DFBCD69193943741A2BB2BFD8ADBC2D19D2FD3D2BBF83C78EA881D97C122C450316022B3F9037FC24BB03CBD91F16357A5C315708'
	$sHexData &= '5DC3A8047526BE70832B4FE522107810780F1FEE447E164092E78319F900968243141616766812351789E9FC154AFDC3D471648A2F9BF0C8F9FBBAFCD3F8F232C8811007F8F3FC202F1E62F4587FDAF90288C9411E82FAD0021C9B908F77A10268537639A48996C0324E4C1CD24C0188343319EC21CD81A018E601D5E4208B0E562006FCF9C2017900F228F7FB599A9795CF09599A33828DF70EF1F2909782FF81047B737F323232F2FEA0A8B0A07902E84D8B191D63FB00C7D0C7F2BC810F1E36F1D119402719899DDEC9CAF3C0504C48085045951F473EC0486690BA8981BB90879C92F6D7F7ECFA5EC0292516105C6F0760801AAA5662CF5B6688B05B60FCE873529796F0AB5E2F7B4F5620D08005857814660CEEB4D64E685D098D694F0FE02F63909323789317FF25BC84C0400F94C7628005372CFA2205FC45DB38F2CEF68164029CFEC75807E4E5907058F9F9030079381E208CF7BFF2FF00CD841C0A97F9C0851B5C2823850BDF0C901A0E0E0FD7C5B25517BC28B0B9D6F5EFFB870C242330483000F9402E010BFF0050C8C37494F6AFFAFE004ED3A3E49FF87417A7740BE450741813243C4534F52A20813C49F8E664C746A738784D5D4C26BAF4908164E4FAD0E89007E4E5D0E5F7EFFD0028E4200778F5DEFDE16910F283F7356C3630790326C35E2ED8F34C51E0842007DF381C48'
	$sHexData &= 'C19B4DC2030D2107398196F4FCFCFF839047A1F6458B7E104585FF756FF5FCFE164139761475100A18750A1CC96025448B697E1FC64588A7194D8CC9CEC44A1D418BF4AFC0D4F08B383F952BB1C1C126FDEE5240AA790544E80B4170FF5AA8002FDC142E0933DB8BFBCF4E62E0895D9864C4343289580E28807D16AA892E63BB08D77E0508D07E64D0B44A5B62BCAE1EC9C98CFA4C9A6D4B76BA3BF90F48829DF8FFF3061820C6453901412BD78955302B0C341414140B4290783848CA2F61F9D0D17A100385C39750A856108786AC7887B8320347003DF2A1D6EAA61950DB459AAF51F1C31AD0636F9DBC837D348859204D46811A981F0D46BD06282B50886E31048DDA80B03C88FE6328A8D203C3303145220CA94D8690DACAE00203FD774830A834EEFFC33B5D347CBFCB5ED50CC38B16E2100957F333C43752107589CB386C2AEF81F82218704D186C4FE5A9782078201630896B284C13180A3470D81FE38A057BF841B8077FB2FA00401090002CD14E51612BD288D171247C8C0B2115F70114DEC13806EC8D0BA4D3757506946AE816E108EB0978750414FE67E4ADAC9D2A0CF0F8418AB043B6755C62D0071DCDCA9DC89DB84464053A14E7A60823B85795192629A7BC70A380BA1C8913768B2D6B618B6E43EB7D5DC9732B3970482FF8A826983032A0905D90F034170805F727509C908C'
	$sHexData &= '95419A56A89019F3195C43772A85554CFC602EF978A2CD0D4CF9FEE808EFBCCF5BD7125D481A100C187303F8AF01474C020F9DE2765C4DAE5A9804ACA2C09EB80E75E88CA854C43863E15FF15CC30F14485AC8F20F101577FF22FBD9A1DB54CA165AC1F3B86F0DD876215684DA6DC9C147D136DAFC0070EF36640177050DAE68492BC88AA5D6205D0CD60F849780F97423C8A2F7A61EBA6F6AB133FFFFDC5A91B4DD54AC1289F42708A62E833EC89D0E54DE18584073C9ED804BEAE0B7EB0C1551960B2A72FB3D04C891E648006A2A046A2A80ADE83645C62B45E00134822FBD0D127D0832403E3988CB211FC4733A820505E0162F17C2495560881DFE709A01691695F2CD02EB37F472489366E8E0D004F205B9FC652FF6D070D00499E419EA043D767068F33FF21D58046958BF1C9041F6C50474390C528165E5FB92E8D05F88C0ACD7F00F7C027439C8419E20FD58054005760881539860F85570AE4E40F3067895BB3890F7266F8764C1672DB6983890B7CEE0BD5E444804FF9C4C39957353F4012CD4662F0C115C747CAF099A9E58CF24859D956D30F2D20404700A9C80BC491346CB04884DC2E6EE142E002CDFBD68663B3A4645D0C8387BEC10CCEC3E4EFB1EC08F134C2752F8C1FD17C14E0AC4442BCE452BC40F88270037D81B2211362A03C03B449A66B6E00F8F1D04C18CD7B80442'
	$sHexData &= '6081B6AFA6FB1D01E9B365F6B0A09A4BD3CC000510C40400082A87D35C040C0D481F93EC8405BF300528B3A84CACC985183F0418F0D200E1BF77004D7072C8C5132D68301812A854F7E3EE0AE2501C73B7306FA87F8590024A0A018017FF34B82131157C84DB753EE998137078066968AC808B98CAA79F76A0885D83F8037C07DC11EC00BEECB9954983168FA7386987EB165848E7408838743001A46D307F539170457E45B28DF2A098CD8D69C08F4DD4E955CB39721908C80C4C93F55054467BCD745D413C294627EB50413F54BEEA3C3840ED7DE089F4C43001070B4E8B3479703401DDD34C48C828015A1D0E2180AEB08CD7724D2241C10CA2C8B701C6903576E4074ED3CC42980469815006A0EB7D83E9167BE8CA80D3531B086A82CA0AA01901FE0E38A800B3F4450F2FCC7257BFD5BA564B24C61303F1F3AE854B265E1145B0AD11F0EEB806A16E02ECDDFA6A58F20CCE73C44CECB5B67A47673B32F6FA04C149186065E48B73DC0A7CF369626234B9AEFAB12D7C4E9AD6C6D04D04AF20A5F953CECFEF69513BFA962B103B55FD1044F3CE460E6C12ACB8C00DC800A569078F85D8A51466E067845C4501AE8D51932F6C059CE04C37690E6255190A442ADF1A85DCC01FADC0DBD91E732B85812B49C788C341BFBC74593AAE68609C2AC3913677C91DC9E085190C8DE81A14557A4D3211'
	$sHexData &= 'EF2458B16D1E744161F5A7928F0A43E90348FFC723D669D162732597BF81C08A2F6D73561ACB8DA0AD80BEE8A5FFEB0B042B01BD484D7858E19E063B254854D08553232C41BB555ACBEC0662DCAC48EB04180A190A4DB42D5B70A558B95C5259CCEA7068586255277C097414293934740AB8152DB4839F7DB6741F21B6AF53165266E7137C9EEB094D0E4CF01D314D6849FFC1180C7860AF1A620186C02866864D40735A9083820F24627C3B2E3565A4B9DB9D9DCDDE39B483E76EBD1DFE11BF1599ED5E8483E688A4E8B04C2BFF49C1F4BF76FCFF064963464CA67E078E4C0F47F84D85F3528DBDEB6C368E30930C03E7D65BF3316BA43A20FBD62A34881344ECEFCB5323FE13C0D04FF48B5FF8A801FE05785001751308FC992BC2D1F803C8286600DB3311D88BD15B3E4D85DEA0E7F94BEE2DD3DB0139A4320F3157FC34F4E6A50DC820161732DA6103FC60FFC63A40493BF7016AFC161C544FE616549987C48085A57422690A8564CFEDF4C07B0F99690F19EF3548D3F2842033F5FBEAD8C37374A60ED04A8A9078F769DBC9C8BC17B460EB13F6870048C8C7EE2F6F445476A30116CA4A980702CF279FD7E024C9E005488BD8CC5EB6433E0806F8AC05B035AD42DE3DFE0C803D4A6E010F45F0DE0F4116F4D6E1B0D697E5B5E420C9E5EDF436C05E81D2AC95FD00EF8F0CEC21CD885FDDEC'
	$sHexData &= '358CD3D61ED244B2EEA736404D5B7248107634980DD91CC8213BACF2E490E64006009CD70E3644F1CDED77AF33D35CF2B54B9167498BD45F085EC381D65125ED05813DA419B45476EB353990439AD0395BA89A0339A41E40A003B239904325D8E8E321CD810C66D4CDF60358C56B2395B16D934F053977442092975D1FADD6C2E520064923DCF2180641A4F80A8DB4A3376290F8D5A4186E39F03D3EB04F52DAD1BD5E600B518DC1A848EBC201CAD9429BDB1ACBA804B94C85962A499D9DD20EECC001A15129872677D59881BC5692EFE0E49348799D202F40FE807D910077E1CA7304DC8CA72F0870BC5D198478BB1717BF18A87C4EE2C788720CFF15EEE015F25A27FB99A864B12EC8C04CD418E43C3980BC22EAABE111EAF2A221916DA9E38245AC15EBB6425EEB9012FD8D484B5606B960483CE3C89303C846E9CFE035E9C502E450DAE2912F99E6018BE049EA545E95757C117009503E9BDF0A5873F0087BE04543D04BC00AE8F97C53B05BA0636E9BC01A8CE3FD16F24DE03F36606490D5028CA15D58D069B8FBF9BAF7418D4BC8B38BCF871C10F833184830FA033F2AE41C837BEB736087ADB0D845D35F10D230D504964CD348DE9327AF305EDF3CE78D94A5ACD052DA857C3326B9E41007D7FC0F604D50137890F13426E453C0D23658782331550687C3BF0BE114021DB2583817E720'
	$sHexData &= 'D6715CA28D850F504D8BE8DDD1BE75BE12604630533866C20221602A76CEFFB212B1D4BA29BEACC57AA1248BDD16387283D97A487F47093810DF76282581A7D088659810FAD06341316459AAC6DFAF040EB7E5025A0772C59BBA03F175A208033C26F0CA83FA24CB41ACE7E1251E802A42C7E2DA76641D59D3CE23A0207083F1026639014883C102FEEDFD3F9649FFC875EBEB0C458BC12C4564C6EB03448BC54963A8B76E39D44E1442FF047305BB71673B32EB0928190F95C0B094C062073F01DB77A309F514928A1D6FE7E989F3B46F400748DEF70F89E4A3E0C0C948FFCEE42219900707055812230FAAE6285F32204731BD0505421E24179140E60F646803E909DBE56C8282232966DF09F572B1930C09757AED069B6027077C1DE106D4796E9ACACECDC1C9751106AEDAAC1D0640CAC5DCCF7241C70414002FA9C67CFC404AE5A4DBEB2223081010851B9C9302042D720949497E1A3FC2E2DCF7506222428160079F96C6C6D6BD55574D69699C89283841F8F0EF0AF8ED75151EC87E28415F5F5D1973FD365C05DDBB582817F55C286068F6376F563F61BE877CF6413BEEEB0F4FC060B4F6F5442BE104EA22CDEE503FA54AD2CDD803C80E811D800AA8E6C06866607FEE00DA0DF9E82FC8760CDCDAFFC1A51AA180EB568D442D0099F7FE98C6CDDBED372C4CDE10F67C41BAFC418BEDF6'
	$sHexData &= 'BE338C003F6E498D0C078BC7040614E010FE031864F66087D3885C1B7CD72603ED0491D6C4D3FC49C6D1D642C110480376BB20BF345C1B18DF1655006AC8F040006C2DE1A547CA70876A300A1EE90988738C0ADDC1B504E0DE7B60A5F163A3D22738307A708D77FFE16F2923FFE133D26AB3A0830859202083DBB90ADC42DA236FD63F7870BEFFDC41623BDE7D0BB93249BB93687025B4DF7CB56D30A2E2E0CCF308EC6C40A85F055C9DD4A09F57F19361BA82B171600868A63603300C681046186C7E5B0A78DCF94D8BF08BF21BDBA7AAE54DFB22CFF457601C282410D30610AF8983CFD1FE32ED3F1B045F9B7AAE58A2D2DA8D4F01BA89D3142C16C6F028184EA0960E8829F13CCF1BD606D7E7F7EF37D4456FFA9CD752EB01805FF769C0A6AB1819905EFB5A0069EDC06E1B411C00D724E3AEBE1C8895C0F8740EAB8C3FBCD2E68EB501B90525A8D50002BE129D551A127B898B55CB05610B846402CE45CE16950CC6836C77EC70ACD8E1205E8E3E93D8AC75CF8F3F30AC1FD2FCD7448D4250FF58E04E55608C28EF7A48D9CFA082304DF4EB4716FFB52305EF897513141F1E2E756C4809D29463A7FE2871FF420C8F6207711AE03BD87BFDD4541CC72787EB3C15CED27CF6760CAC78D94C5F823BDB7D1E61E626F0E7CC2BFB8D573C0CF8C13A18E40878FC4C5E78361372D000AC916D051C'
	$sHexData &= '195CC0197D7F0477C7437A7C45A78C7EDA4EB8161B227DC75726C2B5BE08C1165919D96D5D69FF0A50B81DED69F60B6D8A44F170A02775678A9107C55B10D90C15E926365B20DFD8B001009001846B42F59FC4009F480230A0427C83ACE8AC532023D944DC276300DD26B8002418674D5C4F471ED2D2AED570AD75C25017A06F80693ED7F808C6470801F1A6821AA405018CA761B2F21ACD50083008460E6E045F5B205622401A7C1870494D9B1AD89E0DF67A507B18708A47D8B2C28D381E2369B0371D745E653078A7609030CE48C24BE875829EE324657525C520489D121D80E3AA4EFBB421E45FD444790FB6145CC8C1E1EB02989830683409F88C405EC39F843C1837702C4420D05C8EE03A52BF60F9321FC0E9DA0C6208D6EF8BD8F460C17EF99765605792F69A3A979DCF5AF1E1B4E7D27A512A4B2769EC0CAE8F553D3490412813F289403C383DA1660018EE23DB9E9431362D1179A665164E3A8FA67C59DAAD1861F6591D87137EFB2BF20FB70A0404322BC87508E2AAA0D31A80EDBE0D3C448CF754AB8695C304B15E28B1BA902F6DBC6CABA85B811EC66D31E51E91B12A8C5F9FC669834C2008985E4A9A91469F201808310AE3FF8B52303951300F9CC0FF8053B1AADB7C082ACCD2B311A85CEA4E1E098F29C93E8266FE4D93704B2E0602F09D04F3A2DC05E7D4BA62656BF82E10'
	$sHexData &= 'EB13660E24F4D3744E81D33DA0B80A3B4FCF719A36B5BD3520C8E6A1F15E3C84EFC919140A2A7ED0D50F7CECC884040FCD05BDBF868BC3E9F0685DEFB98C5CA68F319E2481A5AD2E1917848686C7E432ED8217AB87295A1EC8C8651C3D0F75BC07818F39E938D7A51A5A640D28C3A51989EF551C0DCEE2C440D01F48A06C4F3BFA750EB312C8601085C6825DDA1D09FC2BCF49BE25D4041EC6B6143E12A4C1FA0407E83F05662BB4E04C9A30FE10D258E1B93A4413F3509476AFD358CDBB847118ED1B3535CF1A90E33004DBC05F90040C3C9332C04C5005F249EC91FA022C0AD4F272F65C486BF6388DA6C133A7082C8E1045387AD13CA73ACE988D79563FD16BD3D8434FB6D24707909AE6A71F5F5FC7AFB4F7AD124FE05E75D48F4C8F8BC5EB4E02DEED433E4732FD74394C2086906B865F9B6847E90B5379C57C6D3ED202E646C33F0D2018E3692E297BBD59B8440A049F2048E30035856441EA7EC90E9235800C7EA8523B5BC67AB01FE4F484E0728C184045861BF78EEF74FE85DB790422EB0AB86426051D0574464F9F700C1F0CFB1B3A500B83FE08CC17F49B66B6C6EB0232C0FA809BA62886D00A042E45E6EB8D81AFF7034532E4182C3898957E7A91618116C1A70803D645901164103444883E56881CD6FC54EB0D9A7413147816C6C686E6C2B78061CF4D84BA6479BC31F812B080'
	$sHexData &= '417246F044B90ACFA139486B84BF834238C82ADE814C404CEBC9DA323D72F14148D1C5C223BC1784AC41495740F0186F2F54584DA6A6CFF7B86A2D68D8CFCA490F45EB7F7CBDC6CC4F5F4DA8C5F7D021A0B42F0CCB6931D12B8BFF1F2CC4B4348D3C038D140E3BF87C733BB6741EDEFE7F6D3BD17C693B603C7F634452A8356F3185EFBB1F443BCB44CBF445732914334121413B0A9452172E4F3E8AC54160048302F6F0801E6C36E24FF017961C6870A040A6D0C3AB42C3D2E206E355B40A1A553F3B8A3A39151B424BFF023D21FC320DFA5601327C57C6F3190C64CCF67750A9ECFA75358D4828F4E0CED812F050E800B018CF70790078861C30B549D80AA788B59F04CBF39D26E86467A6850E8D9718C204B5D883087BE84B7721BABB3F9F114DBF3786192AFCBC1E377E949A2060AF5289758F346F04DD12B38B85BE1897B770D037D2A79F06A744DA8685D49C4934C7A1B958CAE1EEEED89C5103CE12EBCB44474A815D5578EF3DC49CC9066EEDD0817B8004004E2787E463092D725C4DFF959C4608D7C0004EC8A186DFD2498A55A764874148E1F3CA6FCD204F1E25445593C9703E7C86BE084D774F870E0F86C1DBA0D5C960485948C333864FD2DDDDC718267587B5E1E39410497D8F48A061EBC6F618AF066DB3AF625801F271CD9F0FE4849C82483A54A3C93C4A4E45811FCCF453D4'
	$sHexData &= '90409E43C822342232614583A0857344876583D08744A93C0C48878FD097020C38BA189F1048AD82200F1A1CE1D904CCE78AA5045F081B1EF085104418D650D6AE52180C8F3211B4C4CFCFBB25D3A0525AC7008B72B219905230A85130640AE6C2DD406C95F31B64118053525497E6C6D0040B248B8A04C7DE46223D708990AEF052133E48BC8270A5240704365F3120909ADF4EB500B3D420E1C2A1B840E780298B556BFB53606E1D292FE808692B9D09A1E423A810D29BF202600E45D030039D2388CB5765D302DBC707A700ACB52D8C2B0A4AD2C4106949C02E8F1AAC9C76006BBD4E50848574692929D3324B20AF03699F090B33A8509E8EEBBB0A3C43DE2096173DA469DB02471481F917B85640FFADEBF679048BF7EB053BF0C485ED705BCDF318EFE8E84C7B40FF9B7AE7EB07443BE024A4EFC0F3CC7042CFC8E4400FB6C5833B408544E89741C1E02DA8735DF828440BC012C41F1458B01988B4611B5C9CCA96835151C3A008671AE23AE01F08BCE81C803DB7E8C7EB14094B5B2C8D2BF503EB5875300927ADF187DE1AA44AD45888DB4CF2C7219B3C8D440371A34E07B517B0D5C4B39EDC029D090C74615001380420DDC257FE4038734975DF79815C6CFF50C800BEB75D02318BDF9E774974BACAFB4F8D0D2CA224895BFF4F0E6EC8204F143C04CD37F05C0E90AE580745D41E918C'
	$sHexData &= '538D6896673DB9900BBE9658FB4CE7022EF45B3027AE16E3BC5B02004E9AD01402B6C8D749703FD91792BC9A49E5A1A4FBD0603266FC03D031806FB22770FC2C6FC00063A50B3B55AAAC4D4198ED4078031023AEC42F0996CB6C71807355F7B3CC0F7152C35F8CAC213B7287CE31DB0C3DC727C54A7372705829EB0286E767B101F4954CEFC200816303CAB04EF0573C0172B1DF28D4C600A7E4F390C0C600594E444E2B64E520414E324E72E403E474C6005BC600F14D04720839404D1D4DB18187D814BFFBC500DF90459103E34C3ED84C565E1E72CE4CCB4C3C4B6EC1AC06F295CF20030DB8348506541018486370C0ECBEF9CD5FE603A75923BA383635909695CF344C3EC11F4C1EDF206F410B4C0C4FEEFE9C8353F968FF4A09567AB292DA5D2A6B0D3BD8F0614C4955C3B54AB52254C35D2EBF6DF8FA055381ADC5BEA0188371DE9A7239E767A43613AA204C2D49F74307F8ACEBAAC1FF24C738D2ADF75BF8EEC5384EFCC4DAEDEE59B8ED9EAF96CDA20C97B52F505E059802C98B0DEE433EBA6060A34F0A89621CCFE6ADF186308C224809A393CBF54B4038526984A7EE3216500AB4B56F02A720907FB6FC608DB2C6609F9FBF32C64CD7039187FF05066FA7CAA191E196DA1A0EEE592144BFB57ADD283F7EAE8175B42F2802FFD04D84C1A0E1ABC0ED03FC7231789A572A1CCECA970A'
	$sHexData &= '888905010E798F38661838ED6B4EBB1F47E4496D2D0CB89F523952F918F848923B49858B7BF1322E38BF1FA1AEAEEA9EC5C5A24AA1859E941AEFC9CA6AA8854A6EB3DCC3B5071B8197EDCFBE025A4D264A4A6B750FA661A89B11CC16BCC973E7C7054B01143076897A791FBCDBBEE11A6C229FC9BE38D6E4F5AF0B5803CF1C621918277F1810F7B2E9EB8409E89FC2180FC7AB58A6EE431067C6347F0DA811589FBB417971C1B51006CA90E0C2730E58F5CC9A53C0534FCC342AC7D6000506D2F87554483D2DA1C04118A26111D16FFAF6EF1B1009498D04395AC74A8D0C494814424D03129B769AC0A4BC5B2BDF460997D0CF4C66D7701B291882600C581221C162B49FBF3547763DF8DA8C42184210024C6189C7E24246E14C104C0BC3BF4CA0C65008475004AF7B8601A2F50566CDAD0112645E12372C0C4881C19FCB90E0E01CFB55E7D45C0B7C11180BA231120751823797D91B71F0ADB22DADF618387D2A85CC4762C798B4CCC48788D9129006CE5329DCC75030A78447445F816885E5F9FB1DB3A3840A258D598020CDD800FC03F64378016A8B43A208AFD7294D009A18EB081040341BF9660D581928480D60049FFE31A44B70836378FEE99FB58BCF9785C830E7BDFEF07030005FC108DA4A5EACB63B00C30CE6BF31DE5FAB28F6417808440FB7E2011D0EC7742F1030DC11A8867E5C'
	$sHexData &= '2823E67049501A410125F8A060C622012381DFF587704A492B06D1FA891041B95140B30EC784660D3CABBD64751233ED92C5A5084D6076F65CE6182E9202BC8DEC5D822E7D900A243BC0731E691B1A1FA50CBDB9204172C4F2540F036A02342CC189711406FAF2E1C3EC5B87048BFDEB17A938BB032E2C28BCF02B3849031D84D10BE1DF1410207307B8E8515038C0EB0EA5DF80EFB78B4C7F5CC8483BCF73055475EE95082E6B5516DFC08B85B0D4315C4CD023E04F6097C0343881C29C9E4656B3C04B202926FA14D401A8B2FC3F67C8F4149F0777267553AD73704B8BF752D1EE36BC30D1F913C30966078374D8042140E606C09AA7462858FE224C7BD6D2D996F13358C7A7231CAC094CC5700AC5D6E0463E17D71CCF1AABBC28CDB6484EE006F606308C014B0149412182700B7E7CC30682FB7503CE5E080174198B480B2EC10DFA0F6BF62EAEA4370E8C0031C84D8CD90069DF0CA7586054610B480249E6A800F3149ED4871546C23E15178ACDE29A834B30D270F0FAFD0DBBFDB0E92A60C3381BF97F57E9C8F14140DF9C004122FEA6ED0DF23B007638603BCABE66413B50FE7406DB9CBD8F9E027525365854678A00FE4160A604EF0E1B78709345B10A4811C6C3D7295631751A16AA59DC9BC0EA00C906C0D2750AB8DB9B1D2F0E0A08FE1C24D059580B4937C0DD46FEEAD2730D7AD4'
	$sHexData &= '2E38CDF0754141EC04751AC448FBFFB71FA83449E677064D395170762949394970730449890A3DD7B938406E032C1640DAB2D7E08127018DE3A9C0A7E2A25F570679DA4CD17E5378BC0743399A733AD249ED6E0A8B5063B0B7814DD8F6C22B9945F1F7484DE5309B1F1D04B98543C3133EF946137F84CD7EFF7C4970482B084C03C1EB25300175177C3A17DBC16923CE390845AA03FA034D6A021E5919995642339C833BF4080CA6C12E326F55D4498F58BA49724503C8440FEE9BBF29444A40496303C0480101BC1BE35F97EDB24D2ADB1F6A3FFD36F26A0C40CB9B8909292BD1691AE67EDFE94B7DD4E43F650A79507EA82FB820E00432C1129FCF78346647AEDC97488B10482BC27F1EF9D73BF2837F4103D029108C484863C2EB1476D183D0E8EB0F150ADEB73360A180D1D102122215030E5634418EA9D3428B4DD7C3FFBE158CFA084D6350081CB551484D0310790B692AB8D902BA4E628382C31E28D1F79B17DC747A08DC7F49FE197332C1AD5428D29B142891432EF0591CD0293358736F43560A98D24B409E0274784DC53D1530FA106C58171E854B32431B17EB493BDC14E86741B739FE88F790D81D56886AEF301D2ED68900D15D08ACEE69684C1162414146304209AB2E5FA76700B3D42B84C584568DAD9D481E7A0E2B3A36022BBED323B0CFCC013FA18FE834B0769ACFC3BC90'
	$sHexData &= '093D2FB69AD5DCD801C1648ECF7080770D76B80EA256F20EC262102BAE19ACD1C42E844AA75560F146AA7F42ACC698401C043CF5A0C217D6EE000BBEDF43EEADC6BA07DFE2B74751E26BBF10E7890F044F244F100738CDC82EBF1056DAF1014C16F6359DABD090432570C21A4AEEC8F25C029D50CB3E02B249666A6913831E54DA76CFC6E285C29854C88D259837813C89767E466DD4D08E9B8D1CFA8A4C895E78DB4AF76CC6FB385EC3460E4C964DB4D600EF1649C1FA02D33C16C3DAD3EAB4AF00D683E0FC804D3A784D5BBEBE060B8D0DC0DA9D41BC95894E089DFC6AE3DBE849D5C877224B40080C141493E660C22BC3EA15FCCA0303DFE727682BC724EB32EC3AC04BBB09D8D04C53085706C26FA0FC1C60C5AC84AB707204B3136F0EC51CC3A0024D81FEB603B446B5836090B94225C11B0F2DC38A9FDF7A09E508050F8751CC7EF1F2C11B9AE92FDBC3767CC299BD2F182ECE9ECA0BC72AB63D7B6C24E10F8381CB49BAEEAAA11057F049A0D86565232376691AC164B386431BE05610A644D101C961C0ED467624F96000353E82EE580433D29CB8715042849E4E54C129F80100DB6C864ED71AEC02C2E55C6C0B3D0A3692F65ED51E9E5ADA403B0CD81699633F88DE1A37011F053F3D01FE52271083185B303DF72BD49785E8251535260C3B4CD6C3744A3B9F7B28AC40BE81C23BB374'
	$sHexData &= 'B5003A4FA8431644305B546B0A85F8DC8BDF83854431FE73FF2E392153289030F93EDB9714A98BD8AA056E74106E6AC512B5E478C8B18460524840663504C610D669D7669D409BD9D630F7A03C03F40115AA3CD31885E707EEFEAC203DC211039D8CB4D0AC4DC3A0C0F7BE5455B2C7832B807A49F2E97407CEBEB6B385B348B1C06E4A10FC27090F434138488BF2751749CFEB908F384C8BC02A4B49ED1D97C4751F9F34501075135AD0859E3575D138FA403D74E74C92243C05A787B46C9558BC706A96055439EBEDC3572854014B058CEB0D1A789EFAA505043BC6A2A19F806E16A1C82AAA75865F08123B016340A0D9203970068CF2F96E571E3933756E893BEB407B10F13A1617BFBB30752D74A5CBEB0C07122DF8300E51A983E0FB0CE55CC27C15B87172A19D6F27C3A20F7B0C385AE3E6007B64229B8B146AB0EFB7DF9940109BF3526EEB8A1AA04216415F49664D72830F0A5475C0D8EB1F8B498B583CB3C1605AE2406E0C4CD005D9C0AD3B3A21EB503B38B9D03C8A051C000070EE5C7E5623667E45D3AE50464841084888DC884E48807E48DEFAF6A287A3428CF43928B8D97B192E75A33E6690F0393E58A71F48FC3BF975792F800AD6FAC01E84C64136BEC64348B72C3E58BCDC3332FFED48270BB2FBD44075904110137470745076673E38C6408275C93CE7DCB326249088209C'
	$sHexData &= '0151B3445BF2197CB3E901E91373B0630E770BE7C2B927F3A3740EDC1EBDA6209A3C189C8086B43D2C56E0666F984D14731BBB3708CEDCA619E140E18CC6D6F7C64724837EBE00C2662F2717594E1847B15C10CADF2D4630DA46F6D45700A246342CB1679D38C22AE1743252740A3D81F751918927974512F65A0A0C7F5DC365A102453F24EBBCB38EE4F183B9988FAC5152F49AC26E8188F33902124978A057A92D3E4802BB2BC9C14B5BE948C780BBCB5E95DE299DCAAC9FBFAEDC402CC2323F89C7D8F254225BFAAE003F8C89252FEAAF5F8F2C2C10DFE790C223222EFF6D7020FDA59305FC832B2F0FF9BED444D7030A663B780C52E3810081CC08A78D4B4D411B42BC3A723112F84507B838604963700C763BD1731E41FF7444BBB82A4F28112442E04BC3F22A882274C456BFE1A8D546B7A78A1A10279AE3A564873BC8109C47FAEDB6D42B20577085FFAC30688071AEE938C023977D9C5ED6603162FCAE37C695C2F4FCCEFA558ADC65E7BAAD89551FC64507F5616147B94D073E082044CD3C9524A0FAB698A1C1D574884808AE9ACE82D926F255170E4ADFC0AC067142B810495E126E44A20E0C100EA82416F7237EE93E1FBD894C45EF1A0C4DE9FCD85E044E20978CED8F7E907D087311AD0A47A779E7DCFBA64C8F1044BFD7EF7B9BF031CF8B5DF7A87432E12938C44A215A8F9847'
	$sHexData &= '05A5C218C1EE82BA3BF79ABA858C067D8DC6878A380C3E41270AFD391E392F9F1CEC14001019A59CDC201AD5A12F8CA798C830C26816EEB70C0F8620C05AC442BC8966FBFE777668731DC3A908348DD38DEE7C335D04BE03EB1D9E7521B017D2BD0F0C41448828E9C68CD984EFA3C3DA39F30C065D65BB397408C81FED12880C0369673E8BF0EB71704DB8A7AA1A95D83D973F99908B7260B4EB8FC34BBC5515CB66A49DD9FC8E5FEB26037530230FB74DE7EAACBC5E97717508E6DB2A0F36481B746831142D1AADBC42E07573C6182F4590001A06A2A94048575F8D2798EB1F5CB610344D4F4AA9B2D960FA08F48E6AB50B4C3339087343A8B1BEEA7C26663951FE7534CB457C6917433BFE7442ADB1C378C0E9A04B5568421E23BA934D59FB493DA8C772F71132BB8E2C68230E8ECFFEFAAB554CD07CA5E9C4AA9349391074584B3B4428B2203A547CC2742B0BCDA9164B58AD75F552DF20764B78498911ABCB4F797C3B294281C1BB31DAE1AD02551F8225A336ECD6FF2D07252CF8D49E530A63021B6A819AC4413E7309CE84DB9770CF5B03E3FEFF50382CF8B8B56BF75CE7C7740F341ED73A3020402A00EEF9467F2526022CBFB70FD100438013A726B73F8901FCD4AE1D38B6D6850EA7181012B84A1AE000871C1C2A58895780A36A0C7E4E48800A2507E4537E00ACAFACAA95404B5E3F'
	$sHexData &= 'A80EA323054FD1C0A844D8D844A5B5230A7FBC41CEC3A646C083822F0F9BE4E24B786CD36527CA6517773CAAE163E2DB998BF067992F552779B64A4C13AFAC8B0172D5D55B0F871A7D7E2FBECB73115D19B11C17A8A61EEE57F6CAC34017497C3F178818A1F66CB4726BAD95642F41E9ADA90F408834A06BCCDDFBECA22F2BC60418006978C70EC0A6BDC765C84C524C0EC00C0A794F786D4B27A410F9E501CA5330F7B0FFCD93C30093CFA7AC7B7EF0DD00AB19FABA1AFE2702DDF7100DBDAB46AA338939B42EB800DFE24319852F17C2C0014C03D8FFBA276F011A34EA06D87B795F4183B1174D84104C2BC3E0EC8250F08083A8181C059BEA20DB4CF942FB2028C31E014C2F4CF584109C5DAC9AF7805340D9E2B8A8D63F418AA53D0D172893F1287038C2876E552F44495E0141F0B3C42BC103522F8446C984D42A9EFFCBC1690BF00FBE0C0BF37C0C8060E6DCEE7E0657EBE17B8BCDFA033ED23A6FC8C82EEB34EB1CE878BECB67D8B232A5CA1B24908523F32031FAA743527249A7CC3F90F63562165F176CA1A753C4A1885041F06A2DD33FF14C392060270E30621B6944B90412EE02A380A748C07DE6CB7480C2F40B70750583FD01740F970E83E1A404301E75698610B6532DB92A00BF44BB1A3D0E20CC22144770488B4BAC94CA4F7847CC1D087D51B01693AF8D3EEAFAE48B873DEB'
	$sHexData &= '136C0B709FC1E0A3F93F0B3571AE94598B0E054803A9609155FFE03105E54820EBE8F4870AEAEE4C106300F9492BF157B00A7EE9490330DF9D3070085D3651A65EB218FB2090C20144031C39392E27364165FEA52B120DBE8303E089A5BACD9CCD308A06BF26C3DB9E9E1BE1ADF783C205F0A223C018BCA16210446AF0052AE0C1336EEDB1085881D51F3040624CC7F909FC4B60112EE46E5C750B3AAC969AF806458D4104BCB003C91E1A8C1432944E2C93B13350696C9FC6E901883C0B040C9CC0CEA2383CF01C247D3FD70D5403098F548070601352416BFFC631B061A79A36908C3EF07EC2BF9907E581AFEC5018663BF89F468ED8E0658B8BDCA42F7909466A7D60F2A45BDB9FF4918009D9F7FCDCE0C26C3D89F8882FA37E1642A895A444835735D65630D36429D800D78D65E2A1DF5660202211C373E1E548831E60F6B081C178E9DE67BBC686903F015E088F0256430F05CA910EC29FA11506B0B04BD3BCA431DF40CB6E35037D69321603C683390C8A981C9CEEDF433CAC98648ADDA8389B5E89832183810BAD90624CAC47D0C39E8C1AE703CE457E500EC0ABB402D48BEB52C6573AB52202DF7F1CE4801C172B082BFD2A85C994ACFE2A307D6A208FBE34A37F23923DF5301BA37FB42A405E5EC8501EE40CB6294A1E90297C01010CF27945F24CA28EAC089F1C924301B3DF5A281C'
	$sHexData &= '08399139280A285EC8439E0028B44FB44F45C9814283270F79C8E6B5823C27B57EB57EC84B5E24CFFCB52DCD81E414D05CCFFCB53995BC927ED05C68269047C821B46CCFFC424E21A7D05C079FE491AC1CEC9EE525D125A44A0B295FD91A7476D0FF24107E1605EB70F6047CC17772AC3BCE1CE59DD7C04851CC50D646C73570247183E3E02C2AF8104801B82AD8081DB476F3E168D3742C1FA656A73A4FE51A380E94D3D609A8A4C5AE7E9D412AEFEB5FDC7D0C67090F2680C11D63C2B14A0B773C151224ED1830C0F812D15CFF8E959AB80801997F9F7A5313DF49DEFAD47D446291060A9C777511D300E0EC11EF200A2C54114A22750863E0548DD27C18F677D55060A6BFE769418B049089010A9AAE52C4449004040F080806A6D2540C0C101E465B150AD81495A7E65CE6EA991CC89446910AB19A39738D0A890A85182020C63FF7FA4391565BAF24773D6905CD80B46FD2F9EE43E12D6A0FD33C72E758033BFD49836E484A0948114268A18913C118890807C2DED82266586A2DF6B3951315EB63D7203A0488C927401A3EEE60C5E9D8373F39F978AC18126E66C709640204D46C044C053F4C0C13C8B91758D36114792D1C3B89C2488EB6731A48C883A7B479DD284E0E4236DFD52C756D319281401ED850C1C41B42ED218163251784FBBC2E6B35717F49309A385173E3C1746F33D1C1'
	$sHexData &= '6AE00843110E3694EE88837702E8C5C6743630248CFA1C48465FDA50FE91B20BFF154599CEB7ED991363233EC32860DE75D74C8B163A95057B0F896FA5A7D1B540A1DFFE9493AB0CF30A3E3684D7F412983BD5BCE087073CF27033C08B0CB8A12DB93256DA045D4768CA5C610455197130C358FF0914922702FB61980042C91E0B519FD70C54928E27BCA8F2C834777C82BD1EB17EF00749C76A17B0EED6958F07F84C486BFF380D7326747409A51C2AF61BC664806B13004E081708D80BA007448B5F385E30EB54B9A9549E96AA7B7C7CFF0C204D5306074F56D2C3CE855EB2F7388045032A2C016949BC2A9F8DB170B69DD447448E98BB17B1DA6A940E9492F6C77D51F086C57E96B406E00A958B09C24124385F34BD6CA4638013B4BC2DD9F6267E121A4920258C9649ACE0241B470A4FE212227F9AB8B7822B1C4292961A05A9511F0B830C0313BB69872D211F0696005F3527F93CE5B9586CCF116CCF95088CD426E5FF994A9AE408957D185948741176038B0ECA26E45E10717D5A7826A9DC815EF673330DB53AC0BB73D838C018694A0E893D0050613E949507D06D163E6537A70E4B5B1AE6AE826C92471520D4B9663DCCD53240E72ABF74BEA432FDC5C57A104D8BE0A5369F0F2A6673A65A9FB491492B07DD6A2004CFD6F99C7506329A801F0E07144A6A06320F6EC4881D57EF12CB'
	$sHexData &= '197AC094B8FEA07C270F2F7F279581C131B1329E562FCC0D9DBE494372EFBC336D4E7E3EA0792A6C03363126EB704C6CEB2402E0E6ED3E9DF65DF92A0860EB39800E031B3AD8CEEB31D22C3F36660E72C784506B7FA808E3F930100BCB0C381170CA0492C4911DD2E0D47C31AFB5B6AAA0FF5418B451444673F1FECDFFB84C4D8AB891EA1662D0BA0172337C41A5BBCBD501C1121C406690490107610469AC5C52DF1E5A0F1E66390A741CEE484AE7D6C1DC0EC18B0E8C139C8B1254DE1F00137116BAA2D9495598AAAC41623049AE82C684263A75E949620DC72019FA1B81164444C556D3441516946B6CC36BB96ECDA153AA01F2CDDADF2B2AFC85F04E5A02902BC14C03D08031C4099CC13F106CE28AF82DF998788160A0583BF6C480E37252F320D80948417FF0C0FAF5B048D6949111179851507813E210FC6FF243411B69647F3FFCDF08FBF69A382A812FCFBAB5BF7337D874D6EB02EBAA45F685334320ABE9043C5E92ADFE28124C9987141DDE540EFFC877A00FD46B7C2D91AD37C884A1BCF725C03F8FCC752A42154F4C8D1D189D92DED94A0B01ED424C2E24ECB82BA2F4CC4FC25854267F4AF272747790EE0CA11BAEFAF0D1E88975C0A678A80246DA71FA0AC4111CFC3931DA984D85EA19011E70D4C04CE4DA852EF042DC10BF43A1350D05E2006606D072C2E86C85F88BF5A09E'
	$sHexData &= '025AEA7858C2D883506A66C69A795DA5F2904738ED82124FCB545F19F955C24F0348897543963056B04B180F4D070FFE611A7CC56EA8049BDF7A406B60F0E04939309BDF7E4700726087A4549A20484E810C24918FA860142447270A790472908FFD8E0769861110FF268F52E4429A00C39053820A9F337F2A5F277184238DD2743695CFCA25D08899D28BCAE2C4492887B00136765A50E1FFB948076877FEF7041697B68DEB01B005D021C2EA152A9EEE6BF91FE01560D514B72F2F0FA82FB78D809969E98E4C2C25FF506C15439E9341364252E719799348143D10320E5EDE51D8142749014014BC1C394816DF98C8E8007C2B858C1830EF94618971C21461E0D55F50107556F47551A1C84268F0E4A214F0E4209BB89E149314891411797047677E14E07B14872884A1B7140B2720B536125C747DDF4801F0076BC27554F210C42712AA44C30FA489C400DEF9CEAC6D00E5EB2A1E4A3F1BD0830A0A4C514E126C38C8AE2231C009E823609A7435E8D2D0E2A0B5ABA5A7DCD840758A3A131622C4007F1F7A13098E493D1045622F3C5875591733ED26E0E1455F8BCE3157B95CF2643F088C6B6B153E81B4EF8B576F4974BEF9885E5821DE0287C19E9EB6BE51023B0C1DB432F03F04009AB6B3584F5CE2111324750CBAA7DE2264B15108C3C63A6587DD288A102726716CCC3E17BF119307CC'
	$sHexData &= 'FD73010DBFB112B4B1B1960D62C12C10FDD226B9108917BF0D71069995C07E2EFA8A7F13785CAC42C2E0D6C02FB0C774CA94BA122B29A58ED0422E4BEE308F805B41EB3BA147E0EE9201EB4589BB98996F20AC644E129A576F4EE09EC790B0D9F8C770929EC0C288D90DEB114A916E27C16B8995FC90C64ABB9888009FC23789C202EF0AB18A8C487F9F923ECDC0B1750633FF66C040E832236B8FF092125822222C0F4453676091DCE189A4831318146CF71CA5861F34009E86EE0D5611010941074BC1FC8B7F8595428AABDF4E2530E9657029BB040E619D32DDB517D2BEC50DCD7780B98A195A7C95DE40BB23D3D82B483497DF4179E8C645D0D0E04CC2808CD0E4DABE4D9F324DBF494D70CF825885604D58A150288C3EF401B03071104BC053DF37884BE318D6FF57DC76A3B4CD23A488A78AE5C1261FC4E6D1AB5AC0C8A983D8D55B54880FDE2138CD374F4438ABD368E189731043CBDC24F877103FD2492BC037CD5A498D5808D801BC5387CD4C0473164498F25379D0A3416C381C3165E03D054BF7CE3120E9B4C9C784570C10C8611DE4FCE6F0E6148A6D7DE8AD12EBFEFEEF8814187183E8027402EB0D31724648D33A19258CAEC2935CB8D060059A6A2DD00992E163149ECAF7FECC1AFB9155909C70636C016E5CE885CDC02AEF370921110A8DA29A2FE12B8B493900744151E107'
	$sHexData &= '64203E8D896163E36F632BD2FBF3C3214F5E5A5F406C0E0686420EF2E4520E480E3D0E80ACACBC0786D091B9E1E4214F4E440D86852A0DC8410EF2200D150D0A0D5E210739FF0CEC0C6685AF4056562F9118E19A0CC8431E207A0C760C95833C396B0C610C560C53801C02900CB27C90474D483F790B7950F2A0190BD90AA6948059DA837FBE34A1BA7726B41E0DB97D07B8B582D04AD0DABEDD0267672D718EBE8B038B4BD32EC246C8F939170845DC0C9907CC80157803213BC173120C3BCABF9E4035C4CA2C3CE4ECF804DCED4903C825E81D5D9E498959FFFF1933ED059EBA93FBDB744483CC71491C093FDE74285F21DAE0B8A90A9E422BF0E3E9005E0806106A20DE5ACF6D455BED82557D85F7001AEF2F7806104462929E3D1F7A438A86F7C7C2C148390B91AF23F481F92BDFD51D9F0C170F10C640309DE3C8400141017D405BFF425E5E4EE681AF8D98DD0C66B491CB2D7E161EC30CDF4C4610020B1720BB9D624808099C32E43B5A060BEEE3F803395A07134CF136D66AB60CB8CB950C5A1DA88CADF6BB7CF27E107449FFC086CA0EA0211F6303023FC89E34830484CC48D144ABB7520AC2EE1533E6C9546BAF18400CB1E77C68D12B10A4E24DE823B77FA0C168A5CE585EC1C1710850DD036C06449C2A974BC1054F49E26D0A261A3E190625FF6D0EC9898225E8BCE8BCE1E85640'
	$sHexData &= '153186C0A57A31F97C57811C023F1DE269F5743BAD439E082889938000589A50D863F834FAE880827D4B6F6AE2291FDD886A096880A3B16E6B2569270467080671826235AB48FBDF14B3013904654B06DF64025939B88A027D04271A6E01D106B9E9DC542D3924E9DC9FA5C9815C4A2EC30198D3AF387F476B79A95C352BAA28D8177F0CC0B100FF3F48560EF2028589CF7B271A8E64023F45EB8C54313924EB8CBF16152B8167BF1C8A4532A5BFD888B05D81AC227B9285C921395179EC9CEC9C0119A2697F2021830CC9537D3838B5B10A79287DFF833A8D00FE79FA0A0EFAE3750AD1177AD3A07844A3E602E14C2062133F3CAB52FF503B21589DC43FE965BA808742EAE985C02CA8CA6717877179C0FA13D6683692C148E8643B1C944BB7C5774F07619501FFA992D0EDD90D498D3C284CF279749DD88632792601CAC4059569C8940C708BE53A09F814CC32204C1DC20B78BA94E841440C740197792E5627F1EFC91BE1895AA1BF0914519F9545F4614CC649AB352EC06904724A06068BA849D1150385D6B60B567634517B07EA40ED2BF07ACFA4D1A61248846A16C6A52A9F47B214BFBC9C85F6774B6087915429FBDA6CA5C325B0E71C487D46B5207249EA19E756BBC31A9F989C31404DA4044DF6C640204A4B9C401922103F203E5824ACD7EBE3AA84AB1F8D5404416D3833F6E585FA'
	$sHexData &= '81510B23E5EB7545145BF3B4BA41E0087355693010845ED451D558EAEF28785428AD5A4C2136740D183818F83F57809C8D3A5B098C1A3A4459B0020B50698A346B6CE39B78EB159E10E0015CE1E7B919530A3E1C6A166FE695C0BF2E2EE9B4C057402C185B58FCD3D9E0500183C683E7107660693611120110110AA0E654A30A785ABC8548F80A13B5A544D20EE23358784E7CFBF01D3C63082EE149C1FC02C7EDE9006EA63206BE4A0EA6B03C7B497F8933402530790E1258301F143C104C554DC8770F22E18A125820B89C772151B2B2F2BE77878370D3A01CC8875F50605B07F892A94361E526C6BAAB1EFB24411D63209CBA6219C647EB1CBA8D588A2B2C9081645843A7A0314A14FDBE164C4FA94371DA95A66B1FA22838280F2C2C304BABA934303434F2424F30B233D71F7AE0658104C6DA76A382000B9E7930C8D087D23BC761795E5BCF747C58602AB0BD1F20D361C5B82935782D6805156845C9CEE0EA4A75B8C2840ADE02FD807E1FD5E2987474F1D67EDF062C10CE6F5262104C8B72185E8D527569980324FE2000D5FEE96A1BCD3C5A5A184D3BDB8CFB700A6D18A8E04CA5A6ADA93426DCA6CC4D4ACBD206C80E94187E34664103C140E8F057536240455403EAEA20B1815B44AF62469F089CE1DEE974147F88DE6B4C741F416175BFEC12777930D2B4E4CC1F1613B944E02701'
	$sHexData &= '1B44200860F062A93F86912702CF5450C1C015226011903451C277F6BD870D5C720C1C0CF979DBAE690D86FA04EA12BA7041B8B95432A97DC6EBCD2652E6025A59E0031EDDDCC1C938A11B0F1CAB5F75D037C6D3F0C1A595C2545D7D25914A055EDB7F53ED369C92C82A4E18B3C170607B8C64CB4C5C1E2C81B3CE55D51343D014E0CAC28F1B826AC8E11F9F1A2F9A0EDF489EC2C168722195C1B9B697CA7815907610DC66918001843245F658F53CE848364F842C610D9CBED31C4D892C244D341FA330504EC415B900BF6A011F8B3D1BFC26BE4C386B3E41F870516FC9E0B3D4FB26A11847C5961838CF05CE1D86491848A013D3432838096C44D272D1148C5FD6D3C366C2E00B2103315C72238A0A23D2CBD39CF960E4699E9009C5EBD04979C1201175E0208B3DE2DA804BBBBB5FA9BC540EFA40EE72D8CAF35C0E4D78484C0D443C9D4C142ED6A35AE820BFC1CD35500BAB49FCCF90929757D234C3BF72887EA86B04726CCE3FA0E455F2D0F9E3260AE4843CE309E309420E2587E1FCE3E279157202E34C83F70E14C829E4D3E44007F2BC4A40574058955F232A60EF07483BC84E2F0A2B9F7A986C05E08F6E79516E0DC90D2D4C56D63B5125E60733FFFD093FE30BB25A24C1E105EF1942748E6E1B5009788CF40A233EEB37253B5033F2B2A928A16E6A7A28B6D165E44ECA2890EC8415'
	$sHexData &= '5107EF2575DAEC90184427D8009783F22C32F15912E882D3CE3B261C261EB0C05ABB95049AE303E81D1A4F83E6E05ED9B4E01AF73504893DF5F48DD8B0D6E9178BD2739526E288DE4C0943E9310EC54A33F63E00E711005F5A6E9D9102E0E8735E73DF354057E365DDF3BF46F58420DC0944A9F2B65E1AFA7EAD25E3626B144508CA75B4522BA0DBFA8AF58C28081500E50A806F10A94FC00175FDC790306E8C46C94D800978B47C7EE441DA94901AD2460850102CBF09A8D641FE488B0A80794117B0EB04A9091D74F44C5F9A80F640EB2A5048FE414C3B41D0C8E523404C8BC141400F1A4086DB21CB56F11C0E56E335444C3B2A75871708E1820AC6074785C851B45F1F500170843372595DCAE7DC481FB677006AB0DAAC9B490E49BC99B0568A4201F14ED7E89CD1A4D5265FA44E267C8D850069E76B0D72943DDD34FE3313365821277979056CCE77B2C7771F8175C3131A6C90101F6E54EAE02BD121EEE2F04EA097BA0318EFE005236CDB4C1D54BF1A83FE104B16F3848B49A4B01E205034E834CF6068409F209F2707037676C06818F9CB330659F4BC4C8BE0C06F05E529AE530BB219F7AD9B4EE6D90E4C7C249DF42B0D8E57C1E7064903FC0045C3A93DCAC014F423751A30481AA9BF560C463A37B55C9A6AA71EBF01285930DEB5AE04615EF86A9B734AA7276973C775D4D3388011'
	$sHexData &= '1C9330BF597947183DBF5675A067258FE42004EB0ACBA16473038769A569770A39C96E7552C51B5C1E1F8D61B84E1907C1C449A04DC8C8F7E804111CD187ECB8034A2C58DAB569C38005C825C315C00AF91F52DF2071133748D827BA56F2342A28D81FBA7449AC753F1621F91CA673F0652B5FE972C8EC7FD29519F167929743152B0F68D873A44B1772BCC347DA1727E4951FCBFF15EA670F4C8B673868F270CF2BB3488947A30EE81F9CA1A2417FDACA8B5201FD7F9672B40808C8E064192EAF30417FD38EEDC758A92CFA794D6BC9387CC866F84EF2F20AE666AF7293C20AB9E442C7DCE52ABC1CEDECF666926BDB38A362C87E5E104D6BE4384E94667840D7A5923E00DF19405EC8208866381E0223E4E01F111330E200F3D9434A55C21F0B0D79B1D3E348649BF14F034983CF0F82FE76027600BE0518EB35D78BC840ABC8BEE0B8ABAA01AEE71EEAF828920EBAE4C75FC62EC070ECBEB07704835D828224904F01D41F1600072051E0FF77A4EB64B294CC3A76417883340AC0889417421B49AFC30465D1CD700F64C54EA3C2B1C0EB1AA7082301C7B0BDFE3BC8808BB4894D85E4741A3510C2648C1947D0BBDE5B23F0E233696DD823BE13D0C603932A4BC7C2E066E818DE22E02854A8230741483134A5C37F78C8351CA183E80749C90CBF133D7F493BF995313A6468131234C1878376'
	$sHexData &= '8D80431582F98D2A41722017C68BF43059110B12C7B09463AD39BD242F8F86A5B1637A6FB9E126E45EBF1DCAABAC4C527900ED741C084F9381610192AF082B9563B8BBDAA96B086646D6F004E189246BB5AD1C08B8883F552717492B4F774A4BE8C21367AF6DCB626098B8F489E1F748AC4D84E496AC20C3C2FAA4DC3798027944D029C6D3485D41220608373CB140263B42AF1C9E789D282F522FAE9E38C0EA22F62DD41A932B924341FE486A58E0F6D642F26F807F411591744045083D7924B9DF553A327527693719821571302C0DCA5F24BDA530B45F9C22E97D25F2DA6925F2DA247DA39B79113F68299C4C1F416B74E06751076702041175DAEB1F4381489A675A1FA483E05B8A0D5089500E4AC18A199010F87B3B92EE22601840064A40884204E9ED217D4E40807E40DBF42183926E76220FF41DDC2F1DC187B43B0B17F38DDF666F2CD90A5CC6413E59CDC64340AC5C1E4918F64C41F3C629C837A94012747CDE51784FCA78D17A8E00BF89A4EE8881AC2540AE8841409F9B483A2BCD51C5E99A920607350116ACC131D3483141D1240B44D2E00D1BCFC22B0ADEB04FF2F2EB40471ADB2D0A86823B4CAC2B01290491B5FBACC647255E2036618826E82AAC7E82E9088F625F5A60F82BD35E85EEB80806FF4E203A3C9EB78189E76E2006283019A7E9D22827454E8B49A767E02BE790'
	$sHexData &= '061B63E3600726FBC021267150498E1C42C0CC20FD5D1F0AF993210306FF150C5F51B2B2F22E5FF76AE0BAB2F5C0B4DF48830BC4A95EC9C8E540CB5E947D5618B542B549F8D0025E0972D4EA3F86BCB6A5F63DCD0933F666491720563EB0C80A010ECD350E5F8A284F8AC12FE177B274265A472AA12A63086E5E5596F1B710ABDC48B713D5B5E7963FF617772006D1E0D39E28302C7F807D1344098BA18F68FD74896B1123D2033748A3F515A91AAC76006F84A7F46AA4834259EC296D17AB1E750CAC222BC235034630C33B00E48539A50BB8273E105E2E5BC501BE6A9E1BB962CFA40DBF0D50C1DE90CF02182D10DD00A449D817BF1D089DDAA957F689C684862E4CB189D00DE2DAEF0D40D10EEA71FFD7AC31F10BF092787E0ADAFE7E05492BFE108A0981B8BEEE4AA511464540310715B49BA8082B836810B2100466DED43D50BF9A5A3B5E099F0D7730E8BB0478C5970201F89240311825C0C2127C801CBC60EDFDADFB32407437F92E730FB7543158E9A039C1152DAC5ADBE01483CB048F5B33717BEB0530CFEBCD865963EA6C44E14D8BC60665B659B0C7EF07A2C67409E3EB2EFEB96D764E5EDC5905ABD4C70EC31C8631C7444BAFECC840DC8B3B1704085454284F326070DAD39CE5900C16E120C3B51461312710907F62DED24A8B4C290AE1C29381575AC702C0C5DF9F18511C419F'
	$sHexData &= 'E38B541C001DD8F2A00E1576C24CD4C4B5FF7694BC2079FFBF58E2872B79FA6E36FF742B06CAC9ABB95C58096E3C0A9F29B3D02E499727CD5154B236C7C7D6D287D8C89557C774072CC3FA578E8A5B15C3D0BD0EE504C54534B709582391BCE43A575557C59AAD075A402DB5C62A36561F113062751050C4A5D5015E323DE6DF662F10DB6A8C6E2451FFABD8C0AA740104CC73284C6CE83B811CD64A6045872B615DE0282D60B468208831E8DFEA0CB24BFA042E8E5899C2B8BEE73A6121645801A32194069CBE857625069462D6663BF977B4FCAD17FE394B18742A41B0D2DDDA8C034B1C60A8A825C3256631AAF40DE16C08FBD55BD7FC8D8384C489DED8C07D834F01036C6E0C0B0DA41D7F9FC1600D424CD013C970A6AAD29AA268A41ADE3B87A9AC3ACAA7A3AD544801DF9E04A8F492F86E5E953B097870F5CF6B75574211D44BB53B8DAB1BB802DF1F754E4857A0D20DAF54043CD00D65054FDF15588FAEF4D9747B540556F5663B7834EE21E81873135B8B1CF9AEFD572759FDBFE2DB8078240074145880647832A5F94D0D406EDC602609EF09AF12566B3043E760585E55F80C2A3294865E62275DB241377DDE0EF6B1CC70331B9F83891DAFDE95667D2BEE05BF5329809089E253266B94925C018A6886A840AA7FFE5633290AA8C7FA61F91B9FE00D988E08A07A2C5209346B511049'
	$sHexData &= 'B8C9189BB85CCBC1DC193004243C7F073225901FD3258BC009BE91C2237673176B08C3F1EC7F95BBC3EBDC410F95C2415437232D22D21F55E99FEE1F201B833D54DD6AB7F98EBE8F0CA415053A24B2019808AC51327AC78FD905BAD05DC21940670C2C83B514F4498D4818CA3B00C4403B0F897000FAF1BE87F503CE8ED095E92CEC4F7B6A873A7536B06850C1C3996D21F6BEDC6C1A07D8DA474D1175CD2A480F604D2965C258101E829A81EBFE18A132D4D4C9204844BFD8364C821EEB2E9B193F3B98041D3B019DEE54B66E00BF6B0F444C4BF88D561B0CDC417930BD4E42A12005A6E00F66C3592EA0B7E1FD87217A489BBA55E083E95474898786B4A54739A7266912101D31BED85A93A0170493A0CF1335130F070D2DFC3B1D73DB004153724EB582DB2125DF407D78C1A06020100D8463EB1313F84CD717E81C065CB08E9FC02A254019131DA6251B30BF7C409E3C92DA62DA0D5C2179919C41DA03393E0792CF0D5C034C0339E6D95E1DDB4875BF028A63C128405939C4B1A5BF21BF033903218F089BD802900772906D0D5C0B4C112200211F6906E4277FE849F940069023F9F9859729F94D8BC86C5EE951134F62E63257C807F2395B5BB54D0104AE542A9F072C5B864D4B1C5CD4D2845ECFEFC36A62914394AD83A88941C380C5109A41BFA889470CB4DF2F6AA2159E8D35EA5BF0'
	$sHexData &= '3D1A069001CECABCCBC03A805CD301100F47C6C328BD05BDB932C9CAFC59564CD2063A4840F9A6A411CD5D2743B5D06C787356C1BB3620D81279521019797899C813EC3C4EA6616CC960582F1CFC31089F1103EDDA280B8D9D0DDD354E553004700CDB4CA7025604AA171FB102182B3FD70B07167E3D3593D51F8086865E224E08B772DAC9F58B1AEE40B7014538C0E9077015E3AFC400107E184183CC8AF60F18530CBF49B222632C6C7259280866350BED14913BDB5AD3E4D3A458F8A03CFFA516EB0AD69B85030CA440DFF08D3C90228BC622647314D62C2CB0A2EB18C5DE5868D71CDF302CC63D400F98C71C0B9726C605440478B1602C2672714953BC911E7858F7CCACD4B5DF5DC1E068D612DD5C3F0806493B2E830508D77524A03BE25B4223B478CD77417E3730B0860801BD0C24E92252807D0066FD057D5D10EB556630381598E3205D753E9E05CE9B287F2B52087F070E065C180DEE10808B538178D8E18CAF01F803098CD7F20783758A9E9EF53A0E32750B4A20B11E63E0B46E7C48CDBC3E305C3ED67AB0561C2471004E1A70A4758050216FE14EB3BFA3D360BE844F21A1C98D3BC1723A827103271D05AC5F304EE951FE576565D37D34EA4B20C0EB56F548D682EE53FF04054AB0EE952EA1F3C637A41790FEBC763A1F271418121DEB2984D2D0FB8430C21F0D53F26DDB8A6E'
	$sHexData &= '1CEB1526581028F8D275EE298398C221A7D3007B406DC4130B2282094F1DFFB048BE2363672E2B0A2A06414984F81B3DF51C34E94F6DD048621DEE08A90CF471EEBCB5947EDE885E3CEB3402CB482B87847904CA08EB2611B2676869664A45F9AC16F6B6D017FE013518A1F7410DE16DD6E59D40CE204847DF46E8C5AECE0B10E12D10DCE6AC214717B882A0331DB72EFD4107460D3A4180571473518208DC5A0FEF41620DF1AB75200F8FD1135B55F05D7D1F107489525110F540F0BE3541F70DF52D02307925D1AB2902700C8BAB1D8A6766508E80CF8B093CCC806BF472EBD0C119CF200AB01AF4EC015DDF421E25A72BD00C60400E72880BAC44CF2179801C0CF60D5C013944C90D420C738B9017C885476C4723E400C83BCE154779C8CACBFA46FB510544E2CD0739210FCFCDBFCDB5CDC82B3921A5CD90CD2007905710120F1A901C08790F2420CD80BC9213B8CC100053202779100010001004C92B900427CCF9BC0A39BDCB0E2C83CB05099002CAC2582AD76FF1EA20D705BDF60111DC495BF08C7C76C6C645D94E33414017D4BE202D6AC65A937C1E4930AA4E04412812CEA3194711C6819ECEB80048512C2A1A8CEC32C9E70BE95F40AB19EF56155DD81956B0422C36224CC66104438D5F0A77B1DD7544B4034C03E6EEB6505825158512C27258E882F0621677EEAA19FE420A391D'
	$sHexData &= 'D82364B9A5CABD27926C5BD478E96D09565465E0E3016B82B3A8484C085D5017405D5059280DB002CE614DCD8493543FE3D500464884493C3013416E41303EDC0DE0AC80F14906C1FB064061AC34A6DB1359FC524305C40026D3A41006E1487C49E76AA9A1A780449C7043B9BC0EE6155FC600688043F6742BA4F7245C6A06DA83962769FE6DE03412C04C3BC7FF89A13EEEF83A42F9A4944B622A09281220FD249624BCC5BC9FB189812978B92C225C0E1C8824DCCC41E056126F1D36B125610057DB03D820C95F257D0EFF1598419381789A6014E3DCD042600498F026EB3294AE9A2C7799CECD3AA71813D7D203F1CDB20AD4053C40307D3E230C5E73C213E0609979DC448293FF621A9C6AB6E0DABA18D117474418A04FE90F061211F1F949F7EA5B77096C0F848EC81497FEF6155C1A1FC2C0AB7AC3A20ABF4D43171654AC697525213098B9B1E1207D20AA999CBC99E608171AF01503F0216D772DD6F7EE422BCF9A88BB505338C0793C366910E61E7D122A01D2982D15606B21E8E81091CA23E0EE52F00DCBBFC2B49A46FABDABB98F64BC12017E0E0D2184516FB1208CBD3B02DD0ABD657E0A9EECD056BBAAA5911D5C1B12A3850F1FDCEBA7FF22BE6B90BBF14416ACCFD305ED0194186068987C813C2B488BD69D5AA1583C28941F72A0BD85D0DF172CA1405A107F4B083989D8A74E'
	$sHexData &= '14525807CE904525AF50BAC465797B3E79400420C1E83D4EDA422E2F8A476FFA33B1792FC7282DFE75895C54A81B47BF18023A2DCC31612EEFD900D64DC508124348457DF6E9153866C7922110D9807AC47B283D7BF148F26A2B59DA98FFB13C79C8414E55C44BC440C419B99C00A83C715A185452225F0164A1505D6ABA806B4CA2E4CDA67755401D48F03AC89ACF4D8B1EA31A92E9C2FB149D0AE024B4E2DF30A25121E0EC1C427374E0A6EE0AFB73319F7BC8525408CF7141FFD55C15F3D8753BC819E8DF60FF72DA24F67334F27D8AFF57CE167D47E976415709324E72DD85D4689624A4EB5F0D6C601B487332A71CDB189FF6E0DC96397442247C86C7FE72DBAF0DB7CE08EF763D8665C8E5CC75654F5B43968F83EB3821EF0383B975ED5EC0FD72CA7C754D9734BE31907471B00B9163C21DBC4D15E1F58E96ED3BDE2DE984188C6A28B62094E6B570EB64BF04B176F7EE38729461F4D816246B73270E38357065EDC018D170088575E31A819B20C9D78A3F6038438C0B75EE19BE0558B0B0A11076D295D80ED8D821EBE2248C38C35D608399DF7F234097825041609F9AA06BA295503C4D9C044807104CBBF624082C1F48D2495D3A6F27CD0E105009F4C2E06D044908061008C9501B6C051F1220EBAB09F5E2B32D0D3EDF307B21940441EE9EE65D04B20C080C0CBE0C3858F2EBA914'
	$sHexData &= '9E14EBCB9F14608049DF68B7C8243C8C68DE9E075C785FC85A945841A93C9D026603434378688053435E404E018B4A3B6583E202A7C3403DA0DCC6242C40A16B63E090010A3F351CBAA3D025771DF51C18C983542828EBC61C3240BBDFFF4930B992870B8912080874920C01663420255FF0492738EBB2173201125551F2D15721BCA54DE7DF282F38D43E1D4AE842A204ECA3369E8999D383E20752004398F160F820A744B1F497086BF670380F142E71AA440EF42CFC975C684D3E4E1C374CF4F6BA8905E829DC68265F6B619CDF2DD316CE334D48ACCE11A14EAC7548A45A415D4B2D367B754541E991D6239AC0BA047F92A588896F9CB30705B950CE98B20077501A8442F1D6315D2AF10E79F249C1FE14C6F03E00ADC0126F6BD04A101D3F1F785C0075C1D0E8ECC86EEA7C3130E0C12069AC3F49FF595A03B8EA83EF380C9494BACDD59047D0F130E3E692D22B70DE4C99424915C4D8E93806BEE15A458B4C47C00A486C80FB497264BFBDA2A0B4F8CF7E8D510AB1506D9AD428241AF034CDFFCFDFDBED2A04CD26AC300C042E1633082EBC274601455C3330C22E800DB87DF6EF3A7C3F02C47CA8B0A755E0CE7CDE894433C88B162C07C8D0F8886EFF1811D60682CDB0531709F6ED7D603FC1C4205E2F03DEF1CB6700C16C3640C903C5534FC18B5DC15B4D47BF7CA749B66E6D792430'
	$sHexData &= '7BCF3885C449E91D8E87C5E80C23E30F35B967B9F08DCF304B201DECCD4397E9044FC8702E01232EDD3EBA3130926C95FA49BC7A8C8D69F5C90490C0003314EA7B4DE0289A8105D918DB9FFE6474C68FD7E1275C48D4A8D03DB492E43B0BA49E51037F348F8C50482DE9D3AEC4224E0F224FB15B23572CE270B310603B5D3AA6AF6F60E64CE8300097768D713888F22A76168BB85EBB06462711D01D2265662E5A2DC03943C7300BAD942BB75EDBD5DA6E73580D276BAEEE743D207E68E9E34E2BDF998D5F98551C8D46D39464A03AD853025F6A4E5FC81F5AD1068E956B2E288D4D21309CD49B43585D30EB76AC65C4B52196FB3D9FE12A9AE8CE07234F215FB6071BE2B8FB1E7975C8F1327023F6724FEF766A09F3A08987324DF5569581D7D820DD9B6208C680173598A2369E5EF3C3BF571044085C010453078F30C21A33FFBF34849256A4746CD74E5BA1AD105689030840420C85D33CCD08080C0C85E6375649A179397E29301C68313E584C42AC8223CD593E120F889540EB8FEF38080EC646AFFF2B6D298C49D5E24D34AD026AA7E14A16E9440606F9E052473056C4FA1F60EA3411FFC58F998C6EF8A22A30C24AED7AE90F8D695C0107164CED7E7B4DF390C64B635EFDF41B66D34BEA4CE606DF000336B61C3788B0490F44DD909516F81A17285CE02A807B598258ABE07D05A87668'
	$sHexData &= '606434DC26D81DFDC97853DDDCCF48FFCD264021C993F1D67FB4A450D4116160A03FC3BAA290201F73A393069506D1C6ABF03526CF38822CD7AD2DCD78DB2D175B5BAC50A25AE9A52A13AE481F42C35EF534B13D18BA82AB440E4A490FE13BD2FFEC43A049C743C0FD1AB8667058231A78A3EA57C487E820A821463095ECD3B14FD889064355704E0843308DCACC4548252D9281C92A3D355DB008451D6C2FB10A0B442A1EA00ED0746154DF0D302E428F1D4B0760CA12024D0A2578006EE8D18604C70A026A7D5BA80B7A60C49FBDDF674E78C00885C6B798F0C009AEEA9F273C7B86FDFD1EF6759A77426A5782DD93C09440171396D0A96D829C6474ACCE875A6CA447956455266CD568C6E4115016307ADA0124195FC98D7078095DDD8F530FA594D999D2F46E10B510B114C37D7FD4023034D7934C3025421C4C9F78D8A1A76944D5F6F7D30A135C590806935656E020F2723B08943BC39D1E540C88F0FF9C3B308B992A40866BE9F33E2F10F27D5CE881975501784E8F3AB5974BA3858EFD5E6ABEF91CA56E23C4209FA864699C0E70CAD1079920FAE242C8CD4083C21F3C7192D047F86E4B894739F08894C8914FD03389110A9E5DF8C2FD5C0D2212D7B3EFA923BD442FA74CD633D29170E2A4F4255C9009722CC8456C6968BFB8AA150C16FEFBC14AB2D29002739DA74356134B2901A1'
	$sHexData &= '9EE6699E0E0C3408380C3C3AB4852214411008111293540EF906D8D04818428E143A3D0C3B954AA5BE464646466139AA071E56188ADD30CD336BAA1EC36E0C38086C70D73C3C0CFEF4007C1F22EDBA5748784E5360284D362748482B548A9A8AAD20FD6369A88927B97C969C047C522BDE2DE521C2ACBCEDAA3D202E81FB8805DF1A0BEFC0588C6F95E6E4F7C22184279C2F4D05AFAA02B6030BDB2C2D29BD70134CD5CBDD841D83756E07C65648C591934D653CA751FF2576920A7872B95C2E7A7C7E80629CCBE58284860B847D9AB03F3B0DA9A50CC205BFD91EC1C11066F7C1747502F3DEF71928A8C9E03020D67229BF878173F90B7E90EF8823334E0DD8260E268C0763A3E58E0BF70C7F1D2FE6EEEB7EB9088D317C905AB9B42197AA44229032AE0AEE3B40732C1BD88D208DB3DE35437E7626EF4DF77B250C5B055D740FFA252361308E9D43E1708B37B578425EE7A4C6D81BC05370206D06FFC8F79A328F42A1808FF5F6D709A2F9C202742A44F860B0A43430F0DBBA18ADD040F6C6B62EDF4F514AF85D8D43F8EB1632344808A191402A8FFB1682307CE208287F56F629182C0BD8B727BA8C33DDE6F81D84B80D75B80B05C201EB23E805A0AE6423FF32864B0640C2E49B355416FB0502136A8C295C6901B0193878F06020279FB8351E33FF3DE985D28C2D7CA45380A9F221692DA2'
	$sHexData &= '115E2B3034E6EFF40E24657E15A5ED6825303D96EB10B6EAC189D21A7466A3231BD2C7DFDBF0480FB11DC0B780E385886BA7DABD98A81E2E173FCEF00FB91F61337A982E79D4ADDE1330BBB777472494E284169CCB7B87308C2E34FC4CFFAFD603F6E89C08493BEC725A48397D0074F1534D3B15E6701645E5741F4E1B165A00675731897D83CBDBFFD3924AE43C1835B7201880BB45D41E76B05649C9D028A5C0FB041DD7D32A6297B5217AA0130A7FBE2304B70D05893DDFB6FDA09130DAFE857B873DD7833AC1C11E93F00D7E8206C31832374F7053C3E521A7792296B60551380D5F777F1C080C4FEB3E3F9B9503E27F0C0D69C705575EA7707B46013D7422844047CED36647E0272D6E4A31025CF00C095804C756052BB637F27D3B593D2C0C74215823B62CE0AD0DD78C1152BA5DB709C002117F32FF051FA62F3C203495A57C2B58743C4600F3B4B732E158CB70975EFBE63A8215BB873AE8891561A110A0BCDE0CDE123915C3C36A9F0F27162C6E2FAF74050802696283124408F227845E90CE79FFD09B204362C00EB94A8843727B2E1B012C4435C000F23DFCC8907C83FF0175353E313633D2E1B64B0B1986181D85D8EC187609340B31416685FF7F0D661E18FF03E14CC923CBA8B5105C083EEE10849D144E4A104D5F8A0352A2F0950ADABABAC36BA09200EB8CAEC9F3634850CF'
	$sHexData &= '0767C96C826522104B2D44343D8830F5585CD67F6FDFA00A0B197D76BCAF72D9E3579288770D09A5CBB3200991CD8664F4A503188B981B60B935C450C7545AEFB2C05294539E8FB05C997800284865BF921CD7F305B4A48F670D8F42364C09739C7429D9DD33C99754EB221A3783EF5BF75880E78D1D83C008240DA51B3AF2CEEF66A541D7A35E90D8A41CA7FD35B1AE28090400C012A8F359276FF6051D9FED681619709B9E76722E406E18B917FCF04AC4343C6ADE1F03B725D21EF7ABD918833DF298D84F4F04DCD330C8BAD52FB3A1D2701A78C4C39E64228D73B9340A56483A754027CD4F4458836F418BA0BCF1418B3FB4AE0D62CF84780F492BE18F60A4DC896ED6EBE9EB00FB53920FCC685E4C8948209A2209935F4098501053FA63FD91D2E149E4F28360C8F1FCE6B4630FAF1603D9B7B710482BDE3389CF9A0EEBE84184513A69443F36F0013E02D5A733344E05DDE1AB1F1D1F473D3D1847C6578A78EB033183C308556DE04342ED8F5F252FBB6FF746F0468BC1B919C48FA14D70390874032E1B5D90583E3CBAD0EE176F1281395045100CBA0B023B51188271A18C88124C6341DB3824B4105E03BEFAB777FCB74014450858064A8D4C001845301E8B51E1C45D028ED2720AD44193A134A8DC0FF038FB5D40C1CC28453BCB72E2AEE30895A804DF643612E82A42C2BCFE663D32'
	$sHexData &= 'ACB85D500F5E224D8ED023DC7C21A137850F8B40240EF7D083E001E92E0C09A2033070BC5CDA740A2204F61F0770E85AA503429CEB277E73C2020ABF32A2DF2D999877642BA12C0C489EA66C3C0336D2534A19071D8DB90E70DE1404784944B686BBA416005445B0211B58271CEB1C26261AAD7D9B24732223D81833DD124CC4861ADF4C50DFB7881DA4F69BA8D312F49B4EC12A65D5897632CACBE5D20A241608A21C5E88959794EE1EE01E0F458B6F225E221C164183E3F8C45217E841F60004D17413C09BC6B613F404F7E403D1887851814618D15EC34A8B141BFE0FE21610E8034B08F641030F080FB60E9BDD600CAAF04898524C33CAEABA38A32111E9CA4768F59428D641D430D1296CF3C10D1FD3FF501BA21F038C44086FD9C06B0FC047725980F2DD260B561F9C4304945AB396A6A0CEF9DA040655198941B128C38653843D0EFF4123C04180E2664468B7144C7C5E801DE2BE7B6A301A705703D0C001412C2703EF06BF8D8ADBC41C9F06721240551BEA2C9584E3C3909D3596640C612F2DC8D7FE88018B080E24483380010716606C5FFE349A6D7BB336DA937C3895450608145932FCCF3030813863736DE0E60311CC543E0022021452591AC93442A4C3C0865876923C34D80073457048AC4D606D849CB082954EC9B4922AB84D320EC18B02677E90B642F78A4003EE4C4884BD'
	$sHexData &= 'F51F9487DAF6411D94535B3F678E34ACBE50B21A1F8CF6DADA28BF1FD7AC9F865E481DA270B303CA44A1276EF8BA0400BF1BB5B5317C5D13C3A9DFBF72E4456DDFD71AC4C0663BB1549F4D587AB11C574B5B0C33015F81903403214851EE9043718D9F018C5381F5B8A597285D40A3081C8F489C27FF5C3A0800E14CB124BF8BE201FD3AE01439728C4A0982F3D457195FB467E953A006FFC1802C1685A3418C43C7BEB34A6A2A78015F7079B09A48BC561B91AA230D565BFF6091EC2D000E340D60484C2071F00738836548FE324084C0A1E1A5F00E1928B80B16BFB17E32FF0ACE81C5FFE80ADF44C8240FC29C94B164E4C908661A078255C5211A03FFFF13F26A419F1E5A18BC31F8B4862C3F17607EA2143CA484017C555F155B21AFD1199FC9330472812830178134177831E1877CAC9218BE60600379792A50A8901672259F9150086616682588B537AA375F28E45D4CC5AC7FBEBF1E30A8567172B70CC0BF5FAA98BC3D3D32EDEF155F8F1DAD64283E574B5D68CB3822367D607C1596C047E13F5A505CF4DA6278043EEF53BFAFE4281958507F2C0852843C387FEF008609503F40BE50012948291F2A2F84BCDB9A16F820D28C4C1D4BBF580DC0E4C1D25812BEEC1E84BCE2BF2116F11541D25CC15F50C1ADDAD1A23F5F1F3C030B553F3F3840C772B585BF38862E7621B73F40781F40'
	$sHexData &= 'BF3830C17FBE9F2083E0022639C92F20FD488DA0128DCB00D54B2E12DF019027A7BDBA37FF03CC40014A2A778C1F20287911F2C17F5001E011235772086E11A0FA26402E90117830C19D36405F85608E651CD2C03D1883A5196B302A5E303A96FF7FAE101964E44A303A30449EA92DFF0113157CD9186AC5719301B9825F90908605F290407A10DFE4C895BCFE0F408A10C905A9ADFF5F90C9695AC8805F505C06AC5DC828301FDF8578C08A1FBB6940023988111E28BDA9112B5F55CA5D03462EE314CBCD680731E33E11FF015FE41908101F40489246ABB63FFF30C8D546C7BF3F380AADE2E4B91C703F5FDC89909CBC2CD0B0CEFED5F52A969F047C1F646ABC8E7F116CBF20B7924DF3456208BF0DDFF255B0E446CC1FE14CFF64A20343581FA861218F969F2E4844D7B0A41E287FBC9F9C565B32485F9E140939511F40029E64429EFC1F00FC9000F01826F6E43C1801BF08DF854C94B8E07FB84D737062BF0818F71E572A4335992D0C0458C14A9FFF769157320BE808845FE41413C31F685F250CA0C0091905F807BF7B9999311F10F30004312C3621CCBF5F6A814DECA0031F80BF5FBD482EE4403068062F76B1893F585FD01F20087662382110011F1004BFC8C8C526485FE8BA32212F36B8FF9006A81D27AE1708021FA8059FE46838581FFF02B1D5D0E3FD5F06FF5F82E2737204FB'
	$sHexData &= 'D0055F9B085EEC58FF70013F803F0EBC18421F0006DF8B69620FC0037F987F82C52E86B8DFE05FFF36114DC5FF047FC0FF17C3C52E981FE0FF0893DF131D7F0D29911FB6FCDBA9C092193FD26C3FC091DB9191D0BC1FF9C99CDCFDBC45DF2695C705830A0018540E61FE064EF4D6213C1BB718CC5F8C1C72B91503552E92F762808AC61C3A205A0899363F9C2C4990E7B577259FCDCC70A39162A7CE4C3FB5F33D014B46865ED1DCDAE924E8F08F2F057A72E4044B46381F2902FD79A72D6646324F89057190C0B7E4ADED5E497F909697DF3DDF02661FC98FF204F936F3097F9EDC488B9138DD0800CE53C17F8B15347D23A09EA1120BC73C90159367E4761C38A17FF39C803E833D1488B90DF38724EE23E97D082F05F81C8AC4F67ED705E914660CD287219F078BFF3EB11CE88F1C9243C83D1C8FCEEC593887AC048F01544318757FA050089C77AACAFC186B4D49DC294D056716920400AFF2590B61C8E67C4F61CA66E183A19E9033F606EF8D029F38DB743E163DEB2A292D813C22C3780772B5BEBF4B28B8544735A58D0DA6AAB27046A7DB6FE760C35F1265CB120791775F41B30551260C524E4B5BC3538ECD0174225F7E2C1F4AFB029EC18629B400BEAF0A6001A8C9010F924E4E4E4E786AC8D4230F664EB87F3CC74A4E4E4E6E0F5A627A904E4E4E4EA0B0C2DA234F4E4EE6FE0EC8'
	$sHexData &= '266E2727273642204F5EDC26674F06DDAFF0DC0FC09C9C9C9CAC927E6A9C9C9C9C4C301C08234F469EF8DBE810C7FA27B79391C6E6D81FCAB8D396912750C8A8018E234F66B7D6AFC90FFE26CA54272727277690ACCE93912727F024CB4CC9C8939386D010CC5264E4C9C9A0DE4ECDBE93DBF6E4F63ECE9F0FCA4F4E6EDB08CF6F0F92D61AD0272727235EA2E64493939391D180C2FAE4C9C9C84AD28CDC28D3F2E4E46474B6FE42D41979723284C206D53E3D39393980C0EC18D6E444F1B64F0F5CD66F20DAF2E4647632DD0F2A20C4DB727272328E7A64527272727244342C1E197972721204FADAF239393939E4DAC29E8C3C39399280A4D8B29E9C9C9CBEDAFA1CD94E4E4E463E5A7EA24E4E4E4EACB6C6D04E4E4E4EDAE4F0FA6F27234F02DA0C161F3CDD4E6E27232E3E0F48545B4E4E4E5E6872014E6E374D76EF840F96AA4E4E4E4EB6CADAE632034E4EF8084F10323232321E2C38486F93B31322D74F06D71FD6D64E6E2723566C0F849C4E4E4E4EA8BCD8F62727234F14D8304658452727276C86CA9F231F6DC9EC0F014B804A93939393205080A06EB79393C0D0F07F100F504438C14E60CF2BE85F491EFC0522059319016E98893D16F6BC031EA0209DAF70C0F10513BBBF78DFD46FD54030541B36DF4749006E00614E006936E7788EF7927016749A126F960AEBDEF36B2A6D22'
	$sHexData &= '67006566271F74A5AF2DEF1D461A697A652E6F0A6CDE3B5A3B2A647B722E6D13AF1BB4CB63008F6F740E73F7DEF3DE006376651A6E640E765663AB4BF09EDF63526E007478576B73D9BD8360B35336753A6F0E185EBB702A62653A730F73BBEBCEF736615B68FB6732D637303038EF701E63936283A760C18FDE61653B6AEFC3DD726A2A0776476F9B7647C14B286A1F2922B5B71EDE624E74ABFCEF6B6FC13254FF7E741FF9DE1B607F523273C26C5F53ECC5F7DA5A69A667976F0F4C87C33D4C6DDF6D27A7BFD972B15F1A6401553A6B006EF03A3C3C77771372573E01FE0DA6067D005B005DDA577BE03803DC677CEF8ACFF0C1BF7F5C007E0043004106480045005F0E4D0047BABBBD775E4526522B265600320A572F827BCFC76F4400454F0F362FBEAF3B385F6323000B53BB4CD06DDDF64A441FB7325B1F6344B7EF70EB121F292F2C3B771E8E1D3A5F2CD26FCBFBE830586D2FBF67AA742B83E390EF2C43702C77DCBB7D6AB63D27665B0027000FB6F35EB36942652F27531C1E6E051B002E4256A4E11D54924DBA50BFEE5D695B31C3E73007FFE8C7B58BDF3D2F636B74EF9037EC1FEF593E4E70363DD6BA730C773A072DAFEF6ADB1C8777AFDB2BCA8DF606E6E2633F6BC1EA0EDE0E2E5FCE698E3B2F3BBADA363F6817377F352D765DDF73AFD7EDC177B05F629F6BFF64339E76F0'
	$sHexData &= 'DE282A7592664FCE737A4F5B28D66F4B01C3BBE5284E16EF702B7200FBB6749C790B5FDB755FB60B3018C2B76467C5067CDA4FEF647F7AD67404D3783F2AAB180A4CCC5E014953531DF07405F344224C8F337DBCDF7B7632125BDE36003412589E5DBDBBF7DA0E2A3006353E31120A1620C5AF77E7023A0E86618B56BBBE93AFD7AAC26F2D1F523BEE0EAE3E4E5A2EA216AFEF7D3F165EB76F1F433655A3EB80871DBBF225A77C27DEC15BD5071E25CE780B7B0C02CC3F4D4B0E37DFC9E35C4C839E4B435C326C1C9B932F422A2996067F535062DBD1E89AAF5353477BF170B9C37572AE322B6C02C8FFFFAFCF657450726F6365737344504941776172652A3E75C247AB627F5C5BBBE20A0E5F2F5F5B0063FCFF6C1501696E76616C6964206D61702F737DB0FF17363C543E206974657261746F723F737472DB66D8BE146720706F733C696F8A1F4C6F20977677B36C1CEE766563647C2501352935B376522B8E61AB6D61B6A66361361FFF55D945F4109E78109D502FCE6E17C877E81871400D2A9C6E2747CE2F7648BDF0540F4EDE82CCCE2F94906F975F97A0999DDC4E420F3C365F9872BBDD1E009A102F300F2AFF2460E43B397250906B008893B3238FCF71D0BE800FA4301372829350703FA7C0A8B70982B3E00FA9503FAC3FADB793236F0FAED0AF60C0FF3839C57827B0AB2FD48A64'
	$sHexData &= '1BC903DC8A12F03F856D3410730F7CEDE5D78F025E088F01A08B0A058EC88B1C39B4844F108C388C266483942F788C807B66BBD54F7FF08C9FC80A4EE4C809E4188DC08D988D1D395170E88D3F648E8C8E44F264A24FE0188FF0AD64E4748F9C8F8F232FDA44DFE89F5C90843621279090EC90F44F6EB0552C9F077FBE6667BE044A5895DF5892087EA8924DF2DA4ADF1C93072157B0E71E24585F841E4CF25D032A5094051E6894E44090427F08955BEC447278956098BF1E4E0CAFD8385F5499BF5C9990133BD1BF0C9A5F149AF7BAD3A53F782F500E065E27232F92900C9BE424912327929BC09C989CC89113C9D89C5C9D349DD370A2E1749DFFA89E3F06849C483EC89E649F973C53EC6C9F5F02F8093613D27CBE08A0983F7F4328B89043A0A01F189E899DDCAF5FA4289F7C9A23AF48A120A10760A1F2E43691EC5FC404A23272E44490A268A2A83872E4452CA304A344A379939C28DFE8A3025EF07220C00A0F2CA47232013B54A45FA8D0141C790120A548A5DFD89BB84E94A59FAC1E28BFEC4D54AF90A6DFFCBF5F9089E414A0A6D8B41371BCE8A6DF2CA75E2879915C8160B0A7EC44C109B8A77F18A85FC5260A4E28A87F5C5F905714CF64305FA0A89057C89BB8BE380CA99237859C14A9065E48D6AD90E70878FFDCBF37C550B90AAA9F055E21CF33C980AA06A82034EC965CF8'
	$sHexData &= 'DF20AB9F146F424E74AB7C5FBFC944EC1428AC5FA0EB999017DCACE4281D39F28AFF50AD28AD68ADBC911350DFCCADF45E30511D799548AE70AE7F422E6213AC5FE889103DB09FAF9F086F22AAAFDFE41E287B07728ADF24B0341E68BFF226E415E4B08C5E2078117205B4ECB0963027A0BF30B13F9ED989AAB17FD4B15FAC0D5D8134BFFE04B23888D1F2AAE05E301F78B3FE8C7C5348B3065E18B4F04C249BF00C9E48B4B49F2B36717FF05F0238B52B823721481E401F90E6D7B5E85F26FE18B6E87BDE24AF70B7171E98B05F6792CF053494B87BDD7A92CD34BA08046F1283442090DFFC8A411E803F0AD7233CA040CDCCCC3D898213723CBE44BEFFDD8A4DC874BE845FBFBC908B7621C45FECFFFF0BB9F452534453E63360C642DCDC439F1A7FDCD47F0FE167DE4AF3443A5C446174615C5FDF6E8DAABA6A165F12432B2B0A496D616765B6DDB6FF536561726368444C4C2858502652656C2073653717C1DF2D357836342E706462370042ECB48BE0668006C0014AEC4DFB33813F101787ED3E6E55370C6FAE8F64F3B91DA8EF4060813881DFB5162907783FA816E883D0CED36C06C884A0780125AF822EC7042F608121CF73320090F8D0BE49F6585710822816A086C9D5BAD45F9701FFF8F60D9E5C9860E2AE9E5082FFBBA6BC0903B60A3F20F6354C0CDFE81E1084378057A4E0405B'
	$sHexData &= '0110E27E086F80E0AE10AE2F0936308682602537E0C945D01DAEDF102FA8C792A75B3704C08745322C04CF00298213ECC0E14F38841F9B0039C95084108428CE90074BBA8F081F85C8C5DE9950234F508150F05C80022FBB36C5F14E84F70887F816B00AD6DE280680270F474C83D12F0EAF681B346C6B37B70FAFAD5B2003A82BC0374532B577D0179F00A161B5875F4F5F21AFAE568709EF788222D96C4E10E3708648869B404E728886F086C0FED6825DC900C8B70C425BB3F737A7378A60B02A8F5E9F13C9667368E44087188758E095BC6B3FC05E70E8E3DA1472829F9887B037806010DAA7369FE0C114F24087980698883C436C33CE474FC0380C1462F71006013705F2B4D73F40896670C8853C60BAF68F9888D321B636B037B768E38F292CD804F0AF4CED4DF2860818179F00ABC96191478F6F07620F6F01110F8A0652023064FBEDC41E408E660197904516B8396E360771D5390F3A1F01677D336C9E06326E41A7DF36453B66FB0FAE069E6EC71D0C001DC40B06740AD7FC37DB54091634083219F017E015D01915152E7DBA3FCD156415155211C08F02CF3EEC7923BF0EB53F0100B945C91EBBE9AFFB8BA0360C4A0C2A0917060017340D06BE37047F7213C0117010600FAB0EC406034633FF0DCF0F111805BE6214C01270116010304F88D36C0E0742274244AF39AD79C60904'
	$sHexData &= '423F37436A0670CC7EF0B3460F010A04000A560A3206701706232F37BC1113760A52064871A00571537C66B0E727C60EA79BEF869B1F016F2126D0BE0BBFAE2D92195E3E21052E05A68377C1C027012701376E06C4D6FC2C05002C6403341E0270D4B0B9880BFE4AE404CE0E3DC4C3E0BFD9DE38640D4209F00770065005D6CEE67DA733D7D4448EDE19139EE2543EA8741F600A7632065090A2CD3CCD3E0D621F5F670B74B607B48B8F01B0466627F8428F702AEF802A67B5F69E81DB1F3E930E01DF72D2542EB81209B83C999057248CD0B0292723478EC229E029D0DDD9B0412711BA196436143473D2C2B006726247E0F064478E346E0C0A0DC30D1FB1E839104F087225562564C3BEBDDDE6213410967210F00E70D00AC008703F021772C15D57036FDC204E7757914747160F10C3996D0BE30F3F6028B3A98833476B090F0957AABA37F2791E02620A880E043F0ABB7C84CB5FCF8E2D474A1F349581609F30282892CBFFFF20340F0012720EF00CE00AD008C0063072C984BCAC8D907091CF057280D010073972E4C89007C107CF078308C166881CAC08B45F9F2A5772729DBA581E559E8BAB2E1E740DD60C0AC089E0D1B137A7788E973B3BC193E090FB5F3FFC2F57FC1F9B928F2E3E87ED4780F7855CC91580F465AFADE310482F0F80F71717D98D45881F59F85EC1AFC807171D483F'
	$sHexData &= '82273CCF2113A80B0A08905732D252A8888F91133C994070EB07D1EB80C9C9C8E9EB404D95CABB799F72111404D60970D0C8D9A73B5F70B0D9973ADA9B7F40E470DA0660BF000076F48A387FB0237406270139040039037CC7068F190B0A0B42075006B63B689506D7F81D361DCE18F6AF824F810EF7F8CF709091CF3321804840ABE464E4C8C2ABD2AB808ED336CAE8215398A0A706BDB190A6D9D49E2604040CAF037727542F207317C891B3CFD0C0A3DFEDA34AA4A01976FC191F05000D7E010DA002709F66735F472008EF06305A799491F7F42E70E69278C06822F440E4C022DDE4095892C41168D74C6C9324540E5F9A9A46F39EE7BD912E316E2C27196614E1EDC0CE055F01073E07A203603590CC99DAF740582B92E7799ED7F70CE60405060D7BCDC49E0B0A4F7474466F2ED6CEF3DE1E5E13560E054F288FB0CCD7623782E830547CFFD76086215598ADCF1F21B682F0E428B62FF703949239719C0AA780921F30498E27D73417F080FE4739193972EEFE4AFFF0FD3BCF9B3D0549CF140E003EDFBEC2B6E00FFE0F74120A548F193037FC822B1E2264185E34170012011206C00570C053F0A10460457E908A3974F66DB77E50B91737BAB7939391B3BC1F1DBC6B81F776C3939D1934D626641BCE341A0016C6674F06F60BE009D0CFC070B45F8DB5236733025FB71F0CB7289DAA27'
	$sHexData &= '274866B777C0AD064BD5A706F610997727B6CDC7059AD43E7490A42E4AA54FDBF16C2C940FFE0FE4360A05B1379A13063F9727BC35C03A014FAC546E084204C0215795785EAE04E60A23A778D3F0C7804927B049D05B27373D10E7566FC26D8E53078CA327AC3F21269B55C607BE1C8DB494AA3FD68A960127E70F344A92E444ED0F010802605F30080D5F420E20741F233E1A51B485E87216D0632F5F3921A7E444953095204A30E0C029E0499FDD2EBAA82FC07F35239D19F9E8DC0E1F7E9FED49F53E448E3C044A2E4A4FFD7CC6E1D306FA2C964E1E6407AD4162637E2F6FFB063F97D82B3E5896AF64062796069692CD80F98AF92F014236BB199770A720F62FF66EB6D35A27DE860F340F320BAAB1A7FD5F10F406338CBF166D22AC9D052F8E260634946C179836C2BFEF6613F0AE4231B02F01C0A654060E742013C6B317DFC1A7ED7F90EA06E32097C717F93BB5177C060C9767C4082754066FDD30B40701361E3616EE093AE8D42E65D665C66F54A75D7E60C6153211C0E7062E0EDBEBB6FCDF173C067CB79ED31EEDD42754076F30E5064C4E81ADE230985F27A7DA7B49B6E366EEE31706E8D7056B37DA27D66FB610D9DD6C37107409066408320CC0F760E103D612B27CE1278F421C8AC1EEB7192CCEE9027CD36414A63413000ED2BB45B2DD80A7A6506A874A1640DB399B9B38CA'
	$sHexData &= 'DBBFDD1F50DD1D8737196CBFDE76064BB3E021C0CFF662D0D82D583B15460027B77261A4892701BF46640D06BC1320C17CC15FED670EAD27C7076001148614D1FC779B6414540E261432107001121275E1FFF1DE121E12320EC00C700B60CF13343FE28074E7804AB3C7D06683C891B31AB40FE0AEA3064BC814AF27AEECB420DBAC1E70AC460027B9606D1B6F277F0E49F35CB0B206570C0B52FE4DA5DB1F261F54116E46099205C003700260472A270D6FB090327833AE019123F23397E0D85E787B11D6085204707A270CF87604C7020F649A013F71664B0EC04A27A0EAEC6C98EF2517A12597AF256606DC12B72FC00FCD53722070C727107510C90B999275F8104B2727974CF010313FF9E026276F7E271D4B57285D6267588E062EEF7BC5B09B77D4091FDE269C0045682BD61F9747BFDB181FEC0897010D064606320906D5B6E0C00760EE01A03EA07427EC16D28A6EA126A1DF9005A76DC0D7B206109C9F3E0505F9026B2F6F5409EEA8349BEC664B014C627662740B7FB0A76BD47709083F010C03000C1AE850A4428A50CE16A833506C674740E3D207CDFF50470E3FF2AAF8A092474067AC9C2779251360404BD0F01F7A2367370FF1240E01A6F153F2C72D7E8F6D4B702B1922383770C9954CC8489DB0908F1B397BA0EC37DFECF40E9F30E4E4723CED00B0BD7D7CCD09E730E9F1'
	$sHexData &= '06DC9D7F81C576AC1F5747B2D90D6957646750D86ED85A8BC11227E7CF063AB0962A7C349EB727F7B45F1A47A780C206A98C5F47A5F93C1EA901C3780E56D89758650E74062FAF0877466FAF3801071115660C520870BE6FC888AF87A075372F104CF8DC120AF70F4020C07CF91267775BC08F7AC09BC0006C90BDACDEC006C14F6F40B0C1D725D969B017C106309F6FD882C1122F6F771E6A3C1C1927C6B2BB60035049C727B8F606D0750E523B72E4C150469083F783B7871F345B9B2B3F12A656CE7B57BC21ED60A73E1F084608BED9EE7A01160E165414063413B212F010E0C35F3505C728462834240018011C8EC3CA017FF00AE008D006C0232E9F3C1839B20076604C7036204D4E66478E4B4E994E1F9ED2A5704E4ED7E47F45A8C89B3F9E5407E4C883DB30D780FE7021AA21382CC2DB1F192D2634148669F5B42B534576605A9718544C691C4C19171A87D81B3EED90173E05E0A047744D27106CAF34A74E06014A269E8FE6DC5740020CC63250891D56479076E742156228C27DC37A7F269F2EE413F234A14C0100484CBBCBCB43603289329A0E133999E2ADED1F013317C0CFA87214DCD63FC01BA8C026A0161BDE0D52F67042AE82279A2F47D8729253F2204D004DE02BBD8D1C39092C1A2C4F1F5E2C13C807973F8E472F4DF0E4955C20F07CA2BCE44A268060C02A9123478EE9'
	$sHexData &= '2AFA2A402B17C8C9C84F2B808F200ADE8A8077E71F375CC9843C18A3E0C093939197F0290715E464E4C8562A652AE0EFC05D30D0C78611260128F560AB1F27011E7E1C22FD4A561E321AF018E06F5001663BC62AB654343E0B06540AE4E162D952102E5820540DFE0C1C79670F5077FF404E160B790B23E451018F8077F2401EC9807740A4704EF2F2F2F2A0FC68FD78FD704E188047F27D4EB077675CC8D58EB0F7BCA09123479EC0F821F931F940B99293A0ADE0B5214D434F0C0BEFC893097934A5D0F0F4723272E4ADF5BDF5D0DDD35738D60E02163019294F3C05E89426B2DF107801B79CF8B1AF4FB70F203131CE3E10D33792D36FD49E5B7709970EB7DE1FF402E24164052672D5B70B9C7D57E09F2F40D670CC9FE8CC6E23478E20CD3ECD761F213633B0E9D7CB25CB4F001789DDA46F4B0658A6970927FBD226B412E6124E0D548F7FFCC09558FF70570A397282837078E7C053E0532D0371E4C861C86F1F20443ECFC6010DC9B9E176E97F92A08F307750333B72F6F0C58F40C68AC61FA10A8727CF01EB1928572E5D782AFF1534110008D20470D06276B0B139A1232AC257C2679123474E007970AFEAAF84F9A1C809B0B70E566833FEBF14924E3109361501775FB83B2E14010E029B689E1598139530083F52EE8D9D8A2367AF838EEA8E57008980CDC5A7062EFCA70E1313C498'
	$sHexData &= '434B29F69E1637A11711180146097205D071E8C0D006307F6079179C608884A734F79B34E899428EA634906A3972F6B98F80802C773D2D1E2E9AA60530071E1D1815276F40C086B2908FA47314CEA01FA725B56E3A727D264227372757DDCECE342807FA291F2E2AE72B79933437EE1C3416F2F0D6723672E43BC0B650235D2423D753FE0AE119F61A3434000901307740FDB8B3207AC0CF50BD802D48E74F90CF20093ECFBB2F300E0440C01C5F3C6FE4C8581D741D951E03B8463ECFF304C6051E1E4CF14658E00E6FC61A013600E7C1603AEB309A275064E4C893AA1051805090DDB975E7C0CFE02FF0CF00C891137C5EA01A37FA1A1D1B1B3972E4391B5A1B6D1B8E1E85B327CF06AC6E1C6F821CE1665009B72A20642D655614E0EE2BF601300870232F986CDF807A474051809107B79151A02FB02640163872E4C88E16AD16CE16E916F9719B82E717E722B6681532FB45ED311601000A7009600850EFB08D3C19797052F03052503BBBE2BB2F600E30124F12AF8412FF478E1C39A312C412DF128A13A16A3F22CF64FE1896A793ECF1D7CFE07AA052CE21B7BA9138DF0E874D3901522A7DED703616A306BCAB57DB81BDE136EE2F01131E135466A16FB60B06340A320FE00DC0C7AED85DFA9A357F06F4AB870C27FF52B83D01175E17540E775213F011E0DF2E85DF0FD0766019269E14'
	$sHexData &= 'D6D206700560E004F85A8F107B8F8034DB498E9C57E13429351033DE7133A92370366CACCF0ACD97D6763F37B6060BACD83E3A6B19384E29546025349E74FF3F4ED214E012D010C00E700D60EF409030337253471E311F9F046AB7D77676213419BED2C3CA0DCF4B76706ACFC08CC991B32D87392E862FE3ADE4447FA07B2FC90339740F3CADF0529191491ED0522B2BA5DB6E972BBB1FCA3F8F00380A812317A7D061161E36472506660CEF0886426C2F57E04746CE8EBC5053F021D7AB22E9485E3EE20E4650535E53248F38F1007C37007C978950785CAE5F53E074377286F947A9FAB90EF7905D5C989C9D3730E73672F6C620CA7793CAB61FD49FA14D9E01FA776030C9E4C89123A3C9BFC9DDC9A74D3370FCC9A78906D4EEA878D13CAF9FE41A06D4D604C4181A10EBC1370114BE1FE1D682AEDB60053076192A0560F53257E62FE6EA19A9C46F9F7CAE0154C2970A1A541F0E60825E5E5E3E979F82AB82D9821C83DB4E5E5E2F83498308365413DEF272E1898E5BDEC023DB4D80F74FBEA06027DB8ADB916133617F378E2968061901BCC481A21BDEF0938EF0C51478F2B2305447540EE00CD4C85D9659885E7B79C29B2F970E00D65F7B5F1F21727B32BB0960FF60CEBE062A3F7DD6354F2635E43325473B3550202C74C53872362C4B2CCF0A6C171DB8FF807DB7546E702A04DA468E'
	$sHexData &= 'BE2AD61F0FB855880B5EA72F876EB84C821F44B1DF1A7920605421FF6C32C3C8EC22238F60E79803AC0CE033099E3EC715F02AA609E007D0337F3E2191DE3F24FF261772F2F605C0D0C289AA23EF54279054375EBB65A70FB03F901E0F8F30F2F2341F391F1FD74F3CB397971F8820BD201FDF0429357BF2EA0E215F28BFC281B69135D0D7E61A09D3814DDE0DF2BF7E3FE3A6F82760374FC80E4E2338017400CD2408079E06A88023A07F8628DA88811D36D8B2DF27871C94C0FF547043ED7DA32CA6061CB36E163616585B8AB0DE11645E543FFCBFE200CF46E004D002C001210C0021640DC1FFCDD7540E340B321DF01BE019D017C0157047B1D279DE861414165412E740BF407EBF13F2E8B81E8F8CB3F0E4C8912780A090A1D8A12EECAE64F070CF2E07EEB3C97380690668663723A4390216787603043C92077204B440551055DB79DE02570E02303F32B7BE44A71E709B7F9CF33C6FF6569CBF791E02CA0353EAECC9E3589D3F6C9DFF9D83C9F3667FAE1E05C61F1A7EC3074F7BE67BC4100037E4BE2B950A6F22660A8206D0E3951B5718562626741396662253F3711E5411C610B28B01F0455EDEA07E96808F8F7276022A8307911F1591CE806132CF8E4E3060378A561B560EF2B705419AA6C67E7A376506A0C190FF2E5F2F7C009CE0B331BFCB3167763281F069939306D9A8B59F'
	$sHexData &= '065BB4760B279F2709A52906AE17B257E03F3D97403E0E002D78278628001D8C7088FF0153000FF00DE00BD06F1AF91BCD4716007F0217A05510797979B055C055D055C891CA739E04F03FA5401B3972E45D419441C541E52FCE760B6AE29764424F6E421C6F23479F42BD3F7B440FDDF5D6DD990ED7FA1F04458745656A616A0745CF45E464E4EC4F455F8745919BF1E4E4E4A5D7DC03462347CECE9F0F467F3D464746516DE4C891465D468F46945F1C3972E4964CA04CAA4CB64C9E8093735FE64C36E632D44A2169287F7FB98B3CD9AE320156010E1F3CCFFBD00E4F0E0330C0A6236A22F76C338E82B7D99136CB362FE31F3E373F79DE6D764E372F531F6B2E027A478E3C397F5B386A386F38DB0413439F38DF981F236F27B7CD4FDCE12F053F147E90DB1D3F191F2866190055C863C34D9BB84DB89045A892DE9893FCABFF26169400337895002F68967638011C012E01871C584CE313470E9D50CF60F2C7F056279E5B646456562FB004799EE779C005D006E07793A73D20570E0830003FB63C6FE44057500E0C603F27CF3B9970800E0690A03D272727B0C0D01501B669CFEF0F8FF03FD07272F2340E19102030F33CCF73401D501E60A5D03DCF1F7020EF58DFE74EBA2D1F0F1F24EF0E9975D4B925DF0E15590F10549AEE392025DF592A59BCDBA4D22B590F701E2EA6C373F28090'
	$sHexData &= '30DF5939F1269531590E807F1CE738D3DB6217FE6207266347EF8D9CE94163F715642E0E061EA4868E7C9A650BB4651666F23CEFA5370E0C1EA60DB389A95C7A4067CFC527CFF3BC721E068E079E9A9FE67BB3688FCC0E081E2669093369CFF33C239E04CB00D0C8A7F93C06356A114C6ACE9E3CCF881298FA6AAF2C6B792347CE7F376B3C6B801E0F9F76F6E490F26B5F246C6E10416CDEC891236E6C736C9C1E13A4E03D79AC0E6D6FDE6DCFF33C6F4B1E0050067E148DBC9D3C8EF05F226E2DD37C9EE70E003206026F159B3D79CF27161E3420703FAB0EF6E479DE191EE11AF564710E9EE7799E1B7819C91DF2EF5EFE781EF69A72004373961E499EE79E91737C21AF0E23EA3D4FF379242274254726344F9EE71E8127919D7528E4D31CF9AE757C76290A77E6F33C6F170E254C2A3E7876E47B9E2BE12C1E057912793F79DEC96D461F53660E25BBE0799EE724C51FF72D1B393BF2AE117A597A3F637A952FF9349F77EF1E2EBE7B2F077C3CCF9B1D1F7C3F6B1E307E31D991CFF39132CF7E017F3F3CCFF3BC310E1E671D9719F33CCFF3A406B105E13B72E4C533AEFE7F2B804C805F674F9E37691E1C79DB807FE8803989B32318816F23816F4F9EE7791E06451855B7819F5B4F665FE9F481A7250900170E5F08C90118DEB85F06E3141B37A07F9FADBB916D5916105B1D0E01371F7A'
	$sHexData &= 'BBBA80EFD06FE05905CEBECCF70EB05C27C95CE7B5EDEDF2E25CFA1F085D9FFF767B52E40080B0707F7D5F3C2753F2955C3010D0222627175FDDF5D834DDF0976619191934D0A2823E7E3215C0773D92CD15CF18A23A40E4BDB590A94F96CFBFA506A9405EBFD72BE956BB577837BF687C814CC9F00DB02011006ADF02B9D8C7C8373925A776DEC92798D8BCD68099DA29B4DA9F5E0E28A0502F007E1144D0010156E00041FCFF4F2E3F4156747970655F696E666F404000C4C6FEBBF5CD5D20D266D4FF017F0DFC9F903F2462617369635F6F66BD7DFB0BFA65616D405F575526636861725F1E61697473D55E66ED20403464BC0B939F04A3D87F66696C656275669D3F68A108463D8F8690EDF65F49866240484F6904C182D7485FE2653D3F8AAD0A6E6E377F8A6041C3FD1F1B1804C59FE05FA83D37A75C616C6C6F63D5B6EE6CD42932CBBF82551C6FB3635F0B5170839E59AFDB3CFE17476469706C757342351752670129820F0E2638DA6A534F422252C25F6CB570514F6578636570205CC1004A4732902A3AD1B507007F50C52980665F4F21BC3DB84F5B5175982C114450AF073ECD0004103FB88A799E67B2B616ADF4B0DEC8F6E4F24011467B11900EBAA03E00F806208B37C473CF282D90D8AAA60EF40746F63DCDE60014751470800E9D145F77369B6FA02E0D15408F10150680D3'
	$sHexData &= '6CBE608E40168417643CAF5B95770F7D16CC7DEF3ECFF3C2BCD09C1A6616A0FD04E07D0EB81C06A9A78C1EC31A59A7162CA67D9AEFFB5088B1010FE51650A0F04323D82DBDA72E14500E7640F0A8F7657D019F3E7C16EF97F5760F172CEEB0D77816A4784AFB3C80642E46400EAE7E22B0415818B55F3E34F9B46FF08E99960E0535A6081035D96CBEB35C3576C035623F48B7B63BCDE63F204DBC0FDE545E361B9ECDD49FE05455E60C926055AB14B06C3756E7799EE77D0FAF16A8AFC8981938CDE7C83C5884BE17BB55BF0F165A4691010F1AE03CCFF316B01A23A0A7EDD665F70F0C5D1664799EE7750F1016341019247C67C1E7205E604EFCAF0E92601DFCB452EFCE7A614EAF6C3DFC45475E82E6B829F8DE7DBF8583162E900EE2874E44019C8EEFF00EFD880640E779DD8A2F0FAA16E0AA5BB3759EB3D0F70F29D89DB6B98B16200F598C1E103030784F0EEC060C4E9EE775EB670F3816A038B97BDAE77990C02A8FBE8C300EE08795E0628F974E8B926E61BDB366C4B40E870E16D376ABB0270F97941694F3BAF3BA0FAC16700FB02E80D93E27CFB0B4C02396D60CBE463667DF0E7B97178097E898F0C97A67FB0EA1997678A70E6ED9F093B5AEDC99BF0F9FFED0B39ACD77DF0E7AA0162E80A0F0A168DE9B60770F22A27F300ECB16B8786FF679D002A32F100EBA063CFB0EEF33F7'
	$sHexData &= '0E59A4BE906E0E83A447F7ADCAC0E70FBD16DF4DD19D180EB6A7F893010F3AADD9C07916E0F7D37C9ED70FDE16C0DE25A8ACECFADCBA0F33174003AA0E308BF79466F3AA31AB280EE516040E3ECD4490F04FAC7FF3BC6FC5BF0FCD16BCCDE7D69DE61CADA80FAF17AF6B34709EBD98175E770A760F2EAE66CC0FD916BC756B2670EF0F5E7CEE75E716840F690674EF0E3FB05641E0542E44F7DA6E1D04370F57B316185E775E770F8D16080FA906F87CEFDDF75EB00E29B48EA6300E4B16208B82CFC9D3506B70B9B74EEF6C369F5CC0B7DEB890980E43EFDE77B6B95E24900ED7BCA6922E0ED54EB379B6BD163099BDC9BE1CBD6EC5108F0F56060C807078DDBE0F5E06FCB64FF375ABCF0F8F1610908FB19BE2F33C00C016C017200E0603A7E013C196946FF3345FB70F9E16FC989EA9C0E17CCFECB0467FC26EBA08AC8AD717BCEEB4DD0F97C516600FCD1650F83CCFF3CDE940F0FCC64223F52996C810C736C7F64DF025070E09C837100E1DC9460138FA9E7C300E0CCA26AE50C0697F6F0ACB1698DD0A05AC0F5F0F81799EE7791644819434949B7D9EE79A24A022CCD7300ED9F479DE5A16F4708BCD76E4A566DF14DF0EE3CD5FF00E74CE5F807D6EDDF60EB0170FE717F020CFFBD648E08FC70FA61681F33CCF20A6B4103BAF5B91C70F7816C40FCFF33CAF992EA099CAB0CAE9F89C3C'
	$sHexData &= 'E3F0D6D166E8856F824F0E8AD24FA00E04D3EFE07C4D2FD3FED596B00E134D7BDF8AD6EF0E97D6475ED638709ACD91D740D742D870BCEEEC248F0FC716FC8710BEEE0FD706EC9D0F9BE07BDAA5DA76CCB00E39DB07400EBDD9F7EE9EDDA616D00E47DEBF500EE8D971F6B917F0F2DFD7D0E0065AE1FB5668603F0FC016CFF33CCF1CC0DB0CE068BDD9F7B4E28EC4800E36E377400EC9036B34B017373B6DB746770F37E416D40FCFF33CAF4516C44561B470AB34F03E26E58F573C4FF37D0F9616A09796B190DEB7EAF3C049E6F70E06E71769609D06173CCFFBD6570F772E687785F33CCFF35885A148B069369BCFFAC48A00E82FE930D7AD9BC0270FFD16BAD37C9EC8FD40EAB80F701BD8BAF316A40F832F5BB3813517570F3CEFFBB521EBFE96010F3816E8381A3ECFF363D8702BECE658DAB7E37B300E95EC7F0E8AEDBE181B38ED530EE7EF5EC0CFF3BA55AF0FB716A8B7C1B3E0F33C98D0A5F1DE7C9CEB36F0BE0E07F4778FEE75E77D0FD716740FE10664BF00E1747C0E1BF67E083CCFEB56570F7116447173DF3BDFF33480467AF7D6800EB6F8C2F79D1576C48E0E80F9D6A4019EF7AD013F0FDB1614DB75BEE779DD04E04624FBEBD600E1FEEF0F8716E7799EE6F0958789E090A7D97CDA99FCD648A0FCE1FD106DB70A08AF0F4BFE5EC8F83CAF3B0F4D16B85073FEEFA9FAE69780'
	$sHexData &= '0EBAFFDE40C00E11550654B801868E67BC91CD6E0F9E042C9CB9161C9EC5E3F3C025052E770E88BB7519D817570F410648F84636F89B4C06E89B24E1007FD7F608268D814FB3FD8FD10A0EA88CE0470B77B6EF21AF500E2F0CBECCA30EAFD7BDD9990C2FA00EDA06E7E00EE1013F6D890DA60EF0120E01FBBE70EF4F0EA90666B00E08115E9B6FF6BD8E100E681317700ECA14D49B7D4FFB34D0561616C0600E6E18D937FBDCCE470E411BA7500E5E1D2F8D0696CD601D3120F7FC566960370FEE201CD91CD9889B1521789BBC21BC6E1BD9649BE02F0FEE16547C9ACDF7F00E6B23B4AD70082569BEA7D944B3107A16D09A803A099CE7EA3C1F79239BDD0F94268CA3A7167C6E6FC5E7B026275F0EA617B076EF3C7CAA2956B30EE806787C9ACD77A60E6D2AF4A270BD2A91CDA7D9908BC0602B50A2627C9ACDDB2B10ADE06F2CB6F9AD19C0A101470F4E2D38B239B2FCB2602DECB2D12FDFAA01BC16AC012F0F75799EE77916C4758BB4904EFB9EF655325E7C600E0B33D60CBC6E9D027F0FEC1650A63ACFF3ECFC405806C0DDF79E7AA08E800E5935762EB2D9AD58E70F0236E0AB9A059C376816D09F9B239BDD0F2237A4AB603794ABFB9482235638AE6C0E82392E7D737C4F58840EB7394FB80E233A9FE7BDB3A688890E8F16B8A8583BBD56377BBE890F6F0637780EE1DFEC3BC206F6E8'
	$sHexData &= '0E463C8F480EA13EF7B4EF695EDCA40EDD3F16F8E00E7B9ACDD91D405F24406E4134880ED21C7DDED2163CD4416E426427CFF39E0E7816B878B030169FF7B44306718C168C3F44A67CEEBD21860ED306F48F0EF144F669221C4FD8068F0CBF0EDC46369F9EB945749CDE17E93CB76FF903460F44174470CF73F29C7092B0C780DAE7799A8BD0E7F0204706589ACDA77D988C0E90C847688DE0BCA7D97C0148388E103116B4DE0ACFF3406148A60EC1DDF3BDF1063076162249920626F83CEF0E6F161C80AE4907E0BC77C0F90E124A1E0E6116CCF5DEECF3A0BC4A47C00EE1062CF7DEC867072E0C4B100E3106C02EE77927CD404B5C60168164EEF39CBC902EACB0D1089DDE733B6DF92E0E4CD64C17C08D6C36DFFB2EF8A0004D1C4D20CFC9BBEF0E5B0698A1BE2E7C80BB7C27EF3E3CA2762EDCE00E384EDE864FF3A240664EFF0E919EE79DE306A8EE16C17CD023DBE779F1F8304F5E4C4F5036C83B79166CD02EFE50169B27CFC8765090BE10513651C9737323400F17C0EFDCDE82CFF024521F0E9F17D09DED73F2ECF04853D6FCAC07DFBBEF0E8106A0ADD60EB1061CAEDE53DA9B9B3C5357176054067BEF79DE7D04B1A616EB78F00E0B95F7349B5558B3405B16C05673E49B6CEE71561619577058DA9367649958A0C9205A064B73F22C7167167B803C27CFC999A0B9D0E9E61BF99C'
	$sHexData &= 'F0095B100E475BEDB93DCD2CBE60802FB0CB47E47D9ACFD0075CDC105CEE5CCF7B4EDE166B70A746ACB07D2F8D3E795D169E0EB25D888826E347A1001403215128226441E38F40BC12011F01546C73FF6FDF68E656616C7565184C6F61644C6962726172795768E3BF6F1A536C656570286561766543721BDA6856AE634A5663BF0147FF7EBB6D68466940417474326275746573572754656D70770378F7506174681AAC4C1A457272ED9736DE6F7237B0416464725673FE36926FAA456E746572A46E64436C6F73656F376BED600C62324D9E7A795374A47573457866F6BD67414E6578F7C644650C8E8D83B9648C834D2ED91BC08378C541DC12DA765FFBB54861846CC0496E6A68697A7F0506829762464C09865373051798790854696DD673EB13C536DA9F31437572466E5D9DC10EEC49642954680A94275156D00A0CAE7E5006668C6DF0F60F0C6E52436F75F346746C436170745E5FF104772422A425026F6B757046FEEC7060461348785556697238385C6042556E770076733833DB20F2EA676790FCF8E010F0417BDD3A68B76445A4D81BACC360D933D6DA601491FA2E6DCA613C78D66E6323D8636F641C6F2A676848B1262C1DED7C6BD89A7F3BB33857616974466F725367DE4F62787658CAD8F0C09E440E878F27881A6FB9BFD52DC5E4D10947C5067245B6258A73C71991B146'
	$sHexData &= '1085D2F2FCDB0489DD41F86CB2DB1EA4B69989140CC432444327CB0E9E0459796DD385199E1B5B3A423E2301E7D39441680668552B686727C43547728868F8732747136C2FC9B5C0D653A85B09B86C6E3035CE8E61FB89AD18E93948656967682745F6BE764450772952F64927C68B0753D3531E6FBCB84D04EE4891652BDFC4C46B16A3BF450B17E595697370F0EC9020789B51BA336D81FD17484249544D41503925F33882CC3B4F394921D482A1C0636C61C3414C3D86155C0D146E2305868F093E576964BCA1506978CC0E056CF64F666673149DB9B66A3BBD201EF15589CF85D3B76D46E36A1F88024E829F513F5F4F380E05228B5FF8405F0154067250406B0147303EFBDDFF175145414158585A574241444F46462D335F4A4276BBD1F62669143F24F5637674805744483BE0B67678325630D4A8651240414CA300C3E85870D0E8C5095EDB0F59413F4155745F53044A024D9FBB744A403109CF586C740F13D806645F7234705D5850450283C3DF42443F3F315F716E416C53F29D4058313048A576C0061314968A5F65F4A5973D7EA5B86175CE3E190F090E0B475F4EF78E67DFB6C001FDAF0D4B430253D714F2D456F769968E37B6CD04E465468C70659144E19E05D8555F226640EE058196A05648354E370B193A3C11364240A57048108A5A80B8C187E1C0E029CE01DD4F73D6B1'
	$sHexData &= '88E0667840072B6C2A8207680F498140C1A162C47498933BC6F131428656CF0530B4010513796988595CC10D40F9EB512C311B3104DBD90CA3A0288240B92730998A11BC50383D1B24A8BF56288BFC16745CFE674789303026481A1D87E763473014F2E224C73155A0A901C71766CC6853B4C3B51841682D3F368143B5151730944D40835F4CD546B54B85834A53351C9548114E13B84725503612C3AAB314A7221D32BDA7511C8BD5D35E2BA8BA6022AE4067A368D0022548AECD989EA660EF699C65C10A14B8A94DF5F84219407B4D355D79729904EDC270480B5E830C112ED350E50CD17885648226654D1B3250B292676541666482860664774747A260236C687E8479631BD8332E10CB55876AC785A9550F4C8131401B2D647940869B40283FD4C56918201953E64B80AC1B51FB566661634E404B25340C09B621270594696683BBEFEA0BD448B8488C4431100F0835319E6B526303D25DF66C65107F33693E6C1BB56E7DE10E6FEB8244D87B9B8C7761797680EDB6FC6E6F846E76408F1B8F83488B918362AC691B0C06A405492BB612875D3340A5552DA6C6155B42F43574A360A4D44B532A70A5513F3C588E5531F7002CA101A3E6F2535F09B9BD37E99B166D656D637010737172741B0181EC7B1A425F63385E062B08A531F6613602BA76CD49EF146CFA3764666AC74CB2355F4E74'
	$sHexData &= '58FC2A0D04633D9D74C132BBB7C65425D2F082BC13EA5CA75F680A2943707058B8EC2E152E74575FFA16F452836E6A69376509649D850E1624756C106607D2375E609D36BB5F19146D95779BE19CC9184A8524CDF4AE86DD6438277537F9439D1AD434708A664679DBECE1DF722B787846720A482533266973F01B17F777646967784E32F7582E7BD8AEF3307072206690F8327072E7746F7787F6B134E32053552733107B9032516F3BD589F1811030C97768283AFAD6A0A9B9429F6269A3ADEB3BF53516686FD8BB51FD43FF1578B8C76D6D6F76F266F47714B68E876399C718B4E7D2BDF72D110E660C2E4E59E2BE49BE656B69363426ED076B93EF3745736E73E678D6E206781E765666108370CA20CAA4A345687BFE5033D1415866DFBC666D91B9096677D2659B71AFF5767A6D702063D95F227B4C6D5042750500398A54016E6BAC0EB320369276BCA5684E15E2572EDC56DC339B6A2D7C354D10AA7C0A7EAC344540756D4368EA64ACE962CD2173CD734250DED9596AB73FD33049B50340A9541A6B518EB5BDBB3A489F953F7670D9CB195D1080DE931E952DB56067F06C9FD96901664D6F12A27601F114F23FE41C56081858081008F0D008DF773CDA1823F0C80050504828300A6050FECF96FE583028285828000A648606004376F468863F9C22990022200B020AA4EEF2CE9E4EB0'
	$sHexData &= '27E03F16108012B44E69CD1702C009000F6CECBDE7004004132240690D893A4B2D1F1560DDA673509F505A0CC0D07B20AD5AC6D9E8038BE013EBEC5C08A21EB8B8906601014D21DB14605E10062E0E0E5467AB0030219FCA670FF7048B8C461DF4202E72473611A0641EA6E27E0080E153923809EC961EA29E677BEF05042E4D4E9814E006DE833C79D2161FC02E70517CEB0E0BC7765FD89F361F287C383B860C726327206A926F37C7EC4F561F3C6593E755DA7634F8304E02F0F77DEFE4661FC24237007B450FD10C499224090000002AFF0000000000000000000048894C240848895424104C8944241880FA010F856C02000053565755488D35BD3DFFFF488DBE0070FEFF5731DB31C94883CDFFE85000000001DB7402F3C38B1E4883EEFC11DB8A16F3C3488D042F83F9058A1076214883FDFC771B83E9048B104883C00483E9048917488D7F0473EF83C1048A10741048FFC0881783E9018A10488D7F0175F0F3C3FC415BEB0848FFC6881748FFC78A1601DB750A8B1E4883EEFC11DB8A1672E68D4101EB07FFC841FFD311C041FFD311C001DB750A8B1E4883EEFC11DB8A1673E483E8037219C1E0080FB6D209D048FFC683F0FF7458D1F84863E87238EB0E01DB75088B1E4883EEFC11DB7228FFC101DB75088B1E4883EEFC11DB721841FFD311C901DB75088B1E4883EEFC11DB73ED'
	$sHexData &= '83C102EB0541FFD311C94881FD00FBFFFF83D102E81AFFFFFFE960FFFFFF5E4889F7564889F748C7C6004E0100B2055357488D4C37FD5E565BEB2F4839CE7332565EAC3C80720A3C8F7706807EFE0F74062CE83C0177E44839CE731656AD28D075DF5F0FC829F801D8AB4839CE7303ACEBDF5B5E4883EC28488DBE003002008B0709C0744F8B5F04488D8C30E46302004801F34883C708FF15BC10000048958A0748FFC708C074D74889F94889FAFFC8F2AE4889E9FF15A61000004809C074094889034883C308EBD64883C4285D5F5E5B31C0C34883C4284883C704488D5EFC31C08A0748FFC709C074233CEF77114801C3488B03480FC84801F0488903EBE0240FC1E010668B074883C702EBE1488B2D54100000488DBE00F0FFFFBB00100000504989E141B8040000004889DA4889F94883EC20FFD5488D872702000080207F8060287F4C8D4C24204D8B014889DA4889F9FFD54883C4285D5F5E5B488D4424806A004839C475F94883EC804C8B442418488B542410488B4C2408E928DBFEFF00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010010000000180000800000000000000000814152A90000010001000000300000800000000000000000814152A90000010009040000480000005C70020088030000000000000000000060200200880334000000560053005F00560045005200530049004F004E005F0049004E0046004F0000000000BD04EFFE000001000A00E90702000F0002000300000000003F000000000000000400040002000000000000000000000000000000E8020000010053007400720069006E006700460069006C00650049006E0066006F000000C4020000010030003400300039003000340042003000000082002D000100460069006C0065004400650073006300720069007000740069006F006E000000000049006D'
	$sHexData &= '0061006700650053006500610072006300680044004C004C00200058005000200078003600340020002D00200049006D00610067006500200053006500610072006300680020004C00690062007200610072007900000000004C0016000100500072006F0064007500630074004E0061006D0065000000000049006D0061006700650053006500610072006300680044004C004C002000580050002000780036003400000054001A00010049006E007400650072006E0061006C004E0061006D006500000049006D0061006700650053006500610072006300680044004C004C005F00580050005F007800360034002E0064006C006C0000005C001A0001004F0072006900670069006E0061006C00460069006C0065006E0061006D006500000049006D0061006700650053006500610072006300680044004C004C005F00580050005F007800360034002E0064006C006C00000034000A00010043006F006D00700061006E0079004E0061006D00650000000000540052004F004E0047002E00500052004F000000340008000100500072006F006400750063007400560065007200730069006F006E00000033002E0032002E0030002E00300000003A000D000100460069006C006500560065007200730069006F006E000000000032003000320035002E00310030002E00310035002E0032'
	$sHexData &= '00000000008800320001004C006500670061006C0043006F0070007900720069006700680074000000A9002000440061006F002000560061006E002000540072006F006E00670020002D002000540052004F004E0047002E00500052004F002E00200041006C006C0020007200690067006800740073002000720065007300650072007600650064002E000000440000000100560061007200460069006C00650049006E0066006F00000000002400040000005400720061006E0073006C006100740069006F006E00000000000904B004000000000000000000000000E074020070740200000000000000000000000000EA74020080740200000000000000000000000000F67402009074020000000000000000000000000003750200B074020000000000000000000000000010750200C07402000000000000000000000000001D750200D0740200000000000000000000000000000000000000000028750200000000000000000000000000307502000000000000000000000000004A750200000000003A7502000000000058750200000000000000000000000000687502000000000000000000000000007C7502000000000000000000000000008275020000000000000000000000000047444933322E646C6C00676469706C75732E646C6C004B45524E454C33322E444C4C004D535643'
	$sHexData &= '503130302E646C6C004D535643523130302E646C6C005553455233322E646C6C000000426974426C7400004764697046726565000047657450726F634164647265737300004C6F61644C6962726172794100005669727475616C50726F7465637400003F5F4241444F4646407374644040335F4A4200007371727400004765744443000000000000004176F4680000000018760200010000000A0000000A000000B4750200DC75020004760200C0940000F0980000709B0000209F0000109F000030960000909200009093000080970000B0990000327602003E760200587602006F760200867602009D760200B1760200C8760200E2760200F67602000000010002000300040005000600070008000900496D616765536561726368444C4C5F58505F7836342E646C6C00496D61676553656172636800496D6167655365617263685F4361707475726553637265656E00496D6167655365617263685F436C656172436163686500496D6167655365617263685F476574537973496E666F00496D6167655365617263685F47657456657273696F6E00496D6167655365617263685F496E496D61676500496D6167655365617263685F4D6F757365436C69636B00496D6167655365617263685F4D6F757365436C69636B57696E00496D6167655365617263685F684269746D617000496D616765'
	$sHexData &= '5365617263685F684269746D61704C6F6164000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	Return $sHexData
EndFunc   ;==>__ImgSearch_GetEmbeddedDLL_XP_x64

; #FUNCTION# ====================================================================================================================
; Name ..........: __ImgSearch_GetEmbeddedDLL_XP_x64()
; Description ...: Returns hex string of embedded XP-compatible x64 DLL
; Syntax ........: __ImgSearch_GetEmbeddedDLL_XP_x64()
; Return values .: Hex string of DLL binary (empty = not embedded)
; Author ........: Dao Van Trong
; Remarks .......: PLACEHOLDER - Add hex string here to enable embedded DLL
;                  To generate hex: FileRead(DLL_PATH, 16) then convert to string
; ===============================================================================================================================

Func __ImgSearch_GetEmbeddedDLL_XP_x86()
	; This function holds the hex data for ImageSearchDLL_MD_XP_x86.dll
	; File size: 42.5 KB
	; Architecture: x86
	Local $sHexData = '0x4D5A90000300000004000000FFFF0000B800000000000000400000000000000000000000000000000000000000000000000000000000000000000000080100000E1FBA0E00B409CD21B8014CCD21546869732070726F6772616D2063616E6E6F742062652072756E20696E20444F53206D6F64652E0D0D0A2400000000000000E2146ED2A6750081A6750081A6750081353B9881A7750081BDE89E81A4750081BDE8AA81AA750081BDE89C81A2750081BDE8AB81A3750081AF0D9381AF750081A67501817A750081BDE8AF81A7750081BDE89B81A7750081BDE89A81A7750081BDE89D81A775008152696368A6750081000000000000000000000000000000000000000000000000504500004C0103004F76F4680000000000000000E00002210B010A0000A000000010000000600100100A0200007001000010020000000010001000000002000005000100000000000500010000000000002002000004000000000000020040010000100000100000000010000010000000000000100000005415020084010000E41302007001000000100200E403000000000000000000000000000000000000D8160200100000000000000000000000000000000000000000000000000000000000000000000000F80B020048000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '00000000000000000000000000555058300000000000600100001000000000000000040000000000002C461D7400000000800000E0555058310000000000A0000000700100009E00000004000000000000ECA61E7400000000400000E02E7273726300000000100000001002000008000000A2000000000000EC461FF200000000400000C000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000332E303200555058210D0908'
	$sHexData &= '0A6C2EF20DB467E5B00EE001000B9A000000860100460100F1FFFFE5FF8B0851FF15000122A8C3CC558BEC51568BF18B4E048D45FC50517CFFBFF3C745FC0045E485C074038946086A101DC8FEBFFDFF178B76088B4DFC897008C748012BB88948045E8BE55DC333C04E06D8F60D9F99E88BC815BBCD2653005FB4811DCC56590600AC65066FFBF7D08B55FC628956048BC6675FBDF6FDBFEF46045061B853CCF645080174075619B85A999D20FB5E5DC20400BFAF8B405E53D8FEA36DF3FF6A00680A2026000C51520496602A13AC710A13380C6B08DFA38BF0DB86056F703BF0742370088DAA5150956B58F299898DD4BF3EB437785A50A0FFFA505DE01698C1790791A69D69783F0F9F01C4FF2546101F472E970F17C4100A50832314ACB9C4D77FFF17FCD9BD7E85C97414E721880A8B108B8C026A015F2018C3FFD0DF646A481481E1FFDAC01AFEF3FFFF81C9000228891C783FCB117239EF20F9D1ACB99208830423509DFB0B363B20BA564E33C97E08663BC20F95C05DDEE2ED6E5F6AFF6842116264A13F5081ECCCDD15BFFD0C5356571C01701833C5509AF464A3358B7E5FD1FE750833FF897DFC04F01AA0C78540FFA6012C18B87F30D2125020ED0C578D855820EF6EF7B9E001BB02AB8D8D5B895DF0457EF627B204106A238B5104345B8415F6D266D89F7900AA582BD4DEC9C7E7'
	$sHexData &= 'B5940498AA30535090045A83A0E020CC9E0C7884808850FF2BFEEB9BF8773C8BD885DBDC8B0B8B410403C38B570CDB87DFFE8B4F08528B175144D2F30F10450C088BCB121104F082CEBE245D24B2F5E85A78BC56BFDDFFCE6667C4F00322A1EC8D480183C839777FC757FC6C837E140872FE0EEB028BCEC01066837C41FE2E140469DB8D4C08750B4AFFB16C0996FC45A08945ECD3B8FC053C38CD8E78DC23C634131CB0E1E93BAE11482E4DEC890D9C6255EFEF595F5E5B5F5F08CC8E0F509C5153018D71604690F7FB0DCC75F0C74431A0F74EB8F0003641F6DE1DF6FFF1A0EF67FFB9C0EBE9DE83E4F883EC4CC9C489866273FB442448568D080850C70C40114855FFF68FD6588B4C24148B2A100FACC81EC1E91E7A8BEEFFFFFED10FA4C10203C002F013D185D27707721C83FE647217BE6470CD3DF89D90624833CC22FD26D9363276DB97DA7E122D1C4C5E1EB8326829B0E153347F538B5D08088B3DFE5B147C0CA033F692B885D3E06BC0645053FFD70B6EBF0F30143D8022740D46BE037CDFAE32C0AFEF67B35B7E0CB001807804328B003F83B1ED2F552054B5BFF2FB7B7FC377AC021083F802744C6818B04F408B0D3177836B95DC752CA2F655F05237A055B8473C76FFF2F404F839B98A10530C0557155B7118A6812CBFEC14835C86AB4EB39472EC54530FA2D0BBE2ADFD75EC66'
	$sHexData &= '0FEFC006D6AA8906895EA24E08B8BD0E857BFB560C394E7D175EC6B10C403DB331B44D74CD397A7876C77B379A858E958558808B3A4243D77EC1E81A24015E70A2895BB72C871CB83F10E2BC74553D75E9F133DBAEEC349CCD811CD23C4C53543ABFAF242A3C1BEC0241BADCE53334193A3C07B99091543C545D90FD8A5D6F04948D460A2F4A097746FF24855CB97BBB1208C4B8402374EB3D0DB0EB36F0EB2F9F2B57FE2470EB28B0EB21E8EB1A2528B772E5CAEB1368EB0C98EB05C468081D28BAAEE074510FE420A10F7A7DD817F0505CCA8C97C827343360B161210C116E4A7C7090AEC83C8B95B94204765DE93A56D4EC894DE850053C753792435423885DFC9C6B25C34F9C4D8BFFE73DEF1BF47007FE0E0CF706F084E7799EE9E2DBD4CD9B56556D5F51102E7E564114027E17888FFFFF560FAFC28B71042B31C1FE023BF05E7506D91C9BB0437C25BF90E3A3393A780C840E468A10D70DE8BAA20433D2A214A4461052300D38DB3596076A0C10002C6641B0E0C18911889FAC585DC22026B323745FA21002056F0BFE98538E9DF0FDDC391D2673487430DFDBFE6DFCC747A5895F10539807682C38AE8BCF5BB18866ADC7FB4A8DF47CDF2FC08A6404013D34483D0216777AF1E9EF140A338D5002EB068D9B378BCFCDFE5F38C002663BCB75F52BC2D1F85041BEC39EB1AE82DDEF0C83'
	$sHexData &= '3D144C08A121730282534914A4A060A448D2292146C183823A17275E29EE711008898653B9F48BF7A9EA56F7C952086403D235394CFF0C085A24A60A2A1CF04C3F4591F75F650A897DD057397810751EE2462ADAF79E7E06E1DDE9080C62C93D871DDC264DD4506A8D0CD4300B91FDCFB0507DE4A34DE43BCF76328D6424A8308FDFEC7F8B5DD48BC383F9087305D412D80FB7147852FBEC8D78DBFC82047B56472A3BF972D4FAF619DCCFE46E408B7D67BF8BD383FF7276DFFBFF038D55D466833C422F751C8BCB21E8BA5C3E8406EDDFDD14415E8A64403BC172CE8ABB699FA0006794445C14793BF1745A28B1C109FD5616521574CD8E5946418386AA248CB54B1CDC50090252505063DE55B0393281B2BA585C1014EBDBDDCECCBCAC27890E251379720D30C5A2C2D851B5019832356049F03F55418AC20CFC45CCFDED9F36E645A87C884DB033F6344DB48BFA8975ACD29501F7B40A8CEE350260641820808A9110D7A20306B848CE95182B0C68CE9A64525E53B0AC8B5D2EBC33037F10B60D7203DC9000678B1833BFE675B5BF0A589E8D55B4DF011E14E250320532CCEC805F080AFE3977107449575ED055145151E09D0105041021F8525A06D2E0B215F56D85BEE425DE0EB945D050567AD86A01518BCBAE44C51EB09151950B31E025F8538BF80F1BDF26830C0B21380CED7965270E'
	$sHexData &= '3BDE710403F347560F17F2898F5232D10E148B8C6105DA75A8DE004B792C60B810C417B651E06034085499C407450ADC524B1D014D8B4169B8C05306C8876310C7B6C7BF0FAD740DC821E0EC05D116506108AA34C8A631FECABC1ED3762183796F39B0FFFFAAF97A3C478BDEC1E3052BDE408D343B3BC272E16264A870E9DA7C1C6D2C1E1735278D345139BFF937222B1DE21C7DCC2E5969AE708B951C664234051CAC1C8E0934DB9D709D74972208A05CD004095FD7638B6CD5FC5615C009A01E26085C72D48551522C1E510CC62E3654FD3C0F304191E4D2F06A04FCE4D0E1604853DCC81C9FDF0A3F7CE2851945CC2F45C4C7841572C865901C34063475AD785891E18372285632ECBF146B7B3D851CFC8965F085F183FFAA8000E6320C90894DAC0FE8B47F05400163774485C0180818D4D5B1F032C00CA032A4106EDCE0537CEA05ACE21E09A30F166F8450DF8B208BCE637A8B952BAA0E6A022220022AC0FF07FA5C83F8FF742F5033C98D754CD7A2778706BCBAB4BF0872399E7799F0091E6A0021423E1468ABF79DECEB053DD842E4CEDF7776ADE898F8FE3C90DCB203831B16ECA1BD61742A84A85164F8FE816255272CCF5260055956A16AC8334B2DB70FAEEF11ECB472189FEB09B8E811C0C35CE342828BEAF65395E40D4A55039F0BDCC8A71A149D6C8D4EA4BFF10C7A86BC978B'
	$sHexData &= '56A02820EC2A35E84F08894C30A0AB901715164FBF142050FC8A6230F95D08837BF4B12CD0A976B21476E4FF6F5C6DC3B59CE43B1CE074228B781C83C00037C5EA0C8378F048A88B4310BD0A1D9FD246D81C42788CC8EB0BA1FEDD65CD5162CC185A303B356D168B4628B7029FCADEA452827C42CC511258F063DA842CD12C5FCBF5BCB0B8F8650ABEE10639754481FB56E157601CE96017818A1D7C130A0BE490C71F0F184031980E0E18331C091C720132B8B8BC5D83BBCAFE383B3D1BB6EEB7477DBA06FD28F30F7E40246C06112C4608DB5C8CF1CF9CE8CA7701394C8EF8AF345EAF772833D25283C60856A142E08C86C042D407FAD0E7887BC98955C0B194461C7E43A5BB14DC1324E4998B7F2857D7F15D1AF4BCA1BB309C0494C1EFFEEF51525628B2BCB95C74D1052BF1F083F901730BF63CCBFDE02B7423DCFF052B0E50007FB075EE027BD7835C893015431AA2F4D4BBC0A334B9D67EB001EB0D1D6E6719E445481F0EDB900B84620AF149E55C710F8A51856C205725B13328907655EB11543C0F4C5D856D0B6957F970C8E906BE40240813CE9107452C573C02DB53D1025F28625B8D78A98C51B88A9D555D885895215F16631D32949C7657E2850A83BD015D9751383C82638B8D296F5BE08F626B8C080F828C98A9025463C5797618F2AF2118B8856F888281A4261F23057CB8B4'
	$sHexData &= 'D6A42C55901F1545C0EA0033AC60B9522531506084354A8D5A205D83CB83E03BCF5B013D3D01DEBF4D035B48018327640F86C6A6EB01DE162B44A424E1908B6077FFBA2154859A0886C6D00E469A0945D0E4E03D9B914EBC92CA6802B39CD0BA2D04D2CF8B952B513AB5740047DEC3A53C7004183A757CEF1F688B0E89088B161089CE7E1C6409297EA3AB1608CCD724C9C7620E3EA1FECF5E185666D62BFFABE09D4A170E010931C54576878B8744D4CB270DD6B4DC5DB85F4DA45153C7815515BBBF1320350093F8B51035585A95CC09F8AEEFC082AFB83BF874548B107708EC3BF3744258B0EB1FEE385E1974F6063BF07405336E931DEE1FAC24C3740AB9BCECA54E891ED05E0823A5FCF8B607FC3FA675AE70923C03A05F1EDE42DA52B9109CC8ABA08940BC5F6DB3040F001D7DCC891D1AA402600A12F1595049425E008C3F13C811FC02BC7508EB807E1900A5A25A392C16CF200676800A98A5561A9644D08C53660D083281D1B00AB8FE31FF07C970E00E989F30CC49E58F039BB92C8945C4A004152E43C4A008705D33C1BEA5A0E2B6B2F4543299AC66387E57D5695AD520CC6B257DE06AFFC556CBD03B7628633B307457568D749D57DF7DC82E91CC65F8C85240237E2B3228B64CB926075C3280F425FCFCF89CA6A08D92C630CF2C2160C9912A1ECCFCD02D915C7E2BCC3301227B'
	$sHexData &= 'A02A1C18D8EE1B979D70D525181B46144EE99043C504C33F15E03487500C31790C45C822D110154B0CF5A2C02DC40CF36F8E566CC4F7B1444208E95C193C1320C1EE3AF756522B937CEE695453D1A85684D05373E1C845D51C3A46E484D475FCB95C1484DFD05158D0BA9A21E47215FA480103664042EB30DF1BFE99A6FCC145EB038D49285804080478FF8D4B0C518D55C0525CB72AC4D07BB42AD25020756FAA6C8B7332C0D3E52282CC33FF5E4090E785163B56F8DEBA9EBE7FDB771B33241FF0D1F13F20B6B9FCF24E0C51D2C746201C560C2A0EC10D7879FBFC818749CA4DD195A301E88F9168D253628F2554E390C35F0BA81463FD0C806248AEDC488935F85E5C8E0B4C503A6A241578017EACD639EC78C6744BA3EAE365C0C87474090E080A6884FF5B74B001884120B574AA711FFC7E58884221795CB81ABD052A55100CF0AEE06E9A74DF270714643568207EE0115DF006AE9E010728FF6A020E790A481CB953A23F5EEBD71E9FF31498038BF48D83FF46250006BA8107C10568F2E81E6045739E7097CA75E8097405A85869975610EA14208B2912982E144A48AED83CFDC5587F5C075E14395E1088431EA81C37EAC93BC15089894BF038680AD304F4A278F01FE9C6C74D0F08DAD81881DCBEFDADD8508BD6890F894F0404083BFA74103CDC8B552AF4DE351607235708EB2DAC29'
	$sHexData &= '6DCDBB6B6D1D93CA538BF7524747601101B3089FC7BD81636051C13F38070679C56818E7C4503CB0AB1854207DC82A813014FC869B897F8E3B604C8748C1F9121073347359FB791708D07C240C0610148D4C171603E6E128AC96EE3BC7C7C15D13DDA1859E8CAE0CEDD3203A0889680EDA99A6692E03F8061608086A814188081C71AA20AE0B4723D5BD5F9820111346567BAB092B18843575416258445DA36866A0E18A08182BCB1602A650D8FDF1F181F9E8C47205BE0D3BCE0923B0AABAC1D0F7F636762F12EC9414D81A8D3C85998D7306FEBF011F803EFF720E03D003F73BD172F30EE11970D5D3DF42861B2F16A67FB8E0913330F5D28BF9903C034AC6FE5D005403AAD402391BA901D3146C05B4251AC35DD7B5980A7453D4C0B9781DEBE1A400B00296BEB48B1AE37B5590D05C1ECF1C4663C891D684A2B407A828128AE0521CC8F3986B27C675C4DE7F19CCAB38FFDFD4619C164B0650148953148A4818508BC3884B96E213BE1840805CEEC6437E580D43222954C2231B2E02A65103C9F62C837F51B8A011A13F22A850570F4B299C0B4DA8AD10C89CC25E0AB275A85A3430479DD1050AB588DA698C076F7C75146A02C538FB39ECCAC8F5EB2A061F1C473B5037C220C7FC106A010EA50AA070D269390BD44683E4F4A4185152ED69100BA5027D2E4562A118F8AC5051897DE0AC89'
	$sHexData &= '45B1DA8785FF7E83D0AC7A8E47336BB1BD81FF007D420F8F8B3D1561985268A05BA0A1F4643F0B074103A0A944052F855F88750A884DB33DE8750492B32828B0D601891722804AACD47463E63A1788E450B98283071F11847601BC081BF561AC02249CD09728FA3F681519308943140FAFC7897B1032BDFD05CC508FE4126C8BF88BF3DAF0166FB8A11C8DB540E3A7C1AB06642A4B082B0B36905C0307F03BC87336B67F95FBCBDF820B85C4FBF7168B02CA003B90D050BEAF7AA8C131CE9E3C641F58B8CFAF834B6094F20F102CA80EDB023585AC2C905689555CFF8D83F6A0006C00CB981C8C62EFB45F021F39759C8A06A48BEF868B1CDAC6C1E9106EB28BD012EA4FDBEBB718180884D20F84B1B7FAFF10A80F6FFB6DFBB6DA0F57C0C02AC3125DB2660F28DA165ED8256DAD79DA5AC332E4E3C918D81259DC21CDC8B60658D9302CDBE1C0E4694F33CBDBD82AC0D87EC701F0C3895279C0DBEB0B81FBFFC9F354FFEA7E0380CBFFB224C9F97F8E3FCDC9C0C0EB1581A227040CFFBE37C1EDEB088A7AEB038A38DAD2C1E208E60BD07677CBFE0FC90BD10824948B0B1E28E283EA8A898C4683C730068CFC26834B501C5A3A1ABB1F6EF890037D8C0164FF4DA0580F85D9322E47569A0A5C4F8550513BC67CA1FA3F4708807DB3007412903020FC847F39C0256C1578C06A1CD3FE75E18843'
	$sHexData &= '18A91689308970040408350AF64D101466C74018382C53FC4DD374285643464B18538BC6884EBCBAFE8A9C56C64628523306C03CFA221B6C310718C0AE244736CF51901CE49DFF16498401502C27700CE0DE47FD519F5E084DC4895DEB4D2C571485D2A38B06A71442F4DED02B0F7695CE5CC33BCA202F2562512455FBAE3A95C4193B0E1481FA17F1FF863302AE83F964760FB81F85EB51F7E14F558BD5AE05FEEB07F2C0EDA921E8C075CC6A33A888B4E28222BED6FAC1E8023C06031570FF4F0420008D9402B979379E8B5804002AFE6FC0030CC833F2394DCC72D41C70810545E78F781CCAC40EBC517821BA3F5C264C520B50894714F87E4F257E52313C000260495C5D31D00BA9A060808817B013D82360525F733A81D55631E5144186DA210C5EA17E010B6FCBC6DA57C8EFC0014585C0745BE86805C220D4EBA6B40F726E9145B68D20AD2FB24B1C28C016049C36088857AE7C1006F81E57106A9AF90A0C941A013CD4478D453C02CE87A9B0C04F785102838985024EB840D8FA98523889B56C71A686A39370E0A842F49823DA82A568CC16D409FB90E4E8C053F0CC8E3F0A6B75C8397714A43A01A5144E01AE34FFAEC7931E4655B462F6F9053CA5007F882E481703C676823411554D18D3F0D3C6D034BF00039082088845BB84DBA6A07CE5A9FB9D57C00FB6DB09F03A0A7BDB581D'
	$sHexData &= '015E5780E049C390B3C0004732E2D37200CFF32687D92AC3C1C898CC03BC2F093D2B0244414E9E039CD2FACA04039C9310086C607F0F7003AED3C1E36FD80FD20BDA0FB76B6808C166CC1A24891CB11046540628C03BF068FE356402B801910C90648BD314EDCC4189123BBE30DB2E3DB0A400602CBF45C585EDA19C25BC8D5AA456D92FDC4EE2563C855C528EA9B824561E29F464B4DB0376CC10B460C8525038C89F8019B8B058DA7531F28DF0A46A0756E4C3DCA8985CC8D778561FECC04EA21524036498EE6A24E3973DB3523BBC451542CC8B358A8C7EDCA550FFD6BFD6F9F1012BFAB1862BA8FF074AC1B7082931B9C4F844C31DD7C00D2E846A011B8288FA899A828BF65152901F7580E482AF239721E450034D3777E6E751559852B5298B4575C4895D5875C74310F0C68973404F18061CD785B63248FCC00D05FF5F2E231C0453082B13C1FA023B55C07348B0C0A82FCE49ED2C9B9B782AC8DD56F9F751C32E5CC0011AC48C5E545A618E6BFB4C5457BC5C08539885A500B20B0703D81A000C8583974EEDB0288C8E5F0C17EE9D763975BC92091FC60D845C801CD3D325172143C4C4855C20CFC4D3ACBF0B86CBBC8B4209C0FF4DB40CC5D30F600F85D4A15D513355E8B0A6F0064F6ABDC346851CD3281638897804112A98A6040810D3902B2ABCA4DBD4227260941C9623553A1D35'
	$sHexData &= '550D2F8D5A697A7DE4104D28AC3F404D0C328078993C60563CF185F6B0EA0249858F560ACE7C6A1831200800DEF059D684CD5D803A84895DD8B4AF432146326EB92685FF0EB18C74972BE2FBA517FF991A083EF056D07CF83A26C021F1B6A0FC769C6B588C95694E08E03BC8BE5B1AD44908D58D6C108AA369D6286E9425D0C4C8789003572B0CA3B431CE43B5D220EAB6588BC7148DCEC0FF50DC3D00E1F505763A499451638BCCE5C471A34B8450841F0A54265D55BEAC7FA08495B346AE800175277BD161082CCCFC9AF7A97218A90B8C0A591DCDA8584D9C24A4FBD4B79A463AED9DE0E6E85432A30803547CDBEB08E4A4C924DCB2EF0A392264EFE425950119E4E4EF3F7D726A26E3D8DF8B015F9A2E4DE207F3E0FAE08B4D3A0ECD9BD4D0014DE8DC5420D86AA111F02F14CF949DA1480C49EADF88E23168C54718B763B00156006CA9B7496C0A519F89D59109FF180C0889219C813B21D85E0057148A042B0F6FFF109E6EE80208B8676600F7E9C1FACC5213FF038BC2C1E81F03C24ADE11CF7226B1DE8C3A446F58DF195FD342B9258DB610017EF3853AC976FCCCA112D03BC8751F0F4E28BA3CD4F046F9A070BC2A7B394F5C7E422BC162F00549260DEAC67F298D14B6EFE0E07E464491EC8D06220763D7B05D217C1A4708D5C3FDAE0CC085D39F42A8E95599145C57C22124EC94A8'
	$sHexData &= '6816AE325C157CEC6C15ED262A20F4502BF392FCEFBF11B0FCEB1D1D746A4CFFD76A4D8BD80A4ED860F60E680C4F800A681083D664726DE8740442A20C33644ADDAD00ECF9841483F9264A1B9A8A8D0A00031E55B2CA3973FC9BE25D087DAC0803DE8D53FF395508CA2C9A1BBF0BF60C423BF27D5E12F2224D6508B4A81A7EEFE6EE24EA423BFA35A8FA3BFB2A5D9EE1DC6F10FB8D56013BCA29A6882845A2EBEAD810C82B8C2B5EDFFCBA139D00DB7E52F07E4E557F46A40F5CA0BAE13EBF90EC547430573F36543D2136F002741A0050531FB0E36623049AFC751927100B9700B7576B84A48333B403B53018143C6884CC16F7D169E0578B888DD66D3C664114348AD85B6F5B105A9D4875137C14522D5665B4B10CB98614F138896B26FF1008BAC0F00AA885D278296644F67822C03D8D17AA7B10331003D7D6BFE1EF243BD17F0D8B531430EC03D63B50147E093B83FEB7169B807D1800750959EB4C80EEAE744D1C7F1FFF3C69F60FB881801C910EDB00F7EE76CE07F0636CC7EE2F2BC279751167E50FA3E3607F03EC837A00E7EA85598858E600A8338D03141D9F9C23AF1203241C30011D2A1E38883AF701A5AD40F4905C10A690BFBED50D5AF12EE0438B140E74F2183BC3D0E5DAAE7C63849EF88050D461064C0E14CA10087D981A1B682BF7AC04C4E85E706BBF4E102BFBC1EA101A'
	$sHexData &= 'D804C22BC3A47902F7DE1D05CFCFAA0ADF85C0D8141C960F53698ED78FF5CC8C0EEDC2C517F657E5EDE48BEA664083C1E65E0709F0103BC74C0A5A0A1608CC6AEA506CDC428ABD08031354036A2470009C665828B64E3CB1182160C13F386F50D8493614FA3C2214007C23354D5361B43A485572F962C5F69B7A74FE7F0E8B4014D6E003CA3B42668062B13C0832EC3B7D90271990FFFFEF03D785C28A8BE008BF459E6F7CDA56842D5A900E6EC0066170F0179DF16546C9BB3612C14CD1D5533F0CAAB57175D33C4CD37555C30EF8C1DB583534B2101CDC7DE8756DF90362C94E59C2602B5581A974B33C7703F4EB05BE0CA54F70BFADF3E61402F0E17423EC1614DAA0EDF36E0672D318F8E066E306D7D4EF7DF7C0565A74685A26DBC54AF80ED5B3D9CE3660F96360D906F9DF68C1D9BCF7DED116D02EC306F916FBEEC7F6BEF36E561EFA5665C606DE4E8B85729F63DFE0CF7C858BF041A1B0082483C652108CF04FA7E23BF1984FBA3BF90F8D9F060C2B60A1C034BAA8F42CA5A1B89C7F73D3FC4C3730CAC1E94D608B06E80AA251C82C76AE084BA313CE2136102BCF813C2795F8C7C9D994239BE7FEFDCAF6FDEEFD060FA6A9D2F08BAE474C7034180C46103637B84951714CFF5C0326BB011C83812156A478A607213A8B2A63FE28D920A726611F136009469CC46099069D28DA14705D'
	$sHexData &= '75286698D2455AEB965940B89C1AB804A8C5E88669ACB6FCC23BD3B4C78257C88FB46C1614146FFF084FA838704074096A04B8102670EB0700E750CA6A062CB1ECE6FEB4D42B4B06B80F88DCDD8433F62B94E16E68D41ABC7F06DF3D87FA2AE8F0B618549A565357740722FC73344D3CBA0C349C83C418589A30C3A17A68C4A9040EFDAC2403CE891E446A0BCEC4ADC2D214C08B4B189A9C52894C8258E0F3033C09ABEA45CCFD6218A11294E005B8D69CB82C8D7DA81F980200589390091743810828EA1020524EB06E0B5D6C836ABCB846928E383A220E06B4B04012C146A285CC6451A086A239A18655640822040883C16CB8C674628B828B55C6864D4C14E35004C63B224CC09EE1102A061650F2B3C124150A7CD08B5515379D40069489C6EB6528A836006EF63BD8741B36FA7DAC5017E81CDBAB120E2853813F39C2743ADC1E80184CF64C528BC60D0C2283562A42D844700417310817A9125655E5DE28472E075214282414C30679510A1F13282833B49041F0F1A58F0493BE7EC1741E30F30D5A8C52514EE78C8B7E0460E92004B661BECC465E619A8817B27310766CE47C8886F79138538BCE3EA078F826FE4D582A7503403BFE763D8BD72BD619138B25E6768BD3A3A884C46E6FDC80156A52499C9905540B2E219E2E07AACBF802BE013BF0729B451CE415403F1956A405B93620'
	$sHexData &= '91AC854002BE88B3B20A9D4CBD50F5D37EA82715802A2040463CD7A14FFC83F902742B748448153B03175A00B8750FA66C085A4E1AC98084083A95F699F86855B00FF966E08B86578D9530FB30FBC04BBB23BD34120A383CFB96DD03CD878D8DE0407809392764F63790FB83771DD098FB01805DC3451E1D50080780FB391DC6FB678415AB9F98FBBBD9A56038D4D8B50479432C7026BAF63BC77D082614B4C4560FBF9E634C1FA4215E4B3803DF9A428BC84C5A7EDEDFABAE2F3A23C4760A120D78EB083483BEDF2CCA76030F28CA22433CBA54E1DEBBFBF5E3908D0092541210D9DDBECBD72CC21D9508128510D940749FB4E7A005C0CAC17C770DB7053A85DADBC2EBC0C1B0C846EBB8B64D29854EB92C2C57C960E06069DA2AC83AC95E5E66976E83DD60BD94FCFD20FD0B2428D3F05C2E303466023872E92BB46E33CD3EED6FDFA5BD34688B035408040B0CB95C2A2FB8FCBC98C03FDC8873A044851E04CC73183B54BB61FEF77E2D8DBD70FB7350FCACECFE1D8BB5780B9574857C2BF72BB9991B17C62C8B1D85D3F8FFD3CEBE71AE85E3128AD3621B409B81A38BF98A03F7FA6EDCBEC6CF7C098D7EEACF7F02A25B0C3BDA7D67CDA6AD14DAEB2648D97ED9665B6C6FEFED4A10447E063BCE4CF118143BCB7E54C87F9DA92E760A8D1C13BCD2138B72C1CFE6C62BC7C0FCC12BC3DF9D97'
	$sHexData &= '068EE5B5E0D385FE735633F30F8D2D01CE0E25FF8367A144F842185114D232ECCBC2D48D10742B7E0350FD6CC0B6CE2EBC88855777556643F70C80BD17006EC7009D011764F5F82BBD89AA34CEE70EB0B95776070CBDD9B82CE490FB0A978DB50227C0C24F63EDE9704EBD7E837A44007E0527BB672E42E010B8FC3328E0EA5ED321BDB98BF78D4B01672A432D0309277D8351A596FB04F00A6654AA7C53310472C741B96311605C706D9D83137C5BB927142E4939BD2B85B90517E7E795BC52B9FCD1C51B8F8072852207CEF0FB745D0683BDD062087210D675F5148C09E7E10F8EBE4E6327CEF1268CE64818F6B2DBBD065D8BBF2D90520BA456A1B2B2B2B853C857D451F44608F9B0BB535126820D9DC534219DE281252F0FF3E204250EA417F7BD72F9EABC7742A945083885392877BFF85C74198B613B8D241B0B8DB5C4C2FB29258531638056D70D8BE4639307DC3DDBE0359548FDE40DE902A601AC0915175872227798FBF5E81F131AE1C33D83F8019FBF5EABDC60EE431C62841B01C88DBB6984FE7102668B1183C1EAD7CED1F932E93B064E011DB676FC35F85EB69323C40E2A215C704DA79DC0BAB2270F94B21579258769B984A7DCFD36C90452FE0FDCA31C724D07837BAD955252710A84989BBC309499AD858A410BB4878B382880FB773C7CA416E96B49D88426F02BB5034C77'
	$sHexData &= '1A7A6E98630B810C32219884ABE402C1398A8956E800564527FD72E5F25123453B4D0C388D6402E47295958D8C5C9A0B51858515E37F38970D72431B8B4B243BCF073CA4A2D4E7F97DB2311C2095A0FA108903E47932B495A052C3198001C1B585C73AA140D325CB0F6E2226934685CA100354EEF3C602DA814D43D3A06080A89E85F530FB4424A76811F530FE35B71BFE73083BF77513B00C750E081075091482DEB17408B64D08EC26994E3846A3E330768F1BEAB40CA244BD138D4124175C25761D3BF09AB5130917C2A938520C00340233D685D685DB485E46D322DA3FD385AFE242C0BD71C754C11794C25BC824143BC37E62946957B852C6FA2CE0D80E246922ECD0869558780A5CE7FAF37360C68571012B8D4465128BB85D38E2F18A8DBE9A856C3485DD801B5A4E00FBB90173F3A268888D705B4837E00CDC9F6213B35FC0BB4E8BE7958195901790756F5257FD092EB02BC0B5357E45D78AFDD0A1DD358D041E568543031D8BD6C46F7FAD2281406887D103C902518D14975052F057849FE82246A83B897CBB8D99571970ABC62F820CD79D4AC36B08FC9989B11C0E778A725350898DAA00F072342AE546066A07892B2DAEF624E8B7B44AB86B7733947D2424B434D182AF33E31A8B1F34C12B83A8140FC234D0CE73D51523066827147344C920137C68249D00794A3F8B0B9154C6'
	$sHexData &= 'FF4A75058B4304EB086C305F6714604320D874B7780B7C33ACD01C3D15B87538B38DBD37DB841037158D6148160D4FEFFC289ACC8DB567D7F4EBBD9AD08047BE57D9744184DB61B298F8FDA28F170C4BB6D3C43B94F8FD2490AE950D338B8D3BF1CAA31186EBDA74C050D75151499E5E1395CB5218AA18AFB5C0B1CA6FF0E03A388D39D19A8CCE7769F8CA40FC94484C0B504F9D1CDADE3002442F9D535E1ABB57B0BDD47D45940B908185146A802AA14EA8835EE81AF25970E95CA981ED7640D2F214425AD1E39F424FFB80660F54D12CDBDA9DB00F2FD3064E3BDF762D8A9D1072DB56E8DEA16F5AC060C114FED0C65F31BB2185017707110038B8932449927A27D0F4F7E903D12A04C9BD18B8061BBC0FDC6F840FC8A0DA3D4B63D446E0260AE4167AB95CE8F0F4FCDF7F5899CEF8FCF2179A1B83385499ED37198B48283BCB900C08ACE0758A3DA085EB129A1B5117D88803BDDD04E12ABBCF9D07B3708ED31B85F4C3F264D569C5A0E4FCE0FCFDE0F90E85AB6BDC8A118885FC54916DF00A398D9B731ECB7AA4E9161F5C0FCC3E1C838D8D02EB1E42902263534888701999A9D21D3FD09E978A0D8275B86ACC5DC8B8F7CBE7F94E9F3C1EF685741283A511C8CCB159FBCC93E635024B835B30FD033578C4705175BEC80BCC4C38313366DFDA042C034AB0DFACF24A66B34D9C2239F4349F'
	$sHexData &= '015ED27998A77C7814A302A8D1E0518A8DE18D8092E0D00795148D4B0C2C5865D341678185236D2E14FE0F14E3C8227DAC8E47F2CEA59D8AC0EEA80AACB0CE0CD29BF54D9C7037B4FC72CF932F5F137C0D01BD63F0915EA974A8CF2B8D04E98508C5F6086C9D88EBE0C90EE3007B0FB7FBA7F003D03B9530378FCF2783C1BBB482960B2834BA51B7A11D9FAD520CFCFC51507B6DB002C84F150187017D1006E5AC8531F452535038D250015ACFE058091C34000E0736BDE5D02420107698033B2C780261707EEFFF85D8D686D37DB8D58520E6300951212881F496BAFAE41723B4C2013015C080C100C11196543C5BA53218048A5FFCA980515DAE4E4B047747505152AF4408E6144A0198508338DC7C778683245625BC2B3BB5540A7473901BFA5B6D744C8B3E8B5E044D4D2E1F085C42899D44489DC340B537287B8ECE010687C04B6F5357DADE1A2546E5106C8DE27E54B82734522EB0601C5C3E726964E26C8B8D0D4DCC71006E7CAB217C781BDE7F542A937CB09FD4D02DE9F0409B101F180E16FC67550B83F8037C11D29C520C1299E01170324FCF551085B6E02D2D03EDE7B8AC34968485888C1385006239EB59072B9A5EB39A58CE04D286D1782627B4D7B18AA48F829C9FC0ACB102EC239C95E472C9C91C67D0402D6225BD03A91FBC3B43445B9B9CF4C3148D94FC2E70CF255AC0B0'
	$sHexData &= '2FD0112CC8DF332B83EF1407DEDBA120B52BC28BF0164A34524833BE4B293C160AB286C5D615030BDE13EB0781DA952F85F7371504BA0F5BD2B8D237D33D27D734D1C245E13B70F274AFDE9C5B4BC0D0CAD55DE01EDC37C2082F0F858673D94D6123178D3417DEA552E981282D2C3038D089944B3C40A44D52CDABA9AA4EE007278426AC086A9C952AB0D542137D1D3C3D9D00933D09C901501D7C80A68DF64E17E492441692B251C5927C7AC44BA060FBAE393B9893288A3734640AB4ECAA02D024C848FD7A55204AEDE7B3743A092E0629EC608B055F0885D28851FA26153660B5A65F478E826FCEEB2165630BD67345675745E3C41B993BD3740A8FA150E328C99DF304069507373F3BDF748448FD4C5F322CA4E057535164488405C5D2D1B0381FBD5D4129816DC1186A0459F9FFEB4611769DFF33DBA2CC082B4C982C90D645B8092FDF3469F8B82A388DF9FCE0FC43445E3EE4FCE0FCAE4B15E404581DE61895C12DC0EB10EB3911011D80D316FD5B1C882F8770D4083B8281F792CA832D2555AD978D51231F1824936E55B852C0256136266E8FB99CDC20772938858D8D6BDCD00A3BE9618675D5FD2BF5FE6BE17F5FB8ABAAAA2A69FAC1EF1F03FAB8427687073082FAF8760240A7D15304BA7207DF0EA3CCFABDD22BC7EB7898130A3B84BBD03BA8EF9F90C60C83BD35762268FA000C'
	$sHexData &= 'D95B8D2734BA52F84F4B34018B4EF48B5EF8BA46FC998898FD9BBA03C88B0611D87D8190BA3456FC50DB0B5122448791A3DD2AAE28987140463069C7604EC6C1E8375EB5F4F8C58DB3C87A75AE2738003D1E52003C25F0092C895A3231253029800D89FD1608FF5BECAD3790F4D9A5B8FA3E514EF99CFC2B9B70A99C5BEF12DA2D9F70FA1A0EA19E3CB8FAEB140F27443E3D8043851D783C0370BFE11057BB487C75050D7C7004D69A39ACD8FE4C151002436A02DB1B3CE0370054EA2EC264FF8754466E4352F8002FC06019E48D85485006DFCF8C47058DC7308B51185224275087C60B30783D4BF1021C20D4CB2E4C077C4698CDB88FBC1D87C03A07F52B7EB9733D276451940F8451569C5157513B0A3CD26AD827A05101D4278D7CB777A424DC17B82E34FDD487E0A3F19FF42E28102F30FDBA050C2328289A2D287502523928A74F51AC8EAB4AFF0C3B040CC5042F30145B4BD195EF23B62A2A20CF213F264DA3845CC3504ED45501E5C9BB48A46B81BBCD9D4C9A0A21C50BFC668948A8399A8CA353707955BC5C9AD391607874856425E4EABCECFE58D825078542F3DB2EE00A9A5E945E1D49808043E080D7C2D3590E1E0776AC4977803BF72175099D748BF151E5DAFB92B971649D503C44780A989575A9C298BF8439A4D389147510E856012E16921C54D8BD123667326F4B33093C02'
	$sHexData &= '364B196D0A5BC15219F9A95531389585856B54FE42480439BD2018DB70CFEA8195C0B9B0122C40234F9251236026BF324FE212AC84CF8D067997B5BF12F7EC1FA19F425D08817B101A0F82A57661841FA454B9F72F95D4D048547B056BEAA1F657929ED209DD397D0BB89FF03370550CD9D3003C08FF28A412267F3AB21EC01AFF50F16844E5F27B86AF294164071308FFAAB6E968B8746F02B00CFA8FBB1885E8D79558C6D3C31A03D2AB6800943D0D035DAA4361350F6CB4083D20196C432E0323BF14865C614568552C8CA07F5F67EB007A450854FFB33E720B82FC9E2D5E58E568A8750EDFB40198A8195165574282B5EEA6C8C776345B07C44EDF21FCFFAB66903FFE7DD48BC7F8700CC2E1355BF677624650901609C514D808F8040783A1A04EF794900917375BEFB92A6C0F7EBB153C20FF1AF819750B83C2020E4E757E82EEFFEFEB081BC983E1FE41750D8115780C0F95C11ADF62973E840CC61A7304F64C6C1390B3AD781BA20139429A3A391060B5C74B904B84BFC968A92531FA3532A0ED81DC7D81E43994BF05747331C82713F2A0BF045AACBF0978209F4C41C0BF067514A30DD4DBD0F00020EF9A8178680140D6442645DE96AB6C020200C74A586565D32655010802101277D52D285F4F014301848AA57F248A0138946414F89B28F847240FDB228C6978B19AEF92017FDE01'
	$sHexData &= '4FA007C6925D0C93708BCBD29771FB8BD70FAFCB04D703CA39A6E0DD2A6451A42FC87615381424FE60A610FC9703C099F7FE7F1AA12776A27DFEF483FE017C4CB8A88E38DCF89E63697F4D886CFC4403450C5012F81855564EFD02F463C1661F252801527CF8FF9475D1AC1F8402239F0857336EA41270BEF8488EB646AF52BEDC5EA8087E42539E0CC603E88AC36C1461B191A9F003AE6A0A9319664ADE7023528A4AB66DFF483BF87D046A3232473B7E7CCEE4329EA63D3BDF54EE8B3E785070F868F15980F433DB518227680A04F4F0B004F405110FE11F885DFF3BFB0F85C02AEB323E3B7418F81340C11A9F46DA69C0C8F4FFD406BA194DCC699C6AD52CACD6F1D45B9EE470E606B3AA80DC018019276890E65C720CF9B60D020D60AA4C22D2740204A9B812361608C42A10CCF0FF016A0595381A16C2E0DBA51665F8668EC3146A38C3E312B892B05350401C5EEE89F74C8824FA2850A85D06AED31AC2C0FEC8A011CCEF1D7EC7755972A3504DCC51693019CF2D478D569C7C084D482CDE3DAA10D6C68A1828F07C98EB599EE0865862A392520ED60220C18EA9605CDF2E1618BF42AA62642B75F8EB62F45361C5E2E3001566A28D4E34180FBB8CF97D17AD2BF74E5E566A40703EC8612CAC7C30C05B8383970AED75EC0AC76032C6E190E3C70D63380C6D7D14E3D7A15B0D43C6D59F9C'
	$sHexData &= '234953D4DD0EF4ECE8E704300A1BBF04089C5CC62556CEF6283A016A7409FCF2D30E0F2314FDF02A041662AACA8D5131883A8D95223C04E021ED9896580BF85FC6185FC6460401B2E3D95502CFC2A61E5F0C606D4E115E2D2EA02557DF08108762004E021F2EF07262001F85C974715872805C4BF8F8F7034127441B564775338885F428DCAFC6697CF0F40063D05789B519410CE0FD677C8A3D516A4D30BC1B84110E21AE669F2B1F300825F1344210388EE102C3FE4004538985C8EF70E24BD840F0C140B933F69025565151E47322BB95F05253800496E034621995C4CB0AAE3B6DACC43DFAE0B8CCBF809F760AE033C95693B5663C70BFD01AC785E40AEC7486664E7021EDF656EC43CE28B534D8C8101BC06A2D107F10B237993FCE51A5711BB63DD7CF255F105811751E19EDAF5D82D274151850745102750F86048BBFE373B62875DE62EB051BC083D8FF3E3881ED067AFBEB1828DD3995EA550B309B3E40330D3EB27BC0DD8DCC9C3C83BD8CF0B1E427DBD0F7DF0216A538F9030B84D7DD7F20160E4A2041AF22EB0F9CC05F15A8B158C9707F1CA8C353190A58DEEA0018C361F069ABA19B5F3ED86A02078B29AE18D09560C0F4049FA429FE6A10EB130FB70393E8194C8F847F5D7E564B21F08BF856FE747A666FA239BD783A2E157023B0037AB086B2645CECF7D804A302DA9C23C7'
	$sHexData &= 'B56C24933CFB5D98928C382D6A0F89C9226A148B87926B0DD262538B241F279964190E75BAA2754065888DDC8856C0A73F1656EC52A48D4DDCF26520FA94865138F46B9474416FE8FCDC82E03BFB75DCDC4D2960043CF0FBDB01FD4F8BCB2BCFB8398EE338ED52417B11FFD0BE66E003C250F7E46CD62A013B9280C4CE4F9B60E832FA1F782003CA4095C6E9743D89FA7245B537043BBB3C104F5872832824F77631C1743C77436E3BD972C9B0EA85DE12BE74508F5C413A164A964EE7670887FF0145DC8D14DB8B349030035066AF9C08CE5179552BA94B11FB789F48091940BF2CF40DD02226750B23C218006BE6807EC93945147608D218C6EF578E0DE0180C83F8647E071A64B9B0923AEC1850352511F4B75D32FBFF740A837D100C0DF8821A2C020E18C209F6F11DB6C64A1001181100063EE62001810766108990171357981A367C2420A90FA66D5752EE24035C2003453E447EF039D88B5464AB545250145153EB0F740AE8EC267CF2EE269C704295801440B2062C381082758D9F436620B0BF1C9D4E141AD2084018DEA913325230D1DFC28D70024ED381357E3FC6BF8DB13C80B4A9DCC0B39858CC57522266FC08A185A8F9AC9756651C88149C439A3314FD6885AC7885037AE2708E6CFE837F039B1803F13BD07C613B947F5B3BF17C573BF50BA08BEA207F511083FF017D05BF1C'
	$sHexData &= 'E009FDCD452022920A6C7C0E3BE905B8E5C094C1048E624BE8C052AAF104CEE220ED7FB2CCEE7D14A5C21C0020242E268A98E418D62B1F6C687A41E0C866C8CB50DA25F3912C20AC8675329A808342091120402A9A8702E37808D233017DB66B157950526273200F3D0D73144D552CDBE65D2C1476DEEC1C1A240D306C9E2428288902482043D7D47A2840E40A4863307C4D6D5E8F3D443A3C3850DDD4CAD7345C3D5034514C34A69DE77906383C40BC5C7A36E97D47586A5C3E3C4CF8E9E3765403394A6C720E7C586992C2A53FD350785CB96486FF397E1472168B06936834C760665A745F9A322BBF564668589FE570820C87C812A766D2800CC9C1AF31D2A6B8B0CA1050542D2120010F4E0C06B4A90CA70E46E24330C6775CC8844C202824602ACEAB1C0620087C8739C8F3962C30C7449D7244C8839E34243040C9012455698E8E0CC54F24381CE7A179E7181A20248C2426A2D370DAE4454F4C03F9E77954444538518D54549E938752B61C3438021C24178D3C57444454D24CD2504C5116C54860DF81475AAAF614D60851308095616B814CA13D906D0F1400FFF333DB0AB16A5C3C8473012DACCBE035644E599130E93FCD24461E49A8100F854101ED155646A3833BC38C2C74F6292ACEC23000BC0CB31923DD0BEB025E08403B2CD20F5D573B5CD3008C648ED9B4C042187AEAFC86'
	$sHexData &= '917AA3BFC500E9BA888DDB33BF427C0EBBB1EC2DF0F8028BD8183BD2EB0EBA315F49220CD01EB633C9EB475AE0910D05B9FF9D180C3D46D457662DB80FB6FBC1E71BE14AFBB4FA14C90BF90ED0FA578DA0BF2C8380F09B4E042AE924CDF3D8CA5CE6D8249D12E8FFD002A66F0639875BB11DE2228BC75F5E658BB9215F4925721A0584233F157612B0420ECC02FC1812692BA89BE0390450015E51D0DC16DB5B06E00E001DE4093004F077D562C1A817DD7DB44C3A1C6BC59F0901395DC4721101872BE4B236CED752402400CF02F065995C430C27016698260B98EA0188788EA1B0F42DB4AAAEBEE1839CAE40E04B21BF922908E0D098F9E37C9BE2CE61B5B3528E98A3D03CE0AC31D2C60490FD5C1614B62A7F50E03301972506081E029C40408C883F1BEB22571C9838561FEB68606F69C6455220576AF015004DACA86E98594C9374B2AA9885E6119AA52EEB44176866E4C074EB69E0D03A07CAB45140C54AC005B0D112742E45B40DE14A6B3BEB0F11EE7371C341A118C0EAF918400635C4609F26B0A3F0FEB528C37446C99F0164230D0E3B4EC895CB14A715420450002380512368B77671B8BE101F9D8056E091812B83672C54AE821ACF27850DC9E2A53481D77FD6B63AA5075E1C613FDD6F0661D713BFF8A075050DA84B4AA774B2962BA5D6FA129525FC2D747BB8C3389F27733452'
	$sHexData &= '35B2D1D3EC6D64510452506A4FE6084E57611EE059A9B0411278C921C097BA1FAA305FB81D5E5D56C8A340FF153016420604AFEE2BC727E8789A3AAB48FAF7E92A7CD0DD800FDB246AF774192194B397091FED3C90C419A8DD870302C20CABB847A86D9FC0E05757A38B600C774454750EE6FA36D32DE823555C57D6D83519E90DCCFC81AD355C62CD7AC9AC00F8BF6753397D107548EAA1D15DD795E9740DDD6E3D21390B3C26DF7351660FEB50FE6160FBB8184CE7E4007684C3455855CA4D953E0CD9341338081006501FD20C04620FC60600064C181405543830F062A5065154CEB028521825610F4D2580F8B2630E8BC821F026F0578D79021E7FC151A63985D2CF07E0741B58A2FFC85630B05542DA7F5FCE1B208499AE658706B055008E8BCC3BC1A150A47C1F384AC16CE00BD65242361C56145383FA9E5973487C1EDE1516D66C6D173A56030C043904430C0C4A6006DC07151C987A36811FDA37AE89165B560E33D29026B837CBC65FC3EA17F00CE380DF465251C81F5C58E984448B450852A3D60460493F2125077909A20853435F047885128D4E60C706AF91803913C4A17E18A121AC03B057C47C606BE78AF101FC4816040ED0C4D063A68BCFD9075293DAC6D15F3C044073D4C01F083F515683C01850CB8B344178A2ECCBBE02916EF0F6464001269CA82AC4EBE9C08B088B56'
	$sHexData &= '10C7DAA429FF3E3420C700114E30C7013414C14706192434836640FEBCE013845F3C27CE5E9621704A9D0D2E3FEB08576E8BD685AA569402481A7A3CEA562C02F0138B7E34AA0478E4890A08820BFFA0D1F8890166520EC067C05D4A0A5FC2E604E10EC1F0BB0A4C04FEB75666128BC18D14503BC27316C9CB5ED88C903866BC59DC6E4E00D50216288F2553CD3E0D03AC2010907E18D2FFFF07D403DB2B1A03D9D1FB8BC3D1E883F8207307B820557800F091203FACEDEFC0B9AE7F2BC83BCB73044875F11D10065B700E5BABC0C824781D1E66CF8CA68B8A050D44B73CF83415AA1B0A47A300DA57EF5E7CA8D2589A4C528C897E5D421854883A981C380E070DAC481224ED0262042C3D4883292028D62030D1FF4DD79DAD1EE9B87B2E6D203D30013083213B2FF27880D7BEEB6B1A47A68956C8102BD1D1FAAE57D4941DFEE08B1A2BBE8D1C5F891AF0C6BEEDC35700D22BD00003D7468910E7170D7753F1B3E356E73AEB27CC20E2B336384E245EA71457893BCC7866A96020226630209046080B6F3E51706A1476834E1E2700524053615C3208C100E15CAB3D5F4120B41718E8FF4D6A3B0276465655F8ED737460BE01F25E740C0A50FE7406F6419AFD99AFA0752A8B4130608300FEB843C2106E503B884980018E0FB7C25A6A6F94BF3DE1C6203845A78A1832AC0789C35375BC1A6F4E'
	$sHexData &= '79308B1FCA1C583BC3066652F0B302705B2CDC0475313424F1DAFFFF283BC6770539713C761F39413C730389088B0412CD9C65BF148907FD4E6D06AD440335847F51E7E92F75D80879538ABAF6C3010F84A51E0F63F2018644400085CCCCF0CE5D4BDFA1D8171ECC2B020B9D66B7BFD49903F813DAEB3436017519784018013A80A13B16EE0C1AAD8826E020C4F05802A9D94D9D3642FE8C187FAEB9558480820E01E92E30BD1BF8DFBFC6743BDA0F8FF9F27C083BF80F87EFAC78FFF0F275FCD6D1FE03F7294A20F436012A059B2112180242DB71AC2A0D0AA2065DCE511CC3AAA675DE500E20891E49342F6DF008C03E7DFD4E4A9294540A2A02C218851E88A0722F14B97C3F7F7E11ECA4E87239A97F287C049E34050BA077229818344924135D99551198B7640B5D57429A68A98A38E8C9B46967E95E895808040C105B38A4D23A3FFE92B803E102AA0C53990C038CB24061F74D8BDA139ACBA7F79600D6E0408E66673BFE7586850C0DC0D804B29850A587A6246C7C674E0C92836C71B26F49BA00B00045530EB99389AE0C4BBCA2005724F27A43747129F80E513BEB57514D0224345F1E0E40923C368B51326BC82E971D17E395F3009899049424207438C4F050FEFC812000F022799C1F08A05D0CED6FA09E86C0178B7B042B3B18C1AFB11BD9FF021F464281EA3F765B4B67019560C0'
	$sHexData &= 'BA5ADAF3C7DCA755B315CDAC0EB8854E4BBA1B52E04226DC1BC2EE722A95ACA015AE0878155BA58033AE590004A1418C0A9E067763704344D5D38B938476B607D88B4FB257D2B0119B37041F895F04085BD1CE2F2D865F0B0A1D66E30F2A038BD12BD0729C3E498D30580E58333B538590AC108489C12B686D09F0C1C1F8023E9D53F733901ADF4E03DF5FEC3574C2519C3F5642D709E00DD6889C772E2BC8405B2B4878FB505FA4EDAE75DE2BCE0E6C9914817456B442CA903850F9744DA3EC83DF458D3C908BD71D03D2024CC183F709D9832C2BC719ECE1415E57511803C39E5FA3C5201470C7117D789A80C5789CA4101747BA85C80B50A2B48C131E4607B25F733F9816D8F01C1E08C271F657F4762C0B356C28489A3E745837A3D55923F0733190C8C0F79CDAB5020774362BF8CB6C2874F0E1779142F3AB8BFCC82B0E002BD9A580F7D03298305F711F5634424E0B50E03B908A56F0AB265011893189B0709851E6D3B4A58071F81608088D028A4C08109BF9A29ECA42DFFCE06B0127E6AA7AEC744955C4544CE35327E330118B7207461CBB20705E6B180608321862020338FF5B8AB4194556C7F11F12E9915EE230D1ECEE71BBB91EE7F085FF43D9579980C8ADFEF948E21CC997E1EFC94E204CD8DDE5F73B3D8175CB32810CF2629F0DF82C18B8B978F15662B5D83B1D11F8D77B49'
	$sHexData &= '1D837B200E1C8D430C43E2C85FB946108BCE79541C3112D8A557CED056B945CD4860B1ECBD843FED261CFCE5AC505392506324F014D1DC030B1B88AFBF8D4628604328F4451A559F7280782D0074AC6E2D0D831C86D8EFA66FC0482CAA6E0B80793E051B5ED1057408EB1BE4085B3CF9EB94B70E810E1A78038FF5B78766807F2E8B7304750389773739E23B00B4581805AEEB0B391E75B9BF82B6F43EEB327E5415351A75205F4B00E7FA8ECEEB14ECA7F6667C4546C4CE01BFF69EE300DFBF0D535908757B5509DAD1D996428180086CF1A6BC95C2361D77235A8941C60B3B435875CB830BDC1A1770F01E3F29050C205308FC03340DC2AB35410E7ED401CFC139193A0141C64ADFFE9D7A8A532C8A482C8850044B2C807B2C3C1840700A2AA73B653A05109CACF61FB4EA019CCE3BCD0EA075678060BF619E582C007512C64130355F032B324616550C2F2D78ED50D036C8807A327596418E1074CE167CA763C02312F0C6400E6EDD5BF0F9635C8A5600512C86015AB7E4FBDD983C99EB79C711633D8761DF0EC515A322950E1AC18B90C0A7EDC076043B79843B171C6B854394311B12F9DA358702E8F701F579D5237B5CC647108D357F4B0C0EF072A4D64320431C973A12788B94530C3371E4B45E7D0688760691E4A8119AF0DECA0856220539C1DF0A193987D412A6F986A211811FE92C'
	$sHexData &= '2C48FC395F5472D1FEAE7F4F108D5748391175CC573C8B474089112220B9BFC64DAA18302BC0CE385F5074345357427B244CA7EC42547578D46CABF4D5CF883A044AF9F876330270551254894F4C0A98AE92B044380170A10D43B115BDCC90540EF6A50370084C59C31F24A792C3657F094875080CD7DF1A198BD9663BC675073A6A3F645A2089EDFE8B231034D0BFFFD08E513BCA7317FF082D8D51028946C533ED8E316650575054DE0446D56CD5748E3635C78B624B483BC16A1640123C50203DE806A6CBA314EE0C427B9275262C54AF20B9EDEB78070AD5C10FB7C6569883969268AFC1A51ED6E44858881FD59E4D8AD4003A01D41FBAD07E25F1D183FF10D855D48BCA03C25ACCE030A0105CA40D74EC24A4D2515C443463861D71434CF97CA288D807EA05BCD48FE097B727D1D9DA8703607E2BF074D044E1B529214B54C20109A003C8A3D41061FF1B5B3885E973BCC6434A013955C82AB0D8218C188C856776FD4A376930460F83BFE0CAFF2BD0E0DFEA4A140F86A691700883FEFE0F879A4F83817F64FE7345505ED4C004E7BBBBACA46EAA8A842409669A82B01EF3041642070EC00CB25C96395C851EE9C60430AA887142D3280BA1EC146DC600313E753E6AD8C10ED6A88502CCA07DCA04800864C8B869BBF4427B769C10EB2619D8F2894D56EB1846503190A2C0473BBF421B29'
	$sHexData &= '9C503FBB8E1E844266163AB01039C9FE17EA017333457406663978FE75234628C6AE1F347620830646C2663C091ACA0925341C8B86F8D94687839942B973CF7438837E8EE394F872154328BA6DC208B57D81FC4E208D4648D07413B043DB604377A9DCACA1B52035E9CB9F59E076C81F4E138B56300578015EC3A9E0DC0C22501C5738F84660003C8366F4AD5B78E7421057C51F1C5C5730DFF9264C2B5F8F08507428768B8B0A904FED69FC48173630CA472067D35F583A9C8DD4397754C25A02B9342395474F4757CFDA86EB164A447525E32262B7C036D9746EC504BE1F029E370B3B18E38BD2FAC645D48443A0990090541D29E000871A985E7464F528801F9EC98CC883F9010670A96C7901016D92258ABE4D3BD6D6C1E116948916869B559D48346A2627A2FC93881CC4776E1587BB5A4CBB100A68AE36933BD3A075CE17E26A3BAA06D2528B0203F0562A573555A84F47A18038E20A46477E62D8BA2A1A2FBF0F15460225D0395DE86A02F5F4E0B4500AD0BF5812D05A3DAC699C0723814CD7343058436C9D8CF4BADA884326394DD482755D440C969CFB92077804D0B842363432FB2A35FE202BC65003D652FD87831996C924EE1932ABC14E07134DC4FF1D82058185024DEB60AC2DB05533E2B90E2BF8A3273EA8D0D87E1DFB2C4FFC8A0B484FF2BE4416FF4E51F4A80C1E0BC27FE9'
	$sHexData &= 'E9EB95A8A51B177E4DD5C22854CDDA1FA93A1A1D0CB4A04F08470BA502F0751A9026837FD4DF48575E0E351083C3FE83D6FFD20376260614203254844A33B43C62A78630CBBDB1FD0BD6750679CC1374A043FC4F465335348451B716569C63ACC40402AA3810DF30A8B0B497714DF2101B68F62D2CF5F27214937EAD97324F4C79C740047FC4443AA950E89F1489DB0D357614648A49045FA4425D49403C10BDA2418314FF5ADE109694C26759F9980AD5A5605C7F1AA482332005D490818FC78B4626406BE20BCB7415080F046EE16A012EF1508B4F1901D934445051D000EF50189AC789574CE1AC82816051CB72D0CB5C702AEA20505CD5313383061FCF305ED4F404515BA9419B4DDFE1520376CA5457625A5EDCFA8503E7C6C70B4510489CC4A5E0D13E03E64F7C828BDB487F2D8B7E5478683608F150018846F12EF44E8035E128544656ABE0F66EE9040C0ED8833E00742CF4818D5E120C6835FFD2211237553888D8565B3882B7403B217905965EC376C35F06FCA4274D085770BC8BF8451FD8366984A60D5FE7D9F8408282CE10442F6960C881107F0B2DA912AE427D7DBEF0BD9EA88024605318CDB0097147258D77047156FA544EAE108108070F08F37BD06C0C75EC3B00C63474049EF6CE212C6A501705CA3471157A54AD04A1318656772C3C198089FDA63AD00F6282025CAE50'
	$sHexData &= 'E203E944A44B254A657B255372D09FF820E4E472C80C201C0C06B94006FCF8F8B990414EFCF820262D36190CEF9EF2A0E4920CF8B92C414E0132A0E69C840C11C8D3CCA045320472A0A005C955C8A0A022E400C9A0ABAC01E410F2ABFCABAC072047C9ABFCA0AB17242790ACABFCA4AD53C924A4127AF7807044FFF1013BF0736308FF835DA9483BCE77532BF123CFDE07635075E2AC5C433B83E6F0500C8884030C70778AA511AA04503EE66673B8F5C3178B3189308B78560AD661DE4C080411895104A444A17CC897E40A8B31C67420C2A30186BF4E759600B8C26CCE16A8702782E0205510A30F447715502D50BBCC1E0183039F61F19B8A4710A44A04A4211E57AA9ED3C08B3B9F790C73104982EEEE6636D18B093974EC5FF3551E75755D84D19E333B72B80F5208804482A0385E2455888034876622FC882106037F4E012620036798AE06543F56F1735226084C2BF07F2C3C1679EE60F2C1EE1F03762BA0E9F23B4FC607EFAD3C8BD488FA32FE04B68D0C818B479C493A2574A8C401CC001A41ED8302501C4210833C1427CF37D85EC37161180646309834274E483FCCA1551069160A8D882A342E565FD3549EE7800001404170092AF44D71C01483C114B0DEE82105C7C085B4BF2C9DCBC30802726C1BEE8261A803BC560476C11A0828F1E0040307629CCA348970BA3C8C24BA70D3'
	$sHexData &= '346D042E0A08080C0C06067F0CD94010D95910323C1CA92C0C4475DC9C0EF6A30A50EB4FB147A742C7C715C1EC3BC569DE34CD0AA0484658108D48C081070EA4C6019021C0F1E0779A30F90E2FE1065F064142E865A8805758791C08E15458EB085F17238455A97F12D04A4EACAAFF5356FE300207264BC2AC033BC72CE0C17AF8F74F8F61E8A704AD4B3FCBAE1DD835340BAAF5772BC6A48D2F9C98C03C81AF743F5038D8CF04C94616EB296D7331B0376968C0581255064429BC33A78343041C9D0970F870CC17161118522D1CD5F10F38D6400CFFBF7838978B544E15399A5A10F904C7AEB0A7B56115D476BB381EEFCF744888832D711E6D1D074F835C1944B5408E383F1C2068C0E81810D62B0CDDB1851BCE252BF15447C9C7EFB7A92ABC91243AD904B15ED91894040A953DB5C33F04D90637D712C914C7FF5C6218F24A0DCAE19D8D7339645EFEC71D4E50A1AF1CDDF68B77E63E757C481C708B13D08DF64110D86CE1D0AA43A8E760FF355BD94B20894E20EB45957DB7AEE85F1BD8281689138D4B60E602998D464E9254E8B6F05224D12B0AE1E03F1C9FF00A80E98D59A080944360F052084282260278B800560842302B2BA9E0E0531C34604F3AE435880553C3F11E3462633F9BC436A80671012262204D9E08FDE4B175A447BF86BCBC81C2001F0D9C70073D858D79A0658D7760'
	$sHexData &= '1384E8A6C71B57CCD1A0243504A643E8225B017371D7586F1E506A0856CDB935394E1452004D13D006C657126C73EB67083309C64018B826E12DA8C6460D3F6941780F60A7837FEE151D0AE11F3244120250575671EB0A01A75C13E23CC75B781979A7A37747864747499F843B0FF09F0C39B8889F278B79103BFB95FB20DE2041081D8D0C1F487836D447C39A112896C9985F08C0267FD82CF24B9550E676732731823A74355733C29C90E961212677B80D54787B1A6E680939464E2E2C3075E7897EC24C81D068F410BC9D766103EAB9CA205E582534EC1B18049A3F5383690D2D205151A32269C121719F7E72117846033516C7626E5F21BF7C50E9A16842108E1157C0ADBB21DA352BC17CCF4505B42E837AA4F88E02C218985E5D8C487CC684B64E09ED06B7135F3910140E8E6F1AC0024975F5483CB0D206640300F0BE9592F83741B6668B19663B1A75210B87A5B49083C2C20E6FB7034FA7FE5FD45ED1F86B1BD217F067FD83E2FE4274E040C82B4DFC4E8150F87ED1F92BD103FAC0EB83FF4F0134114071E4CE6A2708371A7E7605831639F01051A42D600717810BFEC78D34489050168BCB6A16884CC1345F16A1E3D76EC0A27C1E2783EEBED5750E8B4FCB28C12A27E1E6485407691BC1477250092C1FBE56166EC0792E137B03C3C32788973FCF8C70862FD2A0C11035FF1F1A5D'
	$sHexData &= '100C6C801ECEFFC347507082CA13C1963A1F6D5A442173A34633CE295F33C02A428D158F94684677119009508A6854B89F2C296197503A38F05B05F9D72DE955F052C745F0967044219F0D141181637940393B7E44E048403F7DB4F61875611968CDFA8B5024393A6C12303C81C040818E6ECAAA780D2891622BC8EC66B6808A6C452FA359EDAB1FBCD3897BA28BFEC7439F8BF33603EE66E911179C0F82AA27D26D0308A1A423CA04755DC0BA89055A39391409DCADC94E42A06A0EC402C94B84800CBEC9D1DC20EE407245C82AEF690BE1338245CC7EC8B874021DD89F29667B8D7DB86FCC080DD2C54447C3E92320D64601A1280D141A080306156F44408751D6B88FB5427C56B001FF6E572B583B16DCC2BF718AF994737F90702D1C03C19ECA25C6EFBF7AC1762C8BD1D1EA552BFADE620304051CFC42C889DD001CACBF76C303CA1F754B58045FA16B518B587FBF00CFC63CEBCC3B0875453BD0D5C65BC475416001CE7A1242C55F0F45C48FE4E1115EC68D5E3AEB135E645EA275C9500BA275465DB81E2A75F54A41750F3F5FE67F6D410AEB1F4C0439133B4808750ED1AD6F70281441ED225230B6302054C7D48AED3F663B4D0C75A0890EE371F0DB4C3AD010FC08807A8E03894A47C1A4B5BAC877998052DB3B1C750988AC4AFA9A66D2C3323B0AA8021D686088B42E089F013962A7'
	$sHexData &= '9278A6089F0A2695165648A1220808573C04F6B648A7773D59365ECA742856AF12845DDF6E7FC12374F5C3BFB0D6D7DA4C7810E14110DDC47827E026EA98C05FC37F3F0A317908785D33DB395EFA84CFF5540F85A4276A406A1263E86048FC4A67FB0F843802046426375E203E029A800834523654895E025E3C023DFC2C5DFC8A80EF289D0540EB0B8D485C6C440354CB2A62C91BC73BC39DAC764E0C2F33C02980BC1844BF09E840C18360EB5F44B45EBE81F6C8B05F4A10554E072C25AC0289349B0A58202D923C2A01038317091D9EDCBC67C049797728FEF606CEC0058DCA022ADCC06AE0E403D1FC52500185BB2D38782BA21B480483E8027C85C6C0B1B1BB3203330A0F7EA981CA97034A68C22B75D02BF108FA90992999C14F936F7068DF78799002F32C6B9EE04EEA830C4DE4C00F869E2B414C39F558A32606E1CF8FD673445107D44AB84DF9748630E566F6E009B9A308881C306EA85464BCC314D475CB9A6388182B08611F3C43B9EB0A17C501042A9F45A59952893F9F50FB4568DAD84839EA1836568B703C89322460FA1540205640302BC9D1F9FE5318B708A35EFE64DD6C1385AA15895C62308B366E5B683C20200A347720405F76EC64BCFB81D085D183C24A508BA752CD748BBD4232811C8B5736C8A70ADF11D121FEA25442E010C3581942EDCD01B43B5E751E2374EE14'
	$sHexData &= 'C394928B015FA0E9804011255FA0401E4452A4937CA67D26B15F75D87E21F0A7815736FE754CDFE05BC42CDF47FA508D4043E0ADF73FE77A27E046327DF774C15B5C556004933FA003241381A0A09C1EC0250D13152A74E32B440481F9F00FFF0DB7DF0567502BC241C1F8BAC8762D9CD0B55AC4F7D1EABEFF4A2BF228730EFEA478056EC10F42C13A84B9DC03C28054A9C01BFF624F0A2F7D3391385758B8426A1046218B06FA343054BF0EBE4E815D24AD7EC61064DF8B03396397440DEC897A524B08056728115FB2BBB84938BCF52270392BD7681FF2024DEA81FECB3C0CA1B5D320D188511063080F0146A2308578A66AD09E0C6AF83D0BB129E4C6C6506ABBCCC280B505891F1BFF2BF9C11E49FF5455550547BBAB6D51FF5560FFBC0D6B9790BC1EFFD8196AD603895682F389CD84BC6057044892240979FBCB15AC5BF7092D3C973CB849640911BD5B1827B6150B213FA1133BE17C323DFEB5137FE2133040332D572270E281355F730F376B31BC75351DBF957C621866BFC6711C0786BB1AB651BFC760BFD82E218FBF8C1EF7A9E435593F0778083888070BE03A1C9784CF1E17F43108BD508C1F928B440238BC098142A1C218493C3B289833525F401029185CFA1E3BC2BD4EC3804E70A9D8468A18785DEA4C778385DBF1EAD21F21578D3C18C14E14395C6A04D027BB550C5C5C32'
	$sHexData &= 'A079B2C12C98E64FD32AE4F908720FD2ED9D2E1528C6420C1D19C623E8A042BA5351A1429794EEC13B6BB0B49B580C785F7B1FA08384A17F574B476DC54EDD5AF6296872351DB562EB6EDED67820294648F0E12BF874D1FFBC93511BC0615FA11B81FB4BCB7F80D9C373198B4E1051539ABFC0A0426567FBDB75F2895EF896D95B4E5BA30C1D19082EC1C1EE57E9FAEF83A59D6446035B6F1F8A20511A3F49FF69E9D4225362CB1BC05EF7D8E06FD85231807D7A745483FB08734FA0491D0E7456DF22FB1C543CDC65218B1E840E303FBBC2250052219FDA0699B530F536492D4D0ED2DA147EB05FB3FC0F24F600A0A4362606F5C6202F629F0AD510F8D243CCC3106A348FCF2208A6C2A4BA897D74B44E8D4602165689E657A8A154002808F63A183850842F48F44B8918F21B900E5354A57B3648431C1C13242492F648CDA1DD551073D844EB3F4128DA80040C208A09C6E4678D0AD20CE4E41F2B84F8D01C535856D880B61C84E07B2D00C0E48560DE3EA04DE889E40F851D01A856E31DD071BC148975F894EB0E4FCFDA17FF188B4E36BAC725300ACA55F0065676FF17BF4FFE64F739532072058B530C620890FF0200C614668B02663B01752CB19DCA4076925A720BB63600700CFEBB791E0A42186496D8E8CE7E00B012B4D00974D555EBDE65BFA0B4AEA47466F3AF2498F2C54B087B3B'
	$sHexData &= '5070B282F700FEF8B204F0ACA9E769328C13D308E08E28006D0B5E19A833D209F808164678419FAC0FE8AFE00C20C0130C16051A43452C245A2D024636DC25F542628A005124744968496AE4AFC4768DCBF08C0E83784C581C8D48419784870C57B6D7503C8F3F4FFAFBF3837DF05B1412E02CC9355322BA50615330D243FB142CF4EB10BC60C1BB3FDF40EB4E400815748DAB11F11D43A6E85F4704234DB372B91F0CEB1C2C5450535AF87EF8C727E8D289086051141C58709B870A5504902642F40806437BB08E797A3CF7CF675C70004477185E6018C8057F20C3ADCA05FF128B5B08EB127BE1EA20B006AB988B1B1574DE1890A95A21211FB5BAEA21E414F96A281DF02174A2E01C7BECC10C2CBA02682CBE48D51BC224021C4615A844C6C2823CB048128E1938868B5234690644A4925753F50411BF4DEC5105D282B8D2C85252553C52F2158210506D45E050E4AAC652011FA01521070AB860E441840CA00D3016C008390837E849AB4237E448C314415F3CC7736F46C3FA1BC44A95D55DEC4FE85A70C7E1DF4EA2BB50FCF17D4804FA8C95F0C4082BF0B0B6BACCE18F091F1D2BC1E70403FBB2BDA56F4EF3215089A9ABD1BA226A48E3228646A1EC5565B56B9DD0DF6257330FC4D01348C7733E897E50A5166A087EEC44E137565A94099E9C4EF3500B6C3B36329D50B715EFC8EC3630'
	$sHexData &= '0B001FC0537815242DAC06F9501C178BC766DF0C7248A11E212121910CC8812121230B35C9215620C3FC9FC98162D8890FE7CCDF085091FE06528481F9CC4B460A1543565816ED20AB8D8FC1736186C6BC5806C28DC84433F4A750FC1E30EF7C8B1E5A624404DECB7305572488F1F403FAB69D456F5F14484A808C8D148810BF89560822E182385E06CD231854ACDDDF48CD00C80D107DFF4460E8D0C8470E02450BD271DFD4771DC40A1E311CE3240E862E301830E5BC34A91B7AB80E78D1A903D790E2DAC18C1EDA0B3BE010841EC08BC10FC7215CA53F08036AFA829F40C1E104F65B03C8C1E202D004A31884180BFF9C1402391250BE51696015FF154E939187027F7C2B5F8CAA06E28A5732EBF08C9961B0CA8B3ECF7F05085F0B9DEAFFE46306991652E82C4C1C07C5ABC8833ABEE113DD1DCB1F00A91256F0242C04761630D5D45EC5C053A29F2B494F2EAD1AC82B06C60E4E5D48C6C053576150D885321D92C6F553D421BD8083E82B3825250EC9022F0833497F42BB8D04935F84A4C817A34BC0FF9F0CF0ABE2A510A89F5CB426008D9F73253984BCC83CF15C8E64A0E87D9F5CC0BDA262288FDB838B0017C3014253E0C33F0846CCD42418C355C0260CC4C4CE0FBA767FAB1F182C278B5F1455AAF7E68BCBD1E9FF0F12FED1EA3BCA7613B8522BC18D34193BD876055BD820B8BE1B'
	$sHexData &= '0C4E01C21516ECF6C83C9A7713516C4A201DFF50B2EB4D5E2488450408DCC745ECD8906648DCDC5A8C181864A8201EA6EED402FDC89C84A45E166E04C181C307E8D225344036461A1A10E846A87B5350709B3210BD26EBA00F030CC69E4C30768A7C7714C4A67E0C08C2C4F8C6041FAB5A1E60938B750807DB60160EF43B1EC606C869484C9BDF09C0BCF45172140781FE96C9814C34E57F7FE134957018F114A427804F81EFDAEB50C8ABB19286ECE073699E4C0DED0CA28CEC2B3114F3C2F8F0DAD52A85A310B01C0F93A532DCC3F138F70A0E456FC3177329B41FF1D058789E4F08FB654F701420C5F51B51180698081D511EF5FF01A3B01F0F7711C1E104517C807C2045FC504DF05105B33594518010AE42E3C201100139343E53CA3C0621486564902110882185466B215104090A8DFB5B2F1875235A03168499642121E2024CB7CD8308B95D07CECD08ECE4EB7BB2650B8DA7BC23BC66F53A5DFE02390A1F214D088B4904F0371F4253048950208A48208850202CF8067BA4884B0AD0B3013858202A0301603434CEE429F1CA85C6FB1E385F2AE71D69B789BB8C2026885920F042C36EF3462000C97C2E9422BE81C39D219548840E7465CC6E9A45622C8852C64163DC44C6AD8D8A808822885E20046FC9F7CBA59DEB78CB146770B3DBB00EC913AB1ED917AD738486C8B81D447B03BB'
	$sHexData &= 'D9323B1558F77C8FF5081EA64D09F3DC1488E4F8014061E6807436A09A4CE81214FC6E06D02330E05DD7D6058130682E19A68940887ED804144CE710A618C921368ED56951D0945B484816F0A81C4D4F8F24AF3F461105D6639777108900700B279067AA27CCB5BF4576EAAB9AC368BE49DEBDF6126662BF4C36771BBEAC2AE888CD13C18B751822313FAA3F8DC48582C17B3FBE0608E93016FFC93C2C2AC015120EF3AF916469BFB22130C02F045EDF7571569EEEB8D08549C077C81B33D290C842B1103DAE39293760FE8AB4A746F460EE50358D74A39D86C7A9F175DCEDD7293182C9578957100414FC008F351829807B06FB74915E210C5229291E3C3F2057A52155088B52040C0017829B6BABD60A61ACB80223B1A910121CDFC5B642B860BF0DC5C1109254DE280808F225ACB6D0C73E212417327221212117960CC92121DFB0400639082A1C585B7019661C276B390A087FCFC7150F1E04068BD44C32884432208C6EDFBE707C1F7F9619427C157F053B3E760E2B26A2D1D8081BCF46D8A249EC4DE1EB0F66CF134588F1DF4CAA04208B5432388975D885D218DD8C6D45DE96FFD28106DC56F02A36837C300CA210B8303CACBDA43B0C0B20C29EE81290374A884C4228B6224184C996746F880FAD0499CC0B30741403FB4ED225C00540743A85FF6842EF52278CDB5A0E1A0FB754309A'
	$sHexData &= '85EF82407A303852AF18E740D891F185AC764D9AE0C47A8F0E752EE56C5B35F840FA31573720F0CA079B798510D70F848EFD79865E1302C0582006248809982D939A8C521A1F8C0E3603D14437F0B02EFF20D8BD6D610CA4CF98EC39E01B68AB384A1411B55CFC542CDD83C3FF83D75EE4C4ADA11EF6CB8DFF39180F8C700AEE1BB8DC087A6612EFDF7506877509E785F5A155E4FFEBB929F8E05A6010686A016A70C870EE780E01B858CCE8C3E490C3F1E1BF07EA14B86007831A58B29A1D39F1141E2420F11E7C4CB8048B1C4CDE089CE31788F2063BCB760C2BCB881467B76417BEE8EB0A4C20CD5B18813E14BC80E09BC47130E980D0D38180DCB123B981E4D1A42379CE4E38E62A5B72D0F11575B41A5613DF50B6F91906F9F8EB7C464C31388653C392A992398584E09C87872B4FBAE9B2CE7708E1005E54747FF8AB8E03E369CE384FEBCA60989CE85DCF17BC43FFD6B5BF8CF08D47075C12085FF9D6EC749D7126D6388BF872588CD08C915EC1D22279073F0CC9A110C01EFB33C07B388338021CC3710EC90436F98D7842F3A2A09F85C9C75A8D101406D6C220D8241470288C7423BC2E493C2293B16E154A246C12215B63038B066378EA33705A0A3D4AB07C3CF17B54132950578BD5BC0E08834521D9FF83458D201057A7362F0067607DC7B86A58947A7F0777188FACDF9E2BC156'
	$sHexData &= 'C434856344D65309565150CBC9512D4EE79FA90401262803C68B800F9560E57588D5407A48781B2098FC072F6C519092073B700C73238BBBE9C16E3C0CB1841D80280074177300C5E6EEE435128B5034B200C0534497AEE3AC09B4B708C875F00C4BDAF0DA6498A95174B4751C7C77020FA288C3041A353083520FF151254D8498F0535F830BF96D8CAEFC75F06D1AB8545631F1D4E03082D5F3FF455736EE323118938500561B351F101E99A9A58840D009F893458D9896BA7128A73C506391DA275CF6DA2CB4C36F078C42A03257123BC3EC699216690F89631095764CF4AB46835F857B245011B642502C83C182C71A814B7E3644567F23A5143803512A1522184DF164084C8B0199536FBC795C20F8048F6E68E022E20AD2E7A68B43E29EDC3D207384DE6A231148BDFC510A6E150C7E7D2A75DD6E62D36D89FB6876AE334B78F42CFBAA1C0AEC0321AA8F7414D40EC018E90669CE3F505FB6D6C01D0E19E10E18DF20301D2E5590541358B7971055F851F18CCCD635560C301337DB420E9928DFA4A022392A19DEDE89422699A0DEAE4A8E90DEDEA03942AE40DEDE03A3E402DC04442712DF64566728202ED0025ED872485FFC07A6FBC376332917D62BD851D1FB5356EC1E200BF3D12854201EC4D52D461A72F981BCE1CAD20F849A10DA18C8101941DC017DAD87F2EC80864CC510874C'
	$sHexData &= 'E55241C85E0812A182B820C253C5B11D509211872142B06E6A5F2AFC690D0AC43958263C773850106F345D2FC322AC0D37B92310E6EB70AA88C756E03C9A89610F413BD737FFB86A20BE94108D1C38B705B14140C21C50F5A998480A67467CF803D16CA328E42602A041F34EF0F80D5B9D08BD27E8C39B0CC6DBF2CE170158C15C677A03FF51084463C203C75F9399187A801F3C795E58BBF7A40E585BBDBA1B67EAF4F8BFB8082B33D2F8788BD63BF7BCD7C501100D1A35C1863A27CDF34BB81F1851C04A0E771AA8C40FB7000104E0FE40B103DAA9753C73090F82D28009DC1CF86D5C291507DFB580588AC40CEA57048053D8649F7CB0012C4C95B143D68845BEA02EE18F22DFFC837A3E8CF14BF87A1C8D420C687DF483430A3618B8383BCE62F18C50304B68E334076F1C6023FE5E3938D3824BF3C1C00853E2E0213530945822EB071A781AB751DA7304F00056B0CAFE0A86868BD1D2A8786940988A6D558B58D05FF873B8F435A05BFA4E313B0F16C3628D8E400B340252B78B644B582DADACEA040194450A90C1A8F46830E5E045514E0C6443086B850D0411277A14D3524B4666F7460C51DCA51010C6457A4B533D846E49F11BE3D3C2572833280E52E0243693F7A4AED78C8400E185A34B9F8199B733B590BA61723ACB49036AC9351FC9F990B10C4B1094616EC326E4301B325509'
	$sHexData &= 'D8BECACA7516895A1D880DA8EC6FFA0F59081E38450C74108919330AB5DD56FE153AEB1132215E03E8F80BC0705C8BCB38422C409C27F07812C43E96E818C422503B160D40B55F0C5C38462C751CC642014259A3EEB43A02881A0E2A75AB7104E4DC68953DD0B54FA95220326E32622DBEA301F5B64E044AE635DF4ED1FCF9005C5671043B0E1601750C4048FE0A7AB6BDB53BAAC0B911908B7E3E9AA6E9BE3A083847A1577A7E3DB011D05C570B36868354DA350B6536240808898654C1B13739368B42826B82833F8811F87E31088901A59D830F180881412C9788AE048D5503B01496687A1ECE10C1088D052E7A7707E76917378616EA80C0B185163584665FAAE8B1984FE86618545913279219FF21471472A0A02453811CDEDE3205C845A4A4C9C9840CA0A0A004C8901CA0A005720072E0A04CC9840CA0E0E0804C2587E0E02153810CA0A0D6182013A03A360E0D25400D91C00BAAD682F2460C979145F1CFD742202C872601628C182335DD14E39334874CF4302E1146A63419E50D5223B64DE91DF323232E94C25103C3F51204FA2B153BF075406083A0ED3D7403F852F8DDA78D524629772E79C7CCBD50C5EB391BA41F2939965E125B503A0851728DBDC8395614EF9A574208A4A782AB4E6E10BCE01072128314488521402A8DB15B23B10A1568559F68068CD9C05201243C1AF017'
	$sHexData &= 'D405D64A0D0C90922AE809D617E0F5E8AAA0661111DFEFA0A26E223F02883FE8A8013830FFF1F2817034722B16B7E9F8E8AC7803DA749601983E2BCF0CA0A18271B98B6216860482295A6D696203FE61B798841828068312A700E60374C8D1E90BF9986F9E3AD0790CEB0503C189394CBC00AEC485D6419870503390427801F70E5250BCDE9CE083707E397A30DCEEA75826402D0C522FF6BC280899181A764E1E10E220EE1436974F19C74A41EED8F20D1B0771729B2A07314817E744A15B6902A4E4108B75E41A4934008A504F3658B5102B894604C25881E0F08883677025B12C3C5B6A8D040F857FD46A834D9C140184D54A086D283F144A14E26B800E9408E4D36719800907E5985B8403F0FA8EA83B57F0719ADA14B93C570ED3DF87053501C8B5720DE1B78BF00D03F0B2F8A875DA200A972BCBBC6A9DA821E503CA420F436CE23B4DF0EAD41B42B785F4D0536EE254F5C15D501294EE73E1550052522710B88381E2D8CBB7FC35104FCFC66F688F5B38F8010674703B5315DE571BF53CC603826D21F3487EDFD3C3AB0F766B0E57A7F3EC8704A41550B85FC781F4DC1803A9E48A8DDF95CC6826E4086A300879ABCE1678C60F7582214F081B7700BB3C472CD88D770C4D998D33052150029A3A3642E3FF64D84F86E8D7431C3868AC0A4270C58C516C9C48E960329E9EDD4808E31124'
	$sHexData &= '9066E4E4DFC8C945C975A0A0A1420432A04427025F04FC3B75087472EE5AD25BF8FE745D837F7F07B5C03508BB0FEA6AC84B17B856565748B802A8F60EAA1E34A800EB801EFC896C0BE0232142D80B83DC90E01392DC7594B6DE0DC620048178341A98B85B7228833EEDEC7960624006EBFC1519781BEC94308D56D88AD102E4602E727F18C58296E624E8DF015D491CF9B2FB7430F00FEA8AC22C1ACEF78C2BF35080211F59EE2440203C09A538FA1A6CBC9DF5C38B0DFFDC1406390868B93276B6424B458C46A1849D7E4D9C37B190F8EBD8F58D14F29C5058B8BFA262C3DC8D3497880F043C703E3EF20C3C5E2473ED501AA81446DCC496AE041870A4FF5550801680638F36C6561D076D06A551CAA49C788A7321FBDE1676E09C56A233139EDFF5DA301508C024407292FB8975A03B3EF642C07D9465578D0F85A5411924F91E5657C996706240645EF8BE14DBA82A51439A8955A3215E262BD308660283347C73E02F82D0BF0789018D472871CC40CC9AE404494F1A4EBA9FEDA08E4A20329C890751AF4F015808DC3D5583A023213F3E895720E624827F99D881823F5D394590983C18149CD946DC5202565157F0CE148BAAD852D1ADC80DDE127BDC83EE24B5CC8D46060816B41498BBAF566081D00CA9DAA4559E99060A6AAA06DC6BC99EC10AF9AF5A013A336105636EEDD242664007'
	$sHexData &= '82B54120D1E1BB4C2CCE6DA42483E8CF48C181BCB1C82D3B85517837B848F2F3842B0310DF1783073317068B518C41572E1605852E0203890683E7E4C98D438D4E4BCC6E1449CB8645184E209013910C0417B7EC734B1CBE28C3C62436982E1C8B3A4D7ACC627DA457A789C56A3A6889C3B189041E41BA8E91CA49450310116B78B89DB74EA4E8988942C760362460C1820FFD6458A04155CAE9B7C585C769389AAC2E18180C881330BA8F47041B16899971A025570E226A04168D461E06873B46204EAC16B33280341FAD4E164E8A0F2846BB83EB24C16211908105248862518AC77BC75401644378D2FBAA66AA340D2BA847AC6DAC79261969C0BC57C8EAE092A7A0060F2B34DC92ADE54294D262F0D63C60082AB7AC7248739373C8A0C0134B136E4273FD66344EF068C2ACE65D458C890E0B5DBC58E15B065D197F320BC0413FE0568BFFF65F1083C5C492689CCA75EB0B0CA408C7DF8358C9C80C9557837401B81ECE7438F82CC1AD80C02439627912784208C0B4040A088581E04EC0E09810EBC4631BAC6218BF29742DA90260AC561F8A1807EDB52971D4709814A41480DA92EB32265F75B651608CDEF91D164C7A38EC44D27454066E8870E9F3743B8BE85D065E8B47560CD94710B63A5E44C8D95E105342149066EB63598959105F154F16AAE488B00C4C3027D1C1C2EBAA0D0C02AD'
	$sHexData &= '92385F0C383B2CDCF4E653A026F48D5399CB01983AB43F15F0605F119FF18BFAA85F51103A688F7F8670FB0AE934CE5E085DF9AFB45E492747108D540002529A861D2E0625891F2810104444A8691414A12878F4886D32F0BCAD532147113F555EB958292729F40E04E2A542081A12F7401EA41A2424EBBC1B3C38008F52151B1F1029C4815E0508E7E3FD0B1BFF5603C2575383F82831409983E20786D258B122C1F8103CC0FEC3687FCD06FF02C98D341F0856B04A2078688816DC55143C28A6AC6770FB10F85C1EB1E064F6EE03C320F8B2CF06C63B9875F841F78019CE68464C735D0CBED79AA06B1E2BB7F01AC6F84DF36376C06E7FF079F85C5705B37690077D1094F4620A90660223FC6808A0F482210EEF054520184FB4A64506105A8B57D5990A6EDA082DFCD62F08C564622B5752211864DA632863075576400E2920520D0C535DB19A0CA59F0A1942CB6AFF304DC36D62FA70F8A5D6D1F2C80F078BC399E0B27E8A24D1FE8AC1F42F2268768D14F68D7C972096A819C46AC883EFBC5708972314ECCCDA524EB6EF85ABD70760BE45C86055CC91130E2467E5ECC0E92074F95DD0368A6121EA7F9106C2AA839F7F904FFA1B7CDE2BDFFBEBB612163A2A7877FA7C172C35152647D6B01802373E527FD6959F821B55A9BEF11C9312D0534280BC20923E750F8422E60B22423142A8C8'
	$sHexData &= '536CA15D611557ED4AEC48C506C552FA4DD6C41306FFD74F4E2D8D73448B469858F0B8B642424EC8E0510CE4A6E0D8FC67D956DC8D46BC89163B75D950F0BA23AA16CCF7890E52B699D9EDA849EC6E20EB66945A40E242B819EDC135BF50A1102BB48C50E61146005FE4176938ECE0BA1AFEF2396975D18D6C3794C4376355B830D837004ABB63A0C58D3DD2AA13C44AECDEFE5F602A69C0BE35101B2E6465FBA8758A7B07D0057C117E3BAC72DE80DA19CF3A8B300E7EDE3646809658848700D3160C66890137FF0F83C01453897914F77F828960A0D23A13108945A45148E508EBA8104C030C3EBF82D4F61370009CF05B24BCA8BD28F32BCFF717F18AF52B9B8BC1460A101DD2FE8A0FB211784710B81CFA6A958557B0F13BC17D667E6CC6776B9540461C402CC1E304035DB737F450AAEB06FE0E8D3C13A9E7EBFE8BCB3B7D0C7503E6BEFB0C7F03103A4F4ECF3BF07D05037D0BA95D617054D62BD06C526E42D3BB51C4FB75C9DC487630102BD81AA97FABA1A600F24E9F11B83B4DBB9AB146516548041369D4A053033306F24182208A014F3C163942F453D38B1A2333405A45D0802B151D8F4302610D5F3B7038A1E370743F02D4727D58057E13086A04838D48DC515003F607BC0E014EB6341C938B133A1992A705103C817C434B30ECFE169E36023B750C7CB2BEA8AF40DBABA04576'
	$sHexData &= 'DB2A90DC280C78886A88464CF79AC022102D881433FCD8384C03960886ECBF620A7CB07547FF7F397083CEB8F07D4F536710851C8152240AEE8819352E7116C081131E917D16FF241EFC9A7CB35B71FF9A0F8D3490BCBE905FCA570447205FCCF22A02628C9F14A84C938C013D45D3D10CA8DC5A32C84ACC2C1D5468E053B54668098E4304530F0F8052C04CA0EB55475C964EBCC0007BFD2BF780FDEE1116E187505733C01B0D94A2B45277811CC141FF88BF3893742563F7FA0050810562463212B8D0583CC05052096A6E2DD4140B83A50C851CA2F0C190A6565FC83CE4DD0AD0680D1736479446608D73694F57A41A1E13188EA214560B3D752ADCCA08A71066078075A72B2E4FA6D147FA7602410E0996E27C86758378610D0D427202A926CDC0C6703F0841A01271F772682AFF35DFFE585932653CB4215373906EE60E0C2C1E06A3217ED16C67199D07495200F662C1CB358AF08365D0EB17C45303023418536DB8BBCF765F56A10659A121F10D6BDA40E0831EFF65D4840229E56B2D412C634C0B505C2E97CB54585C6064EF5B7CB9686C3B0D6D7502F3C3E98C3DF072F97E8B22000B080C186AB069EF05CC3448D60396358020700277A20D94FFD63EF00FDB9B284E750821D459EB64F73283EF025050599B5B4F15018CA1CA1CE0AEA0AEF9EE666A919850674A5C33584D4E3CDC4A'
	$sHexData &= '20A371186320D212E02C673B7B4AFBFDE68B445BA1C3D3445901A2BB0CF39A3D4EBC0786ED98065948CC7B503971239A04B60262742556F706C783B2125A86366A0C577EB1E141D2043855CD93916F0744EB142C070E57CDECBCA9C7C1A321EC8A8041ED060363D036F08C8383C1CC8C675D5FF72B1A7605DA405EC3832600C7A7EA04D16897F7063DEFC70424169F59465EFC05B02779515153168C39450C75323905CA754D5FD4987E23BC18C90D1BDC0F0E0A309F5D8CBFC427082F644AFFD2227412286A0EAF7A0C64B365778E46F464E6698B59045542E8D70D52505DEB117E1BC443D0176808C3A09090EBF4B8DD7305E8A0E7AE0CA1501FBC341698025EB06A1F6E0654EBE7DCDD83397223380930C7053F51359DF9A6704E5959C47A3B2C4258DC7D0822FC31485989D532FEEEAEE059395D9C08CCE190391D2A18741C68B1A2D3E70D50058C490D6C1056A0C141DB7D2CFFF5A7ACA630E4A05DFC50F42E78F40D595971430E44888458B3C750428803C0CEBFE0BBA1100883EB043B46724E833B00749C75FF8BF39DC8390374E9FF3371F8218903EFB2105540F89187368C4D98BD42F1F8394D10CE660874B7891204A70A694D8EA0AA20FAF8C87E75CC59832539938626F8036864FC2E505D42111E4E9CC9526AA11A4BA710CF68BAF29501A7530438766CE80AAA5607AB70ACFC6D'
	$sHexData &= '987D9834822B882EE15702CDF200A8D00583FE02752E574829C7287094F253FFD0890D287076C0AD58876B742299AC28258025CC7E78F7754C4FFE017524BE2057502757D624DE3BE45FA55C85DD89EE22A6B6D50375439F0A1B76F6680321D1742E5925FF24609041B622DA62B7E178AA50514F5A592A2313BD6265E89E5EA7B5E011693E0101C1E4D630CA74C6D8923FC3A7854767D47B605807AEBC89DF1A00B659812F9F402CBAA0643C410C667C08D82B64066C288BE873102135DA70FC3B4317028580F3A9F7E2F1EC51E181EC28F65A74A83E7E7E7E6E0AA48915A0891D9C893598893D334FEB7E94668C31C00C0DB441CDD33CAD9028058C25882DF6092A3F849C8FB8C6ADAC0EC144E59504B08D4508BCDC29DAE02DF173F8BC01387D5DBF0F0CAC27A009FCC013A4CE260A16F9A889BD3F3E6074151CBFA3789AE90E7769F06A015BFC596A817CACC2FDB549681580833D41C40847FE143E3768B73784B988C98DCB3707A7E00BDCD8491B0EA1FF03B9D3027DDA0F56896C068D2BE04B06C583E131CADC89C4E21A4E2BCCF88BC168EEF0AB259043F06414F275FBC022A5D200FD2A5C70982096B007024274C3034A231790C3849D9AB014783AC02BFA0ADAFB78ACCCEBEDDEEE8B04000EE0C458FA6FE0813863736DE0740B845830EA76EDDCC3A108083749B74AF8E9AE6A0CBBB0'
	$sHexData &= '0CE858A23E80C61DA6104060EA48DB0B29660AF11901D5F04C018B043B859D0483951D75119D6069A30B33DA7FFB1EA1834D146E56B8DE08BE09E1F40F8242C6730F8B077AFC6CC3642CD8FE72F108A90C33384B10107C3914C0038BFFB84D5ACE129E05D80140E068782164413C03C11050F8DDF8BB5F2AEF33D2B90B013948180F94C28BC225B7A7806D7F5262C80FB766A1898541980A71F6140ED6A657AE081874AC37113FC4480C3BF936580803D974C9FE0B180A4283C0283BD672E8D4A11F1D3E222DFEB1D077481DA348AD60F801CE62D2B3F02F1359EA840A30F081FC735877F05F542E2D275033053C8AF13504A74024B8F7538F151B22E0011DC684476034CF8A81A4E381C8C2DEF3C3077C8F52137461EFC40BC0B87570B9BCB8A7A4FE97021C7B90F85357BF4EE640BBBB82EF28C37C350D85C374091ED42061ABA33BEB65A2F87FA66ECDB768163330176C33F023CD9C7C0F709C4EF07437C9AC5BA2331822CC07BE4F06EA467FB2EB1085F3210D1147D048DF3DF0E0100BF02718F7D60F1C5CBEA1520A6FB40BB0ACE2327BB9A8A4CC3FA09C2706873A43441FB662E216D02543542408048146C6F88B4AEC33C8F50B2057C441F1DCF0BF4584C02339C9B18C5DF83580C9B50B79BEECE061AC3DF19C185ABF6A931C0BFB4CAF9CC32381DC36909316921C7ED8D8924B26E4D8'
	$sHexData &= 'D48C492E39164BE8E42739E4838DFF023CF4371072802125AD3768632F3E4A0A83E9483F1C5732B5E592F0949FD8BDB6484E3820F7384C9D2A01899F88E9B9A44D8E57C46AFC78F4494D0470E60F8406D0A621FE0CF0FE838AEDFB1BDB48C31918F9F186B4D5702170A39EB441B0BC2AAB39403F16E40032D0396CEC755CBB90F0F04D98DF38C0F03975D8A50CAF39C480A845AA9FF92C974AA6D8723A243F84E54472EDDC7FE43B442497A986FF0840E0F820B8296286BC2101AA811C3B88AF1A3E03D13BB4FFCC977C6EAC6D68E2408A72657C526072E8085E96FCADE42487E43C145C4D0B1A7F5885C73C879C8E75373F9DECCCA8980C203C7F8FD8726DE845CC3F3D2CC9955C248CF838C84171C87F0C7508480E103D7FC35F3DF48A2D07D8F53E4C3F0D938B64A4FC9FC8C4306D0FB8BFB0862896033F387FF021312DD753649F8CAAB58969F45B6107C884141ED0C4C4C88BC4208DC10ED81DF2541B3CC0226BF05D3271453FE0E4DA007B502540245FCCD9B876C8C13EC0F87350CB0D4E293F71D5A407EC0E05C58A60428B8CDAAA1860FFC05F73D2740539B89ECC025DD7820ECCFD079FA5C093A45B9F92BE8E250FA5DD90075F9758049C8E8427CF14AFCCC84DC42E90824521A21041829336BA23BEB4AC4F4B06E4B9AC6018255332C9FCFE00AC4DC920CFACA8B0F498E46AC2941F'
	$sHexData &= '41ECC7B0B93C2F0DEC4D9C79540B72C925383C787930B9AE19EC19E8C9E97229859F41CC39A4792CFFF0FDA05840924B2E2C409992B62019FEEC24E41CCB034220DF3B2CB41C6038BC5F424C900336C03FD05F781438408EC0A4BF13966B07F045E0F579D0805743383F43902B0A11405F7F25570BC943C45FF0142D53D1F8DFB8740894142D93D442B409CB09E404FF443C341CB0A33FC85B68DF26A462A71F997CAEBC90232407A85F449C6462EBC2842F5BC8904B06DA5FF0F45C055A06F05D453CA8C091CC0F457848058E6E7FA00805D7772C29911411EC2BEAC0C5569BE81F901C209AC8DF4614646253B00FA8BF88DAD08C0370DFEC3B0AD680E5C8DFFF91C3925007851E94B014AFEB24FD262627181547405E0D3A0D5CE9FB476C2D0F06185FD0BF8B90B440D7D2AF82A0A958C8A66E70FEA4949A3913FA011C8D851545C19290ED60C2E950F8A5688202436CE747585E0DAB1B5F85680546D2CC44B84087C62053C35F488BA051D890E383E0B8132486D15D2AA51DF707431EF20AB198FB54FA23B58CC880FB874063E82935C30FB02037231B15BCE11BC621CD0112FE8CA27DE0BD195820EF1801BC61590BAE6D04E9382C1B36C592DCCF8475A0491CD9189B301573246DF21D6FEE00FB5A68C924EDE574FC9A8CFC10F83DC09B36C01548579F80C91024B9C3E0F702D8DB4963C9'
	$sHexData &= '5B430C6906E404FB88B749DA0A61D6C49C2514CD8037056B5863A2505005930593FBC284FCF2280D9936206045E0A44D56A415D458FAC8F49203D910C49CFCD85185A56164533C3B39813CA6194CFA4A0017728634A1C3C22B79814532AD2C4A34F2AAC6A93FAC4CD54A60B11C209F3EA6AC8C7D92F0243D5B0C6CDE6863205C1C80E91625F1C71F5F844A2013728090AC0AC0D7D36A346F3281780DCA77ECE3BAB732E077B67D3E85A0B4DE708B2069408EBF303CE764E47296E0E0E0E024EF0EA02CD5F7C088422D58C11B161F1D20F548E1482EDF28F8AE82A0CB236160DFA092414E46A0A0A05FA2826EA0F780DF741D9CD61E71748FE03F1E6035DB1A743F41582BA04BC83521042602ACAD835D24960C7B50C31F84D86E23C4A7737F212FA6B49F19D5FA252D18D7E30818287D4956F0CB2A404969E05B67B63466C11F720FB809674B2393F89E821B0533EE1348B702F753F50D4D45D02017059792F87FB505EF2407A0AA6C217F020BC716BF92FF5E264981DFCFC24CBC19FBBE2270C2FBBC742736F3747DECBB0C21270A5DDAF37B489142170A2C4B53CC945E8911A59BCCD5120EF913D0D4A1F07499484964C3B920E61EE2B816FC9F53079EF6F9032261010008FA600E485841EDCEF306643801CC5EA7F93CCFB606EAF20A5F209EE7799E3040526A768EF39EE7799EB6C6B05EE0'
	$sHexData &= 'CFF334DFEE06F473DEC8AE9A3CCFF33C7C604C381A349FE6F304F472E4A05E8AF83CCFF376685A48D25FBCB31D393E5E9A6DF6786106B6799EE6F3E404621E3858D37C9EE77AAED40E6356E7693ECF96D8266464D2E7DDE7DD40659606BE0466D60684CFF3349FC80A674C8ED012DDE7799A6854AEEA2C698EF3349FE706B2F2406A8AD4F3799EE6166B5CA0E2206CA7F93C4F6092CC046D30DAE6049F5A7E6DAF08711874060EE779DE7DC0729606765C46CDE7799E3426180CFC71F43CCFF33CE6DCD4C6BCE7799EF798368C7C6E649ACFF3CEAE6F06C8E2007020799EE779405C7C9EA8E7799EE7B2C2CCD6E09EF6799EECF6FE2274A6121CF33CCFF3062A3A44505ACFF33EBA01C8BE060614263A3CCFF33C465A6A7688FB3CCFF398A0AEBCD8607F693EAF392C6E100642E06DD49EE779EF601676068EA6B2C63C4FF379E2006F1E3A50363CCFF3627690F6C7E7D96CFE07011C6719F01A2050BCF73C9BB01B3040A03E2006400390D3F450B9FE62E2FD66EF012D80561017739673F830144930287E810A160061006C006916EAED7EA267F41674C2126F00720A2A6D7DDD7B7E2267006566271F748D463BAF2DEF12697265266F0A6C2AF6DE3B5A6473722E6D13637FDEFB75235F740E73006366651A6EF3FBDE7B640E7656639B63526E0074106C09DE785753739BBD37BB775336753A'
	$sHexData &= '6F702A78126CF3BDABE0BA00F77336615B68CEFBEEBAFB6732D637701E6358300C0C83627770F7E029CE61653B726A2A51B0DA7BF776476F4B87F7A6DD286A1F2922624E749B0B96ABB59600EF54FFDE005B7B7E741F7F523273BED7CEF7C26C5F535A69A667976F616A632F0F4C87DF6DF3E038EE27A6476EB7781D1E9EDF55326B006E776F0B7297FF297A4F343E7D005B005D003C7B1A8E7B277C062E82AF5C07BF7FEFF20043004106480045005FFE4D00474E45AFBBDB7B26522B265600320A572F67E0C1BDE74400454F0F2E175FDBFBBEEE431B000B53AB4C42441F0D0EBE77A68E001763C3EAD54164F51F29CF0B3AF78C75E7219A6FB36D27EDFB3E3A7F676A74232C3B7017DBA3B78E2F9E3DDF66533BEFC5BD0027006B6942652FE056F06007530300DE41C5E1F6421E4A4D9250BF0D334C1AAE5DA730C7C7ED62BA368F3D276313B137ECE374C7DF59064E3F75E718DE70363D3F3A072D561D3BACA756772C7B7A6BF7E022CF6E3E375B70BA8377A6265FAE697ED3DA3EAE76163F68AF37B6A6B5EB5F739F5F87777B7039626BCF6457282AB6A71DBC7582662FB6737A67C5D10D2B0BD7618797549CBBA57023720079B30106EFBB4EBB754F820F027C3A7664576D3F64BF1D41543112789F2A1830B335035E014983C0D11560533344824CF17EEF3DCF336632125BCE38003612'
	$sHexData &= '58EEDE6BF7265D0E2A3006353E31120AAFDE9DF71620023A0E8661CB564EBE56175BEA026F2DB83B78FA1F52964E4A2EA216F7FD58EC470E4F6F1F4336011E76BC55A353B225E7076F4DEF7C27771E25BE788BC1357B0B3F4D8BBE3B799C6B374C2B264B4B868DE34343932F37D1568042BF53509A17DB8E464F531347A275752F1E2E72A632136C02A1F9FFFF2F657450726F6365737344504941776172654707159F1ACB62075C4B5F8A5D71051F5F4B00632FF8FFB600696E76616C6964206D61702F73263C543E20F6ED63FF6974657261746F7237737472146720706F73349BDD36C3696F6A1F446F206C1CDE766598D9B4BB635C7425006E4A236BCBDF943862616420636173DF7BC7D4694501315EF0061016739F76C9E0601402FD4677369BEDBD4F309DC01E24061E1679507B70F33CDF3D180E36FD120C06B6F79D4F7C7E2070FCFA06F47E2E6CC4E44974802660CFF33D33F12E4E9E06878090A09BF68D6CAE8A108ADE8B10AF8E60CEF669DD8F07F08C9090A07E2DC8A51675F095743F18815D9BFD24803FFF017F0F000718F4676701E03F0D6F406F12833A19487EF3FF24400AD7233C0000A040A09999B93F48156E26AD280632D04D5E3FF1FFFF534453D2E8AF3DF62D8D409FA89E7F797CD0946A443A767174C75C4400615C5F426AB65F1245A215ED432B2B0A4968676570'
	$sHexData &= '5663A6B0DBF668444C4C2858502652656C20903791DADF2D357838362E706462FFEF27C63E66941D87A406AC061B6E633F01FF39F0419E69C767608F28DCF36CB69105EC062E042F50E42590C3CEC8AF280465D8BB8D874F6886342702D5867DE72E44A6311047019795F70AB66F4771D0268003F32C1879460A904E30C4BC292918C86C6E10916113EC718C08D7D8AF0924CF53B0E8A770A40310690F21782F2C75577E3CB69221E4500036E76217C8715088B7986B1EBA926CE03757A92D0F677F3074EEFC37B1C8C5BE3CEE1802375027C5F3BCD9031F1006A820589FEBB80472FCAE0FB4050B796E4C576FF2B4417474BF66848CEC60250300873707443F1297097F2E13F29990807230F43101F214BD0406447F7224002D6983E0312C6F663C1C4A5B10E7D7077BDEC5DE706674FF8406C49047BB900B7618FFACBC672298405E700174FFE3B990EDDC263244F808EC79DE890F321806B02860772DD8122CC79F9FD43E2EE46E54B8E76FC8D306C17C9F668C940E83950C005F0FB4ED040987B9139A49170678799ACFF3A9EA2A185078D37C9EE7A0CCF01C1948BCFB3CCFA2C0E8191A260675EF3ECFF395B5D52D1B8606809EE6F33CA8DC081C35557C9EE7797595C9F0101DF73CCFD330689CCAF8269ACFF3CE2B1E0698DB181FAD9EE6D37C5520E2622188B8E779F779E815228E0650'
	$sHexData &= '7ACFD37C9EAAF728236098C89FA7F93CF820246BA80825F3799E776E0676A8E04926E7799A4FA6562992B5D5C53FC36A01B7010740220593193F2B024C383418651787D8B693FECC0F1F2EEAB78764603FD00101C50F91B0F7017B06A58F010401062A8F03D24232D43E87D8B1A6F7607F061B062EBF92DB04C7770C474C3754163CBCBC4002350806ECA794DCDA210726246F709C9CE4E4D1C04001F05C21E42437BF7035782E924B6E57A0A4D0F6B628DE0FD8014F2FE240736E93AB0FCEF602FE02C7F8043B1BE9DF05D0273608DF2127D8ED08100F18DF082239C92939CCC736683640D8DA0039780FAF546E819DBCA257AFC0025820178CB08FD02A58C9AB1F3708079DE494BC8005374437342EB920B8275407C08C20348440EF6AB8D88927C1E71F63FFCA24437037D86F03E813C8C89104B802F809B44172921038445640DF5B2497703F0E8D01308C74239977039C3F2527E404C362C2DA3DC9293938F8390838D80227B920982718E7E0396424D74E240A1057901B24975C40BC216C27881E0A7077E7475F7692331CF7DF3A0039E8274E2017ED10BF90E6579C9CE44A603A4870E4047291B0B6E592934BAEC0A8D0949C402ED0B362A0B89DE43B203B08273057C9EC29B0FF0B0D010F19842A4EC7228F3B6887DB89E474373BAC0B70DFB04846BA786702D8928B763ACF3C0C57D0'
	$sHexData &= '68C989E4380C00401E45A2FF0F0C309DE4949CF7583CA83C9013C845BB27B8BF50F16E979C92DC3D08F02718E40472D1BF70F04EC9C949AE683D5078C5B520179077B00FC923C245B0C73DEC218FE401C0743E28C989D8ED3E182738AFBDA82127975C8070902EB99213BAB3D8C86E27184EE8670D500F5841721A4E60DF3F202219A9B45690015C9291CAEDC0BFC502884EB44B2EF0BC570E20382E924BE860B7C945F21B02BF401490D9ED22394048C00FCB5F4E2423DD0ED3CF03740F107B58B44BB0577F0EE4C8F689A71E5D02690048773272750480A1B707939313C9DC1030E0E4ECEC91E81005030F11011F1D059297EE642849C7084138DD4E662FBD0077809AC7A6C94877BB2FB20FD6A7069CF2BD9113113D11270E1A025C3B193926003256A7F0B453719180274244575C24975CB070E080502596DF14E091FC2F121042C8D724A7AE13E3F9774314C989D8ED43042724AFBEB20C27975C6C5C7C2F826BB71312700F7567B4C72E39698366A0A5E89639B448971279E975D21EE3EBEF04441C24DF642213576032D29D5D50211358BF028C8B76C94590C057928BE492C0ECF02A4ED07145187F1379CF04B7939C45544544276407662FB0131450271E586F14CFF3D41E8FDF11BA0103049391CA24C4059C994A2C90D4FF46E4E4762214D00FD8E0E5ECE4E4E8F0F8B715003941C3'
	$sHexData &= '4807389F7678753BC92946A4469427B44E6638D14715509F1515182E0811DF576BAF48F7489E736400471CEF06920986DBEC272CAF15B1E044F2476415D0473991740B4767901610E4C8916D0E1B0123022B03C948377236043EA706C4DC6E271780881F88A790E8D6749C9BAF483F4E4E4E10471741D0F3E4C89F4D360F170B04160532F2647623062F3663006EE4E4E4762B3F939EA9C9C9EDE47D881FB8C3E4ED64E4CE12DBE60FF11579727264FC1807121D1923478E1C281A301B3D1C5FC8C9C9911D811C1B20723272E4892194229CA73272F676B23FBD26DFC827D5F9F2E476EB0FE0F62B19011A478E3C190C1B2E262F3104F2A191233C0047EE333A6983A44866808A2A2CDA29274A2467CF975C24DD5758D08492030B2123F85F4A4EC916266E61A26F9E925372C66DAE71EE60018502682677D42A2880013EA20A20A501236024F0FF7F2E3F4156747970655F696E666F4040003FE4B6BF36B119BF44FF014E3F246261DB5F40FD7369635F6F66DA65616D405F5755266368326BEFED61725F1E61697473204034648C0BD5FE01E66F7F66696C656275667D11DB022CF7F577D7AE763B5F496E6240483769305FB20A6EC0B66535E746576303920369E70B86140CC7CF5FA81D51278F5C616CA57523EA6C6F63BC2932B3AAE296ACA79A7B28B851C12F862959FCAF059F47646970'
	$sHexData &= '6C7573427D16D45AB617EF470644D9B5C13B3742E2D45F6CCDE439776578636570096EDD56BAF7325B5345AD8207D7AE5FECE0AD869737BB39A8020880070022888D01661640C819E720A008DAFF37FFFF2020010001546C7353657456616C75654C6F61644C69627261727900F1DFB7571A536C65657028656176654372ED60352BCE634A5663222A4785B5CA6A68468E4174987AF7FF7E6275746573572754656D70506174681AAC4C00F1BEC1024572726F7237917C6BBFB8416464725673AA456E746572596BF7B7A46E64436C6F7365600C62324D9EEF3D7BBB7A795374A475734578414E6578F7C6446D6134B3650C8E8DB01E1C4C034D2E78C541A5B6DD76DC12496E504E697A651808DE9693AD62464C09155C60168653790854696DD67381687FCE3D1331437572466E063BB0176549642954680A94412B30742751AE7E5006668CECC0E0766D616E52436F75F31673E40F67661B70676716827E6663DB616DED556E684A64CA6445F9E646418ABD5F3317C7ACA10D802E6D52613C234915027B70EB086B80044661DA6076675C8EA2FE37292CDC76413A636F647E6F69395F3BB029D61DD569609B1A59055556577846C0530ECF2E50AC404F626A74284304C3E1E3DD564469721A6FA1A74B31F9B1BDF147AD060E7E84627273AF0748DF798AA6D891995772C4006DB307DB2D9974'
	$sHexData &= 'A850444327E0E0CA4AC55D6FCBC6912D671D1B77D3B6E77D56426C2253635DDCCE3B7B59E7A8224C594772A4607B594168F87327E5D1D6653B9AC0F253AA6E3035EA3A08C65EFE612E788EED27267639486569676827E9D9C1D7DCD364773D520A49271E4C4D1F3C531A6FB8CC4D0B47962D182BF33A7FFE162D38162BE1956973704382E01D9A9B65E805F65FB033484249544D415039C70832B7253B4F398D0C0546DBD46C6139414CACE062103DBE6E237C4CF03119576964BCA15069784B6BB337E04F666673149D552FB1AEAF1C721E7075592B4F47A746CB7448E7C03F5F7CB446824F24A2E0405FD65613590154F272E7BFFB1FAD30FF51414558585A554241444F464662CBDE672B335F4A4226691442DB2AB67EF86376741044CB30B69E873256303A6C303AB86232414C4270CE5B1F4ECBD25B59413F417713AC16B85F53EA48744A4031090D009A3ECF586A748A5FF8FB30A772D35D585042443D3F315FBE47607059563D40582F306082ED48489F0E90AAD10ED86E5F65D49F4285CB1E055075C43C7180870487455F4EEF84678A6F5BE0DBA70D4B43D413084CDD4156F169A684DA63D61CC136468A7065D3505F2041835D5BA040E848482D220A002D4A376B2D33747836424097703AB4190EA1B8E11EBE88E087C1FDCB4F73667840038207D2B1175409A1226C2A497FAE74DB'
	$sHexData &= '7A15C18A8B424514AC188742B7F10F05C3D002776905D9406C3140E4E5310445816289D7D5684084601805B11D4085C954AC501CAFC7D5D274505C57D66733A2AE2E418530EA5AC92FEE80E4634722B331300C4321550B668AAB981A866835415671403B5C033F367D305132515B7E4D407F4B517251724A49DA68517248FE818A01D88950360EF18BC3EA59201A3285AD60A7501CAB2E20012BA8BA404031C146D1E0DE48A2A1EB53313D1D696C65914D4D35067CF04142D12979C2F0027437944801D8D301ED5C8150B30A044D18227B780B14C96FC94D8F67651A1A6CC841625477378DB0910347687C1279B8C0828263157B15A662C7CB8555AD6C94A80D4C7F46770161C45540CD78801302A09F53414946D45E61D86C5666616348402BDB9012F622340A8B8469667BADEA1756694888488244684EBCB3310C0A61C6599277EE31465B0C6E330F7B8976696E6DC70A377541223C6B1D60776179D8D776DBA46E6F746E76407FC37F0C0EEDC4DDB276A66C10186ED7471191622BF133400751E7626A5C1F424033495BDCECC600F4009821776373540EECB65A7E6D656D6370794E0E5F5F636608A100372EF9CD48022B74613680BF146CC744BA76B8376482355F0563266A4E749CD83D21DE199B7474F12E25B7AF3433ED5F7B7234EE6F6D6DB9D4DA4BE210722A64074AF60D6E3C6F6F'
	$sHexData &= '6B2B43707058635C064F25B8B75F2882846964726BE6ECD60CC11E19C3D0C18256376482756C6E66E7AC21F089489DC91861F7DD66DE8524CD64382727D9BFA775ABF978784672E2F378B8DFB4B6336973776469674CD63270E884D6D558015FAED82EF5DE2675693634E06363547072ECDC657A46663660746F77841A4C029E67197426E46D21303051A840880D68375E3022D0E99CD57768909FD0569D6355C27A4D37DD6CB1B4E733B151EC2E1EDB8A3DEF7874760C06050ED4663A0277C26E2C60A42A184CBAF79E922D110E660CF6111916E94E86656B5B72987CCFFB374573F9164A199C6C68BF26056BF77A3676C438CB4833A2EDD942BD415866E5BF8E8AB543F56677B665F3001E243A2E6DB45433D4529285545753DC4B66D0C4A8647EC013EBA90E7876F61C35C53A4A3156E270F5536966364D10A09645C59A15FC18756D43681A642173077A9641D3734A5097FF9650D38EAAE754F2DBBB5BFB2B513A489F9597BDEC5317195D100CAE436507DA49966CCEFF3FB3D14D6FF884580109230F132B30260D2406FEFFF6FF0D130A290D2717092B1E0610182D8C2215240A06231D060A1C080E2FFFF6FF1904501F1F081F1A8744150D0642160609060F3B451FFFC33EDB142048222105070105080C114A21091904FBFFFF209214172E340B060C12342C0F364E7D273435124B6AEC'
	$sHexData &= 'DF6E7F1F1B0C0A442A102E060E11002F321F0C10311259E505FFFFFF200605052608081D35210C3012327B4405151E0A0F2C0E000FC17DB3ED250F27EE290E4694240C0F1A081634DB070B6C5E1B6C641B08262A121CCE2C142F7AE6EF22611E2413204B1BC01786190C2306F6CDFEED301011B03C120514090A08075A16061C0F30518AFBBF7577A33C131A8910680C063A2A0F1D090F2A0E5525E06EB6E00F9E693D8C2C38101598EC0BFEDB9076090B05090520050B16720D1D0E411764E017FCDD5241101D3C092C082CB60D359E121712FF3FFCE00222263151121BAC1E32223BC34B08F05C012CDEAEE1EC7312E93458500802004F7F0E40ED6FF8DF30570126091D1610101E0B78520E096C0A0739285D78F6F6ED01145A1C1D122BB8761854016A6A6AF40AFFB7172C74380B823CC80D125011210D07090EF06A02F0FF8561C1FF010F59090932909F275484A33E74528A0666E6080F5ECA13105882BFE19F2546120A510B7616BEF0F100135008FDEF03FB670B4E181754211A1100364B35220C844AFBDBDDBC1A514A0D0C49351D0F345622F011024419DFFEFFF76E941A831A13F04701B36AAF866F1CF03901A48AF08B0141B8E11DFE6511BB1162833A76962E7C1531238AC386AFBA0D1F060766B80C1892B4CBEED6D90E1412DF0B0C112C11015911FBFFBDFD1F393125E83EFA'
	$sHexData &= '2C125E2BF434250C2D0F614D57391D1900FFFF177C6F3D212B437A07562C0A1D490C521F2A1B12200D0B50131C361DFF57FC104A4C0E190DB42A321625E3541240071511B22FA4F87F6B362E042A0C3E853D6BAC401B14A10BBC87E8EC5F5CA7570C14102207A00DAC559B571EFEC2A3E3EB5E12140A37EC1D2D1D40259EA4E32BA20F1026240868462B1007066DF1C13EA8070C0A1E07CA27EB78DB80071764091B10283A07E214A38AB3B3099E0A3642520FC6BF6D7F7B860C642BD0063B6A121D2E1713204F0E4D72F08517E0FF1562F07C01DD5554F01A011040066430421BFEFF07FF1E71540D6B1C2E19140938170F226B3B06461B2A213539BE4FA28B3619240017438A182C10FAEF78D3FF7B116B40EE06989E0F6DACA63A7B25FC1B379B16DE60D8FE5A6E1B164C5E1964301C08E61E130B0E6235621BD232349FEEC745EFF8143F0C37BC3A01053E9EF0190178FB1B366DCF320E390EE24620090D362739226E28FAFF26FA0D8D2C4B063506F08F018D871BFA2182C38B2E76250150F606592F1438E69DFDEDFFA75036074E0E251C061514004A0F794FD406D133A374FFFF26C3344624056C1D282D067A067D06690668065D0DF1B705555C4A9A0619A60601069F065558CBFB515B04761CB21494AF12196A1421FDE98EAB1E581BEC0517076E1C31EC125D7F7E178C5E3B1E0A70'
	$sHexData &= '3F1872472A3D2418AEAF3ABA2D213B0C62342237429EBAB783DF6A1F103C414A5829C44B021F161534FF3CCFF3063C2F352B502115F031BCFD57FD0112AC58353DE26A330C13128D523E3C145C2D0C54D315AE17CC33011E1182C41F290DF870F0FB7614081998160A96C88806F0F4007AAC6FBAFE6F080C6315B1BA8C51152A6A09121170515D0B8EF674532F24234280066F986383ADFD056A125606DC50F0A6EC1AA60956D8FFBFFDB11D1337285B4083F06D01DBBAABAC91987F0DB312BE0F74B4DBFF42285712F07E020FAE93323C0135168E7C8C1B9DFE7B87A52C0FCC272411091082A0012FC08E77100607F813154E180A3B14C35D7FD13E0E1C05160C210E0D05D0111815A2012F32511828467606141818FF45B6AB5811A228074282111D0D18532D482F1C9119E91C500107088430D8AA5305406605100E022BFE5FB4ACDE1D4D05115E610F052105BE054A1406E4EC29BE67600D23525829047D61FB6FF8661716D4164140262828161624043624DAB7B6BD8A161E32112A2D20002118212333251E2C2395FF7FFB1A3427084214162E3836303A463A240D0C382334216BFF2B3730191E2D3030231B202A303723482E16B5FFF67F324E3A601E5325426C5D293A723BF063013B230C23940AD050F03DF81B06240CA6239E15C143FB2E6B12101923151B11EC99FF37450C7C640F'
	$sHexData &= '07190E1811071E08070B0A9B6E76981922A43A0E081B0B800A182A7EDA5578F05204010C2804080418088F36B4DB1D18011423A004D6DEBBED7C04105218100A671321A5B5C5C62B170C04111B1BBB97153308251D15C181D116599A27EDDAB7AE20086A201C2014020A080CCCCDECB320024E140C080D30155C04D63DFB0B48200F47081BDB4DEC351618142009189D2C20A516F6BDDA451420AB30C7F76D6F7BD95B382021010F3776B6B6110DB514388B859C4D30D808997B10040D60DBE7C6A3291610102BB33F6C14FB0C240801DFF06C2528401150FBFF3C3C1C1C383C44544450241C2016000FB785FF3A4C0105004F76F475E0000277B635F0210B010A31007827D0111665FBEE9E10300E1217021E0500ECBDE7B10F00D004133C40695F3B4BBB040F0D407434823AB05A4682AC6B088C29A00EF4A979DBD101B02E08135E331C40DA296EE9D03C3E4006A6906D305EFC022E293879B076AEB71E206F89377BE88C461DE22404602E72643E4736A752A6C2450046240177EF5BFE4FEC661EEE402E4D3497FFE4DB49806A4FB61E3AC02E727372637CBB3D38D7A04F6E9FD61EBE9ECD8D8772654A4E981BB03BF9E4791472E61E4042A249C2BF3327CCD40F3BDE0100804A9224FF0000000000807C2408010F85D701000060BE007001108DBE00A0FEFF5783CDFFEB0D9090908A0646'
	$sHexData &= '88074701DB75078B1E83EEFC11DB72EDB80100000001DB75078B1E83EEFC11DB11C001DB730B75288B1E83EEFC11DB721F4801DB75078B1E83EEFC11DB11C0EBD401DB75078B1E83EEFC11DB11C9EB5231C983E8037211C1E0088A064683F0FF7475D1F889C5EB0B01DB75078B1E83EEFC11DB72CC4101DB75078B1E83EEFC11DB72BE01DB75078B1E83EEFC11DB11C901DB73EF75098B1E83EEFC11DB73E483C10281FD00FBFFFF83D1028D142F83FDFC760E8A02428807474975F7E942FFFFFF8B0283C204890783C70483E90477F101CFE92CFFFFFF5E89F7B900200100EB238A07472CE83C01771A83F904721A8B072C01750F86C4C1C01086C429F801F083E904AB83E90173D88DBE00C001008B0709C0743C8B5F048D8430E403020001F35083C708FF9680040200958A074708C074DC89F95748F2AE55FF968404020009C07407890383C304EBE16131C0C20C0083C7048D5EFC31C08A074709C074223CEF771101C38B0386C4C1C01086C401F08903EBE2240FC1E010668B0783C702EBE28BAE880402008DBE00F0FFFFBB0010000050546A045357FFD58D872702000080207F8060287F585054505357FFD558618D4424806A0039C475FA83EC80E9D905FFFF004800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '0000000000000000000000000018800110D04201104D00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001001000000018000080000000'
	$sHexData &= '0000000000814152A90000010001000000300000800000000000000000814152A90000010009040000480000005C10020088030000000000000000000060A00100880334000000560053005F00560045005200530049004F004E005F0049004E0046004F0000000000BD04EFFE000001000A00E90702000F0002000300000000003F000000000000000400040002000000000000000000000000000000E8020000010053007400720069006E006700460069006C00650049006E0066006F000000C4020000010030003400300039003000340042003000000082002D000100460069006C0065004400650073006300720069007000740069006F006E000000000049006D0061006700650053006500610072006300680044004C004C00200058005000200078003800360020002D00200049006D00610067006500200053006500610072006300680020004C00690062007200610072007900000000004C0016000100500072006F0064007500630074004E0061006D0065000000000049006D0061006700650053006500610072006300680044004C004C002000580050002000780038003600000054001A00010049006E007400650072006E0061006C004E0061006D006500000049006D0061006700650053006500610072006300680044004C004C005F00580050005F007800380036002E'
	$sHexData &= '0064006C006C0000005C001A0001004F0072006900670069006E0061006C00460069006C0065006E0061006D006500000049006D0061006700650053006500610072006300680044004C004C005F00580050005F007800380036002E0064006C006C00000034000A00010043006F006D00700061006E0079004E0061006D00650000000000540052004F004E0047002E00500052004F000000340008000100500072006F006400750063007400560065007200730069006F006E00000033002E0032002E0030002E00300000003A000D000100460069006C006500560065007200730069006F006E000000000032003000320035002E00310030002E00310035002E003200000000008800320001004C006500670061006C0043006F0070007900720069006700680074000000A9002000440061006F002000560061006E002000540072006F006E00670020002D002000540052004F004E0047002E00500052004F002E00200041006C006C0020007200690067006800740073002000720065007300650072007600650064002E000000440000000100560061007200460069006C00650049006E0066006F00000000002400040000005400720061006E0073006C006100740069006F006E00000000000904B004000000000000000000000000A8140200701402000000000000000000000000'
	$sHexData &= '00B214020078140200000000000000000000000000BE14020080140200000000000000000000000000CB14020090140200000000000000000000000000D814020098140200000000000000000000000000E5140200A01402000000000000000000000000000000000000000000F014020000000000F81402000000000012150200021502002015020000000000301502000000000044150200000000004A1502000000000047444933322E646C6C00676469706C75732E646C6C004B45524E454C33322E444C4C004D535643503130302E646C6C004D535643523130302E646C6C005553455233322E646C6C000000426974426C7400004764697046726565000047657450726F634164647265737300004C6F61644C6962726172794100005669727475616C50726F7465637400003F5F4241444F4646407374644040335F4A4200006672656500004765744443000000000000004E76F46800000000E0150200010000000A0000000A0000007C150200A4150200CC150200807B0000807F0000908100006084000050840000D07C000080790000707A0000207E000000800000FA1502000616020020160200371602004E160200651602007916020090160200AA160200BE1602000000010002000300040005000600070008000900496D616765536561726368444C4C5F58505F7838362E64'
	$sHexData &= '6C6C00496D61676553656172636800496D6167655365617263685F4361707475726553637265656E00496D6167655365617263685F436C656172436163686500496D6167655365617263685F476574537973496E666F00496D6167655365617263685F47657456657273696F6E00496D6167655365617263685F496E496D61676500496D6167655365617263685F4D6F757365436C69636B00496D6167655365617263685F4D6F757365436C69636B57696E00496D6167655365617263685F684269746D617000496D6167655365617263685F684269746D61704C6F616400000000000200100000001D3A343C383C000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	$sHexData &= '000000000000000000000000000000000000000000'
	Return $sHexData
EndFunc   ;==>__ImgSearch_GetEmbeddedDLL_XP_x86

; ===============================================================================================================================
; AUTO-INITIALIZATION
; ===============================================================================================================================

_ImageSearch_Startup()
