# ImageSearch UDF - Image Recognition Library

## 📖 Overview

**ImageSearchDLL** is an image recognition library for AutoIt, utilizing advanced SIMD optimizations (AVX512/AVX2/SSE2) for fast pixel-perfect and fuzzy image matching.

### Key Features

- ⚡ **Ultra-Fast Performance**: SIMD-accelerated algorithms (AVX512/AVX2/SSE2)
- 🎯 **Multiple Search Modes**: Screen capture, image-in-image, HBITMAP support
- 🔍 **Fuzzy Matching**: Tolerance-based color matching (0-255)
- 🖼️ **Multi-Scale Search**: Scale from 0.1x to 5.0x with custom steps
- 🎨 **Transparency Support**: Automatic alpha channel detection
- 💾 **Smart Caching**: Location and bitmap caching for repeated searches
- 🖥️ **Multi-Monitor**: Full support for multi-monitor setups
- 🔄 **Thread-Safe**: Safe for concurrent operations
- 📊 **Detailed Debug Info**: Performance metrics and backend information

### System Requirements

- **OS**: Windows XP SP3+ to Windows 11 (x86/x64)
- **AutoIt**: v3.3.16.1 or higher
- **RAM**: 1GB minimum, 4GB+ recommended
- **CPU**: Any x86/x64 processor (SSE2+ for best performance)

---

## 🚀 Quick Start

### 1. Installation

1. Download the appropriate DLL for your system:
   - `ImageSearchDLL_x64.dll` (x64) - for 64-bit AutoIt
   - `ImageSearchDLL_x86.dll` (x86) - for 32-bit AutoIt
   - `ImageSearchDLL_xp_x64.dll` (x64) - for Windows XP compatibility
   - `ImageSearchDLL_xp_x86.dll` (x86) - for Windows XP compatibility
   (This is /MT static version; /MD is embedded in UDF)

2. Place the DLL in one of these locations:
   - Same folder as your script OR
   - `C:\Windows\System32` (x64)
   - `C:\Windows\SysWOW64` (x86 on x64 systems)

3. Include the UDF in your script:
```autoit
#include "ImageSearch_UDF.au3"
```

### 2. Basic Usage Example

```autoit
#include "ImageSearch_UDF.au3"

; Initialize the library
_ImageSearch_Startup()

; Simple search on entire screen
Local $aResult = _ImageSearch("image.png")
If $aResult[0][0] = 0 Then
    MsgBox(0, "Error", "Image not found!")
Else
    MsgBox(0, "Found", "Image found at: " & $aResult[1][1] & ", " & $aResult[1][2])
    MouseMove($aResult[1][1], $aResult[1][2])
EndIf

; Cleanup
_ImageSearch_Shutdown()
```

---

## 📚 Function Reference

### Startup/Shutdown Functions

#### _ImageSearch_Startup()
```autoit
_ImageSearch_Startup()
```
Initializes the ImageSearch library and loads the DLL.

**Returns:**
- Success: 1
- Failure: 0, sets @error (1: No DLL found, 2: DllOpen failed, 3: Architecture mismatch)

**Example:**
```autoit
_ImageSearch_Startup()
If @error Then Exit MsgBox(16, "Error", "Failed to load ImageSearchDLL")
```

#### _ImageSearch_Shutdown()
```autoit
_ImageSearch_Shutdown()
```
Closes the DLL and cleans up resources.

**Returns:** None

#### _ImageSearch_SetDllPath()
```autoit
_ImageSearch_SetDllPath($sPath)
```
Sets a custom DLL path (must be called before _ImageSearch_Startup).

**Parameters:**
- `$sPath` - Full path to the DLL file

**Returns:**
- Success: 1
- Failure: 0 (file not found)

### Core Search Functions

#### _ImageSearch()
```autoit
_ImageSearch($sImagePath, [$iTolerance = 10], [$iResults = 1], [$iCenterPos = 1], [$fMinScale = 1.0], [$fMaxScale = 1.0], [$fScaleStep = 0.1], [$iReturnDebug = 0])
```
Searches for an image on the entire screen.

**Parameters:**
- `$sImagePath` - Image file path(s), multiple separated by "|"
- `$iTolerance` - Color tolerance (0-255, default: 10)
- `$iResults` - Max results to return (1-1024, default: 1)
- `$iCenterPos` - Return center (1) or top-left (0) coordinates (default: 1)
- `$fMinScale` - Minimum scale factor (0.1-5.0, default: 1.0)
- `$fMaxScale` - Maximum scale factor (0.1-5.0, default: 1.0)
- `$fScaleStep` - Scale increment (0.01-1.0, default: 0.1)
- `$iReturnDebug` - Include debug info (1) or not (0, default: 0)

**Returns:**
- Success: 2D array [[matches], [index, x, y, width, height, ...]]
- Failure: Array with [0][0] = 0, sets @error

**Examples:**

1. **Simple search:**
```autoit
Local $aPos = _ImageSearch("button.png")
If $aPos[0][0] > 0 Then MouseClick("left", $aPos[1][1], $aPos[1][2])
```

2. **Fuzzy matching:**
```autoit
Local $aPos = _ImageSearch("image.png", 30)
```

3. **Find all matches:**
```autoit
Local $aMatches = _ImageSearch("coin.png", 10, 100)
For $i = 1 To $aMatches[0][0]
    ConsoleWrite("Match " & $i & ": " & $aMatches[$i][1] & ", " & $aMatches[$i][2] & @CRLF)
Next
```

4. **Multi-scale search:**
```autoit
Local $aResult = _ImageSearch("logo.png", 10, 1, 1, 0.8, 1.2, 0.1)
```

#### _ImageSearch_Area()
```autoit
_ImageSearch_Area($sImagePath, [$iLeft = 0], [$iTop = 0], [$iRight = 0], [$iBottom = 0], [$iTolerance = 10], [$iResults = 1], [$iCenterPos = 1], [$fMinScale = 1.0], [$fMaxScale = 1.0], [$fScaleStep = 0.1], [$iReturnDebug = 0])
```
Searches for an image in a specified region.

**Parameters:**
- `$sImagePath` - Image file path(s), multiple separated by "|"
- `$iLeft, $iTop` - Search region top-left (0 = full screen)
- `$iRight, $iBottom` - Search region bottom-right (0 = full screen)
- `$iTolerance` - Color tolerance (0-255, default: 10)
- `$iResults` - Max results to return (1-1024, default: 1)
- `$iCenterPos` - Return center (1) or top-left (0) coordinates (default: 1)
- `$fMinScale` - Minimum scale factor (0.1-5.0, default: 1.0)
- `$fMaxScale` - Maximum scale factor (0.1-5.0, default: 1.0)
- `$fScaleStep` - Scale increment (0.01-1.0, default: 0.1)
- `$iReturnDebug` - Include debug info (1) or not (0, default: 0)

**Returns:**
- Success: 2D array [[matches], [index, x, y, width, height, ...]]
- Failure: Array with [0][0] = 0, sets @error

**Example:**
```autoit
Local $aPos = _ImageSearch_Area("icon.png", 100, 100, 500, 400, 30)
```

#### _ImageSearch_InImage()
```autoit
_ImageSearch_InImage($sSourcePath, $sImagePath, [$iTolerance = 10], [$iResults = 1], [$iCenterPos = 1], [$fMinScale = 1.0], [$fMaxScale = 1.0], [$fScaleStep = 0.1], [$iReturnDebug = 0])
```
Searches for a target image inside a source image file (no screen capture).

**Parameters:**
- `$sSourcePath` - Path to source image
- `$sImagePath` - Path to target image(s), multiple separated by "|"
- Other parameters same as _ImageSearch()

**Returns:**
- Success: 2D array [[matches], [index, x, y, width, height, ...]]
- Failure: Array with [0][0] = 0, sets @error

**Example:**
```autoit
Local $aMatches = _ImageSearch_InImage("screenshot.png", "icon.png", 20, 10)
ConsoleWrite("Found " & $aMatches[0][0] & " matches" & @CRLF)
```

#### _ImageSearch_hBitmap()
```autoit
_ImageSearch_hBitmap($hSourceBitmap, $sImagePath, [$iLeft = 0], [$iTop = 0], [$iRight = 0], [$iBottom = 0], [$iTolerance = 10], [$iResults = 1], [$iCenterPos = 1], [$fMinScale = 1.0], [$fMaxScale = 1.0], [$fScaleStep = 0.1], [$iReturnDebug = 0])
```
Searches within HBITMAP handles (no file I/O).

**Parameters:**
- `$hSourceBitmap` - Source bitmap handle
- `$sImagePath` - Path to target image(s), multiple separated by "|"
- `$iLeft, $iTop, $iRight, $iBottom` - Optional search region within source
- Other parameters same as _ImageSearch()

**Returns:**
- Success: 2D array [[matches], [index, x, y, width, height, ...]]
- Failure: Array with [0][0] = 0, sets @error

**Example:**
```autoit
Local $hScreen = _ImageSearch_CaptureScreen(0, 0, @DesktopWidth, @DesktopHeight)
Local $aResult = _ImageSearch_hBitmap($hScreen, "icon.png", 0, 0, 0, 0, 15)
_WinAPI_DeleteObject($hScreen)
```

### Advanced Functions

#### _ImageSearch_CaptureScreen()
```autoit
_ImageSearch_CaptureScreen([$iLeft = 0], [$iTop = 0], [$iRight = 0], [$iBottom = 0])
```
Captures a screen region and returns an HBITMAP handle.

**Parameters:**
- `$iLeft, $iTop` - Top-left coordinates (0 = full screen)
- `$iRight, $iBottom` - Bottom-right coordinates (0 = full screen)

**Returns:**
- Success: HBITMAP handle
- Failure: 0, sets @error

**Example:**
```autoit
Local $hBitmap = _ImageSearch_CaptureScreen(100, 100, 500, 400)
If $hBitmap Then _WinAPI_DeleteObject($hBitmap) ; Cleanup
```

#### _ImageSearch_hBitmapLoad()
```autoit
_ImageSearch_hBitmapLoad($sImagePath)
```
Loads an image file as HBITMAP.

**Parameters:**
- `$sImagePath` - Path to image file

**Returns:**
- Success: HBITMAP handle
- Failure: 0, sets @error

**Example:**
```autoit
Local $hBitmap = _ImageSearch_hBitmapLoad("icon.png")
```

### Utility Functions

#### _ImageSearch_ClearCache()
```autoit
_ImageSearch_ClearCache()
```
Clears all cached bitmaps and location data.

**Returns:** 1

#### _ImageSearch_GetVersion()
```autoit
_ImageSearch_GetVersion()
```
Returns the DLL version string.

**Returns:** Version string

#### _ImageSearch_GetSysInfo()
```autoit
_ImageSearch_GetSysInfo()
```
Returns detailed system information including CPU features, cache status, and monitor count.

**Returns:** System info string

#### _ImageSearch_GetLastResult()
```autoit
_ImageSearch_GetLastResult()
```
Returns debug information from the last search operation.

**Returns:** Debug info string

#### _ImageSearch_WarmUpCache()
```autoit
_ImageSearch_WarmUpCache($sImagePath)
```
Warms up the cache for a specific image.

**Parameters:**
- `$sImagePath` - Image file path

**Returns:** 1 on success, 0 on failure

### Wait & Click Functions

#### _ImageSearch_Wait()
```autoit
_ImageSearch_Wait($sImagePath, [$iTimeout = 30000], [$iCheckInterval = 100], [$iTolerance = 10], [$iResults = 1], [$iCenterPos = 1], [$fMinScale = 1.0], [$fMaxScale = 1.0], [$fScaleStep = 0.1])
```
Waits for an image to appear on screen with timeout.

**Parameters:**
- `$sImagePath` - Image to wait for
- `$iTimeout` - Maximum wait time in ms (default: 30000)
- `$iCheckInterval` - Check frequency in ms (default: 100)
- Other parameters same as _ImageSearch()

**Returns:**
- Success: 2D array with position
- Failure: Array with [0][0] = 0 if timeout, sets @error

**Example:**
```autoit
Local $aPos = _ImageSearch_Wait("loading_complete.png", 60000)
If $aPos[0][0] > 0 Then ConsoleWrite("Page loaded!" & @CRLF)
```

#### _ImageSearch_WaitClick()
```autoit
_ImageSearch_WaitClick($sImagePath, [$iTimeout = 30000], [$iCheckInterval = 100], [$iTolerance = 10], [$iCenterPos = 1], [$fMinScale = 1.0], [$fMaxScale = 1.0], [$fScaleStep = 0.1], [$sButton = "left"], [$iClicks = 1], [$iXOffset = 0], [$iYOffset = 0])
```
Waits for an image and clicks it.

**Parameters:**
- `$sImagePath` - Image to wait for
- `$iTimeout` - Maximum wait time in ms (default: 30000)
- `$iCheckInterval` - Check frequency in ms (default: 100)
- `$iTolerance` - Color tolerance (default: 10)
- `$iCenterPos` - Return center (1) or top-left (0) coordinates (default: 1)
- `$fMinScale` - Minimum scale factor (default: 1.0)
- `$fMaxScale` - Maximum scale factor (default: 1.0)
- `$fScaleStep` - Scale increment (default: 0.1)
- `$sButton` - Mouse button (default: "left")
- `$iClicks` - Number of clicks (default: 1)
- `$iXOffset` - X offset for click (default: 0)
- `$iYOffset` - Y offset for click (default: 0)

**Returns:**
- Success: 1
- Failure: 0, sets @error

#### _ImageSearch_MouseClick()
```autoit
_ImageSearch_MouseClick($sButton[, $iX=-1[, $iY=-1[, $iClicks=1[, $iSpeed=0[, $iScreen=0]]]]])
```
Clicks mouse at coordinates (screen or current position)
**Parameters**
- `$sButton` - Button: "left", "right", "middle" (default: "left")
- `$iX, $iY` - Coordinates (-1 = current position)
- `$iClicks` - Number of clicks (default: 1)
- `$iSpeed`  - Speed 0-100 (0=instant, default: 0)
- `$iScreen` - Monitor index (default: 0)
**Return values**
- 1 on success
- 0 on failure


#### _ImageSearch_MouseClickWin
```autoit
_ImageSearch_MouseClickWin($sTitle, $sText, $iX, $iY[, $sButton="left"[, $iClicks=1[, $iSpeed=0]]])
```
Clicks mouse in a window

**Parameters**
- `$sTitle`  - Window title/class/handle
- `$sText`   - Window text
- `$iX, $iY` - Relative coordinates in window
- `$sButton` - Button (default: "left")
- `$iClicks` - Number of clicks (default: 1)
- `$iSpeed`  - Speed 0-100 (default: 0)
**Return values**
- 1 on success
- 0 on failure


---

## 💡 Usage Tips & Best Practices

### Performance Optimization

1. **Use specific search regions:**
```autoit
$aResult = _ImageSearch_Area("icon.png", 100, 100, 800, 600)
```

2. **Adjust tolerance:**
- Strict (0): May miss variations
- Optimal (10-30): Balances accuracy
- Loose (100+): May get false positives

3. **Cache benefits:** Automatic; clear when needed:
```autoit
_ImageSearch_ClearCache()
```

4. **Use HBITMAP for multiple searches:**
```autoit
Local $hScreen = _ImageSearch_CaptureScreen(0, 0, @DesktopWidth, @DesktopHeight)
Local $aResult1 = _ImageSearch_hBitmap($hScreen, "icon1.png", 0, 0, 0, 0, 10)
Local $aResult2 = _ImageSearch_hBitmap($hScreen, "icon2.png", 0, 0, 0, 0, 10)
_WinAPI_DeleteObject($hScreen)
```

### Image Preparation

1. Use PNG with transparency for UI elements
2. Keep target images small (<500x500 optimal)
3. Avoid JPEG for UI (compression artifacts)
4. Capture at native resolution

### Transparency Handling

Automatic based on alpha channel; tolerance affects semi-transparent pixels.

---

## ❌ Error Handling 

Functions set `@error` on failure:

### Error Codes Reference

| Code | Constant | Description |
|------|----------|-------------|
| 0 | $IMGSEARCH_ERROR_SUCCESS | Success |
| -1 | $IMGSEARCH_ERROR_INVALID_PATH | Invalid path or image format |
| -2 | $IMGSEARCH_ERROR_FAILED_TO_LOAD_IMAGE | Failed to load image from file |
| -3 | $IMGSEARCH_ERROR_FAILED_TO_GET_SCREEN_DC | Failed to capture screen |
| -4 | $IMGSEARCH_ERROR_INVALID_SEARCH_REGION | Invalid search region |
| -5 | $IMGSEARCH_ERROR_INVALID_PARAMETERS | Invalid parameters |
| -6 | $IMGSEARCH_ERROR_INVALID_SOURCE_BITMAP | Invalid source bitmap |
| -7 | $IMGSEARCH_ERROR_INVALID_TARGET_BITMAP | Invalid target bitmap |
| -9 | $IMGSEARCH_ERROR_RESULT_TOO_LARGE | Result string too large |
| -10 | $IMGSEARCH_ERROR_INVALID_MONITOR | Invalid monitor index |

### Example: Robust Error Handling
```autoit
Local $aResult = _ImageSearch("button.png")
If @error Then
    Switch @error
        Case $IMGSEARCH_ERROR_INVALID_PATH
            MsgBox(16, "Error", "Image file not found!")
        Case Else
            MsgBox(16, "Error", "Unknown error: " & @error)
    EndSwitch
EndIf
```

---

## 🔧 Troubleshooting

**1. "DLL not found" error:**
```autoit
_ImageSearch_SetDllPath(@ScriptDir & "\ImageSearchDLL.dll")
_ImageSearch_Startup()
```

**2. Image not found but visible:**
- Increase tolerance
- Check scaling/DPI
- Recapture reference image

**3. Slow performance:**
- Reduce region
- Lower tolerance
- Use x64 DLL on x64 AutoIt

**4. Multiple monitors:** Use regions; full screen searches all.


---

## 🎯 Advanced Examples

### Example 1: Game Automation
```autoit
While True
    Local $aButton = _ImageSearch_Area("play_button.png", 100, 400, 800, 600, 20)
    If $aButton[0][0] > 0 Then MouseClick("left", $aButton[1][1], $aButton[1][2])
    Sleep(100)
WEnd
```

### Example 2: UI Testing
```autoit
Local $aLogin = _ImageSearch("login_btn.png")
If $aLogin[0][0] > 0 Then MouseClick("left", $aLogin[1][1], $aLogin[1][2])
Local $aDashboard = _ImageSearch_Wait("dashboard_logo.png", 10000)
If $aDashboard[0][0] > 0 Then ConsoleWrite("Login successful!" & @CRLF)
```

---

## 📝 License

**OpenSource** - Free for personal and commercial use.

**Author:** Đào Văn Trong

**Copyright © 2025** - Dao Van Trong - TRONG.PRO. All rights reserved.

---

## 🔗 Support & Contact

- **Website:** [TRONG.PRO](https://www.trong.pro)
- **GitHub Issues:** [Report bugs and feature requests](https://github.com/daovantrong/ImageSearchDLL)
- **AutoIt Forum:** [ImageSearch Thread](https://www.autoitscript.com/forum/)
---

**🎉 Thank you for using ImageSearch UDF!**