#include-once

; #INDEX# =======================================================================================================================
; Title .........: MemoryUDF v1.0
; AutoIt Version : 3.3.14+
; Language ......: English
; Description ...: Memory reading/writing and inline assembly execution
; Author(s) .....: Dao Van Trong - TRONG.PRO
; Dll(s) ........: Kernel32.dll, Psapi.dll, User32.dll
; ===============================================================================================================================

; #CONSTANTS - MEMORY# ==========================================================================================================
Global Const $PROCESS_ALL_ACCESS = 0x1F0FFF
Global Const $PROCESS_VM_READ = 0x0010
Global Const $PROCESS_VM_WRITE = 0x0020
Global Const $PROCESS_VM_OPERATION = 0x0008
Global Const $PROCESS_QUERY_INFORMATION = 0x0400

Global Const $TH32CS_SNAPPROCESS = 0x00000002
Global Const $TH32CS_SNAPMODULE = 0x00000008
Global Const $TH32CS_SNAPMODULE32 = 0x00000010

Global Const $TOKEN_ADJUST_PRIVILEGES = 0x0020
Global Const $TOKEN_QUERY = 0x0008
Global Const $SE_PRIVILEGE_ENABLED = 0x0002

Global Const $MEM_COMMIT = 0x1000
Global Const $MEM_RESERVE = 0x2000
Global Const $MEM_RELEASE = 0x8000
Global Const $PAGE_EXECUTE_READWRITE = 0x40
; ===============================================================================================================================

; #CONSTANTS - ASM REGISTERS# ===================================================================================================
Global Const $AL = 0, $CL = 1, $DL = 2, $BL = 3, $AH = 4, $CH = 5, $DH = 6, $BH = 7
Global Const $AX = 0, $CX = 1, $DX = 2, $BX = 3, $SP = 4, $BP = 5, $SI = 6, $DI = 7
Global Const $EAX = 0, $ECX = 1, $EDX = 2, $EBX = 3, $ESP = 4, $EBP = 5, $ESI = 6, $EDI = 7
Global Const $RAX = 0, $RCX = 1, $RDX = 2, $RBX = 3, $RSP = 4, $RBP = 5, $RSI = 6, $RDI = 7
Global Const $R8 = 8, $R9 = 9, $R10 = 10, $R11 = 11, $R12 = 12, $R13 = 13, $R14 = 14, $R15 = 15

Global Const $MM0 = 0, $MM1 = 1, $MM2 = 2, $MM3 = 3, $MM4 = 4, $MM5 = 5, $MM6 = 6, $MM7 = 7
Global Const $XMM0 = 0, $XMM1 = 1, $XMM2 = 2, $XMM3 = 3, $XMM4 = 4, $XMM5 = 5, $XMM6 = 6, $XMM7 = 7
Global Const $ES = 0, $CS = 1, $SS = 2, $DS = 3, $FS = 4, $GS = 5

Global Const $ModRM_Special = 1
Global Const $ModRM_Standard = 2

Global Const $ModRM32_EAX = 0, $ModRM32_ECX = 1, $ModRM32_EDX = 2, $ModRM32_EBX = 3
Global Const $ModRM32_SIB = 4, $ModRM32_Disp32 = 5, $ModRM32_ESI = 6, $ModRM32_EDI = 7
; ===============================================================================================================================

; #GLOBALS# =====================================================================================================================
Global $g_hKernel32 = -1
Global $g_sInstructionSetPath = @ScriptDir & "\IA-32 Instruction Set.txt"
Global $g_aInstructions = 0
; ===============================================================================================================================

; ===============================================================================================================================
; LIST FUNCTIONS
; ===============================================================================================================================
; _Memory_Open($vProcess, $iAccess = $PROCESS_ALL_ACCESS, $bInherit = False) ; Opens a process handle with specified access rights
; _Memory_Close($ahHandle) ; Closes process handle and kernel32.dll
; _Memory_Read($ahHandle, $iAddress, $sType = "dword") ; Reads value from memory address (x86/x64 compatible)
; _Memory_Write($ahHandle, $iAddress, $vData, $sType = "dword") ; Writes value to memory address (x86/x64 compatible)
; _Memory_ReadPointer($ahHandle, $iBaseAddress, $aOffsets, $sType = "dword") ; Reads value from pointer chain (x86/x64 compatible)
; _Memory_WritePointer($ahHandle, $iBaseAddress, $aOffsets, $vData, $sType = "dword") ; Writes value to pointer chain (x86/x64 compatible)
; _Memory_CalculatePointer($ahHandle, $iBaseAddress, $aOffsets) ; Calculates final address from pointer chain (x86/x64 compatible)
; _Memory_GetModuleBaseAddress($ahHandle, $sModule) ; Gets base address of a module in target process (x86/x64 compatible)
; _Memory_GetProcessBaseAddress($ahHandle) ; Gets base address of the main executable (x86/x64 compatible)
; _Memory_SetPrivilege($sPrivilege, $bEnable = True) ; Enables/disables privilege for current process
; _ASM_Initialize($sFilePath = "") ; Initializes instruction set from file (optional, for advanced assembly)
; _ASM_Compile($sHexCode) ; Compiles hex machine code string into executable memory
; _ASM_Execute($pCode, $iParam1 = 0, $iParam2 = 0, $iParam3 = 0, $iParam4 = 0) ; Executes compiled assembly code
; _ASM_Free($pCode) ; Frees memory allocated by _ASM_Compile
; _ASM_VirtualAlloc($pAddress, $iSize, $iAllocType, $iProtect) ; Allocates executable memory region
; _ASM_Inject($ahHandle, $sHexCode, $bAutoFree = True) ; Injects and executes assembly code in remote process
; _ASM_CreateCodeCave($ahHandle, $iAddress, $iSize) ; Creates a code cave (NOP sled) in remote process
; _Mem_QuickRead($vProcess, $iAddress, $sType = "dword") ; Quick read without opening/closing handle
; _Mem_QuickWrite($vProcess, $iAddress, $vData, $sType = "dword") ; Quick write without opening/closing handle
; _ASM_QuickExecute($sHexCode) ; Compile and execute assembly in one call (auto cleanup)
; _ASM_CreateJump($iFrom, $iTo, $bShort = False) ; Creates JMP instruction from source to destination
; _ASM_CreateCall($iFrom, $iTo) ; Creates CALL instruction
; _ASM_CreatePush($iValue) ; Creates PUSH instruction for immediate value
; _ASM_CreateMov($iRegister, $iValue) ; Creates MOV instruction for register
; _ASM_CreateNOP($iCount = 1) ; Creates NOP sled of specified length
; _ASM_CreateRet($iPopBytes = 0) ; Creates RET instruction
; _ASM_CreateAdd($iRegister, $iValue) ; Creates ADD instruction for register
; _ASM_CreateSub($iRegister, $iValue) ; Creates SUB instruction for register
; _ASM_CreateXor($iRegister, $iTarget = -1) ; Creates XOR instruction for register (common for zeroing)
; _ASM_CreateInc($iRegister) ; Creates INC instruction for register
; _ASM_CreateDec($iRegister) ; Creates DEC instruction for register
; _ASM_CreatePushad() ; Creates PUSHAD instruction (push all registers)
; _ASM_CreatePopad() ; Creates POPAD instruction (pop all registers)
; _ASM_HookFunction($ahHandle, $iTargetAddress, $iHookAddress, $iNOPCount = 0) ; Hooks a function by replacing first bytes with JMP
; _ASM_UnhookFunction($ahHandle, $iTargetAddress, $sOriginalBytes) ; Restores original bytes to unhook a function
; _Memory_PatternScan($ahHandle, $iStartAddress, $iSize, $sPattern) ; Scans memory for byte pattern (supports wildcards)
; _Memory_PatternScanAll($ahHandle, $iStartAddress, $iSize, $sPattern, $iMaxResults = 0) ; Scans memory for all occurrences of byte pattern
; _Memory_StringSearch($ahHandle, $iStartAddress, $iSize, $sString, $bUnicode = False, $bCaseSensitive = True) ; Searches for ASCII/Unicode string in memory
; _Memory_StringSearchAll($ahHandle, $iStartAddress, $iSize, $sString, $bUnicode = False, $bCaseSensitive = True, $iMaxResults = 0) ; Searches for all occurrences of string in memory
; _Memory_IntegerSearch($ahHandle, $iStartAddress, $iSize, $iValue, $sType = "int") ; Searches for integer value in memory
; _Memory_IntegerSearchAll($ahHandle, $iStartAddress, $iSize, $iValue, $sType = "int", $iMaxResults = 0) ; Searches for all occurrences of integer value
; _Memory_FloatSearch($ahHandle, $iStartAddress, $iSize, $fValue, $bDouble = False) ; Searches for float/double value in memory
; _Memory_FloatSearchAll($ahHandle, $iStartAddress, $iSize, $fValue, $bDouble = False, $iMaxResults = 0) ; Searches for all occurrences of float value
; _Memory_HexSearch($ahHandle, $iStartAddress, $iSize, $sHex) ; Searches for hex byte sequence in memory
; _Memory_HexSearchAll($ahHandle, $iStartAddress, $iSize, $sHex, $iMaxResults = 0) ; Searches for all occurrences of hex sequence
; _Memory_AOBScan($ahHandle, $iStartAddress, $iSize, $sAOB) ; Array of Bytes scan with wildcard support (alias for PatternScan)
; _Memory_AOBScanAll($ahHandle, $iStartAddress, $iSize, $sAOB, $iMaxResults = 0) ; Array of Bytes scan all occurrences
; _Memory_ScanRegion($ahHandle, $sModule, $sPattern) ; Scans entire memory region of a module
; _Memory_ScanRegionAll($ahHandle, $sModule, $sPattern, $iMaxResults = 0) ; Scans entire memory region for all occurrences
; _SwapEndian($iValue, $iSize = 4) ; Swaps endianness for little-endian (Intel) format
; _Memory_GetProcessModules($ahHandle) ; Gets list of all modules in a process
; _Memory_GetModuleSize($ahHandle, $sModule) ; Gets size of a module in bytes
; _Memory_Protect($ahHandle, $iAddress, $iSize, $iProtection) ; Changes memory protection
; _Memory_ReadString($ahHandle, $iAddress, $iMaxLength = 256, $bUnicode = False) ; Reads null-terminated string from memory
; _Memory_WriteString($ahHandle, $iAddress, $sString, $bUnicode = False, $bNullTerminate = True) ; Writes null-terminated string to memory
; _Memory_ReadArray($ahHandle, $iAddress, $iCount, $sType = "int") ; Reads array of values from memory
; _Memory_WriteArray($ahHandle, $iAddress, $aArray, $sType = "int") ; Writes array of values to memory
; _Memory_Freeze($ahHandle, $iAddress, $vValue, $sType = "int") ; Continuously writes value to address (use in loop or with AdlibRegister)
; _Memory_Compare($ahHandle, $iAddress1, $iAddress2, $iSize) ; Compares two memory regions
; _Memory_Copy($ahHandle, $iSourceAddress, $iDestAddress, $iSize) ; Copies memory from one location to another
; _Memory_Fill($ahHandle, $iAddress, $iSize, $iByte = 0) ; Fills memory region with specified byte value
; _Memory_DumpRegion($ahHandle, $iAddress, $iSize) ; Dumps memory region to hex string
; _Memory_FindDifferences($aSnapshot1, $aSnapshot2) ; Finds differences between two scans (useful for value changes)
; _Memory_GetPageInfo($ahHandle, $iAddress) ; Gets information about a memory page


; ===============================================================================================================================
; MEMORY FUNCTIONS
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_Open
; Description ...: Opens a process handle with specified access rights
; Syntax ........: _Memory_Open($vProcess [, $iAccess = $PROCESS_ALL_ACCESS [, $bInherit = False]])
; Parameters ....: $vProcess  - Process name (string) or PID (integer).
;                  $iAccess   - Desired access rights flag. Defaults to $PROCESS_ALL_ACCESS.
;                  $bInherit  - When True, the returned handle is inheritable. Defaults to False.
; Return values .: Success    - Array: [0] kernel32.dll handle, [1] process handle, [2] PID.
;                  Failure    - Returns 0 and sets @error:
;                              |1| Invalid process reference or process not found.
;                              |2| Unable to load kernel32.dll.
;                              |3| Failed to open target process.
; Remarks .......: Requires appropriate privileges; call _Memory_SetPrivilege("SeDebugPrivilege", True) for remote processes.
;                  Always pair with _Memory_Close() to release resources and kernel handles.
; Related .......: _Memory_Close(), _Memory_SetPrivilege(), _Mem_QuickRead()
; Example .......: See README.md > Quick Start > Basic Memory Reading.
; ===============================================================================================================================
Func _Memory_Open($vProcess, $iAccess = $PROCESS_ALL_ACCESS, $bInherit = False)
    Local $iPID = (IsString($vProcess)) ? ProcessExists($vProcess) : $vProcess

    If Not $iPID Then Return SetError(1, 0, 0)

    Local $ahHandle[3] = [DllOpen("kernel32.dll"), 0, $iPID]

    If $ahHandle[0] = -1 Then Return SetError(2, 0, 0)

    Local $aCall = DllCall($ahHandle[0], "handle", "OpenProcess", _
            "dword", $iAccess, "bool", $bInherit, "dword", $iPID)

    If @error Or Not $aCall[0] Then
        DllClose($ahHandle[0])
        Return SetError(3, 0, 0)
    EndIf

    $ahHandle[1] = $aCall[0]
    Return $ahHandle
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_Close
; Description ...: Closes process handle and kernel32.dll
; Syntax ........: _Memory_Close($ahHandle)
; Parameters ....: $ahHandle - Array returned from _Memory_Open().
; Return values .: Success    - True
;                  Failure    - False and sets @error:
;                              |1| Invalid handle array structure.
;                              |2| CloseHandle failed (process handle).
;                  Note ......: When @extended = 1, CloseHandle succeeded but DllClose failed.
; Remarks .......: Always call after _Memory_Open() or helper wrappers to avoid handle leaks.
;                  Safe to call multiple times; silently ignores already-closed handles.
; Related .......: _Memory_Open(), _Mem_QuickRead(), _Mem_QuickWrite()
; Example .......: See README.md > Quick Start > Basic Memory Reading.
; ===============================================================================================================================
Func _Memory_Close($ahHandle)
    If Not IsArray($ahHandle) Or UBound($ahHandle) < 2 Then Return SetError(1, 0, False)

    Local $bResult = True
    If $ahHandle[1] Then
        Local $aCall = DllCall($ahHandle[0], "bool", "CloseHandle", "handle", $ahHandle[1])
        If @error Or Not $aCall[0] Then $bResult = False
    EndIf

    If $ahHandle[0] Then DllClose($ahHandle[0])
    Return $bResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_Read
; Description ...: Reads value from memory address (x86/x64 compatible)
; Syntax ........: _Memory_Read($ahHandle, $iAddress [, $sType = "dword"])
; Parameters ....: $ahHandle - Array returned from _Memory_Open().
;                  $iAddress - Absolute memory address to read.
;                  $sType    - DllStruct type ("byte", "word", "int", "double", "ptr", ...). Defaults to "dword".
; Return values .: Success    - Value read, cast to requested type.
;                  Failure    - 0 and sets @error:
;                              |1| Invalid process handle array.
;                              |2| Invalid struct type string.
;                              |3| ReadProcessMemory failed (see @extended for GetLastError()).
;                              |4| Process handle invalid or closed.
;                              |5| Address invalid (< 0).
; Remarks .......: Supports x86/x64 transparently. Ensure target address is readable and size matches data type.
; Related .......: _Memory_Write(), _Memory_ReadPointer(), _Mem_QuickRead()
; Example .......: README.md > Quick Start > Basic Memory Reading.
; ===============================================================================================================================
Func _Memory_Read($ahHandle, $iAddress, $sType = "dword")
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)
    If $ahHandle[1] <= 0 Then Return SetError(4, 0, 0) ; Validate process handle
    If $iAddress <= 0 Then Return SetError(5, 0, 0) ; Validate memory address

    ; Validate requested struct type
    Local $tBuffer = DllStructCreate($sType)
    If @error Then Return SetError(2, 0, 0)

    ; Read from process memory
    Local $aCall = DllCall($ahHandle[0], "bool", "ReadProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iAddress, _
            "ptr", DllStructGetPtr($tBuffer), _
            "ulong_ptr", DllStructGetSize($tBuffer), "ulong_ptr*", 0)

    If @error Then Return SetError(3, @error, 0)
    If Not $aCall[0] Then Return SetError(3, DllCall("kernel32.dll", "dword", "GetLastError"), 0)

    Return DllStructGetData($tBuffer, 1)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_Write
; Description ...: Writes value to memory address (x86/x64 compatible)
; Syntax ........: _Memory_Write($ahHandle, $iAddress, $vData [, $sType = "dword"])
; Parameters ....: $ahHandle - Array returned from _Memory_Open().
;                  $iAddress - Absolute memory address to write.
;                  $vData    - Value to write (cast to type).
;                  $sType    - DllStruct type string (default "dword").
; Return values .: Success    - True
;                  Failure    - False and sets @error:
;                              |1| Invalid process handle array.
;                              |2| Invalid struct type.
;                              |3| DllStructSetData failed for provided value.
;                              |4| WriteProcessMemory failed (see @extended for GetLastError()).
;                              |5| Process handle invalid or closed.
;                              |6| Address invalid (< 0).
; Remarks .......: If target page is read-only, call _Memory_Protect() before writing.
; Related .......: _Memory_Read(), _Memory_WritePointer(), _Mem_QuickWrite()
; Example .......: README.md > Detailed Examples > Quick Read / Quick Write Helpers.
; ===============================================================================================================================
Func _Memory_Write($ahHandle, $iAddress, $vData, $sType = "dword")
    If Not IsArray($ahHandle) Then Return SetError(1, 0, False)
    If $ahHandle[1] <= 0 Then Return SetError(5, 0, False) ; Validate process handle
    If $iAddress <= 0 Then Return SetError(6, 0, False) ; Validate memory address

    ; Validate requested struct type
    Local $tBuffer = DllStructCreate($sType)
    If @error Then Return SetError(2, 0, False)

    ; Populate buffer with data
    DllStructSetData($tBuffer, 1, $vData)
    If @error Then Return SetError(3, 0, False)

    ; Write data into process memory
    Local $aCall = DllCall($ahHandle[0], "bool", "WriteProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iAddress, _
            "ptr", DllStructGetPtr($tBuffer), _
            "ulong_ptr", DllStructGetSize($tBuffer), "ulong_ptr*", 0)

    If @error Then Return SetError(4, @error, False)
    If Not $aCall[0] Then Return SetError(4, DllCall("kernel32.dll", "dword", "GetLastError"), False)

    Return True
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_ReadPointer
; Description ...: Reads value from pointer chain (x86/x64 compatible)
; Syntax ........: _Memory_ReadPointer($ahHandle, $iBaseAddress, $aOffsets [, $sType = "dword"])
; Parameters ....: $ahHandle     - Array returned from _Memory_Open().
;                  $iBaseAddress - Base pointer address.
;                  $aOffsets     - 0-based array of offsets to traverse.
;                  $sType        - Final read type (default "dword").
; Return values .: Success       - Value at the resolved pointer.
;                  Failure       - 0 and propagates @error from _Memory_CalculatePointer() or _Memory_Read().
; Remarks .......: Automatically handles pointer size based on @AutoItX64. Validates offsets array.
; Related .......: _Memory_CalculatePointer(), _Memory_Read(), _Memory_WritePointer()
; Example .......: README.md > Quick Start > Pointer Chain Navigation.
; ===============================================================================================================================
Func _Memory_ReadPointer($ahHandle, $iBaseAddress, $aOffsets, $sType = "dword")
    Local $iAddress = _Memory_CalculatePointer($ahHandle, $iBaseAddress, $aOffsets)
    If @error Then Return SetError(@error, 0, 0)
    Return _Memory_Read($ahHandle, $iAddress, $sType)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_WritePointer
; Description ...: Writes value to pointer chain (x86/x64 compatible)
; Syntax ........: _Memory_WritePointer($ahHandle, $iBaseAddress, $aOffsets, $vData [, $sType = "dword"])
; Parameters ....: $ahHandle     - Array returned from _Memory_Open().
;                  $iBaseAddress - Base pointer address.
;                  $aOffsets     - 0-based array of offsets to traverse.
;                  $vData        - Value to write at final address.
;                  $sType        - Write type (default "dword").
; Return values .: Success       - True
;                  Failure       - False and propagates @error from _Memory_CalculatePointer() or _Memory_Write().
; Remarks .......: Ensures pointer chain is resolved before writing; use with caution on protected memory.
; Related .......: _Memory_CalculatePointer(), _Memory_Write(), _Memory_ReadPointer()
; Example .......: README.md > Quick Start > Pointer Chain Navigation.
; ===============================================================================================================================
Func _Memory_WritePointer($ahHandle, $iBaseAddress, $aOffsets, $vData, $sType = "dword")
    Local $iAddress = _Memory_CalculatePointer($ahHandle, $iBaseAddress, $aOffsets)
    If @error Then Return SetError(@error, 0, False)
    Return _Memory_Write($ahHandle, $iAddress, $vData, $sType)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_CalculatePointer
; Description ...: Calculates final address from pointer chain (x86/x64 compatible)
; Syntax ........: _Memory_CalculatePointer($ahHandle, $iBaseAddress, $aOffsets)
; Parameters ....: $ahHandle     - Array returned from _Memory_Open().
;                  $iBaseAddress - Base pointer address.
;                  $aOffsets     - 0-based array of offsets to traverse.
; Return values .: Success       - Final absolute address.
;                  Failure       - 0 and sets @error:
;                              |1| Offsets parameter is not an array.
;                              |2| Intermediate read failed; @extended contains failing offset index.
; Remarks .......: Uses pointer-size aware reads automatically. Does not validate target readability.
; Related .......: _Memory_ReadPointer(), _Memory_WritePointer(), _Memory_Read()
; Example .......: README.md > Quick Start > Pointer Chain Navigation.
; ===============================================================================================================================
Func _Memory_CalculatePointer($ahHandle, $iBaseAddress, $aOffsets)
    If Not IsArray($aOffsets) Then Return SetError(1, 0, 0)

    Local $iAddress = $iBaseAddress
    Local $iCount = UBound($aOffsets)
    Local $sPtrType = (@AutoItX64) ? "ptr" : "dword"

    For $i = 0 To $iCount - 1
        $iAddress = _Memory_Read($ahHandle, $iAddress, $sPtrType)
        If @error Then Return SetError(2, $i, 0)
        $iAddress += $aOffsets[$i]
    Next

    Return $iAddress
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_GetModuleBaseAddress
; Description ...: Gets base address of a module in target process (x86/x64 compatible)
; Syntax ........: _Memory_GetModuleBaseAddress($ahHandle, $sModule)
; Parameters ....: $ahHandle - Array returned from _Memory_Open() or PID (integer).
;                  $sModule  - Module name (case-sensitive string) including extension.
; Return values .: Success   - Pointer to the module base address.
;                  Failure   - 0 and sets @error:
;                              |1| Process not found.
;                              |2| Module parameter invalid.
;                              |3| psapi.dll failed to load.
;                              |4| OpenProcess failed.
;                              |5| Module not found in target process.
; Remarks .......: Accepts either handle array or PID for convenience. Requires PROCESS_QUERY_INFORMATION + PROCESS_VM_READ.
;                  For the main executable base, prefer _Memory_GetProcessBaseAddress().
; Related .......: _Memory_GetProcessBaseAddress(), _Memory_GetProcessModules(), _Memory_GetModuleSize()
; Example .......: README.md > Quick Start > Pointer Chain Navigation.
; ===============================================================================================================================
Func _Memory_GetModuleBaseAddress($ahHandle, $sModule)
    Local $iPID = IsArray($ahHandle) ? $ahHandle[2] : $ahHandle

    If Not ProcessExists($iPID) Then Return SetError(1, 0, 0)
    If Not IsString($sModule) Then Return SetError(2, 0, 0)

    Local $hPsapi = DllOpen("psapi.dll")
    If $hPsapi = -1 Then Return SetError(3, 0, 0)

    Local $iPermission = BitOR($PROCESS_QUERY_INFORMATION, $PROCESS_VM_READ)
    Local $aCall = DllCall("kernel32.dll", "handle", "OpenProcess", _
            "dword", $iPermission, "bool", False, "dword", $iPID)

    If @error Or Not $aCall[0] Then
        DllClose($hPsapi)
        Return SetError(4, 0, 0)
    EndIf

    Local $hProcess = $aCall[0]
    Local $tModules = DllStructCreate("ptr[2048]")

    $aCall = DllCall($hPsapi, "bool", "EnumProcessModules", _
            "handle", $hProcess, "ptr", DllStructGetPtr($tModules), _
            "dword", DllStructGetSize($tModules), "dword*", 0)

    Local $iResult = 0
    If Not @error And $aCall[0] And $aCall[4] > 0 Then
        Local $iModCount = $aCall[4] / (@AutoItX64 ? 8 : 4)

        For $i = 1 To $iModCount
            $aCall = DllCall($hPsapi, "dword", "GetModuleBaseNameW", _
                    "handle", $hProcess, "ptr", DllStructGetData($tModules, 1, $i), _
                    "wstr", "", "dword", 260)

            If Not @error And $aCall[3] = $sModule Then
                $iResult = DllStructGetData($tModules, 1, $i)
                ExitLoop
            EndIf
        Next
    EndIf

    DllCall("kernel32.dll", "bool", "CloseHandle", "handle", $hProcess)
    DllClose($hPsapi)

    If Not $iResult Then Return SetError(5, 0, 0)
    Return $iResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_GetProcessBaseAddress
; Description ...: Gets base address of the main executable (x86/x64 compatible)
; Syntax ........: _Memory_GetProcessBaseAddress($ahHandle)
; Parameters ....: $ahHandle - Array returned from _Memory_Open().
; Return values .: Success   - Pointer to the main module base.
;                  Failure   - 0 and sets @error:
;                              |1| Invalid handle array.
;                              |2| Failed to create Toolhelp snapshot.
;                              |3| Module enumeration failed.
; Remarks .......: Use when you specifically need the executable module. For other modules, call _Memory_GetModuleBaseAddress().
;                  Relies on Toolhelp32 APIs that require PROCESS_QUERY_INFORMATION rights.
; Related .......: _Memory_GetModuleBaseAddress(), _Memory_GetProcessModules()
; Example .......: README.md > Quick Start > Pointer Chain Navigation.
; ===============================================================================================================================
Func _Memory_GetProcessBaseAddress($ahHandle)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $aCall = DllCall($ahHandle[0], "handle", "CreateToolhelp32Snapshot", _
            "dword", BitOR($TH32CS_SNAPMODULE, $TH32CS_SNAPMODULE32), "dword", $ahHandle[2])

    If @error Or Not $aCall[0] Then Return SetError(2, 0, 0)

    Local $hSnapshot = $aCall[0]
    Local $tModEntry = DllStructCreate("dword dwSize;dword th32ModuleID;" & _
            "dword th32ProcessID;dword GlblcntUsage;dword ProccntUsage;" & _
            "ptr modBaseAddr;dword modBaseSize;handle hModule;" & _
            "char szModule[256];char szExePath[260]")

    DllStructSetData($tModEntry, "dwSize", DllStructGetSize($tModEntry))

    $aCall = DllCall($ahHandle[0], "bool", "Module32First", _
            "handle", $hSnapshot, "ptr", DllStructGetPtr($tModEntry))

    Local $iBaseAddr = 0
    If Not @error And $aCall[0] Then
        $iBaseAddr = DllStructGetData($tModEntry, "modBaseAddr")
    EndIf

    DllCall($ahHandle[0], "bool", "CloseHandle", "handle", $hSnapshot)

    If Not $iBaseAddr Then Return SetError(3, 0, 0)
    Return $iBaseAddr
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_SetPrivilege
; Description ...: Enables/disables privilege for current process
; Syntax ........: _Memory_SetPrivilege($sPrivilege [, $bEnable = True])
; Parameters ....: $sPrivilege - Privilege name (e.g., "SeDebugPrivilege").
;                  $bEnable    - True to enable, False to disable. Defaults to True.
; Return values .: Success    - True
;                  Failure    - False
; Remarks .......: Required before accessing protected processes. Call with False to revert changes when finished.
; Related .......: _Memory_Open(), _Mem_QuickRead(), _Mem_QuickWrite()
; Example .......: README.md > Quick Start > Basic Memory Reading.
; ===============================================================================================================================
Func _Memory_SetPrivilege($sPrivilege, $bEnable = True)
    Local $aCall = DllCall("kernel32.dll", "handle", "GetCurrentProcess")
    If @error Then Return False

    Local $hCurrProcess = $aCall[0]

    $aCall = DllCall("advapi32.dll", "bool", "OpenProcessToken", _
            "handle", $hCurrProcess, _
            "dword", BitOR($TOKEN_ADJUST_PRIVILEGES, $TOKEN_QUERY), "handle*", 0)

    If @error Or Not $aCall[0] Then Return False

    Local $hToken = $aCall[3]
    Local $tLUID = DllStructCreate("dword;int")

    $aCall = DllCall("advapi32.dll", "bool", "LookupPrivilegeValueW", _
            "wstr", "", "wstr", $sPrivilege, "ptr", DllStructGetPtr($tLUID))

    If @error Or Not $aCall[0] Then
        DllCall("kernel32.dll", "bool", "CloseHandle", "handle", $hToken)
        Return False
    EndIf

    Local $tPriv = DllStructCreate("dword PrivilegeCount;dword LowPart;" & _
            "int HighPart;dword Attributes")

    DllStructSetData($tPriv, "PrivilegeCount", 1)
    DllStructSetData($tPriv, "LowPart", DllStructGetData($tLUID, 1))
    DllStructSetData($tPriv, "HighPart", DllStructGetData($tLUID, 2))
    DllStructSetData($tPriv, "Attributes", $bEnable ? $SE_PRIVILEGE_ENABLED : 0)

    $aCall = DllCall("advapi32.dll", "bool", "AdjustTokenPrivileges", _
            "handle", $hToken, "bool", False, "ptr", DllStructGetPtr($tPriv), _
            "dword", 0, "ptr", 0, "ptr", 0)

    Local $bResult = (Not @error And $aCall[0])
    DllCall("kernel32.dll", "bool", "CloseHandle", "handle", $hToken)

    Return $bResult
EndFunc

; ===============================================================================================================================
; ASSEMBLY FUNCTIONS
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_Initialize
; Description ...: Initializes instruction set from file (optional, for advanced assembly)
; Syntax ........: _ASM_Initialize([$sFilePath = ""])
; Parameters ....: $sFilePath - Optional path to "IA-32 Instruction Set.txt". Defaults to bundled file.
; Return values .: Success    - True and loads instructions into memory.
;                  Failure    - False and sets @error:
;                              |1| Instruction set file not found.
;                              |2| Unable to open instruction set file.
; Remarks .......: Call once before using mnemonic-based assembly helpers. Resets global instruction cache when file missing.
; Related .......: _ASM_Compile(), _ASM_CreateJump(), _ASM_CreateCall()
; Example .......: README.md > Detailed Examples > Remote Code Injection Workflow.
; ===============================================================================================================================
Func _ASM_Initialize($sFilePath = "")
    If $sFilePath = "" Then $sFilePath = $g_sInstructionSetPath

    If Not FileExists($sFilePath) Then
        $g_aInstructions = 0
        Return SetError(1, 0, False)
    EndIf

    Local $hFile = FileOpen($sFilePath, 0)
    If $hFile = -1 Then Return SetError(2, 0, False)

    Global $g_aInstructions = StringSplit(FileRead($hFile), @CRLF, 1)
    FileClose($hFile)

    Return True
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_Compile
; Description ...: Compiles hex machine code string into executable memory
; Syntax ........: _ASM_Compile($sHexCode)
; Parameters ....: $sHexCode - Hex string of machine code (e.g., "C3" for RET).
; Return values .: Success    - Pointer to executable code buffer.
;                  Failure    - 0 and sets @error:
;                              |1| VirtualAlloc failed to reserve executable memory.
; Remarks .......: Buffer size automatically calculated from hex length. Ensure code ends with RET when using CallWindowProc.
; Related .......: _ASM_Execute(), _ASM_Free(), _ASM_QuickExecute()
; Example .......: README.md > Quick Start > Assembly Code Execution.
; ===============================================================================================================================
Func _ASM_Compile($sHexCode)
    Local $iBinSize = StringLen($sHexCode) / 2

    Local $pCode = _ASM_VirtualAlloc(0, $iBinSize, $MEM_COMMIT, $PAGE_EXECUTE_READWRITE)
    If Not $pCode Then Return SetError(1, 0, 0)

    Local $tCode = DllStructCreate("byte[" & $iBinSize & "]", $pCode)
    DllStructSetData($tCode, 1, "0x" & $sHexCode)

    Return $pCode
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_Execute
; Description ...: Executes compiled assembly code
; Syntax ........: _ASM_Execute($pCode [, $iParam1 = 0 [, $iParam2 = 0 [, $iParam3 = 0 [, $iParam4 = 0]]]])
; Parameters ....: $pCode   - Pointer previously returned by _ASM_Compile().
;                  $iParam1 - Optional first parameter passed to code.
;                  $iParam2 - Optional second parameter.
;                  $iParam3 - Optional third parameter.
;                  $iParam4 - Optional fourth parameter.
; Return values .: Success   - Value returned from executed code.
;                  Failure   - 0 and sets @error:
;                              |1| CallWindowProcW invocation failed.
; Remarks .......: Uses CallWindowProcW trampoline; ensure code preserves stack and ends with RET. Parameters passed via registers/stack per WinAPI.
; Related .......: _ASM_Compile(), _ASM_Free(), _ASM_QuickExecute()
; Example .......: README.md > Quick Start > Assembly Code Execution.
; ===============================================================================================================================
Func _ASM_Execute($pCode, $iParam1 = 0, $iParam2 = 0, $iParam3 = 0, $iParam4 = 0)
    Local $aResult = DllCall("user32.dll", "ptr_ptr", "CallWindowProcW", _
            "ptr", $pCode, "ptr", $iParam1, "ptr", $iParam2, "ptr", $iParam3, "ptr", $iParam4)

    If @error Then Return SetError(1, 0, 0)
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_Free
; Description ...: Frees memory allocated by _ASM_Compile
; Syntax ........: _ASM_Free($pCode)
; Parameters ....: $pCode - Pointer returned from _ASM_Compile().
; Return values .: Success - True
;                  Failure - False and sets @error:
;                              |1| VirtualFree failed to release memory.
; Remarks .......: Always call after _ASM_Compile() when buffer no longer needed to prevent executable memory leaks.
; Related .......: _ASM_Compile(), _ASM_QuickExecute()
; Example .......: README.md > Quick Start > Assembly Code Execution.
; ===============================================================================================================================
Func _ASM_Free($pCode)
    Local $aCall = DllCall("kernel32.dll", "bool", "VirtualFree", _
            "ptr", $pCode, "ulong_ptr", 0, "dword", $MEM_RELEASE)

    If @error Or Not $aCall[0] Then Return SetError(1, 0, False)
    Return True
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_VirtualAlloc
; Description ...: Allocates executable memory region
; Syntax ........: _ASM_VirtualAlloc($pAddress, $iSize, $iAllocType, $iProtect)
; Parameters ....: $pAddress  - Desired base address (or 0 to let the system choose).
;                  $iSize     - Allocation size in bytes.
;                  $iAllocType- Allocation type flags (e.g., $MEM_COMMIT | $MEM_RESERVE).
;                  $iProtect  - Memory protection flags (e.g., $PAGE_EXECUTE_READWRITE).
; Return values .: Success    - Pointer to allocated memory.
;                  Failure    - 0 and sets @error:
;                              |1| VirtualAlloc failed.
; Remarks .......: Thin wrapper around kernel32!VirtualAlloc for local process allocations. Pair with _ASM_Free().
; Related .......: _ASM_Compile(), _ASM_Free()
; Example .......: README.md > Detailed Examples > Remote Code Injection Workflow.
; ===============================================================================================================================
Func _ASM_VirtualAlloc($pAddress, $iSize, $iAllocType, $iProtect)
    Local $aCall = DllCall("kernel32.dll", "ptr", "VirtualAlloc", _
            "ptr", $pAddress, "ulong_ptr", $iSize, "dword", $iAllocType, "dword", $iProtect)

    If @error Or Not $aCall[0] Then Return SetError(1, 0, 0)
    Return $aCall[0]
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_Inject
; Description ...: Injects and executes assembly code in remote process
; Syntax ........: _ASM_Inject($ahHandle, $sHexCode [, $bAutoFree = True])
; Parameters ....: $ahHandle - Array returned from _Memory_Open().
;                  $sHexCode - Hex machine code string to inject.
;                  $bAutoFree- When True, releases remote buffer after thread exit.
; Return values .: Success    - Return value retrieved from remote thread.
;                  Failure    - 0 and sets @error:
;                              |1| Invalid handle array.
;                              |2| VirtualAllocEx failed.
;                              |3| WriteProcessMemory failed.
;                              |4| CreateRemoteThread failed.
; Remarks .......: Requires PROCESS_VM_OPERATION, PROCESS_VM_WRITE, PROCESS_CREATE_THREAD rights. Ensure shellcode ends with RET.
; Related .......: _Memory_Open(), _ASM_QuickExecute(), _ASM_Free()
; Example .......: README.md > Detailed Examples > Remote Code Injection Workflow.
; ===============================================================================================================================
Func _ASM_Inject($ahHandle, $sHexCode, $bAutoFree = True)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $iBinSize = StringLen($sHexCode) / 2

    Local $aCall = DllCall($ahHandle[0], "ptr", "VirtualAllocEx", _
            "handle", $ahHandle[1], "ptr", 0, "ulong_ptr", $iBinSize, _
            "dword", $MEM_COMMIT, "dword", $PAGE_EXECUTE_READWRITE)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, 0)

    Local $pRemote = $aCall[0]
    Local $tCode = DllStructCreate("byte[" & $iBinSize & "]")
    DllStructSetData($tCode, 1, "0x" & $sHexCode)

    $aCall = DllCall($ahHandle[0], "bool", "WriteProcessMemory", _
            "handle", $ahHandle[1], "ptr", $pRemote, _
            "ptr", DllStructGetPtr($tCode), "ulong_ptr", $iBinSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then
        DllCall($ahHandle[0], "bool", "VirtualFreeEx", _
                "handle", $ahHandle[1], "ptr", $pRemote, "ulong_ptr", 0, "dword", $MEM_RELEASE)
        Return SetError(3, 0, 0)
    EndIf

    $aCall = DllCall($ahHandle[0], "handle", "CreateRemoteThread", _
            "handle", $ahHandle[1], "ptr", 0, "ulong_ptr", 0, _
            "ptr", $pRemote, "ptr", 0, "dword", 0, "dword*", 0)

    If @error Or Not $aCall[0] Then
        If $bAutoFree Then
            DllCall($ahHandle[0], "bool", "VirtualFreeEx", _
                    "handle", $ahHandle[1], "ptr", $pRemote, "ulong_ptr", 0, "dword", $MEM_RELEASE)
        EndIf
        Return SetError(4, 0, 0)
    EndIf

    Local $hThread = $aCall[0]
    DllCall($ahHandle[0], "dword", "WaitForSingleObject", "handle", $hThread, "dword", -1)

    Local $iExitCode = 0
    $aCall = DllCall($ahHandle[0], "bool", "GetExitCodeThread", "handle", $hThread, "dword*", 0)
    If Not @error And $aCall[0] Then $iExitCode = $aCall[2]

    DllCall($ahHandle[0], "bool", "CloseHandle", "handle", $hThread)

    If $bAutoFree Then
        DllCall($ahHandle[0], "bool", "VirtualFreeEx", _
                "handle", $ahHandle[1], "ptr", $pRemote, "ulong_ptr", 0, "dword", $MEM_RELEASE)
    EndIf

    Return $iExitCode
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateCodeCave
; Description ...: Creates a code cave (NOP sled) in remote process
; Syntax ........: _ASM_CreateCodeCave($ahHandle, $iAddress, $iSize)
; Parameters ....: $ahHandle - Array returned from _Memory_Open().
;                  $iAddress - Starting address to overwrite with NOPs.
;                  $iSize    - Length of NOP sled in bytes.
; Return values .: Success    - True
;                  Failure    - False and sets @error propagated from _Memory_Write().
; Remarks .......: Writes $iSize bytes of 0x90. Ensure target memory is writable before calling.
; Related .......: _Memory_Write(), _ASM_CreateNOP(), _ASM_HookFunction()
; Example .......: README.md > Detailed Examples > Code Cave Creation & Hooking.
; ===============================================================================================================================
Func _ASM_CreateCodeCave($ahHandle, $iAddress, $iSize)
    Local $sNOPs = StringRepeat("90", $iSize)

    Local $tNOPs = DllStructCreate("byte[" & $iSize & "]")
    DllStructSetData($tNOPs, 1, "0x" & $sNOPs)

    Return _Memory_Write($ahHandle, $iAddress, DllStructGetData($tNOPs, 1), "byte[" & $iSize & "]")
EndFunc

; ===============================================================================================================================
; QUICK ACCESS FUNCTIONS (Combine Memory + ASM operations)
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _Mem_QuickRead
; Description ...: Quick read without opening/closing handle
; Syntax ........: _Mem_QuickRead($vProcess, $iAddress [, $sType = "dword"])
; Parameters ....: $vProcess - Process name (string) or PID (integer).
;                  $iAddress - Memory address to read.
;                  $sType    - Data type string. Defaults to "dword".
; Return values .: Success    - Value read.
;                  Failure    - 0 and sets @error propagated from _Memory_Open()/_Memory_Read().
; Remarks .......: Convenience wrapper that opens and closes handle automatically. Use for sporadic reads.
; Related .......: _Memory_Open(), _Memory_Read(), _Mem_QuickWrite()
; Example .......: README.md > Detailed Examples > Quick Read / Quick Write Helpers.
; ===============================================================================================================================
Func _Mem_QuickRead($vProcess, $iAddress, $sType = "dword")
    Local $ahHandle = _Memory_Open($vProcess)
    If @error Then Return SetError(@error, 0, 0)

    Local $vResult = _Memory_Read($ahHandle, $iAddress, $sType)
    Local $iError = @error

    _Memory_Close($ahHandle)

    If $iError Then Return SetError($iError, 0, 0)
    Return $vResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Mem_QuickWrite
; Description ...: Quick write without opening/closing handle
; Syntax ........: _Mem_QuickWrite($vProcess, $iAddress, $vData [, $sType = "dword"])
; Parameters ....: $vProcess - Process name or PID.
;                  $iAddress - Target memory address.
;                  $vData    - Value to write.
;                  $sType    - Data type string. Defaults to "dword".
; Return values .: Success    - True
;                  Failure    - False and sets @error propagated from _Memory_Open()/_Memory_Write().
; Remarks .......: Opens, writes, then closes handle automatically. Suitable for infrequent writes.
; Related .......: _Memory_Open(), _Memory_Write(), _Mem_QuickRead()
; Example .......: README.md > Detailed Examples > Quick Read / Quick Write Helpers.
; ===============================================================================================================================
Func _Mem_QuickWrite($vProcess, $iAddress, $vData, $sType = "dword")
    Local $ahHandle = _Memory_Open($vProcess)
    If @error Then Return SetError(@error, 0, False)

    Local $bResult = _Memory_Write($ahHandle, $iAddress, $vData, $sType)
    Local $iError = @error

    _Memory_Close($ahHandle)

    If $iError Then Return SetError($iError, 0, False)
    Return $bResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_QuickExecute
; Description ...: Compile and execute assembly in one call (auto cleanup)
; Syntax ........: _ASM_QuickExecute($sHexCode)
; Parameters ....: $sHexCode - Machine code hex string.
; Return values .: Success    - Value returned from executed code.
;                  Failure    - 0 and sets @error:
;                              |1| _ASM_Compile() failed.
;                              |2| _ASM_Execute() failed (cleanup still performed).
; Remarks .......: High-level helper that ensures compiled buffer is freed automatically.
; Related .......: _ASM_Compile(), _ASM_Execute(), _ASM_Free()
; Example .......: README.md > Quick Start > Assembly Code Execution.
; ===============================================================================================================================
Func _ASM_QuickExecute($sHexCode)
    Local $pCode = _ASM_Compile($sHexCode)
    If @error Then Return SetError(1, 0, 0)

    Local $iResult = _ASM_Execute($pCode)
    Local $iError = @error

    _ASM_Free($pCode)

    If $iError Then Return SetError(2, 0, 0)
    Return $iResult
EndFunc


; ===============================================================================================================================
; ADVANCED ASSEMBLY HELPERS
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateJump
; Description ...: Creates JMP instruction from source to destination
; Syntax ........: _ASM_CreateJump($iFrom, $iTo [, $bShort = False])
; Parameters ....: $iFrom  - Source address (address of instruction start).
;                  $iTo    - Destination address to jump to.
;                  $bShort - Force short jump (1-byte offset) if possible.
; Return values .: Hex string representing the JMP instruction.
; Remarks .......: Calculates relative offset automatically. Returns short jump when range fits signed byte.
; Related .......: _ASM_CreateCall(), _ASM_CreateNOP(), _ASM_HookFunction()
; Example .......: README.md > Advanced Assembly Helpers.
; ===============================================================================================================================
Func _ASM_CreateJump($iFrom, $iTo, $bShort = False)
    Local $iOffset = $iTo - $iFrom - 5 ; JMP near = 5 bytes (E9 + 4 bytes offset)

    ; Check if short jump is possible (-128 to +127)
    If $bShort Or ($iOffset >= -128 And $iOffset <= 127) Then
        Local $iShortOffset = $iTo - $iFrom - 2
        If $iShortOffset >= -128 And $iShortOffset <= 127 Then
            ; EB xx = Short jump
            Return "EB" & Hex(BitAND($iShortOffset, 0xFF), 2)
        EndIf
    EndIf

    ; E9 xx xx xx xx = Near jump
    Return "E9" & _SwapEndian($iOffset, 4)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateCall
; Description ...: Creates CALL instruction
; Syntax ........: _ASM_CreateCall($iFrom, $iTo)
; Parameters ....: $iFrom - Source address (address of instruction start).
;                  $iTo   - Destination function address.
; Return values .: Hex string representing the CALL instruction.
; Remarks .......: Uses relative near call (E8 + 4-byte offset). Ensure destination resides within 2GB range on x86.
; Related .......: _ASM_CreateJump(), _ASM_CreatePush(), _ASM_HookFunction()
; Example .......: README.md > Advanced Assembly Helpers.
; ===============================================================================================================================
Func _ASM_CreateCall($iFrom, $iTo)
    Local $iOffset = $iTo - $iFrom - 5 ; CALL = 5 bytes (E8 + 4 bytes offset)
    Return "E8" & _SwapEndian($iOffset, 4)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreatePush
; Description ...: Creates PUSH instruction for immediate value
; Syntax ........: _ASM_CreatePush($iValue)
; Parameters ....: $iValue - Value to push (supports 8-bit and 32-bit immediates).
; Return values .: Hex string representing the PUSH instruction.
; Remarks .......: Emits 6A for imm8 when value fits signed byte range, otherwise emits 68 with 4-byte immediate.
; Related .......: _ASM_CreateMov(), _ASM_CreateCall()
; Example .......: README.md > Advanced Assembly Helpers.
; ===============================================================================================================================
Func _ASM_CreatePush($iValue)
    ; Check if value fits in signed byte (-128 to +127)
    If $iValue >= -128 And $iValue <= 127 Then
        ; 6A xx = PUSH imm8
        Return "6A" & Hex(BitAND($iValue, 0xFF), 2)
    Else
        ; 68 xx xx xx xx = PUSH imm32
        Return "68" & _SwapEndian($iValue, 4)
    EndIf
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateMov
; Description ...: Creates MOV instruction for register
; Syntax ........: _ASM_CreateMov($iRegister, $iValue)
; Parameters ....: $iRegister - Register constant (e.g., $EAX, $ECX).
;                  $iValue    - Immediate value written to register.
; Return values .: Hex string representing the MOV instruction.
; Remarks .......: Uses opcode B8+r (r=register index). For 64-bit immediates, extend logic before use.
; Related .......: _ASM_CreatePush(), _ASM_CreateAdd(), _ASM_CreateSub()
; Example .......: README.md > Advanced Assembly Helpers.
; ===============================================================================================================================
Func _ASM_CreateMov($iRegister, $iValue)
    ; B8+r = MOV r32, imm32 (EAX=B8, ECX=B9, EDX=BA, etc.)
    Return Hex(0xB8 + $iRegister, 2) & _SwapEndian($iValue, 4)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateNOP
; Description ...: Creates NOP sled of specified length
; Syntax ........: _ASM_CreateNOP([$iCount = 1])
; Parameters ....: $iCount - Number of NOP instructions to emit. Defaults to 1.
; Return values .: Hex string of repeated 0x90 bytes.
; Remarks .......: Use to pad instruction space or create code caves. For optimized sleds, consider multi-byte NOPs.
; Related .......: _ASM_CreateCodeCave(), _ASM_HookFunction()
; Example .......: README.md > Advanced Assembly Helpers.
; ===============================================================================================================================
Func _ASM_CreateNOP($iCount = 1)
    Return StringRepeat("90", $iCount)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateRet
; Description ...: Creates RET instruction
; Syntax ........: _ASM_CreateRet([$iPopBytes = 0])
; Parameters ....: $iPopBytes - Number of bytes to pop from stack (default 0).
; Return values .: Hex string representing the RET instruction.
; Remarks .......: Returns C3 for plain RET, or C2 + imm16 for stack cleanup (stdcall style).
; Related .......: _ASM_CreateCall(), _ASM_QuickExecute()
; Example .......: README.md > Advanced Assembly Helpers.
; ===============================================================================================================================
Func _ASM_CreateRet($iPopBytes = 0)
    If $iPopBytes = 0 Then
        Return "C3" ; RET near
    Else
        ; C2 xx xx = RET imm16
        Return "C2" & _SwapEndian($iPopBytes, 2)
    EndIf
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateAdd
; Description ...: Creates ADD instruction for register
; Syntax ........: _ASM_CreateAdd($iRegister, $iValue)
; Parameters ....: $iRegister - Register constant (e.g., $EAX, $ECX).
;                  $iValue    - Immediate value added to register.
; Return values .: Hex string representing the ADD instruction.
; Remarks .......: Uses opcode 83 C0+r for imm8, or 81 C0+r for imm32.
; Related .......: _ASM_CreateSub(), _ASM_CreateMov()
; Example .......: README.md > Advanced Assembly Helpers.
; ===============================================================================================================================
Func _ASM_CreateAdd($iRegister, $iValue)
    ; Check if value fits in signed byte
    If $iValue >= -128 And $iValue <= 127 Then
        ; 83 C0+r xx = ADD r32, imm8
        Return "83" & Hex(0xC0 + $iRegister, 2) & Hex(BitAND($iValue, 0xFF), 2)
    Else
        ; 81 C0+r xx xx xx xx = ADD r32, imm32
        Return "81" & Hex(0xC0 + $iRegister, 2) & _SwapEndian($iValue, 4)
    EndIf
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateSub
; Description ...: Creates SUB instruction for register
; Parameters ....: $iRegister - Register constant
;                  $iValue - Value to subtract
; Return values .: Hex string of SUB instruction
; ===============================================================================================================================
Func _ASM_CreateSub($iRegister, $iValue)
    ; Check if value fits in signed byte
    If $iValue >= -128 And $iValue <= 127 Then
        ; 83 E8+r xx = SUB r32, imm8
        Return "83" & Hex(0xE8 + $iRegister, 2) & Hex(BitAND($iValue, 0xFF), 2)
    Else
        ; 81 E8+r xx xx xx xx = SUB r32, imm32
        Return "81" & Hex(0xE8 + $iRegister, 2) & _SwapEndian($iValue, 4)
    EndIf
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateXor
; Description ...: Creates XOR instruction for register (common for zeroing)
; Parameters ....: $iRegister - Register constant
;                  $iTarget - Target register (same as source for zero)
; Return values .: Hex string of XOR instruction
; ===============================================================================================================================
Func _ASM_CreateXor($iRegister, $iTarget = -1)
    If $iTarget = -1 Then $iTarget = $iRegister
    ; 33 C0+r = XOR r32, r32 (example: 33 C0 = XOR EAX, EAX)
    Return "33" & Hex(0xC0 + ($iRegister * 8) + $iTarget, 2)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateInc
; Description ...: Creates INC instruction for register
; Parameters ....: $iRegister - Register constant
; Return values .: Hex string of INC instruction
; ===============================================================================================================================
Func _ASM_CreateInc($iRegister)
    If @AutoItX64 Then
        ; FF C0+r = INC r32/r64
        Return "FF" & Hex(0xC0 + $iRegister, 2)
    Else
        ; 40+r = INC r32 (x86 only)
        Return Hex(0x40 + $iRegister, 2)
    EndIf
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreateDec
; Description ...: Creates DEC instruction for register
; Parameters ....: $iRegister - Register constant
; Return values .: Hex string of DEC instruction
; ===============================================================================================================================
Func _ASM_CreateDec($iRegister)
    If @AutoItX64 Then
        ; FF C8+r = DEC r32/r64
        Return "FF" & Hex(0xC8 + $iRegister, 2)
    Else
        ; 48+r = DEC r32 (x86 only)
        Return Hex(0x48 + $iRegister, 2)
    EndIf
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreatePushad
; Description ...: Creates PUSHAD instruction (push all registers)
; Return values .: Hex string "60"
; ===============================================================================================================================
Func _ASM_CreatePushad()
    If @AutoItX64 Then
        ; PUSHAD is not available on 64-bit, build manual PUSH sequence per register
        Local $sPushRegs = ""
        $sPushRegs &= "50"          ; PUSH RAX
        $sPushRegs &= "51"          ; PUSH RCX
        $sPushRegs &= "52"          ; PUSH RDX
        $sPushRegs &= "53"          ; PUSH RBX
        $sPushRegs &= "54"          ; PUSH RSP
        $sPushRegs &= "55"          ; PUSH RBP
        $sPushRegs &= "56"          ; PUSH RSI
        $sPushRegs &= "57"          ; PUSH RDI
        Return $sPushRegs
    Else
        ; 60 = PUSHAD (x86 only)
        Return "60"
    EndIf
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_CreatePopad
; Description ...: Creates POPAD instruction (pop all registers)
; Return values .: Hex string "61"
; ===============================================================================================================================
Func _ASM_CreatePopad()
    If @AutoItX64 Then
        ; POPAD is not available on 64-bit, emit POP sequence mirroring PUSHAD
        Local $sPopRegs = ""
        $sPopRegs &= "5F"          ; POP RDI
        $sPopRegs &= "5E"          ; POP RSI
        $sPopRegs &= "5D"          ; POP RBP
        $sPopRegs &= "5C"          ; POP RSP
        $sPopRegs &= "5B"          ; POP RBX
        $sPopRegs &= "5A"          ; POP RDX
        $sPopRegs &= "59"          ; POP RCX
        $sPopRegs &= "58"          ; POP RAX
        Return $sPopRegs
    Else
        ; 61 = POPAD (x86 only)
        Return "61"
    EndIf
EndFunc

; ===============================================================================================================================
; HOOK HELPERS
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_HookFunction
; Description ...: Hooks a function by replacing first bytes with JMP
; Parameters ....: $ahHandle - Process handle
;                  $iTargetAddress - Address to hook
;                  $iHookAddress - Address to jump to
;                  $iNOPCount - Additional NOPs after JMP (default: 0)
; Return values .: Success: Original bytes (for unhooking)
;                  Failure: "", @error set
; ===============================================================================================================================
Func _ASM_HookFunction($ahHandle, $iTargetAddress, $iHookAddress, $iNOPCount = 0)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, "")

    Local $iHookSize = 5 + $iNOPCount

    ; Read original bytes
    Local $sOriginal = ""
    For $i = 0 To $iHookSize - 1
        Local $iByte = _Memory_Read($ahHandle, $iTargetAddress + $i, "byte")
        If @error Then Return SetError(2, 0, "")
        $sOriginal &= Hex($iByte, 2)
    Next

    ; Create hook code
    Local $sHook = _ASM_CreateJump($iTargetAddress, $iHookAddress, False)
    If $iNOPCount > 0 Then $sHook &= _ASM_CreateNOP($iNOPCount)

    ; Write hook
    Local $tHook = DllStructCreate("byte[" & $iHookSize & "]")
    DllStructSetData($tHook, 1, "0x" & $sHook)

    Local $bResult = _Memory_Write($ahHandle, $iTargetAddress, DllStructGetData($tHook, 1), "byte[" & $iHookSize & "]")

    If Not $bResult Then Return SetError(3, 0, "")

    Return $sOriginal
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ASM_UnhookFunction
; Description ...: Restores original bytes to unhook a function
; Parameters ....: $ahHandle - Process handle
;                  $iTargetAddress - Hooked address
;                  $sOriginalBytes - Original bytes from _ASM_HookFunction
; Return values .: Success: True
;                  Failure: False, @error set
; ===============================================================================================================================
Func _ASM_UnhookFunction($ahHandle, $iTargetAddress, $sOriginalBytes)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, False)

    Local $iSize = StringLen($sOriginalBytes) / 2
    Local $tOriginal = DllStructCreate("byte[" & $iSize & "]")
    DllStructSetData($tOriginal, 1, "0x" & $sOriginalBytes)

    Return _Memory_Write($ahHandle, $iTargetAddress, DllStructGetData($tOriginal, 1), "byte[" & $iSize & "]")
EndFunc

; ===============================================================================================================================
; PATTERN SCANNING & MEMORY SEARCH
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_PatternScan
; Description ...: Scans memory for byte pattern (supports wildcards)
; Syntax ........: _Memory_PatternScan($ahHandle, $iStartAddress, $iSize, $sPattern)
; Parameters ....: $ahHandle     - Array returned from _Memory_Open().
;                  $iStartAddress - Start address to begin scanning.
;                  $iSize        - Number of bytes to read.
;                  $sPattern     - Pattern (e.g., "8B 0D ?? ?? ?? ?? 85 C9").
; Return values .: Success       - Address of first match.
;                  Failure       - 0 and sets @error:
;                              |1| Handle array invalid.
;                              |2| Pattern string empty or ReadProcessMemory failed.
;                              |3| Pattern not found.
;                              |4| Scan size invalid (<= 0).
;                              |5| Pattern tokenization produced zero length.
;                              |6| Invalid hex byte in pattern (index in @extended).
; Remarks .......: Supports wildcard bytes using "??". Reads region once for efficiency.
; Related .......: _Memory_PatternScanAll(), _Memory_StringSearch(), _Memory_IntegerSearch()
; Example .......: README.md > Detailed Examples > Pattern Scan with Wildcards.
; ===============================================================================================================================
Func _Memory_PatternScan($ahHandle, $iStartAddress, $iSize, $sPattern)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)
    If $iSize <= 0 Then Return SetError(4, 0, 0)
    If StringLen($sPattern) = 0 Then Return SetError(2, 0, 0)

    ; Normalize and tokenize pattern
    Local $sNormalizedPattern = StringStripWS($sPattern, 8)
    Local $aPattern = StringSplit($sNormalizedPattern, " ")
    Local $iPatternSize = $aPattern[0]

    If $iPatternSize = 0 Then Return SetError(5, 0, 0)

    ; Validate pattern tokens and convert to integers
    Local $aPatternBytes[$iPatternSize + 1]
    Local $aPatternMask[$iPatternSize + 1]

    For $i = 1 To $iPatternSize
        $aPatternMask[$i] = ($aPattern[$i] = "??") ? False : True
        If $aPatternMask[$i] Then
            ; Validate hex token
            If Not StringRegExp($aPattern[$i], "^[0-9A-Fa-f]{2}$") Then
                Return SetError(6, $i, 0) ; Invalid hex value
            EndIf
            $aPatternBytes[$i] = Dec($aPattern[$i])
        EndIf
    Next

    ; Read target memory region
    Local $tBuffer = DllStructCreate("byte[" & $iSize & "]")
    Local $aCall = DllCall($ahHandle[0], "bool", "ReadProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iStartAddress, _
            "ptr", DllStructGetPtr($tBuffer), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, 0)

    ; Scan against pattern
    For $i = 0 To $iSize - $iPatternSize
        Local $bMatch = True

        For $j = 1 To $iPatternSize
            If Not $aPatternMask[$j] Then ContinueLoop ; Skip wildcard token ??

            Local $iByte = DllStructGetData($tBuffer, 1, $i + $j - 1)
            If $iByte <> $aPatternBytes[$j] Then
                $bMatch = False
                ExitLoop
            EndIf
        Next

        If $bMatch Then Return $iStartAddress + $i
    Next

    Return SetError(3, 0, 0) ; Pattern not found
EndFunc
; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_PatternScanAll
; Description ...: Scans memory for all occurrences of byte pattern
; Syntax ........: _Memory_PatternScanAll($ahHandle, $iStartAddress, $iSize, $sPattern [, $iMaxResults = 0])
; Parameters ....: $ahHandle     - Array returned from _Memory_Open().
;                  $iStartAddress - Start address to begin scanning.
;                  $iSize        - Number of bytes to read.
;                  $sPattern     - Pattern to match (supports "??" wildcards).
;                  $iMaxResults  - Maximum results to return (0 = unlimited).
; Return values .: Success       - Array of matching addresses.
;                  Failure       - 0 and sets @error:
;                              |1| Handle array invalid.
;                              |2| ReadProcessMemory failed.
;                              |3| Pattern not found.
;                              |4| Scan size invalid (<= 0).
;                              |5| Pattern tokenization produced zero length.
;                              |6| Invalid hex byte in pattern (index in @extended).
; Remarks .......: Returns dynamic array sized to matches. Use $iMaxResults to limit results for performance.
; Related .......: _Memory_PatternScan(), _Memory_StringSearchAll(), _Memory_IntegerSearchAll()
; Example .......: README.md > Detailed Examples > Pattern Scan with Wildcards.
; ===============================================================================================================================
Func _Memory_PatternScanAll($ahHandle, $iStartAddress, $iSize, $sPattern, $iMaxResults = 0)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)
    If $iSize <= 0 Then Return SetError(4, 0, 0)

    ; Normalize and tokenize pattern
    Local $sNormalizedPattern = StringStripWS($sPattern, 8)
    Local $aPattern = StringSplit($sNormalizedPattern, " ")
    Local $iPatternSize = $aPattern[0]

    If $iPatternSize = 0 Then Return SetError(5, 0, 0)

    ; Validate pattern tokens and convert to integers
    Local $aPatternBytes[$iPatternSize + 1]
    Local $aPatternMask[$iPatternSize + 1]

    For $i = 1 To $iPatternSize
        $aPatternMask[$i] = ($aPattern[$i] = "??") ? False : True
        If $aPatternMask[$i] Then
            ; Validate hex token
            If Not StringRegExp($aPattern[$i], "^[0-9A-Fa-f]{2}$") Then
                Return SetError(6, $i, 0) ; Invalid hex value
            EndIf
            $aPatternBytes[$i] = Dec($aPattern[$i])
        EndIf
    Next

    ; Read target memory region
    Local $tBuffer = DllStructCreate("byte[" & $iSize & "]")
    Local $aCall = DllCall($ahHandle[0], "bool", "ReadProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iStartAddress, _
            "ptr", DllStructGetPtr($tBuffer), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, 0)

    ; Results array
    Local $aResults[1]
    Local $iCount = 0

    ; Scan against pattern
    For $i = 0 To $iSize - $iPatternSize
        Local $bMatch = True

        For $j = 1 To $iPatternSize
            If Not $aPatternMask[$j] Then ContinueLoop ; Skip wildcard token ??

            Local $iByte = DllStructGetData($tBuffer, 1, $i + $j - 1)
            If $iByte <> $aPatternBytes[$j] Then
                $bMatch = False
                ExitLoop
            EndIf
        Next

        If $bMatch Then
            ReDim $aResults[$iCount + 1]
            $aResults[$iCount] = $iStartAddress + $i
            $iCount += 1

            If $iMaxResults > 0 And $iCount >= $iMaxResults Then ExitLoop
        EndIf
    Next

    If $iCount = 0 Then Return SetError(3, 0, 0)

    Return $aResults
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_StringSearch
; Description ...: Searches for ASCII/Unicode string in memory
; Syntax ........: _Memory_StringSearch($ahHandle, $iStartAddress, $iSize, $sString [, $bUnicode = False [, $bCaseSensitive = True]])
; Parameters ....: $ahHandle      - Array returned from _Memory_Open().
;                  $iStartAddress - Start address to begin scanning.
;                  $iSize         - Number of bytes to read.
;                  $sString       - String to search for.
;                  $bUnicode      - True for Unicode (UTF-16 LE), False for ASCII.
;                  $bCaseSensitive- True for case-sensitive search.
; Return values .: Success        - Address of first match.
;                  Failure        - 0 and sets @error.
; Remarks .......: Converts string into hex pattern. For case-insensitive search, delegates to helper.
; Related .......: _Memory_StringSearchAll(), _Memory_PatternScan()
; Example .......: README.md > Pattern Scanning > String Search.
; ===============================================================================================================================
Func _Memory_StringSearch($ahHandle, $iStartAddress, $iSize, $sString, $bUnicode = False, $bCaseSensitive = True)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    ; Convert string to byte pattern
    Local $sPattern = ""

    If $bUnicode Then
        For $i = 1 To StringLen($sString)
            Local $iChar = AscW(StringMid($sString, $i, 1))
            $sPattern &= Hex(BitAND($iChar, 0xFF), 2) & " " & Hex(BitShift($iChar, 8), 2) & " "
        Next
    Else
        For $i = 1 To StringLen($sString)
            $sPattern &= Hex(Asc(StringMid($sString, $i, 1)), 2) & " "
        Next
    EndIf

    $sPattern = StringTrimRight($sPattern, 1)

    ; If case insensitive, need different approach
    If Not $bCaseSensitive Then
        Return _Memory_StringSearchCaseInsensitive($ahHandle, $iStartAddress, $iSize, $sString, $bUnicode)
    EndIf

    Return _Memory_PatternScan($ahHandle, $iStartAddress, $iSize, $sPattern)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_StringSearchAll
; Description ...: Searches for all occurrences of string in memory
; Syntax ........: _Memory_StringSearchAll($ahHandle, $iStartAddress, $iSize, $sString [, $bUnicode = False [, $bCaseSensitive = True [, $iMaxResults = 0]]])
; Parameters ....: $ahHandle      - Array returned from _Memory_Open().
;                  $iStartAddress - Start address to begin scanning.
;                  $iSize         - Number of bytes to read.
;                  $sString       - String to search for.
;                  $bUnicode      - True for Unicode (UTF-16 LE), False for ASCII.
;                  $bCaseSensitive- True for case-sensitive search.
;                  $iMaxResults   - Maximum number of matches to return (0 = unlimited).
; Return values .: Success        - Array of matching addresses.
;                  Failure        - 0 and sets @error.
; Remarks .......: Case-insensitive mode processed by dedicated helper for ASCII/Unicode.
; Related .......: _Memory_StringSearch(), _Memory_PatternScanAll()
; Example .......: README.md > Pattern Scanning > String Search.
; ===============================================================================================================================
Func _Memory_StringSearchAll($ahHandle, $iStartAddress, $iSize, $sString, $bUnicode = False, $bCaseSensitive = True, $iMaxResults = 0)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $sPattern = ""

    If $bUnicode Then
        For $i = 1 To StringLen($sString)
            Local $iChar = AscW(StringMid($sString, $i, 1))
            $sPattern &= Hex(BitAND($iChar, 0xFF), 2) & " " & Hex(BitShift($iChar, 8), 2) & " "
        Next
    Else
        For $i = 1 To StringLen($sString)
            $sPattern &= Hex(Asc(StringMid($sString, $i, 1)), 2) & " "
        Next
    EndIf

    $sPattern = StringTrimRight($sPattern, 1)

    If Not $bCaseSensitive Then
        Return _Memory_StringSearchAllCaseInsensitive($ahHandle, $iStartAddress, $iSize, $sString, $bUnicode, $iMaxResults)
    EndIf

    Return _Memory_PatternScanAll($ahHandle, $iStartAddress, $iSize, $sPattern, $iMaxResults)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_IntegerSearch
; Description ...: Searches for integer value in memory
; Parameters ....: $ahHandle - Process handle
;                  $iStartAddress - Start address
;                  $iSize - Size to scan
;                  $iValue - Integer value to find
;                  $sType - Data type ("byte", "word", "short", "int", "dword", "int64", "uint64")
; Return values .: Success: Address of first match
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_IntegerSearch($ahHandle, $iStartAddress, $iSize, $iValue, $sType = "int")
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $iTypeSize = 4
    Switch $sType
        Case "byte"
            $iTypeSize = 1
        Case "word", "short"
            $iTypeSize = 2
        Case "int", "dword"
            $iTypeSize = 4
        Case "int64", "uint64"
            $iTypeSize = 8
    EndSwitch

    Local $sPattern = _SwapEndian($iValue, $iTypeSize)
    $sPattern = StringRegExpReplace($sPattern, "(.{2})", "$1 ")
    $sPattern = StringTrimRight($sPattern, 1)

    Return _Memory_PatternScan($ahHandle, $iStartAddress, $iSize, $sPattern)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_IntegerSearchAll
; Description ...: Searches for all occurrences of integer value
; Parameters ....: $ahHandle - Process handle
;                  $iStartAddress - Start address
;                  $iSize - Size to scan
;                  $iValue - Integer value to find
;                  $sType - Data type
;                  $iMaxResults - Maximum results (0 = unlimited)
; Return values .: Success: Array of addresses
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_IntegerSearchAll($ahHandle, $iStartAddress, $iSize, $iValue, $sType = "int", $iMaxResults = 0)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $iTypeSize = 4
    Switch $sType
        Case "byte"
            $iTypeSize = 1
        Case "word", "short"
            $iTypeSize = 2
        Case "int", "dword"
            $iTypeSize = 4
        Case "int64", "uint64"
            $iTypeSize = 8
    EndSwitch

    Local $sPattern = _SwapEndian($iValue, $iTypeSize)
    $sPattern = StringRegExpReplace($sPattern, "(.{2})", "$1 ")
    $sPattern = StringTrimRight($sPattern, 1)

    Return _Memory_PatternScanAll($ahHandle, $iStartAddress, $iSize, $sPattern, $iMaxResults)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_FloatSearch
; Description ...: Searches for float/double value in memory
; Parameters ....: $ahHandle - Process handle
;                  $iStartAddress - Start address
;                  $iSize - Size to scan
;                  $fValue - Float value to find
;                  $bDouble - True for double (8 bytes), False for float (4 bytes)
; Return values .: Success: Address of first match
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_FloatSearch($ahHandle, $iStartAddress, $iSize, $fValue, $bDouble = False)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $sType = $bDouble ? "double" : "float"
    Local $tFloat = DllStructCreate($sType)
    DllStructSetData($tFloat, 1, $fValue)

    Local $sPattern = ""
    Local $iTypeSize = $bDouble ? 8 : 4

    For $i = 1 To $iTypeSize
        $sPattern &= Hex(DllStructGetData(DllStructCreate("byte", DllStructGetPtr($tFloat) + $i - 1), 1), 2) & " "
    Next

    $sPattern = StringTrimRight($sPattern, 1)

    Return _Memory_PatternScan($ahHandle, $iStartAddress, $iSize, $sPattern)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_FloatSearchAll
; Description ...: Searches for all occurrences of float value
; Return values .: Success: Array of addresses
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_FloatSearchAll($ahHandle, $iStartAddress, $iSize, $fValue, $bDouble = False, $iMaxResults = 0)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $sType = $bDouble ? "double" : "float"
    Local $tFloat = DllStructCreate($sType)
    DllStructSetData($tFloat, 1, $fValue)

    Local $sPattern = ""
    Local $iTypeSize = $bDouble ? 8 : 4

    For $i = 1 To $iTypeSize
        $sPattern &= Hex(DllStructGetData(DllStructCreate("byte", DllStructGetPtr($tFloat) + $i - 1), 1), 2) & " "
    Next

    $sPattern = StringTrimRight($sPattern, 1)

    Return _Memory_PatternScanAll($ahHandle, $iStartAddress, $iSize, $sPattern, $iMaxResults)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_HexSearch
; Description ...: Searches for hex byte sequence in memory
; Parameters ....: $ahHandle - Process handle
;                  $iStartAddress - Start address
;                  $iSize - Size to scan
;                  $sHex - Hex string (e.g., "8B0D" or "8B 0D")
; Return values .: Success: Address of first match
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_HexSearch($ahHandle, $iStartAddress, $iSize, $sHex)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    ; Normalize hex string
    $sHex = StringStripWS($sHex, 8)
    $sHex = StringRegExpReplace($sHex, "(.{2})", "$1 ")
    $sHex = StringTrimRight($sHex, 1)

    Return _Memory_PatternScan($ahHandle, $iStartAddress, $iSize, $sHex)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_HexSearchAll
; Description ...: Searches for all occurrences of hex sequence
; Return values .: Success: Array of addresses
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_HexSearchAll($ahHandle, $iStartAddress, $iSize, $sHex, $iMaxResults = 0)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    $sHex = StringStripWS($sHex, 8)
    $sHex = StringRegExpReplace($sHex, "(.{2})", "$1 ")
    $sHex = StringTrimRight($sHex, 1)

    Return _Memory_PatternScanAll($ahHandle, $iStartAddress, $iSize, $sHex, $iMaxResults)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_AOBScan
; Description ...: Array of Bytes scan with wildcard support (alias for PatternScan)
; Parameters ....: $ahHandle - Process handle
;                  $iStartAddress - Start address
;                  $iSize - Size to scan
;                  $sAOB - AOB string (e.g., "55 8B EC ?? ?? 5D C3")
; Return values .: Success: Address of first match
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_AOBScan($ahHandle, $iStartAddress, $iSize, $sAOB)
    Return _Memory_PatternScan($ahHandle, $iStartAddress, $iSize, $sAOB)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_AOBScanAll
; Description ...: Array of Bytes scan all occurrences
; Return values .: Success: Array of addresses
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_AOBScanAll($ahHandle, $iStartAddress, $iSize, $sAOB, $iMaxResults = 0)
    Return _Memory_PatternScanAll($ahHandle, $iStartAddress, $iSize, $sAOB, $iMaxResults)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_ScanRegion
; Description ...: Scans entire memory region of a module
; Parameters ....: $ahHandle - Process handle
;                  $sModule - Module name (e.g., "game.exe", "engine.dll")
;                  $sPattern - Pattern to search
; Return values .: Success: Address of first match
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_ScanRegion($ahHandle, $sModule, $sPattern)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, $sModule)
    If @error Then Return SetError(2, 0, 0)

    Local $iModuleSize = _Memory_GetModuleSize($ahHandle, $sModule)
    If @error Then Return SetError(3, 0, 0)

    Return _Memory_PatternScan($ahHandle, $iModuleBase, $iModuleSize, $sPattern)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_ScanRegionAll
; Description ...: Scans entire memory region for all occurrences
; Return values .: Success: Array of addresses
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_ScanRegionAll($ahHandle, $sModule, $sPattern, $iMaxResults = 0)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, $sModule)
    If @error Then Return SetError(2, 0, 0)

    Local $iModuleSize = _Memory_GetModuleSize($ahHandle, $sModule)
    If @error Then Return SetError(3, 0, 0)

    Return _Memory_PatternScanAll($ahHandle, $iModuleBase, $iModuleSize, $sPattern, $iMaxResults)
EndFunc

; Internal helper functions
Func _Memory_StringSearchCaseInsensitive($ahHandle, $iStartAddress, $iSize, $sString, $bUnicode)
    Local $tBuffer = DllStructCreate("byte[" & $iSize & "]")
    Local $aCall = DllCall($ahHandle[0], "bool", "ReadProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iStartAddress, _
            "ptr", DllStructGetPtr($tBuffer), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, 0)

    Local $iStringLen = StringLen($sString)
    Local $sSearchLower = StringLower($sString)

    For $i = 0 To $iSize - ($iStringLen * ($bUnicode ? 2 : 1))
        Local $sFound = ""

        If $bUnicode Then
            For $j = 0 To $iStringLen - 1
                Local $iLow = DllStructGetData($tBuffer, 1, $i + ($j * 2) + 1)
                Local $iHigh = DllStructGetData($tBuffer, 1, $i + ($j * 2) + 2)
                $sFound &= ChrW($iLow + ($iHigh * 256))
            Next
        Else
            For $j = 0 To $iStringLen - 1
                $sFound &= Chr(DllStructGetData($tBuffer, 1, $i + $j + 1))
            Next
        EndIf

        If StringLower($sFound) = $sSearchLower Then Return $iStartAddress + $i
    Next

    Return SetError(3, 0, 0)
EndFunc

Func _Memory_StringSearchAllCaseInsensitive($ahHandle, $iStartAddress, $iSize, $sString, $bUnicode, $iMaxResults)
    Local $tBuffer = DllStructCreate("byte[" & $iSize & "]")
    Local $aCall = DllCall($ahHandle[0], "bool", "ReadProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iStartAddress, _
            "ptr", DllStructGetPtr($tBuffer), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, 0)

    Local $aResults[1]
    Local $iCount = 0
    Local $iStringLen = StringLen($sString)
    Local $sSearchLower = StringLower($sString)

    For $i = 0 To $iSize - ($iStringLen * ($bUnicode ? 2 : 1))
        Local $sFound = ""

        If $bUnicode Then
            For $j = 0 To $iStringLen - 1
                Local $iLow = DllStructGetData($tBuffer, 1, $i + ($j * 2) + 1)
                Local $iHigh = DllStructGetData($tBuffer, 1, $i + ($j * 2) + 2)
                $sFound &= ChrW($iLow + ($iHigh * 256))
            Next
        Else
            For $j = 0 To $iStringLen - 1
                $sFound &= Chr(DllStructGetData($tBuffer, 1, $i + $j + 1))
            Next
        EndIf

        If StringLower($sFound) = $sSearchLower Then
            ReDim $aResults[$iCount + 1]
            $aResults[$iCount] = $iStartAddress + $i
            $iCount += 1

            If $iMaxResults > 0 And $iCount >= $iMaxResults Then ExitLoop
        EndIf
    Next

    If $iCount = 0 Then Return SetError(3, 0, 0)
    Return $aResults
EndFunc

; ===============================================================================================================================
; UTILITY FUNCTIONS
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _SwapEndian
; Description ...: Swaps endianness for little-endian (Intel) format
; Parameters ....: $iValue - Value to swap
;                  $iSize - Size in bytes (2, 4, or 8)
; Return values .: Hex string with swapped bytes
; ===============================================================================================================================
Func _SwapEndian($iValue, $iSize = 4)
    Local $tStruct = DllStructCreate("byte[" & $iSize & "]")
    DllStructSetData($tStruct, 1, $iValue)

    Local $sResult = ""
    For $i = $iSize To 1 Step -1
        $sResult &= Hex(DllStructGetData(DllStructCreate("byte", DllStructGetPtr($tStruct) + $i - 1), 1), 2)
    Next
    Return $sResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_GetProcessModules
; Description ...: Gets list of all modules in a process
; Parameters ....: $ahHandle - Process handle
; Return values .: Success: 2D array [n][2] where [n][0] = name, [n][1] = base address
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_GetProcessModules($ahHandle)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $aCall = DllCall($ahHandle[0], "handle", "CreateToolhelp32Snapshot", _
            "dword", BitOR($TH32CS_SNAPMODULE, $TH32CS_SNAPMODULE32), "dword", $ahHandle[2])

    If @error Or Not $aCall[0] Then Return SetError(2, 0, 0)

    Local $hSnapshot = $aCall[0]
    Local $tModEntry = DllStructCreate("dword dwSize;dword th32ModuleID;" & _
            "dword th32ProcessID;dword GlblcntUsage;dword ProccntUsage;" & _
            "ptr modBaseAddr;dword modBaseSize;handle hModule;" & _
            "char szModule[256];char szExePath[260]")

    DllStructSetData($tModEntry, "dwSize", DllStructGetSize($tModEntry))

    $aCall = DllCall($ahHandle[0], "bool", "Module32First", _
            "handle", $hSnapshot, "ptr", DllStructGetPtr($tModEntry))

    If @error Or Not $aCall[0] Then
        DllCall($ahHandle[0], "bool", "CloseHandle", "handle", $hSnapshot)
        Return SetError(3, 0, 0)
    EndIf

    Local $aModules[1][2]
    Local $iCount = 0

    Do
        ReDim $aModules[$iCount + 1][2]
        $aModules[$iCount][0] = DllStructGetData($tModEntry, "szModule")
        $aModules[$iCount][1] = DllStructGetData($tModEntry, "modBaseAddr")
        $iCount += 1

        $aCall = DllCall($ahHandle[0], "bool", "Module32Next", _
                "handle", $hSnapshot, "ptr", DllStructGetPtr($tModEntry))
    Until @error Or Not $aCall[0]

    DllCall($ahHandle[0], "bool", "CloseHandle", "handle", $hSnapshot)

    Return $aModules
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_GetModuleSize
; Description ...: Gets size of a module in bytes
; Parameters ....: $ahHandle - Process handle
;                  $sModule - Module name
; Return values .: Success: Module size in bytes
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_GetModuleSize($ahHandle, $sModule)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $aCall = DllCall($ahHandle[0], "handle", "CreateToolhelp32Snapshot", _
            "dword", BitOR($TH32CS_SNAPMODULE, $TH32CS_SNAPMODULE32), "dword", $ahHandle[2])

    If @error Or Not $aCall[0] Then Return SetError(2, 0, 0)

    Local $hSnapshot = $aCall[0]
    Local $tModEntry = DllStructCreate("dword dwSize;dword th32ModuleID;" & _
            "dword th32ProcessID;dword GlblcntUsage;dword ProccntUsage;" & _
            "ptr modBaseAddr;dword modBaseSize;handle hModule;" & _
            "char szModule[256];char szExePath[260]")

    DllStructSetData($tModEntry, "dwSize", DllStructGetSize($tModEntry))

    $aCall = DllCall($ahHandle[0], "bool", "Module32First", _
            "handle", $hSnapshot, "ptr", DllStructGetPtr($tModEntry))

    If @error Or Not $aCall[0] Then
        DllCall($ahHandle[0], "bool", "CloseHandle", "handle", $hSnapshot)
        Return SetError(3, 0, 0)
    EndIf

    Local $iSize = 0

    Do
        If DllStructGetData($tModEntry, "szModule") = $sModule Then
            $iSize = DllStructGetData($tModEntry, "modBaseSize")
            ExitLoop
        EndIf

        $aCall = DllCall($ahHandle[0], "bool", "Module32Next", _
                "handle", $hSnapshot, "ptr", DllStructGetPtr($tModEntry))
    Until @error Or Not $aCall[0]

    DllCall($ahHandle[0], "bool", "CloseHandle", "handle", $hSnapshot)

    If Not $iSize Then Return SetError(4, 0, 0)
    Return $iSize
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_Protect
; Description ...: Changes memory protection
; Parameters ....: $ahHandle - Process handle
;                  $iAddress - Memory address
;                  $iSize - Size in bytes
;                  $iProtection - Protection flags (e.g., $PAGE_EXECUTE_READWRITE)
; Return values .: Success: Old protection value
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_Protect($ahHandle, $iAddress, $iSize, $iProtection)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $aCall = DllCall($ahHandle[0], "bool", "VirtualProtectEx", _
            "handle", $ahHandle[1], "ptr", $iAddress, _
            "ulong_ptr", $iSize, "dword", $iProtection, "dword*", 0)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, 0)

    Return $aCall[5] ; Old protection
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_ReadString
; Description ...: Reads null-terminated string from memory
; Parameters ....: $ahHandle - Process handle
;                  $iAddress - Memory address
;                  $iMaxLength - Maximum string length (default: 256)
;                  $bUnicode - True for Unicode, False for ASCII
; Return values .: Success: String value
;                  Failure: "", @error set
; ===============================================================================================================================
Func _Memory_ReadString($ahHandle, $iAddress, $iMaxLength = 256, $bUnicode = False)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, "")

    Local $iCharSize = $bUnicode ? 2 : 1
    Local $sResult = ""

    For $i = 0 To $iMaxLength - 1
        Local $iChar = _Memory_Read($ahHandle, $iAddress + ($i * $iCharSize), $bUnicode ? "word" : "byte")
        If @error Or $iChar = 0 Then ExitLoop

        $sResult &= $bUnicode ? ChrW($iChar) : Chr($iChar)
    Next

    Return $sResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_WriteString
; Description ...: Writes null-terminated string to memory
; Parameters ....: $ahHandle - Process handle
;                  $iAddress - Memory address
;                  $sString - String to write
;                  $bUnicode - True for Unicode, False for ASCII
;                  $bNullTerminate - Add null terminator (default: True)
; Return values .: Success: True
;                  Failure: False, @error set
; ===============================================================================================================================
Func _Memory_WriteString($ahHandle, $iAddress, $sString, $bUnicode = False, $bNullTerminate = True)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, False)

    Local $iCharSize = $bUnicode ? 2 : 1
    Local $iLength = StringLen($sString)

    For $i = 1 To $iLength
        Local $iChar = $bUnicode ? AscW(StringMid($sString, $i, 1)) : Asc(StringMid($sString, $i, 1))

        If Not _Memory_Write($ahHandle, $iAddress + (($i - 1) * $iCharSize), $iChar, $bUnicode ? "word" : "byte") Then
            Return SetError(2, $i, False)
        EndIf
    Next

    If $bNullTerminate Then
        _Memory_Write($ahHandle, $iAddress + ($iLength * $iCharSize), 0, $bUnicode ? "word" : "byte")
    EndIf

    Return True
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_ReadArray
; Description ...: Reads array of values from memory
; Parameters ....: $ahHandle - Process handle
;                  $iAddress - Memory address
;                  $iCount - Number of elements
;                  $sType - Data type
; Return values .: Success: Array of values
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_ReadArray($ahHandle, $iAddress, $iCount, $sType = "int")
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $iTypeSize = 4
    Switch $sType
        Case "byte"
            $iTypeSize = 1
        Case "word", "short"
            $iTypeSize = 2
        Case "int", "dword", "float"
            $iTypeSize = 4
        Case "int64", "uint64", "double"
            $iTypeSize = 8
    EndSwitch

    Local $aResult[$iCount]

    For $i = 0 To $iCount - 1
        $aResult[$i] = _Memory_Read($ahHandle, $iAddress + ($i * $iTypeSize), $sType)
        If @error Then Return SetError(2, $i, 0)
    Next

    Return $aResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_WriteArray
; Description ...: Writes array of values to memory
; Parameters ....: $ahHandle - Process handle
;                  $iAddress - Memory address
;                  $aArray - Array of values
;                  $sType - Data type
; Return values .: Success: True
;                  Failure: False, @error set
; ===============================================================================================================================
Func _Memory_WriteArray($ahHandle, $iAddress, $aArray, $sType = "int")
    If Not IsArray($ahHandle) Or Not IsArray($aArray) Then Return SetError(1, 0, False)

    Local $iTypeSize = 4
    Switch $sType
        Case "byte"
            $iTypeSize = 1
        Case "word", "short"
            $iTypeSize = 2
        Case "int", "dword", "float"
            $iTypeSize = 4
        Case "int64", "uint64", "double"
            $iTypeSize = 8
    EndSwitch

    For $i = 0 To UBound($aArray) - 1
        If Not _Memory_Write($ahHandle, $iAddress + ($i * $iTypeSize), $aArray[$i], $sType) Then
            Return SetError(2, $i, False)
        EndIf
    Next

    Return True
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_Freeze
; Description ...: Continuously writes value to address (use in loop or with AdlibRegister)
; Parameters ....: $ahHandle - Process handle
;                  $iAddress - Memory address
;                  $vValue - Value to write
;                  $sType - Data type
; Return values .: Success: True
;                  Failure: False, @error set
; ===============================================================================================================================
Func _Memory_Freeze($ahHandle, $iAddress, $vValue, $sType = "int")
    If Not IsArray($ahHandle) Then Return SetError(1, 0, False)
    If $iAddress <= 0 Then Return SetError(2, 0, False)

    Local $bResult = _Memory_Write($ahHandle, $iAddress, $vValue, $sType)
    If @error Then Return SetError(3, 0, False)

    Return $bResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_Compare
; Description ...: Compares two memory regions
; Parameters ....: $ahHandle - Process handle
;                  $iAddress1 - First address
;                  $iAddress2 - Second address
;                  $iSize - Size to compare in bytes
; Return values .: Success: True if identical, False if different
;                  Failure: False, @error set
; ===============================================================================================================================
Func _Memory_Compare($ahHandle, $iAddress1, $iAddress2, $iSize)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, False)

    Local $tBuffer1 = DllStructCreate("byte[" & $iSize & "]")
    Local $tBuffer2 = DllStructCreate("byte[" & $iSize & "]")

    Local $aCall = DllCall($ahHandle[0], "bool", "ReadProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iAddress1, _
            "ptr", DllStructGetPtr($tBuffer1), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, False)

    $aCall = DllCall($ahHandle[0], "bool", "ReadProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iAddress2, _
            "ptr", DllStructGetPtr($tBuffer2), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(3, 0, False)

    For $i = 1 To $iSize
        If DllStructGetData($tBuffer1, 1, $i) <> DllStructGetData($tBuffer2, 1, $i) Then
            Return False
        EndIf
    Next

    Return True
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_Copy
; Description ...: Copies memory from one location to another
; Parameters ....: $ahHandle - Process handle
;                  $iSourceAddress - Source address
;                  $iDestAddress - Destination address
;                  $iSize - Size to copy in bytes
; Return values .: Success: True
;                  Failure: False, @error set
; ===============================================================================================================================
Func _Memory_Copy($ahHandle, $iSourceAddress, $iDestAddress, $iSize)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, False)

    Local $tBuffer = DllStructCreate("byte[" & $iSize & "]")

    Local $aCall = DllCall($ahHandle[0], "bool", "ReadProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iSourceAddress, _
            "ptr", DllStructGetPtr($tBuffer), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, False)

    $aCall = DllCall($ahHandle[0], "bool", "WriteProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iDestAddress, _
            "ptr", DllStructGetPtr($tBuffer), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(3, 0, False)

    Return True
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_Fill
; Description ...: Fills memory region with specified byte value
; Parameters ....: $ahHandle - Process handle
;                  $iAddress - Memory address
;                  $iSize - Size in bytes
;                  $iByte - Byte value to fill (0-255)
; Return values .: Success: True
;                  Failure: False, @error set
; ===============================================================================================================================
Func _Memory_Fill($ahHandle, $iAddress, $iSize, $iByte = 0)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, False)
    If $iSize <= 0 Then Return SetError(3, 0, False)

    ; Create a buffer with every byte set to the specified value
    Local $tBuffer = DllStructCreate("byte[" & $iSize & "]")

    ; Use DLLCall with memset for better performance when filling the buffer
    DllCall("msvcrt.dll", "ptr", "memset", "ptr", DllStructGetPtr($tBuffer), "int", $iByte, "size_t", $iSize)

    ; Write the filled buffer into the target memory region
    Local $aCall = DllCall($ahHandle[0], "bool", "WriteProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iAddress, _
            "ptr", DllStructGetPtr($tBuffer), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, False)

    Return True
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_DumpRegion
; Description ...: Dumps memory region to hex string
; Parameters ....: $ahHandle - Process handle
;                  $iAddress - Memory address
;                  $iSize - Size to dump in bytes
; Return values .: Success: Hex string
;                  Failure: "", @error set
; ===============================================================================================================================
Func _Memory_DumpRegion($ahHandle, $iAddress, $iSize)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, "")

    Local $tBuffer = DllStructCreate("byte[" & $iSize & "]")

    Local $aCall = DllCall($ahHandle[0], "bool", "ReadProcessMemory", _
            "handle", $ahHandle[1], "ptr", $iAddress, _
            "ptr", DllStructGetPtr($tBuffer), "ulong_ptr", $iSize, "ulong_ptr*", 0)

    If @error Or Not $aCall[0] Then Return SetError(2, 0, "")

    Local $sHex = ""
    For $i = 1 To $iSize
        $sHex &= Hex(DllStructGetData($tBuffer, 1, $i), 2) & " "
    Next

    Return StringTrimRight($sHex, 1)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_FindDifferences
; Description ...: Finds differences between two scans (useful for value changes)
; Parameters ....: $aSnapshot1 - First snapshot array
;                  $aSnapshot2 - Second snapshot array
; Return values .: Success: Array of addresses that changed
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_FindDifferences($aSnapshot1, $aSnapshot2)
    If Not IsArray($aSnapshot1) Or Not IsArray($aSnapshot2) Then Return SetError(1, 0, 0)
    If UBound($aSnapshot1) <> UBound($aSnapshot2) Then Return SetError(2, 0, 0)

    Local $aDifferences[1]
    Local $iCount = 0

    For $i = 0 To UBound($aSnapshot1) - 1
        If $aSnapshot1[$i] <> $aSnapshot2[$i] Then
            ReDim $aDifferences[$iCount + 1]
            $aDifferences[$iCount] = $i
            $iCount += 1
        EndIf
    Next

    If $iCount = 0 Then Return SetError(3, 0, 0)
    Return $aDifferences
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Memory_GetPageInfo
; Description ...: Gets information about a memory page
; Parameters ....: $ahHandle - Process handle
;                  $iAddress - Memory address
; Return values .: Success: Array [BaseAddress, AllocationBase, AllocationProtect, RegionSize, State, Protect, Type]
;                  Failure: 0, @error set
; ===============================================================================================================================
Func _Memory_GetPageInfo($ahHandle, $iAddress)
    If Not IsArray($ahHandle) Then Return SetError(1, 0, 0)

    Local $tMBI = DllStructCreate("ptr BaseAddress;" & _
            "ptr AllocationBase;" & _
            "dword AllocationProtect;" & _
            "ulong_ptr RegionSize;" & _
            "dword State;" & _
            "dword Protect;" & _
            "dword Type")

    Local $aCall = DllCall($ahHandle[0], "ulong_ptr", "VirtualQueryEx", _
            "handle", $ahHandle[1], _
            "ptr", $iAddress, _
            "ptr", DllStructGetPtr($tMBI), _
            "ulong_ptr", DllStructGetSize($tMBI))

    If @error Or $aCall[0] = 0 Then Return SetError(2, 0, 0)

    Local $aInfo[7]
    $aInfo[0] = DllStructGetData($tMBI, "BaseAddress")
    $aInfo[1] = DllStructGetData($tMBI, "AllocationBase")
    $aInfo[2] = DllStructGetData($tMBI, "AllocationProtect")
    $aInfo[3] = DllStructGetData($tMBI, "RegionSize")
    $aInfo[4] = DllStructGetData($tMBI, "State")
    $aInfo[5] = DllStructGetData($tMBI, "Protect")
    $aInfo[6] = DllStructGetData($tMBI, "Type")
    Return $aInfo
EndFunc

Func StringRepeat($sString, $iRepeatCount)
    ; Ensure $iRepeatCount is an integer
    $iRepeatCount = Int($iRepeatCount)

    ; Handle special cases
    If $iRepeatCount = 0 Then Return "" ; Return empty string when repeating zero times
    If StringLen($sString) < 1 Or $iRepeatCount < 0 Then Return SetError(1, 0, "")
    If $iRepeatCount = 1 Then Return $sString ; Return original string when repeating once

    ; Use dynamic approach to optimize performance
    Local $sResult = ""

    ; Binary exponentiation approach
    While $iRepeatCount > 0
        If BitAND($iRepeatCount, 1) Then $sResult &= $sString ; If lowest bit is set, append string to result
        $sString &= $sString ; Double the string
        $iRepeatCount = BitShift($iRepeatCount, 1) ; Shift right (divide by 2)
    WEnd

    Return $sResult
EndFunc   ;==>StringRepeat



; ===============================================================================================================================
; EXAMPLE CODE (Commented)
; ===============================================================================================================================
#cs
; Enable debug privilege
_Memory_SetPrivilege("SeDebugPrivilege", True)

; Example 1: Read memory from external process
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $iHealth = _Memory_Read($ahHandle, 0x12345678, "int")
    ConsoleWrite("Health: " & $iHealth & @CRLF)
    _Memory_Close($ahHandle)
EndIf

; Example 2: Read pointer chain
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, "game.dll")
    Local $aOffsets[3] = [0x28, 0x1D8, 0x6C0]
    Local $iValue = _Memory_ReadPointer($ahHandle, $iModuleBase + 0x123456, $aOffsets)
    ConsoleWrite("Value: " & $iValue & @CRLF)
    _Memory_Close($ahHandle)
EndIf

; Example 3: Execute inline assembly (return 42)
; MOV EAX, 42 (B8 2A000000) + RET (C3)
Local $iResult = _ASM_QuickExecute("B82A000000C3")
ConsoleWrite("Result: " & $iResult & @CRLF)

; Example 4: Inject code into remote process
Local $ahHandle = _Memory_Open("target.exe")
If Not @error Then
    ; Simple NOP + RET code
    _ASM_Inject($ahHandle, "90909090C3", True)
    _Memory_Close($ahHandle)
EndIf

; Example 5: Create code cave
Local $ahHandle = _Memory_Open("target.exe")
If Not @error Then
    _ASM_CreateCodeCave($ahHandle, 0x00401000, 100) ; Fill 100 bytes with NOPs
    _Memory_Close($ahHandle)
EndIf

; Example 6: Quick operations
Local $iHP = _Mem_QuickRead("game.exe", 0x12345678, "int")
_Mem_QuickWrite("game.exe", 0x12345678, 999, "int")

#ce

; ===============================================================================================================================
; MORE EXAMPLES
; ===============================================================================================================================
#cs

; Example 7: Hook a function
_Memory_SetPrivilege("SeDebugPrivilege")
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; Hook function at 0x00401000 to jump to 0x00402000
    Local $sOriginalBytes = _ASM_HookFunction($ahHandle, 0x00401000, 0x00402000)

    ; ... do something ...

    ; Unhook
    _ASM_UnhookFunction($ahHandle, 0x00401000, $sOriginalBytes)
    _Memory_Close($ahHandle)
EndIf

; Example 8: Pattern scanning
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, "game.dll")
    Local $iAddress = _Memory_PatternScan($ahHandle, $iModuleBase, 0x100000, "8B 0D ?? ?? ?? ?? 85 C9")

    If Not @error Then
        ConsoleWrite("Pattern found at: 0x" & Hex($iAddress) & @CRLF)
    EndIf

    _Memory_Close($ahHandle)
EndIf

; Example 9: List all modules
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $aModules = _Memory_GetProcessModules($ahHandle)

    For $i = 0 To UBound($aModules) - 1
        ConsoleWrite($aModules[$i][0] & " @ 0x" & Hex($aModules[$i][1]) & @CRLF)
    Next

    _Memory_Close($ahHandle)
EndIf

; Example 10: Create custom assembly
Local $sCode = ""
$sCode &= _ASM_CreatePushad()              ; Save all registers
$sCode &= _ASM_CreateMov($EAX, 1337)       ; MOV EAX, 1337
$sCode &= _ASM_CreateAdd($EAX, 100)        ; ADD EAX, 100
$sCode &= _ASM_CreatePopad()               ; Restore all registers
$sCode &= _ASM_CreateRet()                 ; RET

Local $iResult = _ASM_QuickExecute($sCode)
ConsoleWrite("Result: " & $iResult & @CRLF) ; Should be 1437

; Example 11: Inject custom code
Local $ahHandle = _Memory_Open("target.exe")
If Not @error Then
    Local $sInjectedCode = ""
    $sInjectedCode &= _ASM_CreatePush(0)           ; PUSH 0 (uType)
    $sInjectedCode &= _ASM_CreatePush(0x00401234)  ; PUSH caption_ptr
    $sInjectedCode &= _ASM_CreatePush(0x00401244)  ; PUSH text_ptr
    $sInjectedCode &= _ASM_CreatePush(0)           ; PUSH hWnd (NULL)
    $sInjectedCode &= _ASM_CreateMov($EAX, 0x76543210) ; MessageBoxA address
    $sInjectedCode &= "FFD0"                       ; CALL EAX
    $sInjectedCode &= _ASM_CreateRet()             ; RET

    _ASM_Inject($ahHandle, $sInjectedCode)
    _Memory_Close($ahHandle)
EndIf

; Example 12: Search for string in memory
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, "game.exe")
    Local $iModuleSize = _Memory_GetModuleSize($ahHandle, "game.exe")

    ; Search for ASCII string
    Local $iAddress = _Memory_StringSearch($ahHandle, $iModuleBase, $iModuleSize, "PlayerName", False, True)
    If Not @error Then
        ConsoleWrite("String found at: 0x" & Hex($iAddress) & @CRLF)
    EndIf

    ; Search for Unicode string
    $iAddress = _Memory_StringSearch($ahHandle, $iModuleBase, $iModuleSize, "WeaponName", True, True)
    If Not @error Then
        ConsoleWrite("Unicode string found at: 0x" & Hex($iAddress) & @CRLF)
    EndIf

    _Memory_Close($ahHandle)
EndIf

; Example 13: Search for integer value
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, "game.exe")

    ; Search for health value (e.g., 100)
    Local $aAddresses = _Memory_IntegerSearchAll($ahHandle, $iModuleBase, 0x100000, 100, "int", 10)

    If Not @error Then
        ConsoleWrite("Found " & UBound($aAddresses) & " occurrences:" & @CRLF)
        For $i = 0 To UBound($aAddresses) - 1
            ConsoleWrite("  [" & $i & "] 0x" & Hex($aAddresses[$i]) & @CRLF)
        Next
    EndIf

    _Memory_Close($ahHandle)
EndIf

; Example 14: Search for float value
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, "game.exe")

    ; Search for speed multiplier (e.g., 1.5)
    Local $iAddress = _Memory_FloatSearch($ahHandle, $iModuleBase, 0x100000, 1.5, False)

    If Not @error Then
        ConsoleWrite("Float found at: 0x" & Hex($iAddress) & @CRLF)
    EndIf

    _Memory_Close($ahHandle)
EndIf

; Example 15: AOB (Array of Bytes) scan
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; Find game function signature
    Local $sAOB = "55 8B EC 83 EC ?? 53 56 57 8B F9"
    Local $iAddress = _Memory_ScanRegion($ahHandle, "game.exe", $sAOB)

    If Not @error Then
        ConsoleWrite("Function found at: 0x" & Hex($iAddress) & @CRLF)
    EndIf

    _Memory_Close($ahHandle)
EndIf

; Example 16: Read/Write strings
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; Read string from memory
    Local $sName = _Memory_ReadString($ahHandle, 0x12345678, 256, False)
    ConsoleWrite("Player name: " & $sName & @CRLF)

    ; Write new string
    _Memory_WriteString($ahHandle, 0x12345678, "NewPlayerName", False, True)

    _Memory_Close($ahHandle)
EndIf

; Example 17: Read/Write arrays
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; Read array of 10 integers
    Local $aInventory = _Memory_ReadArray($ahHandle, 0x12345678, 10, "int")

    ; Modify array
    For $i = 0 To UBound($aInventory) - 1
        $aInventory[$i] = 999
    Next

    ; Write back
    _Memory_WriteArray($ahHandle, 0x12345678, $aInventory, "int")

    _Memory_Close($ahHandle)
EndIf

; Example 18: Memory freeze (continuously write value)
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; Freeze health at 999
    AdlibRegister("_FreezeHealth", 100)

    ; Wait for 10 seconds
    Sleep(10000)

    AdlibUnRegister("_FreezeHealth")
    _Memory_Close($ahHandle)
EndIf

Func _FreezeHealth()
    Global $ahHandle
    _Memory_Freeze($ahHandle, 0x12345678, 999, "int")
EndFunc

; Example 19: Copy memory region
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; Copy 100 bytes from one location to another
    _Memory_Copy($ahHandle, 0x12345678, 0x12346000, 100)

    _Memory_Close($ahHandle)
EndIf

; Example 20: Fill memory with NOPs
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; Fill 10 bytes with NOP (0x90)
    _Memory_Fill($ahHandle, 0x00401000, 10, 0x90)

    _Memory_Close($ahHandle)
EndIf

; Example 21: Dump memory to hex
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $sHexDump = _Memory_DumpRegion($ahHandle, 0x00401000, 32)
    ConsoleWrite("Hex dump:" & @CRLF & $sHexDump & @CRLF)

    _Memory_Close($ahHandle)
EndIf

; Example 22: Compare two memory regions
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    If _Memory_Compare($ahHandle, 0x12345678, 0x12346000, 100) Then
        ConsoleWrite("Memory regions are identical" & @CRLF)
    Else
        ConsoleWrite("Memory regions differ" & @CRLF)
    EndIf

    _Memory_Close($ahHandle)
EndIf

; Example 23: Get memory page information
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $aPageInfo = _Memory_GetPageInfo($ahHandle, 0x00401000)

    If Not @error Then
        ConsoleWrite("Base Address: 0x" & Hex($aPageInfo[0]) & @CRLF)
        ConsoleWrite("Region Size: " & $aPageInfo[3] & " bytes" & @CRLF)
        ConsoleWrite("Protection: 0x" & Hex($aPageInfo[5]) & @CRLF)
    EndIf

    _Memory_Close($ahHandle)
EndIf

; Example 24: Change detection (find changed values)
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; First scan
    Local $aSnapshot1 = _Memory_ReadArray($ahHandle, 0x12345000, 1000, "int")

    ; Wait for value change (e.g., player does something)
    Sleep(5000)

    ; Second scan
    Local $aSnapshot2 = _Memory_ReadArray($ahHandle, 0x12345000, 1000, "int")

    ; Find differences
    Local $aDifferences = _Memory_FindDifferences($aSnapshot1, $aSnapshot2)

    If Not @error Then
        ConsoleWrite("Found " & UBound($aDifferences) & " changed values" & @CRLF)
        For $i = 0 To UBound($aDifferences) - 1
            Local $iOffset = $aDifferences[$i]
            ConsoleWrite("  Offset " & $iOffset & ": " & $aSnapshot1[$iOffset] & " -> " & $aSnapshot2[$iOffset] & @CRLF)
        Next
    EndIf

    _Memory_Close($ahHandle)
EndIf

; Example 25: Multi-level pointer with dynamic base
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; Get dynamic module base
    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, "game.dll")

    ; Static offset from module
    Local $iStaticOffset = 0x123456

    ; Pointer chain
    Local $aOffsets[4] = [0x28, 0x1D8, 0x6C0, 0x14]

    ; Read final value
    Local $iValue = _Memory_ReadPointer($ahHandle, $iModuleBase + $iStaticOffset, $aOffsets, "int")
    ConsoleWrite("Final value: " & $iValue & @CRLF)

    ; Write new value
    _Memory_WritePointer($ahHandle, $iModuleBase + $iStaticOffset, $aOffsets, 9999, "int")

    _Memory_Close($ahHandle)
EndIf

#ce