# MemoryUDF - AutoIt Memory Management & Assembly Library

A comprehensive AutoIt User Defined Function (UDF) library for advanced memory manipulation and inline assembly execution. This library provides powerful tools for reading/writing process memory, pointer chain traversal, pattern scanning, and assembly code injection.

## Table of Contents

1. [Features](#features)
2. [Requirements](#requirements)
3. [Installation](#installation)
4. [Quick Start](#quick-start)
5. [Detailed Examples](#detailed-examples)
6. [Function Reference Overview](#function-reference-overview)
7. [Data Types](#data-types)
8. [Constants](#constants)
9. [Error Handling](#error-handling)
10. [Best Practices](#best-practices)
11. [Security Considerations](#security-considerations)
12. [Compatibility](#compatibility)
13. [Author](#author)
14. [License](#license)

## Features

### Memory Operations
- **Process Memory Access**: Read/write memory from external processes
- **Pointer Chain Support**: Navigate complex pointer structures with x86/x64 compatibility
- **Module Management**: Get base addresses and sizes of loaded modules
- **Memory Protection**: Change memory protection flags
- **Memory Utilities**: Copy, fill, compare, and dump memory regions

### Assembly & Code Injection
- **Inline Assembly**: Compile and execute machine code directly
- **Code Injection**: Inject assembly code into remote processes
- **Function Hooking**: Hook and unhook functions with jump patches
- **Code Cave Creation**: Create NOP sleds for code modification
- **Assembly Helpers**: Generate common x86 instructions programmatically

### Pattern Scanning & Search
- **Pattern Scanning**: Search for byte patterns with wildcard support
- **String Search**: Find ASCII/Unicode strings in memory (case-sensitive/insensitive)
- **Value Search**: Search for integers, floats, and hex sequences
- **AOB Scanning**: Array of Bytes scanning with wildcard support
- **Region Scanning**: Scan entire module memory regions

### Advanced Features
- **Memory Snapshots**: Compare memory states to detect changes
- **Array Operations**: Read/write arrays of values efficiently
- **String Operations**: Handle null-terminated strings (ASCII/Unicode)
- **Memory Freezing**: Continuously write values to addresses
- **Page Information**: Query memory page properties

## Requirements

- **AutoIt Version**: 3.3.14+
- **Operating System**: Windows (x86/x64)
- **Required DLLs**: Kernel32.dll, Psapi.dll, User32.dll
- **Privileges**: SeDebugPrivilege recommended for external process access

## Installation

1. Download `MemoryUDF.au3`
2. Include in your AutoIt script:
```autoit
#include "MemoryUDF.au3"
```

## Quick Start

### Basic Memory Reading

> Goal: Attach to a target process, read a value, then safely release resources.
```autoit
; Enable debug privilege for external process access
_Memory_SetPrivilege("SeDebugPrivilege", True)

; Open process handle
Local $ahHandle = _Memory_Open("notepad.exe")
If Not @error Then
    ; Read a 4-byte integer from memory
    Local $iValue = _Memory_Read($ahHandle, 0x12345678, "int")
    ConsoleWrite("Value: " & $iValue & @CRLF)
    
    ; Close handle when done
    _Memory_Close($ahHandle)
EndIf
```

### Pointer Chain Navigation

> Goal: Resolve multi-level pointers using module base + offsets.
```autoit
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    ; Get module base address
    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, "game.dll")
    
    ; Define pointer chain offsets
    Local $aOffsets[3] = [0x28, 0x1D8, 0x6C0]
    
    ; Read value through pointer chain
    Local $iValue = _Memory_ReadPointer($ahHandle, $iModuleBase + 0x123456, $aOffsets, "int")
    ConsoleWrite("Player Health: " & $iValue & @CRLF)
    
    _Memory_Close($ahHandle)
EndIf
```

### Assembly Code Execution

> Goal: Run lightweight machine code directly from AutoIt without touching external processes.
```autoit
; Execute inline assembly (MOV EAX, 42; RET)
Local $iResult = _ASM_QuickExecute("B82A000000C3")
ConsoleWrite("Assembly result: " & $iResult & @CRLF) ; Output: 42
```

### Pattern Scanning

> Goal: Locate a signature in memory to build reliable offsets or hooks.
```autoit
Local $ahHandle = _Memory_Open("game.exe")
If Not @error Then
    Local $iModuleBase = _Memory_GetModuleBaseAddress($ahHandle, "game.exe")
    
    ; Search for byte pattern with wildcards
    Local $iAddress = _Memory_PatternScan($ahHandle, $iModuleBase, 0x100000, "8B 0D ?? ?? ?? ?? 85 C9")
    
    If Not @error Then
        ConsoleWrite("Pattern found at: 0x" & Hex($iAddress) & @CRLF)
    EndIf
    
    _Memory_Close($ahHandle)
EndIf
```

## Detailed Examples

### Remote Code Injection Workflow

```autoit
_Memory_SetPrivilege("SeDebugPrivilege", True)

Local $ahHandle = _Memory_Open("game.exe", BitOR($PROCESS_VM_OPERATION, $PROCESS_VM_WRITE, $PROCESS_VM_READ))
If @error Then Exit ConsoleWrite("Failed to open process" & @CRLF)

Local $sShellcode = "B833000000C3" ; MOV EAX, 51 / RET
Local $iExit = _ASM_Inject($ahHandle, $sShellcode)
If Not @error Then ConsoleWrite("Remote thread exited with: " & $iExit & @CRLF)

_Memory_Close($ahHandle)
```

### Code Cave Creation & Hooking

```autoit
Local $ahHandle = _Memory_Open("target.exe")
Local $sOriginal = _ASM_HookFunction($ahHandle, 0x401000, 0x500000, 3)

If Not @error Then
    ConsoleWrite("Hook installed" & @CRLF)
    ; ... perform operations ...
    _ASM_UnhookFunction($ahHandle, 0x401000, $sOriginal)
EndIf

_Memory_Close($ahHandle)
```

### Quick Read / Quick Write Helpers

```autoit
Local $iHealth = _Mem_QuickRead("game.exe", 0x123456, "int")
If Not @error Then ConsoleWrite("Current Health: " & $iHealth & @CRLF)

_Mem_QuickWrite("game.exe", 0x123456, 999, "int")
```

### Memory Freezing Pattern

```autoit
Func FreezeHealth()
    _Memory_Write($g_ahGame, $g_iHealthAddress, 999, "int")
EndFunc

$g_ahGame = _Memory_Open("game.exe")
$g_iHealthAddress = 0x123456

AdlibRegister("FreezeHealth", 250)
; Remember to call AdlibUnRegister and _Memory_Close when done
```

### Pattern Scan with Wildcards

```autoit
Local $ahHandle = _Memory_Open("game.exe")
Local $iAddress = _Memory_PatternScan($ahHandle, 0x00400000, 0x00200000, "55 8B EC ?? ?? 5D C3")

If Not @error Then ConsoleWrite("Signature located at 0x" & Hex($iAddress) & @CRLF)
_Memory_Close($ahHandle)
```

## Function Reference Overview

The UDF follows the AutoIt documentation standard. Each function header inside `MemoryUDF.au3` exposes:

- **Syntax**: How to call the function, including optional parameters.
- **Parameters**: Argument descriptions, accepted types, and defaults.
- **Return Values**: Expected success payload and detailed `@error` information.
- **Remarks**: Implementation notes, architecture hints, and safety guidance.
- **Related**: Links to complementary helper functions.
- **Example**: Inline usage snippet or the README section to consult.

> Tip: Open `MemoryUDF.au3` directly to view the complete per-function documentation injected into the header blocks.

### Functional Categories

| Category | Key Functions | Notes |
|----------|---------------|-------|
| Memory Handles | `_Memory_Open`, `_Memory_Close`, `_Memory_SetPrivilege` | Always close handles to avoid leaks. |
| Pointer & Module Utilities | `_Memory_ReadPointer`, `_Memory_CalculatePointer`, `_Memory_GetModuleBaseAddress` | Pointer helpers auto-adapt to x86/x64. |
| Read/Write Helpers | `_Memory_Read`, `_Memory_Write`, `_Memory_ReadString` | Support a variety of data types and array operations. |
| Assembly Runtime | `_ASM_Compile`, `_ASM_Execute`, `_ASM_Inject`, `_ASM_QuickExecute` | Combine to compile, run, or inject custom shellcode. |
| Pattern & Value Scans | `_Memory_PatternScan`, `_Memory_StringSearch`, `_Memory_IntegerSearch`, `_Memory_FloatSearch` | Provide wildcard and case-insensitive options. |
| Hooking & Code Caves | `_ASM_HookFunction`, `_ASM_CreateJump`, `_ASM_UnhookFunction` | Ensure you store original bytes for safe removal. |

Consult the in-file headers for exhaustive parameter lists and `@error` codes per function.

## Data Types

Supported data types for memory operations:
- `"byte"` - 1 byte (0-255)
- `"word"`, `"short"` - 2 bytes
- `"int"`, `"dword"` - 4 bytes (default)
- `"int64"`, `"uint64"` - 8 bytes
- `"float"` - 4-byte floating point
- `"double"` - 8-byte floating point
- `"ptr"` - Pointer size (4 bytes on x86, 8 bytes on x64)

## Constants

### Process Access Rights
- `$PROCESS_ALL_ACCESS` - Full access rights
- `$PROCESS_VM_READ` - Read memory access
- `$PROCESS_VM_WRITE` - Write memory access
- `$PROCESS_VM_OPERATION` - Memory operation access

### Memory Protection
- `$PAGE_EXECUTE_READWRITE` - Execute, read, and write access
- `$MEM_COMMIT` - Commit memory pages
- `$MEM_RESERVE` - Reserve memory pages
- `$MEM_RELEASE` - Release memory pages

### Assembly Registers
- **8-bit**: `$AL`, `$CL`, `$DL`, `$BL`, `$AH`, `$CH`, `$DH`, `$BH`
- **16-bit**: `$AX`, `$CX`, `$DX`, `$BX`, `$SP`, `$BP`, `$SI`, `$DI`
- **32-bit**: `$EAX`, `$ECX`, `$EDX`, `$EBX`, `$ESP`, `$EBP`, `$ESI`, `$EDI`
- **64-bit**: `$RAX`, `$RCX`, `$RDX`, `$RBX`, `$RSP`, `$RBP`, `$RSI`, `$RDI`, `$R8`-`$R15`

## Error Handling

All functions use AutoIt's `@error` system for error reporting:
- `@error = 0` - Success
- `@error > 0` - Error occurred (check function documentation for specific error codes)

Always check `@error` after function calls:
```autoit
Local $iValue = _Memory_Read($ahHandle, $iAddress, "int")
If @error Then
    ConsoleWrite("Error reading memory: " & @error & @CRLF)
Else
    ConsoleWrite("Value: " & $iValue & @CRLF)
EndIf
```

## Best Practices

1. **Enable Debug Privilege**: Call `_Memory_SetPrivilege("SeDebugPrivilege", True)` before accessing external processes
2. **Handle Cleanup**: Always call `_Memory_Close()` to free resources
3. **Check Errors**: Verify `@error` after each function call
4. **Use Appropriate Types**: Choose the correct data type for your memory operations
5. **Validate Addresses**: Ensure memory addresses are valid before access
6. **Test Patterns**: Verify pattern strings are correctly formatted with spaces

## Security Considerations

- This library requires elevated privileges for external process access
- Memory manipulation can cause application crashes or system instability
- Always validate input parameters and memory addresses
- Use appropriate error handling to prevent unexpected behavior
- Be cautious when injecting code into critical system processes

## Compatibility

- **Architecture**: Supports both x86 and x64 processes
- **AutoIt**: Compatible with AutoIt 3.3.14 and later versions
- **Windows**: Works on Windows Vista and later versions
- **Processes**: Can access both 32-bit and 64-bit processes (with appropriate AutoIt version)

## Author

**Dao Van Trong** - TRONG.PRO

## License

This UDF is provided as-is for educational and development purposes. Use responsibly and in accordance with applicable laws and regulations.
