#include-once
#include <Array.au3>

; ===============================================================================
; Title .........: HardwareMonitor UDF
; Description ...: AutoIt wrapper for HardwareMonitor.dll
;                  Complete hardware monitoring with 65 functions
; Author ........: Dao Van Trong - TRONG.PRO
; Version .......: 2.0.0
; Functions .....: 65 total
;                  - Core: 3 functions (Initialize, Shutdown, GetLastError)
;                  - Intel CPU: 4 functions
;                  - AMD CPU: 3 functions
;                  - NVIDIA GPU: 13 functions
;                  - AMD Radeon GPU: 13 functions
;                  - Storage: 4 functions
;                  - RAM Monitor: 10 functions
;                  - Battery Monitor: 15 functions
;                  - Motherboard Monitor: 14 functions (71 chips supported)
; ===============================================================================

; ===============================================================================
; Global Variables
; ===============================================================================
Global $__g_hHWMonDLL = -1
Global $__g_sHWMonDLLPath = ""
Global $__g_bHWMonInitialized = False

; Module info struct (matches C++ HWMonModuleInfo exactly)
; All bool fields are uint8_t (byte), all counts are uint32_t (dword)
Global Const $tagHWMonModuleInfo = "align 1;" & _
    "byte isInitialized;" & _
    "byte hasPawnIO;" & _
    "byte hasIntelCPU;" & _
    "byte hasAMDCPU;" & _
    "dword amdCPUFamily;" & _
    "byte hasNVIDIAGPU;" & _
    "dword nvidiaGPUCount;" & _
    "byte hasAMDRadeonGPU;" & _
    "dword amdRadeonGPUCount;" & _
    "byte hasStorage;" & _
    "dword storageCount"

; ===============================================================================
; Core Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Startup
; Description ...: Initialize the HardwareMonitor library
; Syntax ........: _HWMon_Startup([$sDLLPath = ""])
; Parameters ....: $sDLLPath - [optional] Path to HardwareMonitor.dll
; Return values .: Success - True
;                  Failure - False, sets @error
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Startup($sDLLPath = "")
    If $__g_bHWMonInitialized Then Return True

    ; Determine DLL path
    If ($sDLLPath = "") Then
		$sDLLPath = @ScriptDir & "\HardwareMonitor.dll"
    EndIf

    ; Check if DLL exists
    If Not FileExists($sDLLPath) Then
        Return SetError(1, 0, False) ; DLL not found
    EndIf

    ; Open DLL
    $__g_hHWMonDLL = DllOpen($sDLLPath)
    If ($__g_hHWMonDLL = -1) Then
        Return SetError(2, 0, False) ; Failed to open DLL
    EndIf

    $__g_sHWMonDLLPath = $sDLLPath

    ; Initialize library
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Initialize")
    If @error Or ($aResult[0] <> 0) Then
        DllClose($__g_hHWMonDLL)
        $__g_hHWMonDLL = -1
        Return SetError(3, 0, False) ; Initialization failed
    EndIf

    $__g_bHWMonInitialized = True
    Return True
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Shutdown
; Description ...: Cleanup and close the HardwareMonitor library
; Syntax ........: _HWMon_Shutdown()
; Return values .: None
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Shutdown()
    If Not $__g_bHWMonInitialized Then Return

    ; Cleanup
    DllCall($__g_hHWMonDLL, "none", "HWMon_Cleanup")

    ; Close DLL
    DllClose($__g_hHWMonDLL)
    $__g_hHWMonDLL = -1
    $__g_bHWMonInitialized = False
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_GetModuleInfo
; Description ...: Get information about available hardware modules
; Syntax ........: _HWMon_GetModuleInfo()
; Return values .: Success - Array with module information
;                  Failure - False, sets @error
; Author ........: Dao Van Trong - TRONG.PRO
; Remarks .......: Array format:
;                  [0] = isInitialized
;                  [1] = hasPawnIO
;                  [2] = hasIntelCPU
;                  [3] = hasAMDCPU
;                  [4] = amdCPUFamily
;                  [5] = hasNVIDIAGPU
;                  [6] = nvidiaGPUCount
;                  [7] = hasAMDRadeonGPU
;                  [8] = amdRadeonGPUCount
;                  [9] = hasStorage
;                  [10] = storageCount
; ===============================================================================
Func _HWMon_GetModuleInfo()
    If Not $__g_bHWMonInitialized Then Return SetError(1, 0, False)

    ; Create struct
    Local $tInfo = DllStructCreate($tagHWMonModuleInfo)
    If @error Then Return SetError(2, 0, False)

    ; Call DLL function with pointer (returns int: 1=success, 0=fail)
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_GetModuleInfo", "ptr", DllStructGetPtr($tInfo))

    If @error Then
        Return SetError(3, @error, False)
    EndIf

    If Not $aResult[0] Then
        Return SetError(4, 0, False)
    EndIf

    ; Read struct data with error checking
    Local $aInfo[11]
    $aInfo[0] = DllStructGetData($tInfo, "isInitialized")
    If @error Then $aInfo[0] = 0

    $aInfo[1] = DllStructGetData($tInfo, "hasPawnIO")
    If @error Then $aInfo[1] = 0

    $aInfo[2] = DllStructGetData($tInfo, "hasIntelCPU")
    If @error Then $aInfo[2] = 0

    $aInfo[3] = DllStructGetData($tInfo, "hasAMDCPU")
    If @error Then $aInfo[3] = 0

    $aInfo[4] = DllStructGetData($tInfo, "amdCPUFamily")
    If @error Then $aInfo[4] = 0

    $aInfo[5] = DllStructGetData($tInfo, "hasNVIDIAGPU")
    If @error Then $aInfo[5] = 0

    $aInfo[6] = DllStructGetData($tInfo, "nvidiaGPUCount")
    If @error Then $aInfo[6] = 0

    $aInfo[7] = DllStructGetData($tInfo, "hasAMDRadeonGPU")
    If @error Then $aInfo[7] = 0

    $aInfo[8] = DllStructGetData($tInfo, "amdRadeonGPUCount")
    If @error Then $aInfo[8] = 0

    $aInfo[9] = DllStructGetData($tInfo, "hasStorage")
    If @error Then $aInfo[9] = 0

    $aInfo[10] = DllStructGetData($tInfo, "storageCount")
    If @error Then $aInfo[10] = 0

    Return $aInfo
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_GetLastError
; Description ...: Get the last error message
; Syntax ........: _HWMon_GetLastError()
; Return values .: Error message string
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_GetLastError()
    If Not $__g_bHWMonInitialized Then Return ""

    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_GetLastError")
    If @error Then Return ""

    Return $aResult[0]
EndFunc

; ===============================================================================
; Intel CPU Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Intel_GetCoreCount
; Description ...: Get Intel CPU core count
; Syntax ........: _HWMon_Intel_GetCoreCount()
; Return values .: Success - Number of cores
;                  Failure - 0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Intel_GetCoreCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Intel_GetCoreCount")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Intel_GetPackageTemp
; Description ...: Get Intel CPU package temperature
; Syntax ........: _HWMon_Intel_GetPackageTemp()
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Intel_GetPackageTemp()
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Update first
    DllCall($__g_hHWMonDLL, "bool", "HWMon_Intel_Update")

    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Intel_GetPackageTemp")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Intel_GetCoreTemp
; Description ...: Get Intel CPU core temperature
; Syntax ........: _HWMon_Intel_GetCoreTemp($iCoreIndex)
; Parameters ....: $iCoreIndex - Core index (0-based)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Intel_GetCoreTemp($iCoreIndex)
    If Not $__g_bHWMonInitialized Then Return 0.0

    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Intel_GetCoreTemp", "int", $iCoreIndex)
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Intel_GetAllCoreTemps
; Description ...: Get all Intel CPU core temperatures
; Syntax ........: _HWMon_Intel_GetAllCoreTemps()
; Return values .: Success - Array of temperatures
;                  Failure - Empty array
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Intel_GetAllCoreTemps()
    If Not $__g_bHWMonInitialized Then Return SetError(1, 0, 0)

    Local $iCoreCount = _HWMon_Intel_GetCoreCount()
    If $iCoreCount = 0 Then Return SetError(2, 0, 0)

    ; Update first
    DllCall($__g_hHWMonDLL, "bool", "HWMon_Intel_Update")

    Local $aTemps[$iCoreCount]
    For $i = 0 To $iCoreCount - 1
        $aTemps[$i] = _HWMon_Intel_GetCoreTemp($i)
    Next

    Return $aTemps
EndFunc

; ===============================================================================
; AMD CPU Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMD_GetTctlTemp
; Description ...: Get AMD CPU Tctl temperature
; Syntax ........: _HWMon_AMD_GetTctlTemp()
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_AMD_GetTctlTemp()
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Update first
    DllCall($__g_hHWMonDLL, "bool", "HWMon_AMD_Update")

    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_AMD_GetTctlTemp")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMD_GetCCDCount
; Description ...: Get AMD CPU CCD count
; Syntax ........: _HWMon_AMD_GetCCDCount()
; Return values .: Success - Number of CCDs
;                  Failure - 0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_AMD_GetCCDCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_AMD_GetCCDCount")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMD_GetCCDTemp
; Description ...: Get AMD CPU CCD temperature
; Syntax ........: _HWMon_AMD_GetCCDTemp($iCCDIndex)
; Parameters ....: $iCCDIndex - CCD index (0-based)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_AMD_GetCCDTemp($iCCDIndex)
    If Not $__g_bHWMonInitialized Then Return 0.0

    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_AMD_GetCCDTemp", "int", $iCCDIndex)
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; ===============================================================================
; NVIDIA GPU Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetGPUCount
; Description ...: Get NVIDIA GPU count
; Syntax ........: _HWMon_NVIDIA_GetGPUCount()
; Return values .: Success - Number of GPUs
;                  Failure - 0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_NVIDIA_GetGPUCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_NVIDIA_EnumerateGPUs")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetGPUTemp
; Description ...: Get NVIDIA GPU temperature
; Syntax ........: _HWMon_NVIDIA_GetGPUTemp([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_NVIDIA_GetGPUTemp($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Select GPU
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)

    ; Update
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")

    ; Get temperature
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetGPUTemperature")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetGPUName
; Description ...: Get NVIDIA GPU name
; Syntax ........: _HWMon_NVIDIA_GetGPUName([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - GPU name
;                  Failure - ""
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_NVIDIA_GetGPUName($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return ""

    ; Select GPU
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)

    ; Get name
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_NVIDIA_GetGPUName")
    If @error Then Return ""

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetCoreClock
; Description ...: Get NVIDIA GPU core clock frequency
; Syntax ........: _HWMon_NVIDIA_GetCoreClock([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Clock frequency in MHz
;                  Failure - 0.0
; ===============================================================================
Func _HWMon_NVIDIA_GetCoreClock($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetCoreClock")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetMemoryClock
; Description ...: Get NVIDIA GPU memory clock frequency
; Syntax ........: _HWMon_NVIDIA_GetMemoryClock([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Clock frequency in MHz
;                  Failure - 0.0
; ===============================================================================
Func _HWMon_NVIDIA_GetMemoryClock($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetMemoryClock")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetShaderClock
; Description ...: Get NVIDIA GPU shader clock frequency
; Syntax ........: _HWMon_NVIDIA_GetShaderClock([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Clock frequency in MHz
;                  Failure - 0.0
; ===============================================================================
Func _HWMon_NVIDIA_GetShaderClock($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetShaderClock")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetFanSpeed
; Description ...: Get NVIDIA GPU fan speed in RPM
; Syntax ........: _HWMon_NVIDIA_GetFanSpeed([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Fan speed in RPM
;                  Failure - 0
; ===============================================================================
Func _HWMon_NVIDIA_GetFanSpeed($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_NVIDIA_GetFanSpeed")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetFanSpeedPercent
; Description ...: Get NVIDIA GPU fan speed in percentage
; Syntax ........: _HWMon_NVIDIA_GetFanSpeedPercent([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Fan speed in %
;                  Failure - 0.0
; ===============================================================================
Func _HWMon_NVIDIA_GetFanSpeedPercent($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetFanSpeedPercent")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetGPUUsage
; Description ...: Get NVIDIA GPU usage percentage
; Syntax ........: _HWMon_NVIDIA_GetGPUUsage([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - GPU usage in %
;                  Failure - 0.0
; ===============================================================================
Func _HWMon_NVIDIA_GetGPUUsage($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetGPUUsage")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetMemoryUsage
; Description ...: Get NVIDIA GPU memory controller usage percentage
; Syntax ........: _HWMon_NVIDIA_GetMemoryUsage([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Memory usage in %
;                  Failure - 0.0
; ===============================================================================
Func _HWMon_NVIDIA_GetMemoryUsage($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetMemoryUsage")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetVideoEngineUsage
; Description ...: Get NVIDIA GPU video engine usage percentage
; Syntax ........: _HWMon_NVIDIA_GetVideoEngineUsage([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Video engine usage in %
;                  Failure - 0.0
; ===============================================================================
Func _HWMon_NVIDIA_GetVideoEngineUsage($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetVideoEngineUsage")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetMemoryTotal
; Description ...: Get NVIDIA GPU total memory in bytes
; Syntax ........: _HWMon_NVIDIA_GetMemoryTotal([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Total memory in bytes
;                  Failure - 0
; ===============================================================================
Func _HWMon_NVIDIA_GetMemoryTotal($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "uint64", "HWMon_NVIDIA_GetMemoryTotal")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetMemoryUsed
; Description ...: Get NVIDIA GPU used memory in bytes
; Syntax ........: _HWMon_NVIDIA_GetMemoryUsed([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Used memory in bytes
;                  Failure - 0
; ===============================================================================
Func _HWMon_NVIDIA_GetMemoryUsed($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "uint64", "HWMon_NVIDIA_GetMemoryUsed")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetMemoryFree
; Description ...: Get NVIDIA GPU free memory in bytes
; Syntax ........: _HWMon_NVIDIA_GetMemoryFree([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Free memory in bytes
;                  Failure - 0
; ===============================================================================
Func _HWMon_NVIDIA_GetMemoryFree($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "uint64", "HWMon_NVIDIA_GetMemoryFree")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetPowerUsage
; Description ...: Get NVIDIA GPU power usage in Watts
; Syntax ........: _HWMon_NVIDIA_GetPowerUsage([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Power usage in Watts
;                  Failure - 0.0
; ===============================================================================
Func _HWMon_NVIDIA_GetPowerUsage($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetPowerUsage")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; ===============================================================================
; AMD Radeon GPU Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMDRadeon_GetGPUCount
; Description ...: Get AMD Radeon GPU count
; Syntax ........: _HWMon_AMDRadeon_GetGPUCount()
; Return values .: Success - Number of GPUs
;                  Failure - 0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_AMDRadeon_GetGPUCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_AMDRadeon_EnumerateGPUs")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMDRadeon_GetGPUTemp
; Description ...: Get AMD Radeon GPU temperature
; Syntax ........: _HWMon_AMDRadeon_GetGPUTemp([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_AMDRadeon_GetGPUTemp($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Select GPU
    DllCall($__g_hHWMonDLL, "bool", "HWMon_AMDRadeon_SelectGPU", "int", $iGPUIndex)

    ; Update
    DllCall($__g_hHWMonDLL, "bool", "HWMon_AMDRadeon_Update")

    ; Get temperature
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_AMDRadeon_GetGPUTemperature")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMDRadeon_GetGPUName
; Description ...: Get AMD Radeon GPU name
; Syntax ........: _HWMon_AMDRadeon_GetGPUName([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - GPU name
;                  Failure - ""
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_AMDRadeon_GetGPUName($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return ""

    ; Select GPU
    DllCall($__g_hHWMonDLL, "bool", "HWMon_AMDRadeon_SelectGPU", "int", $iGPUIndex)

    ; Get name
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_AMDRadeon_GetGPUName")
    If @error Then Return ""

    Return $aResult[0]
EndFunc

; ===============================================================================
; Storage Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Storage_GetDriveCount
; Description ...: Get storage drive count
; Syntax ........: _HWMon_Storage_GetDriveCount()
; Return values .: Success - Number of drives
;                  Failure - 0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Storage_GetDriveCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Storage_EnumerateDrives")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Storage_GetDriveTemp
; Description ...: Get storage drive temperature
; Syntax ........: _HWMon_Storage_GetDriveTemp($iDriveIndex)
; Parameters ....: $iDriveIndex - Drive index (0-based)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Storage_GetDriveTemp($iDriveIndex)
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Select drive
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Storage_SelectDrive", "int", $iDriveIndex)
    If @error Or Not $aResult[0] Then Return 0.0

    ; Update
    DllCall($__g_hHWMonDLL, "bool", "HWMon_Storage_Update")

    ; Get temperature
    $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Storage_GetTemperature")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Storage_GetDriveModel
; Description ...: Get storage drive model
; Syntax ........: _HWMon_Storage_GetDriveModel($iDriveIndex)
; Parameters ....: $iDriveIndex - Drive index (0-based)
; Return values .: Success - Drive model string
;                  Failure - ""
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Storage_GetDriveModel($iDriveIndex)
    If Not $__g_bHWMonInitialized Then Return ""

    ; Select drive
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Storage_SelectDrive", "int", $iDriveIndex)
    If @error Or Not $aResult[0] Then Return ""

    ; Get model
    $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Storage_GetDriveModel")
    If @error Then Return ""

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Storage_GetDriveType
; Description ...: Get storage drive type
; Syntax ........: _HWMon_Storage_GetDriveType($iDriveIndex)
; Parameters ....: $iDriveIndex - Drive index (0-based)
; Return values .: Success - Drive type string (HDD/SSD/NVMe)
;                  Failure - ""
; Author ........: Dao Van Trong - TRONG.PRO
; ===============================================================================
Func _HWMon_Storage_GetDriveType($iDriveIndex)
    If Not $__g_bHWMonInitialized Then Return ""

    ; Select drive
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Storage_SelectDrive", "int", $iDriveIndex)
    If @error Or Not $aResult[0] Then Return ""

    ; Get type
    $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Storage_GetDriveType")
    If @error Then Return ""

    Return $aResult[0]
EndFunc

; ===============================================================================
; RAM Monitor Functions (10 functions)
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_Initialize
; Description ...: Initialize RAM Monitor
; Syntax ........: _HWMon_RAM_Initialize()
; Return values .: Success - True
;                  Failure - False
; ===============================================================================
Func _HWMon_RAM_Initialize()
    If Not $__g_bHWMonInitialized Then Return False
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_RAM_Initialize")
    If @error Then Return False
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_Update
; Description ...: Update RAM Monitor data
; Syntax ........: _HWMon_RAM_Update()
; Return values .: Success - True
;                  Failure - False
; ===============================================================================
Func _HWMon_RAM_Update()
    If Not $__g_bHWMonInitialized Then Return False
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_RAM_Update")
    If @error Then Return False
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_GetTotalMemory
; Description ...: Get total physical memory in bytes
; Syntax ........: _HWMon_RAM_GetTotalMemory()
; Return values .: Total memory in bytes
; ===============================================================================
Func _HWMon_RAM_GetTotalMemory()
    If Not $__g_bHWMonInitialized Then Return 0
    _HWMon_RAM_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "uint64", "HWMon_RAM_GetTotalMemory")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_GetUsedMemory
; Description ...: Get used physical memory in bytes
; Syntax ........: _HWMon_RAM_GetUsedMemory()
; Return values .: Used memory in bytes
; ===============================================================================
Func _HWMon_RAM_GetUsedMemory()
    If Not $__g_bHWMonInitialized Then Return 0
    _HWMon_RAM_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "uint64", "HWMon_RAM_GetUsedMemory")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_GetAvailableMemory
; Description ...: Get available physical memory in bytes
; Syntax ........: _HWMon_RAM_GetAvailableMemory()
; Return values .: Available memory in bytes
; ===============================================================================
Func _HWMon_RAM_GetAvailableMemory()
    If Not $__g_bHWMonInitialized Then Return 0
    _HWMon_RAM_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "uint64", "HWMon_RAM_GetAvailableMemory")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_GetMemoryLoad
; Description ...: Get memory load percentage
; Syntax ........: _HWMon_RAM_GetMemoryLoad()
; Return values .: Memory load in %
; ===============================================================================
Func _HWMon_RAM_GetMemoryLoad()
    If Not $__g_bHWMonInitialized Then Return 0.0
    _HWMon_RAM_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_RAM_GetMemoryLoad")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_GetMemorySpeed
; Description ...: Get RAM speed in MHz
; Syntax ........: _HWMon_RAM_GetMemorySpeed()
; Return values .: RAM speed in MHz
; ===============================================================================
Func _HWMon_RAM_GetMemorySpeed()
    If Not $__g_bHWMonInitialized Then Return 0
    _HWMon_RAM_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_RAM_GetMemorySpeed")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_GetMemoryVoltage
; Description ...: Get RAM voltage in Volts
; Syntax ........: _HWMon_RAM_GetMemoryVoltage()
; Return values .: RAM voltage in V
; ===============================================================================
Func _HWMon_RAM_GetMemoryVoltage()
    If Not $__g_bHWMonInitialized Then Return 0.0
    _HWMon_RAM_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_RAM_GetMemoryVoltage")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_GetModuleCount
; Description ...: Get number of RAM modules
; Syntax ........: _HWMon_RAM_GetModuleCount()
; Return values .: Number of RAM modules
; ===============================================================================
Func _HWMon_RAM_GetModuleCount()
    If Not $__g_bHWMonInitialized Then Return 0
    _HWMon_RAM_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_RAM_GetModuleCount")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_RAM_GetModuleInfo
; Description ...: Get RAM module information
; Syntax ........: _HWMon_RAM_GetModuleInfo($iModuleIndex)
; Parameters ....: $iModuleIndex - Module index (0-based)
; Return values .: Module info string (Manufacturer, Part Number, Size)
; ===============================================================================
Func _HWMon_RAM_GetModuleInfo($iModuleIndex)
    If Not $__g_bHWMonInitialized Then Return ""
    _HWMon_RAM_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_RAM_GetModuleInfo", "int", $iModuleIndex)
    If @error Then Return ""
    Return $aResult[0]
EndFunc

; ===============================================================================
; Battery Monitor Functions (15 functions)
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_Initialize
; Description ...: Initialize Battery Monitor
; Syntax ........: _HWMon_Battery_Initialize()
; Return values .: Success - True, Failure - False
; ===============================================================================
Func _HWMon_Battery_Initialize()
    If Not $__g_bHWMonInitialized Then Return False
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Battery_Initialize")
    If @error Then Return False
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_Update
; Description ...: Update Battery Monitor data
; Syntax ........: _HWMon_Battery_Update()
; Return values .: Success - True, Failure - False
; ===============================================================================
Func _HWMon_Battery_Update()
    If Not $__g_bHWMonInitialized Then Return False
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Battery_Update")
    If @error Then Return False
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_IsPresent
; Description ...: Check if battery is present
; Syntax ........: _HWMon_Battery_IsPresent()
; Return values .: True if battery present, False otherwise
; ===============================================================================
Func _HWMon_Battery_IsPresent()
    If Not $__g_bHWMonInitialized Then Return False
    ; Update disabled - static values from init
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Battery_IsPresent")
    If @error Then Return False
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetStatus
; Description ...: Get battery status string
; Syntax ........: _HWMon_Battery_GetStatus()
; Return values .: Status string (Charging/Discharging/Full/Unknown)
; ===============================================================================
Func _HWMon_Battery_GetStatus()
    If Not $__g_bHWMonInitialized Then Return ""
    ; Don't call Update - it may crash
    ; _HWMon_Battery_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Battery_GetStatusString")
    If @error Then Return ""
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetChargeLevel
; Description ...: Get battery charge level percentage
; Syntax ........: _HWMon_Battery_GetChargeLevel()
; Return values .: Charge level in %
; ===============================================================================
Func _HWMon_Battery_GetChargeLevel()
    If Not $__g_bHWMonInitialized Then Return 0.0
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Battery_GetChargeLevel")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetDesignCapacity
; Description ...: Get battery design capacity in mWh
; Syntax ........: _HWMon_Battery_GetDesignCapacity()
; Return values .: Design capacity in mWh
; ===============================================================================
Func _HWMon_Battery_GetDesignCapacity()
    If Not $__g_bHWMonInitialized Then Return 0
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Battery_GetDesignCapacity")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetFullChargeCapacity
; Description ...: Get battery full charge capacity in mWh
; Syntax ........: _HWMon_Battery_GetFullChargeCapacity()
; Return values .: Full charge capacity in mWh
; ===============================================================================
Func _HWMon_Battery_GetFullChargeCapacity()
    If Not $__g_bHWMonInitialized Then Return 0
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Battery_GetFullChargeCapacity")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetCurrentCapacity
; Description ...: Get battery current capacity in mWh
; Syntax ........: _HWMon_Battery_GetCurrentCapacity()
; Return values .: Current capacity in mWh
; ===============================================================================
Func _HWMon_Battery_GetCurrentCapacity()
    If Not $__g_bHWMonInitialized Then Return 0
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Battery_GetRemainingCapacity")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetHealthPercent
; Description ...: Get battery health percentage
; Syntax ........: _HWMon_Battery_GetHealthPercent()
; Return values .: Health percentage
; ===============================================================================
Func _HWMon_Battery_GetHealthPercent()
    If Not $__g_bHWMonInitialized Then Return 0.0
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Battery_GetHealth")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetVoltage
; Description ...: Get battery voltage in Volts
; Syntax ........: _HWMon_Battery_GetVoltage()
; Return values .: Voltage in V
; ===============================================================================
Func _HWMon_Battery_GetVoltage()
    If Not $__g_bHWMonInitialized Then Return 0.0
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Battery_GetVoltage")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetChargeRate
; Description ...: Get battery charge/discharge rate in Watts
; Syntax ........: _HWMon_Battery_GetChargeRate()
; Return values .: Charge rate in W (positive=charging, negative=discharging)
; ===============================================================================
Func _HWMon_Battery_GetChargeRate()
    If Not $__g_bHWMonInitialized Then Return 0.0
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Battery_GetChargeRate")
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetTimeRemaining
; Description ...: Get estimated time remaining in minutes
; Syntax ........: _HWMon_Battery_GetTimeRemaining()
; Return values .: Time remaining in minutes
; ===============================================================================
Func _HWMon_Battery_GetTimeRemaining()
    If Not $__g_bHWMonInitialized Then Return 0
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Battery_GetTimeRemaining")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetTimeToFullCharge
; Description ...: Get estimated time to full charge in minutes
; Syntax ........: _HWMon_Battery_GetTimeToFullCharge()
; Return values .: Time to full charge in minutes
; ===============================================================================
Func _HWMon_Battery_GetTimeToFullCharge()
    If Not $__g_bHWMonInitialized Then Return 0
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Battery_GetTimeToFullCharge")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetCycleCount
; Description ...: Get battery cycle count
; Syntax ........: _HWMon_Battery_GetCycleCount()
; Return values .: Cycle count
; ===============================================================================
Func _HWMon_Battery_GetCycleCount()
    If Not $__g_bHWMonInitialized Then Return 0
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Battery_GetCycleCount")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Battery_GetManufacturer
; Description ...: Get battery manufacturer name
; Syntax ........: _HWMon_Battery_GetManufacturer()
; Return values .: Manufacturer string
; ===============================================================================
Func _HWMon_Battery_GetManufacturer()
    If Not $__g_bHWMonInitialized Then Return ""
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Battery_GetManufacturer")
    If @error Then Return ""
    Return $aResult[0]
EndFunc

; ===============================================================================
; Motherboard Monitor Functions (14 functions)
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_Initialize
; Description ...: Initialize Motherboard Monitor
; Syntax ........: _HWMon_Motherboard_Initialize()
; Return values .: Success - True, Failure - False
; Remarks .......: Requires PawnIO driver. May not work on laptops.
; ===============================================================================
Func _HWMon_Motherboard_Initialize()
    If Not $__g_bHWMonInitialized Then Return False
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Motherboard_Initialize")
    If @error Then Return False
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_Update
; Description ...: Update Motherboard Monitor data
; Syntax ........: _HWMon_Motherboard_Update()
; Return values .: Success - True, Failure - False
; ===============================================================================
Func _HWMon_Motherboard_Update()
    If Not $__g_bHWMonInitialized Then Return False
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Motherboard_Update")
    If @error Then Return False
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_IsDetected
; Description ...: Check if Super I/O chip is detected
; Syntax ........: _HWMon_Motherboard_IsDetected()
; Return values .: True if chip detected, False otherwise
; ===============================================================================
Func _HWMon_Motherboard_IsDetected()
    If Not $__g_bHWMonInitialized Then Return False
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Motherboard_IsDetected")
    If @error Then Return False
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetChipName
; Description ...: Get Super I/O chip name
; Syntax ........: _HWMon_Motherboard_GetChipName()
; Return values .: Chip name string (71 chips supported)
; ===============================================================================
Func _HWMon_Motherboard_GetChipName()
    If Not $__g_bHWMonInitialized Then Return ""
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Motherboard_GetChipName")
    If @error Then Return ""
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetTemperatureCount
; Description ...: Get number of temperature sensors
; Syntax ........: _HWMon_Motherboard_GetTemperatureCount()
; Return values .: Number of temperature sensors
; ===============================================================================
Func _HWMon_Motherboard_GetTemperatureCount()
    If Not $__g_bHWMonInitialized Then Return 0
    _HWMon_Motherboard_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Motherboard_GetTemperatureCount")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetTemperature
; Description ...: Get temperature sensor value
; Syntax ........: _HWMon_Motherboard_GetTemperature($iIndex)
; Parameters ....: $iIndex - Sensor index (0-based)
; Return values .: Temperature in Celsius
; ===============================================================================
Func _HWMon_Motherboard_GetTemperature($iIndex)
    If Not $__g_bHWMonInitialized Then Return 0.0
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Motherboard_GetTemperature", "int", $iIndex)
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetTemperatureName
; Description ...: Get temperature sensor name
; Syntax ........: _HWMon_Motherboard_GetTemperatureName($iIndex)
; Parameters ....: $iIndex - Sensor index (0-based)
; Return values .: Sensor name string
; ===============================================================================
Func _HWMon_Motherboard_GetTemperatureName($iIndex)
    If Not $__g_bHWMonInitialized Then Return ""
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Motherboard_GetTemperatureName", "int", $iIndex)
    If @error Then Return ""
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetVoltageCount
; Description ...: Get number of voltage sensors
; Syntax ........: _HWMon_Motherboard_GetVoltageCount()
; Return values .: Number of voltage sensors
; ===============================================================================
Func _HWMon_Motherboard_GetVoltageCount()
    If Not $__g_bHWMonInitialized Then Return 0
    _HWMon_Motherboard_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Motherboard_GetVoltageCount")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetVoltage
; Description ...: Get voltage sensor value
; Syntax ........: _HWMon_Motherboard_GetVoltage($iIndex)
; Parameters ....: $iIndex - Sensor index (0-based)
; Return values .: Voltage in Volts
; ===============================================================================
Func _HWMon_Motherboard_GetVoltage($iIndex)
    If Not $__g_bHWMonInitialized Then Return 0.0
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Motherboard_GetVoltage", "int", $iIndex)
    If @error Then Return 0.0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetVoltageName
; Description ...: Get voltage sensor name
; Syntax ........: _HWMon_Motherboard_GetVoltageName($iIndex)
; Parameters ....: $iIndex - Sensor index (0-based)
; Return values .: Sensor name string
; ===============================================================================
Func _HWMon_Motherboard_GetVoltageName($iIndex)
    If Not $__g_bHWMonInitialized Then Return ""
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Motherboard_GetVoltageName", "int", $iIndex)
    If @error Then Return ""
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetFanCount
; Description ...: Get number of fan sensors
; Syntax ........: _HWMon_Motherboard_GetFanCount()
; Return values .: Number of fan sensors
; ===============================================================================
Func _HWMon_Motherboard_GetFanCount()
    If Not $__g_bHWMonInitialized Then Return 0
    _HWMon_Motherboard_Update()
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Motherboard_GetFanCount")
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetFanSpeed
; Description ...: Get fan speed in RPM
; Syntax ........: _HWMon_Motherboard_GetFanSpeed($iIndex)
; Parameters ....: $iIndex - Fan index (0-based)
; Return values .: Fan speed in RPM
; ===============================================================================
Func _HWMon_Motherboard_GetFanSpeed($iIndex)
    If Not $__g_bHWMonInitialized Then Return 0
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Motherboard_GetFanSpeed", "int", $iIndex)
    If @error Then Return 0
    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Motherboard_GetFanName
; Description ...: Get fan sensor name
; Syntax ........: _HWMon_Motherboard_GetFanName($iIndex)
; Parameters ....: $iIndex - Fan index (0-based)
; Return values .: Fan name string
; ===============================================================================
Func _HWMon_Motherboard_GetFanName($iIndex)
    If Not $__g_bHWMonInitialized Then Return ""
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Motherboard_GetFanName", "int", $iIndex)
    If @error Then Return ""
    Return $aResult[0]
EndFunc
