#pragma once

#ifdef HARDWAREMONITOR_EXPORTS
#define HWMON_API __declspec(dllexport)
#else
#define HWMON_API __declspec(dllimport)
#endif

#include <windows.h>
#include <cstdint>

// Error codes
#define HWMON_SUCCESS           0
#define HWMON_ERROR_INIT        -1
#define HWMON_ERROR_NOT_FOUND   -2
#define HWMON_ERROR_ACCESS      -3
#define HWMON_ERROR_INVALID     -4

// Module availability info structure
// Note: Packed to avoid alignment issues with AutoIt
// Using uint8_t instead of bool for guaranteed 1-byte size
#pragma pack(push, 1)
typedef struct {
    uint8_t isInitialized;      // Library initialized (0=false, 1=true)
    uint8_t hasPawnIO;          // PawnIO driver available
    uint8_t hasIntelCPU;        // Intel CPU available
    uint8_t hasAMDCPU;          // AMD CPU available
    uint32_t amdCPUFamily;      // AMD CPU family (0 if not AMD)
    uint8_t hasNVIDIAGPU;       // NVIDIA GPU available
    uint32_t nvidiaGPUCount;    // Number of NVIDIA GPUs (changed from int to uint32_t)
    uint8_t hasAMDRadeonGPU;    // AMD Radeon GPU available
    uint32_t amdRadeonGPUCount; // Number of AMD Radeon GPUs (changed from int to uint32_t)
    uint8_t hasStorage;         // Storage available
    uint32_t storageCount;      // Number of storage drives (changed from int to uint32_t)
} HWMonModuleInfo;
#pragma pack(pop)

extern "C" {
    // Library initialization
    HWMON_API int HWMon_Initialize();
    HWMON_API int HWMon_InitializeEx(HWMonModuleInfo* outInfo);  // Extended version
    HWMON_API void HWMon_Cleanup();
    HWMON_API bool HWMon_IsInitialized();
    HWMON_API int HWMon_GetModuleInfo(HWMonModuleInfo* outInfo);  // Get current module info (returns 1=success, 0=fail)
    HWMON_API int HWMon_GetModuleInfoSize();  // Get sizeof(HWMonModuleInfo) for debugging
    
    // PawnIO status
    HWMON_API bool HWMon_IsPawnIOInstalled();
    HWMON_API const char* HWMon_GetLastError();
    
    // Intel CPU Temperature
    HWMON_API int HWMon_Intel_Initialize(const wchar_t* msrBinPath);
    HWMON_API bool HWMon_Intel_Update();
    HWMON_API int HWMon_Intel_GetCoreCount();
    HWMON_API float HWMon_Intel_GetTjMax();
    HWMON_API float HWMon_Intel_GetCoreTemp(int coreIndex);
    HWMON_API float HWMon_Intel_GetPackageTemp();
    HWMON_API float HWMon_Intel_GetAverageTemp();
    HWMON_API float HWMon_Intel_GetMaxTemp();
    HWMON_API void HWMon_Intel_Close();
    
    // AMD CPU Temperature
    HWMON_API int HWMon_AMD_Initialize(const wchar_t* binPath);
    HWMON_API bool HWMon_AMD_Update();
    HWMON_API uint32_t HWMon_AMD_GetModel();
    HWMON_API float HWMon_AMD_GetTctlTemp();
    HWMON_API float HWMon_AMD_GetTdieTemp();
    HWMON_API int HWMon_AMD_GetCCDCount();
    HWMON_API float HWMon_AMD_GetCCDTemp(int ccdIndex);
    HWMON_API float HWMon_AMD_GetAverageCCDTemp();
    HWMON_API float HWMon_AMD_GetMaxCCDTemp();
    HWMON_API float HWMon_AMD_GetSoCVoltage();
    HWMON_API float HWMon_AMD_GetCoreVoltage();
    HWMON_API float HWMon_AMD_GetPackagePower();
    HWMON_API void HWMon_AMD_Close();
    
    // NVIDIA GPU
    HWMON_API bool HWMon_NVIDIA_IsAvailable();
    HWMON_API int HWMon_NVIDIA_Initialize();
    HWMON_API int HWMon_NVIDIA_EnumerateGPUs();
    HWMON_API bool HWMon_NVIDIA_SelectGPU(int index);
    HWMON_API int HWMon_NVIDIA_GetGPUCount();
    HWMON_API const char* HWMon_NVIDIA_GetGPUName();
    HWMON_API bool HWMon_NVIDIA_Update();
    
    // Temperature
    HWMON_API int HWMon_NVIDIA_GetSensorCount();
    HWMON_API float HWMon_NVIDIA_GetTemperature(int sensorIndex);
    HWMON_API float HWMon_NVIDIA_GetGPUTemperature();
    
    // Clock frequencies (MHz)
    HWMON_API float HWMon_NVIDIA_GetCoreClock();
    HWMON_API float HWMon_NVIDIA_GetMemoryClock();
    HWMON_API float HWMon_NVIDIA_GetShaderClock();
    
    // Fan speed
    HWMON_API int HWMon_NVIDIA_GetFanSpeed();        // RPM
    HWMON_API float HWMon_NVIDIA_GetFanSpeedPercent(); // %
    
    // Load/Usage (%)
    HWMON_API float HWMon_NVIDIA_GetGPUUsage();
    HWMON_API float HWMon_NVIDIA_GetMemoryUsage();
    HWMON_API float HWMon_NVIDIA_GetVideoEngineUsage();
    
    // Memory (bytes)
    HWMON_API uint64_t HWMon_NVIDIA_GetMemoryTotal();
    HWMON_API uint64_t HWMon_NVIDIA_GetMemoryUsed();
    HWMON_API uint64_t HWMon_NVIDIA_GetMemoryFree();
    
    // Power (Watts)
    HWMON_API float HWMon_NVIDIA_GetPowerUsage();
    
    HWMON_API void HWMon_NVIDIA_Close();
    
    // AMD Radeon GPU
    HWMON_API bool HWMon_AMDRadeon_IsAvailable();
    HWMON_API int HWMon_AMDRadeon_Initialize();
    HWMON_API int HWMon_AMDRadeon_EnumerateGPUs();
    HWMON_API bool HWMon_AMDRadeon_SelectGPU(int index);
    HWMON_API int HWMon_AMDRadeon_GetGPUCount();
    HWMON_API const char* HWMon_AMDRadeon_GetGPUName();
    HWMON_API bool HWMon_AMDRadeon_Update();
    
    // Temperature
    HWMON_API float HWMon_AMDRadeon_GetGPUTemperature();
    HWMON_API float HWMon_AMDRadeon_GetHotspotTemperature();
    HWMON_API float HWMon_AMDRadeon_GetMemoryTemperature();
    HWMON_API bool HWMon_AMDRadeon_HasHotspot();
    HWMON_API bool HWMon_AMDRadeon_HasMemory();
    
    // Clock frequencies (MHz)
    HWMON_API float HWMon_AMDRadeon_GetCoreClock();
    HWMON_API float HWMon_AMDRadeon_GetMemoryClock();
    
    // Fan speed
    HWMON_API int HWMon_AMDRadeon_GetFanSpeed();        // RPM
    HWMON_API float HWMon_AMDRadeon_GetFanSpeedPercent(); // %
    HWMON_API int HWMon_AMDRadeon_GetFanSpeedMax();     // Max RPM
    
    // Load/Usage (%)
    HWMON_API float HWMon_AMDRadeon_GetGPUUsage();
    
    // Voltage (V)
    HWMON_API float HWMon_AMDRadeon_GetCoreVoltage();
    
    // Memory (bytes)
    HWMON_API uint64_t HWMon_AMDRadeon_GetMemoryTotal();
    HWMON_API uint64_t HWMon_AMDRadeon_GetMemoryUsed();
    HWMON_API uint64_t HWMon_AMDRadeon_GetMemoryFree();
    
    // Power (Watts)
    HWMON_API float HWMon_AMDRadeon_GetPowerUsage();
    
    HWMON_API int HWMon_AMDRadeon_GetOverdriveVersion();
    HWMON_API void HWMon_AMDRadeon_Close();
    
    // Storage SMART Temperature
    HWMON_API int HWMon_Storage_EnumerateDrives();
    HWMON_API int HWMon_Storage_GetDriveCount();
    HWMON_API bool HWMon_Storage_SelectDrive(int index);
    HWMON_API const char* HWMon_Storage_GetDriveName();
    HWMON_API const char* HWMon_Storage_GetDriveModel();
    HWMON_API const char* HWMon_Storage_GetDriveType();
    HWMON_API bool HWMon_Storage_IsSSD();
    HWMON_API bool HWMon_Storage_IsNVMe();
    HWMON_API bool HWMon_Storage_HasSMART();
    HWMON_API bool HWMon_Storage_Update();
    HWMON_API float HWMon_Storage_GetTemperature();
    HWMON_API bool HWMon_Storage_HasTemperature();
    HWMON_API void HWMon_Storage_Close();
    
    // RAM Monitor
    HWMON_API bool HWMon_RAM_Initialize();
    HWMON_API bool HWMon_RAM_Update();
    HWMON_API uint64_t HWMon_RAM_GetTotalMemory();
    HWMON_API uint64_t HWMon_RAM_GetUsedMemory();
    HWMON_API uint64_t HWMon_RAM_GetAvailableMemory();
    HWMON_API float HWMon_RAM_GetMemoryLoad();
    HWMON_API uint32_t HWMon_RAM_GetMemorySpeed();
    HWMON_API uint32_t HWMon_RAM_GetMemorySpeedConfigured();
    HWMON_API float HWMon_RAM_GetMemoryVoltage();
    HWMON_API int HWMon_RAM_GetModuleCount();
    HWMON_API void HWMon_RAM_Close();
    
    // Battery Monitor
    HWMON_API bool HWMon_Battery_IsAvailable();
    HWMON_API bool HWMon_Battery_Initialize();
    HWMON_API bool HWMon_Battery_Update();
    HWMON_API bool HWMon_Battery_IsPresent();
    HWMON_API int HWMon_Battery_GetStatus();
    HWMON_API const char* HWMon_Battery_GetStatusString();
    HWMON_API float HWMon_Battery_GetChargeLevel();
    HWMON_API uint32_t HWMon_Battery_GetRemainingCapacity();
    HWMON_API uint32_t HWMon_Battery_GetFullChargeCapacity();
    HWMON_API uint32_t HWMon_Battery_GetDesignCapacity();
    HWMON_API float HWMon_Battery_GetHealth();
    HWMON_API uint32_t HWMon_Battery_GetCycleCount();
    HWMON_API int HWMon_Battery_GetChargeRate();
    HWMON_API int HWMon_Battery_GetDischargeRate();
    HWMON_API float HWMon_Battery_GetVoltage();
    HWMON_API uint32_t HWMon_Battery_GetRemainingTime();
    HWMON_API uint32_t HWMon_Battery_GetTimeToFull();
    HWMON_API void HWMon_Battery_Close();
    
    // Motherboard Monitor
    HWMON_API bool HWMon_Motherboard_Initialize();
    HWMON_API bool HWMon_Motherboard_Update();
    HWMON_API bool HWMon_Motherboard_IsDetected();
    HWMON_API const char* HWMon_Motherboard_GetChipName();
    HWMON_API int HWMon_Motherboard_GetTemperatureCount();
    HWMON_API float HWMon_Motherboard_GetTemperature(int index);
    HWMON_API const char* HWMon_Motherboard_GetTemperatureName(int index);
    HWMON_API int HWMon_Motherboard_GetVoltageCount();
    HWMON_API float HWMon_Motherboard_GetVoltage(int index);
    HWMON_API const char* HWMon_Motherboard_GetVoltageName(int index);
    HWMON_API int HWMon_Motherboard_GetFanCount();
    HWMON_API int HWMon_Motherboard_GetFanSpeed(int index);
    HWMON_API const char* HWMon_Motherboard_GetFanName(int index);
    HWMON_API void HWMon_Motherboard_Close();
}
