# HardwareMonitor AutoIt UDF

A comprehensive hardware monitoring library for AutoIt with 65 read-only functions to monitor CPU, GPU, RAM, Storage, Battery, and Motherboard sensors.

## 🎯 Features

- **100% Read-Only** - Safe monitoring without hardware control
- **Cross-Platform Compatible** - Works on different Windows systems
- **Exception Handling** - Crash-free operation with graceful fallbacks
- **Multi-GPU Support** - NVIDIA and AMD Radeon GPUs
- **Extensive Sensor Coverage** - Temperature, clocks, voltages, fan speeds, and more

## 📦 Package Contents

- `HardwareMonitor.dll` - Native C++ monitoring library (x64)
- `HardwareMonitor.au3` - AutoIt UDF wrapper with 65 functions
- `test_udf.au3` - Comprehensive test script demonstrating all features

## 🔧 Requirements

- **AutoIt 3.3.14.0+** (x64 version recommended)
- **Windows 7/8/10/11** (x64)
- **Administrator Rights** - Required for low-level hardware access
- **Optional:** PawnIO driver for enhanced motherboard monitoring

## 🚀 Quick Start

### 1. Basic Usage

```autoit
#RequireAdmin
#include "HardwareMonitor.au3"

; Initialize library
If Not _HWMon_Initialize() Then
    MsgBox(16, "Error", "Failed to initialize: " & _HWMon_GetLastError())
    Exit
EndIf

; Get hardware information
Local $aModuleInfo = _HWMon_GetModuleInfo()
If @error Then
    MsgBox(16, "Error", "Failed to get module info")
    _HWMon_Shutdown()
    Exit
EndIf

; Check available modules
ConsoleWrite("Intel CPU: " & ($aModuleInfo[2] ? "Yes" : "No") & @CRLF)
ConsoleWrite("AMD Radeon GPU: " & ($aModuleInfo[7] ? "Yes" : "No") & @CRLF)
ConsoleWrite("Storage: " & ($aModuleInfo[9] ? "Yes" : "No") & @CRLF)

; Cleanup
_HWMon_Shutdown()
```

### 2. CPU Temperature Monitoring

```autoit
#RequireAdmin
#include "HardwareMonitor.au3"

_HWMon_Initialize()

; Get Intel CPU temperatures
Local $coreCount = _HWMon_IntelCPU_GetCoreCount()
ConsoleWrite("CPU Cores: " & $coreCount & @CRLF)

For $i = 0 To $coreCount - 1
    Local $temp = _HWMon_IntelCPU_GetCoreTemperature($i)
    ConsoleWrite("Core " & $i & ": " & $temp & "°C" & @CRLF)
Next

_HWMon_Shutdown()
```

### 3. GPU Monitoring

```autoit
#RequireAdmin
#include "HardwareMonitor.au3"

_HWMon_Initialize()

; AMD Radeon GPU
Local $gpuCount = _HWMon_AMDRadeon_EnumerateGPUs()
If $gpuCount > 0 Then
    _HWMon_AMDRadeon_SelectGPU(0)
    _HWMon_AMDRadeon_Update()
    
    ConsoleWrite("GPU: " & _HWMon_AMDRadeon_GetGPUName() & @CRLF)
    ConsoleWrite("Temperature: " & _HWMon_AMDRadeon_GetGPUTemperature() & "°C" & @CRLF)
    ConsoleWrite("Core Clock: " & _HWMon_AMDRadeon_GetCoreClock() & " MHz" & @CRLF)
    ConsoleWrite("Memory Clock: " & _HWMon_AMDRadeon_GetMemoryClock() & " MHz" & @CRLF)
EndIf

_HWMon_Shutdown()
```

## 📚 Available Modules

### Core Functions (3)
- `_HWMon_Initialize()` - Initialize library
- `_HWMon_GetModuleInfo()` - Get available hardware modules
- `_HWMon_Shutdown()` - Cleanup and shutdown

### Intel CPU Monitor (4)
- `_HWMon_IntelCPU_GetCoreCount()` - Number of CPU cores
- `_HWMon_IntelCPU_GetPackageTemperature()` - Package temperature
- `_HWMon_IntelCPU_GetCoreTemperature($core)` - Individual core temperature
- `_HWMon_IntelCPU_GetTjMax()` - Maximum junction temperature

### AMD CPU Monitor (3)
- `_HWMon_AMDCPU_GetCoreCount()` - Number of CPU cores
- `_HWMon_AMDCPU_GetCoreTemperature($core)` - Core temperature
- `_HWMon_AMDCPU_GetCCDTemperature($ccd)` - CCD temperature (Zen 2+)

### NVIDIA GPU Monitor (13)
- `_HWMon_NVIDIAGPU_EnumerateGPUs()` - Count NVIDIA GPUs
- `_HWMon_NVIDIAGPU_SelectGPU($index)` - Select GPU
- `_HWMon_NVIDIAGPU_Update()` - Update sensor readings
- `_HWMon_NVIDIAGPU_GetGPUName()` - GPU name
- `_HWMon_NVIDIAGPU_GetGPUTemperature()` - GPU temperature
- `_HWMon_NVIDIAGPU_GetCoreClock()` - Core clock (MHz)
- `_HWMon_NVIDIAGPU_GetMemoryClock()` - Memory clock (MHz)
- `_HWMon_NVIDIAGPU_GetShaderClock()` - Shader clock (MHz)
- `_HWMon_NVIDIAGPU_GetFanSpeed()` - Fan speed (RPM)
- `_HWMon_NVIDIAGPU_GetFanSpeedPercent()` - Fan speed (%)
- `_HWMon_NVIDIAGPU_GetGPUUsage()` - GPU usage (%)
- `_HWMon_NVIDIAGPU_GetMemoryUsage()` - Memory usage (MB)
- `_HWMon_NVIDIAGPU_GetPowerUsage()` - Power usage (W)

### AMD Radeon GPU Monitor (13)
- `_HWMon_AMDRadeon_EnumerateGPUs()` - Count AMD GPUs
- `_HWMon_AMDRadeon_SelectGPU($index)` - Select GPU
- `_HWMon_AMDRadeon_Update()` - Update sensor readings
- `_HWMon_AMDRadeon_GetGPUName()` - GPU name
- `_HWMon_AMDRadeon_GetGPUTemperature()` - GPU temperature
- `_HWMon_AMDRadeon_GetHotspotTemperature()` - Hotspot temperature
- `_HWMon_AMDRadeon_GetMemoryTemperature()` - Memory temperature
- `_HWMon_AMDRadeon_GetCoreClock()` - Core clock (MHz)
- `_HWMon_AMDRadeon_GetMemoryClock()` - Memory clock (MHz)
- `_HWMon_AMDRadeon_GetFanSpeed()` - Fan speed (RPM)
- `_HWMon_AMDRadeon_GetFanSpeedPercent()` - Fan speed (%)
- `_HWMon_AMDRadeon_GetGPUUsage()` - GPU usage (%)
- `_HWMon_AMDRadeon_GetPowerUsage()` - Power usage (W)

### Storage Monitor (4)
- `_HWMon_Storage_EnumerateDrives()` - Count storage drives
- `_HWMon_Storage_GetDriveName($index)` - Drive name
- `_HWMon_Storage_GetDriveType($index)` - Drive type (SSD/HDD)
- `_HWMon_Storage_GetTemperature($index)` - Drive temperature

### RAM Monitor (10)
- `_HWMon_RAM_Initialize()` - Initialize RAM monitoring
- `_HWMon_RAM_GetTotalMemory()` - Total RAM (GB)
- `_HWMon_RAM_GetUsedMemory()` - Used RAM (GB)
- `_HWMon_RAM_GetAvailableMemory()` - Available RAM (GB)
- `_HWMon_RAM_GetMemoryLoad()` - Memory load (%)
- `_HWMon_RAM_GetMemorySpeed()` - RAM speed (MHz)
- `_HWMon_RAM_GetMemoryVoltage()` - RAM voltage (V)
- `_HWMon_RAM_GetModuleCount()` - Number of RAM modules
- `_HWMon_RAM_GetModuleSize($index)` - Module size (GB)
- `_HWMon_RAM_GetModuleSpeed($index)` - Module speed (MHz)

### Battery Monitor (15)
- `_HWMon_Battery_Initialize()` - Initialize battery monitoring
- `_HWMon_Battery_IsPresent()` - Check if battery present
- `_HWMon_Battery_GetStatus()` - Battery status string
- `_HWMon_Battery_GetChargeLevel()` - Charge level (%)
- `_HWMon_Battery_GetVoltage()` - Voltage (V)
- `_HWMon_Battery_GetChargeCurrent()` - Charge current (mA)
- `_HWMon_Battery_GetDischargeCurrent()` - Discharge current (mA)
- `_HWMon_Battery_GetChargeRate()` - Charge rate (W)
- `_HWMon_Battery_GetDischargeRate()` - Discharge rate (W)
- `_HWMon_Battery_GetRemainingCapacity()` - Remaining capacity (mWh)
- `_HWMon_Battery_GetFullChargeCapacity()` - Full charge capacity (mWh)
- `_HWMon_Battery_GetDesignCapacity()` - Design capacity (mWh)
- `_HWMon_Battery_GetHealth()` - Battery health (%)
- `_HWMon_Battery_GetEstimatedTime()` - Estimated time (minutes)
- `_HWMon_Battery_GetCycleCount()` - Charge cycle count

### Motherboard Monitor (14)
- `_HWMon_Motherboard_Initialize()` - Initialize motherboard monitoring
- `_HWMon_Motherboard_IsChipDetected()` - Check if Super I/O chip detected
- `_HWMon_Motherboard_GetChipName()` - Super I/O chip name
- `_HWMon_Motherboard_GetTemperatureCount()` - Number of temperature sensors
- `_HWMon_Motherboard_GetTemperature($index)` - Temperature sensor value
- `_HWMon_Motherboard_GetTemperatureName($index)` - Temperature sensor name
- `_HWMon_Motherboard_GetVoltageCount()` - Number of voltage sensors
- `_HWMon_Motherboard_GetVoltage($index)` - Voltage sensor value
- `_HWMon_Motherboard_GetVoltageName($index)` - Voltage sensor name
- `_HWMon_Motherboard_GetFanCount()` - Number of fan sensors
- `_HWMon_Motherboard_GetFanSpeed($index)` - Fan speed (RPM)
- `_HWMon_Motherboard_GetFanName($index)` - Fan name
- `_HWMon_Motherboard_GetFanControlPercent($index)` - Fan control (%)
- `_HWMon_Motherboard_Update()` - Update all sensor readings

## 🔍 Hardware Detection

### Module Info Array Structure
```autoit
Local $aModuleInfo = _HWMon_GetModuleInfo()
; [0] = Library Initialized (1/0)
; [1] = PawnIO Available (1/0)
; [2] = Intel CPU (1/0)
; [3] = AMD CPU (1/0)
; [4] = AMD CPU Family
; [5] = NVIDIA GPU (1/0)
; [6] = NVIDIA GPU Count
; [7] = AMD Radeon GPU (1/0)
; [8] = AMD Radeon GPU Count
; [9] = Storage (1/0)
; [10] = Storage Count
```

### AMD GPU Detection
The library uses a 3-tier fallback system for AMD GPU detection:

1. **ADL2 API** (Primary) - Full monitoring support with temperature, clocks, fan, etc.
2. **ADL1 API** (Fallback) - Legacy driver support
3. **WMI** (Final Fallback) - Detection only, limited monitoring

**Note:** Some AMD GPUs may be detected but not fully monitorable due to driver limitations. If ADL fails, the GPU will be detected via WMI but monitoring functions will return 0.

## 🛠️ Supported Hardware

### CPUs
- Intel Core (all generations with DTS support)
- AMD Ryzen (Zen, Zen+, Zen 2, Zen 3, Zen 4)

### GPUs
- **NVIDIA:** GeForce GTX/RTX series (NVAPI support required)
- **AMD:** Radeon RX series (ADL or WMI detection)

### Motherboards
Supports 71 Super I/O chips from:
- **ITE:** IT8705F, IT8712F, IT8716F, IT8718F, IT8720F, IT8721F, IT8728F, IT8792E
- **Nuvoton:** NCT6775F, NCT6776F, NCT6779D, NCT6791D, NCT6792D, NCT6793D, NCT6795D, NCT6796D, NCT6797D, NCT6798D
- **Winbond:** W83627HF, W83627DHG, W83627EHF, W83667HG
- **Fintek:** F71858, F71862, F71869, F71882, F71889ED, F71889AD
- **SMSC:** SCH5027

## ⚠️ Important Notes

### Administrator Rights
All hardware monitoring requires administrator privileges. Always use `#RequireAdmin` in your scripts.

### Error Handling
Always check for errors after function calls:
```autoit
Local $result = _HWMon_SomeFunction()
If @error Then
    ConsoleWrite("Error: " & _HWMon_GetLastError() & @CRLF)
EndIf
```

### Update Functions
Some modules require calling `Update()` before reading sensors:
- `_HWMon_NVIDIAGPU_Update()`
- `_HWMon_AMDRadeon_Update()`
- `_HWMon_Motherboard_Update()`

### Cleanup
Always call `_HWMon_Shutdown()` before exiting to properly cleanup resources.

## 🐛 Troubleshooting

### "Library not initialized"
- Ensure you called `_HWMon_Initialize()` first
- Check if DLL is in the same folder as your script
- Verify you're running as administrator

### "Module not available"
- Check `_HWMon_GetModuleInfo()` to see which modules are detected
- Some hardware may not be supported (e.g., laptop motherboards)
- AMD GPU may be detected but not monitorable (driver limitation)

### Crashes or Unexpected Behavior
- Ensure you're using the x64 version of AutoIt
- Update your GPU drivers to the latest version
- Check `_HWMon_GetLastError()` for detailed error messages

## 📄 License

This library is provided as-is for hardware monitoring purposes only. No hardware control or modification is performed.

## 🙏 Credits

Built using:
- **NVAPI** - NVIDIA GPU monitoring
- **ADL** - AMD GPU monitoring  
- **WMI** - Windows Management Instrumentation fallback
- **PawnIO** - Low-level I/O access for motherboard monitoring

## 📞 Support

For issues, questions, or feature requests, please check:
1. This README for common solutions
2. `test_udf.au3` for usage examples
3. Error messages from `_HWMon_GetLastError()`

---

**Version:** 1.0.0  
**Last Updated:** November 2024  
**Total Functions:** 65  
**Status:** Production Ready ✅
