#include-once
#include <Array.au3>
#RequireAdmin
#AutoIt3Wrapper_UseX64=y
#pragma compile(x64, True)
; ===============================================================================
; Title .........: HardwareMonitor UDF
; Description ...: AutoIt wrapper for HardwareMonitor.dll (ONLY X64)
; Author(s) .....: Dao Van Trong - TRONG.PRO
; Version .......: 1.0.0
; ===============================================================================

; ===============================================================================
; Global Variables
; ===============================================================================
Global $__g_hHWMonDLL = -1
Global $__g_sHWMonDLLPath = ""
Global $__g_bHWMonInitialized = False

; Module info struct
Global Const $tagHWMonModuleInfo = "align 1;" & _
    "byte isInitialized;" & _
    "byte hasPawnIO;" & _
    "byte hasIntelCPU;" & _
    "byte hasAMDCPU;" & _
    "dword amdCPUFamily;" & _
    "byte hasNVIDIAGPU;" & _
    "dword nvidiaGPUCount;" & _
    "byte hasAMDRadeonGPU;" & _
    "dword amdRadeonGPUCount;" & _
    "byte hasStorage;" & _
    "dword storageCount"

; ===============================================================================
; Core Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Startup
; Description ...: Initialize the HardwareMonitor library
; Syntax ........: _HWMon_Startup([$sDLLPath = ""])
; Parameters ....: $sDLLPath - [optional] Path to HardwareMonitor.dll
; Return values .: Success - True
;                  Failure - False, sets @error
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Startup($sDLLPath = "")
    If $__g_bHWMonInitialized Then Return True

    ; Determine DLL path
    If $sDLLPath = "" Then
        If @AutoItX64 Then
            $sDLLPath = @ScriptDir & "\HardwareMonitor.dll"
        Else
            Return False
        EndIf
    EndIf

    ; Check if DLL exists
    If Not FileExists($sDLLPath) Then
        Return SetError(1, 0, False) ; DLL not found
    EndIf

    ; Open DLL
    $__g_hHWMonDLL = DllOpen($sDLLPath)
    If $__g_hHWMonDLL = -1 Then
        Return SetError(2, 0, False) ; Failed to open DLL
    EndIf

    $__g_sHWMonDLLPath = $sDLLPath

    ; Initialize library
    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Initialize")
    If @error Or $aResult[0] <> 0 Then
        DllClose($__g_hHWMonDLL)
        $__g_hHWMonDLL = -1
        Return SetError(3, 0, False) ; Initialization failed
    EndIf

    $__g_bHWMonInitialized = True
    Return True
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Shutdown
; Description ...: Cleanup and close the HardwareMonitor library
; Syntax ........: _HWMon_Shutdown()
; Return values .: None
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Shutdown()
    If Not $__g_bHWMonInitialized Then Return

    ; Cleanup
    DllCall($__g_hHWMonDLL, "none", "HWMon_Cleanup")

    ; Close DLL
    DllClose($__g_hHWMonDLL)
    $__g_hHWMonDLL = -1
    $__g_bHWMonInitialized = False
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_GetModuleInfo
; Description ...: Get information about available hardware modules
; Syntax ........: _HWMon_GetModuleInfo()
; Return values .: Success - Array with module information
;                  Failure - False, sets @error
; Author ........: Your Name
; Remarks .......: Array format:
;                  [0] = isInitialized
;                  [1] = hasPawnIO
;                  [2] = hasIntelCPU
;                  [3] = hasAMDCPU
;                  [4] = amdCPUFamily
;                  [5] = hasNVIDIAGPU
;                  [6] = nvidiaGPUCount
;                  [7] = hasAMDRadeonGPU
;                  [8] = amdRadeonGPUCount
;                  [9] = hasStorage
;                  [10] = storageCount
; ===============================================================================
Func _HWMon_GetModuleInfo()
    If Not $__g_bHWMonInitialized Then Return SetError(1, 0, False)

    Local $tInfo = DllStructCreate($tagHWMonModuleInfo)
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_GetModuleInfo", "struct*", $tInfo)

    If @error Or Not $aResult[0] Then
        Return SetError(2, 0, False)
    EndIf

    Local $aInfo[11]
    $aInfo[0] = DllStructGetData($tInfo, "isInitialized")
    $aInfo[1] = DllStructGetData($tInfo, "hasPawnIO")
    $aInfo[2] = DllStructGetData($tInfo, "hasIntelCPU")
    $aInfo[3] = DllStructGetData($tInfo, "hasAMDCPU")
    $aInfo[4] = DllStructGetData($tInfo, "amdCPUFamily")
    $aInfo[5] = DllStructGetData($tInfo, "hasNVIDIAGPU")
    $aInfo[6] = DllStructGetData($tInfo, "nvidiaGPUCount")
    $aInfo[7] = DllStructGetData($tInfo, "hasAMDRadeonGPU")
    $aInfo[8] = DllStructGetData($tInfo, "amdRadeonGPUCount")
    $aInfo[9] = DllStructGetData($tInfo, "hasStorage")
    $aInfo[10] = DllStructGetData($tInfo, "storageCount")

    Return $aInfo
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_GetLastError
; Description ...: Get the last error message
; Syntax ........: _HWMon_GetLastError()
; Return values .: Error message string
; Author ........: Your Name
; ===============================================================================
Func _HWMon_GetLastError()
    If Not $__g_bHWMonInitialized Then Return ""

    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_GetLastError")
    If @error Then Return ""

    Return $aResult[0]
EndFunc

; ===============================================================================
; Intel CPU Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Intel_GetCoreCount
; Description ...: Get Intel CPU core count
; Syntax ........: _HWMon_Intel_GetCoreCount()
; Return values .: Success - Number of cores
;                  Failure - 0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Intel_GetCoreCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Intel_GetCoreCount")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Intel_GetPackageTemp
; Description ...: Get Intel CPU package temperature
; Syntax ........: _HWMon_Intel_GetPackageTemp()
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Intel_GetPackageTemp()
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Update first
    DllCall($__g_hHWMonDLL, "bool", "HWMon_Intel_Update")

    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Intel_GetPackageTemp")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Intel_GetCoreTemp
; Description ...: Get Intel CPU core temperature
; Syntax ........: _HWMon_Intel_GetCoreTemp($iCoreIndex)
; Parameters ....: $iCoreIndex - Core index (0-based)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Intel_GetCoreTemp($iCoreIndex)
    If Not $__g_bHWMonInitialized Then Return 0.0

    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Intel_GetCoreTemp", "int", $iCoreIndex)
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Intel_GetAllCoreTemps
; Description ...: Get all Intel CPU core temperatures
; Syntax ........: _HWMon_Intel_GetAllCoreTemps()
; Return values .: Success - Array of temperatures
;                  Failure - Empty array
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Intel_GetAllCoreTemps()
    If Not $__g_bHWMonInitialized Then Return SetError(1, 0, 0)

    Local $iCoreCount = _HWMon_Intel_GetCoreCount()
    If $iCoreCount = 0 Then Return SetError(2, 0, 0)

    ; Update first
    DllCall($__g_hHWMonDLL, "bool", "HWMon_Intel_Update")

    Local $aTemps[$iCoreCount]
    For $i = 0 To $iCoreCount - 1
        $aTemps[$i] = _HWMon_Intel_GetCoreTemp($i)
    Next

    Return $aTemps
EndFunc

; ===============================================================================
; AMD CPU Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMD_GetTctlTemp
; Description ...: Get AMD CPU Tctl temperature
; Syntax ........: _HWMon_AMD_GetTctlTemp()
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_AMD_GetTctlTemp()
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Update first
    DllCall($__g_hHWMonDLL, "bool", "HWMon_AMD_Update")

    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_AMD_GetTctlTemp")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMD_GetCCDCount
; Description ...: Get AMD CPU CCD count
; Syntax ........: _HWMon_AMD_GetCCDCount()
; Return values .: Success - Number of CCDs
;                  Failure - 0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_AMD_GetCCDCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_AMD_GetCCDCount")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMD_GetCCDTemp
; Description ...: Get AMD CPU CCD temperature
; Syntax ........: _HWMon_AMD_GetCCDTemp($iCCDIndex)
; Parameters ....: $iCCDIndex - CCD index (0-based)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_AMD_GetCCDTemp($iCCDIndex)
    If Not $__g_bHWMonInitialized Then Return 0.0

    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_AMD_GetCCDTemp", "int", $iCCDIndex)
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; ===============================================================================
; NVIDIA GPU Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetGPUCount
; Description ...: Get NVIDIA GPU count
; Syntax ........: _HWMon_NVIDIA_GetGPUCount()
; Return values .: Success - Number of GPUs
;                  Failure - 0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_NVIDIA_GetGPUCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_NVIDIA_EnumerateGPUs")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetGPUTemp
; Description ...: Get NVIDIA GPU temperature
; Syntax ........: _HWMon_NVIDIA_GetGPUTemp([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_NVIDIA_GetGPUTemp($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Select GPU
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)

    ; Update
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_Update")

    ; Get temperature
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_NVIDIA_GetGPUTemperature")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_NVIDIA_GetGPUName
; Description ...: Get NVIDIA GPU name
; Syntax ........: _HWMon_NVIDIA_GetGPUName([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - GPU name
;                  Failure - ""
; Author ........: Your Name
; ===============================================================================
Func _HWMon_NVIDIA_GetGPUName($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return ""

    ; Select GPU
    DllCall($__g_hHWMonDLL, "bool", "HWMon_NVIDIA_SelectGPU", "int", $iGPUIndex)

    ; Get name
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_NVIDIA_GetGPUName")
    If @error Then Return ""

    Return $aResult[0]
EndFunc

; ===============================================================================
; AMD Radeon GPU Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMDRadeon_GetGPUCount
; Description ...: Get AMD Radeon GPU count
; Syntax ........: _HWMon_AMDRadeon_GetGPUCount()
; Return values .: Success - Number of GPUs
;                  Failure - 0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_AMDRadeon_GetGPUCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_AMDRadeon_EnumerateGPUs")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMDRadeon_GetGPUTemp
; Description ...: Get AMD Radeon GPU temperature
; Syntax ........: _HWMon_AMDRadeon_GetGPUTemp([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_AMDRadeon_GetGPUTemp($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Select GPU
    DllCall($__g_hHWMonDLL, "bool", "HWMon_AMDRadeon_SelectGPU", "int", $iGPUIndex)

    ; Update
    DllCall($__g_hHWMonDLL, "bool", "HWMon_AMDRadeon_Update")

    ; Get temperature
    Local $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_AMDRadeon_GetGPUTemperature")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_AMDRadeon_GetGPUName
; Description ...: Get AMD Radeon GPU name
; Syntax ........: _HWMon_AMDRadeon_GetGPUName([$iGPUIndex = 0])
; Parameters ....: $iGPUIndex - [optional] GPU index (default: 0)
; Return values .: Success - GPU name
;                  Failure - ""
; Author ........: Your Name
; ===============================================================================
Func _HWMon_AMDRadeon_GetGPUName($iGPUIndex = 0)
    If Not $__g_bHWMonInitialized Then Return ""

    ; Select GPU
    DllCall($__g_hHWMonDLL, "bool", "HWMon_AMDRadeon_SelectGPU", "int", $iGPUIndex)

    ; Get name
    Local $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_AMDRadeon_GetGPUName")
    If @error Then Return ""

    Return $aResult[0]
EndFunc

; ===============================================================================
; Storage Functions
; ===============================================================================

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Storage_GetDriveCount
; Description ...: Get storage drive count
; Syntax ........: _HWMon_Storage_GetDriveCount()
; Return values .: Success - Number of drives
;                  Failure - 0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Storage_GetDriveCount()
    If Not $__g_bHWMonInitialized Then Return 0

    Local $aResult = DllCall($__g_hHWMonDLL, "int", "HWMon_Storage_EnumerateDrives")
    If @error Then Return 0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Storage_GetDriveTemp
; Description ...: Get storage drive temperature
; Syntax ........: _HWMon_Storage_GetDriveTemp($iDriveIndex)
; Parameters ....: $iDriveIndex - Drive index (0-based)
; Return values .: Success - Temperature in Celsius
;                  Failure - 0.0
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Storage_GetDriveTemp($iDriveIndex)
    If Not $__g_bHWMonInitialized Then Return 0.0

    ; Select drive
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Storage_SelectDrive", "int", $iDriveIndex)
    If @error Or Not $aResult[0] Then Return 0.0

    ; Update
    DllCall($__g_hHWMonDLL, "bool", "HWMon_Storage_Update")

    ; Get temperature
    $aResult = DllCall($__g_hHWMonDLL, "float", "HWMon_Storage_GetTemperature")
    If @error Then Return 0.0

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Storage_GetDriveModel
; Description ...: Get storage drive model
; Syntax ........: _HWMon_Storage_GetDriveModel($iDriveIndex)
; Parameters ....: $iDriveIndex - Drive index (0-based)
; Return values .: Success - Drive model string
;                  Failure - ""
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Storage_GetDriveModel($iDriveIndex)
    If Not $__g_bHWMonInitialized Then Return ""

    ; Select drive
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Storage_SelectDrive", "int", $iDriveIndex)
    If @error Or Not $aResult[0] Then Return ""

    ; Get model
    $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Storage_GetDriveModel")
    If @error Then Return ""

    Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================
; Name ..........: _HWMon_Storage_GetDriveType
; Description ...: Get storage drive type
; Syntax ........: _HWMon_Storage_GetDriveType($iDriveIndex)
; Parameters ....: $iDriveIndex - Drive index (0-based)
; Return values .: Success - Drive type string (HDD/SSD/NVMe)
;                  Failure - ""
; Author ........: Your Name
; ===============================================================================
Func _HWMon_Storage_GetDriveType($iDriveIndex)
    If Not $__g_bHWMonInitialized Then Return ""

    ; Select drive
    Local $aResult = DllCall($__g_hHWMonDLL, "bool", "HWMon_Storage_SelectDrive", "int", $iDriveIndex)
    If @error Or Not $aResult[0] Then Return ""

    ; Get type
    $aResult = DllCall($__g_hHWMonDLL, "str", "HWMon_Storage_GetDriveType")
    If @error Then Return ""

    Return $aResult[0]
EndFunc
