# 📚 HardwareMonitor UDF - AutoIt Wrapper

## 🎯 Introduction

AutoIt UDF wrapper for HardwareMonitor.dll - Powerful hardware monitoring library.

## ✨ Features

- ✅ **Intel CPU** - Temperature monitoring (Core 2+)
- ✅ **AMD CPU** - Temperature monitoring (K8, K10, Bulldozer, Ryzen)
- ✅ **NVIDIA GPU** - Temperature monitoring
- ✅ **AMD Radeon GPU** - Temperature monitoring
- ✅ **Storage** - SMART temperature (HDD/SSD/NVMe)
- ✅ **Lazy Initialization** - Automatically initialize modules
- ✅ **Module Detection** - Know what hardware is available
- ✅ **Error Handling** - Good error handling

## 📦 Files

```
HardwareMonitor.au3 - UDF wrapper
test_udf.au3 - Full test script
example_simple.au3 - Simple example
example_realtime.au3 - Example GUI real-time
```

## 🚀 Quick Start

### **1. Include UDF:**
```autoit
#include "HardwareMonitor.au3"
```

### **2. Initialize:**
```autoit
If Not _HWMon_Startup() Then
    MsgBox(16, "Error", "Failed to initialize!")
    Exit
EndIf
```

### **3. Get temperatures:**
```autoit
; Intel CPU
Local $fCPUTemp = _HWMon_Intel_GetPackageTemp()
ConsoleWrite("CPU: " & $fCPUTemp & "°C" & @CRLF)

; NVIDIA GPU
Local $fGPUTemp = _HWMon_NVIDIA_GetGPUTemp()
ConsoleWrite("GPU: " & $fGPUTemp & "°C" & @CRLF)

; Storage
Local $fHDDTemp = _HWMon_Storage_GetDriveTemp(0)
ConsoleWrite("HDD: " & $fHDDTemp & "°C" & @CRLF)
```

### **4. Cleanup:**
```autoit
_HWMon_Shutdown()
```

## 📖 API Reference

### **Core Functions**

#### `_HWMon_Startup([$sDLLPath = ""])`
Initialize the library.

**Parameters:**
- `$sDLLPath` - [optional] Path to HardwareMonitor.dll

**Returns:**
- Success: `True`
- Failure: `False`, sets `@error`

**Example:**
```autoit
If Not _HWMon_Startup() Then
    ConsoleWrite("Failed to initialize!" & @CRLF)
EndIf
```

#### `_HWMon_Shutdown()`
Cleanup and close the library.

**Example:**
```autoit
_HWMon_Shutdown()
```

#### `_HWMon_GetModuleInfo()`
Get information about available hardware modules.

**Returns:**
- Success: Array with module information
- Failure: `False`, sets `@error`

**Array format:**
```
[0] = isInitialized
[1] = hasPawnIO
[2] = hasIntelCPU
[3] = hasAMDCPU
[4] = amdCPUFamily
[5] = hasNVIDIAGPU
[6] = nvidiaGPUCount
[7] = hasAMDRadeonGPU
[8] = amdRadeonGPUCount
[9] = hasStorage
[10] = storageCount
```

**Example:**
```autoit
Local $aInfo = _HWMon_GetModuleInfo()
If $aInfo[2] Then
    ConsoleWrite("Intel CPU available!" & @CRLF)
EndIf
```

### **Intel CPU Functions**

#### `_HWMon_Intel_GetCoreCount()`
Get Intel CPU core count.

**Returns:** Number of cores

#### `_HWMon_Intel_GetPackageTemp()`
Get Intel CPU package temperature.

**Returns:** Temperature in Celsius

#### `_HWMon_Intel_GetCoreTemp($iCoreIndex)`
Get Intel CPU core temperature.

**Parameters:**
- `$iCoreIndex` - Core index (0-based)

**Returns:** Temperature in Celsius

#### `_HWMon_Intel_GetAllCoreTemps()`
Get all Intel CPU core temperatures.

**Returns:** Array of temperatures

**Example:**
```autoit
Local $aTemps = _HWMon_Intel_GetAllCoreTemps()
For $i = 0 To UBound($aTemps) - 1
    ConsoleWrite("Core " & $i & ": " & $aTemps[$i] & "°C" & @CRLF)
Next
```

### **AMD CPU Functions**

#### `_HWMon_AMD_GetTctlTemp()`
Get AMD CPU Tctl temperature.

**Returns:** Temperature in Celsius

#### `_HWMon_AMD_GetCCDCount()`
Get AMD CPU CCD count (Ryzen only).

**Returns:** Number of CCDs

#### `_HWMon_AMD_GetCCDTemp($iCCDIndex)`
Get AMD CPU CCD temperature.

**Parameters:**
- `$iCCDIndex` - CCD index (0-based)

**Returns:** Temperature in Celsius

### **NVIDIA GPU Functions**

#### `_HWMon_NVIDIA_GetGPUCount()`
Get NVIDIA GPU count.

**Returns:** Number of GPUs

#### `_HWMon_NVIDIA_GetGPUTemp([$iGPUIndex = 0])`
Get NVIDIA GPU temperature.

**Parameters:**
- `$iGPUIndex` - [optional] GPU index (default: 0)

**Returns:** Temperature in Celsius

#### `_HWMon_NVIDIA_GetGPUName([$iGPUIndex = 0])`
Get NVIDIA GPU name.

**Parameters:**
- `$iGPUIndex` - [optional] GPU index (default: 0)

**Returns:** GPU name string

### **AMD Radeon GPU Functions**

#### `_HWMon_AMDRadeon_GetGPUCount()`
Get AMD Radeon GPU count.

**Returns:** Number of GPUs

#### `_HWMon_AMDRadeon_GetGPUTemp([$iGPUIndex = 0])`
Get AMD Radeon GPU temperature.

**Parameters:**
- `$iGPUIndex` - [optional] GPU index (default: 0)

**Returns:** Temperature in Celsius

#### `_HWMon_AMDRadeon_GetGPUName([$iGPUIndex = 0])`
Get AMD Radeon GPU name.

**Parameters:**
- `$iGPUIndex` - [optional] GPU index (default: 0)

**Returns:** GPU name string

### **Storage Functions**

#### `_HWMon_Storage_GetDriveCount()`
Get storage drive count.

**Returns:** Number of drives

#### `_HWMon_Storage_GetDriveTemp($iDriveIndex)`
Get storage drive temperature.

**Parameters:**
- `$iDriveIndex` - Drive index (0-based)

**Returns:** Temperature in Celsius

#### `_HWMon_Storage_GetDriveModel($iDriveIndex)`
Get storage drive model.

**Parameters:**
- `$iDriveIndex` - Drive index (0-based)

**Returns:** Drive model string

#### `_HWMon_Storage_GetDriveType($iDriveIndex)`
Get storage drive type.

**Parameters:**
- `$iDriveIndex` - Drive index (0-based)

**Returns:** Drive type string (HDD/SSD/NVMe)

## 💡 Examples

### **Example 1: Simple Temperature Display**
```autoit
#include "HardwareMonitor.au3"

_HWMon_Startup()

Local $aInfo = _HWMon_GetModuleInfo()

If $aInfo[2] Then ; Intel CPU
    ConsoleWrite("CPU: " & _HWMon_Intel_GetPackageTemp() & "°C" & @CRLF)
EndIf

If $aInfo[5] Then ; NVIDIA GPU
    ConsoleWrite("GPU: " & _HWMon_NVIDIA_GetGPUTemp() & "°C" & @CRLF)
EndIf

_HWMon_Shutdown()
```

### **Example 2: Monitor All Hardware**
```autoit
#include "HardwareMonitor.au3"

_HWMon_Startup()

Local $aInfo = _HWMon_GetModuleInfo()

; CPU
If $aInfo[2] Then
    Local $aTemps = _HWMon_Intel_GetAllCoreTemps()
    ConsoleWrite("Intel CPU Cores:" & @CRLF)
    For $i = 0 To UBound($aTemps) - 1
        ConsoleWrite("  Core " & $i & ": " & $aTemps[$i] & "°C" & @CRLF)
    Next
EndIf

; GPU
If $aInfo[5] Then
    For $i = 0 To $aInfo[6] - 1
        ConsoleWrite("NVIDIA GPU " & $i & ": " & _HWMon_NVIDIA_GetGPUTemp($i) & "°C" & @CRLF)
    Next
EndIf

; Storage
If $aInfo[9] Then
    For $i = 0 To $aInfo[10] - 1
        ConsoleWrite("Drive " & $i & ": " & _HWMon_Storage_GetDriveTemp($i) & "°C" & @CRLF)
    Next
EndIf

_HWMon_Shutdown()
```

### **Example 3: Real-time Monitoring Loop**
```autoit
#include "HardwareMonitor.au3"

_HWMon_Startup()

While True
    ; Update every second
    ConsoleWrite("CPU: " & _HWMon_Intel_GetPackageTemp() & "°C | ")
    ConsoleWrite("GPU: " & _HWMon_NVIDIA_GetGPUTemp() & "°C" & @CRLF)
    
    Sleep(1000)
WEnd

_HWMon_Shutdown()
```

## ⚠️ Requirements

- **Windows x64** - Only 64-bit supported
- **Administrator rights** - Required for PawnIO driver (CPU temp)
- **AutoIt x64** - Must use 64-bit AutoIt

## 🔧 Error Handling

```autoit
If Not _HWMon_Startup() Then
    Switch @error
        Case 1
            ConsoleWrite("DLL not found!" & @CRLF)
        Case 2
            ConsoleWrite("Failed to open DLL!" & @CRLF)
        Case 3
            ConsoleWrite("Initialization failed!" & @CRLF)
    EndSwitch
    Exit
EndIf

; Check for warnings
Local $sError = _HWMon_GetLastError()
If $sError <> "" Then
    ConsoleWrite("Warning: " & $sError & @CRLF)
EndIf
```

## 📝 Notes

- **PawnIO driver** is optional - GPU and Storage work without it
- **Lazy initialization** - Modules auto-initialize on first use
- **Module detection** - Check `_HWMon_GetModuleInfo()` before use
- **Error messages** - Use `_HWMon_GetLastError()` for details

## 🎯 Best Practices

1. **Always check module availability:**
```autoit
Local $aInfo = _HWMon_GetModuleInfo()
If $aInfo[2] Then
    ; Intel CPU available
EndIf
```

2. **Handle errors gracefully:**
```autoit
If Not _HWMon_Startup() Then
    ; Fallback or exit
EndIf
```

3. **Cleanup on exit:**
```autoit
_HWMon_Shutdown()
```

4. **Update before reading:**
```autoit
; UDF handles this automatically
Local $fTemp = _HWMon_Intel_GetPackageTemp()
```

## 🚀 Performance

- **Initialization:** ~100ms (one-time)
- **Temperature read:** ~1-5ms per sensor
- **Module detection:** ~150ms (one-time)

## 📄 License

Same as HardwareMonitor.dll

---

**Happy monitoring!** 🎉
