; No architecture directive - works on both x86 and x64 now!
#include "JSONc_UDF.au3"
#include <String.au3>
#include <File.au3>
Global $sLogFile = StringReplace(StringReplace(@ScriptFullPath, '.au3', ''), '.exe', '') & '.log'
FileDelete($sLogFile)
; ===============================================================================================================================
; TEST hardware_data.json - Demonstration of JSONc_UDF.au3 with real hardware monitoring data
; ===============================================================================================================================

; Load the correct DLL path
;~ Local $sDllPath = @AutoItX64 ? @ScriptDir & "\json-c.dll" : @ScriptDir & "\json-c_x86.dll"
;~ If Not _JsonC_Startup($sDllPath) Then
If Not _JsonC_Startup() Then
;~ 	MsgBox(16, "Error", "Failed to load JSON-C DLL:" & $sDllPath)
	MsgBox(16, "Error", "Failed to load JSON-C DLL !")
	Exit
EndIf

_LogWrite(@CRLF & "=== TEST hardware_data.json USING JSONc_UDF.au3 ===" & @CRLF & @CRLF)

; ===============================================================================================================================
; 1. LOAD AND PARSE JSON FILE
; ===============================================================================================================================
_LogWrite("1. LOADING HARDWARE DATA" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

; Read JSON file (using fixed version with large numbers as strings)
Local $sJsonFile = @ScriptDir & "\hardware_data.json"
If Not FileExists($sJsonFile) Then
	_LogWrite("Fixed JSON not found, using original..." & @CRLF)
	$sJsonFile = @ScriptDir & "\hardware_data.json"
EndIf
Local $sJsonContent = FileRead($sJsonFile)

If @error Or $sJsonContent = "" Then
	_LogWrite("Error: Cannot read file " & $sJsonFile & @CRLF)
	Exit
EndIf

_LogWrite("✓ File loaded: " & $sJsonFile & @CRLF)
_LogWrite("✓ File size: " & StringLen($sJsonContent) & " characters" & @CRLF)

; Parse JSON directly (skip validation - it crashes on x86)
_LogWrite("Parsing JSON..." & @CRLF)
Local $mHardware = _JsonC_Parse($sJsonContent)
Local $iParseError = @error
If $iParseError Then
	_LogWrite("✗ ERROR: JSON không hợp lệ hoặc parse thất bại" & @CRLF)
	_LogWrite("  Error code: " & $iParseError & @CRLF)
	_LogWrite("  File: " & $sJsonFile & @CRLF)
	_LogWrite("  Vấn đề có thể:" & @CRLF)
	_LogWrite("    - JSON syntax sai (thiếu/thừa dấu ngoặc)" & @CRLF)
	_LogWrite("    - JSON bị cắt hoặc dính nhiều objects" & @CRLF)
	_LogWrite("    - Số quá lớn (cần chuyển thành string)" & @CRLF)
	_LogWrite("    - File bị corrupt" & @CRLF)
	_JsonC_Shutdown()
	Exit 1
EndIf

_LogWrite("✓ JSON parsed successfully!" & @CRLF & @CRLF)

; ===============================================================================================================================
; 2. BASIC STATUS INFORMATION
; ===============================================================================================================================
_LogWrite("2. BASIC STATUS" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

_LogWrite("Initialized: " & (_JsonC_Get($mHardware, "initialized") ? "Yes" : "No") & @CRLF)
_LogWrite("Has Pawn IO: " & (_JsonC_Get($mHardware, "hasPawnIO") ? "Yes" : "No") & @CRLF)
_LogWrite(@CRLF)

; ===============================================================================================================================
; 3. INTEL CPU INFORMATION
; ===============================================================================================================================
_LogWrite("3. INTEL CPU INFORMATION" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

If _JsonC_Get($mHardware, "intelCPU.available", False) Then
	_LogWrite("✓ Intel CPU Available" & @CRLF)
	_LogWrite("  Core Count: " & _JsonC_Get($mHardware, "intelCPU.coreCount") & @CRLF)
	_LogWrite("  TjMax: " & _JsonC_Get($mHardware, "intelCPU.tjMax") & "°C" & @CRLF)
	_LogWrite("  Package Temp: " & _JsonC_Get($mHardware, "intelCPU.packageTemp") & "°C" & @CRLF)
	_LogWrite("  Average Temp: " & _JsonC_Get($mHardware, "intelCPU.averageTemp") & "°C" & @CRLF)
	_LogWrite("  Max Temp: " & _JsonC_Get($mHardware, "intelCPU.maxTemp") & "°C" & @CRLF)

	; Show first 5 core temperatures
	Local $aCoreTemps = _JsonC_Get($mHardware, "intelCPU.coreTemps")
	_LogWrite(@CRLF & "  First 5 Core Temperatures:" & @CRLF)
	For $i = 0 To 4
		If $i < UBound($aCoreTemps) Then
			Local $mCore = $aCoreTemps[$i]
			_LogWrite("    Core " & $mCore["core"] & ": " & $mCore["temp"] & "°C" & @CRLF)
		EndIf
	Next
Else
	_LogWrite("✗ Intel CPU not available" & @CRLF)
EndIf
_LogWrite(@CRLF)

; ===============================================================================================================================
; 4. AMD CPU INFORMATION
; ===============================================================================================================================
_LogWrite("4. AMD CPU INFORMATION" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

If _JsonC_Get($mHardware, "amdCPU.available", False) Then
	_LogWrite("✓ AMD CPU Available" & @CRLF)
Else
	_LogWrite("✗ AMD CPU not available" & @CRLF)
EndIf
_LogWrite(@CRLF)

; ===============================================================================================================================
; 5. GPU INFORMATION
; ===============================================================================================================================
_LogWrite("5. GPU INFORMATION" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

; NVIDIA GPUs
Local $aNvidiaGPUs = _JsonC_Get($mHardware, "nvidiaGPUs")
If Not IsArray($aNvidiaGPUs) Then Local $aNvidiaGPUs[0]
_LogWrite("NVIDIA GPUs: " & UBound($aNvidiaGPUs) & " found" & @CRLF)

; AMD Radeon GPUs
Local $aAmdGPUs = _JsonC_Get($mHardware, "amdRadeonGPUs")
If Not IsArray($aAmdGPUs) Then Local $aAmdGPUs[0]
_LogWrite("AMD Radeon GPUs: " & UBound($aAmdGPUs) & " found" & @CRLF)

If UBound($aAmdGPUs) > 0 Then
	_LogWrite(@CRLF & "AMD GPU Details:" & @CRLF)
	For $i = 0 To UBound($aAmdGPUs) - 1
		Local $mGPU = $aAmdGPUs[$i]
		_LogWrite("  GPU #" & $mGPU["index"] & ": " & $mGPU["name"] & @CRLF)
		_LogWrite("    Temperature: " & $mGPU["temperature"] & "°C" & @CRLF)
		_LogWrite("    Hotspot: " & $mGPU["hotspotTemp"] & "°C" & @CRLF)
		_LogWrite("    Memory Temp: " & $mGPU["memoryTemp"] & "°C" & @CRLF)
		_LogWrite("    Core Clock: " & $mGPU["coreClock"] & " MHz" & @CRLF)
		_LogWrite("    Memory Clock: " & $mGPU["memoryClock"] & " MHz" & @CRLF)
		_LogWrite("    Fan Speed: " & $mGPU["fanSpeedPercent"] & "%" & @CRLF)
		_LogWrite("    GPU Usage: " & $mGPU["gpuUsage"] & "%" & @CRLF)
		_LogWrite("    Power Usage: " & $mGPU["powerUsage"] & " W" & @CRLF)
	Next
EndIf
_LogWrite(@CRLF)

; ===============================================================================================================================
; 6. STORAGE DRIVES INFORMATION
; ===============================================================================================================================
_LogWrite("6. STORAGE DRIVES" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

Local $aStorageDrives = _JsonC_Get($mHardware, "storageDrives")
If Not IsArray($aStorageDrives) Then Local $aStorageDrives[0]
_LogWrite("Total Drives: " & UBound($aStorageDrives) & @CRLF & @CRLF)

For $i = 0 To UBound($aStorageDrives) - 1
	Local $mDrive = $aStorageDrives[$i]
	_LogWrite("Drive #" & $mDrive["index"] & ": " & $mDrive["model"] & @CRLF)
	_LogWrite("  Type: " & $mDrive["type"] & ($mDrive["isSSD"] ? " (SSD)" : " (HDD)") & @CRLF)
	_LogWrite("  Name: " & $mDrive["name"] & @CRLF)
	If $mDrive["hasTemperature"] Then
		_LogWrite("  Temperature: " & $mDrive["temperature"] & "°C" & @CRLF)
	EndIf
	_LogWrite("  SMART: " & ($mDrive["hasSMART"] ? "Available" : "Not Available") & @CRLF)
	_LogWrite(@CRLF)
Next

; ===============================================================================================================================
; 7. RAM INFORMATION
; ===============================================================================================================================
_LogWrite("7. MEMORY (RAM) INFORMATION" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

If _JsonC_Get($mHardware, "ram.available", False) Then
	Local $iTotalMemory = _JsonC_Get($mHardware, "ram.totalMemory")
	Local $iUsedMemory = _JsonC_Get($mHardware, "ram.usedMemory")
	Local $iAvailableMemory = _JsonC_Get($mHardware, "ram.availableMemory")

	_LogWrite("✓ RAM Information Available" & @CRLF)
	_LogWrite("  Total Memory: " & Round($iTotalMemory / 1024 / 1024 / 1024, 2) & " GB" & @CRLF)
	_LogWrite("  Used Memory: " & Round($iUsedMemory / 1024 / 1024 / 1024, 2) & " GB" & @CRLF)
	_LogWrite("  Available Memory: " & Round($iAvailableMemory / 1024 / 1024 / 1024, 2) & " GB" & @CRLF)
	_LogWrite("  Memory Load: " & _JsonC_Get($mHardware, "ram.memoryLoad") & "%" & @CRLF)
	_LogWrite("  Memory Speed: " & _JsonC_Get($mHardware, "ram.memorySpeed") & " MHz" & @CRLF)
	_LogWrite("  Configured Speed: " & _JsonC_Get($mHardware, "ram.memorySpeedConfigured") & " MHz" & @CRLF)
	_LogWrite("  Module Count: " & _JsonC_Get($mHardware, "ram.moduleCount") & @CRLF)

	; Show memory modules
	Local $aModules = _JsonC_Get($mHardware, "ram.modules")
	If Not IsArray($aModules) Then Local $aModules[0]
	If UBound($aModules) > 0 Then
		_LogWrite(@CRLF & "  Memory Modules:" & @CRLF)
		For $i = 0 To UBound($aModules) - 1
			Local $mModule = $aModules[$i]
			_LogWrite("    Module #" & $mModule["index"] & ": " & _
					Round($mModule["capacity"] / 1024 / 1024 / 1024, 2) & " GB" & _
					" @ " & $mModule["speed"] & " MHz" & @CRLF)
			_LogWrite("      Manufacturer: " & $mModule["manufacturer"] & @CRLF)
			_LogWrite("      Part Number: " & StringStripWS($mModule["partNumber"], 3) & @CRLF)
		Next
	EndIf
Else
	_LogWrite("✗ RAM information not available" & @CRLF)
EndIf
_LogWrite(@CRLF)

; ===============================================================================================================================
; 8. PATH-BASED QUERIES DEMONSTRATION
; ===============================================================================================================================
_LogWrite("8. PATH-BASED QUERIES DEMO" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

_LogWrite("Using _JsonC_Get() with paths:" & @CRLF)
_LogWrite("  CPU Package Temp: " & _JsonC_Get($mHardware, "intelCPU.packageTemp", "N/A") & "°C" & @CRLF)
_LogWrite("  First GPU Name: " & _JsonC_Get($mHardware, "amdRadeonGPUs.0.name", "N/A") & @CRLF)
_LogWrite("  First Drive Model: " & _JsonC_Get($mHardware, "storageDrives.0.model", "N/A") & @CRLF)
_LogWrite("  Memory Load: " & _JsonC_Get($mHardware, "ram.memoryLoad", "N/A") & "%" & @CRLF)
_LogWrite(@CRLF)

; ===============================================================================================================================
; 9. CHECK DATA TYPES
; ===============================================================================================================================
_LogWrite("9. DATA TYPE INFORMATION" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

_LogWrite("Type of 'initialized': " & _JsonC_GetTypeStr($mHardware, "initialized") & @CRLF)
_LogWrite("Type of 'intelCPU': " & _JsonC_GetTypeStr($mHardware, "intelCPU") & @CRLF)
_LogWrite("Type of 'intelCPU.coreCount': " & _JsonC_GetTypeStr($mHardware, "intelCPU.coreCount") & @CRLF)
_LogWrite("Type of 'intelCPU.coreTemps': " & _JsonC_GetTypeStr($mHardware, "intelCPU.coreTemps") & @CRLF)
_LogWrite("Type of 'amdRadeonGPUs': " & _JsonC_GetTypeStr($mHardware, "amdRadeonGPUs") & @CRLF)
_LogWrite(@CRLF)

; ===============================================================================================================================
; 10. MODIFY DATA AND SAVE
; ===============================================================================================================================
_LogWrite("10. MODIFY DATA" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

; Add a timestamp
_JsonC_Set($mHardware, "lastUpdated", @YEAR & "-" & @MON & "-" & @MDAY & " " & @HOUR & ":" & @MIN & ":" & @SEC)
_LogWrite("✓ Added timestamp: " & _JsonC_Get($mHardware, "lastUpdated") & @CRLF)

; Add test note
_JsonC_Set($mHardware, "testNote", "Data tested with JSONc_UDF.au3")
_LogWrite("✓ Added test note" & @CRLF)

; Modify CPU temperature (for demo purposes)
_JsonC_Set($mHardware, "intelCPU.packageTemp", "45")
_LogWrite("✓ Updated CPU package temp to: " & _JsonC_Get($mHardware, "intelCPU.packageTemp") & "°C" & @CRLF)
_LogWrite(@CRLF)

; ===============================================================================================================================
; 11. GENERATE AND SAVE MODIFIED JSON
; ===============================================================================================================================
_LogWrite("11. SAVE MODIFIED DATA" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

; Generate pretty JSON
Local $sModifiedJson = _JsonC_GeneratePretty($mHardware)

; Save to new file
Local $sOutputFile = @ScriptDir & "\hardware_data_new.json"
Local $hFile = FileOpen($sOutputFile, 2) ; 2 = overwrite mode
If $hFile = -1 Then
	_LogWrite("Error: Cannot create output file" & @CRLF)
Else
	FileWrite($hFile, $sModifiedJson)
	FileClose($hFile)
	_LogWrite("✓ Modified data saved to: " & $sOutputFile & @CRLF)
EndIf
_LogWrite(@CRLF)

; ===============================================================================================================================
; 12. UTILITY FUNCTIONS DEMO
; ===============================================================================================================================
_LogWrite("12. UTILITY FUNCTIONS" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

; Get all root-level keys
Local $aRootKeys = _JsonC_Keys($mHardware)
_LogWrite("Root-level keys (" & UBound($aRootKeys) & "):" & @CRLF)
_LogWrite("  " & __ArrayToStr($aRootKeys, ", ") & @CRLF)
_LogWrite(@CRLF)

; Get keys from nested object
If _JsonC_Has($mHardware, "intelCPU") Then
	Local $aCpuKeys = _JsonC_Keys($mHardware, "intelCPU")
	_LogWrite("Intel CPU keys (" & UBound($aCpuKeys) & "):" & @CRLF)
	_LogWrite("  " & __ArrayToStr($aCpuKeys, ", ") & @CRLF)
	_LogWrite(@CRLF)
EndIf

; ===============================================================================================================================
; 13. VALIDATION TEST
; ===============================================================================================================================
_LogWrite("13. VALIDATION TEST" & @CRLF)
_LogWrite(_StringRepeat("-", 70) & @CRLF)

; Test if original content is valid
_LogWrite("Original JSON is valid: " & (_JsonC_IsValid($sJsonContent) ? "Yes" : "No") & @CRLF)

; Test if modified JSON is valid
_LogWrite("Modified JSON is valid: " & (_JsonC_IsValid($sModifiedJson) ? "Yes" : "No") & @CRLF)

; Test invalid JSON
Local $sInvalidJson = '{"test": invalid}'
_LogWrite("Invalid JSON test: " & (_JsonC_IsValid($sInvalidJson) ? "Valid" : "Invalid (as expected)") & @CRLF)
_LogWrite(@CRLF)

; ===============================================================================================================================
; SUMMARY
; ===============================================================================================================================
_LogWrite("=== TEST COMPLETED SUCCESSFULLY ===" & @CRLF)
_LogWrite("Summary:" & @CRLF)
_LogWrite("  ✓ Loaded and parsed hardware_data.json" & @CRLF)
_LogWrite("  ✓ Read CPU, GPU, Storage, and RAM information" & @CRLF)
_LogWrite("  ✓ Used path-based queries (_JsonC_Get)" & @CRLF)
_LogWrite("  ✓ Modified data with _JsonC_Set" & @CRLF)
_LogWrite("  ✓ Generated and saved modified JSON" & @CRLF)
_LogWrite("  ✓ Used utility functions (_JsonC_Keys, _JsonC_Has)" & @CRLF)
_LogWrite("  ✓ Validated JSON data" & @CRLF)
_LogWrite(@CRLF)

; Cleanup
_JsonC_Shutdown()

; ===============================================================================================================================
; HELPER FUNCTIONS
; ===============================================================================================================================

; Convert array to comma-separated string (custom version to avoid conflict)
Func __ArrayToStr($aArray, $sDelimiter = ",")
	Local $sResult = ""
	For $vElement In $aArray
		$sResult &= $vElement & $sDelimiter
	Next
	Return StringTrimRight($sResult, StringLen($sDelimiter))
EndFunc   ;==>__ArrayToStr

Func _LogWrite($s)
	ConsoleWrite($s)
	FileWrite($sLogFile, $s)
EndFunc   ;==>_LogWrite
