#include-once

; #INDEX# =======================================================================================================================
; Title .........: JsonC UDF Enhanced v1.2
; AutoIt Version : 3.3.16.0
; Description ...: Enhanced JSON library using JSON-C with high-level API, path-based queries, and utility functions.
;                  Combines the best features from JSON_C.au3 and JSON_UDF.au3 with optimizations.
; Author(s) .....: Sean Griffin (original), Enhanced by Dao Van Trong - TRONG.PRO
; Dll ...........: json-c.dll (based on v0.18-20240915)
; Link ..........: https://github.com/json-c/json-c
; ===============================================================================================================================
;
; #ENHANCEMENTS# ================================================================================================================
; 1. High-Level API:
;    - _JsonC_Parse: Parse JSON string to AutoIt Maps/Arrays (easy to work with)
;    - _JsonC_Generate: Convert AutoIt data to JSON string with formatting options
;    - _JsonC_GeneratePretty: Pretty-print JSON output
;
; 2. Path-Based Queries:
;    - _JsonC_Get: Get value by path (e.g., "user.address.city" or "items[0].name")
;    - _JsonC_Set: Set value by path (creates nested structures automatically)
;    - _JsonC_Delete: Delete value by path
;    - _JsonC_Has: Check if path exists
;
; 3. Utility Functions:
;    - _JsonC_Keys: Get all keys from a Map/Object
;    - _JsonC_Values: Get all values from a Map/Object or Array
;    - _JsonC_IsValid: Validate JSON string
;    - _JsonC_GetTypeStr: Get human-readable type name
;
; 4. Optimizations:
;    - Auto-loading DLL on first use
;    - Improved error handling with meaningful error codes
;    - Better memory management
;    - String escaping/unescaping helpers
; ===============================================================================================================================

; #VARIABLES# ===================================================================================================================
Global $__g_hDll_JsonC = 0
Global $__g_bDll_AutoLoaded = False
Global $PtrSize = @AutoItX64 ? 8 : 4
; ===============================================================================================================================

; #CONSTANTS# ===================================================================================================================
Global Enum _
		$JSONC_TYPE_NULL, _
		$JSONC_TYPE_BOOLEAN, _
		$JSONC_TYPE_DOUBLE, _
		$JSONC_TYPE_INT, _
		$JSONC_TYPE_OBJECT, _
		$JSONC_TYPE_ARRAY, _
		$JSONC_TYPE_STRING

Global Const $tagJSONC_OBJECT = _
		"struct;" & _
		"int   o_type;" & _
		"uint  _ref_count;" & _
		"ptr   _to_json_string;" & _
		"ptr   _pb;" & _
		"ptr   _user_delete;" & _
		"ptr   _userdata;" & _
		"endstruct;"

; JSON formatting flags
Global Const $JSON_C_TO_STRING_PLAIN = 0
Global Const $JSON_C_TO_STRING_SPACED = 1
Global Const $JSON_C_TO_STRING_PRETTY = 2
Global Const $JSON_C_TO_STRING_PRETTY_TAB = 8
Global Const $JSON_C_TO_STRING_NOZERO = 4

; Iterator structure for efficient object traversal
Global Const $tagJSON_OBJECT_ITERATOR = "ptr opaque_;"
; ===============================================================================================================================

; #CURRENT# =====================================================================================================================
; === Core Functions ===
; _JsonC_Startup             - Load json-c.dll
; _JsonC_Shutdown            - Unload json-c.dll

; === High-Level API (ENHANCED) ===
; _JsonC_Parse               - Parse JSON string to AutoIt Map/Array (easy to use!)
; _JsonC_Generate            - Convert AutoIt data to JSON string
; _JsonC_GeneratePretty      - Convert AutoIt data to pretty-printed JSON
; _JsonC_IsValid             - Validate JSON string

; === Path-Based Queries (NEW) ===
; _JsonC_Get                 - Get value by path (e.g., "user.name" or "items[0]")
; _JsonC_Set                 - Set value by path (auto-creates nested structures)
; _JsonC_Delete              - Delete value by path
; _JsonC_Has                 - Check if path exists
; _JsonC_GetTypeStr          - Get type of value at path

; === Utility Functions (NEW) ===
; _JsonC_Keys                - Get all keys from Map/Object
; _JsonC_Values              - Get all values from Map/Object or Array

; === Low-Level DLL Functions ===
; _JsonC_ObjectNewObject
; _JsonC_ObjectNewBoolean
; _JsonC_ObjectNewInt
; _JsonC_ObjectNewInt64
; _JsonC_ObjectNewDouble
; _JsonC_ObjectNewString
; _JsonC_ObjectNewArray
; _JsonC_ObjectArrayAdd
; _JsonC_ObjectObjectAdd
; _JsonC_ObjectObjectDel
; _JsonC_ObjectObjectGet
; _JsonC_TokenerParse
; _JsonC_ObjectToJsonString
; _JsonC_ObjectIsType
; _JsonC_ObjectGetType
; _JsonC_ObjectGetString
; _JsonC_ObjectGetBoolean
; _JsonC_ObjectGetDouble
; _JsonC_ObjectGetInteger
; _JsonC_ObjectGetObject
; _JsonC_ObjectGetValue
; _JsonC_ObjectGetFieldValue
; _JsonC_ObjectGetArray
; _JsonC_ArrayListLength
; _JsonC_ArrayListGetIndex
; _JsonC_ObjectArrayLength
; _JsonC_ObjectArrayGetIndex
; _JsonC_ObjectArrayGetObjects
; ===============================================================================================================================

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Startup
; Description ...: Loads json-c.dll
; Syntax ........: _JsonC_Startup($sDll_Filename = "")
; Parameters ....: $sDll_Filename - [Optional] DLL path or "" for auto-detect
; Return values .: Success - Return DLL filename
;                  Failure - Return "" and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_Startup($sDll_Filename = "")
	If $__g_hDll_JsonC <> 0 Then Return True ; Already loaded

	If $sDll_Filename = Default Or $sDll_Filename = -1 Then $sDll_Filename = ""
	If $sDll_Filename = "" Then $sDll_Filename = @AutoItX64 ? @ScriptDir & "\json-c.dll" : @ScriptDir & "\json-c_x86.dll"

	Local $hDll = DllOpen($sDll_Filename)
	If $hDll = -1 Then
		$__g_hDll_JsonC = 0
		Return SetError(1, 0, False)
	EndIf

	$__g_hDll_JsonC = $hDll
	Return $sDll_Filename
EndFunc   ;==>_JsonC_Startup

; #INTERNAL FUNCTION# ===============================================================================
; Name ..........: __JsonC_EnsureLoaded
; Description ...: Ensure DLL is loaded (auto-load if needed)
; Return values .: True if loaded, False otherwise
; =================================================================================================
Func __JsonC_EnsureLoaded()
	If $__g_hDll_JsonC <> 0 Then Return True

	; Auto-load DLL
	$__g_bDll_AutoLoaded = True
	Return _JsonC_Startup() <> False
EndFunc   ;==>__JsonC_EnsureLoaded

; ===============================================================================================================================
; #HIGH-LEVEL API FUNCTIONS# ====================================================================================================
; ===============================================================================================================================

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Parse
; Description ...: Parse JSON string into AutoIt data structure (Map for objects, Array for arrays)
; Syntax ........: _JsonC_Parse($sJsonString)
; Parameters ....: $sJsonString - JSON formatted string
; Return values .: Success - AutoIt data structure (Map[], Array[], String, Number, Boolean, Null)
;                  Failure - "" and sets @error:
;                     @error = 1: DLL not loaded or load failed
;                     @error = 2: Parse error (invalid JSON)
;                     @error = 3: Memory allocation error
; Example .......: $mData = _JsonC_Parse('{"name":"John","age":30}')
;                  ConsoleWrite($mData["name"] & @CRLF)  ; Output: John
; Author ........: Dao Van Trong - TRONG.PRO
; =================================================================================================
Func _JsonC_Parse($sJsonString)
	If Not __JsonC_EnsureLoaded() Then Return SetError(1, 0, "")

	; Parse JSON string with error checking
	; Use json_tokener_parse_verbose to get error code
	Local $tError = DllStructCreate("int")
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_tokener_parse_verbose", "str", $sJsonString, "ptr", DllStructGetPtr($tError))

	If @error Or Not IsArray($avRval) Then
		Return SetError(2, 0, "")
	EndIf

	; Check error code (0 = success)
	Local $iError = DllStructGetData($tError, 1)
	If $iError <> 0 Then
		; Parse error
		Return SetError(2, $iError, "")
	EndIf

	Local $pJsonObj = $avRval[0]

	; Even for null, we should get a valid pointer
	; Only return error if pointer is 0 AND error code was set
	If $pJsonObj = 0 And $iError <> 0 Then
		Return SetError(2, $iError, "")
	EndIf

	; Convert json_object to AutoIt structure
	Local $vResult = __JsonC_ObjectToAutoIt($pJsonObj)

	; Free json_object (decrease ref count)
	If $pJsonObj <> 0 Then
		DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_put", "ptr", $pJsonObj)
	EndIf

	If @error Then Return SetError(3, 0, "")

	Return $vResult
EndFunc   ;==>_JsonC_Parse

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Generate
; Description ...: Convert AutoIt data structure to JSON string
; Syntax ........: _JsonC_Generate($vData, $iFlags = $JSON_C_TO_STRING_PLAIN)
; Parameters ....: $vData  - AutoIt data structure (Map, Array, String, Number, Boolean, Null)
;                  $iFlags - [optional] Formatting flags (default: plain, no whitespace)
; Return values .: Success - JSON formatted string
;                  Failure - "" and sets @error:
;                     @error = 1: DLL not loaded
;                     @error = 2: Conversion error
; Example .......: Local $mData[]
;                  $mData["name"] = "John"
;                  $mData["age"] = 30
;                  ConsoleWrite(_JsonC_Generate($mData) & @CRLF)
; Author ........: Dao Van Trong - TRONG.PRO
; =================================================================================================
Func _JsonC_Generate($vData, $iFlags = $JSON_C_TO_STRING_PLAIN)
	If Not __JsonC_EnsureLoaded() Then Return SetError(1, 0, "")

	; Convert AutoIt structure to json_object
	Local $pJsonObj = __JsonC_AutoItToObject($vData)
	If @error Or $pJsonObj = 0 Then Return SetError(2, 0, "")

	; Convert to JSON string with formatting
	Local $avRval = DllCall($__g_hDll_JsonC, "str:cdecl", "json_object_to_json_string_ext", "ptr", $pJsonObj, "int", $iFlags)

	Local $sJson = ""
	If Not @error And IsArray($avRval) Then
		$sJson = $avRval[0]
	EndIf

	; Free json_object
	DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_put", "ptr", $pJsonObj)

	If $sJson = "" Then Return SetError(2, 0, "")

	Return $sJson
EndFunc   ;==>_JsonC_Generate

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_GeneratePretty
; Description ...: Convert AutoIt data structure to pretty-printed JSON string
; Syntax ........: _JsonC_GeneratePretty($vData, $bUseTabs = False)
; Parameters ....: $vData    - AutoIt data structure
;                  $bUseTabs - [optional] Use tabs instead of 2 spaces (default: False)
; Return values .: Success - Pretty-printed JSON string
;                  Failure - "" and sets @error
; Author ........: Dao Van Trong - TRONG.PRO
; =================================================================================================
Func _JsonC_GeneratePretty($vData, $bUseTabs = False)
	Local $iFlags = $bUseTabs ? $JSON_C_TO_STRING_PRETTY_TAB : $JSON_C_TO_STRING_PRETTY
	Return _JsonC_Generate($vData, $iFlags)
EndFunc   ;==>_JsonC_GeneratePretty

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_IsValid
; Description ...: Validate if a string is valid JSON
; Syntax ........: _JsonC_IsValid($sJsonString)
; Parameters ....: $sJsonString - String to validate
; Return values .: True if valid JSON, False otherwise
; =================================================================================================
Func _JsonC_IsValid($sJsonString)
	If Not __JsonC_EnsureLoaded() Then Return False

	; Use json_tokener_parse_verbose to properly check errors
	Local $tError = DllStructCreate("int")
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_tokener_parse_verbose", "str", $sJsonString, "ptr", DllStructGetPtr($tError))

	If @error Or Not IsArray($avRval) Then Return False

	; Check error code (0 = success)
	Local $iError = DllStructGetData($tError, 1)
	Local $pObj = $avRval[0]

	; Free the parsed object if valid
	If $pObj <> 0 Then
		DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_put", "ptr", $pObj)
	EndIf

	Return ($iError = 0)
EndFunc   ;==>_JsonC_IsValid

; ===============================================================================================================================
; #INTERNAL CONVERSION FUNCTIONS# ===============================================================================================
; ===============================================================================================================================

; #INTERNAL FUNCTION# ===============================================================================
; Name ..........: __JsonC_ObjectToAutoIt
; Description ...: Convert json_object pointer to AutoIt data structure
; Parameters ....: $pJsonObj - Pointer to json_object
; Return values .: AutoIt data structure (Map, Array, or scalar)
; =================================================================================================
Func __JsonC_ObjectToAutoIt($pJsonObj)
	If $pJsonObj = 0 Then Return Null

	; Get object type
	Local $avType = DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_get_type", "ptr", $pJsonObj)
	If @error Or Not IsArray($avType) Then Return Null

	Local $iType = $avType[0]

	Switch $iType
		Case $JSONC_TYPE_NULL
			Return Null

		Case $JSONC_TYPE_BOOLEAN
			Local $avBool = DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_get_boolean", "ptr", $pJsonObj)
			Return IsArray($avBool) ? ($avBool[0] <> 0) : False

		Case $JSONC_TYPE_INT
			Local $avInt = DllCall($__g_hDll_JsonC, "int64:cdecl", "json_object_get_int64", "ptr", $pJsonObj)
			Return IsArray($avInt) ? $avInt[0] : 0

		Case $JSONC_TYPE_DOUBLE
			Local $avDouble = DllCall($__g_hDll_JsonC, "double:cdecl", "json_object_get_double", "ptr", $pJsonObj)
			Return IsArray($avDouble) ? $avDouble[0] : 0.0

		Case $JSONC_TYPE_STRING
			Local $avStr = DllCall($__g_hDll_JsonC, "str:cdecl", "json_object_get_string", "ptr", $pJsonObj)
			Return IsArray($avStr) ? $avStr[0] : ""

		Case $JSONC_TYPE_ARRAY
			Return __JsonC_ArrayToAutoIt($pJsonObj)

		Case $JSONC_TYPE_OBJECT
			Return __JsonC_ObjectMapToAutoIt($pJsonObj)

		Case Else
			Return Null
	EndSwitch
EndFunc   ;==>__JsonC_ObjectToAutoIt

; #INTERNAL FUNCTION# ===============================================================================
; Name ..........: __JsonC_ArrayToAutoIt
; Description ...: Convert json array to AutoIt Array
; =================================================================================================
Func __JsonC_ArrayToAutoIt($pJsonArray)
	; Get array length (size_t = 32-bit on x86, 64-bit on x64)
	Local $sLenType = @AutoItX64 ? "uint64:cdecl" : "uint:cdecl"
	Local $avLen = DllCall($__g_hDll_JsonC, $sLenType, "json_object_array_length", "ptr", $pJsonArray)
	If @error Or Not IsArray($avLen) Then Return SetError(1, 0, Null)

	Local $iLen = $avLen[0]
	If $iLen = 0 Then
		Local $aEmpty[0]
		Return $aEmpty
	EndIf

	; Create AutoIt array
	Local $aResult[$iLen]

	For $i = 0 To $iLen - 1
		; Get element at index
		Local $avElem = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_array_get_idx", "ptr", $pJsonArray, "int", $i)
		If Not @error And IsArray($avElem) And $avElem[0] <> 0 Then
			$aResult[$i] = __JsonC_ObjectToAutoIt($avElem[0])
		Else
			$aResult[$i] = Null
		EndIf
	Next

	Return $aResult
EndFunc   ;==>__JsonC_ArrayToAutoIt

; #INTERNAL FUNCTION# ===============================================================================
; Name ..........: __JsonC_ObjectMapToAutoIt
; Description ...: Convert json object to AutoIt Map using ITERATOR API (OPTIMIZED!)
; Performance ...: O(n) instead of O(n³) - Uses json-c iterator functions for efficiency
; =================================================================================================
Func __JsonC_ObjectMapToAutoIt($pJsonObj)
	Local $mResult[]

	; Get begin iterator (returns iterator struct by value, stored as pointer)
	Local $avBegin = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_iter_begin", "ptr", $pJsonObj)
	If @error Or Not IsArray($avBegin) Then Return $mResult
	Local $pIterBegin = $avBegin[0]

	; Get end iterator
	Local $avEnd = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_iter_end", "ptr", $pJsonObj)
	If @error Or Not IsArray($avEnd) Then Return $mResult
	Local $pIterEnd = $avEnd[0]

	; Create iterator struct for current position
	Local $tIter = DllStructCreate($tagJSON_OBJECT_ITERATOR)
	Local $tIterEnd = DllStructCreate($tagJSON_OBJECT_ITERATOR)
	DllStructSetData($tIter, "opaque_", $pIterBegin)
	DllStructSetData($tIterEnd, "opaque_", $pIterEnd)

	; Check if object is empty (begin == end)
	Local $avEqual = DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_iter_equal", _
			"ptr", DllStructGetPtr($tIter), "ptr", DllStructGetPtr($tIterEnd))
	If Not @error And IsArray($avEqual) And $avEqual[0] <> 0 Then
		; Empty object
		Return $mResult
	EndIf

	; Iterate through all key-value pairs
	Local $iMaxIter = 10000 ; Safety limit to prevent infinite loops
	Local $iCount = 0

	While $iCount < $iMaxIter
		$iCount += 1

		; Get key name
		Local $avKey = DllCall($__g_hDll_JsonC, "str:cdecl", "json_object_iter_peek_name", _
				"ptr", DllStructGetPtr($tIter))
		If @error Or Not IsArray($avKey) Then ExitLoop
		Local $sKey = $avKey[0]

		; Get value pointer
		Local $avVal = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_iter_peek_value", _
				"ptr", DllStructGetPtr($tIter))
		If Not @error And IsArray($avVal) Then
			Local $pValue = $avVal[0]
			If $pValue <> 0 Then
				; Recursively convert the value
				$mResult[$sKey] = __JsonC_ObjectToAutoIt($pValue)
			Else
				$mResult[$sKey] = Null
			EndIf
		EndIf

		; Move to next pair
		DllCall($__g_hDll_JsonC, "none:cdecl", "json_object_iter_next", "ptr", DllStructGetPtr($tIter))
		If @error Then ExitLoop

		; Check if we reached the end
		Local $avEq = DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_iter_equal", _
				"ptr", DllStructGetPtr($tIter), "ptr", DllStructGetPtr($tIterEnd))
		If Not @error And IsArray($avEq) And $avEq[0] <> 0 Then ExitLoop
	WEnd

	Return $mResult
EndFunc   ;==>__JsonC_ObjectMapToAutoIt

; #INTERNAL FUNCTION# ===============================================================================
; Name ..........: __JsonC_AutoItToObject
; Description ...: Convert AutoIt data structure to json_object pointer
; Parameters ....: $vData - AutoIt data (Map, Array, String, Number, Boolean, Null)
; Return values .: Pointer to json_object (must be freed with json_object_put)
; =================================================================================================
Func __JsonC_AutoItToObject($vData)
	Local $sType = VarGetType($vData)
	Local $avResult, $pObj

	Switch $sType
		Case "String"
			$avResult = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_string", "str", $vData)
			Return IsArray($avResult) ? $avResult[0] : 0

		Case "Int32", "Int64"
			$avResult = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_int64", "int64", $vData)
			Return IsArray($avResult) ? $avResult[0] : 0

		Case "Float", "Double"
			$avResult = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_double", "double", $vData)
			Return IsArray($avResult) ? $avResult[0] : 0

		Case "Bool"
			$avResult = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_boolean", "int", $vData ? 1 : 0)
			Return IsArray($avResult) ? $avResult[0] : 0

		Case "Keyword"
			; Null
			$avResult = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_null")
			Return IsArray($avResult) ? $avResult[0] : 0

		Case "Array"
			Return __JsonC_AutoItArrayToObject($vData)

		Case "Map"
			Return __JsonC_AutoItMapToObject($vData)

		Case Else
			; Unknown type, return null
			$avResult = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_null")
			Return IsArray($avResult) ? $avResult[0] : 0
	EndSwitch
EndFunc   ;==>__JsonC_AutoItToObject

; #INTERNAL FUNCTION# ===============================================================================
; Name ..........: __JsonC_AutoItArrayToObject
; Description ...: Convert AutoIt Array to json array object
; =================================================================================================
Func __JsonC_AutoItArrayToObject($aData)
	; Create new array object
	Local $avNew = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_array")
	If @error Or Not IsArray($avNew) Or $avNew[0] = 0 Then Return 0

	Local $pArray = $avNew[0]

	; Add elements
	For $vElement In $aData
		Local $pElement = __JsonC_AutoItToObject($vElement)
		If $pElement <> 0 Then
			; Add to array (array takes ownership, increases ref count)
			DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_array_add", "ptr", $pArray, "ptr", $pElement)
		EndIf
	Next

	Return $pArray
EndFunc   ;==>__JsonC_AutoItArrayToObject

; #INTERNAL FUNCTION# ===============================================================================
; Name ..........: __JsonC_AutoItMapToObject
; Description ...: Convert AutoIt Map to json object
; =================================================================================================
Func __JsonC_AutoItMapToObject($mData)
	; Create new object
	Local $avNew = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_object")
	If @error Or Not IsArray($avNew) Or $avNew[0] = 0 Then Return 0

	Local $pObject = $avNew[0]

	; Add key-value pairs
	For $sKey In MapKeys($mData)
		Local $vValue = $mData[$sKey]
		Local $pValue = __JsonC_AutoItToObject($vValue)

		If $pValue <> 0 Then
			; Add to object (object takes ownership)
			DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_object_add", _
					"ptr", $pObject, "str", $sKey, "ptr", $pValue)
		EndIf
	Next

	Return $pObject
EndFunc   ;==>__JsonC_AutoItMapToObject

; ===============================================================================================================================
; #PATH-BASED QUERY FUNCTIONS# ==================================================================================================
; ===============================================================================================================================

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Get
; Description ...: Get value from JSON data by path (supports dot notation)
; Syntax ........: _JsonC_Get($vData, $sPath, $vDefault = Null)
; Parameters ....: $vData    - JSON data (Map/Array from _JsonC_Parse)
;                  $sPath    - Path to value (e.g., "user.address.city" or "items[0].name")
;                  $vDefault - [Optional] Default value if path not found (Default: Null)
; Return values .: Value at path, or $vDefault if not found
; Example .......: _JsonC_Get($mData, "user.name")        ; Returns user name
;                  _JsonC_Get($mData, "items[0]")         ; Returns first item
;                  _JsonC_Get($mData, "missing", "N/A")   ; Returns "N/A" if not found
; Author ........: Dao Van Trong - TRONG.PRO
; =================================================================================================
Func _JsonC_Get($vData, $sPath, $vDefault = Null)
	If $sPath = "" Then Return $vData

	Local $aParts = StringSplit($sPath, ".", 2) ; Flag 2 = no count
	Local $vCurrent = $vData

	For $sPart In $aParts
		; Check for array index: key[0]
		Local $aMatch = StringRegExp($sPart, "^([^\[]+)\[(\d+)\]$", 1)
		If IsArray($aMatch) Then
			; Has array index
			Local $sKey = $aMatch[0]
			Local $iIndex = Int($aMatch[1])

			If IsMap($vCurrent) And MapExists($vCurrent, $sKey) Then
				$vCurrent = $vCurrent[$sKey]
				If IsArray($vCurrent) And $iIndex >= 0 And $iIndex < UBound($vCurrent) Then
					$vCurrent = $vCurrent[$iIndex]
				Else
					Return $vDefault
				EndIf
			Else
				Return $vDefault
			EndIf
		Else
			; Normal key
			If IsMap($vCurrent) Then
				If MapExists($vCurrent, $sPart) Then
					$vCurrent = $vCurrent[$sPart]
				Else
					Return $vDefault
				EndIf
			ElseIf IsArray($vCurrent) Then
				Local $iIdx = Int($sPart)
				If $iIdx >= 0 And $iIdx < UBound($vCurrent) Then
					$vCurrent = $vCurrent[$iIdx]
				Else
					Return $vDefault
				EndIf
			Else
				Return $vDefault
			EndIf
		EndIf
	Next

	Return $vCurrent
EndFunc   ;==>_JsonC_Get

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Has
; Description ...: Check if path exists in JSON data
; Syntax ........: _JsonC_Has($vData, $sPath)
; Parameters ....: $vData - JSON data (Map/Array)
;                  $sPath - Path to check
; Return values .: True if path exists, False otherwise
; =================================================================================================
Func _JsonC_Has($vData, $sPath)
	Local $vResult = _JsonC_Get($vData, $sPath, Default)
	Return ($vResult <> Default)
EndFunc   ;==>_JsonC_Has

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_GetTypeStr
; Description ...: Get the type of value at path as string
; Syntax ........: _JsonC_GetTypeStr($vData, $sPath = "")
; Parameters ....: $vData - JSON data
;                  $sPath - [Optional] Path to value (Default: "" = root)
; Return values .: "Map", "Array", "String", "Int64", "Double", "Bool", "Keyword", or "Unknown"
; =================================================================================================
Func _JsonC_GetTypeStr($vData, $sPath = "")
	Local $vValue = ($sPath = "") ? $vData : _JsonC_Get($vData, $sPath, Default)
	If $vValue = Default Then Return "Unknown"
	Return VarGetType($vValue)
EndFunc   ;==>_JsonC_GetTypeStr

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Set
; Description ...: Set value in JSON data by path (creates intermediate objects if needed)
; Syntax ........: _JsonC_Set(ByRef $vData, $sPath, $vValue)
; Parameters ....: $vData  - [ByRef] JSON data to modify
;                  $sPath  - Path where to set value (e.g., "user.name" or "items[0].name")
;                  $vValue - Value to set
; Return values .: True on success, False on failure
; Example .......: _JsonC_Set($mData, "user.name", "John")
;                  _JsonC_Set($mData, "items[0].price", 19.99)
; Author ........: Dao Van Trong - TRONG.PRO
; =================================================================================================
Func _JsonC_Set(ByRef $vData, $sPath, $vValue)
	If $sPath = "" Then
		$vData = $vValue
		Return True
	EndIf

	Local $aParts = StringSplit($sPath, ".", 2)
	Local $iLast = UBound($aParts) - 1

	; Ensure root is Map
	If Not IsMap($vData) Then
		Local $mNew[]
		$vData = $mNew
	EndIf

	; For simple paths (no dots), set directly
	If $iLast = 0 Then
		Local $sFinalPart = $aParts[0]
		Local $aMatch = StringRegExp($sFinalPart, "^([^\[]+)\[(\d+)\]$", 1)
		If IsArray($aMatch) Then
			; Array notation: key[index]
			Local $sKey = $aMatch[0]
			Local $iIndex = Int($aMatch[1])
			If Not MapExists($vData, $sKey) Then
				Local $aNew[$iIndex + 1]
				$vData[$sKey] = $aNew
			EndIf
			Local $aArr = $vData[$sKey]
			If $iIndex >= UBound($aArr) Then
				ReDim $aArr[$iIndex + 1]
			EndIf
			$aArr[$iIndex] = $vValue
			$vData[$sKey] = $aArr
		Else
			; Simple key
			$vData[$sFinalPart] = $vValue
		EndIf
		Return True
	EndIf

	; For nested paths, recursively build structure
	Local $sFirstPart = $aParts[0]
	Local $sRestPath = ""
	For $i = 1 To $iLast
		$sRestPath &= ($i > 1 ? "." : "") & $aParts[$i]
	Next

	; Handle array notation in first part
	Local $aMatch = StringRegExp($sFirstPart, "^([^\[]+)\[(\d+)\]$", 1)
	If IsArray($aMatch) Then
		Local $sKey = $aMatch[0]
		Local $iIndex = Int($aMatch[1])

		; Ensure key exists as array
		If Not MapExists($vData, $sKey) Then
			Local $aNew[$iIndex + 1]
			$vData[$sKey] = $aNew
		EndIf

		; Get array and ensure index exists
		Local $aArr = $vData[$sKey]
		If $iIndex >= UBound($aArr) Then
			ReDim $aArr[$iIndex + 1]
			$vData[$sKey] = $aArr
		EndIf

		; Ensure element is a Map
		If Not IsMap($aArr[$iIndex]) Then
			Local $mNew[]
			$aArr[$iIndex] = $mNew
			$vData[$sKey] = $aArr
		EndIf

		; Recursively set in nested map
		Local $mNested = $vData[$sKey][$iIndex]
		_JsonC_Set($mNested, $sRestPath, $vValue)
		$aArr[$iIndex] = $mNested
		$vData[$sKey] = $aArr
	Else
		; Normal key - ensure it's a Map
		If Not MapExists($vData, $sFirstPart) Then
			Local $mNew[]
			$vData[$sFirstPart] = $mNew
		ElseIf Not IsMap($vData[$sFirstPart]) Then
			Local $mNew[]
			$vData[$sFirstPart] = $mNew
		EndIf

		; Recursively set in nested map
		Local $mNested = $vData[$sFirstPart]
		_JsonC_Set($mNested, $sRestPath, $vValue)
		$vData[$sFirstPart] = $mNested
	EndIf

	Return True
EndFunc   ;==>_JsonC_Set

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Delete
; Description ...: Delete value from JSON data by path
; Syntax ........: _JsonC_Delete(ByRef $vData, $sPath)
; Parameters ....: $vData - [ByRef] JSON data
;                  $sPath - Path to delete
; Return values .: True if deleted, False if path not found
; Author ........: Dao Van Trong - TRONG.PRO
; =================================================================================================
Func _JsonC_Delete(ByRef $vData, $sPath)
	If $sPath = "" Then Return False

	Local $aParts = StringSplit($sPath, ".", 2)
	Local $iLast = UBound($aParts) - 1

	If $iLast = 0 Then
		; Direct key - delete from root
		Local $aMatch = StringRegExp($aParts[0], "^([^\[]+)\[(\d+)\]$", 1)
		If IsArray($aMatch) Then
			; Array notation: key[index]
			Return False ; Cannot delete array elements (would need to shift indices)
		Else
			; Simple key
			If IsMap($vData) And MapExists($vData, $aParts[0]) Then
				MapRemove($vData, $aParts[0])
				Return True
			EndIf
		EndIf
		Return False
	EndIf

	; For nested paths, navigate to parent and delete from there
	Local $sParentPath = ""
	For $i = 0 To $iLast - 1
		$sParentPath &= ($i > 1 ? "." : "") & $aParts[$i]
	Next

	; Use recursive approach to maintain proper references
	Return __JsonC_DeleteNested($vData, $aParts, 0, $iLast)
EndFunc   ;==>_JsonC_Delete

; Internal recursive helper for delete
Func __JsonC_DeleteNested(ByRef $vCurrent, ByRef $aParts, $iIndex, $iLast)
	If $iIndex = $iLast Then
		; We're at the parent, delete the key
		Local $sKey = $aParts[$iIndex]
		If IsMap($vCurrent) And MapExists($vCurrent, $sKey) Then
			MapRemove($vCurrent, $sKey)
			Return True
		EndIf
		Return False
	EndIf

	; Navigate deeper
	Local $sPart = $aParts[$iIndex]
	Local $aMatch = StringRegExp($sPart, "^([^\[]+)\[(\d+)\]$", 1)

	If IsArray($aMatch) Then
		; Array notation
		Local $sKey = $aMatch[0]
		Local $iIdx = Int($aMatch[1])
		If MapExists($vCurrent, $sKey) Then
			Local $aArr = $vCurrent[$sKey]
			If $iIdx >= 0 And $iIdx < UBound($aArr) Then
				Local $bResult = __JsonC_DeleteNested($aArr[$iIdx], $aParts, $iIndex + 1, $iLast)
				$vCurrent[$sKey] = $aArr
				Return $bResult
			EndIf
		EndIf
	Else
		; Normal key
		If MapExists($vCurrent, $sPart) Then
			Local $mNested = $vCurrent[$sPart]
			Local $bResult = __JsonC_DeleteNested($mNested, $aParts, $iIndex + 1, $iLast)
			$vCurrent[$sPart] = $mNested
			Return $bResult
		EndIf
	EndIf

	Return False
EndFunc   ;==>__JsonC_DeleteNested

; ===============================================================================================================================
; #UTILITY FUNCTIONS# ===========================================================================================================
; ===============================================================================================================================

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Keys
; Description ...: Get all keys from a Map/Object
; Syntax ........: _JsonC_Keys($vData, $sPath = "")
; Parameters ....: $vData - JSON data
;                  $sPath - [Optional] Path to object (Default: "" = root)
; Return values .: Array of keys, or empty array if not a Map
; Example .......: $aKeys = _JsonC_Keys($mData)
;                  For $sKey In $aKeys
;                      ConsoleWrite($sKey & @CRLF)
;                  Next
; Author ........: Dao Van Trong - TRONG.PRO
; =================================================================================================
Func _JsonC_Keys($vData, $sPath = "")
	Local $vObj = ($sPath = "") ? $vData : _JsonC_Get($vData, $sPath, Default)

	Local $aKeys[0]
	If Not IsMap($vObj) Then Return $aKeys

	For $sKey In MapKeys($vObj)
		ReDim $aKeys[UBound($aKeys) + 1]
		$aKeys[UBound($aKeys) - 1] = $sKey
	Next

	Return $aKeys
EndFunc   ;==>_JsonC_Keys

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Values
; Description ...: Get all values from a Map/Object or Array
; Syntax ........: _JsonC_Values($vData, $sPath = "")
; Parameters ....: $vData - JSON data
;                  $sPath - [Optional] Path to object/array (Default: "" = root)
; Return values .: Array of values, or empty array if not a Map/Array
; Author ........: Dao Van Trong - TRONG.PRO
; =================================================================================================
Func _JsonC_Values($vData, $sPath = "")
	Local $vObj = ($sPath = "") ? $vData : _JsonC_Get($vData, $sPath, Default)

	Local $aValues[0]

	If IsMap($vObj) Then
		For $sKey In MapKeys($vObj)
			ReDim $aValues[UBound($aValues) + 1]
			$aValues[UBound($aValues) - 1] = $vObj[$sKey]
		Next
	ElseIf IsArray($vObj) Then
		Return $vObj ; Already an array
	EndIf

	Return $aValues
EndFunc   ;==>_JsonC_Values

; ===============================================================================================================================
; #LOW-LEVEL DLL FUNCTIONS# =====================================================================================================
; ===============================================================================================================================

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Tokener_Parse
; Description ...: Parse a string and return a json object
; Syntax ........: _JsonC_Tokener_Parse($sString)
; Parameters ....: $sString     - a string formatted as JSON
; Return values .: Success 		- return a non-NULL json_object if a valid JSON value is found
;                  Failure 		- Return a "" (NULL) and set @error to:
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_TokenerParse($sString)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_tokener_parse", "str", $sString)
	If @error Then Return SetError(1, @error, 0) ; DllCall error
	If $avRval[0] = "" Then
		Return SetError(-1, $avRval[0], 0)
	EndIf
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, 0) ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_TokenerParse

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectToJsonString
; Description ...: Stringify object to json format.
; Syntax ........: _JsonC_ObjectToJsonString($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- return the string equivalent of the json object
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectToJsonString($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "str:cdecl", "json_object_to_json_string", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectToJsonString

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectGetType
; Description ...: Get the type of the json object
; Syntax ........: _JsonC_ObjectGetType($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- return type being one of:
;									$JSONC_TYPE_NULL
;									$JSONC_TYPE_BOOLEAN
;									$JSONC_TYPE_DOUBLE
;									$JSONC_TYPE_INT
;									$JSONC_TYPE_OBJECT
;									$JSONC_TYPE_ARRAY
;									$JSONC_TYPE_STRING
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Remarks .......: See also _JsonC_TypeToName to turn this into a string suitable, for instance, for logging.
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectGetType($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_get_type", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectGetType

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectIsType
; Description ...: Checks if the json object is of a given type
; Syntax ........: _JsonC_ObjectIsType($pObject, $iType)
; Parameters ....: $pObject     - the json object instance
;				   $iType		- one of:
;									$JSONC_TYPE_NULL
;									$JSONC_TYPE_BOOLEAN
;									$JSONC_TYPE_DOUBLE
;									$JSONC_TYPE_INT
;									$JSONC_TYPE_OBJECT
;									$JSONC_TYPE_ARRAY
;									$JSONC_TYPE_STRING
; Return values .: Success 		- return a True or False depending on whether object is the type
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectIsType($pObject, $iType)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_is_type", "ptr", $pObject, "int", $iType)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectIsType

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_TypeToName
; Description ...: Return a string describing the type of the object. e.g. "int", or "object", etc...
; Syntax ........: _JsonC_TypeToName($iType)
; Parameters ....: $iType      - one of:
;									$JSONC_TYPE_NULL
;									$JSONC_TYPE_BOOLEAN
;									$JSONC_TYPE_DOUBLE
;									$JSONC_TYPE_INT
;									$JSONC_TYPE_OBJECT
;									$JSONC_TYPE_ARRAY
;									$JSONC_TYPE_STRING
; Return values .: Success 		- return the string name of the type
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_TypeToName($iType)
	Local $avRval = DllCall($__g_hDll_JsonC, "str:cdecl", "json_type_to_name", "int", $iType)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_TypeToName

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectObjectAdd
; Description ...: Add an object field to a json object of type $JSONC_TYPE_OBJECT
; Syntax ........: _JsonC_ObjectObjectAdd($pObject, $sKey, $pObjectToAdd)
; Parameters ....: $pObject     - the json object instance
;				   $sKey		- the object field name
;				   $pObjectToAdd- the json object to add
; Return values .: Success 		- the given json object field is added
;                  Failure 		- set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectObjectAdd($pObject, $sKey, $pObjectToAdd)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	If IsPtr($pObjectToAdd) = False Then $pObjectToAdd = DllStructGetPtr($pObjectToAdd)
	DllCall($__g_hDll_JsonC, "none:cdecl", "json_object_object_add", "ptr", $pObject, "str", $sKey, "ptr", $pObjectToAdd)
	If @error Then Return SetError(1, @error, 0) ; DllCall error
EndFunc   ;==>_JsonC_ObjectObjectAdd

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectObjectDel
; Description ...: Delete the given json object field
; Syntax ........: _JsonC_ObjectObjectDel($pObject, $sKey)
; Parameters ....: $pObject     - the json object instance
;				   $sKey		- the object field name
; Return values .: Success 		- the given json object field is deleted
;                  Failure 		- set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectObjectDel($pObject, $sKey)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	DllCall($__g_hDll_JsonC, "none:cdecl", "json_object_object_del", "ptr", $pObject, "str", $sKey)
	If @error Then Return SetError(1, @error, 0) ; DllCall error
EndFunc   ;==>_JsonC_ObjectObjectDel

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectObjectGet
; Description ...: Get the json object associated with a given object field
; Syntax ........: _JsonC_ObjectObjectGet($pObject, $sKey)
; Parameters ....: $pObject     - the json object instance
;				   $sKey		- the object field name
; Return values .: Success 		- return the json object associated with the given field name
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectObjectGet($pObject, $sKey)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_object_get", "ptr", $pObject, "str", $sKey)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_ObjectObjectGet

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectGetBoolean
; Description ...: Get the $JSONC_TYPE_BOOLEAN value of a json object
; Syntax ........: _JsonC_ObjectGetBoolean($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- a boolean
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectGetBoolean($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "boolean:cdecl", "json_object_get_boolean", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectGetBoolean

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectGetDouble
; Description ...: Get the $JSONC_TYPE_DOUBLE value of a json object
; Syntax ........: _JsonC_ObjectGetDouble($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- return a double floating point number
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectGetDouble($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "double:cdecl", "json_object_get_double", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectGetDouble

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectGetInt
; Description ...: Get the $JSONC_TYPE_INT value of a json object
; Syntax ........: _JsonC_ObjectGetInt($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- return an int
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectGetInt($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_get_int", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectGetInt

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectGetObject
; Description ...: Get the hashtable of a json object of type json type object
; Syntax ........: _JsonC_ObjectGetObject($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- return a linkhash
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectGetObject($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_get_object", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectGetObject

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectGetArray
; Description ...: Get the arraylist of a json object of type json type array
; Syntax ........: _JsonC_ObjectGetArray($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- return an arraylist
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectGetArray($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_get_array", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectGetArray

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ArrayListLength
; Description ...: Get the length of an arraylist
; Syntax ........: _JsonC_ArrayListLength($tArrayList)
; Parameters ....: $tArrayList  - the arraylist (from _JsonC_ObjectGetArray)
; Return values .: Success 		- return the length of the arraylist
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ArrayListLength($tArrayList)
	; size_t return type
	Local $sRetType = @AutoItX64 ? "uint64:cdecl" : "uint:cdecl"
	Local $avRval = DllCall($__g_hDll_JsonC, $sRetType, "array_list_length", "ptr", $tArrayList)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ArrayListLength

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ArrayListGetIndex
; Description ...: Get the json object for an item in the arraylist
; Syntax ........: _JsonC_ArrayListGetIndex($tArrayList, $iIndex)
; Parameters ....: $tArrayList  - the arraylist (from _JsonC_ObjectGetArray)
; Return values .: Success 		- return the json object
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ArrayListGetIndex($tArrayList, $iIndex)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "array_list_get_idx", "ptr", $tArrayList, "int", $iIndex)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ArrayListGetIndex

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectGetString
; Description ...: Get the string value of a json object
; Syntax ........: _JsonC_ObjectGetString($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- return a string
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectGetString($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "str:cdecl", "json_object_get_string", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectGetString

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectGetValue
; Description ...: Get the value of a json object of any type
; Syntax ........: _JsonC_ObjectGetValue($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- return the value of the object
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectGetValue($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $iType, $iLength
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_get_value", "ptr", $pObject, "int*", $iType, "int*", $iLength)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	$iType = $avRval[2]
	$iLength = $avRval[3]
	Switch $iType
		Case $JSONC_TYPE_NULL
			Return ""
		Case $JSONC_TYPE_BOOLEAN
			Return DllStructGetData(DllStructCreate("BOOLEAN", $avRval[0]), 1)
		Case $JSONC_TYPE_DOUBLE
			Return DllStructGetData(DllStructCreate("DOUBLE", $avRval[0]), 1)
		Case $JSONC_TYPE_INT
			Return DllStructGetData(DllStructCreate("INT", $avRval[0]), 1)
		Case $JSONC_TYPE_OBJECT
			Return DllStructGetData(DllStructCreate("PTR", $avRval[0]), 1)
		Case $JSONC_TYPE_ARRAY
			Return DllStructGetData(DllStructCreate("PTR", $avRval[0]), 1)
		Case $JSONC_TYPE_STRING
			Return DllStructGetData(DllStructCreate("CHAR[" & $iLength & "]", $avRval[0]), 1)
	EndSwitch
	Return SetError(1, @error, "")
EndFunc   ;==>_JsonC_ObjectGetValue

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectGetFieldValue
; Description ...: Get the value of a field in a json object
; Syntax ........: _JsonC_ObjectGetFieldValue($pObject, $sKey)
; Parameters ....: $pObject     - the json object instance
;				   $sKey		- the object field name
; Return values .: Success 		- return the value of the field
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectGetFieldValue($pObject, $sKey)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $iType, $iLength
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_get_field_value", "ptr", $pObject, "str", $sKey, "int*", $iType, "int*", $iLength)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	$iType = $avRval[3]
	$iLength = $avRval[4]
	Switch $iType
		Case $JSONC_TYPE_NULL
			Return ""
		Case $JSONC_TYPE_BOOLEAN
			Return DllStructGetData(DllStructCreate("BOOLEAN", $avRval[0]), 1)
		Case $JSONC_TYPE_DOUBLE
			Return DllStructGetData(DllStructCreate("DOUBLE", $avRval[0]), 1)
		Case $JSONC_TYPE_INT
			Return DllStructGetData(DllStructCreate("INT", $avRval[0]), 1)
		Case $JSONC_TYPE_OBJECT
			Return DllStructGetData(DllStructCreate("PTR", $avRval[0]), 1)
		Case $JSONC_TYPE_ARRAY
			Return DllStructGetData(DllStructCreate("PTR", $avRval[0]), 1)
		Case $JSONC_TYPE_STRING
			Return DllStructGetData(DllStructCreate("CHAR[" & $iLength & "]", $avRval[0]), 1)
	EndSwitch
	Return SetError(1, @error, "")
EndFunc   ;==>_JsonC_ObjectGetFieldValue

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectNewObject
; Description ...: Create a new empty json object of type $JSONC_TYPE_OBJECT
; Syntax ........: _JsonC_ObjectNewObject()
; Parameters ....:
; Return values .: Success 		- return a json object of type $JSONC_TYPE_OBJECT
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectNewObject()
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_object")
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_ObjectNewObject

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectNewArray
; Description ...: Create a new empty json object of type $JSONC_TYPE_ARRAY
; Syntax ........: _JsonC_ObjectNewArray()
; Parameters ....:
; Return values .: Success 		- return a json object of type $JSONC_TYPE_ARRAY
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectNewArray()
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_array")
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_ObjectNewArray

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectArrayAdd
; Description ...: Add an element to the end of a json object of type $JSONC_TYPE_ARRAY
; Syntax ........: _JsonC_ObjectArrayAdd($pObject, $pObjectToAdd)
; Parameters ....: $pObject     - the json object instance of type $JSONC_TYPE_ARRAY
;				   $pObjectToAdd- the json object to add
; Return values .: Success 		- return an int
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectArrayAdd($pObject, $pObjectToAdd)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	If IsPtr($pObjectToAdd) = False Then $pObjectToAdd = DllStructGetPtr($pObjectToAdd)
	Local $avRval = DllCall($__g_hDll_JsonC, "int:cdecl", "json_object_array_add", "ptr", $pObject, "ptr", $pObjectToAdd)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectArrayAdd

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectNewBoolean
; Description ...: Create a new empty json object of type $JSONC_TYPE_BOOLEAN
; Syntax ........: _JsonC_ObjectNewBoolean($bBoolean)
; Parameters ....: $bBoolean   	- a Boolean TRUE or FALSE (0 or 1)
; Return values .: Success 		- return a json_object of type $JSONC_TYPE_BOOLEAN
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectNewBoolean($bBoolean)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_boolean", "boolean", $bBoolean)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_ObjectNewBoolean

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectNewInt
; Description ...: Create a new empty json object of type $JSONC_TYPE_INT
; Syntax ........: _JsonC_ObjectNewInt($iInteger)
; Parameters ....: $iInteger    - the integer
; Return values .: Success 		- return a json_object of type $JSONC_TYPE_INT
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectNewInt($iInteger)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_int", "int", $iInteger)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_ObjectNewInt

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectNewInt64
; Description ...: Create a new empty json object of type $JSONC_TYPE_INT
; Syntax ........: _JsonC_ObjectNewInt64($iInteger)
; Parameters ....: $iInteger    - the integer
; Return values .: Success 		- return a json_object of type $JSONC_TYPE_INT
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectNewInt64($iInteger)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_int64", "int64", $iInteger)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_ObjectNewInt64

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectNewDouble
; Description ...: Create a new empty json object of type $JSONC_TYPE_DOUBLE
; Syntax ........: _JsonC_ObjectNewDouble($fDouble)
; Parameters ....: $fDouble     - the double
; Return values .: Success 		- return a json_object of type $JSONC_TYPE_DOUBLE
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectNewDouble($fDouble)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_double", "double", $fDouble)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_ObjectNewDouble

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectNewString
; Description ...: Create a new empty json object of type $JSONC_TYPE_STRING
; Syntax ........: _JsonC_ObjectNewString($sString)
; Parameters ....: $sString     - the string
; Return values .: Success 		- return a json_object of type $JSONC_TYPE_STRING
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectNewString($sString)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_new_string", "str", $sString)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_ObjectNewString

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectArrayLength
; Description ...: Get the length of a json object of type $JSONC_TYPE_ARRAY
; Syntax ........: _JsonC_ObjectArrayLength($pObject)
; Parameters ....: $pObject     - the json object instance
; Return values .: Success 		- return an int
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectArrayLength($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	; size_t return type
	Local $sRetType = @AutoItX64 ? "uint64:cdecl" : "uint:cdecl"
	Local $avRval = DllCall($__g_hDll_JsonC, $sRetType, "json_object_array_length", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $avRval[0]
EndFunc   ;==>_JsonC_ObjectArrayLength

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectArrayGetIndex
; Description ...: Get the element at specificed index of the array
; Syntax ........: _JsonC_ObjectArrayGetIndex($pObject, $iIndex)
; Parameters ....: $pObject     - a json object of type json_type_array
;				   $iIndex		- the index to get the element at
; Return values .: Success 		- return the json object at the specified index
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectArrayGetIndex($pObject, $iIndex)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_array_get_idx", "ptr", $pObject, "int", $iIndex)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Local $tJsonObject = DllStructCreate($tagJSONC_OBJECT, $avRval[0])
	If @error Then Return SetError(1, @error, "") ; DllCall error
	Return $tJsonObject
EndFunc   ;==>_JsonC_ObjectArrayGetIndex

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_ObjectArrayGetObjects
; Description ...: Get all the json objects from an array
; Syntax ........: _JsonC_ObjectArrayGetObjects($pObject)
; Parameters ....: $pObject     - a json object of type json_type_array
; Return values .: Success 		- return an AutoIt array of json objects
;                  Failure 		- Return a "" (NULL) and set @error to 1
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_ObjectArrayGetObjects($pObject)
	If IsPtr($pObject) = False Then $pObject = DllStructGetPtr($pObject)
	Local $avRval = DllCall($__g_hDll_JsonC, "ptr:cdecl", "json_object_array_get_objects", "ptr", $pObject)
	If @error Then Return SetError(1, @error, "") ; DllCall error
	$ArrayLength = _JsonC_ObjectArrayLength($pObject)
	Local $pObjects[$ArrayLength]
	For $i = 0 To $ArrayLength - 1
		Local $ptrObject = DllStructGetData(DllStructCreate("ptr", $avRval[0] + ($i * $PtrSize)), 1)
		$pObjects[$i] = $ptrObject
	Next
	Return $pObjects
EndFunc   ;==>_JsonC_ObjectArrayGetObjects

; #FUNCTION# ======================================================================================
; Name ..........: _JsonC_Shutdown
; Description ...: Unloads json-c.dll and cleanup
; Syntax ........: _JsonC_Shutdown()
; Return values .: None
; Author ........: SeanGriffin
; =================================================================================================
Func _JsonC_Shutdown()
	If $__g_hDll_JsonC <> 0 Then
		DllClose($__g_hDll_JsonC)
		$__g_hDll_JsonC = 0
		$__g_bDll_AutoLoaded = False
	EndIf
EndFunc   ;==>_JsonC_Shutdown
