#AutoIt3Wrapper_UseX64=y
#include "JSONc_UDF.au3"
#include <MsgBoxConstants.au3>
#include <String.au3>
; ===============================================================================================================================
; COMPREHENSIVE TEST SUITE FOR JSONc_UDF.au3
; Dao Van Trong - TRONG.PRO
; ===============================================================================================================================

Global $iTestsPassed = 0
Global $iTestsFailed = 0
Global $sOutput = ""

ConsoleWrite(@CRLF & "=== JSONc_UDF.au3 COMPREHENSIVE TEST SUITE ===" & @CRLF & @CRLF)

; ===============================================================================================================================
; INITIALIZE DLL
; ===============================================================================================================================
TestSection("1. DLL INITIALIZATION")

If Not _JsonC_Startup() Then
	TestFail("Failed to load JSON-C DLL")
	MsgBox($MB_ICONERROR, "Error", "Failed to load JSON-C DLL!" & @CRLF & "Stopping tests.")
	Exit
Else
	TestPass("JSON-C DLL loaded successfully")
EndIf

; ===============================================================================================================================
; TEST: PARSE SIMPLE JSON
; ===============================================================================================================================
TestSection("2. PARSE SIMPLE JSON")

Local $sJson = '{"name":"John","age":30,"active":true}'
Local $mData = _JsonC_Parse($sJson)
If @error Then
	TestFail("Failed to parse simple JSON")
Else
	TestPass("Parsed simple JSON")
	If $mData["name"] = "John" Then
		TestPass("Retrieved string value correctly: " & $mData["name"])
	Else
		TestFail("String value mismatch")
	EndIf

	If $mData["age"] = 30 Then
		TestPass("Retrieved integer value correctly: " & $mData["age"])
	Else
		TestFail("Integer value mismatch")
	EndIf

	If $mData["active"] = True Then
		TestPass("Retrieved boolean value correctly: " & $mData["active"])
	Else
		TestFail("Boolean value mismatch")
	EndIf
EndIf

; ===============================================================================================================================
; TEST: PARSE NESTED JSON
; ===============================================================================================================================
TestSection("3. PARSE NESTED JSON")

$sJson = '{"user":{"name":"Alice","address":{"city":"New York","zip":"10001"}}}'
$mData = _JsonC_Parse($sJson)
If @error Then
	TestFail("Failed to parse nested JSON")
Else
	TestPass("Parsed nested JSON")
	Local $mUser = $mData["user"]
	Local $mAddress = $mUser["address"]

	If $mAddress["city"] = "New York" Then
		TestPass("Retrieved nested value: " & $mAddress["city"])
	Else
		TestFail("Nested value mismatch")
	EndIf
EndIf

; ===============================================================================================================================
; TEST: PARSE ARRAY
; ===============================================================================================================================
TestSection("4. PARSE JSON ARRAY")

$sJson = '[1,2,3,4,5]'
Local $aData = _JsonC_Parse($sJson)
If @error Then
	TestFail("Failed to parse array")
Else
	TestPass("Parsed array")
	If IsArray($aData) And UBound($aData) = 5 Then
		TestPass("Array has correct length: " & UBound($aData))
	Else
		TestFail("Array length mismatch")
	EndIf

	If $aData[0] = 1 And $aData[4] = 5 Then
		TestPass("Array values correct")
	Else
		TestFail("Array values mismatch")
	EndIf
EndIf

; ===============================================================================================================================
; TEST: PARSE COMPLEX ARRAY
; ===============================================================================================================================
TestSection("5. PARSE COMPLEX ARRAY")

$sJson = '[{"name":"Alice","age":25},{"name":"Bob","age":30}]'
$aData = _JsonC_Parse($sJson)
If @error Then
	TestFail("Failed to parse complex array")
Else
	TestPass("Parsed complex array")
	If IsArray($aData) And UBound($aData) = 2 Then
		TestPass("Array has correct length: " & UBound($aData))

		Local $mFirst = $aData[0]
		If $mFirst["name"] = "Alice" And $mFirst["age"] = 25 Then
			TestPass("First array element correct")
		Else
			TestFail("First array element mismatch")
		EndIf
	Else
		TestFail("Array structure mismatch")
	EndIf
EndIf

; ===============================================================================================================================
; TEST: GENERATE JSON
; ===============================================================================================================================
TestSection("6. GENERATE JSON")

Local $mTest[]
$mTest["name"] = "Bob"
$mTest["age"] = 35
$mTest["verified"] = False

Local $sGenerated = _JsonC_Generate($mTest)
If @error Then
	TestFail("Failed to generate JSON")
Else
	TestPass("Generated JSON: " & $sGenerated)

	; Verify it can be parsed back
	Local $mParsedBack = _JsonC_Parse($sGenerated)
	If Not @error And $mParsedBack["name"] = "Bob" Then
		TestPass("Round-trip conversion successful")
	Else
		TestFail("Round-trip conversion failed")
	EndIf
EndIf

; ===============================================================================================================================
; TEST: GENERATE PRETTY JSON
; ===============================================================================================================================
TestSection("7. GENERATE PRETTY JSON")

; Create a proper nested structure (not circular reference)
Local $mPrettyTest[]
$mPrettyTest["name"] = "Test User"
$mPrettyTest["age"] = 25

Local $mAddress[]
$mAddress["city"] = "Hanoi"
$mAddress["country"] = "Vietnam"
$mPrettyTest["address"] = $mAddress

Local $sPretty = _JsonC_GeneratePretty($mPrettyTest)
If @error Then
	TestFail("Failed to generate pretty JSON")
Else
	TestPass("Generated pretty JSON")
	; Check for line breaks (JSON-C uses \n, not \r\n on Windows)
	If StringInStr($sPretty, @CRLF) Or StringInStr($sPretty, @LF) Then
		TestPass("Pretty JSON contains line breaks")
	Else
		TestFail("Pretty JSON format issue")
	EndIf
EndIf

; ===============================================================================================================================
; TEST: PATH-BASED GET
; ===============================================================================================================================
TestSection("8. PATH-BASED GET QUERIES")

$sJson = '{"user":{"name":"Charlie","address":{"city":"Boston","zip":"02101"}},"items":[{"id":1,"name":"Item1"},{"id":2,"name":"Item2"}]}'
$mData = _JsonC_Parse($sJson)

; Simple path
Local $sName = _JsonC_Get($mData, "user.name")
If $sName = "Charlie" Then
	TestPass("Simple path get: user.name = " & $sName)
Else
	TestFail("Simple path get failed")
EndIf

; Nested path
Local $sCity = _JsonC_Get($mData, "user.address.city")
If $sCity = "Boston" Then
	TestPass("Nested path get: user.address.city = " & $sCity)
Else
	TestFail("Nested path get failed")
EndIf

; Array index path
Local $mItem = _JsonC_Get($mData, "items[0]")
If IsMap($mItem) And $mItem["name"] = "Item1" Then
	TestPass("Array index get: items[0].name = " & $mItem["name"])
Else
	TestFail("Array index get failed")
EndIf

; Default value
Local $vDefault = _JsonC_Get($mData, "nonexistent.path", "DEFAULT")
If $vDefault = "DEFAULT" Then
	TestPass("Default value returned for missing path")
Else
	TestFail("Default value not returned")
EndIf

; ===============================================================================================================================
; TEST: PATH-BASED SET
; ===============================================================================================================================
TestSection("9. PATH-BASED SET")

Local $mNewData[]

; Set simple value
_JsonC_Set($mNewData, "name", "David")
If _JsonC_Get($mNewData, "name") = "David" Then
	TestPass("Set simple value: name = David")
Else
	TestFail("Set simple value failed")
EndIf

; Set nested value (auto-create structure)
_JsonC_Set($mNewData, "user.email", "david@example.com")
If _JsonC_Get($mNewData, "user.email") = "david@example.com" Then
	TestPass("Set nested value (auto-created): user.email = david@example.com")
Else
	TestFail("Set nested value failed")
EndIf

; Set deep nested value
_JsonC_Set($mNewData, "settings.display.theme", "dark")
If _JsonC_Get($mNewData, "settings.display.theme") = "dark" Then
	TestPass("Set deep nested value: settings.display.theme = dark")
Else
	TestFail("Set deep nested value failed")
EndIf

; ===============================================================================================================================
; TEST: PATH-BASED HAS
; ===============================================================================================================================
TestSection("10. PATH-BASED HAS")

If _JsonC_Has($mNewData, "name") Then
	TestPass("Has() correctly found existing path: name")
Else
	TestFail("Has() failed to find existing path")
EndIf

If Not _JsonC_Has($mNewData, "nonexistent") Then
	TestPass("Has() correctly reported missing path: nonexistent")
Else
	TestFail("Has() incorrectly found nonexistent path")
EndIf

If _JsonC_Has($mNewData, "settings.display.theme") Then
	TestPass("Has() found nested path: settings.display.theme")
Else
	TestFail("Has() failed to find nested path")
EndIf

; ===============================================================================================================================
; TEST: PATH-BASED DELETE
; ===============================================================================================================================
TestSection("11. PATH-BASED DELETE")

_JsonC_Set($mNewData, "toDelete", "value")
If _JsonC_Has($mNewData, "toDelete") Then
	_JsonC_Delete($mNewData, "toDelete")
	If Not _JsonC_Has($mNewData, "toDelete") Then
		TestPass("Delete() successfully removed path: toDelete")
	Else
		TestFail("Delete() failed to remove path")
	EndIf
Else
	TestFail("Failed to create path for deletion test")
EndIf

; ===============================================================================================================================
; TEST: GET TYPE
; ===============================================================================================================================
TestSection("12. GET TYPE")

Local $mTypeTest[]
$mTypeTest["string"] = "text"
$mTypeTest["number"] = 42
$mTypeTest["boolean"] = True
$mTypeTest["array"] = StringSplit('1|2|3', '|')

If _JsonC_GetTypeStr($mTypeTest, "string") = "String" Then
	TestPass("GetTypeStr() correctly identified String type")
Else
	TestFail("GetTypeStr() failed for String")
EndIf

Local $sNumType = _JsonC_GetTypeStr($mTypeTest, "number")
If $sNumType = "Int32" Or $sNumType = "Int64" Then
	TestPass("GetTypeStr() correctly identified Integer type")
Else
	TestFail("GetTypeStr() failed for Integer")
EndIf

If _JsonC_GetTypeStr($mTypeTest, "boolean") = "Bool" Then
	TestPass("GetTypeStr() correctly identified Bool type")
Else
	TestFail("GetTypeStr() failed for Bool")
EndIf

If _JsonC_GetTypeStr($mTypeTest, "array") = "Array" Then
	TestPass("GetTypeStr() correctly identified Array type")
Else
	TestFail("GetTypeStr() failed for Array")
EndIf

; ===============================================================================================================================
; TEST: UTILITY - KEYS
; ===============================================================================================================================
TestSection("13. UTILITY - KEYS")

Local $mKeysTest[]
$mKeysTest["key1"] = "value1"
$mKeysTest["key2"] = "value2"
$mKeysTest["key3"] = "value3"

Local $aKeys = _JsonC_Keys($mKeysTest)
If IsArray($aKeys) And UBound($aKeys) = 3 Then
	TestPass("Keys() returned correct number of keys: " & UBound($aKeys))
Else
	TestFail("Keys() returned incorrect number of keys")
EndIf

; ===============================================================================================================================
; TEST: UTILITY - VALUES
; ===============================================================================================================================
TestSection("14. UTILITY - VALUES")

Local $aValues = _JsonC_Values($mKeysTest)
If IsArray($aValues) And UBound($aValues) = 3 Then
	TestPass("Values() returned correct number of values: " & UBound($aValues))
Else
	TestFail("Values() returned incorrect number of values")
EndIf

; ===============================================================================================================================
; TEST: JSON VALIDATION
; ===============================================================================================================================
TestSection("15. JSON VALIDATION")

Local $sValidJson = '{"valid":true}'
If _JsonC_IsValid($sValidJson) Then
	TestPass("IsValid() correctly validated valid JSON")
Else
	TestFail("IsValid() failed on valid JSON")
EndIf

Local $sInvalidJson = '{"invalid": missing quotes}'
If Not _JsonC_IsValid($sInvalidJson) Then
	TestPass("IsValid() correctly rejected invalid JSON")
Else
	TestFail("IsValid() incorrectly accepted invalid JSON")
EndIf

Local $sEmptyJson = '{}'
If _JsonC_IsValid($sEmptyJson) Then
	TestPass("IsValid() correctly validated empty object")
Else
	TestFail("IsValid() failed on empty object")
EndIf

Local $sEmptyArray = '[]'
If _JsonC_IsValid($sEmptyArray) Then
	TestPass("IsValid() correctly validated empty array")
Else
	TestFail("IsValid() failed on empty array")
EndIf

; ===============================================================================================================================
; TEST: SPECIAL VALUES
; ===============================================================================================================================
TestSection("16. SPECIAL VALUES")

$sJson = '{"null_value":null,"empty_string":"","zero":0,"false_val":false}'
$mData = _JsonC_Parse($sJson)
If Not @error Then
	TestPass("Parsed JSON with special values")

	If $mData["null_value"] = Null Then
		TestPass("Null value handled correctly")
	Else
		TestFail("Null value not handled correctly")
	EndIf

	If $mData["empty_string"] = "" Then
		TestPass("Empty string handled correctly")
	Else
		TestFail("Empty string not handled correctly")
	EndIf

	If $mData["zero"] = 0 Then
		TestPass("Zero value handled correctly")
	Else
		TestFail("Zero value not handled correctly")
	EndIf

	If $mData["false_val"] = False Then
		TestPass("False value handled correctly")
	Else
		TestFail("False value not handled correctly")
	EndIf
Else
	TestFail("Failed to parse JSON with special values")
EndIf

; ===============================================================================================================================
; TEST: UNICODE STRINGS
; ===============================================================================================================================
TestSection("17. UNICODE STRINGS")

Local $mUnicode[]
$mUnicode["vietnamese"] = "Xin chào"
$mUnicode["chinese"] = "你好"
$mUnicode["emoji"] = "😀🎉"

$sJson = _JsonC_Generate($mUnicode)
If Not @error Then
	TestPass("Generated JSON with Unicode strings")

	$mData = _JsonC_Parse($sJson)
	If Not @error Then
		If $mData["vietnamese"] = "Xin chào" Then
			TestPass("Unicode Vietnamese text preserved")
		Else
			TestFail("Unicode Vietnamese text not preserved")
		EndIf
	Else
		TestFail("Failed to parse Unicode JSON")
	EndIf
Else
	TestFail("Failed to generate JSON with Unicode")
EndIf

; ===============================================================================================================================
; TEST: LARGE NUMBERS
; ===============================================================================================================================
TestSection("18. LARGE NUMBERS")

$sJson = '{"big_number":9223372036854775807}'
$mData = _JsonC_Parse($sJson)
If Not @error Then
	TestPass("Parsed JSON with large number")
	ConsoleWrite("    Big number value: " & $mData["big_number"] & @CRLF)
Else
	TestFail("Failed to parse large number")
EndIf

; ===============================================================================================================================
; TEST: FLOATING POINT NUMBERS
; ===============================================================================================================================
TestSection("19. FLOATING POINT NUMBERS")

Local $mFloat[]
$mFloat["pi"] = 3.14159265359
$mFloat["e"] = 2.71828182846

$sJson = _JsonC_Generate($mFloat)
$mData = _JsonC_Parse($sJson)
If Not @error Then
	TestPass("Round-trip conversion of floating point numbers")
	If Abs($mData["pi"] - 3.14159265359) < 0.0001 Then
		TestPass("Pi value preserved with acceptable precision")
	Else
		TestFail("Pi value precision issue")
	EndIf
Else
	TestFail("Failed to convert floating point numbers")
EndIf

; ===============================================================================================================================
; TEST: ESCAPE SEQUENCES
; ===============================================================================================================================
TestSection("20. ESCAPE SEQUENCES")

Local $mEscape[]
$mEscape["quote"] = 'He said "Hello"'
$mEscape["newline"] = "Line1" & @CRLF & "Line2"
$mEscape["tab"] = "Col1" & @TAB & "Col2"

$sJson = _JsonC_Generate($mEscape)
If Not @error Then
	TestPass("Generated JSON with escape sequences")

	$mData = _JsonC_Parse($sJson)
	If Not @error Then
		TestPass("Parsed JSON with escape sequences")
	Else
		TestFail("Failed to parse JSON with escape sequences")
	EndIf
Else
	TestFail("Failed to generate JSON with escape sequences")
EndIf

; ===============================================================================================================================
; CLEANUP
; ===============================================================================================================================
TestSection("21. CLEANUP")

_JsonC_Shutdown()
TestPass("JSON-C DLL unloaded successfully")

; ===============================================================================================================================
; TEST SUMMARY
; ===============================================================================================================================
ConsoleWrite(@CRLF & "=== TEST SUMMARY ===" & @CRLF)
ConsoleWrite("Tests Passed: " & $iTestsPassed & @CRLF)
ConsoleWrite("Tests Failed: " & $iTestsFailed & @CRLF)
ConsoleWrite("Total Tests:  " & ($iTestsPassed + $iTestsFailed) & @CRLF)

Local $fSuccessRate = ($iTestsPassed / ($iTestsPassed + $iTestsFailed)) * 100
ConsoleWrite("Success Rate: " & Round($fSuccessRate, 2) & "%" & @CRLF)

If $iTestsFailed = 0 Then
	ConsoleWrite(@CRLF & "🎉 ALL TESTS PASSED! 🎉" & @CRLF)
	MsgBox($MB_ICONINFORMATION, "Test Results", "All tests passed successfully!" & @CRLF & @CRLF & _
			"Passed: " & $iTestsPassed & @CRLF & _
			"Failed: " & $iTestsFailed)
Else
	ConsoleWrite(@CRLF & "⚠️ SOME TESTS FAILED ⚠️" & @CRLF)
	MsgBox($MB_ICONWARNING, "Test Results", "Some tests failed!" & @CRLF & @CRLF & _
			"Passed: " & $iTestsPassed & @CRLF & _
			"Failed: " & $iTestsFailed)
EndIf

; ===============================================================================================================================
; HELPER FUNCTIONS
; ===============================================================================================================================

Func TestSection($sName)
	ConsoleWrite(@CRLF & $sName & @CRLF)
	ConsoleWrite(_StringRepeat("-", 70) & @CRLF)
EndFunc

Func TestPass($sMessage)
	$iTestsPassed += 1
	ConsoleWrite("  ✓ PASS: " & $sMessage & @CRLF)
EndFunc

Func TestFail($sMessage)
	$iTestsFailed += 1
	ConsoleWrite("  ✗ FAIL: " & $sMessage & @CRLF)
EndFunc
