#include-once
#include <A3LMemory.au3>

Opt("MustDeclareVars", 1)

; #INDEX# =======================================================================================================================
; Title .........: Header
; Description ...: A header control is a window that is usually positioned above columns of text or numbers.  It contains a title
;                  for each column, and it can be divided into parts.
; Author ........: Paul Campbell (PaulIA)
; ===============================================================================================================================

; #CONSTANTS# ===================================================================================================================
  Global Const $HDF_LEFT                  = 0x00000000
  Global Const $HDF_RIGHT                 = 0x00000001
  Global Const $HDF_CENTER                = 0x00000002
  Global Const $HDF_JUSTIFYMASK           = 0x00000003

  Global Const $HDF_BITMAP_ON_RIGHT       = 0x00001000
  Global Const $HDF_BITMAP                = 0x00002000
  Global Const $HDF_STRING                = 0x00004000
  Global Const $HDF_OWNERDRAW             = 0x00008000
  Global Const $HDF_DISPLAYMASK           = 0x0000F000

  Global Const $HDF_RTLREADING            = 0x00000004
  Global Const $HDF_SORTDOWN              = 0x00000200
  Global Const $HDF_IMAGE                 = 0x00000800
  Global Const $HDF_SORTUP                = 0x00000400
  Global Const $HDF_FLAGMASK              = 0x00000E04

  Global Const $HDI_WIDTH                 = 0x00000001
  Global Const $HDI_TEXT                  = 0x00000002
  Global Const $HDI_FORMAT                = 0x00000004
  Global Const $HDI_PARAM                 = 0x00000008
  Global Const $HDI_BITMAP                = 0x00000010
  Global Const $HDI_IMAGE                 = 0x00000020
  Global Const $HDI_DI_SETITEM            = 0x00000040
  Global Const $HDI_ORDER                 = 0x00000080
  Global Const $HDI_FILTER                = 0x00000100

  Global Const $HHT_NOWHERE               = 0x00000001
  Global Const $HHT_ONHEADER              = 0x00000002
  Global Const $HHT_ONDIVIDER             = 0x00000004
  Global Const $HHT_ONDIVOPEN             = 0x00000008
  Global Const $HHT_ONFILTER              = 0x00000010
  Global Const $HHT_ONFILTERBUTTON        = 0x00000020
  Global Const $HHT_ABOVE                 = 0x00000100
  Global Const $HHT_BELOW                 = 0x00000200
  Global Const $HHT_TORIGHT               = 0x00000400
  Global Const $HHT_TOLEFT                = 0x00000800
; ===============================================================================================================================

; #MESSAGES# ====================================================================================================================
  Global Const $HDM_FIRST                 = 0x1200
  Global Const $HDM_GETITEMCOUNT          = $HDM_FIRST + 0
  Global Const $HDM_INSERTITEM            = $HDM_FIRST + 1
  Global Const $HDM_DELETEITEM            = $HDM_FIRST + 2
  Global Const $HDM_GETITEM               = $HDM_FIRST + 3
  Global Const $HDM_SETITEM               = $HDM_FIRST + 4
  Global Const $HDM_LAYOUT                = $HDM_FIRST + 5
  Global Const $HDM_HITTEST               = $HDM_FIRST + 6
  Global Const $HDM_GETITEMRECT           = $HDM_FIRST + 7
  Global Const $HDM_SETIMAGELIST          = $HDM_FIRST + 8
  Global Const $HDM_GETIMAGELIST          = $HDM_FIRST + 9
  Global Const $HDM_INSERTITEMW           = $HDM_FIRST + 10
  Global Const $HDM_GETITEMW              = $HDM_FIRST + 11
  Global Const $HDM_SETITEMW              = $HDM_FIRST + 12
  Global Const $HDM_ORDERTOINDEX          = $HDM_FIRST + 15
  Global Const $HDM_CREATEDRAGIMAGE       = $HDM_FIRST + 16
  Global Const $HDM_GETORDERARRAY         = $HDM_FIRST + 17
  Global Const $HDM_SETORDERARRAY         = $HDM_FIRST + 18
  Global Const $HDM_SETHOTDIVIDER         = $HDM_FIRST + 19
  Global Const $HDM_SETBITMAPMARGIN       = $HDM_FIRST + 20
  Global Const $HDM_GETBITMAPMARGIN       = $HDM_FIRST + 21
  Global Const $HDM_SETFILTERCHANGETIMEOUT= $HDM_FIRST + 22
  Global Const $HDM_EDITFILTER            = $HDM_FIRST + 23
  Global Const $HDM_CLEARFILTER           = $HDM_FIRST + 24
  Global Const $HDM_GETUNICODEFORMAT      = $CCM_GETUNICODEFORMAT
  Global Const $HDM_SETUNICODEFORMAT      = $CCM_SETUNICODEFORMAT
; ===============================================================================================================================

; #NOTIFICATIONS# ===============================================================================================================
  Global Const $HDN_ITEMCHANGING          = 0xFFFFFED4    ; Sent when a header item is about to change
  Global Const $HDN_ITEMCHANGED           = 0xFFFFFED3    ; Sent when a header item has changed
  Global Const $HDN_ITEMCLICK             = 0xFFFFFED2    ; Sent when the user clicks the control
  Global Const $HDN_ITEMDBLCLICK          = 0xFFFFFED1    ; Sent when the user double clicks the control
  Global Const $HDN_DIVIDERDBLCLICK       = 0xFFFFFECF    ; Sent when the user double clicks the divider
  Global Const $HDN_BEGINTRACK            = 0xFFFFFECE    ; Sent when the user has begun dragging a divider
  Global Const $HDN_ENDTRACK              = 0xFFFFFECD    ; Sent when the user has finished dragging a divider
  Global Const $HDN_TRACK                 = 0xFFFFFECC    ; Sent when the user is dragging a divider
  Global Const $HDN_GETDISPINFO           = 0xFFFFFECB    ; Sent when the control needs information about a callback
  Global Const $HDN_BEGINDRAG             = 0xFFFFFECA    ; Sent when a drag operation has begun
  Global Const $HDN_ENDDRAG               = 0xFFFFFEC9    ; Sent when a drag operation has ended
  Global Const $HDN_FILTERCHANGE          = 0xFFFFFEC8    ; Sent when the attributes of a header control filter are being changed
  Global Const $HDN_FILTERBTNCLICK        = 0xFFFFFEC7    ; Sent when filter button is clicked
  Global Const $HDN_ITEMCHANGINGW         = 0xFFFFFEC0    ; [Unicode] Sent when a header item is about to change
  Global Const $HDN_ITEMCHANGEDW          = 0xFFFFFEBF    ; [Unicode] Sent when a header item has changed
  Global Const $HDN_ITEMCLICKW            = 0xFFFFFEBE    ; [Unicode] Sent when the user clicks the control
  Global Const $HDN_ITEMDBLCLICKW         = 0xFFFFFEBD    ; [Unicode] Sent when the user double clicks the control
  Global Const $HDN_DIVIDERDBLCLICKW      = 0xFFFFFEBB    ; [Unicode] Sent when the user double clicks the divider
  Global Const $HDN_BEGINTRACKW           = 0xFFFFFEBA    ; [Unicode] Sent when the user has begun dragging a divider
  Global Const $HDN_ENDTRACKW             = 0xFFFFFEB9    ; [Unicode] Sent when the user has finished dragging a divider
  Global Const $HDN_TRACKW                = 0xFFFFFEB8    ; [Unicode] Sent when the user is dragging a divider
  Global Const $HDN_GETDISPINFOW          = 0xFFFFFEB7    ; [Unicode] Sent when the control needs information about a callback
; ===============================================================================================================================

; #STYLES# ======================================================================================================================
  Global Const $HDS_HORZ                  = 0x00000000    ; Creates a header control with a horizontal orientation
  Global Const $HDS_BUTTONS               = 0x00000002    ; Each item in the control looks and behaves like a push button
  Global Const $HDS_HOTTRACK              = 0x00000004    ; Enables hot tracking
  Global Const $HDS_HIDDEN                = 0x00000008    ; Indicates a header control that is intended to be hidden
  Global Const $HDS_DRAGDROP              = 0x00000040    ; Allows drag-and-drop reordering of header items
  Global Const $HDS_FULLDRAG              = 0x00000080    ; Column contents are displayed while the user resizes a column
  Global Const $HDS_FILTERBAR             = 0x00000100    ; Include a filter bar as part of the standard header control
  Global Const $HDS_FLAT                  = 0x00000200    ; Control is drawn flat when XP is running in classic mode
  Global Const $HDS_DEFAULT               = 0x00000046    ; Default header style
; ===============================================================================================================================

; #STRUCTURE# ===================================================================================================================
; Description ...: tagHDHITTESTINFO structure
; Fields ........: X     - Horizontal postion to be hit test, in client coordinates
;                  Y     - Vertical position to be hit test, in client coordinates
;                  Flags - Information about the results of a hit test
;                  Item  - If the hit test is successful, contains the index of the item at the hit test point
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: Contains information about a hit test. This structure is used with the $HDM_HITTEST message.
; ===============================================================================================================================
Global Const $tagHDHITTESTINFO = "int X;int Y;int Flags;int Item"

; #STRUCTURE# ===================================================================================================================
; Description ...: tagHDITEM structure
; Fields ........: Mask    - Flags indicating which other structure members contain valid data or must be filled in
;                  XY      - Width or height of the item
;                  Text    - Item string
;                  hBMP    - Handle to the item bitmap
;                  TextMax - Length of the item string
;                  Fmt     - Flags that specify the item's format
;                  Param   - Application-defined item data
;                  Image   - Zero-based index of an image within the image list
;                  Order   - Order in which the item appears within the header control, from left to right
;                  Type    - Type of filter specified by pFilter
;                  pFilter - Address of an application-defined data item
;                  State   - Item state
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: Contains information about an item in the control
; ===============================================================================================================================
Global Const $tagHDITEM = "int Mask;int XY;ptr Text;int hBMP;int TextMax;int Fmt;int Param;int Image;int Order;int Type;ptr pFilter;int State"

; #STRUCTURE# ===================================================================================================================
; Description ...: tagHDLAYOUT structure
; Fields ........: Rect      - Pointer to a RECT structure that contains the rectangle that the header control will occupy
;                  WindowPos - Pointer to a WINDOWPOS structure that receives information about the size/position of the control
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: Contains information used to set the size and position of  the  control.  This  structure  is  used  with  the
;                  $HDM_LAYOUT message
; ===============================================================================================================================
Global Const $tagHDLAYOUT = "ptr Rect;ptr WindowPos"

; #STRUCTURE# ===================================================================================================================
; Description ...: tagHDTEXTFILTER structure
; Fields ........: Text    - Pointer to the buffer containing the filter
;                  TextMax - The maximum size, in characters, for an edit control buffer
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: Contains information about header control text filters
; ===============================================================================================================================
Global Const $tagHDTEXTFILTER = "ptr Text;int TextMax"

; #STRUCTURE# ===================================================================================================================
; Description ...: tagNMHDDISPINFO structure
; Fields ........: WndFrom - Window handle to the control sending a message
;                  IDFrom  - Identifier of the control sending a message
;                  Code    - Notification code
;                  Item    - Zero based index of the item in the header control
;                  Mask    - Set of bit flags specifying which members of the structure must be filled in by  the  owner  of  the
;                  +control. This value can be a combination of the following values:
;                  |$HDI_TEXT       - The Text field must be filled in
;                  |$HDI_IMAGE      - The Image field must be filled in
;                  |$HDI_LPARAM     - The lParam field must be filled in
;                  |$HDI_DI_SETITEM - A return value. Indicates that the control should store the item information  and  not  ask
;                  +for it again.
;                  Text    - Pointer to a null terminated string containing the text that will be displayed for the header item
;                  TextMax - Size of the buffer that Text points to
;                  Image   - Zero based index of an image within the image list.  The specified image will be displayed with  the
;                  +header item, but it does not take the place of the item's bitmap.  If iImage is set to $I_IMAGECALLBACK,  the
;                  +control requests image information for this item by using an $HDN_GETDISPINFO notification message.
;                  lParam  - An application-defined value to associate with the item
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: Contains information used in handling $HDN_GETDISPINFO notification messages
; ===============================================================================================================================
Global Const $tagNMHDDISPINFO = "int WndFrom;int IDFrom;int Code;int Item;int Mask;ptr Text;int TextMax;int Image;int lParam"

; #STRUCTURE# ===================================================================================================================
; Description ...: tagNMHDFILTERBTNCLICK structure
; Fields ........: WndFrom - Window handle to the control sending a message
;                  IDFrom  - Identifier of the control sending a message
;                  Code    - Notification code
;                  Item    - Zero based index of the control to which this structure refers
;                  Left    - X coordinate of the upper left corner of the rectangle
;                  Top     - Y coordinate of the upper left corner of the rectangle
;                  Right   - X coordinate of the lower right corner of the rectangle
;                  Bottom  - Y coordinate of the lower right corner of the rectangle
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: Specifies or receives the attributes of a filter button click
; ===============================================================================================================================
Global Const $tagNMHDFILTERBTNCLICK = "int WndFrom;int IDFrom;int Code;int Item;int Left;int Top;int Right;int Bottom"

; #STRUCTURE# ===================================================================================================================
; Description ...: tagNMHEADER structure
; Fields ........: WndFrom - Window handle to the control sending a message
;                  IDFrom  - Identifier of the control sending a message
;                  Code    - Notification code
;                  Item    - Zero based index of the control to which this structure refers
;                  Button  - Index of the mouse button used to generate the notification message.  This member can be one of  the
;                  +following values:
;                  |0 - Left button
;                  |1 - Right button
;                  |2 - Middle button
;                  pItem   - Optional pointer to a tagHDITEM structure containing information about the item specified  by  Item.
;                  +The mask member of the tagHDITEM structure indicates which of its members are valid.
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: Contains information about control notification messages
; ===============================================================================================================================
Global Const $tagNMHEADER = "int WndFrom;int IDFrom;int Code;int Item;int Button;ptr pItem"

; #VARIABLES# ===================================================================================================================
Global $ghHDLastWnd
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Description ...: Adds a new header item
; Parameters ....: $hWnd        - Handle to control
;                  $sText       - Item text
;                  $iWidth      - Item width
;                  $iAlign      - Text alignment:
;                  |0 - Text is left-aligned
;                  |1 - Text is right-aligned
;                  |2 - Text is centered
;                  $iImage      - Zero based index of an image within the image list
;                  $fOnRight    - If True, the column image appears to the right of text
; Return values .: Success      - The index of the new item
;                  Failure      - -1
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_InsertItem
; ===============================================================================================================================
Func _Header_AddItem($hWnd, $sText, $iWidth=50, $iAlign=0, $iImage=-1, $fOnRight=False)
  Return _Header_InsertItem($hWnd, 999999, $sText, $iWidth, $iAlign, $iImage, $fOnRight)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Clears the filter
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_ClearFilterAll
; ===============================================================================================================================
Func _Header_ClearFilter($hWnd, $iIndex)
  Return _API_SendMessage($hWnd, $HDM_CLEARFILTER, $iIndex, 0) <> 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Clears all of the filters
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_ClearFilter
; ===============================================================================================================================
Func _Header_ClearFilterAll($hWnd)
  Return _API_SendMessage($hWnd, $HDM_CLEARFILTER, -1, 0) <> 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Click on a item
; Parameters ....: $hWnd        - Handle to the control
;                  $iIndex      - Zero based item index
;                  $sButton     - Button to click
;                  $fMove       - If True, the mouse will be moved. If False, the mouse does not move.
;                  $iClicks     - Number of clicks
;                  $iSpeed      - Mouse movement speed
;                  $fPopupScan  - If True, a popup scan will be done after the click
; Return values .: Success      - Zero based item index
;                  Failure      - -1
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_Create
; ===============================================================================================================================
Func _Header_Click($hWnd, $iIndex, $sButton="left", $fMove=False, $iClicks=1, $iSpeed=0, $fPopupScan=False)
  Local $tPoint, $tRect, $iX, $iY

  $tRect  = _Header_GetItemRectEx($hWnd, $iIndex)
  $tPoint = _Lib_PointFromRect($tRect)
  _API_ClientToScreen($hWnd, $tPoint)
  _Lib_GetXYFromPoint($tPoint, $iX, $iY)
  _Lib_MouseClick($sButton, $iX, $iY, $fMove, $iClicks, $iSpeed, $fPopupScan)
  Return $iIndex
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Creates a Header control
; Parameters ....: $hWnd        - Handle to parent or owner window
;                  $iStyle      - Control styles:
;                  |$HDS_BUTTONS   - Each item in the control looks and behaves like a push button
;                  |$HDS_DRAGDROP  - Allows drag-and-drop reordering of header items
;                  |$HDS_FILTERBAR - Include a filter bar as part of the standard header control
;                  |$HDS_FLAT      - Causes the header control to be drawn flat
;                  |$HDS_FULLDRAG  - Causes the header control to display column contents
;                  |$HDS_HIDDEN    - Indicates a header control that is intended to be hidden
;                  |$HDS_HORZ      - Creates a header control with a horizontal orientation
;                  |$HDS_HOTTRACK  - Enables hot tracking
;                  -
;                  |Default: $HDS_BUTTONS, $HDS_HOTTRACK, $HDS_DRAGDROP
;                  |Forced : $WS_CHILD, $WS_VISIBLE
; Return values .: Success      - Handle to the Header control
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_Click
; ===============================================================================================================================
Func _Header_Create($hWnd, $iStyle=0x00000046)
  Local $hHeader, $tRect, $tWndPos, $iFlags

  $iStyle  = BitOR($iStyle, $WS_CHILD, $WS_VISIBLE)
  $hHeader = _API_CreateWindowEx(0, "SysHeader32", "", $iStyle, 0, 0, 0, 0, $hWnd)
  $tRect   = _API_GetClientRect($hWnd)
  $tWndPos = _Header_Layout($hHeader, $tRect)
  $iFlags  = BitOR(DllStructGetData($tWndPos, "Flags"), $SWP_SHOWWINDOW)
  _API_SetWindowPos($hHeader, DllStructGetData($tWndPos, "InsertAfter"), _
                              DllStructGetData($tWndPos, "X"          ), _
                              DllStructGetData($tWndPos, "Y"          ), _
                              DllStructGetData($tWndPos, "CX"         ), _
                              DllStructGetData($tWndPos, "CY"         ), $iFlags)
  _Lib_SetFont($hHeader, _API_GetStockObject($DEFAULT_GUI_FONT))
  _Header_SetUnicodeFormat($hHeader, False)
  Return $hHeader
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Creates a semi-transparent version of an item's image for use as a dragging image
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index.  The image assigned to the item is the basis for the transparent image.
; Return values .: Success      - Handle to an image list that contains the new image as its only element
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetImageList
; ===============================================================================================================================
Func _Header_CreateDragImage($hWnd, $iIndex)
  Return _API_SendMessage($hWnd, $HDM_CREATEDRAGIMAGE, $iIndex, 0)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Deletes a header item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_AddItem, _Header_InsertItem
; ===============================================================================================================================
Func _Header_DeleteItem($hWnd, $iIndex)
  Return _API_SendMessage($hWnd, $HDM_DELETEITEM, $iIndex, 0) <> 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Starts editing the specified filter
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $fDiscard    - Flag that specifies how to handle the user's editing changes.  Use this flag to specify what to
;                  +do if the user is in the process of editing the filter when the message is sent:
;                  | True - Discard the changes made by the user
;                  |False - Accept the changes made by the user
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_ClearFilter, _Header_ClearFilterAll
; ===============================================================================================================================
Func _Header_EditFilter($hWnd, $iIndex, $fDiscard=True)
  Return _API_SendMessage($hWnd, $HDM_EDITFILTER, $iIndex, $fDiscard) <> 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the width of the bitmap margin
; Parameters ....: $hWnd        - Handle to control
; Return values .: Success      - The width of the bitmap margin in pixels.  If the bitmap margin was not  previously  specified,
;                  +the default value of 3 * GetSystemMetrics(SM_CXEDGE) is returned.
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetBitmapMargin
; ===============================================================================================================================
Func _Header_GetBitmapMargin($hWnd)
  Return _API_SendMessage($hWnd, $HDM_GETBITMAPMARGIN, 0, 0)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the handle to the image list
; Parameters ....: $hWnd        - Handle to control
; Return values .: Success      - A handle to the image list set for the header control
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetImageList
; ===============================================================================================================================
Func _Header_GetImageList($hWnd)
  Return _API_SendMessage($hWnd, $HDM_GETIMAGELIST, 0, 0)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves information about an item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $tItem       - DllStructCreate($tagHDITEM) structure
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: When the message is sent, the mask member indicates the type of information being requested.  When the message
;                  returns, the other members receive the requested information.  If the mask member specifies zero, the  message
;                  returns True but copies no information to the structure.
; Related .......: _Header_SetItem, DllStructCreate($tagHDITEM)
; ===============================================================================================================================
Func _Header_GetItem($hWnd, $iIndex, ByRef $tItem)
  Local $iItem, $pItem, $pMemory, $tMemMap, $iResult

  $pItem = DllStructGetPtr($tItem)
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    $iResult = _API_SendMessage($hWnd, $HDM_GETITEM, $iIndex, $pItem)
  else
    $iItem   = DllStructGetSize($tItem)
    $pMemory = _Mem_CtrlInit($hWnd, $iItem, $tMemMap)
    _Mem_CtrlWrite($tMemMap, $pItem)
    $iResult = _API_SendMessage($hWnd, $HDM_GETITEM, $iIndex, $pMemory)
    _Mem_CtrlRead($tMemMap, $pMemory, $pItem, $iItem)
    _Mem_CtrlFree($tMemMap)
  endif
  Return $iResult <> 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the item text alignment
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Item text alignment:
;                  |0 - Left
;                  |1 - Right
;                  |2 - Center
;                  Failure      - -1
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemAlign
; ===============================================================================================================================
Func _Header_GetItemAlign($hWnd, $iIndex)
  Switch BitAND(_Header_GetItemFormat($hWnd, $iIndex), $HDF_JUSTIFYMASK)
    case $HDF_LEFT
      Return 0
    case $HDF_RIGHT
      Return 1
    case $HDF_CENTER
      Return 2
    case else
      Return -1
  EndSwitch
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the item bitmap handle
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Item bitmap handle
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemBitmap
; ===============================================================================================================================
Func _Header_GetItemBitmap($hWnd, $iIndex)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask", $HDI_BITMAP)
  _Header_GetItem($hWnd, $iIndex, $tItem)
  Return DllStructGetData($tItem, "hBmp")
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves a count of the items
; Parameters ....: $hWnd        - Handle to control
; Return values .: Success      - The number of items
;                  Failure      - -1
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......:
; ===============================================================================================================================
Func _Header_GetItemCount($hWnd)
  Return _API_SendMessage($hWnd, $HDM_GETITEMCOUNT, 0, 0)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Returns the item display information
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Item display information:
;                  |1 - The item displays a bitmap
;                  |2 - The bitmap appears to the right of text
;                  |4 - The control's owner draws the item
;                  |8 - The item displays a string
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemDisplay
; ===============================================================================================================================
Func _Header_GetItemDisplay($hWnd, $iIndex)
  Local $iFormat, $iResult=0

  $iFormat = _Header_GetItemFormat($hWnd, $iIndex)
  if BitAND($iFormat, $HDF_BITMAP         ) <> 0 then $iResult = BitOR($iResult, 1)
  if BitAND($iFormat, $HDF_BITMAP_ON_RIGHT) <> 0 then $iResult = BitOR($iResult, 2)
  if BitAND($iFormat, $HDF_OWNERDRAW      ) <> 0 then $iResult = BitOR($iResult, 4)
  if BitAND($iFormat, $HDF_STRING         ) <> 0 then $iResult = BitOR($iResult, 8)
  Return $iResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Returns the item flag information
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Item flag information:
;                  |1 - Displays an image from an image list
;                  |2 - Text reads in the opposite direction from the text in the parent window
;                  |4 - Draws a down arrow on this item
;                  |8 - Draws a up arrow on this item
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemFlags
; ===============================================================================================================================
Func _Header_GetItemFlags($hWnd, $iIndex)
  Local $iFormat, $iResult=0

  $iFormat = _Header_GetItemFormat($hWnd, $iIndex)
  if BitAND($iFormat, $HDF_IMAGE     ) <> 0 then $iResult = BitOR($iResult, 1)
  if BitAND($iFormat, $HDF_RTLREADING) <> 0 then $iResult = BitOR($iResult, 2)
  if BitAND($iFormat, $HDF_SORTDOWN  ) <> 0 then $iResult = BitOR($iResult, 4)
  if BitAND($iFormat, $HDF_SORTUP    ) <> 0 then $iResult = BitOR($iResult, 8)
  Return $iResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Returns the format of the item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Item format:
;                  |HDF_CENTER          - The item's contents are centered
;                  |HDF_LEFT            - The item's contents are left-aligned.
;                  |HDF_RIGHT           - The item's contents are right-aligned.
;                  |HDF_BITMAP          - The item displays a bitmap.
;                  |HDF_BITMAP_ON_RIGHT - The bitmap appears to the right of text.
;                  |HDF_OWNERDRAW       - The control's owner draws the item.
;                  |HDF_STRING          - The item displays a string.
;                  |HDF_IMAGE           - Display an image from an image list
;                  |HDF_RTLREADING      - Text will read in the opposite direction
;                  |HDF_SORTDOWN        - Draw a down-arrow on this item
;                  |HDF_SORTUP          - Draw an up-arrow on this item
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemFormat
; ===============================================================================================================================
Func _Header_GetItemFormat($hWnd, $iIndex)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask", $HDI_FORMAT)
  _Header_GetItem($hWnd, $iIndex, $tItem)
  Return DllStructGetData($tItem, "Fmt")
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrives the index of an image within the image list
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Zero based index of the image
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemImage
; ===============================================================================================================================
Func _Header_GetItemImage($hWnd, $iIndex)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask", $HDI_IMAGE)
  _Header_GetItem($hWnd, $iIndex, $tItem)
  Return DllStructGetData($tItem, "Image")
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrives the order in which the item appears
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Zero based item order
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemOrder
; ===============================================================================================================================
Func _Header_GetItemOrder($hWnd, $iIndex)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask", $HDI_ORDER)
  _Header_GetItem($hWnd, $iIndex, $tItem)
  Return DllStructGetData($tItem, "Order")
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrives the param value of the item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Item param value
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemParam
; ===============================================================================================================================
Func _Header_GetItemParam($hWnd, $iIndex)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask", $HDI_PARAM)
  _Header_GetItem($hWnd, $iIndex, $tItem)
  Return DllStructGetData($tItem, "Param")
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the bounding rectangle for a given item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Array with the following format:
;                  |[0] - X coordinate of the upper left corner of the rectangle
;                  |[1] - Y coordinate of the upper left corner of the rectangle
;                  |[2] - X coordinate of the lower right corner of the rectangle
;                  |[3] - Y coordinate of the lower right corner of the rectangle
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemRectEx
; ===============================================================================================================================
Func _Header_GetItemRect($hWnd, $iIndex)
  Local $tRect, $aRect[4]

  $tRect = _Header_GetItemRectEx($hWnd, $iIndex)
  $aRect[0] = DllStructGetData($tRect, "Left"  )
  $aRect[1] = DllStructGetData($tRect, "Top"   )
  $aRect[2] = DllStructGetData($tRect, "Right" )
  $aRect[3] = DllStructGetData($tRect, "Bottom")
  Return $aRect
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the bounding rectangle for a given item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - tagRECT structure that receives the bounding rectangle
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemRect
; ===============================================================================================================================
Func _Header_GetItemRectEx($hWnd, $iIndex)
  Local $iRect, $pRect, $tRect, $pMemory, $tMemMap

  $tRect = DllStructCreate($tagRECT)
  $pRect = DllStructGetPtr($tRect)
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    _API_SendMessage($hWnd, $HDM_GETITEMRECT, $iIndex, $pRect)
  else
    $iRect   = DllStructGetSize($tRect)
    $pMemory = _Mem_CtrlInit($hWnd, $iRect, $tMemMap)
    _Mem_CtrlWrite($tMemMap, $pRect)
    _API_SendMessage($hWnd, $HDM_GETITEMRECT, $iIndex, $pMemory)
    _Mem_CtrlRead($tMemMap, $pMemory, $pRect, $iRect)
    _Mem_CtrlFree($tMemMap)
  endif
  Return $tRect
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the item text
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Item text
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemText
; ===============================================================================================================================
Func _Header_GetItemText($hWnd, $iIndex)
  Local $pBuffer, $tBuffer, $iItem, $pItem, $tItem, $pMemory, $tMemMap, $pText

  $tBuffer = DllStructCreate("char Text[4096]")
  $pBuffer = DllStructGetPtr($tBuffer  )
  $tItem   = DllStructCreate($tagHDITEM)
  $pItem   = DllStructGetPtr($tItem    )
  DllStructSetData($tItem, "Mask", $HDI_TEXT)
  DllStructSetData($tItem, "TextMax", 4096)
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    DllStructSetData($tItem, "Text", $pBuffer)
    _API_SendMessage($hWnd, $HDM_GETITEM, $iIndex, $pItem)
  else
    $iItem   = DllStructGetSize($tItem)
    $pMemory = _Mem_CtrlInit($hWnd, $iItem + 4096, $tMemMap)
    $pText   = $pMemory + $iItem
    DllStructSetData($tItem, "Text", $pText)
    _Mem_CtrlWrite($tMemMap, $pItem, $pMemory, $iItem)
    _API_SendMessage($hWnd, $HDM_GETITEM, $iIndex, $pMemory)
    _Mem_CtrlRead($tMemMap, $pText, $pBuffer, 4096)
    _Mem_CtrlFree($tMemMap)
  endif
  Return DllStructGetData($tBuffer, "Text")
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the item's width
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
; Return values .: Success      - Width of the item
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemWidth
; ===============================================================================================================================
Func _Header_GetItemWidth($hWnd, $iIndex)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask", $HDI_WIDTH)
  _Header_GetItem($hWnd, $iIndex, $tItem)
  Return DllStructGetData($tItem, "XY")
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the current left-to-right order of items in a header control
; Parameters ....: $hWnd        - Handle to control
; Return values .: Success      - Array with the following format:
;                  |[0] - Number of items in array
;                  |[1] - Item index 1
;                  |[2] - Item index 2
;                  |[n] - Item index n
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetOrderArray
; ===============================================================================================================================
Func _Header_GetOrderArray($hWnd)
  Local $iI, $iBuffer, $pBuffer, $tBuffer, $iItems, $pMemory, $tMemMap

  $iItems  = _Header_GetItemCount($hWnd)
  $tBuffer = DllStructCreate("int[" & $iItems & "]")
  $pBuffer = DllStructGetPtr($tBuffer)
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    _API_SendMessage($hWnd, $HDM_GETORDERARRAY, $iItems, $pBuffer)
  else
    $iBuffer = DllStructGetSize($tBuffer)
    $pMemory = _Mem_CtrlInit($hWnd, $iBuffer, $tMemMap)
    _API_SendMessage($hWnd, $HDM_GETORDERARRAY, $iItems, $pMemory)
    _Mem_CtrlRead($tMemMap, $pMemory, $pBuffer, $iBuffer)
    _Mem_CtrlFree($tMemMap)
  endif

  Dim $aBuffer[$iItems + 1]
  $aBuffer[0] = $iItems
  for $iI = 1 to $iItems
    $aBuffer[$iI] = DllStructGetData($tBuffer, 1, $iI)
  next
  Return $aBuffer
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the Unicode character format flag for the control
; Parameters ....: $hWnd        - Handle to control
; Return values .: True         - Control uses Unicode characters
;                  False        - Control uses ANSI characters
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetUnicodeFormat
; ===============================================================================================================================
Func _Header_GetUnicodeFormat($hWnd)
  Return _API_SendMessage($hWnd, $HDM_GETUNICODEFORMAT, 0, 0) <> 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Tests a point to determine which item is at the specified point
; Parameters ....: $hWnd        - Handle to control
;                  $iX          - X position to test
;                  $iY          - Y position to text
; Return values .: Success      - Array with the following format:
;                  |[ 0] - Zero based index of the item at the specified position, or -1 if no item was found
;                  |[ 1] - If True, position is in control's client window but not on an item
;                  |[ 2] - If True, position is in the control's bounding rectangle
;                  |[ 3] - If True, position is on the divider between two items
;                  |[ 4] - If True, position is on the divider of an item that has a zero width
;                  |[ 5] - If True, position is over the filter area
;                  |[ 6] - If True, position is on the filter button
;                  |[ 7] - If True, position is above the control's bounding rectangle
;                  |[ 8] - If True, position is below the control's bounding rectangle
;                  |[ 9] - If True, position is to the right of the control's bounding rectangle
;                  |[10] - If True, position is to the left of the control's bounding rectangle
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......:
; ===============================================================================================================================
Func _Header_HitTest($hWnd, $iX, $iY)
  Local $iTest, $pTest, $tTest, $pMemory, $tMemMap, $iFlags, $aTest[11]

  $tTest = DllStructCreate($tagHDHITTESTINFO)
  $pTest = DllStructGetPtr($tTest)
  DllStructSetData($tTest, "X", $iX)
  DllStructSetData($tTest, "Y", $iY)
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    $aTest[0] = _API_SendMessage($hWnd, $HDM_HITTEST, 0, $pTest  )
  else
    $iTest   = DllStructGetSize($tTest)
    $pMemory = _Mem_CtrlInit($hWnd, $iTest, $tMemMap)
    _Mem_CtrlWrite($tMemMap, $pTest)
    $aTest[0] = _API_SendMessage($hWnd, $HDM_HITTEST, 0, $pMemory)
    _Mem_CtrlRead($tMemMap, $pMemory, $pTest, $iTest)
    _Mem_CtrlFree($tMemMap)
  endif
  $iFlags = DllStructGetData($tTest, "Flags")
  $aTest[ 1] = BitAND($iFlags, $HHT_NOWHERE       ) <> 0
  $aTest[ 2] = BitAND($iFlags, $HHT_ONHEADER      ) <> 0
  $aTest[ 3] = BitAND($iFlags, $HHT_ONDIVIDER     ) <> 0
  $aTest[ 4] = BitAND($iFlags, $HHT_ONDIVOPEN     ) <> 0
  $aTest[ 5] = BitAND($iFlags, $HHT_ONFILTER      ) <> 0
  $aTest[ 6] = BitAND($iFlags, $HHT_ONFILTERBUTTON) <> 0
  $aTest[ 7] = BitAND($iFlags, $HHT_ABOVE         ) <> 0
  $aTest[ 8] = BitAND($iFlags, $HHT_BELOW         ) <> 0
  $aTest[ 9] = BitAND($iFlags, $HHT_TORIGHT       ) <> 0
  $aTest[10] = BitAND($iFlags, $HHT_TOLEFT        ) <> 0
  Return $aTest
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Inserts a new header item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Index of the item after which the new item is to be inserted.  The new item is inserted at  the
;                  +end of the control if index is greater than or equal to the number of items in the control. If index is zero,
;                  +the new item is inserted at the beginning of the control.
;                  $sText       - Item text
;                  $iWidth      - Item width
;                  $iAlign      - Text alignment:
;                  |0 - Text is left-aligned
;                  |1 - Text is right-aligned
;                  |2 - Text is centered
;                  $iImage      - Zero based index of an image within the image list
;                  $fOnRight    - If True, the column image appears to the right of text
; Return values .: Success      - The index of the new item
;                  Failure      - -1
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_AddItem
; ===============================================================================================================================
Func _Header_InsertItem($hWnd, $iIndex, $sText, $iWidth=50, $iAlign=0, $iImage=-1, $fOnRight=False)
  Local $iBuffer, $pBuffer, $tBuffer, $iItem, $pItem, $tItem, $pMemory, $tMemMap, $pText, $iMask, $iFmt, $iResult
  Local $aAlign[3]=[$HDF_LEFT, $HDF_RIGHT, $HDF_CENTER]

  $iBuffer = StringLen($sText) + 1
  $tBuffer = DllStructCreate("char Text[" & $iBuffer & "]")
  $pBuffer = DllStructGetPtr($tBuffer)
  $tItem   = DllStructCreate($tagHDITEM)
  $pItem   = DllStructGetPtr($tItem)
  $iFmt    = $aAlign[$iAlign]
  $iMask   = BitOR($HDI_WIDTH, $HDI_FORMAT)
  if $sText  <> "" then
    $iMask = BitOR($iMask, $HDI_TEXT  )
    $iFmt  = BitOR($iFmt , $HDF_STRING)
  endif
  if $iImage <> -1 then
    $iMask = BitOR($iMask, $HDI_IMAGE)
    $iFmt  = BitOR($iFmt , $HDF_IMAGE)
  endif
  if $fOnRight then $iFmt = BitOR($iFmt, $HDF_BITMAP_ON_RIGHT)
  DllStructSetData($tBuffer, "Text" , $sText )
  DllStructSetData($tItem  , "Mask" , $iMask )
  DllStructSetData($tItem  , "XY"   , $iWidth)
  DllStructSetData($tItem  , "Fmt"  , $iFmt  )
  DllStructSetData($tItem  , "Image", $iImage)
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    DllStructSetData($tItem, "Text", $pBuffer)
    $iResult = _API_SendMessage($hWnd, $HDM_INSERTITEM, $iIndex, $pItem  )
  else
    $iItem   = DllStructGetSize($tItem)
    $pMemory = _Mem_CtrlInit($hWnd, $iItem + $iBuffer, $tMemMap)
    $pText   = $pMemory + $iItem
    DllStructSetData($tItem, "Text", $pText)
    _Mem_CtrlWrite($tMemMap, $pItem  , $pMemory, $iItem  )
    _Mem_CtrlWrite($tMemMap, $pBuffer, $pText  , $iBuffer)
    $iResult = _API_SendMessage($hWnd, $HDM_INSERTITEM, $iIndex, $pMemory)
    _Mem_CtrlFree($tMemMap)
  endif
  Return $iResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the correct size and position of the control
; Parameters ....: $hWnd        - Handle to control
;                  DllStructGetData($tRect - tagRECT structure that contains the rectangle the control will occupy, "")
; Return values .: Success      - tagWINDOWPOS structure that contains the control size and position
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......:
; ===============================================================================================================================
Func _Header_Layout($hWnd, ByRef $tRect)
  Local $iLayout, $pLayout, $tLayout, $iWindowPos, $pWindowPos, $tWindowPos, $iRect, $pRect, $pMemory, $tMemMap

  $tLayout    = DllStructCreate($tagHDLAYOUT)
  $pLayout    = DllStructGetPtr($tLayout)
  $pRect      = DllStructGetPtr($tRect)
  $tWindowPos = DllStructCreate($tagWINDOWPOS)
  $pWindowPos = DllStructGetPtr($tWindowPos)
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    DllStructSetData($tLayout, "Rect", $pRect)
    DllStructSetData($tLayout, "WindowPos", $pWindowPos)
    _API_SendMessage($hWnd, $HDM_LAYOUT, 0, $pLayout)
  else
    $iLayout    = DllStructGetSize($tLayout   )
    $iRect      = DllStructGetSize($tRect     )
    $iWindowPos = DllStructGetSize($tWindowPos)
    $pMemory    = _Mem_CtrlInit($hWnd, $iLayout + $iRect + $iWindowPos, $tMemMap)
    DllStructSetData($tLayout, "Rect", $pMemory + $iLayout)
    DllStructSetData($tLayout, "WindowPos", $pMemory + $iLayout + $iRect)
    _Mem_CtrlWrite($tMemMap, $pLayout, $pMemory, $iLayout)
    _Mem_CtrlWrite($tMemMap, $pRect  , $pMemory + $iLayout, $iRect)
    _API_SendMessage($hWnd, $HDM_LAYOUT, 0, $pMemory)
    _Mem_CtrlRead($tMemMap, $pMemory + $iLayout + $iRect, $pWindowPos, $iWindowPos)
    _Mem_CtrlFree($tMemMap)
  endif
  Return $tWindowPos
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves an index value for an item based on its order
; Parameters ....: $hWnd        - Handle to control
;                  $iOrder      - Order in which the item appears within the header control, from left to right
; Return values .: Success      - Returns the item index
;                  Failure      - iOrder
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......:
; ===============================================================================================================================
Func _Header_OrderToIndex($hWnd, $iOrder)
  Return _API_SendMessage($hWnd, $HDM_ORDERTOINDEX, $iOrder, 0)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the width of the margin, specified in pixels, of a bitmap
; Parameters ....: $hWnd        - Handle to control
;                  $iWidth      - Width, specified in pixels, of the bitmap margin
; Return values .: Success      - The previous bitmap margin width
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetBitmapMargin
; ===============================================================================================================================
Func _Header_SetBitmapMargin($hWnd, $iWidth)
  Return _API_SendMessage($hWnd, $HDM_SETBITMAPMARGIN, $iWidth, 0)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the filter change timeout interval
; Parameters ....: $hWnd        - Handle to control
;                  $iTimeOut    - Timeout value, in milliseconds
; Return values .: Success      - The index of the filter control being modified
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......:
; ===============================================================================================================================
Func _Header_SetFilterChangeTimeout($hWnd, $iTimeOut)
  Return _API_SendMessage($hWnd, $HDM_SETFILTERCHANGETIMEOUT, 0, $iTimeOut)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Changes the hot divider color
; Parameters ....: $hWnd        - Handle to control
;                  $iFlag       - Value specifying the type of value represented by $iInputValue.  This value can be one  of  the
;                  +following:
;                  | True - Indicates that $iInputValue holds the client coordinates of the pointer
;                  |False - Indicates that $iInputValue holds a divider index value
;                  $iInputValue  - Value interpreted by $iFlag
; Return values .: Success      - A value equal to the index of the divider that the control highlighted
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: This function creates an effect that a header control automatically produces  when  it  has  the  HDS_DRAGDROP
;                  style. It is intended to be used when the owner of the control handles drag-and-drop operations manually.
; Related .......:
; ===============================================================================================================================
Func _Header_SetHotDivider($hWnd, $iFlag, $iInputValue)
  Return _API_SendMessage($hWnd, $HDM_SETHOTDIVIDER, $iFlag, $iInputValue)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Assigns an image list
; Parameters ....: $hWnd        - Handle to control
;                  $hImage      - Handle to an image list
; Return values .: Success      - The handle to the image list previously associated with the control
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetImageList
; ===============================================================================================================================
Func _Header_SetImageList($hWnd, $hImage)
  Return _API_SendMessage($hWnd, $HDM_SETIMAGELIST, 0, $hImage)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets information about an item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $tItem       - DllStructCreate($tagHDITEM) structure
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItem, DllStructCreate($tagHDITEM)
; ===============================================================================================================================
Func _Header_SetItem($hWnd, $iIndex, ByRef $tItem)
  Local $iItem, $pItem, $pMemory, $tMemMap, $iResult

  $pItem = DllStructGetPtr($tItem)
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    $iResult = _API_SendMessage($hWnd, $HDM_SETITEM, $iIndex, $pItem)
  else
    $iItem   = DllStructGetSize($tItem)
    $pMemory = _Mem_CtrlInit($hWnd, $iItem, $tMemMap)
    _Mem_CtrlWrite($tMemMap, $pItem)
    $iResult = _API_SendMessage($hWnd, $HDM_SETITEM, $iIndex, $pMemory)
    _Mem_CtrlFree($tMemMap)
  endif
  Return $iResult <> 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the item text alignment
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $iAlign      - Text alignment:
;                  |0 - Text is left-aligned
;                  |1 - Text is right-aligned
;                  |2 - Text is centered
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemAlign
; ===============================================================================================================================
Func _Header_SetItemAlign($hWnd, $iIndex, $iAlign)
  Local $iFormat, $aAlign[3]=[$HDF_LEFT, $HDF_RIGHT, $HDF_CENTER]

  $iFormat = _Header_GetItemFormat($hWnd, $iIndex)
  $iFormat = BitAND($iFormat, BitNOT($HDF_JUSTIFYMASK))
  $iFormat = BitOR ($iFormat, $aAlign[$iAlign]        )
  Return _Header_SetItemFormat($hWnd, $iIndex, $iFormat)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the item bitmap handle
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $hBmp        - Item bitmap handle
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: After calling this function, make sure you call _Header_SetItemDisplay to set the items display information
; Related .......: _Header_GetItemBitmap, _Header_SetItemDisplay
; ===============================================================================================================================
Func _Header_SetItemBitmap($hWnd, $iIndex, $hBmp)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask", BitOR($HDI_FORMAT, $HDI_BITMAP))
  DllStructSetData($tItem, "Fmt" , $HDF_BITMAP)
  DllStructSetData($tItem, "hBMP", $hBmp      )
  Return _Header_SetItem($hWnd, $iIndex, $tItem)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Returns the item display information
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $iDisplay    - Item display information. Can be a combination of the following:
;                  |1 - The item displays a bitmap
;                  |2 - The bitmap appears to the right of text
;                  |4 - The control's owner draws the item
;                  |8 - The item displays a string
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemDisplay
; ===============================================================================================================================
Func _Header_SetItemDisplay($hWnd, $iIndex, $iDisplay)
  Local $iFormat

  $iFormat = BitAND(_Header_GetItemFormat($hWnd, $iIndex), not $HDF_DISPLAYMASK)
  if BitAND($iDisplay, 1) <> 0 then $iFormat = BitOR($iFormat, $HDF_BITMAP         )
  if BitAND($iDisplay, 2) <> 0 then $iFormat = BitOR($iFormat, $HDF_BITMAP_ON_RIGHT)
  if BitAND($iDisplay, 4) <> 0 then $iFormat = BitOR($iFormat, $HDF_OWNERDRAW      )
  if BitAND($iDisplay, 8) <> 0 then $iFormat = BitOR($iFormat, $HDF_STRING         )
  Return _Header_SetItemFormat($hWnd, $iIndex, $iFormat)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Returns the item flag information
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $iFlags      - Item flag information.  Can be a combination of the following:
;                  |1 - Displays an image from an image list
;                  |2 - Text reads in the opposite direction from the text in the parent window
;                  |4 - Draws a down arrow on this item
;                  |8 - Draws a up arrow on this item
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemFlags
; ===============================================================================================================================
Func _Header_SetItemFlags($hWnd, $iIndex, $iFlags)
  Local $iFormat

  $iFormat = _Header_GetItemFormat($hWnd, $iIndex)
  $iFormat = BitAND($iFormat, BitNOT($HDF_FLAGMASK))
  if BitAND($iFlags, 1) <> 0 then $iFormat = BitOR($iFormat, $HDF_IMAGE     )
  if BitAND($iFlags, 2) <> 0 then $iFormat = BitOR($iFormat, $HDF_RTLREADING)
  if BitAND($iFlags, 4) <> 0 then $iFormat = BitOR($iFormat, $HDF_SORTDOWN  )
  if BitAND($iFlags, 8) <> 0 then $iFormat = BitOR($iFormat, $HDF_SORTUP    )
  Return _Header_SetItemFormat($hWnd, $iIndex, $iFormat)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the format of the item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $iFmt        - Combination of the following format identifiers:
;                  |$HDF_CENTER          - The item's contents are centered
;                  |$HDF_LEFT            - The item's contents are left-aligned
;                  |$HDF_RIGHT           - The item's contents are right-aligned
;                  |$HDF_BITMAP          - The item displays a bitmap
;                  |$HDF_BITMAP_ON_RIGHT - The bitmap appears to the right of text
;                  |$HDF_OWNERDRAW       - The header control's owner draws the item
;                  |$HDF_STRING          - The item displays a string
;                  |$HDF_IMAGE           - Display an image from an image list
;                  |$HDF_RTLREADING      - Text will read in the opposite direction from the text in the parent window
;                  |$HDF_SORTDOWN        - Draws a down-arrow on this item
;                  |$HDF_SORTUP          - Draws an up-arrow on this item
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemFormat
; ===============================================================================================================================
Func _Header_SetItemFormat($hWnd, $iIndex, $iFormat)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask", $HDI_FORMAT)
  DllStructSetData($tItem, "Fmt", $iFormat)
  Return _Header_SetItem($hWnd, $iIndex, $tItem)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the index of an image within the image list
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $iImage      - Zero based image index
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemImage
; ===============================================================================================================================
Func _Header_SetItemImage($hWnd, $iIndex, $iImage)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask" , $HDI_IMAGE)
  DllStructSetData($tItem, "Image", $iImage   )
  Return _Header_SetItem($hWnd, $iIndex, $tItem)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the order in which the item appears
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $iOrder      - Zero based item order
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemOrder
; ===============================================================================================================================
Func _Header_SetItemOrder($hWnd, $iIndex, $iOrder)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask" , $HDI_ORDER)
  DllStructSetData($tItem, "Order", $iOrder   )
  Return _Header_SetItem($hWnd, $iIndex, $tItem)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the param value of the item
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $iParam      - Item param value
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemParam
; ===============================================================================================================================
Func _Header_SetItemParam($hWnd, $iIndex, $iParam)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask" , $HDI_PARAM)
  DllStructSetData($tItem, "Param", $iParam   )
  Return _Header_SetItem($hWnd, $iIndex, $tItem)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the item text
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $sText       - New item text
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_SetItemText
; ===============================================================================================================================
Func _Header_SetItemText($hWnd, $iIndex, $sText)
  Local $iBuffer, $pBuffer, $tBuffer, $iItem, $pItem, $tItem, $pMemory, $tMemMap, $pText, $iResult

  $iBuffer = StringLen($sText) + 1
  $tBuffer = DllStructCreate("char Text[" & $iBuffer & "]")
  $pBuffer = DllStructGetPtr($tBuffer)
  $tItem   = DllStructCreate($tagHDITEM)
  $pItem   = DllStructGetPtr($tItem)
  DllStructSetData($tBuffer, "Text"   , $sText   )
  DllStructSetData($tItem  , "Mask"   , $HDI_TEXT)
  DllStructSetData($tItem  , "TextMax", $iBuffer )
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    DllStructSetData($tItem, "Text", $pBuffer)
    $iResult = _API_SendMessage($hWnd, $HDM_SETITEM, $iIndex, $pItem)
  else
    $iItem   = DllStructGetSize($tItem)
    $pMemory = _Mem_CtrlInit($hWnd, $iItem + $iBuffer, $tMemMap)
    $pText   = $pMemory + $iItem
    DllStructSetData($tItem, "Text", $pText)
    _Mem_CtrlWrite($tMemMap, $pItem  , $pMemory, $iItem)
    _Mem_CtrlWrite($tMemMap, $pBuffer, $pText  , $iBuffer)
    $iResult = _API_SendMessage($hWnd, $HDM_SETITEM, $iIndex, $pMemory)
    _Mem_CtrlFree($tMemMap)
  endif
  Return $iResult <> 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the item's width
; Parameters ....: $hWnd        - Handle to control
;                  $iIndex      - Zero based item index
;                  $iWidth      - New width for item
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetItemWidth
; ===============================================================================================================================
Func _Header_SetItemWidth($hWnd, $iIndex, $iWidth)
  Local $tItem

  $tItem = DllStructCreate($tagHDITEM)
  DllStructSetData($tItem, "Mask", $HDI_WIDTH)
  DllStructSetData($tItem, "XY"  , $iWidth   )
  Return _Header_SetItem($hWnd, $iIndex, $tItem)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the current left-to-right order of items
; Parameters ....: $hWnd        - Handle to control
;                  $aOrder      - Array that specifies the index values for items in the header:
;                  |[0] - Number of items in array
;                  |[1] - Item index 1
;                  |[2] - Item index 2
;                  |[n] - Item index n
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetOrderArray
; ===============================================================================================================================
Func _Header_SetOrderArray($hWnd, ByRef $aOrder)
  Local $iI, $iBuffer, $pBuffer, $tBuffer, $pMemory, $tMemMap, $iResult

  $tBuffer = DllStructCreate("int[" & $aOrder[0] & "]")
  $pBuffer = DllStructGetPtr($tBuffer)
  for $iI = 1 to $aOrder[0]
    DllStructSetData($tBuffer, 1, $aOrder[$iI], $iI)
  next
  if _Lib_InProcess($hWnd, $ghHDLastWnd) then
    $iResult = _API_SendMessage($hWnd, $HDM_SETORDERARRAY, $aOrder[0], $pBuffer)
  else
    $iBuffer = DllStructGetSize($tBuffer)
    $pMemory = _Mem_CtrlInit($hWnd, $iBuffer, $tMemMap)
    _Mem_CtrlWrite($tMemMap, $pBuffer)
    $iResult = _API_SendMessage($hWnd, $HDM_SETORDERARRAY, $aOrder[0], $pMemory)
    _Mem_CtrlFree($tMemMap)
  endif
  Return $iResult <> 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Sets the Unicode character format flag for the control
; Parameters ....: $hWnd        - Handle to control
;                  $fUnicode    - Unicode flag:
;                  | True - Control uses Unicode characters
;                  |False - Control uses ANSI characters
; Return values .: Success      - The previous Unicode format flag
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Header_GetUnicodeFormat
; ===============================================================================================================================
Func _Header_SetUnicodeFormat($hWnd, $fUnicode)
  Return _API_SendMessage($hWnd, $HDM_SETUNICODEFORMAT, $fUnicode, 0)
EndFunc

; ===============================================================================================================================
; Description ...: Debug display WM_NOTIFY messages
; Parameters ....: $hWnd         - Window handle
;                 $iMsg         - Message
;                 $iwParam      - wParam
;                 $ilParam      - lParam
; Return values .: None
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Lib_WM_NOTIFY_EX
; ===============================================================================================================================
Func _Header_WM_NOTIFY($hWnd, $iMsg, $iwParam, $ilParam)
  Local $tNMHDR, $hFrom, $iID, $iCode

  $tNMHDR = DllStructCreate($tagNMHDR, $ilParam)
  $hFrom  = DllStructGetData($tNMHDR, "hWndFrom")
  $iID    = DllStructGetData($tNMHDR, "IDFrom"  )
  $iCode  = DllStructGetData($tNMHDR, "Code"    )
  Select
    case _Lib_WM_NOTIFY_EX("HDN_ITEMCHANGING"    , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ITEMCHANGED"     , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ITEMCLICK"       , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ITEMDBLCLICK"    , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_DIVIDERDBLCLICK" , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_BEGINTRACK"      , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ENDTRACK"        , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_TRACK"           , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_GETDISPINFO"     , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_BEGINDRAG"       , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ENDDRAG"         , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_FILTERCHANGE"    , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_FILTERBTNCLICK"  , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ITEMCHANGINGW"   , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ITEMCHANGEDW"    , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ITEMCLICKW"      , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ITEMDBLCLICKW"   , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_DIVIDERDBLCLICKW", $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_BEGINTRACKW"     , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_ENDTRACKW"       , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_TRACKW"          , $hFrom, $iID, $iCode)
    case _Lib_WM_NOTIFY_EX("HDN_GETDISPINFOW"    , $hFrom, $iID, $iCode)
    case else
      _Lib_WM_NOTIFY($hWnd, $iMsg, $iwParam, $ilParam)
  EndSelect
  Return $GUI_RUNDEFMSG
EndFunc

Opt("MustDeclareVars", 0)