#include-once
#include <A3LWinAPI.au3>

Opt("MustDeclareVars", 1)

; #INDEX# =======================================================================================================================
; Title .........: Memory
; Description ...: The memory manager implements virtual memory, provides a core set of services such  as  memory  mapped  files,
;                  copy-on-write memory, large memory support, and underlying support for the cache manager.
; Author ........: Paul Campbell (PaulIA)
; ===============================================================================================================================

; #CONSTANTS# ===================================================================================================================
  Global Const $GMEM_FIXED              = 0x0000
  Global Const $GMEM_MOVEABLE           = 0x0002
  Global Const $GMEM_NOCOMPACT          = 0x0010
  Global Const $GMEM_NODISCARD          = 0x0020
  Global Const $GMEM_ZEROINIT           = 0x0040
  Global Const $GMEM_MODIFY             = 0x0080
  Global Const $GMEM_DISCARDABLE        = 0x0100
  Global Const $GMEM_NOT_BANKED         = 0x1000
  Global Const $GMEM_SHARE              = 0x2000
  Global Const $GMEM_DDESHARE           = 0x2000
  Global Const $GMEM_NOTIFY             = 0x4000
  Global Const $GMEM_LOWER              = 0x1000
  Global Const $GMEM_VALID_FLAGS        = 0x7F72
  Global Const $GMEM_INVALID_HANDLE     = 0x8000

  Global Const $GPTR                    = 0x0040
  Global Const $GHND                    = 0x0042
; ===============================================================================================================================

; #STRUCTURE# ===================================================================================================================
; Description ...: tagMEMMAP structure
; Fields ........: hProc - Handle to the external process
;                  Size  - Size, in bytes, of the memory block allocated
;                  Mem   - Pointer to the memory block
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; ===============================================================================================================================
Global Const $tagMEMMAP = "int hProc;int Size;ptr Mem"

; #FUNCTION# ====================================================================================================================
; Description ...: Releases a memory map structure for a control
; Parameters ....: $tMemMap     - tagMEMMAP structure
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: This function is used internally by Auto3Lib and should not normally be called
; Related .......: _Mem_CtrlInit
; ===============================================================================================================================
Func _Mem_CtrlFree(ByRef $tMemMap)
  Local $hProcess, $pMemory, $bResult

  $pMemory  = DllStructGetData($tMemMap, "Mem"  )
  $hProcess = DllStructGetData($tMemMap, "hProc")
  ; Thanks to jpm for his tip on using @OSType instead of @OSVersion
  if @OSTYPE = "WIN32_WINDOWS" then
    $bResult = _Mem_VirtualFree  (           $pMemory, 0, $MEM_RELEASE)
  else
    $bResult = _Mem_VirtualFreeEx($hProcess, $pMemory, 0, $MEM_RELEASE)
  endif
  _API_CloseHandle($hProcess)
  Return $bResult
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Initializes a tagMEMMAP structure for a control
; Parameters ....: $hWnd        - Window handle of the process where memory will be mapped
;                  $iSize       - Size, in bytes, of memory space to map
;                  $tMemMap     - tagMEMMAP structure that will be initialized
; Return values .: Success      - Pointer to reserved memory block
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: This function is used internally by Auto3Lib and should not normally be called
; Related .......: _Mem_CtrlFree
; ===============================================================================================================================
Func _Mem_CtrlInit($hWnd, $iSize, ByRef $tMemMap)
  Local $iAccess, $iAlloc, $pMemory, $hProcess, $iProcessID

  _API_GetWindowThreadProcessId($hWnd, $iProcessID)
  if $iProcessID = 0 then _Lib_ShowError("_Mem_CtrlInit: Invalid window handle [0x" & Hex($hWnd) & "]")

  $iAccess  = BitOR($PROCESS_VM_OPERATION, $PROCESS_VM_READ, $PROCESS_VM_WRITE)
  $hProcess = _API_OpenProcess($iAccess, False, $iProcessID, True)
  ; Thanks to jpm for his tip on using @OSType instead of @OSVersion
  if @OSTYPE = "WIN32_WINDOWS" then
    $iAlloc  = BitOR($MEM_RESERVE, $MEM_COMMIT, $MEM_SHARED)
    $pMemory = _Mem_VirtualAlloc  (           0, $iSize, $iAlloc, $PAGE_READWRITE)
  else
    $iAlloc  = BitOR($MEM_RESERVE, $MEM_COMMIT)
    $pMemory = _Mem_VirtualAllocEx($hProcess, 0, $iSize, $iAlloc, $PAGE_READWRITE)
  endif

  if $pMemory = 0 then _Lib_ShowError("_Mem_CtrlInit: Unable to allocate memory")
  $tMemMap = DllStructCreate($tagMEMMAP)
  DllStructSetData($tMemMap, "hProc", $hProcess)
  DllStructSetData($tMemMap, "Size" , $iSize   )
  DllStructSetData($tMemMap, "Mem"  , $pMemory )
  Return $pMemory
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Transfer memory from external address space to internal address space
; Parameters ....: $tMemMap     - tagMEMMAP structure
;                  $pSrce       - Pointer to external memory
;                  $pDest       - Pointer to internal memory
;                  $iSize       - Size in bytes of memory to read
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: This function is used internally by Auto3Lib and should not normally be called
; Related .......: _Mem_CtrlWrite
; ===============================================================================================================================
Func _Mem_CtrlRead(ByRef $tMemMap, $pSrce, $pDest, $iSize)
  Local $iRead

  Return _API_ReadProcessMemory(DllStructGetData($tMemMap, "hProc"), $pSrce, $pDest, $iSize, $iRead)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Transfer memory to external address space from internal address space
; Parameters ....: $tMemMap     - tagMEMMAP structure
;                  $pSrce       - Pointer to internal memory
;                  $pDest       - Pointer to external memory
;                  $iSize       - Size in bytes of memory to write
;                  $sSrce       - Contains the data type for $pSrce
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: This function is used internally by Auto3Lib and should not normally be called
; Related .......: _Mem_CtrlRead
; ===============================================================================================================================
Func _Mem_CtrlWrite(ByRef $tMemMap, $pSrce, $pDest=0, $iSize=0, $sSrce="ptr")
  Local $iWritten

  if $pDest  = 0 then $pDest = DllStructGetData($tMemMap, "Mem" )
  if $iSize  = 0 then $iSize = DllStructGetData($tMemMap, "Size")
  Return _API_WriteProcessMemory(DllStructGetData($tMemMap, "hProc"), $pDest, $pSrce, $iSize, $iWritten, $sSrce)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Allocates the specified number of bytes from the heap
; Parameters ....: $iBytes      - The number of bytes to allocate. If this parameter is zero and the $iFlags parameter  specifies
;                  +$GMEM_MOVEABLE, the function returns a handle to a memory object that is marked as discarded.
;                  $iFlags      - The memory allocation attributes:
;                  |$GMEM_FIXED    - Allocates fixed memory. The return value is a pointer.
;                  |$GMEM_MOVEABLE - Allocates movable memory.  Memory blocks are never moved in physical memory, but they can be
;                  +moved within the default heap.  The return value is a handle to the memory object.  To translate  the  handle
;                  +into a pointer, use the _Mem_GlobalLock function. This value cannot be combined with $GMEM_FIXED.
;                  |$GMEM_ZEROINIT - Initializes memory contents to zero
;                  |$GHND          - Combines $GMEM_MOVEABLE and $GMEM_ZEROINIT
;                  |$GPTR          - Combines $GMEM_FIXED and $GMEM_ZEROINIT
; Return values .: Success      - Handle to the newly allocated memory object
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......: Windows memory management does not provide a separate local heap and global heap.  If this function  succeeds,
;                  it allocates at least the amount of memory requested.  If the actual amount  allocated  is  greater  than  the
;                  amount requested, the process can use the entire amount.  To determine the actual number of  bytes  allocated,
;                  use the _Mem_GlobalSize function.  If the heap does not contain sufficient free space to satisfy the  request,
;                  this function returns NULL.  Memory allocated with this function is guaranteed to be  aligned  on  an  8  byte
;                  boundary. To execute dynamically generated code, use the _Mem_VirtualAlloc function to allocate memory and the
;                  _Mem_VirtualProtect function to grant $PAGE_EXECUTE access.  To  free  the  memory,  use  the  _Mem_GlobalFree
;                  function. It is not safe to free memory allocated with _Mem_GlobalAlloc using _Mem_LocalFree.
; Related .......: _Mem_GlobalLock, _Mem_GlobalSize, _Mem_VirtualAlloc
; ===============================================================================================================================
Func _Mem_GlobalAlloc($iBytes, $iFlags=0)
  Local $aResult

  $aResult = DllCall("Kernel32.dll", "hwnd", "GlobalAlloc", "int", $iFlags, "int", $iBytes)
  Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Frees the specified global memory object and invalidates its handle
; Parameters ....: $hMem        - Handle to the global memory object
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Mem_GlobalAlloc
; ===============================================================================================================================
Func _Mem_GlobalFree($hMem)
  Local $aResult

  $aResult = DllCall("Kernel32.dll", "int", "GlobalFree", "hwnd", $hMem)
  Return $aResult[0] = 0
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Locks a global memory object and returns a pointer to the first byte of the object's memory block
; Parameters ....: $hMem        - Handle to the global memory object
; Return values .: Success      - Pointer to the first byte of the memory block
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Mem_GlobalUnlock
; ===============================================================================================================================
Func _Mem_GlobalLock($hMem)
  Local $aResult

  $aResult = DllCall("Kernel32.dll", "ptr", "GlobalLock", "hwnd", $hMem)
  Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Retrieves the current size of the specified global memory object
; Parameters ....: $hMem        - Handle to the global memory object
; Return values .: Success      - size of the specified global memory object, in bytes
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Mem_GlobalAlloc
; ===============================================================================================================================
Func _Mem_GlobalSize($hMem)
  Local $aResult

  $aResult = DllCall("Kernel32.dll", "int", "GlobalSize", "hwnd", $hMem)
  Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Decrements the lock count associated with a memory object that was allocated with GMEM_MOVEABLE
; Parameters ....: $hMem        - Handle to the global memory object
; Return values .:
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Mem_GlobalLock
; ===============================================================================================================================
Func _Mem_GlobalUnlock($hMem)
  Local $aResult

  $aResult = DllCall("Kernel32.dll", "int", "GlobalUnlock", "hwnd", $hMem)
  Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Moves memory either forward or backward, aligned or unaligned
; Parameters ....: $pSource     - Pointer to the source of the move
;                  $pDest       - Pointer to the destination of the move
;                  $iLength     - Specifies the number of bytes to be copied
; Return values .:
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......:
; ===============================================================================================================================
Func _Mem_MoveMemory($pSource, $pDest, $iLength)
  DllCall("Kernel3232.dll", "none", "RtlMoveMemory", "ptr", $pDest, "ptr", $pSource, "dword", $iLength)
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Reserves or commits a region of pages in the virtual address space of the calling process
; Parameters ....: $pAddress    - Specifies the desired starting address of the region to allocate
;                  $iSize       - Specifies the size, in bytes, of th  region
;                  $iAllocation - Specifies the type of allocation:
;                  |$MEM_COMMIT   - Allocates physical storage in memory or in the paging file on disk for the  specified  region
;                  +of pages.
;                  |$MEM_RESERVE  - Reserves a range of the process's virtual  address  space  without  allocating  any  physical
;                  +storage.
;                  |$MEM_TOP_DOWN - Allocates memory at the highest possible address
;                  $iProtect    - Type of access protection:
;                  |$PAGE_READONLY          - Enables read access to the committed region of pages
;                  |$PAGE_READWRITE         - Enables read and write access to the committed region
;                  |$PAGE_EXECUTE           - Enables execute access to the committed region
;                  |$PAGE_EXECUTE_READ      - Enables execute and read access to the committed region
;                  |$PAGE_EXECUTE_READWRITE - Enables execute, read, and write access to the committed region of pages
;                  |$PAGE_GUARD             - Pages in the region become guard pages
;                  |$PAGE_NOACCESS          - Disables all access to the committed region of pages
;                  |$PAGE_NOCACHE           - Allows no caching of the committed regions of pages
; Return values .: Success      - Memory address pointer
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Mem_VirtualFree
; ===============================================================================================================================
Func _Mem_VirtualAlloc($pAddress, $iSize, $iAllocation, $iProtect)
  Local $aResult

  $aResult = DllCall("Kernel32.dll", "ptr", "VirtualAlloc", "ptr", $pAddress, "int", $iSize, "int", $iAllocation, "int", $iProtect)
  Return SetError($aResult[0]=0, 0, $aResult[0])
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Reserves a region of memory within the virtual address space of a specified process
; Parameters ....: $pAddress    - Specifies the desired starting address of the region to allocate
;                  $iSize       - Specifies the size, in bytes, of th  region
;                  $iAllocation - Specifies the type of allocation:
;                  |$MEM_COMMIT   - Allocates physical storage in memory or in the paging file on disk for the  specified  region
;                  +of pages.
;                  |$MEM_RESERVE  - Reserves a range of the process's virtual  address  space  without  allocating  any  physical
;                  +storage.
;                  |$MEM_TOP_DOWN - Allocates memory at the highest possible address
;                  $iProtect    - Type of access protection:
;                  |$PAGE_READONLY          - Enables read access to the committed region of pages
;                  |$PAGE_READWRITE         - Enables read and write access to the committed region
;                  |$PAGE_EXECUTE           - Enables execute access to the committed region
;                  |$PAGE_EXECUTE_READ      - Enables execute and read access to the committed region
;                  |$PAGE_EXECUTE_READWRITE - Enables execute, read, and write access to the committed region of pages
;                  |$PAGE_GUARD             - Pages in the region become guard pages
;                  |$PAGE_NOACCESS          - Disables all access to the committed region of pages
;                  |$PAGE_NOCACHE           - Allows no caching of the committed regions of pages
; Return values .: Success      - Memory address pointer
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _API_VirtualFreeEx
; ===============================================================================================================================
Func _Mem_VirtualAllocEx($hProcess, $pAddress, $iSize, $iAllocation, $iProtect)
  Local $aResult

  $aResult = DllCall("Kernel32.dll", "ptr", "VirtualAllocEx", "int", $hProcess, "ptr", $pAddress, "int", $iSize, "int", $iAllocation, "int", $iProtect)
  Return SetError($aResult[0]=0, 0, $aResult[0])
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Releases a region of pages within the virtual address space of a process
; Parameters ....: $pAddress    - Points to the base address of the region of pages to be freed
;                  $iSize       - Specifies the size, in bytes, of the region to be freed
;                  $iFreeType   - Specifies the type of free operation:
;                  |$MEM_DECOMMIT - Decommits the specified region of committed pages
;                  |$MEM_RELEASE  - Releases the specified region of reserved pages
; Return values .: Success      - True
;                  Failure      - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Mem_VirtualAlloc
; ===============================================================================================================================
Func _Mem_VirtualFree($pAddress, $iSize, $iFreeType)
  Local $aResult

  $aResult = DllCall("Kernel32.dll", "ptr", "VirtualFree", "ptr", $pAddress, "int", $iSize, "int", $iFreeType)
  Return $aResult[0]
EndFunc

; #FUNCTION# ====================================================================================================================
; Description ...: Releases a region of pages within the virtual address space of a process
; Parameters ....: $hProcess     - Handle to a process
;                  $pAddress     - A pointer to the starting address of the region of memory to be freed
;                  $iSize        - The size of the region of memory to free, in bytes
;                  $iFreeType   - Specifies the type of free operation:
;                  |$MEM_DECOMMIT - Decommits the specified region of committed pages
;                  |$MEM_RELEASE  - Releases the specified region of reserved pages
; Return values .: Success       - True
;                  Failure       - False
; Author ........: Paul Campbell (PaulIA)
; Remarks .......:
; Related .......: _Mem_VirtualAllocEx
; ===============================================================================================================================
Func _Mem_VirtualFreeEx($hProcess, $pAddress, $iSize, $iFreeType)
  Local $aResult

  $aResult = DllCall("Kernel32.dll", "ptr", "VirtualFreeEx", "hwnd", $hProcess, "ptr", $pAddress, "int", $iSize, "int", $iFreeType)
  Return $aResult[0]
EndFunc

Opt("MustDeclareVars", 0)