#include-once
#region Header
#cs
    Title:          MsAgent Automation UDF Library for AutoIt3
    Filename:       msagentlib.au3
    Description:    A collection of functions for manipulating MsAgent characters
    Author:         Alexsandro Percy
    Version:        1.1.1
    Last Update:    11/01/06
    Requirements:   AutoIt with COM support (3.1.1.63 or higher) and MsAgent Control
    Notes:          Errors associated with incorrect objects will be common user errors.  AutoIt beta 3.1.1.63 has added an ObjName()
                    function that will be used to trap and report most of these errors.
					
		Update History:
	===================================================
	1.1.1 11/01/06
	
	Corrected Function
	_MAScriptPlay() corrected - variable "$i" now is declarated in local scope
	
	---------------------------------------------------
	1.1 10/27/06
    Enhanced Functions
    _MALoadCharacter() enhanced - now it feeds $g_oSinkObject_MsAgent and $g_oSinkObject_Agent, that Handles incoming events from the agent control
    
	New Functions
	_MACommandAdd() added
	_MACommandRemove() added
	---------------------------------------------------
	1.0 10/24/06
	Initial Release
	---------------------------------------------------
	
#ce
#endregion
#region Function List
#cs
    >> Core Functions
    
    _MALoadCharacter()
    Create a Character object suitable for animating
        - Also starts handling events from MsAgent Control
        - To handle 'commands' (see _MACommandAdd()) comming from agent, create a function called MsAgent_Command, like:
            ;Func MsAgent_Command( $UserInput )
            ;    If IsObj( $UserInput ) Then
            ;        Select
            ;            Case $UserInput.Name = "MyCommand"
            ;                ;Do anything
            ;            Case $UserInput.Name = "MyCommand2"
            ;                ;Do anything
            ;        EndSelect
            ;    EndIf
            ;EndFunc
        - There is a manner to handle events comming from agent, creating functions started by "Agent_", like "Agent_Click" or "Agent_DragComplete"
          For more information, read Agent Platform SDK, there are many details about events 
    
    _MAGetRequestStatus()
    Returns a Request Status Code

    _MAWaitAction()
    Wait queued request ends
    
    _MAGetLanguageId()
    Gets Character Language Id
    
    _MAShowChar()
    Shows the character

    _MAHideChar()
    Hides the character
    
    _MAMoveChar()
    Move Character across the screen
    
    _MAMoveCharToWindowBorder()
    Move Character to a window border
    
    _MACharSpeak()
    Speaks the specified text for the character.
    
    _MACharThink()
    Displays the specified text for the character in a "thought" word balloon
    
    _MACharPlayAnim()
    Character does an indicated animation
    
    _MAListCharAnimations()
    Lists all characters animations to a vector
    
    _MAGestureAt()
    Plays the gesturing animation for the character at the specified location
    
    _MAGetCharSize()
    Get size of a character

    _MAGetCharPos()
    Get position of a character
    
    _MAStop()
    Stops the animation for the specified character
    
    _MAStopAll()
    Stops all animation requests
    
    _MAIsCharVisible()
    Return if char is visible
    
    _MAScriptPlay()
    Runs a script - very util when load it from file before
    
    _MAQuitChar()
    Quits the character
    
    >> Command Functions
    
    _MACommandAdd()
    Adds a command to a character
    
    _MACommandRemove()
    Removes a command from character

#ce
#endregion
#region Global Variables and Constants
Global $g_oAgentCtrl
Global $g_oSinkObject_MsAgent   ; <--- Handles incoming events from the agent control
Global $g_oSinkObject_Agent     ; <--/
#endregion
#region Core functions
;===============================================================================
;
; Function Name:    _MALoadCharacter()
; Description:      Create a Character object suitable for animating
; Parameter(s):     $AgentFileName  - string - Filename or FullPath of agent
;                   $AgentName      - string - Agent name
;                   $Language       - hex string - Language number in hex - "0409"; "0416"
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a Character object reference
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MALoadCharacter( $AgentFileName, $AgentName, $Language = "0409" )
    Local $RetVal = 0
    If Not IsObj($g_oAgentCtrl) Then $g_oAgentCtrl = ObjCreate("Agent.Control.2")
        
    If IsObj($g_oAgentCtrl) Then
        $g_oAgentCtrl.Connected = True
        
        ;Handles incoming events from the agent control
        $g_oSinkObject_MsAgent = ObjEvent( $g_oAgentCtrl, "MsAgent_" )
        $g_oSinkObject_Agent = ObjEvent( $g_oAgentCtrl, "Agent_" )
        
        Local $AgentPath = $AgentFileName
        If Not FileExists($AgentPath) Then $AgentPath = @WindowsDir & "\Msagent\Chars\" & $AgentFileName
        
        ;load character
        If FileExists( $AgentPath ) Then
            $g_oAgentCtrl.Characters.Load( $AgentName, $AgentFileName )
            Local $oCharacter = $g_oAgentCtrl.Characters( $AgentName )
            $oCharacter.Activate( True )
            
            ;Initial commands
            $oCharacter.Commands.RemoveAll
            $oCharacter.LanguageID = Dec( $Language )
            
            $RetVal = $oCharacter
        EndIf
    EndIf
    Return $RetVal
EndFunc   ;==>_MALoadCharacter

;===============================================================================
;
; Function Name:    _MAGetRequestStatus()
; Description:      Returns a Request Status Code
; Parameter(s):     $oRequest - object variable that points to a Request
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request status code:
;                                   0 - Request successfully completed
;                                   1 - Request failed
;                                   2 - Request pending (in the queue, but not complete)
;                                   3 - Request interrupted
;                                   4 - Request in progress
;                   On Failure - Returns -1
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAGetRequestStatus( ByRef $oRequest )
    Local $RetVal = -1
    If IsObj( $oRequest ) Then
        $RetVal = $oRequest.Status
    EndIf
    Return $RetVal
EndFunc   ;==>_MAGetRequestStatus

;===============================================================================
;
; Function Name:    _MAWaitAction()
; Description:      Wait queued request ends
; Parameter(s):     $oRequest - object variable that points to a Request
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request status code - see _MAGetRequestStatus
;                   On Failure - Returns -1
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAWaitAction( ByRef $oRequest )
    Local $RetVal = -1
    If IsObj( $oRequest ) Then
        While $oRequest.Status <> 0 and $oRequest.Status <> 1 and $oRequest.Status <> 3
            Sleep( 100 )
        Wend
        $RetVal = $oRequest.Status
    EndIf
    Return $RetVal
EndFunc   ;==>_MAWaitAction

;===============================================================================
;
; Function Name:    _MAGetLanguageId()
; Description:      Gets Character Language Id
; Parameter(s):     $oCharacter - object variable that points to a Character
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a Language Hex Code
;                   On Failure - Returns empty
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAGetLanguageId( ByRef $oCharacter )
    Local $RetVal = ""
    If IsObj( $oCharacter ) Then
        $RetVal = Hex( $oCharacter.LanguageID, 4 )
    EndIf
    Return $RetVal
EndFunc   ;==>_MAGetLanguageId

;===============================================================================
;
; Function Name:    _MAShowChar()
; Description:      Shows the character
; Parameter(s):     $oCharacter - object variable that points to a Character
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request object (related: _MAGetRequestStatus)
;                   On Failure or 'no action to do' - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAShowChar( ByRef $oCharacter )
    Local $RetVal = 0
    If IsObj( $oCharacter ) Then
        If _MAIsCharVisible( $oCharacter ) = 0 Then
            $RetVal = $oCharacter.Show
        EndIf
    EndIf
    Return $RetVal
EndFunc   ;==>_MAShowChar

;===============================================================================
;
; Function Name:    _MAHideChar()
; Description:      Hides the character
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $WaitQueued - integer -	0 - do not wait
;                                           1 - default: wait all queued actions end
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request object (related: _MAGetRequestStatus)
;                   On Failure or 'no action to do' - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAHideChar( ByRef $oCharacter, $WaitQueued = 1 )
    Local $RetVal = 0
    If IsObj( $oCharacter ) Then
        If _MAIsCharVisible( $oCharacter ) Then
            $RetVal = $oCharacter.Hide
            If $WaitQueued = 1 Then
                While $RetVal.Status > 0
                    Sleep( 100 )
                Wend
            EndIf
        EndIf
    EndIf
    Return $RetVal
EndFunc   ;==>_MAHideChar

;===============================================================================
;
; Function Name:    _MAMoveChar()
; Description:      Move Character across the screen
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $ScreenX    - integer - X position
;                   $ScreenY    - integer - Y Position
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request object (related: _MAGetRequestStatus)
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAMoveChar( ByRef $oCharacter, $ScreenX, $ScreenY )
    Local $RetVal = 0
    If IsObj( $oCharacter ) and $ScreenX >= 0 and $ScreenY >= 0 Then
        $RetVal = $oCharacter.MoveTo ( $ScreenX, $ScreenY )
    EndIf
    Return $RetVal
EndFunc   ;==>_MAMoveChar

;===============================================================================
;
; Function Name:    _MAMoveCharToWindowBorder()
; Description:      Move Character to a window border
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $Window     - string or hwnd - Destination Window
;                   $Pos        - integer - Position across X axis - in pixels - use '-1' to center (default)
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request object (related: _MAGetRequestStatus)
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAMoveCharToWindowBorder( ByRef $oCharacter, $Window, $Pos = -1 )
    Local $RetVal = 0
    If IsObj( $oCharacter ) Then
        If WinExists( $Window ) Then
            Local $WinPos = WinGetPos( $Window )
            Local $CharSize = _MAGetCharSize( $oCharacter )
            Local $MidChar = $CharSize[0] / 2
            If $WinPos[2] < $CharSize[0] Then $Pos = -1
            
            Local $ScreenX = 0
            If $Pos = -1 Then
                ;gets mid of window
                $ScreenX = $WinPos[0] + ( $WinPos[2] / 2 )
                ;now sub mid of window than mid width of character
                $ScreenX = $ScreenX - $MidChar
            Else
                If $Pos > ( $WinPos[2] - $MidChar ) Then $Pos = $WinPos[2] - $MidChar
                If $Pos < $MidChar Then $Pos = $MidChar
                $ScreenX = $WinPos[0] + $Pos
            EndIf
            Local $ScreenY = $WinPos[1] - $CharSize[1] + 10
            $RetVal = $oCharacter.MoveTo ( $ScreenX, $ScreenY )
        EndIf
    EndIf
    Return $RetVal
EndFunc   ;==>_MAMoveCharToWindowBorder

;===============================================================================
;
; Function Name:    _MACharSpeak()
; Description:      Speaks the specified text for the character.
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $Str2Speak 	- string - String to speak
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request object (related: _MAGetRequestStatus)
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MACharSpeak( ByRef $oCharacter, $Str2Speak )
    Local $RetVal = 0
    If IsObj( $oCharacter ) Then
        $RetVal = $oCharacter.Speak( $Str2Speak )
    EndIf
    Return $RetVal
EndFunc   ;==>_MACharSpeak

;===============================================================================
;
; Function Name:    _MACharThink()
; Description:      Displays the specified text for the character in a "thought" word balloon
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $Str2Think 	- string - String to think
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request object (related: _MAGetRequestStatus)
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MACharThink( ByRef $oCharacter, $Str2Think )
    Local $RetVal = 0
    If IsObj( $oCharacter ) Then
        $RetVal = $oCharacter.Think( $Str2Think )
    EndIf
    Return $RetVal
EndFunc   ;==>_MACharThink

;===============================================================================
;
; Function Name:    _MACharPlayAnim()
; Description:      Character does an indicated animation
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $Animation 	- string - String that names a valid animation
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request object (related: _MAGetRequestStatus)
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MACharPlayAnim( ByRef $oCharacter, $Animation )
    Local $RetVal = 0
    If IsObj( $oCharacter ) and StringLen( $Animation ) Then
        Local $Animations = _MAListCharAnimations( $oCharacter )
        Local $i = 1
        For $i = 1 to $Animations[0]
            If $Animation = $Animations[$i] Then
                $RetVal = $oCharacter.Play ( $Animations[$i] )
                ExitLoop
            EndIf
        Next
    EndIf
    Return $RetVal
EndFunc   ;==>_MACharPlayAnim

;===============================================================================
;
; Function Name:    _MAListCharAnimations()
; Description:      Lists all characters animations to a vector
; Parameter(s):     $oCharacter - object variable that points to a Character
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns array with all animations - Index[0] returns total occurrencies
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAListCharAnimations( ByRef $oCharacter )
    Local $RetVal[1]
    $RetVal[0] = 0
    If IsObj( $oCharacter ) Then
        For $AnimName in $oCharacter.AnimationNames
            Redim $RetVal[ Ubound( $RetVal ) + 1 ]
            $RetVal[0] = Ubound( $RetVal ) - 1
            $RetVal[$RetVal[0]] = $AnimName
        Next
    EndIf
    Return $RetVal
EndFunc   ;==>_MAListCharAnimations

;===============================================================================
;
; Function Name:    _MAGestureAt()
; Description:      Plays the gesturing animation for the character at the specified location
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $X - integer - horizontal screen coordinate
;                   $Y - integer - vertical screen coordinate
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a request object (related: _MAGetRequestStatus)
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAGestureAt( ByRef $oCharacter, $X, $Y )
    Local $RetVal = 0
    If IsObj( $oCharacter ) Then
        $RetVal = $oCharacter.GestureAt( $X, $Y )
    EndIf
    Return $RetVal
EndFunc   ;==>_MAGestureAt

;===============================================================================
;
; Function Name:    _MAGetCharSize()
; Description:      Get size of a character
; Parameter(s):     $oCharacter - object variable that points to a Character
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns array with sizes - Index[0] = X ; Index[1] = Y
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAGetCharSize( ByRef $oCharacter )
    Local $RetVal[2]
    $RetVal[0] = 0
    $RetVal[1] = 0
    If IsObj( $oCharacter ) Then
        $RetVal[0] = $oCharacter.OriginalWidth
        $RetVal[1] = $oCharacter.OriginalHeight
    EndIf
    Return $RetVal
EndFunc   ;==>_MAGetCharSize

;===============================================================================
;
; Function Name:    _MAGetCharPos()
; Description:      Get position of a character
; Parameter(s):     $oCharacter - object variable that points to a Character
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns array with pos - Index[0] = X ; Index[1] = Y
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAGetCharPos( ByRef $oCharacter )
    Local $RetVal[2]
    $RetVal[0] = 0
    $RetVal[1] = 0
    If IsObj( $oCharacter ) Then
        $RetVal[0] = $oCharacter.Left
        $RetVal[1] = $oCharacter.Top
    EndIf
    Return $RetVal
EndFunc   ;==>_MAGetCharPos

;===============================================================================
;
; Function Name:    _MAStop()
; Description:      Stops the animation for the specified character
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $Request    - optional: object variable that points to a request (stop especified request)
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success  - Returns 1
;                   On Failure  - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAStop( ByRef $oCharacter, $Request = 0 )
    Local $RetVal = 0
    If IsObj( $oCharacter ) Then
        If IsObj( $Request ) Then
            $oCharacter.Stop( $Request )
        Else
            $oCharacter.Stop
        EndIf
        $RetVal = 1
    EndIf
    Return $RetVal
EndFunc   ;==>_MAStop

;===============================================================================
;
; Function Name:    _MAStopAll()
; Description:      Stops all animation requests
; Parameter(s):     $oCharacter - object variable that points to a Character
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns 1
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAStopAll( ByRef $oCharacter )
    Local $RetVal = 0
    If IsObj( $oCharacter ) Then
        $oCharacter.StopAll
        $RetVal = 1
    EndIf
    Return $RetVal
EndFunc   ;==>_MAStopAll

;===============================================================================
;
; Function Name:    _MAIsCharVisible()
; Description:      Return if char is visible
; Parameter(s):     $oCharacter - object variable that points to a Character
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns 1
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAIsCharVisible( ByRef $oCharacter )
    Local $RetVal = 0
    If IsObj( $oCharacter ) Then
        $RetVal = $oCharacter.Visible
    EndIf
    Return $RetVal
EndFunc   ;==>_MAIsCharVisible

;===============================================================================
;
; Function Name:    _MAScriptPlay()
; Description:      Runs a script - very util when load it from file before
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $Script     - string - commands sequence
;                           How it does? Simple:
;                               Script = "play;Pleased|speak;Hello!|play;RestPose|sleep;1000|speak;I'm a character!"
;                               Put commands separated by a "|"
;                               The commands are: play, speak, think, sleep
;                               Commands and their actions are separated by a ";" like this: "play;Pleased"
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns 1
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAScriptPlay( $oCharacter, $Script )
    
    ;Start the monologue
    Local $Lines = StringSplit( $Script, "|" )
    
    If UBound( $Lines ) Then
        Local $CharCmd = ""
        Local $oRequest
		Local $i
        For $i = 1 to $Lines[0]
            $CharCmd = StringSplit( $Lines[$i], ";" )
            Select
                case $CharCmd[1] = "play"
                    $oRequest = _MACharPlayAnim( $oCharacter, $CharCmd[2] )
                case $CharCmd[1] = "speak"
                    $oRequest = $oCharacter.Speak ( $CharCmd[2] )
                case $CharCmd[1] = "think"
                    $oRequest = $oCharacter.Think ( $CharCmd[2] )
                case $CharCmd[1] = "sleep"
                    If Number( $CharCmd[2] ) Then
                        _MAWaitAction( $oRequest )
                        Sleep( $CharCmd[2] )
                    EndIf
            EndSelect
        Next
        _MAWaitAction( $oRequest )
    EndIf
    
EndFunc   ;==>_MAScriptPlay

;===============================================================================
;
; Function Name:    _MAQuitChar()
; Description:      Quits the character
; Parameter(s):     $oCharacter - object variable that points to a Character
;                   $WaitQueued - integer - 0 - do not wait
;                                           1 - default: wait all queued actions end
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns 1
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MAQuitChar( ByRef $oCharacter, $WaitQueued = 1 )
	Local $RetVal = 0
	If IsObj( $oCharacter ) Then
		If $WaitQueued = 0 Then
			_MAStopAll( $oCharacter )
		EndIf
		_MAHideChar( $oCharacter, $WaitQueued )
        $g_oAgentCtrl.Characters.Unload( $oCharacter.Name )
		$oCharacter = 0
		$RetVal = 1
	EndIf
	Return $RetVal
EndFunc   ;==>_MAQuitChar

;===============================================================================
;
; Function Name:    _MACommandAdd()
; Description:      Adds a command to a character
; Parameter(s):     $oCharacter     - object variable that points to a Character
;                   $CmdIdentifier  - string - the ID you assign for the command
;                   $CmdCaption     - string - the name that will appear in the character pop-up menu
;                   $Voice          - string - the words or phrase to be used by the speech engine for recognizing this command
;                   $Enabled        - boolean - indicating whether the command is enabled
;                   $Visible        - boolean - indicating whether the command is visible
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  On Success - Returns a Command object reference
;                   On Failure - Returns 0
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MACommandAdd( $oCharacter, $CmdIdentifier, $CmdCaption, $Voice = "", $Enabled = 1, $Visible = 1 )
    Local $RetVal = 0
    Local $CmdId = StringStripWS( $CmdIdentifier, 8 )
    Local $Command = $oCharacter.Commands.Add ( $CmdId, $CmdCaption, $Voice, $Enabled, $Visible )
    If IsObj( $Command ) Then $RetVal = $Command
    Return $RetVal
EndFunc   ;==>_MACommandAdd

;===============================================================================
;
; Function Name:    _MACommandRemove()
; Description:      Removes a command from character
; Parameter(s):     $oCharacter     - object variable that points to a Character
;                   $CmdIdentifier  - string - the ID of command
; Requirement(s):   AutoIt with COM support (post 3.1.1)
; Return Value(s):  none
; Author(s):        Alexsandro Percy
;
;===============================================================================
Func _MACommandRemove( $oCharacter, $CmdIdentifier )
    Local $CmdId = StringStripWS( $CmdIdentifier, 8 )
    $oCharacter.Commands.Remove ( $CmdId )
EndFunc   ;==>_MACommandRemove

#endregion
