#include-once
#region Header
#comments-start
	Title:			Computer Information Automation UDF Library for AutoIt3
	Filename:		CompInfo.au3
	Description:	A collection of UDF's that assist in gathering and setting information about a computer (Software and Hardware).
	Version:		00.03.08
	Last Update:	11.09.06
	Requirements:	AutoIt v3.2 +, Developed/Tested on WindowsXP Pro Service Pack 2
	Notes:			Errors associated with incorrect objects will be common user errors. AutoIt beta 3.1.1.63 has added an ObjName()
	function that will be used to trap and report most of these errors.
	
	Special thanks to JdeB (WMI Introduction), SvenP (Scriptomatic), Firestorm (testing, ideas), gafrost (WMI Support),
	piccaso (COM Errors), RagnaroktA (testing, ideas, bug fixes), SmOke_N (testing), and everyone else that has helped in
	the creation of this UDF.
#comments-end
#endregion Header
Global $avRET, $iPID
#region Global Variables and Constants
If Not(IsDeclared("$cI_CompName")) Then
	Global	$cI_CompName = @ComputerName
EndIf
Global Const $cI_VersionInfo		= "00.03.08"
Global Const $cI_aName				= 0, _
			 $cI_aDesc				= 4
Global	$wbemFlagReturnImmediately	= 0x10, _	;DO NOT CHANGE
$wbemFlagForwardOnly		= 0x20				;DO NOT CHANGE
Global	$ERR_NO_INFO				= "Array contains no information", _
		$ERR_NOT_OBJ				= "$colItems isnt an object"
#endregion Global Variables and Constants

#region Software Functions


;===============================================================================
; Description:      Returns the drive information based on $sDriveType in a two
;					dimensional array. First dimension is the index for each set
;					of drive information.
; Parameter(s):     $aDriveInfo - By Reference - Drive information in an array.
;					$sDriveType - 	Type of drive to return the information on.
;									Options: "ALL", "CDROM", "REMOVABLE", "FIXED",
;									"NETWORK", "RAMDISK", or "UNKNOWN"
;									Defaults to "FIXED" drives.
; Requirement(s):   None
; Return Value(s):  On Success - Returns array of drive information.
;						$aDriveInfo[0][0] = Number of Drives
;						The second dimension is as follows: ($i starts at 1)
;							[$i][0] - Drive Letter (ex. C:\)
;							[$i][1] - File System
;							[$i][2] - Label
;							[$i][3] - Serial Number
;							[$i][4] - Free Space
;							[$i][5] - Total Space
;                   On Failure - Return 0 - @error - 1
;								@extended:	 1 = DriveGetDrive		Error
;											 2 = DriveGetFileSystem Error
;											 3 = DriveGetLabel		Error
;											 4 = DriveGetSerial		Error
;											 5 = DriveSpaceFree		Error
;											 6 = DriveSpaceTotal	Error
; Note(s):
;===============================================================================
Func _ComputerGetDrives(ByRef $aDriveInfo, $sDriveType = "FIXED")
	Local $drive
	$drive = DriveGetDrive($sDriveType)
	If NOT @error Then
		Dim $aDriveInfo[UBound($drive)][6]
		$aDriveInfo[0][0] = $drive[0]
		For $i = 1 To $aDriveInfo[0][0] Step 1
			$aDriveInfo[$i][0] = StringUpper($drive[$i] & "\")
			$aDriveInfo[$i][1] = DriveGetFileSystem($drive[$i])
			If @error Then SetError(1, 2, 0)
			$aDriveInfo[$i][2] = DriveGetLabel($drive[$i])
			If @error Then SetError(1, 3, 0)
			$aDriveInfo[$i][3] = DriveGetSerial($drive[$i])
			If @error Then SetError(1, 4, 0)
			$aDriveInfo[$i][4] = DriveSpaceFree($drive[$i])
			If @error Then SetError(1, 5, 0)
			$aDriveInfo[$i][5] = DriveSpaceTotal($drive[$i])
			If @error Then SetError(1, 6, 0)
		Next
	Else
		SetError(1, 1, 0)
	EndIf
EndFunc ;_ComputerGetDrives
;===============================================================================
; Function Name:    _ProcessListProperties()
; Description:   Get various properties of a process, or all processes
; Call With:       _ProcessListProperties( [$Process [, $sComputer]] )
; Parameter(s):     (optional) $Process - PID or name of a process, default is "" (all)
;           (optional) $sComputer - remote computer to get list from, default is local
; Requirement(s):   AutoIt v3.2.4.9+
; Return Value(s):  On Success - Returns a 2D array of processes, as in ProcessList()
;             with additional columns added:
;             [0][0] - Number of processes listed (can be 0 if no matches found)
;             [1][0] - 1st process name
;             [1][1] - 1st process PID
;             [1][2] - 1st process Parent PID
;             [1][3] - 1st process owner
;             [1][4] - 1st process priority (0 = low, 31 = high)
;             [1][5] - 1st process executable path
;             [1][6] - 1st process CPU usage
;             [1][7] - 1st process memory usage
;             [1][8] - 1st process creation date/time = "MM/DD/YYY hh:mm:ss" (hh = 00 to 23)
;             [1][9] - 1st process command line string
;             ...
;             [n][0] thru [n][9] - last process properties
; On Failure:       Returns array with [0][0] = 0 and sets @Error to non-zero (see code below)

; Notes:            If an integer PID or string process name is provided and no match is found,
;             then [0][0] = 0 and @error = 0 (not treated as an error, same as ProcessList)
;           This function requires admin permissions to the target computer.
;           All properties come from the Win32_Process class in WMI.
;             To get time-base properties (CPU and Memory usage), a 100ms SWbemRefresher is used.
;===============================================================================
Func _ProcessListProperties($Process = "", $sComputer = ".")
    Local $sUserName, $sMsg, $sUserDomain, $avProcs, $dtmDate
    Local $avProcs[1][2] = [[0, ""]], $n = 1

    ; Convert PID if passed as string
    If StringIsInt($Process) Then $Process = Int($Process)

    ; Connect to WMI and get process objects
    $oWMI = ObjGet("winmgmts:{impersonationLevel=impersonate,authenticationLevel=pktPrivacy, (Debug)}!\\" & $sComputer & "\root\cimv2")
    If IsObj($oWMI) Then
        ; Get collection processes from Win32_Process
        If $Process = "" Then
            ; Get all
            $colProcs = $oWMI.ExecQuery("select * from win32_process")
        ElseIf IsInt($Process) Then
            ; Get by PID
            $colProcs = $oWMI.ExecQuery("select * from win32_process where ProcessId = " & $Process)
        Else
            ; Get by Name
            $colProcs = $oWMI.ExecQuery("select * from win32_process where Name = '" & $Process & "'")
        EndIf

        If IsObj($colProcs) Then
            ; Return for no matches
            If $colProcs.count = 0 Then Return $avProcs

            ; Size the array
            ReDim $avProcs[$colProcs.count + 1][10]
            $avProcs[0][0] = UBound($avProcs) - 1

            ; For each process...
            For $oProc In $colProcs
                ; [n][0] = Process name
                $avProcs[$n][0] = $oProc.name
                ; [n][1] = Process PID
                $avProcs[$n][1] = $oProc.ProcessId
                ; [n][2] = Parent PID
                $avProcs[$n][2] = $oProc.ParentProcessId
                ; [n][3] = Owner
                If $oProc.GetOwner($sUserName, $sUserDomain) = 0 Then $avProcs[$n][3] = $sUserDomain & "\" & $sUserName
                ; [n][4] = Priority
                $avProcs[$n][4] = $oProc.Priority
                ; [n][5] = Executable path
                $avProcs[$n][5] = $oProc.ExecutablePath
                ; [n][8] = Creation date/time
                $dtmDate = $oProc.CreationDate
                If $dtmDate <> "" Then
                    ; Back referencing RegExp pattern from weaponx
                    Local $sRegExpPatt = "\A(\d{4})(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})(?:.*)"
                    $dtmDate = StringRegExpReplace($dtmDate, $sRegExpPatt, "$2/$3/$1 $4:$5:$6")
                EndIf
                $avProcs[$n][8] = $dtmDate
                ; [n][9] = Command line string
                $avProcs[$n][9] = $oProc.CommandLine

                ; increment index
                $n += 1
            Next
        Else
            SetError(2); Error getting process collection from WMI
        EndIf
        ; release the collection object
        $colProcs = 0

        ; Get collection of all processes from Win32_PerfFormattedData_PerfProc_Process
        ; Have to use an SWbemRefresher to pull the collection, or all Perf data will be zeros
        Local $oRefresher = ObjCreate("WbemScripting.SWbemRefresher")
        $colProcs = $oRefresher.AddEnum($oWMI, "Win32_PerfFormattedData_PerfProc_Process" ).objectSet
        $oRefresher.Refresh

        ; Time delay before calling refresher
        Local $iTime = TimerInit()
        Do
            Sleep(20)
        Until TimerDiff($iTime) >= 100
        $oRefresher.Refresh

        ; Get PerfProc data
        For $oProc In $colProcs
            ; Find it in the array
            For $n = 1 To $avProcs[0][0]
                If $avProcs[$n][1] = $oProc.IDProcess Then
                    ; [n][6] = CPU usage
                    $avProcs[$n][6] = $oProc.PercentProcessorTime
                    ; [n][7] = memory usage
                    $avProcs[$n][7] = $oProc.WorkingSet
                    ExitLoop
                EndIf
            Next
        Next
    Else
        SetError(1); Error connecting to WMI
    EndIf

    ; Return array
    Return $avProcs
EndFunc  ;==>_ProcessListProperties


;===============================================================================
; Description:      Returns the OS information in an array.
; Parameter(s):     $aOSInfo - By Reference - OS Information array.
; Requirement(s):   None
; Return Value(s):  On Success - Returns array of OS Information.
;						$aOSInfo[$i][0]  = Name
;						$aOSInfo[$i][1]  = Boot Device
;						$aOSInfo[$i][2]  = Build Number
;						$aOSInfo[$i][3]  = Build Type
;						$aOSInfo[$i][4]  = Description
;						$aOSInfo[$i][5]  = CodeSet
;						$aOSInfo[$i][6]  = Country Code
;						$aOSInfo[$i][7]  = Creation Class Name
;						$aOSInfo[$i][8]  = CS Creation Class Name
;						$aOSInfo[$i][9]  = CSD Version
;						$aOSInfo[$i][10] = CS Name
;						$aOSInfo[$i][11] = Current Time Zone
;						$aOSInfo[$i][12] = DataExecutionPrevention_32BitApplications
;						$aOSInfo[$i][13] = DataExecutionPrevention_Available
;						$aOSInfo[$i][14] = DataExecutionPrevention_Drivers
;						$aOSInfo[$i][15] = DataExecutionPrevention_SupportPolicy
;						$aOSInfo[$i][16] = Debug
;						$aOSInfo[$i][17] = Distributed
;						$aOSInfo[$i][18] = Encryption Level
;						$aOSInfo[$i][19] = Foreground Application Boost
;						$aOSInfo[$i][20] = Free Physical Memory
;						$aOSInfo[$i][21] = Free Space In Paging Files
;						$aOSInfo[$i][22] = Free Virtual Memory
;						$aOSInfo[$i][23] = Install Date
;						$aOSInfo[$i][24] = Large System Cache
;						$aOSInfo[$i][25] = Last Boot Up Time
;						$aOSInfo[$i][26] = Local Date Time
;						$aOSInfo[$i][27] = Locale
;						$aOSInfo[$i][28] = Manufacturer
;						$aOSInfo[$i][29] = Max Number Of Processes
;						$aOSInfo[$i][30] = Max Process Memory Size
;						$aOSInfo[$i][31] = Number Of Licensed Users
;						$aOSInfo[$i][32] = Number Of Processes
;						$aOSInfo[$i][33] = Number Of Users
;						$aOSInfo[$i][34] = Organization
;						$aOSInfo[$i][35] = OS Language
;						$aOSInfo[$i][36] = OS Product Suite
;						$aOSInfo[$i][37] = OS Type
;						$aOSInfo[$i][38] = Other Type Description
;						$aOSInfo[$i][39] = Plus Product ID
;						$aOSInfo[$i][40] = Plus Version Number
;						$aOSInfo[$i][41] = Primary
;						$aOSInfo[$i][42] = Product Type
;						$aOSInfo[$i][43] = Quantum Length
;						$aOSInfo[$i][44] = Quantum Type
;						$aOSInfo[$i][45] = Registered User
;						$aOSInfo[$i][46] = Serial Number
;						$aOSInfo[$i][47] = Service Pack Major Version
;						$aOSInfo[$i][48] = Service Pack Minor Version
;						$aOSInfo[$i][49] = Size Stored In Paging Files
;						$aOSInfo[$i][50] = Status
;						$aOSInfo[$i][51] = Suite Mask
;						$aOSInfo[$i][52] = System Device
;						$aOSInfo[$i][53] = System Directory
;						$aOSInfo[$i][54] = System Drive
;						$aOSInfo[$i][55] = Total Swap Space Size
;						$aOSInfo[$i][56] = Total Virtual Memory Size
;						$aOSInfo[$i][57] = Total Visible Memory Size
;						$aOSInfo[$i][58] = Version
;						$aOSInfo[$i][59] = Windows Directory
;                   On Failure - @error = 1 and Returns 0
;								@extended = 1 - Array contains no information
;											2 - $colItems isnt an object
; Note(s):
;===============================================================================
Func _ComputerGetOSs(ByRef $aOSInfo)
	Local $colItems, $objWMIService, $objItem
	Dim $aOSInfo[1][60], $i = 1
	
	$objWMIService = ObjGet("winmgmts:\\" & $cI_Compname & "\root\CIMV2")
	$colItems = $objWMIService.ExecQuery("SELECT * FROM Win32_OperatingSystem", "WQL", $wbemFlagReturnImmediately + $wbemFlagForwardOnly)
	
	If IsObj($colItems) Then
		For $objItem In $colItems
			ReDim $aOSInfo[UBound($aOSInfo) + 1][60]
			$aOSInfo[$i][0]  = $objItem.Name
			$aOSInfo[$i][1]  = $objItem.BootDevice
			$aOSInfo[$i][2]  = $objItem.BuildNumber
			$aOSInfo[$i][3]  = $objItem.BuildType
			$aOSInfo[$i][4]  = $objItem.Description
			$aOSInfo[$i][5]  = $objItem.CodeSet
			$aOSInfo[$i][6]  = $objItem.CountryCode
			$aOSInfo[$i][7]  = $objItem.CreationClassName
			$aOSInfo[$i][8]  = $objItem.CSCreationClassName
			$aOSInfo[$i][9]  = $objItem.CSDVersion
			$aOSInfo[$i][10] = $objItem.CSName
			$aOSInfo[$i][11] = $objItem.CurrentTimeZone
			$aOSInfo[$i][12] = $objItem.DataExecutionPrevention_32BitApplications
			$aOSInfo[$i][13] = $objItem.DataExecutionPrevention_Available
			$aOSInfo[$i][14] = $objItem.DataExecutionPrevention_Drivers
			$aOSInfo[$i][15] = $objItem.DataExecutionPrevention_SupportPolicy
			$aOSInfo[$i][16] = $objItem.Debug
			$aOSInfo[$i][17] = $objItem.Distributed
			$aOSInfo[$i][18] = $objItem.EncryptionLevel
			$aOSInfo[$i][19] = $objItem.ForegroundApplicationBoost
			$aOSInfo[$i][20] = $objItem.FreePhysicalMemory
			$aOSInfo[$i][21] = $objItem.FreeSpaceInPagingFiles
			$aOSInfo[$i][22] = $objItem.FreeVirtualMemory
			$aOSInfo[$i][23] = __StringToDate($objItem.InstallDate)
			$aOSInfo[$i][24] = $objItem.LargeSystemCache
			$aOSInfo[$i][25] = __StringToDate($objItem.LastBootUpTime)
			$aOSInfo[$i][26] = __StringToDate($objItem.LocalDateTime)
			$aOSInfo[$i][27] = $objItem.Locale
			$aOSInfo[$i][28] = $objItem.Manufacturer
			$aOSInfo[$i][29] = $objItem.MaxNumberOfProcesses
			$aOSInfo[$i][30] = $objItem.MaxProcessMemorySize
			$aOSInfo[$i][31] = $objItem.NumberOfLicensedUsers
			$aOSInfo[$i][32] = $objItem.NumberOfProcesses
			$aOSInfo[$i][33] = $objItem.NumberOfUsers
			$aOSInfo[$i][34] = $objItem.Organization
			$aOSInfo[$i][35] = $objItem.OSLanguage
			$aOSInfo[$i][36] = $objItem.OSProductSuite
			$aOSInfo[$i][37] = $objItem.OSType
			$aOSInfo[$i][38] = $objItem.OtherTypeDescription
			$aOSInfo[$i][39] = $objItem.PlusProductID
			$aOSInfo[$i][40] = $objItem.PlusVersionNumber
			$aOSInfo[$i][41] = $objItem.Primary
			$aOSInfo[$i][42] = $objItem.ProductType
			$aOSInfo[$i][43] = $objItem.QuantumLength
			$aOSInfo[$i][44] = $objItem.QuantumType
			$aOSInfo[$i][45] = $objItem.RegisteredUser
			$aOSInfo[$i][46] = $objItem.SerialNumber
			$aOSInfo[$i][47] = $objItem.ServicePackMajorVersion
			$aOSInfo[$i][48] = $objItem.ServicePackMinorVersion
			$aOSInfo[$i][49] = $objItem.SizeStoredInPagingFiles
			$aOSInfo[$i][50] = $objItem.Status
			$aOSInfo[$i][51] = $objItem.SuiteMask
			$aOSInfo[$i][52] = $objItem.SystemDevice
			$aOSInfo[$i][53] = $objItem.SystemDirectory
			$aOSInfo[$i][54] = $objItem.SystemDrive
			$aOSInfo[$i][55] = $objItem.TotalSwapSpaceSize
			$aOSInfo[$i][56] = $objItem.TotalVirtualMemorySize
			$aOSInfo[$i][57] = $objItem.TotalVisibleMemorySize
			$aOSInfo[$i][58] = $objItem.Version
			$aOSInfo[$i][59] = $objItem.WindowsDirectory
			$i += 1
		Next
		$aOSInfo[0][0] = UBound($aOSInfo) - 1
		If $aOSInfo[0][0] < 1 Then
			SetError(1, 1, 0)
		EndIf
	Else
		SetError(1, 2, 0)
	EndIf
EndFunc ;_ComputerGetOSs


;===============================================================================
; Description:      Returns the Processor information in an array.
; Parameter(s):     $aProcessorInfo - By Reference - Processor Information array.
; Requirement(s):   None
; Return Value(s):  On Success - Returns array of Processor Information.
;						$aProcessorInfo[0][0]   = Number of Processors
;						$aProcessorInfo[$i][0]  = Name ($i starts at 1)
;						$aProcessorInfo[$i][1]  = Address Width
;						$aProcessorInfo[$i][2]  = Architecture
;						$aProcessorInfo[$i][3]  = Availability
;						$aProcessorInfo[$i][4]  = Description
;						$aProcessorInfo[$i][5]  = Config Manager Error Code
;						$aProcessorInfo[$i][6]  = Config Manager User Config
;						$aProcessorInfo[$i][7]  = CPU Status
;						$aProcessorInfo[$i][8]  = Creation Class Name
;						$aProcessorInfo[$i][9]  = Current Clock Speed
;						$aProcessorInfo[$i][10] = Current Voltage
;						$aProcessorInfo[$i][11] = Data Width
;						$aProcessorInfo[$i][12] = Device ID
;						$aProcessorInfo[$i][13] = Error Cleared
;						$aProcessorInfo[$i][14] = Error Description
;						$aProcessorInfo[$i][15] = Ext Clock
;						$aProcessorInfo[$i][16] = Family
;						$aProcessorInfo[$i][17] = L2 Cache Size
;						$aProcessorInfo[$i][18] = L2 Cache Speed
;						$aProcessorInfo[$i][19] = Last Error Code
;						$aProcessorInfo[$i][20] = Level
;						$aProcessorInfo[$i][21] = Load Percentage
;						$aProcessorInfo[$i][22] = Manufacturer
;						$aProcessorInfo[$i][23] = Max Clock Speed
;						$aProcessorInfo[$i][24] = Other Family Description
;						$aProcessorInfo[$i][25] = PNP Device ID
;						$aProcessorInfo[$i][26] = Power Management Capabilities
;						$aProcessorInfo[$i][27] = Power Management Supported
;						$aProcessorInfo[$i][28] = Processor ID
;						$aProcessorInfo[$i][29] = Processor Type
;						$aProcessorInfo[$i][30] = Revision
;						$aProcessorInfo[$i][31] = Role
;						$aProcessorInfo[$i][32] = Socket Designation
;						$aProcessorInfo[$i][33] = Status
;						$aProcessorInfo[$i][34] = Status Info
;						$aProcessorInfo[$i][35] = Stepping
;						$aProcessorInfo[$i][36] = System Creation Class Name
;						$aProcessorInfo[$i][37] = System Name
;						$aProcessorInfo[$i][38] = Unique ID
;						$aProcessorInfo[$i][39] = Upgrade Method
;						$aProcessorInfo[$i][40] = Version
;						$aProcessorInfo[$i][41] = Voltage Caps
;                   On Failure - @error = 1 and Returns 0
;								@extended = 1 - Array contains no information
;											2 - $colItems isnt an object

; Note(s):
;===============================================================================
Func _ComputerGetProcessors(ByRef $aProcessorInfo)
	Local $colItems, $objWMIService, $objItem
	Dim $aProcessorInfo[1][42], $i = 1
	
	$objWMIService = ObjGet("winmgmts:\\" & $cI_Compname & "\root\CIMV2")
	$colItems = $objWMIService.ExecQuery("SELECT * FROM Win32_Processor", "WQL", $wbemFlagReturnImmediately + $wbemFlagForwardOnly)
	
	If IsObj($colItems) Then
		For $objItem In $colItems
			ReDim $aProcessorInfo[UBound($aProcessorInfo) + 1][42]
			$aProcessorInfo[$i][0]  = StringStripWS($objItem.Name, 1)
			$aProcessorInfo[$i][1]  = $objItem.AddressWidth
			$aProcessorInfo[$i][2]  = $objItem.Architecture
			$aProcessorInfo[$i][3]  = $objItem.Availability
			$aProcessorInfo[$i][4]  = $objItem.Description
			$aProcessorInfo[$i][5]  = $objItem.ConfigManagerErrorCode
			$aProcessorInfo[$i][6]  = $objItem.ConfigManagerUserConfig
			$aProcessorInfo[$i][7]  = $objItem.CpuStatus
			$aProcessorInfo[$i][8]  = $objItem.CreationClassName
			$aProcessorInfo[$i][9]  = $objItem.CurrentClockSpeed
			$aProcessorInfo[$i][10] = $objItem.CurrentVoltage
			$aProcessorInfo[$i][11] = $objItem.DataWidth
			$aProcessorInfo[$i][12] = $objItem.DeviceID
			$aProcessorInfo[$i][13] = $objItem.ErrorCleared
			$aProcessorInfo[$i][14] = $objItem.ErrorDescription
			$aProcessorInfo[$i][15] = $objItem.ExtClock
			$aProcessorInfo[$i][16] = $objItem.Family
			$aProcessorInfo[$i][17] = $objItem.L2CacheSize
			$aProcessorInfo[$i][18] = $objItem.L2CacheSpeed
			$aProcessorInfo[$i][19] = $objItem.LastErrorCode
			$aProcessorInfo[$i][20] = $objItem.Level
			$aProcessorInfo[$i][21] = $objItem.LoadPercentage
			$aProcessorInfo[$i][22] = $objItem.Manufacturer
			$aProcessorInfo[$i][23] = $objItem.MaxClockSpeed
			$aProcessorInfo[$i][24] = $objItem.OtherFamilyDescription
			$aProcessorInfo[$i][25] = $objItem.PNPDeviceID
			$aProcessorInfo[$i][26] = $objItem.PowerManagementCapabilities(0)
			$aProcessorInfo[$i][27] = $objItem.PowerManagementSupported
			$aProcessorInfo[$i][28] = $objItem.ProcessorId
			$aProcessorInfo[$i][29] = $objItem.ProcessorType
			$aProcessorInfo[$i][30] = $objItem.Revision
			$aProcessorInfo[$i][31] = $objItem.Role
			$aProcessorInfo[$i][32] = $objItem.SocketDesignation
			$aProcessorInfo[$i][33] = $objItem.Status
			$aProcessorInfo[$i][34] = $objItem.StatusInfo
			$aProcessorInfo[$i][35] = $objItem.Stepping
			$aProcessorInfo[$i][36] = $objItem.SystemCreationClassName
			$aProcessorInfo[$i][37] = $objItem.SystemName
			$aProcessorInfo[$i][38] = $objItem.UniqueId
			$aProcessorInfo[$i][39] = $objItem.UpgradeMethod
			$aProcessorInfo[$i][40] = $objItem.Version
			$aProcessorInfo[$i][41] = $objItem.VoltageCaps
			$i += 1
		Next
		$aProcessorInfo[0][0] = UBound($aProcessorInfo) - 1
		If $aProcessorInfo[0][0] < 1 Then
			SetError(1, 1, 0)
		EndIf
	Else
		SetError(1, 2, 0)
	EndIf
EndFunc ;_ComputerGetProcessors

;===============================================================================
; Description:      Returns the services information in an array.
; Parameter(s):     $aServicesInfo - By Reference - Services Information array.
;					$sState - OPTIONAL - Accepted values 'All' or 'Stopped' or
;										'Running'
; Requirement(s):   None
; Return Value(s):  On Success - Returns array of Services Information.
;						$aServicesInfo[0][0]   = Number of Services
;						$aServicesInfo[$i][0]  = Name ($i starts at 1)
;						$aServicesInfo[$i][1]  = Accept Pause
;						$aServicesInfo[$i][2]  = Accept Stop
;						$aServicesInfo[$i][3]  = Check Point
;						$aServicesInfo[$i][4]  = Description
;						$aServicesInfo[$i][5]  = Creation Class Name
;						$aServicesInfo[$i][6]  = Desktop Interact
;						$aServicesInfo[$i][7]  = Display Name
;						$aServicesInfo[$i][8]  = Error Control
;						$aServicesInfo[$i][9]  = Exit Code
;						$aServicesInfo[$i][10] = Path Name
;						$aServicesInfo[$i][11] = Process ID
;						$aServicesInfo[$i][12] = Service Specific Exit Code
;						$aServicesInfo[$i][13] = Service Type
;						$aServicesInfo[$i][14] = Started
;						$aServicesInfo[$i][15] = Start Mode
;						$aServicesInfo[$i][16] = Start Name
;						$aServicesInfo[$i][17] = State
;						$aServicesInfo[$i][18] = Status
;						$aServicesInfo[$i][19] = System Creation Class Name
;						$aServicesInfo[$i][20] = System Name
;						$aServicesInfo[$i][21] = Tag ID
;						$aServicesInfo[$i][22] = Wait Hint
;
;                   On Failure - @error = 1 and Returns 0
;								@extended = 1 - Array contains no information
;											2 - $colItems isnt an object
; Note(s):
;===============================================================================
Func _ComputerGetServices(ByRef $aServicesInfo, $sState = "All")
	Local $cI_Compname = @ComputerName, $wbemFlagReturnImmediately = 0x10, $wbemFlagForwardOnly = 0x20
	Local $colItems, $objWMIService, $objItem
	Dim $aServicesInfo[1][23], $i = 1
	
	$objWMIService = ObjGet("winmgmts:\\" & $cI_Compname & "\root\CIMV2")
	$colItems = $objWMIService.ExecQuery("SELECT * FROM Win32_Service", "WQL", $wbemFlagReturnImmediately + $wbemFlagForwardOnly)
	
	If IsObj($colItems) Then
		For $objItem In $colItems
			If $sState <> "All" Then
				If $sState = "Stopped" AND $objItem.State <> "Stopped" Then ContinueLoop
				If $sState = "Running" AND $objItem.State <> "Running" Then ContinueLoop
			EndIf
			ReDim $aServicesInfo[UBound($aServicesInfo) + 1][23]
			$aServicesInfo[$i][0]  = $objItem.Name
			$aServicesInfo[$i][1]  = $objItem.AcceptPause
			$aServicesInfo[$i][2]  = $objItem.AcceptStop
			$aServicesInfo[$i][3]  = $objItem.CheckPoint
			$aServicesInfo[$i][4]  = $objItem.Description
			$aServicesInfo[$i][5]  = $objItem.CreationClassName
			$aServicesInfo[$i][6]  = $objItem.DesktopInteract
			$aServicesInfo[$i][7]  = $objItem.DisplayName
			$aServicesInfo[$i][8]  = $objItem.ErrorControl
			$aServicesInfo[$i][9]  = $objItem.ExitCode
			$aServicesInfo[$i][10] = $objItem.PathName
			$aServicesInfo[$i][11] = $objItem.ProcessId
			$aServicesInfo[$i][12] = $objItem.ServiceSpecificExitCode
			$aServicesInfo[$i][13] = $objItem.ServiceType
			$aServicesInfo[$i][14] = $objItem.Started
			$aServicesInfo[$i][15] = $objItem.StartMode
			$aServicesInfo[$i][16] = $objItem.StartName
			$aServicesInfo[$i][17] = $objItem.State
			$aServicesInfo[$i][18] = $objItem.Status
			$aServicesInfo[$i][19] = $objItem.SystemCreationClassName
			$aServicesInfo[$i][20] = $objItem.SystemName
			$aServicesInfo[$i][21] = $objItem.TagId
			$aServicesInfo[$i][22] = $objItem.WaitHint
			$i += 1
		Next
		$aServicesInfo[0][0] = UBound($aServicesInfo) - 1
		If $aServicesInfo[0][0] < 1 Then
			SetError(1, 1, 0)
		EndIf
	Else
		SetError(1, 2, 0)
	EndIf
EndFunc ;_ComputerGetServices






#region Internal Functions
; **********************************************************************************
; Internal Functions with names starting with two underscores will not be documented
; as user functions.
; **********************************************************************************
Func __StringVersion()
	Return $cI_VersionInfo
EndFunc ;_StringVersion

Func __StringToDate($dtmDate)
	Return (StringMid($dtmDate, 5, 2) & "/" & _
			StringMid($dtmDate, 7, 2) & "/" & StringLeft($dtmDate, 4) _
			& " " & StringMid($dtmDate, 9, 2) & ":" & StringMid($dtmDate, 11, 2) & ":" & StringMid($dtmDate,13, 2))
EndFunc ;__StringToDate Function created by SvenP Modified by JSThePatriot
#endregion Internal Functions