; #EXAMPLE# =====================================================================================================================
MsgBox(0, "PathFindOnPath Example", _
	StringFormat(	"Full path of notepad.exe:\n%s\n\n" & _
					"Find ntuser.dat in profile folder, using custom paths:\n%s", _
					_WinAPI_PathFindOnPath("notepad.exe"), _WinAPI_PathFindOnPath("ntuser.dat", @UserProfileDir) _
))
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name...........: _WinAPI_PathFindOnPath
; Description ...: Searchs for a file in the default system paths
; Syntax.........: _WinAPI_PathFindOnPath($szFile, $aExtraPaths="", $szPathDelimiter=@LF)
; Parameters ....: $szFile          - Filename to search for
;                  $aExtraPaths     - Extra paths to check before any others.
;                  $szPathDelimiter - Delimiter used to split $aExtraPaths if it's an non-empty string (StringSplit with flag 2).
; Return values .: Success      - Full path of found file
;                  Failure      - Unchanged filename, @error=1
; Author ........: Daniel Miranda (danielkza)
; Modified.......:
; Remarks .......: $aExtraPaths can contain a list of paths to be checked before any system defaults.
;                  It can be an array or a string. If the former, it shall not have a count in it's first element.
;                  If the latter, it will be split using $szPathDelimiter as the delimiter, that defaults to @LF.
; Related .......:
; Link ..........; @@MsdnLink@@ PathFindOnpath
; Example .......; No
; ===============================================================================================================================

Func _WinAPI_PathFindOnPath(Const $szFile, $aExtraPaths = "", Const $szPathDelimiter=@LF)
	Local $iExtraCount = 0
	If IsString($aExtraPaths) Then
		If StringLen($aExtraPaths) Then
			$aExtraPaths = StringSplit($aExtraPaths, $szPathDelimiter, 1+2)
			$iExtraCount = UBound($aExtraPaths, 1)
		EndIf
	ElseIf IsArray($aExtraPaths) Then
		$iExtraCount = UBound($aExtraPaths)
	EndIf
	
	Local $tPaths, $tPathPtrs
	If $iExtraCount Then
		Local $szStruct = ""
		For $path In $aExtraPaths
			$szStruct &= StringFormat("wchar[%d];", StringLen($path)+1)
		Next
		
		$tPaths = DllStructCreate($szStruct)
		$tPathPtrs = DLLStructCreate(StringFormat("ptr[%d];", $iExtraCount+1))
		
		For $i=1 To $iExtraCount
			DLLStructSetData($tPaths, $i, $aExtraPaths[$i-1])
			DLLStructSetData($tPathPtrs, 1, DLLStructGetPtr($tPaths, $i), $i)
		Next
		DLLStructSetData($tPathPtrs, 1, Ptr(0), $iExtraCount+1)
	EndIf
	
	Local $aRet = DLLCall("shlwapi.dll", "int", "PathFindOnPathW", _
		"wstr"	, $szFile, _
		"ptr"	, DLLStructGetPtr($tPathPtrs) _
	)
	If @error OR NOT $aRet[0] Then Return SetError(1, 0, $szFile)
	
	Return $aRet[1]
EndFunc
