#include <GuiListView.au3>

; #FUNCTION# ;===============================================================================
;
; Name...........: _GUICtrlListView_SetBkHBITMAP
; Description ...: Set in the ListView a background image based on the Bitmap handle
; Syntax.........: _GUICtrlListView_SetBkHBITMAP($hLVWnd, $hBitmap, [$iStyle = 0, [$iXOffset = 0, [$iYOffset = 0, [$bDeleteBitmap = False]]]])
; Parameters ....: $hLVWnd - Control ID/Handle to the control
;                  $hBitmap - Handle to a HBITMAP for background
;                  $iStyle - Determines the background image style:
;					|	0 - Normal
;					|	1 - Tiled
;                  $iXOffset - The value to which  the control's client area that the image should be offset horizontally (see remarks)
;                  $iYOffset - The value to which  the control's client area that the image should be offset vertically (see remarks)
;                  $bDeleteBitmap - Specifies whether to delete the bitmap handle at the end of the function, valid values:
;					|	True - Bitmap will be deleted if the function succeeds
;					|	False - Do not delete, you must release the bitmap when you are finished using it (Default).
; Return values .:	Success:	True
;					Failure:	False
; Author ........: Alofa
; Modified.......:
; Remarks .......: If $hBitmap is not a handle, then the background of the element will be cleared.
;					|If $iStyle = 1, then the values of $iXOffset and $iYOffset will indicate the amount of the image offset relative
;					|to the upper-left corner of the client area of the element in percent. So for example, at 50% the drawing will
;					|be displayed in the center of the client area of the element, at 100% it will be shifted to the right and bottom
;					|borders of the element, respectively. If $iStyle = 0, the values of $iXOffset and $iYOffset will indicate the
;					|amount of the offset  first tile in pixels.
; Related .......: _GUICtrlListView_SetBkImage
; Link ..........; http://msdn.microsoft.com/library/windows/desktop/bb761155(v=vs.85).aspx
; Example .......; Yes
;
; ;==========================================================================================

Func _GUICtrlListView_SetBkHBITMAP($hWnd, $hBitmap, $iStyle = 0, $iXOffset = 0, $iYOffset = 0, $bDeleteBitmap = False)
	Local $LVM_MSG = $LVM_SETBKIMAGEA
	If _GUICtrlListView_GetUnicodeFormat($hWnd) Then $LVM_MSG = $LVM_SETBKIMAGEW
	Local $iRet, $iFlags = BitOR($LVBKIF_STYLE_TILE, $LVBKIF_FLAG_TILEOFFSET)
	
	If $iXOffset = Default Then $iXOffset = 0
	If $iYOffset = Default Then $iYOffset = 0
	
	If IsPtr($hBitmap) Then
		If (Not $iStyle) Or ($iStyle = Default) Then
			$iFlags = $LVBKIF_STYLE_NORMAL
		Else
			$iXOffset *= -1
			$iYOffset *= -1
		EndIf
		$iFlags = BitOR($iFlags, $LVBKIF_SOURCE_HBITMAP)
	Else
		$iFlags = $LVBKIF_SOURCE_NONE
	EndIf

	Local $tLVBKImage = DllStructCreate($tagLVBKIMAGE)
	Local $pLVBKImage = DllStructGetPtr($tLVBKImage)
	DllStructSetData($tLVBKImage, 'Flags', $iFlags)
	DllStructSetData($tLVBKImage, 'hBmp', $hBitmap)
	DllStructSetData($tLVBKImage, 'XOffPercent', $iXOffset)
	DllStructSetData($tLVBKImage, 'YOffPercent', $iYOffset)

	If IsHWnd($hWnd) Then
		If _WinAPI_InProcess($hWnd, $__g_hLVLastWnd) Then
			$iRet = _SendMessage($hWnd, $LVM_SETBKIMAGEW, 0, $pLVBKImage)
		Else
			Local $tMemMap, $iLVBKImageSize = DllStructGetSize($tLVBKImage)
			Local $pMemory = _MemInit($hWnd, $iLVBKImageSize, $tMemMap)
			_MemWrite($tMemMap, $pLVBKImage, $pMemory, $iLVBKImageSize)
			$iRet = _SendMessage($hWnd, $LVM_MSG, 0, $pMemory, 0, 'wparam', 'ptr')
			_MemFree($tMemMap)
		EndIf
	Else
		$iRet = GUICtrlSendMsg($hWnd, $LVM_MSG, 0, $pLVBKImage)
	EndIf
	
	If $bDeleteBitmap And $iRet Then DllCall('gdi32.dll', 'bool', 'DeleteObject', 'handle', $hBitmap)

	If $iFlags = $LVBKIF_SOURCE_NONE Then Return True
	Return $iRet <> 0
EndFunc   ;==>_GUICtrlListView_SetBkHBITMAP
